/************************************************************************
 *                                                                      *
 *            Build and Analyse the various SDK files                   *
 *            =======================================                   *
 *                                                                      *
 *  Copyright 2013 Sirius XM Radio, Inc.                                *
 *  All Rights Reserved.                                                *
 *  Licensed Materials - Property of Sirius XM Radio, Inc.              *
 *                                                                      *
 *    Create and dump tfiles; analyse cfiles                            *
 *                                                                      *
 ************************************************************************/

#ifdef SDKFILES_STANDALONE_BUILD
#ifdef _MSC_VER
    #define _CRTDBG_MAP_ALLOC
    #include <stdlib.h>
    #include <crtdbg.h>
#endif

#include <ctype.h>
#include <sys/stat.h>
#include <time.h>
#include "sdkfiles.h"

/** \name Compiler vs Code compatibility on data types level
 * @{
 */
TYPE_SIZE_STATIC_ASSERT(char, 1);
TYPE_SIZE_STATIC_ASSERT(byte, 1);
TYPE_SIZE_STATIC_ASSERT(ushort, 2);
TYPE_SIZE_STATIC_ASSERT(short, 2);
TYPE_SIZE_STATIC_ASSERT(fix, 4);
TYPE_SIZE_STATIC_ASSERT(int, 4);
TYPE_SIZE_STATIC_ASSERT(uint, 4);
TYPE_SIZE_STATIC_ASSERT(CUINT8, 1);
TYPE_SIZE_STATIC_ASSERT(UINT8, 1);
TYPE_SIZE_STATIC_ASSERT(INT8, 1);
TYPE_SIZE_STATIC_ASSERT(UINT16, 2);
TYPE_SIZE_STATIC_ASSERT(INT16, 2);
TYPE_SIZE_STATIC_ASSERT(CUINT16, 2);
TYPE_SIZE_STATIC_ASSERT(UINT32, 4);
TYPE_SIZE_STATIC_ASSERT(INT32, 4);

/** Command line input options */
#define SDKFILES_CMD_OPT_FEATURE        (0x01)
#define SDKFILES_CMD_OPT_INPUT          (0x02)
#define SDKFILES_CMD_OPT_OUTPUT         (0x04)
#define SDKFILES_CMD_OPT_TFILE          (0x08)
#define SDKFILES_CMD_OPT_CFILE          (0x10)

/** Command line options combinations */
#define SDKFILES_CMD_OPTS_BUILD_TFILE   (SDKFILES_CMD_OPT_FEATURE | \
                                         SDKFILES_CMD_OPT_INPUT | \
                                         SDKFILES_CMD_OPT_OUTPUT)
#define SDKFILES_CMD_OPTS_CHECK_TFILE   (SDKFILES_CMD_OPT_FEATURE | \
                                         SDKFILES_CMD_OPT_TFILE)
#define SDKFILES_CMD_OPTS_CHECK_CFILE   (SDKFILES_CMD_OPT_FEATURE | \
                                         SDKFILES_CMD_OPT_CFILE)

/** For 32-bit int value the structure has to have 8 bytes size 
 * in order to fulfill SXe needs
 * \note There are some systems which has int in 64-bit in 64-bit build
 *       configuration. In this case the check shall fail. Please, contact
 *       to SXe development team if needed.
 */
typedef struct sdkfiles_alignment_check_struct
{
    char one_byte;
    int four_bytes;
} SDKFILES_ALIGNMENT_CHECK_STRUCT;
TYPE_SIZE_STATIC_ASSERT(SDKFILES_ALIGNMENT_CHECK_STRUCT, 8);
/** @} */

typedef struct {
    const char *moduleName;
    /* This function parses file name to get version */
    void(*tfileVersion)(const char *pFileInput, BOOL isArchive, void *pUserData);
    /* This is build function to be used in all builders */
    int (*tfileBuildFunc)(const SXMFilesInput *pFilesInput, int version);
    void (*tfileCheckFunc)(const char *pQuery);
    void (*cfileCheckFunc)(void);
} SdkfilesInterface;

static const SdkfilesInterface lookupTable[] = {
#if (SXM_USE_APOGEE_FULL == 1)
    {"apogee_ptrn", apogee_patterns_ver,    apogee_patterns_build,  apogee_patterns_check_tfile,    NULL},
#endif
    {"apogee_sramp",apogee_sxmramp_ver,     apogee_sxmramp_build,   apogee_sxmramp_check_tfile,     NULL},
    {"apogee",      locations_ver,          locations_build,        locations_check_tfile,          NULL},
    {"canfuel",     fuel_ver,               canfuel_build,          canfuel_check_tfile,            canfuel_check_cfile},
    {"fuel",        fuel_ver,               fuel_build,             fuel_check_tfile,               fuel_check_cfile},
    {"parking",     parking_ver,            parking_build,          parking_check_tfile,            parking_check_cfile},
    {"sports",      NULL,                   NULL,                   NULL,                           sports_check_cfile},
    {"states",      NULL,                   states_build,           states_check_tfile,             NULL},
};

static const int lc = sizeof(lookupTable) / sizeof(SdkfilesInterface);

void fatal(const char *err, ...) {
    va_list ap;
    char errs[1024];
    int res;

    va_start(ap, err);
    res = vsnprintf(errs, sizeof(errs), err, ap);
    va_end(ap);
    if (res < 0)
        strcpy(errs, "log creation failed");

    fprintf(stderr, "ERROR|%s\n", errs);

    return;
}

void non_fatal(const char *err, ...) {
    va_list ap;
    char errs[1024];
    int res;

    va_start(ap, err);
    res = vsnprintf(errs, sizeof(errs), err, ap);
    va_end(ap);
    if (res < 0)
        strcpy(errs, "log creation failed");

    fprintf(stderr, "WARNING|%s\n", errs);

    return;
}

fix fixread(char *inp) {
    uint i = 0, f = 0;
    uint dp = 1;
    int sign = 1;
    char *c = inp;

    /* sign */
    if  (*c == '-') {
        c++;
        sign = -1;
    }

    /* integer part */
    while (*c >= '0' && *c <= '9')
        i = i * 10 + (uint)(*c++ - '0');

    /* fraction */
    if (*c == '.') {
    c++;

    while (*c >= '0' && *c <= '9') {
        f = f * 10 + (uint)(*c++ - '0');
        dp *= 10;
        if (dp == 100000)
        break;
    }
    }

    return sign * (int)((i << 15) + ((f << 15) / dp));
}

byte *bput4(byte *dp, int i) {
    *(int *)(void*)dp = i;
    return dp+4;
}

byte *bput2(byte *dp, short i) {
    *(short *)(void*)dp = i;
    return dp+2;
}

byte *bput1(byte *dp, unsigned char i) {
    *(unsigned char *)dp = i;
    return dp+1;
}

void salign(SXMTFileStream *stream, size_t size) {
    sxm_tstream_alignup(stream, size);
}

void sputb(SXMTFileStream *stream, const void *b, size_t s) {
    sxm_tstream_putb(stream, b, s);
}

void sputs(SXMTFileStream *stream, const char *s) {
    sxm_tstream_puts(stream, s);
}

void sput4(SXMTFileStream *stream, int i) {
    sxm_tstream_put(stream, i);
}

void sput2(SXMTFileStream *stream, short i) {
    sxm_tstream_put(stream, i);
}

void sput1(SXMTFileStream *stream, unsigned char i) {
    sxm_tstream_put(stream, i);
}

byte *bputs(byte *dp, String s) {
    strcpy((char*)dp, s);
    return dp + strlen(s) + 1;
}

byte *bget1(byte *dp, unsigned char* pi) {
    *pi = *(unsigned char*)dp;
    return dp + sizeof(unsigned char);
}

byte *bget2(byte *dp, ushort *pi) {
    *pi = *(ushort*)(void*)dp;
    return dp + sizeof(ushort);
}

byte *bget4(byte *dp, int *pi) {
    *pi = *(int*)(void*)dp;
    return dp + sizeof(int);
}

byte *bgets(byte *dp, String *ps) {
    size_t len = strlen((const char*)dp);
    *ps = (String)dp;
    return dp + len + 1;
}

static void usage(String err, ...) {
    va_list va;
    char errs[256];
    int i;

    if (err) {
        va_start(va, err);
        vsnprintf(errs, sizeof(errs), err, va);
        errs[255] = '\0';
        va_end(va);
        fprintf(stderr, "ERROR: %s\n\n", errs);
    }

    fputs("usage:  sdkfiles <options>\n", stderr);
    fputs("    <options> are:\n", stderr);
    fprintf(stderr, "        -f  ");
    for (i = 0; i < (lc - 1); ++i) {
        fprintf(stderr, " %s |", lookupTable[i].moduleName);
        if ((i % 5 == 0) && (i > 1)) {
            fprintf(stderr, "\n            ");
        }
    }
    fprintf(stderr, " %s\n", lookupTable[i].moduleName);
    fputs("            (define file format)\n", stderr);
    fputs("        -d  base directory (optional)\n", stderr);
    fputs("        -i  semicolon delimited input file(s)\n", stderr);
    fputs("        -o  output file path and template\n", stderr);
    fputs("            (template format: ...*T...*M...*F...\n", stderr);
    fputs("             for example: C:/Folder/*T/*M/*F)\n", stderr);
    fputs("        -v  version\n", stderr);
    fputs("            (force baseline version)\n", stderr);
    fputs("        -t  transaction file path or template\n", stderr);
    fputs("            (analyse a transaction/baseline file)\n", stderr);
    fputs("        -c  cycle file path or template\n", stderr);
    fputs("            (analyse an engine-cycle file)\n", stderr);
    fputs("        -q  tql query\n", stderr);
    fputs("            (filter output based on tql)\n", stderr);
    exit(0);
}

void dump(const byte *b, uint len) {
    char text[40];
    uint i, j;

    text[32] = '\0';
    for (i = 0, j = 0; i < len; i++) {
        if (j == 0)
            printf("%6u: ", i);
        if (*b >= ' ' && *b < 0x7f)
            text[j] = (char)*b;
        else
            text[j] = '.';

        printf("%02x", *b++);
        j++;
        if (j > 0 && (j % 4) == 0)
            printf(" ");
        if (j == 32) {
            printf("  '%s'\n", text);
            fflush(stdout);
            j = 0;
        }
    }

    if (j > 0) {
        while (j < 32) {
            printf("  ");
            if ((j % 4) == 0) {
                printf(" ");
            }
            text[j] = '.';
            if (++j == 32) {
                printf("  '%s'\n", text);
            }
        }
    }
}

static int fileSize(const char *path) {
    struct stat ss;

    if (stat(path, &ss) == -1) {
        return -1;
    }

    return (int)ss.st_size;
}

byte *fileLoad(const char *path, uint *len) {
    FILE *ft;
    int size;
    byte *ret;

    size = fileSize(path);
    if (size <= 0) {
        return NULL;
    }

    *len = (uint)size;

    ft = fopen(path, "rb");
    if (ft == NULL) {
        return NULL;
    }

    ret = (byte *)sxe_malloc(*len + 16);
    if (ret != NULL) {
        if (*len != fread(ret, 1, *len, ft)) {
            fclose(ft);
            sxe_free(ret);
            return NULL;
        }
        ret[*len] = '\0';
    }

    fclose(ft);
    return ret;
}

const char *get_file_name_pointer(const char *pFilePath) {

    const char *pFileName = strrchr(pFilePath, '/');

    if (NULL == pFileName) {
        pFileName = strrchr(pFilePath, '\\');
    }

    if (NULL != pFileName) {
        ++pFileName;
    }
    else {
        pFileName = pFilePath;
    }

    return pFileName;
}

int strncmp_lowercase(const char *pStr1, const char *pStr2, size_t len) {

    /* Case-independent comparison */
    while (len > 0) {
        if (('\0' == *pStr1) || ('\0' == *pStr2)) {
            break;
        }

        if (tolower(*(pStr1++)) != tolower(*(pStr2++))) {
            break;
        }

        len--;
    }

    return (int)len;
}

const char * strstr_lowercase(const char *pStr1, const char *pStr2) {

    size_t str1Len = strlen(pStr1);
    size_t str2Len = strlen(pStr2);

    while (str1Len >= str2Len) {
        if (0 == strncmp_lowercase(pStr1, pStr2, str2Len)) {
            return pStr1;
        }
        pStr1++;
        str1Len--;
    }

    return NULL;
}

int create_buffer_psv_parser(const SXMFilesInputRecord *pFileRecord,
                             char separator, uint lineSizeMax,
                             SxmCsvParser **ppRet)
{
    int rc;
    void *pData;
    size_t dataSize;

    rc = sxm_files_input_alloc_and_read(pFileRecord, &pData, &dataSize);
    if (SXM_E_OK != rc) {
        non_fatal("Input file read failed");
    }
    else {
        rc = sxm_csv_buffer_new(pData, dataSize,
                                lineSizeMax,
                                SXM_CSV_OPTION_DATA_OWNER,
                                separator, ppRet);
        if (SXM_E_OK != rc) {
            non_fatal("CSV parser creation failed");
            sxe_free(pData);
        }
    }

    return rc;
}

/************************************************************************
 *                                                                      *
 *            CRC-32 generator                                          *
 *            ================                                          *
 *                                                                      *
 *    IEEE 802.3 CRC as used in PNG                                     *
 *                                                                      *
 ************************************************************************/

static int initcrc = 0;
static uint crc[256];

static void buildCrC(void) {
    uint c;
    uint n,k;

    for (n = 0; n < 256; n++) {
        c = n;
        for (k = 0; k < 8; k++)
            if (c & 1)
                c = 0xedb88320 ^ (c >> 1);
            else
                c =c >> 1;
        crc[n] = c;
    }
    initcrc = 1;
}

uint doCRC_part(uint c, const void *buf, uint len) {
    uint i;

    if  (initcrc == 0)
        buildCrC();

    for (i = 0; i < len; i++)
        c = crc[(c ^ ((const byte*)buf)[i]) & 0xff] ^ (c >> 8);
    return c;
}

uint doCRC(const void *buf, uint len) {
    return doCRC_part(0xffffffff, buf, len) ^ 0xffffffff;
}

/************************************************************************
 *                                                                      *
 *            CRC-16 generator                                          *
 *            ================                                          *
 *                                                                      *
 ************************************************************************/

static int initcrc16 = 0;
/* Precomputed values for CRC calculation */
static ushort sxm_crc16_table[256];

static void sxm_crc16_initialize(void) {
    /* Generate CRC table */
    ushort c;
    ushort n, k;

    for (n = 0; n < 256; n++) {
        c = (ushort)(n << 8);
        for (k = 0; k < 8; k++)
            if (c & 0x8000)
                c = (ushort)(0x1021 ^ (c << 1));
            else
                c = (ushort)(c << 1);

        sxm_crc16_table[n] = c;
    }
    initcrc16 = 1;
}

ushort sxm_crc16_calculate(const void *buf, uint count) {
    uint i;
    ushort crcval;

    if (initcrc16 == 0)
        sxm_crc16_initialize();

    crcval = 0x1d0f;

    for (i = 0; i < count; ++i)
        crcval = (ushort)((crcval << 8) ^ sxm_crc16_table[*((const byte*)buf + i) ^ (byte)(crcval>>8)]);
    return crcval;
}

#if SXM_DEBUG_MEMORY_TRACKING
/************************************************************************
 *                                                                      *
 *            SXe Memory Tracking Functions                             *
 *            ==================                                        *
 *                                                                      *
 ************************************************************************/

#endif

/************************************************************************
 *                                                                      *
 *            SXe File Emulation                                        *
 *            ==================                                        *
 *                                                                      *
 ************************************************************************/

/** Makepath function specified in sdkfiles_start() */
static SXMMakePathCallback_t gMakePathFunc = NULL;
/** Output path from command line */
static char *gMakePathParam = NULL;
/** Buffer used in makepath callback */
static char gMakePathBuffer[SDK_FILE_PATH_LEN_MAX + 1];

/* The function prses pFormatStr in format :<original>:<replacement>:...
   compares pOriginalStr parameter with <original>.
   The function returns pointer to pOriginalStr
   if it does not match <original> or if pFormatStr format does not correspond
   to the format described above.
   The function returns pointer to <replacement> if pOriginalStr matches <original>.
   pStrLen is populated with the length of the returned string.
   pFmtStrLen is populated with format length counting from the
   opening ':' to the closing ':'. */
static const char * check_replacement_format(const char *pFormatStr,
                                             const char *pOriginalStr,
                                             size_t *pStrLen,
                                             size_t *pFmtStrLen) {
    *pStrLen = strlen(pOriginalStr);
    *pFmtStrLen = 0;

    /* 1 - seeking for original string */
    if (':' == *pFormatStr) {
        int result;
        /* 1a  - determining start and finish in our search string */
        const char *closing_column;
        const char *start = pFormatStr + 1;

        const char *finish = strchr(pFormatStr, SDKFILES_MAKEPATH_TEMPLATE_SPECIFIER);
        if (finish == NULL) {
            size_t len = strlen(pFormatStr);
            finish = start + len;
        }

        while ((finish > pFormatStr) && (*finish != ':')) {
            finish--;
        }

        *pFmtStrLen = (size_t)(finish - pFormatStr + 1);

        if (finish == pFormatStr) {
            /* looks like no replacement here */
            return pOriginalStr;
        }

        do {
            /* 1b - looking for closing column */
            closing_column = strchr(start, ':');
            if ((closing_column == NULL) || (closing_column > finish)) {
                /* if closing column is farer than finish of format specifier or even does not
                   exist, returning original string pointer */
                return pOriginalStr;
            }

            /* 1b - comparing string between columns to find out original string */
            result = strncmp(start, pOriginalStr, (size_t)(closing_column - start));
            if (result == 0) {
                break;
            }
            /* 1c - moving to next search position. it should be after the replacement string */
            start = strchr(closing_column + 1, ':');
            if (start == NULL) {
                return pOriginalStr;
            }
            start++;
        } while (start < finish);

        if (result != 0) {
            /* 1d - nothing was found, original string should be used */
            return pOriginalStr;
        }

        /* 2 - filling replacement string */
        start = closing_column + 1;
        closing_column = strchr(start, ':');
        if ((closing_column != NULL) && (closing_column <= finish)) {
            *pStrLen = (size_t)(closing_column - start);
            return start;
        }
    }

    return pOriginalStr;
}

/***************************************************************************//**
 * The function builds destination path based on function paramaters.
 *
 * \param[in,out] path buffer used to build the destination path
 * \param[in] pathSize path buffer size in bytes
 * \param[in] type file type
 * \param[in] module module name
 * \param[in] file file name
 *
 * \note: gMakePathParam variable points to direct file path
 * or to the destination path template with following specifiers:
 * ...*T...*M...*F...
 * where *T will be replaced with the type parameter value
 *       *M will be replaced with the module parameter value
 *       *F will be replaced with the file parameter value
 * for example:
 * template "C:\sirius-xm\*T\*M\*F"
 * will result following path for ev DB file:
 * C:\sirius-xm\W\ev\evdb
 *
 * \return destination path pointer
 *
 ******************************************************************************/
static char *sdkfiles_makepath(char *path,
                               uint pathSize,
                               char type,
                               const char *module,
                               const char *file) {

    if ((NULL != module) && (NULL != file)) {

        /* gMakePathParam should point to the direct file path
           or to the template */
        const char *pInputPtr = gMakePathParam;
        const char *pSpecifierPtr = NULL;
        size_t numPrinted = 0;
        size_t formatPartLen = 0;

        while ('\0' != *pInputPtr) {
            pSpecifierPtr = strchr(pInputPtr,
                                   SDKFILES_MAKEPATH_TEMPLATE_SPECIFIER);
            if (NULL == pSpecifierPtr) {
                /* No more specifiers found */
                break;
            }

            formatPartLen = (size_t)(pSpecifierPtr - pInputPtr);
            if (formatPartLen >= (pathSize - numPrinted)) {
                non_fatal("Insufficient makepath buffer size");
                return NULL;
            }

            /* Print next part */
            strncpy(&path[numPrinted], pInputPtr, formatPartLen);
            numPrinted += formatPartLen;

            /* Move input pointer to the position right after found specifier */
            pInputPtr = pSpecifierPtr + 2;

            /* Print next specifier */
            switch (pSpecifierPtr[1]) {
                case 't':
                case 'T': {
                    size_t formatLen = 0;
                    char type_str[2];
                    type_str[0] = type;
                    type_str[1] = '\0';
                    pSpecifierPtr = check_replacement_format(&pSpecifierPtr[2],
                        type_str, &formatPartLen,
                        &formatLen);
                    pInputPtr += formatLen;
                }
                break;

                case 'm':
                case 'M': {
                    size_t formatLen = 0;
                    pSpecifierPtr = check_replacement_format(&pSpecifierPtr[2],
                        module, &formatPartLen,
                        &formatLen);
                    pInputPtr += formatLen;
                }
                break;

                case 'f':
                case 'F': {
                    size_t formatLen = 0;
                    pSpecifierPtr = check_replacement_format(&pSpecifierPtr[2],
                                                             file, &formatPartLen,
                                                             &formatLen);
                    pInputPtr += formatLen;
                }
                break;

                default: {
                    non_fatal("Invalid path template specifier: '%c'",
                              *(pSpecifierPtr + 1));
                    return NULL;
                }
                break;
            }

            if (formatPartLen >= (pathSize - numPrinted)) {
                non_fatal("Insufficient makepath buffer size");
                return NULL;
            }

            /* Print appropriate specifier */
            strncpy(&path[numPrinted], pSpecifierPtr, formatPartLen);
            numPrinted += formatPartLen;
        }

        /* Print the tail if required */
        if ('\0' != *pInputPtr) {
            if (formatPartLen >= (pathSize - numPrinted)) {
                non_fatal("Insufficient makepath buffer size");
                return NULL;
            }

            formatPartLen =
                (size_t)(&gMakePathParam[strlen(gMakePathParam)] - pInputPtr);

            strncpy(&path[numPrinted], pInputPtr, formatPartLen);
            numPrinted += formatPartLen;
        }

        path[numPrinted] = '\0';

        return &path[0];
    }

    return NULL;
}

/** Initialize file generators
 * \param[in] makePath callback for real file path generations
 */
void sdkfiles_start(SXMMakePathCallback_t makePath) {
    gMakePathFunc = makePath;
}

/** De-initialize file generators */
void sdkfiles_stop(void) {
    gMakePathFunc =NULL;
}

const char *sdkfiles_get_path(char type, const char *module, const char *file) {

    if (NULL != gMakePathFunc) {
        return gMakePathFunc(gMakePathBuffer,
                             sizeof(gMakePathBuffer),
                             type, module, file);
    }
    return NULL;
}

/** Replica of the SXe API \ref sxm_file_open */
FILE *sdkfiles_file_open(char type,
                         const char *module,
                         const char *file,
                         const char *mode) {

    FILE *pFile = NULL;

    if ((NULL != gMakePathFunc) && (NULL != mode)) {
        const char* pFilePath = gMakePathFunc(gMakePathBuffer,
                                              sizeof(gMakePathBuffer),
                                              type, module, file);
        if (NULL != pFilePath) {
            pFile = fopen(pFilePath, mode);
            if (NULL == pFile) {
                non_fatal("Failed to open file '%s' with mode '%s'",
                          pFilePath, mode);
            }
        }
        else {
            non_fatal("Makepath failed for '%c/%s/%s'",
                      type, module, file);
        }
    }

    return pFile;
}

/** Replica of the SXe API \ref sxm_file_open_or_create */
FILE *sdkfiles_file_open_or_create(char type,
                                   const char *module,
                                   const char *file,
                                   const char *mode) {

    FILE *pFile = NULL;

    if ((NULL != gMakePathFunc) && (NULL != mode)) {
        const char* pFilePath = gMakePathFunc(gMakePathBuffer,
                                              sizeof(gMakePathBuffer),
                                              type, module, file);
        if (NULL != pFilePath) {
            pFile = fopen(pFilePath, "r+b");

            if (NULL == pFile) {
                pFile = fopen(pFilePath, "w+b");
            }

            if (NULL != pFile) {
                fclose(pFile);
                pFile = fopen(pFilePath, mode);
                if (NULL == pFile) {
                    non_fatal("Failed to open file '%s' with mode '%s'",
                              pFilePath, mode);
                }
            }
        }
        else {
            non_fatal("Makepath failed for '%c/%s/%s'",
                      type, module, file);
        }
    }
    return pFile;
}

/** Replica of the SXe API \ref sxm_file_remove */
void sdkfiles_file_remove(char type,const char* module, const char* file) {

    if (NULL != gMakePathFunc) {
        const char* pFilePath = gMakePathFunc(gMakePathBuffer,
                                              sizeof(gMakePathBuffer),
                                              type, module, file);
        if (NULL != pFilePath) {
            remove(pFilePath);
        }
        else {
            non_fatal("Makepath failed for '%c/%s/%s'",
                      type, module, file);
        }
    }
}

/** Replica of the SXe API \ref sxm_file_close */
int sdkfiles_file_close(FILE *pFile) {

    if (NULL == pFile) {
        return SXM_E_FAULT;
    }

    if (0 != fclose(pFile)) {
        return SXM_E_PIPE;
    }

    return SXM_E_OK;
}

/** Replica of the SXe API \ref sxm_GMD_extract_time */
int sdkfiles_sxi_extract_time(SXMTime *v) {
    (void)v;
    return SXM_E_OK;
}

void printColumn(uint colWidth,
                 const char *format,
                 ...)
{
    int rc;
    uint numPrinted = 0;
    uint numPaddingChars = 0;

    va_list args;
    va_start(args, format);

    /* Print arguments */
    numPrinted += (uint)printf(" ");
    rc = vprintf(format, args);
    if (rc > 0) {
        numPrinted += (uint)rc;
    }

    /* Calculate number of padding characters */
    if (colWidth > numPrinted) {
        numPaddingChars = colWidth - numPrinted;
    }

    /* Print padding characters and separator */
    printf("%*s", numPaddingChars, "|");

    va_end(args);

    return;
}

void printFooter(uint length)
{
    while (0 != length--) {
        putchar('-');
    }

    putchar('\n');
}

int main(int argc, char *argv[]) {

    int rc = SXM_E_OK;
    uint inputOptions = 0;
    int ix;
    char *pFeature = NULL;
    char *pBuildInput = NULL;
    char *pBuildOutput = NULL;
    char *pInputTfile = NULL;
    char *pInputCfile = NULL;
    char *pQuery = NULL;
    char *pBaseDir = NULL;
    int version = -1;

#if defined(_MSC_VER) && defined(_DEBUG)
    /* Get current flag */
    const int tmpFlag = _CrtSetDbgFlag(_CRTDBG_REPORT_FLAG) | _CRTDBG_ALLOC_MEM_DF | _CRTDBG_LEAK_CHECK_DF;
    _CrtSetDbgFlag(tmpFlag);
#endif

    setbuf(stdout, NULL);

    if (argc == 1) {
        usage("Options expected");
    }

    ix = 1;
    while (ix < argc) {

        char *opt = argv[ix];

        if (*opt != '-') {
            usage("Option expected at '%s'", opt);
        }

        opt++;
        switch (*opt) {
            case 'f': {
                if ('\0' != opt[1]) {
                    pFeature = opt + 1;
                } else if ((ix + 1) < argc) {
                    pFeature = argv[++ix];
                } else {
                    usage("Value expected after '-f' option");
                }
                inputOptions |= SDKFILES_CMD_OPT_FEATURE;
            }
            break;

            case 'i': {
                if ('\0' != opt[1]) {
                    pBuildInput = opt + 1;
                } else if ((ix + 1) < argc) {
                    pBuildInput = argv[++ix];
                } else {
                    usage("Value expected after '-i' option");
                }
                inputOptions |= SDKFILES_CMD_OPT_INPUT;
            }
            break;

            case 'o': {
                if ('\0' != opt[1]) {
                    pBuildOutput = opt + 1;
                } else if ((ix + 1) < argc) {
                    pBuildOutput = argv[++ix];
                } else {
                    usage("Value expected after '-o' option");
                }
                inputOptions |= SDKFILES_CMD_OPT_OUTPUT;
            }
            break;

            case 't': {
                if ('\0' != opt[1]) {
                    pInputTfile = opt + 1;
                } else if ((ix + 1) < argc) {
                    pInputTfile = argv[++ix];
                } else {
                    usage("Value expected after '-t' option");
                }
                inputOptions |= SDKFILES_CMD_OPT_TFILE;
            }
            break;

            case 'c': {
                if ('\0' != opt[1]) {
                    pInputCfile = opt + 1;
                } else if ((ix + 1) < argc) {
                    pInputCfile = argv[++ix];
                } else {
                    usage("Value expected after '-c' option");
                }
                inputOptions |= SDKFILES_CMD_OPT_CFILE;
            }
            break;

            case 'q': {
                if ('\0' != opt[1]) {
                    pQuery = opt + 1;
                } else if ((ix + 1) < argc) {
                    pQuery = argv[++ix];
                } else {
                    usage("Value expected after '-q' option");
                }
            }
            break;

            case 'v': {
                if ('\0' != opt[1]) {
                    version = atoi(opt + 1);
                } else if ((ix + 1) < argc) {
                    version = atoi(argv[++ix]);
                } else {
                    usage("Value expected after '-v' option");
                }
            }
            break;

            case 'd': {
                if ('\0' != opt[1]) {
                    pBaseDir = opt + 1;
                } else if ((ix + 1) < argc) {
                    pBaseDir = argv[++ix];
                } else {
                    usage("Value expected after '-d' option");
                }
            }
            break;

            default: {
                usage("Unknown option '%s'", opt);
            }
            break;
        }

        ix++;
    }

    if (0 == (inputOptions & SDKFILES_CMD_OPT_FEATURE)) {
        usage("Missing file format");
    }

    /* Valid inputOptions combinations:
       SDKFILES_CMD_OPTS_BUILD_TFILE
       SDKFILES_CMD_OPTS_CHECK_TFILE
       SDKFILES_CMD_OPTS_CHECK_CFILE */

    for (ix = 0; ix < lc; ix++) {
        if (strequ(pFeature, lookupTable[ix].moduleName)) {
            time_t start_time, end_time;
            time(&start_time);

            /* Set makepath function */
            sdkfiles_start(sdkfiles_makepath);

            if (SDKFILES_CMD_OPTS_BUILD_TFILE == inputOptions) {
                /* verifying ouput path - it should contain *T, *M and *F */
                if ((strstr(pBuildOutput, "*T") == NULL) ||
                    (strstr(pBuildOutput, "*M") == NULL) ||
                    (strstr(pBuildOutput, "*F") == NULL)) {
                    usage("Template in output path %s is wrong: it "
                        "should contain *T, *M and *F combinations", pBuildOutput);
                }

                printf("Building %s: '%s' into '%s'\n",
                       lookupTable[ix].moduleName, pBuildInput, pBuildOutput);

                gMakePathParam = pBuildOutput;

                if (NULL != lookupTable[ix].tfileBuildFunc) {

                    /* Allocate and init files input object */
                    SXMFilesInput *pFilesInput = NULL;
                    int *pVersion = NULL;
                    SXM_FSINPUT_PATH_CALLBACK pCallback = NULL;

                    if (-1 == version) {
                        pCallback = lookupTable[ix].tfileVersion;
                        pVersion = &version;
                    }

                    rc = sxm_files_input_create(';', pBaseDir, pBuildInput,
                        pCallback, pVersion, &pFilesInput);
                    if (SXM_E_OK != rc) {
                        non_fatal("Failed to process files input (rc=%d)", rc);
                        break;
                    }

                    /* Call appropriate builder */
                    rc = lookupTable[ix].tfileBuildFunc(pFilesInput, version);

                    /* Cleanup */
                    sxm_files_input_destroy(pFilesInput);
                }
                else {
                    non_fatal("No build method specified");
                }

                if (SXM_E_OK != rc) {
                    non_fatal("DB Creation error (rc=%d)", rc);
                }
            }
            else if ((SDKFILES_CMD_OPTS_CHECK_TFILE == inputOptions) &&
                     (NULL != lookupTable[ix].tfileCheckFunc)) {
                printf("Checking DB File: %s\n", pInputTfile);
                gMakePathParam = pInputTfile;
                lookupTable[ix].tfileCheckFunc(pQuery);
            }
            else if ((SDKFILES_CMD_OPTS_CHECK_CFILE == inputOptions) &&
                     (NULL != lookupTable[ix].cfileCheckFunc)) {
                printf("Checking Cycle File: %s\n", pInputCfile);
                gMakePathParam = pInputCfile;
                lookupTable[ix].cfileCheckFunc();
            }
            else {
                usage("Invalid options for '%s' format.", pFeature);
            }

            sdkfiles_stop();

#if SXM_DEBUG_MEMORY_TRACKING
            {
                size_t totalBlocks = 0, totalSize = 0,
                       maxSize = 0, maxBlocks = 0;
                sxe_mem_stat(&totalBlocks, &totalSize, &maxBlocks, &maxSize);
                printf("Memory statistics:\n"
                       "  Total: %u byte(s) in %u block(s)"
                       "    Max: %u byte(s) in %u block(s)\n",
                       totalSize, totalBlocks, maxSize, maxBlocks);
            }
#endif
            time(&end_time);
            printf("Operation done in %u sec(s)\n",
                   (unsigned int)(end_time - start_time + 1));
            break;
        }
    }

    if (ix == lc) {
        usage("Unknown format: %s", pFeature);
    }

    return rc;
}

#endif /* #ifdef SDKFILES_STANDALONE_BUILD */
