/************************************************************************
 *                                                                      *
 *            Build and Analyze the various SDK files                   *
 *            =======================================                   *
 *                                                                      *
 *  Copyright 2015 Sirius XM Radio, Inc.                                *
 *  All Rights Reserved.                                                *
 *  Licensed Materials - Property of Sirius XM Radio, Inc.              *
 *                                                                      *
 ************************************************************************/

#ifndef SXM_TQL_MATH_SUPPORT
    /** Triggers the real mathematical function support */
    #define SXM_TQL_MATH_SUPPORT 0
#endif

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>
#if SXM_TQL_MATH_SUPPORT
    #include <math.h>
#endif

#include "sdkfiles.h"
#include "sdk_tql.h"

/**
 * \name Logging
 * @{
 */
#define SXMTQL_LEX_LOG (1)
#define SXMTQL_SYN_LOG (2)
#define SXMTQL_RPN_LOG (4)
#define SXMTQL_TQL_LOG (8)
#ifndef SXMTQL_LOG
    #define SXMTQL_LOG 0
#endif

#if SXMTQL_LOG
    #define LOG(_comp, _f, ...) \
        fprintf(stdout, "[%s] %s-> " _f "\n", _comp, __FUNCTION__, __VA_ARGS__)
#else
    #define LOG(...)
#endif

#if SXMTQL_LOG & SXMTQL_LEX_LOG
    #define LOG_LEX(_f, ...) LOG("lex", _f, __VA_ARGS__)
#else 
    #define LOG_LEX(...)
#endif
#if SXMTQL_LOG & SXMTQL_SYN_LOG
    #define LOG_SYN(_f, ...) LOG("syn", _f, __VA_ARGS__)
#else
    #define LOG_SYN(...)
#endif
#if SXMTQL_LOG & SXMTQL_RPN_LOG
    #define LOG_RPN(_f, ...) LOG("rpn", _f, __VA_ARGS__)
#else
    #define LOG_RPN(...)
#endif
#if SXMTQL_LOG & SXMTQL_TQL_LOG
    #define LOG_TQL(_f, ...) LOG("tql", _f, __VA_ARGS__)
#else
    #define LOG_TQL(...)
#endif

/**
 * @}
 */

/**
 * \name LEXICAL ANALYZER
 * @{
 */
 
 /** Defines max lexeme length */

typedef enum
{
    LEX_E_OK,
    LEX_E_EOL,
    LEX_E_UNKNOWN
} LexError;

/** Lexeme types */
typedef enum
{
    /** Special cases */
    LEX_TOKEN_UNKNOWN     = -1, //!< Unknown lexeme

    /** One symbol length */
    LEX_TOKEN_APOSTROPHE  = '\'',
    LEX_TOKEN_COMMA       = ',',
    LEX_TOKEN_EOL         = '\0',
    LEX_TOKEN_EQUAL       = '=',
    LEX_TOKEN_GREATER     = '>',
    LEX_TOKEN_INCH        = '"',
    LEX_TOKEN_LESS        = '<',
    LEX_TOKEN_MINUS       = '-',
    LEX_TOKEN_PERC        = '%',
    LEX_TOKEN_PERIOD      = '.',
    LEX_TOKEN_PLUS        = '+',
    LEX_TOKEN_ROUND_LEFT  = '(',
    LEX_TOKEN_ROUND_RIGHT = ')',
    LEX_TOKEN_SEMICOLON   = ';',
    LEX_TOKEN_SLASH       = '/',
    LEX_TOKEN_SPACE       = ' ',
    LEX_TOKEN_STAR        = '*',
    LEX_TOKEN_UNDERSCORE  = '_',
    
    /** SEPARATOR to make sure that no one value after this
      * overlaps ASCII table
      */
    LEX_TOKEN_MAX =  (1 << (8 * sizeof(char))), 

    /** Double symbols length */
    LEX_TOKEN_LE, //!< \c <=
    LEX_TOKEN_GE, //!< \c >=
    LEX_TOKEN_NOT_EQUAL, //!< \c <>
    
    /** Multiple symbols length */
    LEX_TOKEN_IDENTIFIER, //!< Sequence of the alpha, digits and underscore
                          ///  starting with alpha.
    LEX_TOKEN_STRING, //!< Sequence of the symbols between inch or apostrophe symbols.
    LEX_TOKEN_NUMBER, //!< Sequence of the digits starting from the digit and
                      ///  may contain one period symbol to specify real value.
} LexTokenType;

typedef struct
{
    LexTokenType type; //!< Toke type
    union
    {
        const char *s; //!< String representation
        double real; //!< Real representation
    } value; //!< Token own value
} LexToken; //!< Token descriptor

typedef struct
{
    const char *original;  //!< Original string
    const char *tokenizer; //!< Currently processing
    char *buffer;          //!< Temp buffer for strings
    char *buffer_ptr;      //!< Temp buffer iterator
    LexToken sCurToken; //!< Keeps the latest read token
    LexError error; //!< Current error code.
} Lex; //!< Lexical analyzer

#define lex_token_str(_t)     ((_t)->value.s)
#define lex_token_real(_t)    ((_t)->value.real)
#define lex_error_code(_lex)  ((_lex)->error)
#define lex_failure(_lex, _c) (lex_error_code(_lex) = (_c))
#define lex_is_ok(_lex)       (lex_error_code(_lex) == LEX_E_OK)

/** Skips over the next symbol
 * \param[in] _lex lexical analyzer instance
 * \return none
 */
#define lex_skip(_lex) ((_lex)->tokenizer++)

/** Reads out the next symbol
 * \param[in] _lex lexical analyzer instance
 * \return just read symbol
 */
#define lex_next(_lex) (*((_lex)->tokenizer++))

/** Provides current symbol
 * \param[in] _lex lexical analyzer instance
 * \return current symbol
 */
 #define lex_cur(_lex) (*(_lex)->tokenizer)

/** Initialize the lexical analyzer
 * \param[in] lex lexical analyzer instance
 * \param[in] query the input string for the parsing
 */
static void lex_init(Lex *lex, const char *query)
{
    LOG_LEX("query '%s'", query);
    memset(lex, 0, sizeof(*lex));
    lex->original = lex->tokenizer = query;
    lex->buffer = (char*)sxe_calloc(1, strlen(query) + 1);
    lex->buffer_ptr = lex->buffer;
    lex_failure(lex, LEX_E_OK);
}

/** Initialize the lexical analyzer
 * \param[in] lex lexical analyzer instance
 */
static void lex_uninit(Lex *lex)
{
    if (lex->buffer)
    {
        sxe_free(lex->buffer);
    }
    memset(lex, 0, sizeof(*lex));
}

#if SXMTQL_LOG & SXMTQL_TQL_LOG
/** Dumps the lex token to the file
 * \param[in] f output file
 * \param[in] token the lex-token instance to be dumped
 */
static void lex_token_dump(FILE* f, const LexToken *token)
{
    fprintf(f, "(%3d) ", token->type);
    switch (token->type)
    {
        case LEX_TOKEN_NUMBER:
            fprintf(f, "real(%lf)", lex_token_real(token));
            break;
        case LEX_TOKEN_STRING:
            fprintf(f, "string(%s)", lex_token_str(token));   
            break;
        case LEX_TOKEN_IDENTIFIER:
            fprintf(f, "identifier(%s)", lex_token_str(token));
            break;
        case LEX_TOKEN_GE:
            fprintf(f, "op(>=)");
            break;
        case LEX_TOKEN_LE:
            fprintf(f, "op(<=)");
            break;
        case LEX_TOKEN_EQUAL:
            fprintf(f, "op(=)");
            break;
        case LEX_TOKEN_NOT_EQUAL:
            fprintf(f, "op(<>)");
            break;
        default:
            fprintf(f, "op(%c)", (int)token->type);
            break;
    }
}
#endif

/** Returns the lexeme type for the given symbol
 * \param[in] s address of the string which is being parsed
 * \return lexeme type
 */
static LexTokenType lex_token_type(int s)
{
    LexTokenType result;
    if (s < 0)
        result = LEX_TOKEN_UNKNOWN;
    else if (isspace(s))
        result = LEX_TOKEN_SPACE;
    else if (isalpha(s))
        result = LEX_TOKEN_IDENTIFIER;
    else if (isdigit(s))
        result = LEX_TOKEN_NUMBER;
    else switch (s)
    {
        case LEX_TOKEN_APOSTROPHE:
        case LEX_TOKEN_COMMA:
        case LEX_TOKEN_EOL:
        case LEX_TOKEN_EQUAL:
        case LEX_TOKEN_GREATER:
        case LEX_TOKEN_INCH:
        case LEX_TOKEN_LESS:
        case LEX_TOKEN_MINUS:
        case LEX_TOKEN_PERC:
        case LEX_TOKEN_PERIOD:
        case LEX_TOKEN_PLUS:
        case LEX_TOKEN_ROUND_LEFT:
        case LEX_TOKEN_ROUND_RIGHT:
        case LEX_TOKEN_SEMICOLON:
        case LEX_TOKEN_SLASH:
        case LEX_TOKEN_STAR:
        case LEX_TOKEN_UNDERSCORE:
            result = (LexTokenType)s;
            break;
        default:
            result = LEX_TOKEN_UNKNOWN;
            break;
    }

    LOG_LEX("%d", (int)result);

    return result;
}

/** Reads next token for the passes string
 *
 * This function should be called while the string has not been completely parsed. 
 *
 * As a rule the tokens shall be pop-up from the string until UNKNOWN lexeme or some obvious
 * end of the expression. The \c type argument should be used to detect the end of the parsing
 *
 * \param[in] lex lexical analyzer instance
 * \param[out] token place holder for read token.
 */
static void lex_token_next(Lex *lex, LexToken *token)
{
    LexTokenType tType;
    const char *str_start;
    char *str_out;
    
    /*
     * Skip all spaces until meaningful symbols.
     */
    do
    {
        tType = lex_token_type(lex_cur(lex));
        if (tType == LEX_TOKEN_SPACE)
        {
            lex_skip(lex);
        }
    } while (tType == LEX_TOKEN_SPACE);

    LOG_LEX("type: %d", (int)tType);

    /*
     * Read number
     */
    switch (tType)
    {
        /*
         * Read while digits or . and digits
         */
        case LEX_TOKEN_NUMBER:
        {
            str_start = str_out = lex->buffer_ptr;
            *str_out++ = lex_next(lex);

            tType = lex_token_type(lex_cur(lex));
            while ((tType == LEX_TOKEN_NUMBER) ||
                   (tType == LEX_TOKEN_PERIOD))
            {
                *str_out++ = lex_next(lex);
                tType = lex_token_type(lex_cur(lex));    
            }
            *str_out = '\0';
            token->type = LEX_TOKEN_NUMBER;
            lex_token_real(token) = atof(str_start);
        }
        break;
        /*
         * Read identifiers in lower case always
         */
        case LEX_TOKEN_IDENTIFIER:
        {
            str_start = str_out = lex->buffer_ptr;
            *str_out++ = (char)tolower(lex_next(lex));
            tType = lex_token_type(lex_cur(lex));
            while ((tType == LEX_TOKEN_IDENTIFIER) ||
                   (tType == LEX_TOKEN_NUMBER) ||
                   (tType == LEX_TOKEN_UNDERSCORE))
            {
                *str_out++ = (char)tolower(lex_next(lex));
                tType = lex_token_type(lex_cur(lex));    
            }
            *str_out = '\0'; // make sure it's nil-terminated
            lex->buffer_ptr = str_out + 1; // move pointer to the next position
            token->type = LEX_TOKEN_IDENTIFIER;
            lex_token_str(token) = str_start;
        }
        break;
        /*
         * Read string bordered by inch or apostrophe symbols
         */
        case LEX_TOKEN_INCH:
        case LEX_TOKEN_APOSTROPHE:
        {
            const LexTokenType end_token = tType;
            str_start = str_out = lex->buffer_ptr;

            /*
             * Read the next token
             */
            *str_out++ = lex_next(lex);
            do
            {
                tType = lex_token_type(lex_cur(lex));
                *str_out++ = lex_next(lex); 
            } while ((tType != end_token) &&
                     (tType != LEX_TOKEN_EOL) &&
                     (tType != LEX_TOKEN_UNKNOWN));

            // Check the reason for the parsing end
            if (tType == end_token)
            {
                *(str_out - 1) = '\0';
                lex->buffer_ptr = str_out; // move pointer to the next position
                token->type = LEX_TOKEN_STRING;
                lex_token_str(token) = str_start + 1;
            }
            else
            {
                LOG_LEX("Unexpected lexeme %d during string processing", tType);
                token->type = tType;
            }
        }
        break;
        case LEX_TOKEN_LESS:
        case LEX_TOKEN_GREATER:
        {
            const LexTokenType tOriginToken = tType;

            lex_skip(lex);
            tType = lex_token_type(lex_cur(lex));
            switch (tType)
            {
                case LEX_TOKEN_EQUAL:
                    lex_skip(lex);
                    token->type =
                        (tOriginToken == LEX_TOKEN_LESS) ? LEX_TOKEN_LE :
                                                           LEX_TOKEN_GE;
                    break;
                case LEX_TOKEN_GREATER:
                    if (tOriginToken == LEX_TOKEN_LESS)
                    {
                        lex_skip(lex);
                        token->type = LEX_TOKEN_NOT_EQUAL;
                    }
                    break;
                default:
                    token->type = tOriginToken;
                    break;
            }
        }
        break;
        case LEX_TOKEN_UNKNOWN:
        {
            LOG_LEX("%s", "Unknown");
            lex_failure(lex, LEX_E_UNKNOWN);
            token->type = tType;
        }
        break;
        case LEX_TOKEN_EOL:
        {
            LOG_LEX("%s", "End-of-Line");
            lex_failure(lex, LEX_E_EOL);
            token->type = tType;
        }
        break;
        default:
        {
            lex_skip(lex);
            token->type = tType;
        }
        break;
    }

    if (lex_is_ok(lex))
    {
        // Put the token in the Lex if ok
        lex->sCurToken = *token;        
    }
}

/** Peeks the next token from the current one
 * \param[in] lex lexical analyzer instance
 * \param[out] token next token instance
 */
static LexError lex_token_next_peek(Lex *lex, LexToken *token)
{
    // Backup lex internal data
    const LexToken cur_token = lex->sCurToken;
    const char *cur_tokenizer = lex->tokenizer;
    const LexError cur_error = lex->error;
    LexError rc;

    // Rad the next token
    lex_token_next(lex, token);
    rc = lex->error;

    // Restore the parser
    lex->sCurToken = cur_token;
    lex->tokenizer = cur_tokenizer;        
    lex->error = cur_error;
    return rc;
}

/**
 * @}
 */
 
/**
 * \name SYNTAX ANALYZER
 * @{
 */
typedef enum
{
    SYN_E_OK,
    SYN_E_EOL,
    SYN_E_UNKNOWN,
    SYN_E_UNEXPECTED,
    SYN_E_ERROR
} SynError;

typedef struct
{
    Lex lex; //!< Lexical analyzer
    SynError error; //!< Error code
} Syn; //!< Syntax analyzer

#define syn_token_str(_syn)    (lex_token_str(&((_syn)->lex.sCurToken)))
#define syn_token_type(_syn)   ((_syn)->lex.sCurToken.type)
#define syn_token_is(_syn, _t) (syn_token_type(_syn) == (_t))
#define syn_token_cur(_syn)    (&(_syn)->lex.sCurToken)
#define syn_error_code(_syn)   ((_syn)->error)
#define syn_is_ok(_syn)        (syn_error_code((_syn)) == SYN_E_OK)
#define syn_failure(_syn, _c)  (syn_error_code((_syn)) = (_c))

/** Initialize syn analyzer
 * \param[in] analyzer syn-analyzer instance
 * \param[in] query TQL query to evaluate
 */
static void syn_init(Syn *syn, const char *query)
{
    LOG_SYN("syn: %p, query: %s", syn, query);
    memset(syn, 0, sizeof(*syn));
    lex_init(&syn->lex, query);
    syn->error = SYN_E_OK;
}

/** Uninitialize syn analyzer
 * \param[in] analyzer syn-analyzer instance
 */
static void syn_uninit(Syn *syn)
{
    LOG_SYN("syn: %p", syn);
    lex_uninit(&syn->lex);
    memset(syn, 0, sizeof(*syn));
}

#define syn_error(_syn, _fmt, ...)                               \
    non_fatal("Syntax error:\n%s\n%*.*s\nMessage:" _fmt,         \
        (_syn)->lex.original,                                    \
        (int)((_syn)->lex.tokenizer - (_syn)->lex.original + 1), \
        (int)((_syn)->lex.tokenizer - (_syn)->lex.original + 1), \
        "^", __VA_ARGS__);

/** Reads out next token and keep it inside the analyzer
 *
 * In case of error need to check internal error status
 * \param[in] syn syn-analyzer instance
 */
static void syn_next_token(Syn *syn)
{
    LexToken token;
    lex_token_next(&syn->lex, &token);
    LOG_SYN("%d", token.type);
    if (!lex_is_ok(&syn->lex))
    {
        LOG_SYN("Lex failure detected %d", lex_error_code(&syn->lex));
        if (lex_error_code(&syn->lex) == LEX_E_UNKNOWN)
            syn_failure(syn, SYN_E_UNKNOWN);
        else if (lex_error_code(&syn->lex) == LEX_E_EOL)
            syn_failure(syn, SYN_E_EOL);
    }
}

/** Peeks the next token w/o modifying the syn instance
 * \param[in] syn syn-analyzer
 * \param[in] token lex token as result of peek
 * \return SynError code
 */
static SynError syn_next_token_peek(Syn *syn, LexToken *token)
{
    SynError rc;
    const LexError lex_error = lex_token_next_peek(&syn->lex, token);
    if (lex_error == LEX_E_EOL)
        rc = SYN_E_EOL;
    else if (lex_error == LEX_E_UNKNOWN)
        rc = SYN_E_UNKNOWN;
    else
        rc = SYN_E_OK;
    return rc;
}

/** Skips lexeme by its type
 * \param[in] syn the syntax analyzer
 * \param[in] type lexeme type to be skipped
 */
static void syn_skip_type(Syn *syn, const LexTokenType type)
{
    LOG_SYN("skipping type %d", (int)type);

    syn_next_token(syn);
    if (syn_is_ok(syn))
    {
        if (!syn_token_is(syn, type))
        {
            syn_failure(syn, SYN_E_UNEXPECTED);
            syn_error(syn, "Found token %d instead of %d\n",
                syn_token_type(syn), type);
        }
    }
}

/** Skips the identifier with certain name
 * \param[in] syn the syntax analyzer
 * \param[in] identifier identifier
 */
static void syn_skip_identifier(Syn *syn, const char *identifier)
{
    LOG_SYN("skipping identifier '%s'", identifier);
    syn_next_token(syn);
    if (syn_is_ok(syn))
    {
        if (!syn_token_is(syn, LEX_TOKEN_IDENTIFIER) ||
            strcmp(syn_token_str(syn), identifier))
        {
            syn_failure(syn, SYN_E_UNEXPECTED);
            syn_error(syn, "Found token %d instead of %d\n",
                      syn_token_type(syn), LEX_TOKEN_IDENTIFIER);
        }
    }
}

/**
 * @}
 */
 
 /**
  * \name Equation parser and RPN Processor
  * @{
  */  

#define RPN_EXPRESSION_LEN (512)

typedef enum
{
    RPN_E_OK
} RpnError;

typedef enum
{
    /* This must be the first value initialized by 0 */
    RPN_TYPE_UNKNOWN = 0,

    /**
     * \name Operations
     * @{
     */
    RPN_TYPE_OP_FIRST_MARK,
    RPN_TYPE_OP_AND = RPN_TYPE_OP_FIRST_MARK,
    RPN_TYPE_OP_COMMA,
    RPN_TYPE_OP_DIVIDE,
    RPN_TYPE_OP_EQUAL,
    RPN_TYPE_OP_GREATER,
    RPN_TYPE_OP_GREATER_EQUAL,
    RPN_TYPE_OP_LESS,
    RPN_TYPE_OP_LESS_EQUAL,
    RPN_TYPE_OP_LIKE,
    RPN_TYPE_OP_MINUS,
    RPN_TYPE_OP_MOD,
    RPN_TYPE_OP_MULTIPLY,
    RPN_TYPE_OP_NOT,
    RPN_TYPE_OP_NOT_EQUAL,
    RPN_TYPE_OP_OR,
    RPN_TYPE_OP_PLUS,
    RPN_TYPE_OP_LAST_MARK = RPN_TYPE_OP_PLUS,
    /**
     * @}
     */

    RPN_TYPE_FUNCTION,
    RPN_TYPE_VARIABLE,
    RPN_TYPE_CONST_NUMBER,
    RPN_TYPE_CONST_STRING,
    RPN_TYPE_BRACE_LEFT

} RpnCode;

typedef struct
{
    RpnCode code; //!< Op-code
    int priority; //!< Op-code priority   
    const char *txt; //!< String representation is available
} RpnKeyword; //!< Keyword definition

typedef struct
{
    int argc; //<! Number of arguments
    const char *txt; //!< Name
    int (*func)(void *); //!< Reference to implementation
} RpnFunction; //!< Function descriptor

typedef struct
{
    RpnCode op_code; //!< Op-code
    union
    {
        int variable_id; //!< Variable unique id if the token is variable
        const RpnFunction *func; //!< Attached function;
    } data;
    LexToken lex; //!< lex-token representation
} RpnToken; //!< RPN token

/** Maps the variable to the unique id which can be used later during
 *        expression evaluation.
 * \param[out] id variable unique identifier in case of success
 * \param[in] name variable name
 * \param[in] arg extra argument
 * \return SXM_E_OK in case of success and some other code in case of failure.
 */
typedef int (*RpnVariableLookup)(int *id, const char *name, void *arg);

typedef struct
{
    struct
    {
        LexToken tokens[RPN_EXPRESSION_LEN];
        int tokens_count;
    } sLex; //!< Keeps pre-processed tokens

    struct
    {
        RpnToken tokens[RPN_EXPRESSION_LEN];
        int tokens_count;
    } sStack, sOutput;

    const RpnKeyword *keywords; //!< Recognizable keywords set
    const RpnFunction *functions; //!< Recognizable functions

    RpnVariableLookup var_lookup; //!< Variable lookup callback
    void *var_lookup_arg; //!< Variable lookup argument

    RpnError error; //!< RPN current error code
} Rpn; //!< RPN Handling structure

/** Defines operations
 */
static const RpnKeyword gKeywords[] =
{
    /** OPERATORS */
    /* ARIPHMETICAL OPERATIONS */
    {RPN_TYPE_OP_MULTIPLY,      3, NULL},
    {RPN_TYPE_OP_DIVIDE,        3, NULL},
    {RPN_TYPE_OP_MOD,           3, NULL},
    {RPN_TYPE_OP_MINUS,         4, NULL},
    {RPN_TYPE_OP_PLUS,          4, NULL},

    /* LOGICAL OPERATIONS */
    {RPN_TYPE_OP_NOT,           2, "not"},
    {RPN_TYPE_OP_LESS,          6, "less"},
    {RPN_TYPE_OP_LESS_EQUAL,    6, "le"},
    {RPN_TYPE_OP_EQUAL,         7, "eq"},
    {RPN_TYPE_OP_NOT_EQUAL,     7, "ne"},
    {RPN_TYPE_OP_LIKE,          7, "like"},
    {RPN_TYPE_OP_GREATER_EQUAL, 6, "ge"},
    {RPN_TYPE_OP_GREATER,       6, "gr"},
    {RPN_TYPE_OP_AND,          11, "and"},
    {RPN_TYPE_OP_OR,           12, "or"},

    /* TERMINATOR - MUST BE THE LAST ITEM */
    {RPN_TYPE_UNKNOWN,         -1, NULL}
};

/** RPN Initialization routine
 * \param[in] rpn RPN instance to be initialized
 * \param[in] keywords the set of supported keywords
 * \param[in] functions the set of supported functions
 * \param[in] var_lookup Variable lookup callback
 * \param[in] var_lookup_arg the address will be passed to each lookup callback call
 */
static int rpn_init(Rpn **rpn,
                    const RpnKeyword *keywords,
                    const RpnFunction *functions,
                    RpnVariableLookup var_lookup,
                    void *var_lookup_arg)
{
    LOG_RPN("%p, %p", rpn, keywords);
    *rpn = (Rpn*)sxe_calloc(1, sizeof(Rpn));
    if (!(*rpn))
    {
        return SXM_E_NOMEM;
    }
    else
    {
        (*rpn)->keywords = keywords;
        (*rpn)->functions = functions;
        (*rpn)->var_lookup = var_lookup;
        (*rpn)->var_lookup_arg = var_lookup_arg;
        (*rpn)->error = RPN_E_OK;
        return SXM_E_OK;    
    }
}

/** Detects operation op-code
 * \param[in] _c RPN op-code from RpnCode
 */
#define rpn_is_operator(_c) \
    (((_c)>=RPN_TYPE_OP_FIRST_MARK) && \
     ((_c)<=RPN_TYPE_OP_LAST_MARK))

/** Appends the lex token to the RPN internal collection
 * \param[in] rpn RPN instance
 * \param[in] token the LexToken instance to be appended
 */
static int rpn_lex_token_append(Rpn *rpn, const LexToken *token)
{
    rpn->sLex.tokens[rpn->sLex.tokens_count++] = *token;
    LOG_RPN("count=%d, %d", (int)rpn->sLex.tokens_count, (int)token->type);
    return !0;
}

/** Lookups for the RPN keyword based on the name
 * \param[in] rpn RPN instance
 * \param[in] name the keyword textual representation
 * \return PRN keyword description instance in case of success or
 *         \c NULL if it doesn't exits
 */
static const RpnKeyword* rpn_keyword(Rpn *rpn, const char *txt)
{
    const RpnKeyword* result = rpn->keywords;
    while (result->code != RPN_TYPE_UNKNOWN)
    {
        if (result->txt && !strcmp(txt, result->txt))
        {
            break;   
        }
        ++result; // Move to the next item
    }
    if (result->code == RPN_TYPE_UNKNOWN)
    {
        // End-Of-List detected
        result = NULL;
    }
    LOG_RPN("'%s' is %d", txt, (result ? result->code : RPN_TYPE_UNKNOWN));
    return result;
}

/** Lookups for the RPN function based on the name
 * \param[in] rpn RPN instance
 * \param[in] name the function textual representation
 * \return PRN function description instance in case of success or
 *         \c NULL if it doesn't exits
 */
static const RpnFunction* rpn_function(Rpn *rpn, const char *txt)
{
    const RpnFunction* result = rpn->functions;
    while (result->func)
    {
        if (result->txt && !strcmp(txt, result->txt))
        {
            break;   
        }
        ++result; // Move to the next item
    }
    if (!result->func)
    {
        // End-Of-List detected
        result = NULL;
    }
    LOG_RPN("'%s' is %p", txt, (result ? result->func : NULL));
    return result;
}

/** Converts the next token into the RPN one
 * \param[in] rpn RPN instance
 * \param[in] token the lex-token
 * \param[out] rpn_token the keeper of the converted value
 */
static int rpn_token_from_lex(Rpn *rpn,
                              const LexToken *token,
                              RpnToken *rpn_token)
{
    memset(rpn_token, 0, sizeof(*rpn_token));
    rpn_token->op_code = RPN_TYPE_UNKNOWN;
    rpn_token->lex = *token;
    switch (token->type)
    {
        case LEX_TOKEN_NUMBER:
            rpn_token->op_code = RPN_TYPE_CONST_NUMBER;
            break;
        case LEX_TOKEN_STRING:
            rpn_token->op_code = RPN_TYPE_CONST_STRING;
            break;
        case LEX_TOKEN_IDENTIFIER:
        {
            const RpnKeyword *keyword;
            // Since the identifier can be either variable 
            // or embedded function the check shall find the
            // answer to this question.
            keyword = rpn_keyword(rpn, lex_token_str(token));
            if (keyword)
            {
                rpn_token->op_code = keyword->code;
            }
            else
            {
                const RpnFunction *func = rpn_function(rpn, lex_token_str(token));
                if (func)
                {
                    rpn_token->data.func = func;
                    rpn_token->op_code = RPN_TYPE_FUNCTION;
                }
                else
                {
                    rpn_token->op_code = RPN_TYPE_VARIABLE;
                }
            }
        }
        break;
        case LEX_TOKEN_STAR:
            rpn_token->op_code = RPN_TYPE_OP_MULTIPLY;
            break;
        case LEX_TOKEN_MINUS:
            rpn_token->op_code = RPN_TYPE_OP_MINUS;
            break;
        case LEX_TOKEN_PLUS:
            rpn_token->op_code = RPN_TYPE_OP_PLUS;
            break;
        case LEX_TOKEN_SLASH:
            rpn_token->op_code = RPN_TYPE_OP_DIVIDE;
            break;
        case LEX_TOKEN_PERC:
            rpn_token->op_code = RPN_TYPE_OP_MOD;
            break;
        case LEX_TOKEN_ROUND_LEFT:
            rpn_token->op_code = RPN_TYPE_BRACE_LEFT;
            break;
        case LEX_TOKEN_LESS:
            rpn_token->op_code = RPN_TYPE_OP_LESS;
            break;
        case LEX_TOKEN_LE:
            rpn_token->op_code = RPN_TYPE_OP_LESS_EQUAL;
            break;
        case LEX_TOKEN_EQUAL:
            rpn_token->op_code = RPN_TYPE_OP_EQUAL;
            break;
        case LEX_TOKEN_NOT_EQUAL:
            rpn_token->op_code = RPN_TYPE_OP_NOT_EQUAL;
            break;
        case LEX_TOKEN_GE:
            rpn_token->op_code = RPN_TYPE_OP_GREATER_EQUAL;
            break;
        case LEX_TOKEN_GREATER:
            rpn_token->op_code = RPN_TYPE_OP_GREATER;
            break;
        case LEX_TOKEN_COMMA:
            rpn_token->op_code = RPN_TYPE_OP_COMMA;
            break;
        /* Unhandled tokens */
        case LEX_TOKEN_UNKNOWN:
        case LEX_TOKEN_APOSTROPHE:
        case LEX_TOKEN_EOL:
        case LEX_TOKEN_INCH:
        case LEX_TOKEN_PERIOD:
        case LEX_TOKEN_ROUND_RIGHT:
        case LEX_TOKEN_SEMICOLON:
        case LEX_TOKEN_SPACE:
        case LEX_TOKEN_UNDERSCORE:
        case LEX_TOKEN_MAX:
            break;
    }
    LOG_RPN("lex %d to rpn %d", (int)token->type, (int)rpn_token->op_code);
    return (rpn_token->op_code != RPN_TYPE_UNKNOWN) ? SXM_E_OK : SXM_E_UNSUPPORTED;
}

/** Composes the set of tokens via syn-analyzer
 * \param[in] rpn RPN instance
 * \param[in] analyzer syn-analyzer instance
 */
static int rpn_lex_tokens_build(Rpn *rpn, Syn *analyzer)
{
    int nBracesCount = 0;
    int multiplyNumberByMinus1 = 0;
    int rc = SXM_E_OK;

    LOG_RPN("(%p)", analyzer);

    while (syn_is_ok(analyzer))
    {
        int bDoAppend = 0;

        syn_next_token(analyzer);
        if (!syn_is_ok(analyzer))
        {
            syn_error(analyzer, "%s\n", "interrupted");
            rc = SXM_E_INVAL;
        }
        else if (syn_token_type(analyzer) == LEX_TOKEN_SEMICOLON)
        {
            LOG_RPN("End of expression detected", "");
            break;
        }

        switch (syn_token_type(analyzer))
        {
            case LEX_TOKEN_ROUND_LEFT:
                LOG_RPN("(%s)", "left");
                ++nBracesCount;
                bDoAppend = !0;
                break;
            case LEX_TOKEN_ROUND_RIGHT:
                LOG_RPN("(%s)", "right");
                --nBracesCount;
                bDoAppend = !0;
                break;
            case LEX_TOKEN_MINUS:
            {
                LexToken nextToken;
                if ((syn_next_token_peek(analyzer, &nextToken) != SYN_E_OK) ||
                    (nextToken.type == LEX_TOKEN_SEMICOLON))
                {
                    syn_error(analyzer, "%s\n", "Unexpected end of the stream");
                }
                else
                {
                    LexToken *prevToken;
                    RpnToken rpn_token;

                    memset(&rpn_token, 0, sizeof(rpn_token));
                    prevToken =
                        (rpn->sLex.tokens_count) ?
                            &rpn->sLex.tokens[rpn->sLex.tokens_count - 1] : NULL;

                    if (prevToken)
                    {
                        rpn_token_from_lex(rpn, prevToken, &rpn_token);
                    }

                    if (!prevToken ||
                        (prevToken->type == LEX_TOKEN_ROUND_LEFT) ||
                        rpn_is_operator(rpn_token.op_code))
                    {
                        if ((nextToken.type == LEX_TOKEN_ROUND_LEFT) ||
                            (nextToken.type == LEX_TOKEN_IDENTIFIER))
                        {
                            LexToken tmpToken;
                            tmpToken.type = LEX_TOKEN_NUMBER;
                            lex_token_real(&tmpToken) = -1.0f;
                            rpn_lex_token_append(rpn, &tmpToken);
                            tmpToken.type = LEX_TOKEN_STAR;
                            rpn_lex_token_append(rpn, &tmpToken);
                        }
                        else if (nextToken.type == LEX_TOKEN_NUMBER)
                        {
                            multiplyNumberByMinus1 = !0;
                        }
                    }
                    else if (nextToken.type == LEX_TOKEN_NUMBER)
                    {
                        LexToken plusOp;
                        plusOp.type = LEX_TOKEN_PLUS;
                        rpn_lex_token_append(rpn, &plusOp);
                        multiplyNumberByMinus1 = !0;
                    }
                    else
                    {
                        bDoAppend = !0;
                    }
                }
            }
            break;
            case LEX_TOKEN_NUMBER:
                if (multiplyNumberByMinus1)
                {
                    lex_token_real(syn_token_cur(analyzer)) *= -1.0f;
                    multiplyNumberByMinus1 = 0;
                }
                bDoAppend = !0;
                break;
            default:
                bDoAppend = !0;
                break;
        }

        if (bDoAppend)
        {
            rpn_lex_token_append(rpn, syn_token_cur(analyzer));
        }
    }

    return (rc != SXM_E_OK) ? (rc) : ((nBracesCount == 0) ? SXM_E_OK : SXM_E_INVAL);
}
    
/** Lookups for the RPN operation priority based on op-code
 * \param[in] rpn RPN instance
 * \param[in] op_code operation code
 * \return operation priority or \c -1 if it's not defined
 */
static int rpn_op_priority(Rpn *rpn, RpnCode op_code)
{
    const RpnKeyword* kw = rpn->keywords;
    int result = -1;
    while ((kw->code != RPN_TYPE_UNKNOWN) && (kw->code != op_code))
    {
        ++kw; // Move to the next item
    }
    if (kw->code != RPN_TYPE_UNKNOWN)
    {
        // End-Of-List detected
        result = kw->priority;
    }
    LOG_RPN("%d has priority %d", (int)op_code, result);
    return result;
}

/** Pushes the RPN token on the top of the stack
 * \param[in] rpn RPN instance
 * \param[in] token rpn-token to be pushed
 */
static int rpn_stack_push(Rpn *rpn, const RpnToken *token)
{
    int rc;
    if ((size_t)rpn->sStack.tokens_count < ARRAY_SIZE(rpn->sStack.tokens))
    {
        rpn->sStack.tokens[rpn->sStack.tokens_count++] = *token;    
        rc = SXM_E_OK;
    }
    else
    {
        rc = SXM_E_NOMEM;
    }    
    LOG_RPN("count=%d, %d", rpn->sStack.tokens_count, (int)token->op_code);
    return rc;
}

/** Pops the top RPN token form the stack
 * \param[in] rpn RPN instance
 * \param[in] token copy of the RPN token from the stack's top
 * \return not-0 if the items has been pop-up or 0 in case if the stack empty
 */
static int rpn_stack_pop(Rpn *rpn, RpnToken *token)
{
    if (rpn->sStack.tokens_count > 0)
    {
        --rpn->sStack.tokens_count;
        *token = rpn->sStack.tokens[rpn->sStack.tokens_count];
        LOG_RPN("count=%d, %d", rpn->sStack.tokens_count, (int)token->op_code);
        return SXM_E_OK;
    }
    LOG_RPN("%s", "(empty)");
    return SXM_E_NOENT;
}

/** Peeks the top RPN token form the stack
 * \param[in] rpn RPN instance
 * \param[in] token rpn-token to be peeked.
 * \return not-0 if the items has been peeked-up or 0 in case if the stack empty
 */
static int rpn_stack_peek(Rpn *rpn, RpnToken *token)
{
    if (rpn->sStack.tokens_count > 0)
    {
        *token = rpn->sStack.tokens[rpn->sStack.tokens_count - 1];  
        LOG_RPN("count=%d, %d", rpn->sStack.tokens_count, (int)token->op_code);
        return SXM_E_OK;
    }
    LOG_RPN("%s", "(empty)");
    return SXM_E_NOENT;
}

/** Append the RPN token to the RPN conveyor
 * \param[in] rpn RPN instance
 * \param[in] token rpn-token to be added to the stack  
 */
static int rpn_output_append(Rpn *rpn, const RpnToken *token)
{
    int rc;
    if ((size_t)rpn->sOutput.tokens_count < ARRAY_SIZE(rpn->sOutput.tokens))
    {
        rpn->sOutput.tokens[rpn->sOutput.tokens_count++] = *token;
        rc = SXM_E_OK;
    }
    else
    {
        rc = SXM_E_NOMEM;
    }
    
    LOG_RPN("size=%d, %d", rpn->sOutput.tokens_count, (int)token->op_code);
    return rc;
}

#if SXMTQL_LOG & SXMTQL_TQL_LOG
/** Dumps the RPN token to the file
 * \param[in] f opened in write mode file
 * \param[in] token the RPN token to be dumped
 */
static int rpn_token_dump(FILE *f, const RpnToken *token)
{
    fprintf(f, "RPN(%3d) ", token->op_code);
    lex_token_dump(f, &token->lex);
    if (token->op_code == RPN_TYPE_FUNCTION)
    {
        fprintf(f, " FUN:%p",  token->data.func);
    }
    else if (token->op_code == RPN_TYPE_VARIABLE)
    {
        fprintf(f, " ID:%d", token->data.variable_id);
    }
    
    return SXM_E_OK;
}
#endif

/** Build the expression for \b operator
 * \param[in] rpn PRN instance
 * \param[in] token origin token
 */
static int rpn_expression_build_op(Rpn *rpn, const RpnToken *token)
{
    RpnToken rpn_token;
    while (rpn_stack_peek(rpn, &rpn_token) == SXM_E_OK)
    {
        if (rpn_is_operator(rpn_token.op_code))
        {
            if (rpn_op_priority(rpn, token->op_code)
                >=
                rpn_op_priority(rpn, rpn_token.op_code))
            {
                rpn_stack_pop(rpn, &rpn_token);
                rpn_output_append(rpn, &rpn_token);
            }
            else
            {
                LOG_RPN("(%s)", "leave for the next time");
                break;
            }
        }
        else
        {
            LOG_RPN("(%s)", "not operator");
            break;
        }
    }
    rpn_stack_push(rpn, token); 
    return SXM_E_OK;      
}

/** Build the RPN expression
 * \param[in] rpn RPN instance
 */
static int rpn_expression_build(Rpn *rpn)
{
    int lex_token_itr;
    int rc = SXM_E_OK;
    LexToken *lex_token;
    RpnToken rpn_token;

    LOG_RPN("------------------ START ------------------ (%p)", rpn);

    for (lex_token_itr = 0, lex_token = &rpn->sLex.tokens[0];
         (lex_token_itr < rpn->sLex.tokens_count) && (rc == SXM_E_OK);
         ++lex_token_itr, ++lex_token)
    {
        memset(&rpn_token, 0, sizeof(rpn_token));

        switch (lex_token->type)
        {
            case LEX_TOKEN_NUMBER:
                LOG_RPN("(%s)", "number");
                rpn_token_from_lex(rpn, lex_token, &rpn_token);
                rc = rpn_output_append(rpn, &rpn_token);
                break;
            case LEX_TOKEN_ROUND_LEFT:
                rpn_token_from_lex(rpn, lex_token, &rpn_token);
                rc = rpn_stack_push(rpn, &rpn_token);
                break;
            case LEX_TOKEN_ROUND_RIGHT:
            {
                int hasPopUp;
                for(;;)
                {
                    hasPopUp = rpn_stack_pop(rpn, &rpn_token);
                    if ((hasPopUp != SXM_E_OK) || (rpn_token.lex.type == LEX_TOKEN_ROUND_LEFT))
                    {
                        break;
                    }
                    rpn_output_append(rpn, &rpn_token);
                }
                if (hasPopUp == SXM_E_OK)
                {
                    hasPopUp = rpn_stack_peek(rpn, &rpn_token);
                    if ((hasPopUp == SXM_E_OK) && (rpn_token.op_code == RPN_TYPE_FUNCTION))
                    {
                        rpn_output_append(rpn, &rpn_token);
                        rpn_stack_pop(rpn, &rpn_token);
                    }
                }
                rc = SXM_E_OK;
            }
            break;
            case LEX_TOKEN_IDENTIFIER:
            {
                LOG_RPN("(%s)", "identifier");
                rpn_token_from_lex(rpn, lex_token, &rpn_token);
                if (rpn_is_operator(rpn_token.op_code))
                {
                    LOG_RPN("symbolic operator %d", (int)rpn_token.op_code);
                    rc = rpn_expression_build_op(rpn, &rpn_token);
                }
                else switch (rpn_token.op_code)
                {
                    case RPN_TYPE_FUNCTION:
                        LOG_RPN("function %s", lex_token_str(&rpn_token.lex));
                        rpn_stack_push(rpn, &rpn_token);
                        rc = SXM_E_OK;
                        break;
                    case RPN_TYPE_VARIABLE:
                    {
                        int var_id;

                        // Do the check for the variable existence via callback
                        // provide by the application
                        rc = rpn->var_lookup(&var_id,
                                             lex_token_str(&rpn_token.lex),
                                             rpn->var_lookup_arg);
                        if (rc == SXM_E_OK)
                        {
                            LOG_RPN("variable %s (id=%d)",
                                lex_token_str(&rpn_token.lex), var_id);
                            rpn_token.data.variable_id = var_id;
                            rpn_output_append(rpn, &rpn_token);  
                        }
                        else
                        {
                            LOG_RPN("There is no such variable '%s'", lex_token_str(&rpn_token.lex));
                        }
                    }
                    break;
                    default:
                        rc = SXM_E_INVAL;
                        break;
                }
            }
            break;
            case LEX_TOKEN_STRING:
            {
                LOG_RPN("(%s)", "string");
                rpn_token_from_lex(rpn, lex_token, &rpn_token);
                rpn_output_append(rpn, &rpn_token);   
                rc = SXM_E_OK;     
            }
            break;
            default:
            {
                LOG_RPN("default case for %d", lex_token->type);
                rpn_token_from_lex(rpn, lex_token, &rpn_token);
                if (rpn_is_operator(rpn_token.op_code))
                {
                    LOG_RPN("operator %d", (int)rpn_token.op_code);
                    rpn_expression_build_op(rpn, &rpn_token);
                }
                else
                {
                    LOG_RPN("(%s)", "not an operator");
                }
                rc = SXM_E_OK;
            }
            break;
        }
    }

    if (rc == SXM_E_OK)
    {
        LOG_RPN("flush stack's tokens (count=%d)", (int)rpn->sStack.tokens_count);
        while (rpn_stack_pop(rpn, &rpn_token) == SXM_E_OK)
        {
            rc = rpn_output_append(rpn, &rpn_token);
        }
    }
    return rc;
}

#if SXMTQL_LOG & SXMTQL_TQL_LOG
/** Dumps all already built RPN expression to the file
 * \param[in] f output file
 * \param[in] rpn RPN instance
 */
static int rpn_dump(FILE *f, Rpn *rpn)
{
    int token_itr;
    RpnToken *token;
    for (token_itr = 0, token = &rpn->sOutput.tokens[0];
         token_itr < rpn->sOutput.tokens_count;
         ++token_itr, ++token)
    {
        rpn_token_dump(f, token);
        fprintf(f, "\n");
    }
    return !0;
}
#endif

/** Full RPN building process from the expression
 * \param[in] rpn RPN instance
 * \param[in] analyzer syn-analyzer instance to feed the RPN
 */
static int rpn_build(Rpn *rpn, Syn *analyzer)
{
    int rc;
    rc = rpn_lex_tokens_build(rpn, analyzer);
    if (rc == SXM_E_OK)
    {
        rc = rpn_expression_build(rpn);
        if (rc)
        {
            LOG_RPN("failed to build RPN expression collection (%d)", rc);
        }
    }
    else
    {
        LOG_RPN("failed to build lex-tokens collection (%d)", rc);
    }
    return rc;
}
  
 /**
  * @}
  */

/**
 * \name TQL IMPLEMENTATION
 * @{
 */
#define TQL_DB_FIELDS_COUNT_MAX (32)
#define TQL_CONVEYOR_STACK_SIZE (16)

typedef enum
{
    TQL_E_OK,
    TQL_E_ERROR
} TqlError;

typedef struct
{
    const SXMTqlTableColumn *column; //!< Reference to the DB's table
    const char *name; //!< Column name keeper during query processing
} TqlQueryColumn; //!< Query column descriptor

typedef struct
{
    SXMTqlTable *table; //!< TQL table descriptor
    SXMTqlTableColumn columns[TQL_DB_FIELDS_COUNT_MAX]; //!< Keeps full set of the table's columns
    int columns_count; // Number of used columns
} TqlTable; //!< TQL table keeper

struct _sxm_tql_statement_struct
{
    Syn syn; //!< Syn-analyzer
    Rpn *rpn; //!< RPN instance which exists 
    const SXMTqlCallback *callback; //!< Callback interface
    const void *callback_arg; // !< Callback interface argument

    struct
    {
        SXMTqlTable db_tables[SXMTQL_TABLE_MAX_COUNT]; //!< Original tables
        TqlTable tables[SXMTQL_TABLE_MAX_COUNT]; //!< Preprocessed tables
        int tables_count; //!< Number of tables within \c tables array

        struct
        {
            TqlTable *table_ref; //!< Reference to the DB's table
            TqlQueryColumn columns[TQL_DB_FIELDS_COUNT_MAX]; //!< Keeps columns within the query
            int columns_count; // Number of used columns 
        } query; //!< Useful query information
    } sDB; //!< Description of the table under query

    struct
    {
        SXMTqlValue stack[TQL_CONVEYOR_STACK_SIZE]; //!< Data stack
        int stack_size; //!< Data stack size
    } sConveyor; //!< Expression execution conveyor

    TqlError error; //<! Current error code
};

#define tql_error_code(_tql)  ((_tql)->error)
#define tql_failure(_tql, _c) (tql_error_code(_tql) = (_c))
#define tql_is_ok(_tql)       (tql_error_code(_tql) == TQL_E_OK)

/** Maps the string representation of the column to it's unique identifier
 * \param[in] statement TQL statement
 * \param[in] table the reference to the table where to look for the column
 * \param[in] name column name
 * \return table column descriptor or NULL if there is no such column
 */
static const SXMTqlTableColumn* tql_column_by_name(SXMTqlStatement statement,
                                                   const TqlTable *table,
                                                   const char *name)
{
    const SXMTqlTableColumn *result = NULL, *iter;
    int column_iter;

    UNUSED_VAR(statement);

    for (column_iter = 0, iter = table->columns;
         column_iter < table->columns_count;
         ++column_iter, ++iter)
     {
        if (!strcmp(name, iter->name))
        {
            result = iter;
            break;
        }
     }  

     return result;
}

/** Links the query columns and table to the real tables from the DB.
 * \param[in] statement the TQL statement
 * \return SXM_E_OK in case of success or any other error code in case of error
 */
static int tql_link_query_to_schema(SXMTqlStatement statement)
{
    int column_iter;
    int rc = SXM_E_OK;
    TqlQueryColumn *q_column;

    if (statement->sDB.query.columns_count)
    {
        for (column_iter = 0, q_column = &statement->sDB.query.columns[0];
             column_iter < statement->sDB.query.columns_count;
             ++column_iter, ++q_column)
        {
            q_column->column =
                tql_column_by_name(statement,
                                   statement->sDB.query.table_ref,
                                   q_column->name);
            if (!q_column->column)
            {
                LOG_TQL("There is no column called '%s'", q_column->name);
                rc = SXM_E_NOENT;
                break;
            }
        }
    }
    else
    {
        // Put all columns which are available for the table
        // to the query
        SXMTqlTableColumn *t_column;
        for (column_iter = 0,
                q_column = &statement->sDB.query.columns[0],
                t_column = &statement->sDB.query.table_ref->columns[0];
             column_iter < statement->sDB.query.table_ref->columns_count;
             ++column_iter, ++q_column, ++t_column)
        {
            q_column->column = t_column;
        }
        statement->sDB.query.columns_count =
            statement->sDB.query.table_ref->columns_count;
    }

    return rc;    
}

static int tql_var_lookup(int *id, const char *name, void *arg)
{
    SXMTqlStatement statement = (SXMTqlStatement)arg;
    const SXMTqlTableColumn* column =
        tql_column_by_name(statement, statement->sDB.query.table_ref, name);
    if (column)
    {
        *id = column->id;
        return SXM_E_OK;
    }  
    return SXM_E_NOENT;
}

/** Dumps the tql value structure
 * \param[in] value the value to be dumped
 */
static void tql_value_dump(const SXMTqlValue *value)
{
    switch (value->type)
    {
        case SXM_TQL_TYPE_INT:
            LOG_TQL("int(%d)", value->value.intV);
            break;
        case SXM_TQL_TYPE_REAL:
            LOG_TQL("real(%f)", value->value.dblV);
            break;
        case SXM_TQL_TYPE_BOOL:
            LOG_TQL("bool(%s)", ((value->value.boolV == SXM_TQL_TRUE) ? "true" : "false"));
            break;
        case SXM_TQL_TYPE_STR:
            LOG_TQL("text(%s)", value->value.str.strV);
            break;
        default:
            LOG_TQL("unknown(%p)", value->value.str.strV);
            break;
    }
}

/** Pushes the value on top 
 * \param[in] statement the TQL statement
 * \param[in] value the value to be pushed
 * \return SXM_E_OK in case of success or any other code in case of failure
 */
static int tql_push(SXMTqlStatement statement, const SXMTqlValue *value)
{
    LOG_TQL("type=%c", value->type);
    tql_value_dump(value);
    if (statement->sConveyor.stack_size < TQL_CONVEYOR_STACK_SIZE)
    {
        statement->sConveyor.stack[statement->sConveyor.stack_size++] = *value;
        return SXM_E_OK;
    }
    return SXM_E_NOMEM;
}

/** Pops up the top element from the stack
 * \param[in] statement the TQL statement
 * \return SXM_E_OK in case of success or any other code in case of failure
 */
static int tql_pop(SXMTqlStatement statement)
{
    LOG_TQL("depth=%d", statement->sConveyor.stack_size);
    if (statement->sConveyor.stack_size > 0)
    {
        --statement->sConveyor.stack_size;
        tql_value_dump(&statement->sConveyor.stack[statement->sConveyor.stack_size]);
        return SXM_E_OK;
    }
    return SXM_E_NOENT;
}

/** Gives address of the N-top element of the stack
 * \param[in] _s the TQL statement
 * \param[in] _offset the offset from the top to provide the values
 * \return address of the requested element
 */
#define tql_stack_top(_s, _offset) \
    (&((_s)->sConveyor.stack[(_s)->sConveyor.stack_size - (1 + (_offset))]))

/** Give the depth of the stack
 * \param[in] _s the TQL statement
 */
#define tql_stack_depth(_s) \
    (_s)->sConveyor.stack_size

/** Check the stack to make sure that N-top element have retired data type
 * \param[in] statement the TQL statement
 * \param[in] type required data type to N-top values
 * \param[in] count number of top items to be checked
 * \return SXM_E_OK in case if the retirement is mean or any other code in case
 *         of failure.
 */
static int tql_has_top(SXMTqlStatement statement, SXMTqlType type, int count)
{
    int rc;
    if (tql_stack_depth(statement) >= count)
    {
        rc = SXM_E_OK;
        while ((count --> 0) && (rc != SXM_E_INVAL))
        {
            if (tql_stack_top(statement, count)->type != type)
            {
                rc = SXM_E_INVAL;
            }
        }
    }
    else
    {
        rc = SXM_E_NOENT;
    }
    return rc;
}

/** Checks the N-top items have the same data type
 * \param[in] statement the TQL statement
 * \param[in] count Sumner of top items to be checked
 * \return SXM_E_OK in case if the requirement is meant or any other code in case
 *         of failure.
 */
static int tql_has_same_top(SXMTqlStatement statement, int count)
{
    int rc;
    if (tql_stack_depth(statement) >= count)
    {
        const SXMTqlType type = tql_stack_top(statement, 0)->type;

        rc = SXM_E_OK;
        while ((count --> 0) && (rc != SXM_E_INVAL))
        {
            if (tql_stack_top(statement, count)->type != type)
            {
                rc = SXM_E_INVAL;
            }
        }
    }
    else
    {
        rc = SXM_E_NOENT;
    }
    return rc;
}

/** Binary logical operation
 * \param[in] _s TQL statement
 * \param[in] _op C-language operations
 * \param[out] _ret calculation return value
 */
#define tql_op_bin_logical(_s, _op, _ret)                                  \
    if (tql_has_top((_s), SXM_TQL_TYPE_BOOL, 2) == SXM_E_OK) {             \
        (_ret).type = SXM_TQL_TYPE_BOOL;                                   \
        (_ret).value.boolV = (tql_stack_top((_s), 1)->value.boolV          \
                              _op                                          \
                              tql_stack_top((_s), 0)->value.boolV) ?       \
                                    SXM_TQL_TRUE : SXM_TQL_FALSE;          \
        tql_pop((_s)); tql_pop((_s)); tql_push((_s), &(_ret));             \
    } else { LOG_TQL("(%s)", "expected boolean operands"); }               \

/** Provides access to the named field within SXMTqlValue
 * \param[in] _v instance of the SXMTqlValue data structure
 * \param[in] _field the field data
 */
#define tql_value_typed(_v, _field) \
    ((_v)->value._field)

#define tql_stack_top_typed(_s, _offset, _field) \
    tql_value_typed(tql_stack_top((_s), (_offset)), _field)
       
#define tql_op_typed(_s, _op, _field)         \
    (tql_stack_top_typed(_s, 1, _field)       \
     _op                                      \
     tql_stack_top_typed(_s, 0, _field))      \

#define tql_op_logical_typed(_s, _op, _field) \
    (tql_op_typed(_s, _op, _field) ? SXM_TQL_TRUE : SXM_TQL_FALSE);
    
/** Binary compare operation
 * \param[in] _s TQL statement
 * \param[in] _op C-language operations
 * \param[out] _ret calculation return value
 */
#define tql_op_bin_compare(_s, _op, _ret)                                  \
    if (tql_has_same_top((_s), 2) == SXM_E_OK) {                           \
        SXM_TQL_BOOL res = SXM_TQL_FALSE, valid = SXM_TQL_TRUE;            \
        LOG_TQL("type=%c, op(%s)", tql_stack_top((_s), 0)->type, #_op);    \
        switch (tql_stack_top((_s), 0)->type) {                            \
            case SXM_TQL_TYPE_REAL:                                        \
                res = (tql_real_compare(tql_stack_top((_s), 1),            \
                                        tql_stack_top((_s), 0)) _op 0) ?   \
                                    SXM_TQL_TRUE : SXM_TQL_FALSE;          \
                break;                                                     \
            case SXM_TQL_TYPE_INT:                                         \
                res = tql_op_logical_typed((_s), _op, intV);               \
                break;                                                     \
            case SXM_TQL_TYPE_BOOL:                                        \
                res = tql_op_logical_typed((_s), _op, boolV);              \
                break;                                                     \
            case SXM_TQL_TYPE_STR:                                         \
                res = (tql_string_compare(tql_stack_top((_s), 1),          \
                                          tql_stack_top((_s), 0)) _op 0) ? \
                                    SXM_TQL_TRUE : SXM_TQL_FALSE;          \
                break;                                                     \
            default:                                                       \
                LOG_TQL("unable to do %s for %c data type", #_op,          \
                        tql_stack_top((_s), 0)->type);                     \
                valid = SXM_TQL_FALSE;                                     \
                break;                                                     \
        }                                                                  \
        if (valid == SXM_TQL_TRUE) {                                       \
            sxm_tql_val_set_bool(&(_ret), res)                             \
            tql_pop((_s)); tql_pop((_s)); tql_push((_s), &(_ret));         \
        }                                                                  \
    } else {LOG_TQL("failed to perform %s due to difference types", #_op);}

/** Binary arithmetically operation
 * \param[in] _s TQL statement
 * \param[in] _op C-language operations
 * \param[out] _ret calculation return value
 */
#define tql_op_bin_ariphetical(_s, _op, _ret)                              \
    if (tql_has_same_top((_s), 2) == SXM_E_OK) {                           \
        LOG_TQL("type=%c, op(%s)", tql_stack_top((_s), 0)->type, #_op);    \
        (_ret).type = tql_stack_top((_s), 0)->type;                        \
        switch (tql_stack_top((_s), 0)->type) {                            \
            case SXM_TQL_TYPE_INT:                                         \
                tql_value_typed(&(_ret), intV) =                           \
                    tql_op_typed((_s), _op, intV);                         \
                tql_pop((_s)); tql_pop((_s)); tql_push((_s), &(_ret));     \
                break;                                                     \
            case SXM_TQL_TYPE_REAL:                                        \
                tql_value_typed(&(_ret), dblV) =                           \
                    tql_op_typed((_s), _op, dblV);                         \
                tql_pop((_s)); tql_pop((_s)); tql_push((_s), &(_ret));     \
                break;                                                     \
            default:                                                       \
                LOG_TQL("Unsupported type %x for %s",                      \
                    tql_stack_top((_s), 0)->type, #_op);                   \
                break;                                                     \
        }                                                                  \
    } else { LOG_TQL("unsupported operands for %s", #_op); }               \

/** Binary arithmetically operation over integer values
 * \param[in] _s TQL statement
 * \param[in] _op C-language operations
 * \param[out] _ret calculation return value
 */
#define tql_op_bin_int_ariphetical(_s, _op, _ret)                          \
    if (tql_has_same_top((_s), 2) == SXM_E_OK) {                           \
        LOG_TQL("type=%c, op(%s)", tql_stack_top((_s), 0)->type, #_op);    \
        (_ret).type = tql_stack_top((_s), 0)->type;                        \
        if (tql_stack_top((_s), 0)->type == SXM_TQL_TYPE_INT) {            \
                tql_value_typed(&(_ret), intV) =                           \
                    tql_op_typed((_s), _op, intV);                         \
                tql_pop((_s)); tql_pop((_s)); tql_push((_s), &(_ret));     \
        } else {LOG_TQL("Unsupported type %c for %s",                      \
                    tql_stack_top((_s), 0)->type, #_op);}                  \
    } else { LOG_TQL("unsupported operands for %s", #_op); }               \

/** \name Helper functions
 * @{
 */

/** Real value comparison function
 * \param[in] v1 the first value to compare
 * \param[in] v2 the second value to compare
 * \retval >0 \p v1 greater than \p v2
 * \retval <0 \p v2 less than \p v2
 * \retval 0 values are equal within 0.0000001
 *           (enough for coordinates representation)
 */
static int tql_real_compare(const SXMTqlValue *v1, const SXMTqlValue *v2)
{
    static const double epsilon = 0.0000001;
    const double diff = v1->value.dblV - v2->value.dblV;
    /* Check if the v1 greater than v2; delta greater than positive epsilon */
    if (diff > epsilon) {
        return 1;
    }
    /* Check if the v2 greater than v1; delta less than positive epsilon */
    if (diff < -epsilon) {
        return -1;
    }
    /* Abs delta less than epsilon, thus, values are considerably equal */ 
    return 0;
}

/** String comparison function
 * \param[in] v1 the first value to compare
 * \param[in] v2 the second value to compare
 * \retval >0 \p v1 greater than \p v2
 * \retval <0 \p v2 less than \p v2
 * \retval 0 values are equal
 */
static int tql_string_compare(const SXMTqlValue *v1, const SXMTqlValue *v2)
{
   return strcmp(v1->value.str.strV, v2->value.str.strV);
}

/** @} */

/**
 * \name TQL Embedded function
 * @{
 */
#define tql_type_to_field_REAL  dblV
#define tql_type_to_field_INT   intV
#define tql_type_to_field_BOOL  boolV
#define tql_type_to_field_STR   str.strV

#define tql_uno_arg_func(_name, _func, _ret, _arg)                    \
    static int _name(void* arg)  {                                    \
        SXMTqlStatement s = (SXMTqlStatement)arg;                     \
        const int rc = tql_has_top(s, SXM_TQL_TYPE_##_arg, 1);        \
        if (rc == SXM_E_OK) {                                         \
            SXMTqlValue *val = tql_stack_top(s, 0);                   \
            val->type = SXM_TQL_TYPE_##_ret;                          \
            val->value.tql_type_to_field_##_ret =                     \
                _func(val->value.tql_type_to_field_##_arg);           \
            tql_pop(s); tql_push(s, val);                             \
        } else { LOG_TQL("Failed to execute function %s (%s)",        \
                         #_name, #_func);}                            \
        return rc; }                                                  \

static int sxm_cast_int(double v)
{
    return (int)v;
}

static SXM_TQL_BOOL sxm_str_is_null(const char *v)
{
    return ((!v) || !strlen(v)) ? SXM_TQL_TRUE : SXM_TQL_FALSE;
}

#if SXM_TQL_MATH_SUPPORT
    tql_uno_arg_func(tql_func_cos, cos, REAL, REAL)
    tql_uno_arg_func(tql_func_sin, sin, REAL, REAL)
    tql_uno_arg_func(tql_func_sqrt, sqrt, REAL, REAL)
    tql_uno_arg_func(tql_func_abs, abs, INT, INT)
#else
    static int sxm_abs(int v)
    {
        return (v < 0) ? -v : v;
    }
    tql_uno_arg_func(tql_func_abs, sxm_abs, INT, INT)
#endif
tql_uno_arg_func(tql_func_int, sxm_cast_int, INT, REAL)
tql_uno_arg_func(tql_func_is_null, sxm_str_is_null, BOOL, STR)

/** Set of embedded TQL functions
 */
static const RpnFunction tql_functions[] =
{
#if SXM_TQL_MATH_SUPPORT
    {1, "sin", tql_func_sin},
    {1, "cos", tql_func_cos},
    {1, "sqrt", tql_func_sqrt},
#endif
    {1, "abs", tql_func_abs},
    {1, "int", tql_func_int},
    {1, "isnull", tql_func_is_null},

    /* TERMINATOR [MUST BE THE LAST]*/
    {0, NULL, NULL}
};

/**
 * @}
 */

/**
 * ----------------------------------------------------------------------------
 */
int sxm_tql_statement(SXMTqlStatement *statement, const SXMTqlCallback *callback,
                      const char *query, const void *arg)
{
    int rc, special_case = 0;
    if (!statement || !callback || !query ||
        !callback->tprobe || !callback->cprobe || !callback->row_lookup)
    {
        rc = SXM_E_FAULT;
    }
    else if (query[0] == '\0')
    {
        rc = SXM_E_INVAL;
    }
    else if ((*statement =
                (SXMTqlStatement)sxe_calloc(1, sizeof(struct _sxm_tql_statement_struct))) == NULL)
    {
        rc = SXM_E_NOMEM;
    }
    else
    {
        (*statement)->callback = callback;
        (*statement)->callback_arg = arg;

        syn_init(&(*statement)->syn, query);

        //////////////////////////////////////////////////////
        // Probe the DB to fetch out DB information
        rc = (*statement)->callback->tprobe(
                        &(*statement)->sDB.db_tables[0],
                        SXMTQL_TABLE_MAX_COUNT,
                        &(*statement)->sDB.tables_count,
                        arg);
        if (rc == SXM_E_OK)
        {
            rc = SXM_E_INVAL;

            syn_next_token(&(*statement)->syn);
            if (syn_token_is(&(*statement)->syn, LEX_TOKEN_PERIOD))
            {
                syn_next_token(&(*statement)->syn);
                if (syn_token_is(&(*statement)->syn, LEX_TOKEN_IDENTIFIER))
                {
                    rc = SXM_E_OK;
                    special_case = !0;

                    // If this is a special command?
                    if (!strcmp(syn_token_str(&(*statement)->syn), "help"))
                    {
                        printf(
                            ".help - show this message\n"
                            ".schema - show schema for all tables kept by DB\n"
                            "select [*|columns] from [table] where [expression];\n");
                    }
                    else if (!strcmp(syn_token_str(&(*statement)->syn), "schema"))
                    {
                        int table_iter, column_iter;
                        SXMTqlTable *tql_table;
                        TqlTable *table;
                        for (table_iter = 0,
                             tql_table = &(*statement)->sDB.db_tables[0],
                             table = &(*statement)->sDB.tables[0];
                             table_iter < (*statement)->sDB.tables_count;
                             ++table_iter, ++tql_table)
                        {
                            const SXMTqlTableColumn *column;
                            printf("TABLE %s (", tql_table->name);
                            // Do the request through the callback to learn about the table
                            rc = (*statement)->callback->cprobe(
                                        tql_table->id,
                                        table->columns,
                                        TQL_DB_FIELDS_COUNT_MAX,
                                        &table->columns_count,
                                        arg);
                            for (column_iter = 0, column = table->columns;
                                 column_iter < table->columns_count;
                                 ++column_iter, ++column)
                            {
                                const char *type = "UNKNOWN";
                                switch (column->type)
                                {
                                    case SXM_TQL_TYPE_REAL: type = "REAL"; break;
                                    case SXM_TQL_TYPE_INT: type = "INT"; break;
                                    case SXM_TQL_TYPE_STR: type = "STRING"; break;
                                    case SXM_TQL_TYPE_BOOL: type = "BOOL"; break;
                                    case SXM_TQL_TYPE_BLOB: type = "BLOB"; break;
                                }
                                printf("%s %s%c", column->name, type,
                                    (column_iter + 1 == table->columns_count ? ' ' : ','));
                            }
                            printf(");\n");
                        }                           
                    }
                    else 
                    {
                        special_case = 0;
                        rc = SXM_E_INVAL;    
                    }
                }
            }
            else if (syn_token_is(&(*statement)->syn, LEX_TOKEN_IDENTIFIER))
            {
                rc = !strcmp(syn_token_str(&(*statement)->syn), "select") ? SXM_E_OK : SXM_E_INVAL;
            }
            else
            {
                rc = SXM_E_INVAL;
            }

            if ((rc == SXM_E_OK) && !special_case)
            {
                syn_next_token(&(*statement)->syn);
                if (syn_token_is(&(*statement)->syn, LEX_TOKEN_STAR))
                {
                    // Requested ALL columns of the table
                    LOG_TQL("(%s)", "ALL FIELDS REQUESTED");
                    syn_skip_identifier(&(*statement)->syn, "from");
                }
                else if (syn_token_is(&(*statement)->syn, LEX_TOKEN_IDENTIFIER))
                {
                    // The field name has specified. Thus, need to read all them out
                    LOG_TQL("(%s)", "certain field requested");
                    for(;;)
                    {
                        LOG_TQL("column %s", syn_token_str(&(*statement)->syn));
                        (*statement)->sDB.query.columns[
                                    (*statement)->sDB.query.columns_count++].name = 
                               syn_token_str(&(*statement)->syn);
                
                        syn_next_token(&(*statement)->syn);
                        if (syn_token_is(&(*statement)->syn, LEX_TOKEN_COMMA))
                        {
                            // Continue reading column names
                            syn_next_token(&(*statement)->syn);
                            if (!syn_token_is(&(*statement)->syn, LEX_TOKEN_IDENTIFIER))
                            {
                                syn_error(&(*statement)->syn,
                                    "%s\n","Syntax error - identifier required");
                                tql_failure(*statement, TQL_E_ERROR);
                                rc = SXM_E_INVAL;
                                break;
                            }
                        }
                        else if (syn_token_is(&(*statement)->syn, LEX_TOKEN_IDENTIFIER))
                        {
                            if (strcmp(syn_token_str(&(*statement)->syn), "from"))
                            {
                                syn_error(&(*statement)->syn,
                                    "Syntax error - FROM statement required, but '%s' found\n",
                                    syn_token_str(&(*statement)->syn));
                                tql_failure(*statement, TQL_E_ERROR);
                            }
                            break;
                        }
                        else
                        {
                            syn_error(&(*statement)->syn,
                                "Syntax error - Unexpected %d token type\n",
                                syn_token_type(&(*statement)->syn));
                            tql_failure(*statement, TQL_E_ERROR);
                            break;
                        }
                    }
                }

                // Read out table name
                syn_skip_type(&(*statement)->syn, LEX_TOKEN_IDENTIFIER);
                if (rc == SXM_E_OK)
                {
                    int table_iter;

                    LOG_TQL("probed %d tables", (*statement)->sDB.tables_count);

                    for (table_iter = 0;
                         table_iter < (*statement)->sDB.tables_count;
                         ++table_iter)    
                    {
                        (*statement)->sDB.tables[table_iter].table =
                            &(*statement)->sDB.db_tables[table_iter];
                        if (!strcmp(syn_token_str(&(*statement)->syn),
                                    (*statement)->sDB.db_tables[table_iter].name))
                        {
                            (*statement)->sDB.query.table_ref =
                                &(*statement)->sDB.tables[table_iter];
                        }
                    }

                    if (!(*statement)->sDB.query.table_ref)
                    {
                        LOG_TQL("there is no requested table '%s'",
                            syn_token_str(&(*statement)->syn));
                        tql_failure(*statement, TQL_E_ERROR);
                        rc = SXM_E_INVAL;
                    }
                }
                else
                {
                    LOG_TQL("Failed to probe the tables (%d)", rc);
                    tql_failure(*statement, TQL_E_ERROR);
                    rc = SXM_E_INVAL;
                }
            }
            else
            {
                rc = SXM_E_INVAL;
            }
        }
        
        if (rc == SXM_E_OK)
        {
            // Do the request through the callback to learn about the table
            rc = (*statement)->callback->cprobe(
                       (*statement)->sDB.query.table_ref->table->id,
                       (*statement)->sDB.query.table_ref->columns,
                       TQL_DB_FIELDS_COUNT_MAX,
                       &(*statement)->sDB.query.table_ref->columns_count,
                       arg);
            if (rc == SXM_E_OK)
            {
                rc = tql_link_query_to_schema(*statement);
                if (rc == SXM_E_OK)
                {
                    // The only WHERE word supported
                    syn_next_token(&(*statement)->syn);
                    if (syn_token_is(&(*statement)->syn, LEX_TOKEN_SEMICOLON))
                    {
                        LOG_TQL("(%s)", "There is no conditions, pop-up all data from the table");
                    }
                    else if (syn_token_is(&(*statement)->syn, LEX_TOKEN_IDENTIFIER))
                    {
                        if (!strcmp(syn_token_str(&(*statement)->syn), "where"))
                        {
                            LOG_TQL("(%s)",
                                "Condition found. Use the expression parser prepare the RPN");
                            rc = rpn_init(&(*statement)->rpn, gKeywords, tql_functions,
                                          tql_var_lookup, *statement);
                            if (rc == SXM_E_OK)
                            {
                                rc = rpn_build((*statement)->rpn, &(*statement)->syn);
                                #if SXMTQL_LOG & SXMTQL_TQL_LOG
                                if (rc == SXM_E_OK)
                                {
                                    rpn_dump(stdout, (*statement)->rpn);
                                }
                                #endif
                            }

                            if (rc != SXM_E_OK)
                            {
                                tql_failure(*statement, TQL_E_ERROR);
                            }
                        }
                        else
                        {
                            syn_error(&(*statement)->syn,
                                "Syntax error - WHERE statement required, but %s found",
                                syn_token_str(&(*statement)->syn));
                            tql_failure(*statement, TQL_E_ERROR);
                        }
                    }
                    else
                    {
                        syn_error(&(*statement)->syn,
                            "Syntax error - Unexpected %d token type",
                            syn_token_type(&(*statement)->syn));
                        tql_failure(*statement, TQL_E_ERROR);
                    }  
                }
            }
            else
            {
                LOG_TQL("Unable to fetch data for table '%s'",
                    (*statement)->sDB.query.table_ref->table->name);
                tql_failure(*statement, TQL_E_ERROR);
            }
        }
    }

    if (statement &&
        ((rc != SXM_E_OK) || ((*statement) && !tql_is_ok(*statement))))
    {
        sxm_tql_statement_destroy(*statement);
        *statement = NULL;

        if (special_case)
        {
            rc = SXM_E_OK;
        }
    }

    return rc;
}

/**
 * ----------------------------------------------------------------------------
 */
int sxm_tql_evaluate(SXMTqlStatement statement, const void *row_data)
{
    int rc;

    if (!statement)
    {
        rc = SXM_E_FAULT;
    }
    else
    {
        LOG_TQL("--- EVALUATION for %p", row_data);
        if (statement->rpn)
        {
            int rpn_term;
            RpnToken *token;
            SXMTqlValue value;

            rc = SXM_E_NOENT;

            statement->sConveyor.stack_size = 0;

            for (rpn_term = 0, token = statement->rpn->sOutput.tokens;
                 rpn_term < statement->rpn->sOutput.tokens_count;
                 ++rpn_term, ++token)
            {
                switch (token->op_code)
                {
                    case RPN_TYPE_CONST_STRING:
                        sxm_tql_val_set_str(&value, lex_token_str(&token->lex));
                        tql_push(statement, &value);
                        break;
                    case RPN_TYPE_CONST_NUMBER:
                        sxm_tql_val_set_real(&value, lex_token_real(&token->lex));
                        tql_push(statement, &value);
                        break;
                    case RPN_TYPE_VARIABLE:
                        if (statement->callback->row_lookup(
                                       statement->sDB.query.table_ref->table->id,
                                       token->data.variable_id,
                                       &value, row_data, statement->callback_arg) == SXM_E_OK)
                        {
                            // Do the conversion from int to double
                            if (value.type == SXM_TQL_TYPE_INT)
                            {
                                sxm_tql_val_set_real(&value, tql_value_typed(&value, intV));
                            }
                            tql_push(statement, &value);    
                        }
                        else
                        {
                            LOG_TQL("Failed to query column by id %d",
                                token->data.variable_id);
                        }
                        break;
                    case RPN_TYPE_FUNCTION:
                        LOG_TQL("Execution function '%s'", token->data.func->txt);
                        token->data.func->func(statement);
                        break;
                    case RPN_TYPE_OP_LIKE:
                        if (tql_has_top(statement, SXM_TQL_TYPE_STR, 2) == SXM_E_OK)
                        {
                            SXMTqlValue ret;
                            ret.type = SXM_TQL_TYPE_BOOL;
                            ret.value.boolV =
                                (strstr(tql_stack_top(statement, 1)->value.str.strV,
                                        tql_stack_top(statement, 0)->value.str.strV)) ?
                                            SXM_TQL_TRUE : SXM_TQL_FALSE;
                            tql_pop(statement);
                            tql_pop(statement);
                            tql_push(statement, &ret);
                        }
                        else
                        {
                            LOG_TQL("(%s)", "Failed to query process LIKE statement");
                        }
                        break;
                    case RPN_TYPE_OP_NOT:
                        if (tql_has_top(statement, SXM_TQL_TYPE_BOOL, 1) == SXM_E_OK)
                        {
                            tql_stack_top(statement, 0)->value.boolV = 
                                (tql_stack_top(statement, 0)->value.boolV ==
                                    SXM_TQL_TRUE) ? SXM_TQL_FALSE : SXM_TQL_TRUE;
                        }
                        else
                        {
                            LOG_TQL("(%s)", "Failed to query process NOT statement");
                        }
                        break;
                    case RPN_TYPE_OP_AND:
                        tql_op_bin_logical(statement, &&, value);
                        break;
                    case RPN_TYPE_OP_OR:
                        tql_op_bin_logical(statement, ||, value);
                        break;
                    case RPN_TYPE_OP_LESS:
                        tql_op_bin_compare(statement,  <, value);
                        break;
                    case RPN_TYPE_OP_LESS_EQUAL:
                        tql_op_bin_compare(statement, <=, value);
                        break;
                    case RPN_TYPE_OP_EQUAL:
                        tql_op_bin_compare(statement, ==, value);
                        break;
                    case RPN_TYPE_OP_GREATER_EQUAL:
                        tql_op_bin_compare(statement, >=, value);
                        break;
                    case RPN_TYPE_OP_GREATER:
                        tql_op_bin_compare(statement,  >, value);
                        break;
                    case RPN_TYPE_OP_NOT_EQUAL:
                        tql_op_bin_compare(statement, !=, value);
                        break;
                    case RPN_TYPE_OP_PLUS:
                        tql_op_bin_ariphetical(statement, +, value);
                        break;
                    case RPN_TYPE_OP_MINUS:
                        tql_op_bin_ariphetical(statement, -, value);
                        break;
                    case RPN_TYPE_OP_MULTIPLY:
                        tql_op_bin_ariphetical(statement, *, value);
                        break;
                    case RPN_TYPE_OP_DIVIDE:
                        tql_op_bin_ariphetical(statement, /, value);
                        break;
                    case RPN_TYPE_OP_MOD:
                        tql_op_bin_int_ariphetical(statement, %, value);
                        break;
                    case RPN_TYPE_OP_COMMA:
                        // Keep working assuming
                        break;
                    
                    /* Unhandled tokens */
                    case RPN_TYPE_BRACE_LEFT:
                    case RPN_TYPE_UNKNOWN:
                        break;
                }    
            }

            if (tql_stack_depth(statement) == 1)
            {
                rc = tql_has_top(statement, SXM_TQL_TYPE_BOOL, 1);
                if (rc == SXM_E_OK)
                {
                    rc = (tql_stack_top(statement, 0)->value.boolV == SXM_TQL_TRUE)
                        ? SXM_E_OK : SXM_E_NOENT;
                }
                else
                {
                    LOG_TQL("The condition must be boolean expression (%d)", rc);
                    tql_failure(statement, TQL_E_ERROR);
                }
            }
            else
            {
                LOG_TQL("Invalid expression. Calculation stack is %d items",
                    tql_stack_depth(statement));
                tql_failure(statement, TQL_E_ERROR);
                rc = SXM_E_STATE;
            }
        }
        else
        {
            rc = SXM_E_OK;
        }
        if (rc == SXM_E_OK)
        {
            const TqlQueryColumn *q_column;
            int column_iter;
            for (column_iter = 0, q_column = statement->sDB.query.columns;
                 (column_iter < statement->sDB.query.columns_count) && (rc == SXM_E_OK);
                 ++column_iter, ++q_column)
            {
                SXMTqlValue v;
                rc = statement->callback->row_lookup(statement->sDB.query.table_ref->table->id,
                                                     q_column->column->id, &v, row_data,
                                                     statement->callback_arg);
                if (rc == SXM_E_OK)
                {
                    rc = statement->callback->value_output(statement->sDB.query.table_ref->table->id,
                                                           q_column->column->id, &v, statement->callback_arg);
                    if (rc != SXM_E_OK)
                    {

                        LOG_TQL("Failed to print data for table %d, columt %d",
                            statement->sDB.query.table_ref->table->id,
                            q_column->column->id);
                    }
                }
                else
                {
                    LOG_TQL("Failed to query column by id %d", q_column->column->id);
                }  
            }
            printf("|\n");
        }
        
    }
    return rc;
}

/**
 * ----------------------------------------------------------------------------
 */
int sxm_tql_query_table(SXMTqlStatement statement, const char *name)
{
    int rc;
    if (!statement)
    {
        rc = SXM_E_FAULT;
    }
    else if (strcmp(statement->sDB.query.table_ref->table->name, name) != 0)
    {
        rc = SXM_E_INVAL;
    }
    else
    {
        rc = SXM_E_OK;
    }
    return rc;
}

/**
 * ----------------------------------------------------------------------------
 */
int sxm_tql_statement_destroy(SXMTqlStatement statement)
{
    int rc;
    if (!statement)
    {
        rc = SXM_E_FAULT;
    }
    else
    {
        if (statement->rpn)
        {
            sxe_free(statement->rpn);
        }
        syn_uninit(&(statement->syn));
        sxe_free(statement);
        rc = SXM_E_OK;
    }
    return rc;
}
/**
 * @}
 */

/**
 * \name TQL Helpers
 * @{
 */

/**
 * ----------------------------------------------------------------------------
 */
static const SXMTqlTableDef *tqlhelper_find_table(const SXMTqlTableDef *tables, int id)
{
    const SXMTqlTableDef *ptable_def = &tables[0];

    while ((ptable_def->table.id >= 0) && (id != ptable_def->table.id))
    {
        ++ptable_def; // Move to the next table
    }

    return (ptable_def->table.id >= 0) ? ptable_def : NULL;
}

/**
 * ----------------------------------------------------------------------------
 */
static const SXMTqlTableColumnDef *tqlhelper_find_column(const SXMTqlTableDef *table, int id)
{
    const SXMTqlTableColumnDef *pcolumn_def = &table->columns[0];

    while ((pcolumn_def->column.id >= 0) && (id != pcolumn_def->column.id))
    {
        ++pcolumn_def; // Move to the next table
    }

    return (pcolumn_def->column.id >= 0) ? pcolumn_def : NULL;
} 

/**
 * ----------------------------------------------------------------------------
 */
static int sxmtql_default_probe(SXMTqlTable *tables, int capacity,
                                int *size, const void *arg)
{
    SXMTqlShemeCallback* callback = (SXMTqlShemeCallback*)arg;
    const SXMTqlTableDef *ptable_def = (*callback)();
    SXMTqlTable *out_table = tables;

    *size = 0; // Assume no tables defined

    // Copy as many as it's possible
    while ((ptable_def->table.id >= 0) && (*size < capacity))
    {
        *out_table = ptable_def->table;
        ++(*size);
        ++out_table;
        ++ptable_def;
    }

    return SXM_E_OK;
}

/**
 * ----------------------------------------------------------------------------
 */
static int sxmtql_default_column_probe(int table_id, SXMTqlTableColumn *columns,
                                       int capacity, int *size, const void *arg)
{
    SXMTqlShemeCallback* callback = (SXMTqlShemeCallback*)arg;
    const SXMTqlTableDef *table = tqlhelper_find_table((*callback)(), table_id);

    if (table)
    {
        // Give back the columns
        const SXMTqlTableColumnDef *pcolumn_def = &(table->columns[0]);
        SXMTqlTableColumn *out_column = columns;

        *size = 0; // Assuming no columns

        // Copy as many as it's possible
        while ((pcolumn_def->column.id >= 0) && (*size < capacity))
        {
            *out_column = pcolumn_def->column;
            ++out_column;
            ++pcolumn_def;
            ++(*size);
        }
        return SXM_E_OK;
    }
    return SXM_E_NOENT;
}

/**
 * ----------------------------------------------------------------------------
 */
#define TQLHELPER_TYPED_FIELD(_type, _row, _column) \
    *((_type*)((size_t)(_row) + (_column)->field_offset))

/**
 * ----------------------------------------------------------------------------
 */
static int sxmtql_default_row_lookup(int table_id, int id, SXMTqlValue *value,
                                     const void *row_data, const void *arg)
{
    SXMTqlShemeCallback* callback = (SXMTqlShemeCallback*)arg;
    const SXMTqlTableDef *table = tqlhelper_find_table((*callback)(), table_id);
    if (table)
    {
        const SXMTqlTableColumnDef *column = 
            tqlhelper_find_column(table, id);
        if (column)
        {
            switch (column->column.type)
            {
                case SXM_TQL_TYPE_INT:
                    switch (column->field_byte_size)
                    {
                        case sizeof(int):
                            sxm_tql_val_set_int(value,
                                TQLHELPER_TYPED_FIELD(int, row_data, column));
                            break;
                        case sizeof(ushort):
                            sxm_tql_val_set_int(value,
                                TQLHELPER_TYPED_FIELD(ushort, row_data, column));
                            break;
                        case sizeof(byte):
                            sxm_tql_val_set_int(value,
                                TQLHELPER_TYPED_FIELD(byte, row_data, column));
                            break;
                    }
                    break;
                case SXM_TQL_TYPE_REAL:
                    switch (column->field_byte_size)
                    {
                        case sizeof(double):
                            sxm_tql_val_set_real(value,
                                TQLHELPER_TYPED_FIELD(double, row_data, column));
                            break;
                        case sizeof(float):
                            sxm_tql_val_set_real(value,
                                TQLHELPER_TYPED_FIELD(float, row_data, column));
                            break;
                    }
                    break;
                case SXM_TQL_TYPE_STR:
                    if (column->field_byte_size == sizeof(char))
                    {
                        char * const buff = &value->value.str.symbV[0];
                        value->type = SXM_TQL_TYPE_STR;
                        buff[0] = TQLHELPER_TYPED_FIELD(char, row_data, column);
                        buff[1] = '\0';
                        value->value.str.strV = buff;
                    }
                    else 
                    {
                        sxm_tql_val_set_str(value,
                            TQLHELPER_TYPED_FIELD(const char*, row_data, column));
                    }
                    break;
                case SXM_TQL_TYPE_BOOL:
                    sxm_tql_val_set_bool(value,
                        TQLHELPER_TYPED_FIELD(SXM_TQL_BOOL, row_data, column));
                    break;
                case SXM_TQL_TYPE_BLOB:
                    // Here we've got no idea how to process this. Thus, let's fill
                    // something to let caller do something.
                    value->type = SXM_TQL_TYPE_BLOB;
                    value->value.blob.data = (void*)row_data;
                    value->value.blob.count = 1;
                    value->value.blob.size = 1;
                    break;
            }
            return SXM_E_OK;
        }
    }
    return SXM_E_NOENT;
}

/**
 * ----------------------------------------------------------------------------
 */
static int sxmtql_default_value_output(int table, int column,
                                       const SXMTqlValue *value,
                                       const void *arg)
{
    UNUSED_VAR(table);
    UNUSED_VAR(column);
    UNUSED_VAR(arg);

    switch (value->type)
    {
        case SXM_TQL_TYPE_REAL:
            printf("|%lf", value->value.dblV);
            break;
        case SXM_TQL_TYPE_INT:
            printf("|%d", value->value.intV);
            break;
        case SXM_TQL_TYPE_STR:
            printf("|%s", value->value.str.strV ? value->value.str.strV : "");
            break;
        case SXM_TQL_TYPE_BOOL:
            printf("|%s", (value->value.boolV == SXM_TQL_TRUE) ? "true" : "false");
            break;
        case SXM_TQL_TYPE_BLOB:
        {
            int block;
            const char *p_data = (const char*)value->value.blob.data;
            
            for (block = 0; block < value->value.blob.count; ++block)
            {
                int block_remain;
                printf("(");
                block_remain = value->value.blob.size;
                while (block_remain --> 0)
                {
                    printf("%2X", (int)(*p_data));
                    ++p_data;
                }
                printf(")");
            }
            
            break;
        }
    }
    return SXM_E_OK;
}

/**
 * ----------------------------------------------------------------------------
 */
static const SXMTqlCallback sxmtql_default_callback_interface = 
{
    sxmtql_default_probe,
    sxmtql_default_column_probe,
    sxmtql_default_row_lookup,
    sxmtql_default_value_output
};

const SXMTqlCallback * const SXMTQL_DEFAULT_INTERFACE = &sxmtql_default_callback_interface;

/**
 * @}
 */

#ifdef TQL_STANDALONE
/**
 * \name APPLICATION EXAMPLE
 * @{
 */
typedef struct
{
   int id;
   char type;
   int count;
   const char *desc;
   double lat;
   double lon;
} TestRowData;

SXMTQL_TABLES()
    SXMTQL_TABLE(test)
        SXMTQL_COLUMN(id, INT, TestRowData, id)
        SXMTQL_COLUMN(type, STR, TestRowData, type)
        SXMTQL_COLUMN(count, INT, TestRowData, count)
        SXMTQL_COLUMN(desc, STR, TestRowData, desc)
        SXMTQL_COLUMN(lat, REAL, TestRowData, lat)
        SXMTQL_COLUMN(lon, REAL, TestRowData, lon)
    SXMTQL_TABLE_END()
    SXMTQL_TABLE(test2)
        SXMTQL_COLUMN(id, INT, TestRowData, id)
        SXMTQL_COLUMN(type, STR, TestRowData, type)
        SXMTQL_COLUMN(count, INT, TestRowData, count)
        SXMTQL_COLUMN(desc, STR, TestRowData, desc)
        SXMTQL_COLUMN(lon, REAL, TestRowData, lon)
        SXMTQL_COLUMN(lat, REAL, TestRowData, lat)
    SXMTQL_TABLE_END()
SXMTQL_TABLES_END()

static const TestRowData table_1_rows[] = 
{
    {1, 'A', 123, "ALASKA",   80.0, -170.0},
    {2, 'A', 132, "FLORIDA",  30.0,  -40.0},
    {3, 'Z', 259, "MICHIRAN", 40.0,  -83.0}
};

static const TestRowData table_2_rows[] = 
{
    {1, 'Z', 345, "Ontario",     80.0, -170.0},
    {2, 'V', 128, "Quebec",      85.0,  -40.0},
    {3, '@', 100, "Nova Scotia", 70.0, -83.0}
};

int main(int argc, char **argv)
{ 
    int rc;
    SXMTqlStatement s = NULL;

    printf("sizeof(SXMTqlStatement)=%d\n", sizeof(*s));

    rc = sxm_tql_statement(&s, SXMTQL_DEFAULT_INTERFACE, argv[1], SXMTQL_SERVICE_TABLES);
    if (rc == SXM_E_OK)
    {
        const TestRowData *table;
        size_t table_size, i_row;

        if (sxm_tql_query_table(s, "test") == SXM_E_OK)
        {
            table = table_1_rows;
            table_size = ARRAY_SIZE(table_1_rows);
        }
        else if (sxm_tql_query_table(s, "test2") == SXM_E_OK)
        {
            table = table_2_rows;
            table_size = ARRAY_SIZE(table_2_rows);
        }
        else
        {
            table = NULL;
            table_size = 0;
        }
        
        for (i_row = 0; i_row < table_size; ++i_row)
        {
            sxm_tql_evaluate(s, &table[i_row]);
        }

        sxm_tql_statement_destroy(s);
    }
    else
    {
        non_fatal("Failed to create the statement for query\n'%s'\n", argv[1]);
    }
    return EXIT_SUCCESS;
}

/**
 * @}
 */
#endif

