/************************************************************************
 *                                                                      *
 *            Build and Analyse the parking database and cycle file     *
 *            ==================================================        *
 *                                                                      *
 *                                 Copyright 2013 Sirius XM Radio, Inc. *
 *                                                 All Rights Reserved. *
 *               Licensed Materials - Property of Sirius XM Radio, Inc. *
 *                                                                      *
 ************************************************************************/

#include <ctype.h>
#include "sdkfiles.h"

/** NB: TO BE SURE THAT ALL DATA STRUCTURES AND MACROSES ARE THE SAME **/
#define SDKFILES_BUILDER
#include <parking/sxm_parking_internal.h>
#undef SDKFILES_BUILDER

enum {
    /** Defines parking block size */
    PARKING_BLOCK_SIZE = (5120U),
    /** Defines Fuel t-file use data size in single block */
    PARKING_BLOCK_USER_SIZE = sxm_tfile_bsize_static(PARKING_BLOCK_SIZE),
    /** Number of bytes for Parking Station storage by Regions */
    PARKING_TFILE_DATA_SIZE = 1024U * 8192U,
    /** Defines parking root size in blocks */
    PARKING_ROOT_BLOCK_COUNT = 
        (((PARKING_MAX_REGIONS * (2 * sizeof(ushort))) + PARKING_BLOCK_SIZE - 1) / PARKING_BLOCK_SIZE),
    /** Defines size of the root in bytes aligned by block size */
    PARKING_ROOT_BLOCK_SIZE = 
       PARKING_ROOT_BLOCK_COUNT * PARKING_BLOCK_SIZE,
    /** Defines overall T-File size on blocks */
    PARKING_TFILE_BLOCK_COUNT =
        (((sizeof(SXMTFileHeader) + PARKING_BLOCK_SIZE - 1) / PARKING_BLOCK_SIZE) +
        (PARKING_ROOT_BLOCK_COUNT * SXM_TFILE_TRANSACTION_MAX) +
        ((PARKING_TFILE_DATA_SIZE + PARKING_BLOCK_USER_SIZE - 1) / PARKING_BLOCK_USER_SIZE)),
    /** Final T-File size on bytes, just FYI */
    PARKING_TFILE_FILE_SIZE = PARKING_TFILE_BLOCK_COUNT * PARKING_BLOCK_SIZE,
    /** Maximum size of the line in psv files */
    PARKING_PSV_LINE_SIZE_MAX = 1024,
    /** Maximum value of the PRICE field */
    PARKING_BASELINE_PRICE_VAL_MAX = 99,
    /** Minimum value of VERSION */
    PARKING_VERSION_MIN = 0,
    /** Maximum value of VERSION */
    PARKING_VERSION_MAX = 999,
};

typedef struct {
    uint region, id, h1, h2;
    int price, spaces;
    ushort ver;
    const char *name, *address1, *address2,
               *city, *state, *zip, *phone, *comm, *operation;
    double lat1, lon1, lat2, lon2;
    ushort amen1, amen2;
} Parking;

/*******************************************************************************
*
* Function    :  verifyOperatingHoursFormat
*
* Description :  Checks format of operating hours in input string
*
********************************************************************************/
static int verifyOperatingHoursFormat(const char **ppText) {

    // SX-9845-0303 - 3.3.18
    // The format for each day's operating hours is 
    // a) 'HH:MM-HH:MM' or b) 'HH:MM-H:MM', where the
    // first HH:MM carries the opening time and 
    // the second carries the closing time.
    // If b) variant is used, it means that time period 
    // overlaps midnight.
    // All times are local and expressed using a 24 hour clock.
    // If a Parking Location is open for 24 hours a day
    // then the text string for each day of 24 hour operation is 'O'.
    // If a Parking Location is closed for a day
    // then the text string for that day is 'C'.

    if ('O' == **ppText || 'C' == **ppText) {
        (*ppText)++;
        return SXM_E_OK;
    }

    switch (0) { default: {
        if ((isdigit(*((*ppText)++)) == 0) || (isdigit(*((*ppText)++)) == 0)) {
            /* hours should start from two digits */
            break;
        }

        if (*((*ppText)++) != ':') {
            /* next symbol should be separator */
            break;
        }

        if ((isdigit(*((*ppText)++)) == 0) || (isdigit(*((*ppText)++)) == 0)) {
            /* next two symbols should be digits */
            break;
        }

        if (*((*ppText)++) != '-') {
            /* now it should be dash */
            break;
        }

        if (isdigit((*(*ppText)++)) == 0) {
            /* hours should start from digit */
            break;
        }

        if (isdigit(**ppText) == 0) {
            if (**ppText == ':') {
                (*ppText)++;
            }
            else {
                /* next symbol should be digit or separator */
                break;
            }
        }
        else {
            (*ppText)++;
            if (**ppText == ':') {
                (*ppText)++;
            }
            else {
                /* next symbol should be separator */
                break;
            }
        }

        if ((isdigit(*((*ppText)++)) == 0) || (isdigit(*((*ppText)++)) == 0)) {
            /* next two symbols should be digits */
            break;
        }

        return SXM_E_OK;
    }}

    return SXM_E_INVAL;
}

static int verifyOperationFormat(const char *pOperationText) {

    int rc = SXM_E_OK;
    byte numDays = 0;

    // SX-9845-0303 - 3.3.18
    // This field is an array of text strings which provide the normal
    // operating hours of the Parking Location.
    // The array contains 7 space (ASCII 0x20) separated text strings
    // where the first string represents the hours of operation for Monday
    // followed by that for Tuesday etc. with the last text string
    // representing the hours of operation for Sunday.

    // Some Parking Locations may not have any information on Operating hours.
    if ('\0' == *pOperationText) {
        return SXM_E_OK;
    }

    while ('\0' != *pOperationText) {

        rc = verifyOperatingHoursFormat(&pOperationText);
        if (SXM_E_OK != rc) {
            break;
        }

        // Next character must be space or '\0'
        if (' ' == *pOperationText) {
            pOperationText++;
        } else if ('\0' != *pOperationText) {
            rc = SXM_E_INVAL;
            break;
        }

        numDays++;
    }

    if (numDays != 7)
    {
        rc = SXM_E_INVAL;
    }

    return rc;
}

static int parking_read_optional_int_val(SxmCsvParser *pPsv,
                                         uint index, int *pRet) {

    int rc = sxm_csv_get_int_val(pPsv, index, pRet);
    if (SXM_E_OK != rc) {
        if (SXM_E_NOENT == rc) {
            *pRet = 0;
            rc = SXM_E_OK;
        }
    }

    return rc;
}

static int parking_read_optional_fix_val(SxmCsvParser *pPsv,
                                         uint index, fix *pRet) {

    int rc = sxm_csv_get_fix_val(pPsv, index, pRet);
    if (SXM_E_OK != rc) {
        if (SXM_E_NOENT == rc) {
            *pRet = 0;
            rc = SXM_E_OK;
        }
    }

    return rc;
}

static int parking_build_locations(SXMTFile *pDbFile,
                                   SxmCsvParser *pPsv,
                                   uint *pMaxVer)
{
    int rc= SXM_E_OK;
    SXMTFileStream *pStream = NULL;
    uint *pParkingIndex = NULL; /* offset index (1 int/index) */

    switch (0) { default: {

        int iver, iregion, iid, iname, iaddress1, iaddress2, icity, istate, izip,
            iphone, ilat1, ilon1, ilat2, ilon2, ihgt1, ihgt2, ispaces, ioper,
            iprice, icomm, iamen1, iamen2;
        ushort *pIndexBlock;
        int prevRegion = -1;
        uint rmask[(PARKING_MAX_REGIONS + SXM_ARCH_INT_BITS - 1) / SXM_ARCH_INT_BITS];

        /* Allocate space for parking index table */
        pParkingIndex = (uint*)sxe_malloc(PARKING_MAX_STATIONS * sizeof(*pParkingIndex));
        if (NULL == pParkingIndex) {
            non_fatal("Failed to allocate space for parking index");
            rc = SXM_E_NOMEM;
            break;
        }

        /* 2048 regions.  2 byte offset, 2 byte count */
        pIndexBlock = (ushort *)sxm_tfile_root(pDbFile);
        if (NULL == pIndexBlock) {
            non_fatal("Failed to get root");
            rc = SXM_E_ERROR;
            break;
        }

        /* Create the stream for update */
        rc = sxm_tstream_create(pDbFile, &pStream);
        if (rc != SXM_E_OK) {
            non_fatal("Failed to create stream (rc=%d)", rc);
            break;
        }

        /* Read fields indexes */

        iregion = sxm_csv_index(pPsv, "PREGION");
        if (iregion < 0) {
            non_fatal("Column 'PREGION' is not found");
            rc = SXM_E_INVAL;
            break;
        }

        iid = sxm_csv_index(pPsv, "PUID");
        if (iid < 0) {
            non_fatal("Column 'PUID' is not found");
            rc = SXM_E_INVAL;
            break;
        }

        iver = sxm_csv_index(pPsv, "VER");
        if (iver < 0) {
            non_fatal("Column 'VER' is not found");
            rc = SXM_E_INVAL;
            break;
        }

        iname = sxm_csv_index(pPsv, "NAME");
        if (iname < 0) {
            non_fatal("Column 'NAME' is not found");
            rc = SXM_E_INVAL;
            break;
        }

        iaddress1 = sxm_csv_index(pPsv, "ADDRESS1");
        if (iaddress1 < 0) {
            non_fatal("Column 'ADDRESS1' is not found");
            rc = SXM_E_INVAL;
            break;
        }

        iaddress2 = sxm_csv_index(pPsv, "ADDRESS2");
        if (iaddress2 < 0) {
            non_fatal("Column 'ADDRESS2' is not found");
            rc = SXM_E_INVAL;
            break;
        }

        icity = sxm_csv_index(pPsv, "CITY");
        if (icity < 0) {
            non_fatal("Column 'CITY' is not found");
            rc = SXM_E_INVAL;
            break;
        }

        istate = sxm_csv_index(pPsv, "STATE");
        if (istate < 0) {
            non_fatal("Column 'STATE' is not found");
            rc = SXM_E_INVAL;
            break;
        }

        izip = sxm_csv_index(pPsv, "ZIP");
        if (izip < 0) {
            non_fatal("Column 'ZIP' is not found");
            rc = SXM_E_INVAL;
            break;
        }

        iphone = sxm_csv_index(pPsv, "PHONE");
        if (iphone < 0) {
            non_fatal("Column 'PHONE' is not found");
            rc = SXM_E_INVAL;
            break;
        }

        ilat1 = sxm_csv_index(pPsv, "LAT1");
        if (ilat1 < 0) {
            non_fatal("Column 'LAT1' is not found");
            rc = SXM_E_INVAL;
            break;
        }

        ilon1 = sxm_csv_index(pPsv, "LON1");
        if (ilon1 < 0) {
            non_fatal("Column 'LON1' is not found");
            rc = SXM_E_INVAL;
            break;
        }

        ilat2 = sxm_csv_index(pPsv, "LAT2");
        if (ilat2 < 0) {
            non_fatal("Column 'LAT2' is not found");
            rc = SXM_E_INVAL;
            break;
        }

        ilon2 = sxm_csv_index(pPsv, "LON2");
        if (ilon2 < 0) {
            non_fatal("Column 'LON2' is not found");
            rc = SXM_E_INVAL;
            break;
        }

        ihgt1 = sxm_csv_index(pPsv, "HGT1");
        if (ihgt1 < 0) {
            non_fatal("Column 'HGT1' is not found");
            rc = SXM_E_INVAL;
            break;
        }

        ihgt2 = sxm_csv_index(pPsv, "HGT2");
        if (ihgt2 < 0) {
            non_fatal("Column 'HGT2' is not found");
            rc = SXM_E_INVAL;
            break;
        }

        ispaces = sxm_csv_index(pPsv, "SPACES");
        if (ispaces < 0) {
            non_fatal("Column 'SPACES' is not found");
            rc = SXM_E_INVAL;
            break;
        }

        ioper = sxm_csv_index(pPsv, "OPERATION");
        if (ioper < 0) {
            non_fatal("Column 'OPERATION' is not found");
            rc = SXM_E_INVAL;
            break;
        }

        iprice = sxm_csv_index(pPsv, "PRICE");
        if (iprice < 0) {
            non_fatal("Column 'PRICE' is not found");
            rc = SXM_E_INVAL;
            break;
        }

        icomm = sxm_csv_index(pPsv, "COMM");
        if (icomm < 0) {
            non_fatal("Column 'COMM' is not found");
            rc = SXM_E_INVAL;
            break;
        }

        iamen1 = sxm_csv_index(pPsv, "AMEN1");
        if (iamen1 < 0) {
            non_fatal("Column 'AMEN1' is not found");
            rc = SXM_E_INVAL;
            break;
        }

        iamen2 = sxm_csv_index(pPsv, "AMEN2");
        if (iamen2 < 0) {
            non_fatal("Column 'AMEN2' is not found");
            rc = SXM_E_INVAL;
            break;
        }

        /* Initialize regions mask */
        memset(rmask, 0, sizeof(rmask));

        while (SXM_E_OK == rc) {

            int region, id, ver, hgt1, hgt2, spaces, price;
            //char *slat1, *slon1, *slat2, *slon2;
            fix flon1, flat1, flon2, flat2, dlon, dlat;
            char phone[SXM_GRID_PHONE_BYTELEN];
            const char *pText;

            rc = sxm_csv_next_line(pPsv);
            if (SXM_E_OK != rc)
            {
                if (SXM_E_NOENT == rc)
                {
                    rc = SXM_E_OK;
                }

                break;
            }

            /* Read and verify REGION */
            rc = sxm_csv_get_int_val(pPsv, (uint)iregion, &region);
            if (SXM_E_OK != rc)
            {
                non_fatal("Failed to get PREGION. Line: %u",
                          sxm_csv_current_line(pPsv));
                break;
            }

            if (!SXM_GRID_REGION_IS_VALID(region)) {
                non_fatal("Unsupported Region ID: %d "
                          "ignoring this record. Line: %u",
                          region, sxm_csv_current_line(pPsv));
                continue;
            }

            /* Read and verify PUID */
            rc = sxm_csv_get_int_val(pPsv, (uint)iid, &id);
            if (SXM_E_OK != rc)
            {
                non_fatal("Failed to get PUID. Line: %u",
                          sxm_csv_current_line(pPsv));
                break;
            }

            if (!IS_IN_RANGE(id, PARKING_MIN_STATION_ID, PARKING_MAX_STATION_ID)) {
                non_fatal("Unsupported station ID: %d region %d,"
                          " ignoring this record. Line: %u",
                          id, region, sxm_csv_current_line(pPsv));
                continue;
            }

            /* Skip this record if region has already been committed */
            if (BITP(rmask, region)) {
                non_fatal("Repeat Region %d, ignoring this record. "
                          "Line: %u", region, sxm_csv_current_line(pPsv));
                continue;
            }

            /* Read and verify VER */
            rc = sxm_csv_get_int_val(pPsv, (uint)iver, &ver);
            if (SXM_E_OK != rc)
            {
                non_fatal("Failed to get VER. Line: %u",
                          sxm_csv_current_line(pPsv));
                break;
            }

            if (ver < 0) {
                non_fatal("Unsupported VER: %d. Line: %u",
                          ver, sxm_csv_current_line(pPsv));
                continue;
            }

            if ((uint)ver > *pMaxVer) {
                *pMaxVer = (uint)ver;
            }

            if (region != prevRegion) {
                if (prevRegion >= 0) {
                    /* Save the region and set the saved region mask */
                    rc = grid_saveregion(pDbFile, pStream, prevRegion,
                                         pIndexBlock, pParkingIndex);
                    if (rc != SXM_E_OK) {
                        non_fatal("Failed to save region %d (rc=%d)",
                                  prevRegion, rc);
                        if (rc == SXM_E_NOMEM) {
                            /* stop processing but complete DB even 
                            if number of blocks is not enough */
                            rc = SXM_E_OK;
                        }
                        break;
                    }
                    BITS(rmask, prevRegion);
                }

                memset(&pParkingIndex[0], 0xFF,
                       sizeof(*pParkingIndex) * PARKING_MAX_STATIONS);
                pParkingIndex[0] = 0;
                prevRegion = region;
            }

            /* Read LON1 */
            rc = sxm_csv_get_fix_val(pPsv, (uint)ilon1, &flon1);
            if (SXM_E_OK != rc)
            {
                non_fatal("Failed to get LON1. Line: %u",
                          sxm_csv_current_line(pPsv));
                break;
            }

            /* Read LAT1 */
            rc = sxm_csv_get_fix_val(pPsv, (uint)ilat1, &flat1);
            if (SXM_E_OK != rc)
            {
                non_fatal("Failed to get LAT1. Line: %u",
                          sxm_csv_current_line(pPsv));
                break;
            }

            /* Read LON2 */
            rc = parking_read_optional_fix_val(pPsv, (uint)ilon2, &flon2);
            if (SXM_E_OK != rc)
            {
                non_fatal("Failed to get LON2. Line: %u",
                            sxm_csv_current_line(pPsv));
                break;
            }

            /* Read LAT2 */
            rc = parking_read_optional_fix_val(pPsv, (uint)ilat2, &flat2);
            if (SXM_E_OK != rc)
            {
                non_fatal("Failed to get LAT2. Line: %u",
                          sxm_csv_current_line(pPsv));
                break;
            }

            if ((0 == flon2) && (0 == flat2)) {
                dlon = 0;
                dlat = 0;
            }
            else {
                dlon = flon2 - flon1;
                dlat = flat2 - flat1;
            }

            /* Read and verify HGT1 */
            rc = parking_read_optional_int_val(pPsv, (uint)ihgt1, &hgt1);
            if (SXM_E_OK != rc)
            {
                non_fatal("Failed to get HGT1. Line: %u",
                          sxm_csv_current_line(pPsv));
                break;
            }
            if (hgt1 < 0) {
                non_fatal("Unsupported HGT1: %d. Line: %u",
                          hgt1, sxm_csv_current_line(pPsv));
                continue;
            }

            /* Read and verify HGT2 */
            rc = parking_read_optional_int_val(pPsv, (uint)ihgt2, &hgt2);
            if (SXM_E_OK != rc)
            {
                non_fatal("Failed to get HGT2. Line: %u",
                          sxm_csv_current_line(pPsv));
                break;
            }
            if (hgt2 < 0) {
                non_fatal("Unsupported HGT2: %d. Line: %u",
                          hgt2, sxm_csv_current_line(pPsv));
                continue;
            }

            /* Read and verify SPACES */
            rc = parking_read_optional_int_val(pPsv, (uint)ispaces, &spaces);
            if (SXM_E_OK != rc)
            {
                non_fatal("Failed to get SPACES. Line: %u",
                          sxm_csv_current_line(pPsv));
                break;
            }
            if (spaces < 0) {
                non_fatal("Unsupported SPACES: %d. Line: %u",
                          spaces, sxm_csv_current_line(pPsv));
                continue;
            }

            /* Read and verify PRICE */
            rc = parking_read_optional_int_val(pPsv, (uint)iprice, &price);
            if (SXM_E_OK != rc)
            {
                non_fatal("Failed to get PRICE. Line: %u",
                          sxm_csv_current_line(pPsv));
                break;
            }
            if (!IS_IN_RANGE(price, 0, PARKING_BASELINE_PRICE_VAL_MAX)) {
                non_fatal("Unsupported PRICE: %d. Line: %u",
                          price, sxm_csv_current_line(pPsv));
                continue;
            }

            pParkingIndex[id] = (uint)sxm_tstream_tell(pStream);
            sput4(pStream, flon1);
            sput4(pStream, flat1);
            sput4(pStream, dlon);
            sput4(pStream, dlat);
            sput2(pStream, (short)ver);
            sput2(pStream, (short)hgt1);
            sput2(pStream, (short)hgt2);
            sput2(pStream, (short)spaces);
            sput2(pStream, (short)price);
            pText = sxm_csv_str_val(pPsv, (uint)iamen1, NULL);
            sput2(pStream, (short)grid_encode_amen(pText));
            pText = sxm_csv_str_val(pPsv, (uint)iamen2, NULL);
            sput2(pStream, (short)grid_encode_amen(pText));
            /* In order to save space in t-file the phone number is converted
               into binary form where each digit is represented by 4 bits.
               Based on the protocol spec the format shall be (XXX)YYY-ZZZZ,
               thus, the whole phone is 40-bits or 5-bytes length */
            pText = sxm_csv_str_val(pPsv, (uint)iphone, NULL);
            sputb(pStream,
                  grid_encode_phone(&phone[0], sizeof(phone), pText),
                  sizeof(phone));

            pText = sxm_csv_str_val(pPsv, (uint)iname, NULL);
            if (strlen(pText) < SXM_PARKING_NAME_SIZE_MAX)
            {
                sputs(pStream, pText);
            }
            else
            {
                char truncatedName[SXM_PARKING_NAME_SIZE_MAX];
                memcpy(truncatedName, pText, SXM_PARKING_NAME_SIZE_MAX - 1);
                truncatedName[SXM_PARKING_NAME_SIZE_MAX - 1] = '\0';
                sputs(pStream, truncatedName);

                string_limit_warning("NAME", pText,
                    SXM_PARKING_NAME_SIZE_MAX,
                    truncatedName, sxm_csv_current_line(pPsv));
            }

            pText = sxm_csv_str_val(pPsv, (uint)iaddress1, NULL);
            if (strlen(pText) < SXM_PARKING_ADDR_SIZE_MAX)
            {
                sputs(pStream, pText);
            }
            else
            {
                char truncatedAddress[SXM_PARKING_ADDR_SIZE_MAX];
                memcpy(truncatedAddress, pText, SXM_PARKING_ADDR_SIZE_MAX - 1);
                truncatedAddress[SXM_PARKING_ADDR_SIZE_MAX - 1] = '\0';
                sputs(pStream, truncatedAddress);

                string_limit_warning("ADDRESS1", pText,
                    SXM_PARKING_ADDR_SIZE_MAX,
                    truncatedAddress, sxm_csv_current_line(pPsv));
            }

            pText = sxm_csv_str_val(pPsv, (uint)iaddress2, NULL);
            if (strlen(pText) < SXM_PARKING_ADDR_SIZE_MAX)
            {
                sputs(pStream, pText);
            }
            else
            {
                char truncatedAddress[SXM_PARKING_ADDR_SIZE_MAX];
                memcpy(truncatedAddress, pText, SXM_PARKING_ADDR_SIZE_MAX - 1);
                truncatedAddress[SXM_PARKING_ADDR_SIZE_MAX - 1] = '\0';
                sputs(pStream, truncatedAddress);

                string_limit_warning("ADDRESS2", pText,
                    SXM_PARKING_ADDR_SIZE_MAX,
                    truncatedAddress, sxm_csv_current_line(pPsv));
            }

            pText = sxm_csv_str_val(pPsv, (uint)icity, NULL);
            if (strlen(pText) < SXM_PARKING_CITY_SIZE_MAX)
            {
                sputs(pStream, pText);
            }
            else
            {
                char truncatedCity[SXM_PARKING_CITY_SIZE_MAX];
                memcpy(truncatedCity, pText, SXM_PARKING_CITY_SIZE_MAX - 1);
                truncatedCity[SXM_PARKING_CITY_SIZE_MAX - 1] = '\0';
                sputs(pStream, truncatedCity);

                string_limit_warning("CITY", pText,
                    SXM_PARKING_CITY_SIZE_MAX,
                    truncatedCity, sxm_csv_current_line(pPsv));
            }

            pText = sxm_csv_str_val(pPsv, (uint)istate, NULL);
            if (strlen(pText) < SXM_PARKING_STATE_SIZE_MAX)
            {
                sputs(pStream, pText);
            }
            else
            {
                char truncatedState[SXM_PARKING_STATE_SIZE_MAX];
                memcpy(truncatedState, pText, SXM_PARKING_STATE_SIZE_MAX - 1);
                truncatedState[SXM_PARKING_STATE_SIZE_MAX - 1] = '\0';
                sputs(pStream, truncatedState);

                string_limit_warning("STATE", pText,
                    SXM_PARKING_STATE_SIZE_MAX,
                    truncatedState, sxm_csv_current_line(pPsv));
            }

            pText = sxm_csv_str_val(pPsv, (uint)izip, NULL);
            if (strlen(pText) < SXM_PARKING_ZIP_SIZE_MAX)
            {
                sputs(pStream, pText);
            }
            else
            {
                char truncatedZip[SXM_PARKING_ZIP_SIZE_MAX];
                memcpy(truncatedZip, pText, SXM_PARKING_ZIP_SIZE_MAX - 1);
                truncatedZip[SXM_PARKING_ZIP_SIZE_MAX - 1] = '\0';
                sputs(pStream, truncatedZip);

                string_limit_warning("ZIP", pText,
                    SXM_PARKING_ZIP_SIZE_MAX,
                    truncatedZip, sxm_csv_current_line(pPsv));
            }

            pText = sxm_csv_str_val(pPsv, (uint)ioper, NULL);
            if (SXM_E_OK != verifyOperationFormat(pText)) {
                non_fatal("Inappropriate OPERATION format: '%s'. "
                    "Line: %u", pText, sxm_csv_current_line(pPsv));
            }
            sputs(pStream, pText);

            pText = sxm_csv_str_val(pPsv, (uint)icomm, NULL);
            if (strlen(pText) < SXM_PARKING_COMMENT_SIZE_MAX)
            {
                sputs(pStream, pText);
            }
            else
            {
                char truncatedComment[SXM_PARKING_COMMENT_SIZE_MAX];
                memcpy(truncatedComment, pText, SXM_PARKING_COMMENT_SIZE_MAX - 1);
                truncatedComment[SXM_PARKING_COMMENT_SIZE_MAX - 1] = '\0';
                sputs(pStream, truncatedComment);

                string_limit_warning("COMM", pText,
                    SXM_PARKING_COMMENT_SIZE_MAX,
                    truncatedComment, sxm_csv_current_line(pPsv));
            }

            salign(pStream, sizeof(uint));

            /* Do check that the station coordinates do belong to the region */
            if (!grid_check_location((ushort)region, flat1, flon1)) {
                non_fatal("Parking (primary) %u outside the region %d"
                    " (lat: %s [%d], lon: %s [%d]). Line: %u",
                    id, region,
                    sxm_csv_str_val(pPsv, (uint)ilat1, NULL), (int)flat1,
                    sxm_csv_str_val(pPsv, (uint)ilon1, NULL), (int)flon1,
                    sxm_csv_current_line(pPsv));
            }
            if (flon2 && flat2 && !grid_check_location((ushort)region, flat2, flon2)) {
                non_fatal("Parking (second) %d outside the region %d "
                    "(lat: %s [%d], lon: %s [%d]). Line: %u",
                    id, region,
                    sxm_csv_str_val(pPsv, (uint)ilat2, NULL), (int)flat2,
                    sxm_csv_str_val(pPsv, (uint)ilon2, NULL), (int)flon2,
                    sxm_csv_current_line(pPsv));
            }

            /* Index[0] is our "max index"/"max # of records" */
            if ((uint)id > pParkingIndex[0]) {
                pParkingIndex[0] = (uint)id;
            }
        }

        /* Save the last region */
        if ((rc == SXM_E_OK) && (prevRegion >= 0)) {
            rc = grid_saveregion(pDbFile, pStream, prevRegion, pIndexBlock, pParkingIndex);
            if (rc != SXM_E_OK) {
                non_fatal("Failed to save region %d (rc=%d)", prevRegion, rc);
                if (rc == SXM_E_NOMEM) {
                    /* stop processing but complete DB even if number of blocks
                    is not enough */
                    rc = SXM_E_OK;
                }
                break;
            }
        }

    }} /* switch (0) { default: { */

    if (NULL != pStream) {
        sxm_tstream_destroy(pStream);
    }

    if (NULL != pParkingIndex) {
        sxe_free(pParkingIndex);
    }

    return rc;
}

typedef struct
{
    SXMTFile *pDbFile;
    uint maxVersion;
} ParkingSdkFilesInputData;

static int parking_files_input_callback(const SXMFilesInputProps *pProps,
                                        const SXMFilesInputRecord *pRecord,
                                        void *pUserData)
{
    int rc = SXM_E_OK;
    SxmCsvParser *pPsvParser = NULL;
    ParkingSdkFilesInputData *pInputData = (ParkingSdkFilesInputData*)pUserData;

    UNUSED_VAR(pProps);

    rc = create_buffer_psv_parser(pRecord, '|',
                                  PARKING_PSV_LINE_SIZE_MAX,
                                  &pPsvParser);
    if (SXM_E_OK == rc) {

        rc = parking_build_locations(pInputData->pDbFile, pPsvParser,
                                     &pInputData->maxVersion);

        sxm_csv_delete(pPsvParser);
    }

    return rc;
}

int parking_build(const SXMFilesInput *pFilesInput, int version) {

    int rc = SXM_E_OK;
    ParkingSdkFilesInputData filesInputData;

    memset(&filesInputData, 0, sizeof(ParkingSdkFilesInputData));

    switch (0) { default: {

        SXMTFileStat st;

        filesInputData.pDbFile =
            sxm_tfile_create(PARKING_SERVICE_DB_FILE_TYPE,
                             PARKING_SERVICE_NAME,
                             PARKING_SERVICE_DB_NAME,
                             PARKING_SERVICE_NAME,
                             SXM_GRID_PACK_VERSION(PARKING_TFILE_SCHEMA_VERSION),
                             SXM_TFILE_VERSION,
                             PARKING_ROOT_BLOCK_SIZE,
                             PARKING_TFILE_BLOCK_COUNT,
                             PARKING_BLOCK_SIZE, &rc);
        if (rc != SXM_E_OK) {
            non_fatal("Failed to create DB file (rc=%d)", rc);
            break;
        }

        rc = sxm_tfile_start(filesInputData.pDbFile);
        if (rc != SXM_E_OK) {
            non_fatal("Failed to start DB transaction (rc=%d)", rc);
            break;
        }

        /* Enumarate psv files */
        rc = sxm_files_input_enumerate(pFilesInput, "psv",
                                       parking_files_input_callback,
                                       &filesInputData);
        if (SXM_E_OK != rc) {
            non_fatal("Input files enumeration failed (rc=%d)", rc);
            break;
        }

        if ((version != -1) && ((uint)version > filesInputData.maxVersion)) {
            filesInputData.maxVersion = (uint)version;
        }

        rc = sxm_tfile_update_version(filesInputData.pDbFile,
                                      filesInputData.maxVersion);
        if (rc != SXM_E_OK) {
            non_fatal("Failed to update DB version (rc=%d)", rc);
            break;
        }

        rc = sxm_tfile_commit(filesInputData.pDbFile);
        if (rc != SXM_E_OK) {
            non_fatal("Failed to commit DB file (rc=%d)", rc);
            break;
        }

        rc = sxm_tfile_stat(filesInputData.pDbFile, &st);
        if (rc != SXM_E_OK) {
            non_fatal("Failed to gather stat from the t-file (rc=%d)", rc);
            break;
        }

        printf("Versions\n"
            "  Baseline Overall: %u\n"
            "  Schema          : %u/%u\n"
            "Statistics: %u block(s) by %u byte(s), used %u block(s) [%u%%]\n",
            st.dversion, SXM_GRID_EXTRACT_SERVICE_SCHEMA(st.dschema),
            SXM_GRID_EXTRACT_OWN_SCHEMA(st.dschema), st.fsize, st.ubsize, st.usize,
            (st.usize * 100) / st.fsize);

    }} /* switch (0) { default: { */

    if (NULL != filesInputData.pDbFile) {
        sxm_tfile_close(filesInputData.pDbFile);
    }

    return rc;
}

/*******************************************************************************
*
* Function    :  parking_ver
*
* Description :  Parses input file names to get version
*
* Parameters  :
*          1  :  pSource = pointer to file name with path
*          2  :  isArchive = TRUE if passed file name is archive
*          3  :  pUserData = pointer to version variable
*
* Returns     :  Nothing
*
********************************************************************************/
void parking_ver(const char *pSource, BOOL isArchive, void *pUserData) {
    const char *start;
    int *pVersion = (int *)pUserData;

    UNUSED_VAR(isArchive);

    start = strpbrk(pSource, "0123456789");
    if (NULL != start) {
        int ver = atoi(start);

        if (IS_IN_RANGE(ver, PARKING_VERSION_MIN, PARKING_VERSION_MAX)) {
            *pVersion = ver;
        }
    }

    return;
}

#ifdef SDKFILES_STANDALONE_BUILD

SXMTQL_TABLES()
    SXMTQL_TABLE(parkings)
        SXMTQL_COLUMN(region, INT, Parking, region)
        SXMTQL_COLUMN(id, INT, Parking, id)
        SXMTQL_COLUMN(ver, INT, Parking, ver)
        SXMTQL_COLUMN(name, STR, Parking, name)
        SXMTQL_COLUMN(address1, STR, Parking, address1)
        SXMTQL_COLUMN(address2, STR, Parking, address2)
        SXMTQL_COLUMN(city, STR, Parking, city)
        SXMTQL_COLUMN(state, STR, Parking, state)
        SXMTQL_COLUMN(zip, STR, Parking, zip)
        SXMTQL_COLUMN(phone, STR, Parking, phone)
        SXMTQL_COLUMN(lat1, REAL, Parking, lat1)
        SXMTQL_COLUMN(lon1, REAL, Parking, lon1)
        SXMTQL_COLUMN(lat2, REAL, Parking, lat2)
        SXMTQL_COLUMN(lon2, REAL, Parking, lon2)
        SXMTQL_COLUMN(h1, INT, Parking, h1)
        SXMTQL_COLUMN(h2, INT, Parking, h2)
        SXMTQL_COLUMN(spaces, INT, Parking, spaces)
        SXMTQL_COLUMN(operation, STR, Parking, operation)
        SXMTQL_COLUMN(price, INT, Parking, price)
        SXMTQL_COLUMN(comm, STR, Parking, comm)
        SXMTQL_COLUMN(amen1, INT, Parking, amen1)
        SXMTQL_COLUMN(amen2, INT, Parking, amen2)
    SXMTQL_TABLE_END()
SXMTQL_TABLES_END()

void parking_check_tfile(const char *query) {
    SXMTFile *dbfile = NULL;
    int rc = SXM_E_ERROR;
    const uint *dbr = NULL;
    SXMTqlStatement tql = NULL;

    switch (0) { default: {
        const ushort *ib;
        Parking parking;
        uint dbschema;
        if (query) {
            if (SXM_E_OK != sxm_tql_statement(&tql, SXMTQL_DEFAULT_INTERFACE,
                                              query, SXMTQL_SERVICE_TABLES)) {
                non_fatal("Failed to create statement for TQL query: %s", query);
                break;
            }
            else if (!tql) {
                break;
            }
        }

        dbfile = sxm_tfile_open('W', PARKING_SERVICE_NAME,
                                PARKING_SERVICE_DB_NAME, "rb",
                                PARKING_SERVICE_NAME, &dbschema, &rc);
        if (rc != SXM_E_OK) {
            non_fatal("Failed to open DB file (rc=%d)", rc);
            break;
        }

        if (dbschema != SXM_GRID_PACK_VERSION(PARKING_TFILE_SCHEMA_VERSION)) {
            non_fatal("Unsupported schema version %u, Grid %u\n",
                SXM_GRID_EXTRACT_SERVICE_SCHEMA(dbschema),
                SXM_GRID_EXTRACT_OWN_SCHEMA(dbschema));
            break;
        }
        ib = (ushort *)sxm_tfile_root(dbfile);
        if (!ib) {
            non_fatal("Failed to access memory for root!");
            break;
        }
        memset(&parking, 0, sizeof(parking));
        for  (parking.region = 0; parking.region <= SXM_GRID_REGION_ID_MAX; ++parking.region, ib += 2) {
            if  (ib[1] > 0) {
                if (!tql) {
                    printf("      Region %6d:     Offset %6d     Count %6d\n",
                           parking.region, ib[0], ib[1]);
                }

                // Load region's data
                dbr = (uint*)sxm_tfile_alloc_and_read(dbfile, ib[0], ib[1], NULL, &rc);
                if (!dbr) {
                    non_fatal("Failed to load %d block(s) from %d (rc=%d)\n",
                        (int)ib[1], (int)ib[0], rc);
                    break;
                }

                for (parking.id = PARKING_MIN_STATION_ID; parking.id <= dbr[0]; ++parking.id) {
                    if (dbr[parking.id] != SXM_DB_INVALID_ENTRY) {
                        const char *rec;
                        fix flon1, flat1, flon2, flat2, dlon, dlat;
                        char sphone[SXM_GRID_PHONE_LENGTH];
                        rec = (const char *)dbr + (dbr[0] + 1) * 2 * sizeof(ushort) + dbr[parking.id];
                        if (!tql) {
                            printf("   %4d: ", parking.id);
                        }

                        flon1 = *(int *)(void*)rec;
                        flat1 = *(int *)(void*)(rec+4);
                        dlon = *(int *)(void*)(rec+8);
                        dlat = *(int *)(void*)(rec+12);
                        if (!dlon && !dlat) {
                            flon2 = 0;
                            flat2 = 0;
                        }
                        else {
                            flon2 = flon1 + dlon;
                            flat2 = flat1 + dlat;
                        }
                        parking.lon1 = fix2float(flon1);
                        parking.lat1 = fix2float(flat1);
                        parking.lon2 = fix2float(flon2);
                        parking.lat2 = fix2float(flat2);
                        parking.ver = *(ushort *)(void*)(rec+16);
                        parking.h1 = *(ushort *)(void*)(rec+18);
                        parking.h2 = *(ushort *)(void*)(rec+20);
                        parking.spaces = *(short *)(void*)(rec+22);
                        parking.price = *(short *)(void*)(rec+24);
                        parking.amen1 = *(ushort *)(void*)(rec+26);
                        parking.amen2 = *(ushort *)(void*)(rec+28);
                        rec += 30;
                        parking.phone = grid_decode_phone(&sphone[0], sizeof(sphone), rec);
                        rec += SXM_GRID_PHONE_BYTELEN;
                        parking.name = (const char*)rec;
                        rec += strlen((const char*)rec)+1;
                        parking.address1 = (const char*)rec;
                        rec += strlen((const char*)rec)+1;
                        parking.address2 = (const char*)rec;
                        rec += strlen((const char*)rec)+1;
                        parking.city = (const char*)rec;
                        rec += strlen((const char*)rec)+1;
                        parking.state = (const char*)rec;
                        rec += strlen((const char*)rec)+1;
                        parking.zip = (const char*)rec;
                        rec += strlen((const char*)rec)+1;
                        parking.operation = (const char*)rec;
                        rec += strlen((const char*)rec)+1;
                        parking.comm = (const char*)rec;

                        if (tql) {
                            rc = sxm_tql_evaluate(tql, &parking);
                            if ((rc != SXM_E_OK) && (rc != SXM_E_NOENT)) {
                                non_fatal("Failed to evaluate TQL statement (%d)", rc);
                                break;
                            }
                        }
                        else {
                            printf("%f,%f h %d %f,%f h %d ver %2d. amen %04x.%04x %d spaces price %d\n",
                                parking.lon1, parking.lat1, parking.h1, 
                                parking.lon2, parking.lat2, parking.h2, 
                                parking.ver, parking.amen2, parking.amen1, parking.spaces, parking.price);
                            printf("            N: '%s'\n", parking.name);
                            printf("            a1: '%s' a2: '%s'\n", parking.address1, parking.address2);
                            printf("            c: '%s' s: '%s' z: '%s' p: '%s'\n h: '%s' C: '%s'\n", 
                                parking.city, parking.state, parking.zip, parking.phone, parking.operation, parking.comm);
                        }
                    }
                }

                sxe_free((void*)dbr);
                dbr = NULL;
            }
        }
    }}

    if (dbr) {
        sxe_free((void*)dbr);
    }
    if (dbfile) {
        sxm_tfile_close(dbfile);
    }
    if (tql) {
        sxm_tql_statement_destroy(tql);
    }
}

void parking_check_cfile(void) {

    ParkingSave *cfiledata = NULL;
    FILE *cfile;

    switch (0) { default: {
        int rc;
        uint fileschema;
        SXMCFileValidationResult val;
        uint idx, jdx;
        uint sec;
        // Try to load the file
        cfile = sxm_cfile_open(PARKING_SERVICE_NAME, PARKING_CFILE_NAME);
        if (!cfile) {
            non_fatal(": failed to open CFile");
            break;
        }
        // load the cycle file
        rc = sxm_cfile_alloc_and_load(cfile, sizeof(*cfiledata), (ptr*)&cfiledata);
        if ((sizeof(SXMSector) * (uint)rc) != sizeof(*cfiledata)) {
            non_fatal(": failed read %d bytes, rc = %d\n", sizeof(*cfiledata), rc);
            break;
        }
        // Check version and format
        rc = sxm_cfile_check_format(&cfiledata->g, PARKING_SERVICE_NAME, &fileschema);
        if (rc != SXM_E_OK) {
            non_fatal(": failed to check format/schema, rc=%d\n", rc);
            break;
        }
        printf("Schema version %u\n", fileschema);
        if (PARKING_CFILE_SCHEMA_VERSION != fileschema) {
            non_fatal("Unsupported schema version %u\n", fileschema);
            break;
        }
        // Validate the file and print data for valid section(s) only
        val = sxm_cfile_validate(&cfiledata->g);
        printf("Overall validation result is %d\n", val);
        // Do section by sections validation and output
        for (sec = PARKING_SECTION_FIRST; sec < PARKING_SECTIONS_COUNT; ++sec) {
            // Do section check
            if (!sxm_cfile_check(&cfiledata->g, sec, FALSE)) {
                printf(": sec #%u is corrupted via light check\n", sec);
            }
            else if (!sxm_cfile_check(&cfiledata->g, sec, TRUE)) {
                printf(": sec #%u is corrupted via deep check\n", sec);
            }
            else {
                const SXMCFileSection *psec = &cfiledata->g.sections[sec];
                grid_cfile_sec_stat(&cfiledata->g, (int)sec);
                switch (sec) {
                    case PARKING_BASE: {
                        const SXMGridBaseLine *b = &cfiledata->baseb.baseb[0];
                        printf("BASEB\n");
                        for (idx = 0; idx < ARRAY_SIZE(cfiledata->baseb.baseb); ++idx, ++b) {
                            printf("    [%1d]:  region %d\n", idx, b->region);
                            for (jdx = 0; jdx < ARRAY_SIZE(b->safe); ++jdx) {
                                printf("        [%2d]:  pktix %d crc %08x pl %d\n", jdx,
                                    b->safe[jdx].pktix, b->safe[jdx].crc, b->safe[jdx].pl);
                            }
                        }
                    }
                    break;
                    case PARKING_HEAP: {
                        printf("HEAP\n");
                        for (idx = 0; idx < psec->count; ++idx) {
                            if (idx) {
                                if (!(idx % (sizeof(uint) * 8))) {
                                    printf ("\n");
                                }
                                else if (!(idx % 4)) {
                                    printf(" ");
                                }
                            }
                            // - means not used, * - used one
                            printf("%c", BITP(cfiledata->g.map, idx + psec->heap_map_off) ? '-' : '*');
                        }
                        printf("\n");
                    }
                    break;
                    case PARKING_PPOI: {
                        const ParkingFavPersData *ppoi = &cfiledata->fav_loc.fav_loc[0];
                        const ParkingFavLocation *ppoi_loc; 
                        printf("PPOI storage has %u item(s)\n", ppoi->count);
                        for (ppoi_loc = &ppoi->locations[0], idx = 0; idx < ppoi->count; ++idx, ++ppoi_loc) {
                            printf("    [%4d]: ID:%08x, flags:%08x\n",
                                idx, ppoi_loc->ppoi_spec.ppoi_id, ppoi_loc->ppoi_spec.type_flags);
                        }
                    }
                    break;
                }
            }
        }
    }}

    if (cfiledata) {
        sxe_free(cfiledata);
    }
    if (cfile) {
        sxm_cfile_close(cfile, NULL);
    }
}

#endif
