/************************************************************************
 *                                                                      *
 *            CFILE.C                                                   *
 *            =======                                                   *
 *                                                                      *
 *  Copyright 2013 Sirius XM Radio, Inc.                                *
 *  All Rights Reserved.                                                *
 *  Licensed Materials - Property of Sirius XM Radio, Inc.              *
 *                                                                      *
 *    Helper Routines for verifying cycle files                         *
 *                                                                      *
 ************************************************************************/

#include <stddef.h>
#include "sdkfiles.h"
#include <util/sxm_cfile.h>

STATIC_ASSERT(!(sizeof(SXMCFileRootBlock) % sizeof(SXMSector)),
              SXMCFileRootBlock_must_be_divided_by_sizeof_SXMSector);

static FILE *dbfile;
static SXMCFileRootBlock root;
static char tbuff[64];

#ifdef _WIN32
    #define snprintf _snprintf
#endif

#define  DAYS_CYCLE (366 + 365*3)
#define  SXM_SXE_TIME_TO_UNIX_OFFSET (1325376000)

static uint dsf[] = {0, 366, 366+365, 366+365+365};

// 31 28 31 30 31 30 31 31 30 31 30 31

static uint msf[] = {
    0, 31, 59, 90,
    120, 151, 181, 212,
    243, 273, 304, 334
};

char *unix_time_decode(uint now) {
    return time_decode((now - SXM_SXE_TIME_TO_UNIX_OFFSET) / 60);
}

char *time_decode(uint now)
{
    uint tmin = now % MINUTES_IN_HOUR;
    uint thour = (now / MINUTES_IN_HOUR) % HOURS_IN_DAY;
    uint tday = now / MINUTES_IN_DAY;
    uint lc = tday / DAYS_CYCLE;
    uint rest = tday % DAYS_CYCLE;
    int yy = 0;
    int mm = 0;

    while (yy < 3)
    {
        if (rest >= dsf[yy + 1])
        {
            yy++;
        }
        else
        {
            break;
        }
    }
    rest -= dsf[yy];

    while (mm < 11)
    {
        if (rest >= (msf[mm + 1] + (yy == 0 && mm > 0)))
        {
            mm++;
        }
        else
        {
            break;
        }
    }

    rest -= (msf[mm] + (yy == 0 && mm > 1));

    snprintf(tbuff, sizeof(tbuff), "%04u-%02u-%02u %02u:%02u",
        2012 + lc * 4 + yy, mm + 1, rest + 1, thour, tmin);
    return tbuff;
}

uint *cfile_user(void) {
    return &root.u[0];
}

uint cfile_count(uint i) {
    return (i < ARRAY_SIZE(root.sections)) ? root.sections[i].count : 0U;
}

int cfile_version(uint i) {
    return (i < ARRAY_SIZE(root.sections)) ? root.sections[i].version : 0;
}

ushort cfile_length(uint i) {

    if (i < ARRAY_SIZE(root.sections)) {
        return root.sections[i].length;
    }

    return 0U;
}

uint cfile_timestamp(uint i) {
    return (i < ARRAY_SIZE(root.sections)) ? root.sections[i].ts : 0U;
}

byte *cfile_load(uint i) {
    SXMSector *ret;
    uint crc;
    const SXMCFileSection *sec;

    if (i >= ARRAY_SIZE(root.sections)) {
        fatal("cfile:: incorrect section id %d", i);
        return NULL;
    }

    sec = &root.sections[i];
    if ((ret = (SXMSector*)sxe_calloc(sec->length, sizeof(SXMSector))) == NULL) {
        fatal("cfile:: Out of Memory");
        return NULL;
    }

    fseek(dbfile, (long)(sec->off * sizeof(SXMSector)), SEEK_SET);
    if (fread(ret, sizeof(SXMSector), sec->length, dbfile) != sec->length ) {
        sxe_free(ret);
        fatal("cfile:: Read Length error");
        return NULL;
    }
    
    if (sec->mask & SXM_CFILE_MASK_HEAP) {
        uint idx;
        crc = 0xFFFFFFFFU;
        for (idx = 0; idx < sec->count; ++idx) {
            if (!BITP(root.map, idx + sec->heap_map_off)) {
                const uint item_len = sec->length / sec->count;
                crc = doCRC_part(crc, &ret[idx * item_len],
                                 (uint)sizeof(SXMSector) * item_len);
            }
        }
        crc ^= 0xFFFFFFFFU;
    }
    else {
        crc = doCRC(ret, (uint)sizeof(SXMSector) * sec->length);
    }
    if  (crc != sec->crc) {
        sxe_free(ret);
        fatal("cfile:: CRC error on section");
        return NULL;
    }

    return (byte*)ret;
}

int cfile_open(const char *module, const char *file, const char *format) {
    uint crc;
    int i;

    dbfile = sxm_cfile_open(module, file);
    if (NULL == dbfile) {
        fatal("Failed to open %s\n", file);
        return SXM_E_ERROR;
    }

    if (fread(&root, sizeof(SXMCFileRootBlock), 1, dbfile) != 1) {
        fclose(dbfile);
        fatal("%s::Failed to read root block\n", file);
        return SXM_E_ERROR;
    }

    if ((root.magic != SXM_CFILE_MAGIC) ||
        (root.cver != SXM_CFILE_VERSION))
    {
        fclose(dbfile);
        fatal("%s::Unsupported file format magic=%x, tver=%u\n",
            file, root.magic, root.cver);
        return SXM_E_ERROR;
    }

    if (strncmp(root.format, format, sizeof(root.format)))
    {
        char f1[SXM_CFILE_FORMAT_LEN + 1];
        char f2[SXM_CFILE_FORMAT_LEN + 1];
        memset(f1, 0, sizeof(f1));
        memset(f2, 0, sizeof(f2));
        strncpy(f1, format, SXM_CFILE_FORMAT_LEN);
        strncpy(f2, root.format, SXM_CFILE_FORMAT_LEN);

        fclose(dbfile);
        fatal("%s::Unsupported file format format=%s vs %s\n",
            __FUNCTION__, f1, f2);
        return SXM_E_ERROR;
    }

    crc = doCRC(&root, offsetof(SXMCFileRootBlock, crc));
    if  (crc != root.crc) {
        fclose(dbfile);
        fatal("%s::Root Block is corrupt\n", file);
        return SXM_E_ERROR;
    }

    printf("\nCFile Root Block"
           "\n================\n");
    printf("Schema:    %u\n", root.sver);
    printf("Root CRC:  0x%X\n", root.crc);
    printf("Timestamp: %d (%s)\n", root.ts, time_decode(root.ts));

    // Print header
    printf("\nSECTIONS:\n");
    printFooter(76);
    printColumn(8, "INDEX");
    printColumn(13, "CRC");
    printColumn(19, "TIMESTAMP");
    printColumn(9, "OFFSET");
    printColumn(9, "LENGTH");
    printColumn(10, "VERSION");
    printColumn(8, "COUNT");
    printf("\n"); printFooter(76);

    for (i = 0; i < 8; i++)
    {
        if (root.sections[i].length)
        {
            // Print values
            printColumn(8, "[%d]", i);
            printColumn(13, "0x%X", root.sections[i].crc);
            printColumn(19, "%s", time_decode(root.sections[i].ts));
            printColumn(9, "%u", root.sections[i].off);
            printColumn(9, "%u", root.sections[i].length);
            printColumn(10, "%u", root.sections[i].version);
            printColumn(8, "%u", root.sections[i].count);
            // Print footer
            printf("\n"); printFooter(76);
        }
    }

    return SXM_E_OK;
}

uint cfile_schema(void)
{
    return root.sver;
}

void cfile_close(void)
{
    if (dbfile)
    {
        fclose(dbfile);
        dbfile = NULL;
    }
}
