/******************************************************************************/
/*                    Copyright (c) Sirius XM Radio, Inc.                     */
/*                            All Rights Reserved                             */
/*         Licensed Materials - Property of Sirius XM Radio, Inc.             */
/******************************************************************************/
/*******************************************************************************
 *
 * DESCRIPTION
 *
 *  This module includes the various STI APIs. The STI is used as a common
 *  interface to all supported protocols. It establishes a Framework for
 *  easily creating various protocol drivers by automatically creating and
 *  servicing any required OS tasks & resources needed. All the protocol design
 *  needs to supply are the interface functions.
 *
 ******************************************************************************/

#include <stdio.h>
#include <stdarg.h>
#include <string.h>

#include "standard.h"
#include "osal.h"

#include "sti_api.h"
#include "sti_protocol.h"

#include "sti_framework.h"
#include "_sti_framework.h"


/*****************************************************************************
*
*   STI_bInit
*
*****************************************************************************/
BOOLEAN STI_bInit( void )
{
    BOOLEAN bSuccess = FALSE;
    OSAL_RETURN_CODE_ENUM eReturnCode;

    // Create a mutex for a list of protocol instances installed
    eReturnCode = OSAL.eSemCreate(
        &gsFramework.hProtocolListMutex,
        "STI-Framework:ProtocolListMutex", 0,
        1, OSAL_SEM_OPTION_NONE);
    if(eReturnCode == OSAL_SUCCESS)
    {
        // Create a protocol based linked list based on device
        // handles and protocol specific information.
        eReturnCode = OSAL.eLinkedListCreate(
            &gsFramework.hProtocols,
            "STI-Framework:ProtocolList",
            n16CompareProtocolDevices,
            (OSAL_LL_OPTION_USE_PRE_ALLOCATED_ELEMENTS |
                OSAL_LL_OPTION_LINEAR | OSAL_LL_OPTION_UNIQUE));
        if(eReturnCode == OSAL_SUCCESS)
        {
            // STI Framework is initialized
            bSuccess = TRUE;
        }
        else
        {
            // Error!
            printf("STI Error! Cannot create protocol framework "
                "linked list.\n");
        }

        // Unlock the list, regardless
        OSAL.eSemGive(gsFramework.hProtocolListMutex);
    }
    else
    {
        // Error!
        printf("STI Error! Cannot create protocol framework "
            "linked list mutex.\n");
    }

    // Check if anything didn't go so well.
    if(bSuccess == FALSE)
    {
        // Error!
        STI_vUninit();
    }

    return bSuccess;
}

/*****************************************************************************
*
*   STI_vUninit
*
*****************************************************************************/
void STI_vUninit( void )
{
    OSAL_RETURN_CODE_ENUM eReturnCode;

    // Take control of the protocol list
    eReturnCode = OSAL.eSemTake(gsFramework.hProtocolListMutex,
        OSAL_OBJ_TIMEOUT_INFINITE);
    if(eReturnCode == OSAL_SUCCESS)
    {
        // Destroy the protocol's linked list itself (if it exists)
        if(gsFramework.hProtocols != OSAL_INVALID_OBJECT_HDL)
        {
            OSAL_RETURN_CODE_ENUM eReturnCode;

            // Remove all entries from the list and destroy each content entry.
            // at this point there actually shouldn't be any.

            // Since these are pre-allocated LL entries, we can't use eRemoveAll
            // and free the entry in the RemoveAll callback because that would
            // end up freeing the LL portion of the memory and then eRemoveAll
            // gets confused. So do it the old fashioned way.
            OSAL_LINKED_LIST_ENTRY hEntry;
            STI_PROTOCOL_ENTRY_STRUCT *psProtocolEntry;

            do
            {
                hEntry = OSAL.hLinkedListFirst(
                    gsFramework.hProtocols,
                    (void**)&psProtocolEntry);
                if (hEntry != OSAL_INVALID_LINKED_LIST_ENTRY)
                {
                    // Remove this protocol from our list.
                    eReturnCode = OSAL.eLinkedListRemove(hEntry);
                    if(eReturnCode == OSAL_SUCCESS)
                    {
                        // Release protocol
                        vProtocolRelease(psProtocolEntry);
                        psProtocolEntry = NULL;
                    }
                }

            } while (hEntry != OSAL_INVALID_LINKED_LIST_ENTRY);

            // Delete the list
            eReturnCode = OSAL.eLinkedListDelete(gsFramework.hProtocols);
            if(eReturnCode == OSAL_SUCCESS)
            {
                // Invalidate list
                gsFramework.hProtocols = OSAL_INVALID_OBJECT_HDL;
            }
        }

        if (gsFramework.hProtocols == OSAL_INVALID_OBJECT_HDL)
        {
            // Delete the mutex
            eReturnCode = OSAL.eSemDelete(
                gsFramework.hProtocolListMutex);
            if(eReturnCode == OSAL_SUCCESS)
            {
                gsFramework.hProtocolListMutex =
                    OSAL_INVALID_OBJECT_HDL;
            }
        }
        
        if (gsFramework.hProtocolListMutex != OSAL_INVALID_OBJECT_HDL) 
        {
            // Return it if we still have it.
            OSAL.eSemGive(gsFramework.hProtocolListMutex);
        }
    }

    return;
}

/*****************************************************************************
*
*   STI_hConnect
*
*   This API is used to create an STI connection for the purpose of
*   transmitting and receiving payload data. The connection requested
*   binds a specified protocol and physical device together to create
*   a bi-directional communication path. The connection created is
*   represented by an STI Handle which is returned from the API to the
*   caller. This handle must be used when allocating and sending payloads.
*   It is also provided via callback when payload data is received.
*
*   Inputs:
*       pcName - A pointer to an ASCII null terminated string which
*           identifies this connection. A copy of the provided name is
*           retained as part of the connection handle.
*       vCallback - A pointer to a callback function which is invoked
*           when a transport layer payload is received over this connection.
*           If this pointer is NULL no responses will be processed. If
*           something other than NULL is provided upon receipt of a payload
*           the callback function is invoked with the STI_HDL over which it
*           was received and, and the payload data. See Section 8.7 for a
*           description of this function's prototype requirements.
*       pvCallbackArg - A pointer to void which represents a data type
*           known to the caller and must be cast in the callback function.
*       pvConnectionArgument - A pointer to void which represents an
*           application specific object known to the caller and any other
*           entity wishing to access this object using the connection handle.
*           This connection specific argument can be obtained by anyone with
*           the STI handle by calling STI_pvGetConnectionArgument() and it
*           must be cast appropriately by the caller.
*       psProtocolIntf - A pointer to a STI_PROTOCOL_INTERFACE which represents
*           the Protocol Driver to use over this connection. This may be SLP,
*           SDTP or any other valid transport protocol driver. The following
*           variable arguments are dependent upon the protocol driver
*           interface specified.
*       pvDevice - A void * representing the device this protocol connects to.
*           This may be a FILE * if it connects to a physical device driver,
*           a connection handle to connect to another lower-level protocol, or
*           any other "protocol driver" specific handle.
*       ... - Variable arguments based on eProtocol.
*
*   Outputs:
*       A valid STI_HDL on success. Otherwise returns STI_HDL_INVALID.
*
*****************************************************************************/
STI_HDL STI_hConnect (
    const char *pacName,
    STI_PROTOCOL_CALLBACK vCallback,
    void *pvCallbackArg,
    void *pvConnectionArgument,
    const STI_PROTOCOL_INTERFACE *psProtocolIntf,
    void *pvDevice,
    ...
        )
{
    BOOLEAN bConnectionInitialized = FALSE;
    STI_HDL hSTI = STI_INVALID_HDL;
    STI_CONNECTION_ENTRY_STRUCT *psConnection; // STI_HDL
    char acName[OSAL_MAX_OBJECT_NAME_LENGTH_WITH_NULL];
    va_list tList; // variable arguments list

    // Check inputs. We need a name and an interface to use.
    if( (pacName == NULL) || (psProtocolIntf == NULL) )
    {
        // Error!
        return STI_INVALID_HDL;
    }

    // Construct a unique name for this object
    snprintf(acName, sizeof(acName), "STI:Cxn:%s", pacName);

    // Allocate a connection entry structure (zero initialized)
    psConnection =
        (STI_CONNECTION_ENTRY_STRUCT *)
            OSAL.pvLinkedListMemoryAllocate(
                acName,
                sizeof(STI_CONNECTION_ENTRY_STRUCT),  // STI_HDL
                TRUE );
    if(psConnection == NULL)
    {
        // Error!
        return STI_INVALID_HDL;
    }

    do
    {
        BOOLEAN bAddedProtocol;
        OSAL_RETURN_CODE_ENUM eReturnCode;

        // Populate connection details with inputs, and initialized values
        strncpy(&psConnection->acName[0], acName,
            sizeof(psConnection->acName)-1);
        psConnection->vCallback = vCallback;
        psConnection->pvCallbackArg = pvCallbackArg;
        psConnection->pvConnectionArgument = pvConnectionArgument;
        psConnection->hEntry = OSAL_INVALID_LINKED_LIST_ENTRY;
        psConnection->psProtocolEntry = NULL;
        psConnection->pvProtocolConnectionData = NULL;
        OSAL.bMemSet(&psConnection->sCallback, 0,
            sizeof(psConnection->sCallback)); // Populated as needed.

        // Install protocol instance for this connection.
        // Note: This only needs to be done once per unique physical interface.
        // If THIS protocol on THIS device has already been installed via
        // an earlier 'connect' call, we can just use that handle instead,
        // which is the one which will be returned.

        // Construct a unique name for this protocol entry
        snprintf(acName, sizeof(acName), "STI:Proto:%s",
            psConnection->acName);

        // Allocate a protocol entry for our master list of protocols.
        // We allocate one now, but may decide we already have one and
        // reuse that instead and get rid of this one.
        psConnection->psProtocolEntry = (STI_PROTOCOL_ENTRY_STRUCT *)
            OSAL.pvLinkedListMemoryAllocate(
                acName, sizeof(STI_PROTOCOL_ENTRY_STRUCT), TRUE);
        if(psConnection->psProtocolEntry == NULL)
        {
            // Error!
            printf("STI Error! Cannot allocate protocol list entry (%s).\n",
                acName);

            OSAL.vLinkedListMemoryFree(psConnection);
            break;
        }

        // alias the protocol connection pointer to the STI handle which
        // will eventually be given back to the caller.
        hSTI = (STI_HDL)psConnection;

        // Initialize protocol/device entry.
        // Most of this will get filled in later.
        psConnection->psProtocolEntry->hEntry =
            OSAL_INVALID_LINKED_LIST_ENTRY;
        psConnection->psProtocolEntry->pvDevice = pvDevice;
        psConnection->psProtocolEntry->psProtocolIntf =
            psProtocolIntf;
        psConnection->psProtocolEntry->hProtocol =
            STI_PROTOCOL_INVALID_HDL;
        psConnection->psProtocolEntry->hConnectionListMutex =
            OSAL_INVALID_OBJECT_HDL;
        psConnection->psProtocolEntry->hConnectionList =
            OSAL_INVALID_OBJECT_HDL;

        // Add this protocol based on the device specified. The result of this
        // call will either add it to the device or reuse an existing
        // installation based on previous connections. Either way multiple
        // protocols on the same device will not be installed.
        bAddedProtocol = bAddProtocol(
            &psConnection->psProtocolEntry, psConnection->vCallback,
            psConnection->pvCallbackArg);
        if(bAddedProtocol == FALSE)
        {
            // Error!
            printf("STI Error! Cannot add this protocol (%s) to this device,"
                " it may already have one installed.\n",
                psConnection->psProtocolEntry->psProtocolIntf->pacName);

            // Protocol entry memory will be released as part of 
            // the STI disconnect.
            break;
        }

        // At this point we have either installed(added) a protocol
        // or referenced an existing installed(added) protocol on that
        // device (psConnection->psProtocolEntry).

        // Initialize this connection

        // Check if protocol connection initialization method exists
        if(psProtocolIntf->sConnection._pvInit !=
            (STI_PROTOCOL_CONNECTION_INIT_HANDLER_PROTOTYPE)NULL)
        {
            // Run protocol connection initialization

            // grab variable arguments (pop)
            va_start(tList, pvDevice);

            // Initialize protocol specific connection data
            psConnection->pvProtocolConnectionData =
                psProtocolIntf->sConnection._pvInit(
                    &psConnection->acName[0],
                    psConnection->psProtocolEntry->hProtocol,
                    hSTI,
                    &tList
                        );

            // restore stack (push)
            va_end(tList);
        }

        // If we can take the connection semaphore, we got it
        eReturnCode = OSAL.eSemTake(
            psConnection->psProtocolEntry->hConnectionListMutex,
            OSAL_OBJ_TIMEOUT_INFINITE);
        if(eReturnCode == OSAL_SUCCESS)
        {
            // Add this connection to our list of connections
            eReturnCode = OSAL.eLinkedListAdd(
                psConnection->psProtocolEntry->hConnectionList,
                &psConnection->hEntry, psConnection);
            if(eReturnCode == OSAL_ERROR_LIST_ITEM_NOT_UNIQUE)
            {
                // Error!

                // Woah! Hang on, this is not a unique connection, so
                // there is no way to distinguish them, so sorry after all
                // that we cannot add this connection.

                printf("STI Error! Cannot make this connection, it already "
                    "exists (%s).\n", acName);

                // Note: It is not possible to have a non-unique connection
                // item on the connection list if we are using
                // a connection list which we just created via bAddProtocol().
            }
            else if(eReturnCode == OSAL_SUCCESS)
            {
                // Added the connection.
                // Connection is initialized.
                bConnectionInitialized = TRUE;
            }
            else
            {
                // Error!
                printf("STI Error! Cannot add connection into list (%s).\n",
                    acName);

                // Note: It is not possible to unsuccessfully add an
                // item on the connection list if we are using
                // a connection list which we just created via bAddProtocol().
            }

            // Now we can release the connection lock
            OSAL.eSemGive(psConnection->psProtocolEntry->hConnectionListMutex);
        }
        else
        {
            // Error!
            printf("STI Error! Unable to lock the connection list (%s).\n",
                acName);

            // Can't lock, can't connect. Sorry!
        }

    } while (FALSE);

    // Check if anything failed
    if(bConnectionInitialized == FALSE)
    {
        // Something failed, clean-up this mess.
        STI_bDisconnect(hSTI);
        hSTI = STI_INVALID_HDL;
    }

    return hSTI;
}

/*****************************************************************************
*
*   STI_bDisconnect
*
*   This API is used to disconnect an existing STI connection.
*
*   Inputs:
*       hSTI - A valid STI connection handle which is to be disconnected.
*
*   Outputs:
*       TRUE on success. Otherwise a value of FALSE is returned on error.
*
*****************************************************************************/
BOOLEAN STI_bDisconnect ( STI_HDL hSTI )
{
    BOOLEAN bSuccess = FALSE;

    // Check input
    if(hSTI != STI_INVALID_HDL)
    {
        // Take protocol list, this is so we have exclusive control
        // over the available protocols while we disconnect.
        OSAL_RETURN_CODE_ENUM eReturnCode;

        // Take control of the protocol list
        eReturnCode = OSAL.eSemTake(gsFramework.hProtocolListMutex,
            OSAL_OBJ_TIMEOUT_INFINITE);
        if(eReturnCode == OSAL_SUCCESS)
        {
            UN32 un32NumConnections = 0;
            STI_CONNECTION_ENTRY_STRUCT *psConnection =
                (STI_CONNECTION_ENTRY_STRUCT *)hSTI; // STI_HDL
            STI_PROTOCOL_ENTRY_STRUCT *psProtocolEntry =
                psConnection->psProtocolEntry;

            // If protocol was not installed, we have almost nothing 
            // to do here
            if (psProtocolEntry != (STI_PROTOCOL_ENTRY_STRUCT*)NULL)
            {
                // Take the connection list mutex
                eReturnCode = OSAL.eSemTake(
                    psProtocolEntry->hConnectionListMutex,
                    OSAL_OBJ_TIMEOUT_INFINITE);
                if(eReturnCode == OSAL_SUCCESS)
                {
                    // Delete this connection from this protocol handler's 
                    // connection list. This way, it at least cannot be found
                    // again.
                    // But we shouldn't delete the connection itself yet.
                    if(psConnection->hEntry != OSAL_INVALID_LINKED_LIST_ENTRY)
                    {
                        // Remove this connection from our list.
                        eReturnCode = OSAL.eLinkedListRemove(
                            psConnection->hEntry);
                        if(eReturnCode == OSAL_SUCCESS)
                        {
                            // Find out how many entries remain
                            (void)OSAL.eLinkedListItems(
                                psProtocolEntry->hConnectionList,
                                &un32NumConnections);

                            // Release connection
                            vConnectionRelease(psConnection);
                        }
                    }
                    else
                    {
                        // Release connection
                        vConnectionRelease(psConnection);
                    }

                    // At this the connection has been removed from the list
                    // and can no longer be found by anyone. We can now
                    // release the connection  list since we
                    // have effectively separated this connection now.
                    OSAL.eSemGive(psProtocolEntry->hConnectionListMutex);
                }

                // Finally request removal of the protocol associated with this
                // connection. If the protocol still has connections, nothing
                // needs to happen, but if all connections
                // are gone it will be removed.
                if(un32NumConnections == 0)
                {
                    // Remove this protocol from our list.
                    eReturnCode = OSAL.eLinkedListRemove(
                        psProtocolEntry->hEntry);
                    if(eReturnCode == OSAL_SUCCESS)
                    {
                        // Release protocol
                        vProtocolRelease(psProtocolEntry);
                    }
                }
            }

            // Give back protocol list mutex
            OSAL.eSemGive(gsFramework.hProtocolListMutex);

            // Done!
            bSuccess = TRUE;
        }
    }

    return bSuccess;
}

/*****************************************************************************
*
*   psFindConnection
*
*   This API is used by Protocol Drivers to query a connection handle which
*   belongs or maps to a particular set of protocol specific parameters. This
*   might be an address, index, id or other identifiers defined by the Protocol
*   Driver. To perform this search, a valid protocol handle must be provided
*   along with any connection specific data. This data must be cast by the
*   Protocol Driver's Connection Interface method OSAL_LL_COMPARE_HANDLER
*   provided in the Protocol Driver's STI_PROTOCOL_CONNECTION_HANDLER_STRUCT.
*
*   Inputs:
*   	hProtocol - A valid STI protocol handle for which to search for
*   		connection handles which match the supplied set of connection
*   		parameters.
*   	pvProtocolConnectionData - A void * type which is Protocol Driver
*   		specific and must be cast to the appropriate type during the
*   		compare process. Protocol Drivers use this data to locate a
*   		connection which maps to this data.
*       ppvData - A pointer populated with the connection's data
*
*   Outputs:
*   	A valid STI_HDL on success. Otherwise, STI_INVALID_HDL is returned
*   	if no active connection can be found.
*
*****************************************************************************/
static STI_CONNECTION_ENTRY_STRUCT *psFindConnection(
    STI_PROTOCOL_HDL hProtocol,
    void *pvProtocolConnectionData,
    void **ppvData
        )
{
    STI_CONNECTION_ENTRY_STRUCT *psConnectionEntry = NULL;
    const STI_PROTOCOL_ENTRY_STRUCT *psProtocolEntry;

    // Check input protocol handle
    if(hProtocol == STI_PROTOCOL_INVALID_HDL)
    {
        return NULL;
    }

    // From the protocol handle, extract the protocol entry
    psProtocolEntry = STIP_psGetProtocolEntry(hProtocol);
    if(psProtocolEntry != NULL)
    {
        // Search connection list for something that matches
        OSAL_LINKED_LIST_ENTRY hFoundEntry =
            OSAL_INVALID_LINKED_LIST_ENTRY; // search from start(top) of list
        OSAL_RETURN_CODE_ENUM eReturnCode;

        // Take control of the connection list
        eReturnCode = OSAL.eSemTake(
            psProtocolEntry->hConnectionListMutex,
            OSAL_OBJ_TIMEOUT_INFINITE);
        if(eReturnCode == OSAL_SUCCESS)
        {
            STI_CONNECTION_ENTRY_STRUCT sDummyConnection;

            // Initialize dummy connection for use in list search
            sDummyConnection.pvProtocolConnectionData = pvProtocolConnectionData;

            // Search for connections with these protocol specific parameters
            eReturnCode =
                OSAL.eLinkedListSearch(
                    psProtocolEntry->hConnectionList,
                    &hFoundEntry, &sDummyConnection);
            if(eReturnCode == OSAL_SUCCESS)
            {
                // Found it!
                psConnectionEntry = (STI_CONNECTION_ENTRY_STRUCT *)
                    OSAL.pvLinkedListThis(hFoundEntry);

                if (psConnectionEntry != (STI_CONNECTION_ENTRY_STRUCT *)NULL)
                {
                    // Provide the caller with the connection handle and data
                    if(ppvData != NULL)
                    {
                        *ppvData = pvGetProtocolConnectionData(psConnectionEntry);
                    }
                }
            }

            // Give-up connection list control if nothing was found
            if(psConnectionEntry == NULL)
            {
                OSAL.eSemGive(psProtocolEntry->hConnectionListMutex);
            }
        }

    }

    return psConnectionEntry;
}

/*****************************************************************************
*
*   STI_hAllocatePayload
*
*   This API is used to allocate an application tx payload buffer from
*   the specified STI connection. The payload buffer allocated is an
*   OSAL Dynamic Buffer which may be populated by the application using
*   OSAL Dynamic Buffer APIs (such as OSAL.tBufferWriteHead() and
*   OSAL.tBufferWriteTail()). Once allocated this buffer belongs to and
*   is associated with this connection.
*
*   Inputs:
*       hSTI - A valid STI connection handle from which the allocated buffer
*           will be obtained.
*
*   Outputs:
*       A valid OSAL_BUFFER_HDL is returned on success, otherwise
*       OSAL_INVALID_BUFFER_HDL is returned.
*
*****************************************************************************/
OSAL_BUFFER_HDL STI_hAllocatePayload ( STI_HDL hSTI )
{
    OSAL_BUFFER_HDL hPayload = OSAL_INVALID_BUFFER_HDL;

    // Check inputs
    if(hSTI != STI_INVALID_HDL)
    {
        STI_CONNECTION_ENTRY_STRUCT *psConnection =
            (STI_CONNECTION_ENTRY_STRUCT *)hSTI; // STI_HDL

        hPayload = STIP_hAllocateTxPayload(
            psConnection->psProtocolEntry->hProtocol);
    }

    return hPayload;
}

/*****************************************************************************
*
*   STI_bFreePayload
*
*   This API is used to de-allocate or free a previously obtained payload.
*   Once freed, the payload buffer handle is no longer valid.
*
*   Inputs:
*       hPayload - A valid STI payload buffer handle that is to be
*           freed or released.
*
*   Outputs:
*       TRUE on success. Otherwise a value of FALSE is returned on error.
*
*****************************************************************************/
BOOLEAN STI_bFreePayload ( OSAL_BUFFER_HDL hPayload )
{
    BOOLEAN bSuccess = FALSE;

    // Check input
    if(hPayload != OSAL_INVALID_BUFFER_HDL)
    {
        OSAL_RETURN_CODE_ENUM eReturnCode;

        // Free buffer
        eReturnCode = OSAL.eBufferFree(hPayload);
        if(eReturnCode == OSAL_SUCCESS)
        {
            // All is well
            bSuccess = TRUE;
        }
        else
        {
            // Error!
            printf("STI Error! Unable to free buffer. %s\n",
                   OSAL.pacGetReturnCodeName(eReturnCode));
        }
    }

    return bSuccess;
}

/*****************************************************************************
*
*   STI_eSendPayload
*
*   This API is used to send or transmit a previously allocated and populated
*   payload via the associated connection. Once successfully sent, the payload
*   is queued by the connection and is the responsibility of the STI Protocol
*   Driver to eventually free the payload once it has been transmitted or
*   transmission failure occurs.
*
*   Inputs:
*       hSTI - A valid STI connection handle for which the allocated buffer
*           will be transmitted over.
*       bBlock - TRUE to block on send or FALSE to not block.
*       hPayload - A valid STI payload buffer handle that is to queued
*           for transmission.
*       un8Retries - The number of retries to be attempted if the
*           protocol driver does not receive a response.
*       n32Timeout - The timeout in msec to wait for a response.
*
*   Outputs:
*       STI_PROTOCOL_RESULT_CODE_ENUM.
*
*****************************************************************************/
STI_PROTOCOL_RESULT_CODE_ENUM STI_eSendPayload (
    STI_HDL hSTI,
    BOOLEAN bBlock,
    OSAL_BUFFER_HDL hPayload,
    UN8 un8Retries,
    N32 n32Timeout
        )
{
    STI_PROTOCOL_RESULT_CODE_ENUM eResultCode =
        STI_RESULT_CODE_BAD_INPUT;

    // Check inputs...
    if(hSTI != STI_INVALID_HDL)
    {
        STI_CONNECTION_ENTRY_STRUCT *psConnection =
            (STI_CONNECTION_ENTRY_STRUCT *)hSTI; // STI_HDL

        eResultCode = STIP_eSendPayload(
            hSTI,
            psConnection->psProtocolEntry->hProtocol,
            bBlock,
            hPayload,
            un8Retries,
            n32Timeout
                );
    }

    return eResultCode;
}

/*****************************************************************************
*
*   STI_bReceivePayload
*
*   This API is used by Protocol Drivers to post received payload back
*   to the application via the connection it was received over. This API
*   will result in calling the application defined protocol callback as
*   configured using the STI_hConnect() call.
*
*   Inputs:
*   	hProtocol - A valid STI protocol handle for which to receive this
*   		payload over.
*       pvProtocolConnectionData - Protocol data used to identify a
*           specific connection on this protocol.
*   	psRx - A pointer to the rx payload structure populated by the
*           framework.
*
*   Outputs:
*   	TRUE on success (payload received or posted). Otherwise a value
*   	of FALSE is returned when the payload could not be posted.
*
*****************************************************************************/
BOOLEAN STI_bReceivePayload (
    STI_PROTOCOL_HDL hProtocol,
    void *pvProtocolConnectionData,
    STI_PROTOCOL_RX_PAYLOAD_STRUCT *psRx
        )
{
    BOOLEAN bReceived = FALSE;

    // Check inputs
    if( (psRx != NULL) && (psRx->hPayload != OSAL_INVALID_BUFFER_HDL))
    {
        STI_CONNECTION_ENTRY_STRUCT *psConnectionEntry;

        // Find a connection which maps to this connection data.
        psConnectionEntry = psFindConnection(
            hProtocol, pvProtocolConnectionData, NULL);

        // Does such a connection exist?
        if(psConnectionEntry != NULL)
        {
            // Call connection callback function
            if(psConnectionEntry->vCallback != (STI_PROTOCOL_CALLBACK)NULL)
            {
                psConnectionEntry->sCallback.eType =
                    STI_PROTOCOL_CALLBACK_MSG_TYPE_RX_PAYLOAD;
                psConnectionEntry->sCallback.sMsg.psRx =
                    psRx;
                psConnectionEntry->sCallback.pvArg =
                    psConnectionEntry->pvCallbackArg;

                bReceived = psConnectionEntry->vCallback(
                    (STI_HDL)psConnectionEntry,
                    psConnectionEntry->psProtocolEntry->hProtocol,
                    &psConnectionEntry->sCallback);
            }

            // Give back connection
            OSAL.eSemGive(
                psConnectionEntry->psProtocolEntry->hConnectionListMutex);
        }
    }

    return bReceived;
}

/*****************************************************************************
*
*   STI_bTransmitPayload
*
*   This API is used by Protocol Drivers to post transmit payload back
*   to the application via the connection it was received over. This API
*   will result in calling the application defined protocol callback as
*   configured using the STI_hConnect() call.
*
*   Inputs:
*   	hSTI - A valid STI connection handle for which to receive this
*   		payload over.
*   	psTx - A pointer to the tx payload structure populated by the
*           framework.
*
*   Outputs:
*   	TRUE on success (payload received or posted). Otherwise a value
*   	of FALSE is returned when the payload could not be posted.
*
*****************************************************************************/
BOOLEAN STI_bTransmitPayload (
    STI_HDL hSTI,
    STI_PROTOCOL_TX_PAYLOAD_STRUCT *psTx
        )
{
    BOOLEAN bReceived = FALSE;

    if(hSTI != STI_INVALID_HDL)
    {
        STI_CONNECTION_ENTRY_STRUCT *psConnection =
            (STI_CONNECTION_ENTRY_STRUCT *)hSTI; // STI_HDL

        // Call connection callback function
        if(psConnection->vCallback != (STI_PROTOCOL_CALLBACK)NULL)
        {
            psConnection->sCallback.eType =
                STI_PROTOCOL_CALLBACK_MSG_TYPE_TX_PAYLOAD;
            psConnection->sCallback.sMsg.psTx =
                psTx;
            psConnection->sCallback.pvArg =
                psConnection->pvCallbackArg;

            bReceived = psConnection->vCallback(
                hSTI, psConnection->psProtocolEntry->hProtocol,
                &psConnection->sCallback);
        }
    }

    return bReceived;
}

/*****************************************************************************
*
*   STI_bSendTimeout
*
*   This API is used by Protocol Drivers to post transmit timeout back
*   to the application via the connection it was received over. This API
*   will result in posting a timeout event to the protocol txrx handler.
*
*   Inputs:
*   	hSTI - A valid STI connection handle for which to receive this
*   		timeout over.
*
*   Outputs:
*   	TRUE on success (posted). Otherwise a value
*   	of FALSE is returned when the msg could not be posted.
*
*****************************************************************************/
BOOLEAN STI_bSendTimeout (
    STI_HDL hSTI
        )
{
    BOOLEAN bSuccess = FALSE;

    // Check inputs...
    if(hSTI != STI_INVALID_HDL)
    {
        STI_CONNECTION_ENTRY_STRUCT *psConnection =
            (STI_CONNECTION_ENTRY_STRUCT *)hSTI; // STI_HDL

        bSuccess = STIP_bPostTimeout(
            psConnection->psProtocolEntry->hProtocol, hSTI, FALSE);
    }

    return bSuccess;
}

/*****************************************************************************
*
*   STI_bReceiveTimeout
*
*   This API is used by Protocol Drivers to post timeout information
*   to the application via the connection it was received over. This API
*   will result in calling the application defined protocol callback as
*   configured using the STI_hConnect() call.
*
*   Inputs:
*   	hSTI - A valid STI connection handle for which to receive this
*   		timeout msg over.
*
*   Outputs:
*   	TRUE on success (payload received or posted). Otherwise a value
*   	of FALSE is returned when the timeout msg could not be posted.
*
*****************************************************************************/
BOOLEAN STI_bReceiveTimeout (
    STI_HDL hSTI
        )
{
    BOOLEAN bReceived = FALSE;

    if(hSTI != STI_INVALID_HDL)
    {
        STI_CONNECTION_ENTRY_STRUCT *psConnection =
            (STI_CONNECTION_ENTRY_STRUCT *)hSTI; // STI_HDL

        // Call connection callback function
        if(psConnection->vCallback != (STI_PROTOCOL_CALLBACK)NULL)
        {
            psConnection->sCallback.eType =
                STI_PROTOCOL_CALLBACK_MSG_TYPE_TX_TIMEOUT;
            psConnection->sCallback.pvArg =
                psConnection->pvCallbackArg;

            bReceived = psConnection->vCallback(
                hSTI, psConnection->psProtocolEntry->hProtocol,
                &psConnection->sCallback);
        }
    }

    return bReceived;
}

/*****************************************************************************
*
*   STI_bSendOOB
*
*   This API is used by Protocol Drivers to send Out-of-band information
*   to the application via the connection provided.
*
*   Inputs:
*   	hSTI - A valid STI connection handle for which to send this
*   		oob msg over.
*       eOOBType - The type of OOB message.
*       pvData - Some protocol specific OOB message data
*
*   Outputs:
*   	TRUE on success (payload received or posted). Otherwise a value
*   	of FALSE is returned when the oob msg could not be posted.
*
*****************************************************************************/
BOOLEAN STI_bSendOOB (
    STI_HDL hSTI,
    STI_OOB_TYPE_ENUM eOOBType,
    void *pvEventData
        )
{
    BOOLEAN bSuccess = FALSE;

    // Check inputs...
    if(hSTI != STI_INVALID_HDL)
    {
        STI_CONNECTION_ENTRY_STRUCT *psConnection =
            (STI_CONNECTION_ENTRY_STRUCT *)hSTI; // STI_HDL

        bSuccess = STIP_bPostOOB(
            psConnection->psProtocolEntry->hProtocol,
            psConnection->pvProtocolConnectionData,
            TRUE, eOOBType, pvEventData
                );
    }

    return bSuccess;
}

/*****************************************************************************
*
*   STI_bReceiveOOB
*
*   This API is used by Protocol Drivers to post Out-of-band information
*   to the application via the connection it was received over.  This API
*   will result in calling the application defined protocol callback as
*   configured using the STI_hConnect() call.
*
*   Inputs:
*   	hProtocol - A valid STI protocol handle for which to receive this
*   		oob msg over.
*       pvProtocolConnectionData - Protocol data used to identify a
*           specific connection on this protocol.
*       eOOBType - The type of OOB message.
*       pvData - Some protocol specific OOB message data
*
*   Outputs:
*   	TRUE on success (payload received or posted). Otherwise a value
*   	of FALSE is returned when the oob msg could not be posted.
*
*****************************************************************************/
BOOLEAN STI_bReceiveOOB (
    STI_PROTOCOL_HDL hProtocol,
    void *pvProtocolConnectionData,
    STI_OOB_TYPE_ENUM eOOBType,
    void *pvData
        )
{
    BOOLEAN bReceived = FALSE;
    STI_CONNECTION_ENTRY_STRUCT *psConnectionEntry;

    // Find a connection which maps to this connection data.
    psConnectionEntry = psFindConnection(
        hProtocol, pvProtocolConnectionData, NULL);

    // Does such a connection exist?
    if(psConnectionEntry != NULL)
    {
        // Call connection callback function
        if(psConnectionEntry->vCallback != (STI_PROTOCOL_CALLBACK)NULL)
        {
            STI_PROTOCOL_OOB_STRUCT sOOB;

            sOOB.eType = eOOBType;
            sOOB.pvData = pvData;

            psConnectionEntry->sCallback.eType =
                STI_PROTOCOL_CALLBACK_MSG_TYPE_RX_OOB;
            psConnectionEntry->sCallback.sMsg.psOOB =
                &sOOB;
            psConnectionEntry->sCallback.pvArg =
                psConnectionEntry->pvCallbackArg;

            bReceived = psConnectionEntry->vCallback(
                (STI_HDL)psConnectionEntry,
                psConnectionEntry->psProtocolEntry->hProtocol
                , &psConnectionEntry->sCallback);
        }

        // Give back connection
        OSAL.eSemGive(
            psConnectionEntry->psProtocolEntry->hConnectionListMutex);
    }

    return bReceived;
}

/*****************************************************************************
*
*   STI_pvGetConnectionArgument
*
*   This API is used by Applications to query a connection's associated
*   Connection Argument.
*
*   Inputs:
*   	hSTI - A valid STI connection handle for which to locate it's
*   		connection argument.
*
*   Outputs:
*   	A void * on success. Otherwise, NULL on error. The returned value
*   	is application specific and must be cast to the appropriate
*   	type once it is received.
*
*****************************************************************************/
void *STI_pvGetConnectionArgument ( STI_HDL hSTI )
{
    void *pvConnectionArgument = NULL;

    if(hSTI != STI_INVALID_HDL)
    {
        STI_CONNECTION_ENTRY_STRUCT *psConnection =
            (STI_CONNECTION_ENTRY_STRUCT *)hSTI; // STI_HDL

        pvConnectionArgument =
            (void *)psConnection->pvConnectionArgument;
    }

    return pvConnectionArgument;
}

/*****************************************************************************
*
*   STI_bSetError
*
*   This API is used by Protocol Drivers to indicate the STI connection
*   is in error.
*
*   Inputs:
*   	hSTI - A valid STI connection handle for which to set the error.
*
*   Outputs:
*   	TRUE on success (posted). Otherwise a value
*   	of FALSE is returned when the API could not be processed.
*
*****************************************************************************/
BOOLEAN STI_bSetError (
    STI_HDL hSTI
        )
{
    BOOLEAN bSuccess = FALSE;

    // Check inputs...
    if(hSTI != STI_INVALID_HDL)
    {
        STI_CONNECTION_ENTRY_STRUCT *psConnection =
            (STI_CONNECTION_ENTRY_STRUCT *)hSTI; // STI_HDL

        bSuccess = STIP_bSetError(
            psConnection->psProtocolEntry->hProtocol);
    }

    return bSuccess;
}


/*****************************************************************************
*
*   pvGetProtocolConnectionData
*
*   This API is used by Protocol Drivers to query a connection's associated
*   Protocol Specific data.
*
*   Inputs:
*   	hSTI - A valid STI connection handle for which to locate it's
*   		associated protocol data.
*
*   Outputs:
*   	A void * on success. Otherwise, NULL on error. The returned value
*   	is Protocol Driver specific and must be cast to the appropriate
*   	type once it is received.
*
*****************************************************************************/
static void *pvGetProtocolConnectionData (
    STI_CONNECTION_ENTRY_STRUCT const *psConnection
        )
{
    void *pvProtocolConnectionData = NULL;

    if(psConnection != NULL)
    {
        pvProtocolConnectionData =
            (void *)psConnection->pvProtocolConnectionData;
    }

    return pvProtocolConnectionData;
}

/*****************************************************************************
*
*   n16CompareProtocolDevices
*
*   This function is used to compare the 2 protocol entries provided
*   specifically comparing if the two devices are a match.
*
*   Inputs:
*       *pvArg1 - pointer to one of the protocols being compared
*       *pvArg2 - pointer to the other protocol being compared
*
*   Outputs:
*       n16Result - The comparison between the protocol and its' device
*           passed in.
*
*****************************************************************************/
static N16 n16CompareProtocolDevices( void *pvArg1, void *pvArg2 )
{
    N16 n16Result = -1;
    const STI_PROTOCOL_ENTRY_STRUCT *psProtocolEntry1 =
        (STI_PROTOCOL_ENTRY_STRUCT *)pvArg1;
    const STI_PROTOCOL_ENTRY_STRUCT *psProtocolEntry2 =
        (STI_PROTOCOL_ENTRY_STRUCT *)pvArg2;

    // Check pointer
    if( (psProtocolEntry1 != NULL) && (psProtocolEntry2 != NULL) )
    {
        // A protocol for a specific device handle is considered
        // 'existing' if the device handles match since there can be
        // only one protocol installed on a single physical device.
        if(psProtocolEntry1->pvDevice == psProtocolEntry2->pvDevice)
        {
            // They are the same
            n16Result = 0;
        }
    }

    return n16Result;
}

/*****************************************************************************
 *
 *   vProtocolRelease
 *
 *****************************************************************************/
static void vProtocolRelease (
    STI_PROTOCOL_ENTRY_STRUCT *psProtocolEntry
        )
{
    OSAL_RETURN_CODE_ENUM eReturnCode;
    BOOLEAN bUninstalled;

    // To remove a protocol, all connections using this protocol
    // must have already been disconnected (i.e. available)

    // Invalidate master entry in order to
    // avoid removing it from master list
    psProtocolEntry->hEntry = OSAL_INVALID_LINKED_LIST_ENTRY;

    // Take the connection list mutex
    eReturnCode = OSAL.eSemTake(
        psProtocolEntry->hConnectionListMutex,
        OSAL_OBJ_TIMEOUT_INFINITE);
    if(eReturnCode == OSAL_SUCCESS)
    {
        // Destroy connection list itself (if it exists)
        if(psProtocolEntry->hConnectionList != OSAL_INVALID_OBJECT_HDL)
        {
            OSAL_RETURN_CODE_ENUM eReturnCode;

            // Remove all entries from the list and destroy each content entry.
            // at this point there actually shouldn't be any.

            // Since these are pre-allocated LL entries, we can't use eRemoveAll
            // and free the entry in the RemoveAll callback because that would
            // end up freeing the LL portion of the memory and then eRemoveAll
            // gets confused. So do it the old fashioned way.
            OSAL_LINKED_LIST_ENTRY hEntry;
            STI_CONNECTION_ENTRY_STRUCT *psConnection;

            do
            {
                hEntry = OSAL.hLinkedListFirst(
                    psProtocolEntry->hConnectionList,
                    (void**)&psConnection);
                if (hEntry != OSAL_INVALID_LINKED_LIST_ENTRY)
                {
                    // Remove this connection from our list.
                    eReturnCode = OSAL.eLinkedListRemove(hEntry);
                    if(eReturnCode == OSAL_SUCCESS)
                    {
                        // Release connection
                        vConnectionRelease(psConnection);
                        psConnection = NULL;
                    }
                }

            } while (hEntry != OSAL_INVALID_LINKED_LIST_ENTRY);

            // Delete the list
            eReturnCode = OSAL.eLinkedListDelete(
                psProtocolEntry->hConnectionList);
            if(eReturnCode == OSAL_SUCCESS)
            {
                // Invalidate list
                psProtocolEntry->hConnectionList =
                    OSAL_INVALID_OBJECT_HDL;
            }
        }

        // Delete mutex
        eReturnCode = OSAL.eSemDelete(psProtocolEntry->hConnectionListMutex);
        if(eReturnCode == OSAL_SUCCESS)
        {
            psProtocolEntry->hConnectionListMutex =
                OSAL_INVALID_OBJECT_HDL;
        }
    }

    // Uninstall protocol
    bUninstalled = STIP_bUninstallProtocol(psProtocolEntry->hProtocol);
    if(bUninstalled == TRUE)
    {
        psProtocolEntry->hProtocol = STI_PROTOCOL_INVALID_HDL;
    }

    // Uninitialize remaining elements
    psProtocolEntry->psProtocolIntf = NULL;
    psProtocolEntry->pvDevice = NULL;

    // Destroy entry
    OSAL.vLinkedListMemoryFree(psProtocolEntry);

    return;
}

/*****************************************************************************
 *
 *   vConnectionRelease
 *
 *****************************************************************************/
static void vConnectionRelease (
    STI_CONNECTION_ENTRY_STRUCT *psConnectionEntry
        )
{
    STI_PROTOCOL_ENTRY_STRUCT const *psProtocolEntry =
        psConnectionEntry->psProtocolEntry;

    // Invalidate master entry in order to
    // avoid removing it from master list
    psConnectionEntry->hEntry = OSAL_INVALID_LINKED_LIST_ENTRY;

    // Check if protocol data uninitialization method exists
    if(psProtocolEntry->psProtocolIntf->sConnection._vUninit !=
        (STI_PROTOCOL_CONNECTION_UNINIT_HANDLER_PROTOTYPE)NULL)
    {
        // Uninitialize protocol specific data
        psProtocolEntry->psProtocolIntf->sConnection._vUninit(
            psConnectionEntry->pvProtocolConnectionData);

        // Clear the data pointer
        psConnectionEntry->pvProtocolConnectionData = NULL;
    }

    // Now take care of the remaining connection business
    OSAL.bMemSet(psConnectionEntry->acName, 0,
        sizeof(psConnectionEntry->acName));

    psConnectionEntry->vCallback = (STI_PROTOCOL_CALLBACK)NULL;
    psConnectionEntry->pvCallbackArg = NULL;

    OSAL.bMemSet(&psConnectionEntry->sCallback, 0,
        sizeof(psConnectionEntry->sCallback)); // Populated as needed.

    // Destroy entry
    OSAL.vLinkedListMemoryFree(psConnectionEntry);

    return;
}

/*****************************************************************************
*
*   n16CompareCnxn
*
*   This function is used to compare the 2 connections.
*   This is used when searching the LL for a specific connection.
*
*   Inputs:
*       *pvArg1 - pointer to one of the connections being compared
*       *pvArg2 - pointer to the other connection being compared
*
*   Outputs:
*       n16Result - The comparison between the connections passed in
*           and the connection of the struct currently being searched.
*
*****************************************************************************/
static N16 n16CompareCnxn( void *pvArg1, void *pvArg2)
{
    N16 n16Result = -1;
    const STI_CONNECTION_ENTRY_STRUCT *psCnxn1 =
        (const STI_CONNECTION_ENTRY_STRUCT *)pvArg1;
    const STI_CONNECTION_ENTRY_STRUCT *psCnxn2 =
        (const STI_CONNECTION_ENTRY_STRUCT *)pvArg2;

    // Check if connection handles to compare to exists.
    if ((psCnxn1 != NULL) && (psCnxn2 != NULL))
    {
        const STI_PROTOCOL_CONNECTION_HANDLER_STRUCT *psConnectionHdlr =
            &psCnxn1->psProtocolEntry->psProtocolIntf->sConnection;

        // Execute protocol specific compare fxn...
        if(psConnectionHdlr->_n16Compare != (OSAL_LL_COMPARE_HANDLER)NULL)
        {
            n16Result = psConnectionHdlr->_n16Compare(
                psCnxn1->pvProtocolConnectionData,
                psCnxn2->pvProtocolConnectionData);
        }
        else
        {
            n16Result = 0; // Match by default (only one connection allowed)
        }
    }

    return n16Result;
}

/*****************************************************************************
*
*   bAddProtocol
*
*****************************************************************************/
static BOOLEAN bAddProtocol (
    STI_PROTOCOL_ENTRY_STRUCT **ppsProtocolEntry,
    STI_PROTOCOL_CALLBACK vCallback,
    void *pvCallbackArg
        )
{
    BOOLEAN bAddedProtocol = FALSE;

    do
    {
        OSAL_RETURN_CODE_ENUM eReturnCode;

        // First, just add this protocol to see if it is unique
        // for a specific device. If the device already has a protocol
        // installed this won't work.
        eReturnCode = OSAL.eLinkedListAdd(gsFramework.hProtocols,
            &(*ppsProtocolEntry)->hEntry,
            *ppsProtocolEntry
                );
        if(eReturnCode == OSAL_ERROR_LIST_ITEM_NOT_UNIQUE) // A protocol exists
        {
            STI_PROTOCOL_ENTRY_STRUCT *psExistingProtocol;

            // Extract the existing protocol entry
            psExistingProtocol =
                (STI_PROTOCOL_ENTRY_STRUCT *)
                    OSAL.pvLinkedListThis(
                        (*ppsProtocolEntry)->hEntry);

            // Now we already have a protocol running on this device,
            // but all is not lost yet. We need to further understand
            // if what is really being requested is an additional
            // connection for the existing protocol on this device.
            // We will know if this is the case by comparing the
            // protocol interfaces.
            if(psExistingProtocol->psProtocolIntf ==
                (*ppsProtocolEntry)->psProtocolIntf)
            {
                // Indeed these are the same. So we can reuse the
                // installed protocol, just add a new connection.

                // Release allocated protocol memory
                OSAL.vLinkedListMemoryFree(
                    (void *)*ppsProtocolEntry);

                // Reassign local pointer to the existing protocol
                *ppsProtocolEntry = psExistingProtocol;

                //  Indicate protocol has been added
                bAddedProtocol = TRUE;

                // There is nothing left to do.
                break;
            }
            else // Protocol exists, but not the same.
            {
                // Error!

                printf("STI Error! Cannot add protocol (%s) to this device"
                    " since a protocol already exists on it (%s).\n",
                    (*ppsProtocolEntry)->psProtocolIntf->pacName,
                    psExistingProtocol->psProtocolIntf->pacName);
                break;
            }
        }
        else if(eReturnCode == OSAL_SUCCESS)
        {
            char acCxnName[OSAL_MAX_OBJECT_NAME_LENGTH_WITH_NULL];

            // New entry added. So this is a new device which
            // needs to have a protocol installed on it.

            // Let's first create the things that protocol
            // installations will need, such as a connection list.

            // Construct a unique name for the connection linked list mutex
            snprintf(&acCxnName[0], sizeof(acCxnName),
                    "%s:Mutex", (*ppsProtocolEntry)->psProtocolIntf->pacName);

            // Create a connection linked list mutex (available)
            eReturnCode = OSAL.eSemCreate(
                &(*ppsProtocolEntry)->hConnectionListMutex,
                &acCxnName[0], 1,
                1, OSAL_SEM_OPTION_NONE);
            if(eReturnCode == OSAL_SUCCESS)
            {
                // Construct a unique name for the connection list
                snprintf(&acCxnName[0], sizeof(acCxnName), "%s:List:",
                    (*ppsProtocolEntry)->psProtocolIntf->pacName);

                // Create a Connection Linked List based protocol specific information.
                // This linked list will store each connection (via handle) associated
                // with this instance of the protocol handler.
                eReturnCode = OSAL.eLinkedListCreate(
                    &(*ppsProtocolEntry)->hConnectionList,
                    &acCxnName[0],
                    n16CompareCnxn,
                    (OSAL_LL_OPTION_USE_PRE_ALLOCATED_ELEMENTS |
                        OSAL_LL_OPTION_LINEAR | OSAL_LL_OPTION_UNIQUE)
                        );
                if(eReturnCode != OSAL_SUCCESS)
                {
                    // Error!
                    printf("STI Error! Cannot create connection linked list"
                        " (%s).\n", acCxnName);
                    break;
                }

                // Finally, install protocol onto this device.
                (*ppsProtocolEntry)->hProtocol =
                    STIP_hInstallProtocol (
                        *ppsProtocolEntry,
                        vCallback,
                        pvCallbackArg
                            );
                if((*ppsProtocolEntry)->hProtocol == STI_PROTOCOL_INVALID_HDL)
                {
                    // Error!
                    printf("STI Error! Cannot install protocol onto device"
                        " (%s).\n",
                        (*ppsProtocolEntry)->psProtocolIntf->pacName);
                    break;
                }

                //  Indicate protocol has been added
                bAddedProtocol = TRUE;

                // There is nothing left to do.
                break;
            }
            else
            {
                // Error!
                printf("STI Error! Cannot create cxn ll-mutex mutex (%s).\n",
                    acCxnName);
            }

        }
        else
        {
            // Error!

            printf("STI Error! Cannot add protocol into list (%s).\n",
                (*ppsProtocolEntry)->psProtocolIntf->pacName);
            break;
        }

    } while (FALSE);

    return bAddedProtocol;
}
