/******************************************************************************/
/*                    Copyright (c) Sirius XM Radio, Inc.                     */
/*                            All Rights Reserved                             */
/*         Licensed Materials - Property of Sirius XM Radio, Inc.             */
/******************************************************************************/
/*******************************************************************************
 *
 * DESCRIPTION
 *
 *  This module contains implementation of the Ring Buffer API.
 *
 ******************************************************************************/

#include "sxill_ringbuffer.h"

/*****************************************************************************
*
*   SXILL_RB_vInit
*
*****************************************************************************/
void SXILL_RB_vInit(
    SXILL_RING_BUFFER_DATA_STRUCT *psObj,
    void *pvBuffer,
    size_t tBufferSize
        )
{
    psObj->tCapacity = tBufferSize;
    psObj->tSize = 0;
    psObj->tStart = 0;
    psObj->pun8Data = (UN8*)pvBuffer;

    return;
}

/*****************************************************************************
*
*   SXILL_RB_vAttach
*
*****************************************************************************/
void SXILL_RB_vAttach(
    SXILL_RING_BUFFER_DATA_STRUCT *psObj,
    SXILL_RING_BUFFER_DATA_STRUCT *psMaster
        )
{
    *psObj = *psMaster;

    return;
}

/*****************************************************************************
*
*   SXILL_RB_tRead
*
*****************************************************************************/
size_t SXILL_RB_tRead(
    SXILL_RING_BUFFER_DATA_STRUCT *psObj,
    void *pData,
    size_t tDataSize
        )
{
    UN8 *pun8Val = (UN8*)pData;
    size_t tResult = 0;

    // Make sure caller will get data no more that the stream really has
    if (tDataSize > psObj->tSize)
    {
        tDataSize = psObj->tSize;
    }

    // Byte-by-byte reading
    while (tResult < tDataSize)
    {
        *pun8Val++ = psObj->pun8Data[psObj->tStart];
        psObj->tStart = (psObj->tStart + 1) % psObj->tCapacity;
        ++tResult;
    }
    psObj->tSize -= tResult;

    return tResult;
}

/*****************************************************************************
*
*   SXILL_RB_tSkip
*
*****************************************************************************/
size_t SXILL_RB_tSkip(
    SXILL_RING_BUFFER_DATA_STRUCT *psObj,
    size_t tDataSize
        )
{
    // Make sure caller will get data no more that the stream really has
    if (tDataSize > psObj->tSize)
    {
        tDataSize = psObj->tSize;
    }

    psObj->tSize -= tDataSize;
    psObj->tStart = (psObj->tStart + tDataSize) % psObj->tCapacity;

    return tDataSize;
}

/*****************************************************************************
*
*   SXILL_RB_tCommit
*
*****************************************************************************/
size_t SXILL_RB_tCommit(
    SXILL_RING_BUFFER_DATA_STRUCT *psObj,
    size_t tDataSize
        )
{
    const size_t tLeft = psObj->tCapacity - psObj->tSize;
    if (tLeft < tDataSize)
    {
        tDataSize = tLeft;
    }
    psObj->tSize += tDataSize;

    return tDataSize;
}

/*****************************************************************************
*
*   SXILL_RB_tSize
*
*****************************************************************************/
size_t SXILL_RB_tSize(
    SXILL_RING_BUFFER_DATA_STRUCT *psObj
        )
{
    return psObj->tSize;
}

/*****************************************************************************
*
*   SXILL_RB_vSolidUnusedBlock
*
*****************************************************************************/
void SXILL_RB_vSolidUnusedBlock(
    SXILL_RING_BUFFER_DATA_STRUCT *psObj,
    UN8 **ppun8Block,
    size_t *ptBlockSize
        )
{
    if (psObj->tSize < psObj->tCapacity)
    {
        const size_t tEnd =
            (psObj->tStart + psObj->tSize) % psObj->tCapacity;

        *ppun8Block = &psObj->pun8Data[tEnd];
        if (tEnd < psObj->tStart)
        {
            *ptBlockSize = psObj->tStart - tEnd;
        }
        else
        {
            *ptBlockSize = psObj->tCapacity - tEnd;
        }
    }
    else
    {
        *ppun8Block = (UN8*)NULL;
        *ptBlockSize = 0;
    }
    return;
}

/*****************************************************************************
*
*   SXILL_RB_vSolidUsedBlocks
*
*****************************************************************************/
void SXILL_RB_vSolidUsedBlocks(
    SXILL_RING_BUFFER_DATA_STRUCT *psObj,
    UN8 **ppun8Block1,
    size_t *ptBlock1Size,
    UN8 **ppun8Block2,
    size_t *ptBlock2Size
        )
{
    if (psObj->tSize == 0)
    {
        *ppun8Block1 = *ppun8Block2 = (UN8*)NULL;
        *ptBlock1Size = *ptBlock2Size = 0;
    }
    else
    {
        const size_t tEnd =
            (psObj->tStart + psObj->tSize) % psObj->tCapacity;

        if (psObj->tStart < tEnd)
        {
            // Seems like the stream has one solid memory block to provide
            *ppun8Block1 = psObj->pun8Data + psObj->tStart;
            *ptBlock1Size = psObj->tSize;
            *ppun8Block2 = (UN8*)NULL;
            *ptBlock2Size = 0;
        }
        else
        {
            // Seems like there are two solid memory blocks
            *ppun8Block1 = psObj->pun8Data + psObj->tStart;
            *ptBlock1Size = psObj->tCapacity - psObj->tStart;
            *ppun8Block2 = psObj->pun8Data;
            *ptBlock2Size = psObj->tSize - *ptBlock1Size;
        }
    }
}
