/******************************************************************************/
/*                    Copyright (c) Sirius XM Radio, Inc.                     */
/*                            All Rights Reserved                             */
/*         Licensed Materials - Property of Sirius XM Radio, Inc.             */
/******************************************************************************/
/*******************************************************************************
 *
 * DESCRIPTION
 *
 *  This file contains the SXILL implementation using the STI Framework.
 *
 ******************************************************************************/

#include <stdio.h>
#include <string.h>
#include <stddef.h>

#include "standard.h"
#include "osal.h"

#include "sms.h" // for debug only

#include "com.h"

#include "sti_api.h"

#include "sxiapi.h"
#include "sxi.h"
#include "sxi_rw.h"
#include "sxill.h"
#include "_sxill.h"

// Connection Methods

/*****************************************************************************
 *
 *   SXILL_pvConnectionInit
 *
 *   Performs any required connection initialization. Each connection which
 *   is made to this protocol will execute this function. Any connection
 *   specific data which must be persistent for a connection should be
 *   passed back to the caller via the return.sValue (void *).
 *
 *   _pvInit
 *
 *   This connection interface method is invoked when a new connection to an
 *   existing protocol instance is being created. The Protocol Driver
 *   implementation should use this function to allocate any persistent resources
 *   (i.e. memory) required to maintain a connection. A pointer to any persistent
 *   data needs to be returned to the caller and it will be maintained by the STI
 *   Framework. If a Protocol Driver does not require an initialization step,
 *   this member may be assigned to NULL.
 *
 *   Inputs:
 *   	pacName - A pointer to an ASCII null terminated string which represents
 *   		the name of the connection to be initialized.
 *   	hSTI - The STI Connection Handle assigned to this connection.
 *   	ptArgList - A pointer to a variable argument list which contains the
 *   		variable argument parameters supplied by Application via the
 *   		STI_hConnect() API.
 *
 *   Outputs:
 *   	The Protocol Driver implementation must return any persistent data
 *   	required by this connection. Implementation may return NULL if no
 *   	persistent data is required.
 *
 *****************************************************************************/
static STI_PROTOCOL_CONNECTION_INIT_HANDLER(SXILL_pvConnectionInit)
{
    SXILL_PAYLOAD_TYPE tPayloadType;

    // Check if we were given an argument list to parse.
    if (NULL == ptArgList)
    {
        // No arguments, no parse!
        return NULL;
    }

    tPayloadType = (SXILL_PAYLOAD_TYPE) va_arg( *ptArgList, int );
    tPayloadType =
        (SXILL_PAYLOAD_TYPE)(tPayloadType & SXILL_PAYLOAD_TYPE_MASK);

    printf("SXILL: Initialized connection (%s) Payload Type = %u\n",
        pacName, tPayloadType);

    return (void *) (size_t)tPayloadType;
}

/*****************************************************************************
 *
 *   SXILL_vConnectionUninit
 *
 *   Performs any required connection uninitialization. Each connection which
 *   is disconnected from this protocol will execute this function using
 *   STI_bDisconnect(). Any connection specific data which was persistent
 *   for a connection should be returned and is provided by the input parameter
 *   pvProtocolConnectionData.
 *
 *   _vUninit
 *
 *   This connection interface method is invoked when an existing connection to
 *   a protocol instance is being disconnected. The Protocol Driver
 *   implementation should use this function to release any allocated persistent
 *   resources (i.e. memory) required to maintain a connection. A pointer to any
 *   persistent data returned during the connection initialization is provided
 *   by the caller and needs to be cast to the appropriate type required by the
 *   specific Protocol Driver. If a Protocol Driver does not require an
 *   un-initialization step, this member may be assigned to NULL.
 *
 *   Inputs:
 *   	pvProtocolConnectionData - A pointer to any Protocol Driver specific
 *   		persistent data being maintained by the STI Framework. This pointer
 *   		must be cast locally to the appropriate type by the Protocol Driver
 *   		implementation.
 *
 *   Outputs:
 *   	None.
 *
 *****************************************************************************/
static STI_PROTOCOL_CONNECTION_UNINIT_HANDLER(SXILL_vConnectionUninit)
{
    SXILL_PAYLOAD_TYPE tPayloadType = (SXILL_PAYLOAD_TYPE) (size_t)pvProtocolConnectionData;

    printf("SXILL: Uninitializing connection with Payload Type = %u\n",
        tPayloadType);

    return;
}

/*****************************************************************************
 *
 *   SXILL_n16ConnectionCompare
 *
 *   This function is used to compare two connections. This is used when
 *   searching the LL for a specific connection given the input tuple.
 *
 *   Inputs:
 *       pvArg1 - In this case it is SXI_CONNECTION_STRUCT *. See
 *          SXILL_pvConnectionInit().
 *       pvArg2 - In this case it is also a SXI_CONNECTION_STRUCT *.
 *
 *   _n16Compare
 *
 *   This connection interface method is invoked when the Protocol Driver
 *   need to look-up or find an active connection which matches the connection
 *   criteria supplied. The STI Framework will
 *   call this function in order to compare all its active connections until
 *   a match has been found. A match is indicated by returning the.sValue 0.
 *   This member function has the same prototype as the OSAL Linked List
 *   Compare Function identified as OSAL_LL_COMPARE_HANDLER. Please refer
 *    to the OSAL Documentation for more details on this prototype.
 *
 *   Inputs:
 *   	pvObj1 - pointer to one of the connections being compared to. This
 *   		should be cast to the object provided as 'connection data' during
 *   		connection initialization.
 *   	pvObj2 - A pointer to the structure used to compare an available
 *   		connection with. This pointer must be cast to the structure being
 *   		provided to the STI APIs.
 *
 *   Outputs:
 *   	The comparison between the connection passed in and the connection
 *   	currently being searched for. A return.sValue of 0 indicates a match
 *   	has been found. If there is an implied "order" which connections can
 *   	be represented in then -1 may be returned to indicate the connection
 *   	being compared is less than the one being searched for. Likewise, 1 may
 *   	be returned to indicate the connection being compared is greater than
 *   	the one being searched for.
 *
 *****************************************************************************/
static N16 SXILL_n16ConnectionCompare(void *pvArg1, void *pvArg2)
{
    N16 n16Result = -1; // default is 'not equal'
    SXILL_PAYLOAD_TYPE tPayloadType1 = (SXILL_PAYLOAD_TYPE) (size_t)pvArg1;
    SXILL_PAYLOAD_TYPE tPayloadType2 = (SXILL_PAYLOAD_TYPE) (size_t)pvArg2;

    // An SXILL connection is defined by the type
    if (tPayloadType1 == tPayloadType2)
    {
        n16Result = 0;

    }

    return n16Result;
}

// TxRx Methods

/*****************************************************************************
 *
 *   SXILL_bTxRxInit
 *
 *   _bInit
 *
 *   This Transceiver Interface method is invoked when the STI requires the
 *   protocol specified to be installed, or initialized for the first-time.
 *   This is in response to the application creating the first connection to a
 *   specified Protocol. The Protocol Driver implementation should use this
 *   function to allocate any persistent resources (i.e. memory) required to
 *   implement the protocol. A pointer to any persistent data needs to be
 *   returned to the caller via the parameter *ppvProtocolData and it will
 *   be maintained by the STI Framework. If a Protocol Driver does not require
 *   a Transceiver or a Transceiver initialization step, this member may be
 *   assigned to NULL.
 *
 *   Inputs:
 *   	pacTaskName - A pointer to the name of the protocol to be initialized.
 *   	hProtocol - The STI Protocol Handle assigned to this protocol instance.
 *   	pvDevice - A void * representing the device this protocol connects
 *   		to. This may be a FILE * if it connects to a physical device driver,
 *   		a connection handle to connect to another lower-level protocol, or
 *   		any other "protocol driver" specific handle.
 *       pun32EventQueueSize - A pointer used to return the number of TxRx events
 *           required by this protocol driver. The.sValue returned via this
 *           parameter will be maintained by the STI Framework.
 *   	ppvProtocolData - A pointer to a void pointer which may be initialized
 *   	    by the protocol driver to maintain any persistent data required by
 *           the implementation. The.sValue returned via this parameter will be
 *   	    maintained by the STI Framework.
 *
 *   Outputs:
 *   	TRUE on success. Otherwise, FALSE on error. If FALSE is returned the
 *   	Protocol will not be installed.
 *
 *****************************************************************************/
static STI_PROTOCOL_TXRX_INIT_HANDLER(SXILL_bTxRxInit)
{
    OSAL_RETURN_CODE_ENUM eReturnCode;
    SXILL_TXRX_CONTROL_STRUCT *psCtrl;
    char acName[OSAL_MAX_OBJECT_NAME_LENGTH_WITH_NULL];

    printf("SXILL: SXILL_bTxRxInit()\n");

    // Construct a unique name for the memory of this handler's control struct
    snprintf(&acName[0], sizeof(acName), "%s:Ctrl", pacTaskName);

    // Allocate memory for control structure
    psCtrl = *((SXILL_TXRX_CONTROL_STRUCT **) ppvProtocolData) =
                    (SXILL_TXRX_CONTROL_STRUCT *) OSAL.pvMemoryAllocate(
                        &acName[0], sizeof(SXILL_TXRX_CONTROL_STRUCT), TRUE);
    if ((SXILL_TXRX_CONTROL_STRUCT *) NULL == psCtrl)
    {
        // Error! Memory allocation failed.
        printf("SXILL Error! %s: Out of memory.\n", pacTaskName);
        SXILL_vTxRxUnInit(hProtocol, ppvProtocolData);
        return FALSE;
    }

    // Populate control structure with inputs...

    // Initialize Rx-Task ctrl struct ptr (we don't know it yet)
    psCtrl->psRxCtrl = NULL;

    // Initialize Tx.sValues
    psCtrl->tTxSeq = 0;

    // Remember the physical device handle for which this protocol runs on
    psCtrl->psDevice = (FILE *) pvDevice;

    // Return the number of TxRx events required to the STI Framework
    *pun32EventQueueSize = SXILL_TXRX_EVENTS;

    // Construct a unique name for this payload block pool
    snprintf( &acName[0], sizeof(acName), "%s:TxPayload", pacTaskName);

    // Create a tx payload block pool
    eReturnCode =
        OSAL.eBlockPoolCreate (
            phTxBlockPool,
            &acName[0],
            SXILL_TX_PAYLOAD_UNIT_SIZE,
            SXILL_TX_PAYLOAD_UNITS,
            OSAL_BLOCK_POOL_OPTION_NONE
            );
    if(eReturnCode != OSAL_SUCCESS)
    {
        // Error!
        printf("STI Error! Cannot create tx payload block pool. %s in %s.\n",
               OSAL.pacGetReturnCodeName(eReturnCode), pacTaskName);
        SXILL_vTxRxUnInit(hProtocol, ppvProtocolData);
        return FALSE;
    }

    // Construct a unique name for the rx frame block pool
    snprintf( &acName[0], sizeof(acName), "%s:RxPayload", pacTaskName);

    // Create an rx frame block pool
    eReturnCode = OSAL.eBlockPoolCreate(
        phRxBlockPool, &acName[0],
        (UN16)SXILL_RX_FRAME_BLOCK_SIZE,
        SXILL_RX_FRAME_NUM_BLOCKS,
        OSAL_BLOCK_POOL_OPTION_NONE);
    if (OSAL_SUCCESS != eReturnCode)
    {
        // Error!
        printf("STI Error! Cannot create tx frame block pool. %s in %s.\n",
               OSAL.pacGetReturnCodeName(eReturnCode), pacTaskName);
        SXILL_vTxRxUnInit(hProtocol, ppvProtocolData);
        return FALSE;
    }

#ifdef SXILL_TXRX_STATS_OUTPUT
    {
        OSAL_RETURN_CODE_ENUM eReturnCode;

        // Initialize statistics
        OSAL.bMemSet(&psCtrl->sStatistics, 0, sizeof(psCtrl->sStatistics));

        // Open our stats log
        eReturnCode = OSAL.eLogCreate(&psCtrl->hLog, SXILL_TXRX_STATS_LOG_NAME,
            SXILL_TXRX_STATS_LOG_SIZE, OSAL_LOG_OPTION_FORMAT);

        if (OSAL_SUCCESS == eReturnCode)
        {
            printf("SXILLTxRx: Stats log created.\n");

            // Write the header
            OSAL.n32LogWrite(psCtrl->hLog, SXILL_TXRX_STATS_HEADER);

            // Set the last write time to now
            OSAL.eTimeGetLocal(&psCtrl->un32LastWrite);
        }
        else
        {
            printf("SXILLTx: ERROR!! creating stats log.\n");
        }
    }
#endif

    return TRUE;
}

/*****************************************************************************
 *
 *   SXILL_vTxRxUnInit
 *
 *   _vUninit
 *
 *   This Transceiver Interface method is invoked when the STI requires
 *   the protocol specified to be uninstalled, or un-initialized which
 *   occurs when no more connections to it exist. The Protocol Driver
 *   implementation should use this function to release any allocated persistent
 *   resources (i.e. memory) required to run the protocol. A pointer to any
 *   persistent data returned during the protocol initialization is provided
 *   by the STI Framework and needs to be cast to the appropriate type required
 *   by the specific Protocol Driver. If a Protocol Driver does not require a
 *   Transceiver or an un-initialization step, this member may be assigned to
 *   NULL.
 *
 *   Inputs:
 *   	hProtocol - The STI Protocol Handle assigned to this protocol instance.
 *   	pvProtocolData - A pointer to any Protocol Driver specific persistent
 *   		data being maintained by the STI Framework. This pointer must be
 *   		cast locally to the appropriate type by the Protocol Driver
 *   		implementation.
 *
 *   Outputs:
 *   	None.
 *
 *****************************************************************************/
static STI_PROTOCOL_UNINIT_HANDLER(SXILL_vTxRxUnInit)
{
    SXILL_TXRX_CONTROL_STRUCT *psTxRxCtrl =
        (SXILL_TXRX_CONTROL_STRUCT *) pvProtocolData;
    N32 n32Err;

    printf("SXILL: SXILL_vTxRxUnInit()\n");

    if (pvProtocolData == NULL)
    {
        return;
    }

#ifdef SXILL_TXRX_STATS_OUTPUT
    // close our stats log
    OSAL.eLogDelete(psTxRxCtrl->hLog);
    psTxRxCtrl->hLog = OSAL_INVALID_OBJECT_HDL;
#endif

    // Unprotect the device to allow it to be
    // closed by whomever owns it. We do this in the TxRx thread
    // since it is the Rx thread which informs the TxRx thread to
    // shutdown.

    n32Err = ioctl(psTxRxCtrl->psDevice, FILE_IOCTL_USAGE_UNPROTECT);
    if (0 != n32Err)
    {
        printf("SXILL Error! Failed to unprotect read device\n");
    }

    // Free allocated memory
    OSAL.vMemoryFree(psTxRxCtrl);
    pvProtocolData = NULL;

    return;
}

/*****************************************************************************
 *
 *   SXILL_n32TxRxHdlr
 *
 *   _n32Hdlr
 *
 *   This Transceiver Interface method is invoked when the STI has a Transceiver
 *   event to process. Otherwise the handler is idle (i.e. not running). When an
 *   event occurs this handle is called and the event type and data are passed to
 *   the handler for processing by the Protocol Driver. As long as no fatal
 *   errors exist in the Protocol Driver, the handler should always return
 *   OSAL_TASK_REPORT_NO_ERROR. Otherwise if an error occurs and the Protocol
 *   needs to shutdown, the Protocol Driver should report some Protocol Specific
 *   error code not equal to OSAL_TASK_REPORT_NO_ERROR. If a Protocol Driver does
 *   not require a Transceiver this member may be assigned to NULL.
 *
 *   Inputs:
 *   	hProtocol - The STI Protocol Handle assigned to this protocol instance.
 *   	pvProtocolData - A pointer to any Protocol Driver specific persistent
 *   		data being maintained by the STI Framework. This pointer must be
 *   		cast locally to the appropriate type by the Protocol Driver
 *   		implementation.
 *   	un32EventType - A UN32 describing a Protocol Driver specific event type.
 *   		This can be used by protocol drivers as a code or key into event
 *   		types. This is the same event type specified in the STI_bPostEvent()
 *   		API (See Section 9.4).
 *   	pvEventData - A void * type which is Protocol Driver specific and must
 *   	be cast to the appropriate type during event processing. Protocol
 *       Drivers may use this as a way to communicate event data to the
 *       Transceiver Handler. This is the same event data specified in the
 *       STI_bPostEvent() API (See Section 9.4).
 *
 *   Outputs:
 *   	OSAL_TASK_REPORT_NO_ERROR if no error. Otherwise any.sValue not equal to
 *   	OSAL_TASK_REPORT_NO_ERROR.
 *
 *****************************************************************************/
static STI_PROTOCOL_TXRX_HANDLER(SXILL_n32TxRxHdlr)
{
    N32 n32Error = 0;
    SXILL_TXRX_CONTROL_STRUCT *psCtrl =
        (SXILL_TXRX_CONTROL_STRUCT *) pvProtocolData;

    printf("SXILL: SXILL_n32TxRxHdlr()\n");

#ifdef SXILL_TXRX_STATS_OUTPUT
    vWriteTxRxStats(psCtrl);
#endif

    // Process event...
    switch (psEvent->eType)
    {
        case STI_PROTOCOL_EVENT_RX_PAYLOAD:
            vHandleRx(hProtocol, psCtrl, &psEvent->sMsg.sRx);
        break;

        case STI_PROTOCOL_EVENT_TX_PAYLOAD:
            SXILL_vHandleTx(hProtocol, &psEvent->sMsg.sTx);
        break;

        case STI_PROTOCOL_EVENT_OOB:
            vHandleOOB(hProtocol, psCtrl, &psEvent->sMsg.sOOB);
        break;

        case STI_PROTOCOL_EVENT_TIMEOUT:
            STI_bReceiveTimeout(psEvent->sMsg.sTimeout.hSTI);
        break;

        case STI_PROTOCOL_EVENT_PROTOCOL_SPECIFIC:
            vHandleProtocolSpecific(hProtocol, psCtrl, &psEvent->sMsg.sProto);
        break;

        default:

            printf("SXILL: Rx UNKNOWN Event\n");

#ifdef SXILL_TXRX_STATS_OUTPUT
            // Increment Statistics
            psCtrl->sStatistics.sRx.un32UnknownEvent++;
#endif

        break;
    }

    // check if error was detected
    n32Error = ferror(psCtrl->psDevice);
    if (0 != n32Error)
    {
        // error was detected
        OSAL.vReportError(OSAL_ERROR_TASK_MONITOR_DEVICE_DRIVER_FAILURE);

        // Uninstall TxRx Task.
        return !OSAL_TASK_REPORT_NO_ERROR;
    }

    return OSAL_TASK_REPORT_NO_ERROR;
}

// RxFrame Methods

/*****************************************************************************
 *
 *   SXILL_bRxFrameInit
 *
 *   _bInit
 *   	This Receiver Interface method is invoked when the STI requires the
 *   	protocol specified to be installed, or initialized for the first-time.
 *   	This is in response to the application creating the first connection to
 *   	a specified Protocol. The Protocol Driver implementation should use this
 *   	function to allocate any persistent resources (i.e. memory) required to
 *   	implement the protocol. A pointer to any persistent data needs to be
 *   	returned to the caller via the parameter *ppvProtocolData and it will
 *   	be maintained by the STI Framework. If a Protocol Driver does not
 *   	require a Receiver or a Receiver initialization step, this member may
 *   	be assigned to NULL.
 *
 *   Inputs:
 *   	pacTaskName - A pointer to the name of the protocol to be initialized.
 *   	hProtocol - The STI Protocol Handle assigned to this protocol instance.
 *   	n32TimeoutMsec - The maximum time in milliseconds the Receiver handler
 *   		can suspend waiting on an event or data. When this time has expired
 *   		the Receiver Handler must return to the caller.
 *   	ppvProtocolData - A pointer to a void pointer which may be initialized
 *   		by the protocol driver to maintain any persistent data required by
 *   		the implementation. The.sValue returned via this parameters will be
 *   		maintained by the STI Framework.
 *
 *   Outputs:
 *   	TRUE on success. Otherwise, FALSE on error. If FALSE is returned the
 *   	Protocol will not be installed.
 *
 *****************************************************************************/
static STI_PROTOCOL_RX_INIT_HANDLER(SXILL_bRxFrameInit)
{
    N32 n32Err;
    SXILL_RX_FRAME_CONTROL_STRUCT *psCtrl;
    char acName[OSAL_MAX_OBJECT_NAME_LENGTH_WITH_NULL];
    // TxRx Ctrl structure pointer
    SXILL_TXRX_CONTROL_STRUCT *psTxRxCtrl = (SXILL_TXRX_CONTROL_STRUCT*)*ppvProtocolData;

    printf("SXILL: SXILL_bRxFrameInit()\n");

    // Check for valid TxRx protocol data provided to us, if not data
    // then we cannot continue.
    if (NULL == *ppvProtocolData)
    {
        return FALSE;
    }

    // Construct a unique name for the memory of the Rx Handler
    // control structure
    snprintf(&acName[0], sizeof(acName), "%s:Ctrl", pacTaskName);

    // Allocate memory for control structure...
    psCtrl = (SXILL_RX_FRAME_CONTROL_STRUCT *)
        OSAL.pvMemoryAllocate(&acName[0],
            sizeof(SXILL_RX_FRAME_CONTROL_STRUCT), TRUE);
    if (NULL == psCtrl)
    {
        printf("SXILL Error! %s: Out of memory.\n",
                        pacTaskName);
        SXILL_vRxFrameUnInit(hProtocol, psCtrl);
        return FALSE;
    }

    // Populate control structure...

    // Extract device handle for Rx
    psCtrl->psDevice = psTxRxCtrl->psDevice;

    // Initialize no rx counter
    psCtrl->un32NoRxCount = 0;

    // Initialize rx frame buffer
    OSAL.bMemSet(&psCtrl->sRingBuffer, 0, sizeof(psCtrl->sRingBuffer));
    vInitRingBuffer(&psCtrl->sRingBuffer);

    // Reset state machine
    vResetSyncStateMachine(&psCtrl->sContext);

    // Protect the file to ensure it can't be
    // closed while we're using it
    n32Err = ioctl(psCtrl->psDevice, FILE_IOCTL_USAGE_PROTECT);
    if (0 != n32Err)
    {
        printf("SXILL Error! %s: Failed to protect read device\n",
                        pacTaskName);
        SXILL_vRxFrameUnInit(hProtocol, psCtrl);
        return FALSE;
    }

#ifdef SXILL_RX_STATS_OUTPUT

    {
        OSAL_RETURN_CODE_ENUM eReturnCode;

        // Initialize statistics
        OSAL.bMemSet(&psCtrl->sStatistics, 0, sizeof(psCtrl->sStatistics));

        // Initialize last write counter
        psCtrl->un32LastWrite = 0;

        // Open our rx stats log
        eReturnCode = OSAL.eLogCreate(&psCtrl->hLog, SXILL_RX_STATS_LOG_NAME,
            SXILL_RX_STATS_LOG_SIZE, OSAL_LOG_OPTION_FORMAT);

        if (OSAL_SUCCESS == eReturnCode)
        {
            printf("SXILLRx: Stats log created.\n");

            // Write the header
            OSAL.n32LogWrite(psCtrl->hLog, SXILL_RX_STATS_HEADER);

            // Set the last write time to now
            OSAL.eTimeGetLocal(&psCtrl->un32LastWrite);
        }
        else
        {
            printf("SXILLRx: ERROR!! creating stats log.\n");
        }

        // Open our rx request log
        eReturnCode = OSAL.eLogCreate(&psCtrl->hReqLog, SXILL_RX_REQ_LOG_NAME,
            SXILL_RX_REQ_LOG_SIZE, OSAL_LOG_OPTION_FORMAT);

        if (OSAL_SUCCESS == eReturnCode)
        {
            printf("SXILLRx: Request log created.\n");

            // Write the header
            OSAL.n32LogWrite(psCtrl->hReqLog, SXILL_RX_REQ_HEADER);
        }
        else
        {
            printf("SXILLRx: ERROR!! creating request log.\n");
        }

    }
#endif

    // Configure RTO of input device (msec). This allows us to
    // suspend on reads for a provided timeout period.
    n32Err = ioctl(psCtrl->psDevice,
        COM_IOCTL_SET_RTO,
        SXILL_RTO_MSEC
            );
    if (0 != n32Err)
    {
        printf("SXILL Error! %s: Failed RTO config!\n\n",
                        pacTaskName);

        // Report a device failure
        OSAL.vReportError(OSAL_ERROR_TASK_MONITOR_DEVICE_DRIVER_FAILURE);

        SXILL_vRxFrameUnInit(hProtocol, psCtrl);
        return FALSE;
    }

    // Inform TxRx task of our control structure so they can
    // uninitialize it for us.
    psTxRxCtrl->psRxCtrl = psCtrl;

    // Replace the provided Protocol Data Ptr (from TxRx) with
    // our own Rx data structure.
    *ppvProtocolData = psCtrl;

    return TRUE;
}

/*****************************************************************************
 *
 *   SXILL_vRxFrameUnInit
 *
 *   _vUninit
 *
 *   This Receiver Interface method is invoked when the STI requires the
 *   protocol specified to be uninstalled, or un-initialized which occurs when
 *   no more connections to it exist. The Protocol Driver implementation should
 *   use this function to release any allocated persistent resources
 *   (i.e. memory) required to run the protocol. A pointer to any persistent
 *   data returned during the protocol initialization is provided by the STI
 *   Framework and needs to be cast to the appropriate type required by the
 *   specific Protocol Driver. If a Protocol Driver does not require a Receiver
 *   or an un-initialization step, this member may be assigned to NULL.
 *
 *   Inputs:
 *   	hProtocol - The STI Protocol Handle assigned to this protocol instance.
 *   	pvProtocolData - A pointer to any Protocol Driver specific persistent
 *   		data being maintained by the STI Framework. This pointer must be
 *   		cast locally to the appropriate type by the Protocol Driver
 *   		implementation.
 *
 *   Outputs:
 *   	None.
 *
 *****************************************************************************/
static STI_PROTOCOL_UNINIT_HANDLER(SXILL_vRxFrameUnInit)
{
    SXILL_RX_FRAME_CONTROL_STRUCT *psRxCtrl =
        (SXILL_RX_FRAME_CONTROL_STRUCT *) pvProtocolData;

    if (NULL == psRxCtrl)
    {
        return;
    }

    printf("SXILL: SXILL_vRxFrameUnInit()\n");

#ifdef SXILL_RX_STATS_OUTPUT
    // close our stats logs
    OSAL.eLogDelete(psRxCtrl->hReqLog);
    psRxCtrl->hReqLog = OSAL_INVALID_OBJECT_HDL;
    OSAL.eLogDelete(psRxCtrl->hLog);
    psRxCtrl->hLog = OSAL_INVALID_OBJECT_HDL;
#endif

    // Free allocated memory
    OSAL.vMemoryFree(psRxCtrl);

    // Leave the rest to the TxRx thread when it is done.
    pvProtocolData = NULL;

    return;
}

/*****************************************************************************
 *
 *   SXILL_n32RxFrameHdlr
 *
 *   This function's only purpose in life is to receive complete frames from
 *   the input device and check the frame validity (Checksum).   It is up to
 *   the Transciever task (TxRx) to actually decide what it wants to do with
 *   this frame. It is then up to the application to handle any errors or
 *   retransmissions since the SXILL doesn't handle that.  It just supports
 *   an end-to-end protocol.
 *
 *   SXILL_n32RxFrameHdlr
 *
 *   This Receiver Interface method is invoked when the STI Receiver Task
 *   becomes ready to run. The STI Framework will continually call this
 *   handler in an infinite loop until something other than
 *   OSAL_TASK_REPORT_NO_ERROR is returned. As long as no fatal errors exist
 *   in the Protocol Driver, the handler should always return
 *   OSAL_TASK_REPORT_NO_ERROR. Otherwise if an error occurs and the Protocol
 *   needs to shutdown, the Protocol Driver should report some Protocol
 *   Specific error code not equal to OSAL_TASK_REPORT_NO_ERROR.
 *   If a Protocol Driver does not require a Receiver this member may be
 *   assigned to NULL.
 *
 *   Inputs:
 *   	hProtocol - The STI Protocol Handle assigned to this protocol instance.
 *   	pvProtocolData - A pointer to any Protocol Driver specific persistent
 *   	data being maintained by the STI Framework. This pointer must be cast
 *   	locally to the appropriate type by the Protocol Driver implementation.
 *
 *   Outputs:
 *   	OSAL_TASK_REPORT_NO_ERROR if no error. Otherwise any.sValue not equal
 *   	to OSAL_TASK_REPORT_NO_ERROR.
 *
 *****************************************************************************/
static STI_PROTOCOL_RX_HANDLER(SXILL_n32RxFrameHdlr)
{
    size_t tNumRead, tNumToRead;
#ifndef SXILL_USE_LEGACY_RING_BUFFER
    UN8 *pun8In;
#endif
    N32 n32Error;
    SXILL_RX_FRAME_CONTROL_STRUCT *psRxFrameCtrl =
        (SXILL_RX_FRAME_CONTROL_STRUCT *) pvProtocolData;
    SXILL_RING_BUFFER_STRUCT *psRingBuffer = &psRxFrameCtrl->sRingBuffer;

    puts("SXILL: SXILL_n32RxFrameHdlr()");

    // Assign protocol handler to local structure
    psRxFrameCtrl->hProtocol = hProtocol;

#ifdef SXILL_RX_STATS_OUTPUT
    vWriteRxStats(psRxFrameCtrl);
#endif

    // Determine how far it is to the end of the ring buffer
    // linearly, with no wrapping. This is the maximum number of bytes
    // I can receive at one time.
#ifdef SXILL_USE_LEGACY_RING_BUFFER
    tNumToRead = NUM_INGRESS_BYTES_AVAILABLE;
#else
    SXILL_RB_vSolidUnusedBlock(&psRingBuffer->sMaster,
                               &pun8In, &tNumToRead);
#endif

    // Check if we have any space in the buffer to read anything into.
    // If not, the only thing we can do is reset the ring buffer, reset
    // the sync-state machine and try again.
    if (0 == tNumToRead)
    {
        // Woah! Slow down.
        puts("SXILL: Reinitialize buffers due to overflow");

#ifdef SXILL_RX_STATS_OUTPUT
        // Update statistics
        psRxFrameCtrl->sStatistics.sRxBytes.un32Overflow++;
#endif

        // Initialize rx frame buffer
        vInitRingBuffer(psRingBuffer);

        // Reset state machine
        vResetSyncStateMachine(&psRxFrameCtrl->sContext);
#ifdef SXILL_USE_LEGACY_RING_BUFFER
        tNumToRead = NUM_INGRESS_BYTES_AVAILABLE;
#else
        SXILL_RB_vSolidUnusedBlock(&psRingBuffer->sMaster,
                                   &pun8In, &tNumToRead);
#endif
    }

    // Read into a local buffer a chunk of data from the input
    // device. This chunk of data will be processed in it's entirety
    // before looping back to acquire more data.
#ifdef SXILL_USE_LEGACY_RING_BUFFER
    tNumRead = fread(psRingBuffer->pun8In, sizeof(UN8), tNumToRead,
        psRxFrameCtrl->psDevice);
#else
    tNumRead = fread(pun8In, sizeof(UN8), tNumToRead,
        psRxFrameCtrl->psDevice);
#endif

    // check if error was detected
    n32Error = ferror(psRxFrameCtrl->psDevice);
    if (0 != n32Error)
    {
#ifdef SXILL_RX_STATS_OUTPUT
        // A read error was detected
        psRxFrameCtrl->sStatistics.un32Errors++;
#endif
        // Update OSAL monitor status
        OSAL.vReportError(OSAL_ERROR_TASK_MONITOR_DEVICE_DRIVER_FAILURE);

        // Post event
        STIP_bPostOOB(
            psRxFrameCtrl->hProtocol,
            (void *)SXILL_PAYLOAD_TYPE_CONTROL,
            FALSE,
            STI_OOB_TYPE_LINK_ERROR,
            (void *)SXILL_RX_ERROR_DEVICE
                );

        // Bug 1501: In case of UART failure the Application has to be notified
        //           about this. However, due to Rx-task high priority the
        //           OS doesn't provide time to lower tasks making Rx one 
        //           working w/o interruptions. In the same time we cannot
        //           use yeild() function since it can be implemented in various
        //           ways where in some cases no one of lower priority threads
        //           will get rights to run.
        OSAL.eTaskDelay(3000);

        // Uninstall Rx Framer.
        return !OSAL_TASK_REPORT_NO_ERROR;
    }

#ifdef SXILL_RX_STATS_OUTPUT
    if (tNumRead > tNumToRead)
    {
        // Error! A driver error was detected
        psRxFrameCtrl->sStatistics.un32ReadExceedsRequest++;
    }
#endif

    // Truncate any read request to the maximum requested.
    tNumRead = MIN(tNumRead, tNumToRead);

#ifdef SXILL_RX_STATS_OUTPUT
    vWriteRxReq(psRxFrameCtrl->hReqLog, tNumToRead, tNumRead);

    // Update statistics with the number of link layer
    // byte received. In this case all bytes received are link layer.
    psRxFrameCtrl->sStatistics.sRxBytes.un32LinkLayer += tNumRead;
#endif

    printf("SXILL: Rx %u bytes...\n", tNumRead);
#ifdef SXILL_USE_LEGACY_RING_BUFFER
    OSAL.vDump(stdout, 0, psRingBuffer->pun8In, tNumRead);
#else
    OSAL.vDump(stdout, 0, pun8In, tNumRead);
#endif

    // Update rx activity
    if(0 == tNumRead)
    {
        psRxFrameCtrl->un32NoRxCount++;
        if(SXILL_NO_RX_LIMIT <= psRxFrameCtrl->un32NoRxCount)
        {
            BOOLEAN bPostedEvent;

            // Post event
            bPostedEvent = STIP_bPostOOB(
                psRxFrameCtrl->hProtocol,
                (void *)SXILL_PAYLOAD_TYPE_CONTROL,
                FALSE,
                STI_OOB_TYPE_LINK_ERROR,
                (void *)SXILL_RX_ERROR_NO_RX
                    );
            if(TRUE == bPostedEvent)
            {
                // Reset counter
                psRxFrameCtrl->un32NoRxCount = 0;
            }
        }
    }
    else // any activity resets the counter
    {
        // Reset counter
        psRxFrameCtrl->un32NoRxCount = 0;
    }

    // Adjust input ring-buffer pointers per read request results and
    // add these bytes to the number of bytes which remain unprocessed.
#ifdef SXILL_USE_LEGACY_RING_BUFFER
    MOVE_INGRESS(&tNumRead);
#else
    if (tNumRead != 0)
    {
        // Commit added read data to let state machine process it
        SXILL_RB_tCommit(&psRingBuffer->sMaster, tNumRead);

        // Update progress stream just in case by the same number of bytes
        // added to the master stream
        SXILL_RB_tCommit(&psRingBuffer->sProgress, tNumRead);
    }
#endif

    // Call Sync State Machine, receive frames, payloads and confirm
    // them as necessary.
    vSyncStateMachine(psRxFrameCtrl);

    // Everything has been processed which could be, move on.

#ifdef SXILL_USE_LEGACY_RING_BUFFER
#ifdef SXILL_RX_STATS_OUTPUT
    // Is that true? Let's check
    if (0 < psRingBuffer->tUnprocessed)
    {
        // Error!
        psRxFrameCtrl->sStatistics.un32NumUnprocessedBytesIsNotZero++;
        vWriteRingBufferState(__LINE__, SXILL_RING_BUFFER_UNPROCESSED_NOT_ZERO,
            psRingBuffer, psRingBuffer->tUnprocessed);
    }
#endif
#endif
    return OSAL_TASK_REPORT_NO_ERROR;
}

/*****************************************************************************
 *
 *   SXILL_bTxPayload
 *
 *  A direct SXi-LL API to tx a payload to the device.
 *
 *****************************************************************************/
BOOLEAN SXILL_bTxPayload (
    STI_PROTOCOL_HDL hProtocol,
    SXILL_PAYLOAD_TYPE tPayloadType,
    OSAL_BUFFER_HDL hPayload
        )
{
    BOOLEAN bSentPayload = FALSE;
    BOOLEAN bSuccess;
    void *pvProtocolData = NULL;

    bSuccess = STIP_bGetProtocolData(hProtocol, &pvProtocolData);
    if(bSuccess == TRUE)
    {
        SXILL_TXRX_CONTROL_STRUCT *psCtrl =
            (SXILL_TXRX_CONTROL_STRUCT *)pvProtocolData;

        if(psCtrl != NULL)
        {
            // Prepare tx frame buffer from payload provided.
            psCtrl->tTxFrameBufferLen =
                tWritePayload(
                    psCtrl->aun8TxFrameBuffer,
                    sizeof(psCtrl->aun8TxFrameBuffer),
                    tPayloadType,
                    hPayload
                        );
            if(psCtrl->tTxFrameBufferLen > 0)
            {
                // At this point we have a frame buffer populated and ready
                // to transmit.

                // Send de-serialized payload as a complete frame
                bSentPayload = bSendFrame(
                    psCtrl,
                    psCtrl->aun8TxFrameBuffer,
                    psCtrl->tTxFrameBufferLen
                        );
            }
#ifdef SXILL_TXRX_STATS_OUTPUT
            else
            {
                // Increment Statistics
                psCtrl->sStatistics.sTx.un32Nothing++;
            }
#endif
        }
    }

    return bSentPayload;
}

// Local Helper/Utility Functions

/*****************************************************************************
 *
 *   vProcessMessage
 *
 *****************************************************************************/
static void vProcessMessage(SXILL_RX_FRAME_CONTROL_STRUCT *psRxFrameCtrl)
{
    BOOLEAN bPostedEvent = FALSE, bPostEvent = TRUE;
    size_t tFrameLength;
#ifndef SXILL_USE_LEGACY_RING_BUFFER
    size_t tFrameLengthRemains,
           tNumWritten1 = 0, tNumWritten2 = 0,
           tOut1Size = 0, tOut2Size = 0;
    UN8 *pun8Out1 = (UN8*)NULL,
        *pun8Out2 = (UN8*)NULL;
#endif
    OSAL_BUFFER_HDL hRxFrame;
    SXILL_SYNC_STATE_MACHINE_CONTEXT_STRUCT *psContext =
                    &psRxFrameCtrl->sContext;
    SXILL_RING_BUFFER_STRUCT *psRingBuffer = &psRxFrameCtrl->sRingBuffer;
#ifdef SXILL_USE_LEGACY_RING_BUFFER
    size_t tCheckValueSize = sizeof(SXILL_CHECK_VALUE);
#endif
    // Ok, there we have it, we have synced, have a valid header
    // and we have a complete validated frame

    // Allocate a buffer for an Rx Frame (no waiting)
    hRxFrame = STIP_hAllocateRxPayload(psRxFrameCtrl->hProtocol);
    if (OSAL_INVALID_BUFFER_HDL == hRxFrame)
    {
        // Error! Cannot allocate buffer
        printf("SXILL Error! Cannot allocate rx frame buffer.\n");

#ifdef SXILL_RX_STATS_OUTPUT
        // Update statistics
        psRxFrameCtrl->sStatistics.un32NoMoreBuffers++;
#endif

        return;
    }

    // Copy frame into RxFrame buffer in linear chunks of memory
    // The frame copied into the buffer is the header plus payload
    tFrameLength = SXILL_FRAME_HDR_LEN + psContext->tNumPayloadBytes;
#ifdef SXILL_USE_LEGACY_RING_BUFFER
    do
    {
        size_t tLinearSegmentLength;
        size_t tNumWritten;

        // Determine how much we can linearly work on at one time.
        // This may include bytes which are NOT part of this frame.
        tLinearSegmentLength = NUM_EGRESS_BYTES_AVAILABLE;

        // Now that we know how much we can process in one
        // linear access, we truncate this number by the number
        // of payload bytes remaining to process.
        tLinearSegmentLength = MIN(tFrameLength, tLinearSegmentLength);

        // Write what we can, as much as we can as the entire
        // request might not be granted (i.e. buffer space is low).
        // Any remaining data to write will get handled next time.
        tNumWritten = OSAL.tBufferWriteTail(hRxFrame, psRingBuffer->pun8Out,
            tLinearSegmentLength);

        // Move ring buffer number of bytes written out
        MOVE_EGRESS(&tNumWritten);

        // Reduce number of frame bytes remaining to copy
        tFrameLength -= tNumWritten;

        // At this point we only care if NOTHING was written.
        // Otherwise we just keep trying.
        if (0 == tNumWritten)
        {

#ifdef SXILL_RX_STATS_OUTPUT
            if (0 == tLinearSegmentLength)
            {
                // Update statistics, we requested to write nothing.
                psRxFrameCtrl->sStatistics.un32WroteNothingToBuffer++;
                if (SXILL_ILOOP_STAGE_OK == psRxFrameCtrl->sStatistics.eILoopStage)
                {
                    psRxFrameCtrl->sStatistics.eILoopStage =
                        SXILL_ILOOP_DETECTED_AND_AVERTED;
                    vWriteRingBufferState(__LINE__, SXILL_RING_BUFFER_LINEAR_SEGMENT_IS_ZERO,
                        psRingBuffer, tLinearSegmentLength);
                }
            }
            else
            {
                // Update statistics, wanted to write something but
                // we could not.
                psRxFrameCtrl->sStatistics.un32BufferOverflow++;
            }
#endif

            // Error! Cannot write to buffer.
            bPostEvent = FALSE;

            // Consume everything remaining about this frame

            // Move ring buffer the number of bytes remaining in this frame
            MOVE_EGRESS(&tFrameLength);

            printf("SXILL Error! Unable to write to buffer.\n");

            // We cannot continue
            break;
        }

    } while (0 < tFrameLength);

    // Consume check.sValue
    MOVE_EGRESS(&tCheckValueSize);
#else
    tFrameLengthRemains = tFrameLength;

    // Determine how much we can linearly work on at one time.
    // This may include bytes which are NOT part of this frame.
    SXILL_RB_vSolidUsedBlocks(&psRingBuffer->sMaster,
                    &pun8Out1, &tOut1Size, &pun8Out2, &tOut2Size);

    // Now that we know how much we can process in one
    // linear access, we truncate this number by the number
    // of payload bytes remaining to process.
    tOut1Size = MIN(tOut1Size, tFrameLengthRemains);

    tNumWritten1 = OSAL.tBufferWriteTail(hRxFrame, pun8Out1, tOut1Size);
    tFrameLengthRemains -= tNumWritten1;
    if (tFrameLengthRemains > 0)
    {
        tOut2Size = MIN(tOut2Size, tFrameLengthRemains);
        tNumWritten2 = OSAL.tBufferWriteTail(hRxFrame, pun8Out2, tOut2Size);
        tFrameLengthRemains -= tNumWritten2;
    }

    if ((tFrameLengthRemains != 0) &&
        ((tNumWritten1 == 0) || (tNumWritten2 == 0)))
    {
#ifdef SXILL_RX_STATS_OUTPUT
        if ((tOut1Size == 0) || (tOut2Size == 0))
        {
            // Update statistics, we requested to write nothing.
            psRxFrameCtrl->sStatistics.un32WroteNothingToBuffer++;
            if (SXILL_ILOOP_STAGE_OK == psRxFrameCtrl->sStatistics.eILoopStage)
            {
                psRxFrameCtrl->sStatistics.eILoopStage =
                    SXILL_ILOOP_DETECTED_AND_AVERTED;
                vWriteRingBufferState(__LINE__, SXILL_RING_BUFFER_LINEAR_SEGMENT_IS_ZERO,
                    psRingBuffer, tOut1Size);
                vWriteRingBufferState(__LINE__, SXILL_RING_BUFFER_LINEAR_SEGMENT_IS_ZERO,
                    psRingBuffer, tOut2Size);
            }
        }
        else
        {
            // Update statistics, wanted to write something but
            // we could not.
            psRxFrameCtrl->sStatistics.un32BufferOverflow++;
        }
#endif
        // Error! Cannot write to buffer.
        bPostEvent = FALSE;

        // Consume everything remaining about this frame

        printf("SXILL Error! Unable to write to buffer.\n");
    }

    // Pass through whole frame
    (void) SXILL_RB_tSkip(&psRingBuffer->sMaster,
                          tFrameLength + sizeof(SXILL_CHECK_VALUE));
#endif

    // Check for link-layer frames, if so process them before
    // passing along to the transceiver.
    if (SXILL_PAYLOAD_TYPE_LINK ==
        psRxFrameCtrl->sContext.sFrameHeader.uFormat.sHeader.tPayloadType)
    {
        // Let link layer processor decide if we are
        // going to post or absorb this message.
        vProcessLinkLayerMessages(psRxFrameCtrl, hRxFrame, &bPostEvent);
    }

    // Post event?
    if (TRUE == bPostEvent)
    {
        // Post Frame
        bPostedEvent = STIP_bReceivePayload(
            psRxFrameCtrl->hProtocol, FALSE, hRxFrame);
        if (FALSE == bPostedEvent)
        {
            printf("SXILL Error! Unable to post frame event.\n");

#ifdef SXILL_RX_STATS_OUTPUT
            psRxFrameCtrl->sStatistics.un32CannotPostEvent++;
#endif
        }
#ifdef SXILL_RX_STATS_OUTPUT
        else
        {
            // We posted a message

            // If the condition occurred and we averted the infinite loop,
            // we want to verify that we are still able to process a message
            // this should indicate the ring buffer is still able to operate.
            //
            if (SXILL_ILOOP_DETECTED_AND_AVERTED ==
                psRxFrameCtrl->sStatistics.eILoopStage)
            {
                psRxFrameCtrl->sStatistics.eILoopStage =
                    SXILL_ILOOP_MESSAGES_PROCESSED_OK;
            }
        }
#endif

    }

    // Event posted?
    if (FALSE == bPostedEvent)
    {
        // Forget about everything at this point

        // Release rx frame
        OSAL.eBufferFree(hRxFrame);
    }

    return;
}

/*****************************************************************************
 *
 *   vProcessLinkLayerMessages
 *
 *****************************************************************************/
static void vProcessLinkLayerMessages(
    SXILL_RX_FRAME_CONTROL_STRUCT *psRxFrameCtrl,
    OSAL_BUFFER_HDL hRxFrame,
    BOOLEAN *pbPostEvent
        )
{
    BOOLEAN bSuccess;
    UN16 un16OpCode;

    // Peek into rx-frame enough to get the opcode so we can figure
    // out what to do with it.
    bSuccess = SXIAPI_bPeekUint16(
        hRxFrame, &un16OpCode, sizeof(SXILL_FRAME_HDR_STRUCT));
    if (TRUE == bSuccess)
    {
        SXIAPI_OPCODE tOpcode = (SXIAPI_OPCODE)un16OpCode;

        // Check for keep-alive
        if (SXIAPI_MESSOP_UARTLINKKEEPALIVE == tOpcode)
        {
#ifdef SXILL_RX_STATS_OUTPUT
            // Keep track of the keep-alives we see.
            psRxFrameCtrl->sStatistics.un32KeepAlives++;
#endif
            // Do not post this, it is a waste of time.
            *pbPostEvent = FALSE;
        }
    }

    return;
}

/*****************************************************************************
 *
 *   vResetSyncStateMachine
 *
 *****************************************************************************/
static void vResetSyncStateMachine(
    SXILL_SYNC_STATE_MACHINE_CONTEXT_STRUCT *psContext)
{
    puts("SXILL: vResetSyncStateMachine()");

    psContext->eSyncState = SXILL_SYNC_STATE_SEARCHING;

#ifdef SXILL_USE_LEGACY_RING_BUFFER
    psContext->eSyncDetectedState = SXILL_SYNC_STATE_HEADER;

    psContext->pun8CurrentSyncByte = gaun8SyncBytes;
    psContext->pun8LastSyncByte = gaun8SyncBytes + sizeof(gaun8SyncBytes) - 1;

    psContext->sFrameHeader.uFormat.sHeader.tPayloadType = 0;
    psContext->sFrameHeader.uFormat.sHeader.tSequenceNumber = 0;
    psContext->sFrameHeader.uFormat.sHeader.tNumPayloadBytes = 0;
    psContext->pun8CurrentHeaderByte = psContext->sFrameHeader.uFormat.sData.aun8Bytes;
    psContext->pun8LastHeaderByte = psContext->sFrameHeader.uFormat.sData.aun8Bytes
                    + sizeof(psContext->sFrameHeader.uFormat.sData.aun8Bytes) - 1;

    psContext->tNumPayloadBytes = 0;
    psContext->tNumPayloadBytesProcessed = 0;

    psContext->tComputedCheckValue = 0;
    psContext->sCheckValue.uFormat.sValue.tCheckValue = 0;
    psContext->pun8CurrentCheckValueByte = psContext->sCheckValue.uFormat.sData.aun8Bytes;
    psContext->pun8LastCheckValueByte = psContext->sCheckValue.uFormat.sData.aun8Bytes
                    + sizeof(psContext->sCheckValue.uFormat.sData.aun8Bytes) - 1;
#else
    psContext->tSync = SXILL_SYNC_BYTE_NONE;

    psContext->sFrameHeader.uFormat.sHeader.tPayloadType = 0;
    psContext->sFrameHeader.uFormat.sHeader.tSequenceNumber = 0;
    psContext->sFrameHeader.uFormat.sHeader.tNumPayloadBytes = 0;
    psContext->tFrameHeaderSize = 0;

    psContext->tNumPayloadBytes = 0;

    psContext->tComputedCheckValue = 0;
    psContext->sCheckValue.uFormat.sValue.tCheckValue = 0;
    psContext->tCheckValueSize = 0;
#endif
    return;
}

/*****************************************************************************
 *
 *   vSyncStateMachine
 *
 *****************************************************************************/
static void vSyncStateMachine(SXILL_RX_FRAME_CONTROL_STRUCT *psRxFrameCtrl)
{
#ifdef SXILL_USE_LEGACY_RING_BUFFER
    SXILL_SYNC_STATE_MACHINE_CONTEXT_STRUCT *psContext =
                    &psRxFrameCtrl->sContext;
    SXILL_RING_BUFFER_STRUCT *psRingBuffer = &psRxFrameCtrl->sRingBuffer;

    // For each byte we have still to process, keep running.
    while (0 < psRingBuffer->tUnprocessed)
    {
        // Based on our current state, handle the input
        switch (psContext->eSyncState)
        {
            case SXILL_SYNC_STATE_SEARCHING:
            {
                UN8 un8Data;
                BOOLEAN bRepeat;
                size_t tNumToProcess = sizeof(*psRingBuffer->pun8Out);

                // Extract data from ring buffer
                un8Data = *psRingBuffer->pun8Out;

                // Decrement the number of bytes we processed.
                MOVE_IN_PROCESS(&tNumToProcess);

                // Remove these bytes
                MOVE_EGRESS(&tNumToProcess);

                do
                {
                    // No need to repeat from here
                    bRepeat = FALSE;

                    // Check for SYNC
                    if (*psContext->pun8CurrentSyncByte == un8Data)
                    {
                        // Do we have more sync bytes to get?
                        if (psContext->pun8LastSyncByte
                                        > psContext->pun8CurrentSyncByte)
                        {
                            // Don't have them all yet. Increment to next sync byte
                            psContext->pun8CurrentSyncByte++;
                        }
                        else
                        {
                            // At this point, we are SYNC'd and have a place to start
                            // building a complete frame. We will not manipulate our
                            // ring buffer out pointer again until we have an error
                            // or complete the frame. This is because we may need to
                            // resync at some time in the future.

                            // Move on to next state
                            psContext->eSyncState = SXILL_SYNC_STATE_DETECTED;
                            psContext->eSyncDetectedState = SXILL_SYNC_STATE_HEADER;
                        }
                    }
                    else
                    {
#ifdef SXILL_RX_STATS_OUTPUT
                        // Update statistics
                        psRxFrameCtrl->sStatistics.un32OutOfSync++;
#endif
                        if (psContext->pun8CurrentSyncByte != gaun8SyncBytes)
                        {
                            // Sorry no sync yet, reset to beginning
                            psContext->pun8CurrentSyncByte = gaun8SyncBytes;

                            // Check this byte again
                            bRepeat = TRUE;

                            puts("SXILL: Repeat sync check to the same "
                                        "byte after reset");
                        }
                    }
                } while (bRepeat == TRUE);
            }
            break;
            case SXILL_SYNC_STATE_DETECTED:
            {
                // Now that we have detected SYNC, we now need to process
                // the frame header, payload and check.

                switch (psContext->eSyncDetectedState)
                {
                    case SXILL_SYNC_STATE_HEADER:
                    {
                        size_t tNumToProcess =
                            sizeof(*psRingBuffer->pun8InProcess);

                        // Extract header byte to local header
                        *psContext->pun8CurrentHeaderByte =
                                        *psRingBuffer->pun8InProcess;

                        // Decrement number of bytes unprocessed
                        MOVE_IN_PROCESS(&tNumToProcess);

                        // Do we have more header bytes to get?
                        if (psContext->pun8LastHeaderByte
                                        > psContext->pun8CurrentHeaderByte)
                        {
                            // No complete header yet
                            psContext->pun8CurrentHeaderByte++;
                        }
                        else
                        {
                            SXILL_PAYLOAD_TYPE tPayloadType;

                            // Process the header

                            // Mask off non-type bits
                            psContext->sFrameHeader.uFormat.sHeader.tPayloadType
                                &= SXILL_PAYLOAD_TYPE_MASK;
                            tPayloadType =
                                (SXILL_PAYLOAD_TYPE)psContext->sFrameHeader.uFormat.sHeader.tPayloadType;

                            // Verify type is something we care about
                            if (FALSE == gasPayloadTable[tPayloadType].bProcess)
                            {
                                // Error! Ignored payload type
                                printf("SXILL Error! Ignored payload type: %X\n", tPayloadType);
#ifdef SXILL_RX_STATS_OUTPUT
                                // Update statistics
                                psRxFrameCtrl->sStatistics.un32IgnoredFrames++;
#endif

                                // Reset for next frame
                                vResetSyncStateMachine(psContext);

                                RESET_RING_BUFFER(psRingBuffer);

                                break;
                            }

                            // Translate length MODULE to HOST
                            psContext->tNumPayloadBytes =
                                            SXILL_LEN_MODULE_TO_HOST(psContext->sFrameHeader.uFormat.sHeader.tNumPayloadBytes);

                            // Verify payload length based on payload type
                            if (!((psContext->tNumPayloadBytes
                                            >= gasPayloadTable[tPayloadType].un16NumPayloadBytesMin)
                                            && (psContext->tNumPayloadBytes
                                                            <= gasPayloadTable[tPayloadType].un16NumPayloadBytesMax)))
                            {
                                // Error! Bad payload length
                                printf("SXILL Error! Incorrect payload length.\n");

#ifdef SXILL_RX_STATS_OUTPUT
                                // Update statistics
                                psRxFrameCtrl->sStatistics.un32InvalidLength++;
#endif

                                // Reset for next frame
                                vResetSyncStateMachine(psContext);

                                RESET_RING_BUFFER(psRingBuffer);
                                break;
                            }

                            // Compute header check.sValue
                            psContext->tComputedCheckValue =
                                            (SXILL_CHECK_VALUE) un16CalculateCheckValue(
                                                SXILL_INITIAL_CHECK_VALUE,
                                                psContext->sFrameHeader.uFormat.sData.aun8Bytes,
                                                sizeof(psContext->sFrameHeader.uFormat.sData.aun8Bytes));

                            // Move to payload state
                            psContext->eSyncDetectedState =
                                            SXILL_SYNC_STATE_PAYLOAD;

                            puts("SXILL: Sync detected. Moved to STATE_DETECTED(PAYLOAD)");
                        }
                    }
                    break;
                    case SXILL_SYNC_STATE_PAYLOAD:
                    {
                        size_t tNumPayloadBytesRemaining =
                                        psContext->tNumPayloadBytes
                                                        - psContext->tNumPayloadBytesProcessed;

                        do
                        {
                            size_t tLinearSegmentLength;
                            size_t tNumToProcess;

                            // Any data remaining is potentially part of this frame.
                            // Determine how much we can linearly work on at one time.
                            // This may include bytes which are NOT part of this frame.
                            tLinearSegmentLength =
                                            NUM_IN_PROCESS_BYTES_AVAILABLE;

                            // Now that we know how much we can process in one
                            // linear access, we truncate this number by the number
                            // of payload bytes expected.
                            tNumToProcess =
                                            MIN(tNumPayloadBytesRemaining, tLinearSegmentLength);

                            // Compute payload check.sValue
                            psContext->tComputedCheckValue =
                                            (SXILL_CHECK_VALUE) un16CalculateCheckValue(
                                                psContext->tComputedCheckValue,
                                                psRingBuffer->pun8InProcess,
                                                tNumToProcess);

                            // Decrement number of bytes unprocessed by the
                            // number which has been processed.
                            MOVE_IN_PROCESS(&tNumToProcess);

                            // Reduce number of payload bytes remaining
                            tNumPayloadBytesRemaining -= tNumToProcess;

                            // Increment length of current frame processed
                            psContext->tNumPayloadBytesProcessed +=
                                            tNumToProcess;

                        } while ((0 < tNumPayloadBytesRemaining)
                                        && (psRingBuffer->tUnprocessed));

                        // Check to see if we have all the payload bytes required
                        if (0 == tNumPayloadBytesRemaining)
                        {
                            // Move to check.sValue state
                            psContext->eSyncDetectedState =
                                            SXILL_SYNC_STATE_CHECK_VALUE;
                            puts("SXILL: Sync detected. Moved to STATE_DETECTED(CHECK_VALUE)");
                        }
                    }
                    break;
                    case SXILL_SYNC_STATE_CHECK_VALUE:
                    {
                        size_t tNumToProcess =
                            sizeof(*psRingBuffer->pun8InProcess);

                        // Extract check.sValue byte to local header
                        *psContext->pun8CurrentCheckValueByte =
                                        *psRingBuffer->pun8InProcess;

                        // Decrement number of bytes unprocessed
                        MOVE_IN_PROCESS(&tNumToProcess);

                        // Do we have more check.sValue bytes to get?
                        if (psContext->pun8LastCheckValueByte
                                        > psContext->pun8CurrentCheckValueByte)
                        {
                            // No complete check.sValue yet
                            psContext->pun8CurrentCheckValueByte++;
                        }
                        else
                        {
                            SXILL_CHECK_VALUE tReceivedCheckValue;

                            // Translate check.sValue MODULE to HOST
                            tReceivedCheckValue =
                                            SXILL_CHECK_VALUE_MODULE_TO_HOST(psContext->sCheckValue.uFormat.sValue.tCheckValue);

                            // Verify check.sValue
                            if (psContext->tComputedCheckValue
                                            == tReceivedCheckValue)
                            {
                                // Check.sValue is correct!

#ifdef SXILL_RX_STATS_OUTPUT
                                // Update statistics
                                psRxFrameCtrl->sStatistics.un32MsgFrames++;
                                psRxFrameCtrl->sStatistics.sRxBytes.un32Payload +=
                                                psContext->tNumPayloadBytes;
#endif

                                // Indicate message is ready
                                vProcessMessage(psRxFrameCtrl);

                                // Reset for next frame
                                vResetSyncStateMachine(psContext);
                            }
                            else
                            {
                                // Error! Bad check.sValue
                                printf("SXILL Error! Incorrect check: %04X != %04X\n",
                                    psContext->tComputedCheckValue, tReceivedCheckValue);

#ifdef SXILL_RX_STATS_OUTPUT
                                // Update statistics
                                psRxFrameCtrl->sStatistics.un32ChecksumFailures++;
#endif

                                // Reset for next frame
                                vResetSyncStateMachine(psContext);

                                RESET_RING_BUFFER(psRingBuffer);
                                break;
                            }
                        }
                    }
                    break;
                    default:
                        psContext->eSyncDetectedState = SXILL_SYNC_STATE_HEADER;
                    break;
                } // eSyncDetectedState
            } // SXILL_SYNC_STATE_DETECTED
            break;
            default:
                psContext->eSyncState = SXILL_SYNC_STATE_SEARCHING;
            break;
        } // eSyncState
    } // while

#else
    SXILL_SYNC_STATE_MACHINE_CONTEXT_STRUCT *psContext =
                    &psRxFrameCtrl->sContext;
    SXILL_RING_BUFFER_STRUCT *psRingBuffer = &psRxFrameCtrl->sRingBuffer;
    BOOLEAN bContinue = TRUE;

    // For each byte we have still to process, keep running.
    while (bContinue == TRUE)
    {
        // Based on our current state, handle the input
        switch (psContext->eSyncState)
        {
            case SXILL_SYNC_STATE_RESET:
            {
                printf("SXILL: Dropped: Type %d, SeqId %d, Len %d, (M: %d, *%d), (P: %d, *%d)\n",
                    psContext->sFrameHeader.uFormat.sHeader.tPayloadType,
                    psContext->sFrameHeader.uFormat.sHeader.tSequenceNumber,
                    SXILL_LEN_MODULE_TO_HOST(psContext->sFrameHeader.uFormat.sHeader.tNumPayloadBytes),
                    psRingBuffer->sMaster.tStart, psRingBuffer->sMaster.tSize,
                    psRingBuffer->sProgress.tStart, psRingBuffer->sProgress.tSize);
            }
            /* Break not needed */
            case SXILL_SYNC_STATE_RESET_AFTER_READY:
            {
                // Reset for next frame
                vResetSyncStateMachine(psContext);

                psContext->eSyncState = SXILL_SYNC_STATE_SEARCHING;
                printf("SXILL[%d] Move to state SXILL_SYNC_STATE_SEARCHING\n", __LINE__);
            }
            break;
            case SXILL_SYNC_STATE_SEARCHING:
            {
                UN8 un8Data;
                size_t tSize;

                while (SXILL_RB_tRead(&psRingBuffer->sMaster, &un8Data, sizeof(un8Data)) > 0)
                {
                    if (psContext->tSync == SXILL_SYNC_BYTE0_MASK)
                    {
                        // we've got the first sync byte. Does the new one
                        // is the second sync byte?
                        if (un8Data == SXILL_SYNC_BYTE1)
                        {
                            // Just write the byte
                            psContext->tSync |= SXILL_SYNC_BYTE1_MASK;

                            // Prepare the second buffer for progress reading
                            SXILL_RB_vAttach(&psRingBuffer->sProgress,
                                             &psRingBuffer->sMaster);

                            // At this point, we are SYNC'd and have a place to start
                            // building a complete frame. We will not manipulate our
                            // ring buffer out pointer again until we have an error
                            // or complete the frame. This is because we may need to
                            // resync at some time in the future.

                            // Move on to next state
                            psContext->eSyncState = SXILL_SYNC_STATE_HEADER;
                            printf("SXILL[%d] Move to state SXILL_SYNC_STATE_HEADER\n",
                                __LINE__);
                            break;
                        }
                        else if (un8Data != SXILL_SYNC_BYTE0)
                        {
#ifdef SXILL_RX_STATS_OUTPUT
                            // Update statistics
                            psRxFrameCtrl->sStatistics.un32OutOfSync++;
#endif
                            // It might happen that we've got several first bytes
                            // back to back. In this case we simple need to read next byte
                            // again. If the second byte after the first sync byte found is
                            // nether first nor second sync bytes we will invalidate mask
                            // to start looking for the first sync byte again
                            psContext->tSync = SXILL_SYNC_BYTE_NONE;
                        }
                    }
                    else if (un8Data == SXILL_SYNC_BYTE0)
                    {
                        // We've got the first sync byte from the stream
                        psContext->tSync = SXILL_SYNC_BYTE0_MASK;
                    }
                }

                // Do we have something to process?
                tSize = SXILL_RB_tSize(&psRingBuffer->sMaster);
                if (tSize == 0)
                {
                    bContinue = FALSE;
                }
            }
            break;
            case SXILL_SYNC_STATE_HEADER:
            {
                size_t tReadSize;

                tReadSize =
                    SXILL_RB_tRead(&psRingBuffer->sProgress,
                        psContext->sFrameHeader.uFormat.sData.aun8Bytes +
                            psContext->tFrameHeaderSize,
                        sizeof(psContext->sFrameHeader) -
                            psContext->tFrameHeaderSize);
                psContext->tFrameHeaderSize += tReadSize;
                if (psContext->tFrameHeaderSize == sizeof(psContext->sFrameHeader))
                {
                    SXILL_PAYLOAD_TYPE tPayloadType;

                    // Process the header

                    // Mask off non-type bits
                    psContext->sFrameHeader.uFormat.sHeader.tPayloadType
                        &= SXILL_PAYLOAD_TYPE_MASK;
                    tPayloadType =
                        (SXILL_PAYLOAD_TYPE)psContext->sFrameHeader.uFormat.sHeader.tPayloadType;

                    printf("SXILL: Found Type %d, SeqId %d, Len %d\n",
                        psContext->sFrameHeader.uFormat.sHeader.tPayloadType,
                        psContext->sFrameHeader.uFormat.sHeader.tSequenceNumber,
                        SXILL_LEN_MODULE_TO_HOST(psContext->sFrameHeader.uFormat.sHeader.tNumPayloadBytes));

                    // Verify type is something we care about
                    if (FALSE == gasPayloadTable[tPayloadType].bProcess)
                    {
                        // Error! Ignored payload type
                        printf("SXILL Error! Ignored payload type: %X\n",
                            (int) tPayloadType);

#ifdef SXILL_RX_STATS_OUTPUT
                        // Update statistics
                        psRxFrameCtrl->sStatistics.un32IgnoredFrames++;
#endif
                        // Reset for next frame
                        psContext->eSyncState = SXILL_SYNC_STATE_RESET;
                        printf("SXILL[%d] Move to state SXILL_SYNC_STATE_RESET\n",
                            __LINE__);
                        break;
                    }

                    // Translate length MODULE to HOST
                    psContext->tNumPayloadBytes =
                        SXILL_LEN_MODULE_TO_HOST(psContext->sFrameHeader.uFormat.sHeader.tNumPayloadBytes);

                    // Verify payload length based on payload type
                    if (!((psContext->tNumPayloadBytes
                                    >= gasPayloadTable[tPayloadType].un16NumPayloadBytesMin)
                                    && (psContext->tNumPayloadBytes
                                                    <= gasPayloadTable[tPayloadType].un16NumPayloadBytesMax)))
                    {
                        // Error! Bad payload length
                        printf("SXILL Error! Incorrect payload length.\n");

#ifdef SXILL_RX_STATS_OUTPUT
                        // Update statistics
                        psRxFrameCtrl->sStatistics.un32InvalidLength++;
#endif

                        // Reset for next frame
                        psContext->eSyncState = SXILL_SYNC_STATE_RESET;
                        break;
                    }

                    // Compute header check.sValue
                    psContext->tComputedCheckValue =
                                    (SXILL_CHECK_VALUE) un16CalculateCheckValue(
                                        SXILL_INITIAL_CHECK_VALUE,
                                        psContext->sFrameHeader.uFormat.sData.aun8Bytes,
                                        sizeof(psContext->sFrameHeader.uFormat.sData.aun8Bytes));

                    // Move to payload state
                    psContext->eSyncState = SXILL_SYNC_STATE_PAYLOAD;
                    printf("SXILL[%d] Move to state SXILL_SYNC_STATE_PAYLOAD\n",
                        __LINE__);
                }
                else
                {
                    bContinue = FALSE;
                }
            }
            break;
            case SXILL_SYNC_STATE_PAYLOAD:
            {
                size_t tPayloadSize;

                tPayloadSize = SXILL_RB_tSize(&psRingBuffer->sProgress);
                if (tPayloadSize >= psContext->tNumPayloadBytes)
                {
                    UN8 *pun8Block1 = NULL, *pun8Block2 = NULL;
                    size_t tBlock1Size = 0, tBlock2Size = 0;
                    size_t tDataToVerify = psContext->tNumPayloadBytes;

                    SXILL_RB_vSolidUsedBlocks(&psRingBuffer->sProgress,
                        &pun8Block1, &tBlock1Size, &pun8Block2, &tBlock2Size);
                    if (tBlock1Size > tDataToVerify)
                    {
                        tBlock1Size = tDataToVerify;
                    }
                    // Compute payload check.sValue
                    psContext->tComputedCheckValue =
                                    (SXILL_CHECK_VALUE) un16CalculateCheckValue(
                                        psContext->tComputedCheckValue,
                                        pun8Block1,
                                        tBlock1Size);
                    tDataToVerify -= tBlock1Size;
                    if (tDataToVerify > 0)
                    {
                        if (tBlock2Size > tDataToVerify)
                        {
                            tBlock2Size = tDataToVerify;
                        }
                        // Compute payload check.sValue
                        psContext->tComputedCheckValue =
                                        (SXILL_CHECK_VALUE) un16CalculateCheckValue(
                                            psContext->tComputedCheckValue,
                                            pun8Block2,
                                            tBlock2Size);
                        tDataToVerify -= tBlock2Size;
                    }

                    if (tDataToVerify == 0)
                    {
                        // Skip payload data via in-progress RB instance
                        SXILL_RB_tSkip(&psRingBuffer->sProgress,
                            psContext->tNumPayloadBytes);

                        // Move to check message by checksum
                        psContext->eSyncState = SXILL_SYNC_STATE_CHECK_VALUE;
                        printf("SXILL[%d] Move to state SXILL_SYNC_STATE_CHECK_VALUE\n",
                            __LINE__);
                    }
                    else
                    {
                        psContext->eSyncState = SXILL_SYNC_STATE_RESET;
                        printf("SXILL[%d] Move to state SXILL_SYNC_STATE_RESET\n",
                            __LINE__);
                    }
                }
                else
                {
                    printf("SXILL: has %d of %d byte(s)\n",
                        tPayloadSize, psContext->tNumPayloadBytes);
                    bContinue = FALSE;
                }
            }
            break;
            case SXILL_SYNC_STATE_CHECK_VALUE:
            {
                size_t tReadSize;

                tReadSize =
                    SXILL_RB_tRead(&psRingBuffer->sProgress,
                        psContext->sCheckValue.uFormat.sData.aun8Bytes +
                            psContext->tCheckValueSize,
                        sizeof(psContext->sCheckValue) -
                            psContext->tCheckValueSize);
                psContext->tCheckValueSize += tReadSize;
                if (psContext->tCheckValueSize == sizeof(psContext->sCheckValue))
                {
                    SXILL_CHECK_VALUE tReceivedCheckValue;

                    // Translate check.sValue MODULE to HOST
                    tReceivedCheckValue =
                        SXILL_CHECK_VALUE_MODULE_TO_HOST(
                            psContext->sCheckValue.uFormat.sValue.tCheckValue);

                    // Verify check.sValue
                    if (psContext->tComputedCheckValue == tReceivedCheckValue)
                    {
                        // Check.sValue is correct!
                        psContext->eSyncState = SXILL_SYNC_STATE_READY;
                        printf("SXILL[%d] Move to state SXILL_SYNC_STATE_READY\n",
                            __LINE__);
                    }
                    else
                    {
                        // Error! Bad check.sValue
                        printf("SXILL Error! Incorrect check: %04X != %04X\n",
                            psContext->tComputedCheckValue, tReceivedCheckValue);

#ifdef SXILL_RX_STATS_OUTPUT
                        // Update statistics
                        psRxFrameCtrl->sStatistics.un32ChecksumFailures++;
#endif
                        psContext->eSyncState = SXILL_SYNC_STATE_RESET;
                        printf("SXILL[%d] Move to state SXILL_SYNC_STATE_RESET\n",
                            __LINE__);
                    }
                }
                else
                {
                    bContinue = FALSE;
                }
            }
            break;
            case SXILL_SYNC_STATE_READY:
            {
#ifdef SXILL_RX_STATS_OUTPUT
                // Update statistics
                psRxFrameCtrl->sStatistics.un32MsgFrames++;
                psRxFrameCtrl->sStatistics.sRxBytes.un32Payload +=
                                psContext->tNumPayloadBytes;
#endif
                printf("SXILL: READY Type %d, SeqId %d, Len %d\n",
                    psContext->sFrameHeader.uFormat.sHeader.tPayloadType,
                    psContext->sFrameHeader.uFormat.sHeader.tSequenceNumber,
                    SXILL_LEN_MODULE_TO_HOST(psContext->sFrameHeader.uFormat.sHeader.tNumPayloadBytes));

                // Indicate message is ready
                vProcessMessage(psRxFrameCtrl);

                // Check.sValue is correct!
                psContext->eSyncState = SXILL_SYNC_STATE_RESET_AFTER_READY;
                printf("SXILL[%d] Move to state SXILL_SYNC_STATE_READY\n",
                    __LINE__);
            }
            break;
            default:
            {
                psContext->eSyncState = SXILL_SYNC_STATE_RESET;
                printf("SXILL[%d] Move to state SXILL_SYNC_STATE_RESET\n",
                    __LINE__);
            }
            break;
        } // eSyncState
    } // while
#endif
    return;
}

/*****************************************************************************
 *
 *   un16CalculateCheckValue
 *
 *****************************************************************************/
/*
 Note that this.sValue is designed so that with a little rewriting and
 static storage of the check.sValue, this function can be used to calculate
 a check.sValue across multiple buffers. This function can also be used to
 calculate the check.sValue on a byte-byte basis.

 Note that this version of Fletcher Check Sum does not match the standards
 proposed in RFC. It adds two features:

 1) An integer of 1 is added to the upper octet for every byte, preventing
 an all zero buffer from resulting in a static check.
 2) The carry bits from each octet are fed into the other to insure that all
 bits have equal "spread" characteristics.
 */
static UN16 un16CalculateCheckValue(UN16 un16CheckValue, void const *pvBuffer,
    UN32 un32Length)
{
    // Use a 32 bit intermediate state to allow carry's to be captured.
    UN32 un32CheckValue = un16CheckValue;
    UN8 *pun8Buffer = (UN8*)pvBuffer;
    UN32 un32Index;

    for (un32Index = 0; un32Index < un32Length; ++un32Index, ++pun8Buffer)
    {
        // Basic XOR Check Sum
        un32CheckValue += *pun8Buffer;
        // Make it a Fletcher Check Value. Add a constant.sValue
        // to insure that the 'all zero' case won't cause the result
        // to stay static.
        // The 0xFF mask insures that the following carry wrap will not
        // XOR the same.sValue back into the current state.
        un32CheckValue += ((un32CheckValue & 0xFF) << 8) + 0x100;
        // Wrap the carry bit(s) back to the bottom of the check.
        // Mask off the carry bits so that this calculation can be
        // interrupted and continued from the interruption point while
        // only storing the 16 bit check.
        un32CheckValue = (un32CheckValue ^ (un32CheckValue >> 16)) & 0xFFFF;
        // At this point, the CheckValue is a valid returnable 16 bit.
        // It could be stored and then used as a seed for a following calculation
        // over a second buffer. The result of this would return the same.sValue as
        // as if the two buffers were concatenated and this function were run over
        // over them once.
    }

    return ((UN16) un32CheckValue);
}

/*****************************************************************************
 *
 *   tWritePayload
 *
 *****************************************************************************/
static size_t tWritePayload(
    UN8 *pun8Dst,
    size_t tSize,
    SXILL_PAYLOAD_TYPE tPayloadType,
    OSAL_BUFFER_HDL hSrcBuffer
        )
{
    size_t tNumWritten = 0;

    if ((pun8Dst != NULL) && (tSize != 0) &&
        (OSAL_INVALID_BUFFER_HDL != hSrcBuffer))
    {
        SXILL_LEN tNumPayloadBytes;
        size_t tComputedFrameLength;

        // Grab the payload length
        tNumPayloadBytes = OSAL.tBufferGetSize(hSrcBuffer);

        // We now know enough to compute the entire length of the frame
        tComputedFrameLength =
            sizeof(SXILL_SYNC) + SXILL_FRAME_LEN(tNumPayloadBytes);

        // Will this frame fit?
        if(tComputedFrameLength < tSize)
        {
            // Note! This is tightly coupled to what is specified in SX-9845-0098
            // version 1.3 Section 2.1 Link Packet Format

            // Write SYNC.sValue
            // See SX-9845-0098 version 1.3 Section 2.1
            *pun8Dst++ = (UN8) SXILL_SYNC_BYTE0;
            *pun8Dst++ = (UN8) SXILL_SYNC_BYTE1;

            // Populate SXi-LL header
            *pun8Dst++ = (UN8) 0; // Nothing for now, populated on tx
            *pun8Dst++ = (UN8) tPayloadType;
            *pun8Dst++ = BYTE1(tNumPayloadBytes);
            *pun8Dst++ = BYTE0(tNumPayloadBytes);

            // Adjust number written (and what will be written)
            tNumWritten = sizeof(SXILL_SYNC) + SXILL_FRAME_LEN(0);

            // Read all data from tx frame(msg) into our local buffer right
            // after the SYNC and SXiLL header (payload).
            tNumWritten += OSAL.tBufferPeek(
                hSrcBuffer, pun8Dst, tNumPayloadBytes, 0);

            // Make sure everything expected to be written, was written
            if(tNumWritten != tComputedFrameLength)
            {
                // It's ok to report nothing was written since this was
                // only to a buffer provided by the caller (i.e. no
                // state information has been impacted).
                tNumWritten = 0;
            }
        }
    }

    return tNumWritten;
}

/*****************************************************************************
 *
 *   bSendFrame
 *
 *   This function takes a populated SXi frame and sends it.
 *   The buffer is expected to contain the SXI frame's payload.
 *
 *   Inputs:
 *      psTxRxCtrl - The transceiver control structure.
 *
 *   Outputs:
 *       None.
 *
 *****************************************************************************/
static BOOLEAN bSendFrame(
    SXILL_TXRX_CONTROL_STRUCT *psTxRxCtrl,
    UN8 *pun8Src,
    size_t tSrcLength
        )
{
    int iResult;
#ifdef SXILL_TXRX_STATS_OUTPUT
    int iSent = 0;
#endif
    SXILL_CHECK_VALUE tCheckValue = 0;
    UN8 *pun8CheckValueStart;
    SXIAPI_LINK_OPCODE tLinkOpcode;
    UN8 un8OpcodeLB, un8OpcodeHB;

    if ((pun8Src == NULL) ||
        (tSrcLength < sizeof(SXILL_SYNC) + SXILL_FRAME_LEN(0)))
    {
        return FALSE;
    }

    // Before we can populate the sequence number, we need
    // to peek into the message we are sending and see if
    // it is a LinkStartCmd. If so, mama always says seq=0.
    un8OpcodeHB = pun8Src[sizeof(SXILL_SYNC) + SXILL_FRAME_HDR_LEN];
    un8OpcodeLB = pun8Src[sizeof(SXILL_SYNC) + SXILL_FRAME_HDR_LEN + 1];
    tLinkOpcode = ((un8OpcodeHB << 8) + un8OpcodeLB);
    if(tLinkOpcode == SXIAPI_MESSOP_UARTLINKSTARTCMD)
    {
        // Reset sequence numbers
        psTxRxCtrl->tTxSeq = 0;
    }

    // Populate sequence number
    pun8Src[sizeof(SXILL_SYNC) + SXILL_SEQ_OFFSET] = psTxRxCtrl->tTxSeq;

    // Compute check.sValue for message over header plus payload
    tCheckValue = un16CalculateCheckValue(
        SXILL_INITIAL_CHECK_VALUE,
        pun8Src + sizeof(SXILL_SYNC), // header starts after sync
        tSrcLength - (sizeof(SXILL_SYNC) + sizeof(SXILL_CHECK_VALUE))
            );

    // Populate check.sValue
    pun8CheckValueStart = pun8Src + (tSrcLength - sizeof(SXILL_CHECK_VALUE));
    *pun8CheckValueStart++ = BYTE1(tCheckValue);
    *pun8CheckValueStart++ = BYTE0(tCheckValue);

    // Transmit entire message
    do
    {
        iResult = fwrite(pun8Src, 1, tSrcLength,
            psTxRxCtrl->psDevice);
        if (0 < iResult)
        {
            // Adjust by number actually sent
            tSrcLength -= iResult;

#ifdef SXILL_TXRX_STATS_OUTPUT
            // Update statistics
            iSent += iResult;
            psTxRxCtrl->sStatistics.sTx.sBytes.un32LinkLayer += iResult;
#endif
        }
    } while ((0 < tSrcLength) && (0 < iResult));

    if (0 <= iResult)
    {
        // Increment the seq number and frames sent since
        // we sent this successfully
        psTxRxCtrl->tTxSeq++;

#ifdef SXILL_TXRX_STATS_OUTPUT
        // Update statistics
        psTxRxCtrl->sStatistics.sTx.un32MsgFrames++;
        psTxRxCtrl->sStatistics.sTx.sBytes.un32Payload += iSent;
#endif
    }
    else
    {
#ifdef SXILL_TXRX_STATS_OUTPUT
        // Update statistics for errors
        psTxRxCtrl->sStatistics.sTx.un32DeviceErrors++;
#endif
    }

    return TRUE;
}

/*****************************************************************************
 *
 *   bExtractFrameHdr
 *
 *  This API extracts the SXI-LL frame header from a received frame.
 *
 *   Inputs:
 *   	hRxFrame - The received frame, containing information for which
 *           when it is parsed identifies a unique connection.
 *      psFrameHdr - A pointer to where to place the parsed frame header.
 *
 *   Outputs:
 *   	TRUE on successful extraction of a header.
 *      Otherwise FALSE on error.
 *
 *****************************************************************************/
static BOOLEAN bExtractFrameHdr(
    OSAL_BUFFER_HDL hRxFrame,
    SXILL_FRAME_HDR_STRUCT *psFrameHdr
        )
{
    BOOLEAN bSuccess;
    size_t tNum;

    // Extract frame header data from Frame...
    tNum = OSAL.tBufferReadHead(hRxFrame, psFrameHdr->uFormat.sData.aun8Bytes,
        sizeof(psFrameHdr->uFormat.sData.aun8Bytes));
    bSuccess = (SXILL_FRAME_HDR_LEN == tNum) ? TRUE : FALSE;

    return bSuccess;
}

#ifdef SXILL_USE_LEGACY_RING_BUFFER
/*****************************************************************************
 *
 *   vAdvanceRingBuffer
 *
 *****************************************************************************/
__INLINE__ static void vAdvanceRingBuffer(
    SXILL_RING_BUFFER_STRUCT *psRingBuffer, size_t tMemberOffset,
    size_t *ptOffset)
{
    UN8 const **ppun8Ptr = (UN8 const * *) (((UN8 const *) psRingBuffer)
                    + tMemberOffset);

    // Check that advance request is within the allowed limits. If it is not
    // go ahead and truncate the request to the maximium offset allowed and then
    // adjust counters as necessary.
    if (offsetof(SXILL_RING_BUFFER_STRUCT, pun8In) == tMemberOffset) // Ingress
    {
        // The offset requested cannot take us beyond the ring buffer size
        if (*ptOffset > psRingBuffer->tRemaining)
        {
            // Truncate!
#ifdef SXILL_RX_STATS_OUTPUT
            psRingBuffer->un32IngressOverflow++;
            vWriteRingBufferState(__LINE__, SXILL_RING_BUFFER_INGRESS_OVERFLOW,
                psRingBuffer, *ptOffset);
#endif
            *ptOffset = psRingBuffer->tRemaining;
        }

        // Advance ingress counters
        psRingBuffer->tCount += *ptOffset;
        psRingBuffer->tRemaining -= *ptOffset;

        // Add to number of bytes unprocessed
        psRingBuffer->tUnprocessed += *ptOffset;
    }
    else if (offsetof(SXILL_RING_BUFFER_STRUCT, pun8Out) == tMemberOffset) // Egress
    {
        // The offset requested cannot take us below an empty buffer
        if (psRingBuffer->tCount < *ptOffset)
        {
            // Truncate!
#ifdef SXILL_RX_STATS_OUTPUT
            psRingBuffer->un32EgressUnderflow++;
            vWriteRingBufferState(__LINE__, SXILL_RING_BUFFER_EGRESS_UNDERFLOW,
                psRingBuffer, *ptOffset);
#endif
            *ptOffset = psRingBuffer->tCount;
        }

        // Advance Egress counters, these bytes are gone
        psRingBuffer->tCount -= *ptOffset;
        psRingBuffer->tRemaining += *ptOffset;
    }
    else if (offsetof(SXILL_RING_BUFFER_STRUCT, pun8InProcess) == tMemberOffset) // In-Process
    {
        // The offset requested cannot take us below what
        // we have yet to process.
        if (psRingBuffer->tUnprocessed < *ptOffset)
        {
            // Truncate!
#ifdef SXILL_RX_STATS_OUTPUT
            psRingBuffer->un32InProcessUnderflow++;
            vWriteRingBufferState(__LINE__, SXILL_RING_BUFFER_INPROCESS_UNDERFLOW,
                psRingBuffer, *ptOffset);
#endif
            *ptOffset = psRingBuffer->tUnprocessed;
        }

        // Remove from number of bytes unprocessed
        psRingBuffer->tUnprocessed -= *ptOffset;

        // The number of unprocessed bytes left cannot be larger
        // than the number of ingress bytes available.
        if (psRingBuffer->tCount < psRingBuffer->tUnprocessed)
        {
            // Truncate!
#ifdef SXILL_RX_STATS_OUTPUT
            psRingBuffer->un32UnprocessedOverflow++;
            vWriteRingBufferState(__LINE__, SXILL_RING_BUFFER_UNPROCESSED_OVERFLOW,
                psRingBuffer, 0);
#endif
            psRingBuffer->tUnprocessed = psRingBuffer->tCount;
        }
    }
    else // Error!
    {
#ifdef SXILL_RX_STATS_OUTPUT
        psRingBuffer->un32IllegalAdvancement++;
        vWriteRingBufferState(__LINE__, SXILL_RING_BUFFER_ILLEGAL_ADVANCEMENT,
            psRingBuffer, tMemberOffset);
#endif
    }

    // Move ring buffer forward
    *ppun8Ptr += *ptOffset;

    // Check for wrap around
    if (psRingBuffer->pun8End <= *ppun8Ptr)
    {
        // Wrap around including anything beyond the end ptr
        *ppun8Ptr = (*ppun8Ptr - psRingBuffer->pun8End)
                        + psRingBuffer->pun8Start;
    }

    return;
}

/*****************************************************************************
 *
 *   tNumToEndOfEgressBytes
 *
 *  This function simply calculated the number of bytes from the current
 *  output/in-process pointer to the end of the ring buffer without wrapping.
 *  which really returns the largest available contiguous memory
 *  segment, but that is quite a mouthful.
 *
 *****************************************************************************/
__INLINE__ static size_t tNumToEndOfEgressBytes(
    SXILL_RING_BUFFER_STRUCT *psRingBuffer, size_t tMemberOffset)
{
    size_t tNumBytes = 0;
    UN8 const **ppun8Ptr = (UN8 const * *) (((UN8 const *) psRingBuffer)
                    + tMemberOffset);

    // Determine number of remaining egress bytes
    // that are in a contiguous memory segment.
    if (*ppun8Ptr < psRingBuffer->pun8In)
    {
        // Compute number of bytes up to the end of the ring buffer
        tNumBytes = psRingBuffer->pun8In - *ppun8Ptr;
    }
    else if ((*ppun8Ptr > psRingBuffer->pun8In) || (0 < psRingBuffer->tCount))
    {
        // Compute number of bytes up to the in pointer
        tNumBytes = psRingBuffer->pun8End - *ppun8Ptr;
    }

    return tNumBytes;
}

/*****************************************************************************
 *
 *   tNumAvailableInRingBuffer
 *
 *  This function simply calculated the number of bytes from the current
 *  input pointer to the end of the ring buffer without wrapping.
 *  which really returns the largest available contiguous memory
 *  segment, but that is quite a mouthful.
 *
 *****************************************************************************/
__INLINE__ static size_t tNumAvailableInRingBuffer(
    SXILL_RING_BUFFER_STRUCT *psRingBuffer)
{
    size_t tNumBytes;

    // If our ring buffer is empty, snap back to the beginning of the buffer
    // to allow larger, more efficient reads, by providing a larger contiguous
    // memory segment rather than just the last place we were.
    if (0 == psRingBuffer->tCount)
    {
        // Snap back to the start
        vInitRingBuffer(psRingBuffer);
        tNumBytes = sizeof(psRingBuffer->aun8Data);
    }
    // Determine the amount of space available in a contiguous
    // chunk of memory.
    else if (psRingBuffer->pun8Out < psRingBuffer->pun8In)
    {
        // Compute number of bytes upto the end of the ring buffer
        tNumBytes = psRingBuffer->pun8End - psRingBuffer->pun8In;
    }
    else
    {
        // Compute number of bytes up to the in pointer
        tNumBytes = psRingBuffer->pun8Out - psRingBuffer->pun8In;
    }

    return tNumBytes;
}
#endif
/*****************************************************************************
 *
 *   vInitRingBuffer
 *
 *****************************************************************************/
static void vInitRingBuffer(SXILL_RING_BUFFER_STRUCT *psRingBuffer)
{
#ifdef SXILL_USE_LEGACY_RING_BUFFER
    psRingBuffer->pun8Start = psRingBuffer->aun8Data;
    psRingBuffer->pun8End = psRingBuffer->aun8Data
                    + sizeof(psRingBuffer->aun8Data);
    psRingBuffer->pun8In = (UN8 *) psRingBuffer->pun8Start;
    psRingBuffer->pun8Out = psRingBuffer->pun8Start;
    psRingBuffer->pun8InProcess = psRingBuffer->pun8Start;
    psRingBuffer->tCount = 0;
    psRingBuffer->tRemaining = sizeof(psRingBuffer->aun8Data);
    psRingBuffer->tUnprocessed = 0;
#else
    SXILL_RB_vInit(&psRingBuffer->sMaster,
        psRingBuffer->aun8Data, sizeof(psRingBuffer->aun8Data));

    SXILL_RB_vAttach(&psRingBuffer->sProgress, &psRingBuffer->sMaster);
#endif
    return;
}

/*****************************************************************************
 *
 *   vHandleRx
 *
 *****************************************************************************/
static void vHandleRx(
    STI_PROTOCOL_HDL hProtocol,
    SXILL_TXRX_CONTROL_STRUCT *psCtrl,
    STI_PROTOCOL_RX_PAYLOAD_STRUCT *psRx
        )
{
    BOOLEAN bOk;
    SXILL_FRAME_HDR_STRUCT sRxFrameHdr;

    puts("SXILL: Rx MSG Frame");

#ifdef SXILL_TXRX_STATS_OUTPUT
    // Increment Statistics
    psCtrl->sStatistics.sRx.un32MsgFrames++;
    psCtrl->sStatistics.sRx.un32MsgBytes +=
        OSAL.tBufferGetSize(psRx->hPayload);
#endif

#if DEBUG_OBJECT != 0
    // Dump the frame
    vDumpFrame(psRx->hPayload);
#endif

    // Extract the frame header
    bOk = bExtractFrameHdr(psRx->hPayload, &sRxFrameHdr);
    if(bOk == TRUE)
    {
        SXILL_PAYLOAD_TYPE tPayloadType =
            (SXILL_PAYLOAD_TYPE)sRxFrameHdr.uFormat.sHeader.tPayloadType;

        // Pass frame to callback
        bOk = STI_bReceivePayload(
            hProtocol, (void *)(size_t)tPayloadType, psRx);
        if (FALSE == bOk)
        {
            // Error! Failed to pass payload to application.
            printf("SXILL Error! Failed to pass rx payload to "
                            "application.\n");

#ifdef SXILL_TXRX_STATS_OUTPUT
            // Increment Statistics
            psCtrl->sStatistics.sRx.un32DroppedFrames++;
#endif
        }
        else
        {
            // No connection ready to accept this frame, this is not
            // an error, just a message may have been received which
            // currently no-one has established a connection for.
            printf("SXILL: Dropped frame due to no active connection.\n");

#ifdef SXILL_TXRX_STATS_OUTPUT
            // Increment Statistics
            psCtrl->sStatistics.sRx.un32DroppedFrames++;
#endif
        }
    }
    else
    {
        // No frame header available.
        printf("SXILL: Dropped frame due to no frame extraction.\n");

#ifdef SXILL_TXRX_STATS_OUTPUT
        // Increment Statistics
        psCtrl->sStatistics.sRx.un32DroppedFrames++;
#endif
    }

    // Discard frame if we still own it
    if(psRx->hPayload != OSAL_INVALID_BUFFER_HDL)
    {
        OSAL.eBufferFree(psRx->hPayload);
        psRx->hPayload = OSAL_INVALID_BUFFER_HDL;
    }

    return;
}

/*****************************************************************************
 *
 *   SXILL_vHandleTx
 *
 *****************************************************************************/
void SXILL_vHandleTx(
    STI_PROTOCOL_HDL hProtocol,
    STI_PROTOCOL_TX_PAYLOAD_STRUCT *psTx
        )
{
#ifdef SXILL_TXRX_STATS_OUTPUT
    SXILL_TXRX_CONTROL_STRUCT *psCtrl = NULL;
    {
        BOOLEAN bSuccess;
        void *pvProtocolData = NULL;

        bSuccess = STIP_bGetProtocolData(hProtocol, &pvProtocolData);
        if(bSuccess == TRUE)
        {
            psCtrl = (SXILL_TXRX_CONTROL_STRUCT *)pvProtocolData;
            if(psCtrl == NULL)
            {
                // Error!
                return;
            }
        }
        else
        {
            // Error!
            return;
        }
    }
#endif

    // Check if a payload was provided.
    if (psTx->hPayload != OSAL_INVALID_BUFFER_HDL)
    {
        SXILL_PAYLOAD_TYPE tPayloadType;
        BOOLEAN bSentFrame;

        // A payload was provided so we must determine the connection
        // type and write the payload to our buffer for sending.

        SXI_n32SXiLogWritePayload(psTx->hSTI, psTx->hPayload, "OK");

        // Extract type
        tPayloadType = SXI_tGetTypeForConnection(psTx->hSTI);

        // Now transmit this payload
        bSentFrame = SXILL_bTxPayload(
            hProtocol, tPayloadType, psTx->hPayload);
        if(TRUE == bSentFrame)
        {
            BOOLEAN bHandled;

            bHandled =
                STI_bTransmitPayload(psTx->hSTI, psTx);
            if (FALSE == bHandled)
            {
#ifdef SXILL_TXRX_STATS_OUTPUT
                // Increment Statistics
                psCtrl->sStatistics.sTx.un32TxHandlerErrors++;
#endif
            }
        }
        else
        {
#ifdef SXILL_TXRX_STATS_OUTPUT
            // Increment Statistics
            psCtrl->sStatistics.sTx.un32SendErrors++;
#endif
        }

        // If this payload still exists, free it
        if(psTx->hPayload != OSAL_INVALID_BUFFER_HDL)
        {
            // Free this payload
            STI_bFreePayload(psTx->hPayload);
            psTx->hPayload = OSAL_INVALID_BUFFER_HDL;
        }
    }

    return;
}

/*****************************************************************************
 *
 *   vHandleProtocolSpecific
 *
 *****************************************************************************/
static void vHandleProtocolSpecific(
    STI_PROTOCOL_HDL hProtocol,
    SXILL_TXRX_CONTROL_STRUCT *psCtrl,
    STI_PROTOCOL_SPECIFIC_STRUCT const *psProtocol
        )
{
    printf("SXILL: Protocol Specific.\n");
    return;
}

/*****************************************************************************
 *
 *   vHandleOOB
 *
 *****************************************************************************/
static void vHandleOOB(
    STI_PROTOCOL_HDL hProtocol,
    SXILL_TXRX_CONTROL_STRUCT *psCtrl,
    STI_PROTOCOL_OOB_STRUCT const *psOOB
        )
{
    BOOLEAN bOk;

    printf("SXILL: Rx OOB (eType=%d).\n", psOOB->eType);

    // Act on the type of OOB event that occurred
    switch(psOOB->eType)
    {
        case STI_OOB_TYPE_LINK_ERROR:
        {
            // Event data is an SXiLL error code
            SXILL_RX_ERROR_ENUM eError =
                (SXILL_RX_ERROR_ENUM)((size_t)psOOB->pvData);

            printf("\tLink Error: %u\n", eError);
        }
        break;

        case STI_OOB_TYPE_PROTOCOL_SPECIFIC:
        {
            // Event data...? ((size_t)psOOB->pvData)

            puts("OOB-Msg: Protocol Specific");
        }
        break;

        default:
        {
            // Unhandled OOB event type
            return;
        }
    }

    // Rx OOB event via the STI connection
    bOk = STI_bReceiveOOB(
        hProtocol, psOOB->pvProtocolConnectionData,
        psOOB->eType, psOOB->pvData
            );
    if (FALSE == bOk)
    {
        // Error! Failed to pass oob msg to application.
        puts("SXILL Error! Failed to pass oob msg to "
                        "application.");
    }

    return;
}

#if DEBUG_OBJECT != 0

/*****************************************************************************
 *
 *   vDumpFrame
 *
 *   Inputs:
 *   	hBuffer - The handle of the buffer for which to dump.
 *
 *   Outputs:
 *   	None.
 *
 *****************************************************************************/
static void vDumpFrame(OSAL_BUFFER_HDL hRxFrame)
{
    SXILL_FRAME_HDR_STRUCT sFrameHdr;
    size_t tNum;

    // Extract frame header data from Frame...
    tNum = OSAL.tBufferPeek(hRxFrame, sFrameHdr.uFormat.sData.aun8Bytes,
                    sizeof(sFrameHdr.uFormat.sData.aun8Bytes), 0);
    if (SXILL_FRAME_HDR_LEN == tNum)
    {
        size_t tOffset = 0;
        UN16 un16Row = 0;

        printf(" Frame: "); printf("SEQ=%u, TYPE=%s, LEN=%u\n",
                        sFrameHdr.uFormat.sHeader.tSequenceNumber,
                        sFrameHdr.uFormat.sHeader.tPayloadType == SXILL_PAYLOAD_TYPE_LINK ? "LINK" :
                        sFrameHdr.uFormat.sHeader.tPayloadType == SXILL_PAYLOAD_TYPE_CONTROL ? "CONTROL" :
                        sFrameHdr.uFormat.sHeader.tPayloadType == SXILL_PAYLOAD_TYPE_DATA ? "DATA" :
                        sFrameHdr.uFormat.sHeader.tPayloadType == SXILL_PAYLOAD_TYPE_AUDIO ? "AUDIO" :
                        sFrameHdr.uFormat.sHeader.tPayloadType == SXILL_PAYLOAD_TYPE_DEBUG ? "DEBUG" :
                        "UNKNOWN",
                        SXILL_LEN_MODULE_TO_HOST(sFrameHdr.uFormat.sHeader.tNumPayloadBytes)
        );

        // Read data from buffer and dump it
        do
        {
            UN8 aun8Data[32];
            tNum = OSAL.tBufferPeek(hRxFrame, &aun8Data[0], sizeof(aun8Data),
                            tOffset);
            tOffset += tNum;
            OSAL.vDump(stdout, &un16Row, &aun8Data[0], tNum);

        }while (0 < tNum);
    }

    return;
}

#endif /*  DEBUG_OBJECT != 0 */

#ifdef SXILL_RX_STATS_OUTPUT

/*****************************************************************************
 *
 *   vWriteRxStats
 *
 *****************************************************************************/
static void vWriteRxStats(SXILL_RX_FRAME_CONTROL_STRUCT *psRxFrameCtrl)
{
    UN32 un32Seconds;

    OSAL.eTimeGetLocal(&un32Seconds);
    if ((OSAL_INVALID_OBJECT_HDL != psRxFrameCtrl->hLog)
                    && ((un32Seconds - psRxFrameCtrl->un32LastWrite)
                                    > SXILL_RX_STATS_POLLRATE_IN_SECONDS))
    {
        struct tm sOutputLocalTime;
        TIME_T tTime;
        char acOutputLocalTime[OSAL_MAX_OBJECT_NAME_LENGTH_WITH_NULL];

        // Update the write time
        OSAL.eTimeGetLocal(&psRxFrameCtrl->un32LastWrite);
        tTime = psRxFrameCtrl->un32LastWrite;

        // Get the local time and build a string output
        OSAL.localtime_r(&tTime, &sOutputLocalTime);
        OSAL.asctime_r(&sOutputLocalTime, acOutputLocalTime);

        // Print out our row of data
        printf("SXILLRx: Writing Stats\n");
        OSAL.n32LogWrite(psRxFrameCtrl->hLog, SXILL_RX_STATS_ROW,
            acOutputLocalTime, psRxFrameCtrl->sStatistics.un32MsgFrames,
            psRxFrameCtrl->sStatistics.un32CannotPostEvent,
            psRxFrameCtrl->sStatistics.un32NoMoreBuffers,
            psRxFrameCtrl->sStatistics.un32InvalidLength,
            psRxFrameCtrl->sStatistics.un32ChecksumFailures,
            psRxFrameCtrl->sStatistics.un32BufferOverflow,
            psRxFrameCtrl->sStatistics.un32IgnoredFrames,
            psRxFrameCtrl->sStatistics.un32OutOfSync,
            psRxFrameCtrl->sStatistics.un32KeepAlives,
            psRxFrameCtrl->sStatistics.un32Errors,
            psRxFrameCtrl->sStatistics.un32ReadExceedsRequest,
            psRxFrameCtrl->sStatistics.un32WroteNothingToBuffer,
            psRxFrameCtrl->sStatistics.un32NumUnprocessedBytesIsNotZero,
            psRxFrameCtrl->sStatistics.sRxBytes.un32LinkLayer,
            psRxFrameCtrl->sStatistics.sRxBytes.un32Payload,
            psRxFrameCtrl->sStatistics.sRxBytes.un32Overflow,
            psRxFrameCtrl->sRingBuffer.un32IngressOverflow,
            psRxFrameCtrl->sRingBuffer.un32EgressUnderflow,
            psRxFrameCtrl->sRingBuffer.un32InProcessUnderflow,
            psRxFrameCtrl->sRingBuffer.un32UnprocessedOverflow,
            psRxFrameCtrl->sRingBuffer.un32IllegalAdvancement
                );
    }
}

/*****************************************************************************
 *
 *   vWriteRxReq
 *
 *****************************************************************************/
static void vWriteRxReq(OSAL_OBJECT_HDL hLog, size_t tNumToRead,
    size_t tNumRead)
{
    if (OSAL_INVALID_OBJECT_HDL != hLog)
    {
        // Print out our row of data
        puts("SXILLRx: Writing Request");
        OSAL.n32LogWrite(hLog, SXILL_RX_REQ_ROW, tNumToRead, tNumRead);
    }
}

/*****************************************************************************
 *
 *   vWriteRingBufferState
 *
 *****************************************************************************/
static void vWriteRingBufferState(
    long lLineNum,
    SXILL_RING_BUFFER_REASON_CODE_ENUM eReasonCode,
    const SXILL_RING_BUFFER_STRUCT *psRingBuffer,
    size_t tArg
        )
{
    static char acBuffer[255];
    const char *pacSMSPath;
    UN32 un32Seconds = 0;
    FILE *psFile;

    OSAL.eTimeGetLocal(&un32Seconds);

    pacSMSPath = SMS_pacGetPath();

    if (NULL != pacSMSPath)
    {
        snprintf( &acBuffer[0], sizeof(acBuffer),
            "%s/RingBufferState.txt", pacSMSPath);

        psFile = fopen(&acBuffer[0], "a+");

        if (NULL != psFile)
        {
            TIME_T tTod = un32Seconds;
            char cBuf[OSAL_ASCBUFSIZE];

            snprintf( &acBuffer[0], sizeof(acBuffer),
                "RingBuffer: %s "
                "line#: %ld "
                "reason: %u "
#ifdef SXILL_USE_LEGACY_RING_BUFFER
                "pun8Start: %p "
                "pun8End: %p "
                "pun8In: %p "
                "pun8Out: %p "
                "pun8InProcess: %p "
                "tCount: %u "
                "tRemaining: %u "
                "tUnprocessed: %u "
#else
                "Master: %d (start %u, size %u)"
                "Progress: %d (start %u, size %u)"
#endif
                "tArg: %u\n",
                OSAL.ctime_r(&tTod, cBuf),
                lLineNum,
                eReasonCode,
#ifdef SXILL_USE_LEGACY_RING_BUFFER
                psRingBuffer->pun8Start,
                psRingBuffer->pun8End,
                psRingBuffer->pun8In,
                psRingBuffer->pun8Out,
                psRingBuffer->pun8InProcess,
                psRingBuffer->tCount,
                psRingBuffer->tRemaining,
                psRingBuffer->tUnprocessed,
#else
                psRingBuffer->sMaster.tCapacity,
                psRingBuffer->sMaster.tStart,
                psRingBuffer->sMaster.tSize,
                psRingBuffer->sProgress.tCapacity,
                psRingBuffer->sProgress.tStart,
                psRingBuffer->sProgress.tSize,
#endif
                tArg
                    );

            fprintf(psFile, &acBuffer[0]);

            fclose(psFile);
        }
    }

    return;
}

#endif

#ifdef SXILL_TXRX_STATS_OUTPUT

/*****************************************************************************
 *
 *   vWriteTxRxStats
 *
 *****************************************************************************/
static void vWriteTxRxStats(SXILL_TXRX_CONTROL_STRUCT *psTxRxCtrl)
{
    UN32 un32Seconds;
    OSAL_RETURN_CODE_ENUM eReturnCode;

    eReturnCode = OSAL.eTimeGetLocal(&un32Seconds);
    if ((OSAL_SUCCESS == eReturnCode)
                    && (OSAL_INVALID_OBJECT_HDL != psTxRxCtrl->hLog)
                    && ((un32Seconds - psTxRxCtrl->un32LastWrite)
                                    > SXILL_TXRX_STATS_POLLRATE_IN_SECONDS))
    {
        struct tm sOutputLocalTime;
        TIME_T tTime;
        char acOutputLocalTime[OSAL_MAX_OBJECT_NAME_LENGTH_WITH_NULL];

        // Update the write time
        OSAL.eTimeGetLocal(&psTxRxCtrl->un32LastWrite);
        tTime = psTxRxCtrl->un32LastWrite;

        // Get the local time and build a string output
        OSAL.localtime_r(&tTime, &sOutputLocalTime);
        OSAL.asctime_r(&sOutputLocalTime, acOutputLocalTime);

        // Print out our row of data
        printf("SXILLRx: Writing Stats\n");
        OSAL.n32LogWrite(psTxRxCtrl->hLog, SXILL_TXRX_STATS_ROW,
            acOutputLocalTime,
            psTxRxCtrl->sStatistics.sRx.un32MsgFrames,
            psTxRxCtrl->sStatistics.sRx.un32DroppedFrames,
            psTxRxCtrl->sStatistics.sRx.un32UnknownEvent,
            psTxRxCtrl->sStatistics.sRx.un32MsgBytes,

            psTxRxCtrl->sStatistics.sTx.un32MsgFrames,
            psTxRxCtrl->sStatistics.sTx.un32DeviceErrors,
            psTxRxCtrl->sStatistics.sTx.un32SendErrors,
            psTxRxCtrl->sStatistics.sTx.un32Nothing,
            psTxRxCtrl->sStatistics.sTx.sBytes.un32LinkLayer,
            psTxRxCtrl->sStatistics.sTx.sBytes.un32Payload,
            psTxRxCtrl->sStatistics.sTx.sBytes.un32Overflow,

            psTxRxCtrl->sStatistics.un32Errors
                );
    }

    return;
}

#endif
