/******************************************************************************/
/*                    Copyright (c) Sirius XM Radio, Inc.                     */
/*                            All Rights Reserved                             */
/*         Licensed Materials - Property of Sirius XM Radio, Inc.             */
/******************************************************************************/
/*******************************************************************************
 *
 * DESCRIPTION
 *
 *  This module contains various APIs which implement simple task
 *  synchronization. This is really just a set of helper apis which
 *  assist in the robust, just do-it synchronization between two threads.
 *  The idea is that maybe if this is deemed useful it may make it into
 *  OSAL or perhaps, within OSAL task creatiion itself to sync creation of
 *  threads for everyone. One API to sync them all!
 *
 ******************************************************************************/

#include <standard.h>

#include "osal.h"

#include "sti_sync.h"
#include "_sti_sync.h"

/*****************************************************************************
*
*   STIS_hCreate
*
*****************************************************************************/
STI_SYNC_HDL STIS_hCreate ( const char *pacName )
{
    STI_SYNC_HDL hSync = STI_SYNC_INVALID_HDL;

    do
    {
        OSAL_RETURN_CODE_ENUM eReturnCode;
        STI_SYNC_STRUCT *psSync = NULL;
        char acName[OSAL_MAX_OBJECT_NAME_LENGTH_WITH_NULL];
        char acMutexName[OSAL_MAX_OBJECT_NAME_LENGTH_WITH_NULL];

        // Construct a unique name for this object
        snprintf(&acName[0], sizeof(acName), "STISync:%s", pacName);

        // Allocate a sync structure (zero initialized)
        psSync = (STI_SYNC_STRUCT *)
            OSAL.pvMemoryAllocate(
                &acName[0],
                sizeof(STI_SYNC_STRUCT),
                TRUE );
        if(psSync == NULL)
        {
            // Error!
            break;
        }

        // Mutex A
        snprintf(&acMutexName[0], sizeof(acMutexName), "%s:A", acName);

        eReturnCode = OSAL.eSemCreate(
            &psSync->sSyncA.hMutex,
            acMutexName, 0, 1, OSAL_SEM_OPTION_NONE);
        if (eReturnCode != OSAL_SUCCESS)
        {
            // Error!
            printf("STIS Error: Cannot create sync (%s) mutex!\n",
                acMutexName);
            STIS_bDestroy((STI_SYNC_HDL)psSync);
            break;
        }

        // Mutex B
        snprintf(&acMutexName[0], sizeof(acMutexName), "%s:B", acName);

        eReturnCode = OSAL.eSemCreate(
            &psSync->sSyncB.hMutex,
            acMutexName, 0, 1, OSAL_SEM_OPTION_NONE);
        if (eReturnCode != OSAL_SUCCESS)
        {
            // Error!
            printf("STIS Error: Cannot create sync (%s) mutex!\n",
                acMutexName);
            STIS_bDestroy((STI_SYNC_HDL)psSync);
            break;
        }

        // Assign sync object
        hSync = (STI_SYNC_HDL)psSync;

    } while (FALSE);

    return hSync;
}

/*****************************************************************************
*
*   STIS_bDestroy
*
*****************************************************************************/
BOOLEAN STIS_bDestroy (
    STI_SYNC_HDL hSync
        )
{
    BOOLEAN bSuccess = FALSE;
    STI_SYNC_STRUCT *psSync =
        (STI_SYNC_STRUCT *)hSync;

    if(psSync != NULL)
    {
        OSAL_RETURN_CODE_ENUM eReturnCode;

        // Note: The idea is that both mutex's are not available
        // so we must just take them with no timeout.

        if(psSync->sSyncA.hMutex != OSAL_INVALID_OBJECT_HDL)
        {
            eReturnCode = OSAL.eSemTake(
                psSync->sSyncA.hMutex, OSAL_OBJ_TIMEOUT_NONE);
            if((eReturnCode == OSAL_SUCCESS) ||
                (eReturnCode == OSAL_SEM_NOT_AVAILABLE))
            {
                eReturnCode = OSAL.eSemDelete(psSync->sSyncA.hMutex);
                if(eReturnCode == OSAL_SUCCESS)
                {
                    psSync->sSyncA.hMutex = OSAL_INVALID_OBJECT_HDL;
                }
            }
        }

        if(psSync->sSyncB.hMutex != OSAL_INVALID_OBJECT_HDL)
        {
            eReturnCode = OSAL.eSemTake(
                psSync->sSyncB.hMutex, OSAL_OBJ_TIMEOUT_NONE);
            if((eReturnCode == OSAL_SUCCESS) ||
                (eReturnCode == OSAL_SEM_NOT_AVAILABLE))
            {
                eReturnCode = OSAL.eSemDelete(psSync->sSyncB.hMutex);
                if(eReturnCode == OSAL_SUCCESS)
                {
                    psSync->sSyncB.hMutex = OSAL_INVALID_OBJECT_HDL;
                }
            }
        }

        OSAL.vMemoryFree(psSync);
        bSuccess = TRUE;
    }

    return bSuccess;
}

/*****************************************************************************
*
*   STIS_bWait
*
*****************************************************************************/
BOOLEAN STIS_bWait (
    STI_SYNC_HDL hSync,
    void ** ppvData
        )
{
    BOOLEAN bSuccess = FALSE;
    STI_SYNC_STRUCT *psSync =
        (STI_SYNC_STRUCT *)hSync;

    if(psSync != NULL)
    {
        OSAL_RETURN_CODE_ENUM eReturnCode;

        if(ppvData != NULL)
        {
            // Give data to B
            psSync->sSyncB.pvData = *ppvData;
        }

        // Signal B
        eReturnCode = OSAL.eSemGive(psSync->sSyncB.hMutex);
        if(eReturnCode == OSAL_SUCCESS)
        {
            // Wait for A
            eReturnCode = OSAL.eSemTake(
                psSync->sSyncA.hMutex, OSAL_OBJ_TIMEOUT_INFINITE);
            if(eReturnCode == OSAL_SUCCESS)
            {
                if(ppvData != NULL)
                {
                    // Get data for A
                    *ppvData = psSync->sSyncA.pvData;

                    // Leave nothing behind
                    psSync->sSyncA.pvData = NULL;
                }

                bSuccess = TRUE;
            }
        }
    }

    return bSuccess;
}

/*****************************************************************************
*
*   STIS_bSignal
*
*****************************************************************************/
BOOLEAN STIS_bSignal (
    STI_SYNC_HDL hSync,
    void ** ppvData
        )
{
    BOOLEAN bSuccess = FALSE;
    STI_SYNC_STRUCT *psSync =
        (STI_SYNC_STRUCT *)hSync;

    if(psSync != NULL)
    {
        OSAL_RETURN_CODE_ENUM eReturnCode;

        // Wait for B
        eReturnCode = OSAL.eSemTake(
            psSync->sSyncB.hMutex, OSAL_OBJ_TIMEOUT_INFINITE);
        if(eReturnCode == OSAL_SUCCESS)
        {
            if(ppvData != NULL)
            {
                // Give data to A
                psSync->sSyncA.pvData = *ppvData;

                // Get data for B
                *ppvData = psSync->sSyncB.pvData;

                // Leave no evidence
                psSync->sSyncB.pvData = NULL;
            }

            // Signal A
            eReturnCode = OSAL.eSemGive(psSync->sSyncA.hMutex);
            if(eReturnCode == OSAL_SUCCESS)
            {
                bSuccess = TRUE;
            }
        }
    }

    return bSuccess;
}
