/******************************************************************************/
/*                    Copyright (c) Sirius XM Radio, Inc.                     */
/*                            All Rights Reserved                             */
/*         Licensed Materials - Property of Sirius XM Radio, Inc.             */
/******************************************************************************/
/*******************************************************************************
 *
 * DESCRIPTION
 *
 *  This module contains the Object:WIND implementation for the
 *  Sirius Module Services (SMS)
 *
 ******************************************************************************/
#include "standard.h"
#include "osal.h"

#include "sms_api.h"
#include "sms_obj.h"
#include "_wind_obj.h"
#include "wind_obj.h"
#include "weather_interface.h"

#include "sms_api_debug.h"

/*****************************************************************************
                             PUBLIC FUNCTIONS
*****************************************************************************/

/*****************************************************************************
*
*   eDirection
*
*****************************************************************************/
static SMSAPI_RETURN_CODE_ENUM eDirection (
    WIND_OBJECT hWind,
    UN16 *pun16WindDirection
        )
{
    BOOLEAN bOwner = FALSE;
    SMSAPI_RETURN_CODE_ENUM eResult = SMSAPI_RETURN_CODE_ERROR;
    WIND_OBJECT_STRUCT *psObj = (WIND_OBJECT_STRUCT *) hWind;

    do
    {
        if (pun16WindDirection == NULL)
        {
            eResult = SMSAPI_RETURN_CODE_INVALID_INPUT;
            break;
        }

        bOwner = SMSO_bOwner((SMS_OBJECT)hWind);
        if (bOwner == FALSE)
        {
            eResult = SMSAPI_RETURN_CODE_NOT_OWNER;
            break;
        }

        if (psObj->bAvailable == FALSE)
        {
            eResult = SMSAPI_RETURN_CODE_NOT_FOUND;
            break;
        }
        eResult = SMSAPI_RETURN_CODE_SUCCESS;
        *pun16WindDirection = psObj->sData.un16Direction;
    } while (FALSE);

    return eResult;
}


/*****************************************************************************
*
*   eSpeed
*
*****************************************************************************/
static SMSAPI_RETURN_CODE_ENUM eSpeed (
    WIND_OBJECT hWind,
    UN8 *pun8WindSpeed
        )
{
    BOOLEAN bOwner = FALSE;
    SMSAPI_RETURN_CODE_ENUM eResult = SMSAPI_RETURN_CODE_ERROR;
    WIND_OBJECT_STRUCT *psObj = (WIND_OBJECT_STRUCT *) hWind;

    do
    {
        if (pun8WindSpeed == NULL)
        {
            eResult = SMSAPI_RETURN_CODE_INVALID_INPUT;
            break;
        }

        bOwner = SMSO_bOwner((SMS_OBJECT)hWind);
        if (bOwner == FALSE)
        {
            eResult = SMSAPI_RETURN_CODE_NOT_OWNER;
            break;
        }

        if (psObj->bAvailable == FALSE)
        {
            eResult = SMSAPI_RETURN_CODE_NOT_FOUND;
            break;
        }

        *pun8WindSpeed = psObj->sData.un8Speed;
        eResult = SMSAPI_RETURN_CODE_SUCCESS;
    } while (FALSE);

    return eResult;
}


/*****************************************************************************
*
*   n32FPrintf
*
*****************************************************************************/
static N32 n32FPrintf (
    WIND_OBJECT hWind,
    FILE *psFile
        )
{
    N32 n32Return = 0;
    WIND_OBJECT_STRUCT *psObj = (WIND_OBJECT_STRUCT *)hWind;
    BOOLEAN bOwner;

    // Determine if we are the owner
    bOwner = SMSO_bOwner((SMS_OBJECT)hWind);
    if ((bOwner == FALSE) || (psFile == NULL))
    {
        return EOF;
    }

    if (psObj->bAvailable == TRUE)
    {
        // Print WIND information header
        n32Return += fprintf(psFile, "\nWind (Object: 0x%X):", hWind);

        n32Return += fprintf(psFile, "\n\tDirection: %u\n\tSpeed: %u\n",
                psObj->sData.un16Direction, psObj->sData.un8Speed);
    }

    return n32Return;
}


/*****************************************************************************
                             FRIEND FUNCTIONS
*****************************************************************************/

/*****************************************************************************
*
*   WIND_hCreate
*
*****************************************************************************/
WIND_OBJECT WIND_hCreate(
        SMS_OBJECT hParent,
        BOOLEAN bAvailable,
        WIND_UNPARSED_DATA tData
            )
{
    WIND_OBJECT_STRUCT *psObj =
        (WIND_OBJECT_STRUCT *)WIND_INVALID_OBJECT;

    // Create an instance of the WIND object
    psObj = (WIND_OBJECT_STRUCT *)
        SMSO_hCreate(
            WIND_OBJECT_NAME,
            sizeof(WIND_OBJECT_STRUCT),
            hParent,
            FALSE
                );

    if(psObj == NULL)
    {
        // Error!
        return WIND_INVALID_OBJECT;
    }

    psObj->bAvailable = bAvailable;

    if (bAvailable == TRUE)
    {
        psObj->bAvailable = GsWeatherIntf.bProcessWindData(tData, &psObj->sData);
    }

    return (WIND_OBJECT)psObj;
}

/*****************************************************************************
*
*   WIND_vDestroy
*
*****************************************************************************/
void WIND_vDestroy (
    WIND_OBJECT hWind
        )
{
    BOOLEAN bOwner;

    bOwner = SMSO_bOwner((SMS_OBJECT)hWind);
    if(bOwner == TRUE)
    {
        // Free object instance
        SMSO_vDestroy((SMS_OBJECT)hWind);
    }

    return;
}


/*****************************************************************************
                             PRIVATE FUNCTIONS
*****************************************************************************/
