/******************************************************************************/
/*                    Copyright (c) Sirius XM Radio, Inc.                     */
/*                            All Rights Reserved                             */
/*         Licensed Materials - Property of Sirius XM Radio, Inc.             */
/******************************************************************************/
/*******************************************************************************
 *
 * DESCRIPTION
 *
 *  This module contains the Object:WEATHER_MGR implementation for the
 *  Simple Module Services (SMS)
 *
 ******************************************************************************/
#include <ctype.h>
#include <string.h>

#include "standard.h"
#include "osal.h"

#include "sms.h"
#include "sms_obj.h"
#include "sms_event.h"
#include "string_obj.h"
#include "dataservice_mgr_impl.h"
#include "device_obj.h"
#include "forecast_obj.h"
#include "dsrl_target_obj.h"
#include "dsrl_entry_obj.h"
#include "weather_msg_obj.h"
#include "weather_mgr_obj.h"
#include "_weather_mgr_obj.h"
#include "locid_obj.h"
#include "ski_conditions_obj.h"

#include "sms_api_debug.h"
#include "db_util.h"

static const char *gpacThisFile = __FILE__;

/*****************************************************************************
                             PUBLIC FUNCTIONS
*****************************************************************************/

/*****************************************************************************
*
*   hStart
*
*   This function will create all the basics needed for this service to
*   operate.  However, all initial processing to actually get this service
*   running is done at a later time.
*
*****************************************************************************/
static WEATHER_SERVICE_OBJECT hStart (
    const char *pacSRHDriverName,
    DATASERVICE_EVENT_MASK tEventRequestMask,
    DATASERVICE_EVENT_CALLBACK vEventCallback,
    void *pvAppEventCallbackArg,
    DATASERVICE_OPTIONS_STRUCT const *psOptions
        )
{
    WEATHER_MGR_OBJECT_STRUCT *psObj = NULL;
    BOOLEAN bOk = FALSE;
    DATASERVICE_CREATE_STRUCT sCreate;
    DATASERVICE_OPTION_VALUES_STRUCT sOptionValues;
    va_list tList; // variable arguments list

    printf(WEATHER_MGR_OBJECT_NAME" hStart enter\n");

    // grab variable arguments (pop)
    bOk = DATASERVICE_IMPL_bProcessOptions(
        WEATHER_SUPPORTED_OPTIONS, psOptions, &sOptionValues);
    // restore stack (push)
    va_end(tList);
    if (bOk == FALSE)
    {
        // Bad options
        return WEATHER_SERVICE_INVALID_OBJECT;
    }

    // Populate our data service creation structure
    DATASERVICE_IMPL_vInitCreateStruct(&sCreate);
    sCreate.pacSRHDriverName = pacSRHDriverName;
    sCreate.pacServiceObjectName = WEATHER_MGR_OBJECT_NAME;
    sCreate.tServiceObjectSize = sizeof(WEATHER_MGR_OBJECT_STRUCT);
    sCreate.tDataID = (DATASERVICE_ID)GsWeatherIntf.tDSI;

    // Get the interface to tell us how much memory
    // this service needs at a minimum for the given
    // startup options
    sCreate.tSuggestedOTABufferByteSize =
        GsWeatherIntf.tMinimumOTABufferByteSize(sOptionValues.bUpdateRefDB);

    // Configure the data service's static event attributes
    sCreate.vEventCallback = vEventHandler;
    sCreate.tEventRequestMask = (
        DATASERVICE_EVENT_ALL |
        DATASERVICE_INTERNAL_EVENT_DSRL |
        DATASERVICE_INTERNAL_EVENT_SERVICE_SPECIFIC);

    // Ask the data service manager controller to
    // create our manager object and do everything
    // necessary to create the underlying objects required
    // in order to support this service
    psObj = (WEATHER_MGR_OBJECT_STRUCT *)
        DATASERVICE_IMPL_hCreateNewService(&sCreate);
    if (psObj == NULL)
    {
        // Can't create the service, fail out!
        // Free options memory
        DATASERVICE_IMPL_vFreeOptions(&sOptionValues);

        return WEATHER_SERVICE_INVALID_OBJECT;
    }

    // Update the flag & path for the reference db
    psObj->bRefDBUpdatesEnabled = sOptionValues.bUpdateRefDB;
    psObj->pacRefDatabaseDirPath = sOptionValues.pcRefDBPath;

    // Initialize asynchronous update configuration
    SMSU_vInitialize(
        &psObj->sEvent,
        psObj,
        DATASERVICE_EVENT_ALL,
        tEventRequestMask,
        (SMSAPI_OBJECT_EVENT_CALLBACK)vEventCallback,
        pvAppEventCallbackArg);

    psObj->hInterface = WEATHER_INTERFACE_INVALID_OBJECT;

    // Initialize the app facing object
    bOk = bInitAppFacingObject(psObj);

    if (bOk == TRUE)
    {
        bOk = DATASERVICE_IMPL_bStart((DATASERVICE_IMPL_HDL)psObj);

        if (bOk == FALSE)
        {
            printf(WEATHER_MGR_OBJECT_NAME" DATASERVICE_MGR_bStart has failed (destroying service)\n");
            // Error!
            vUninitObject( psObj, TRUE );
            DATASERVICE_IMPL_vDestroy((DATASERVICE_IMPL_HDL)psObj);

            return WEATHER_SERVICE_INVALID_OBJECT;
        }
    }

    return (WEATHER_SERVICE_OBJECT)psObj;
}

/*******************************************************************************
*
*        eEnableSkiReports
*
*******************************************************************************/
static SMSAPI_RETURN_CODE_ENUM eEnableSkiReports(
    WEATHER_SERVICE_OBJECT hWeatherService
        )
{
    SMSAPI_RETURN_CODE_ENUM eResult = SMSAPI_RETURN_CODE_ERROR;
    BOOLEAN bValid;

    // Verify the manager
    bValid = DATASERVICE_IMPL_bValid((DATASERVICE_IMPL_HDL)hWeatherService);
    if(bValid == TRUE)
    {
        WEATHER_MGR_EVENT_STRUCT *psEventData = NULL;
        SMS_EVENT_HDL hEvent;

        hEvent = DATASERVICE_IMPL_hAllocateEvent(
            (DATASERVICE_IMPL_HDL)hWeatherService, (void **)&psEventData);

        if (SMS_INVALID_EVENT_HDL != hEvent)
        {
            BOOLEAN bPosted;

            // Indicate that ski processing should be enabled
            psEventData->bEnableSkiReports = TRUE;

            // Post it now
            bPosted = SMSE_bPostEvent(hEvent);
            if (bPosted == TRUE)
            {
                eResult = SMSAPI_RETURN_CODE_SUCCESS;
            }
            else
            {
                SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    WEATHER_MGR_OBJECT_NAME": Failed to post event.");
            }
        }
    }

    return eResult;
}

/*******************************************************************************
*
*        eDisableSkiReports
*
*******************************************************************************/
static SMSAPI_RETURN_CODE_ENUM eDisableSkiReports(
    WEATHER_SERVICE_OBJECT hWeatherService
        )
{
    SMSAPI_RETURN_CODE_ENUM eResult = SMSAPI_RETURN_CODE_ERROR;
    BOOLEAN bValid;

    // Verify the manager
    bValid = DATASERVICE_IMPL_bValid((DATASERVICE_IMPL_HDL)hWeatherService);
    if(bValid == TRUE)
    {
        WEATHER_MGR_EVENT_STRUCT *psEventData = NULL;
        SMS_EVENT_HDL hEvent;

        hEvent = DATASERVICE_IMPL_hAllocateEvent(
            (DATASERVICE_IMPL_HDL)hWeatherService, (void **)&psEventData);

        if (SMS_INVALID_EVENT_HDL != hEvent)
        {
            BOOLEAN bPosted;

            // Indicate that ski processing should be disabled
            psEventData->bEnableSkiReports = FALSE;

            // Post it now
            bPosted = SMSE_bPostEvent(hEvent);
            if (bPosted == TRUE)
            {
                eResult = SMSAPI_RETURN_CODE_SUCCESS;
            }
            else
            {
                SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    WEATHER_MGR_OBJECT_NAME": Failed to post event.");
            }
        }
    }

    return eResult;
}

/*******************************************************************************
*
*        eIterateLocations
*
*******************************************************************************/
static SMSAPI_RETURN_CODE_ENUM eIterateLocations(
    WEATHER_SERVICE_OBJECT hWeatherService,
    BOOLEAN bWeatherSkiFlag,
    STRING_OBJECT hSortName,
    STRING_OBJECT hState,
    WEATHER_LOCATION_ITERATOR bIterator,
    void *pvArg
        )
{
    WEATHER_FILTER_LOCATIONS_LIST_STRUCT sFilter;
    SMSAPI_RETURN_CODE_ENUM eReturnCode = SMSAPI_RETURN_CODE_ERROR;
    WEATHER_APP_OBJECT_STRUCT *psAppObj = NULL;

    OSAL.bMemSet(&sFilter, 0, sizeof(sFilter));

    do
    {
        BOOLEAN bSuccess;

        // Validate input
        if (NULL == bIterator)
        {
            eReturnCode = SMSAPI_RETURN_CODE_INVALID_INPUT;
            break;
        }

        psAppObj = psGetAppFacingObject(hWeatherService);
        if (psAppObj == NULL)
        {
            break;
        }

        // Populate the filter structure
        sFilter.tStateId = LOCATION_tStateIDForAbbrv(hState);
        sFilter.bWeatherSkiFlag = bWeatherSkiFlag;
        sFilter.bIterator = bIterator;
        sFilter.pvArg = pvArg;
        if (hSortName != STRING_INVALID_OBJECT)
        {
            sFilter.hFilterString = STRING.hDuplicate(hSortName);
            if (sFilter.hFilterString == STRING_INVALID_OBJECT)
            {
                break;
            }

            // converting string to upper letters since SQL request
            // performs upper letter conversion too
            STRING_bToUpper(sFilter.hFilterString);
        }

        // Iterate now
        bSuccess = bQueryLocationsFromDB(psAppObj, &sFilter);
        if (bSuccess == FALSE)
        {
            //error !
            break;
        }

        eReturnCode = SMSAPI_RETURN_CODE_SUCCESS;
    } while (FALSE);

    vUnlockAppFacingObject(psAppObj);

    if (sFilter.hFilterString != STRING_INVALID_OBJECT)
    {
        STRING_vDestroy(sFilter.hFilterString);
    }

    return eReturnCode;
}

/*******************************************************************************
*
*        bRefDBBank
*
*******************************************************************************/
static BOOLEAN bRefDBBank (
    WEATHER_SERVICE_OBJECT hWeatherService,
    STRING_OBJECT *phInUseDB,
    STRING_OBJECT *phNextDB
        )
{
    BOOLEAN bLocked, bSuccess = FALSE;

    if ((phInUseDB == NULL) || (phNextDB == NULL))
    {
        return FALSE;
    }

    // Initialize inputs
    *phInUseDB = *phNextDB = STRING_INVALID_OBJECT;

    bLocked = DATASERVICE_IMPL_bLock((DATASERVICE_IMPL_HDL)hWeatherService);
    if (bLocked == TRUE)
    {
        WEATHER_MGR_OBJECT_STRUCT *psObj =
            (WEATHER_MGR_OBJECT_STRUCT *)hWeatherService;

        do
        {
            // Provide the caller with the path info
            *phInUseDB = STRING.hCreate(
                psObj->pacCurRefDatabaseFilePath,
                strlen(psObj->pacCurRefDatabaseFilePath));

            if (*phInUseDB == STRING_INVALID_OBJECT)
            {
                // Error! Stop here
                break;
            }

            *phNextDB = STRING.hCreate(
                psObj->pacNextRefDatabaseFilePath,
                strlen(psObj->pacNextRefDatabaseFilePath));

            if (*phNextDB == STRING_INVALID_OBJECT)
            {
                // Clear the other string now
                STRING_vDestroy(*phInUseDB);
                *phInUseDB = STRING_INVALID_OBJECT;

                break;
            }

            // All went well
            bSuccess = TRUE;
        } while (FALSE);

        DATASERVICE_IMPL_vUnlock((DATASERVICE_IMPL_HDL)hWeatherService);
    }

    return bSuccess;
}

/*******************************************************************************
*
*        bDBUpdateBegin
*
*******************************************************************************/
static BOOLEAN bDBUpdateBegin (
    SQL_INTERFACE_OBJECT hConnection,
    char *pacBuffer,
    size_t tBufferSize
        )
{
    BOOLEAN bSuccess;

    // Build our update string for this new version using our
    // DB's schema version to find the version row
    snprintf(&pacBuffer[0], tBufferSize,
        WEATHER_UPDATE_LOCATION_VERSION,
        DB_UTIL_DB_UNDER_CONSTRUCTION_VER);

    // update in database
    bSuccess = SQL_INTERFACE.bExecuteCommand(
            hConnection, &pacBuffer[0]);

    if (bSuccess == FALSE)
    {
        SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
            WEATHER_MGR_OBJECT_NAME": Unable to set baseline version");
    }

    return bSuccess;
}

/*******************************************************************************
*
*        bDBUpdateEnd
*
*******************************************************************************/
static BOOLEAN bDBUpdateEnd (
    SQL_INTERFACE_OBJECT hConnection,
    char *pacBuffer,
    size_t tBufferSize,
    UN8 un8DBVer
        )
{
    BOOLEAN bResult = FALSE;

    do
    {
        N32 n32Result;

        // While the DB_UTIL function will print its own error, we also print
        // one here to make it easier to trace the DB failure back to a
        // particular service.
        bResult = DB_UTIL_bUpdateTimestamp( hConnection, pacBuffer, tBufferSize );
        if ( bResult == FALSE )
        {
            SMSAPI_DEBUG_vPrintErrorFull( gpacThisFile, __LINE__,
                    WEATHER_MGR_OBJECT_NAME": Unable to set timestamp.");
        }

        n32Result = snprintf( pacBuffer, tBufferSize, WEATHER_UPDATE_LOCATION_VERSION,
                un8DBVer );
        if ( n32Result <= 0 )
        {
            bResult = FALSE;
            break;
        }

        // Updating the baseline version comes last, as this effectively stamps
        // the DB as "ready."
        bResult = SQL_INTERFACE.bExecuteCommand( hConnection, pacBuffer );
        if ( bResult == FALSE )
        {
            SMSAPI_DEBUG_vPrintErrorFull( gpacThisFile, __LINE__,
                    WEATHER_MGR_OBJECT_NAME": Unable to set baseline version.");
            break;
        }

    } while ( FALSE );

    return bResult;
}

/*******************************************************************************
*
*        bEntrySetForecast
*
*******************************************************************************/
static BOOLEAN bEntrySetForecast(
    DSRL_ENTRY_OBJECT hDSRLEntry,
    WEATHER_APP_OBJECT_STRUCT *psAppObj,
    WEATHER_FORECAST_TYPE_ENUM eType,
    FORECAST_DATA_STRUCT *psData
        )
{
    BOOLEAN bResult = FALSE;
    FORECAST_OBJECT hNewForecast = FORECAST_INVALID_OBJECT;

    do
    {
        FORECAST_OBJECT hOldForecast = FORECAST_INVALID_OBJECT;
        SMSAPI_RETURN_CODE_ENUM eResult = SMSAPI_RETURN_CODE_ERROR;
        BOOLEAN bIsEqual = FALSE, bSuccess = FALSE;
        WEATHER_MSG_OBJECT hMsg = WEATHER_MSG_INVALID_OBJECT;

        if ((hDSRLEntry == DSRL_ENTRY_INVALID_OBJECT) ||
            (psAppObj == NULL) ||
            (psData == NULL))
        {
            break;
        }

        hMsg = DSRL_ENTRY.hWeatherMsg(hDSRLEntry);
        if (hMsg == WEATHER_MSG_INVALID_OBJECT)
        {
            break;
        }

        //compare forecast item to be sure that data is not same
        eResult = WEATHER_MSG.eGetForecast(
            hMsg,
            eType,
            &hOldForecast
                );

        if (eResult == SMSAPI_RETURN_CODE_SUCCESS)
        {
            bIsEqual = FORECAST_bCompareData(hOldForecast, psData);
        }
        else
        {
            bIsEqual = FALSE;
        }

        if (bIsEqual == TRUE)
        {
            FORECAST_bUpdateHashEntry(hOldForecast, psData->hHashEntry);
            FORECAST_vUpdateTimeStamp(hOldForecast);

            DATASERVICE_IMPL_vLog(WEATHER_MGR_OBJECT_NAME
                ": DSRL forecast entry timestamp has been updated\n"
                    );
        }
        else
        {
            // creating the new forecast object from the provided data
            hNewForecast = FORECAST_hCreate((SMS_OBJECT)psAppObj, psData, 0);
            if (hNewForecast == FORECAST_INVALID_OBJECT)
            {
                SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    WEATHER_MGR_OBJECT_NAME": cannot create forecast"
                        );
                break;
            }

            // moving entry DSRLs to updating state
            vChangeEntryDSRLsState(hDSRLEntry, DSRL_STATE_UPDATING);

            bSuccess = WEATHER_MSG_bSetForecast(hMsg,
                eType, hNewForecast
                    );
            if (bSuccess == FALSE)
            {
                SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    WEATHER_MGR_OBJECT_NAME": failed to assign forecast"
                        );
                break;
            }

            // updating the entry state to CHANGED in all DSRLs which include
            // this entry
            bSuccess = bEntryChange(hDSRLEntry, psAppObj);
            if (bSuccess == FALSE)
            {
                SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    WEATHER_MGR_OBJECT_NAME": failed to update DSRL state"
                        );
                break;
            }

            //reset forecast to avoid dealloc
            hNewForecast = FORECAST_INVALID_OBJECT;

            DATASERVICE_IMPL_vLog(WEATHER_MGR_OBJECT_NAME
                ": DSRL forecast entry has been updated\n"
                    );
        }

        bResult = TRUE;
    } while (FALSE);

    if (hNewForecast != FORECAST_INVALID_OBJECT)
    {
        FORECAST_vDestroy(hNewForecast);
    }

    return bResult;
}

/*******************************************************************************
*
*        bProcessForecastReport
*
*******************************************************************************/
static BOOLEAN bProcessForecastReport (
    WEATHER_SERVICE_OBJECT hWeatherService,
    LOC_ID tID,
    WEATHER_FORECAST_TYPE_ENUM eType,
    FORECAST_DATA_STRUCT *psForecastData
        )
{
    BOOLEAN bResult = FALSE;

    do
    {
        WEATHER_APP_OBJECT_STRUCT *psAppObj = NULL;
        DSRL_ENTRY_OBJECT hDSRLEntry = DSRL_ENTRY_INVALID_OBJECT;

        if (hWeatherService == WEATHER_SERVICE_INVALID_OBJECT)
        {
            break;
        }
        psAppObj = ((WEATHER_MGR_OBJECT_STRUCT*)hWeatherService)->psAppObj;

        // looking if we have the entry for this loc id
        hDSRLEntry = hSearchEntry(psAppObj, tID);
        if (hDSRLEntry == DSRL_ENTRY_INVALID_OBJECT)
        {
            printf(WEATHER_MGR_OBJECT_NAME": entry for loc_id %x"
                " not found, skipping\n", tID);
            bResult = TRUE;
            break;
        }

        // entry has been found, lets try to update the forecast data
        bResult = bEntrySetForecast(hDSRLEntry, psAppObj,
            eType, psForecastData
                );
    } while (FALSE);

    return bResult;
}

/*******************************************************************************
*
*        bStartProcessing
*
*******************************************************************************/
static BOOLEAN bStartProcessing(
    WEATHER_SERVICE_OBJECT hWeatherService
        )
{
    BOOLEAN bResult = TRUE;
    WEATHER_APP_OBJECT_STRUCT *psAppObj = NULL;

    // data processing could result in update of DSRLs and DSRL entries
    // lets lock the app object since it is DSRL data parent
    psAppObj = psGetAppFacingObject(hWeatherService);
    if (psAppObj == NULL)
    {
        bResult = FALSE;
    }

    return bResult;
}

/*******************************************************************************
*
*        bEndProcessing
*
*******************************************************************************/
static BOOLEAN bEndProcessing(
    WEATHER_SERVICE_OBJECT hWeatherService
        )
{
    BOOLEAN bResult = FALSE;

    if (hWeatherService != WEATHER_SERVICE_INVALID_OBJECT)
    {
        OSAL_RETURN_CODE_ENUM eReturnCode = OSAL_ERROR;

        WEATHER_MGR_OBJECT_STRUCT *psObj =
            (WEATHER_MGR_OBJECT_STRUCT *)hWeatherService;

        // data processing is completed. Need to set the READY state for all
        // the DSRLs, excluding the DSRLs in ERROR state
        eReturnCode = OSAL.eLinkedListIterate(psObj->psAppObj->hDSRLList,
            (OSAL_LL_ITERATOR_HANDLER)bDSRLChangeState,
            (void *)(size_t)DSRL_STATE_READY);
        if ((eReturnCode != OSAL_SUCCESS) &&
            (eReturnCode != OSAL_NO_OBJECTS))
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                WEATHER_MGR_OBJECT_NAME": failed to set DSRLs state (%s)",
                OSAL.pacGetReturnCodeName(eReturnCode)
                    );
        }
        else
        {
            bResult = TRUE;
        }

        // unlocking the App object locked before by bStartProcessing
        vUnlockAppFacingObject(psObj->psAppObj);
    }

    return bResult;
}

/*******************************************************************************
*
*        bEntrySetSkiReport
*
*******************************************************************************/
static BOOLEAN bEntrySetSkiReport(
    DSRL_ENTRY_OBJECT hDSRLEntry,
    WEATHER_APP_OBJECT_STRUCT *psAppObj,
    SKI_DATA_STRUCT *psData
        )
{
    BOOLEAN bResult = FALSE;
    SKI_CONDITIONS_OBJECT hNewSkiCond = SKI_CONDITIONS_INVALID_OBJECT;

    do
    {
        SMSAPI_RETURN_CODE_ENUM eResult = SMSAPI_RETURN_CODE_ERROR;
        SKI_CONDITIONS_OBJECT hOldSkiCond = SKI_CONDITIONS_INVALID_OBJECT;
        BOOLEAN bIsEqual = FALSE, bSuccess = FALSE;
        WEATHER_MSG_OBJECT hMsg = WEATHER_MSG_INVALID_OBJECT;

        if ((hDSRLEntry == DSRL_ENTRY_INVALID_OBJECT) ||
            (psAppObj == NULL) ||
            (psData == NULL))
        {
            break;
        }

        hMsg = DSRL_ENTRY.hWeatherMsg(hDSRLEntry);
        if (hMsg == WEATHER_MSG_INVALID_OBJECT)
        {
            break;
        }

        //compare ski cond item to be sure that data is not same
        eResult = WEATHER_MSG.eGetSkiConditions(hMsg,
                &hOldSkiCond
                    );
        if (eResult == SMSAPI_RETURN_CODE_SUCCESS)
        {
            bIsEqual = SKI_CONDITIONS_bCompareData(hOldSkiCond, psData);
        }
        else
        {
            bIsEqual = FALSE;
        }

        if (bIsEqual == TRUE)
        {
            //just change hash entry and timestamp
            //DSRL update not needed
            SKI_CONDITIONS_bUpdateHashEntry(hOldSkiCond,
                psData->hHashEntry);
            SKI_CONDITIONS_vUpdateTimeStamp(hOldSkiCond);

            DATASERVICE_IMPL_vLog(WEATHER_MGR_OBJECT_NAME
                ": DSRL ski conditions entry timestamp has been updated\n"
                    );
        }
        else
        {
            // creating new ski conditions object based on incoming data
            hNewSkiCond = SKI_CONDITIONS_hCreate((SMS_OBJECT)psAppObj,
                psData, 0);
            if (hNewSkiCond == SKI_CONDITIONS_INVALID_OBJECT)
            {
                SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    WEATHER_MGR_OBJECT_NAME": cannot create ski conditions"
                        );
                break;
            }

            // moving entry DSRLs to updating state
            vChangeEntryDSRLsState(hDSRLEntry, DSRL_STATE_UPDATING);

            bSuccess = WEATHER_MSG_bSetSkiCond(
                hMsg,
                hNewSkiCond
                    );
            if (bSuccess == FALSE)
            {
                SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    WEATHER_MGR_OBJECT_NAME": cannot assign ski conditions"
                        );
                break;
            }

            // updating the entry state to CHANGED in all the entry DSRLs
            bSuccess = bEntryChange(hDSRLEntry, psAppObj);
            if (bSuccess == FALSE)
            {
                SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    WEATHER_MGR_OBJECT_NAME": cannot change DSRL entry"
                    " state");
                break;
            }

            //reset variable to avoid dealloc
            hNewSkiCond = SKI_CONDITIONS_INVALID_OBJECT;

            DATASERVICE_IMPL_vLog(WEATHER_MGR_OBJECT_NAME
                ": DSRL ski conditions entry has been updated\n"
                    );
        }

        bResult = TRUE;
    } while (FALSE);

    if (hNewSkiCond != SKI_CONDITIONS_INVALID_OBJECT)
    {
        SKI_CONDITIONS_vDestroy(hNewSkiCond);
    }

    return bResult;
}

/*******************************************************************************
*
*        bProcessSkiReport
*
*******************************************************************************/
static BOOLEAN bProcessSkiReport (
    WEATHER_SERVICE_OBJECT hWeatherService,
    LOC_ID tID,
    SKI_DATA_STRUCT *psData
        )
{
    BOOLEAN bResult = FALSE;

    do
    {
        DSRL_ENTRY_OBJECT hDSRLEntry = DSRL_ENTRY_INVALID_OBJECT;

        if ((hWeatherService == WEATHER_SERVICE_INVALID_OBJECT) ||
            (psData == NULL) || (tID == LOC_INVALID_ID))
        {
            break;
        }

        // searching for the entry by locid
        hDSRLEntry = hSearchEntry(
            ((WEATHER_MGR_OBJECT_STRUCT *)hWeatherService)->psAppObj, tID);
        if (hDSRLEntry == DSRL_ENTRY_INVALID_OBJECT)
        {
            printf(WEATHER_MGR_OBJECT_NAME": entry for loc_id %x"
                " not found, skipping\n", tID);
            bResult = TRUE;
            break;
        }

        // if entry found, applying the incoming data to entry
        bResult = bEntrySetSkiReport(hDSRLEntry,
            ((WEATHER_MGR_OBJECT_STRUCT *)hWeatherService)->psAppObj,
            psData);
    } while (FALSE);

    return bResult;
}

/*****************************************************************************
*
*   bAddLocation
*
*****************************************************************************/
static BOOLEAN bAddLocation (
    SQL_INTERFACE_OBJECT hConnection,
    char *pacBuffer,
    size_t tBufferSize,
    WEATHER_LOCATION_ROW_STRUCT *psLocationData
        )
{
    BOOLEAN bResult = FALSE, bInDB;

    if ((hConnection == SQL_INTERFACE_INVALID_OBJECT) ||
        (psLocationData == NULL))
    {
        return FALSE;
    }

    bInDB = bIsLocInDB(hConnection, pacBuffer, tBufferSize, psLocationData);
    if (bInDB == TRUE)
    {
        //remove location
        bResult = bDeleteLocFromDB(
            hConnection, pacBuffer, tBufferSize, psLocationData);
        if (bResult == TRUE)
        {
            //adding location
            bResult = bAddLocToDB(
                hConnection, pacBuffer, tBufferSize, psLocationData);
        }
    }
    else
    {
        //adding location
        bResult = bAddLocToDB(hConnection, pacBuffer, tBufferSize, psLocationData);
    }

    printf(WEATHER_MGR_OBJECT_NAME": [%s] end with result %u\n", __FUNCTION__, bResult);

    return bResult;
}

/*****************************************************************************
*
*   bModifyLocation
*
*****************************************************************************/
static BOOLEAN bModifyLocation (
    SQL_INTERFACE_OBJECT hConnection,
    char *pacBuffer,
    size_t tBufferSize,
    WEATHER_LOCATION_ROW_STRUCT *psLocationData
        )
{
    BOOLEAN bResult = FALSE, bInDB;

    if ((hConnection == SQL_INTERFACE_INVALID_OBJECT) ||
        (psLocationData == NULL))
    {
        return FALSE;
    }


    bInDB = bIsLocInDB(
        hConnection, pacBuffer, tBufferSize, psLocationData);
    if (bInDB == TRUE)
    {
        //remove location
        bResult = bDeleteLocFromDB(
            hConnection, pacBuffer, tBufferSize, psLocationData);
        if (bResult == TRUE)
        {
            //add location
            bResult = bAddLocToDB(
                hConnection, pacBuffer, tBufferSize, psLocationData);
        }
    }
    else
    {
        //add location
        bResult = bAddLocToDB(
            hConnection, pacBuffer, tBufferSize, psLocationData);
    }

    return bResult;
}

/*****************************************************************************
*
*   bDeleteLocation
*
*****************************************************************************/
static BOOLEAN bDeleteLocation (
    SQL_INTERFACE_OBJECT hConnection,
    char *pacBuffer,
    size_t tBufferSize,
    WEATHER_LOCATION_ROW_STRUCT *psLocationData
        )
{
    BOOLEAN bResult = FALSE, bInDB;

    if ((hConnection == SQL_INTERFACE_INVALID_OBJECT) ||
        (psLocationData == NULL))
    {
        return FALSE;
    }

    // Is this location in the DB?
    bInDB = bIsLocInDB(
        hConnection, pacBuffer, tBufferSize, psLocationData);
    if (bInDB == TRUE)
    {
        //remove location
        bResult = bDeleteLocFromDB(
            hConnection, pacBuffer, tBufferSize, psLocationData);
    }

    return bResult;
}

/*****************************************************************************
*
*   bForecastHashRemoved
*
*****************************************************************************/
static BOOLEAN bForecastHashRemoved (
    WEATHER_SERVICE_OBJECT hWeatherService,
    WEATHER_HASH_OBJECT hHashEntry
        )
{
    BOOLEAN bResult = FALSE;
    WEATHER_APP_OBJECT_STRUCT *psAppObj = NULL;

    if ((hWeatherService == WEATHER_SERVICE_INVALID_OBJECT) ||
        (hHashEntry == WEATHER_HASH_INVALID_OBJECT))
    {
        return FALSE;
    }

    psAppObj = psGetAppFacingObject(hWeatherService);
    if (psAppObj != NULL)
    {
        vSkipForecastHashEntry(psAppObj, hHashEntry);

        bResult = TRUE;

        vUnlockAppFacingObject(psAppObj);
    }

    return bResult;
}

/*****************************************************************************
*
*   bSkiHashRemoved
*
*****************************************************************************/
static BOOLEAN bSkiHashRemoved (
    WEATHER_SERVICE_OBJECT hWeatherService,
    WEATHER_HASH_OBJECT hHashEntry
        )
{
    BOOLEAN bResult = FALSE;
    WEATHER_APP_OBJECT_STRUCT *psAppObj = NULL;

    if ((hWeatherService == WEATHER_SERVICE_INVALID_OBJECT) ||
        (hHashEntry == WEATHER_HASH_INVALID_OBJECT))
    {
        return FALSE;
    }

    psAppObj = psGetAppFacingObject(hWeatherService);
    if (psAppObj != NULL)
    {
        vSkipSkiCondHashEntry(psAppObj, hHashEntry);

        bResult = TRUE;

        vUnlockAppFacingObject(psAppObj);
    }

    return bResult;
}

/*****************************************************************************
*
*   bForecastHashRepeated
*
*****************************************************************************/
static BOOLEAN bForecastHashRepeated (
    WEATHER_SERVICE_OBJECT hWeatherService,
    WEATHER_HASH_OBJECT hHashEntry
        )
{
    BOOLEAN bResult = FALSE;
    WEATHER_APP_OBJECT_STRUCT *psAppObj = NULL;

    if ((hWeatherService == WEATHER_SERVICE_INVALID_OBJECT) ||
        (hHashEntry == WEATHER_HASH_INVALID_OBJECT))
    {
        return FALSE;
    }

    psAppObj = psGetAppFacingObject(hWeatherService);
    if (psAppObj != NULL)
    {
        vChangeForecastTimestamp(psAppObj, hHashEntry);
        bResult = TRUE;
        vUnlockAppFacingObject(psAppObj);
    }

    return bResult;
}

/*****************************************************************************
*
*   bSkiHashRepeated
*
*****************************************************************************/
static BOOLEAN bSkiHashRepeated (
    WEATHER_SERVICE_OBJECT hWeatherService,
    WEATHER_HASH_OBJECT hHashEntry
        )
{
    BOOLEAN bResult = FALSE;
    WEATHER_APP_OBJECT_STRUCT *psAppObj = NULL;

    if ((hWeatherService == WEATHER_SERVICE_INVALID_OBJECT) ||
        (hHashEntry == WEATHER_HASH_INVALID_OBJECT))
    {
        return FALSE;
    }

    psAppObj = psGetAppFacingObject(hWeatherService);
    if (psAppObj != NULL)
    {
        vChangeSkiCondTimestamp(psAppObj, (OSAL_LINKED_LIST_ENTRY)hHashEntry);
        bResult = TRUE;
        vUnlockAppFacingObject(psAppObj);
    }

    return bResult;
}

/*****************************************************************************
*
*   eGetReferenceDataVersion
*
*****************************************************************************/
DATASERVICE_ERROR_CODE_ENUM eGetReferenceDataVersion (
    const char *pcContainingDirectoryPath,
    DATASERVICE_REF_DATA_VER *ptCurrentRefDataVer,
    DATASERVICE_REF_DATA_VER *ptNextRefDataVer
        )
{
    DATASERVICE_ERROR_CODE_ENUM eReturnCode = DATASERVICE_ERROR_CODE_GENERAL;
    BOOLEAN bOk;
    char *pacDatabaseFilePathA = NULL,
         *pacDatabaseFilePathB = NULL;

    if (ptCurrentRefDataVer == NULL)
    {
        return DATASERVICE_ERROR_CODE_GENERAL;
    }

    do
    {
        // Create DB path resources
        bOk = DB_UTIL_bCreateFilePath(
            pcContainingDirectoryPath,
            WEATHER_DATABASE_FOLDER,
            WEATHER_REF_DATABASE_FILENAMEA,
            &pacDatabaseFilePathA);
        if (bOk != TRUE)
        {
            break;
        }

        bOk = DB_UTIL_bCreateFilePath(
            pcContainingDirectoryPath,
            WEATHER_DATABASE_FOLDER,
            WEATHER_REF_DATABASE_FILENAMEB,
            &pacDatabaseFilePathB);
        if (bOk != TRUE)
        {
            break;
        }

        // Check Weather databases
        eReturnCode =
            DB_UTIL_eCheckReferenceBanks (
                &pacDatabaseFilePathA[0],
                &pacDatabaseFilePathB[0],
                n32ExtractDataVersion, NULL,
                GsWeatherIntf.tMaxVersionBitlen,
                ptCurrentRefDataVer, ptNextRefDataVer );

    } while (FALSE);

    // Remove database path resources
    if (pacDatabaseFilePathA != NULL)
    {
        SMSO_vDestroy(
            (SMS_OBJECT)pacDatabaseFilePathA);
    }

    if (pacDatabaseFilePathB != NULL)
    {
        SMSO_vDestroy(
            (SMS_OBJECT)pacDatabaseFilePathB);
    }

    return eReturnCode;
}

/*****************************************************************************
                             PRIVATE FUNCTIONS
*****************************************************************************/

/*****************************************************************************
*
*   vEventHandler
*
*   This function runs in the context of an SMS resource which has been
*   assigned to this service.
*
*****************************************************************************/
static void vEventHandler (
    DATASERVICE_MGR_OBJECT hDataService,
    DATASERVICE_EVENT_MASK tCurrentEvent,
    void *pvEventArg,
    void *pvEventCallbackArg
        )
{
    WEATHER_MGR_OBJECT_STRUCT *psObj;
    BOOLEAN bValid, bStopEvent = FALSE;
    SMSAPI_EVENT_MASK tEventMask = DATASERVICE_EVENT_NONE;

    // Get our weather handle from the callback argument
    psObj = (WEATHER_MGR_OBJECT_STRUCT *)pvEventCallbackArg;

    // Is this object valid?
    bValid = DATASERVICE_IMPL_bValid((DATASERVICE_IMPL_HDL)psObj);

    printf(WEATHER_MGR_OBJECT_NAME" [%s] enter\n", __FUNCTION__);

    // Only handle events for valid objects...
    if (bValid == TRUE)
    {
        switch(tCurrentEvent)
        {
            // Handle Weather Service events here...

            // State has changed
            case DATASERVICE_EVENT_STATE:
            {
                BOOLEAN bStateChanged;
                DATASERVICE_STATE_CHANGE_STRUCT const *psStateChange =
                    (DATASERVICE_STATE_CHANGE_STRUCT const *)pvEventArg;

                // Process the state transition
                bStateChanged = DATASERVICE_IMPL_bStateFSM(
                    (DATASERVICE_IMPL_HDL)psObj,
                    psStateChange,
                    &GsTabWeatherStateHandlers,
                    (void *)psObj);

                if (bStateChanged == TRUE)
                {
                    // The state has been updated
                    tEventMask |= DATASERVICE_EVENT_STATE;

                    // Is the service stopped now?
                    if (psStateChange->eCurrentState ==
                            DATASERVICE_STATE_STOPPED)
                    {
                        bStopEvent = TRUE;
                    }
                }
            }
            break;

            // This service has a message to process
            case DATASERVICE_EVENT_NEW_DATA:
            {
                BOOLEAN bOk;

                // Event argument is a message payloadReceived
                OSAL_BUFFER_HDL hPayload =
                    (OSAL_BUFFER_HDL)pvEventArg;

                printf(WEATHER_MGR_OBJECT_NAME" New payload size: %u \n",
                       OSAL.tBufferGetSize(hPayload));
                // Handle the message reception
                bOk = bHandleMessageReception(psObj, hPayload);

                if (bOk != TRUE)
                {
                    // If an error occurred, indicate a state change
                    // to the application
                    puts(WEATHER_MGR_OBJECT_NAME" Payload processing failed");

                    DATASERVICE_IMPL_vLog(WEATHER_MGR_OBJECT_NAME
                        " Payload processing failed\n"
                            );
                }
                vStartCancelTimer(psObj);
            }
            break;

            // here we process weather service specific events
            case DATASERVICE_INTERNAL_EVENT_SERVICE_SPECIFIC:
            {
                WEATHER_MGR_EVENT_STRUCT *psEventData =
                    (WEATHER_MGR_EVENT_STRUCT *)pvEventArg;
                BOOLEAN bSuccess;

                if (psEventData == NULL)
                {
                    // looks like critical error
                    vSetError(psObj, DATASERVICE_ERROR_CODE_GENERAL);
                    break;
                }

                if (psEventData->bEnableSkiReports == TRUE)
                {
                    bSuccess = GsWeatherIntf.bEnableSkiReports(
                        psObj->hInterface);
                }
                else
                {
                    bSuccess = GsWeatherIntf.bDisableSkiReports(
                        psObj->hInterface);
                }

                if (bSuccess == FALSE)
                {
                    SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                        WEATHER_MGR_OBJECT_NAME": failed to %s ski reports",
                        ((psEventData->bEnableSkiReports == TRUE) ? "enable" 
                        : "disable"));
                }
            }
            break;

            // We need to do some work with our DSRLs
            case DATASERVICE_INTERNAL_EVENT_DSRL:
            {
                BOOLEAN bSuccess = FALSE;
                DSRL_ARG_STRUCT *psDSRLArg =
                    (DSRL_ARG_STRUCT *)pvEventArg;
                WEATHER_APP_OBJECT_STRUCT *psAppObj;

                if ((psDSRLArg == NULL) ||
                    (psDSRLArg->hDSRL == DSRL_INVALID_OBJECT))
                {
                    // error!
                    vSetError(psObj, DATASERVICE_ERROR_CODE_GENERAL);
                    break;
                }

                printf(WEATHER_MGR_OBJECT_NAME" Receiving DSRL event action type %d\n", psDSRLArg->eAction);

                psAppObj = psGetAppFacingObject((WEATHER_SERVICE_OBJECT)psObj);
                if (psAppObj == NULL)
                {
                    break;
                }

                switch (psDSRLArg->eAction)
                {
                    case DSRL_ACTION_ADD:
                    {
                        bSuccess = bCreateList(psAppObj, psDSRLArg);
                    }
                    break;

                    case DSRL_ACTION_MODIFY:
                    {
                        bSuccess = bModifyList(psAppObj, psDSRLArg);
                    }
                    break;

                    case DSRL_ACTION_REMOVE:
                    {
                        bSuccess = bDeleteList(psAppObj, psDSRLArg);
                    }
                    break;

                    case DSRL_ACTION_REFRESH:
                    {
                        bSuccess = bRefreshList(psAppObj, psDSRLArg);
                    }
                    break;

                    case DSRL_ACTION_INVALID:
                    default:
                    break;
                }

                // Go to the error state if the create/modify failed
                // or if we aren't ready
                if (bSuccess == FALSE)
                {
                    DATASERVICE_IMPL_vLog(WEATHER_MGR_OBJECT_NAME
                        " DSRL event processing failed\n"
                            );

                    bDSRLChangeState(psDSRLArg->hDSRL, DSRL_STATE_ERROR);

                    // If an error occurred, indicate a state change
                    // to the application
                    tEventMask |= DATASERVICE_EVENT_STATE;

                    // Indicate an error
                    vSetError(psObj, DATASERVICE_ERROR_CODE_GENERAL);

                    vUnlockAppFacingObject(psAppObj);
                }
                else
                {
                    if (psDSRLArg->hDSRL != DSRL_INVALID_OBJECT)
                    {
                        bSuccess = bLoadDataToEntries(psObj);
                        if (bSuccess == FALSE)
                        {
                            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile,
                                __LINE__,WEATHER_MGR_OBJECT_NAME
                                ": failed to load data for new entries"
                                    );
                        }
                        bDSRLChangeState(psDSRLArg->hDSRL, DSRL_STATE_READY);
                    }

                    bSuccess = bRemoveUnusedEntries(psAppObj);
                    if (bSuccess != TRUE)
                    {
                        SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                            WEATHER_MGR_OBJECT_NAME": failed to remove unused"
                            " entries"
                                );
                    }

                    vUnlockAppFacingObject(psAppObj);
                    vStartCancelTimer(psObj);
                }
            }
            break;

            case DATASERVICE_EVENT_TIMEOUT:
            {
                puts(WEATHER_MGR_OBJECT_NAME" Checking for msgs to cancel");
                vCheckForItemsToCancel(psObj);
            }
            break;

            default:
            {
                SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    WEATHER_MGR_OBJECT_NAME": Got unknown event (%u)",
                    tCurrentEvent);
            }
            break;
        }

        if (bStopEvent == TRUE)
        {
            // Uninitialize the object, but leave
            // enough to be useful to the callback
            vUninitObject( psObj, FALSE );
        }

        // Update event mask with any relevant events which have occurred
        SMSU_tUpdate(&psObj->sEvent, tEventMask);

        // Notify of any change via any registered callback which may be present
        SMSU_bNotify(&psObj->sEvent);

        if (bStopEvent == TRUE)
        {
            // Filter out all further weather manager updates
            SMSU_tFilter(&psObj->sEvent, DATASERVICE_EVENT_ALL);

            vDestroyObject(psObj);
        }
    }

    return;
}

/*****************************************************************************
*
*   vUninitObject
*
*****************************************************************************/
static void vUninitObject (
    WEATHER_MGR_OBJECT_STRUCT *psObj,
    BOOLEAN bFullDelete
        )
{
    BOOLEAN bLock;

    bLock = SMSO_bLock((SMS_OBJECT)psObj->psAppObj, OSAL_OBJ_TIMEOUT_INFINITE);
    if (bLock == TRUE)
    {
        printf(WEATHER_MGR_OBJECT_NAME" Stopping timer\n");

        // Clear the timed event handle, we're done with it
        psObj->hCancelEvent = DATASERVICE_TIMED_EVENT_INVALID_HDL;

        if (psObj->pacCurRefDatabaseFilePath != NULL)
        {
            // Remove database path resources
            SMSO_vDestroy((SMS_OBJECT)
                          psObj->pacCurRefDatabaseFilePath);
        }

        if (psObj->pacNextRefDatabaseFilePath != NULL)
        {
            // Remove database path resources
            SMSO_vDestroy((SMS_OBJECT)
                          psObj->pacNextRefDatabaseFilePath);
        }

        if (psObj->pacRefDatabaseDirPath != NULL)
        {
            // Remove database path resources
            SMSO_vDestroy((SMS_OBJECT)
                          psObj->pacRefDatabaseDirPath);
        }

        SMSO_vUnlock((SMS_OBJECT)psObj->psAppObj);

        // Releasing App interface object
        vUninitAppFacingObject(psObj);

        // Release plugin object
        if (psObj->hInterface != WEATHER_INTERFACE_INVALID_OBJECT)
        {
            GsWeatherIntf.vUnInit(psObj->hInterface);
            psObj->hInterface = WEATHER_INTERFACE_INVALID_OBJECT;
        }

        if (bFullDelete == TRUE)
        {
            vDestroyObject(psObj);
        }

        printf(WEATHER_MGR_OBJECT_NAME" %s end\n", __FUNCTION__);
    }
    else
    {
        SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
            WEATHER_MGR_OBJECT_NAME": Cannot lock App object. Uninit failed");
    }

    return;
}

/*****************************************************************************
*
*   vDestroyObject
*
*****************************************************************************/
static void vDestroyObject (
    WEATHER_MGR_OBJECT_STRUCT *psObj
        )
{
    // Destroy the SMS Update Event object
    SMSU_vDestroy(&psObj->sEvent);

    printf(WEATHER_MGR_OBJECT_NAME" %s end\n", __FUNCTION__);

    return;
}

/*****************************************************************************
*
*   bHandleServiceReady
*
*   This function is called when SMS is ready for the service to startup.
*   At this time, the service has a context in which to operate (SMS
*   assigned a resource to us). When this call is made, this service will
*   perform all of its time-intensive startup procedures.
*
*****************************************************************************/
static BOOLEAN bHandleServiceReady (
    WEATHER_MGR_OBJECT_STRUCT *psObj
        )
{
    BOOLEAN bOk = FALSE, bServiceUpdated = FALSE;
    WEATHER_APP_OBJECT_STRUCT *psAppObj = NULL;
    DATASERVICE_ERROR_CODE_ENUM eErrorCode =
        DATASERVICE_ERROR_CODE_NONE;
    char *pacDatabaseFilePath,
         *pacDatabaseFilePathB;
    UN8 un8DBVer = 0;

    do
    {
        // Validate the hService, then get a pointer to the
        // locked app-facing object
        psAppObj = psGetAppFacingObject((WEATHER_SERVICE_OBJECT)psObj);
        if (psAppObj == NULL)
        {
            // Error, not sure why this would happen
            vSetError( psObj, DATASERVICE_ERROR_CODE_GENERAL );
            break;
        }

        // Register for a timed event
        psObj->hCancelEvent =
            DATASERVICE_IMPL_hRegisterTimedEvent(
                (DATASERVICE_IMPL_HDL)psObj, (void *)NULL);

        if(psObj->hCancelEvent == DATASERVICE_TIMED_EVENT_INVALID_HDL)
        {
            // Error!
            vSetError( psObj, DATASERVICE_ERROR_CODE_GENERAL );
            break;
        }

        // Construct the path needed for the reference database
        bOk = DB_UTIL_bCreateFilePath(
            psObj->pacRefDatabaseDirPath,
            WEATHER_DATABASE_FOLDER,
            WEATHER_REF_DATABASE_FILENAMEA,
            &pacDatabaseFilePath);

        if (bOk == FALSE)
        {
            // Error!  Couldn't build database path
            eErrorCode = DATASERVICE_ERROR_CODE_GENERAL;
            break;
        }

        // Construct the path needed for the reference database
        bOk = DB_UTIL_bCreateFilePath(
            psObj->pacRefDatabaseDirPath,
            WEATHER_DATABASE_FOLDER,
            WEATHER_REF_DATABASE_FILENAMEB,
            &pacDatabaseFilePathB);

        if (bOk == FALSE)
        {
            // Error!  Couldn't build database path
            eErrorCode = DATASERVICE_ERROR_CODE_GENERAL;
            break;
        }

        // Connect to the ref database bank, but
        // never perform an integrity check
        psAppObj->hSQLRefConnection =
            DB_UTIL_hConnectToReferenceBank(
                &pacDatabaseFilePath[0],
                &pacDatabaseFilePathB[0],
                &psObj->pacCurRefDatabaseFilePath,
                n32ExtractDataVersion, NULL,
                TRUE, &bServiceUpdated,
                GsWeatherIntf.tMaxVersionBitlen,
                (DB_UTIL_CHECK_VERSION_HANDLER)bVerifyDBSchemaVersion,
                (void *)(size_t)WEATHER_DATABASE_FILE_VERSION,
                &eErrorCode,
                SQL_INTERFACE_OPTIONS_READONLY |
                SQL_INTERFACE_OPTIONS_SKIP_CORRUPTION_CHECK);

        if (psObj->pacCurRefDatabaseFilePath == pacDatabaseFilePath)
        {
            psObj->pacNextRefDatabaseFilePath = pacDatabaseFilePathB;
        }
        else
        {
            psObj->pacNextRefDatabaseFilePath = pacDatabaseFilePath;
        }

        // Now, we have to actually use the data version from
        // the selected connection
        un8DBVer = (UN8)n32ExtractDataVersion(psAppObj->hSQLRefConnection, NULL);

        // If the connection succeeded, continue with
        // initial processing
        if (psAppObj->hSQLRefConnection == SQL_INTERFACE_INVALID_OBJECT)
        {
            vSetError(psObj, DATASERVICE_ERROR_CODE_DATABASE_NOT_FOUND);
            break;
        }

        if (psObj->hInterface == WEATHER_INTERFACE_INVALID_OBJECT)
        {
            // Start the interface
            psObj->hInterface = GsWeatherIntf.hInit(
                (WEATHER_SERVICE_OBJECT)psObj,
                DATASERVICE_IMPL_hSMSObj((DATASERVICE_IMPL_HDL)psObj),
                psObj->bRefDBUpdatesEnabled,
                bServiceUpdated, un8DBVer);
            if (psObj->hInterface == WEATHER_INTERFACE_INVALID_OBJECT)
            {
                vSetError(psObj, DATASERVICE_ERROR_CODE_GENERAL);
                break;
            }
        }

    } while(FALSE);

    vUnlockAppFacingObject(psAppObj);

    vStartCancelTimer(psObj);

    return bOk;
}

/*****************************************************************************
*
*   bHandleServiceError
*
*****************************************************************************/
static BOOLEAN bHandleServiceError (
    WEATHER_MGR_OBJECT_STRUCT *psObj
        )
{
    // Stop any ageout timer we have now 'cause
    // who knows what's going on up there
    vStopCancelTimer(psObj);

    return TRUE;
}

/*****************************************************************************
*
*   vSetError
*
*****************************************************************************/
static void vSetError (
    WEATHER_MGR_OBJECT_STRUCT *psObj,
    DATASERVICE_ERROR_CODE_ENUM eErrorCode
        )
{
    SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
       WEATHER_MGR_OBJECT_NAME": vSetError (%d)", eErrorCode
           );

    // Tell the DSM about it
    DATASERVICE_IMPL_vError((DATASERVICE_IMPL_HDL)psObj, eErrorCode);

    return;
}

/*****************************************************************************
*
*   bHandleMessageReception
*
*   This function is called when a new message is received by the
*   weather service.
*
*****************************************************************************/
static BOOLEAN bHandleMessageReception (
    WEATHER_MGR_OBJECT_STRUCT *psObj,
    OSAL_BUFFER_HDL hPayload
        )
{
    BOOLEAN bOk = TRUE;

    // Ensure the payload handle is valid.
    // If it isn't, whatevs, we'll just
    // ignore it.
    if (hPayload == OSAL_INVALID_BUFFER_HDL)
    {
        // Set the error
        vSetError(psObj, DATASERVICE_ERROR_CODE_GENERAL);
        return FALSE;
    }

    bOk = GsWeatherIntf.bProcessMessage(psObj->hInterface, &hPayload);

    // Free the payload, we're done with it
    DATASERVICE_IMPL_bFreeDataPayload(hPayload);

    return bOk;
}

/*****************************************************************************
*
*   psGetAppFacingObject
*
*****************************************************************************/
static WEATHER_APP_OBJECT_STRUCT *psGetAppFacingObject(
    WEATHER_SERVICE_OBJECT hService
        )
{
    WEATHER_MGR_OBJECT_STRUCT *psObj =
        (WEATHER_MGR_OBJECT_STRUCT *)hService;
    BOOLEAN bValid, bLocked;

    do
    {
        bValid = DATASERVICE_IMPL_bValid((DATASERVICE_IMPL_HDL)hService);

        if (bValid == FALSE)
        {
            break;
        }

        bLocked = SMSO_bLock((SMS_OBJECT)psObj->psAppObj,
            OSAL_OBJ_TIMEOUT_INFINITE);

        if (bLocked == FALSE)
        {
            break;
        }

        return psObj->psAppObj;

    } while (FALSE);

    return NULL;
}

/*****************************************************************************
*
*   vUnlockAppFacingObject
*
*****************************************************************************/
void vUnlockAppFacingObject(
    WEATHER_APP_OBJECT_STRUCT *psAppObj
        )
{
    if (psAppObj != NULL)
    {
        SMSO_vUnlock((SMS_OBJECT)psAppObj);
    }

    return;
}

/*****************************************************************************
*
*   n16SortByLocID
*
*****************************************************************************/
static N16 n16SortByLocID(
    WEATHER_MSG_OBJECT hWeatherMsg1,
    WEATHER_MSG_OBJECT hWeatherMsg2
        )
{
    N16 n16Result = N16_MIN;

    if ((hWeatherMsg1 != WEATHER_MSG_INVALID_OBJECT) &&
        (hWeatherMsg2 != WEATHER_MSG_INVALID_OBJECT))
    {
        LOC_ID tID1, tID2;

        tID1 = WEATHER_MSG_tGetLOCID(hWeatherMsg1);
        tID2 = WEATHER_MSG_tGetLOCID(hWeatherMsg2);

        n16Result = COMPARE(tID1, tID2);
    }

    return n16Result;
}

/*****************************************************************************
*
*   bInitAppFacingObject
*
*****************************************************************************/
static BOOLEAN bInitAppFacingObject(
    WEATHER_MGR_OBJECT_STRUCT *psObj
        )
{

    OSAL_RETURN_CODE_ENUM eReturnCode = OSAL_ERROR;
    BOOLEAN bResult = FALSE;

    do
    {
        DATASERVICE_DSRL_CONFIG_STRUCT sDSRLConfig;
        LOCID_OBJECT hDummyLocId;

        // Begin DSRL config
        DATASERVICE_IMPL_vInitializeDSRLConfig(&sDSRLConfig);

        // Set service type
        sDSRLConfig.eServiceType = DATASERVICE_TYPE_WEATHER;

        // Configure object size
        sDSRLConfig.tParentObjectSize = sizeof(WEATHER_APP_OBJECT_STRUCT);

        // Set the DSLR descriptor size
        sDSRLConfig.tServiceDataSize = sizeof(WEATHER_DSRL_DESC_STRUCT);

        // Configure the favorites feature
        sDSRLConfig.bFavoritesEnabled = TRUE;
        sDSRLConfig.hCreateTargetFromTag = LOCATION_hCreateTargetFromTag;
        sDSRLConfig.hGetTagForTarget = LOCATION_hGetTargetTag;

        // Configure the device feature
        sDSRLConfig.bDeviceEnabled = TRUE;
        sDSRLConfig.un32DeviceNotifyDistance =
            WEATHER_DEVICE_DISTANCE_THRESHOLD;
        sDSRLConfig.eDeviceNotifyUnits = DISTANCE_UNIT_TYPE_MILES;

        // Create it
        psObj->psAppObj = (WEATHER_APP_OBJECT_STRUCT *)
            DATASERVICE_IMPL_hConfigureDSRL(
                (DATASERVICE_IMPL_HDL)psObj, &sDSRLConfig);

        if (psObj->psAppObj == NULL)
        {
            break;
        }

        // creating the unsorted list of DSRL handles
        eReturnCode = OSAL.eLinkedListCreate(
                &(psObj->psAppObj->hDSRLList),
                WEATHER_MGR_OBJECT_NAME":DSRLList",
                NULL,
                OSAL_LL_OPTION_NONE
                    );
        if (eReturnCode != OSAL_SUCCESS)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                WEATHER_MGR_OBJECT_NAME": cannot create linked list (%s)",
                OSAL.pacGetReturnCodeName(eReturnCode));
            break;
        }

        // creating the list of entries sorted by handles
        eReturnCode = OSAL.eLinkedListCreate(
                &(psObj->psAppObj->hEntriesList),
                WEATHER_MGR_OBJECT_NAME":EntriesList",
                (OSAL_LL_COMPARE_HANDLER)n16SortByLocID,
                OSAL_LL_OPTION_NONE
                    );
        if (eReturnCode != OSAL_SUCCESS)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                WEATHER_MGR_OBJECT_NAME": cannot create linked list (%s)",
                OSAL.pacGetReturnCodeName(eReturnCode));
            break;
        }

        eReturnCode = OSAL.eLinkedListCreate(
            &( psObj->psAppObj->hRemovedEntriesList ),
            WEATHER_MGR_OBJECT_NAME":RemovedEntriesList",
            n16CompareHandles,
            OSAL_LL_OPTION_UNIQUE
                );
        if (eReturnCode != OSAL_SUCCESS)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                WEATHER_MGR_OBJECT_NAME": cannot create linked list (%s)",
                OSAL.pacGetReturnCodeName(eReturnCode));
            break;
        }

        eReturnCode = OSAL.eLinkedListCreate(
                &(psObj->psAppObj->hNewForecastEntriesList),
                WEATHER_MGR_OBJECT_NAME":NewForecastEntriesList",
                NULL,
                OSAL_LL_OPTION_LINEAR|OSAL_LL_OPTION_UNIQUE
                    );
        if (eReturnCode != OSAL_SUCCESS)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                WEATHER_MGR_OBJECT_NAME": cannot create linked list (%s)",
                OSAL.pacGetReturnCodeName(eReturnCode));
            break;
        }

        eReturnCode = OSAL.eLinkedListCreate(
                &(psObj->psAppObj->hNewSkiEntriesList),
                WEATHER_MGR_OBJECT_NAME":NewSkiEntriesList",
                NULL,
                OSAL_LL_OPTION_LINEAR|OSAL_LL_OPTION_UNIQUE
                    );
        if (eReturnCode != OSAL_SUCCESS)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                WEATHER_MGR_OBJECT_NAME": cannot create linked list (%s)",
                OSAL.pacGetReturnCodeName(eReturnCode));
            break;
        }

        eReturnCode = OSAL.eLinkedListCreate(
                &(psObj->psAppObj->hNewDSRLEntriesList),
                WEATHER_MGR_OBJECT_NAME":hNewDSRLEntriesList",
                n16CompareHandles,
                OSAL_LL_OPTION_UNIQUE
                    );
        if (eReturnCode != OSAL_SUCCESS)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                WEATHER_MGR_OBJECT_NAME": cannot create linked list (%s)",
                OSAL.pacGetReturnCodeName(eReturnCode)
                    );
            break;
        }

        eReturnCode = OSAL.eLinkedListCreate(
                &(psObj->psAppObj->hEntriesToAddList),
                WEATHER_MGR_OBJECT_NAME":EntriesToAddList",
                NULL,
                OSAL_LL_OPTION_NONE
                    );
        if (eReturnCode != OSAL_SUCCESS)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                WEATHER_MGR_OBJECT_NAME": cannot create add entry list (%s)",
                OSAL.pacGetReturnCodeName(eReturnCode)
                    );
            break;
        }

        eReturnCode = OSAL.eLinkedListCreate(
                &(psObj->psAppObj->hEntriesToRemoveList),
                WEATHER_MGR_OBJECT_NAME":EntriesToRemoveList",
                NULL,
                OSAL_LL_OPTION_NONE
                    );
        if (eReturnCode != OSAL_SUCCESS)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                WEATHER_MGR_OBJECT_NAME": cannot create remove entry"
                " list (%s)",
                OSAL.pacGetReturnCodeName(eReturnCode)
                    );
            break;
        }

        // Creating dummy LocID so valid LOCATION can be created for it.
        hDummyLocId = LOCID_hCreate(0, LOCID_TYPE_WEATHER, NULL);

        psObj->psAppObj->hDummyLocation = LOCATION_hCreate(
            (SMS_OBJECT)psObj->psAppObj,
            hDummyLocId, 0, 0, 0, 0, DISTANCE_INVALID_OBJECT,
            NULL, FALSE);
        if (psObj->psAppObj->hDummyLocation == LOCATION_INVALID_OBJECT)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                WEATHER_MGR_OBJECT_NAME": cannot create dummy location");
            break;
        }

        psObj->psAppObj->hDummyMsg = WEATHER_MSG_hCreateDummy(
            (SMS_OBJECT)psObj->psAppObj, psObj->psAppObj->hDummyLocation);
        if (psObj->psAppObj->hDummyMsg == WEATHER_MSG_INVALID_OBJECT)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                WEATHER_MGR_OBJECT_NAME": cannot create dummy message");
            break;
        }

        bResult = TRUE;
    } while (FALSE);

    if (NULL != psObj->psAppObj)
    {
        // Relinquish ownership
        SMSO_vUnlock((SMS_OBJECT)psObj->psAppObj);
    }

    return bResult;
}

/*****************************************************************************
*
*   vDSRLEntryRelease
*
*****************************************************************************/
static void vDSRLEntryRelease(
    DSRL_ENTRY_OBJECT hDSRLEntry
        )
{
    if (hDSRLEntry != DSRL_ENTRY_INVALID_OBJECT)
    {
        WEATHER_DSRL_ENTRY_DESC_STRUCT *psEntryDesc;

        printf(WEATHER_MGR_OBJECT_NAME": Releasing entry %p\n", hDSRLEntry);

        psEntryDesc =
            (WEATHER_DSRL_ENTRY_DESC_STRUCT *)DSRL_ENTRY_pvServiceData(
            hDSRLEntry);
        if (psEntryDesc != NULL)
        {
            if (psEntryDesc->hDSRLList != OSAL_INVALID_OBJECT_HDL)
            {
                OSAL_RETURN_CODE_ENUM eReturnCode;

                eReturnCode = OSAL.eLinkedListRemoveAll(psEntryDesc->hDSRLList,
                    NULL);
                if (eReturnCode != OSAL_SUCCESS)
                {
                    SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                        WEATHER_MGR_OBJECT_NAME": cannot cleanup list (%s)",
                        OSAL.pacGetReturnCodeName(eReturnCode));
                }

                eReturnCode = OSAL.eLinkedListDelete(psEntryDesc->hDSRLList);
                if (eReturnCode != OSAL_SUCCESS)
                {
                    SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                        WEATHER_MGR_OBJECT_NAME": cannot delete list (%s)",
                        OSAL.pacGetReturnCodeName(eReturnCode));
                }

                psEntryDesc->hDSRLList = OSAL_INVALID_OBJECT_HDL;
            }
        }

        WEATHER_MSG_vDestroy((WEATHER_MSG_OBJECT)hDSRLEntry);
    }

    return;
}

/*****************************************************************************
*
*   vUninitAppFacingObject
*
*****************************************************************************/
static void vUninitAppFacingObject(
    WEATHER_MGR_OBJECT_STRUCT *psObj
        )
{
    BOOLEAN bLocked;
    WEATHER_APP_OBJECT_STRUCT *psAppObj = psObj->psAppObj;
    OSAL_RETURN_CODE_ENUM eReturnCode = OSAL_ERROR;

    // Lock the app object
    bLocked = SMSO_bLock((SMS_OBJECT)psAppObj, OSAL_OBJ_TIMEOUT_INFINITE);
    if (bLocked == FALSE)
    {
        SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
            WEATHER_MGR_OBJECT_NAME": Unable to lock app object");
        return;
    }

    // Disconnect from the database application
    if (psAppObj->hSQLRefConnection != SQL_INTERFACE_INVALID_OBJECT)
    {
        SQL_INTERFACE.vDisconnect(psAppObj->hSQLRefConnection);
        psAppObj->hSQLRefConnection = SQL_INTERFACE_INVALID_OBJECT;
    }

    puts(WEATHER_MGR_OBJECT_NAME" Destroying DSRL nodes.");

    if (psAppObj->hDSRLList != OSAL_INVALID_OBJECT_HDL)
    {
        eReturnCode = OSAL.eLinkedListRemoveAll(psAppObj->hDSRLList,
            (OSAL_LL_RELEASE_HANDLER)vDSRLRelease
                );
        if (eReturnCode != OSAL_SUCCESS)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                WEATHER_MGR_OBJECT_NAME": failed to remove DSRL items (%s)",
                OSAL.pacGetReturnCodeName(eReturnCode)
                    );
        }

        eReturnCode = OSAL.eLinkedListDelete(psAppObj->hDSRLList);
        if (eReturnCode != OSAL_SUCCESS)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                WEATHER_MGR_OBJECT_NAME": failed to delete DSRLs list (%s)",
                OSAL.pacGetReturnCodeName(eReturnCode)
                    );
        }

        psAppObj->hDSRLList = OSAL_INVALID_OBJECT_HDL;
    }

    puts(WEATHER_MGR_OBJECT_NAME" Destroying Entry nodes");
    if (psAppObj->hEntriesList != OSAL_INVALID_OBJECT_HDL)
    {
        eReturnCode = OSAL.eLinkedListRemoveAll(psAppObj->hEntriesList,
            (OSAL_LL_RELEASE_HANDLER)vDSRLEntryRelease
                );
        if (eReturnCode != OSAL_SUCCESS)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                WEATHER_MGR_OBJECT_NAME": failed to remove entry items (%s)",
                OSAL.pacGetReturnCodeName(eReturnCode)
                    );
        }

        eReturnCode = OSAL.eLinkedListDelete(psAppObj->hEntriesList);
        if (eReturnCode != OSAL_SUCCESS)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                WEATHER_MGR_OBJECT_NAME": failed to delete entry list (%s)",
                OSAL.pacGetReturnCodeName(eReturnCode)
                    );
        }

        psAppObj->hEntriesList = OSAL_INVALID_OBJECT_HDL;
    }

    if (psAppObj->hRemovedEntriesList != OSAL_INVALID_OBJECT_HDL)
    {
        eReturnCode =
            OSAL.eLinkedListRemoveAll(psAppObj->hRemovedEntriesList, NULL);
        if (eReturnCode != OSAL_SUCCESS)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                WEATHER_MGR_OBJECT_NAME": failed to clean removed entries"
                " list (%s)",
                OSAL.pacGetReturnCodeName(eReturnCode)
                    );
        }

        eReturnCode =
            OSAL.eLinkedListDelete(psAppObj->hRemovedEntriesList);
        if (eReturnCode != OSAL_SUCCESS)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                WEATHER_MGR_OBJECT_NAME": failed to delete removed"
                " entries list (%s)",
                OSAL.pacGetReturnCodeName(eReturnCode)
                    );
        }

        psAppObj->hRemovedEntriesList = OSAL_INVALID_OBJECT_HDL;
    }

    if (psAppObj->hNewForecastEntriesList != OSAL_INVALID_OBJECT_HDL)
    {
        OSAL.eLinkedListRemoveAll(psAppObj->hNewForecastEntriesList, NULL);
        OSAL.eLinkedListDelete(psAppObj->hNewForecastEntriesList);
    }

    if (psAppObj->hNewSkiEntriesList != OSAL_INVALID_OBJECT_HDL)
    {
        OSAL.eLinkedListRemoveAll(psAppObj->hNewSkiEntriesList, NULL);
        OSAL.eLinkedListDelete(psAppObj->hNewSkiEntriesList);
    }

    if (psAppObj->hNewDSRLEntriesList != OSAL_INVALID_OBJECT_HDL)
    {
        eReturnCode = OSAL.eLinkedListRemoveAll(psAppObj->hNewDSRLEntriesList,
            NULL);
        if (eReturnCode != OSAL_SUCCESS)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                WEATHER_MGR_OBJECT_NAME": cannot cleanup linked list (%s)",
                OSAL.pacGetReturnCodeName(eReturnCode)
                    );
        }

        eReturnCode = OSAL.eLinkedListDelete(psAppObj->hNewDSRLEntriesList);
        if (eReturnCode != OSAL_SUCCESS)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                WEATHER_MGR_OBJECT_NAME": cannot delete linked list (%s)",
                OSAL.pacGetReturnCodeName(eReturnCode)
                    );
        }

        psAppObj->hNewDSRLEntriesList = OSAL_INVALID_OBJECT_HDL;
    }

    if (psAppObj->hEntriesToAddList != OSAL_INVALID_OBJECT_HDL)
    {
        eReturnCode = OSAL.eLinkedListRemoveAll(psAppObj->hEntriesToAddList,
            NULL);
        if (eReturnCode != OSAL_SUCCESS)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                WEATHER_MGR_OBJECT_NAME": cannot cleanup"
                " add entry list (%s)",
                OSAL.pacGetReturnCodeName(eReturnCode)
                    );
        }

        eReturnCode = OSAL.eLinkedListDelete(psAppObj->hEntriesToAddList);
        if (eReturnCode != OSAL_SUCCESS)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                WEATHER_MGR_OBJECT_NAME": cannot delete add entry"
                " list (%s)",
                OSAL.pacGetReturnCodeName(eReturnCode)
                    );
        }

        psAppObj->hEntriesToAddList = OSAL_INVALID_OBJECT_HDL;
    }

    if (psAppObj->hEntriesToRemoveList != OSAL_INVALID_OBJECT_HDL)
    {
        eReturnCode = OSAL.eLinkedListRemoveAll(psAppObj->hEntriesToRemoveList,
            NULL);
        if (eReturnCode != OSAL_SUCCESS)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                WEATHER_MGR_OBJECT_NAME": cannot cleanup"
                " remove entry list (%s)",
                OSAL.pacGetReturnCodeName(eReturnCode)
                    );
        }

        eReturnCode = OSAL.eLinkedListDelete(psAppObj->hEntriesToRemoveList);
        if (eReturnCode != OSAL_SUCCESS)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                WEATHER_MGR_OBJECT_NAME": cannot delete remove entry"
                " list (%s)",
                OSAL.pacGetReturnCodeName(eReturnCode)
                    );
        }
        psAppObj->hEntriesToRemoveList = OSAL_INVALID_OBJECT_HDL;
    }

    puts(WEATHER_MGR_OBJECT_NAME" Entry nodes destroyed.");

    if (psAppObj->hDummyMsg != WEATHER_MSG_INVALID_OBJECT)
    {
        WEATHER_MSG_vDestroy(psAppObj->hDummyMsg);
        psAppObj->hDummyMsg = WEATHER_MSG_INVALID_OBJECT;
        psAppObj->hDummyLocation = LOCATION_INVALID_OBJECT;
    }
    else if (psAppObj->hDummyLocation != LOCATION_INVALID_OBJECT)
    {
        LOCATION.vDestroy(psAppObj->hDummyLocation);
        psAppObj->hDummyLocation = LOCATION_INVALID_OBJECT;
    }

    // Destroy the app object
    DATASERVICE_IMPL_vDestroyDSRLParent((DATASERVICE_IMPL_HDL)psObj, (SMS_OBJECT)psAppObj);
    psObj->psAppObj = NULL;

    return;
}

/*****************************************************************************
*
*   n32ExtractDataVersion
*
*****************************************************************************/
static N32 n32ExtractDataVersion (
    SQL_INTERFACE_OBJECT hConnection,
    void *pvArg
        )
{
    N32 n32Result = DB_UTIL_DB_UNDER_CONSTRUCTION_VER;
    N16 n16ContentVer = 0;
    BOOLEAN bLoaded;

    if (hConnection == SQL_INTERFACE_INVALID_OBJECT)
    {
        return DB_UTIL_DB_UNDER_CONSTRUCTION_VER;
    }

    // Get the Content Version (table!)
    bLoaded = bLoadDBContentVersion(
        hConnection, &n16ContentVer);
    if (bLoaded == TRUE)
    {
        n32Result = (N32)n16ContentVer;
    }

    return n32Result;
}

/*****************************************************************************
*
*   bLoadDBContentVersion
*
*****************************************************************************/
static BOOLEAN bLoadDBContentVersion (
    SQL_INTERFACE_OBJECT hSQLConnection,
    N16 *pn16DBVer
        )
{
    WEATHER_DB_QUERY_RESULT_STRUCT sQueryResult;
    BOOLEAN bOk;

    if ((hSQLConnection == SQL_INTERFACE_INVALID_OBJECT) ||
        (pn16DBVer == NULL))
    {
        return FALSE;
    }

    sQueryResult.bResultantRows = FALSE;

    // Perform the SQL query and process the result
    // (it will provide us with a data row)
    bOk = SQL_INTERFACE.bQuery(
        hSQLConnection,
        WEATHER_SELECT_UPDATE_VERSION,
        (SQL_QUERY_RESULT_HANDLER)bProcessSelectUpdateVersionResult,
        &sQueryResult);

    if ((bOk == TRUE) &&
        (sQueryResult.bResultantRows == TRUE))
    {
        *pn16DBVer = sQueryResult.uDbRow.sUpdateVersion.n16Version;
    }

    return sQueryResult.bResultantRows;
}

/*****************************************************************************
*
*   bVerifyDBSchemaVersion
*
*****************************************************************************/
static BOOLEAN bVerifyDBSchemaVersion (
    SQL_INTERFACE_OBJECT hSQLConnection,
    UN8 un8SchemaVer
        )
{
    WEATHER_DB_QUERY_RESULT_STRUCT sQueryResult;
    BOOLEAN bOk, bVersionMatched = FALSE;

    if (hSQLConnection == SQL_INTERFACE_INVALID_OBJECT)
    {
        return FALSE;
    }

    // Perform the SQL query and process the result
    // (it will provide us with a data row)
    bOk = SQL_INTERFACE.bQuery(
        hSQLConnection,
        WEATHER_SELECT_DB_VERSION,
        bProcessSelectDBVersionResult,
        &sQueryResult);

    if ((bOk == TRUE) &&
        (sQueryResult.bResultantRows == TRUE))
    {
        WEATHER_VERSION_ROW_STRUCT *psVersionRow =
            &sQueryResult.uDbRow.sVersion;

        if (psVersionRow->un8DBVer == un8SchemaVer)
        {
            // Version match!
            bVersionMatched = TRUE;
        }
    }

    return bVersionMatched;
}

/*******************************************************************************
*
*   bProcessSelectDBVersionResult
*
*   This function is provided to the SQL_INTERFACE_OBJECT in order to process
*   a "select all" query made on the database version relation.  It populates a
*   pointer to a WEATHER_VERSION_ROW_STRUCT with the information found
*   in the database.
*
*******************************************************************************/
static BOOLEAN bProcessSelectDBVersionResult (
    SQL_QUERY_COLUMN_STRUCT *psColumns,
    N32 n32NumberOfColumns,
    void *pvArg
        )
{
    DB_VERSION_FIELDS_ENUM eCurrentField =
        (DB_VERSION_FIELDS_ENUM)0;
    WEATHER_DB_QUERY_RESULT_STRUCT *psResult =
        (WEATHER_DB_QUERY_RESULT_STRUCT *)pvArg;
    WEATHER_VERSION_ROW_STRUCT *psVersionRow;
    SQL_QUERY_COLUMN_STRUCT *psCurrentCol;

    // Verify input
    if (psResult == NULL)
    {
        return FALSE;
    }

    // If there are the correct
    // number of columns, then we have good results
    if (n32NumberOfColumns == DB_VERSION_MAX_FIELDS)
    {
        psResult->bResultantRows = TRUE;
    }
    else
    {
        psResult->bResultantRows = FALSE;

        return FALSE;
    }

    // Get a more convenient pointer
    psVersionRow = &psResult->uDbRow.sVersion;

    // Clear the version structure
    OSAL.bMemSet(psVersionRow, 0, sizeof(WEATHER_VERSION_ROW_STRUCT));

    // We have just performed an SQL query in order to
    // determine the version of the database.
    // Process the results (should be only one row)
    do
    {

        // Grab the current column data
        psCurrentCol = &psColumns[(UN8)eCurrentField];

        // Decode the current field based on it's type
        switch (eCurrentField)
        {
            case DB_VERSION_FIELD_DB_VER:
            {
                psVersionRow->un8DBVer =
                        (UN8)psCurrentCol->uData.sUN32.un32Data;
            }
            break;

            case DB_VERSION_FIELD_DSI:
            {
                psVersionRow->tDSI =
                        (UN8)psCurrentCol->uData.sUN32.un32Data;
            }
            break;

            default: // Shouldn't happen
            break;
        }
    } while ( ++eCurrentField < DB_VERSION_MAX_FIELDS);

    if (eCurrentField == DB_VERSION_MAX_FIELDS)
    {
        return TRUE;
    }

    return FALSE;
}

/*******************************************************************************
*
*   bProcessSelectUpdateVersionResult
*
*   This function is provided to the SQL_INTERFACE_OBJECT in order to process
*   a "select all" query made on the location update database version relation.
*   It populates a pointer to a UPDATE_VERSION_ROW_STRUCT with the information
*   found in the database.
*
*******************************************************************************/
static BOOLEAN bProcessSelectUpdateVersionResult (
    SQL_QUERY_COLUMN_STRUCT *psColumns,
    N32 n32NumberOfColumns,
    WEATHER_DB_QUERY_RESULT_STRUCT *psResult
        )
{
    UPDATE_VERSION_FIELDS_ENUM eCurrentField =
        (UPDATE_VERSION_FIELDS_ENUM)0;

    // Verify input
    if (psResult == NULL)
    {
        return FALSE;
    }

    // If there are the correct
    // number of columns, then we have good results
    if (n32NumberOfColumns == UPDATE_VERSION_MAX_FIELDS)
    {
        psResult->bResultantRows = TRUE;
    }
    else
    {
        psResult->bResultantRows = FALSE;

        return FALSE;
    }

    // We have just performed an SQL query in order to
    // determine the update version in the database.
    // Process the results (should be only one row)
    do
    {
        // Decode the current field based on it's type
        switch (eCurrentField)
        {
            case UPDATE_VERSION:
            {
                psResult->uDbRow.sUpdateVersion.n16Version =
                    (N16)psColumns[eCurrentField].uData.sUN32.un32Data;
            }
            break;

            default: // Shouldn't happen
            break;
        }
    } while (++eCurrentField < UPDATE_VERSION_MAX_FIELDS);

    return FALSE;
}

/*******************************************************************************
*
*        bIsLocInDB
*
*******************************************************************************/
static BOOLEAN bIsLocInDB(
    SQL_INTERFACE_OBJECT hConnection,
    char *pacBuffer,
    size_t tBufferSize,
    WEATHER_LOCATION_ROW_STRUCT *psLocationData
            )
{
    BOOLEAN bResult = FALSE;
    int iReturn;

    iReturn = snprintf(
        &pacBuffer[0], tBufferSize,
        WEATHER_SELECT_LOCATION,
        psLocationData->tID);

    if (iReturn > 0)
    {
        // look in database
        bResult = SQL_INTERFACE.bQuery(
            hConnection, &pacBuffer[0],
            (SQL_QUERY_RESULT_HANDLER)bCheckLocInDB,
            &bResult);

        if (bResult == FALSE)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                WEATHER_MGR_OBJECT_NAME
                ": Unable to get location record loc %d.",
                psLocationData->tID);
        }
    }

    return bResult;
}

/*******************************************************************************
*
*        bCheckLocInDB
*
*******************************************************************************/
static BOOLEAN bCheckLocInDB(
        SQL_QUERY_COLUMN_STRUCT *psColumn,
        N32 n32NumberOfColumns,
        BOOLEAN *pbLocExists
            )
{
    // Verify input
    if (pbLocExists == NULL)
    {
        return FALSE;
    }

    // If there is at least one data row and the correct
    // number of columns (just make sure we have enough),
    // then we have good results
    if (n32NumberOfColumns < LOCATION_MAX_FIELDS)
    {
        return FALSE;
    }

    *pbLocExists = TRUE;

    return TRUE;
}

/*******************************************************************************
*
*        bExtractLocationFromDB
*
*******************************************************************************/
static BOOLEAN bExtractLocationFromDB (
    SQL_QUERY_COLUMN_STRUCT *psColumn,
    N32 n32NumberOfColumns,
    WEATHER_FILTER_LOCATIONS_LIST_STRUCT *psFilterList
        )
{
    BOOLEAN bContinue = FALSE;
    LOCID_OBJECT hLocID = LOCID_INVALID_OBJECT;

    do
    {
        LOC_ID tLocId = (LOC_ID)
            psColumn[LOCATION_FIELD_LOC_ID].uData.sUN32.un32Data;
        size_t tLength = 0;
        const char *pacAreaName = (const char *)
            psColumn[LOCATION_FIELD_AREA_NAME].uData.sCString.pcData;
        STATE_ID tState = GET_STATE_ID_FROM_LOC_ID(tLocId);

        // Do we have the correct state?
        if ((psFilterList->tStateId != 0) &&
            (tState != psFilterList->tStateId))
        {
            // if state is different from requested, just skip to the next
            bContinue = TRUE;
            break;
        }

        // creating loc_id object
        hLocID = LOCID_hCreate(tLocId, LOCID_TYPE_WEATHER, NULL);
        if (hLocID == LOCID_INVALID_OBJECT)
        {
            // error!
            break;
        }

        if (pacAreaName == NULL)
        {
            // area name could be empty in some cases. Lets assign space to
            // avoid issues in the string creation
            pacAreaName = " ";
        }

        if (psFilterList->hAreaName == STRING_INVALID_OBJECT)
        {
            // if area name string is not created yet, create new string
            tLength = strlen(pacAreaName);
            psFilterList->hAreaName = STRING_hCreate(SMS_INVALID_OBJECT,
                pacAreaName, tLength, 0);
        }
        else
        {
            // update the existing string. Since the string handle is provided
            // to the location object before, location description will
            // be updated by this call
            STRING.bModifyCStr(psFilterList->hAreaName, pacAreaName);
        }

        if ((psFilterList->hStateName == STRING_INVALID_OBJECT) ||
            (psFilterList->tCurrentStateId != tState))
        {
            // if state name is not created yet, or state id is different 
            // from the current, lets just create(update) it for this state

            STRING_OBJECT hStateName;

            // state name could not exist or could not be created for this
            // state. Let's put one space string as an initial value
            const char * pacStateName = " ";
            tLength = 1;

            hStateName = LOCATION_hStateAbbrvForID(tState);
            if (hStateName != STRING_INVALID_OBJECT)
            {
                // since its a constant string, we'll just extract the 
                // content 
                pacStateName = STRING.pacCStr(hStateName);
                tLength = strlen(pacStateName);
            }

            if (psFilterList->hStateName == STRING_INVALID_OBJECT)
            {
                psFilterList->hStateName = STRING_hCreate(
                    SMS_INVALID_OBJECT, pacStateName, tLength, 0);
            }
            else
            {
                STRING.bModifyCStr(psFilterList->hStateName, 
                    pacStateName);
            }

            // now we storing the state id to detect the change
            // of the state id later (for next locations)
            psFilterList->tCurrentStateId = tState;

            // cleanup
            if (hStateName != STRING_INVALID_OBJECT)
            {
                STRING_vDestroy(hStateName);
            }
        }

        if (psFilterList->hLocation == LOCATION_INVALID_OBJECT)
        {
            LOCATION_ATTRIBUTE_STRUCT sAttrs;

            OSAL.bMemSet(&sAttrs, 0, sizeof(sAttrs));

            sAttrs.hDescription = psFilterList->hAreaName;

            // Set the state name
            sAttrs.hState = psFilterList->hStateName;

            // create location using the loc id, area name string
            // and coordinates
            psFilterList->hLocation = LOCATION_hCreate(SMS_INVALID_OBJECT,
                hLocID,
                (N32)
                psColumn[LOCATION_FIELD_LATITUDE].uData.sUN32.un32Data,
                LOCATION_BINPOINT,
                (N32)
                psColumn[LOCATION_FIELD_LONGITUDE].uData.sUN32.un32Data,
                LOCATION_BINPOINT,
                DISTANCE_INVALID_OBJECT,
                &sAttrs,
                FALSE);
            if (psFilterList->hLocation == LOCATION_INVALID_OBJECT)
            {
                // error!
                // need to remove the LOCID object
                LOCID.vDestroy(hLocID);

                // need to remove the STRING objects
                STRING_vDestroy(psFilterList->hAreaName);
                psFilterList->hAreaName = STRING_INVALID_OBJECT;

                STRING_vDestroy(psFilterList->hStateName);
                psFilterList->hStateName = STRING_INVALID_OBJECT;

                break;
            }
        }
        else
        {
            BOOLEAN bSuccess = FALSE;

            // updating the existing object coordinates
            bSuccess = LOCATION_bUpdateCoordinates(psFilterList->hLocation,
                (N32)
                psColumn[LOCATION_FIELD_LATITUDE].uData.sUN32.un32Data,
                LOCATION_BINPOINT,
                (N32)
                psColumn[LOCATION_FIELD_LONGITUDE].uData.sUN32.un32Data,
                LOCATION_BINPOINT);
            if (bSuccess == FALSE)
            {
                // error!
                break;
            }

            // updating the existing object locid
            bSuccess = LOCATION_bUpdateLocID(psFilterList->hLocation,
                hLocID, FALSE);
            if (bSuccess == FALSE)
            {
                // error!
                // need to remove the LOCID object
                LOCID.vDestroy(hLocID);
                break;
            }
        }

        // calling the app function to provide the location object to app
        bContinue = psFilterList->bIterator(psFilterList->hLocation,
            psFilterList->pvArg);
    } while (FALSE);

    return bContinue;
}

/*******************************************************************************
*
*        bDeleteLocFromDB
*
*******************************************************************************/
static BOOLEAN bDeleteLocFromDB (
    SQL_INTERFACE_OBJECT hConnection,
    char *pacBuffer,
    size_t tBufferSize,
    WEATHER_LOCATION_ROW_STRUCT *psLocationData
            )
{
    BOOLEAN bResult = FALSE;
    int iReturn;

    // Build our update string for this new version using our
    // DB's schema version to find the version row
    iReturn = snprintf( &pacBuffer[0], tBufferSize,
        WEATHER_DELETE_LOCATION, psLocationData->tID);

    if (iReturn > 0)
    {
        // update in database
        bResult = SQL_INTERFACE.bExecuteCommand(
            hConnection, &pacBuffer[0]);

        if (bResult == FALSE)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                WEATHER_MGR_OBJECT_NAME
                ": Unable to delete location record state loc %d",
                psLocationData->tID);
        }
        else
        {
            // Build our string to remove the locations_rtree row from the database
            iReturn = snprintf( &pacBuffer[0], tBufferSize,
                WEATHER_DELETE_LOCATION_RTREE, psLocationData->tID);

            if (iReturn > 0)
            {
                bResult = SQL_INTERFACE.bExecuteCommand(
                    hConnection, &pacBuffer[0]);

                if (bResult == FALSE)
                {
                    SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                            WEATHER_MGR_OBJECT_NAME
                            ": Unable to delete location rtree");
                }
            }
        }
    }

    return bResult;
}

/*******************************************************************************
*
*   bQueryLocationsFromDB
*
*******************************************************************************/
static BOOLEAN bQueryLocationsFromDB(
    WEATHER_APP_OBJECT_STRUCT *psAppObj,
    WEATHER_FILTER_LOCATIONS_LIST_STRUCT *psFilterList
        )
{
    BOOLEAN bResult = FALSE;
    int iReturn = -1;

    if (psFilterList->hFilterString == STRING_INVALID_OBJECT)
    {
        // creating the query based on ski flag only
        iReturn = snprintf(
            psAppObj->acBuffer, sizeof(psAppObj->acBuffer),
            WEATHER_SELECT_ALL_LOCATIONS_WITH_FLAG,
            psFilterList->bWeatherSkiFlag);
        if (iReturn > 0)
        {
            // good, lets run our query
            bResult = SQL_INTERFACE.bQuery(
                psAppObj->hSQLRefConnection,
                psAppObj->acBuffer,
                (SQL_QUERY_RESULT_HANDLER)bExtractLocationFromDB,
                psFilterList);
        }
    }
    else
    {
        // creating the query based on filter string and ski flag
        iReturn = snprintf(
            psAppObj->acBuffer, sizeof(psAppObj->acBuffer),
            WEATHER_SELECT_ALL_LOCATIONS_WITH_FLAG_AND_UPPER_AREA_NAME,
            psFilterList->bWeatherSkiFlag);
        if (iReturn > 0)
        {
            SQL_PREPARED_STATEMENT_HANDLE hStatement;
            hStatement = SQL_INTERFACE.hCreatePreparedStatement(
                psAppObj->hSQLRefConnection,
                psAppObj->acBuffer);
            if (hStatement != SQL_PREPARED_STATEMENT_INVALID_HANDLE)
            {
                SQL_BIND_PARAMETER_STRUCT sParam;
                sParam.eType = SQL_BIND_TYPE_STRING_OBJECT;
                sParam.tSize = 0;
                sParam.pvData = psFilterList->hFilterString;
                bResult = SQL_INTERFACE.bExecutePreparedStatement(
                    psAppObj->hSQLRefConnection,
                    hStatement,
                    (SQL_QUERY_RESULT_HANDLER)bExtractLocationFromDB, psFilterList,
                    1, &sParam);

                SQL_INTERFACE.vDestroyPreparedStatement(psAppObj->hSQLRefConnection, hStatement);
            }
        }
    }

    if (psFilterList->hLocation != LOCATION_INVALID_OBJECT)
    {
        LOCATION.vDestroy(psFilterList->hLocation);
        psFilterList->hLocation = LOCATION_INVALID_OBJECT;
    }

    return bResult;
}

/*******************************************************************************
*
*        bAddLocToDB
*
*******************************************************************************/
static BOOLEAN bAddLocToDB(
    SQL_INTERFACE_OBJECT hConnection,
    char *pacBuffer,
    size_t tBufferSize,
    WEATHER_LOCATION_ROW_STRUCT *psLocationData
        )
{
    BOOLEAN bResult = FALSE, bTemp = FALSE;
    N32 n32Lat, n32Lon;
    int iReturn;

    do {

        bTemp = SQL_INTERFACE.bExecutePreparedCommand(
            hConnection,
            WEATHER_INSERT_LOCATION,
            LOCATION_MAX_FIELDS,
            (PREPARED_QUERY_COLUMN_CALLBACK)bPrepareLocationRow,
            (void *)psLocationData);

        if (bTemp == FALSE)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                WEATHER_MGR_OBJECT_NAME": Failed to insert location"
                    );
            break;
        }

        // Insert record into the R-Tree table
        n32Lat = OSAL_FIXED.n32ScaledValue(
            psLocationData->hLat, LOCATION_BINPOINT);

        n32Lon = OSAL_FIXED.n32ScaledValue(
            psLocationData->hLon, LOCATION_BINPOINT);

        iReturn = snprintf(&pacBuffer[0], tBufferSize,
                WEATHER_INSERT_LOCATION_RTREE,
                psLocationData->tID,
                n32Lat - WEATHER_RTREE_RANGE,
                n32Lat + WEATHER_RTREE_RANGE,
                n32Lon - WEATHER_RTREE_RANGE,
                n32Lon + WEATHER_RTREE_RANGE);

        if (iReturn == 0)
        {
            break;
        }

        bResult = SQL_INTERFACE.bExecuteCommand(
            hConnection, &pacBuffer[0]);
        if (bResult == FALSE)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                WEATHER_MGR_OBJECT_NAME": Failed to insert location rtree"
                    );
        }
    } while (FALSE);

    return bResult;
}

/*******************************************************************************
*
*        bPrepareLocationRow
*
*******************************************************************************/
static BOOLEAN bPrepareLocationRow(
        SQL_COLUMN_INDEX tIndex,
        SQL_BIND_TYPE_ENUM *peType,
        size_t *ptDataSize,
        void **ppvData,
        WEATHER_LOCATION_ROW_STRUCT *psLoc
            )
{
    BOOLEAN bSuccess = TRUE;
    N32 n32FixedValue;

    switch (tIndex)
    {
        case LOCATION_FIELD_LOC_ID:
        {
            *peType = SQL_BIND_TYPE_UN32;
            *ppvData = (void *)(size_t)psLoc->tID;
        }
        break;

        case LOCATION_FIELD_VERSION:
        {
            *peType = SQL_BIND_TYPE_UN32;
            *ppvData = (void *)(size_t)psLoc->un8Version;
        }
        break;

        case LOCATION_FIELD_SKI_FLAG:
        {
            *peType = SQL_BIND_TYPE_UN32;
            *ppvData = (void *)(size_t)psLoc->bSkiFlag;
        }
        break;

        case LOCATION_FIELD_AREA_NAME:
        {
            *peType = SQL_BIND_TYPE_STRING_OBJECT;
            *ppvData = psLoc->hAreaName;
        }
        break;

        case LOCATION_FIELD_ICAO_NAME:
        {
            *peType = SQL_BIND_TYPE_STRING_OBJECT;
            *ppvData = psLoc->hICAOName;
        }
        break;


        case LOCATION_FIELD_LATITUDE:
        {
            n32FixedValue = OSAL_FIXED.n32ScaledValue(
                psLoc->hLat, WEATHER_LOCATION_LATLON_BINPOINT);

            *peType = SQL_BIND_TYPE_UN32;
            *ppvData = (void *)(size_t)(UN32)n32FixedValue;
        }
        break;

        case LOCATION_FIELD_LONGITUDE:
        {
            n32FixedValue = OSAL_FIXED.n32ScaledValue(
                psLoc->hLon, WEATHER_LOCATION_LATLON_BINPOINT);

            *peType = SQL_BIND_TYPE_UN32;
            *ppvData = (void *)(size_t)(UN32)n32FixedValue;
        }
        break;

        case LOCATION_MAX_FIELDS:
        default:
            bSuccess = FALSE;
        break;

    }

    return bSuccess;
}

/*****************************************************************************
*
*   bDSRLDescInit
*
*****************************************************************************/
static BOOLEAN bDSRLDescInit(
    DSRL_ARG_STRUCT *psDSRLArg,
    WEATHER_APP_OBJECT_STRUCT *psAppObj
        )
{
    WEATHER_DSRL_DESC_STRUCT *psDSRLDesc = (WEATHER_DSRL_DESC_STRUCT *)NULL;
    OSAL_RETURN_CODE_ENUM eReturnCode = OSAL_ERROR;
    BOOLEAN bResult = FALSE;

    do
    {
        // get the DSRL desc pointer from DSRL
        psDSRLDesc = (WEATHER_DSRL_DESC_STRUCT *)DSRL_pvServiceData(
            psDSRLArg->hDSRL);
        if (psDSRLDesc == NULL)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                WEATHER_MGR_OBJECT_NAME": cannot get DSRL desc from DSRL"
                    );
            break;
        }

        // creating the list of DSRL targets
        eReturnCode = OSAL.eLinkedListCreate(
            &psDSRLDesc->hTargetList,
            WEATHER_MGR_OBJECT_NAME":DSRLDesc:TgtList",
            NULL,
            OSAL_LL_OPTION_NONE
                );
        if (eReturnCode != OSAL_SUCCESS)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                WEATHER_MGR_OBJECT_NAME": cannot create targets list (%s)",
                OSAL.pacGetReturnCodeName(eReturnCode)
                    );
            break;
        }

        // creating the list of DSRL entries
        eReturnCode = OSAL.eLinkedListCreate(
            &psDSRLDesc->hEntriesList,
            WEATHER_MGR_OBJECT_NAME":DSRLDesc:EntriesList",
            n16CompareHandles,
            OSAL_LL_OPTION_NONE
                );
        if (eReturnCode != OSAL_SUCCESS)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                WEATHER_MGR_OBJECT_NAME": cannot create entries list (%s)",
                OSAL.pacGetReturnCodeName(eReturnCode)
                    );
            break;
        }

        // Add this DSRL to our list of tracked DSRLs
        eReturnCode = OSAL.eLinkedListAdd(psAppObj->hDSRLList,
            &(psDSRLDesc->hDSRLListEntry), psDSRLArg->hDSRL);
        if (eReturnCode != OSAL_SUCCESS)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                WEATHER_MGR_OBJECT_NAME": cannot add DSRL to pool (%s)",
                OSAL.pacGetReturnCodeName(eReturnCode)
                    );
            break;
        }

        bResult = TRUE;
    } while (FALSE);

    if ((bResult == FALSE) && (psDSRLDesc != NULL))
    {
        if (psDSRLDesc->hDSRLListEntry != OSAL_INVALID_LINKED_LIST_ENTRY)
        {
            eReturnCode = OSAL.eLinkedListRemove(psDSRLDesc->hDSRLListEntry);
            if (eReturnCode != OSAL_SUCCESS)
            {
                SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    WEATHER_MGR_OBJECT_NAME": cannot remove DSRL from pool (%s)",
                    OSAL.pacGetReturnCodeName(eReturnCode)
                        );
            }
            psDSRLDesc->hDSRLListEntry = OSAL_INVALID_LINKED_LIST_ENTRY;
        }
        vDSRLRelease(psDSRLArg->hDSRL);
    }

    return bResult;
}

/*******************************************************************************
*
*    bCreateList
*
*******************************************************************************/
static BOOLEAN bCreateList (
    WEATHER_APP_OBJECT_STRUCT *psAppObj,
    DSRL_ARG_STRUCT *psDSRLArg
        )
{
    BOOLEAN bResult = FALSE;

    do
    {
        BOOLEAN bSuccess = FALSE;

        printf(WEATHER_MGR_OBJECT_NAME": Creating DSRL\n");

        // Initialize DSRL
        bSuccess = bDSRLDescInit(psDSRLArg, psAppObj);
        if (bSuccess == FALSE)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                WEATHER_MGR_OBJECT_NAME": Failed to init DSRL desc for"
                " DSRL %p",
                psDSRLArg->hDSRL
                    );
            break;
        }

        // Set the finalizer
        bSuccess = DSRL_bSetFinalizeFunction(psDSRLArg->hDSRL,
            (DSRL_FINALIZE_FUNCTION)vDSRLFinalizeEntry, psAppObj
                );
        if (bSuccess == FALSE)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                WEATHER_MGR_OBJECT_NAME": failed to set finalizer for DSRL %p",
                psDSRLArg->hDSRL
                    );
            break;
        }

        // Add targets
        bSuccess = bDSRLAddTargets(psDSRLArg);
        if (bSuccess == FALSE)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                WEATHER_MGR_OBJECT_NAME": failed to add targets to DSRL %p",
                psDSRLArg->hDSRL
                    );
            break;
        }

        // Update content
        bSuccess = bDSRLUpdateContent(psDSRLArg->hDSRL, psAppObj);
        if (bSuccess == FALSE)
        {
            break;
        }

        // Looks good
        bResult = TRUE;
    } while (FALSE);

    return bResult;
}

/*****************************************************************************
*
*   bDSRLRemoveEntryLink
*
*****************************************************************************/
static BOOLEAN bDSRLRemoveEntryLink(
    DSRL_OBJECT hDSRL,
    DSRL_ENTRY_OBJECT hDSRLEntry
        )
{
    BOOLEAN bResult = FALSE;

    do
    {
        WEATHER_DSRL_DESC_STRUCT *psDSRLDesc = NULL;
        OSAL_LINKED_LIST_ENTRY hEntry = OSAL_INVALID_LINKED_LIST_ENTRY;
        OSAL_RETURN_CODE_ENUM eReturnCode = OSAL_ERROR;

        psDSRLDesc = (WEATHER_DSRL_DESC_STRUCT *)DSRL_pvServiceData(hDSRL);
        if (psDSRLDesc == NULL)
        {
            break;
        }

        // searching for DSRL entry handle in the list of DSRL entries
        // and removing this DSRL entry from list
        eReturnCode = OSAL.eLinkedListSearch(psDSRLDesc->hEntriesList,
            &hEntry, hDSRLEntry);
        if (eReturnCode != OSAL_SUCCESS)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                WEATHER_MGR_OBJECT_NAME": failed to find entry"
                " in list (%s)",
                OSAL.pacGetReturnCodeName(eReturnCode));
            break;
        }

        eReturnCode = OSAL.eLinkedListRemove(hEntry);
        if (eReturnCode != OSAL_SUCCESS)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                WEATHER_MGR_OBJECT_NAME": failed to remove entry"
                " from list (%s)",
                OSAL.pacGetReturnCodeName(eReturnCode));
            break;
        }

        bResult = TRUE;
    } while (FALSE);

    return bResult;
}

/*****************************************************************************
*
*   bEntryRemoveDSRLLink
*
*****************************************************************************/
BOOLEAN bEntryRemoveDSRLLink(
    WEATHER_APP_OBJECT_STRUCT *psAppObj,
    DSRL_ENTRY_OBJECT hDSRLEntry,
    DSRL_OBJECT hDSRL
        )
{
    BOOLEAN bResult = FALSE;
    WEATHER_DSRL_ENTRY_DESC_STRUCT *psEntryDesc = NULL;

    // get Weather DSRL entry desc
    psEntryDesc = (WEATHER_DSRL_ENTRY_DESC_STRUCT *)
        DSRL_ENTRY_pvServiceData(hDSRLEntry);

    if ((psEntryDesc != NULL) &&
        (psEntryDesc->hDSRLList != OSAL_INVALID_OBJECT_HDL))
    {
        OSAL_RETURN_CODE_ENUM eReturnCode = OSAL_ERROR;
        OSAL_LINKED_LIST_ENTRY hEntry = OSAL_INVALID_LINKED_LIST_ENTRY;

        // looking for DSRL handle in list of DSRL handles for this entry
        // and removing the DSRL handle from the list
        eReturnCode = OSAL.eLinkedListSearch(psEntryDesc->hDSRLList,
            &hEntry, hDSRL
                );
        if (eReturnCode == OSAL_SUCCESS)
        {
            eReturnCode = OSAL.eLinkedListRemove(hEntry);
            if (eReturnCode != OSAL_SUCCESS)
            {
                SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    WEATHER_MGR_OBJECT_NAME": failed to remove DSRL desc"
                    " from list (%s)",
                    OSAL.pacGetReturnCodeName(eReturnCode)
                        );
            }

            // adding entry to list of removed entries
            eReturnCode = OSAL.eLinkedListAdd(
                psAppObj->hRemovedEntriesList, NULL, hDSRLEntry
                    );
            if ((eReturnCode != OSAL_SUCCESS) &&
                (eReturnCode != OSAL_ERROR_LIST_ITEM_NOT_UNIQUE))
            {
                SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    WEATHER_MGR_OBJECT_NAME": failed to add DSRL entry to list"
                    " of removed entries (%s)",
                    OSAL.pacGetReturnCodeName(eReturnCode)
                        );
            }
            else
            {
                bResult = TRUE;
            }
        }
        else
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                WEATHER_MGR_OBJECT_NAME": failed to find DSRL desc"
                " in list (%s)",
                OSAL.pacGetReturnCodeName(eReturnCode)
                    );
        }
    }

    return bResult;
}

/*******************************************************************************
*
*    bDeleteList
*
*******************************************************************************/
static BOOLEAN bDeleteList(
    WEATHER_APP_OBJECT_STRUCT *psAppObj,
    DSRL_ARG_STRUCT *psDSRLArg
        )
{
    BOOLEAN bResult = FALSE;

    do
    {
        WEATHER_DSRL_DESC_STRUCT *psDSRLDesc = NULL;
        OSAL_RETURN_CODE_ENUM eReturnCode = OSAL_ERROR;

        printf(WEATHER_MGR_OBJECT_NAME": Removing DSRL %p\n",
            psDSRLArg->hDSRL
                );

        // Remove / free all entries in the DSRL
        DSRL_vRemoveAllEntries(psDSRLArg->hDSRL);

        psDSRLDesc = (WEATHER_DSRL_DESC_STRUCT *)DSRL_pvServiceData(
            psDSRLArg->hDSRL
                );
        if (psDSRLDesc == NULL)
        {
            break;
        }

        eReturnCode = OSAL.eLinkedListRemove(psDSRLDesc->hDSRLListEntry);
        if (eReturnCode != OSAL_SUCCESS)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                WEATHER_MGR_OBJECT_NAME": cannot remove DSRL from pool (%s)",
                OSAL.pacGetReturnCodeName(eReturnCode)
                    );
        }

        // destroy the DSRL descriptor data
        vDSRLRelease(psDSRLArg->hDSRL);
        psDSRLArg->hDSRL = DSRL_INVALID_OBJECT;

        bResult = TRUE;
    } while (FALSE);

    return bResult;
}

/*****************************************************************************
*
*   bEntryRemoveFromDSRL
*
*****************************************************************************/
static BOOLEAN bEntryRemoveFromDSRL(
    DSRL_ENTRY_OBJECT hDSRLEntry,
    WEATHER_DSRL_ENTRIES_LIST_ITERATOR_STRUCT *psIterator
        )
{
    BOOLEAN bContinue = FALSE;

    if ((hDSRLEntry != DSRL_ENTRY_INVALID_OBJECT) &&
        (psIterator != NULL))
    {
        OSAL_RETURN_CODE_ENUM eReturnCode;

        DSRL_vRemoveEntry(psIterator->hDSRL, hDSRLEntry);
        bContinue = TRUE;

        // place entry to the list of unused entries
        // to destroy it later
        eReturnCode = OSAL.eLinkedListAdd(
            psIterator->psAppObj->hRemovedEntriesList, NULL, hDSRLEntry
                );
        if ((eReturnCode != OSAL_SUCCESS) &&
            (eReturnCode != OSAL_ERROR_LIST_ITEM_NOT_UNIQUE))
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                WEATHER_MGR_OBJECT_NAME": failed to add entry to list"
                " of removed entries (%s)",
                OSAL.pacGetReturnCodeName(eReturnCode)
                    );
        }
    }

    return bContinue;
}

/*****************************************************************************
*
*   bDSRLRemoveEntries
*
*****************************************************************************/
static BOOLEAN bDSRLRemoveEntries(
    WEATHER_APP_OBJECT_STRUCT *psAppObj,
    DSRL_OBJECT hDSRL
        )
{
    BOOLEAN bResult = FALSE;

    do
    {
        OSAL_RETURN_CODE_ENUM eReturnCode;
        WEATHER_DSRL_ENTRIES_LIST_ITERATOR_STRUCT sIterator;

        sIterator.hDSRL = hDSRL;
        sIterator.psAppObj = psAppObj;

        // remove obsolete entries from DSRL if there are any
        eReturnCode = OSAL.eLinkedListIterate(psAppObj->hEntriesToRemoveList,
            (OSAL_LL_ITERATOR_HANDLER)bEntryRemoveFromDSRL, &sIterator
                );
        if (eReturnCode == OSAL_NO_OBJECTS)
        {
            printf(WEATHER_MGR_OBJECT_NAME": nothing to remove, "
                "list is empty\n"
                    );
        }
        else if (eReturnCode != OSAL_SUCCESS)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                WEATHER_MGR_OBJECT_NAME": cannot iterate remove entry"
                " list (%s)",
                OSAL.pacGetReturnCodeName(eReturnCode)
                    );
            break;
        }

        // clean up the list
        eReturnCode = OSAL.eLinkedListRemoveAll(psAppObj->hEntriesToRemoveList,
            NULL);
        if (eReturnCode != OSAL_SUCCESS)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                WEATHER_MGR_OBJECT_NAME": cannot cleanup entries"
                " to remove list (%s)", OSAL.pacGetReturnCodeName(eReturnCode));
            break;
        }

        bResult = TRUE;
    } while(FALSE);

    return bResult;
}

/*****************************************************************************
*
*   bEntryAddToDSRL
*
*****************************************************************************/
static BOOLEAN bEntryAddToDSRL(
    DSRL_ENTRY_OBJECT hDSRLEntry,
    WEATHER_DSRL_ENTRIES_LIST_ITERATOR_STRUCT *psIterator
        )
{
    BOOLEAN bContinue = FALSE, bAdded = FALSE;
    OSAL_RETURN_CODE_ENUM eReturnCode = OSAL_ERROR;

    do
    {
        DSRL_ADD_REPLACE_RESULT_ENUM eDSRLResult = DSRL_ADD_REPLACE_ERROR;
        WEATHER_DSRL_DESC_STRUCT *psDSRLDesc = NULL;
        WEATHER_DSRL_ENTRY_DESC_STRUCT *psEntryDesc = NULL;

        if ((psIterator == NULL) ||
            (hDSRLEntry == DSRL_ENTRY_INVALID_OBJECT))
        {
            break;
        }

        psEntryDesc = (WEATHER_DSRL_ENTRY_DESC_STRUCT *)
            DSRL_ENTRY_pvServiceData(hDSRLEntry);
        if (psEntryDesc == NULL)
        {
            break;
        }

        eDSRLResult = DSRL_eAddEntry(psIterator->hDSRL, hDSRLEntry);
        if (eDSRLResult != DSRL_ADD_REPLACE_OK)
        {
            printf(WEATHER_MGR_OBJECT_NAME": entry cannot be "
                "added for some reason.\n"
                    );
            bContinue = TRUE;
            break;
        }
        bAdded = TRUE;

        psDSRLDesc =
            (WEATHER_DSRL_DESC_STRUCT *)DSRL_pvServiceData(psIterator->hDSRL);
        if (psDSRLDesc == NULL)
        {
            break;
        }

        // adding the DSRL entry handle to list of DSRL entries for this DSRL
        eReturnCode = OSAL.eLinkedListAdd(psDSRLDesc->hEntriesList,
            OSAL_INVALID_LINKED_LIST_ENTRY_PTR, hDSRLEntry
                );
        if (eReturnCode != OSAL_SUCCESS)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                WEATHER_MGR_OBJECT_NAME": cannot add entry"
                " to list (%s)",
                OSAL.pacGetReturnCodeName(eReturnCode)
                    );
            break;
        }

        if (psEntryDesc->hDSRLList == OSAL_INVALID_OBJECT_HDL)
        {
            // in case if DSRLs list in EntryDesc struct is not created
            // lets create it
            eReturnCode = OSAL.eLinkedListCreate(&(psEntryDesc->hDSRLList),
                WEATHER_MGR_OBJECT_NAME":DSRLEntry:DSRLList",
                n16CompareHandles, OSAL_LL_OPTION_NONE
                    );
            if (eReturnCode != OSAL_SUCCESS)
            {
                SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    WEATHER_MGR_OBJECT_NAME": cannot create link list (%s)",
                    OSAL.pacGetReturnCodeName(eReturnCode)
                        );
                break;
            }
        }

        // adding the DSRL handle to list of DSRLs for this entry
        eReturnCode = OSAL.eLinkedListAdd(psEntryDesc->hDSRLList,
            OSAL_INVALID_LINKED_LIST_ENTRY_PTR, psIterator->hDSRL
                );
        if (eReturnCode != OSAL_SUCCESS)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                WEATHER_MGR_OBJECT_NAME": cannot add entry"
                " to list (%s)",
                OSAL.pacGetReturnCodeName(eReturnCode)
                    );
            break;
        }

        bContinue = TRUE;
    } while (FALSE);

    if ((bContinue == TRUE) && (bAdded == FALSE))
    {
        // place entry to the list of unused entries
        // to destroy it later
        eReturnCode = OSAL.eLinkedListAdd(
            psIterator->psAppObj->hRemovedEntriesList, NULL, hDSRLEntry
                );
        if ((eReturnCode != OSAL_SUCCESS) &&
            (eReturnCode != OSAL_ERROR_LIST_ITEM_NOT_UNIQUE))
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                WEATHER_MGR_OBJECT_NAME": cannot add entry"
                " to list of removed entries (%s)",
                OSAL.pacGetReturnCodeName(eReturnCode)
                    );
        }

        //lets remove entry from DSRL to avoid any issues
        DSRL_vRemoveEntry(psIterator->hDSRL, hDSRLEntry);
    }

    return bContinue;
}

/*****************************************************************************
*
*   bDSRLAddEntries
*
*****************************************************************************/
static BOOLEAN bDSRLAddEntries(
    WEATHER_APP_OBJECT_STRUCT *psAppObj,
    DSRL_OBJECT hDSRL
        )
{
    BOOLEAN bResult = FALSE;

    do
    {
        OSAL_RETURN_CODE_ENUM eReturnCode;
        WEATHER_DSRL_ENTRIES_LIST_ITERATOR_STRUCT sIterator;

        sIterator.psAppObj = psAppObj;
        sIterator.hDSRL = hDSRL;

        // add all the entries from the list to the DSRL
        eReturnCode = OSAL.eLinkedListIterate(psAppObj->hEntriesToAddList,
            (OSAL_LL_ITERATOR_HANDLER)bEntryAddToDSRL, &sIterator
                );
        if (eReturnCode == OSAL_NO_OBJECTS)
        {
            printf(WEATHER_MGR_OBJECT_NAME": nothing to add, "
                "list is empty\n"
                    );
        }
        else if (eReturnCode != OSAL_SUCCESS)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                WEATHER_MGR_OBJECT_NAME": cannot iterate add entry"
                " list (%s)",
                OSAL.pacGetReturnCodeName(eReturnCode)
                    );
            break;
        }

        // cleanup the list
        eReturnCode = OSAL.eLinkedListRemoveAll(psAppObj->hEntriesToAddList,
            NULL);
        if (eReturnCode != OSAL_SUCCESS)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                WEATHER_MGR_OBJECT_NAME": cannot cleanup"
                " add entry list (%s)",
                OSAL.pacGetReturnCodeName(eReturnCode)
                    );
        }

        bResult = TRUE;
    } while(FALSE);

    return bResult;
}

/*****************************************************************************
*
*   bIsUpdateNeeded
*
*****************************************************************************/
static BOOLEAN bIsUpdateNeeded(
    OSAL_OBJECT_HDL hList1,
    OSAL_OBJECT_HDL hList2
        )
{
    UN32 un32Items = 0;
    OSAL_RETURN_CODE_ENUM eReturnCode = OSAL_ERROR;

    // get number of items in the list of entries
    eReturnCode = OSAL.eLinkedListItems(hList1, &un32Items);
    if (eReturnCode != OSAL_SUCCESS)
    {
        SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
            WEATHER_MGR_OBJECT_NAME": failed to get number of"
            " items in list (%s)",
            OSAL.pacGetReturnCodeName(eReturnCode)
                );
    }
    else if (un32Items > 0)
    {
        return TRUE;
    }

    un32Items = 0;
    // get number of items in the list of entries
    eReturnCode = OSAL.eLinkedListItems(hList2, &un32Items);
    if (eReturnCode != OSAL_SUCCESS)
    {
        SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
            WEATHER_MGR_OBJECT_NAME": failed to get number of"
            " items in list (%s)",
            OSAL.pacGetReturnCodeName(eReturnCode)
                );
    }
    else if (un32Items > 0)
    {
        return TRUE;
    }

    // nothing to add/remove
    DATASERVICE_IMPL_vLog(WEATHER_MGR_OBJECT_NAME
        ": DSRL update is not needed\n"
            );

    return FALSE;
}

/*****************************************************************************
*
*   bDSRLUpdateContent
*
*****************************************************************************/
static BOOLEAN bDSRLUpdateContent(
    DSRL_OBJECT hDSRL,
    WEATHER_APP_OBJECT_STRUCT *psAppObj
        )
{
    BOOLEAN bResult = FALSE;

    do
    {
        BOOLEAN bSuccess;

        // populate lists of entries to add and remove
        bSuccess = bDSRLFillDiffEntriesLists(hDSRL, psAppObj);
        if (bSuccess == FALSE)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                WEATHER_MGR_OBJECT_NAME": cannot fill dsrl entries diff lists"
                    );
            break;
        }

        bSuccess = bIsUpdateNeeded(psAppObj->hEntriesToAddList,
            psAppObj->hEntriesToRemoveList);
        if (bSuccess == FALSE)
        {
            //looks like we don't need to add or remove entries
            printf(WEATHER_MGR_OBJECT_NAME": DSRL %p update not required\n",
                hDSRL);
            bResult = TRUE;
            break;
        }

        // Set the DSRL state to UPDATING since we are going to apply our lists
        // to the DSRL
        bDSRLChangeState(hDSRL, DSRL_STATE_UPDATING);

        // adding entries from generated list
        bSuccess = bDSRLAddEntries(psAppObj, hDSRL);
        if (bSuccess == FALSE)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                WEATHER_MGR_OBJECT_NAME": cannot add entries to DSRL from"
                " list %p",
                psAppObj->hEntriesToAddList
                    );
            break;
        }

        //removing entries from generated list
        bSuccess = bDSRLRemoveEntries(psAppObj, hDSRL);
        if (bSuccess == FALSE)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                WEATHER_MGR_OBJECT_NAME": cannot remove entries "
                "from DSRL from list %p",
                psAppObj->hEntriesToRemoveList
                    );
            break;
        }

        bResult = TRUE;
    } while (FALSE);

    return bResult;
}

/*******************************************************************************
*
*        bModifyList
*
*******************************************************************************/
static BOOLEAN bModifyList(
    WEATHER_APP_OBJECT_STRUCT *psAppObj,
    DSRL_ARG_STRUCT *psDSRLArg
        )
{
    BOOLEAN bResult = FALSE;

    do
    {
        BOOLEAN bSuccess = FALSE;

        // This operation tells us explicitly if
        // we should do this or not
        if (psDSRLArg->uAction.sModify.bForceDSRLStateChange == TRUE)
        {
            bSuccess =
                bDSRLChangeState(psDSRLArg->hDSRL, DSRL_STATE_UPDATING);
            if (bSuccess == FALSE)
            {
                break;
            }
        }

        printf(WEATHER_MGR_OBJECT_NAME" :DSRL %p modify: modify type is %d\n",
            psDSRLArg->hDSRL, psDSRLArg->uAction.sModify.eModifyType
                );

        // Modify the DSRL as the user specified
        switch (psDSRLArg->uAction.sModify.eModifyType)
        {
            case DSRL_MODIFY_OPERATION_ADD:
            {
                bSuccess = bDSRLAddTargets(psDSRLArg);
            }
            break;

            case DSRL_MODIFY_OPERATION_REPLACE:
            {
                bSuccess = bDSRLReplaceTargets(psDSRLArg);
            }
            break;

            case DSRL_MODIFY_OPERATION_REMOVE:
            {
                bSuccess = bDSRLRemoveTargets(psDSRLArg);
            }
            break;

            case DSRL_MODIFY_OPERATION_REMOVEALL:
            {
                bSuccess = bDSRLRemoveAllTargets(psDSRLArg);
            }
            break;

            case DSRL_MODIFY_OPERATION_INVALID:
            default:
                break;
        }

        if (bSuccess == FALSE)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                WEATHER_MGR_OBJECT_NAME": DSRL modify targets operation failed"
                    );
            break;
        }

        bSuccess = bDSRLUpdateContent(psDSRLArg->hDSRL, psAppObj);
        if (bSuccess == FALSE)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                WEATHER_MGR_OBJECT_NAME": DSRL modify entries operation failed"
                    );
            break;
        }

        bResult = TRUE;
    } while (FALSE);

    if (bResult == FALSE)
    {
        bDSRLChangeState(psDSRLArg->hDSRL, DSRL_STATE_ERROR);
    }

    return bResult;
}


/*******************************************************************************
*
*   bRefreshList
*
*******************************************************************************/
static BOOLEAN bRefreshList(
    WEATHER_APP_OBJECT_STRUCT *psAppObj,
    DSRL_ARG_STRUCT *psDSRLArg
        )
{
    BOOLEAN bResult = FALSE;

    bResult = bDSRLUpdateContent(psDSRLArg->hDSRL, psAppObj);
    if (bResult == FALSE)
    {
        SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
            WEATHER_MGR_OBJECT_NAME": failed to update DSRL content"
                );

        bDSRLChangeState(psDSRLArg->hDSRL, DSRL_STATE_ERROR);
    }

    return bResult;
}

/*****************************************************************************
*
*   bGetPredefinedLocFromLoc
*
*****************************************************************************/
static BOOLEAN bGetPredefinedLocFromLoc(
    WEATHER_APP_OBJECT_STRUCT *psObj,
    LOCATION_OBJECT hTargetLocation,
    BOOLEAN bSkiFlag,
    WEATHER_LOCATION_STRUCT *psLoc
        )
{
    LOCID_OBJECT hLocId;
    BOOLEAN bResult = FALSE;

    printf(WEATHER_MGR_OBJECT_NAME" [%s] start\n", __FUNCTION__);

    psLoc->hLocation = LOCATION_INVALID_OBJECT;
    psLoc->hICAO = STRING_INVALID_OBJECT;

    hLocId = LOCATION.hLocID(hTargetLocation);

    if (hLocId == LOCID_INVALID_OBJECT)
    {
        bResult = bGetPredefinedLocFromCoord(
                psObj,
                hTargetLocation,
                bSkiFlag,
                psLoc);
    }
    else
    {
        bResult = bGetPredefinedLocFromLocId(
                psObj,
                hTargetLocation,
                bSkiFlag,
                psLoc);
    }

    printf(WEATHER_MGR_OBJECT_NAME" [%s] end with result %u\n", __FUNCTION__, bResult);

    return bResult;
}

/*****************************************************************************
*
*   bGetPredefinedLocFromCoord
*
*****************************************************************************/
static BOOLEAN bGetPredefinedLocFromCoord(
    WEATHER_APP_OBJECT_STRUCT *psObj,
    LOCATION_OBJECT hTargetLocation,
    BOOLEAN bSkiFlag,
    WEATHER_LOCATION_STRUCT *psLoc
        )
{
    WEATHER_SQL_QUERY_LOCATION_STRUCT sLocationSQLArg;
    int iReturn;
    BOOLEAN bSuccess, bResult = FALSE;
    OSAL_FIXED_OBJECT hTopRightLat, hTopRightLon;
    OSAL_FIXED_OBJECT hBottomLeftLat, hBottomLeftLon;
    OSAL_FIXED_OBJECT_DATA atFixedData[OSAL_FIXED_OBJECT_SIZE * 4];
    UN8 un8NumFixed = 0;    // Keep track of the fixed objects
    N32 n32FixedTopRightLat, n32FixedTopRightLon;
    N32 n32FixedBottomLeftLat, n32FixedhBottomLeftLon;

    printf(WEATHER_MGR_OBJECT_NAME" [%s] start\n", __FUNCTION__);

    OSAL.bMemSet(&sLocationSQLArg, 0, sizeof(sLocationSQLArg));

    do
    {
        if ((LOCATION.hLat(hTargetLocation) == OSAL_FIXED_INVALID_OBJECT) ||
            (LOCATION.hLon(hTargetLocation) == OSAL_FIXED_INVALID_OBJECT))
        {
            break;
        }

        // Create the objects.  If this fails, LOCATION_bTopRight
        // will result in an error
        hTopRightLat = OSAL_FIXED.hCreateInMemory(0, 0,
            &atFixedData[OSAL_FIXED_OBJECT_SIZE*(un8NumFixed++)]);
        hTopRightLon = OSAL_FIXED.hCreateInMemory(0, 0,
            &atFixedData[OSAL_FIXED_OBJECT_SIZE*(un8NumFixed++)]);

        // Get the top right lat, lon
        bSuccess = LOCATION_bTopRight(hTargetLocation,
            hTopRightLat, hTopRightLon);
        if (bSuccess == FALSE)
        {
            break;
        }

        // Create the objects.  If this fails, LOCATION_bBottomLeft
        // will result in an error
        hBottomLeftLat = OSAL_FIXED.hCreateInMemory(0, 0,
            &atFixedData[OSAL_FIXED_OBJECT_SIZE*(un8NumFixed++)]);
        hBottomLeftLon = OSAL_FIXED.hCreateInMemory(0, 0,
            &atFixedData[OSAL_FIXED_OBJECT_SIZE*(un8NumFixed++)]);

        // Get the bottom left lat, lon
        bSuccess = LOCATION_bBottomLeft(hTargetLocation,
            hBottomLeftLat, hBottomLeftLon);

        if (bSuccess == FALSE)
        {
            break;
        }

        n32FixedTopRightLat =
            OSAL_FIXED.n32ScaledValue(hTopRightLat, LOCATION_BINPOINT);
        n32FixedTopRightLon =
            OSAL_FIXED.n32ScaledValue(hTopRightLon, LOCATION_BINPOINT);
        n32FixedBottomLeftLat =
            OSAL_FIXED.n32ScaledValue(hBottomLeftLat, LOCATION_BINPOINT);
        n32FixedhBottomLeftLon =
            OSAL_FIXED.n32ScaledValue(hBottomLeftLon, LOCATION_BINPOINT);

        // Specify a search box within the target area
        iReturn = snprintf( &psObj->acBuffer[0], sizeof(psObj->acBuffer),
            WEATHER_SELECT_LOCATIONS_BY_LOCATION_AND_TYPE,
            n32FixedBottomLeftLat, n32FixedTopRightLat, // lat min / max
            n32FixedhBottomLeftLon, n32FixedTopRightLon, // lon min /max
            bSkiFlag);

        printf(WEATHER_MGR_OBJECT_NAME" SQL request is %s, size is %d\n",
                psObj->acBuffer, iReturn);

        if (iReturn <= 0)
        {
            break;
        }

        // Prepare query argument
        sLocationSQLArg.hTargetLocation = hTargetLocation;
        sLocationSQLArg.hMinimumDistance = DISTANCE_INVALID_OBJECT;
        sLocationSQLArg.hMatchedLocation = LOCATION_INVALID_OBJECT;
        sLocationSQLArg.hLocationUnderCheck = LOCATION_INVALID_OBJECT;
        sLocationSQLArg.sLocationRow.hLat = OSAL_FIXED_INVALID_OBJECT;
        sLocationSQLArg.sLocationRow.hLon = OSAL_FIXED_INVALID_OBJECT;

        // look in database
        bSuccess = SQL_INTERFACE.bQuery(psObj->hSQLRefConnection,
                &psObj->acBuffer[0],
                (SQL_QUERY_RESULT_HANDLER)bGetNearestLoc,
                &sLocationSQLArg);
        if (bSuccess == FALSE)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                WEATHER_MGR_OBJECT_NAME": Failed to get location"
                    );
            break;
        }

        // if there is not nearest location - break
        if (sLocationSQLArg.hMatchedLocation == LOCATION_INVALID_OBJECT)
        {
            break;
        }

        // Update found nearest location
        bResult =
            bUpdateLocationFromRow(sLocationSQLArg.hMatchedLocation,
                &sLocationSQLArg.sLocationRow);
        if (bResult == FALSE)
        {
            break;
        }

        // We're ok for now
        psLoc->hLocation = sLocationSQLArg.hMatchedLocation;
        psLoc->hICAO = sLocationSQLArg.sLocationRow.hICAOName;
        sLocationSQLArg.hMatchedLocation = LOCATION_INVALID_OBJECT;

        printf(WEATHER_MGR_OBJECT_NAME": location is %x, ICAO is %x\n",
                    psLoc->hLocation, psLoc->hICAO);
    } while (FALSE);

    if (sLocationSQLArg.hMinimumDistance != DISTANCE_INVALID_OBJECT)
    {
        DISTANCE.vDestroy(sLocationSQLArg.hMinimumDistance);
    }

    if (sLocationSQLArg.hMatchedLocation != LOCATION_INVALID_OBJECT)
    {
        LOCATION.vDestroy(sLocationSQLArg.hMatchedLocation);
    }

    if (sLocationSQLArg.hLocationUnderCheck != LOCATION_INVALID_OBJECT)
    {
        LOCATION.vDestroy(sLocationSQLArg.hLocationUnderCheck);
    }

    if (psLoc->hLocation == LOCATION_INVALID_OBJECT)
    {
        if (sLocationSQLArg.sLocationRow.hAreaName != STRING_INVALID_OBJECT)
        {
            STRING_vDestroy(sLocationSQLArg.sLocationRow.hAreaName);
        }

        if (sLocationSQLArg.sLocationRow.hICAOName != STRING_INVALID_OBJECT)
        {
            STRING_vDestroy(sLocationSQLArg.sLocationRow.hICAOName);
        }

        psLoc->hICAO = STRING_INVALID_OBJECT;
    }

    printf(WEATHER_MGR_OBJECT_NAME" [%s] end, result is %u\n", __FUNCTION__, bResult);

    return bResult;
}

/*****************************************************************************
*
*   bGetPredefinedLocFromLocId
*
*****************************************************************************/
static BOOLEAN bGetPredefinedLocFromLocId(
    WEATHER_APP_OBJECT_STRUCT *psObj,
    LOCATION_OBJECT hTargetLocation,
    BOOLEAN bSkiFlag,
    WEATHER_LOCATION_STRUCT *psLoc
        )
{
    int iReturn = 0;
    LOCID_OBJECT hLocId = LOCID_INVALID_OBJECT;
    LOC_ID tID = LOC_INVALID_ID;
    BOOLEAN bResult = FALSE;

    printf(WEATHER_MGR_OBJECT_NAME" %s start\n", __FUNCTION__);

    psLoc->hLocation = LOCATION_INVALID_OBJECT;
    psLoc->hICAO = STRING_INVALID_OBJECT;

    do
    {
        BOOLEAN bSuccess = FALSE;

        hLocId = LOCATION.hLocID(hTargetLocation);
        tID = LOCID.tID(hLocId);

        // Build our update string for this new version using our
        // DB's schema version to find the version row
        iReturn = snprintf(
                &psObj->acBuffer[0],
                sizeof(psObj->acBuffer),
                WEATHER_SELECT_LOCATION_WITH_FLAG,
                tID,
                bSkiFlag);

        printf(WEATHER_MGR_OBJECT_NAME" SQL request is %s, size is %d\n", psObj->acBuffer, iReturn);

        if (iReturn > 0)
        {
            // look in database
            bSuccess = SQL_INTERFACE.bQuery(psObj->hSQLRefConnection,
                    &psObj->acBuffer[0],
                    (SQL_QUERY_RESULT_HANDLER)bGetLocationFromDB,
                    psLoc);
            if (bSuccess == FALSE)
            {
                SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    WEATHER_MGR_OBJECT_NAME": Failed to get location"
                        );
                break;
            }

            if (psLoc->hLocation != LOCATION_INVALID_OBJECT)
            {
                printf(WEATHER_MGR_OBJECT_NAME": location is %x\n", psLoc->hLocation);
                printf(WEATHER_MGR_OBJECT_NAME": ICAO is %x\n", psLoc->hICAO);
                bResult = TRUE;
            }
        }
    } while (FALSE);

    printf(WEATHER_MGR_OBJECT_NAME" %s end, result is %u\n", __FUNCTION__, bResult);

    return bResult;
}

/*****************************************************************************
*
*   hCreateLocationFromRow
*
*****************************************************************************/
static LOCATION_OBJECT hCreateLocationFromRow(
    WEATHER_LOCATION_ROW_STRUCT *psRow
        )
{
    LOCATION_OBJECT hResult = LOCATION_INVALID_OBJECT;
    LOCID_OBJECT hLocId = LOCID_INVALID_OBJECT;
    LOCATION_ATTRIBUTE_STRUCT sDescriptiveAttrs;

    // Initializing structure here to correctly handle cleanup after do cycle
    OSAL.bMemSet(&sDescriptiveAttrs, 0, sizeof(sDescriptiveAttrs));

    do {

        hLocId = LOCID_hCreate(
                psRow->tID,
                LOCID_TYPE_WEATHER,
                NULL);

        if (hLocId == LOCID_INVALID_OBJECT)
        {
            break;
        }

        sDescriptiveAttrs.hDescription = psRow->hAreaName;
        sDescriptiveAttrs.hState = LOCATION_hStateAbbrvForID(GET_STATE_ID_FROM_LOC_ID(psRow->tID));

        hResult = LOCATION_hCreate(
                SMS_INVALID_OBJECT,
                hLocId,
                OSAL_FIXED.n32Value(psRow->hLat),
                OSAL_FIXED.un8NumFractionalBits(psRow->hLat),
                OSAL_FIXED.n32Value(psRow->hLon),
                OSAL_FIXED.un8NumFractionalBits(psRow->hLon),
                DISTANCE_INVALID_OBJECT,
                &sDescriptiveAttrs,
                FALSE);

        if (hResult == LOCATION_INVALID_OBJECT)
        {
            break;
        }

    } while (FALSE);

    if (hResult == LOCATION_INVALID_OBJECT)
    {
        if (hLocId != LOCID_INVALID_OBJECT)
        {
            LOCID.vDestroy(hLocId);
        }
        if(sDescriptiveAttrs.hDescription != STRING_INVALID_OBJECT )
        {
            STRING_vDestroy(sDescriptiveAttrs.hDescription);
        }
        if(sDescriptiveAttrs.hState != STRING_INVALID_OBJECT )
        {
            STRING_vDestroy(sDescriptiveAttrs.hState);
        }
    }

    return hResult;
}


/*****************************************************************************
*
*   bUpdateLocationFromRow
*
*****************************************************************************/
static BOOLEAN bUpdateLocationFromRow(
    LOCATION_OBJECT hLocation,
    WEATHER_LOCATION_ROW_STRUCT *psRow
        )
{
    BOOLEAN bResult = FALSE;
    LOCID_OBJECT hLocId = LOCID_INVALID_OBJECT;
    LOCATION_ATTRIBUTE_STRUCT sDescriptiveAttrs;

    // Initializing structure here to correctly handle cleanup after do cycle
    OSAL.bMemSet(&sDescriptiveAttrs, 0, sizeof(sDescriptiveAttrs));

    do
    {
        // Create LOCID object for weather
        hLocId = LOCID_hCreate(
                psRow->tID,
                LOCID_TYPE_WEATHER,
                NULL);
        if (hLocId == LOCID_INVALID_OBJECT)
        {
            break;
        }

        // Update LOCID
        bResult = LOCATION_bUpdateLocID(hLocation, hLocId, FALSE);
        if (bResult == FALSE)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                WEATHER_MGR_OBJECT_NAME": failed to update LOCID");
            break;
        }
        hLocId = LOCID_INVALID_OBJECT;

        // Fill in the descriptive attributes
        sDescriptiveAttrs.hDescription = psRow->hAreaName;
        sDescriptiveAttrs.hState =
            LOCATION_hStateAbbrvForID(GET_STATE_ID_FROM_LOC_ID(psRow->tID));

        // Update attributes
        bResult =
            LOCATION_bUpdateDescriptiveAttributes(
                hLocation, &sDescriptiveAttrs, TRUE, FALSE);
        if (bResult == FALSE)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                WEATHER_MGR_OBJECT_NAME": failed to update descriptive attributes");
            break;
        }
    } while (FALSE);

    if (bResult == FALSE)
    {
        if (hLocId != LOCID_INVALID_OBJECT)
        {
            LOCID.vDestroy(hLocId);
        }
        if (sDescriptiveAttrs.hDescription != STRING_INVALID_OBJECT )
        {
            STRING_vDestroy(sDescriptiveAttrs.hDescription);
        }
        if (sDescriptiveAttrs.hState != STRING_INVALID_OBJECT )
        {
            STRING_vDestroy(sDescriptiveAttrs.hState);
        }
    }

    return bResult;
}


/*****************************************************************************
*
*   hGetNearestLoc
*
*****************************************************************************/
static BOOLEAN bGetNearestLoc(
    SQL_QUERY_COLUMN_STRUCT *psColumn,
    N32 n32NumberOfColumns,
    WEATHER_SQL_QUERY_LOCATION_STRUCT *psLocationSQLArg
        )
{
    BOOLEAN bResult = FALSE, bTemp;
    WEATHER_LOCATION_ROW_STRUCT sLocationRow;
    DISTANCE_OBJECT hCalculatedDistance = DISTANCE_INVALID_OBJECT;
    N16 n16CompareResult;

    do
    {
        // Clear the location structure
        OSAL.bMemSet(&sLocationRow, 0, sizeof(WEATHER_LOCATION_ROW_STRUCT));

        // Gather data from the column
        bTemp = bFillLocationRow(psColumn, n32NumberOfColumns, &sLocationRow);
        if (bTemp != TRUE)
        {
            break;
        }

        // Create location object based on the
        if (psLocationSQLArg->hLocationUnderCheck == LOCATION_INVALID_OBJECT)
        {
            // Create new object
            psLocationSQLArg->hLocationUnderCheck =
                LOCATION.hCreateForRadius(sLocationRow.hLat, sLocationRow.hLon,
                                          DISTANCE_INVALID_OBJECT);
            if (psLocationSQLArg->hLocationUnderCheck ==
                                                LOCATION_INVALID_OBJECT)
            {
                SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    WEATHER_MGR_OBJECT_NAME": failed to create LOCATION");
                break;
            }
        }
        else
        {
            // Update existing object
            bTemp = LOCATION_bUpdateCoordinatesByFixed(
                                    psLocationSQLArg->hLocationUnderCheck,
                                    sLocationRow.hLat,
                                    sLocationRow.hLon);
            if (bTemp == FALSE)
            {
                break;
            }
        }

        // Calculate the distance between two locations
        hCalculatedDistance =
            LOCATION.hDistance(psLocationSQLArg->hLocationUnderCheck,
                               psLocationSQLArg->hTargetLocation);
        if (hCalculatedDistance == DISTANCE_INVALID_OBJECT)
        {
            break;
        }

        if (psLocationSQLArg->hMinimumDistance == DISTANCE_INVALID_OBJECT)
        {
            // This is a first found location
            n16CompareResult = 1;
        }
        else
        {
            n16CompareResult =
                DISTANCE.n16Compare(psLocationSQLArg->hMinimumDistance,
                                    hCalculatedDistance);
        }

        if (n16CompareResult > 0)
        {
            DISTANCE_OBJECT hTmpDist;
            LOCATION_OBJECT hTmpLoc;

            // Swap DISTANCE objects
            hTmpDist = psLocationSQLArg->hMinimumDistance;
            psLocationSQLArg->hMinimumDistance = hCalculatedDistance;
            hCalculatedDistance = hTmpDist;

            // Swap LOCATION objects
            hTmpLoc = psLocationSQLArg->hMatchedLocation;
            psLocationSQLArg->hMatchedLocation =
                psLocationSQLArg->hLocationUnderCheck;
            psLocationSQLArg->hLocationUnderCheck = hTmpLoc;

            if (psLocationSQLArg->sLocationRow.hAreaName != STRING_INVALID_OBJECT)
            {
                STRING_vDestroy(psLocationSQLArg->sLocationRow.hAreaName);
            }

            if (psLocationSQLArg->sLocationRow.hICAOName != STRING_INVALID_OBJECT)
            {
                STRING_vDestroy(psLocationSQLArg->sLocationRow.hICAOName);
            }

            psLocationSQLArg->sLocationRow.bSkiFlag = sLocationRow.bSkiFlag;
            psLocationSQLArg->sLocationRow.tID = sLocationRow.tID;
            psLocationSQLArg->sLocationRow.un8Version = sLocationRow.un8Version;
            psLocationSQLArg->sLocationRow.hAreaName = sLocationRow.hAreaName;
            sLocationRow.hAreaName = STRING_INVALID_OBJECT;
            psLocationSQLArg->sLocationRow.hICAOName = sLocationRow.hICAOName;
            sLocationRow.hICAOName = STRING_INVALID_OBJECT;
        }

        // We're ok
        bResult = TRUE;
    } while (FALSE);

    // Cleanup all
    if (hCalculatedDistance != DISTANCE_INVALID_OBJECT)
    {
        DISTANCE.vDestroy(hCalculatedDistance);
    }

    if (sLocationRow.hAreaName != STRING_INVALID_OBJECT)
    {
        STRING_vDestroy(sLocationRow.hAreaName);
    }

    if (sLocationRow.hICAOName != STRING_INVALID_OBJECT)
    {
        STRING_vDestroy(sLocationRow.hICAOName);
    }

    return bResult;
}

/*****************************************************************************
*
*   bDSRLAddTargets
*
*****************************************************************************/
static BOOLEAN bDSRLAddTargets(
    DSRL_ARG_STRUCT *psDSRLArg
        )
{
    size_t tDSRLTargetIndex;
    BOOLEAN bResult = TRUE;

    for (tDSRLTargetIndex = 0;
         tDSRLTargetIndex < psDSRLArg->tNumTargets;
         tDSRLTargetIndex++)
    {
        bResult = bDSRLAddTarget(psDSRLArg->hDSRL,
            psDSRLArg->ahTargetList[tDSRLTargetIndex]
                );
        if (bResult == FALSE)
        {
            break;
        }
        else
        {
            // if target has been added successfully
            // need to clean it in array to avoid target destroy
            // in general DATASERVICE event processor
            psDSRLArg->ahTargetList[tDSRLTargetIndex] =
                DSRL_TARGET_INVALID_OBJECT;
        }
    }

    return bResult;
}

/*****************************************************************************
*
*   bDSRLAddTarget
*
*****************************************************************************/
static BOOLEAN bDSRLAddTarget(
    DSRL_OBJECT hDSRL,
    DSRL_TARGET_OBJECT hTargetObj
        )
{
    BOOLEAN bResult = FALSE;

    do
    {
        OSAL_RETURN_CODE_ENUM eReturnCode = OSAL_ERROR;
        WEATHER_DSRL_DESC_STRUCT *psDSRLDesc = NULL;

        if (hTargetObj == DSRL_TARGET_INVALID_OBJECT)
        {
            break;
        }

        psDSRLDesc = (WEATHER_DSRL_DESC_STRUCT *)DSRL_pvServiceData(hDSRL);
        if (NULL == psDSRLDesc)
        {
            // Error!
            break;
        }

        // Add this target to our DSRL description
        eReturnCode = OSAL.eLinkedListAdd(
            psDSRLDesc->hTargetList,
            OSAL_INVALID_LINKED_LIST_ENTRY_PTR,
            hTargetObj
                );
        if (eReturnCode != OSAL_SUCCESS)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                WEATHER_MGR_OBJECT_NAME": failed to add target to list (%s)",
                OSAL.pacGetReturnCodeName(eReturnCode)
                    );
            break;
        }

        bResult = TRUE;
    } while (FALSE);

    return bResult;
}

/*****************************************************************************
*
*   bDSRLRemoveAllTargets
*
*****************************************************************************/
static BOOLEAN bDSRLRemoveAllTargets(
    DSRL_ARG_STRUCT *psDSRLArg
        )
{
    BOOLEAN bResult = TRUE;
    OSAL_RETURN_CODE_ENUM eReturnCode = OSAL_ERROR;
    WEATHER_DSRL_DESC_STRUCT *psDSRLDesc = NULL;

    psDSRLDesc =
        (WEATHER_DSRL_DESC_STRUCT *)DSRL_pvServiceData(psDSRLArg->hDSRL);
    if (psDSRLDesc == NULL)
    {
        bResult = FALSE;
    }
    else
    {
        eReturnCode = OSAL.eLinkedListRemoveAll(psDSRLDesc->hTargetList,
            (OSAL_LL_RELEASE_HANDLER)DSRL_TARGET_vDestroyByType
                );
        if (eReturnCode != OSAL_SUCCESS)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                WEATHER_MGR_OBJECT_NAME": failed to remove DSRL targets"
                " (%s)", OSAL.pacGetReturnCodeName(eReturnCode)
                    );
            bResult = FALSE;
        }
    }

    return bResult;
}

/*****************************************************************************
*
*   vDSRLRemoveTarget
*
*****************************************************************************/
static void vDSRLRemoveTarget(
    DSRL_OBJECT hDSRL,
    DSRL_TARGET_OBJECT hTargetObj
        )
{
    do
    {
        OSAL_LINKED_LIST_ENTRY hTargetEntry = OSAL_INVALID_LINKED_LIST_ENTRY;
        OSAL_RETURN_CODE_ENUM eReturnCode = OSAL_ERROR;
        WEATHER_DSRL_DESC_STRUCT *psDSRLDesc = NULL;
        DSRL_TARGET_OBJECT hTargetToRemove = DSRL_TARGET_INVALID_OBJECT;

        if (hTargetObj == DSRL_TARGET_INVALID_OBJECT)
        {
            break;
        }

        psDSRLDesc = (WEATHER_DSRL_DESC_STRUCT *)DSRL_pvServiceData(hDSRL);
        if (psDSRLDesc == NULL)
        {
            break;
        }

        // looking for the target in the DSRL targets list
        eReturnCode = OSAL.eLinkedListLinearSearch(
            psDSRLDesc->hTargetList,
            &hTargetEntry,
            (OSAL_LL_COMPARE_HANDLER)n16FindTargetEntry,
            hTargetObj
                );

        if (eReturnCode != OSAL_SUCCESS)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                WEATHER_MGR_OBJECT_NAME": failed to find target entry (%s)",
                OSAL.pacGetReturnCodeName(eReturnCode)
                    );
            break;
        }

        hTargetToRemove =
            (DSRL_TARGET_OBJECT)OSAL.pvLinkedListThis(hTargetEntry);
        if (hTargetToRemove == DSRL_TARGET_INVALID_OBJECT)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                WEATHER_MGR_OBJECT_NAME": failed to get target handle"
                    );
            break;
        }

        // removing the target from the list
        eReturnCode = OSAL.eLinkedListRemove(hTargetEntry);
        if (eReturnCode != OSAL_SUCCESS)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                WEATHER_MGR_OBJECT_NAME": failed to remove target entry"
                " from list of DSRL targets (%s)",
                OSAL.pacGetReturnCodeName(eReturnCode)
                    );
        }

        // destroying the target
        DSRL_TARGET_vDestroyByType(hTargetToRemove);
    } while (FALSE);

    return;
}

/*****************************************************************************
*
*   bDSRLRemoveTargets
*
*****************************************************************************/
static BOOLEAN bDSRLRemoveTargets(
    DSRL_ARG_STRUCT *psDSRLArg
        )
{
    size_t tDSRLTargetIndex;

    for (tDSRLTargetIndex = 0;
         tDSRLTargetIndex < psDSRLArg->tNumTargets;
         ++tDSRLTargetIndex)
    {
        vDSRLRemoveTarget(
            psDSRLArg->hDSRL,
            psDSRLArg->ahTargetList[tDSRLTargetIndex]
                );
    }

    return TRUE;
}

/*****************************************************************************
*
*   n16FindTargetEntry
*
*****************************************************************************/
static N16 n16FindTargetEntry(
    DSRL_TARGET_OBJECT hTarget1,
    DSRL_TARGET_OBJECT hTarget2
        )
{
    N16 n16Result = N16_MIN;

    if ((hTarget1 != DSRL_TARGET_INVALID_OBJECT) &&
        (hTarget2 != DSRL_TARGET_INVALID_OBJECT))
    {
        BOOLEAN bResult;

        bResult = DSRL_TARGET_bCompare(hTarget1, hTarget2);

        n16Result = (bResult == TRUE) ? 0 : 1;
    }

    return n16Result;
}

/*****************************************************************************
*
*   bEntrySkipForecastHash
*
******************************************************************************/
static BOOLEAN bEntrySkipForecastHash(
    DSRL_ENTRY_OBJECT hDSRLEntry,
    WEATHER_HASH_OBJECT hHashEntry
        )
{
    BOOLEAN bContinue = TRUE;

    do
    {
        WEATHER_MSG_OBJECT hMsg = WEATHER_MSG_INVALID_OBJECT;
        WEATHER_MSG_OBJECT_TYPE_ENUM eType = WEATHER_MSG_TYPE_FORECAST;
        UN8 un8Iterator = 0;
        FORECAST_OBJECT hForecast = FORECAST_INVALID_OBJECT;
        SMSAPI_RETURN_CODE_ENUM eResult = SMSAPI_RETURN_CODE_ERROR;
        WEATHER_HASH_OBJECT hCurHashEntry = WEATHER_HASH_INVALID_OBJECT;

        if ((hDSRLEntry == DSRL_ENTRY_INVALID_OBJECT) ||
            (hHashEntry == WEATHER_HASH_INVALID_OBJECT))
        {
            // skip processing if something wrong with parameters
            bContinue = FALSE;
            break;
        }

        hMsg = DSRL_ENTRY.hWeatherMsg(hDSRLEntry);

        if (hMsg == WEATHER_MSG_INVALID_OBJECT)
        {
            break;
        }

        eType = WEATHER_MSG.eType(hMsg);
        if (eType != WEATHER_MSG_TYPE_FORECAST)
        {
            break;
        }

        for (un8Iterator = 0; un8Iterator < WEATHER_FORECAST_TYPE_MAX;
            un8Iterator++)
        {
            eResult = WEATHER_MSG.eGetForecast(hMsg,
                (WEATHER_FORECAST_TYPE_ENUM)un8Iterator, &hForecast);
            if (eResult != SMSAPI_RETURN_CODE_SUCCESS)
            {
                //Continue if forecast object for this forecast
                //type is not assigned to weather message
                continue;
            }

            hCurHashEntry = FORECAST_hHashEntry(hForecast);
            if (hCurHashEntry == hHashEntry)
            {
                FORECAST_bUpdateHashEntry(hForecast,
                    OSAL_INVALID_LINKED_LIST_ENTRY
                        );
                break;
            }
        }

    } while (FALSE);

    return bContinue;
}

/*****************************************************************************
*
*   vSkipForecastHashEntry
*
******************************************************************************/
static void vSkipForecastHashEntry(
    WEATHER_APP_OBJECT_STRUCT *psObj,
    WEATHER_HASH_OBJECT hHashEntry
        )
{
    if (psObj != NULL)
    {
        OSAL_RETURN_CODE_ENUM eReturnCode;

        eReturnCode = OSAL.eLinkedListIterate(psObj->hEntriesList,
            (OSAL_LL_ITERATOR_HANDLER)bEntrySkipForecastHash, hHashEntry
                );
        if ((eReturnCode != OSAL_SUCCESS) && (eReturnCode != OSAL_NO_OBJECTS))
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                WEATHER_MGR_OBJECT_NAME": failed to iterate entry list"
                " (%s)", OSAL.pacGetReturnCodeName(eReturnCode)
                    );
        }
    }

    return;
}

/*****************************************************************************
*
*   bEntryChangeForecastTimestamp
*
******************************************************************************/
static BOOLEAN bEntryChangeForecastTimestamp(
    DSRL_ENTRY_OBJECT hEntry,
    WEATHER_HASH_OBJECT hHashEntry
        )
{
    BOOLEAN bContinue = TRUE;

    do
    {
        WEATHER_MSG_OBJECT hMsg = WEATHER_MSG_INVALID_OBJECT;
        WEATHER_MSG_OBJECT_TYPE_ENUM eType = WEATHER_MSG_TYPE_FORECAST;
        UN8 un8Iterator = 0;
        FORECAST_OBJECT hForecast = FORECAST_INVALID_OBJECT;
        SMSAPI_RETURN_CODE_ENUM eResult = SMSAPI_RETURN_CODE_ERROR;
        WEATHER_HASH_OBJECT hCurHashEntry =
            WEATHER_HASH_INVALID_OBJECT;

        if ((hEntry == DSRL_ENTRY_INVALID_OBJECT) ||
            (hHashEntry == WEATHER_HASH_INVALID_OBJECT))
        {
            // skip processing if something wrong with parameters
            bContinue = FALSE;
            break;
        }

        // get message object from entry
        hMsg = DSRL_ENTRY.hWeatherMsg(hEntry);

        if (hMsg == WEATHER_MSG_INVALID_OBJECT)
        {
            break;
        }

        eType = WEATHER_MSG.eType(hMsg);
        if (eType != WEATHER_MSG_TYPE_FORECAST)
        {
            break;
        }

        for (un8Iterator = 0; un8Iterator < WEATHER_FORECAST_TYPE_MAX;
            un8Iterator++)
        {
            eResult = WEATHER_MSG.eGetForecast(hMsg,
                (WEATHER_FORECAST_TYPE_ENUM)un8Iterator, &hForecast);
            if (eResult != SMSAPI_RETURN_CODE_SUCCESS)
            {
                //Continue if forecast object for this forecast
                //type is not assigned to weather message
                continue;
            }

            hCurHashEntry = FORECAST_hHashEntry(hForecast);
            if (hCurHashEntry == hHashEntry)
            {
                FORECAST_vUpdateTimeStamp(hForecast);
                break;
            }
        }
    } while (FALSE);

    return bContinue;
}

/*****************************************************************************
*
*   vChangeForecastTimestamp
*
******************************************************************************/
static void vChangeForecastTimestamp(
    WEATHER_APP_OBJECT_STRUCT *psObj,
    WEATHER_HASH_OBJECT hHashEntry
        )
{
    if (psObj != NULL)
    {
        OSAL_RETURN_CODE_ENUM eReturnCode = OSAL_ERROR;

        eReturnCode = OSAL.eLinkedListIterate(psObj->hEntriesList,
            (OSAL_LL_ITERATOR_HANDLER)bEntryChangeForecastTimestamp, hHashEntry
                );
        if ((eReturnCode != OSAL_SUCCESS) && (eReturnCode != OSAL_NO_OBJECTS))
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                WEATHER_MGR_OBJECT_NAME": failed to iterate"
                " items in list (%s)", OSAL.pacGetReturnCodeName(eReturnCode)
                    );
        }
    }

    return;
}

/*****************************************************************************
*
*   bEntrySkipSkiHash
*
******************************************************************************/
static BOOLEAN bEntrySkipSkiHash(
    DSRL_ENTRY_OBJECT hEntry,
    WEATHER_HASH_OBJECT hHashEntry
        )
{
    BOOLEAN bContinue = TRUE;

    do
    {
        WEATHER_MSG_OBJECT_TYPE_ENUM eType = WEATHER_MSG_TYPE_FORECAST;
        SKI_CONDITIONS_OBJECT hSkiCond = SKI_CONDITIONS_INVALID_OBJECT;
        SMSAPI_RETURN_CODE_ENUM eResult = SMSAPI_RETURN_CODE_ERROR;
        WEATHER_HASH_OBJECT hCurHashEntry =
            WEATHER_HASH_INVALID_OBJECT;
        WEATHER_MSG_OBJECT hMsg = WEATHER_MSG_INVALID_OBJECT;

        if ((hEntry == DSRL_ENTRY_INVALID_OBJECT) ||
            (hHashEntry == WEATHER_HASH_INVALID_OBJECT))
        {
            // skip processing if something wrong with parameters
            bContinue = FALSE;
            break;
        }

        hMsg = DSRL_ENTRY.hWeatherMsg(hEntry);

        if (hMsg == WEATHER_MSG_INVALID_OBJECT)
        {
            break;
        }

        eType = WEATHER_MSG.eType(hMsg);
        if (eType != WEATHER_MSG_TYPE_SKI_REPORT)
        {
            break;
        }

        eResult = WEATHER_MSG.eGetSkiConditions(hMsg,
            &hSkiCond
                );
        if (eResult != SMSAPI_RETURN_CODE_SUCCESS)
        {
            //break if ski conditions object
            //is not assigned to weather message
            break;
        }

        hCurHashEntry = SKI_CONDITIONS_hHashEntry(hSkiCond);
        if (hCurHashEntry != hHashEntry)
        {
            break;
        }

        SKI_CONDITIONS_bUpdateHashEntry(hSkiCond,
            OSAL_INVALID_LINKED_LIST_ENTRY
                );
    } while (FALSE);

    return bContinue;
}

/*****************************************************************************
*
*   vSkipSkiCondHashEntry
*
******************************************************************************/
static void vSkipSkiCondHashEntry(
    WEATHER_APP_OBJECT_STRUCT *psObj,
    WEATHER_HASH_OBJECT hOldHashEntry
        )
{
    if (psObj != NULL)
    {
        OSAL_RETURN_CODE_ENUM eReturnCode;

        eReturnCode = OSAL.eLinkedListIterate(psObj->hEntriesList,
            (OSAL_LL_ITERATOR_HANDLER)bEntrySkipSkiHash, hOldHashEntry
                );
        if ((eReturnCode != OSAL_SUCCESS) && (eReturnCode != OSAL_NO_OBJECTS))
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                WEATHER_MGR_OBJECT_NAME": failed to iterate"
                " items in list (%s)", OSAL.pacGetReturnCodeName(eReturnCode)
                    );
        }
    }

    return;
}

/*****************************************************************************
*
*   bLoadDataToEntries
*
******************************************************************************/
static BOOLEAN bLoadDataToEntries(
    WEATHER_MGR_OBJECT_STRUCT *psObj
        )
{
    BOOLEAN bResult = FALSE;
    OSAL_RETURN_CODE_ENUM eReturnCode = OSAL_ERROR;

    do
    {
        BOOLEAN bSuccess = FALSE;

        bSuccess = GsWeatherIntf.bLoadForecastData(psObj->hInterface,
            (SMS_OBJECT)psObj->psAppObj,
            psObj->psAppObj->hNewForecastEntriesList
                );
        if (bSuccess == FALSE)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                WEATHER_MGR_OBJECT_NAME ": failed to load forecast data"
                    );
            break;
        }

        eReturnCode = OSAL.eLinkedListRemoveAll(
            psObj->psAppObj->hNewForecastEntriesList, NULL);
        if (eReturnCode != OSAL_SUCCESS)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                WEATHER_MGR_OBJECT_NAME
                ": failed to remove all items from new Forecast"
                " entries list (%s)", OSAL.pacGetReturnCodeName(eReturnCode));
            break;
        }

        bSuccess = GsWeatherIntf.bLoadSkiData(psObj->hInterface,
            (SMS_OBJECT)psObj->psAppObj, psObj->psAppObj->hNewSkiEntriesList
                );
        if (bSuccess == FALSE)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                WEATHER_MGR_OBJECT_NAME ": failed to load ski data"
                    );
            break;
        }

        eReturnCode = OSAL.eLinkedListRemoveAll(
            psObj->psAppObj->hNewSkiEntriesList, NULL);
        if (eReturnCode != OSAL_SUCCESS)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                WEATHER_MGR_OBJECT_NAME
                ": failed to remove all items from new Ski entries list (%s)",
                OSAL.pacGetReturnCodeName(eReturnCode)
                    );
            break;
        }

        bResult = TRUE;
    } while(FALSE);

    return bResult;
}

/*****************************************************************************
*
*   bEntryChangeSkiTimestamp
*
******************************************************************************/
static BOOLEAN bEntryChangeSkiTimestamp(
    DSRL_ENTRY_OBJECT hEntry,
    WEATHER_HASH_OBJECT hHashEntry
        )
{
    BOOLEAN bContinue = TRUE;

    do
    {
        WEATHER_MSG_OBJECT hMsg = WEATHER_MSG_INVALID_OBJECT;
        WEATHER_MSG_OBJECT_TYPE_ENUM eType = WEATHER_MSG_TYPE_FORECAST;
        SKI_CONDITIONS_OBJECT hSkiCond = SKI_CONDITIONS_INVALID_OBJECT;
        SMSAPI_RETURN_CODE_ENUM eResult = SMSAPI_RETURN_CODE_ERROR;
        WEATHER_HASH_OBJECT hCurHashEntry =
            WEATHER_MSG_INVALID_OBJECT;

        if ((hEntry == DSRL_ENTRY_INVALID_OBJECT ) ||
            (hHashEntry == WEATHER_HASH_INVALID_OBJECT))
        {
            // skip processing if something wrong with parameters
            bContinue = FALSE;
            break;
        }

        hMsg = DSRL_ENTRY.hWeatherMsg(hEntry);

        if (hMsg == WEATHER_MSG_INVALID_OBJECT)
        {
            break;
        }

        eType = WEATHER_MSG.eType(hMsg);
        if (eType != WEATHER_MSG_TYPE_SKI_REPORT)
        {
            break;
        }

        eResult = WEATHER_MSG.eGetSkiConditions(hMsg,
            &hSkiCond
                );
        if (eResult != SMSAPI_RETURN_CODE_SUCCESS)
        {
            //break if ski conditions object
            //is not assigned to weather message
            break;
        }

        hCurHashEntry = SKI_CONDITIONS_hHashEntry(hSkiCond);
        if (hCurHashEntry != hHashEntry)
        {
            break;
        }

        SKI_CONDITIONS_vUpdateTimeStamp(hSkiCond);
    } while (FALSE);

    return bContinue;
}

/*****************************************************************************
*
*   vChangeSkiCondTimestamp
*
******************************************************************************/
static void vChangeSkiCondTimestamp(
    WEATHER_APP_OBJECT_STRUCT *psObj,
    WEATHER_HASH_OBJECT hHashEntry
        )
{
    if (psObj != NULL)
    {
        OSAL_RETURN_CODE_ENUM eReturnCode;

        eReturnCode = OSAL.eLinkedListIterate(psObj->hEntriesList,
            (OSAL_LL_ITERATOR_HANDLER)bEntryChangeSkiTimestamp, hHashEntry
                );
        if ((eReturnCode != OSAL_SUCCESS) && (eReturnCode != OSAL_NO_OBJECTS))
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                WEATHER_MGR_OBJECT_NAME": failed to iterate"
                " items in list (%s)", OSAL.pacGetReturnCodeName(eReturnCode)
                    );
        }
    }

    return;
}

/*****************************************************************************
*
*   vCheckForItemsToCancel
*
*****************************************************************************/
static void vCheckForItemsToCancel (
    WEATHER_MGR_OBJECT_STRUCT *psObj
        )
{
    WEATHER_APP_OBJECT_STRUCT *psAppObj = NULL;

    GsWeatherIntf.bProcessTimerCall(psObj->hInterface);

    psAppObj = psGetAppFacingObject((WEATHER_SERVICE_OBJECT)psObj);

    if (psAppObj != NULL)
    {
        UN32 un32UTCsec = 0;
        OSAL_RETURN_CODE_ENUM eReturnCode;

        eReturnCode = OSAL.eTimeGet(&un32UTCsec);
        if (eReturnCode == OSAL_SUCCESS)
        {
            printf(WEATHER_MGR_OBJECT_NAME
                " Current time is %d seconds UTC.\n", un32UTCsec);

            // iterating list of all weather entries to remove expired items
            vRemoveOldItemsFromEntries(psAppObj, un32UTCsec);
        }
        else
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                WEATHER_MGR_OBJECT_NAME": failed to get OS time (%s)",
                OSAL.pacGetReturnCodeName(eReturnCode));
        }

        vUnlockAppFacingObject(psAppObj);
    }

    vStartCancelTimer(psObj);

    return;
}

/*****************************************************************************
*
*   bEntryRemoveOldForecastItems
*
*****************************************************************************/
static BOOLEAN bEntryRemoveOldForecastItems(
    DSRL_ENTRY_OBJECT hDSRLEntry,
    UN32 un32CurrentTime
        )
{
    UN8 un8Iterator = 0;
    BOOLEAN bUpdated = FALSE;
    FORECAST_OBJECT hForecast = FORECAST_INVALID_OBJECT;
    SMSAPI_RETURN_CODE_ENUM eResult = SMSAPI_RETURN_CODE_ERROR;
    UN32 un32OldTime = 0;

    for (un8Iterator = 0; un8Iterator < WEATHER_FORECAST_TYPE_MAX;
        un8Iterator++)
    {
        eResult = WEATHER_MSG.eGetForecast((WEATHER_MSG_OBJECT)hDSRLEntry,
            (WEATHER_FORECAST_TYPE_ENUM)un8Iterator, &hForecast);
        if ((eResult == SMSAPI_RETURN_CODE_SUCCESS) &&
            (hForecast != FORECAST_INVALID_OBJECT))
        {
            un32OldTime = FORECAST.un32TimeStamp(hForecast);
            if ((un32CurrentTime - un32OldTime) > WEATHER_MSG_TIMEOUT)
            {
                printf(WEATHER_MGR_OBJECT_NAME" Removing forecast"
                    " item %p\n", hForecast
                        );

                if (bUpdated == FALSE)
                {
                    bUpdated = TRUE;

                    // ok, now we are going to move all related DSLRs
                    // to UPDATING state
                    vChangeEntryDSRLsState(hDSRLEntry,
                        DSRL_STATE_UPDATING);
                }

                WEATHER_MSG_bSetForecast((WEATHER_MSG_OBJECT)hDSRLEntry,
                    (WEATHER_FORECAST_TYPE_ENUM)un8Iterator,
                    FORECAST_INVALID_OBJECT);
            }
        }
    }

    return bUpdated;
}

/*****************************************************************************
*
*   bEntryRemoveOldSkiItem
*
*****************************************************************************/
static BOOLEAN bEntryRemoveOldSkiItem(
    DSRL_ENTRY_OBJECT hDSRLEntry,
    UN32 un32CurrentTime
        )
{
    UN32 un32OldTime = 0;
    BOOLEAN bUpdated = FALSE;
    SKI_CONDITIONS_OBJECT hSkiCond = SKI_CONDITIONS_INVALID_OBJECT;
    SMSAPI_RETURN_CODE_ENUM eResult;

    eResult = WEATHER_MSG.eGetSkiConditions((WEATHER_MSG_OBJECT)hDSRLEntry,
        &hSkiCond);
    if ((eResult == SMSAPI_RETURN_CODE_SUCCESS) &&
        (hSkiCond != SKI_CONDITIONS_INVALID_OBJECT))
    {
        un32OldTime = SKI_CONDITIONS.un32TimeStamp(hSkiCond);
        if ((un32CurrentTime - un32OldTime) > WEATHER_MSG_TIMEOUT)
        {
            printf(WEATHER_MGR_OBJECT_NAME" Removing ski conditions item %x\n",
                hSkiCond
                    );
            bUpdated = TRUE;

            // set state of DSRLs for this entry to UPDATING
            vChangeEntryDSRLsState(hDSRLEntry, DSRL_STATE_UPDATING);

            WEATHER_MSG_bSetSkiCond((WEATHER_MSG_OBJECT)hDSRLEntry,
                SKI_CONDITIONS_INVALID_OBJECT);
        }
    }

    return bUpdated;
}

/*****************************************************************************
*
*   bEntryRemoveOldItems
*
*****************************************************************************/
static BOOLEAN bEntryRemoveOldItems(
    DSRL_ENTRY_OBJECT hDSRLEntry,
    WEATHER_REMOVE_OLD_ITEMS_ITERATOR_STRUCT *psIterator
        )
{
    BOOLEAN bContinue = FALSE;

    do
    {
        WEATHER_MSG_OBJECT_TYPE_ENUM eType = WEATHER_MSG_TYPE_INVALID;
        BOOLEAN bEntryUpdated = FALSE;

        if ((hDSRLEntry == DSRL_ENTRY_INVALID_OBJECT) ||
            (psIterator == NULL))
        {
            //something wrong with parameters
            break;
        }

        eType = WEATHER_MSG.eType((WEATHER_MSG_OBJECT)hDSRLEntry);
        switch (eType)
        {
            case WEATHER_MSG_TYPE_FORECAST:
            {
               bEntryUpdated = bEntryRemoveOldForecastItems(hDSRLEntry,
                   psIterator->un32CurrentTime);
            }
            break;

            case WEATHER_MSG_TYPE_SKI_REPORT:
            {
               bEntryUpdated = bEntryRemoveOldSkiItem(hDSRLEntry,
                   psIterator->un32CurrentTime );
            }
            break;

            case WEATHER_MSG_TYPE_INVALID:
            default:
            break;
        }

        if (bEntryUpdated == TRUE)
        {
            bEntryChange(hDSRLEntry, psIterator->psAppObj);

            DATASERVICE_IMPL_vLog(WEATHER_MGR_OBJECT_NAME
                ": Removing expired items from DSRL\n"
                    );
        }

        bContinue = TRUE;
    } while (FALSE);

    return bContinue;
}

/*****************************************************************************
*
*   vRemoveOldItemsFromEntries
*
*****************************************************************************/
static void vRemoveOldItemsFromEntries(
    WEATHER_APP_OBJECT_STRUCT *psAppObj,
    UN32 un32CurrentTime
        )
{
    WEATHER_REMOVE_OLD_ITEMS_ITERATOR_STRUCT sIterator;
    OSAL_RETURN_CODE_ENUM eReturnCode = OSAL_ERROR;

    sIterator.un32CurrentTime = un32CurrentTime;
    sIterator.psAppObj = psAppObj;

    eReturnCode = OSAL.eLinkedListIterate(psAppObj->hEntriesList,
        (OSAL_LL_ITERATOR_HANDLER)bEntryRemoveOldItems, &sIterator
            );
    if ((eReturnCode != OSAL_SUCCESS) &&
        (eReturnCode != OSAL_NO_OBJECTS))
    {
        SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
            WEATHER_MGR_OBJECT_NAME": failed to iterate"
            " items in list (%s)",
            OSAL.pacGetReturnCodeName(eReturnCode));
    }

    eReturnCode = OSAL.eLinkedListIterate(psAppObj->hDSRLList,
        (OSAL_LL_ITERATOR_HANDLER)bDSRLChangeState,
        (void *)(size_t)DSRL_STATE_READY);
    if ((eReturnCode != OSAL_SUCCESS) &&
        (eReturnCode != OSAL_NO_OBJECTS))
    {
        SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
            WEATHER_MGR_OBJECT_NAME": failed to iterate"
            " items in list (%s)",
            OSAL.pacGetReturnCodeName(eReturnCode));
    }

    return;
}

/*****************************************************************************
*
*   vStartCancelTimer
*
*****************************************************************************/
static void vStartCancelTimer (
    WEATHER_MGR_OBJECT_STRUCT *psObj
        )
{
    WEATHER_APP_OBJECT_STRUCT *psAppObj = NULL;

    do
    {
        UN32 un32ForecastCount = 0;
        UN32 un32SkiCondCount = 0;
        BOOLEAN bTimerNeeded = FALSE, bOk;

        psAppObj = psGetAppFacingObject((WEATHER_SERVICE_OBJECT)psObj);
        if (psAppObj == NULL)
        {
            break;
        }

        un32ForecastCount = FORECAST_un32Count();
        un32SkiCondCount = SKI_CONDITIONS_un32Count();

        bTimerNeeded = GsWeatherIntf.bIsTimerNeeded(psObj->hInterface);

        printf(WEATHER_MGR_OBJECT_NAME" un32ForecastCount is %u\n",
            un32ForecastCount);
        printf(WEATHER_MGR_OBJECT_NAME" un32SkiCondCount is %u\n",
            un32SkiCondCount);

        if ((bTimerNeeded != TRUE) && (un32ForecastCount == 0) &&
            (un32SkiCondCount == 0))
        {
            //timer is not needed
            break;
        }

        // Kick off the event
        bOk = DATASERVICE_IMPL_bSetTimedEvent(
            psObj->hCancelEvent, FALSE, FALSE,
            WEATHER_SERVICE_TIMER_PERIOD);
        if (bOk == TRUE)
        {
            puts(WEATHER_MGR_OBJECT_NAME" Starting timer");
        }
    } while (FALSE);

    vUnlockAppFacingObject(psAppObj);

    return;
}

/*****************************************************************************
*
*   vStopAgeoutTimer
*
*****************************************************************************/
static void vStopCancelTimer (
    WEATHER_MGR_OBJECT_STRUCT *psObj
        )
{
    puts(WEATHER_MGR_OBJECT_NAME" Stopping cancel timer");

    // Stop the cancel event from occurring
    DATASERVICE_IMPL_bStopTimedEvent(psObj->hCancelEvent);

    return;
}

/*****************************************************************************
*
*   bFillLocationRow
*
*****************************************************************************/
static BOOLEAN bFillLocationRow(
        SQL_QUERY_COLUMN_STRUCT *psColumn,
        N32 n32NumberOfColumns,
        WEATHER_LOCATION_ROW_STRUCT *psRow
            )
{
    LOCATION_FIELDS_ENUM eCurrentField = LOCATION_FIELD_LOC_ID;
    BOOLEAN bResult = FALSE;
    N32 n32FixedValue = 0;

    do {
        // Clear the location structure
        OSAL.bMemSet(psRow, 0, sizeof(WEATHER_LOCATION_ROW_STRUCT));

        // If there is at least one data row and the correct
        // number of columns (just make sure we have enough),
        // then we have good results
        if (n32NumberOfColumns != LOCATION_MAX_FIELDS)
        {
            break;
        }

        if (psRow == NULL)
        {
            break;
        }

        // Grab the entire station record
        do
        {
            switch (eCurrentField)
            {
            case LOCATION_FIELD_LOC_ID:
            {
                psRow->tID = (LOC_ID)
                            psColumn[eCurrentField].uData.sUN32.un32Data;
            }
            break;

            case LOCATION_FIELD_SKI_FLAG:
            {
                psRow->bSkiFlag = (BOOLEAN)
                            psColumn[eCurrentField].uData.sUN32.un32Data;
            }
            break;

            case LOCATION_FIELD_VERSION:
            {
                psRow->un8Version = (UN8)
                            psColumn[eCurrentField].uData.sUN32.un32Data;
            }
            break;

            case LOCATION_FIELD_AREA_NAME:
            {
                psRow->hAreaName = STRING.hCreate(
                        (const char *)psColumn[eCurrentField].uData.sCString.pcData,
                        strlen((const char *)psColumn[eCurrentField].uData.sCString.pcData));
            }
            break;

            case LOCATION_FIELD_ICAO_NAME:
            {
                psRow->hICAOName = STRING.hCreate(
                        (const char *)psColumn[eCurrentField].uData.sCString.pcData,
                        strlen((const char *)psColumn[eCurrentField].uData.sCString.pcData));
            }
            break;

            case LOCATION_FIELD_LATITUDE:
            {
                n32FixedValue = (N32)
                            psColumn[eCurrentField].uData.sUN32.un32Data;

                psRow->hLat =
                        OSAL_FIXED.hCreateInMemory(n32FixedValue, LOCATION_BINPOINT,
                                &(psRow->atLatFixedData[0]));
            }
            break;

            case LOCATION_FIELD_LONGITUDE:
            {
                n32FixedValue = (N32)
                            psColumn[eCurrentField].uData.sUN32.un32Data;

                psRow->hLon =
                        OSAL_FIXED.hCreateInMemory(n32FixedValue, LOCATION_BINPOINT,
                                &(psRow->atLonFixedData[0]));
            }
            break;

            default: // Shouldn't happen
            break;
            }

        } while (++eCurrentField < LOCATION_MAX_FIELDS);

        bResult = TRUE;
    } while (FALSE);

    return bResult;
}

/*******************************************************************************
*
*        bDSRLChangeEntry
*
*******************************************************************************/
static BOOLEAN bDSRLChangeEntry (
    DSRL_OBJECT hDSRL,
    WEATHER_DSRL_CHANGE_ENTRY_STRUCT *psChangeEntry
        )
{
    BOOLEAN bContinue = FALSE;

    do
    {
        DSRL_ADD_REPLACE_RESULT_ENUM eResult = DSRL_ADD_REPLACE_ERROR;

        if ((hDSRL == DSRL_INVALID_OBJECT) ||
            (psChangeEntry == NULL))
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                WEATHER_MGR_OBJECT_NAME": Invalid parameter"
                    );
            break;
        }

        if (psChangeEntry->hDSRLEntry == DSRL_ENTRY_INVALID_OBJECT)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                WEATHER_MGR_OBJECT_NAME": Invalid parameter"
                    );
            break;
        }

        eResult = DSRL_eReplaceEntry(hDSRL, psChangeEntry->hDSRLEntry,
            psChangeEntry->hDSRLEntry);
        if (eResult == DSRL_ADD_REPLACE_ERROR)
        {
            // remove the entry relations with the DSRL and moving the DSRL
            // to error
            vDSRLFinalizeEntry(hDSRL, psChangeEntry->hDSRLEntry,
                psChangeEntry->psAppObj);
            bDSRLChangeState(hDSRL, DSRL_STATE_ERROR);
            break;
        }
        else if (eResult == DSRL_ADD_REPLACE_NOP)
        {
            // just removing the entry from the DSRL, finalizer will be called
            // inside DSRL_vRemoveEntry
            DSRL_vRemoveEntry(hDSRL, psChangeEntry->hDSRLEntry);
        }

        bContinue = TRUE;
    } while (FALSE);

    return bContinue;
}

/*****************************************************************************
*
*   bGetLocationFromDB
*
*****************************************************************************/
static BOOLEAN bGetLocationFromDB(
    SQL_QUERY_COLUMN_STRUCT *psColumn,
    N32 n32NumberOfColumns,
    WEATHER_LOCATION_STRUCT *psLoc
        )
{
    WEATHER_LOCATION_ROW_STRUCT sLocationRow;
    BOOLEAN bResult = FALSE;

    do {
        bResult = bFillLocationRow(psColumn, n32NumberOfColumns, &sLocationRow);
        if (bResult == FALSE)
        {
            if (sLocationRow.hAreaName != STRING_INVALID_OBJECT)
            {
                STRING_vDestroy(sLocationRow.hAreaName);
            }

            if (sLocationRow.hICAOName != STRING_INVALID_OBJECT)
            {
                STRING_vDestroy(sLocationRow.hICAOName);
            }

            break;
        }

        psLoc->hLocation = hCreateLocationFromRow(&sLocationRow);
        psLoc->hICAO = sLocationRow.hICAOName;
        if (psLoc->hLocation == LOCATION_INVALID_OBJECT)
        {
            bResult = FALSE;
            break;
        }
        bResult = TRUE;
    } while (FALSE);

    return bResult;
}

/*****************************************************************************
*
*   bDSRLChangeState
*
*****************************************************************************/
static BOOLEAN bDSRLChangeState(
    DSRL_OBJECT hDSRL,
    DSRL_STATE_ENUM eState
        )
{
    if (hDSRL != DSRL_INVALID_OBJECT)
    {
        DSRL_STATE_ENUM eCurState = DSRL_STATE_ERROR;

        eCurState = DSRL.eState(hDSRL);
        if (eCurState != DSRL_STATE_ERROR)
        {
            DSRL_vSetState(hDSRL, eState);
        }
    }

    return TRUE;
}

/*****************************************************************************
*
*   vChangeEntryDSRLsState
*
*****************************************************************************/
static void vChangeEntryDSRLsState(
    DSRL_ENTRY_OBJECT hDSRLEntry,
    DSRL_STATE_ENUM eState
        )
{
    WEATHER_DSRL_ENTRY_DESC_STRUCT *psEntryDesc = NULL;

    psEntryDesc =
        (WEATHER_DSRL_ENTRY_DESC_STRUCT *)DSRL_ENTRY_pvServiceData(
        hDSRLEntry);
    if (psEntryDesc != NULL)
    {
        OSAL_RETURN_CODE_ENUM eReturnCode;

        // go through DSRLs changing their state
        eReturnCode = OSAL.eLinkedListIterate(psEntryDesc->hDSRLList,
            (OSAL_LL_ITERATOR_HANDLER)bDSRLChangeState,
            (void *)(size_t)eState);
        if ((eReturnCode != OSAL_SUCCESS) &&
            (eReturnCode != OSAL_NO_OBJECTS))
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                WEATHER_MGR_OBJECT_NAME": failed to iterate DSRLs"
                " list (%s)", OSAL.pacGetReturnCodeName(eReturnCode));
        }
    }
    else
    {
        SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
            WEATHER_MGR_OBJECT_NAME": Entry descriptor is NULL");
    }

    return;
}

/*****************************************************************************
*
*   vDestroyWeatherLocData
*
*****************************************************************************/
static void vDestroyWeatherLocData(
    WEATHER_LOCATION_STRUCT *psObj
        )
{
    printf(WEATHER_MGR_OBJECT_NAME": [%s] start with parameter psLoc=%p\n", __FUNCTION__, psObj);

    if (psObj->hLocation != LOCATION_INVALID_OBJECT)
    {
        printf(WEATHER_MGR_OBJECT_NAME": destroying location %x\n", psObj->hLocation);
        LOCATION.vDestroy(psObj->hLocation);
        psObj->hLocation = LOCATION_INVALID_OBJECT;
    }

    if (psObj->hICAO != STRING_INVALID_OBJECT)
    {
        printf(WEATHER_MGR_OBJECT_NAME": destroying string %x\n", psObj->hICAO);
        STRING_vDestroy(psObj->hICAO);
        psObj->hICAO = STRING_INVALID_OBJECT;
    }

    return;
}

/*****************************************************************************
*
*   hDSRLEntryCreate
*
*****************************************************************************/
static DSRL_ENTRY_OBJECT hDSRLEntryCreate(
    WEATHER_APP_OBJECT_STRUCT *psObj,
    WEATHER_LOCATION_STRUCT *psLoc,
    WEATHER_MSG_OBJECT_TYPE_ENUM eType
        )
{
    DSRL_ENTRY_OBJECT hDSRLEntry = DSRL_ENTRY_INVALID_OBJECT;
    BOOLEAN bResult = FALSE;
    OSAL_RETURN_CODE_ENUM eReturnCode = OSAL_ERROR;
    WEATHER_DSRL_ENTRY_DESC_STRUCT *psEntryDesc = NULL;

    do
    {
        // Create DSRL entry - weather message object
        hDSRLEntry = (DSRL_ENTRY_OBJECT)WEATHER_MSG_hCreate(
            (SMS_OBJECT)psObj, sizeof(WEATHER_DSRL_ENTRY_DESC_STRUCT),
            psLoc->hLocation, psLoc->hICAO, eType);
        if (hDSRLEntry == DSRL_ENTRY_INVALID_OBJECT)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                WEATHER_MGR_OBJECT_NAME": cannot create WEATHER_MGS"
                " object.");
            break;
        }

        psEntryDesc =
            (WEATHER_DSRL_ENTRY_DESC_STRUCT *)DSRL_ENTRY_pvServiceData(
            hDSRLEntry);
        if (psEntryDesc == NULL)
        {
            break;
        }

        eReturnCode = OSAL.eLinkedListAdd(psObj->hEntriesList,
            &psEntryDesc->hPoolEntry, hDSRLEntry);
        if (eReturnCode != OSAL_SUCCESS)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                WEATHER_MGR_OBJECT_NAME": failed to add entry to list (%s)",
                OSAL.pacGetReturnCodeName(eReturnCode));
            break;
        }

        if (eType == WEATHER_MSG_TYPE_FORECAST)
        {
            eReturnCode = OSAL.eLinkedListAdd(
                psObj->hNewForecastEntriesList,
                OSAL_INVALID_LINKED_LIST_ENTRY_PTR, hDSRLEntry);
            if (eReturnCode != OSAL_SUCCESS)
            {
                SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    WEATHER_MGR_OBJECT_NAME": cannot add to new entries"
                    " list (%s)", OSAL.pacGetReturnCodeName(eReturnCode));
                break;
            }
        }
        else
        {
            eReturnCode = OSAL.eLinkedListAdd(psObj->hNewSkiEntriesList,
                OSAL_INVALID_LINKED_LIST_ENTRY_PTR, hDSRLEntry);
            if (eReturnCode != OSAL_SUCCESS)
            {
                SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    WEATHER_MGR_OBJECT_NAME": cannot add to new entries"
                    " list (%s)", OSAL.pacGetReturnCodeName(eReturnCode));
                break;
            }
        }

        bResult = TRUE;
    } while (FALSE);

    if (bResult == FALSE)
    {
        if (psEntryDesc != NULL)
        {
            if (psEntryDesc->hPoolEntry != OSAL_INVALID_LINKED_LIST_ENTRY)
            {
                eReturnCode =
                    OSAL.eLinkedListRemove(psEntryDesc->hPoolEntry);
                if (eReturnCode != OSAL_SUCCESS)
                {
                    SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                        WEATHER_MGR_OBJECT_NAME": cannot add to new entries"
                        " list (%s)", OSAL.pacGetReturnCodeName(eReturnCode));
                }

                psEntryDesc->hPoolEntry = OSAL_INVALID_LINKED_LIST_ENTRY;
            }
        }

        if (hDSRLEntry != DSRL_ENTRY_INVALID_OBJECT)
        {
            WEATHER_MSG_vDestroy((WEATHER_MSG_OBJECT)hDSRLEntry);
        }
    }

    return hDSRLEntry;
}

/*****************************************************************************
*
*   hDSRLEntryGet
*
*   Returns pointer to existing or newly created DSRL ENTRY for requested
*   location and type. If entry cannot be created due to some issues
*   (for example, location radius cannot include any of predefined locations)
*   DSRL_ENTRY_INVALID_OBJECT will be returned
*
*****************************************************************************/
static DSRL_ENTRY_OBJECT hDSRLEntryGet(
    DSRL_TARGET_OBJECT hTarget,
    WEATHER_MSG_OBJECT_TYPE_ENUM eType,
    void *pvCallbackArg
        )
{
    DSRL_ENTRY_OBJECT hDSRLEntry = DSRL_ENTRY_INVALID_OBJECT;
    WEATHER_LOCATION_STRUCT sLoc;
    WEATHER_APP_OBJECT_STRUCT *psObj =
        (WEATHER_APP_OBJECT_STRUCT*)pvCallbackArg;

    OSAL.bMemSet(&sLoc, 0, sizeof(sLoc));

    do
    {
        BOOLEAN bSuccess = FALSE;
        BOOLEAN bSkiFlag =
            (eType == WEATHER_MSG_TYPE_FORECAST) ? FALSE : TRUE;
        LOCATION_OBJECT hLocation;
        LOCID_OBJECT hLocID = LOCID_INVALID_OBJECT;
        LOC_ID tID = LOC_INVALID_ID;

        hLocation = DSRL_TARGET.hLocation(hTarget);
        if (hLocation == LOCATION_INVALID_OBJECT)
        {
            break;
        }

        bSuccess = bGetPredefinedLocFromLoc(psObj, hLocation, bSkiFlag,
            &sLoc);
        if (bSuccess == FALSE)
        {
            //no predefined location for this target. exiting
            break;
        }

        hLocID = LOCATION.hLocID(sLoc.hLocation);
        if (hLocID == LOCID_INVALID_OBJECT)
        {
            break;
        }

        tID = LOCID.tID(hLocID);
        if (tID == LOC_INVALID_ID)
        {
            break;
        }

        //location exist, trying to find entry with such location in pool
        hDSRLEntry = hSearchEntry(psObj, tID);
        if(hDSRLEntry != DSRL_ENTRY_INVALID_OBJECT)
        {
            //existing entry has been found
            break;
        }

        //no entry with this location, lets create new
        hDSRLEntry = hDSRLEntryCreate(psObj, &sLoc, eType);

    } while (FALSE);

    vDestroyWeatherLocData(&sLoc);

    return hDSRLEntry;
}

/*******************************************************************************
*
*        bEntryChange
*
*******************************************************************************/
BOOLEAN bEntryChange(
    DSRL_ENTRY_OBJECT hDSRLEntry,
    WEATHER_APP_OBJECT_STRUCT *psAppObj
        )
{
    BOOLEAN bResult = FALSE;

    do
    {
        OSAL_RETURN_CODE_ENUM eReturnCode = OSAL_ERROR;
        WEATHER_DSRL_ENTRY_DESC_STRUCT *psEntryDesc = NULL;
        WEATHER_DSRL_CHANGE_ENTRY_STRUCT sChangeEntry;

        printf(WEATHER_MGR_OBJECT_NAME": Changing entry %p\n", hDSRLEntry);

        psEntryDesc =
            (WEATHER_DSRL_ENTRY_DESC_STRUCT *)DSRL_ENTRY_pvServiceData(
            hDSRLEntry);
        if (psEntryDesc == NULL)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                WEATHER_MGR_OBJECT_NAME": Entry descriptor is NULL"
                    );
            break;
        }

        sChangeEntry.hDSRLEntry = hDSRLEntry;
        sChangeEntry.psAppObj = psAppObj;

        // go through DSRLs trying to replace an old entry with a new one
        eReturnCode = OSAL.eLinkedListIterate(psEntryDesc->hDSRLList,
            (OSAL_LL_ITERATOR_HANDLER)bDSRLChangeEntry, &sChangeEntry
                );
        if ((eReturnCode != OSAL_SUCCESS) &&
            (eReturnCode != OSAL_NO_OBJECTS))
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                WEATHER_MGR_OBJECT_NAME": failed to iterate DSRLs"
                " list (%s)", OSAL.pacGetReturnCodeName(eReturnCode));
            break;
        }

        bResult = TRUE;
    } while (FALSE);

    return bResult;
}

/*******************************************************************************
*
*        hSearchEntry
*
*******************************************************************************/
static DSRL_ENTRY_OBJECT hSearchEntry(
    WEATHER_APP_OBJECT_STRUCT *psAppObj,
    LOC_ID tID
        )
{
    DSRL_ENTRY_OBJECT hResult = DSRL_ENTRY_INVALID_OBJECT;
    LOCID_OBJECT hLocID = LOCID_INVALID_OBJECT;

    do
    {
        OSAL_RETURN_CODE_ENUM eReturnCode;
        OSAL_LINKED_LIST_ENTRY hEntry = OSAL_INVALID_LINKED_LIST_ENTRY;
        BOOLEAN bSuccess = FALSE;

        hLocID = LOCID_hCreate(tID, LOCID_TYPE_WEATHER, NULL);
        if (hLocID == LOCID_INVALID_OBJECT)
        {
            break;
        }

        bSuccess = LOCATION_bUpdateLocID(psAppObj->hDummyLocation, hLocID,
            FALSE);
        if (bSuccess == FALSE)
        {
            break;
        }

        hLocID = LOCID_INVALID_OBJECT;

        // searching in the list of DSRL entries for requested LOCID
        eReturnCode = OSAL.eLinkedListSearch(psAppObj->hEntriesList,
            &hEntry, psAppObj->hDummyMsg);
        if ((eReturnCode != OSAL_SUCCESS) &&
            (eReturnCode != OSAL_OBJECT_NOT_FOUND))
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                WEATHER_MGR_OBJECT_NAME": failed to search for entry (%s)",
                OSAL.pacGetReturnCodeName(eReturnCode)
                    );
            break;
        }

        if (eReturnCode == OSAL_SUCCESS)
        {
            hResult =
                (DSRL_ENTRY_OBJECT)OSAL.pvLinkedListThis(hEntry);
        }

    } while (FALSE);

    if (hLocID != LOCID_INVALID_OBJECT)
    {
        LOCID.vDestroy(hLocID);
    }

    return hResult;
}

/*****************************************************************************
*
*   bDSRLReplaceTargets
*
*****************************************************************************/
static BOOLEAN bDSRLReplaceTargets(
    DSRL_ARG_STRUCT *psDSRLArg
        )
{
    BOOLEAN bResult = FALSE;
    OSAL_RETURN_CODE_ENUM eReturnCode = OSAL_ERROR;
    WEATHER_DSRL_DESC_STRUCT *psDSRLDesc = NULL;

    psDSRLDesc =
        (WEATHER_DSRL_DESC_STRUCT *)DSRL_pvServiceData(psDSRLArg->hDSRL);
    if (psDSRLDesc == NULL)
    {
        return FALSE;
    }

    eReturnCode = OSAL.eLinkedListRemoveAll(psDSRLDesc->hTargetList,
        (OSAL_LL_RELEASE_HANDLER)DSRL_TARGET_vDestroyByType
            );
    if (eReturnCode !=  OSAL_SUCCESS)
    {
        SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
            WEATHER_MGR_OBJECT_NAME": failed to release target desc"
            " items in DSRL desc (%s)",
            OSAL.pacGetReturnCodeName(eReturnCode)
                );
    }
    else
    {
        //Add new targets
        bResult = bDSRLAddTargets(psDSRLArg);
    }

    return bResult;
}

/*****************************************************************************
*
*   vDestroyUnusedEntry
*
*****************************************************************************/
static void vDestroyUnusedEntry(
    DSRL_ENTRY_OBJECT hDSRLEntry
        )
{
    do
    {
        WEATHER_DSRL_ENTRY_DESC_STRUCT *psEntryDesc = NULL;
        OSAL_RETURN_CODE_ENUM eReturnCode = OSAL_ERROR;
        UN32 un32Items = 0;

        if (hDSRLEntry == DSRL_ENTRY_INVALID_OBJECT)
        {
            // error !
            break;
        }

        psEntryDesc = (WEATHER_DSRL_ENTRY_DESC_STRUCT *)
            DSRL_ENTRY_pvServiceData(hDSRLEntry);
        if (psEntryDesc == NULL)
        {
            break;
        }

        // in some cases DSRL list could even be not created yet
        // in this case we suggest that no assigned DSRLs exist
        if (psEntryDesc->hDSRLList != OSAL_INVALID_OBJECT_HDL)
        {
            eReturnCode = OSAL.eLinkedListItems(psEntryDesc->hDSRLList,
                &un32Items);
            if (eReturnCode != OSAL_SUCCESS)
            {
                SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    WEATHER_MGR_OBJECT_NAME": failed to get number of DSRL"
                    " desc items in list (%s)",
                    OSAL.pacGetReturnCodeName(eReturnCode)
                        );
                break;
            }
        }
        else
        {
            un32Items = 0;
        }

        if (un32Items != 0)
        {
            break;
        }

        eReturnCode = OSAL.eLinkedListRemove(psEntryDesc->hPoolEntry);
        if (eReturnCode != OSAL_SUCCESS)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                WEATHER_MGR_OBJECT_NAME": failed to remove DSRL"
                " entry from pool (%s)",
                OSAL.pacGetReturnCodeName(eReturnCode));
        }

        vDSRLEntryRelease(hDSRLEntry);

    } while (FALSE);

    return;
}

/*****************************************************************************
*
*   bRemoveUnusedEntries
*
*****************************************************************************/
static BOOLEAN bRemoveUnusedEntries(
    WEATHER_APP_OBJECT_STRUCT *psAppObj
        )
{
    OSAL_RETURN_CODE_ENUM eReturnCode;

    // clearing removed entries list
    eReturnCode =
        OSAL.eLinkedListRemoveAll(psAppObj->hRemovedEntriesList,
        (OSAL_LL_RELEASE_HANDLER)vDestroyUnusedEntry);
    if (eReturnCode != OSAL_SUCCESS)
    {
        SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
            WEATHER_MGR_OBJECT_NAME": failed to clean unused DSRL"
            " entries list (%s)",
            OSAL.pacGetReturnCodeName(eReturnCode)
                );
    }

    return TRUE;
}

/*****************************************************************************
*
*   bTargetEntriesGenerator
*
*****************************************************************************/
static BOOLEAN bTargetEntriesGenerator(
    DSRL_TARGET_OBJECT hTarget,
    WEATHER_APP_OBJECT_STRUCT *psAppObj
        )
{
    DSRL_ENTRY_OBJECT hDSRLEntry = DSRL_ENTRY_INVALID_OBJECT;
    OSAL_RETURN_CODE_ENUM eReturnCode = OSAL_ERROR;
    BOOLEAN bContinue = TRUE;

    if ((hTarget == DSRL_TARGET_INVALID_OBJECT) || (psAppObj == NULL))
    {
        return FALSE;
    }

    do
    {
        //get entry for forecast location and add it to list
        hDSRLEntry = hDSRLEntryGet(hTarget, WEATHER_MSG_TYPE_FORECAST,
            psAppObj);
        if (hDSRLEntry != DSRL_ENTRY_INVALID_OBJECT)
        {
            eReturnCode = OSAL.eLinkedListAdd(psAppObj->hNewDSRLEntriesList,
                OSAL_INVALID_LINKED_LIST_ENTRY_PTR, hDSRLEntry);
            if (eReturnCode == OSAL_ERROR_LIST_ITEM_NOT_UNIQUE)
            {
                printf(WEATHER_MGR_OBJECT_NAME": entry already in list\n");
            }
            else if (eReturnCode != OSAL_SUCCESS)
            {
                SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    WEATHER_MGR_OBJECT_NAME": add to list failed (%s)",
                    OSAL.pacGetReturnCodeName(eReturnCode)
                        );
                bContinue = FALSE;
                break;
            }
        }

        //get entry for ski location and add it to list
        hDSRLEntry = hDSRLEntryGet(hTarget, WEATHER_MSG_TYPE_SKI_REPORT,
            psAppObj);
        if (hDSRLEntry != DSRL_ENTRY_INVALID_OBJECT)
        {
            eReturnCode = OSAL.eLinkedListAdd(
                psAppObj->hNewDSRLEntriesList,
                OSAL_INVALID_LINKED_LIST_ENTRY_PTR, hDSRLEntry);
            if (eReturnCode == OSAL_ERROR_LIST_ITEM_NOT_UNIQUE)
            {
                printf(WEATHER_MGR_OBJECT_NAME": entry already in list\n");
            }
            else if (eReturnCode != OSAL_SUCCESS)
            {
                SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    WEATHER_MGR_OBJECT_NAME": add to list failed (%s)",
                    OSAL.pacGetReturnCodeName(eReturnCode));
                bContinue = FALSE;
                break;
            }
        }

    } while (FALSE);

    printf(WEATHER_MGR_OBJECT_NAME": bTargetEntriesGenerator called"
        " with result %u\n", bContinue);

    return bContinue;
}

/*****************************************************************************
*
*   bGenerateDSRLEntriesList
*
*****************************************************************************/
static BOOLEAN bGenerateDSRLEntriesList(
    WEATHER_APP_OBJECT_STRUCT *psAppObj,
    OSAL_OBJECT_HDL hTargetList
        )
{
    OSAL_RETURN_CODE_ENUM eReturnCode;

    // iterating through all the targets and adding the generated entries
    // to the list of new entries in psAppObj
    eReturnCode = OSAL.eLinkedListIterate(hTargetList,
        (OSAL_LL_ITERATOR_HANDLER)bTargetEntriesGenerator, psAppObj);
    if ((eReturnCode != OSAL_SUCCESS) && (eReturnCode != OSAL_NO_OBJECTS))
    {
        SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
            WEATHER_MGR_OBJECT_NAME": Error iterating targets list (%s)",
            OSAL.pacGetReturnCodeName(eReturnCode));
        return FALSE;
    }

    return TRUE;
}

/*****************************************************************************
*
*   n16CompareHandles
*
*****************************************************************************/
static N16 n16CompareHandles(
    void *pvArg1,
    void *pvArg2
        )
{
    return COMPARE(pvArg1, pvArg2);
}

/*****************************************************************************
*
*   vDSRLFinalizeEntry
*
*****************************************************************************/
static void vDSRLFinalizeEntry(
    DSRL_OBJECT hDSRL,
    DSRL_ENTRY_OBJECT hDSRLEntry,
    WEATHER_APP_OBJECT_STRUCT *psAppObj
        )
{
    if ((hDSRL != DSRL_INVALID_OBJECT) &&
        (hDSRLEntry != DSRL_ENTRY_INVALID_OBJECT))
    {
        BOOLEAN bSuccess;

        // removing the DSRL handle from the entry descriptor
        bSuccess = bEntryRemoveDSRLLink(psAppObj, hDSRLEntry, hDSRL);
        if (bSuccess == FALSE)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                WEATHER_MGR_OBJECT_NAME": failed to remove entry"
                " to DSRL link");
        }

        // removing the entry handle from the DSRL descriptor
        bSuccess = bDSRLRemoveEntryLink(hDSRL, hDSRLEntry);
        if (bSuccess == FALSE)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                WEATHER_MGR_OBJECT_NAME": failed to remove DSRL"
                " to entry link");
        }

    }
    else
    {
        SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
            WEATHER_MGR_OBJECT_NAME": failed to finalize entry: hDSRL %p,"
            " hDSRLEntry %p, callback arg %p", hDSRL, hDSRLEntry, psAppObj
                );
    }

    return;
}

/*****************************************************************************
*
*   bDiffListGenerator
*
*****************************************************************************/
static BOOLEAN bDiffListGenerator(
    DSRL_ENTRY_OBJECT hDSRLEntry,
    WEATHER_DIFF_LIST_ITERATOR_STRUCT *psIterator
        )
{
    BOOLEAN bContinue = TRUE;
    OSAL_RETURN_CODE_ENUM eReturnCode = OSAL_ERROR;

    if ((hDSRLEntry == DSRL_ENTRY_INVALID_OBJECT) ||
        (psIterator == NULL))
    {
        return FALSE;
    }

    // searching for the entry in sub list
    eReturnCode = OSAL.eLinkedListSearch(psIterator->hSubList,
        &psIterator->hEntry, hDSRLEntry);
    if (eReturnCode == OSAL_OBJECT_NOT_FOUND)
    {
        // object not found in list, lets add entry to diff
        // list
        eReturnCode = OSAL.eLinkedListAdd(psIterator->hDiffList,
            OSAL_INVALID_LINKED_LIST_ENTRY_PTR, hDSRLEntry);
        if ((eReturnCode != OSAL_SUCCESS) &&
            (eReturnCode != OSAL_ERROR_LIST_ITEM_NOT_UNIQUE))
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                WEATHER_MGR_OBJECT_NAME": add to diff list failed"
                " (%s)", OSAL.pacGetReturnCodeName(eReturnCode));
            bContinue = FALSE;
        }
    }
    else if (eReturnCode != OSAL_SUCCESS)
    {
        SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
            WEATHER_MGR_OBJECT_NAME": search in sub list failed"
            " (%s)", OSAL.pacGetReturnCodeName(eReturnCode));
        bContinue = FALSE;
    }

    return bContinue;
}

/*****************************************************************************
*
*   bGenerateEntriesDiffList
*
*****************************************************************************/
static BOOLEAN bGenerateEntriesDiffList (
    OSAL_OBJECT_HDL hMainList,
    OSAL_OBJECT_HDL hSubList,
    OSAL_OBJECT_HDL hDiffList
        )
{
    BOOLEAN bResult = TRUE;
    OSAL_RETURN_CODE_ENUM eReturnCode = OSAL_ERROR;
    WEATHER_DIFF_LIST_ITERATOR_STRUCT sIterator;

    OSAL.bMemSet(&sIterator, 0, sizeof(sIterator));

    sIterator.hDiffList = hDiffList;
    sIterator.hSubList = hSubList;

    // populate DiffList with entries from SubList
    // which are absent in MainList
    eReturnCode = OSAL.eLinkedListIterate(hMainList,
        (OSAL_LL_ITERATOR_HANDLER)bDiffListGenerator, &sIterator
            );
    if ((eReturnCode != OSAL_SUCCESS) && (eReturnCode != OSAL_NO_OBJECTS))
    {
        SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
            WEATHER_MGR_OBJECT_NAME": cannot iterate linked list (%s)",
            OSAL.pacGetReturnCodeName(eReturnCode));
        bResult = FALSE;
    }

    return bResult;
}

/*****************************************************************************
*
*   bDSRLFillDiffEntriesLists
*
*****************************************************************************/
static BOOLEAN bDSRLFillDiffEntriesLists(
    DSRL_OBJECT hDSRL,
    WEATHER_APP_OBJECT_STRUCT *psAppObj
        )
{
    BOOLEAN bResult = FALSE;
    OSAL_RETURN_CODE_ENUM eReturnCode = OSAL_ERROR;

    do
    {
        BOOLEAN bSuccess = FALSE;
        WEATHER_DSRL_DESC_STRUCT *psDSRLDesc = NULL;

        psDSRLDesc = (WEATHER_DSRL_DESC_STRUCT *)DSRL_pvServiceData(hDSRL);
        if (psDSRLDesc == NULL)
        {
            break;
        }

        bSuccess = bGenerateDSRLEntriesList(psAppObj, psDSRLDesc->hTargetList);
        if (bSuccess == FALSE)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                WEATHER_MGR_OBJECT_NAME": cannot generate new list "
                "of entries");
            break;
        }

        bSuccess = bGenerateEntriesDiffList(psAppObj->hNewDSRLEntriesList,
            psDSRLDesc->hEntriesList, psAppObj->hEntriesToAddList);
        if (bSuccess == FALSE)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                WEATHER_MGR_OBJECT_NAME": cannot generate list "
                "of entries to add");
            break;
        }

        bSuccess = bGenerateEntriesDiffList(psDSRLDesc->hEntriesList,
            psAppObj->hNewDSRLEntriesList, psAppObj->hEntriesToRemoveList);
        if (bSuccess == FALSE)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                WEATHER_MGR_OBJECT_NAME": cannot generate list "
                "of entries to remove");
            break;
        }

        bResult = TRUE;
    } while (FALSE);

    eReturnCode =
        OSAL.eLinkedListRemoveAll(psAppObj->hNewDSRLEntriesList, NULL);
    if (eReturnCode != OSAL_SUCCESS)
    {
        SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
            WEATHER_MGR_OBJECT_NAME": cannot cleanup linked list (%s)",
            OSAL.pacGetReturnCodeName(eReturnCode));
    }

    return bResult;
}

/*****************************************************************************
*
*   vDSRLRelease
*
*****************************************************************************/
static void vDSRLRelease(
    DSRL_OBJECT hDSRL
        )
{
    if (hDSRL != DSRL_INVALID_OBJECT)
    {
        OSAL_RETURN_CODE_ENUM eReturnCode = OSAL_ERROR;
        WEATHER_DSRL_DESC_STRUCT *psDSRLDesc = NULL;

        printf(WEATHER_MGR_OBJECT_NAME": Releasing DSRL %p\n", hDSRL);

        psDSRLDesc = (WEATHER_DSRL_DESC_STRUCT *)DSRL_pvServiceData(hDSRL);
        if (psDSRLDesc != NULL)
        {
            if (psDSRLDesc->hTargetList != OSAL_INVALID_OBJECT_HDL)
            {
                eReturnCode = OSAL.eLinkedListRemoveAll(
                    psDSRLDesc->hTargetList,
                    (OSAL_LL_RELEASE_HANDLER)DSRL_TARGET_vDestroyByType);
                if (eReturnCode != OSAL_SUCCESS)
                {
                    SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                        WEATHER_MGR_OBJECT_NAME": failed to cleanup"
                        " list (%s)",
                        OSAL.pacGetReturnCodeName(eReturnCode));
                }

                eReturnCode = OSAL.eLinkedListDelete(
                    psDSRLDesc->hTargetList);
                if (eReturnCode != OSAL_SUCCESS)
                {
                    SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                        WEATHER_MGR_OBJECT_NAME": failed to delete"
                        " list (%s)",
                        OSAL.pacGetReturnCodeName(eReturnCode));
                }
                psDSRLDesc->hTargetList = OSAL_INVALID_OBJECT_HDL;
            }

            if (psDSRLDesc->hEntriesList != OSAL_INVALID_OBJECT_HDL)
            {
                eReturnCode = OSAL.eLinkedListRemoveAll(
                    psDSRLDesc->hEntriesList, NULL);
                if (eReturnCode != OSAL_SUCCESS)
                {
                    SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                        WEATHER_MGR_OBJECT_NAME": failed to cleanup"
                        " list (%s)",
                        OSAL.pacGetReturnCodeName(eReturnCode));
                }

                eReturnCode = OSAL.eLinkedListDelete(
                    psDSRLDesc->hEntriesList);
                if (eReturnCode != OSAL_SUCCESS)
                {
                    SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                        WEATHER_MGR_OBJECT_NAME": failed to delete"
                        " list (%s)",
                        OSAL.pacGetReturnCodeName(eReturnCode));
                }
                psDSRLDesc->hEntriesList = OSAL_INVALID_OBJECT_HDL;
            }
        }

        DSRL_vDestroy(hDSRL);
    }

    return;
}
