/******************************************************************************/
/*                    Copyright (c) Sirius XM Radio, Inc.                     */
/*                            All Rights Reserved                             */
/*         Licensed Materials - Property of Sirius XM Radio, Inc.             */
/******************************************************************************/
/*******************************************************************************
 *
 * DESCRIPTION
 *
 *  This module contains the Object:WEATHER_EVENT implementation for the
 *  Sirius Module Services (SMS)
 *
 ******************************************************************************/
#include "standard.h"
#include "osal.h"

#include "sms_api.h"
#include "sms_obj.h"
#include "_weather_event_obj.h"
#include "weather_event_obj.h"
#include "string_obj.h"

#include "sms_api_debug.h"

/*****************************************************************************
                             PUBLIC FUNCTIONS
*****************************************************************************/

/*****************************************************************************
*
*   eCode
*
*****************************************************************************/
static SMSAPI_RETURN_CODE_ENUM eCode (
    WEATHER_EVENT_OBJECT hWeatherEvent,
    WEATHER_EVENT_CODE_ENUM *peEventCode
        )
{
    BOOLEAN bOwner = FALSE;
    SMSAPI_RETURN_CODE_ENUM eResult = SMSAPI_RETURN_CODE_ERROR;
    WEATHER_EVENT_OBJECT_STRUCT *psObj = (WEATHER_EVENT_OBJECT_STRUCT *)hWeatherEvent;

    do
    {
        if (peEventCode == NULL)
        {
            eResult = SMSAPI_RETURN_CODE_INVALID_INPUT;
            break;
        }

        bOwner = SMSO_bOwner((SMS_OBJECT)hWeatherEvent);
        if (bOwner == FALSE)
        {
            eResult = SMSAPI_RETURN_CODE_NOT_OWNER;
            break;
        }

        if (psObj->bAvailable == FALSE)
        {
            eResult = SMSAPI_RETURN_CODE_NOT_FOUND;
            break;
        }
        *peEventCode = (WEATHER_EVENT_CODE_ENUM)psObj->sData.un8EventCode;
        eResult = SMSAPI_RETURN_CODE_SUCCESS;
    } while (FALSE);

    return eResult;
}


/*****************************************************************************
*
*   hDescription
*
*****************************************************************************/
static STRING_OBJECT hDescription (
    WEATHER_EVENT_OBJECT hWeatherEvent
        )
{
    BOOLEAN bOwner = FALSE;
    STRING_OBJECT hResult = STRING_INVALID_OBJECT;
    WEATHER_EVENT_OBJECT_STRUCT *psObj = (WEATHER_EVENT_OBJECT_STRUCT *)hWeatherEvent;

    bOwner = SMSO_bOwner((SMS_OBJECT)hWeatherEvent);
    if (bOwner == TRUE)
    {
        if (psObj->bAvailable == TRUE)
        {
            hResult = psObj->sData.hDescription;
        }
    }

    return hResult;
}

/*****************************************************************************
*
*   eCode
*
*****************************************************************************/
static SMSAPI_RETURN_CODE_ENUM eIconCode (
    WEATHER_EVENT_OBJECT hWeatherEvent,
    UN8 *pun8EventIconCode
        )
{
    BOOLEAN bOwner = FALSE;
    SMSAPI_RETURN_CODE_ENUM eResult = SMSAPI_RETURN_CODE_ERROR;
    WEATHER_EVENT_OBJECT_STRUCT *psObj = (WEATHER_EVENT_OBJECT_STRUCT *)hWeatherEvent;

    do
    {
        if (pun8EventIconCode == NULL)
        {
            eResult = SMSAPI_RETURN_CODE_INVALID_INPUT;
            break;
        }

        bOwner = SMSO_bOwner((SMS_OBJECT)hWeatherEvent);
        if (bOwner == FALSE)
        {
            eResult = SMSAPI_RETURN_CODE_NOT_OWNER;
            break;
        }

        if ((psObj->bAvailable != TRUE) || (psObj->sData.un8IconCode == 0))
        {
            eResult = SMSAPI_RETURN_CODE_NOT_FOUND;
            break;
        }
        eResult = SMSAPI_RETURN_CODE_SUCCESS;
        *pun8EventIconCode = psObj->sData.un8IconCode;
    } while (FALSE);

    return eResult;
}


/*****************************************************************************
*
*   n32FPrintf
*
*****************************************************************************/
static N32 n32FPrintf (
    WEATHER_EVENT_OBJECT hWeatherEvent,
    FILE *psFile
        )
{
	N32 n32Return = 0, n32Temp;
    WEATHER_EVENT_OBJECT_STRUCT *psObj =
        (WEATHER_EVENT_OBJECT_STRUCT *)hWeatherEvent;
    BOOLEAN bOwner;

    // Ensure we own the object
    bOwner = SMSO_bOwner((SMS_OBJECT)hWeatherEvent);

    if ((bOwner == FALSE) || (psFile == NULL))
    {
        return EOF;
    }

    if (psObj->bAvailable == TRUE)
       {
        // Print Weatheer Event information header
        n32Return += fprintf(psFile, "\nWeather Event (Object: 0x%X):\n", hWeatherEvent);
        n32Return += fprintf(psFile, "\tEvent: ");
        n32Temp = STRING.n32FWrite(psObj->sData.hDescription, psFile);
        if (n32Temp > 0)
        {
            n32Return += n32Temp;
        }
        n32Return += fprintf(psFile, "\n\tEvent Code: %u",
            psObj->sData.un8EventCode);
        n32Return += fprintf(psFile, "\n\tIcon Code: %u\n",
            psObj->sData.un8IconCode);
    }

    return n32Return;
}


/*****************************************************************************
                             FRIEND FUNCTIONS
*****************************************************************************/

/*****************************************************************************
*
*   WEATHER_EVENT_hCreate
*
*****************************************************************************/
WEATHER_EVENT_OBJECT WEATHER_EVENT_hCreate(
    SMS_OBJECT hParent,
    BOOLEAN bAvailable,
    WEATHER_EVENT_UNPARSED_DATA tData
            )
{
    WEATHER_EVENT_OBJECT_STRUCT *psObj =
        (WEATHER_EVENT_OBJECT_STRUCT *)WEATHER_EVENT_INVALID_OBJECT;

    // Create an instance of the object
    psObj = (WEATHER_EVENT_OBJECT_STRUCT *)
        SMSO_hCreate(
            WEATHER_EVENT_OBJECT_NAME,
            sizeof(WEATHER_EVENT_OBJECT_STRUCT),
            hParent,
            FALSE
                );

    if(psObj == NULL)
    {
        // Error!
        return WEATHER_EVENT_INVALID_OBJECT;
    }

    psObj->bAvailable = bAvailable;

    // Initialize object per inputs
    if (bAvailable == TRUE)
    {
        psObj->bAvailable = GsWeatherIntf.bProcessWeatherEventData(tData, &psObj->sData);
    }

    return (WEATHER_EVENT_OBJECT)psObj;
}


/*****************************************************************************
*
*   WEATHER_EVENT_vDestroy
*
*****************************************************************************/
void WEATHER_EVENT_vDestroy (
    WEATHER_EVENT_OBJECT hWeatherEvent
        )
{
    BOOLEAN bOwner;
    WEATHER_EVENT_OBJECT_STRUCT *psObj = (WEATHER_EVENT_OBJECT_STRUCT *)hWeatherEvent;

    bOwner = SMSO_bOwner((SMS_OBJECT) hWeatherEvent);
    if(bOwner == TRUE)
    {
        if (psObj->sData.hDescription != STRING_INVALID_OBJECT)
        {
            STRING_vDestroy(psObj->sData.hDescription);
        }

        // Free object instance
        SMSO_vDestroy((SMS_OBJECT)hWeatherEvent);
    }
    return;
}


/*****************************************************************************
                             PRIVATE FUNCTIONS
*****************************************************************************/
