/******************************************************************************/
/*                    Copyright (c) Sirius XM Radio, Inc.                     */
/*                            All Rights Reserved                             */
/*         Licensed Materials - Property of Sirius XM Radio, Inc.             */
/******************************************************************************/
/*******************************************************************************
 *
 * DESCRIPTION
 *
 *  This module contains the Object:TW_NOW implementation for the
 *  Sirius Module Services (SMS)
 *
 ******************************************************************************/

#include <string.h>

#include "standard.h"
#include "osal.h"

#include "sms_version.h"
#include "sms_api.h"
#include "sms_obj.h"
#include "sms_event.h"
#include "decoder_obj.h"
#include "report_obj.h"
#include "module_obj.h"
#include "module_version_obj.h"

#include "_tw_now_obj.h"
#include "tw_now_obj.h"

#include "sms_api_debug.h"
static const char *gpacThisFile = __FILE__;

/*****************************************************************************
                             PUBLIC FUNCTIONS
*****************************************************************************/

/*****************************************************************************
*
*   eStart
*
*****************************************************************************/
static SMSAPI_RETURN_CODE_ENUM eStart (
    DECODER_OBJECT hDecoder,
    TW_NOW_EVENT_CALLBACK vEventCallback,
    void *pvEventCallbackArg
        )
{
    BOOLEAN bLocked = FALSE;
    SMSAPI_RETURN_CODE_ENUM eReturnCode = SMSAPI_RETURN_CODE_ERROR;

    do
    {
        MODULE_OBJECT hModule;
        BOOLEAN bSupported;
        UN32 un32ProtocolVer = 0;
        UN8 un8MaxTWNow = 0;
        TRAFFIC_MARKET tMarketID;
        TW_NOW_OBJECT hTWNow;

        // Callback must be provided
        if (vEventCallback == NULL)
        {
            eReturnCode = SMSAPI_RETURN_CODE_INVALID_INPUT;
            break;
        }

        // Verify and lock object
        bLocked = SMSO_bLock((SMS_OBJECT)hDecoder, OSAL_OBJ_TIMEOUT_INFINITE);
        if (bLocked == FALSE)
        {
            eReturnCode = SMSAPI_RETURN_CODE_INVALID_INPUT;
            break;
        }

        // Get TW Now service handle from DECODER
        hTWNow = DECODER_hGetTWNowHandle(hDecoder);
        if (hTWNow != TW_NOW_INVALID_OBJECT)
        {
            // Service is already started
            eReturnCode = SMSAPI_RETURN_CODE_SERVICE_ALREADY_STARTED;
            break;
        }

        // Get the module this decoder is attached to
        hModule = DECODER_hModule(hDecoder);

        // Verify IR is supported
        bSupported = MODULE_bIsIRSupported(hModule);
        if (bSupported == FALSE)
        {
            puts(TW_NOW_OBJECT_NAME
                ": IR is not supported by the Module");
            eReturnCode = SMSAPI_RETURN_CODE_UNSUPPORTED_API;
            break;
        }

        // Verify SXi 3.0 features are supported
        bSupported =
            MODULE_VERSION_bGetProtocolVersion(hModule, &un32ProtocolVer);
        if (bSupported != FALSE)
        {
            if (un32ProtocolVer < TW_NOW_PROTOCOL_SUPPORT_MIN_VERSION)
            {
                puts(TW_NOW_OBJECT_NAME
                    ": SXi 3.x features are not supported by the Module");
                eReturnCode = SMSAPI_RETURN_CODE_UNSUPPORTED_API;
                break;
            }
        }
        else
        {
            puts(TW_NOW_OBJECT_NAME
                ": Module version info is not available");
            break;
        }

        bSupported = MODULE_VERSION_bMaxTWNow(hModule, &un8MaxTWNow);
        if (bSupported != FALSE)
        {
            if (un8MaxTWNow == 0)
            {
                puts(TW_NOW_OBJECT_NAME
                    ": TWNow feature is not supported by the Module");
                eReturnCode = SMSAPI_RETURN_CODE_UNSUPPORTED_API;
                break;
            }
        }
        else
        {
            puts(TW_NOW_OBJECT_NAME
                ": MaxTWNow parameter is not available");
            break;
        }

        // Try to create the service
        eReturnCode = eCreateTWNow(
            hDecoder,
            &tMarketID,
            vEventCallback,
            pvEventCallbackArg
                );

        // Relinquish ownership
        SMSO_vUnlock((SMS_OBJECT)hDecoder);
        bLocked = FALSE;

        if ((eReturnCode == SMSAPI_RETURN_CODE_SUCCESS) &&
            (tMarketID != TRAFFIC_INVALID_MARKET))
        {
            SMS_EVENT_TW_NOW_STRUCT sEvent;
            BOOLEAN bPosted;

            sEvent.eAction = TW_NOW_ACTION_INITIALIZE_MARKET_ID;
            sEvent.uData.tMarketID = tMarketID;

            // We have loaded some market - pass it to the module
            bPosted = DECODER_bTWNow(hDecoder, &sEvent);
            if (bPosted == FALSE)
            {
                SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    TW_NOW_OBJECT_NAME
                    ": Could not post event to set market loaded from config.");
            }
        }
    } while (FALSE);

    if (bLocked == TRUE)
    {
        // Relinquish ownership
        SMSO_vUnlock((SMS_OBJECT)hDecoder);
    }

    return eReturnCode;
}

/*****************************************************************************
*
*   vStop
*
*****************************************************************************/
static void vStop (
    DECODER_OBJECT hDecoder
        )
{
    SMS_EVENT_TW_NOW_STRUCT sEvent;

    sEvent.eAction = TW_NOW_ACTION_STOP_EVENT;
    DECODER_bTWNow(hDecoder, &sEvent);

    return;
}

/*****************************************************************************
*
*   eSetMarketID
*
*****************************************************************************/
static SMSAPI_RETURN_CODE_ENUM eSetMarketID (
    DECODER_OBJECT hDecoder,
    CID_OBJECT hMarketID
        )
{
    SMSAPI_RETURN_CODE_ENUM eReturnCode = SMSAPI_RETURN_CODE_INVALID_INPUT;

    do
    {
        SMS_EVENT_TW_NOW_STRUCT sEvent;
        BOOLEAN bSuccess;

        sEvent.eAction = TW_NOW_ACTION_SET_MARKET_ID;

        // Check if the provided Market ID is valid and registered
        if (hMarketID != CID_INVALID_OBJECT)
        {
            TW_NOW_MARKET_ITERATOR_STRUCT sIterator;
            UN32 un32ID = (UN32)TRAFFIC_INVALID_MARKET;
            void *pvId = &un32ID;

            sIterator.bFound = FALSE;
            sIterator.hMarketID = hMarketID;

            (void) REPORT.bIterateContent(
                        bMarketIterator,
                        &sIterator);
            if (sIterator.bFound == FALSE)
            {
                // Unknown Market ID
                break;
            }
            CID_n32GetValue(hMarketID, &pvId);
            sEvent.uData.tMarketID = (TRAFFIC_MARKET)un32ID;
        }
        else
        {
            sEvent.uData.tMarketID = TRAFFIC_INVALID_MARKET;
        }

        // Post the event to decoder
        bSuccess = DECODER_bTWNow(hDecoder, &sEvent);
        if (bSuccess != FALSE)
        {
            eReturnCode = SMSAPI_RETURN_CODE_SUCCESS;
        }
    } while (FALSE);

    return eReturnCode;
}

/*****************************************************************************
*
*   hGetMarketID
*
*****************************************************************************/
static CID_OBJECT hGetMarketID (
    DECODER_OBJECT hDecoder
        )
{
    BOOLEAN bLocked;
    CID_OBJECT hMarketID = CID_INVALID_OBJECT;

    // Verify and lock object
    bLocked = SMSO_bLock((SMS_OBJECT)hDecoder, OSAL_OBJ_TIMEOUT_INFINITE);
    if (bLocked == TRUE)
    {
        TW_NOW_OBJECT_STRUCT *psObj = NULL;

        psObj = (TW_NOW_OBJECT_STRUCT*)DECODER_hGetTWNowHandle(hDecoder);
        if (((TW_NOW_OBJECT)psObj != TW_NOW_INVALID_OBJECT) &&
             (psObj->hMarketID != CID_INVALID_OBJECT))
        {
            // Duplicate the stored value and give it to the caller
            hMarketID = CID.hDuplicate(psObj->hMarketID);
        }

        SMSO_vUnlock((SMS_OBJECT)hDecoder);
    }

    return hMarketID;
}

/*****************************************************************************
*
*   ePlayBulletinAudio
*
*****************************************************************************/
static SMSAPI_RETURN_CODE_ENUM ePlayBulletinAudio (
    DECODER_OBJECT hDecoder,
    TW_NOW_BULLETIN_ID tBulletinID
        )
{
    BOOLEAN bSuccess;
    SMS_EVENT_TW_NOW_STRUCT sEvent;

    if (tBulletinID == TW_NOW_INVALID_BULLETIN_ID)
    {
        return SMSAPI_RETURN_CODE_INVALID_INPUT;
    }

    sEvent.eAction = TW_NOW_ACTION_PLAY_BULLETIN;
    sEvent.uData.tBulletinID = tBulletinID;

    bSuccess = DECODER_bTWNow(hDecoder, &sEvent);
    if (bSuccess != FALSE)
    {
        return SMSAPI_RETURN_CODE_SUCCESS;
    }

    return SMSAPI_RETURN_CODE_ERROR;
}

/*****************************************************************************
*
*   eAbortBulletinPlayback
*
*****************************************************************************/
static SMSAPI_RETURN_CODE_ENUM eAbortBulletinPlayback (
    DECODER_OBJECT hDecoder
        )
{
    BOOLEAN bSuccess;
    SMS_EVENT_TW_NOW_STRUCT sEvent;

    sEvent.eAction = TW_NOW_ACTION_ABORT_BULLETIN;

    bSuccess = DECODER_bTWNow(hDecoder, &sEvent);
    if (bSuccess != FALSE)
    {
        return SMSAPI_RETURN_CODE_SUCCESS;
    }

    return SMSAPI_RETURN_CODE_ERROR;
}

/*****************************************************************************
*
*   tCurrentBulletinID
*
*****************************************************************************/
static TW_NOW_BULLETIN_ID tCurrentBulletinID (
    DECODER_OBJECT hDecoder
        )
{
    TW_NOW_BULLETIN_ID tBulletinId = TW_NOW_INVALID_BULLETIN_ID;
    TW_NOW_OBJECT hTWNow;

    hTWNow = DECODER_hGetTWNowHandle(hDecoder);
    if (hTWNow != TW_NOW_INVALID_OBJECT)
    {
        TW_NOW_OBJECT_STRUCT *psObj =
            (TW_NOW_OBJECT_STRUCT *)hTWNow;

        tBulletinId = psObj->tCurrentBulletinID;
    }

    return tBulletinId;
}

/*****************************************************************************
 FRIEND FUNCTIONS
 *****************************************************************************/

/*****************************************************************************
 *
 *   TW_NOW_bDecoderEventHandler
 *
 *****************************************************************************/
BOOLEAN TW_NOW_bDecoderEventHandler (
    DECODER_OBJECT hDecoder,
    TW_NOW_OBJECT hTWNow,
    const SMS_EVENT_TW_NOW_STRUCT *psEventData
        )
{
    TW_NOW_OBJECT_STRUCT *psObj = (TW_NOW_OBJECT_STRUCT *)hTWNow;
    BOOLEAN bReturn = FALSE, bOwner, bValid;

    // Verify ownership
    bOwner = SMSO_bOwner((SMS_OBJECT)hDecoder);
    if (bOwner == FALSE)
    {
        // Object is not owned
        return FALSE;
    }

    bValid = SMSO_bValid((SMS_OBJECT)hTWNow);
    if (bValid == FALSE)
    {
        // Object is not valid
        return FALSE;
    }

    // Process event...
    switch (psEventData->eAction)
    {
        case TW_NOW_ACTION_INITIALIZE_MARKET_ID:
        {
            const UN16 un16BulletinParam1 = (UN16)psEventData->uData.tMarketID;

            puts(TW_NOW_OBJECT_NAME": Initialize Market ID event");

            bReturn = RADIO_bBulletinMonitor(
                psObj->hDecoder,
                BULLETIN_MONITOR_TYPE_TW_NOW,
                ((psEventData->uData.tMarketID != TRAFFIC_INVALID_MARKET) ? 1 : 0),
                &un16BulletinParam1
                    );
        }
        break;

        case TW_NOW_ACTION_SET_MARKET_ID:
        {
            puts(TW_NOW_OBJECT_NAME": Set Market ID event");

            bReturn = bHandleSetMarketIDEvent(psObj,
                            psEventData->uData.tMarketID);
        }
        break;

        case TW_NOW_ACTION_PLAY_BULLETIN:
        {
            puts(TW_NOW_OBJECT_NAME": Play Bulletin event");

            if (psObj->tAvailableBulletinID == psEventData->uData.tBulletinID)
            {
                bReturn = DECODER_bTWNowPlayAbort(
                            psObj->hDecoder,
                            psEventData->uData.tBulletinID,
                            FALSE
                                );
            }
        }
        break;

        case TW_NOW_ACTION_ABORT_BULLETIN:
        {
            puts(TW_NOW_OBJECT_NAME": Abort Bulletin event");

            if (psObj->tCurrentBulletinID != TW_NOW_INVALID_BULLETIN_ID)
            {
                bReturn = DECODER_bTWNowPlayAbort(
                            psObj->hDecoder,
                            psObj->tCurrentBulletinID,
                            TRUE
                                );
            }
        }
        break;

        case TW_NOW_ACTION_STOP_EVENT:
        {
            puts(TW_NOW_OBJECT_NAME": Stop service");

            bReturn = bHandleStopEvent(psObj);
        }
        break;
    }

    return bReturn;
}

/*****************************************************************************
 *
 *   TW_NOW_bHandleBulletinEvent
 *
 *****************************************************************************/
BOOLEAN TW_NOW_bHandleBulletinEvent (
    DECODER_OBJECT hDecoder,
    TW_NOW_BULLETIN_ID tBulletinID,
    TW_NOW_BULLETIN_STATUS_ENUM eBulletinStatus,
    TRAFFIC_MARKET tMarketID
        )
{
    BOOLEAN bSuccess = FALSE;

    do
    {
        TW_NOW_OBJECT_STRUCT *psObj;

        psObj = (TW_NOW_OBJECT_STRUCT*)DECODER_hGetTWNowHandle(hDecoder);
        if ((TW_NOW_OBJECT)psObj == TW_NOW_INVALID_OBJECT)
        {
            // Could not get TW Now object which is possibly because
            // the service has been just stopped and it is ok.
            bSuccess = TRUE;
            break;
        }

        if (eBulletinStatus != TW_NOW_NO_BULLETIN_AVAILABLE)
        {
            UN32 un32ID = UN32_MAX;
            void *pvId = &un32ID;

            if (psObj->hMarketID == CID_INVALID_OBJECT)
            {
                // Wrong event, no market is currently active
                break;
            }

            CID_n32GetValue(psObj->hMarketID, &pvId);

            if (un32ID != (UN32)tMarketID)
            {
                // Wrong event, another market is currently active
                break;
            }

            psObj->tAvailableBulletinID = tBulletinID;
        }
        else
        {
            // Clear the available bulletin id
            psObj->tAvailableBulletinID = TW_NOW_INVALID_BULLETIN_ID;
        }

        // Notify the application
        psObj->vEventCallback(
            psObj->hDecoder,
            tBulletinID,
            eBulletinStatus,
            psObj->hMarketID,
            psObj->pvEventCallbackArg
                );

        bSuccess = TRUE;
    } while (FALSE);

    return bSuccess;
}

/*****************************************************************************
 *
 *   TW_NOW_vUpdateStatus
 *
 *****************************************************************************/
void TW_NOW_vUpdateStatus(
    TW_NOW_OBJECT hTWNow,
    TW_NOW_BULLETIN_ID tBulletinId,
    BOOLEAN *pbIsActive,
    BOOLEAN *pbChanged
        )
{
    if (hTWNow != TW_NOW_INVALID_OBJECT)
    {
        TW_NOW_OBJECT_STRUCT *psObj = (TW_NOW_OBJECT_STRUCT*)hTWNow;
        if (psObj->tCurrentBulletinID != tBulletinId)
        {
            if (pbChanged != NULL)
            {
                *pbChanged = TRUE;
            }

            psObj->tCurrentBulletinID = tBulletinId;
        }

        if (pbIsActive != NULL)
        {
            if (psObj->tCurrentBulletinID != TW_NOW_INVALID_BULLETIN_ID)
            {
                *pbIsActive = TRUE;
            }
            else
            {
                *pbIsActive = FALSE;
            }
        }
    }

    return;
}

/*****************************************************************************
 *
 *   TW_NOW_vStop
 *
 *****************************************************************************/
void TW_NOW_vStop (
    TW_NOW_OBJECT hTWNow
        )
{
    if (hTWNow != TW_NOW_INVALID_OBJECT)
    {
        bHandleStopEvent((TW_NOW_OBJECT_STRUCT *)hTWNow);
    }

    return;
}

/*****************************************************************************
 PRIVATE FUNCTIONS
 *****************************************************************************/

/*****************************************************************************
 *
 *   bHandleSetMarketIDEvent
 *
 *****************************************************************************/
static BOOLEAN bHandleSetMarketIDEvent (
    TW_NOW_OBJECT_STRUCT *psObj,
    TRAFFIC_MARKET tMarketID
        )
{
    BOOLEAN bSuccess = FALSE;

    do
    {
        const UN16 un16BulletinParam1 = (UN16) tMarketID;

        // Check if we already have a market
        if (psObj->hMarketID != CID_INVALID_OBJECT)
        {
            if (tMarketID  != TRAFFIC_INVALID_MARKET)
            {
                UN32 un32ID = (UN32)TRAFFIC_INVALID_MARKET;
                void *pvId = &un32ID;

                CID_n32GetValue(psObj->hMarketID, &pvId);
                if (tMarketID == (TRAFFIC_MARKET)un32ID)
                {
                    // The same market is already set, no action required
                    bSuccess = TRUE;
                    break;
                }
            }

            // Delete previous Market
            CID_vDestroy(psObj->hMarketID);
            psObj->hMarketID = CID_INVALID_OBJECT;
        }
        else if (tMarketID == TRAFFIC_INVALID_MARKET)
        {
            // Empty market is already set, no action required
            bSuccess = TRUE;
            break;
        }

        // We are setting new market so clear the available bulletin
        psObj->tAvailableBulletinID = TW_NOW_INVALID_BULLETIN_ID;

        // Set the new market
        if (tMarketID != TRAFFIC_INVALID_MARKET)
        {
            UN32 un32ID = tMarketID;

            psObj->hMarketID = CID_hCreate(CID_POOL_INVALID_OBJECT,
                CID_SXI_MARKET_ID, &un32ID);
            if (psObj->hMarketID == CID_INVALID_OBJECT)
            {
                SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    TW_NOW_OBJECT_NAME": failed to create CID");

                break;
            }
        }

        if (psObj->hTag != TAG_INVALID_OBJECT)
        {
            bSuccess = bSaveTWNow(psObj);
            if (bSuccess == FALSE)
            {
                SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    TW_NOW_OBJECT_NAME
                    ": Could not save TW Now Market ID to config file.");
            }
        }

        bSuccess = RADIO_bBulletinMonitor(
            psObj->hDecoder,
            BULLETIN_MONITOR_TYPE_TW_NOW,
            ((tMarketID != TRAFFIC_INVALID_MARKET) ? 1 : 0),
            &un16BulletinParam1
                );

    } while(FALSE);

    return bSuccess;
}

/*****************************************************************************
*
*   bHandleStopEvent
*
*****************************************************************************/
static BOOLEAN bHandleStopEvent(
    TW_NOW_OBJECT_STRUCT *psObj
        )
{
    BOOLEAN bSuccess = FALSE;

    // Clear the TW Now service handle for this decoder
    DECODER_bSetTWNowHandle(
        psObj->hDecoder,
        TW_NOW_INVALID_OBJECT
            );

    // Abort the currently playing bulletin if exists
    if (psObj->tCurrentBulletinID != TW_NOW_INVALID_BULLETIN_ID)
    {
        bSuccess = DECODER_bTWNowPlayAbort(
                    psObj->hDecoder,
                    psObj->tCurrentBulletinID,
                    TRUE
                        );
        if (bSuccess == FALSE)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                TW_NOW_OBJECT_NAME": Failed to abort bulletin playback");
        }
    }

    // Cancel the bulletin monitor if set
    if (psObj->hMarketID != CID_INVALID_OBJECT)
    {
        bSuccess = RADIO_bBulletinMonitor(
            psObj->hDecoder,
            BULLETIN_MONITOR_TYPE_TW_NOW,
            0,
            NULL
                );
        if (bSuccess == FALSE)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                TW_NOW_OBJECT_NAME": Failed to cancel bulletin monitor");
        }
    }

    // Destroy sub-objects
    vDestroyTWNow(psObj);

    // Destroy main object
    SMSO_vDestroy((SMS_OBJECT)psObj);

    return bSuccess;
}

/*****************************************************************************
 *
 *   eCreateTWNow
 *
 *****************************************************************************/
static SMSAPI_RETURN_CODE_ENUM eCreateTWNow (
    DECODER_OBJECT hDecoder,
    TRAFFIC_MARKET *ptMarketID,
    TW_NOW_EVENT_CALLBACK vEventCallback,
    void *pvEventCallbackArg
        )
{
    SMSAPI_RETURN_CODE_ENUM eReturnCode = SMSAPI_RETURN_CODE_ERROR;
    TW_NOW_OBJECT_STRUCT *psObj = NULL;

    do
    {
        BOOLEAN bOk, bNeedUpdate;

        // Verify ownership
        bOk = SMSO_bOwner((SMS_OBJECT)hDecoder);
        if (bOk == FALSE)
        {
            eReturnCode = SMSAPI_RETURN_CODE_NOT_OWNER;
            break;
        }

        // Create new object
        psObj = (TW_NOW_OBJECT_STRUCT *)SMSO_hCreate(
            TW_NOW_OBJECT_NAME,
            sizeof(TW_NOW_OBJECT_STRUCT),
            (SMS_OBJECT)hDecoder, // Decoder is parent
            FALSE // Not lockable
                );

        // Not enough memory
        if (psObj == NULL)
        {
            eReturnCode = SMSAPI_RETURN_CODE_OUT_OF_MEMORY;

            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                TW_NOW_OBJECT_NAME
                ": Not enough memory to create TW Now object");
            break;
        }

        // Initialize object
        *psObj = gsObjectDefaults;
        psObj->vEventCallback = vEventCallback;
        psObj->pvEventCallbackArg = pvEventCallbackArg;

        // Set decoder object handle into TW Now object struct
        psObj->hDecoder = hDecoder;

        // Get TW Now Tag
        bOk = bCreateTag(psObj);
        if (bOk == FALSE)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                TW_NOW_OBJECT_NAME
                ": Failed to get TW Now tag object");
            break;
        }

        // Restore TW Now configuration from SMS cfg
        bNeedUpdate = bRestoreTWNow(psObj);
        if (bNeedUpdate == FALSE)
        {
            puts(TW_NOW_OBJECT_NAME
                ": Failed to restore TW Now configuration");

            if (ptMarketID != NULL)
            {
                *ptMarketID = TRAFFIC_INVALID_MARKET;
            }
        }
        else if (ptMarketID != NULL)
        {
            UN32 un32ID = TRAFFIC_INVALID_MARKET;
            void *pvID = &un32ID;

            CID_n32GetValue(psObj->hMarketID, &pvID);
            *ptMarketID = (TRAFFIC_MARKET)un32ID;
        }

        // Set TW Now handle into Decoder
        bOk = DECODER_bSetTWNowHandle(hDecoder, (TW_NOW_OBJECT)psObj);
        if (bOk == FALSE)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                TW_NOW_OBJECT_NAME
                ": Failed to set TW Now handle");
            break;
        }

        // Seems to be OK, if we can go so far ...
        eReturnCode = SMSAPI_RETURN_CODE_SUCCESS;

    } while (FALSE);

    if ((eReturnCode != SMSAPI_RETURN_CODE_SUCCESS) &&
        (psObj != NULL))
    {
        // Dissociate from decoder
        DECODER_bSetTWNowHandle(
            hDecoder,
            TW_NOW_INVALID_OBJECT
                );

        // Destroy created objects
        vDestroyTWNow(psObj);

        // and destroy the main object
        SMSO_vDestroy((SMS_OBJECT)psObj);
    }

    return eReturnCode;
}

/*****************************************************************************
 *
 *   vDestroyTWNow
 *
 *****************************************************************************/
static void vDestroyTWNow (
    TW_NOW_OBJECT_STRUCT *psObj
        )
{
    // Zero the events handlers
    psObj->vEventCallback = NULL;
    psObj->pvEventCallbackArg = NULL;

    // Destroy the CID copy
    if (psObj->hMarketID != CID_INVALID_OBJECT)
    {
        CID_vDestroy(psObj->hMarketID);
        psObj->hMarketID = CID_INVALID_OBJECT;
    }

    return;
}

/*****************************************************************************
 *
 *   bCreateTag
 *
 *****************************************************************************/
static BOOLEAN bCreateTag (
    TW_NOW_OBJECT_STRUCT *psObj
        )
{
    TAG_OBJECT hParentTag;
    BOOLEAN bSuccess = FALSE;

    // Get the decoder tag
    hParentTag = DECODER_hGetTag(psObj->hDecoder);
    if (hParentTag != TAG_INVALID_OBJECT)
    {
        TAG_OBJECT hTag;
        SMSAPI_RETURN_CODE_ENUM eReturnCode;

        // Get TW Now Tag
        eReturnCode =
            TAG_eGet(TW_NOW_OBJECT_NAME, hParentTag, &hTag, NULL, TRUE);
        if (eReturnCode == SMSAPI_RETURN_CODE_SUCCESS)
        {
            // Save TW Now Tag Locally
            psObj->hTag = hTag;
            bSuccess = TRUE;
        }
    }

    return bSuccess;
}

/*****************************************************************************
 *
 *   bRestoreTWNow
 *
 *****************************************************************************/
static BOOLEAN bRestoreTWNow (
    TW_NOW_OBJECT_STRUCT *psObj
        )
{
    BOOLEAN bSuccess = FALSE;
    STRING_OBJECT hCIDString = STRING_INVALID_OBJECT;
    STRING_OBJECT *phString;
    size_t tSize;
    const char *pacBuff = NULL;

    phString = &hCIDString;
    tSize = sizeof(STRING_OBJECT);
    TAG_eGetTagValue(
        psObj->hTag,
        TAG_TYPE_STRING,
        (void**)&phString,
        &tSize);

    // check if we got a valid string handle. a valid string implies
    // that the TAG_eGetValue returned success and that the tag has a
    // value so we don't need to actually check the result in this case
    if (hCIDString != STRING_INVALID_OBJECT)
    {
        tSize = STRING.tSize(hCIDString);
        if (tSize > 0) // We have got something in the string
        {
            CID_OBJECT hId;
            // Now use that string
            pacBuff = STRING.pacCStr(hCIDString);
            // Read from memory assumes memory provided as an input is
            // NULL terminated.
            hId = CID_hReadFromMemory((const void **)&pacBuff);
            if (hId != CID_INVALID_OBJECT)
            {
                // Save the CID as new one
                psObj->hMarketID = hId;
                bSuccess = TRUE;
            }
        }

        STRING.vDestroy(hCIDString);
    }

    return bSuccess;
}

/*****************************************************************************
 *
 *   bSaveTWNow
 *
 *****************************************************************************/
static BOOLEAN bSaveTWNow (
    TW_NOW_OBJECT_STRUCT *psObj
        )
{
    SMSAPI_RETURN_CODE_ENUM eReturnCode;
    BOOLEAN bSuccess = FALSE;
    STRING_OBJECT hCIDString = STRING_INVALID_OBJECT;
    char *pacBuff = NULL;

    do
    {
        if (psObj->hMarketID != CID_INVALID_OBJECT)
        {
            size_t tSize;
            N32 n32Result;
            void *pvMem;

            // get the size of this cid
            tSize = CID_tSize(psObj->hMarketID) + 1; // add one for NULL

            // allocate memory for the buffer
            pacBuff = (char *) OSAL.pvMemoryAllocate(
                                    TW_NOW_OBJECT_NAME":CSTR",
                                    tSize, FALSE);
            if (pacBuff == NULL)
            {
                SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    TW_NOW_OBJECT_NAME
                    ": Failed to allocate memory for buffer.");
                break;
            }

            pvMem = pacBuff;
            n32Result =
                CID_n32FWriteToMemory(psObj->hMarketID, (void **)&pvMem);
            if (n32Result == EOF)
            {
                SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    TW_NOW_OBJECT_NAME
                    ": Failed to serialize CID");
                break;
            }

            // create a STRING from the CID info
            hCIDString = STRING.hCreate(pacBuff, tSize);
            if (hCIDString == STRING_INVALID_OBJECT)
            {
                SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    TW_NOW_OBJECT_NAME
                    ": Failed to create string");
                break;
            }
        }

        eReturnCode = TAG_eSetTagValue(
                                psObj->hTag,
                                TAG_TYPE_STRING,
                                &hCIDString,
                                sizeof(STRING_OBJECT),
                                TRUE);
        if (eReturnCode != SMSAPI_RETURN_CODE_SUCCESS)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                TW_NOW_OBJECT_NAME
                ": Failed to clear tag value (%s)",
                SMSAPI_DEBUG_pacReturnCodeText(eReturnCode));
            break;
        }

        bSuccess = TRUE;
    } while (FALSE);

    if (hCIDString != STRING_INVALID_OBJECT)
    {
        STRING.vDestroy(hCIDString);
    }

    if (pacBuff != NULL)
    {
        OSAL.vMemoryFree((void*)pacBuff);
    }

    return bSuccess;
}

/*****************************************************************************
 *
 *   bMarketIterator
 *
 *****************************************************************************/
static BOOLEAN bMarketIterator(
    const char *pacId,
    const char *pacName,
    CID_OBJECT hId,
    void *pvContentIteratorCallbackArg
        )
{
    TW_NOW_MARKET_ITERATOR_STRUCT *psIterator =
        (TW_NOW_MARKET_ITERATOR_STRUCT *)pvContentIteratorCallbackArg;
    BOOLEAN bResult = TRUE;

    if (hId != CID_INVALID_OBJECT)
    {
        N16 n16Equal = CID.n16Equal(hId, psIterator->hMarketID);
        if (n16Equal == 0)
        {
            // Market found, stop iteration
            psIterator->bFound = TRUE;
            bResult = FALSE;
        }
    }

    return bResult;
}
