/*************************************************************************/
/*               Copyright (c) Sirius XM Radio, Inc.                     */
/*                       All Rights Reserved                             */
/*     Licensed Materials - Property of Sirius XM Radio, Inc.            */
/*                    Proprietary & Confidential                         */
/*************************************************************************/
/**************************************************************************
 *
 * DESCRIPTION
 *
 *  This module contains the Object:TUNEMIX implementation for the
 *  Satellite Module Services (SMS)
 *
 *************************************************************************/

#include "standard.h"
#include "osal.h"

#include "sms_api.h"
#include "module_obj.h"
#include "decoder_obj.h"
#include "category_obj.h"
#include "channel_obj.h"
#include "cm.h"
#include "string_obj.h"

#include "sms_obj.h"

#include "tunemix_obj.h"
#include "_tunemix_obj.h"

#include "sms_api_debug.h"

static const char *gpacThisFile = __FILE__;

/*************************************************************************
 *                             PUBLIC FUNCTIONS
 *************************************************************************/
/*************************************************************************
 *
 *   hCreate
 *
 *************************************************************************/
static TUNEMIX_OBJECT hCreate (
    DECODER_OBJECT hDecoder,
    BOOLEAN bOverride,
    BOOLEAN bPlayUnrestricted
        )
{
    BOOLEAN bOk;
    TUNEMIX_OBJECT hTuneMix = TUNEMIX_INVALID_OBJECT;
    UN8 un8Index;
    CHANNEL_ID tChannelId;
    TUNEMIX_OBJECT_STRUCT *psObj = TUNEMIX_INVALID_OBJECT_STRUCT_PTR;

    bOk = SMSO_bLock((SMS_OBJECT)hDecoder, OSAL_OBJ_TIMEOUT_INFINITE);
    if (bOk == FALSE)
    {
        return TUNEMIX_INVALID_OBJECT;
    }

    do
    {
        char acTuneMixName[10];  // 7 characters "TuneMix" 
                                 // + 2 for number
                                 // + 1 for NULL
        size_t tNameSize;

        // Check feature support by the Module
        bOk = DECODER_bIsTuneMixSupported(hDecoder);
        if (bOk == FALSE)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                TUNEMIX_OBJECT_NAME
                ":TuneMIx is not supported by the Module");
            break;
        }

        //To get next available index in the decoder array
        bOk = DECODER_bGetTuneMixIndexAvailable(hDecoder, 
            &un8Index, &tChannelId);
        if (bOk == FALSE)
        {
            puts( TUNEMIX_OBJECT_NAME
                ": There are no any available TuneMix channels.");
            break;
        }

        // Create the instance
        psObj = psCreateObject(hDecoder, tChannelId);
        if (psObj == NULL)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                TUNEMIX_OBJECT_NAME": failed to create the object");
            bOk = FALSE;
            break;
        }

        // To fill the TuneMix structure
        psObj->hDecoder = hDecoder;
        psObj->eStatus = TUNEMIX_STATUS_UNDEFINED;
        psObj->un8Index = un8Index;
        psObj->tChannelID = tChannelId;
        tNameSize = sprintf(acTuneMixName,"TuneMix%02d", un8Index);
        psObj->hName = STRING_hCreate( SMS_INVALID_OBJECT, 
            acTuneMixName, tNameSize, tNameSize );
        psObj->bOverride = bOverride;
        psObj->bPlayUnrestricted = bPlayUnrestricted;

        // Create tag for this TuneMix
        bOk = bCreateTuneMixTag( psObj );
        if (bOk == FALSE)
        {
            SMSAPI_DEBUG_vPrintErrorFull( gpacThisFile, __LINE__,
                TUNEMIX_OBJECT_NAME
                ": Failed to create Tune Mix tag object" );
            break;
        }

        // Populate the object
        hTuneMix = (TUNEMIX_OBJECT)psObj;

        // Put into the decoder's array of TuneMix handlers
        DECODER_vSetTuneMix(hDecoder, un8Index, hTuneMix);

    } while (FALSE);

    if ((bOk == FALSE) && (psObj != TUNEMIX_INVALID_OBJECT_STRUCT_PTR))
    {
        // Uninit in case of error
        vUninitObject(psObj);
    }

    // Relinquish ownership
    SMSO_vUnlock((SMS_OBJECT)hDecoder);
    return hTuneMix;
}

/*************************************************************************
 *
 *   vDestroy
 *
 *************************************************************************/
static void vDestroy (
    TUNEMIX_OBJECT hTuneMix
        )
{
    BOOLEAN bOk;
    TUNEMIX_OBJECT_STRUCT *psObj = (TUNEMIX_OBJECT_STRUCT *)hTuneMix;

    bOk = SMSO_bValid((SMS_OBJECT)hTuneMix);
    if(bOk == TRUE)
    {
        DECODER_OBJECT hDecoder = psObj->hDecoder;
        SMSAPI_RETURN_CODE_ENUM eReturnCode;
        SERVICE_ID tSID = (SERVICE_ID)psObj->tChannelID;

        // Remove handle from the list of TuneMixes
        DECODER_vSetTuneMix(hDecoder, psObj->un8Index,
            TUNEMIX_INVALID_OBJECT);

        // Send configure command to clean the attributes
        eReturnCode = DECODER_eTuneMixConfigure(
                hDecoder,
                psObj->un8Index );

        if (SMSAPI_RETURN_CODE_SUCCESS == eReturnCode)
        {
            // Lock decoder for TuneMix uninitialization
            bOk = SMSO_bLock((SMS_OBJECT)hDecoder, OSAL_OBJ_TIMEOUT_INFINITE);
            if (TRUE == bOk)
            {
                PRESETS_OBJECT hPresets;
                TUNEMIX_PRESETS_ITERATOR_STRUCT sIterator;

                // Free up resources
                vUninitObject(psObj);

                // Getting PRESETS object and removing TuneMix ID from 
                // all presets
                hPresets = DECODER_hPresets(hDecoder);

                sIterator.tSID = tSID;
                sIterator.tLastBandIndex = 0;
                sIterator.tLastPresetIndex = 0;

                eReturnCode = PRESETS.eIterate(hPresets, 
                    n16TuneMixRemoveIterator, 
                    (void*)&sIterator);

                // Release DECODER
                SMSO_vUnlock((SMS_OBJECT)hDecoder);
            }
        }
    }

    if (bOk == FALSE)
    {
        SMSAPI_DEBUG_vPrintErrorFull( gpacThisFile, __LINE__,
            TUNEMIX_OBJECT_NAME": TuneMix failed to destroy." );
    }
    else
    {
        puts(TUNEMIX_OBJECT_NAME": TuneMix is destroyed.");
    }
    return;
}

/*************************************************************************
 *
 *   hActive
 *
 *************************************************************************/
static TUNEMIX_OBJECT hActive (
    DECODER_OBJECT hDecoder
        )
{
    BOOLEAN bOk;
    TUNEMIX_OBJECT hActiveTuneMix = TUNEMIX_INVALID_OBJECT;

    bOk = SMSO_bLock((SMS_OBJECT)hDecoder, OSAL_OBJ_TIMEOUT_INFINITE);
    if(bOk == TRUE)
    {
        hActiveTuneMix = DECODER_hGetTuneMixActive(hDecoder);
        SMSO_vUnlock((SMS_OBJECT)hDecoder);
    }
    return hActiveTuneMix;
}

/*************************************************************************
 *
 *   tChannelID
 *
 *************************************************************************/
static CHANNEL_ID tChannelID (
    TUNEMIX_OBJECT hTuneMix
        )
{
    BOOLEAN bValid;
    CHANNEL_ID tId = CHANNEL_INVALID_ID;
    TUNEMIX_OBJECT_STRUCT *psObj =
        (TUNEMIX_OBJECT_STRUCT *)hTuneMix;

    // Verify inputs
    bValid = SMSO_bValid((SMS_OBJECT)hTuneMix);
    if( bValid != FALSE)
    {
        tId = CHANNEL.tChannelId(psObj->hChannel);
    }

    return tId;
}

/*************************************************************************
 *
 *   eTune
 *
 *************************************************************************/
static SMSAPI_RETURN_CODE_ENUM eTune (
    TUNEMIX_OBJECT hTuneMix
        )
{
    SMSAPI_RETURN_CODE_ENUM eReturnCode = SMSAPI_RETURN_CODE_INVALID_INPUT;
    BOOLEAN bLocked;
    TUNEMIX_OBJECT_STRUCT *psObj = 
        (TUNEMIX_OBJECT_STRUCT *)hTuneMix;

    bLocked = SMSO_bLock((SMS_OBJECT)hTuneMix, OSAL_OBJ_TIMEOUT_INFINITE);
    if (TRUE == bLocked)
    {
        // Here we wish to validate already configured TuneMix channel.
        // If number of qualified desired components is less than
        // minimum the tuning isn't allowed.
        // Since the desired list is configured by the Module
        // this list shouldn't changed here.
        // The non-qualified channels aren't removed from list here.
        eReturnCode = TUNEMIX_eQualified(hTuneMix);

        if (eReturnCode == SMSAPI_RETURN_CODE_SUCCESS)
        {
            // Unlock the object
            SMSO_vUnlock((SMS_OBJECT)hTuneMix);

            // Sending tune request
            eReturnCode = DECODER_eTuneMixTune( psObj->hDecoder,
                psObj->tChannelID,
                psObj->bOverride, psObj->bPlayUnrestricted);
            if (eReturnCode != SMSAPI_RETURN_CODE_SUCCESS)
            {
                puts(TUNEMIX_OBJECT_NAME": Cannot send tune requiest!");
            }
        }
        else
        {
            TUNEMIX_vUpdateTuneMixStatus(
                psObj->hDecoder,
                psObj->tChannelID,
                FALSE,
                TUNEMIX_STATUS_UNAVAILABLE);

            // Unlock the object
            SMSO_vUnlock((SMS_OBJECT)hTuneMix);
        }
    }

    return eReturnCode;
}

/*************************************************************************
 *
 *   eConfigure
 *
 *************************************************************************/
static SMSAPI_RETURN_CODE_ENUM eConfigure (
    TUNEMIX_OBJECT hTuneMix,
    CHANNEL_ID *ptChannelId,
    UN16 un16Count
        )
{
    BOOLEAN bReturn;
    SMSAPI_RETURN_CODE_ENUM eReturnCode;
    SERVICE_ID atServiceId[TUNEMIX_CHANNELS_MAX];
    TUNEMIX_OBJECT_STRUCT *psObj = (TUNEMIX_OBJECT_STRUCT *)hTuneMix;
    UN8 un8Index;

    // Verify inputs
    if ((ptChannelId == NULL) && (un16Count > 0))
    {
        return SMSAPI_RETURN_CODE_INVALID_INPUT;
    }

    if ((0 < un16Count) && (un16Count < TUNEMIX_CHANNELS_MIN))
    {
        return SMSAPI_RETURN_CODE_OP_NOT_SUPPORTED;
    }

    bReturn = SMSO_bLock((SMS_OBJECT)hTuneMix, OSAL_OBJ_TIMEOUT_INFINITE);
    if (bReturn == FALSE)
    {
        return SMSAPI_RETURN_CODE_INVALID_INPUT;
    }

    // Remove all entries in the linked list before new configuration
    OSAL.eLinkedListRemoveAll( psObj->hComponentList, 
        (OSAL_LL_RELEASE_HANDLER)NULL );

    for (un8Index = 0; un8Index < TUNEMIX_CHANNELS_MAX; un8Index++)
    {
        atServiceId[un8Index] = SERVICE_INVALID_ID;
    }

    eReturnCode = eFillComponentList(hTuneMix, &atServiceId[0],
        ptChannelId, un16Count, FALSE);

    // Relinquish ownership
    SMSO_vUnlock((SMS_OBJECT)hTuneMix);

    if (eReturnCode == SMSAPI_RETURN_CODE_SUCCESS)
    {
        eReturnCode = DECODER_eTuneMixConfigure(
            psObj->hDecoder,
            psObj->un8Index );
        if (eReturnCode != SMSAPI_RETURN_CODE_SUCCESS)
        {
            puts(TUNEMIX_OBJECT_NAME": Cannot configure TuneMix");
        }
    }
    else
    {
        puts(TUNEMIX_OBJECT_NAME": Failed to fill component list");
    }

    return eReturnCode;
}

/*************************************************************************
 *
 *   eIterate
 *
 *************************************************************************/
static SMSAPI_RETURN_CODE_ENUM eIterate (
    DECODER_OBJECT hDecoder,
    TUNEMIX_ITERATOR_CALLBACK bIterateFxn,
    void *pvArg
        )
 {
    SMSAPI_RETURN_CODE_ENUM eReturnCode = SMSAPI_RETURN_CODE_INVALID_INPUT;
    BOOLEAN bOk;

    // Check inputs
    if (bIterateFxn == NULL)
    {
        return SMSAPI_RETURN_CODE_INVALID_INPUT;
    }

    // Verify and lock SMS Object
    bOk = SMSO_bLock((SMS_OBJECT)hDecoder, OSAL_OBJ_TIMEOUT_INFINITE);
    if (bOk == TRUE)
    {
        bOk = DECODER_bIsTuneMixSupported(hDecoder);
        if (bOk == FALSE)
        {
            puts(TUNEMIX_OBJECT_NAME": TuneMix functionality is not supported");

            eReturnCode = SMSAPI_RETURN_CODE_UNSUPPORTED_API;
        }
        else
        {
            eReturnCode = DECODER_eIterateTuneMixList(hDecoder, 
                bIterateFxn, pvArg);
        }

        // Unlock decoder
        SMSO_vUnlock((SMS_OBJECT)hDecoder);
    }

    return eReturnCode;
}

/*************************************************************************
 *
 *   eStatus
 *
 *************************************************************************/
static TUNEMIX_STATUS_ENUM eStatus ( 
    TUNEMIX_OBJECT hTuneMix
        )
{
    BOOLEAN bOk;
    TUNEMIX_OBJECT hActive;
    TUNEMIX_STATUS_ENUM eStatus = TUNEMIX_STATUS_UNDEFINED;
    TUNEMIX_OBJECT_STRUCT *psObj =
        (TUNEMIX_OBJECT_STRUCT *)hTuneMix;

    bOk = SMSO_bLock((SMS_OBJECT)hTuneMix, OSAL_OBJ_TIMEOUT_INFINITE);
    if( bOk == TRUE)
    {
        hActive = DECODER_hGetTuneMixActive(psObj->hDecoder);
        if (hActive != hTuneMix)
        {
            SMSAPI_RETURN_CODE_ENUM eReturnCode;

            eReturnCode = TUNEMIX_eQualified(hTuneMix);
            if (eReturnCode == SMSAPI_RETURN_CODE_SUCCESS)
            {
                eStatus = TUNEMIX_STATUS_AVAILABLE;
            }
            else
            {
                eStatus = TUNEMIX_STATUS_UNAVAILABLE;
            }
            psObj->eStatus = eStatus;
        }
        else
        {
            eStatus = psObj->eStatus;
        }

        // Relinquish ownership
        SMSO_vUnlock((SMS_OBJECT)hTuneMix);
    }

    return eStatus;
}

/*************************************************************************
 *
 *   eIterateChannels
 *
 *************************************************************************/
static SMSAPI_RETURN_CODE_ENUM eIterateChannels (
    TUNEMIX_OBJECT hTuneMix,
    TUNEMIX_CHANNELS_ITERATOR_CALLBACK bIterateFxn,
    void *pvArg
        )
{
    SMSAPI_RETURN_CODE_ENUM eReturnCode = SMSAPI_RETURN_CODE_INVALID_INPUT;
    BOOLEAN bLocked;
    TUNEMIX_OBJECT_STRUCT *psObj =
        (TUNEMIX_OBJECT_STRUCT *)hTuneMix;

    if (bIterateFxn == NULL)
    {
        return SMSAPI_RETURN_CODE_INVALID_INPUT;
    }

    // Verify and lock SMS Object
    bLocked = SMSO_bLock((SMS_OBJECT)hTuneMix, OSAL_OBJ_TIMEOUT_INFINITE);
    if((bLocked == TRUE))
    {
        OSAL_RETURN_CODE_ENUM eOsalReturnCode;
        TUNEMIX_COMPONENTS_ITERATOR_STRUCT sIterator;

        sIterator.bIterateFxn = bIterateFxn;
        sIterator.pvArg = pvArg;
        sIterator.hCCache = psObj->hCCache;

        eOsalReturnCode = OSAL.eLinkedListIterate(psObj->hComponentList,
                              (OSAL_LL_ITERATOR_HANDLER)bChannelIterator,
                              (void *)&sIterator);

        if (eOsalReturnCode == OSAL_NO_OBJECTS)
        {
            eReturnCode = SMSAPI_RETURN_CODE_NO_OBJECTS;
        }
        else if (eOsalReturnCode != OSAL_SUCCESS)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                TUNEMIX_OBJECT_NAME
               ": failed to iterate TuneMix %p channels (%s)",
               hTuneMix, OSAL.pacGetReturnCodeName(eOsalReturnCode));
            eReturnCode = SMSAPI_RETURN_CODE_ERROR;
        }
        else
        {
            eReturnCode = SMSAPI_RETURN_CODE_SUCCESS;
        }
         // Unlock object
        SMSO_vUnlock((SMS_OBJECT)hTuneMix);
    }

    return eReturnCode;
}

/*************************************************************************
 *
 *   eSetOverrides
 *
 *************************************************************************/
static SMSAPI_RETURN_CODE_ENUM eSetOverrides (
    TUNEMIX_OBJECT hTuneMix,
    BOOLEAN bOverride,
    BOOLEAN bPlayUnrestricted
        )
{
    SMSAPI_RETURN_CODE_ENUM eReturnCode = SMSAPI_RETURN_CODE_SUCCESS;
    BOOLEAN bResult;
    TUNEMIX_OBJECT_STRUCT *psObj = (TUNEMIX_OBJECT_STRUCT *)hTuneMix;

    bResult = SMSO_bLock((SMS_OBJECT)hTuneMix, OSAL_OBJ_TIMEOUT_INFINITE);
    if (bResult == FALSE)
    {
        eReturnCode = SMSAPI_RETURN_CODE_INVALID_INPUT;
    }
    else
    {
        psObj->bOverride = bOverride;
        psObj->bPlayUnrestricted = bPlayUnrestricted;

        // save overrides to config
        bResult = bSaveOverrides(psObj->hTag, bOverride, bPlayUnrestricted);
        if (bResult != TRUE)
        {
            eReturnCode = SMSAPI_RETURN_CODE_ERROR;
        }

        // Relinquish ownership
        SMSO_vUnlock((SMS_OBJECT)hTuneMix);
    }
    return eReturnCode;
}

/*************************************************************************
 *
 *   eGetOverrides
 *
 *************************************************************************/
static SMSAPI_RETURN_CODE_ENUM eGetOverrides (
    TUNEMIX_OBJECT hTuneMix,
    BOOLEAN *pbOverride,
    BOOLEAN *pbPlayUnrestricted
        )
{
    SMSAPI_RETURN_CODE_ENUM eReturnCode = SMSAPI_RETURN_CODE_INVALID_INPUT;
    BOOLEAN bResult;
    TUNEMIX_OBJECT_STRUCT *psObj = (TUNEMIX_OBJECT_STRUCT *)hTuneMix;

    bResult = SMSO_bLock((SMS_OBJECT)hTuneMix, OSAL_OBJ_TIMEOUT_INFINITE);
    if (bResult == TRUE)
    {
        if (pbOverride != NULL)
        {
            *pbOverride = psObj->bOverride;
        }
        if (pbPlayUnrestricted != NULL)
        {
            *pbPlayUnrestricted = psObj->bPlayUnrestricted;
        }
        // Relinquish ownership
        SMSO_vUnlock((SMS_OBJECT)hTuneMix);
        eReturnCode = SMSAPI_RETURN_CODE_SUCCESS;
    }
    return eReturnCode;
}

/*************************************************************************
 *
 *   hGetName
 *
 *************************************************************************/
static STRING_OBJECT hGetName (
    TUNEMIX_OBJECT hTuneMix
        )
{
    STRING_OBJECT hName = STRING_INVALID_OBJECT;
    BOOLEAN bOk;

    bOk = SMSO_bValid((SMS_OBJECT)hTuneMix);
    if (bOk == TRUE)
    {
        TUNEMIX_OBJECT_STRUCT *psObj = (TUNEMIX_OBJECT_STRUCT *)hTuneMix;

        hName = psObj->hName;
    }

    return hName;
}

/*************************************************************************
 *
 *   eSetName
 *
 *************************************************************************/
static SMSAPI_RETURN_CODE_ENUM eSetName (
    TUNEMIX_OBJECT hTuneMix,
    char *pacName
        )
{
    BOOLEAN bResult;
    SMSAPI_RETURN_CODE_ENUM eReturnCode = SMSAPI_RETURN_CODE_INVALID_INPUT;

    // Lock the object
    bResult = SMSO_bLock((SMS_OBJECT)hTuneMix, OSAL_OBJ_TIMEOUT_INFINITE);
    if (bResult == TRUE)
    {
        TUNEMIX_OBJECT_STRUCT *psObj = (TUNEMIX_OBJECT_STRUCT *)hTuneMix;
        size_t tNameSize = 0;

        // Do we have new name provided?
        if (pacName != NULL)
        {
            tNameSize = strlen(pacName);
        }
        
        // Destroying old name object
        if (psObj->hName != STRING_INVALID_OBJECT)
        {
            STRING_vDestroy(psObj->hName);
            psObj->hName = STRING_INVALID_OBJECT;
        }
        
        // Create new string object if some string was provided
        if (tNameSize > 0)
        {
            psObj->hName = STRING_hCreate(SMS_INVALID_OBJECT, 
                pacName, tNameSize, tNameSize);
        }

        // Saving into configuration file
        bSaveName(psObj->hTag, psObj->hName);

        // Release the object
        SMSO_vUnlock((SMS_OBJECT)hTuneMix);

        // Done
        eReturnCode = SMSAPI_RETURN_CODE_SUCCESS;
    }

    return eReturnCode;
}

/*************************************************************************
 *                       FRIEND FUNCTIONS
 *************************************************************************/

/*************************************************************************
 *
 *    TUNEMIX_hCreateFromTags
 *
 *************************************************************************/
TUNEMIX_OBJECT TUNEMIX_hCreateFromTags (
    DECODER_OBJECT hDecoder,
    TAG_OBJECT hCurrentTMTag,
    UN8 *pun8Index,
    BOOLEAN bBehaviorValue
        )
{
    BOOLEAN bOk = FALSE;
    TUNEMIX_OBJECT_STRUCT *psObj = TUNEMIX_INVALID_OBJECT_STRUCT_PTR;
    UN32 un32Children;
    size_t tBytesAvail = 0;
    SMSAPI_RETURN_CODE_ENUM eReturnCode;
    N32 n32Value, *pn32Value = &n32Value;
    TAG_OBJECT hTagActive, hTagServices, hTagOverrides, hTagIndex, hTagName;
    TUNEMIX_OBJECT hTuneMix = TUNEMIX_INVALID_OBJECT;
    CHANNEL_ID atChannelId[TUNEMIX_CHANNELS_MAX];
    TUNEMIX_RESTORE_STRUCT sData;
    CHANNEL_ID tChannelId;

    do
    {
        BOOLEAN bOwner;
        STRING_OBJECT hString = STRING_INVALID_OBJECT, *phString;

        // Verify we own the decoder
        bOwner = SMSO_bOwner((SMS_OBJECT)hDecoder);
        if (bOwner == FALSE)
        {
            break;
        }

        // Get Services Tag
        eReturnCode = TAG_eGet(
            TUNEMIX_COMPONENTS_TAG_NAME,
            hCurrentTMTag,
            &hTagServices,
            (char *)NULL,
            FALSE);

        if ((eReturnCode != SMSAPI_RETURN_CODE_SUCCESS) ||
            (hTagServices == TAG_INVALID_OBJECT))
        {
            break;
        }

        eReturnCode = TAG_eNumberOfChildren( hTagServices, &un32Children );
        if (eReturnCode != SMSAPI_RETURN_CODE_SUCCESS)
        {
            break;
        }
        // Do not create Tune Mix object if number of components < 2
        if (un32Children < TUNEMIX_CHANNELS_MIN)
        {
            break;
        }

        // Initialize iterator struct
        sData.bSuccess = TRUE;
        sData.un16ChIndex = 0;

        // Get all serviceId
        eReturnCode = TAG_eIterateChildren(
            hTagServices,
            bRestoreIterator,
            (void *)&sData
                );

        if ((eReturnCode != SMSAPI_RETURN_CODE_SUCCESS) ||
            (sData.bSuccess == FALSE))
        {
            break;
        }

        // Get Index Tag
        eReturnCode = TAG_eGet(
            TUNEMIX_INDEX_TAG_NAME,
            hCurrentTMTag,
            &hTagIndex,
            (char *)NULL,
            FALSE);
        if ((eReturnCode != SMSAPI_RETURN_CODE_SUCCESS) ||
            (hTagIndex == TAG_INVALID_OBJECT))
        {
            break;
        }

        // Get available space size
        tBytesAvail = sizeof(N32);
        // Get Overrides
        eReturnCode = TAG_eGetTagValue(
            hTagIndex,
            TAG_TYPE_INTEGER,
            (void **)&pn32Value,
            &tBytesAvail);
        if (eReturnCode != SMSAPI_RETURN_CODE_SUCCESS)
        {
            break;
        }

        *pun8Index = (UN8)n32Value;

        // Getting Tune Mix ID from index
        tChannelId = DECODER_tTuneMixIdFromIndex(
            hDecoder, 
            *pun8Index);

        // Get Name Tag
        eReturnCode = TAG_eGet(
            TUNEMIX_NAME_TAG_NAME,
            hCurrentTMTag,
            &hTagName,
            (char *)NULL,
            FALSE);
        if ((eReturnCode != SMSAPI_RETURN_CODE_SUCCESS) ||
            (hTagName == TAG_INVALID_OBJECT))
        {
            break;
        }

        phString = &hString;
        tBytesAvail = sizeof(STRING_OBJECT);

        // Get Name
        eReturnCode = TAG_eGetTagValue(
            hTagName,
            TAG_TYPE_STRING,
            (void **)&phString,
            &tBytesAvail);
        if ((eReturnCode != SMSAPI_RETURN_CODE_SUCCESS) &&
            (eReturnCode != SMSAPI_RETURN_CODE_CFG_NO_VALUE_SET))
        {
            break;
        }

        // Create the instance
        psObj = psCreateObject(hDecoder, tChannelId);
        if (psObj == NULL)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                TUNEMIX_OBJECT_NAME": failed to create the object");
            break;
        }

        // To fill the TuneMix structure
        psObj->hDecoder = hDecoder;
        psObj->un8Index = *pun8Index;
        psObj->hTag = hCurrentTMTag;
        psObj->bPlayUnrestricted = FALSE;
        psObj->bOverride = FALSE;
        psObj->hName = hString;

        if (sData.un16ChIndex >= TUNEMIX_CHANNELS_MIN)
        {
            int iIndex;

            for (iIndex = 0; iIndex < TUNEMIX_CHANNELS_MAX; iIndex++)
            {
                atChannelId[iIndex] = CHANNEL_INVALID_ID;
            }

            // Get Override Tag
            eReturnCode = TAG_eGet(
                TUNEMIX_OVERRIDES_TAG_NAME,
                hCurrentTMTag,
                &hTagOverrides,
                (char *)NULL,
                FALSE);
            if ((eReturnCode != SMSAPI_RETURN_CODE_SUCCESS) ||
                (hTagOverrides == TAG_INVALID_OBJECT))
            {
                break;
            }

            // Get available space size
            tBytesAvail = sizeof(N32);
            // Get Overrides
            eReturnCode = TAG_eGetTagValue(
                hTagOverrides,
                TAG_TYPE_INTEGER,
                (void **)&pn32Value,
                &tBytesAvail
                    );
            if (eReturnCode != SMSAPI_RETURN_CODE_SUCCESS)
            {
                break;
            }

            // PlayUnrestricted is always retrieved from config
            if (((UN8)n32Value & TUNEMIX_PLAY_UNRESTRICTED)
                == TUNEMIX_PLAY_UNRESTRICTED)
            {
                psObj->bPlayUnrestricted = TRUE;
            }

            // Set Override from config
            if (((UN8)n32Value & TUNEMIX_OVERRIDE)
                == TUNEMIX_OVERRIDE)
            {
                psObj->bOverride = TRUE;
            }

            eReturnCode = eFillComponentList(
                (TUNEMIX_OBJECT)psObj,
                &sData.atServiceId[0],
                &atChannelId[0], sData.un16ChIndex, TRUE);
            if (eReturnCode != SMSAPI_RETURN_CODE_SUCCESS)
            {
                // Error
                break;
            }

            // Now resetting Override flag according to powerup Parental 
            // Control setting to prevent automatic tune
            if ((bBehaviorValue == FALSE) && (psObj->bOverride == TRUE))
            {
                BOOLEAN bResult;

                // Parental Control is set.
                // Override for the TuneMix set to FALSE.
                psObj->bOverride = FALSE;
                // Save overrides to config.
                bResult = bSaveOverrides(psObj->hTag, FALSE, psObj->bPlayUnrestricted);
                if (bResult != TRUE)
                {
                    break;
                }
            }

            bOk = RADIO_bTuneMixConfigure(
                psObj->hDecoder,
                &sData.atServiceId[0],
                // the number of desired music channels
                sData.un16ChIndex,
                // Tune Mix Id
                psObj->un8Index);
            if (TRUE == bOk)
            {
                BOOLEAN bActive;

                psObj->eStatus = TUNEMIX_STATUS_AVAILABLE;
                psObj->tChannelID = tChannelId;

                // Get Active Tag
                eReturnCode = TAG_eGet(
                    TUNEMIX_ACTIVE_TAG_NAME,
                    hCurrentTMTag,
                    &hTagActive,
                    (char *)NULL,
                    FALSE
                        );
                if ((eReturnCode != SMSAPI_RETURN_CODE_SUCCESS) ||
                    (hTagActive == TAG_INVALID_OBJECT))
                {
                    break;
                }

                // Get available space size
                tBytesAvail = sizeof(N32);
                // Get Active value
                eReturnCode = TAG_eGetTagValue(
                    hTagActive,
                    TAG_TYPE_INTEGER,
                    (void **)&pn32Value,
                    &tBytesAvail
                        );
                if (eReturnCode != SMSAPI_RETURN_CODE_SUCCESS)
                {
                    break;
                }
                // Save Active value
                bActive = (BOOLEAN)n32Value;

                if (bActive == TRUE)
                {
                    eReturnCode = TUNEMIX.eTune((TUNEMIX_OBJECT)psObj);
                    if (eReturnCode != SMSAPI_RETURN_CODE_SUCCESS)
                    {
                        SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                            TUNEMIX_OBJECT_NAME
                            ": Failed to tune TuneMix with ID %d",
                            tChannelId);

                        /* Cannot tune to TuneMix, so setting Active to FALSE */
                        n32Value = 0;
                        eReturnCode = TAG_eSetTagValue(hTagActive, 
                            TAG_TYPE_INTEGER, 
                            &n32Value, 
                            sizeof(N32), 
                            TRUE);
                    }
                }

                // Populate the object
                hTuneMix = (TUNEMIX_OBJECT) psObj;
                bOk = TRUE;
            }
        }
    } while (FALSE);

    if ((bOk == FALSE) && (psObj != TUNEMIX_INVALID_OBJECT_STRUCT_PTR))
    {
        TUNEMIX_vDeallocate((TUNEMIX_OBJECT)psObj);
    }

    return hTuneMix;
}

/*************************************************************************
*
*   TUNEMIX_vUpdateTuneMixStatus
*
*      This friend function allows the tunemix to update status and
*      ask the decoder to inform app
*
*************************************************************************/
void TUNEMIX_vUpdateTuneMixStatus (
    DECODER_OBJECT hDecoder,
    CHANNEL_ID tTuneMixId,
    BOOLEAN bActive,
    TUNEMIX_STATUS_ENUM eStatus
        )
{
    TUNEMIX_OBJECT hTuneMix, hTuneMixActive;
    TUNEMIX_OBJECT_STRUCT *psObj, *psObjActive;
    BOOLEAN bChanged = FALSE;

    // Getting TuneMix handles. Any of these can be INVALID
    hTuneMix = DECODER_hGetTuneMix(hDecoder, tTuneMixId);
    hTuneMixActive = DECODER_hGetTuneMixActive(hDecoder);

    psObj = (TUNEMIX_OBJECT_STRUCT *)hTuneMix;
    psObjActive = (TUNEMIX_OBJECT_STRUCT *)hTuneMixActive;

    if (TRUE == bActive)
    {
        // We are updating Active TuneMix
        if (hTuneMixActive != hTuneMix)
        {
            BOOLEAN bOk;

            // Setting new one as Active
            DECODER_vSetTuneMixActive(hDecoder, hTuneMix);

            // Active TuneMix is changed
            if (hTuneMixActive != TUNEMIX_INVALID_OBJECT)
            {
                // We had another Active previously, so deactivating it
                bOk = bSaveActive(psObjActive->hTag, FALSE);
                if (bOk == FALSE)
                {
                    printf(TUNEMIX_OBJECT_NAME
                        ": Cannot clear TuneMix Active flag for TuneMix %u\n",
                        psObjActive->tChannelID);
                }

                // Saving status
                psObjActive->eStatus = eStatus;

                // We need event notification
                bChanged = TRUE;
            }

            if (hTuneMix != TUNEMIX_INVALID_OBJECT)
            {
                bOk = bSaveActive(psObj->hTag, TRUE);
                if (bOk == FALSE)
                {
                    printf(TUNEMIX_OBJECT_NAME
                        ": Cannot set TuneMix Active flag for TuneMix %u\n",
                        psObj->tChannelID);
                }

                // Saving status
                psObj->eStatus = eStatus;

                // We need event notification
                bChanged = TRUE;
            }
            else
            {
                // Status update belongs to previous active Tune Mix
                if ((psObjActive != NULL) && 
                    (psObjActive->eStatus != eStatus))
                {
                    psObjActive->eStatus = eStatus;

                    // We need event notification
                    bChanged = TRUE;
                }
            }
        }
        else
        {
            // Status update belongs to previous active Tune Mix
            if ((psObjActive != NULL) && 
                (psObjActive->eStatus != eStatus))
            {
                psObjActive->eStatus = eStatus;

                // We need event notification
                bChanged = TRUE;
            }
        }
    }
    else
    {
        // We are updating non-active TuneMix
        if (hTuneMix != TUNEMIX_INVALID_OBJECT)
        {
            if (psObj->eStatus != eStatus)
            {
                psObj->eStatus = eStatus;

                // We need event notification
                bChanged = TRUE;
            }
        }
    }

    if (bChanged == TRUE)
    {
        // Now sending notification
        DECODER_vTuneMixEvent(hDecoder);
    }
    return;
}

/*************************************************************************
*
*   TUNEMIX_tGetSIDFromActiveTM
*
*   This friend function allows the decoder to get first saved SID 
*   from components list of active TuneMix service.
*
*************************************************************************/
SERVICE_ID TUNEMIX_tGetSIDFromActiveTM (
    TAG_OBJECT hParentTag
        )
{
    TAG_OBJECT hTagTM;
    SMSAPI_RETURN_CODE_ENUM eResultCode;
    SERVICE_ID tServiceId = SERVICE_INVALID_ID;

    do
    {
        eResultCode = TAG_eGet(
            TUNEMIX_OBJECT_NAME,
            hParentTag,
            &hTagTM,
            (char *)NULL,
            TRUE
                );
        if (eResultCode == SMSAPI_RETURN_CODE_CFG_NO_PARENT)
        {
            break;
        }

        if ((eResultCode != SMSAPI_RETURN_CODE_SUCCESS) ||
           (hTagTM == TAG_INVALID_OBJECT))
        {
            break;
        }

        // Get active ServiceId
        TAG_eIterateChildren(
            hTagTM,
            bTMIterator,
            (void *)&tServiceId
                );

    } while (FALSE);

    return tServiceId;
}

/*************************************************************************
 *
 *   TUNEMIX_bSaveConfiguration
 *
 *************************************************************************/
BOOLEAN TUNEMIX_bSaveConfiguration (
    DECODER_OBJECT hDecoder,
    UN16 un16TuneMixId,
    SERVICE_ID atServiceId[],
    UN16 un16NumberOfServices
        )
{
    TAG_OBJECT hTag, hNextTag, hParentTag;
    BOOLEAN bSuccess = FALSE;
    SMSAPI_RETURN_CODE_ENUM eResultCode;
    UN16 un16Index;

    do
    {
        TUNEMIX_OBJECT hTuneMix;
        TUNEMIX_OBJECT_STRUCT *psObj;

        hTuneMix = DECODER_hGetTuneMix(hDecoder, (SERVICE_ID)un16TuneMixId);
        if (hTuneMix == TUNEMIX_INVALID_OBJECT)
        {
            break;
        }

        psObj = (TUNEMIX_OBJECT_STRUCT *)hTuneMix;

        // Get Services array Tag
        eResultCode = TAG_eGet(
            TUNEMIX_COMPONENTS_TAG_NAME,
            psObj->hTag,
            &hParentTag,
            (char *)NULL,
            TRUE);
        if (eResultCode != SMSAPI_RETURN_CODE_SUCCESS)
        {
            break;
        }

        eResultCode = TAG_eFirstChild(hParentTag, &hNextTag);
        if (eResultCode != SMSAPI_RETURN_CODE_SUCCESS)
        {
            break;
        }

        for (un16Index = 0; un16Index < un16NumberOfServices; un16Index++)
        {
            if (hNextTag == TAG_INVALID_OBJECT)
            {
                // Get Service Tag
                eResultCode = TAG_eAdd(
                    TUNEMIX_SERVICE_ID_TAG_NAME,
                    hParentTag,
                    &hTag,
                    (const char*)NULL);
            }
            else
            {
                // Re-configuration
                hTag = hNextTag;
                eResultCode = TAG_eNextSibling(hTag, &hNextTag);
            }

            if (eResultCode == SMSAPI_RETURN_CODE_SUCCESS)
            {
                N32 n32Value = (N32)atServiceId[un16Index];

                eResultCode = TAG_eSetTagValue(
                    hTag,
                    TAG_TYPE_INTEGER,
                    (void *)&n32Value,
                    sizeof(N32),
                    TRUE);

                if (eResultCode != SMSAPI_RETURN_CODE_SUCCESS)
                {
                    TAG_eRemove(hTag, TRUE);
                    break;
                }
            }
            else
            {
                break;
            }
        }

        while (hNextTag != TAG_INVALID_OBJECT)
        {
            //To remove extra tags
            hTag = hNextTag;
            eResultCode = TAG_eNextSibling(hTag,
                              &hNextTag);

            if (eResultCode != SMSAPI_RETURN_CODE_SUCCESS)
            {
                break;
            }
            TAG_eRemove(hTag, TRUE);
         }

        bSuccess = TRUE;
    } while (FALSE);

    return bSuccess;
}

/*************************************************************************
 *
 *   TUNEMIX_vDeallocate
 *
 *************************************************************************/
void TUNEMIX_vDeallocate (
    TUNEMIX_OBJECT hTuneMix
        )
{
    TUNEMIX_OBJECT_STRUCT *psObj = 
        (TUNEMIX_OBJECT_STRUCT*)hTuneMix;
    OSAL_RETURN_CODE_ENUM eReturnCode;
    BOOLEAN bOwner;

    bOwner = SMSO_bOwner((SMS_OBJECT)hTuneMix);
    if (bOwner == FALSE)
    {
        return;
    }

    if (psObj->hComponentList != OSAL_INVALID_OBJECT_HDL)
    {
        eReturnCode = OSAL.eLinkedListRemoveAll(
            psObj->hComponentList, (OSAL_LL_RELEASE_HANDLER)NULL);
        if (eReturnCode != OSAL_SUCCESS)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__, 
                TUNEMIX_OBJECT_NAME
                ": failed to remove component list items (%s)",
                OSAL.pacGetReturnCodeName(eReturnCode));
        }

        eReturnCode = OSAL.eLinkedListDelete(psObj->hComponentList);
        if (eReturnCode != OSAL_SUCCESS)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                TUNEMIX_OBJECT_NAME
                ": failed to delete Component Channel List (%s)",
                OSAL.pacGetReturnCodeName(eReturnCode));
        }
    }

    // Destroying name string
    if (psObj->hName != STRING_INVALID_OBJECT)
    {
        STRING_vDestroy(psObj->hName);
    }

    // Destroying TuneMix object
    SMSO_vDestroy((SMS_OBJECT)psObj);

    return;
}

/*************************************************************************
*
*   TUNEMIX_eQualified
*
**************************************************************************/
SMSAPI_RETURN_CODE_ENUM TUNEMIX_eQualified (
    TUNEMIX_OBJECT hTuneMix
    )
{
    SMSAPI_RETURN_CODE_ENUM eReturnCode = 
        SMSAPI_RETURN_CODE_OP_NOT_SUPPORTED;
    TUNEMIX_OBJECT_STRUCT *psObj = 
        (TUNEMIX_OBJECT_STRUCT *)hTuneMix;

    do
    {
        OSAL_RETURN_CODE_ENUM eOsalReturnCode;
        TUNEMIX_QUALIFY_ITERATOR_STRUCT sIteratorStruct;

        sIteratorStruct.un16Count = 0;
        sIteratorStruct.hCCache = psObj->hCCache;
        sIteratorStruct.un16Unrestricted = 0;

        eOsalReturnCode = OSAL.eLinkedListIterate( 
            psObj->hComponentList,
            (OSAL_LL_ITERATOR_HANDLER)bIterateForQualified,
            (void *)&sIteratorStruct);

        if (eOsalReturnCode == OSAL_NO_OBJECTS)
        {
            printf (TUNEMIX_OBJECT_NAME
                ": Component list of TuneMix %u is empty\n",
                psObj->tChannelID);
            break;
        }
        else if (eOsalReturnCode != OSAL_SUCCESS )
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                TUNEMIX_OBJECT_NAME
                ": failed iterate Channels (%s)\n",
                OSAL.pacGetReturnCodeName(eOsalReturnCode));
            eReturnCode = SMSAPI_RETURN_CODE_ERROR;
            break;
        }

        if (sIteratorStruct.un16Count < TUNEMIX_CHANNELS_MIN)
        {
            printf(TUNEMIX_OBJECT_NAME
                ": TuneMix %u is not qualified\n", 
                psObj->tChannelID);
            break;
        }
        else
        {
            if (psObj->bOverride == FALSE)
            {
                // User wishes to play only unrestricted components,
                // but there are less than 2 unrestricted
                if ((psObj->bPlayUnrestricted == TRUE) && 
                    (sIteratorStruct.un16Unrestricted < TUNEMIX_CHANNELS_MIN))
                {
                    break;
                }
                // User don't wish to play TuneMix channel if it consists
                // lock/mature components
                else if ((psObj->bPlayUnrestricted == FALSE) && 
                    (sIteratorStruct.un16Unrestricted !=
                    sIteratorStruct.un16Count))
                {
                    eReturnCode = SMSAPI_RETURN_CODE_LOCKED_CHANNEL;
                    break;
                }
            }
        }
        eReturnCode = SMSAPI_RETURN_CODE_SUCCESS;
    } while (FALSE);

    return eReturnCode;
}

/*************************************************************************
 *                  PRIVATE FUNCTIONS
 *************************************************************************/
/*************************************************************************
 *
 *    eFillComponentList
 *
 *************************************************************************/
static SMSAPI_RETURN_CODE_ENUM eFillComponentList (
    TUNEMIX_OBJECT hTuneMix,
    SERVICE_ID *ptServiceId,
    CHANNEL_ID *ptChannelId,
    UN16 un16Count,
    BOOLEAN bCreateIfNotFound
        )
{
    SMSAPI_RETURN_CODE_ENUM eReturnCode = SMSAPI_RETURN_CODE_SUCCESS;
    CCACHE_OBJECT hCCache;
    CHANNEL_OBJECT hChannel;
    UN16 un16Index = 0, un16ChanCount = 0, un16Unrestricted = 0;
    TUNEMIX_OBJECT_STRUCT *psObj = (TUNEMIX_OBJECT_STRUCT *)hTuneMix;

    // We do not check input parameters here, because they are checked
    // before this private function's call.

    // Get the cache handle
    hCCache = psObj->hCCache;

    // We only accept TUNEMIX_CHANNELS_MAX component channels
    // out of full provided list, since radio module does not support
    // more anyway.
    while ((un16Index < un16Count) &&
           (un16Index < TUNEMIX_CHANNELS_MAX))
    {
        // From the channel id provided,
        // determine the channel object handler.
        hChannel = CCACHE_hChannelFromIds(
            hCCache, ptServiceId[un16Index], ptChannelId[un16Index],
            bCreateIfNotFound);

        if (hChannel == CHANNEL_INVALID_OBJECT)
        {
            printf(TUNEMIX_OBJECT_NAME
                ": Cannot get handle for channel %d (SID %d)\n",
                ptChannelId[un16Index], ptServiceId[un16Index]
                    );
        }
        else
        {
            BOOLEAN bIsMusic = FALSE;
            BOOLEAN bIsSubscribed = FALSE;
            BOOLEAN bIsRestricted = FALSE;

            vGetComponentProperties(hChannel, &bIsMusic, &bIsSubscribed, &bIsRestricted);
            // Only subscribed music channels are needed
            if ((bIsMusic == TRUE) && (bIsSubscribed == TRUE))
            {
                OSAL_RETURN_CODE_ENUM eOsalReturnCode;

                if (ptChannelId[un16Index] == CHANNEL_INVALID_ID)
                {
                    ptChannelId[un16Index] = CHANNEL.tChannelId(hChannel);
                }

                // Add it to the list
                eOsalReturnCode = OSAL.eLinkedListAdd(
                    psObj->hComponentList,
                    OSAL_INVALID_LINKED_LIST_ENTRY_PTR,
                    (void *)(size_t)ptChannelId[un16Index] );
                if (eOsalReturnCode == OSAL_SUCCESS)
                {
                    ++un16ChanCount;
                    if (bIsRestricted == FALSE)
                    {
                        ++un16Unrestricted;
                    }
                }
                else if (eOsalReturnCode != OSAL_ERROR_LIST_ITEM_NOT_UNIQUE)
                {
                    SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                        TUNEMIX_OBJECT_NAME
                        ": failed to add Channel %d (%s)",
                        ptChannelId[un16Index],
                        OSAL.pacGetReturnCodeName(eOsalReturnCode));
                }
            }
        }
        ++un16Index;
    }

    if ((un16Count > 0) && (un16ChanCount < TUNEMIX_CHANNELS_MIN))
    {
        // Not enough qualified channels.
        (void)OSAL.eLinkedListRemoveAll( psObj->hComponentList, 
            (OSAL_LL_RELEASE_HANDLER)NULL );
        eReturnCode = SMSAPI_RETURN_CODE_OP_NOT_SUPPORTED;
    }
    else if (un16ChanCount >= TUNEMIX_CHANNELS_MIN)
    {
        if (bCreateIfNotFound == FALSE)
        {
            if (psObj->bOverride == FALSE)
            {
                // User wishes to play only unrestricted components,
                // but there are less than 2 unrestricted
                if ((psObj->bPlayUnrestricted == TRUE) &&
                    (un16Unrestricted < TUNEMIX_CHANNELS_MIN))
                {
                    eReturnCode = SMSAPI_RETURN_CODE_OP_NOT_SUPPORTED;
                }
                // User don't wish to play TuneMix channel if itconists
                // lock/mature components
                else if ((psObj->bPlayUnrestricted == FALSE) &&
                    (un16Unrestricted != un16ChanCount))
                {
                    eReturnCode = SMSAPI_RETURN_CODE_LOCKED_CHANNEL;
                }
            }
        }
    }

    return eReturnCode;
}

/*************************************************************************
 *
 *   vGetComponentProperties
 *
**************************************************************************/
static void vGetComponentProperties(
    CHANNEL_OBJECT hChannel,
    BOOLEAN *pbIsMusic,
    BOOLEAN *pbIsSubscribed,
    BOOLEAN *pbIsRestricted
        )
{
    SMSAPI_RETURN_CODE_ENUM eReturnCode;

    eReturnCode = CHANNEL.eIsSubscribed(hChannel, pbIsSubscribed);
    if(eReturnCode == SMSAPI_RETURN_CODE_SUCCESS)
    {
        *pbIsMusic = CHANNEL.bQualifiedForTuneMix(hChannel);
        CHANNEL.eIsLocked(hChannel, pbIsRestricted);
        if (*pbIsRestricted == FALSE)
        {
            CHANNEL.eIsMature(hChannel, pbIsRestricted);
        }
    }
    return;
}

/*************************************************************************
 *
 *   bIterateForQualified
 *
**************************************************************************/
static BOOLEAN bIterateForQualified (
    CHANNEL_ID tChannelId,
    void *pvArg
        )
{
    TUNEMIX_QUALIFY_ITERATOR_STRUCT *psIteratorStruct =
        (TUNEMIX_QUALIFY_ITERATOR_STRUCT*)pvArg;
    CHANNEL_OBJECT hChannel;


    hChannel = CCACHE_hChannelFromIds(
            psIteratorStruct->hCCache, SERVICE_INVALID_ID,
            tChannelId, FALSE);

    if (hChannel == CHANNEL_INVALID_OBJECT)
    {
        // Indicate that the channel is deleted from channel map.
        // We exclude this from qualifing iteration.
        printf(TUNEMIX_OBJECT_NAME
            ": Cannot get handle for channel %d \n",
            tChannelId);
    }
    else
    {
        BOOLEAN bIsMusic = FALSE;
        BOOLEAN bIsSubscribed = FALSE;
        BOOLEAN bIsRestricted = FALSE;

        vGetComponentProperties(hChannel, &bIsMusic, &bIsSubscribed, &bIsRestricted);
        if ((bIsMusic == TRUE) && (bIsSubscribed == TRUE))
            {
                psIteratorStruct->un16Count++;
                if (bIsRestricted == FALSE)
                {
                    psIteratorStruct->un16Unrestricted++;
                }
            }
        }

    return TRUE;
}

/*************************************************************************
 *
 *   bCreateTuneMixTag
 *
 *************************************************************************/
static BOOLEAN bCreateTuneMixTag (
    TUNEMIX_OBJECT_STRUCT *psObj
        )
{
    TAG_OBJECT hParentTag;
    BOOLEAN bReturn = FALSE;

    do 
    {
        TAG_OBJECT hTag;
        SMSAPI_RETURN_CODE_ENUM eReturnCode;
        N32 n32Value;
        STRING_OBJECT hString;

        // Get the decoder tag
        hParentTag = DECODER_hGetTag( psObj->hDecoder );
        if (hParentTag == TAG_INVALID_OBJECT)
        {
            break;
        }

        // Get Tune Mix Tag
        eReturnCode = TAG_eGet(
            TUNEMIX_OBJECT_NAME,
            hParentTag,
            &hParentTag,
            (const char *)NULL,
            TRUE );

        if (eReturnCode != SMSAPI_RETURN_CODE_SUCCESS)
        {
            break;
        }

        // Add TuneMix channel tag
        eReturnCode = TAG_eAdd(
            TUNEMIX_TAG_NAME,
            hParentTag,
            &hTag, 
            (const char *)NULL);

        if (eReturnCode != SMSAPI_RETURN_CODE_SUCCESS)
        {
            break;
        }

        // Save Tune Mix Tag Locally
        psObj->hTag = hTag;

        // Get Index flag tag
        eReturnCode = TAG_eAdd(
            TUNEMIX_INDEX_TAG_NAME,
            psObj->hTag,
            &hTag,
            (char *)NULL );

        if (eReturnCode != SMSAPI_RETURN_CODE_SUCCESS)
        {
            break;
        }

        // Not active initially
        n32Value = (N32)psObj->un8Index;

        eReturnCode = TAG_eSetTagValue(
            hTag,
            TAG_TYPE_INTEGER,
            (void *)&n32Value,
            sizeof(N32),
            TRUE);

        if (eReturnCode != SMSAPI_RETURN_CODE_SUCCESS)
        {
            break;
        }

        // Get Name flag tag
        eReturnCode = TAG_eAdd(
            TUNEMIX_NAME_TAG_NAME,
            psObj->hTag,
            &hTag,
            (char *)NULL );

        if (eReturnCode != SMSAPI_RETURN_CODE_SUCCESS)
        {
            break;
        }

        hString = psObj->hName;

        eReturnCode = TAG_eSetTagValue(
            hTag,
            TAG_TYPE_STRING,
            (void *)&hString,
            sizeof(STRING_OBJECT),
            TRUE);

        if (eReturnCode != SMSAPI_RETURN_CODE_SUCCESS)
        {
            break;
        }

        // Get Active flag tag
        eReturnCode = TAG_eAdd(
            TUNEMIX_ACTIVE_TAG_NAME,
            psObj->hTag,
            &hTag,
            (char *)NULL );

        if (eReturnCode != SMSAPI_RETURN_CODE_SUCCESS)
        {
            break;
        }

        // Not active initially
        n32Value = (N32)FALSE;

        eReturnCode = TAG_eSetTagValue(
            hTag,
            TAG_TYPE_INTEGER,
            (void *)&n32Value,
            sizeof(N32),
            TRUE);

        if (eReturnCode != SMSAPI_RETURN_CODE_SUCCESS)
        {
            break;
        }

        // save overrides
        bReturn = bSaveOverrides(psObj->hTag, psObj->bOverride, psObj->bPlayUnrestricted);

    } while (FALSE);

    return bReturn;
}

/*************************************************************************
 *
 *   bSaveOverrides
 *
 *************************************************************************/
static BOOLEAN bSaveOverrides (
    TAG_OBJECT hParentTag,
    BOOLEAN bOverride,
    BOOLEAN bPlayUnrestricted
        )
{
    TAG_OBJECT hTag;
    BOOLEAN bSuccess = FALSE;
    SMSAPI_RETURN_CODE_ENUM eReturnCode;

    // Get Overrides Tag
    eReturnCode = TAG_eGet(
        TUNEMIX_OVERRIDES_TAG_NAME,
        hParentTag,
        &hTag,
        (char *)NULL,
        TRUE);

    if (eReturnCode == SMSAPI_RETURN_CODE_SUCCESS)
    {
        N32 n32Value = 0;

        if (TRUE == bOverride)
        {
            n32Value |= TUNEMIX_OVERRIDE;
        }
        if (TRUE == bPlayUnrestricted)
        {
            n32Value |= TUNEMIX_PLAY_UNRESTRICTED;
        }

        eReturnCode = TAG_eSetTagValue(
            hTag,
            TAG_TYPE_INTEGER,
            (void *)&n32Value,
            sizeof(N32),
            TRUE);

        if (eReturnCode == SMSAPI_RETURN_CODE_SUCCESS)
        {
            bSuccess = TRUE;
        }
    }

    return bSuccess;
}

/*************************************************************************
 *
 *   bSaveActive
 *
 *************************************************************************/
static BOOLEAN bSaveActive (
    TAG_OBJECT hParentTag,
    BOOLEAN bActive
        )
{
    TAG_OBJECT hTag;
    BOOLEAN bSuccess = FALSE;
    SMSAPI_RETURN_CODE_ENUM eReturnCode;

    // Get Index Tag
    eReturnCode = TAG_eGet(
        TUNEMIX_ACTIVE_TAG_NAME,
        hParentTag,
        &hTag,
        (char *)NULL,
        TRUE);

    if (eReturnCode == SMSAPI_RETURN_CODE_SUCCESS)
    {
        N32 n32Value = (N32)bActive;

        eReturnCode = TAG_eSetTagValue(
            hTag,
            TAG_TYPE_INTEGER,
            (void *)&n32Value,
            sizeof(N32),
            TRUE);

        if (eReturnCode == SMSAPI_RETURN_CODE_SUCCESS)
        {
            bSuccess = TRUE;
        }
    }

    return bSuccess;
}

/*************************************************************************
 *
 *   bSaveName
 *
 *************************************************************************/
static BOOLEAN bSaveName (
    TAG_OBJECT hParentTag,
    STRING_OBJECT hName
        )
{
    TAG_OBJECT hTag;
    BOOLEAN bSuccess = FALSE;
    SMSAPI_RETURN_CODE_ENUM eReturnCode;

    // Get Name Tag
    eReturnCode = TAG_eGet(
        TUNEMIX_NAME_TAG_NAME,
        hParentTag,
        &hTag,
        (char *)NULL,
        TRUE);

    if (eReturnCode == SMSAPI_RETURN_CODE_SUCCESS)
    {
        eReturnCode = TAG_eSetTagValue(
            hTag,
            TAG_TYPE_STRING,
            (void *)&hName,
            sizeof(STRING_OBJECT),
            TRUE);

        if (eReturnCode == SMSAPI_RETURN_CODE_SUCCESS)
        {
            bSuccess = TRUE;
        }
    }

    return bSuccess;
}

/*************************************************************************
*
*   bRestoreIterator
*
**************************************************************************/
static BOOLEAN bRestoreIterator (
    TAG_OBJECT hTag,
    void *pvArg
        )
{
    size_t tBytesAvail = sizeof(N32);
    SMSAPI_RETURN_CODE_ENUM eReturnCode;
    N32 n32Value, *pn32Value = &n32Value;
    TUNEMIX_RESTORE_STRUCT *psData =
        (TUNEMIX_RESTORE_STRUCT *)pvArg;

    // Get Tag value
    eReturnCode = TAG_eGetTagValue(hTag, TAG_TYPE_INTEGER,
                      (void **)&pn32Value, &tBytesAvail);
    if (eReturnCode != SMSAPI_RETURN_CODE_SUCCESS)
    {
        psData->bSuccess = FALSE;
        return FALSE;
    }

    // Assign Service Id value
    psData->atServiceId[psData->un16ChIndex] = (SERVICE_ID)n32Value;

    // Increment index
    psData->un16ChIndex++;

    return TRUE;
}

/*************************************************************************
*
*   bTMIterator
*
**************************************************************************/
static BOOLEAN bTMIterator (
    TAG_OBJECT hTagTMChannel,
    void * pvArg
        )
{
    size_t tBytesAvail = sizeof(N32);
    BOOLEAN bOk = FALSE;
    SMSAPI_RETURN_CODE_ENUM eReturnCode;
    N32 n32Value, *pn32Value = &n32Value;
    TAG_OBJECT hTagActive;
    SERVICE_ID *ptServiceId =
        (SERVICE_ID *)pvArg;

    do
    {
        // Get Active Tag
        eReturnCode = TAG_eGet(
            TUNEMIX_ACTIVE_TAG_NAME,
            hTagTMChannel,
            &hTagActive,
            (char *)NULL,
            FALSE);

        if ((eReturnCode != SMSAPI_RETURN_CODE_SUCCESS) ||
            (hTagActive == TAG_INVALID_OBJECT))
        {
            break;
        }

        // Get Active value
        eReturnCode = TAG_eGetTagValue(
            hTagActive,
            TAG_TYPE_INTEGER,
            (void **)&pn32Value,
            &tBytesAvail
                );
        if (eReturnCode != SMSAPI_RETURN_CODE_SUCCESS)
        {
            break;
        }

        if ((BOOLEAN)n32Value == FALSE)
        {
            bOk = TRUE;
            break;
        }

        //This TuneMix channel is active - to retrieve first SID component
        // Get Services array Tag
        eReturnCode = TAG_eGet(
            TUNEMIX_COMPONENTS_TAG_NAME,
            hTagTMChannel,
            &hTagActive,
            (char *)NULL,
            FALSE);

        if ((eReturnCode != SMSAPI_RETURN_CODE_SUCCESS) ||
            (hTagActive == TAG_INVALID_OBJECT))
        {
            break;
        }

        eReturnCode = TAG_eFirstChild(hTagActive, &hTagActive);
        if ((eReturnCode != SMSAPI_RETURN_CODE_SUCCESS) &&
            (hTagActive == TAG_INVALID_OBJECT))
        {
            break;
        }

        tBytesAvail = sizeof(N32);
        // Get Tag value
        eReturnCode = TAG_eGetTagValue(hTagActive, TAG_TYPE_INTEGER,
            (void **)&pn32Value, &tBytesAvail);
        if (eReturnCode != SMSAPI_RETURN_CODE_SUCCESS)
        {
            break;
        }
        // Assign Service Id value
        *ptServiceId = (SERVICE_ID)n32Value;
    } while( FALSE );

    return bOk;
}

/*************************************************************************
*
*   bChannelIterator
*
**************************************************************************/
static BOOLEAN bChannelIterator (
    CHANNEL_ID tChannelId,
    void *pvArg
        )
{
    TUNEMIX_COMPONENTS_ITERATOR_STRUCT *psIteratorStruct =
        (TUNEMIX_COMPONENTS_ITERATOR_STRUCT*)pvArg;
    CHANNEL_OBJECT hChannel;
    BOOLEAN bContinue = TRUE;

    hChannel = CCACHE_hChannelFromIds(
            psIteratorStruct->hCCache, SERVICE_INVALID_ID,
            tChannelId, FALSE);

    if (hChannel == CHANNEL_INVALID_OBJECT)
    {
        printf(TUNEMIX_OBJECT_NAME
            ": Cannot get handle for channel %d \n",
            tChannelId);
    }
    else
    {
        bContinue = psIteratorStruct->bIterateFxn(
            hChannel,
            psIteratorStruct->pvArg);
    }

    return bContinue;
}

/*************************************************************************
*
*   vUninitObject
*
**************************************************************************/
static void vUninitObject(
    TUNEMIX_OBJECT_STRUCT *psObj
        )
{
    TAG_eRemove(psObj->hTag, TRUE);
    TUNEMIX_vDeallocate((TUNEMIX_OBJECT)psObj);

    return;
}

/*************************************************************************
 *
 *   psCreateObject
 *
 *************************************************************************/
static TUNEMIX_OBJECT_STRUCT *psCreateObject (
    DECODER_OBJECT hDecoder,
    CHANNEL_ID tChannelId
        )
{
    TUNEMIX_OBJECT_STRUCT *psObj;
    BOOLEAN bOk = FALSE;
    CHANNEL_ID tChannelIdFromCache;
    CHANNEL_OBJECT hChannel;

    // Create the instance
    psObj = (TUNEMIX_OBJECT_STRUCT*)SMSO_hCreate(
        TUNEMIX_OBJECT_NAME":Obj",
        sizeof(TUNEMIX_OBJECT_STRUCT),
        (SMS_OBJECT)hDecoder, FALSE);

    do
    {
        OSAL_RETURN_CODE_ENUM eReturnCode;
        CCACHE_OBJECT hCCache;

        if (psObj == NULL)
        {
            break;
        }

        // Create the list of components
        eReturnCode = OSAL.eLinkedListCreate(
            &psObj->hComponentList,
            TUNEMIX_OBJECT_NAME":hComponentList",
            (OSAL_LL_COMPARE_HANDLER)NULL,
            OSAL_LL_OPTION_LINEAR | OSAL_LL_OPTION_UNIQUE);
        if (eReturnCode != OSAL_SUCCESS)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                TUNEMIX_OBJECT_NAME
                ": failed to create Component List (%s)",
                OSAL.pacGetReturnCodeName(eReturnCode)
                );
            break;
        }

        // Get the ccache attached to this decoder
        hCCache = DECODER_hCCache(hDecoder);

        // If we failed, we cannot create the channel
        if ( hCCache == CCACHE_INVALID_OBJECT )
        {
            break;
        }

        // create a fake channel
        hChannel = CCACHE_hChannelFromIds(hCCache,
            (SERVICE_ID)tChannelId, tChannelId, TRUE);
        // Failed to create channel for some reason
        if (hChannel == CHANNEL_INVALID_OBJECT)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                TUNEMIX_OBJECT_NAME
                ": Failed to create TuneMix channel");
            break;
        }
        tChannelIdFromCache = CHANNEL.tChannelId(hChannel);
        if (tChannelIdFromCache == CHANNEL_INVALID_ID)
        {
            // Channel with current Service Id was found in the cache.
            // Channel Id is needed to update.
            bOk = CHANNEL_bUpdateChannelId(hChannel, tChannelId);
            if (bOk == FALSE)
            {
                puts( TUNEMIX_OBJECT_NAME
                    ": Failed to update TuneMix channel Id.");
                break;
            }
        }
        psObj->hChannel = hChannel;
        psObj->hCCache = hCCache;
        bOk = TRUE;
    } while (FALSE);

    if ((bOk == FALSE) && (psObj != NULL))
    {
        if (psObj->hComponentList != OSAL_INVALID_OBJECT_HDL)
        {
            OSAL.eLinkedListDelete(psObj->hComponentList);
        }
        SMSO_vDestroy((SMS_OBJECT)psObj);
        psObj = (TUNEMIX_OBJECT_STRUCT*)NULL;
    }

    return psObj;
}

/*************************************************************************
 *
 *   n16TuneMixRemoveIterator
 *
 *************************************************************************/
static N16 n16TuneMixRemoveIterator (
    PRESETS_OBJECT hPresets,
    PRESET_BAND_OBJECT hBand,
    size_t tBandIndex,
    CHANNEL_OBJECT hChannel,
    STRING_OBJECT hPresetName,
    size_t tPresetIndex,
    void *pvIterateArg
        )
{
    TUNEMIX_PRESETS_ITERATOR_STRUCT *psIterator = 
        (TUNEMIX_PRESETS_ITERATOR_STRUCT *)pvIterateArg;
    SERVICE_ID tSID;

    // If the preset index wrapped back to zero ...
    if (tPresetIndex < psIterator->tLastPresetIndex)
    {
        // ... and band index also wrapped or the same (for alone band case) ...
        if (tBandIndex <= psIterator->tLastBandIndex)
        {
            // ... it means we have gone through all the presets. So exiting.
            return 0;
        }
    }

    // Storing last iterated preset
    psIterator->tLastBandIndex = tBandIndex;
    psIterator->tLastPresetIndex = tPresetIndex;

    // Is this the SID we are looking for?
    tSID = CHANNEL.tServiceId(hChannel);

    if (tSID == psIterator->tSID)
    {
        // Removing this channel from presets.
        // This is because unlike normal channels TuneMix will not 
        // contain the same (or similar) content on the same SID
        // upon the next creation.
        BAND.eSetPreset(hBand, tPresetIndex, CHANNEL_INVALID_ID);
    }
    
    // Continue iteration
    return 1;
}
