/******************************************************************************/
/*                    Copyright (c) Sirius XM Radio, Inc.                     */
/*                            All Rights Reserved                             */
/*         Licensed Materials - Property of Sirius XM Radio, Inc.             */
/******************************************************************************/
/*******************************************************************************
 *
 * DESCRIPTION
 *
 *  This module contains the Object:TRAFFIC_MSG implementation for the
 *  Sirius Module Services (SMS)
 *
 ******************************************************************************/
#include <string.h>

#include "standard.h"
#include "osal.h"

#include "sms_api.h"
#include "sms_obj.h"
#include "decoder_obj.h"
#include "dataservice_mgr_obj.h"
#include "string_obj.h"
#include "location_obj.h"
#include "alertc_event_obj.h"
#include "traffic_mgr_obj.h"
#include "traffic_db_constants.h"
#include "traffic_msg_obj.h"
#include "_traffic_msg_obj.h"
#include "traffic_msg_base_obj.h"

#include "sms_api_debug.h"
static const char *gpacThisFile = __FILE__;

/*****************************************************************************
                             PUBLIC FUNCTIONS
*****************************************************************************/

/*****************************************************************************
*
*   eType
*
*****************************************************************************/
static TRAFFIC_MSG_TYPE_ENUM eType (
    TRAFFIC_MSG_BASE_OBJECT hTrafficBaseMsg
        )
{
    TRAFFIC_MSG_OBJECT_STRUCT *psObj =
        (TRAFFIC_MSG_OBJECT_STRUCT *)hTrafficBaseMsg;
    TRAFFIC_MSG_TYPE_ENUM eType  = TRAFFIC_MSG_TYPE_UNKNOWN;
    ALERTC_EVENT_OBJECT hEventInfo = hGetEventInfo(psObj);
    ALERTC_EVENT_TYPE_ENUM eAlertCType = ALERTC_EVENT_eType(hEventInfo);

    switch (eAlertCType)
    {
        case ALERTC_EVENT_TYPE_INCIDENT:
            eType  = TRAFFIC_MSG_TYPE_INCIDENT;
            break;
        case ALERTC_EVENT_TYPE_SPEED_AND_FLOW:
            eType  = TRAFFIC_MSG_TYPE_SPEED_AND_FLOW;
            break;
        case ALERTC_EVENT_TYPE_UNKNOWN:
        default:
            eType = TRAFFIC_MSG_TYPE_UNKNOWN;
            break;
    }

    return eType;
}

/*****************************************************************************
*
*   eClass
*
*****************************************************************************/
static TRAFFIC_MSG_CLASS_ENUM eClass (
    TRAFFIC_MSG_BASE_OBJECT hTrafficBaseMsg
        )
{
    TRAFFIC_MSG_OBJECT_STRUCT *psObj =
        (TRAFFIC_MSG_OBJECT_STRUCT *)hTrafficBaseMsg;
    TRAFFIC_MSG_CLASS_ENUM eClass  = TRAFFIC_MSG_CLASS_UNKNOWN;
    ALERTC_EVENT_OBJECT hEventInfo;

    hEventInfo = hGetEventInfo(psObj);

    if (hEventInfo != ALERTC_EVENT_INVALID_OBJECT)
    {
        eClass = ALERTC_EVENT_eClass(hEventInfo);
    }

    return eClass;
}

/*****************************************************************************
*
*   eSpeed
*
*****************************************************************************/
static TRAFFIC_SPEED_ENUM eSpeed (
    TRAFFIC_MSG_BASE_OBJECT hTrafficBaseMsg
        )
{
    TRAFFIC_MSG_OBJECT_STRUCT *psObj =
        (TRAFFIC_MSG_OBJECT_STRUCT *)hTrafficBaseMsg;
    TRAFFIC_SPEED_ENUM eSpeed  = TRAFFIC_SPEED_UNKNOWN;
    TRAFFIC_MSG_TYPE_ENUM eType = LEGACY_TRAFFIC_MSG.eType(hTrafficBaseMsg);

    if (eType == TRAFFIC_MSG_TYPE_SPEED_AND_FLOW)
    {
        // Event code to speed mapping as defined in RX96 Table 3
        switch (psObj->sAlertC.tEventCode)
        {
            case 70:
                eSpeed = TRAFFIC_SPEED_5MPH;
            break;

            case 71:
                eSpeed = TRAFFIC_SPEED_10MPH;
            break;

            case 72:
                eSpeed = TRAFFIC_SPEED_20MPH;
            break;

            case 73:
                eSpeed = TRAFFIC_SPEED_25MPH;
            break;

            case 74:
                eSpeed = TRAFFIC_SPEED_30MPH;
            break;

            case 75:
                eSpeed = TRAFFIC_SPEED_40MPH;
            break;

            case 76:
                eSpeed = TRAFFIC_SPEED_45MPH;
            break;

            case 124:
                eSpeed = TRAFFIC_SPEED_FLOWING_FREELY;
            break;

            default:
                eSpeed = TRAFFIC_SPEED_UNKNOWN;
            break;

        }
    }

    return eSpeed;
}

/*****************************************************************************
*
*   eDuration
*
*****************************************************************************/
static TRAFFIC_DURATION_ENUM eDuration (
    TRAFFIC_MSG_BASE_OBJECT hTrafficBaseMsg
        )
{
    TRAFFIC_MSG_OBJECT_STRUCT *psObj =
        (TRAFFIC_MSG_OBJECT_STRUCT *)hTrafficBaseMsg;
    TRAFFIC_DURATION_ENUM eDuration;

    eDuration = (TRAFFIC_DURATION_ENUM)psObj->sAlertC.un8Duration;

    return eDuration;
}

/*****************************************************************************
*
*   eDirection
*
*****************************************************************************/
static TRAFFIC_DIRECTION_ENUM eDirection (
    TRAFFIC_MSG_BASE_OBJECT hTrafficBaseMsg
        )
{
    TRAFFIC_MSG_OBJECT_STRUCT *psObj =
        (TRAFFIC_MSG_OBJECT_STRUCT *)hTrafficBaseMsg;
    TRAFFIC_DIRECTION_ENUM eDirection;

    eDirection = psObj->sAlertC.eDirection;

    return eDirection;
}

/*****************************************************************************
*
*   bDiversionAdvised
*
*****************************************************************************/
static BOOLEAN bDiversionAdvised (
    TRAFFIC_MSG_BASE_OBJECT hTrafficBaseMsg
        )
{
    BOOLEAN bDiversionAdvised;
    TRAFFIC_MSG_OBJECT_STRUCT *psObj =
        (TRAFFIC_MSG_OBJECT_STRUCT *)hTrafficBaseMsg;

    bDiversionAdvised = psObj->sAlertC.bDiversionAdvised;

    return bDiversionAdvised;
}

/*****************************************************************************
*
*   tEventCode
*
*****************************************************************************/
static TRAFFIC_EVENT_CODE tEventCode (
    TRAFFIC_MSG_BASE_OBJECT hTrafficBaseMsg
        )
{
    TRAFFIC_EVENT_CODE tEventCode;
    TRAFFIC_MSG_OBJECT_STRUCT *psObj =
        (TRAFFIC_MSG_OBJECT_STRUCT *)hTrafficBaseMsg;

    tEventCode = psObj->sAlertC.tEventCode;

    return tEventCode;
}

/*****************************************************************************
*
*   hText
*
*****************************************************************************/
static STRING_OBJECT hText (
    TRAFFIC_MSG_BASE_OBJECT hTrafficBaseMsg
        )
{
    STRING_OBJECT hText;
    ALERTC_EVENT_OBJECT hEventInfo;
    TRAFFIC_MSG_OBJECT_STRUCT *psObj =
        (TRAFFIC_MSG_OBJECT_STRUCT *)hTrafficBaseMsg;

    hEventInfo = hGetEventInfo(psObj);
    hText = ALERTC_EVENT_hText(hEventInfo);

    // Check to see if we need to use a modified text instead
    if (psObj->hTextMod == STRING_INVALID_OBJECT)
    {
        STRING_OBJECT hTextMod;

        if (hEventInfo == ALERTC_EVENT_INVALID_OBJECT)
        {
            // ERROR!
            return STRING_INVALID_OBJECT;
        }

        hTextMod = ALERTC_EVENT_hTextMod(hEventInfo);

        if (hTextMod != STRING_INVALID_OBJECT )
        {
            UN16 aun16Quantifiers[MAX_SUPPLEMENTARY_INFO_CODES];
            UN8 un8NumQuantifiers = 0;
            ALERTC_QUANTIFIER_TYPE_ENUM eQuantType;

            eQuantType = ALERTC_EVENT_eQuantType(hEventInfo);

            if (eQuantType < ALERTC_QUANTIFIER_TYPE_TEMP)
            {
                un8NumQuantifiers =
                    LEGACY_TRAFFIC_MSG.un8GetFreeFormData(
                        (TRAFFIC_MSG_BASE_OBJECT)psObj,
                        ALERTC_FREEFORM_DATA_LABEL_QUANTIFIER_5,
                        &aun16Quantifiers[0],
                        MAX_SUPPLEMENTARY_INFO_CODES);
            }
            else
            {
                un8NumQuantifiers =
                    LEGACY_TRAFFIC_MSG.un8GetFreeFormData(
                        (TRAFFIC_MSG_BASE_OBJECT)psObj,
                        ALERTC_FREEFORM_DATA_LABEL_QUANTIFIER_8,
                        &aun16Quantifiers[0],
                        MAX_SUPPLEMENTARY_INFO_CODES);
            }

            if (un8NumQuantifiers > 0)
            {
                psObj->hTextMod =
                    hBuildTextWithQuantifiers(
                        psObj, hTextMod, aun16Quantifiers[0]);

                hText = psObj->hTextMod;
            }
        }
    }

    return hText;
}

/*****************************************************************************
*
*   hAdditionalText
*
*****************************************************************************/
static STRING_OBJECT hAdditionalText (
    TRAFFIC_MSG_BASE_OBJECT hTrafficBaseMsg
        )
{
    TRAFFIC_MSG_OBJECT_STRUCT *psObj =
        (TRAFFIC_MSG_OBJECT_STRUCT *)hTrafficBaseMsg;
    STRING_OBJECT hAdditionalText = STRING_INVALID_OBJECT;

    if (psObj->hAdditionalText == STRING_INVALID_OBJECT)
    {
        ALERTC_SUPPL_INFO_CODE atCodes[MAX_SUPPLEMENTARY_INFO_CODES];
        ALERTC_SUPPL_INFO_CODE tCurCode;
        const char *pcCurText;
        UN8 un8NumCodes, un8Index;
        TRAFFIC_SERVICE_OBJECT hTrafficService;

        // Get the traffic service handle in order to look-up
        // the text
        hTrafficService = hGetTrafficMgr(psObj);

        // Clear our array
        OSAL.bMemSet(&atCodes[0], 0, sizeof(atCodes));

        // See if we have any codes
        un8NumCodes =
            LEGACY_TRAFFIC_MSG.un8GetFreeFormData(
                (TRAFFIC_MSG_BASE_OBJECT)psObj,
                ALERTC_FREEFORM_DATA_LABEL_SUPPL_INFO,
                (UN16 *)&atCodes[0],
                MAX_SUPPLEMENTARY_INFO_CODES);

        for (un8Index = 0;
                un8Index < un8NumCodes;
                un8Index++)
        {
            tCurCode = atCodes[un8Index];

            pcCurText = TRAFFIC_MGR_pcGetSupplInfoText(
                hTrafficService, tCurCode);

            if (pcCurText != NULL)
            {
                size_t tNumAppended;

                if (psObj->hAdditionalText == STRING_INVALID_OBJECT)
                {
                    psObj->hAdditionalText =
                        STRING_hCreate(SMS_INVALID_OBJECT,
                            pcCurText, strlen(pcCurText),
                            un8NumCodes * strlen(pcCurText));

                    if (psObj->hAdditionalText == STRING_INVALID_OBJECT)
                    {
                        // We can't create the string for some reason
                        // so error out
                        break;
                    }
                    else
                    {
                        // Skip the rest of the loop -- we did our
                        // work already
                        continue;
                    }
                }

                tNumAppended = STRING.tAppendCStr(
                    psObj->hAdditionalText, pcCurText);

                if (tNumAppended != strlen(pcCurText))
                {
                    STRING_vDestroy(psObj->hAdditionalText);
                    break;
                }

                if (un8Index < un8NumCodes-1)
                {
                    tNumAppended = STRING.tAppendCStr(
                        psObj->hAdditionalText, " ");

                    if (tNumAppended != 1)
                    {
                        STRING_vDestroy(psObj->hAdditionalText);
                        break;
                    }
                }
            }
        }
    }

    hAdditionalText = psObj->hAdditionalText;

    return hAdditionalText;
}

/*****************************************************************************
*
*   un32StartTime
*
******************************************************************************/
static UN32 un32StartTime (
    TRAFFIC_MSG_BASE_OBJECT hTrafficBaseMsg
        )
{
    UN32 un32StartTime = 0;
    TRAFFIC_MSG_OBJECT_STRUCT *psObj =
        (TRAFFIC_MSG_OBJECT_STRUCT *)hTrafficBaseMsg;

    un32StartTime = un32GetStartStopTime(psObj,
        ALERTC_FREEFORM_DATA_LABEL_EXPLICIT_START_TIME);

    return un32StartTime;
}

/*****************************************************************************
*
*   un32StopTime
*
******************************************************************************/
static UN32 un32StopTime (
    TRAFFIC_MSG_BASE_OBJECT hTrafficBaseMsg
        )
{
    UN32 un32StopTime = 0;
    TRAFFIC_MSG_OBJECT_STRUCT *psObj =
        (TRAFFIC_MSG_OBJECT_STRUCT *)hTrafficBaseMsg;

    un32StopTime = un32GetStartStopTime(psObj,
        ALERTC_FREEFORM_DATA_LABEL_EXPLICIT_STOP_TIME);

    return un32StopTime;
}

/*****************************************************************************
*
*   un8GetFreeFormData
*
*****************************************************************************/
static UN8 un8GetFreeFormData (
    TRAFFIC_MSG_BASE_OBJECT hTrafficBaseMsg,
    ALERTC_FREEFORM_DATA_LABEL_ENUM eLabel,
    UN16 *paun16Values,
    size_t tCount
        )
{
    UN8 un8NumFound = 0;
    OSAL_RETURN_CODE_ENUM eReturnCode = OSAL_SUCCESS;
    OSAL_LINKED_LIST_ENTRY hEntry = OSAL_INVALID_LINKED_LIST_ENTRY;
    ALERTC_FREEFORM_DATA_STRUCT *psFreeFormData;
    TRAFFIC_MSG_OBJECT_STRUCT *psObj =
        (TRAFFIC_MSG_OBJECT_STRUCT *)hTrafficBaseMsg;

    while (eReturnCode == OSAL_SUCCESS &&
            un8NumFound < tCount)
    {
        eReturnCode = OSAL.eLinkedListLinearSearch(
            psObj->sAlertC.hFreeFormDataList, &hEntry,
            (OSAL_LL_COMPARE_HANDLER)n16IsFreeFormDataEqual,
            (void *) eLabel);

        if (eReturnCode == OSAL_SUCCESS)
        {
            psFreeFormData =
                (ALERTC_FREEFORM_DATA_STRUCT *) OSAL.pvLinkedListThis(hEntry);

            paun16Values[un8NumFound] = psFreeFormData->un16Value;
            un8NumFound++;
        }
    } // while (eReturnCode == OSAL_SUCCESS)

    return un8NumFound;
}

/*****************************************************************************
*
*   un8NumLocations
*
*****************************************************************************/
static UN8 un8NumLocations (
    TRAFFIC_MSG_BASE_OBJECT hTrafficBaseMsg
        )
{
    UN32 un32ListSize = 0;
    TRAFFIC_MSG_OBJECT_STRUCT *psObj =
        (TRAFFIC_MSG_OBJECT_STRUCT *)hTrafficBaseMsg;

    OSAL.eLinkedListItems(psObj->hLocationList, &un32ListSize);

    return (UN8)un32ListSize;
}

/*****************************************************************************
*
*   hGetLocation
*
*****************************************************************************/
static LOCATION_OBJECT hGetLocation (
    TRAFFIC_MSG_BASE_OBJECT hTrafficBaseMsg,
    UN8 un8Index
        )
{
    UN32 un32ListSize = 0;
    LOCATION_OBJECT hLocation = LOCATION_INVALID_OBJECT;
    OSAL_RETURN_CODE_ENUM eReturnCode;
    UN8 un8LocIdx;
    OSAL_LINKED_LIST_ENTRY hEntry = OSAL_INVALID_LINKED_LIST_ENTRY;
    TRAFFIC_MSG_OBJECT_STRUCT *psObj =
        (TRAFFIC_MSG_OBJECT_STRUCT *)hTrafficBaseMsg;

    // Only add a location if our current location count is less than un8MaxNumLocations
    eReturnCode = OSAL.eLinkedListItems(psObj->hLocationList, &un32ListSize);

    if (eReturnCode != OSAL_SUCCESS)
    {
        // Error!
        return LOCATION_INVALID_OBJECT;
    }

    if (un8Index >= (UN8)un32ListSize)
    {
        // Error!
        return LOCATION_INVALID_OBJECT;
    }

    hEntry = OSAL.hLinkedListFirst(psObj->hLocationList, (void**)NULL);

    if (hEntry == OSAL_INVALID_LINKED_LIST_ENTRY)
    {
        // Error!
        return LOCATION_INVALID_OBJECT;
    }

    for (un8LocIdx = 0; un8LocIdx < un8Index; un8LocIdx++)
    {
        hEntry = OSAL.hLinkedListNext(hEntry, NULL);
    }

    if (hEntry != OSAL_INVALID_LINKED_LIST_ENTRY)
    {
        hLocation = (LOCATION_OBJECT)OSAL.pvLinkedListThis(hEntry);
    }

    return hLocation;
}


/*****************************************************************************
*
*   n32FPrintf
*
*****************************************************************************/
static N32 n32FPrintf (
    TRAFFIC_MSG_BASE_OBJECT hTrafficBaseMsg,
    FILE *psFile,
    SMSAPI_OUTPUT_OPTION_ENUM eOutputOption
        )
{
    N32 n32Return = 0, n32TmpReturn = 0;
    LOCATION_OBJECT hStartLoc, hEndLoc;
    TRAFFIC_LOCID_OBJECT hStartLocID, hEndLocID;
    LOC_ID tStartID, tEndID;
    OSAL_LINKED_LIST_ENTRY hEntry = OSAL_INVALID_LINKED_LIST_ENTRY;
    TRAFFIC_MSG_OBJECT_STRUCT *psObj =
        (TRAFFIC_MSG_OBJECT_STRUCT *)hTrafficBaseMsg;
    TRAFFIC_DIRECTION_ENUM eDirection =
        LEGACY_TRAFFIC_MSG.eDirection(hTrafficBaseMsg);
    TRAFFIC_DURATION_ENUM eDuration =
        LEGACY_TRAFFIC_MSG.eDuration(hTrafficBaseMsg);
    TRAFFIC_MSG_TYPE_ENUM eType = LEGACY_TRAFFIC_MSG.eType(hTrafficBaseMsg);
    UN8 un8NumLoc = LEGACY_TRAFFIC_MSG.un8NumLocations(hTrafficBaseMsg);
    BOOLEAN bDiversionAdvised =
        LEGACY_TRAFFIC_MSG.bDiversionAdvised(hTrafficBaseMsg);
    STRING_OBJECT hEventText =
            LEGACY_TRAFFIC_MSG.hText(hTrafficBaseMsg);
    STRING_OBJECT hAdditionalText =
            LEGACY_TRAFFIC_MSG.hAdditionalText(hTrafficBaseMsg);
    TRAFFIC_EVENT_CODE tEventCode =
            LEGACY_TRAFFIC_MSG.tEventCode(hTrafficBaseMsg);
    TRAFFIC_MSG_CLASS_ENUM eClass = LEGACY_TRAFFIC_MSG.eClass(hTrafficBaseMsg);
    UN32 un32ExpStartTime = LEGACY_TRAFFIC_MSG.un32StartTime(hTrafficBaseMsg);
    UN32 un32ExpStopTime = LEGACY_TRAFFIC_MSG.un32StopTime(hTrafficBaseMsg);
    char acEventText[MAX_EVENT_TEXT_OUTPUT_LENGTH];
    size_t tBytesCopied;

    // Verify inputs
    if (psFile == NULL)
    {
        // Error!
        return EOF;
    }

    tBytesCopied =
            STRING.tCopyToCStr(hEventText, &acEventText[0], sizeof(acEventText));

    if (tBytesCopied == sizeof(acEventText))
    {
        acEventText[tBytesCopied-1] = '\0';
    }

    hEntry =
            OSAL.hLinkedListFirst(psObj->hLocationList, (void**)&hStartLoc);

    if (hEntry == OSAL_INVALID_LINKED_LIST_ENTRY)
    {
        return EOF;
    }

    hEntry =
        OSAL.hLinkedListLast(psObj->hLocationList, (void**)&hEndLoc);

    if (hEntry == OSAL_INVALID_LINKED_LIST_ENTRY)
    {
        return EOF;
    }

    hStartLocID = LOCATION.hLocID(hStartLoc);
    tStartID= LOCID.tID(hStartLocID);
    hEndLocID = LOCATION.hLocID(hEndLoc);
    tEndID= LOCID.tID(hEndLocID);

    switch(eOutputOption)
    {
        case SMS_OUTPUT_OPTION_TERSE:
        {
            n32Return +=
                    fprintf(psFile,
                    "| %-4s | %5d | %-3s |   %2d   |  %6d  |  %6d  | %-28s |\n",
                    pacTypeText(eType, eOutputOption),
                    tEventCode,
                    pacDirectionText(eDirection, eOutputOption),
                    un8NumLoc,
                    tStartID,
                    tEndID,
                    acEventText);
        }
        break;
        case SMS_OUTPUT_OPTION_VERBOSE:
        case SMS_OUTPUT_OPTION_GROSS:
        {
            // Print TRAFFIC_MSG information
            n32Return += fprintf(psFile, "TRAFFIC_MSG: hTrafficMsg = 0x%p\n",
                    psObj);

            n32Return += fprintf(psFile, "\tMessage Type = %s\n",
                    pacTypeText(LEGACY_TRAFFIC_MSG.eType(hTrafficBaseMsg),
                    eOutputOption));

            n32Return += fprintf(psFile, "\tMessage Class = %s\n",
                    pacClassText(eClass));

            n32Return += fprintf(psFile, "\tDuration = %s\n",
                    pacDurationText(eDuration, eOutputOption));

            if (un32ExpStartTime > 0)
            {
                char cBuf[OSAL_ASCBUFSIZE];
                TIME_T tExpStartTime = un32ExpStartTime;
                n32Return += fprintf(psFile, "\tExp. Start Time = %s\n",
                        OSAL.ctime_r(&tExpStartTime, cBuf));
            }

            if (un32ExpStopTime > 0)
            {
                char cBuf[OSAL_ASCBUFSIZE];
                TIME_T tExpStopTime = un32ExpStopTime;
                n32Return += fprintf(psFile, "\tExp. Stop Time = %s\n",
                        OSAL.ctime_r(&tExpStopTime, cBuf));
            }

            n32Return += fprintf(psFile, "\tDirection = %s\n",
                    pacDirectionText(eDirection, eOutputOption));

            n32Return += fprintf(psFile, "\tDiversionAdvised? = %s\n",
                    bDiversionAdvised ? "Yes" : "No");

            n32Return += fprintf(psFile, "\tEventCode = %d\n",
                    tEventCode);

            n32Return += fprintf(psFile, "\thText = ");
            n32Return += STRING.n32FWrite(hEventText, psFile);
            n32Return += fprintf(psFile, "\n");

            n32Return += fprintf(psFile, "\thAdditionalText = ");
            n32TmpReturn = STRING.n32FWrite(hAdditionalText, psFile);
            if (n32TmpReturn < 0)
            {
                n32Return += fprintf(psFile, "ERROR GETTING ADDITIONAL TEXT");
            }
            else
            {
                n32Return += n32TmpReturn;
            }

            n32Return += fprintf(psFile, "\n");

            n32Return += fprintf(psFile, "\tNum Locations = %d\n",
                    un8NumLoc);

            n32Return += fprintf(psFile, "\tStart Location ID = %d\n",
                    tStartID);

            n32Return += fprintf(psFile, "\tEnd Location ID = %d\n",
                    tEndID);
        }
        break;
        default:
        {
            return EOF;
        }

    }
    return n32Return;
}

/*****************************************************************************
*
*   eIterateLocations
*
*****************************************************************************/
static SMSAPI_RETURN_CODE_ENUM eIterateLocations (
    TRAFFIC_MSG_BASE_OBJECT hTrafficBaseMsg,
    TRAFFIC_MSG_LOCATION_ITERATOR bIterator,
    void *pvIteratorArg
        )
{
    TRAFFIC_MSG_OBJECT_STRUCT *psObj =
        (TRAFFIC_MSG_OBJECT_STRUCT *)hTrafficBaseMsg;
    SMSAPI_RETURN_CODE_ENUM eReturnCode = SMSAPI_RETURN_CODE_NOT_OWNER;
    OSAL_RETURN_CODE_ENUM eOsalReturnCode;

    eOsalReturnCode = OSAL.eLinkedListIterate(psObj->hLocationList,
        (OSAL_LL_ITERATOR_HANDLER)bIterator, pvIteratorArg);

    if (eOsalReturnCode == OSAL_SUCCESS)
    {
        eReturnCode = SMSAPI_RETURN_CODE_SUCCESS;
    }
    else if (eOsalReturnCode == OSAL_NO_OBJECTS)
    {
        eReturnCode = SMSAPI_RETURN_CODE_NO_OBJECTS;
    }
    else
    {
        eReturnCode = SMSAPI_RETURN_CODE_ERROR;
    }

    return eReturnCode;
}

/*****************************************************************************
                             FRIEND FUNCTIONS
*****************************************************************************/

/*****************************************************************************
*
*   TRAFFIC_MSG_bInstall
*
*****************************************************************************/
BOOLEAN TRAFFIC_MSG_bInstall (
    SMS_OBJECT hOwner,
    TRAFFIC_SERVICE_OBJECT hTrafficMgr
        )
{
    BOOLEAN bRetval = FALSE, bLocked;

    // Lock down the owner object first to gain exclusive access
    // to the object which will be associated as parent to this object.
    bLocked = SMSO_bLock(hOwner, OSAL_OBJ_TIMEOUT_INFINITE);
    if(bLocked == TRUE)
    {
        do
        {
            OSAL_RETURN_CODE_ENUM eReturnCode;

            // Check first to make sure the TrafMsg control is not already created
            if(gpsTrafficMsgCtrl != NULL)
            {
                printf("Error! TrafMsg control already exists.\n");
                break;
            }

            // Allocate a chunk of memory to hold the TrafMsg control structure and
            // assign it to our global pointer.
            gpsTrafficMsgCtrl = (TRAFFIC_MSG_CTRL_STRUCT *)
                SMSO_hCreate(
                    TRAFFIC_MSG_OBJECT_NAME":Ctrl",
                    sizeof(TRAFFIC_MSG_CTRL_STRUCT),
                    hOwner, // This object is a child of the provided object
                    FALSE);
            if(gpsTrafficMsgCtrl == NULL)
            {
                // Error! We cannot proceed
                printf("Error! Traffic Msg Ctrl cannot be created.\n");
                break;
            }

#if DEBUG_OBJECT != 0
            // Initialize Alert-C pool statistics
            gpsTrafficMsgCtrl->un32Available = 0;
            gpsTrafficMsgCtrl->un32Unavailable = 0;
#endif

            // Create a linked list to associate with the Traffic Msg pool. This linked
            // list will maintain a list of Traffic Msg objects which have been
            // allocated for the Traffic Service. As Traffic Msg are created they are added
            // to the list, as they are destroyed they are returned to the list.
            // Creating an Traffic Msg will first look to this list for any already
            // allocated and available Traffic Msg before creating and adding a new one.
            eReturnCode =
                OSAL.eLinkedListCreate(
                    &gpsTrafficMsgCtrl->hPool,
                    TRAFFIC_MSG_OBJECT_NAME":Pool",
                    (OSAL_LL_COMPARE_HANDLER)n16CompareTrafficMsgPool,
                    OSAL_LL_OPTION_LINEAR
                        );
            if(eReturnCode != OSAL_SUCCESS)
            {
                // Error!
                printf("Error! Alert-C Pool list cannot be created.\n");
                TRAFFIC_MSG_vUninstall();
                break;
            }

            gpsTrafficMsgCtrl->hTrafficMgr = hTrafficMgr;

            // If we made it this far the object has been created
            bRetval = TRUE;

        } while(FALSE);

        // Unlock the traffic manager object
        SMSO_vUnlock((SMS_OBJECT)hOwner);
    }

    return bRetval;
}


/*****************************************************************************
*
*   TRAFFIC_MSG_vUninstall
*
*****************************************************************************/
void TRAFFIC_MSG_vUninstall (
    void
        )
{
    BOOLEAN bLocked;

    // Lock down the Alert-C object for access.
    bLocked = SMSO_bLock((SMS_OBJECT)gpsTrafficMsgCtrl, OSAL_OBJ_TIMEOUT_INFINITE);
    if(bLocked == TRUE)
    {
        // Parent of the TrafMsg control structure
        SMS_OBJECT hParent;

        // Obtain this object's parent
        hParent = SMSO_hParent((SMS_OBJECT)gpsTrafficMsgCtrl);

        // Release Alert-C resources...

        // Check if the pool list exists.
        if(gpsTrafficMsgCtrl->hPool != OSAL_INVALID_OBJECT_HDL)
        {
            // Remove all entries from the list and destroy CID which are
            // in the pool.
            OSAL.eLinkedListRemoveAll(
                gpsTrafficMsgCtrl->hPool,
                (OSAL_LL_RELEASE_HANDLER)vReleaseObject
                    );

            // Destroy the list itself
            OSAL.eLinkedListDelete(gpsTrafficMsgCtrl->hPool);
            gpsTrafficMsgCtrl->hPool = OSAL_INVALID_OBJECT_HDL;
        }

        // Destroy the TrafMsg control object itself
        SMSO_vDestroy((SMS_OBJECT)gpsTrafficMsgCtrl);

        // Clear TrafMsg control global pointer
        gpsTrafficMsgCtrl = NULL;

        // Unlock the object we locked. For now on the TrafMsg control
        // object is separated from its association with it's parent object.
        SMSO_vUnlock((SMS_OBJECT)hParent);
    }

    return;
}

/*****************************************************************************
*
*   TRAFFIC_MSG_tInitialPosCode
*
*****************************************************************************/
TRAFFIC_POS_CODE TRAFFIC_MSG_tInitialPosCode (
    TRAFFIC_MSG_OBJECT hTrafficMsg
        )
{
    TRAFFIC_MSG_OBJECT_STRUCT *psObj = NULL;
    TRAFFIC_POS_CODE tPosCode = TRAFFIC_INVALID_POS_CODE;

    psObj =
        (TRAFFIC_MSG_OBJECT_STRUCT *)TRAFFIC_MSG_BASE_pvObjectData(hTrafficMsg);

    // Verify inputs.
    if(psObj != NULL)
    {
        tPosCode = psObj->sAlertC.tPosCode;
    }

    return tPosCode;
}

/*****************************************************************************
*
*   TRAFFIC_MSG_tMarket
*
*****************************************************************************/
TRAFFIC_MARKET TRAFFIC_MSG_tMarket (
    TRAFFIC_MSG_OBJECT hTrafficMsg
        )
{
    TRAFFIC_MSG_OBJECT_STRUCT *psObj = NULL;
    TRAFFIC_MARKET tMarket = TRAFFIC_INVALID_MARKET;

    psObj =
        (TRAFFIC_MSG_OBJECT_STRUCT *)TRAFFIC_MSG_BASE_pvObjectData(hTrafficMsg);

    // Verify inputs.
    if(psObj != NULL)
    {
        tMarket = psObj->sAlertC.tMarket;
    }

    return tMarket;
}

/*****************************************************************************
*
*   TRAFFIC_MSG_tBSA
*
*****************************************************************************/
TRAFFIC_BSA TRAFFIC_MSG_tBSA (
    TRAFFIC_MSG_OBJECT hTrafficMsg
        )
{
    TRAFFIC_MSG_OBJECT_STRUCT *psObj = NULL;
    TRAFFIC_BSA tBSA = TRAFFIC_INVALID_BSA;

    psObj =
        (TRAFFIC_MSG_OBJECT_STRUCT *)TRAFFIC_MSG_BASE_pvObjectData(hTrafficMsg);

    // Verify inputs.
    if(psObj != NULL)
    {
        tBSA = psObj->sAlertC.tBSA;
    }

    return tBSA;
}

/*****************************************************************************
*
*   TRAFFIC_MSG_un8InitialNumLocs
*
*****************************************************************************/
UN8 TRAFFIC_MSG_un8InitialNumLocs (
    TRAFFIC_MSG_OBJECT hTrafficMsg
        )
{
    TRAFFIC_MSG_OBJECT_STRUCT *psObj = NULL;
    UN8 un8Num = 0;

    psObj =
        (TRAFFIC_MSG_OBJECT_STRUCT *)TRAFFIC_MSG_BASE_pvObjectData(hTrafficMsg);

    // Verify inputs.
    if(psObj != NULL)
    {
        un8Num = psObj->sAlertC.un8Extent;
    }

    return un8Num;
}

/*****************************************************************************
*
*   TRAFFIC_MSG_eAlertCType
*
*****************************************************************************/
ALERTC_EVENT_TYPE_ENUM TRAFFIC_MSG_eAlertCType (
    TRAFFIC_MSG_OBJECT hTrafficMsg
        )
{
    TRAFFIC_MSG_OBJECT_STRUCT *psObj = NULL;
    ALERTC_EVENT_TYPE_ENUM eAlertCType  = ALERTC_EVENT_TYPE_UNKNOWN;

    psObj =
        (TRAFFIC_MSG_OBJECT_STRUCT *)TRAFFIC_MSG_BASE_pvObjectData(hTrafficMsg);
    if(psObj != NULL)
    {
        ALERTC_EVENT_OBJECT hEventInfo = hGetEventInfo(psObj);

        eAlertCType = ALERTC_EVENT_eType(hEventInfo);
    }

    return eAlertCType;
}


/*****************************************************************************
*
*   TRAFFIC_MSG_hCreate
*
* This object interface method is a friend function that is used to create a
* TRAFFIC_MSG object.
*
* Inputs:
*
*   hTrafficMgr - The SMS object that owns this object
*   hAlertC - The Alert-C message that is the basis for the TRAFFIC_MSG
*
* Outputs:
*
*   A valid TRAFFIC_MSG on success, a TRAFFIC_MSG_INVALID_OBJECT on error.
*
*****************************************************************************/
TRAFFIC_MSG_OBJECT TRAFFIC_MSG_hCreate (
    SMS_OBJECT hOwner,
    OSAL_BUFFER_HDL hPayload,
    TRAFFIC_LOCID_OBJECT hTrafficLocID
        )
{
    BOOLEAN bOk = FALSE;
    char acListName[OSAL_MAX_OBJECT_NAME_LENGTH_WITH_NULL];
    OSAL_RETURN_CODE_ENUM eReturnCode;
    TRAFFIC_MSG_OBJECT hTrafficMsg = TRAFFIC_MSG_INVALID_OBJECT;

    do
    {
        TRAFFIC_MSG_OBJECT_STRUCT *psObj =
            (TRAFFIC_MSG_OBJECT_STRUCT *)TRAFFIC_MSG_INVALID_OBJECT;

        // Verify the Payload and Location ID are valid
        if (hTrafficLocID == TRAFFIC_LOCID_INVALID_OBJECT ||
            hPayload == OSAL_INVALID_BUFFER_HDL)
        {
            // Error!
            break;
        }

        // See if we have any data to reuse
        hTrafficMsg = hFindReusable();

        if (hTrafficMsg == TRAFFIC_MSG_INVALID_OBJECT)
        {
            // Could not find an object to re-use, so create one

            // Create an instance of the TrafficMsg object
            hTrafficMsg = hCreateObject(hOwner);

            if(hTrafficMsg == TRAFFIC_MSG_INVALID_OBJECT)
            {
                // Error!
                break;
            }
        }

        psObj = (TRAFFIC_MSG_OBJECT_STRUCT *)TRAFFIC_MSG_BASE_pvObjectData(
            hTrafficMsg);
        if (psObj == NULL)
        {
            // Error!
            break;
        }

        // Save the TRAFFIC_LOCID info before we start processing the payload
        psObj->sAlertC.tMarket = TRAFFIC_LOCID.tMarket(hTrafficLocID);
        psObj->sAlertC.tBSA = TRAFFIC_LOCID.tBSA(hTrafficLocID);

        // Parse the payload
        bOk = bParsePayload(psObj, hPayload);

        if (bOk == FALSE)
        {
            break;
        }

        if (psObj->hLocationList == OSAL_INVALID_OBJECT_HDL)
        {
            // Create the linked list for the locations
            snprintf( &acListName[0], sizeof(acListName),
                "%s:LocationList", TRAFFIC_MSG_OBJECT_NAME );

            eReturnCode = OSAL.eLinkedListCreate(
                &psObj->hLocationList,
                &acListName[0],
                NULL,
                OSAL_LL_OPTION_LINEAR
                    );

            if(eReturnCode != OSAL_SUCCESS)
            {
                break;
            }
        }

        // Set the timestamp to record the creation time
        OSAL.eTimeGet(&psObj->un32TimeStamp);

        return hTrafficMsg;

    } while (FALSE);

    // Error!
    // Free object instance
    TRAFFIC_MSG_vDestroy(hTrafficMsg);

    return TRAFFIC_MSG_INVALID_OBJECT;
}


/*****************************************************************************
*
*   TRAFFIC_MSG_n16Equal
*
*****************************************************************************/
N16 TRAFFIC_MSG_n16Equal(
    TRAFFIC_MSG_OBJECT hTrafficMsg1,
    TRAFFIC_MSG_OBJECT hTrafficMsg2
        )
{
    TRAFFIC_MSG_OBJECT_STRUCT
        *psObj1 = NULL,
        *psObj2 = NULL;

    // Verify inputs. Object handles must be valid and caller own both.
    psObj1 = (TRAFFIC_MSG_OBJECT_STRUCT *)TRAFFIC_MSG_BASE_pvObjectData(
        hTrafficMsg1);

    if(psObj1 != NULL)
    {
        psObj2 = (TRAFFIC_MSG_OBJECT_STRUCT *)TRAFFIC_MSG_BASE_pvObjectData(
            hTrafficMsg2);
    }
    else
    {
        // Error!
        return N16_MIN;
    }

    if(psObj2 == NULL)
    {
        // Error!
        return N16_MIN;
    }

    if ( (psObj1->sAlertC.tMarket != psObj2->sAlertC.tMarket) ||
         (psObj1->sAlertC.tPosCode != psObj2->sAlertC.tPosCode) ||
         (psObj1->sAlertC.tEventCode != psObj2->sAlertC.tEventCode) ||
         (psObj1->sAlertC.un8Extent != psObj2->sAlertC.un8Extent) ||
         (psObj1->sAlertC.eDirection != psObj2->sAlertC.eDirection) ||
         (psObj1->sAlertC.un8MultiGroupCount != psObj2->sAlertC.un8MultiGroupCount)
       )
    {
        return -1;
    }

    return 0;
}

/*****************************************************************************
*
*   TRAFFIC_MSG_bAddLocation
*
*****************************************************************************/
BOOLEAN TRAFFIC_MSG_bAddLocation (
    TRAFFIC_MSG_OBJECT hTrafficMsg,
    LOCATION_OBJECT hLocation
        )
{
    TRAFFIC_MSG_OBJECT_STRUCT *psObj = NULL;
    BOOLEAN bOk = FALSE;
    OSAL_LINKED_LIST_ENTRY hEntry = OSAL_INVALID_LINKED_LIST_ENTRY;
    OSAL_RETURN_CODE_ENUM eReturnCode;

    psObj = (TRAFFIC_MSG_OBJECT_STRUCT *)TRAFFIC_MSG_BASE_pvObjectData(
        hTrafficMsg);

    // Verify inputs. Object handle must be valid as well as the file handle.
    if((psObj == NULL) || (hLocation == LOCATION_INVALID_OBJECT))
    {
        // Error!
        SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
            TRAFFIC_MSG_OBJECT_NAME": Can't add location. Invalid params.");
        return FALSE;
    }

    // Add the location to our list if it doesn't exists
    eReturnCode = OSAL.eLinkedListLinearSearch(psObj->hLocationList,
        &hEntry, (OSAL_LL_COMPARE_HANDLER)n16IsLocationEqual,
        (void*)hLocation);

    if (eReturnCode == OSAL_OBJECT_NOT_FOUND)
    {
        eReturnCode = OSAL.eLinkedListAdd(psObj->hLocationList,
                                          OSAL_INVALID_LINKED_LIST_ENTRY_PTR,
                                          hLocation);

        if (eReturnCode == OSAL_SUCCESS)
        {
            bOk = TRUE;
        }
    }

    return bOk;

}

/*****************************************************************************
*
*   TRAFFIC_MSG_bRemoveLocation
*
*****************************************************************************/
BOOLEAN TRAFFIC_MSG_bRemoveLocation (
    TRAFFIC_MSG_OBJECT hTrafficMsg,
    LOCATION_OBJECT hLocation,
    BOOLEAN *pbMsgDestroyed
        )
{
    TRAFFIC_MSG_OBJECT_STRUCT *psObj = NULL;
    BOOLEAN bOk = FALSE;
    OSAL_RETURN_CODE_ENUM eReturnCode;
    OSAL_LINKED_LIST_ENTRY hEntry = OSAL_INVALID_LINKED_LIST_ENTRY;

    psObj = (TRAFFIC_MSG_OBJECT_STRUCT *)TRAFFIC_MSG_BASE_pvObjectData(
        hTrafficMsg);

    // Verify inputs. Object handle must be valid as well as the file handle.
    if((psObj == NULL) || (hLocation == LOCATION_INVALID_OBJECT))
    {
        // Error!
        SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
            TRAFFIC_MSG_OBJECT_NAME": Can't remove location. Invalid params.");
        return FALSE;
    }

    // Find the location in our list
    eReturnCode = OSAL.eLinkedListLinearSearch(
        psObj->hLocationList, &hEntry,
        (OSAL_LL_COMPARE_HANDLER)n16IsLocationEqual,
        (void *) hLocation);

    if (eReturnCode == OSAL_SUCCESS)
    {
        // We found the location, remove it from the list
        eReturnCode = OSAL.eLinkedListRemove(hEntry);

        if (eReturnCode == OSAL_SUCCESS)
        {
            UN32 un32ListSize = 0;
            // The removal was successful, so the function was successful.
            bOk = TRUE;
            OSAL.eLinkedListItems(psObj->hLocationList, &un32ListSize);
            if (un32ListSize == 0)
            {
                printf(TRAFFIC_MSG_OBJECT_NAME
                    ": list size for 0x%p is zero.  Removing\n", hTrafficMsg);
                TRAFFIC_MSG_vDestroy(hTrafficMsg);

                if (pbMsgDestroyed != NULL)
                {
                    *pbMsgDestroyed = TRUE;
                }
            }
        }
        else
        {
            printf(TRAFFIC_MSG_OBJECT_NAME
                ": failed to remove location from the list: %s\n",
                OSAL.pacGetReturnCodeName(eReturnCode));
        }
    }
    else
    {
        printf(TRAFFIC_MSG_OBJECT_NAME
            ": Can't find location in the list (%s)\n",
            OSAL.pacGetReturnCodeName(eReturnCode));
    }

    return bOk;
}


/*****************************************************************************
*
*   TRAFFIC_MSG_vDestroy
*
* This object interface method is used by the caller to destroy the specified
* TRAFFIC_MSG object and all members of the object
*

* Inputs:
*
*   hTrafficMsg - A handle to a valid TRAFFIC_MSG object that the caller wants
*                 to get rid of
*
* Outputs:
*
*   Nada
*
*****************************************************************************/
void TRAFFIC_MSG_vDestroy (
    TRAFFIC_MSG_OBJECT hTrafficMsg
        )
{
    TRAFFIC_MSG_OBJECT_STRUCT *psObj = NULL;

    psObj = (TRAFFIC_MSG_OBJECT_STRUCT *)TRAFFIC_MSG_BASE_pvObjectData(
        hTrafficMsg);
    if(psObj == NULL)
    {
        // Error!
        return;
    }

    printf(TRAFFIC_MSG_OBJECT_NAME": attempting to destroy 0x%p\n",hTrafficMsg);

    // Check if this Traffic Msg belongs to the Traffic Msg pool. If it does, then
    // we can recycle it for future use. So, we skip destroying the
    // object data and actually just re-initialize it later when we need it.
    // This essentially puts this Traffic Msg back into the Traffic Msg pool.
    if(psObj->hEntry != OSAL_INVALID_LINKED_LIST_ENTRY)
    {
        BOOLEAN bLocked;

        // Lock the Traffic Msg ctrl object. Object handle must be valid.
        bLocked = SMSO_bLock((SMS_OBJECT)gpsTrafficMsgCtrl,
            OSAL_OBJ_TIMEOUT_INFINITE);
        if(bLocked == TRUE)
        {
            OSAL_RETURN_CODE_ENUM eReturnCode;

            // Update Traffic Msg in use flag, make it unassigned in the pool.
            psObj->bInUse = FALSE;

            // Re-insert this entry as an unused Traffic Msg using its new flag,
            // but leaving everything else alone. We replace the entry instead
            // of insert the entry because replace requires the allocation
            // of no additional memory, but instead reuses the linked list
            // structure from this entry to re-link it.
            eReturnCode =
                OSAL.eLinkedListReplaceEntry(
                    gpsTrafficMsgCtrl->hPool,
                    psObj->hEntry,
                    hTrafficMsg
                        );
            if(eReturnCode != OSAL_SUCCESS)
            {
                // Error!
                printf(TRAFFIC_MSG_OBJECT_NAME
                    ": Error! Cannot replace Traffic Msg in pool.\n");

                // Remove Traffic Msg from pool
                eReturnCode = OSAL.eLinkedListRemove(
                    psObj->hEntry
                        );
                if(eReturnCode == OSAL_SUCCESS)
                {
                    // Destroy this Traffic Msg now that it is not part of the pool.
                    vDestroyObject(psObj);
                    TRAFFIC_MSG_BASE_vDestroy(hTrafficMsg);

#if DEBUG_OBJECT != 0
                    printf(TRAFFIC_MSG_OBJECT_NAME": Removed Traffic Msg from pool (%u/%u/%u/%u)\n",
                        gpsTrafficMsgCtrl->un32Available,
                        --gpsTrafficMsgCtrl->un32Unavailable,
                        gpsTrafficMsgCtrl->un32Available + gpsTrafficMsgCtrl->un32Unavailable,
                        gpsTrafficMsgCtrl->un32Created
                            );
#endif
                }
                else
                {
                    // Error! But don't destroy it. I'd rather leave it in
                    // as a dangling Traffic Msg than destroy it which might cause
                    // corruption when iterating the list.
                    printf(TRAFFIC_MSG_OBJECT_NAME": Error! Cannot remove Traffic Msg"
                        " from pool -- dangling Traffic Msg exists!\n");
                }
            }
            else
            {
#if DEBUG_OBJECT != 0
                printf(TRAFFIC_MSG_OBJECT_NAME": Recycling Traffic Msg back into pool (%u/%u/%u/%u)\n",
                    ++gpsTrafficMsgCtrl->un32Available,
                    --gpsTrafficMsgCtrl->un32Unavailable,
                    gpsTrafficMsgCtrl->un32Available + gpsTrafficMsgCtrl->un32Unavailable,
                    gpsTrafficMsgCtrl->un32Created
                        );
#endif
            }

            // Unlock Traffic Msg Ctrl object
            SMSO_vUnlock((SMS_OBJECT)gpsTrafficMsgCtrl);
        }
    }
    else // Not inTraffic Msg pool so we just destroy it.
    {
        // Destroy the Traffic Msg since it cannot be recycled.
        vDestroyObject(psObj);
        TRAFFIC_MSG_BASE_vDestroy(hTrafficMsg);
    }
    return;
}

/*****************************************************************************
*
*       TRAFFIC_MSG_hCreateFromDB
*
*****************************************************************************/
TRAFFIC_MSG_OBJECT TRAFFIC_MSG_hCreateFromDB (
    SMS_OBJECT hOwner,
    TRAFFIC_STORAGE_ROW_STRUCT *psStorageRow
        )
{
    TRAFFIC_MSG_OBJECT hTrafficMsg = TRAFFIC_MSG_INVALID_OBJECT;
    char acListName[OSAL_MAX_OBJECT_NAME_LENGTH_WITH_NULL];
    OSAL_RETURN_CODE_ENUM eReturnCode;
    UN8 un8Idx;

    do
    {
        TRAFFIC_MSG_OBJECT_STRUCT *psObj = NULL;

        // Verify the we have a valid storage row
        if (psStorageRow == NULL)
        {
            // Error!
            break;
        }

        // See if we have any data to reuse
        hTrafficMsg = hFindReusable();

        if (hTrafficMsg == TRAFFIC_MSG_INVALID_OBJECT)
        {
            // Could not find an object to re-use, so create one

            // Create an instance of the TrafficMsg object
            hTrafficMsg = hCreateObject(hOwner);

            if(hTrafficMsg == TRAFFIC_MSG_INVALID_OBJECT)
            {
                // Error!
                break;
            }
        }

        psObj = (TRAFFIC_MSG_OBJECT_STRUCT *)TRAFFIC_MSG_BASE_pvObjectData(
            hTrafficMsg);
        if (psObj == NULL)
        {
            break;
        }

        // Copy the data from our storage row into our object
        psObj->sAlertC.tMarket = psStorageRow->tMarket;
        psObj->sAlertC.tBSA = psStorageRow->tBSA;
        psObj->sAlertC.tPosCode = psStorageRow->tPosCode;
        psObj->sAlertC.un8Extent = psStorageRow->un8Extent;
        psObj->sAlertC.tEventCode = psStorageRow->tEventCode;
        psObj->sAlertC.eDirection = psStorageRow->eDirection;
        psObj->sAlertC.un8Duration = psStorageRow->un8Duration;
        psObj->sAlertC.bDiversionAdvised = psStorageRow->bDiversionAdvised;
        psObj->sAlertC.un8MultiGroupCount = psStorageRow->un8MultiGroupCount;

        // Add our free-form data
        for (un8Idx = 0;
             ((un8Idx < (psStorageRow->un8FreeFormDataCount * 2)) &&
              (un8Idx < (sizeof(psStorageRow->aun16FreeFormData) / 
               sizeof(psStorageRow->aun16FreeFormData[0]) - 1)));
             un8Idx = un8Idx + 2)
        {
            printf("Adding free-form data %u-%u\n",
                psStorageRow->aun16FreeFormData[un8Idx],
                psStorageRow->aun16FreeFormData[un8Idx+1]);
            vAddFreeFormData(psObj,(ALERTC_FREEFORM_DATA_LABEL_ENUM)
                psStorageRow->aun16FreeFormData[un8Idx],
                psStorageRow->aun16FreeFormData[un8Idx+1]);
        }

        if (psObj->hLocationList == OSAL_INVALID_OBJECT_HDL)
        {
            // Create the linked list for the locations
            snprintf( &acListName[0], sizeof(acListName),
                "%s:LocationList", TRAFFIC_MSG_OBJECT_NAME );

            eReturnCode = OSAL.eLinkedListCreate(
                &psObj->hLocationList,
                &acListName[0],
                NULL,
                OSAL_LL_OPTION_LINEAR
                    );

            if(eReturnCode != OSAL_SUCCESS)
            {
                break;
            }
        }

        // Set our timestamp
        psObj->un32TimeStamp = psStorageRow->un32UTCsec;

        return hTrafficMsg;

    } while (FALSE);

    // Error!
    // Free object instance
    TRAFFIC_MSG_vDestroy(hTrafficMsg);

    return TRAFFIC_MSG_INVALID_OBJECT;
}

/*****************************************************************************
*
*       TRAFFIC_MSG_bWrittenToDB
*
*****************************************************************************/
BOOLEAN TRAFFIC_MSG_bWrittenToDB (
    TRAFFIC_MSG_OBJECT hTrafficMsg
        )
{
    TRAFFIC_MSG_OBJECT_STRUCT *psObj = NULL;
    BOOLEAN bWrittenToDB = FALSE;

    psObj =
        (TRAFFIC_MSG_OBJECT_STRUCT *)TRAFFIC_MSG_BASE_pvObjectData(hTrafficMsg);
    if (psObj != NULL)
    {
        bWrittenToDB = psObj->bWrittenToDB;
    }

    return bWrittenToDB;
}

/*****************************************************************************
*
*       TRAFFIC_MSG_vSetWrittenToDBFlag
*
*****************************************************************************/
void TRAFFIC_MSG_vSetWrittenToDBFlag (
    TRAFFIC_MSG_OBJECT hTrafficMsg,
    BOOLEAN bDBFlag
        )
{
    TRAFFIC_MSG_OBJECT_STRUCT *psObj = NULL;

    psObj =
        (TRAFFIC_MSG_OBJECT_STRUCT *)TRAFFIC_MSG_BASE_pvObjectData(hTrafficMsg);
    if (psObj != NULL)
    {
        psObj->bWrittenToDB = bDBFlag;
    }

    return;
}

/*****************************************************************************
*
*       TRAFFIC_MSG_un32TimeStamp
*
*****************************************************************************/
UN32 TRAFFIC_MSG_un32TimeStamp (
    TRAFFIC_MSG_OBJECT hTrafficMsg
        )
{
    TRAFFIC_MSG_OBJECT_STRUCT *psObj = NULL;
    UN32 un32TimeStamp = 0;

    psObj =
        (TRAFFIC_MSG_OBJECT_STRUCT *)TRAFFIC_MSG_BASE_pvObjectData(hTrafficMsg);
    if (psObj != NULL)
    {
        un32TimeStamp = psObj->un32TimeStamp;
    }

    return un32TimeStamp;
}

/*****************************************************************************
*
*       TRAFFIC_MSG_vUpdateTimeStamp
*
*****************************************************************************/
void TRAFFIC_MSG_vUpdateTimeStamp (
    TRAFFIC_MSG_OBJECT hTrafficMsg
        )
{
    TRAFFIC_MSG_OBJECT_STRUCT *psObj = NULL;

    psObj =
        (TRAFFIC_MSG_OBJECT_STRUCT *)TRAFFIC_MSG_BASE_pvObjectData(hTrafficMsg);
    if (psObj != NULL)
    {
        // Update the time stamp.  This is done to refresh a message
        // in the case it is repeated.
        OSAL.eTimeGet(&psObj->un32TimeStamp);
    }

    return;
}

/*****************************************************************************
*
*       TRAFFIC_MSG_bInsertIntoDB
*
*****************************************************************************/
BOOLEAN TRAFFIC_MSG_bInsertIntoDB (
    TRAFFIC_MSG_OBJECT hTrafficMsg,
    UN32 un32UnixTimeSecs,
    SQL_INTERFACE_OBJECT hSQL,
    SQL_PREPARED_STATEMENT_HANDLE *phStmt,
    SQL_BIND_PARAMETER_STRUCT *psStmtBindParams
        )
{
    TRAFFIC_MSG_OBJECT_STRUCT *psObj = NULL;
    TRAFFIC_MSG_CLASS_ENUM eClass;
    BOOLEAN bOk;
    OSAL_RETURN_CODE_ENUM eReturnCode;
    TRAFFIC_CREATE_FREE_FORM_BLOB_ITERATOR_STRUCT sIteratorArg;

    // need a UN16 for the label and the value, hence the multiplication
    UN16 aun16FreeFormData[MAX_FREEFORM_ENTRIES*2] = {0x00};

    // Determine if the caller owns this resource
    psObj =
        (TRAFFIC_MSG_OBJECT_STRUCT *)TRAFFIC_MSG_BASE_pvObjectData(hTrafficMsg);

    do
    {
        UN8 un8BlobSize;

        if ((psObj == NULL) || (NULL == phStmt))
        {
            break;
        }

        if (SQL_PREPARED_STATEMENT_INVALID_HANDLE == *phStmt)
        {
            *phStmt = SQL_INTERFACE.hCreatePreparedStatement(
                hSQL, TRAFFIC_ADD_MSG_TO_TRAFFIC_TABLE);

            if (SQL_PREPARED_STATEMENT_INVALID_HANDLE == *phStmt)
            {
                break;
            }
        }

        // Prepare our iterator arg
        sIteratorArg.un8FreeFormDataCount = 0;
        sIteratorArg.pun16BlobData = &aun16FreeFormData[0];
        sIteratorArg.tBlobDataSize = sizeof(aun16FreeFormData);

        if (psObj->sAlertC.hFreeFormDataList != OSAL_INVALID_OBJECT_HDL)
        {
            eReturnCode = OSAL.eLinkedListIterate(
                psObj->sAlertC.hFreeFormDataList,
                (OSAL_LL_ITERATOR_HANDLER)bAddFreeFormDataToBlob,
                (void*)&sIteratorArg);

            if (eReturnCode != OSAL_SUCCESS && eReturnCode != OSAL_NO_OBJECTS)
            {
                break;
            }
        }

        // Fetch the class for this message,
        // used as part of the primary key in the DB
        eClass = TRAFFIC_MSG.eClass(hTrafficMsg);

        un8BlobSize = sIteratorArg.un8FreeFormDataCount * sizeof(UN16) * 2;
        if (0 == un8BlobSize)
        {
            aun16FreeFormData[0] = 0x00;
            un8BlobSize = sizeof(aun16FreeFormData[0]);
        }

        vSetAddStmtParams(
            &psObj->sAlertC, un32UnixTimeSecs, eClass,
            sIteratorArg.un8FreeFormDataCount,
            (void*)aun16FreeFormData,
            (size_t)un8BlobSize,
            psStmtBindParams);

        bOk = SQL_INTERFACE.bExecutePreparedStatement(
            hSQL, *phStmt, NULL, NULL,
            TRAFFIC_ADD_STMT_PARAMS_COUNT, psStmtBindParams);

        return bOk;

    } while (FALSE);

    return FALSE;
}

/*****************************************************************************
                             PRIVATE FUNCTIONS
*****************************************************************************/

/*****************************************************************************
*
*       vSetAddStmtParams
*
*****************************************************************************/
static void vSetAddStmtParams (
    ALERTC_OBJECT_STRUCT const *psAlertC,
    UN32 un32UnixTimeSecs,
    TRAFFIC_MSG_CLASS_ENUM eClass,
    UN8 un8FreeFormDataCount,
    void *pvFreeFormData,
    size_t tFreeFormDataSize,
    SQL_BIND_PARAMETER_STRUCT *psBindParams
        )
{
    psBindParams[TRAFFIC_ADD_STMT_TIME_STAMP_PARAM].eType =
        SQL_BIND_TYPE_UN32;
    psBindParams[TRAFFIC_ADD_STMT_TIME_STAMP_PARAM].pvData =
        (void*)un32UnixTimeSecs;

    psBindParams[TRAFFIC_ADD_STMT_MARKET_PARAM].eType =
        SQL_BIND_TYPE_UN32;
    psBindParams[TRAFFIC_ADD_STMT_MARKET_PARAM].pvData =
        (void*)(size_t)psAlertC->tMarket;

    psBindParams[TRAFFIC_ADD_STMT_BSA_PARAM].eType =
        SQL_BIND_TYPE_UN32;
    psBindParams[TRAFFIC_ADD_STMT_BSA_PARAM].pvData =
        (void*)(size_t)psAlertC->tBSA;

    psBindParams[TRAFFIC_ADD_STMT_POS_CODE_PARAM].eType =
        SQL_BIND_TYPE_UN32;
    psBindParams[TRAFFIC_ADD_STMT_POS_CODE_PARAM].pvData =
        (void*)(size_t)psAlertC->tPosCode;

    psBindParams[TRAFFIC_ADD_STMT_EXTENT_PARAM].eType =
        SQL_BIND_TYPE_UN32;
    psBindParams[TRAFFIC_ADD_STMT_EXTENT_PARAM].pvData =
        (void*)(size_t)psAlertC->un8Extent;

    psBindParams[TRAFFIC_ADD_STMT_EVENT_CLASS_PARAM].eType =
        SQL_BIND_TYPE_UN32;
    psBindParams[TRAFFIC_ADD_STMT_EVENT_CLASS_PARAM].pvData =
        (void*)(size_t)eClass;

    psBindParams[TRAFFIC_ADD_STMT_EVENT_CODE_PARAM].eType =
        SQL_BIND_TYPE_UN32;
    psBindParams[TRAFFIC_ADD_STMT_EVENT_CODE_PARAM].pvData =
        (void*)(size_t)psAlertC->tEventCode;

    psBindParams[TRAFFIC_ADD_STMT_DIRECTION_PARAM].eType =
        SQL_BIND_TYPE_UN32;
    psBindParams[TRAFFIC_ADD_STMT_DIRECTION_PARAM].pvData =
        (void*)(size_t)psAlertC->eDirection;

    psBindParams[TRAFFIC_ADD_STMT_DURATION_PARAM].eType =
        SQL_BIND_TYPE_UN32;
    psBindParams[TRAFFIC_ADD_STMT_DURATION_PARAM].pvData =
        (void*)(size_t)psAlertC->un8Duration;

    psBindParams[TRAFFIC_ADD_STMT_DIVERSION_PARAM].eType =
        SQL_BIND_TYPE_UN32;
    psBindParams[TRAFFIC_ADD_STMT_DIVERSION_PARAM].pvData =
        (void*)(size_t)psAlertC->bDiversionAdvised;

    psBindParams[TRAFFIC_ADD_STMT_MGR_COUNT_PARAM].eType =
        SQL_BIND_TYPE_UN32;
    psBindParams[TRAFFIC_ADD_STMT_MGR_COUNT_PARAM].pvData =
        (void*)(size_t)psAlertC->un8MultiGroupCount;

    psBindParams[TRAFFIC_ADD_STMT_FFD_COUNT_PARAM].eType =
        SQL_BIND_TYPE_UN32;
    psBindParams[TRAFFIC_ADD_STMT_FFD_COUNT_PARAM].pvData =
        (void*)(size_t)un8FreeFormDataCount;

    psBindParams[TRAFFIC_ADD_STMT_FFD].eType = SQL_BIND_TYPE_BLOB;
    psBindParams[TRAFFIC_ADD_STMT_FFD].pvData = pvFreeFormData;
    psBindParams[TRAFFIC_ADD_STMT_FFD].tSize = tFreeFormDataSize;

    return;
}

/*****************************************************************************
*
*       hBuildTextWithQuantifiers
*
*****************************************************************************/
static STRING_OBJECT hBuildTextWithQuantifiers (
    TRAFFIC_MSG_OBJECT_STRUCT *psObj,
    STRING_OBJECT hText,
    UN16 un16Value
        )
{
    // TODO: Flesh out hBuildTextWithQuantifiers
    // For now, return the text unmodified.  Still need to figure out
    // how to handle different styles of text.
    //
    // The caller can handle modifying the text through the use of the
    // freeform data function.  This should be acceptable until we have a
    // business decision that states how to handle this.
    return hText;
}

/*****************************************************************************
*
*       n16IsLocationEqual
*
* This is a local function that is used to find a location in the
* TRAFFIC_MSG's hLocationList member variable.
*
*****************************************************************************/
static N16 n16IsLocationEqual (
    const void *pvObj1,
    const void *pvObj2
        )
{
    LOCATION_OBJECT hLocation1 =
        (LOCATION_OBJECT)pvObj1;
    LOCATION_OBJECT hLocation2 =
        (LOCATION_OBJECT)pvObj2;
    LOCID_OBJECT hLocID1, hLocID2;
    N16 n16Result;

    // Get our location IDs since that is all we care about with traffic
    hLocID1 = LOCATION.hLocID(hLocation1);
    hLocID2 = LOCATION.hLocID(hLocation2);

    // Compare them
    n16Result = LOCID.n16Compare(hLocID1, hLocID2);
    if (n16Result != 0)
    {
        // converting value returned from the function to -1 for non equal
        // location ids to support correct linear search
        n16Result = -1;
    }

    return n16Result;
}

/*****************************************************************************
*
*       hGetEventInfo
*
*****************************************************************************/
static ALERTC_EVENT_OBJECT hGetEventInfo (
    TRAFFIC_MSG_OBJECT_STRUCT *psObj
        )
{
    if (psObj->sAlertC.hEventObj == ALERTC_EVENT_INVALID_OBJECT)
    {
        TRAFFIC_SERVICE_OBJECT hTrafficService =
            hGetTrafficMgr(psObj);

        // Get the event info
        psObj->sAlertC.hEventObj = TRAFFIC_MGR_hGetAlertCEvent(
            hTrafficService, psObj->sAlertC.tEventCode);
    }

    return psObj->sAlertC.hEventObj;
}


/*****************************************************************************
*
*   psFindReusable
*
*****************************************************************************/
static TRAFFIC_MSG_OBJECT hFindReusable (
    void
        )
{
    BOOLEAN bLocked;
    TRAFFIC_MSG_OBJECT hTrafficMsg = TRAFFIC_MSG_INVALID_OBJECT;

    // Lock the Traffic Msg Ctrl object
    bLocked = SMSO_bLock((SMS_OBJECT)gpsTrafficMsgCtrl, OSAL_OBJ_TIMEOUT_INFINITE);
    if(bLocked == TRUE)
    {
        OSAL_RETURN_CODE_ENUM eReturnCode;
        OSAL_LINKED_LIST_ENTRY hEntry = OSAL_INVALID_LINKED_LIST_ENTRY;
        TRAFFIC_MSG_OBJECT_STRUCT *psObj = NULL;

        // Search for an available Traffic Msg
        eReturnCode = OSAL.eLinkedListLinearSearch(
            gpsTrafficMsgCtrl->hPool,
            &hEntry,
            (OSAL_LL_COMPARE_HANDLER)n16FindTrafficMsgToReuse,
            (void*)NULL
                );
        if(eReturnCode == OSAL_SUCCESS)
        {
            // Found one! Let's use it.
            hTrafficMsg = (TRAFFIC_MSG_OBJECT)OSAL.pvLinkedListThis(hEntry);
            psObj = (TRAFFIC_MSG_OBJECT_STRUCT *)TRAFFIC_MSG_BASE_pvObjectData(
                hTrafficMsg);

            if (psObj == NULL)
            {
                // Unlock object
                SMSO_vUnlock((SMS_OBJECT)gpsTrafficMsgCtrl);
                return TRAFFIC_MSG_INVALID_OBJECT;
            }

            // Mark as allocated to SMS by modifying the flag
            psObj->bInUse = TRUE;

            // Replace this entry marking it as unavailable for recycling.
            eReturnCode = OSAL.eLinkedListReplaceEntry(
                gpsTrafficMsgCtrl->hPool,
                psObj->hEntry,
                hTrafficMsg
                    );

            // Check success
            if(eReturnCode == OSAL_SUCCESS)
            {
#if DEBUG_OBJECT != 0
                printf(TRAFFIC_MSG_OBJECT_NAME": Reusing Traffic Msg from the pool (%u/%u/%u/%u)\n",
                    --gpsTrafficMsgCtrl->un32Available,
                    ++gpsTrafficMsgCtrl->un32Unavailable,
                    gpsTrafficMsgCtrl->un32Available + gpsTrafficMsgCtrl->un32Unavailable,
                    gpsTrafficMsgCtrl->un32Created
                        );
#endif

                // Re-init the object members
                psObj->sAlertC.tMarket = 0;
                psObj->sAlertC.tBSA = 0;
                psObj->sAlertC.tPosCode = 0;
                psObj->sAlertC.un8Duration = 0;
                psObj->sAlertC.bDiversionAdvised = 0;
                psObj->sAlertC.eDirection = TRAFFIC_DIRECTION_UNKNOWN;
                psObj->sAlertC.un8Extent = 0;
                psObj->sAlertC.tEventCode = 0;
                psObj->sAlertC.un8MultiGroupCount = 0;
                psObj->bWrittenToDB = FALSE;
                psObj->sAlertC.hEventObj = ALERTC_EVENT_INVALID_OBJECT;

                eReturnCode =
                    OSAL.eLinkedListRemoveAll(psObj->hLocationList, NULL);

                // If we couldn't remove all, attempt to delete the
                // list and mark the pointer as invalid
                // so that at least we don't get a messed up entry when we
                // re-use
                if (eReturnCode != OSAL_SUCCESS)
                {
                    // TODO: Log something to stderr
                    OSAL.eLinkedListDelete(psObj->hLocationList);
                    psObj->hLocationList = OSAL_INVALID_OBJECT_HDL;
                }

                // Mark any freeform data as inactive
                eReturnCode = OSAL.eLinkedListIterate(
                    psObj->sAlertC.hFreeFormDataList,
                    (OSAL_LL_ITERATOR_HANDLER)bMarkFreeFormDataInactive,
                    (void*)NULL);

                // If this failed, delete the free-form data list
                if (eReturnCode != OSAL_SUCCESS)
                {
                    // TODO: Log something to stderr
                    OSAL.eLinkedListRemoveAll(
                        psObj->sAlertC.hFreeFormDataList,
                        vReleaseFreeFormDataListNode);
                    OSAL.eLinkedListDelete(psObj->sAlertC.hFreeFormDataList);
                    psObj->sAlertC.hFreeFormDataList = OSAL_INVALID_OBJECT_HDL;
                }

                if (psObj->hTextMod != STRING_INVALID_OBJECT)
                {
                    STRING_vDestroy(psObj->hTextMod);
                    psObj->hTextMod = STRING_INVALID_OBJECT;
                }

                if (psObj->hAdditionalText != STRING_INVALID_OBJECT)
                {
                    STRING_vDestroy(psObj->hAdditionalText);
                    psObj->hAdditionalText = STRING_INVALID_OBJECT;
                }
            }
            else
            {
                // Error!
                printf(TRAFFIC_MSG_OBJECT_NAME": Error! Cannot replace Alert-C within pool.\n");

                // At this point there isn't much we can do, so we just
                // go ahead and leave it alone, but don't use it.
            }
        }

        // Unlock object
        SMSO_vUnlock((SMS_OBJECT)gpsTrafficMsgCtrl);
    }

    return hTrafficMsg;
}

/*****************************************************************************
*
*   psCreateObject
*
*****************************************************************************/
static TRAFFIC_MSG_OBJECT hCreateObject (
    SMS_OBJECT hOwner
        )
{
    TRAFFIC_MSG_OBJECT hTrafficMsg = TRAFFIC_MSG_INVALID_OBJECT;
    OSAL_RETURN_CODE_ENUM eReturnCode;
    BOOLEAN bLocked;

    // Lock the Traffic Msg Control object
    bLocked = SMSO_bLock(
        (SMS_OBJECT)gpsTrafficMsgCtrl, OSAL_OBJ_TIMEOUT_INFINITE);
    if(bLocked == TRUE)
    {
        TRAFFIC_MSG_OBJECT_STRUCT *psObj = NULL;

        // Create an instance of the Traffic Msg object
        hTrafficMsg = TRAFFIC_MSG_BASE_hCreate(
                (void **)&psObj,
                TRAFFIC_MSG_OBJECT_NAME"ObjData",
                hOwner,
                sizeof(TRAFFIC_MSG_OBJECT_STRUCT),
                0,
                &LEGACY_TRAFFIC_MSG);

        if(psObj != NULL)
        {
            // Initialize object to defaults
            psObj->hEntry = OSAL_INVALID_LINKED_LIST_ENTRY;
            psObj->bInUse = TRUE;
            psObj->hParent = hOwner;

            // Add to Alert-C pool.
            eReturnCode = OSAL.eLinkedListAdd(
                gpsTrafficMsgCtrl->hPool,
                &psObj->hEntry,
                hTrafficMsg
                     );

            // Check success
            if(eReturnCode != OSAL_SUCCESS)
            {
                // Error!
                printf(TRAFFIC_MSG_OBJECT_NAME": Error! Cannot add Traffic Msg to pool.\n");

                // This is not part of the Traffic Msg pool
                psObj->hEntry = OSAL_INVALID_LINKED_LIST_ENTRY;

                // If this happens it is unfortunate it cannot be
                // added to the Traffic Msg pool, but it is not tragic.
                // We can still continue on it just wont get
                // recycled.
            }
            else
            {
#if DEBUG_OBJECT != 0
                gpsTrafficMsgCtrl->un32Created++;
                printf(TRAFFIC_MSG_OBJECT_NAME": Added new Traffic Msg to pool (%u/%u/%u/%u)\n",
                    gpsTrafficMsgCtrl->un32Available,
                    ++gpsTrafficMsgCtrl->un32Unavailable,
                    gpsTrafficMsgCtrl->un32Available +
                    gpsTrafficMsgCtrl->un32Unavailable,
                    gpsTrafficMsgCtrl->un32Created
                        );
#endif

            }
        }

        // Unlock Traffic Msg Ctrl object
        SMSO_vUnlock((SMS_OBJECT)gpsTrafficMsgCtrl);
    }

    return hTrafficMsg;
}

/*****************************************************************************
*
*   vDestroyObject
*
*****************************************************************************/
static void vDestroyObject (
    TRAFFIC_MSG_OBJECT_STRUCT *psObj
        )
{
    OSAL_RETURN_CODE_ENUM eReturnCode;

    // Re-Initialize object
    psObj->hEntry = OSAL_INVALID_LINKED_LIST_ENTRY;
    psObj->bInUse = FALSE;

    eReturnCode =
        OSAL.eLinkedListRemoveAll(psObj->hLocationList, NULL);

    if (eReturnCode == OSAL_SUCCESS)
    {
        OSAL.eLinkedListDelete(psObj->hLocationList);
        psObj->hLocationList = OSAL_INVALID_OBJECT_HDL;
    }

    eReturnCode =
        OSAL.eLinkedListRemoveAll(psObj->sAlertC.hFreeFormDataList,
            vReleaseFreeFormDataListNode);

    if (eReturnCode == OSAL_SUCCESS)
    {
        OSAL.eLinkedListDelete(psObj->sAlertC.hFreeFormDataList);
        psObj->sAlertC.hFreeFormDataList = OSAL_INVALID_OBJECT_HDL;
    }

    if (psObj->hTextMod != STRING_INVALID_OBJECT)
    {
        STRING_vDestroy(psObj->hTextMod);
        psObj->hTextMod = STRING_INVALID_OBJECT;
    }

    if (psObj->hAdditionalText != STRING_INVALID_OBJECT)
    {
        STRING_vDestroy(psObj->hAdditionalText);
        psObj->hAdditionalText = STRING_INVALID_OBJECT;
    }
}


/*****************************************************************************
*
*   vReleaseFreeFormDataListNode
*
*****************************************************************************/
static void vReleaseFreeFormDataListNode (
    void *pvData
        )
{
    ALERTC_FREEFORM_DATA_STRUCT *psFreeFormData =
        (ALERTC_FREEFORM_DATA_STRUCT*)pvData;

    // Free the memory
    SMSO_vDestroy((SMS_OBJECT)psFreeFormData);

    return;
}

/*****************************************************************************
*
*   bParsePayload
*
*****************************************************************************/
static BOOLEAN bParsePayload (
    TRAFFIC_MSG_OBJECT_STRUCT *psObj,
    OSAL_BUFFER_HDL hPayload
        )
{
    BOOLEAN bOk = FALSE,
            bRead = FALSE,
            bFullGroupFlag = FALSE;
    UN8 un8Direction = 0;
    size_t tSizeRead;
    UN16 un16PosCode = 0;

    // We aren't checking the validity of the payload handle
    // since we confirmed it previously

    // Seek pass the first three reserved bits and the tunning flag
    tSizeRead = OSAL.tBufferSeekHeadBits(hPayload,
                             RESERVED_FIELD_BITLEN+ALERT_C_FLAG_FIELD_BITLEN);

    if ( tSizeRead != (RESERVED_FIELD_BITLEN + ALERT_C_FLAG_FIELD_BITLEN))
    {
        // ERROR!
        return FALSE;
    }

    // Read the full-group flag
    tSizeRead = OSAL.tBufferReadHeadBits(hPayload,
                                         &bFullGroupFlag,
                                         0,
                                         FULL_GROUP_FLAG_FIELD_BITLEN);

    if (tSizeRead != FULL_GROUP_FLAG_FIELD_BITLEN)
    {
        // ERROR!
        puts(TRAFFIC_MSG_OBJECT_NAME": Error! Not enough bits for Full Group Flag\n");
        return FALSE;
    }

    if (bFullGroupFlag == TRUE)
    {
        // This is a single-group message
        // Read fields specific to this type
        tSizeRead = OSAL.tBufferReadHeadBits(hPayload,
                                             &psObj->sAlertC.un8Duration,
                                             0,
                                             DURATION_FIELD_BITLEN);

        tSizeRead += OSAL.tBufferReadHeadBits(hPayload,
                                              &psObj->sAlertC.bDiversionAdvised,
                                              0,
                                              DIVERSION_ADVISED_FIELD_BITLEN);
        if (tSizeRead !=
            (DURATION_FIELD_BITLEN + DIVERSION_ADVISED_FIELD_BITLEN))
        {
            // ERROR!
            puts(TRAFFIC_MSG_OBJECT_NAME": Error! Not enough bits for "
                                   "duration and diversion flags");
            return FALSE;
        }

        // Set the multi-group count to 1
        psObj->sAlertC.un8MultiGroupCount = 1;
    }
    else
    {
        // This is a multi-group message
        // Read fields specific to this type
        BOOLEAN bFirstGroupFlag = FALSE;

        tSizeRead = OSAL.tBufferSeekHeadBits(hPayload,
                                             CONTINUTY_INDEX_FIELD_BITLEN);

        tSizeRead += OSAL.tBufferReadHeadBits(hPayload,
                                              &bFirstGroupFlag,
                                              0,
                                              FIRST_GROUP_FLAG_FIELD_BITLEN);
        if (tSizeRead !=
            (CONTINUTY_INDEX_FIELD_BITLEN + FIRST_GROUP_FLAG_FIELD_BITLEN))
        {
            // ERROR!
            puts(TRAFFIC_MSG_OBJECT_NAME": Error! Not enough bits for "
                                   "cont idx and first group flags\n");
            return FALSE;
        }

        // Verify the first group flag is set.  If it isn't set, we have a
        // corrupted message and should bail
        if (bFirstGroupFlag == FALSE)
        {
            // ERROR!
            puts(TRAFFIC_MSG_OBJECT_NAME": Error! First group flag not set\n");
            return FALSE;
        }
    }

    tSizeRead =  OSAL.tBufferReadHeadBits(hPayload,
                                          &un8Direction,
                                          0,
                                          DIRECTION_FLAG_FIELD_BITLEN);

    psObj->sAlertC.eDirection = (TRAFFIC_DIRECTION_ENUM)un8Direction;

    tSizeRead += OSAL.tBufferReadHeadBits(hPayload,
                                          &psObj->sAlertC.un8Extent,
                                          0,
                                          EXTENT_FIELD_BITLEN);

    bRead = OSAL.bBufferReadBitsToUN16(hPayload,
                                       &psObj->sAlertC.tEventCode,
                                       EVENT_CODE_FIELD_BITLEN);

    bRead &= OSAL.bBufferReadBitsToUN16(hPayload,
                                        &un16PosCode,
                                        POS_CODE_FIELD_BITLEN);

    if (tSizeRead != (DIRECTION_FLAG_FIELD_BITLEN +
                      EXTENT_FIELD_BITLEN) ||
        bRead == FALSE)
    {
        // ERROR!
        puts(TRAFFIC_MSG_OBJECT_NAME": Error! Not enough bits for "
                               "main chunk of data\n");
        return FALSE;
    }

    // Copy the UN16 position code to our LOC_ID-based Pos Code
    psObj->sAlertC.tPosCode = un16PosCode;

    // Continue parsing if this is a multi-group message
    if (bFullGroupFlag == FALSE)
    {
        BOOLEAN bFirstGroupFlag, bSecondGroupFlag;
        size_t tBytesLeft;
        UN8 un8MsgRead = 1; // set to one since we already read on msg
        UN8 un8CurGSI = 0;

        do
        {
            bFirstGroupFlag = FALSE;
            bSecondGroupFlag = FALSE;
            un8CurGSI = 0;

            tSizeRead = OSAL.tBufferSeekHeadBits(hPayload,
                RESERVED_FIELD_BITLEN + ALERT_C_FLAG_FIELD_BITLEN);

            if (tSizeRead != (RESERVED_FIELD_BITLEN + ALERT_C_FLAG_FIELD_BITLEN))
            {
                // ERROR!
                return FALSE;
            }

            tSizeRead = OSAL.tBufferSeekHeadBits(hPayload,
                                     FULL_GROUP_FLAG_FIELD_BITLEN);

            tSizeRead += OSAL.tBufferSeekHeadBits(hPayload,
                                     CONTINUTY_INDEX_FIELD_BITLEN);

            if (tSizeRead !=
                (FULL_GROUP_FLAG_FIELD_BITLEN + CONTINUTY_INDEX_FIELD_BITLEN))
            {
                // ERROR!
                return FALSE;
            }

            // Check the full and second group flags to verify this is
            // still a valid packet.  We do a lot of checks.

            tSizeRead =
                OSAL.tBufferReadHeadBits(hPayload,
                                         &bFirstGroupFlag,
                                         0,
                                         FIRST_GROUP_FLAG_FIELD_BITLEN);

            tSizeRead +=
                OSAL.tBufferReadHeadBits(hPayload,
                                         &bSecondGroupFlag,
                                         0,
                                         SECOND_GROUP_FLAG_FIELD_BITLEN);

            // If the size read is wrong or the contents of the flags
            // is off, error out
            if (tSizeRead != (FIRST_GROUP_FLAG_FIELD_BITLEN +
                              SECOND_GROUP_FLAG_FIELD_BITLEN) ||
                bFirstGroupFlag == TRUE || bSecondGroupFlag == FALSE)
            {
                // ERROR!
                puts(TRAFFIC_MSG_OBJECT_NAME": Error! Not enough bits for"
                                       "First and second group flags\n");
                return FALSE;
            }

            tSizeRead = OSAL.tBufferReadHeadBits(hPayload,
                &un8CurGSI, 0, GSI_FIELD_BITLEN);

            if ( (tSizeRead != GSI_FIELD_BITLEN) ||
                  (un8MsgRead == 1)
               )
            {
                // if this is the second message being read, store off
                // the GSI value as our count

                // The GSI is stored as N-2 in the second message, so
                // increment out count by two to show the actual value
                psObj->sAlertC.un8MultiGroupCount = un8CurGSI + 2;
            }

            // Now parse the free-form data
            bOk = bParseFreeFormData(psObj, hPayload);

            if (bOk == FALSE)
            {
                // We had a problem parsing the free form data
                // for this message
                puts(TRAFFIC_MSG_OBJECT_NAME": Error! Couldn't parse freeform data\n");
                return FALSE;
            }

            // Increment the number of messages read
            un8MsgRead++;

            // See what is left
            tBytesLeft = OSAL.tBufferGetSize(hPayload);
            printf(TRAFFIC_MSG_OBJECT_NAME": Num bytes left to read multi-group: %d\n",
                tBytesLeft);


        } while(tBytesLeft >= ALERTC_MESSAGE_MIN_BYTE_LENGTH && un8CurGSI != 0);

        if (psObj->sAlertC.un8MultiGroupCount != un8MsgRead)
        {
            // We didn't read the number of messages we were expecting
            // Lets error out here
            puts(TRAFFIC_MSG_OBJECT_NAME": Error! Wrong number of multi-group msgs read\n");
            return FALSE;
        }
    }

    // Processed the payload, return success
    return TRUE;
}

/*****************************************************************************
*
*   bParsePayload
*
*****************************************************************************/
static size_t tFreeFormDataLength (
    ALERTC_FREEFORM_DATA_LABEL_ENUM eLabel
        )
{
    size_t tLengthInBits = 0;

    // The following field values can be found in:
    // RX0000096 -- Section 4.1.7, Table 2 "Optional Message Content"
    // ISO14189-1 -- Section 5.5, "Optional Message Content"

    switch (eLabel)
    {
        case ALERTC_FREEFORM_DATA_LABEL_DURATION:
        case ALERTC_FREEFORM_DATA_LABEL_CONTROL:
            tLengthInBits = 3;
            break;
        case ALERTC_FREEFORM_DATA_LABEL_LENGTH_OF_AFFECTED_ROUTE:
        case ALERTC_FREEFORM_DATA_LABEL_SPEED_LIMIT:
        case ALERTC_FREEFORM_DATA_LABEL_QUANTIFIER_5:
            tLengthInBits = 5;
            break;
        case ALERTC_FREEFORM_DATA_LABEL_QUANTIFIER_8:
        case ALERTC_FREEFORM_DATA_LABEL_SUPPL_INFO:
        case ALERTC_FREEFORM_DATA_LABEL_EXPLICIT_START_TIME:
        case ALERTC_FREEFORM_DATA_LABEL_EXPLICIT_STOP_TIME:
            tLengthInBits = 8;
            break;
        case ALERTC_FREEFORM_DATA_LABEL_ADDITIONAL_EVENT:
            tLengthInBits = 11;
            break;
        case ALERTC_FREEFORM_DATA_LABEL_DETAILED_DIVERSION_ADVISE:
        case ALERTC_FREEFORM_DATA_LABEL_DESTINATION:
        case ALERTC_FREEFORM_DATA_LABEL_RESERVED_12:
        case ALERTC_FREEFORM_DATA_LABEL_SOURCE_OF_PROBLEM:
            tLengthInBits = 16;
            break;
        case ALERTC_FREEFORM_DATA_LABEL_SEPARATOR:
        case ALERTC_FREEFORM_DATA_LABEL_RESERVED_15:
        default:
            // Length of zero
            break;
    }

    return tLengthInBits;
}

/*****************************************************************************
*
*   bParseFreeFormData
*
*****************************************************************************/
static BOOLEAN bParseFreeFormData (
    TRAFFIC_MSG_OBJECT_STRUCT *psObj,
    OSAL_BUFFER_HDL hPayload
        )
{
    size_t tBitsRead = 0, tTotalRead = 0, tFieldLength;
    ALERTC_FREEFORM_DATA_LABEL_ENUM eLabel =
        (ALERTC_FREEFORM_DATA_LABEL_ENUM)0;
    UN16 un16Value = 0;
    BOOLEAN bRead, bDataUsed;

    // Read the free form data that is available
    while ( tTotalRead< FREEFORM_TOTAL_DATA_BITLEN)
    {
        // Read the label
        tTotalRead += OSAL.tBufferReadHeadBits(hPayload,
                                             &eLabel,
                                             0,
                                             FREEFORM_LABEL_FIELD_BITLEN);

        tFieldLength = tFreeFormDataLength(eLabel);

        if (tTotalRead + tFieldLength >= FREEFORM_TOTAL_DATA_BITLEN &&
            eLabel == ALERTC_FREEFORM_DATA_LABEL_DURATION)
        {
            // We ate some zero padding.  Exit here since there is nothing
            // left to read
            puts(TRAFFIC_MSG_OBJECT_NAME": Not enough bits left to read.  Bailing\n");
            break;
        }

        // Read the value for this label
        un16Value = 0;

        // Use the function that reads values with respect to endiness,
        // regardless of the field length.
        bRead =
            OSAL.bBufferReadBitsToUN16(hPayload, &un16Value, tFieldLength);

        if (bRead == TRUE)
        {
            tBitsRead = tFieldLength;
        }
        else
        {
            // ERROR
            return FALSE;
        }

        // Update our total bits read count
        tTotalRead += tBitsRead;

        // Check if we reached the end of the free form data
        // (label = duration, and value=0 (illegal))
        // or if there isn't enough bits left to read a full label length
        if ( (eLabel == ALERTC_FREEFORM_DATA_LABEL_DURATION &&
               un16Value == 0) ||
             ( (tTotalRead + FREEFORM_LABEL_FIELD_BITLEN) > FREEFORM_TOTAL_DATA_BITLEN)
            )
        {
            // Seek to the end of this message
            // This will all be zero padding
            tTotalRead +=
                OSAL.tBufferSeekHeadBits(hPayload,
                                         FREEFORM_TOTAL_DATA_BITLEN-tTotalRead
                                         );
        }
        else
        {
            // See if we can use this field ourselves, or if it should
            // go in the linked-list of data.
            bDataUsed = bUpdateFieldsBasedOnFreeFormData(psObj, eLabel, un16Value);
            if (bDataUsed == FALSE)
            {
                // we didn't use the free-form data to update one of our
                // fields, so hold on to it since the application may need it
                vAddFreeFormData(psObj, eLabel, un16Value);
            }

        }
        printf(TRAFFIC_MSG_OBJECT_NAME": Freeform Data Found: %u-%u", eLabel, un16Value);
        printf(TRAFFIC_MSG_OBJECT_NAME": Total bits of freeform data read %d\n", tTotalRead);
    }

    return TRUE;
}

/*****************************************************************************
*
*   bUpdateFieldsBasedOnFreeFormData
*
*****************************************************************************/
static BOOLEAN bUpdateFieldsBasedOnFreeFormData (
    TRAFFIC_MSG_OBJECT_STRUCT *psObj,
    ALERTC_FREEFORM_DATA_LABEL_ENUM eLabel,
    UN16 un16Value
        )
{
    BOOLEAN bUsed = FALSE;

    switch(eLabel)
    {
        case ALERTC_FREEFORM_DATA_LABEL_CONTROL:
        {
            switch(un16Value)
            {
                case ALERTC_FREEFORM_DATA_CONTROL_DIVERSION_BIT_SET:
                    psObj->sAlertC.bDiversionAdvised = TRUE;
                    bUsed = TRUE;
                    break;
                case ALERTC_FREEFORM_DATA_CONTROL_INCREASE_EXTENT_BY_8:
                    psObj->sAlertC.un8Extent += 8;
                    bUsed = TRUE;
                    break;
                case ALERTC_FREEFORM_DATA_CONTROL_INCREASE_EXTENT_BY_16:
                    psObj->sAlertC.un8Extent += 16;
                    bUsed = TRUE;
                    break;
                    //TODO: Other control codes to handle?
                default:
                    break;
            }
        }
        break;
        case ALERTC_FREEFORM_DATA_LABEL_DURATION:
        {
            psObj->sAlertC.un8Duration = (UN8)un16Value;
            bUsed = TRUE;
        }
        break;
        //TODO: Other free form labels to handle?
        default:
        break;
    }

    return bUsed;

}

/*****************************************************************************
*
*   vAddFreeFormData
*
*****************************************************************************/
static void vAddFreeFormData (
    TRAFFIC_MSG_OBJECT_STRUCT *psObj,
    ALERTC_FREEFORM_DATA_LABEL_ENUM eLabel,
    UN16 un16Value
        )
{
    OSAL_RETURN_CODE_ENUM eReturnCode;
    OSAL_LINKED_LIST_ENTRY hEntry = OSAL_INVALID_LINKED_LIST_ENTRY;
    ALERTC_FREEFORM_DATA_STRUCT *psFreeFormData = NULL;

    if (psObj->sAlertC.hFreeFormDataList == OSAL_INVALID_OBJECT_HDL)
    {
        eReturnCode =
            OSAL.eLinkedListCreate(
                &psObj->sAlertC.hFreeFormDataList,
                TRAFFIC_MSG_OBJECT_NAME":FFList",
                (OSAL_LL_COMPARE_HANDLER)n16CompareFreeFormData,
                OSAL_LL_OPTION_LINEAR
                    );

        if (eReturnCode != OSAL_SUCCESS)
        {
            return;
        }

        psFreeFormData = psCreateFreeFormData(psObj);
    }
    else
    {
        ALERTC_FREEFORM_DATA_STRUCT sSearchCriteria;

        // looking for unassigned entry
        sSearchCriteria.bInUse = FALSE;

        // We have a list, find a reusable entry
        eReturnCode = OSAL.eLinkedListSearch(
            psObj->sAlertC.hFreeFormDataList,
            &hEntry,
            &sSearchCriteria
                );
        if(eReturnCode == OSAL_SUCCESS)
        {
            // Found one! Let's use it.
            psFreeFormData = (ALERTC_FREEFORM_DATA_STRUCT *)
                OSAL.pvLinkedListThis(hEntry);

            psFreeFormData->bInUse = TRUE;

            // Replace this entry marking it as unavailable for recycling.
            eReturnCode = OSAL.eLinkedListReplaceEntry(
                psObj->sAlertC.hFreeFormDataList,
                hEntry,
                (void *)psFreeFormData
                    );
            if (eReturnCode != OSAL_SUCCESS)
            {
                 // Error!
                 printf(TRAFFIC_MSG_OBJECT_NAME
                 ": Error! Cannot replace entry.\n");
            }
        }
        else if (eReturnCode == OSAL_OBJECT_NOT_FOUND)
        {
            psFreeFormData = psCreateFreeFormData(psObj);
        }
    }

    // Set our data
    if (psFreeFormData != NULL)
    {
        psFreeFormData->eLabel = eLabel;
        psFreeFormData->un16Value = un16Value;
    }

    return;
}

/*****************************************************************************
*
*   psCreateFreeFormData
*
*****************************************************************************/
static ALERTC_FREEFORM_DATA_STRUCT *psCreateFreeFormData(
    TRAFFIC_MSG_OBJECT_STRUCT *psObj
        )
{
    ALERTC_FREEFORM_DATA_STRUCT *psFreeFormData = NULL;

    psFreeFormData = (ALERTC_FREEFORM_DATA_STRUCT *)SMSO_hCreate(
        TRAFFIC_MSG_OBJECT_NAME":FFData",
        sizeof(ALERTC_FREEFORM_DATA_STRUCT),
        psObj->hParent,
        FALSE);

    if (psFreeFormData == NULL)
    {
        return NULL;
    }

    // Set that the data is active
    psFreeFormData->bInUse = TRUE;

    OSAL.eLinkedListAdd(psObj->sAlertC.hFreeFormDataList,
        OSAL_INVALID_LINKED_LIST_ENTRY_PTR, psFreeFormData);

    return psFreeFormData;
}

/*****************************************************************************
*
*   n16CompareTrafficMsgPool
*
*****************************************************************************/
static N16 n16CompareTrafficMsgPool (
    TRAFFIC_MSG_OBJECT hTrafficMsg1,
    TRAFFIC_MSG_OBJECT hTrafficMsg2
        )
{
    N16 n16Return;
    TRAFFIC_MSG_OBJECT_STRUCT *psObj1 = NULL, *psObj2 = NULL;

    psObj1 = (TRAFFIC_MSG_OBJECT_STRUCT *)TRAFFIC_MSG_BASE_pvObjectData(
        hTrafficMsg1);
    psObj2 = (TRAFFIC_MSG_OBJECT_STRUCT *)TRAFFIC_MSG_BASE_pvObjectData(
        hTrafficMsg2);

    if ((psObj1 == NULL) || (psObj2 == NULL))
    {
        return N16_MIN;
    }

    // Order Traffic Messages within the pool first by availability

    // Check first based on availability. Allow unassigned
    // Traffic Messages to percolate to the top (front)
    if (psObj1->bInUse == FALSE && psObj2->bInUse == FALSE)
    {
        // Both are unassigned (available)
        n16Return = 0;
    }
    else if(psObj1->bInUse == FALSE)
    {
        // psObj1 is less than (before) psObj2
        n16Return = -1;
    }
    else
    {
        // psObj1 is greater than (after) psObj2
        n16Return = 1;
    }

    return n16Return;
}

/*****************************************************************************
*
*   n16CompareFreeFormData
*
*****************************************************************************/
static N16 n16CompareFreeFormData (
    const ALERTC_FREEFORM_DATA_STRUCT *psObj1,
    const ALERTC_FREEFORM_DATA_STRUCT *psObj2
        )
{
    N16 n16Return;

    // Order free form data by usibility

    // Check first based on availability. Allow unassigned
    // data to percolate to the top (front)
    if (psObj1->bInUse == FALSE && psObj2 == FALSE)
    {
        // Both are unassigned (available)
        n16Return = 0;
    }
    else if(psObj1->bInUse == FALSE)
    {
        // psObj1 is less than (before) psObj2
        n16Return = -1;
    }
    else
    {
        // psObj1 is greater than (after) psObj2
        n16Return = 1;
    }

    return n16Return;
}

/*****************************************************************************
*
*   n16FindTrafficMsgToReuse
*
*****************************************************************************/
static N16 n16FindTrafficMsgToReuse (
    TRAFFIC_MSG_OBJECT hTrafficMsg,
    const void *pvData
        )
{
    N16 n16Return = N16_MIN;

    TRAFFIC_MSG_OBJECT_STRUCT *psObj = NULL;

    psObj =
        (TRAFFIC_MSG_OBJECT_STRUCT *)TRAFFIC_MSG_BASE_pvObjectData(hTrafficMsg);

    if (psObj != NULL)
    {
        // looking for unassigned item
        n16Return = (psObj->bInUse == FALSE) ? 0 : -1;
    }

    return n16Return;
}


/*****************************************************************************
*
*   bMarkFreeFormDataInactive
*
*****************************************************************************/
static BOOLEAN bMarkFreeFormDataInactive(
    ALERTC_FREEFORM_DATA_STRUCT *psObj,
    const void *pvData
        )
{
    if (psObj->bInUse == TRUE)
    {
        // Mark item as not in use
        psObj->bInUse = FALSE;
    }

    return TRUE;

}

/*****************************************************************************
*
*   bAddFreeFormDataToBlob
*
*****************************************************************************/
static BOOLEAN bAddFreeFormDataToBlob(
    ALERTC_FREEFORM_DATA_STRUCT *psObj,
    TRAFFIC_CREATE_FREE_FORM_BLOB_ITERATOR_STRUCT *psIteratorArg
        )
{
    if (psObj->bInUse == TRUE)
    {
        if ( (psIteratorArg->un8FreeFormDataCount * sizeof(UN16) * 2) <
             (psIteratorArg->tBlobDataSize - (sizeof(UN16) * 2))
           )
        {
            *psIteratorArg->pun16BlobData++ = (UN16)psObj->eLabel;
            *psIteratorArg->pun16BlobData++ = psObj->un16Value;

            printf("Writing free-form data to blob: %u-%u",
                psObj->eLabel, psObj->un16Value);

            psIteratorArg->un8FreeFormDataCount++;
        }
    }

    return TRUE;
}

/*****************************************************************************
*
*   un32GetStartStopTime
*
*****************************************************************************/
static UN32 un32GetStartStopTime (
    TRAFFIC_MSG_OBJECT_STRUCT *psObj,
    ALERTC_FREEFORM_DATA_LABEL_ENUM eLabel
        )
{
    UN32 un32Time = 0;
    UN8 un8NumTimes;
    UN16 un16TimeValue;
    UN32 un32OSALTimeInSecs;
    OSAL_RETURN_CODE_ENUM eReturnCode;
    struct tm sUTCTime;
    TIME_T tSecs;

    do
    {
        if (psObj == NULL)
        {
            break;
        }

        if ( (eLabel != ALERTC_FREEFORM_DATA_LABEL_EXPLICIT_START_TIME) &&
             (eLabel != ALERTC_FREEFORM_DATA_LABEL_EXPLICIT_STOP_TIME)
           )
        {
            break;
        }

        un8NumTimes =
            LEGACY_TRAFFIC_MSG.un8GetFreeFormData(
                (TRAFFIC_MSG_BASE_OBJECT)psObj,
                eLabel,
                &un16TimeValue,
                MAX_NUM_EXPLICIT_TIMES);

        if (un8NumTimes != MAX_NUM_EXPLICIT_TIMES)
        {
            break;
        }

        // Get current time from OSAL
        eReturnCode = OSAL.eTimeGet(&un32OSALTimeInSecs);

        if (eReturnCode != OSAL_SUCCESS)
        {
            printf("Result: %s\n",
                   OSAL.pacGetReturnCodeName(eReturnCode));
            break;
        }

        // Get our unixtime as a tm struct
        tSecs = un32OSALTimeInSecs;
        OSAL.gmtime_r(&tSecs, &sUTCTime);

        // Zero out wday since we'll let mktime figure that out
        sUTCTime.tm_wday = 0;

        // Now adjust our tm struct using un16TimeValue
        // according to sec 5.5.8 of the  Alert-C Spec
        if ( (un16TimeValue >= EXPLICIT_TIME_15MIN_START) &&
             (un16TimeValue <= EXPLICIT_TIME_15MIN_STOP)
           )
        {
            // un16TimeValue represents the number
            // of 15 min intervals from midnight of the current
            // day
            UN32 un32NumMinutes = un16TimeValue * 15;

            // Set the hours, minutes and seconds
            sUTCTime.tm_hour = un32NumMinutes / 60;
            sUTCTime.tm_min = un32NumMinutes % 60;
            sUTCTime.tm_sec = 0;

        }
        else if ( (un16TimeValue >= EXPLICIT_TIME_1HOUR_START) &&
                  (un16TimeValue <= EXPLICIT_TIME_1HOUR_STOP)
                )
        {
            // un16TimeValue represents the number of hours
            // since midnight of the next day in one hour
            // incremenets
            UN16 un16NumDays = un16TimeValue / 24;
            UN16 un16NumHours = un16TimeValue % 24;

            // We'll just use tm_yday and tm_hour and let
            // mktime handle the rest.  So zero out other
            // fields
            sUTCTime.tm_min = 0;
            sUTCTime.tm_sec = 0;
            sUTCTime.tm_mon = 0;
            sUTCTime.tm_mday = 0;

            // Increment the number of days in the year
            // and set the hour
            sUTCTime.tm_yday += un16NumDays;
            sUTCTime.tm_hour = un16NumHours;

            // Handle rollover.  Not sure if it can
            // happen based on the Alert-C spec, but it
            // doesn't say that it can't happen, so we'll
            // be "safe"
            if (IS_LEAP_YEAR(sUTCTime.tm_year) == TRUE)
            {
                if (sUTCTime.tm_yday > 366)
                {
                    sUTCTime.tm_yday -= 366;
                    sUTCTime.tm_year++;
                }
            }
            else
            {
                if (sUTCTime.tm_yday > 365)
                {
                    sUTCTime.tm_yday -= 365;
                    sUTCTime.tm_year++;
                }
            }
        }
        else if ( (un16TimeValue >= EXPLICIT_TIME_1DAY_START) &&
                  (un16TimeValue <= EXPLICIT_TIME_1DAY_STOP)
                )
        {
            UN8 un8CurMDay;

            // un16TimeValue indicates a date during the next 31 days.
            // For example, a stop time received on the 20th August
            // with a code of 218 means "until 18th September".
            //
            // Adjust un16TimeValue to a value between 1 and 31 using
            // the offset value
            un16TimeValue -= EXPLICIT_TIME_1DAY_OFFSET;

            // Zero out the time and yday in our struct
            sUTCTime.tm_hour = 0;
            sUTCTime.tm_min = 0;
            sUTCTime.tm_sec = 0;
            sUTCTime.tm_yday = 0;

            // set the mday to the day listed in un16TimeValue
            // but save the current one first
            un8CurMDay = (UN8)sUTCTime.tm_mday;
            sUTCTime.tm_mday = un16TimeValue;

            // Adjust the month if need be
            if ( ((UN8)sUTCTime.tm_mday) < un8CurMDay)
            {
                sUTCTime.tm_mon++;

                if (sUTCTime.tm_mon > MONTH_TIME_VALUE_DEC)
                {
                    sUTCTime.tm_mon = MONTH_TIME_VALUE_JAN;
                    sUTCTime.tm_year++;
                }
            }
        }
        else if ( (un16TimeValue >= EXPLICIT_TIME_HALF_MONTH_START) &&
                  (un16TimeValue <= EXPLICIT_TIME_HALF_MONTH_STOP)
                )
        {
            // un16TimeValue is a half-month interval from Jan15th - Dec31st

            // Save the current month to adjust if need be at the end
            UN8 un8CurMonth = (UN8)sUTCTime.tm_mon;

            // Zero out the time and the year days
            sUTCTime.tm_hour = 0;
            sUTCTime.tm_min = 0;
            sUTCTime.tm_sec = 0;
            sUTCTime.tm_yday = 0;
            sUTCTime.tm_isdst = 0; // No dst in effect for this time

            switch ((EXPLICIT_TIME_VALUE_ENUM)un16TimeValue)
            {
                case EXPLICIT_TIME_VALUE_JAN_15:
                {
                    sUTCTime.tm_mon = MONTH_TIME_VALUE_JAN;
                    sUTCTime.tm_mday = 15;
                }
                break;

                case EXPLICIT_TIME_VALUE_JAN_31:
                {
                    sUTCTime.tm_mon = MONTH_TIME_VALUE_JAN;
                    sUTCTime.tm_mday =
                        NUM_DAYS_IN_MONTH(MONTH_TIME_VALUE_JAN, sUTCTime.tm_year);
                }
                break;

                case EXPLICIT_TIME_VALUE_FEB_15:
                {
                    sUTCTime.tm_mon = MONTH_TIME_VALUE_FEB;
                    sUTCTime.tm_mday = 15;
                }
                break;

                case EXPLICIT_TIME_VALUE_FEB_END:
                {
                    sUTCTime.tm_mon = MONTH_TIME_VALUE_FEB;
                    sUTCTime.tm_mday =
                        NUM_DAYS_IN_MONTH(MONTH_TIME_VALUE_FEB, sUTCTime.tm_year);
                }
                break;

                case EXPLICIT_TIME_VALUE_MAR_15:
                {
                    sUTCTime.tm_mon = MONTH_TIME_VALUE_MAR;
                    sUTCTime.tm_mday = 15;
                }
                break;

                case EXPLICIT_TIME_VALUE_MAR_31:
                {
                    sUTCTime.tm_mon = MONTH_TIME_VALUE_MAR;
                    sUTCTime.tm_mday =
                        NUM_DAYS_IN_MONTH(MONTH_TIME_VALUE_MAR, sUTCTime.tm_year);
                }
                break;

                case EXPLICIT_TIME_VALUE_APR_15:
                {
                    sUTCTime.tm_mon = MONTH_TIME_VALUE_APR;
                    sUTCTime.tm_mday = 15;
                }
                break;

                case EXPLICIT_TIME_VALUE_APR_30:
                {
                    sUTCTime.tm_mon = MONTH_TIME_VALUE_APR;
                    sUTCTime.tm_mday =
                        NUM_DAYS_IN_MONTH(MONTH_TIME_VALUE_APR, sUTCTime.tm_year);
                }
                break;

                case EXPLICIT_TIME_VALUE_MAY_15:
                {
                    sUTCTime.tm_mon = MONTH_TIME_VALUE_MAY;
                    sUTCTime.tm_mday = 15;
                }
                break;

                case EXPLICIT_TIME_VALUE_MAY_31:
                {
                    sUTCTime.tm_mon = MONTH_TIME_VALUE_MAY;
                    sUTCTime.tm_mday =
                        NUM_DAYS_IN_MONTH(MONTH_TIME_VALUE_MAY, sUTCTime.tm_year);
                }
                break;

                case EXPLICIT_TIME_VALUE_JUN_15:
                {
                    sUTCTime.tm_mon = MONTH_TIME_VALUE_JUN;
                    sUTCTime.tm_mday = 15;
                }
                break;

                case EXPLICIT_TIME_VALUE_JUN_30:
                {
                    sUTCTime.tm_mon = MONTH_TIME_VALUE_JUN;
                    sUTCTime.tm_mday =
                        NUM_DAYS_IN_MONTH(MONTH_TIME_VALUE_JUN, sUTCTime.tm_year);
                }
                break;

                case EXPLICIT_TIME_VALUE_JUL_15:
                {
                    sUTCTime.tm_mon = MONTH_TIME_VALUE_JUL;
                    sUTCTime.tm_mday = 15;
                }
                break;

                case EXPLICIT_TIME_VALUE_JUL_31:
                {
                    sUTCTime.tm_mon = MONTH_TIME_VALUE_JUL;
                    sUTCTime.tm_mday =
                        NUM_DAYS_IN_MONTH(MONTH_TIME_VALUE_JUL, sUTCTime.tm_year);
                }
                break;

                case EXPLICIT_TIME_VALUE_AUG_15:
                {
                    sUTCTime.tm_mon = MONTH_TIME_VALUE_AUG;
                    sUTCTime.tm_mday = 15;
                }
                break;

                case EXPLICIT_TIME_VALUE_AUG_31:
                {
                    sUTCTime.tm_mon = MONTH_TIME_VALUE_AUG;
                    sUTCTime.tm_mday =
                        NUM_DAYS_IN_MONTH(MONTH_TIME_VALUE_AUG, sUTCTime.tm_year);
                }
                break;

                case EXPLICIT_TIME_VALUE_SEP_15:
                {
                    sUTCTime.tm_mon = MONTH_TIME_VALUE_SEP;
                    sUTCTime.tm_mday = 15;
                }
                break;

                case EXPLICIT_TIME_VALUE_SEP_30:
                {
                    sUTCTime.tm_mon = MONTH_TIME_VALUE_SEP;
                    sUTCTime.tm_mday =
                        NUM_DAYS_IN_MONTH(MONTH_TIME_VALUE_SEP, sUTCTime.tm_year);
                }
                break;

                case EXPLICIT_TIME_VALUE_OCT_15:
                {
                    sUTCTime.tm_mon = MONTH_TIME_VALUE_OCT;
                    sUTCTime.tm_mday = 15;
                }
                break;

                case EXPLICIT_TIME_VALUE_OCT_31:
                {
                    sUTCTime.tm_mon = MONTH_TIME_VALUE_OCT;
                    sUTCTime.tm_mday =
                        NUM_DAYS_IN_MONTH(MONTH_TIME_VALUE_OCT, sUTCTime.tm_year);
                }
                break;

                case EXPLICIT_TIME_VALUE_NOV_15:
                {
                    sUTCTime.tm_mon = MONTH_TIME_VALUE_NOV;
                    sUTCTime.tm_mday = 15;
                }
                break;

                case EXPLICIT_TIME_VALUE_NOV_30:
                {
                    sUTCTime.tm_mon = MONTH_TIME_VALUE_NOV;
                    sUTCTime.tm_mday =
                        NUM_DAYS_IN_MONTH(MONTH_TIME_VALUE_NOV, sUTCTime.tm_year);
                }
                break;

                case EXPLICIT_TIME_VALUE_DEC_15:
                {
                    sUTCTime.tm_mon = MONTH_TIME_VALUE_DEC;
                    sUTCTime.tm_mday = 15;
                }
                break;

                case EXPLICIT_TIME_VALUE_DEC_31:
                {
                    sUTCTime.tm_mon = MONTH_TIME_VALUE_DEC;
                    sUTCTime.tm_mday =
                        NUM_DAYS_IN_MONTH(MONTH_TIME_VALUE_DEC, sUTCTime.tm_year);
                }
                break;

                default:
                    break;
            }

            // If the new month is less than the
            // current month, increment the year
            // since it must be for the next year
            if (((UN8)sUTCTime.tm_mon) < un8CurMonth)
            {
                sUTCTime.tm_year++;
            }
        }
        else
        {
            // Error!
            break;
        }

        // Now turn our tm back into a unixtime
        un32Time = (UN32)OSAL.mktime(&sUTCTime); // calendar time

    } while (FALSE);

    return un32Time;
}

/*****************************************************************************
*
*   hGetTrafficMgr
*
*****************************************************************************/
static TRAFFIC_SERVICE_OBJECT hGetTrafficMgr (
    TRAFFIC_MSG_OBJECT_STRUCT *psObj
        )
{
    TRAFFIC_SERVICE_OBJECT hTrafficService =
        TRAFFIC_SERVICE_INVALID_OBJECT;
    BOOLEAN bLocked;

    // Lock the Traffic Msg Control object
    bLocked = SMSO_bLock(
        (SMS_OBJECT)gpsTrafficMsgCtrl, OSAL_OBJ_TIMEOUT_INFINITE);

    if (bLocked == TRUE)
    {
        // The traffic service is the owner of the Traffic Message object
        hTrafficService = gpsTrafficMsgCtrl->hTrafficMgr;

        SMSO_vUnlock((SMS_OBJECT)gpsTrafficMsgCtrl);
    }

    return hTrafficService;
}

/*****************************************************************************
*
*   n16IsFreeFormDataEqual
*
*****************************************************************************/
static N16 n16IsFreeFormDataEqual (
    const void *pvObj1,
    const void *pvObj2
        )
{
    ALERTC_FREEFORM_DATA_STRUCT *psFreeFormData =
        (ALERTC_FREEFORM_DATA_STRUCT *)pvObj1;
    ALERTC_FREEFORM_DATA_LABEL_ENUM eLabel =
        (ALERTC_FREEFORM_DATA_LABEL_ENUM)(size_t) pvObj2;

    N16 n16Result = -1;

    if ( (eLabel == psFreeFormData->eLabel) &&
         (psFreeFormData->bInUse == TRUE)  )
    {
        n16Result = 0;
    }

    return n16Result;
}

/*****************************************************************************
*
*   vReleaseObject
*
*****************************************************************************/
static void vReleaseObject(
    TRAFFIC_MSG_OBJECT hTrafficMsg
        )
{
    if (hTrafficMsg != TRAFFIC_MSG_INVALID_OBJECT)
    {
        TRAFFIC_MSG_OBJECT_STRUCT *psObj;

        psObj = (TRAFFIC_MSG_OBJECT_STRUCT *)TRAFFIC_MSG_BASE_pvObjectData(
            hTrafficMsg);

        if(NULL != psObj)
        {
            vDestroyObject(psObj);
        }
        TRAFFIC_MSG_BASE_vDestroy(hTrafficMsg);
    }

    return;
}

/*****************************************************************************
*
*       pacTypeText
*
* This is a local function which simply maps an enumerated type to
* a textual representation for formatting the enumerated type.
*
*****************************************************************************/
static const char *pacTypeText(
    TRAFFIC_MSG_TYPE_ENUM eType,
    SMSAPI_OUTPUT_OPTION_ENUM eOutputOption
        )
{
    const char *pacReturnString;

    switch (eType)
    {
        case TRAFFIC_MSG_TYPE_INCIDENT:
            if (eOutputOption == SMS_OUTPUT_OPTION_TERSE)
            {
                pacReturnString = "I";
            }
            else
            {
                pacReturnString =
                    MACRO_TO_STRING(TRAFFIC_MSG_TYPE_INCIDENT);
            }
        break;

        case TRAFFIC_MSG_TYPE_SPEED_AND_FLOW:
            if (eOutputOption == SMS_OUTPUT_OPTION_TERSE)
            {
                pacReturnString = "SF";
            }
            else
            {
                pacReturnString =
                    MACRO_TO_STRING(TRAFFIC_MSG_TYPE_SPEED_AND_FLOW);
            }
        break;

        case TRAFFIC_MSG_TYPE_UNKNOWN:
        default:
            if (eOutputOption == SMS_OUTPUT_OPTION_TERSE)
            {
                pacReturnString = "U";
            }
            else
            {
                pacReturnString =
                    MACRO_TO_STRING(TRAFFIC_MSG_TYPE_UNKNOWN);
            }
        break;
    }

    return pacReturnString;
}

/*****************************************************************************
*
*       pacClassText
*
* This is a local function which simply maps an enumerated type to
* a textual representation for formatting the enumerated type.
*
*****************************************************************************/
static const char *pacClassText(
    TRAFFIC_MSG_CLASS_ENUM eClass
        )
{
    const char *pacReturnString = NULL;

    if (eClass < TRAFFIC_MSG_CLASS_COUNT)
    {
        pacReturnString =
            gasMsgClassOutput[eClass].pcFullOutput;
    }

    return pacReturnString;
}

/*****************************************************************************
*
*       pacDurationText
*
* This is a local function which simply maps an enumerated type to
* a textual representation for formatting the enumerated type.
*
*****************************************************************************/
static const char *pacDurationText(
    TRAFFIC_DURATION_ENUM eDuration,
    SMSAPI_OUTPUT_OPTION_ENUM eOutputOption
        )
{
    const char *pacReturnString = NULL;

    if (eDuration < TRAFFIC_DURATION_COUNT)
    {
        if (eOutputOption == SMS_OUTPUT_OPTION_TERSE)
        {
            pacReturnString =
                gasDurationOutput[eDuration].pcTerseOutput;
        }
        else
        {
            pacReturnString =
                gasDurationOutput[eDuration].pcFullOutput;
        }
    }

    return pacReturnString;
}

/*****************************************************************************
*
*       pacDirectionText
*
* This is a local function which simply maps an enumerated type to
* a textual representation for formatting the enumerated type.
*
*****************************************************************************/
static const char *pacDirectionText(
    TRAFFIC_DIRECTION_ENUM eDirection,
    SMSAPI_OUTPUT_OPTION_ENUM eOutputOption
        )
{
    const char *pacReturnString;

    switch (eDirection)
    {
        case TRAFFIC_DIRECTION_POSITIVE:
            if (eOutputOption == SMS_OUTPUT_OPTION_TERSE)
            {
                pacReturnString = "+";
            }
            else
            {
                pacReturnString =
                    MACRO_TO_STRING(TRAFFIC_DIRECTION_POSITIVE);
            }
        break;

        case TRAFFIC_DIRECTION_NEGATIVE:
            if (eOutputOption == SMS_OUTPUT_OPTION_TERSE)
            {
                pacReturnString = "-";
            }
            else
            {
                pacReturnString =
                    MACRO_TO_STRING(TRAFFIC_DIRECTION_NEGATIVE);
            }
        break;

        case TRAFFIC_DIRECTION_UNKNOWN:
        default:
            if (eOutputOption == SMS_OUTPUT_OPTION_TERSE)
            {
                pacReturnString = "?";
            }
            else
            {
                pacReturnString =
                    MACRO_TO_STRING(TRAFFIC_DIRECTION_UNKNOWN);
            }
        break;
    }

    return pacReturnString;
}

