/******************************************************************************/
/*                    Copyright (c) Sirius XM Radio, Inc.                     */
/*                            All Rights Reserved                             */
/*         Licensed Materials - Property of Sirius XM Radio, Inc.             */
/******************************************************************************/
/*******************************************************************************
 *
 * DESCRIPTION
 *
 *  This module contains the Object:TRAFFIC_MSG_IMAGE implementation for the
 *  Simple Module Services (SMS)
 *
 ******************************************************************************/

#include "sms_api.h"

#include <string.h>

#include "image_obj.h"
#include "string_obj.h"
#include "location_obj.h"

#include "_traffic_msg_image_obj.h"
#include "traffic_msg_image_obj.h"
#include "traffic_msg_base_obj.h"
#include "trafcam_interface.h"

#include "sms_api_debug.h"
static const char *gpacThisFile = __FILE__;

/*****************************************************************************
 *                           PUBLIC FUNCTIONS
 *****************************************************************************/

/*****************************************************************************
 *
 *   eType
 *
 *****************************************************************************/
static TRAFFIC_MSG_TYPE_ENUM eType(
    TRAFFIC_MSG_BASE_OBJECT hTrafficBaseMsg
        )
{
    return TRAFFIC_MSG_TYPE_CAMERA_IMAGE;
}

/*****************************************************************************
 *
 *   hText
 *
 *****************************************************************************/
static STRING_OBJECT hText (
    TRAFFIC_MSG_BASE_OBJECT hTrafficBaseMsg
        )
{
    return ((TRAFFIC_MSG_IMAGE_OBJECT_STRUCT *)hTrafficBaseMsg)->hText;
}

/*****************************************************************************
 *
 *   un32StartTime
 *
 *****************************************************************************/
static UN32 un32StartTime (
    TRAFFIC_MSG_BASE_OBJECT hTrafficBaseMsg
        )
{
    return ((TRAFFIC_MSG_IMAGE_OBJECT_STRUCT *)
        hTrafficBaseMsg)->un32ProcessTimeStamp;
}

/*****************************************************************************
 *
 *   un8NumLocations
 *
 *****************************************************************************/
static UN8 un8NumLocations (
    TRAFFIC_MSG_BASE_OBJECT hTrafficBaseMsg
        )
{
    return 1;
}

/*****************************************************************************
 *
 *   hGetLocation
 *
 *****************************************************************************/
static LOCATION_OBJECT hGetLocation (
    TRAFFIC_MSG_BASE_OBJECT hTrafficBaseMsg,
    UN8 un8Index
        )
{
    if (un8Index == 0)
    {
        return ((TRAFFIC_MSG_IMAGE_OBJECT_STRUCT *)hTrafficBaseMsg)->hLocation;
    }
    else
    {
        return LOCATION_INVALID_OBJECT;
    }
}

/*****************************************************************************
 *
 *   n32FPrintf
 *
 *****************************************************************************/
static N32 n32FPrintf (
    TRAFFIC_MSG_BASE_OBJECT hTrafficBaseMsg,
    FILE *psFile,
    SMSAPI_OUTPUT_OPTION_ENUM eOutputOption
        )
{
    N32 n32Return = 0;
    LOCATION_OBJECT hLocation;
    TRAFFIC_LOCID_OBJECT hLocID;
    TRAFFIC_POS_CODE tPosCode;
    TRAFFIC_MARKET tMarket;
    TRAFFIC_BSA tBSA;
    TRAFFIC_MSG_IMAGE_OBJECT_STRUCT *psObj =
        (TRAFFIC_MSG_IMAGE_OBJECT_STRUCT *)hTrafficBaseMsg;
    IMAGE_OBJECT hImage;

    STRING_OBJECT hText = STRING_INVALID_OBJECT;
    TIME_T tProcessTime = 0;
    char acDescriptionText[TRAFFIC_MSG_IMAGE_DESC_TEXT_OUTPUT_LENGTH] = "N/A";
    char acImagePath[TRAFFIC_MSG_IMAGE_PATH_OUTPUT_LENGTH];
    size_t tBytesCopied;
    char acProcessTimeStamp[TRAFFIC_MSG_IMAGE_TIME_TEXT_LENGTH] = "N/A";
    UN16 un16Direction = 0;
    SMSAPI_RETURN_CODE_ENUM eReturnCode = SMSAPI_RETURN_CODE_ERROR;
    BOOLEAN bDirectionAvailable = FALSE;
    OSAL_FIXED_OBJECT hLat, hLon;

    // Verify inputs
    if (psFile == NULL)
    {
        // Error!
        return EOF;
    }

    hText = TRAFFIC_MSG_IMAGE.hText(hTrafficBaseMsg);
    tProcessTime = TRAFFIC_MSG_IMAGE.un32StartTime(hTrafficBaseMsg);

    if (hText != STRING_INVALID_OBJECT)
    {
        tBytesCopied = STRING.tCopyToCStr(hText, &acDescriptionText[0],
            sizeof(acDescriptionText));

        if (tBytesCopied == sizeof(acDescriptionText))
        {
            acDescriptionText[tBytesCopied - 1] = '\0';
        }
    }

    hImage = TRAFFIC_MSG_IMAGE.hImage(hTrafficBaseMsg);
    hText = IMAGE.hFileName(hImage);

    tBytesCopied =
        STRING.tCopyToCStr(hText, &acImagePath[0], sizeof(acImagePath));
    if (tBytesCopied == sizeof(acImagePath))
    {
        acImagePath[tBytesCopied - 1] = '\0';
    }

    hLocation = TRAFFIC_MSG_IMAGE.hGetLocation(hTrafficBaseMsg, 0);
    hLocID = LOCATION.hLocID(hLocation);
    tPosCode = TRAFFIC_LOCID.tPosCode(hLocID);
    tMarket = TRAFFIC_LOCID.tMarket(hLocID);
    tBSA = TRAFFIC_LOCID.tBSA(hLocID);
    hLat = LOCATION.hLat(hLocation);
    hLon = LOCATION.hLon(hLocation);

    eReturnCode = TRAFFIC_MSG_IMAGE.eCameraDirection(hTrafficBaseMsg,
        &un16Direction);
    if (eReturnCode == SMSAPI_RETURN_CODE_SUCCESS)
    {
       bDirectionAvailable = TRUE;
    }

    if (tProcessTime != 0)
    {
        OSAL.ctime_r(&tProcessTime, acProcessTimeStamp);
    }

    switch(eOutputOption)
    {
        case SMS_OUTPUT_OPTION_TERSE:
        {
            n32Return +=
                    fprintf(psFile,
                    "| %-2u | %-2u | %-49s |\n",
                    tMarket,
                    tBSA,
                    acImagePath);
        }
        break;
        case SMS_OUTPUT_OPTION_VERBOSE:
        case SMS_OUTPUT_OPTION_GROSS:
        {
            // Print TRAFFIC_MSG information
            n32Return += fprintf(psFile, "TRAFFIC_MSG: hTrafficMsg = 0x%p\n",
                    psObj);

            n32Return += fprintf(psFile, "\tMarket = %u\n", tMarket);

            n32Return += fprintf(psFile, "\tBSA = %u\n",  tBSA);

            if (tPosCode != TRAFFIC_INVALID_POS_CODE)
            {
                n32Return += fprintf(psFile, "\tPosCode = %u\n", tPosCode);
            }
            else
            {
                n32Return += fprintf(psFile, "\tPosCode = N/A\n");
            }

            if (bDirectionAvailable == TRUE)
            {
                n32Return += fprintf(psFile, "\tDirection = %u\n",
                    un16Direction);
            }

            n32Return += fprintf(psFile, "\thLat\n");
            n32Return += OSAL_FIXED.n32FPrintf(hLat, psFile, FALSE);

            n32Return += fprintf(psFile, "\thLon\n");
            n32Return += OSAL_FIXED.n32FPrintf(hLon, psFile, FALSE);

            n32Return += fprintf(psFile, "\tProcess Time = %s\n",
                acProcessTimeStamp);

            n32Return += fprintf(psFile, "\tTime To Expire = %u\n",
                psObj->un32TimeToExpire);

            n32Return += fprintf(psFile, "\tSeconds To Live = %u\n",
                psObj->un16TTL);

            n32Return += fprintf(psFile, "\tDescription = %s\n",
                acDescriptionText);
            n32Return += fprintf(psFile, "\tImage path = %s\n", acImagePath);

            n32Return += fprintf(psFile, "\n");

        }
        break;
        default:
        {
            return EOF;
        }

    }
    return n32Return;
}

/*****************************************************************************
 *
 *   eIterateLocations
 *
 *****************************************************************************/
static SMSAPI_RETURN_CODE_ENUM eIterateLocations (
    TRAFFIC_MSG_BASE_OBJECT hTrafficBaseMsg,
    TRAFFIC_MSG_LOCATION_ITERATOR bIterator,
    void *pvIteratorArg
        )
{
    // we have only one location for camera image, so single iterator call
    // is enough
    bIterator(((TRAFFIC_MSG_IMAGE_OBJECT_STRUCT *)hTrafficBaseMsg)->hLocation,
        pvIteratorArg);

    return SMSAPI_RETURN_CODE_SUCCESS;
}

/*****************************************************************************
 *
 *   eCameraDirection
 *
 *****************************************************************************/
static SMSAPI_RETURN_CODE_ENUM eCameraDirection (
    TRAFFIC_MSG_BASE_OBJECT hTrafficBaseMsg,
    UN16 *pun16Direction
        )
{
    SMSAPI_RETURN_CODE_ENUM eReturnCode = SMSAPI_RETURN_CODE_NOT_FOUND;
    UN16 un16Direction =
        ((TRAFFIC_MSG_IMAGE_OBJECT_STRUCT *)hTrafficBaseMsg)->un16Direction;

    if (un16Direction != TRAFCAM_INVALID_DIRECTION)
    {
        *pun16Direction = un16Direction;
        eReturnCode = SMSAPI_RETURN_CODE_SUCCESS;
    }

    return eReturnCode;
}


/*****************************************************************************
 *
 *   hImage
 *
 *****************************************************************************/
static IMAGE_OBJECT hImage (
    TRAFFIC_MSG_BASE_OBJECT hTrafficBaseMsg
        )
{
    return ((TRAFFIC_MSG_IMAGE_OBJECT_STRUCT *)hTrafficBaseMsg)->hImage;
}


/*****************************************************************************
 *                           FRIEND FUNCTIONS
 *****************************************************************************/

/*****************************************************************************
*
*   TRAFFIC_MSG_IMAGE_hCreate
*
*****************************************************************************/
TRAFFIC_MSG_OBJECT TRAFFIC_MSG_IMAGE_hCreate (
    SMS_OBJECT hOwner,
    size_t tDataSize,
    TRAFCAM_IMAGE_INFO_STRUCT *psImageInfo,
    const char *pacFilePath
        )
{
    TRAFFIC_MSG_OBJECT hTrafficMsg = TRAFFIC_MSG_INVALID_OBJECT;
    TRAFFIC_LOCID_OBJECT hLocID = TRAFFIC_LOCID_INVALID_OBJECT;

    do
    {
        TRAFFIC_MSG_IMAGE_OBJECT_STRUCT *psObj =
            (TRAFFIC_MSG_IMAGE_OBJECT_STRUCT*)NULL;

        N32 n32Lat, n32Lon;
        UN8 un8LatBits, un8LonBits;
        UN32 un32TimeStamp = 0;

        if ((psImageInfo == NULL) || (pacFilePath == NULL))
        {
            break;
        }

        hTrafficMsg = TRAFFIC_MSG_BASE_hCreate(
            (void **)&psObj,
            TRAFFIC_MSG_IMAGE_OBJECT_NAME"ObjData",
            hOwner,
            sizeof(TRAFFIC_MSG_IMAGE_OBJECT_STRUCT),
            tDataSize,
            &TRAFFIC_MSG_IMAGE);
        if (hTrafficMsg == TRAFFIC_MSG_INVALID_OBJECT)
        {
            break;
        }

        // copying TTL and calculating expiration time
        OSAL.vTimeUp(&un32TimeStamp, NULL);
        psObj->un16TTL = psImageInfo->un16TTL;
        psObj->un32TimeToExpire = un32TimeStamp + psObj->un16TTL;

        psObj->hText = psImageInfo->hDescription;
        psImageInfo->hDescription = STRING_INVALID_OBJECT;

        psObj->hImage = IMAGE_hCreate(hOwner, pacFilePath,
                            IMAGE_FORMAT_JPEG, &GsTrafficMsgImageIntf, NULL,
                            FALSE);
        if (psObj->hImage == IMAGE_INVALID_OBJECT)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                TRAFFIC_MSG_IMAGE_OBJECT_NAME": Failed to create image object"
                    );
            break;
        }

        hLocID = TRAFFIC_LOCID.hCreateExt(psImageInfo->tPosCode,
            psImageInfo->tMarket, psImageInfo->tBSA, psImageInfo->tOffset);
        if (hLocID == TRAFFIC_LOCID_INVALID_OBJECT)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                TRAFFIC_MSG_IMAGE_OBJECT_NAME": Failed to create loc id"
                    );
            break;
        }

        n32Lat = OSAL_FIXED.n32Value(psImageInfo->hLat);
        un8LatBits = OSAL_FIXED.un8NumFractionalBits(psImageInfo->hLat);
        n32Lon = OSAL_FIXED.n32Value(psImageInfo->hLon);
        un8LonBits = OSAL_FIXED.un8NumFractionalBits(psImageInfo->hLon);

        psObj->hLocation = LOCATION_hCreate(hOwner, hLocID, n32Lat, un8LatBits,
            n32Lon, un8LonBits, DISTANCE_INVALID_OBJECT,
            (LOCATION_ATTRIBUTE_STRUCT*)NULL, FALSE);
        if (psObj->hLocation == LOCATION_INVALID_OBJECT)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                TRAFFIC_MSG_IMAGE_OBJECT_NAME": Failed to create location"
                " object"
                    );
            break;
        }

        // set locid to invalid since it belongs to location object now
        hLocID = TRAFFIC_LOCID_INVALID_OBJECT;

        psObj->tHash = psImageInfo->tHash;
        psObj->un32ProcessTimeStamp = psImageInfo->un32ProcessTimeStamp;
        psObj->un16Direction = psImageInfo->un16Direction;

        return hTrafficMsg;

    } while (FALSE);

    // Error!
    // Free the string
    if ((psImageInfo != NULL) &&
        (psImageInfo->hDescription != STRING_INVALID_OBJECT))
    {
        STRING.vDestroy(psImageInfo->hDescription);
        psImageInfo->hDescription = STRING_INVALID_OBJECT;
    }

    // Free the LOCID
    if (hLocID != LOCID_INVALID_OBJECT)
    {
        TRAFFIC_LOCID.vDestroy(hLocID);
    }

    if (TRAFFIC_MSG_INVALID_OBJECT != hTrafficMsg)
    {
        TRAFFIC_MSG_IMAGE_vDestroy(hTrafficMsg);
    }

    return TRAFFIC_MSG_INVALID_OBJECT;
}

/*****************************************************************************
*
*   TRAFFIC_MSG_IMAGE_vDestroy
*
*****************************************************************************/
void TRAFFIC_MSG_IMAGE_vDestroy(
    TRAFFIC_MSG_OBJECT hTrafficMsg
        )
{
    TRAFFIC_MSG_IMAGE_OBJECT_STRUCT *psObj = NULL;

    psObj = (TRAFFIC_MSG_IMAGE_OBJECT_STRUCT *)TRAFFIC_MSG_BASE_pvObjectData(
        hTrafficMsg);

    if (psObj != NULL)
    {
        if (psObj->hLocation != LOCATION_INVALID_OBJECT)
        {
            LOCATION.vDestroy(psObj->hLocation);
        }

        if (psObj->hImage != IMAGE_INVALID_OBJECT)
        {
            STRING_OBJECT hFilePath = STRING_INVALID_OBJECT;

            hFilePath = IMAGE.hFileName(psObj->hImage);
            if (hFilePath != STRING_INVALID_OBJECT)
            {
                const char *pacFilePath = NULL;

                pacFilePath = STRING.pacCStr(hFilePath);
                if (pacFilePath != NULL)
                {
                    remove(pacFilePath);
                }
            }

            IMAGE_vDestroy(psObj->hImage);
        }

        if (psObj->hText != STRING_INVALID_OBJECT)
        {
            STRING_vDestroy(psObj->hText);
        }
    }

    TRAFFIC_MSG_BASE_vDestroy(hTrafficMsg);
}

/*****************************************************************************
*
*   TRAFFIC_MSG_IMAGE_tHash
*
*****************************************************************************/
TRAFCAM_IMAGE_HASH TRAFFIC_MSG_IMAGE_tHash(
    TRAFFIC_MSG_OBJECT hTrafficMsg
        )
{
    TRAFCAM_IMAGE_HASH tResult = TRAFCAM_IMAGE_INVALID_HASH;

    TRAFFIC_MSG_IMAGE_OBJECT_STRUCT *psObj = NULL;

    psObj = (TRAFFIC_MSG_IMAGE_OBJECT_STRUCT *)TRAFFIC_MSG_BASE_pvObjectData(
        hTrafficMsg);

    if (psObj != NULL)
    {
        tResult = psObj->tHash;
    }

    return tResult;
}

/*****************************************************************************
*
*   TRAFFIC_MSG_IMAGE_hCreateDummy
*
*****************************************************************************/
TRAFFIC_MSG_OBJECT TRAFFIC_MSG_IMAGE_hCreateDummy (
    SMS_OBJECT hOwner
        )
{
    TRAFFIC_MSG_OBJECT hTrafficMsg;

    hTrafficMsg = TRAFFIC_MSG_BASE_hCreate(
        NULL,
        TRAFFIC_MSG_IMAGE_OBJECT_NAME"ObjData",
        hOwner,
        sizeof(TRAFFIC_MSG_IMAGE_OBJECT_STRUCT),
        0,
        &TRAFFIC_MSG_IMAGE);

    return hTrafficMsg;
}

/*****************************************************************************
*
*   TRAFFIC_MSG_IMAGE_bUpdateHash
*
*****************************************************************************/
BOOLEAN TRAFFIC_MSG_IMAGE_bUpdateHash(
    TRAFFIC_MSG_OBJECT hTrafficMsg,
    TRAFCAM_IMAGE_HASH tHash
        )
{
    BOOLEAN bResult = FALSE;
    TRAFFIC_MSG_IMAGE_OBJECT_STRUCT *psObj = NULL;

    psObj = (TRAFFIC_MSG_IMAGE_OBJECT_STRUCT *)
        TRAFFIC_MSG_BASE_pvObjectData(hTrafficMsg);
    if (psObj != NULL)
    {
        psObj->tHash = tHash;
        bResult = TRUE;
    }

    return bResult;
}

/*****************************************************************************
*
*   TRAFFIC_MSG_IMAGE_bUpdateTimeToExpire
*
*****************************************************************************/
BOOLEAN TRAFFIC_MSG_IMAGE_bUpdateTimeToExpire(
    TRAFFIC_MSG_OBJECT hTrafficMsg
        )
{
    BOOLEAN bResult = FALSE;
    TRAFFIC_MSG_IMAGE_OBJECT_STRUCT *psObj = NULL;

    psObj = (TRAFFIC_MSG_IMAGE_OBJECT_STRUCT *)
        TRAFFIC_MSG_BASE_pvObjectData(hTrafficMsg);
    if (psObj != NULL)
    {
        UN32 un32TimeStamp = 0;

        OSAL.vTimeUp(&un32TimeStamp, NULL);
        psObj->un32TimeToExpire = un32TimeStamp + psObj->un16TTL;
        bResult = TRUE;
    }

    return bResult;
}

/*****************************************************************************
*
*   TRAFFIC_MSG_IMAGE_un32TimeToExpire
*
*****************************************************************************/
UN32 TRAFFIC_MSG_IMAGE_un32TimeToExpire(
    TRAFFIC_MSG_OBJECT hTrafficMsg
        )
{
    UN32 un32TimeToExpire = 0;
    TRAFFIC_MSG_IMAGE_OBJECT_STRUCT *psObj = NULL;

    psObj = (TRAFFIC_MSG_IMAGE_OBJECT_STRUCT *)
        TRAFFIC_MSG_BASE_pvObjectData(hTrafficMsg);
    if (psObj != NULL)
    {
        un32TimeToExpire = psObj->un32TimeToExpire;
    }

    return un32TimeToExpire;
}

/*****************************************************************************
 *                           PRIVATE FUNCTIONS
 *****************************************************************************/

/*******************************************************************************
*
*        bGetFileNameLength
*
*******************************************************************************/
static BOOLEAN bGetFileNameLength(
    IMAGE_OBJECT hImage,
    const char *pacFilePath,
    void *pvSpecificData,
    size_t *ptLength
        )
{
    (*ptLength) = (strlen(pacFilePath) + 1);

    return TRUE;
}

/*******************************************************************************
*
*        bCopyFileNameCallback
*
*******************************************************************************/
static BOOLEAN bCopyFileNameCallback (
    IMAGE_OBJECT hImage,
    const char *pacFilePath,
    void *pvSpecificData,
    char *pacBuffer,
    size_t tBufferSize
        )
{
    strncpy(pacBuffer, pacFilePath, tBufferSize);

    return TRUE;
}
