/******************************************************************************/
/*                    Copyright (c) Sirius XM Radio, Inc.                     */
/*                            All Rights Reserved                             */
/*         Licensed Materials - Property of Sirius XM Radio, Inc.             */
/******************************************************************************/
/*******************************************************************************
 *
 * DESCRIPTION
 *
 *  This module contains the Object:TRAFCAM_MGR implementation for the
 *  Simple Module Services (SMS)
 *
 ******************************************************************************/
#include <stdio.h>
#include <string.h>
#include "standard.h"
#include "osal.h"

#include "sms_api.h"
#include "sms_obj.h"
#include "sms_update.h"
#include "sql_interface_obj.h"
#include "dataservice_mgr_impl.h"
#include "string_obj.h"
#include "db_util.h"
#include "_trafcam_mgr_obj.h"
#include "trafcam_interface.h"
#include "dsrl_entry_obj.h"
#include "dsrl_obj.h"
#include "dsrl_target_obj.h"
#include "traffic_msg_image_obj.h"
#include "location_obj.h"
#include "image_obj.h"

#include "sms_api_debug.h"
static const char *gpacThisFile = __FILE__;

/*****************************************************************************
 *                           PUBLIC FUNCTIONS
 *****************************************************************************/

/*****************************************************************************
 *
 *   hStart
 *
 *   This function will create all the basics needed for this service to
 *   operate.  However, all initial processing to actually get this service
 *   running is done at a later time.
 *
 *****************************************************************************/
static TRAFFIC_CAMERAS_SERVICE_OBJECT hStart (
    const char *pacSRHDriverName,
    DATASERVICE_EVENT_MASK tEventRequestMask,
    DATASERVICE_EVENT_CALLBACK vEventCallback,
    void *pvAppEventCallbackArg,
    DATASERVICE_OPTIONS_STRUCT const *psOptions
        )
{
    TRAFCAM_MGR_OBJECT_STRUCT *psObj =
        (TRAFCAM_MGR_OBJECT_STRUCT *)TRAFFIC_CAMERAS_SERVICE_INVALID_OBJECT;
    BOOLEAN bSuccess = FALSE;
    DATASERVICE_CREATE_STRUCT sCreate;
    DATASERVICE_OPTION_VALUES_STRUCT sOptionValues;

    OSAL.bMemSet(&sOptionValues, 0, sizeof(sOptionValues));

    do
    {
        OSAL_RETURN_CODE_ENUM eReturnCode = OSAL_ERROR;

        // grab variable arguments (pop)
        bSuccess = DATASERVICE_IMPL_bProcessOptions(
            TRAFCAM_SUPPORTED_OPTIONS, psOptions, &sOptionValues);
        // restore stack (push)
        if (bSuccess == FALSE)
        {
            // Bad options!
            break;
        }

        // Populate our data service creation structure
        DATASERVICE_IMPL_vInitCreateStruct(&sCreate);
        sCreate.pacSRHDriverName = pacSRHDriverName;
        sCreate.pacServiceObjectName = TRAFCAM_MGR_OBJECT_NAME;
        sCreate.tServiceObjectSize = sizeof(TRAFCAM_MGR_OBJECT_STRUCT);
        sCreate.tDataID = (DATASERVICE_ID)GsTrafcamIntf.tDSI;
        sCreate.bEnableAll = FALSE;

        // Suggest an OTA buffer size
        sCreate.tSuggestedOTABufferByteSize = GsTrafcamIntf.tOTABufferByteSize;

        // Configure the data service's static event attributes
        sCreate.vEventCallback = vEventHandler;
        sCreate.tEventRequestMask = (
            DATASERVICE_EVENT_ALL |
            DATASERVICE_INTERNAL_EVENT_DSRL );

        // Ask the data service manager controller to
        // create our manager object and do everything
        // necessary to create the underlying objects required
        // in order to support this service
        psObj = (TRAFCAM_MGR_OBJECT_STRUCT *)
            DATASERVICE_IMPL_hCreateNewService(&sCreate);
        if (psObj == NULL)
        {
            // Can't create the service, fail out!
            break;
        }

        // Update the path for the reference db
        psObj->pacRefDatabaseDirPath = sOptionValues.pcRefDBPath;

        // Initialize asynchronous update configuration
        SMSU_vInitialize(
            &psObj->sEvent,
            psObj,
            DATASERVICE_EVENT_ALL,
            tEventRequestMask,
            (SMSAPI_OBJECT_EVENT_CALLBACK)vEventCallback,
            pvAppEventCallbackArg
                );

        eReturnCode = OSAL.eLinkedListCreate(&psObj->hFilter,
            TRAFCAM_MGR_OBJECT_NAME":Filter",
            (OSAL_LL_COMPARE_HANDLER)n16CompareFilterItems,
            OSAL_LL_OPTION_USE_PRE_ALLOCATED_ELEMENTS
                );
        if (eReturnCode != OSAL_SUCCESS)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                TRAFCAM_MGR_OBJECT_NAME": filter list creation failed (%s)",
                OSAL.pacGetReturnCodeName(eReturnCode)
                    );
            break;
        }

        bSuccess = bInitFilesPath(psObj);
        if (bSuccess != TRUE)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                TRAFCAM_MGR_OBJECT_NAME": failed to prepare files path"
                    );
            break;
        }

        // Initialize facing objects
        bSuccess = bInitAppFacingObject(psObj);
        if (bSuccess != TRUE)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                TRAFCAM_MGR_OBJECT_NAME": failed to initialize facing objects"
                    );
            break;
        }

        // The service may now start
        bSuccess = DATASERVICE_IMPL_bStart((DATASERVICE_IMPL_HDL)psObj);
        if (bSuccess != TRUE)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                TRAFCAM_MGR_OBJECT_NAME": unable to start service"
                    );
            break;
        }

        return (TRAFFIC_CAMERAS_SERVICE_OBJECT)psObj;
    }
    while (FALSE);

    // Error!
    if (psObj != NULL)
    {
        vUninitObject(psObj, TRUE);
        DATASERVICE_IMPL_vDestroy((DATASERVICE_IMPL_HDL)psObj);
    }

    if (sOptionValues.pcRefDBPath != NULL)
    {
        SMSO_vDestroy((SMS_OBJECT)(sOptionValues.pcRefDBPath));
    }

    return TRAFFIC_CAMERAS_SERVICE_INVALID_OBJECT;
}

/*****************************************************************************
*
*   eGetReferenceDataVersion
*
*****************************************************************************/
static DATASERVICE_ERROR_CODE_ENUM eGetReferenceDataVersion (
    const char *pcContainingDirectoryPath,
    DATASERVICE_REF_DATA_VER *ptCurrentRefDataVer,
    DATASERVICE_REF_DATA_VER *ptNextRefDataVer
        )
{
    DATASERVICE_ERROR_CODE_ENUM eReturnCode = DATASERVICE_ERROR_CODE_GENERAL;
    SQL_INTERFACE_OBJECT hSQLRefConnection = SQL_INTERFACE_INVALID_OBJECT;
    char *pacDatabaseFilePath = NULL;

    if (ptCurrentRefDataVer == NULL)
    {
        return DATASERVICE_ERROR_CODE_GENERAL;
    }

    do
    {
        BOOLEAN bSuccess = FALSE;

        bSuccess = DB_UTIL_bCreateFilePath(
            pcContainingDirectoryPath,
            TRAFCAM_DATABASE_FOLDER,
            TRAFCAM_REF_DATABASE_FILENAME,
            &pacDatabaseFilePath
                );
        if (bSuccess != TRUE)
        {
            break;
        }

        // Connect to the Traffic Cameras ref database
        hSQLRefConnection = DB_UTIL_hConnectToReference(
            pacDatabaseFilePath,
            (DB_UTIL_CHECK_VERSION_HANDLER)bVerifyDBSchemaVersion,
            (void *)(size_t)TRAFCAM_DATABASE_FILE_VERSION,
            &eReturnCode,
            SQL_INTERFACE_OPTIONS_READONLY |
            SQL_INTERFACE_OPTIONS_SKIP_CORRUPTION_CHECK
                );
        if ((hSQLRefConnection == SQL_INTERFACE_INVALID_OBJECT) ||
            (eReturnCode != DATASERVICE_ERROR_CODE_NONE))
        {
            break;
        }

        // Grab the actual database version
        *ptCurrentRefDataVer = (DATASERVICE_REF_DATA_VER)
            n32ExtractDataVersion(hSQLRefConnection);
        if (ptNextRefDataVer != NULL)
        {
            *ptNextRefDataVer = *ptCurrentRefDataVer;
        }

        eReturnCode = DATASERVICE_ERROR_CODE_NONE;

    } while (FALSE);

    //Disconnect from Database
    if (hSQLRefConnection != SQL_INTERFACE_INVALID_OBJECT)
    {
        SQL_INTERFACE.vDisconnect(hSQLRefConnection);
    }

    // Remove database path resources
    if (pacDatabaseFilePath != NULL)
    {
        SMSO_vDestroy((SMS_OBJECT)pacDatabaseFilePath);
    }

    return eReturnCode;
}

/*****************************************************************************
*
*   bReportRepeatedImage
*
*****************************************************************************/
static BOOLEAN bReportRepeatedImage (
    TRAFFIC_CAMERAS_SERVICE_OBJECT hTrafficCameras,
    TRAFCAM_IMAGE_HASH tHash
        )
{
    // lets find the expired message using the hash
    OSAL_RETURN_CODE_ENUM eReturnCode = OSAL_ERROR;
    OSAL_LINKED_LIST_ENTRY hEntry = OSAL_INVALID_LINKED_LIST_ENTRY;
    TRAFCAM_APP_OBJECT_STRUCT *psAppObj = NULL;

    psAppObj = psGetAppFacingObject(hTrafficCameras);
    if (psAppObj != NULL)
    {
        BOOLEAN bSuccess = FALSE;

        bSuccess = TRAFFIC_MSG_IMAGE_bUpdateHash(psAppObj->hDummyMsg, tHash);
        if (bSuccess == TRUE)
        {
            eReturnCode = OSAL.eLinkedListSearch(psAppObj->hEntriesList,
                &hEntry, psAppObj->hDummyMsg);
            if (eReturnCode == OSAL_SUCCESS)
            {
                DSRL_ENTRY_OBJECT hDSRLEntry = DSRL_ENTRY_INVALID_OBJECT;

                hDSRLEntry = (DSRL_ENTRY_OBJECT)OSAL.pvLinkedListThis(hEntry);
                TRAFFIC_MSG_IMAGE_bUpdateTimeToExpire(
                    (TRAFFIC_MSG_OBJECT)hDSRLEntry);
            }
        }

        vReleaseAppFacingObject(psAppObj);
    }

    return TRUE;
}

/*****************************************************************************
*
*   bReportRemovedImage
*
*****************************************************************************/
static BOOLEAN bReportRemovedImage (
    TRAFFIC_CAMERAS_SERVICE_OBJECT hTrafficCameras,
    TRAFCAM_IMAGE_HASH tHash
        )
{
    // lets find the removed message using the hash
    OSAL_RETURN_CODE_ENUM eReturnCode = OSAL_ERROR;
    OSAL_LINKED_LIST_ENTRY hEntry = OSAL_INVALID_LINKED_LIST_ENTRY;
    TRAFCAM_APP_OBJECT_STRUCT *psAppObj = NULL;

    psAppObj = psGetAppFacingObject(hTrafficCameras);
    if (psAppObj != NULL)
    {
        BOOLEAN bSuccess = TRAFFIC_MSG_IMAGE_bUpdateHash(
            psAppObj->hDummyMsg, tHash);
        if (bSuccess == TRUE)
        {
            eReturnCode = OSAL.eLinkedListSearch(psAppObj->hEntriesList,
                &hEntry, psAppObj->hDummyMsg);
            if (eReturnCode == OSAL_SUCCESS)
            {
                DSRL_ENTRY_OBJECT hDSRLEntry =
                    (DSRL_ENTRY_OBJECT)OSAL.pvLinkedListThis(hEntry);

                vEntryRemove(hDSRLEntry);
            }
        }

        vReleaseAppFacingObject(psAppObj);
    }

    return TRUE;
}

/*****************************************************************************
*
*    bReportNewImage
*
*****************************************************************************/
static BOOLEAN bReportNewImage(
    TRAFFIC_CAMERAS_SERVICE_OBJECT hTrafficCameras,
    TRAFCAM_IMAGE_INFO_STRUCT *psImageInfo,
    OSAL_BUFFER_HDL hImageBuffer
        )
{
    TRAFFIC_MSG_OBJECT hTrafficMsg = TRAFFIC_MSG_INVALID_OBJECT;
    TRAFCAM_APP_OBJECT_STRUCT *psAppObj = NULL;
    FILE *psFile = NULL;
    TRAFCAM_DSRL_ENTRY_DESC_STRUCT *psEntryDesc = NULL;
    BOOLEAN bFileCreated = FALSE;
    TRAFCAM_MGR_OBJECT_STRUCT *psObj =
        (TRAFCAM_MGR_OBJECT_STRUCT *)hTrafficCameras;

    do
    {
        OSAL_RETURN_CODE_ENUM eReturnCode = OSAL_ERROR;
        BOOLEAN bSuccess = FALSE;


        bSuccess =
            DATASERVICE_IMPL_bOwner((DATASERVICE_IMPL_HDL)hTrafficCameras);
        if ((bSuccess == FALSE) || (psImageInfo == NULL) ||
            (hImageBuffer == OSAL_INVALID_BUFFER_HDL))
        {
            break;
        }

        psFile = psCreateFile(psObj, psImageInfo->tHash);
        if (psFile == NULL)
        {
            // failed to create file
            break;
        }

        bFileCreated = TRUE;

        eReturnCode = OSAL.eBufferWriteToFile(hImageBuffer, TRUE, psFile);
        if (eReturnCode != OSAL_SUCCESS)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                TRAFCAM_MGR_OBJECT_NAME": Failed to copy image to file (%s)",
                OSAL.pacGetReturnCodeName(eReturnCode)
                    );
            break;
        }

        fclose(psFile);
        psFile = NULL;

        psAppObj = psGetAppFacingObject(hTrafficCameras);
        if (psAppObj == NULL)
        {
            break;
        }

        hTrafficMsg = TRAFFIC_MSG_IMAGE_hCreate((SMS_OBJECT)psAppObj,
            sizeof(*psEntryDesc), psImageInfo, psObj->pcFileNameBuffer);
        if (hTrafficMsg == TRAFFIC_MSG_INVALID_OBJECT)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                TRAFCAM_MGR_OBJECT_NAME": Failed to create traffic msg object"
                    );
            break;
        }

        psEntryDesc =
            (TRAFCAM_DSRL_ENTRY_DESC_STRUCT *)DSRL_ENTRY_pvServiceData(
            (DSRL_ENTRY_OBJECT)hTrafficMsg);
        if (psEntryDesc == NULL)
        {
            // error !
            break;
        }

        eReturnCode = OSAL.eLinkedListAdd(psAppObj->hEntriesList,
            &psEntryDesc->hPoolEntry, hTrafficMsg);
        if (eReturnCode != OSAL_SUCCESS)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                TRAFCAM_MGR_OBJECT_NAME": Failed to add entry to list (%s)",
                OSAL.pacGetReturnCodeName(eReturnCode)
                    );
            break;
        }

        vStartAgeoutTimer(psObj);

        bSuccess = bDSRLsReportEntry(psObj, hTrafficMsg);
        if (bSuccess == FALSE)
        {
            break;
        }

        vReleaseAppFacingObject(psAppObj);

        return TRUE;
    } while (FALSE);


    // cleaning all the lost stuff
    if (psFile != NULL)
    {
        fclose(psFile);
    }

    if (hTrafficMsg != TRAFFIC_MSG_INVALID_OBJECT)
    {
        vEntryRemove((DSRL_ENTRY_OBJECT)hTrafficMsg);
    }
    else if (bFileCreated == TRUE)
    {
        // if entry has not been created, but file exists,
        // lets remove the file
        remove(psObj->pcFileNameBuffer);
    }

    vReleaseAppFacingObject(psAppObj);

    return FALSE;
}

/*****************************************************************************
*
*    bReportUpdatedImage
*
*****************************************************************************/
static BOOLEAN bReportUpdatedImage (
    TRAFFIC_CAMERAS_SERVICE_OBJECT hTrafficCameras,
    TRAFCAM_IMAGE_HASH tHash,
    OSAL_BUFFER_HDL hImageBuffer
        )
{
    BOOLEAN bResult = FALSE;
    TRAFCAM_APP_OBJECT_STRUCT *psAppObj = NULL;
    FILE *psFile = NULL;

    do
    {
        TRAFCAM_DSRL_ENTRY_DESC_STRUCT *psEntryDesc = NULL;
        OSAL_RETURN_CODE_ENUM eReturnCode = OSAL_ERROR;
        BOOLEAN bSuccess = FALSE;
        OSAL_LINKED_LIST_ENTRY hEntry = OSAL_INVALID_LINKED_LIST_ENTRY;
        DSRL_ENTRY_OBJECT hDSRLEntry = DSRL_ENTRY_INVALID_OBJECT;
        TRAFFIC_MSG_OBJECT hTrafficMsg = TRAFFIC_MSG_INVALID_OBJECT;
        IMAGE_OBJECT hImage = IMAGE_INVALID_OBJECT;
        STRING_OBJECT hFilePath = STRING_INVALID_OBJECT;
        const char *pacFilePath = NULL;

        psAppObj = psGetAppFacingObject(hTrafficCameras);
        if ((psAppObj == NULL) || (tHash == TRAFCAM_IMAGE_INVALID_HASH) ||
            (hImageBuffer == OSAL_INVALID_BUFFER_HDL))
        {
            break;
        }

        bSuccess = TRAFFIC_MSG_IMAGE_bUpdateHash(psAppObj->hDummyMsg, tHash);
        if (bSuccess == FALSE)
        {
            break;
        }

        // looking for the message in the pool by hash
        eReturnCode = OSAL.eLinkedListSearch(psAppObj->hEntriesList, &hEntry,
            psAppObj->hDummyMsg);
        if (eReturnCode == OSAL_OBJECT_NOT_FOUND)
        {
            // camera image cannot be found, bad case
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                TRAFCAM_MGR_OBJECT_NAME": Failed to find camera image with"
                " hash %p", tHash
                    );
            break;
        }
        else if (eReturnCode != OSAL_SUCCESS)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                TRAFCAM_MGR_OBJECT_NAME": Failed to search in entries"
                " list (%s)", OSAL.pacGetReturnCodeName(eReturnCode)
                    );
            break;
        }

        hDSRLEntry = (DSRL_ENTRY_OBJECT)OSAL.pvLinkedListThis(hEntry);
        psEntryDesc =
            (TRAFCAM_DSRL_ENTRY_DESC_STRUCT *)DSRL_ENTRY_pvServiceData(
            hDSRLEntry);
        if (psEntryDesc == NULL)
        {
            // error, cannot continue
            break;
        }

        // now lets set state for entry as changed in all the DSRLs
        if (psEntryDesc->hDSRLList != OSAL_INVALID_OBJECT_HDL)
        {
            eReturnCode = OSAL.eLinkedListIterate(psEntryDesc->hDSRLList,
                (OSAL_LL_ITERATOR_HANDLER)bDSRLChangeEntry, hDSRLEntry);
            if ((eReturnCode != OSAL_SUCCESS) &&
                (eReturnCode != OSAL_NO_OBJECTS))
            {
                SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    TRAFCAM_MGR_OBJECT_NAME": Failed to iterate linked"
                    " list (%s)", OSAL.pacGetReturnCodeName(eReturnCode)
                        );
                break;
            }
        }

        hTrafficMsg = DSRL_ENTRY.hTrafficMsg(hDSRLEntry);
        if (hTrafficMsg == TRAFFIC_MSG_INVALID_OBJECT)
        {
            // error!
            break;
        }

        hImage = TRAFFIC_MSG.hImage(hTrafficMsg);
        if (hImage == IMAGE_INVALID_OBJECT)
        {
            // error!
            break;
        }

        hFilePath = IMAGE.hFileName(hImage);
        if (hFilePath == STRING_INVALID_OBJECT)
        {
            // error !
            break;
        }

        pacFilePath = STRING.pacCStr(hFilePath);

        psFile = fopen(pacFilePath, "wb+");
        if (psFile == NULL)
        {
            // failed to open file
            break;
        }

        eReturnCode = OSAL.eBufferWriteToFile(hImageBuffer, TRUE, psFile);
        if (eReturnCode != OSAL_SUCCESS)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                TRAFCAM_MGR_OBJECT_NAME": failed to copy image to file (%s)",
                OSAL.pacGetReturnCodeName(eReturnCode));
            break;
        }

        fclose(psFile);
        psFile = NULL;

        bSuccess = TRAFFIC_MSG_IMAGE_bUpdateTimeToExpire(hTrafficMsg);
        if (bSuccess == FALSE)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                TRAFCAM_MGR_OBJECT_NAME": Failed to update time to expire"
                    );
            break;
        }

        if (psEntryDesc->hDSRLList != OSAL_INVALID_OBJECT_HDL)
        {
            // ok, now lets move all the DSRLs back to ready
            eReturnCode = OSAL.eLinkedListIterate(psEntryDesc->hDSRLList,
                (OSAL_LL_ITERATOR_HANDLER)bDSRLChangeState, (void *)(size_t)
                DSRL_STATE_READY);
            if ((eReturnCode != OSAL_SUCCESS) &&
                (eReturnCode != OSAL_NO_OBJECTS))
            {
                SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    TRAFCAM_MGR_OBJECT_NAME": Failed to iterate linked"
                    " list (%s)", OSAL.pacGetReturnCodeName(eReturnCode)
                        );
                break;
            }
        }

        bResult = TRUE;
    } while (FALSE);

    vReleaseAppFacingObject(psAppObj);

    if (psFile != NULL)
    {
        fclose(psFile);
    }

    return bResult;
}

/*****************************************************************************
 *                           PRIVATE FUNCTIONS
 *****************************************************************************/

/*****************************************************************************
*
*   bInitFilesPath
*
*****************************************************************************/
static BOOLEAN bInitFilesPath(
    TRAFCAM_MGR_OBJECT_STRUCT *psObj
        )
{
    size_t tPathSize = 0, tServicePathSize = 0;
    BOOLEAN bSuccess = FALSE;
    UN8 un8Attributes = 0;

    // request path size
    tPathSize = OSAL.tFileSystemGetTempPath(0, NULL);
    tServicePathSize = strlen(TRAFCAM_FILES_PATH);

    psObj->pcFilesPath = (char *)SMSO_hCreate(
        TRAFCAM_MGR_OBJECT_NAME"FilesPath", tPathSize + tServicePathSize + 1,
        (SMS_OBJECT)psObj, FALSE);
    if (psObj->pcFilesPath == NULL)
    {
        SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
            TRAFCAM_MGR_OBJECT_NAME": Cannot allocate memory for path"
                );
        return FALSE;
    }

    // temp path + service dir name + NULL
    psObj->tPathSize = tPathSize + tServicePathSize + 1;

    OSAL.tFileSystemGetTempPath(tPathSize + 1, psObj->pcFilesPath);
    printf(TRAFCAM_MGR_OBJECT_NAME": OSAL Temp Path is %s\n",
        psObj->pcFilesPath);

    strncat(psObj->pcFilesPath, TRAFCAM_FILES_PATH, tServicePathSize);

    printf(TRAFCAM_MGR_OBJECT_NAME" Files path is %s\n", psObj->pcFilesPath);

    // check if directory exist
    bSuccess = OSAL.bFileSystemGetFileAttributes(psObj->pcFilesPath,
        &un8Attributes);
    if (bSuccess == FALSE)
    {
        // looks like path does not exist, lets create the dir
        bSuccess = OSAL.bFileSystemMakeDir(psObj->pcFilesPath);
        return bSuccess;
    }
    else if (((un8Attributes & OSAL_FILE_ATTR_WRITE) == OSAL_FILE_ATTR_WRITE) &&
        ((un8Attributes & OSAL_FILE_ATTR_DIRECTORY) ==
        OSAL_FILE_ATTR_DIRECTORY))
    {
        // directory exist already, lets use it
        return TRUE;
    }
    else
    {
        SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
            TRAFCAM_MGR_OBJECT_NAME": Cannot use directory %s",
            psObj->pcFilesPath
                );
    }

    return FALSE;
}

/*****************************************************************************
*
*   bVerifyBufferSize
*
*****************************************************************************/
static BOOLEAN bVerifyBufferSize(
    TRAFCAM_MGR_OBJECT_STRUCT *psObj,
    size_t tRequestedSize
        )
{
    BOOLEAN bResult = TRUE;

    if (psObj->tFileNameBufferSize < tRequestedSize)
    {
        if (psObj->pcFileNameBuffer != NULL)
        {
            SMSO_vDestroy((SMS_OBJECT)psObj->pcFileNameBuffer);
        }

        psObj->pcFileNameBuffer = (char *)SMSO_hCreate(
            TRAFCAM_MGR_OBJECT_NAME"Buffer", tRequestedSize, (SMS_OBJECT)psObj,
            FALSE);
        if (psObj->pcFileNameBuffer == NULL)
        {
            psObj->tFileNameBufferSize = 0;
            bResult = FALSE;
        }
        else
        {
            psObj->tFileNameBufferSize = tRequestedSize;
        }
    }

    return bResult;
}

/*****************************************************************************
*
*   psCreateFile
*
*****************************************************************************/
static FILE *psCreateFile(
    TRAFCAM_MGR_OBJECT_STRUCT *psObj,
    TRAFCAM_IMAGE_HASH tHash
        )
{
    int iFilePathSize = 0;
    FILE *psResult = NULL;
    BOOLEAN bSuccess = FALSE;

    iFilePathSize =
        snprintf(psObj->pcFileNameBuffer, psObj->tFileNameBufferSize,
        "%p%s", tHash, TRAFCAM_FILES_EXT);
    if (iFilePathSize <= 0)
    {
        return NULL;
    }

    iFilePathSize += psObj->tPathSize;

    bSuccess = bVerifyBufferSize(psObj, iFilePathSize);
    if (bSuccess == FALSE)
    {
        return NULL;
    }

    iFilePathSize = snprintf(psObj->pcFileNameBuffer,
        psObj->tFileNameBufferSize, "%s%p%s", psObj->pcFilesPath,
        tHash, TRAFCAM_FILES_EXT);
    if (iFilePathSize <= 0)
    {
        return NULL;
    }

    psResult = fopen(psObj->pcFileNameBuffer, "wb+");

    return psResult;
}

/*******************************************************************************
*
*        bDSRLChangeEntry
*
*******************************************************************************/
static BOOLEAN bDSRLChangeEntry (
    DSRL_OBJECT hDSRL,
    DSRL_ENTRY_OBJECT hDSRLEntry
        )
{
    BOOLEAN bContinue = FALSE;

    do
    {
        DSRL_ADD_REPLACE_RESULT_ENUM eResult = DSRL_ADD_REPLACE_ERROR;

        if ((hDSRL == DSRL_INVALID_OBJECT) ||
            (hDSRLEntry == DSRL_ENTRY_INVALID_OBJECT))
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                TRAFCAM_MGR_OBJECT_NAME": Invalid parameter"
                    );
            break;
        }

        // moving DSRL to updating state
        bDSRLChangeState(hDSRL, DSRL_STATE_UPDATING);

        eResult = DSRL_eReplaceEntry(hDSRL, hDSRLEntry, hDSRLEntry);
        if (eResult == DSRL_ADD_REPLACE_ERROR)
        {
            BOOLEAN bSuccess = FALSE;

            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                TRAFCAM_MGR_OBJECT_NAME": Entry replace failed"
                    );

            bSuccess =
                bEntryRemoveFromDSRL(hDSRLEntry, hDSRL);
            if (bSuccess == FALSE)
            {
                SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    TRAFCAM_MGR_OBJECT_NAME": failed to remove"
                    " entry from DSRL %p", hDSRL
                        );
            }

            bDSRLChangeState(hDSRL, DSRL_STATE_ERROR);

            break;
        }

        bContinue = TRUE;
    } while (FALSE);

    return bContinue;
}

/*******************************************************************************
*
*        bDSRLRemoveEntry
*
*******************************************************************************/
static BOOLEAN bDSRLRemoveEntry (
    DSRL_OBJECT hDSRL,
    DSRL_ENTRY_OBJECT hDSRLEntry
        )
{
    if ((hDSRL != DSRL_INVALID_OBJECT) &&
        (hDSRLEntry != DSRL_ENTRY_INVALID_OBJECT))
    {
        bDSRLChangeState(hDSRL, DSRL_STATE_UPDATING);
        DSRL_vRemoveEntry(hDSRL, hDSRLEntry);

        return TRUE;
    }

    return FALSE;
}

/*******************************************************************************
*
*        bEntryRemoveFromDSRLs
*
*******************************************************************************/
static BOOLEAN bEntryRemoveFromDSRLs(
    DSRL_ENTRY_OBJECT hDSRLEntry
        )
{
    BOOLEAN bResult = FALSE;

    do
    {
        TRAFCAM_DSRL_ENTRY_DESC_STRUCT *psEntryDesc = NULL;
        OSAL_RETURN_CODE_ENUM eReturnCode = OSAL_ERROR;

        psEntryDesc =
            (TRAFCAM_DSRL_ENTRY_DESC_STRUCT *)DSRL_ENTRY_pvServiceData(
            hDSRLEntry);
        if (psEntryDesc == NULL)
        {
            //error!
            break;
        }

        if (psEntryDesc->hDSRLList != OSAL_INVALID_OBJECT_HDL)
        {
            eReturnCode = OSAL.eLinkedListIterate(psEntryDesc->hDSRLList,
                (OSAL_LL_ITERATOR_HANDLER)bDSRLRemoveEntry, hDSRLEntry);
            if ((eReturnCode != OSAL_SUCCESS) &&
                (eReturnCode != OSAL_NO_OBJECTS))
            {
                SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    TRAFCAM_MGR_OBJECT_NAME": Failed to remove DSRL entry from"
                    " list (%s)", OSAL.pacGetReturnCodeName(eReturnCode)
                        );
                break;
            }
        }

        bResult = TRUE;
    } while (FALSE);

    return bResult;
}

/*****************************************************************************
*
*    bDSRLChangeState
*
*****************************************************************************/
static BOOLEAN bDSRLChangeState(
    DSRL_OBJECT hDSRL,
    DSRL_STATE_ENUM eState
        )
{
    if (hDSRL != DSRL_INVALID_OBJECT)
    {
        DSRL_STATE_ENUM eCurState = DSRL_STATE_UNKNOWN;

        eCurState = DSRL.eState(hDSRL);
        if (eCurState != DSRL_STATE_ERROR)
        {
            DSRL_vSetState(hDSRL, eState);
        }
    }

    return TRUE;
}

/*****************************************************************************
*
*    bTargetMessageCheck
*
*****************************************************************************/
static BOOLEAN bTargetMessageCheck(
    DSRL_TARGET_OBJECT hTarget,
    TRAFCAM_TARGET_MESSAGE_CHECK_ITERATOR *psIteratorArg
        )
{
    BOOLEAN bContinue = FALSE;

    do
    {
        LOCATION_OBJECT hLocation = LOCATION_INVALID_OBJECT;
        TRAFFIC_LOCID_OBJECT hLocID = TRAFFIC_LOCID_INVALID_OBJECT;

        if ((hTarget == DSRL_TARGET_INVALID_OBJECT) || (psIteratorArg == NULL))
        {
            break;
        }

        hLocation = DSRL_TARGET.hLocation(hTarget);
        if (hLocation == LOCATION_INVALID_OBJECT)
        {
            // not fatal issue
            bContinue = TRUE;
            break;
        }

        hLocID = LOCATION.hLocID(hLocation);

        // if target locid is not present, we check coordinates case only
        if (hLocID == TRAFFIC_LOCID_INVALID_OBJECT)
        {
            OSAL_FIXED_OBJECT hLat, hLon;

            hLat = LOCATION.hLat(psIteratorArg->hLocation);
            hLon = LOCATION.hLon(psIteratorArg->hLocation);

            psIteratorArg->bFits = LOCATION_bCoordinatesWithinRectangle(
                hLocation, hLat, hLon);
        }
        else
        {
            TRAFFIC_LOCID_TYPE_ENUM eType = TRAFFIC_LOCID_TYPE_INVALID;
            TRAFFIC_LOCID_OBJECT hMessageLocID = TRAFFIC_LOCID_INVALID_OBJECT;
            TRAFFIC_MARKET tMarket = TRAFFIC_INVALID_MARKET;
            TRAFFIC_MARKET tMessageMarket = TRAFFIC_INVALID_MARKET;

            hMessageLocID = LOCATION.hLocID(psIteratorArg->hLocation);

            eType = TRAFFIC_LOCID.eType(hLocID);
            if (eType == TRAFFIC_LOCID_TYPE_BSA)
            {
                TRAFFIC_BSA tBSA = TRAFFIC_INVALID_BSA;
                TRAFFIC_BSA tMessageBSA = TRAFFIC_INVALID_BSA;

                tBSA = TRAFFIC_LOCID.tBSA(hLocID);
                tMessageBSA = TRAFFIC_LOCID.tBSA(hMessageLocID);
                if (tBSA != tMessageBSA)
                {
                    // target doesn't fit, skip it
                    bContinue = TRUE;
                    break;
                }
            }

            tMarket = TRAFFIC_LOCID.tMarket(hLocID);
            tMessageMarket = TRAFFIC_LOCID.tMarket(hMessageLocID);
            if (tMarket != tMessageMarket)
            {
                // target doesn't fit, skip it
                bContinue = TRUE;
                break;
            }

            psIteratorArg->bFits = TRUE;
        }

        // if target fits, we can stop here
        if (psIteratorArg->bFits == TRUE)
        {
            bContinue = FALSE;
            break;
        }


        bContinue = TRUE;
    } while (FALSE);

    return bContinue;
}

/*****************************************************************************
*
*    bDSRLAddEntry
*
*****************************************************************************/
static BOOLEAN bDSRLAddEntry(
    DSRL_OBJECT hDSRL,
    DSRL_ENTRY_OBJECT hDSRLEntry
        )
{
    BOOLEAN bContinue = FALSE, bAdded = FALSE;
    TRAFCAM_TARGET_MESSAGE_CHECK_ITERATOR sIterator;

    OSAL.bMemSet(&sIterator, 0, sizeof(sIterator));

    do
    {
        DSRL_ADD_REPLACE_RESULT_ENUM eDSRLResult = DSRL_ADD_REPLACE_ERROR;
        TRAFCAM_DSRL_DESC_STRUCT *psDSRLDesc = NULL;
        TRAFCAM_DSRL_ENTRY_DESC_STRUCT *psEntryDesc = NULL;
        OSAL_RETURN_CODE_ENUM eReturnCode = OSAL_ERROR;
        TRAFFIC_MSG_OBJECT hTrafficMsg = TRAFFIC_MSG_INVALID_OBJECT;

        if ((hDSRL == DSRL_INVALID_OBJECT) ||
            (hDSRLEntry == DSRL_ENTRY_INVALID_OBJECT))
        {
            break;
        }

        psDSRLDesc =
            (TRAFCAM_DSRL_DESC_STRUCT *)DSRL_pvServiceData(hDSRL);
        if (psDSRLDesc == NULL)
        {
            break;
        }

        hTrafficMsg = DSRL_ENTRY.hTrafficMsg(hDSRLEntry);
        sIterator.hLocation = TRAFFIC_MSG.hGetLocation(hTrafficMsg, 0);

        eReturnCode = OSAL.eLinkedListIterate(psDSRLDesc->hTargetsList,
            (OSAL_LL_ITERATOR_HANDLER)bTargetMessageCheck, &sIterator);
        if ((eReturnCode != OSAL_SUCCESS) && (eReturnCode != OSAL_NO_OBJECTS))
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                TRAFCAM_MGR_OBJECT_NAME": Failed iterate list of targets (%s)",
                OSAL.pacGetReturnCodeName(eReturnCode)
                    );
            break;
        }

        if (sIterator.bFits == FALSE)
        {
            bContinue = TRUE;
            break;
        }

        bDSRLChangeState(hDSRL, DSRL_STATE_UPDATING);

        psEntryDesc =
            (TRAFCAM_DSRL_ENTRY_DESC_STRUCT *)DSRL_ENTRY_pvServiceData(
            hDSRLEntry);
        if (psEntryDesc == NULL)
        {
            break;
        }

        eDSRLResult = DSRL_eAddEntry(hDSRL, hDSRLEntry);
        if (eDSRLResult != DSRL_ADD_REPLACE_OK)
        {
            printf(TRAFCAM_MGR_OBJECT_NAME": entry cannot be "
                "added for some reason.\n"
                    );
            bContinue = TRUE;
            break;
        }
        bAdded = TRUE;

        eReturnCode = OSAL.eLinkedListAdd(psDSRLDesc->hEntriesList,
            OSAL_INVALID_LINKED_LIST_ENTRY_PTR, hDSRLEntry
                );
        if (eReturnCode != OSAL_SUCCESS)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                TRAFCAM_MGR_OBJECT_NAME": cannot add entry"
                " to list (%s)",
                OSAL.pacGetReturnCodeName(eReturnCode)
                    );
            break;
        }

        if (psEntryDesc->hDSRLList == OSAL_INVALID_OBJECT_HDL)
        {
            // in case if DSRLs list in EntryDesc struct is not created
            // lets create it
            eReturnCode = OSAL.eLinkedListCreate(&(psEntryDesc->hDSRLList),
                TRAFCAM_MGR_OBJECT_NAME":DSRLEntry:DSRLList",
                n16CompareHandles, OSAL_LL_OPTION_NONE
                    );
            if (eReturnCode != OSAL_SUCCESS)
            {
                SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    TRAFCAM_MGR_OBJECT_NAME": cannot create link list (%s)",
                    OSAL.pacGetReturnCodeName(eReturnCode)
                        );
                break;
            }
        }

        eReturnCode = OSAL.eLinkedListAdd(psEntryDesc->hDSRLList,
            OSAL_INVALID_LINKED_LIST_ENTRY_PTR, hDSRL);
        if (eReturnCode != OSAL_SUCCESS)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                TRAFCAM_MGR_OBJECT_NAME": cannot add entry"
                " to list (%s)",
                OSAL.pacGetReturnCodeName(eReturnCode)
                    );
            break;
        }

        bContinue = TRUE;
    } while (FALSE);

    if ((sIterator.bFits == TRUE) && (bContinue == TRUE) && (bAdded == FALSE))
    {
        //lets remove entry from DSRL to avoid any issues
        DSRL_vRemoveEntry(hDSRL, hDSRLEntry);
    }

    if ((sIterator.bFits == TRUE) && (bContinue == TRUE))
    {
        bDSRLChangeState(hDSRL, DSRL_STATE_READY);
    }
    else if (bContinue == FALSE)
    {
        bDSRLChangeState(hDSRL, DSRL_STATE_ERROR);
    }

    return bContinue;
}

/*****************************************************************************
*
*    bApplyMessage
*
*****************************************************************************/
static BOOLEAN bApplyMessage(
    TRAFCAM_BSA_FILTER_STRUCT *psFilterItem,
    TRAFFIC_MSG_OBJECT hTrafficMsg
        )
{
    LOCATION_OBJECT hLocation = LOCATION_INVALID_OBJECT;
    TRAFFIC_LOCID_OBJECT hLocID = TRAFFIC_LOCID_INVALID_OBJECT;
    TRAFFIC_MARKET tMarket = TRAFFIC_INVALID_MARKET;
    TRAFFIC_BSA tBSA = TRAFFIC_INVALID_BSA;

    if ((psFilterItem == NULL) ||
        (hTrafficMsg == TRAFFIC_MSG_INVALID_OBJECT))
    {
        return FALSE;
    }

    hLocation = TRAFFIC_MSG.hGetLocation(hTrafficMsg, 0);
    if (hLocation == LOCATION_INVALID_OBJECT)
    {
        return FALSE;
    }

    hLocID = LOCATION.hLocID(hLocation);
    if (hLocID == LOCID_INVALID_OBJECT)
    {
        return FALSE;
    }

    tMarket = TRAFFIC_LOCID.tMarket(hLocID);
    tBSA = TRAFFIC_LOCID.tBSA(hLocID);
    if ((psFilterItem->tBSA == tBSA) &&
        (tMarket == psFilterItem->tMarket))
    {
        OSAL_RETURN_CODE_ENUM eReturnCode = OSAL_ERROR;

        eReturnCode = OSAL.eLinkedListIterate(psFilterItem->hDSRLList,
            (OSAL_LL_ITERATOR_HANDLER)bDSRLAddEntry, hTrafficMsg);
        if ((eReturnCode != OSAL_SUCCESS) &&
            (eReturnCode != OSAL_NO_OBJECTS))
        {
            return FALSE;
        }
    }

    return TRUE;
}

/*****************************************************************************
*
*    bDSRLsReportEntry
*
*****************************************************************************/
static BOOLEAN bDSRLsReportEntry(
    TRAFCAM_MGR_OBJECT_STRUCT *psObj,
    TRAFFIC_MSG_OBJECT hTrafficMsg
        )
{
    OSAL_RETURN_CODE_ENUM eReturnCode = OSAL_ERROR;

    eReturnCode = OSAL.eLinkedListIterate(psObj->hFilter,
        (OSAL_LL_ITERATOR_HANDLER)bApplyMessage, hTrafficMsg);
    if ((eReturnCode != OSAL_SUCCESS) && (eReturnCode != OSAL_NO_OBJECTS))
    {
        return FALSE;
    }

    return TRUE;
}

/*****************************************************************************
*
*   n16CompareFilterItems
*
*****************************************************************************/
static N16 n16CompareFilterItems(
    TRAFCAM_BSA_FILTER_STRUCT *psBSAFilter1,
    TRAFCAM_BSA_FILTER_STRUCT *psBSAFilter2
        )
{
    N16 n16Result = N16_MIN;

    if ((psBSAFilter1 != NULL) && (psBSAFilter2 != NULL))
    {
        n16Result =
            COMPARE(psBSAFilter1->tMarket, psBSAFilter2->tMarket);
        if (n16Result == 0)
        {
            n16Result =
                COMPARE(psBSAFilter1->tBSA, psBSAFilter2->tBSA);
        }
    }

    return n16Result;
}

/*****************************************************************************
*
*   n16CompareHandles
*
*****************************************************************************/
static N16 n16CompareHandles(
    void *pvArg1,
    void *pvArg2
        )
{
    return COMPARE(pvArg1, pvArg2);
}

/*****************************************************************************
*
*   n16CompareByHash
*
*****************************************************************************/
static N16 n16CompareByHash (
    TRAFFIC_MSG_OBJECT hTrafficMsg1,
    TRAFFIC_MSG_OBJECT hTrafficMsg2
        )
{
    N16 n16Result = N16_MIN;

    if ((hTrafficMsg1 != TRAFFIC_MSG_INVALID_OBJECT) &&
        (hTrafficMsg2 != TRAFFIC_MSG_INVALID_OBJECT))
    {
        TRAFCAM_IMAGE_HASH tHash1 = TRAFCAM_IMAGE_INVALID_HASH,
            tHash2 = TRAFCAM_IMAGE_INVALID_HASH;

        tHash1 = TRAFFIC_MSG_IMAGE_tHash(hTrafficMsg1);
        tHash2 = TRAFFIC_MSG_IMAGE_tHash(hTrafficMsg2);

        n16Result = COMPARE(tHash1, tHash2);
    }

    return n16Result;
}

/*****************************************************************************
*
*   bInitAppFacingObject
*
*****************************************************************************/
static BOOLEAN bInitAppFacingObject (
    TRAFCAM_MGR_OBJECT_STRUCT *psObj
        )
{
    OSAL_RETURN_CODE_ENUM eReturnCode = OSAL_ERROR;
    BOOLEAN bResult = FALSE;

    do
    {
        DATASERVICE_DSRL_CONFIG_STRUCT sDSRLConfig;

        // Begin DSRL config
        DATASERVICE_IMPL_vInitializeDSRLConfig(&sDSRLConfig);

        // Set service type
        sDSRLConfig.eServiceType = DATASERVICE_TYPE_TRAFFIC_CAMERAS;

        // Set the parent object size
        sDSRLConfig.tParentObjectSize = sizeof(TRAFCAM_APP_OBJECT_STRUCT);

        // Set the DSRL descriptor size
        sDSRLConfig.tServiceDataSize = sizeof(TRAFCAM_DSRL_DESC_STRUCT);

        // Configure the device feature
        sDSRLConfig.bDeviceEnabled = TRUE;
        sDSRLConfig.un32DeviceNotifyDistance =
            TRAFCAM_DEVICE_DISTANCE_THRESHOLD;
        sDSRLConfig.eDeviceNotifyUnits = DISTANCE_UNIT_TYPE_MILES;

        // Create it
        psObj->psAppObj = (TRAFCAM_APP_OBJECT_STRUCT *)
            DATASERVICE_IMPL_hConfigureDSRL(
            (DATASERVICE_IMPL_HDL)psObj, &sDSRLConfig);
        if (psObj->psAppObj == NULL)
        {
            break;
        }

        psObj->psAppObj->hDummyMsg =
            TRAFFIC_MSG_IMAGE_hCreateDummy((SMS_OBJECT)psObj->psAppObj);
        if (psObj->psAppObj->hDummyMsg == TRAFFIC_MSG_INVALID_OBJECT)
        {
            break;
        }

        eReturnCode = OSAL.eLinkedListCreate(
            &psObj->psAppObj->hDSRLList,
            TRAFCAM_MGR_OBJECT_NAME":DSRLList",
            NULL,
            OSAL_LL_OPTION_NONE
                );
        if (eReturnCode != OSAL_SUCCESS)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                TRAFCAM_MGR_OBJECT_NAME": Cannot create DSRLList (%s)",
                OSAL.pacGetReturnCodeName(eReturnCode)
                    );
            break;
        }

        eReturnCode = OSAL.eLinkedListCreate(
            &psObj->psAppObj->hEntriesList,
            TRAFCAM_MGR_OBJECT_NAME":EntriesList",
            (OSAL_LL_COMPARE_HANDLER)n16CompareByHash,
            OSAL_LL_OPTION_NONE
                );
        if (eReturnCode != OSAL_SUCCESS)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                TRAFCAM_MGR_OBJECT_NAME": Cannot create EntriesList (%s)",
                OSAL.pacGetReturnCodeName(eReturnCode)
                    );
            break;
        }

        eReturnCode = OSAL.eLinkedListCreate(&psObj->psAppObj->hNewEntriesList,
            TRAFCAM_MGR_OBJECT_NAME":NewEntriesList", n16CompareHandles,
            OSAL_LL_OPTION_UNIQUE
                );
        if (eReturnCode != OSAL_SUCCESS)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                TRAFCAM_MGR_OBJECT_NAME": Cannot create NewEntriesList (%s)",
                OSAL.pacGetReturnCodeName(eReturnCode)
                    );
            break;
        }

        eReturnCode = OSAL.eLinkedListCreate(&psObj->psAppObj->hEntriesToAdd,
            TRAFCAM_MGR_OBJECT_NAME":EntriesToAddList",
            NULL,
            OSAL_LL_OPTION_NONE
                );
        if (eReturnCode != OSAL_SUCCESS)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                TRAFCAM_MGR_OBJECT_NAME": cannot create add entry list (%s)",
                OSAL.pacGetReturnCodeName(eReturnCode)
                    );
            break;
        }

        eReturnCode = OSAL.eLinkedListCreate(&psObj->psAppObj->hEntriesToRemove,
            TRAFCAM_MGR_OBJECT_NAME":EntriesToRemoveList",
            NULL,
            OSAL_LL_OPTION_NONE
                );
        if (eReturnCode != OSAL_SUCCESS)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                TRAFCAM_MGR_OBJECT_NAME": cannot create remove entry"
                " list (%s)",
                OSAL.pacGetReturnCodeName(eReturnCode)
                    );
            break;
        }

        eReturnCode = OSAL.eLinkedListCreate(&psObj->psAppObj->hNewFilter,
            TRAFCAM_MGR_OBJECT_NAME":NewFilter",
            (OSAL_LL_COMPARE_HANDLER)n16CompareFilterItems,
            OSAL_LL_OPTION_USE_PRE_ALLOCATED_ELEMENTS|OSAL_LL_OPTION_UNIQUE
                );
        if (eReturnCode != OSAL_SUCCESS)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                TRAFCAM_MGR_OBJECT_NAME": cannot create new filter"
                " list (%s)",
                OSAL.pacGetReturnCodeName(eReturnCode)
                    );
            break;
        }

        bResult = TRUE;

    } while (FALSE);

    if (NULL != psObj->psAppObj)
    {
        // Relinquish ownership
        SMSO_vUnlock((SMS_OBJECT)psObj->psAppObj);
    }

    return bResult;
}

/*****************************************************************************
*
*   vEntryRelease
*
*****************************************************************************/
static void vEntryRelease(
    DSRL_ENTRY_OBJECT hDSRLEntry
        )
{
    if (hDSRLEntry != DSRL_ENTRY_INVALID_OBJECT)
    {
        TRAFCAM_DSRL_ENTRY_DESC_STRUCT *psEntryDesc = NULL;

        psEntryDesc =
            (TRAFCAM_DSRL_ENTRY_DESC_STRUCT *)DSRL_ENTRY_pvServiceData(
            hDSRLEntry);

        if (psEntryDesc != NULL)
        {
            if (psEntryDesc->hDSRLList != OSAL_INVALID_OBJECT_HDL)
            {
                OSAL_RETURN_CODE_ENUM eReturnCode = OSAL_ERROR;

                eReturnCode =
                    OSAL.eLinkedListRemoveAll(psEntryDesc->hDSRLList, NULL);
                if (eReturnCode != OSAL_SUCCESS)
                {
                    SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                        TRAFCAM_MGR_OBJECT_NAME": failed to cleanup list (%s)",
                        OSAL.pacGetReturnCodeName(eReturnCode)
                            );
                }

                eReturnCode = OSAL.eLinkedListDelete(psEntryDesc->hDSRLList);
                if (eReturnCode != OSAL_SUCCESS)
                {
                    SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                        TRAFCAM_MGR_OBJECT_NAME": failed to delete list (%s)",
                        OSAL.pacGetReturnCodeName(eReturnCode)
                            );
                }

                psEntryDesc->hDSRLList = OSAL_INVALID_OBJECT_HDL;
            }
        }

        TRAFFIC_MSG_IMAGE_vDestroy((TRAFFIC_MSG_OBJECT)hDSRLEntry);
    }
}

/*****************************************************************************
*
*   vEntryRemove
*
*****************************************************************************/
static void vEntryRemove(
    DSRL_ENTRY_OBJECT hDSRLEntry
        )
{
    if (hDSRLEntry != DSRL_ENTRY_INVALID_OBJECT)
    {
        TRAFCAM_DSRL_ENTRY_DESC_STRUCT *psEntryDesc = NULL;

        printf(TRAFCAM_MGR_OBJECT_NAME": removing entry %p\n", hDSRLEntry);

        psEntryDesc =
            (TRAFCAM_DSRL_ENTRY_DESC_STRUCT *)DSRL_ENTRY_pvServiceData(
            hDSRLEntry);
        if ((psEntryDesc != NULL) && (psEntryDesc->hPoolEntry !=
            OSAL_INVALID_LINKED_LIST_ENTRY))
        {
            OSAL_RETURN_CODE_ENUM eReturnCode = OSAL_ERROR;

            eReturnCode = OSAL.eLinkedListRemove(psEntryDesc->hPoolEntry);
            if (eReturnCode != OSAL_SUCCESS)
            {
                SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    TRAFCAM_MGR_OBJECT_NAME": failed to remove entry from"
                    " pool (%s)",
                    OSAL.pacGetReturnCodeName(eReturnCode)
                        );
            }
        }

        vEntryRelease(hDSRLEntry);
    }
}

/*****************************************************************************
*
*   vUninitAppFacingObject
*
*****************************************************************************/
static void vUninitAppFacingObject(
    TRAFCAM_MGR_OBJECT_STRUCT *psObj
        )
{
    BOOLEAN bLocked;
    TRAFCAM_APP_OBJECT_STRUCT *psAppObj = psObj->psAppObj;
    OSAL_RETURN_CODE_ENUM eReturnCode = OSAL_ERROR;

    // Lock the app object
    bLocked = SMSO_bLock((SMS_OBJECT)psAppObj, OSAL_OBJ_TIMEOUT_INFINITE);
    if (bLocked == FALSE)
    {
        SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
            TRAFCAM_MGR_OBJECT_NAME": Unable to lock app object");
        return;
    }

    // Disconnect from the database application
    if (psAppObj->hSQLRefConnection != SQL_INTERFACE_INVALID_OBJECT)
    {
        SQL_INTERFACE.vDisconnect(psAppObj->hSQLRefConnection);
        psAppObj->hSQLRefConnection = SQL_INTERFACE_INVALID_OBJECT;
    }

    // Remove the dummy object
    if (psAppObj->hDummyMsg != TRAFFIC_MSG_INVALID_OBJECT)
    {
        TRAFFIC_MSG_IMAGE_vDestroy(psAppObj->hDummyMsg);
        psAppObj->hDummyMsg = TRAFFIC_MSG_INVALID_OBJECT;
    }

    if (psAppObj->hDSRLList != OSAL_INVALID_OBJECT_HDL)
    {
        eReturnCode = OSAL.eLinkedListRemoveAll(psAppObj->hDSRLList,
            (OSAL_LL_RELEASE_HANDLER)vDSRLRelease
                );
        if (eReturnCode != OSAL_SUCCESS)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                TRAFCAM_MGR_OBJECT_NAME": failed to remove DSRL items (%s)",
                OSAL.pacGetReturnCodeName(eReturnCode)
                    );
        }

        eReturnCode = OSAL.eLinkedListDelete(psAppObj->hDSRLList);
        if (eReturnCode != OSAL_SUCCESS)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                TRAFCAM_MGR_OBJECT_NAME": failed to delete DSRLs list (%s)",
                OSAL.pacGetReturnCodeName(eReturnCode)
                    );
        }

        psAppObj->hDSRLList = OSAL_INVALID_OBJECT_HDL;
    }

    if (psAppObj->hEntriesToAdd != OSAL_INVALID_OBJECT_HDL)
    {
        eReturnCode = OSAL.eLinkedListRemoveAll(psAppObj->hEntriesToAdd, NULL);
        if (eReturnCode != OSAL_SUCCESS)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                TRAFCAM_MGR_OBJECT_NAME": failed to cleanup list (%s)",
                OSAL.pacGetReturnCodeName(eReturnCode)
                    );
        }

        eReturnCode = OSAL.eLinkedListDelete(psAppObj->hEntriesToAdd);
        if (eReturnCode != OSAL_SUCCESS)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                TRAFCAM_MGR_OBJECT_NAME": failed to delete list (%s)",
                OSAL.pacGetReturnCodeName(eReturnCode)
                    );
        }

        psAppObj->hEntriesToAdd = OSAL_INVALID_OBJECT_HDL;
    }

    if (psAppObj->hEntriesToRemove != OSAL_INVALID_OBJECT_HDL)
    {
        eReturnCode =
            OSAL.eLinkedListRemoveAll(psAppObj->hEntriesToRemove, NULL);
        if (eReturnCode != OSAL_SUCCESS)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                TRAFCAM_MGR_OBJECT_NAME": failed to cleanup list (%s)",
                OSAL.pacGetReturnCodeName(eReturnCode)
                    );
        }

        eReturnCode = OSAL.eLinkedListDelete(psAppObj->hEntriesToRemove);
        if (eReturnCode != OSAL_SUCCESS)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                TRAFCAM_MGR_OBJECT_NAME": failed to delete list (%s)",
                OSAL.pacGetReturnCodeName(eReturnCode)
                    );
        }

        psAppObj->hEntriesToRemove = OSAL_INVALID_OBJECT_HDL;
    }

    if (psAppObj->hNewEntriesList != OSAL_INVALID_OBJECT_HDL)
    {
        eReturnCode =
            OSAL.eLinkedListRemoveAll(psAppObj->hNewEntriesList, NULL);
        if (eReturnCode != OSAL_SUCCESS)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                TRAFCAM_MGR_OBJECT_NAME": failed to cleanup list (%s)",
                OSAL.pacGetReturnCodeName(eReturnCode)
                    );
        }

        eReturnCode = OSAL.eLinkedListDelete(psAppObj->hNewEntriesList);
        if (eReturnCode != OSAL_SUCCESS)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                TRAFCAM_MGR_OBJECT_NAME": failed to delete list (%s)",
                OSAL.pacGetReturnCodeName(eReturnCode)
                    );
        }

        psAppObj->hNewEntriesList = OSAL_INVALID_OBJECT_HDL;
    }

    if (psAppObj->hEntriesList != OSAL_INVALID_OBJECT_HDL)
    {
        eReturnCode = OSAL.eLinkedListRemoveAll(psAppObj->hEntriesList,
            (OSAL_LL_RELEASE_HANDLER)vEntryRelease
                );
        if (eReturnCode != OSAL_SUCCESS)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                TRAFCAM_MGR_OBJECT_NAME": failed to remove entry items (%s)",
                OSAL.pacGetReturnCodeName(eReturnCode)
                    );
        }

        eReturnCode = OSAL.eLinkedListDelete(psAppObj->hEntriesList);
        if (eReturnCode != OSAL_SUCCESS)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                TRAFCAM_MGR_OBJECT_NAME": failed to delete entry list (%s)",
                OSAL.pacGetReturnCodeName(eReturnCode)
                    );
        }

        psAppObj->hEntriesList = OSAL_INVALID_OBJECT_HDL;
    }

    if (psAppObj->hNewFilter != OSAL_INVALID_OBJECT_HDL)
    {
        eReturnCode = OSAL.eLinkedListRemoveAll(psAppObj->hNewFilter,
            OSAL.vLinkedListMemoryFree);
        if (eReturnCode != OSAL_SUCCESS)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                TRAFCAM_MGR_OBJECT_NAME": failed to cleanup new filter"
                " list (%s)",
                OSAL.pacGetReturnCodeName(eReturnCode)
                    );
        }

        eReturnCode =
            OSAL.eLinkedListDelete(psAppObj->hNewFilter);
        if (eReturnCode != OSAL_SUCCESS)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                TRAFCAM_MGR_OBJECT_NAME": failed to delete new filter"
                " list (%s)",
                OSAL.pacGetReturnCodeName(eReturnCode)
                    );
        }

        psAppObj->hNewFilter = OSAL_INVALID_OBJECT_HDL;
    }

    // Destroy the app object
    DATASERVICE_IMPL_vDestroyDSRLParent((DATASERVICE_IMPL_HDL)psObj,
        (SMS_OBJECT)psAppObj);
    psObj->psAppObj = NULL;

    return;
}

/*****************************************************************************
*
*   bProcessMessage
*
*   This function is called to provide the Traffic Cameras interface with
*   a newly received message
*
*****************************************************************************/
static BOOLEAN bProcessMessage(
    TRAFCAM_MGR_OBJECT_STRUCT *psObj,
    OSAL_BUFFER_HDL hPayload
        )
{
    BOOLEAN bProcessed = TRUE;

    // Tell the interface it's time to
    // process another message
    bProcessed = GsTrafcamIntf.bProcessMessage(psObj->hInterface, &hPayload);

    // We're all done with this payload
    DATASERVICE_IMPL_bFreeDataPayload(hPayload);

    return bProcessed;
}

/*******************************************************************************
*
*   bDBProcessSelectDBVersion
*
*******************************************************************************/
static BOOLEAN bDBProcessSelectDBVersion (
    SQL_QUERY_COLUMN_STRUCT *psColumn,
    N32 n32NumberOfColumns,
    TRAFCAM_DB_RESULT_STRUCT *psResult
        )
{
    // Verify input
    if ((psResult == NULL) || (psResult->uDbRow.psVersion == NULL))
    {
        return FALSE;
    }

    psResult->uDbRow.psVersion->un32Version =
        psColumn[0].uData.sUN32.un32Data;
    psResult->bResult = TRUE;

    // FALSE will stop iteration; TRUE will keep going
    return psResult->bResult;
}

/*******************************************************************************
*
*   bVerifyDBSchemaVersion
*
*******************************************************************************/
static BOOLEAN bVerifyDBSchemaVersion(
    SQL_INTERFACE_OBJECT hSQLConnection,
    UN32 un32SchemaVer
        )
{

    TRAFCAM_DB_RESULT_STRUCT sResult;
    BOOLEAN bSuccess = FALSE, bVersionMatched = FALSE;
    TRAFCAM_DB_VERSION_ROW_STRUCT sVersion;

    if (hSQLConnection == SQL_INTERFACE_INVALID_OBJECT)
    {
        return FALSE;
    }

    OSAL.bMemSet(&sResult, 0, sizeof(sResult));
    OSAL.bMemSet(&sVersion, 0, sizeof(sVersion));
    sResult.uDbRow.psVersion = &sVersion;

    // Perform the SQL query and process the result
    // (it will provide us with a data row)
    bSuccess = SQL_INTERFACE.bQuery(
        hSQLConnection,
        TRAFCAM_SELECT_DB_SCHEMA_VERSION,
        (SQL_QUERY_RESULT_HANDLER)bDBProcessSelectDBVersion,
        &sResult
            );
    if ((bSuccess == TRUE) && (sResult.bResult == TRUE))
    {
        if ((sResult.uDbRow.psVersion != NULL) &&
            (sResult.uDbRow.psVersion->un32Version == un32SchemaVer))
        {
            // Version match!
            bVersionMatched = TRUE;
        }
    }

    return bVersionMatched;
}

/*******************************************************************************
*
*   vStartAgeoutTimer
*
*******************************************************************************/
static void vStartAgeoutTimer(
    TRAFCAM_MGR_OBJECT_STRUCT *psObj
        )
{
    BOOLEAN bSuccess = FALSE;

    // Kick off the event
    bSuccess = DATASERVICE_IMPL_bSetTimedEvent(
        psObj->hExpireEvent, FALSE, FALSE, TRAFCAM_TIMER_PERIOD);

    if (bSuccess == TRUE)
    {
        puts(TRAFCAM_MGR_OBJECT_NAME" Starting timer");
    }

    return;
}

/*******************************************************************************
*
*   bHandleServiceError
*
*******************************************************************************/
static BOOLEAN bHandleServiceError (
    TRAFCAM_MGR_OBJECT_STRUCT *psObj
        )
{
    // stop the timed event just for case
    DATASERVICE_IMPL_bStopTimedEvent(psObj->hExpireEvent);

    return TRUE;
}

/*******************************************************************************
*
*   bHandleServiceReady
*
*******************************************************************************/
static BOOLEAN bHandleServiceReady (
    TRAFCAM_MGR_OBJECT_STRUCT *psObj
        )
{
    BOOLEAN bResult = FALSE;
    TRAFCAM_APP_OBJECT_STRUCT *psAppObj = NULL;
    DATASERVICE_ERROR_CODE_ENUM eErrorCode =
        DATASERVICE_ERROR_CODE_NONE;
    char *pacDatabaseFilePath = NULL;

    do
    {
        // Validate the hService, then get a pointer to the
        // locked app-facing object
        psAppObj = psGetAppFacingObject((WEATHER_SERVICE_OBJECT)psObj);
        if (psAppObj == NULL)
        {
            // Error, not sure why this would happen
            vSetError(psObj, DATASERVICE_ERROR_CODE_GENERAL);
            break;
        }

        // Register for a timed event
        psObj->hExpireEvent =
            DATASERVICE_IMPL_hRegisterTimedEvent(
            (DATASERVICE_IMPL_HDL)psObj, (void *)NULL);
        if(psObj->hExpireEvent == DATASERVICE_TIMED_EVENT_INVALID_HDL)
        {
            // Error!
            vSetError(psObj, DATASERVICE_ERROR_CODE_GENERAL);
            break;
        }

        // Construct the path needed for the reference database
        bResult = DB_UTIL_bCreateFilePath(
            psObj->pacRefDatabaseDirPath,
            TRAFCAM_DATABASE_FOLDER,
            TRAFCAM_REF_DATABASE_FILENAME,
            &pacDatabaseFilePath);
        if (bResult == FALSE)
        {
            // Error!  Couldn't build database path
            vSetError(psObj, DATASERVICE_ERROR_CODE_GENERAL);
            break;
        }

        // Connect to the ref database bank, but
        // never perform an integrity check
        psAppObj->hSQLRefConnection =
            DB_UTIL_hConnectToReference(
            &pacDatabaseFilePath[0],
            (DB_UTIL_CHECK_VERSION_HANDLER)bVerifyDBSchemaVersion,
            (void *)(size_t)TRAFCAM_DATABASE_FILE_VERSION,
            &eErrorCode,
            SQL_INTERFACE_OPTIONS_READONLY |
            SQL_INTERFACE_OPTIONS_SKIP_CORRUPTION_CHECK);
        // If the connection succeeded, continue with
        // initial processing
        if (psAppObj->hSQLRefConnection == SQL_INTERFACE_INVALID_OBJECT)
        {
            vSetError(psObj, DATASERVICE_ERROR_CODE_DATABASE_NOT_FOUND);
            break;
        }

        if (psObj->hInterface == TRAFCAM_INTERFACE_INVALID_OBJECT)
        {
            SMS_OBJECT hParent = SMS_INVALID_OBJECT;

            hParent = DATASERVICE_IMPL_hSMSObj((DATASERVICE_IMPL_HDL)psObj);

            // Start the interface
            psObj->hInterface = GsTrafcamIntf.hInit(
                (TRAFFIC_CAMERAS_SERVICE_OBJECT)psObj, hParent);
            if (psObj->hInterface == TRAFCAM_INTERFACE_INVALID_OBJECT)
            {
                vSetError(psObj, DATASERVICE_ERROR_CODE_GENERAL);
                break;
            }
        }

    } while(FALSE);

    if (pacDatabaseFilePath != NULL)
    {
        SMSO_vDestroy((SMS_OBJECT)pacDatabaseFilePath);
    }

    vReleaseAppFacingObject(psAppObj);

    return bResult;
}

/*******************************************************************************
*
*   vReleaseAppFacingObject
*
*******************************************************************************/
static void vReleaseAppFacingObject (
    TRAFCAM_APP_OBJECT_STRUCT *psAppObj
        )
{
    if (psAppObj != NULL)
    {
        SMSO_vUnlock((SMS_OBJECT)psAppObj);
    }
}

/*******************************************************************************
*
*   psGetAppFacingObject
*
*******************************************************************************/
static TRAFCAM_APP_OBJECT_STRUCT *psGetAppFacingObject (
    TRAFFIC_CAMERAS_SERVICE_OBJECT hTrafficCamerasService
        )
{
    TRAFCAM_MGR_OBJECT_STRUCT *psObj =
        (TRAFCAM_MGR_OBJECT_STRUCT *)hTrafficCamerasService;
    BOOLEAN bValid, bLocked;

    do
    {
        bValid = DATASERVICE_IMPL_bValid(
            (DATASERVICE_IMPL_HDL)hTrafficCamerasService);
        if (bValid == FALSE)
        {
            break;
        }

        bLocked = SMSO_bLock((SMS_OBJECT)psObj->psAppObj,
            OSAL_OBJ_TIMEOUT_INFINITE);
        if (bLocked == FALSE)
        {
            break;
        }

        return psObj->psAppObj;

    } while (FALSE);

    return NULL;
}

/*******************************************************************************
*
*   vSetError
*
*******************************************************************************/
static void vSetError (
    TRAFCAM_MGR_OBJECT_STRUCT *psObj,
    DATASERVICE_ERROR_CODE_ENUM eErrorCode
        )
{
    SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
        TRAFCAM_MGR_OBJECT_NAME": vSetError (%d)", eErrorCode
            );

    // Tell the DSM about it
    DATASERVICE_IMPL_vError((DATASERVICE_IMPL_HDL)psObj, eErrorCode);

    return;
}

/*****************************************************************************
*
*   bDBGetContentVersion
*
*****************************************************************************/
static BOOLEAN bDBGetContentVersion (
    SQL_INTERFACE_OBJECT hSQLConnection,
    N32 *pn32DBVer
        )
{
    TRAFCAM_DB_RESULT_STRUCT sResult;
    BOOLEAN bResult = FALSE;
    TRAFCAM_DB_VERSION_ROW_STRUCT sVersion;

    OSAL.bMemSet(&sResult, 0, sizeof(sResult));
    OSAL.bMemSet(&sVersion, 0, sizeof(sVersion));

    sResult.uDbRow.psVersion = &sVersion;

    if ((hSQLConnection == SQL_INTERFACE_INVALID_OBJECT) ||
        (pn32DBVer == NULL))
    {
        return FALSE;
    }

    // Perform the SQL query and process the result
    // (it will provide us with a data row)
    bResult = SQL_INTERFACE.bQuery(
        hSQLConnection,
        TRAFCAM_SELECT_DB_DATA_VERSION,
        (SQL_QUERY_RESULT_HANDLER)bDBProcessSelectDBVersion,
        &sResult
            );
    if ((bResult == TRUE) &&
        (sResult.bResult == TRUE))
    {
        *pn32DBVer = (N32)sResult.uDbRow.psVersion->un32Version;
    }

    return sResult.bResult;
}

/*******************************************************************************
*
*   n32ExtractDataVersion
*
*******************************************************************************/
static N32 n32ExtractDataVersion(
    SQL_INTERFACE_OBJECT hConnection
        )
{
    N32 n32Result = DB_UTIL_DB_UNDER_CONSTRUCTION_VER;
    BOOLEAN bLoaded;

    if (hConnection == SQL_INTERFACE_INVALID_OBJECT)
    {
        return DB_UTIL_DB_UNDER_CONSTRUCTION_VER;
    }

    // Get the Content Version (table!)
    bLoaded = bDBGetContentVersion(
        hConnection, &n32Result
            );
    if (bLoaded != TRUE)
    {
        n32Result = DB_UTIL_DB_UNDER_CONSTRUCTION_VER;
    }

    return n32Result;
}

/*****************************************************************************
*
*   vDSRLRelease
*
*****************************************************************************/
static void vDSRLRelease(
    DSRL_OBJECT hDSRL
        )
{
    if (hDSRL != DSRL_INVALID_OBJECT)
    {
        OSAL_RETURN_CODE_ENUM eReturnCode = OSAL_ERROR;
        TRAFCAM_DSRL_DESC_STRUCT *psDSRLDesc = NULL;

        printf(TRAFCAM_MGR_OBJECT_NAME": Releasing DSRL %p\n", hDSRL);

        psDSRLDesc = (TRAFCAM_DSRL_DESC_STRUCT *)DSRL_pvServiceData(hDSRL);
        if (psDSRLDesc != NULL)
        {
            if (psDSRLDesc->hTargetsList != OSAL_INVALID_OBJECT_HDL)
            {
                eReturnCode = OSAL.eLinkedListRemoveAll(
                    psDSRLDesc->hTargetsList,
                    (OSAL_LL_RELEASE_HANDLER)DSRL_TARGET_vDestroyByType
                        );
                if (eReturnCode != OSAL_SUCCESS)
                {
                    SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                        TRAFCAM_MGR_OBJECT_NAME": failed to cleanup"
                        " list (%s)", OSAL.pacGetReturnCodeName(eReturnCode)
                            );
                }

                eReturnCode = OSAL.eLinkedListDelete(
                    psDSRLDesc->hTargetsList);
                if (eReturnCode != OSAL_SUCCESS)
                {
                    SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                        TRAFCAM_MGR_OBJECT_NAME": failed to delete"
                        " list (%s)", OSAL.pacGetReturnCodeName(eReturnCode)
                            );
                }
                psDSRLDesc->hTargetsList = OSAL_INVALID_OBJECT_HDL;
            }

            if (psDSRLDesc->hEntriesList != OSAL_INVALID_OBJECT_HDL)
            {
                eReturnCode = OSAL.eLinkedListRemoveAll(
                    psDSRLDesc->hEntriesList, NULL);
                if (eReturnCode != OSAL_SUCCESS)
                {
                    SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                        TRAFCAM_MGR_OBJECT_NAME": failed to cleanup"
                        " list (%s)", OSAL.pacGetReturnCodeName(eReturnCode)
                            );
                }

                eReturnCode = OSAL.eLinkedListDelete(psDSRLDesc->hEntriesList);
                if (eReturnCode != OSAL_SUCCESS)
                {
                    SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                        TRAFCAM_MGR_OBJECT_NAME": failed to delete"
                        " list (%s)", OSAL.pacGetReturnCodeName(eReturnCode)
                            );
                }
                psDSRLDesc->hEntriesList = OSAL_INVALID_OBJECT_HDL;
            }
        }

        DSRL_vDestroy(hDSRL);
    }

    return;
}

/*****************************************************************************
*
*   bDSRLDescInit
*
*****************************************************************************/
static BOOLEAN bDSRLDescInit(
    DSRL_ARG_STRUCT *psDSRLArg,
    TRAFCAM_APP_OBJECT_STRUCT *psObj
        )
{
    TRAFCAM_DSRL_DESC_STRUCT *psDSRLDesc = NULL;
    OSAL_RETURN_CODE_ENUM eReturnCode = OSAL_ERROR;
    BOOLEAN bResult = FALSE;

    do
    {
        BOOLEAN bSuccess = FALSE;

        bSuccess = DSRL_bSetFinalizeFunction(psDSRLArg->hDSRL,
            (DSRL_FINALIZE_FUNCTION)vDSRLFinalizeEntry, psObj
                );
        if (bSuccess == FALSE)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                TRAFCAM_MGR_OBJECT_NAME": Failed to set finalizer for DSRL %p",
                psDSRLArg->hDSRL
                    );
            break;
        }

        // get the DSRL desc pointer from DSRL
        psDSRLDesc = (TRAFCAM_DSRL_DESC_STRUCT *)DSRL_pvServiceData(
            psDSRLArg->hDSRL
                );
        if (psDSRLDesc == NULL)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                TRAFCAM_MGR_OBJECT_NAME": cannot get DSRL desc from DSRL"
                    );
            break;
        }

        eReturnCode = OSAL.eLinkedListCreate(
            &psDSRLDesc->hTargetsList,
            TRAFCAM_MGR_OBJECT_NAME":DSRLDesc:TgtList",
            NULL,
            OSAL_LL_OPTION_NONE
                );
        if (eReturnCode != OSAL_SUCCESS)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                TRAFCAM_MGR_OBJECT_NAME": cannot create targets list (%s)",
                OSAL.pacGetReturnCodeName(eReturnCode)
                    );
            break;
        }

        eReturnCode = OSAL.eLinkedListCreate(
            &psDSRLDesc->hEntriesList,
            TRAFCAM_MGR_OBJECT_NAME":DSRLDesc:EntriesList",
            n16CompareHandles,
            OSAL_LL_OPTION_NONE
                );
        if (eReturnCode != OSAL_SUCCESS)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                TRAFCAM_MGR_OBJECT_NAME": cannot create entry list (%s)",
                OSAL.pacGetReturnCodeName(eReturnCode)
                    );
            break;
        }

        // Add this to list of tracked DSRL lists of targets for this service
        eReturnCode =
            OSAL.eLinkedListAdd(psObj->hDSRLList,
            &(psDSRLDesc->hDSRLListEntry),
            psDSRLArg->hDSRL);
        if (eReturnCode != OSAL_SUCCESS)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                TRAFCAM_MGR_OBJECT_NAME": cannot add DSRL to pool (%s)",
                OSAL.pacGetReturnCodeName(eReturnCode)
                    );
            break;
        }

        bResult = TRUE;
    } while (FALSE);

    if ((bResult == FALSE) && (psDSRLDesc != NULL))
    {
        if (psDSRLDesc->hDSRLListEntry != OSAL_INVALID_LINKED_LIST_ENTRY)
        {
            eReturnCode = OSAL.eLinkedListRemove(psDSRLDesc->hDSRLListEntry);
            if (eReturnCode != OSAL_SUCCESS)
            {
                SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    TRAFCAM_MGR_OBJECT_NAME": can't remove DSRL from pool (%s)",
                    OSAL.pacGetReturnCodeName(eReturnCode)
                        );
            }
            psDSRLDesc->hDSRLListEntry = OSAL_INVALID_LINKED_LIST_ENTRY;
        }
        vDSRLRelease(psDSRLArg->hDSRL);
    }

    return bResult;
}

/*****************************************************************************
*
*   bDSRLAddTarget
*
*****************************************************************************/
static BOOLEAN bDSRLAddTarget(
    DSRL_OBJECT hDSRL,
    DSRL_TARGET_OBJECT hTargetObj
        )
{
    BOOLEAN bResult = FALSE;

    do
    {
        OSAL_RETURN_CODE_ENUM eReturnCode = OSAL_ERROR;
        TRAFCAM_DSRL_DESC_STRUCT *psDSRLDesc = NULL;

        if (hTargetObj == DSRL_TARGET_INVALID_OBJECT)
        {
            break;
        }

        psDSRLDesc = (TRAFCAM_DSRL_DESC_STRUCT *)DSRL_pvServiceData(hDSRL);
        if (psDSRLDesc == NULL)
        {
            // Error
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                TRAFCAM_MGR_OBJECT_NAME": unknown DSRL 0x%p", hDSRL);
            break;
        }

        // Add this target to our DSRL description
        eReturnCode = OSAL.eLinkedListAdd(
            psDSRLDesc->hTargetsList,
            OSAL_INVALID_LINKED_LIST_ENTRY_PTR,
            hTargetObj);
        if (eReturnCode != OSAL_SUCCESS)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                TRAFCAM_MGR_OBJECT_NAME": failed to add target to list (%s)",
                OSAL.pacGetReturnCodeName(eReturnCode)
                    );
            break;
        }

        bResult = TRUE;
    } while (FALSE);

    return bResult;
}

/*****************************************************************************
*
*   bDSRLAddTargets
*
*****************************************************************************/
static BOOLEAN bDSRLAddTargets(
    DSRL_ARG_STRUCT *psDSRLArg
        )
{
    size_t tDSRLTargetIndex;
    BOOLEAN bResult = TRUE;

    for (tDSRLTargetIndex = 0;
         tDSRLTargetIndex < psDSRLArg->tNumTargets;
         tDSRLTargetIndex++)
    {
        bResult = bDSRLAddTarget(psDSRLArg->hDSRL,
            psDSRLArg->ahTargetList[tDSRLTargetIndex]);
        if (bResult == FALSE)
        {
            break;
        }
        else
        {
            // if target has been added successfully
            // need to clean it in array to avoid target destroy
            // in general DATASERVICE event processor
            psDSRLArg->ahTargetList[tDSRLTargetIndex] =
                DSRL_TARGET_INVALID_OBJECT;
        }
    }

    return bResult;
}

/*****************************************************************************
*
*   bDiffListGenerator
*
*****************************************************************************/
static BOOLEAN bDiffListGenerator(
    DSRL_ENTRY_OBJECT hDSRLEntry,
    TRAFCAM_DIFF_LIST_ITERATOR_STRUCT *psIterator
        )
{
    BOOLEAN bContinue = TRUE;
    OSAL_RETURN_CODE_ENUM eReturnCode = OSAL_ERROR;

    if ((hDSRLEntry == DSRL_ENTRY_INVALID_OBJECT) ||
        (psIterator == NULL))
    {
        return FALSE;
    }

    eReturnCode = OSAL.eLinkedListSearch(
        psIterator->hSubList,
        &psIterator->hEntry,
        hDSRLEntry);
    if (eReturnCode == OSAL_OBJECT_NOT_FOUND)
    {
        // object not found in list, lets add entry to diff
        // list
        eReturnCode = OSAL.eLinkedListAdd(psIterator->hDiffList,
            OSAL_INVALID_LINKED_LIST_ENTRY_PTR, hDSRLEntry);
        if ((eReturnCode != OSAL_SUCCESS) &&
            (eReturnCode != OSAL_ERROR_LIST_ITEM_NOT_UNIQUE))
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                TRAFCAM_MGR_OBJECT_NAME": add to diff list failed"
                " (%s)", OSAL.pacGetReturnCodeName(eReturnCode)
                    );
            bContinue = FALSE;
        }
    }
    else if (eReturnCode != OSAL_SUCCESS)
    {
        SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
            TRAFCAM_MGR_OBJECT_NAME": search in sub list failed"
            " (%s)", OSAL.pacGetReturnCodeName(eReturnCode)
                );
        bContinue = FALSE;
    }

    return bContinue;
}

/*****************************************************************************
*
*   bGenerateEntriesDiffList
*
*****************************************************************************/
static BOOLEAN bGenerateEntriesDiffList (
    OSAL_OBJECT_HDL hMainList,
    OSAL_OBJECT_HDL hSubList,
    OSAL_OBJECT_HDL hDiffList
        )
{
    BOOLEAN bResult = FALSE;

    do
    {
        OSAL_RETURN_CODE_ENUM eReturnCode = OSAL_ERROR;
        TRAFCAM_DIFF_LIST_ITERATOR_STRUCT sIterator;

        OSAL.bMemSet(&sIterator, 0, sizeof(sIterator));

        if ((hMainList == OSAL_INVALID_OBJECT_HDL) ||
            (hSubList == OSAL_INVALID_OBJECT_HDL) ||
            (hDiffList == OSAL_INVALID_OBJECT_HDL))
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                TRAFCAM_MGR_OBJECT_NAME": invalid input parameters in"
                " function (%s)", __FUNCTION__
                    );
            break;
        }

        sIterator.hDiffList = hDiffList;
        sIterator.hSubList = hSubList;

        eReturnCode = OSAL.eLinkedListIterate(hMainList,
            (OSAL_LL_ITERATOR_HANDLER)bDiffListGenerator, &sIterator);
        if ((eReturnCode != OSAL_SUCCESS) && (eReturnCode != OSAL_NO_OBJECTS))
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                TRAFCAM_MGR_OBJECT_NAME": cannot iterate linked list (%s)",
                OSAL.pacGetReturnCodeName(eReturnCode)
                    );
            break;
        }

        bResult = TRUE;
    } while (FALSE);

    return bResult;
}

/*******************************************************************************
*
*   psCreateFilterItem
*
*******************************************************************************/
static TRAFCAM_BSA_FILTER_STRUCT *psCreateFilterItem(
    TRAFFIC_MARKET tMarket,
    TRAFFIC_BSA tBSA
        )
{
    TRAFCAM_BSA_FILTER_STRUCT *psResult = NULL;

    psResult =
        (TRAFCAM_BSA_FILTER_STRUCT *)OSAL.pvLinkedListMemoryAllocate(
        TRAFCAM_MGR_OBJECT_NAME":FilterItem", sizeof(*psResult), TRUE);

    if (psResult != NULL)
    {
        psResult->tMarket = tMarket;
        psResult->tBSA = tBSA;
    }

    return psResult;
}

/*******************************************************************************
*
*   bDBProcessSelectBSAs
*
*******************************************************************************/
static BOOLEAN bDBProcessSelectBSAs(
    SQL_QUERY_COLUMN_STRUCT *psColumn,
    N32 n32NumberOfColumns,
    TRAFCAM_DB_QUERY_OBJECTS_STRUCT *psResult
        )
{
    BOOLEAN bContinue = FALSE;

    do
    {
        TRAFCAM_BSA_FILTER_STRUCT *psBSAFilter = NULL;
        OSAL_RETURN_CODE_ENUM eReturnCode = OSAL_ERROR;

        if (psResult == NULL)
        {
            break;
        }

        psResult->bResult = FALSE;

        if (n32NumberOfColumns != 2)
        {
            break;
        }

        psBSAFilter = psCreateFilterItem(
            (TRAFFIC_MARKET)psColumn[0].uData.sUN32.un32Data,
            (TRAFFIC_BSA)psColumn[1].uData.sUN32.un32Data);
        if (psBSAFilter == NULL)
        {
            break;
        }

        eReturnCode = OSAL.eLinkedListAdd(psResult->hObjectsList,
            &psBSAFilter->hEntry, psBSAFilter);
        if (eReturnCode != OSAL_SUCCESS)
        {
            OSAL.vLinkedListMemoryFree(psBSAFilter);
            if (eReturnCode != OSAL_ERROR_LIST_ITEM_NOT_UNIQUE)
            {
                SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    TRAFCAM_MGR_OBJECT_NAME": cannot iterate linked list (%s)",
                    OSAL.pacGetReturnCodeName(eReturnCode)
                        );
                break;
            }
        }

        psResult->bResult = TRUE;
        bContinue = TRUE;
    } while (FALSE);

    return bContinue;
}

/*******************************************************************************
*
*   bCreateFilterItemsForMarket
*
*******************************************************************************/
static BOOLEAN bCreateFilterItemsForMarket(
    TRAFCAM_APP_OBJECT_STRUCT *psObj,
    TRAFFIC_MARKET tMarket,
    OSAL_OBJECT_HDL hFilter
        )
{
    BOOLEAN bResult = FALSE;

    do
    {
        int iReturn = 0;
        TRAFCAM_DB_QUERY_OBJECTS_STRUCT sResult;

        OSAL.bMemSet(&sResult, 0, sizeof(sResult));
        sResult.hObjectsList = hFilter;
        sResult.psAppObj = psObj;

        // Specify a search box within the target area
        iReturn = snprintf(psObj->acBuffer, sizeof(psObj->acBuffer),
            TRAFCAM_SELECT_BSAS_BY_MARKET, tMarket);
        if (iReturn <= 0)
        {
            break;
        }

        printf(TRAFCAM_MGR_OBJECT_NAME": SQL request %s will be performed.\n",
            psObj->acBuffer
                );

        sResult.bResult = TRUE;

        // Perform the SQL query and process the result
        bResult = SQL_INTERFACE.bQuery(
            psObj->hSQLRefConnection, psObj->acBuffer,
            (SQL_QUERY_RESULT_HANDLER)bDBProcessSelectBSAs,
            &sResult
                );
        if ((bResult == FALSE) || (sResult.bResult == FALSE))
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                TRAFCAM_MGR_OBJECT_NAME": SQL request failed"
                    );
            bResult = FALSE;
            break;
        }

    } while (FALSE);

    return bResult;
}

/*******************************************************************************
*
*   bProcessPredefinedLocation
*
*******************************************************************************/
static BOOLEAN bProcessPredefinedLocation(
    TRAFFIC_LOCID_OBJECT hLocID,
    TRAFCAM_TARGET_LIST_ITERATOR_STRUCT *psIterator
        )
{
    BOOLEAN bResult = FALSE;

    do
    {
        TRAFFIC_LOCID_TYPE_ENUM eType = TRAFFIC_LOCID_TYPE_INVALID;
        OSAL_RETURN_CODE_ENUM eReturnCode = OSAL_ERROR;
        TRAFFIC_MARKET tMarket = TRAFFIC_INVALID_MARKET;

        eType = TRAFFIC_LOCID.eType(hLocID);

        tMarket = TRAFFIC_LOCID.tMarket(hLocID);

        if (eType == TRAFFIC_LOCID_TYPE_MARKET)
        {
            BOOLEAN bSuccess = FALSE;

            // target type is market, need to get all the BSAs for this market
            bSuccess = bCreateFilterItemsForMarket(psIterator->psObj,
                tMarket, psIterator->hEntriesList);
            if (bSuccess == FALSE)
            {
                break;
            }
        }
        else if (eType == TRAFFIC_LOCID_TYPE_BSA)
        {
            // ok, this is BSA type entry, lets create new entry and add it
            TRAFCAM_BSA_FILTER_STRUCT *psBSAFilter = NULL;
            TRAFFIC_BSA tBSA = TRAFFIC_INVALID_BSA;

            tBSA = TRAFFIC_LOCID.tBSA(hLocID);

            psBSAFilter = psCreateFilterItem(tMarket, tBSA);
            if (psBSAFilter == NULL)
            {
                break;
            }

            eReturnCode = OSAL.eLinkedListAdd(psIterator->hEntriesList,
                &psBSAFilter->hEntry, psBSAFilter);
            if (eReturnCode == OSAL_ERROR_LIST_ITEM_NOT_UNIQUE)
            {
                // valid case, we probably have this item in list
                OSAL.vLinkedListMemoryFree(psBSAFilter);
            }
            else if (eReturnCode != OSAL_SUCCESS)
            {
                OSAL.vLinkedListMemoryFree(psBSAFilter);
                SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    TRAFCAM_MGR_OBJECT_NAME": cannot add item to list (%s)",
                    OSAL.pacGetReturnCodeName(eReturnCode)
                        );
                break;
            }
        }
        else
        {
            // locid type should be BSA or MARKET
            break;
        }

        bResult = TRUE;
    } while (FALSE);

    return bResult;
}

/*****************************************************************************
*
*   bEntryCheckForDSRL
*
*****************************************************************************/
static BOOLEAN bEntryCheckForDSRL(
    TRAFFIC_MSG_OBJECT hTrafficMsg,
    TRAFCAM_TARGET_LIST_ITERATOR_STRUCT *psIterator
        )
{
    BOOLEAN bContinue = FALSE;

    do
    {
        OSAL_RETURN_CODE_ENUM eReturnCode = OSAL_ERROR;
        LOCATION_OBJECT hLocation = LOCATION_INVALID_OBJECT;
        TRAFCAM_TARGET_MESSAGE_CHECK_ITERATOR sIterator;

        if ((hTrafficMsg == TRAFFIC_MSG_INVALID_OBJECT) ||
            (psIterator == NULL))
        {
            return FALSE;
        }

        hLocation = TRAFFIC_MSG.hGetLocation(hTrafficMsg, 0);
        if (hLocation == LOCATION_INVALID_OBJECT)
        {
            break;
        }

        // lets check message for targets
        sIterator.bFits = FALSE;
        sIterator.hLocation = hLocation;

        eReturnCode = OSAL.eLinkedListIterate(psIterator->hTargetsList,
            (OSAL_LL_ITERATOR_HANDLER)bTargetMessageCheck, &sIterator);
        if ((eReturnCode != OSAL_SUCCESS) && (eReturnCode != OSAL_NO_OBJECTS))
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                TRAFCAM_MGR_OBJECT_NAME":Error iterating the targets list (%s)",
                OSAL.pacGetReturnCodeName(eReturnCode)
                    );
            break;
        }

        if (sIterator.bFits == TRUE)
        {
            eReturnCode = OSAL.eLinkedListAdd(psIterator->hEntriesList,
                OSAL_INVALID_LINKED_LIST_ENTRY_PTR, hTrafficMsg);
            if (eReturnCode != OSAL_SUCCESS)
            {
                SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    TRAFCAM_MGR_OBJECT_NAME": Error adding to list (%s)",
                    OSAL.pacGetReturnCodeName(eReturnCode)
                        );
                break;
            }
        }

        bContinue = TRUE;
    } while (FALSE);

    return bContinue;
}

/*****************************************************************************
*
*   bGenerateDSRLEntriesList
*
*****************************************************************************/
static BOOLEAN bGenerateDSRLEntriesList(
    TRAFCAM_APP_OBJECT_STRUCT *psObj,
    TRAFCAM_DSRL_DESC_STRUCT *psDSRLDesc,
    OSAL_OBJECT_HDL hEntriesList
        )
{
    TRAFCAM_TARGET_LIST_ITERATOR_STRUCT sIterator;
    OSAL_RETURN_CODE_ENUM eReturnCode = OSAL_ERROR;

    sIterator.hEntriesList = hEntriesList;
    sIterator.psObj = psObj;
    sIterator.hTargetsList = psDSRLDesc->hTargetsList;

    eReturnCode = OSAL.eLinkedListIterate(psObj->hEntriesList,
        (OSAL_LL_ITERATOR_HANDLER)bEntryCheckForDSRL, &sIterator
            );
    if ((eReturnCode != OSAL_SUCCESS) && (eReturnCode != OSAL_NO_OBJECTS))
    {
        SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
            TRAFCAM_MGR_OBJECT_NAME": Error iterating targets list (%s)",
            OSAL.pacGetReturnCodeName(eReturnCode)
                );
        return FALSE;
    }

    return TRUE;
}

/*****************************************************************************
*
*   bDSRLFillDiffEntriesLists
*
*****************************************************************************/
static BOOLEAN bDSRLFillDiffEntriesLists(
    DSRL_OBJECT hDSRL,
    TRAFCAM_APP_OBJECT_STRUCT *psObj
        )
{
    BOOLEAN bResult = FALSE;
    OSAL_RETURN_CODE_ENUM eReturnCode = OSAL_ERROR;

    do
    {
        BOOLEAN bSuccess = FALSE;
        TRAFCAM_DSRL_DESC_STRUCT *psDSRLDesc = NULL;

        psDSRLDesc = (TRAFCAM_DSRL_DESC_STRUCT *)DSRL_pvServiceData(hDSRL);
        if (psDSRLDesc == NULL)
        {
            break;
        }

        bSuccess = bGenerateDSRLEntriesList(psObj, psDSRLDesc,
            psObj->hNewEntriesList);
        if (bSuccess == FALSE)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                TRAFCAM_MGR_OBJECT_NAME": cannot generate new list "
                "of entries");
            break;
        }

        bSuccess = bGenerateEntriesDiffList(psObj->hNewEntriesList,
            psDSRLDesc->hEntriesList, psObj->hEntriesToAdd);
        if (bSuccess == FALSE)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                TRAFCAM_MGR_OBJECT_NAME": cannot generate list "
                "of entries to add");
            break;
        }

        bSuccess = bGenerateEntriesDiffList(psDSRLDesc->hEntriesList,
            psObj->hNewEntriesList, psObj->hEntriesToRemove);
        if (bSuccess == FALSE)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                TRAFCAM_MGR_OBJECT_NAME": cannot generate list "
                "of entries to remove");
            break;
        }

        bResult = TRUE;
    } while (FALSE);

    eReturnCode = OSAL.eLinkedListRemoveAll(psObj->hNewEntriesList, NULL);
    if (eReturnCode != OSAL_SUCCESS)
    {
        SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
            TRAFCAM_MGR_OBJECT_NAME": cannot cleanup linked list (%s)",
            OSAL.pacGetReturnCodeName(eReturnCode));
    }

    return bResult;
}

/*****************************************************************************
*
*   bIsUpdateNeeded
*
*****************************************************************************/
static BOOLEAN bIsUpdateNeeded(
    TRAFCAM_APP_OBJECT_STRUCT *psObj
        )
{
    BOOLEAN bResult = TRUE;

    do
    {
        UN32 un32Items1 = 0, un32Items2 = 0;
        OSAL_RETURN_CODE_ENUM eReturnCode = OSAL_ERROR;

        eReturnCode = OSAL.eLinkedListItems(psObj->hEntriesToAdd, &un32Items1);
        if (eReturnCode != OSAL_SUCCESS)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                TRAFCAM_MGR_OBJECT_NAME": failed to get number of"
                " items in list (%s)",
                OSAL.pacGetReturnCodeName(eReturnCode)
                    );
        }

        eReturnCode = OSAL.eLinkedListItems(psObj->hEntriesToRemove,
            &un32Items2);
        if (eReturnCode != OSAL_SUCCESS)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                TRAFCAM_MGR_OBJECT_NAME": failed to get number of"
                " items in list (%s)",
                OSAL.pacGetReturnCodeName(eReturnCode)
                    );
        }

        if ((un32Items1 != 0) || (un32Items2 != 0))
        {
            printf(TRAFCAM_MGR_OBJECT_NAME": DSRL entries to add - %d,"
                "entries to remove - %d\n", un32Items1, un32Items2
                    );

            DATASERVICE_IMPL_vLog(TRAFCAM_MGR_OBJECT_NAME
                ": DSRL entries to add - %d, entries to remove - %d\n",
                un32Items1, un32Items2
                    );

            break;
        }
        else
        {
            DATASERVICE_IMPL_vLog(TRAFCAM_MGR_OBJECT_NAME
                ": DSRL update is not needed\n"
                    );
        }

        bResult = FALSE;
    } while (FALSE);

    return bResult;
}

/*****************************************************************************
*
*   bEntryAddToDSRL
*
*****************************************************************************/
static BOOLEAN bEntryAddToDSRL(
    DSRL_ENTRY_OBJECT hDSRLEntry,
    DSRL_OBJECT hDSRL
        )
{
    BOOLEAN bContinue = FALSE, bAdded = FALSE;
    OSAL_RETURN_CODE_ENUM eReturnCode = OSAL_ERROR;

    do
    {
        DSRL_ADD_REPLACE_RESULT_ENUM eDSRLResult = DSRL_ADD_REPLACE_ERROR;
        TRAFCAM_DSRL_DESC_STRUCT *psDSRLDesc = NULL;
        TRAFCAM_DSRL_ENTRY_DESC_STRUCT *psEntryDesc = NULL;

        if ((hDSRL == DSRL_INVALID_OBJECT) ||
            (hDSRLEntry == DSRL_ENTRY_INVALID_OBJECT))
        {
            break;
        }

        psEntryDesc =
            (TRAFCAM_DSRL_ENTRY_DESC_STRUCT *)DSRL_ENTRY_pvServiceData(
            hDSRLEntry);
        if (psEntryDesc == NULL)
        {
            break;
        }

        psDSRLDesc =
            (TRAFCAM_DSRL_DESC_STRUCT *)DSRL_pvServiceData(hDSRL);
        if (psDSRLDesc == NULL)
        {
            break;
        }

        eDSRLResult = DSRL_eAddEntry(hDSRL, hDSRLEntry);
        if (eDSRLResult != DSRL_ADD_REPLACE_OK)
        {
            printf(TRAFCAM_MGR_OBJECT_NAME": entry cannot be "
                "added for some reason.\n"
                    );
            bContinue = TRUE;
            break;
        }
        bAdded = TRUE;

        eReturnCode = OSAL.eLinkedListAdd(psDSRLDesc->hEntriesList,
            OSAL_INVALID_LINKED_LIST_ENTRY_PTR, hDSRLEntry
                );
        if (eReturnCode != OSAL_SUCCESS)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                TRAFCAM_MGR_OBJECT_NAME": cannot add entry"
                " to list (%s)",
                OSAL.pacGetReturnCodeName(eReturnCode)
                    );
            break;
        }

        if (psEntryDesc->hDSRLList == OSAL_INVALID_OBJECT_HDL)
        {
            // in case if DSRLs list in EntryDesc struct is not created
            // lets create it
            eReturnCode = OSAL.eLinkedListCreate(&(psEntryDesc->hDSRLList),
                TRAFCAM_MGR_OBJECT_NAME":DSRLEntry:DSRLList",
                n16CompareHandles, OSAL_LL_OPTION_NONE
                    );
            if (eReturnCode != OSAL_SUCCESS)
            {
                SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    TRAFCAM_MGR_OBJECT_NAME": cannot create link list (%s)",
                    OSAL.pacGetReturnCodeName(eReturnCode)
                        );
                break;
            }
        }

        eReturnCode = OSAL.eLinkedListAdd(psEntryDesc->hDSRLList,
            OSAL_INVALID_LINKED_LIST_ENTRY_PTR, hDSRL);
        if (eReturnCode != OSAL_SUCCESS)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                TRAFCAM_MGR_OBJECT_NAME": cannot add entry"
                " to list (%s)",
                OSAL.pacGetReturnCodeName(eReturnCode)
                    );
            break;
        }

        bContinue = TRUE;
    } while (FALSE);

    if ((bContinue == TRUE) && (bAdded == FALSE))
    {
        //lets remove entry from DSRL to avoid any issues
        DSRL_vRemoveEntry(hDSRL, hDSRLEntry);
    }

    return bContinue;
}

/*****************************************************************************
*
*   bDSRLAddEntries
*
*****************************************************************************/
static BOOLEAN bDSRLAddEntries(
    TRAFCAM_APP_OBJECT_STRUCT *psObj,
    DSRL_OBJECT hDSRL
        )
{
    BOOLEAN bResult = TRUE;
    OSAL_RETURN_CODE_ENUM eReturnCode = OSAL_ERROR;

    eReturnCode = OSAL.eLinkedListIterate(psObj->hEntriesToAdd,
        (OSAL_LL_ITERATOR_HANDLER)bEntryAddToDSRL, hDSRL
            );
    if (eReturnCode == OSAL_NO_OBJECTS)
    {
        printf(TRAFCAM_MGR_OBJECT_NAME": nothing to add, "
            "list is empty\n"
                );
    }
    else if (eReturnCode != OSAL_SUCCESS)
    {
        SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
            TRAFCAM_MGR_OBJECT_NAME": cannot iterate add entry"
            " list (%s)",
            OSAL.pacGetReturnCodeName(eReturnCode)
                );
        bResult = FALSE;
    }

    return bResult;
}

/*****************************************************************************
*
*   bEntryRemoveFromDSRL
*
*****************************************************************************/
static BOOLEAN bEntryRemoveFromDSRL(
    DSRL_ENTRY_OBJECT hDSRLEntry,
    DSRL_OBJECT hDSRL
        )
{
    if ((hDSRLEntry != DSRL_ENTRY_INVALID_OBJECT) &&
        (hDSRL != DSRL_INVALID_OBJECT))
    {
        DSRL_vRemoveEntry(hDSRL, hDSRLEntry);
        return TRUE;
    }

    return FALSE;
}

/*****************************************************************************
*
*   bDSRLRemoveEntries
*
*****************************************************************************/
static BOOLEAN bDSRLRemoveEntries(
    TRAFCAM_APP_OBJECT_STRUCT *psObj,
    DSRL_OBJECT hDSRL
        )
{
    BOOLEAN bResult = TRUE;
    OSAL_RETURN_CODE_ENUM eReturnCode = OSAL_ERROR;

    eReturnCode = OSAL.eLinkedListIterate(psObj->hEntriesToRemove,
        (OSAL_LL_ITERATOR_HANDLER)bEntryRemoveFromDSRL, hDSRL);
    if (eReturnCode == OSAL_NO_OBJECTS)
    {
        printf(TRAFCAM_MGR_OBJECT_NAME": nothing to remove, "
            "list is empty\n"
                );
    }
    else if (eReturnCode != OSAL_SUCCESS)
    {
        SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
            TRAFCAM_MGR_OBJECT_NAME": cannot iterate remove entry"
            " list (%s)",
            OSAL.pacGetReturnCodeName(eReturnCode)
                );
        bResult = FALSE;
    }

    return bResult;
}

/*****************************************************************************
*
*   bProcessCoordLocation
*
*****************************************************************************/
static BOOLEAN bProcessCoordLocation(
    LOCATION_OBJECT hLocation,
    TRAFCAM_TARGET_LIST_ITERATOR_STRUCT *psIteratorArg
        )
{
    BOOLEAN bResult = FALSE;

    do
    {
        OSAL_FIXED_OBJECT hTopRightLat,
            hTopRightLon,
            hBottomLeftLat,
            hBottomLeftLon,
            hCenterLat,
            hCenterLon;

        OSAL_FIXED_OBJECT_DATA atFixedData[OSAL_FIXED_OBJECT_SIZE * 4];
        UN8 un8NumFixed = 0;    // Keep track of the fixed objects

        N32 n32FixedTopRightLat,
            n32FixedTopRightLon,
            n32FixedBottomLeftLat,
            n32FixedhBottomLeftLon;
        int iReturn = 0;
        TRAFCAM_DB_QUERY_OBJECTS_STRUCT sResult;

        OSAL.bMemSet(&sResult, 0, sizeof(sResult));
        sResult.hObjectsList = psIteratorArg->hEntriesList;
        sResult.psAppObj = psIteratorArg->psObj;
        sResult.bResult = TRUE;

        // Check the lat/lon provided
        hCenterLat = LOCATION.hLat(hLocation);
        hCenterLon = LOCATION.hLon(hLocation);

        // Is the target location populated?
        if ((hCenterLat == OSAL_FIXED_INVALID_OBJECT) ||
            (hCenterLon == OSAL_FIXED_INVALID_OBJECT))
        {
            // No -- but that's okay...just hold off for now
            bResult = TRUE;
            break;
        }

        // Create the objects.  If this fails, LOCATION_bTopRight
        // will result in an error
        hTopRightLat = OSAL_FIXED.hCreateInMemory(0, 0,
            &atFixedData[OSAL_FIXED_OBJECT_SIZE * (un8NumFixed++)]);
        hTopRightLon = OSAL_FIXED.hCreateInMemory(0, 0,
            &atFixedData[OSAL_FIXED_OBJECT_SIZE * (un8NumFixed++)]);

        // Get the top right lat, lon
        bResult = LOCATION_bTopRight(hLocation,
            hTopRightLat, hTopRightLon);
        if (bResult == FALSE)
        {
            break;
        }

        // Create the objects.  If this fails, LOCATION_bBottomLeft
        // will result in an error
        hBottomLeftLat = OSAL_FIXED.hCreateInMemory(0, 0,
            &atFixedData[OSAL_FIXED_OBJECT_SIZE * (un8NumFixed++)]);
        hBottomLeftLon = OSAL_FIXED.hCreateInMemory(0, 0,
            &atFixedData[OSAL_FIXED_OBJECT_SIZE * (un8NumFixed++)]);

        // Get the bottom left lat, lon
        bResult = LOCATION_bBottomLeft(hLocation,
            hBottomLeftLat, hBottomLeftLon);
        if (bResult == FALSE)
        {
            break;
        }

        n32FixedTopRightLat =
            OSAL_FIXED.n32ScaledValue(hTopRightLat, LOCATION_BINPOINT);
        n32FixedTopRightLon =
            OSAL_FIXED.n32ScaledValue(hTopRightLon, LOCATION_BINPOINT);

        n32FixedBottomLeftLat =
            OSAL_FIXED.n32ScaledValue(hBottomLeftLat, LOCATION_BINPOINT);
        n32FixedhBottomLeftLon =
            OSAL_FIXED.n32ScaledValue(hBottomLeftLon, LOCATION_BINPOINT);


        iReturn = snprintf(psIteratorArg->psObj->acBuffer,
            sizeof(psIteratorArg->psObj->acBuffer),
            TRAFCAM_SELECT_BSAS_BY_COORDINATES,
            n32FixedBottomLeftLat, n32FixedTopRightLat,
            n32FixedhBottomLeftLon, n32FixedTopRightLon,
            n32FixedBottomLeftLat, n32FixedTopRightLat,
            n32FixedhBottomLeftLon, n32FixedTopRightLon,
            n32FixedBottomLeftLat, n32FixedTopRightLat,
            n32FixedhBottomLeftLon, n32FixedTopRightLon,
            n32FixedBottomLeftLat, n32FixedTopRightLat,
            n32FixedhBottomLeftLon, n32FixedTopRightLon,
            n32FixedBottomLeftLat, n32FixedBottomLeftLat,
            n32FixedhBottomLeftLon, n32FixedhBottomLeftLon,
            n32FixedTopRightLat, n32FixedTopRightLat,
            n32FixedhBottomLeftLon, n32FixedhBottomLeftLon,
            n32FixedTopRightLat, n32FixedTopRightLat,
            n32FixedTopRightLon, n32FixedTopRightLon,
            n32FixedBottomLeftLat, n32FixedBottomLeftLat,
            n32FixedTopRightLon, n32FixedTopRightLon );
        if (iReturn <= 0)
        {
            break;
        }

        printf(TRAFCAM_MGR_OBJECT_NAME": SQL request %s will be performed.\n",
            psIteratorArg->psObj->acBuffer
                );

        // Perform the SQL query and process the result
        bResult = SQL_INTERFACE.bQuery(
            psIteratorArg->psObj->hSQLRefConnection,
            psIteratorArg->psObj->acBuffer,
            (SQL_QUERY_RESULT_HANDLER)bDBProcessSelectBSAs, &sResult
                );
        if ((bResult == FALSE) || (sResult.bResult == FALSE))
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                TRAFCAM_MGR_OBJECT_NAME": SQL request failed"
                    );
            bResult = FALSE;
            break;
        }

    } while (FALSE);

    return bResult;
}

/*****************************************************************************
*
*   bTargetFilterGenerator
*
*****************************************************************************/
static BOOLEAN bTargetFilterGenerator(
    DSRL_TARGET_OBJECT hTarget,
    TRAFCAM_TARGET_LIST_ITERATOR_STRUCT *psIteratorArg
        )
{
    BOOLEAN bResult = FALSE;

    do
    {
        LOCATION_OBJECT hLocation = LOCATION_INVALID_OBJECT;
        DSRL_TARGET_TYPE_ENUM eType = DSRL_TARGET_TYPE_UNKNOWN;
        TRAFFIC_LOCID_OBJECT hLocID = TRAFFIC_LOCID_INVALID_OBJECT;

        eType = DSRL_TARGET.eType(hTarget);
        if (eType != DSRL_TARGET_TYPE_LOCATION)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                TRAFCAM_MGR_OBJECT_NAME": target type is incorrect (%u)", eType
                    );
            // not fatal error
            bResult = TRUE;
            break;
        }

        hLocation = DSRL_TARGET.hLocation(hTarget);
        if (hLocation == LOCATION_INVALID_OBJECT)
        {
            break;
        }

        hLocID = (TRAFFIC_LOCID_OBJECT)LOCATION.hLocID(hLocation);
        if (hLocID != TRAFFIC_LOCID_INVALID_OBJECT)
        {
            bResult = bProcessPredefinedLocation(hLocID, psIteratorArg);
        }
        else
        {
            bResult = bProcessCoordLocation(hLocation, psIteratorArg);
        }
    } while (FALSE);

    return bResult;
}

/*****************************************************************************
*
*   bGenerateDSRLFilterList
*
*****************************************************************************/
static BOOLEAN bGenerateDSRLFilterList(
    TRAFCAM_APP_OBJECT_STRUCT *psObj,
    TRAFCAM_DSRL_DESC_STRUCT *psDSRLDesc,
    OSAL_OBJECT_HDL hNewFilter
        )
{
    TRAFCAM_TARGET_LIST_ITERATOR_STRUCT sIterator;
    OSAL_RETURN_CODE_ENUM eReturnCode = OSAL_ERROR;

    OSAL.bMemSet(&sIterator, 0, sizeof(sIterator));

    sIterator.hEntriesList = hNewFilter;
    sIterator.psObj = psObj;

    eReturnCode = OSAL.eLinkedListIterate(psDSRLDesc->hTargetsList,
        (OSAL_LL_ITERATOR_HANDLER)bTargetFilterGenerator, &sIterator
            );
    if ((eReturnCode != OSAL_SUCCESS) && (eReturnCode != OSAL_NO_OBJECTS))
    {
        SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
            TRAFCAM_MGR_OBJECT_NAME": Error iterating targets list (%s)",
            OSAL.pacGetReturnCodeName(eReturnCode)
                );
        return FALSE;
    }

    return TRUE;
}

/*****************************************************************************
*
*   bDSRLGenerateFilter
*
*****************************************************************************/
static BOOLEAN bDSRLGenerateFilter(
    TRAFCAM_APP_OBJECT_STRUCT *psObj,
    DSRL_OBJECT hDSRL
        )
{
    BOOLEAN bResult = FALSE;
    TRAFCAM_DSRL_DESC_STRUCT *psDSRLDesc = NULL;

    psDSRLDesc = (TRAFCAM_DSRL_DESC_STRUCT *)DSRL_pvServiceData(hDSRL);
    if (psDSRLDesc != NULL)
    {
        bResult = bGenerateDSRLFilterList(psObj, psDSRLDesc, psObj->hNewFilter);
    }

    return bResult;
}

/*****************************************************************************
*
*   bDSRLUpdateContent
*
*****************************************************************************/
static BOOLEAN bDSRLUpdateContent(
    DSRL_OBJECT hDSRL,
    TRAFCAM_APP_OBJECT_STRUCT *psObj,
    BOOLEAN bUpdateFilter
        )
{
    BOOLEAN bResult = FALSE;
    OSAL_RETURN_CODE_ENUM eReturnCode = OSAL_ERROR;

    do
    {
        BOOLEAN bSuccess = FALSE;

        bSuccess = bDSRLFillDiffEntriesLists(hDSRL, psObj);
        if (bSuccess == FALSE)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                TRAFCAM_MGR_OBJECT_NAME": cannot fill dsrl entries diff lists"
                    );
            break;
        }

        bSuccess = bIsUpdateNeeded(psObj);
        if (bSuccess == FALSE)
        {
            //looks like we don't need to add or remove entries
            printf(TRAFCAM_MGR_OBJECT_NAME": DSRL %p update not required\n",
                hDSRL);
            bResult = TRUE;
            break;
        }

        bDSRLChangeState(hDSRL, DSRL_STATE_UPDATING);

        // adding entries from generated list
        bSuccess = bDSRLAddEntries(psObj, hDSRL);
        if (bSuccess == FALSE)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                TRAFCAM_MGR_OBJECT_NAME": cannot add entries to DSRL from"
                " list %p", psObj->hEntriesToAdd);
            break;
        }

        //removing entries from generated list
        bSuccess = bDSRLRemoveEntries(psObj, hDSRL);
        if (bSuccess == FALSE)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                TRAFCAM_MGR_OBJECT_NAME": cannot remove entries "
                "from DSRL from list %p", psObj->hEntriesToRemove);
            break;
        }

        bResult = TRUE;
    } while (FALSE);

    eReturnCode = OSAL.eLinkedListRemoveAll(psObj->hEntriesToAdd, NULL);
    if (eReturnCode != OSAL_SUCCESS)
    {
        SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
            TRAFCAM_MGR_OBJECT_NAME": cannot cleanup"
            " add entry list (%s)",
            OSAL.pacGetReturnCodeName(eReturnCode)
                );
    }

    eReturnCode = OSAL.eLinkedListRemoveAll(psObj->hEntriesToRemove, NULL);
    if (eReturnCode != OSAL_SUCCESS)
    {
        SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
            TRAFCAM_MGR_OBJECT_NAME": cannot cleanup"
            " remove entry list (%s)",
            OSAL.pacGetReturnCodeName(eReturnCode)
                );
    }

    if (bUpdateFilter == TRUE)
    {
        bResult = bDSRLGenerateFilter(psObj, hDSRL);
        if (bResult == FALSE)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                TRAFCAM_MGR_OBJECT_NAME": cannot update filter for DSRL %p",
                hDSRL);
        }
    }

    return bResult;
}

/*****************************************************************************
*
*   bHandleCreateList
*
*****************************************************************************/
static BOOLEAN bHandleCreateList(
    TRAFCAM_MGR_OBJECT_STRUCT *psObj,
    DSRL_ARG_STRUCT *psDSRLArg
        )
{
    BOOLEAN bResult = FALSE;
    TRAFCAM_APP_OBJECT_STRUCT *psAppObj = NULL;

    do
    {
        BOOLEAN bSuccess = FALSE;

        printf(TRAFCAM_MGR_OBJECT_NAME": Creating DSRL\n");

        psAppObj = psGetAppFacingObject((TRAFFIC_CAMERAS_SERVICE_OBJECT)psObj);
        if (psAppObj == NULL)
        {
            break;
        }

        // lets init DSRL attributes here
        bSuccess = bDSRLDescInit(psDSRLArg, psAppObj);
        if (bSuccess == FALSE)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                TRAFCAM_MGR_OBJECT_NAME": Failed to init DSRL desc for"
                " DSRL %p",
                psDSRLArg->hDSRL
                    );
            break;
        }

        bSuccess = bDSRLAddTargets(psDSRLArg);
        if (bSuccess == FALSE)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                TRAFCAM_MGR_OBJECT_NAME": failed to add targets to DSRL %p",
                psDSRLArg->hDSRL
                    );
            break;
        }

        // generating content for DSRL and filling it
        bSuccess = bDSRLUpdateContent(psDSRLArg->hDSRL, psAppObj, TRUE);
        if (bSuccess == FALSE)
        {
            break;
        }

        bDSRLChangeState(psDSRLArg->hDSRL, DSRL_STATE_READY);

        // applying newly generated DSRL filter to our global filter
        bSuccess = bUpdateGlobalFilter(psObj, psDSRLArg->hDSRL);
        if (bSuccess == FALSE)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                TRAFCAM_MGR_OBJECT_NAME": failed to update global filter"
                    );
            break;
        }

        // Looks good
        bResult = TRUE;
    } while (FALSE);

    vReleaseAppFacingObject(psAppObj);

    return bResult;
}

/*****************************************************************************
*
*   bDSRLReplaceTargets
*
*****************************************************************************/
static BOOLEAN bDSRLReplaceTargets(
    DSRL_ARG_STRUCT *psDSRLArg
        )
{
    BOOLEAN bResult = FALSE;
    OSAL_RETURN_CODE_ENUM eReturnCode = OSAL_ERROR;
    TRAFCAM_DSRL_DESC_STRUCT *psDSRLDesc = NULL;

    psDSRLDesc =
        (TRAFCAM_DSRL_DESC_STRUCT *)DSRL_pvServiceData(psDSRLArg->hDSRL);
    if (psDSRLDesc == NULL)
    {
        return FALSE;
    }

    // removing existing DSRL targets
    eReturnCode = OSAL.eLinkedListRemoveAll(psDSRLDesc->hTargetsList,
        (OSAL_LL_RELEASE_HANDLER)DSRL_TARGET_vDestroyByType
            );
    if (eReturnCode !=  OSAL_SUCCESS)
    {
        SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
            TRAFCAM_MGR_OBJECT_NAME": failed to release target desc"
            " items in DSRL desc (%s)",
            OSAL.pacGetReturnCodeName(eReturnCode)
                );
    }
    else
    {
        //Add new targets
        bResult = bDSRLAddTargets(psDSRLArg);
    }

    return bResult;
}

/*****************************************************************************
*
*   n16FindTargetEntry
*
*****************************************************************************/
static N16 n16FindTargetEntry(
    DSRL_TARGET_OBJECT hTarget1,
    DSRL_TARGET_OBJECT hTarget2
        )
{
    N16 n16Result = N16_MIN;

    if ((hTarget1 != DSRL_TARGET_INVALID_OBJECT) &&
        (hTarget2 != DSRL_TARGET_INVALID_OBJECT))
    {
        BOOLEAN bResult = FALSE;
        LOCATION_OBJECT hLocation1 = LOCATION_INVALID_OBJECT;
        LOCATION_OBJECT hLocation2 = LOCATION_INVALID_OBJECT;

        bResult = LOCATION_bCompare(hLocation1, hLocation2);

        n16Result = (bResult == TRUE) ? 0 : 1;
    }

    return n16Result;
}

/*****************************************************************************
*
*   vDSRLRemoveTarget
*
*****************************************************************************/
static void vDSRLRemoveTarget(
    DSRL_OBJECT hDSRL,
    DSRL_TARGET_OBJECT hTargetObj
        )
{
    do
    {
        OSAL_LINKED_LIST_ENTRY hTargetEntry = OSAL_INVALID_LINKED_LIST_ENTRY;
        OSAL_RETURN_CODE_ENUM eReturnCode = OSAL_ERROR;
        TRAFCAM_DSRL_DESC_STRUCT *psDSRLDesc = NULL;
        DSRL_TARGET_OBJECT hTargetToRemove = DSRL_TARGET_INVALID_OBJECT;

        if (hTargetObj == DSRL_TARGET_INVALID_OBJECT)
        {
            break;
        }

        psDSRLDesc = (TRAFCAM_DSRL_DESC_STRUCT *)DSRL_pvServiceData(hDSRL);
        if (psDSRLDesc == NULL)
        {
            break;
        }

        eReturnCode = OSAL.eLinkedListLinearSearch(
            psDSRLDesc->hTargetsList,
            &hTargetEntry,
            (OSAL_LL_COMPARE_HANDLER)n16FindTargetEntry,
            hTargetObj
                );

        if (eReturnCode != OSAL_SUCCESS)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                TRAFCAM_MGR_OBJECT_NAME": failed to find target entry (%s)",
                OSAL.pacGetReturnCodeName(eReturnCode)
                    );
            break;
        }

        hTargetToRemove =
            (DSRL_TARGET_OBJECT)OSAL.pvLinkedListThis(hTargetEntry);
        if (hTargetToRemove == DSRL_TARGET_INVALID_OBJECT)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                TRAFCAM_MGR_OBJECT_NAME": failed to get target handle"
                    );
            break;
        }

        eReturnCode = OSAL.eLinkedListRemove(hTargetEntry);
        if (eReturnCode != OSAL_SUCCESS)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                TRAFCAM_MGR_OBJECT_NAME": failed to remove target entry"
                " from list of DSRL targets (%s)",
                OSAL.pacGetReturnCodeName(eReturnCode)
                    );
        }

        DSRL_TARGET_vDestroyByType(hTargetToRemove);
    } while (FALSE);

    return;
}

/*****************************************************************************
*
*   bDSRLRemoveTargets
*
*****************************************************************************/
static BOOLEAN bDSRLRemoveTargets(
    DSRL_ARG_STRUCT *psDSRLArg
        )
{
    size_t tDSRLTargetIndex;

    for (tDSRLTargetIndex = 0;
         tDSRLTargetIndex < psDSRLArg->tNumTargets;
         tDSRLTargetIndex++)
    {
        vDSRLRemoveTarget(
            psDSRLArg->hDSRL,
            psDSRLArg->ahTargetList[tDSRLTargetIndex]
                );
    }

    return TRUE;
}

/*****************************************************************************
*
*   bDSRLRemoveAllTargets
*
*****************************************************************************/
static BOOLEAN bDSRLRemoveAllTargets(
    DSRL_ARG_STRUCT *psDSRLArg
        )
{
    BOOLEAN bResult = TRUE;
    OSAL_RETURN_CODE_ENUM eReturnCode = OSAL_ERROR;
    TRAFCAM_DSRL_DESC_STRUCT *psDSRLDesc = NULL;

    psDSRLDesc =
        (TRAFCAM_DSRL_DESC_STRUCT *)DSRL_pvServiceData(psDSRLArg->hDSRL);
    if (psDSRLDesc == NULL)
    {
        bResult = FALSE;
    }
    else
    {
        eReturnCode = OSAL.eLinkedListRemoveAll(psDSRLDesc->hTargetsList,
            (OSAL_LL_RELEASE_HANDLER)DSRL_TARGET_vDestroyByType
                );
        if (eReturnCode != OSAL_SUCCESS)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                TRAFCAM_MGR_OBJECT_NAME": failed to remove DSRL targets"
                " (%s)", OSAL.pacGetReturnCodeName(eReturnCode)
                    );
            bResult = FALSE;
        }
    }

    return bResult;
}

/*****************************************************************************
*
*   bHandleModifyList
*
*****************************************************************************/
static BOOLEAN bHandleModifyList(
    TRAFCAM_MGR_OBJECT_STRUCT *psObj,
    DSRL_ARG_STRUCT *psDSRLArg
        )
{
    BOOLEAN bResult = FALSE;
    TRAFCAM_APP_OBJECT_STRUCT *psAppObj = NULL;

    do
    {
        BOOLEAN bSuccess = FALSE;

        psAppObj = psGetAppFacingObject((TRAFFIC_CAMERAS_SERVICE_OBJECT)psObj);
        if (psAppObj == NULL)
        {
            break;
        }
        // This operation tells us explicitly if
        // we should do this or not
        if (psDSRLArg->uAction.sModify.bForceDSRLStateChange == TRUE)
        {
            bDSRLChangeState(psDSRLArg->hDSRL, DSRL_STATE_UPDATING);
        }

        printf(TRAFCAM_MGR_OBJECT_NAME" :DSRL %p modify: modify type is %d\n",
            psDSRLArg->hDSRL, psDSRLArg->uAction.sModify.eModifyType
                );

        // Modify the DSRL as the user specified
        switch (psDSRLArg->uAction.sModify.eModifyType)
        {
            case DSRL_MODIFY_OPERATION_ADD:
            {
                bSuccess = bDSRLAddTargets(psDSRLArg);
            }
            break;

            case DSRL_MODIFY_OPERATION_REPLACE:
            {
                bSuccess = bDSRLReplaceTargets(psDSRLArg);
            }
            break;

            case DSRL_MODIFY_OPERATION_REMOVE:
            {
                bSuccess = bDSRLRemoveTargets(psDSRLArg);
            }
            break;

            case DSRL_MODIFY_OPERATION_REMOVEALL:
            {
                bSuccess = bDSRLRemoveAllTargets(psDSRLArg);
            }
            break;

            case DSRL_MODIFY_OPERATION_INVALID:
            default:
                break;
        }

        if (bSuccess == FALSE)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                TRAFCAM_MGR_OBJECT_NAME": DSRL modify targets operation failed"
                    );
            break;
        }

        // updating DSRL content and generating the filter according to changes
        bSuccess = bDSRLUpdateContent(psDSRLArg->hDSRL, psAppObj, TRUE);
        if (bSuccess == FALSE)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                TRAFCAM_MGR_OBJECT_NAME": DSRL modify entries operation failed"
                    );
            break;
        }

        bDSRLChangeState(psDSRLArg->hDSRL, DSRL_STATE_READY);

        // applying changed DSRL filter to global filter
        bSuccess = bUpdateGlobalFilter(psObj, psDSRLArg->hDSRL);
        if (bSuccess == FALSE)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                TRAFCAM_MGR_OBJECT_NAME": failed to update global filter"
                    );
            break;
        }

        bResult = TRUE;
    } while (FALSE);

    if (bResult == FALSE)
    {
        bDSRLChangeState(psDSRLArg->hDSRL, DSRL_STATE_ERROR);
    }

    vReleaseAppFacingObject(psAppObj);

    return bResult;
}

/*****************************************************************************
*
*   bHandleRefreshList
*
*****************************************************************************/
static BOOLEAN bHandleRefreshList(
    TRAFCAM_MGR_OBJECT_STRUCT *psObj,
    DSRL_ARG_STRUCT *psDSRLArg
        )
{
    BOOLEAN bResult = FALSE;
    TRAFCAM_APP_OBJECT_STRUCT *psAppObj = NULL;

    do
    {
        BOOLEAN bSuccess = FALSE;

        psAppObj = psGetAppFacingObject((TRAFFIC_CAMERAS_SERVICE_OBJECT)psObj);
        if (psAppObj == NULL)
        {
            break;
        }

        // reading the entries without generating the filter
        bSuccess = bDSRLUpdateContent(psDSRLArg->hDSRL, psAppObj, FALSE);
        if (bSuccess == FALSE)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                TRAFCAM_MGR_OBJECT_NAME": failed to update DSRL content"
                    );
            break;
        }

        bDSRLChangeState(psDSRLArg->hDSRL, DSRL_STATE_READY);

        bResult = TRUE;
    } while (FALSE);

    if (bResult == FALSE)
    {
        bDSRLChangeState(psDSRLArg->hDSRL, DSRL_STATE_ERROR);
    }

    vReleaseAppFacingObject(psAppObj);

    return bResult;
}

/*****************************************************************************
*
*   bFilterRemoveDSRL
*
*****************************************************************************/
static BOOLEAN bFilterRemoveDSRL(
    TRAFCAM_BSA_FILTER_STRUCT *psFilterItem,
    TRAFCAM_FILTER_ITERATOR_STRUCT *psIteratorArg
        )
{
    BOOLEAN bContinue = FALSE;

    do
    {
        OSAL_RETURN_CODE_ENUM eReturnCode = OSAL_ERROR;
        OSAL_LINKED_LIST_ENTRY hEntry = OSAL_INVALID_LINKED_LIST_ENTRY;
        UN32 un32Items = 0;

        if ((psFilterItem == NULL) || (psIteratorArg == NULL))
        {
            break;
        }

        eReturnCode = OSAL.eLinkedListSearch(psFilterItem->hDSRLList, &hEntry,
            psIteratorArg->hDSRL);
        if (eReturnCode == OSAL_OBJECT_NOT_FOUND)
        {
            bContinue = TRUE;
            break;
        }

        if (eReturnCode != OSAL_SUCCESS)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                TRAFCAM_MGR_OBJECT_NAME": failed to search in list (%s)",
                OSAL.pacGetReturnCodeName(eReturnCode)
                    );
            break;
        }

        eReturnCode = OSAL.eLinkedListRemove(hEntry);
        if (eReturnCode != OSAL_SUCCESS)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                TRAFCAM_MGR_OBJECT_NAME": failed to remove DSRL from"
                " filter (%s)", OSAL.pacGetReturnCodeName(eReturnCode)
                    );
            break;
        }

        eReturnCode = OSAL.eLinkedListItems(psFilterItem->hDSRLList,
            &un32Items);
        if (eReturnCode != OSAL_SUCCESS)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                TRAFCAM_MGR_OBJECT_NAME": failed to get number of items (%s)",
                OSAL.pacGetReturnCodeName(eReturnCode)
                    );
            break;
        }

        if (un32Items > 0)
        {
            // this filter item still references to other DSRLs and
            // cannot be deleted
            bContinue = TRUE;
            break;
        }

        eReturnCode = OSAL.eLinkedListRemove(psFilterItem->hEntry);
        if (eReturnCode != OSAL_SUCCESS)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                TRAFCAM_MGR_OBJECT_NAME": failed to remove filter item from"
                " list (%s)", OSAL.pacGetReturnCodeName(eReturnCode)
                    );
            break;
        }

        OSAL.vLinkedListMemoryFree(psFilterItem);
        psIteratorArg->bChanged = TRUE;

        bContinue = TRUE;
    } while (FALSE);

    return bContinue;
}

/*****************************************************************************
*
*   bDSRLRemoveEntryLink
*
*****************************************************************************/
static BOOLEAN bDSRLRemoveEntryLink(
    DSRL_OBJECT hDSRL,
    DSRL_ENTRY_OBJECT hDSRLEntry
        )
{
    BOOLEAN bResult = FALSE;

    if ((hDSRL != DSRL_INVALID_OBJECT) &&
        (hDSRLEntry != DSRL_ENTRY_INVALID_OBJECT))
    {
        OSAL_RETURN_CODE_ENUM eReturnCode = OSAL_ERROR;
        OSAL_LINKED_LIST_ENTRY hEntry = OSAL_INVALID_LINKED_LIST_ENTRY;
        TRAFCAM_DSRL_DESC_STRUCT *psDSRLDesc = NULL;

        psDSRLDesc = (TRAFCAM_DSRL_DESC_STRUCT *)DSRL_pvServiceData(hDSRL);
        if (psDSRLDesc != NULL)
        {
            eReturnCode = OSAL.eLinkedListSearch(psDSRLDesc->hEntriesList,
                &hEntry, hDSRLEntry);
            if (eReturnCode == OSAL_SUCCESS)
            {
                eReturnCode = OSAL.eLinkedListRemove(hEntry);
                if (eReturnCode != OSAL_SUCCESS)
                {
                    SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                        TRAFCAM_MGR_OBJECT_NAME": failed to remove entry"
                        " from list (%s)",
                        OSAL.pacGetReturnCodeName(eReturnCode)
                            );
                }
            }
            else
            {
                SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    TRAFCAM_MGR_OBJECT_NAME": failed to find entry"
                    " in list (%s)",
                    OSAL.pacGetReturnCodeName(eReturnCode)
                        );
            }
        }

        bResult = TRUE;
    }

    return bResult;
}

/*****************************************************************************
*
*   vDSRLFinalizeEntry
*
*****************************************************************************/
static void vDSRLFinalizeEntry(
    DSRL_OBJECT hDSRL,
    DSRL_ENTRY_OBJECT hDSRLEntry,
    TRAFCAM_APP_OBJECT_STRUCT *psObj
        )
{
    if ((hDSRL != DSRL_INVALID_OBJECT) &&
        (hDSRLEntry != DSRL_ENTRY_INVALID_OBJECT))
    {
        BOOLEAN bSuccess = FALSE;

        bSuccess = bEntryRemoveDSRLLink(hDSRLEntry, hDSRL);
        if (bSuccess == FALSE)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                TRAFCAM_MGR_OBJECT_NAME": failed to remove entry"
                " to DSRL link"
                    );
        }

        bSuccess = bDSRLRemoveEntryLink(hDSRL, hDSRLEntry);
        if (bSuccess == FALSE)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                TRAFCAM_MGR_OBJECT_NAME": failed to remove DSRL"
                " to entry link"
                    );
        }

    }
    else
    {
        SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
            TRAFCAM_MGR_OBJECT_NAME": failed to finalize entry: hDSRL %p,"
            " hDSRLEntry %p, callback arg %p", hDSRL, hDSRLEntry, psObj
                );
    }

    return;
}

/*****************************************************************************
*
*   bEntryRemoveDSRLLink
*
*****************************************************************************/
BOOLEAN bEntryRemoveDSRLLink(
    DSRL_ENTRY_OBJECT hDSRLEntry,
    DSRL_OBJECT hDSRL
        )
{
    BOOLEAN bResult = FALSE;
    OSAL_RETURN_CODE_ENUM eReturnCode = OSAL_ERROR;
    OSAL_LINKED_LIST_ENTRY hEntry = OSAL_INVALID_LINKED_LIST_ENTRY;
    TRAFCAM_DSRL_ENTRY_DESC_STRUCT *psEntryDesc = NULL;

    psEntryDesc = (TRAFCAM_DSRL_ENTRY_DESC_STRUCT *)
        DSRL_ENTRY_pvServiceData(hDSRLEntry);

    if ((psEntryDesc != NULL) &&
        (psEntryDesc->hDSRLList != OSAL_INVALID_OBJECT_HDL))
    {
        eReturnCode = OSAL.eLinkedListSearch(psEntryDesc->hDSRLList,
            &hEntry, hDSRL);
        if (eReturnCode == OSAL_SUCCESS)
        {
            eReturnCode = OSAL.eLinkedListRemove(hEntry);
            if (eReturnCode != OSAL_SUCCESS)
            {
                SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    TRAFCAM_MGR_OBJECT_NAME": failed to remove DSRL desc"
                    " from list (%s)",
                    OSAL.pacGetReturnCodeName(eReturnCode)
                        );
            }
        }
        else
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                TRAFCAM_MGR_OBJECT_NAME": failed to find DSRL desc"
                " in list (%s)",
                OSAL.pacGetReturnCodeName(eReturnCode)
                    );
        }

        bResult = TRUE;
    }

    return bResult;
}

/*****************************************************************************
*
*   bRemoveFromGlobalFilter
*
*****************************************************************************/
static BOOLEAN bRemoveFromGlobalFilter(
    TRAFCAM_MGR_OBJECT_STRUCT *psObj,
    DSRL_OBJECT hDSRL,
    BOOLEAN *pbChanged
        )
{
    OSAL_RETURN_CODE_ENUM eReturnCode = OSAL_ERROR;
    TRAFCAM_FILTER_ITERATOR_STRUCT sIterator;

    OSAL.bMemSet(&sIterator, 0, sizeof(sIterator));
    sIterator.hDSRL = hDSRL;

    // removing all the particular DSRL filter items from global filter
    eReturnCode = OSAL.eLinkedListIterate(psObj->hFilter,
        (OSAL_LL_ITERATOR_HANDLER)bFilterRemoveDSRL,
        &sIterator);
    if ((eReturnCode != OSAL_SUCCESS) && (eReturnCode != OSAL_NO_OBJECTS))
    {
        SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
            TRAFCAM_MGR_OBJECT_NAME": failed to iterate filter"
            " list (%s)", OSAL.pacGetReturnCodeName(eReturnCode)
                );
        return FALSE;
    }

    *pbChanged = sIterator.bChanged;
    return TRUE;
}

/*****************************************************************************
*
*   vHandleDeleteList
*
*****************************************************************************/
static void vHandleDeleteList(
    TRAFCAM_MGR_OBJECT_STRUCT *psObj,
    DSRL_ARG_STRUCT *psDSRLArg
        )
{
    TRAFCAM_APP_OBJECT_STRUCT *psAppObj = NULL;
    DSRL_OBJECT hDSRL = DSRL_INVALID_OBJECT;

    do
    {
        TRAFCAM_DSRL_DESC_STRUCT *psDSRLDesc = NULL;
        OSAL_RETURN_CODE_ENUM eReturnCode = OSAL_ERROR;
        psAppObj = psGetAppFacingObject((TRAFFIC_CAMERAS_SERVICE_OBJECT)psObj);
        if (psAppObj == NULL)
        {
            break;
        }

        printf(TRAFCAM_MGR_OBJECT_NAME": Removing DSRL %p\n",
            psDSRLArg->hDSRL
                );

        // Remove / free all entries in the DSRL
        DSRL_vRemoveAllEntries(psDSRLArg->hDSRL);

        psDSRLDesc = (TRAFCAM_DSRL_DESC_STRUCT *)DSRL_pvServiceData(
            psDSRLArg->hDSRL
                );
        if (psDSRLDesc == NULL)
        {
            break;
        }

        eReturnCode = OSAL.eLinkedListRemove(psDSRLDesc->hDSRLListEntry);
        if (eReturnCode != OSAL_SUCCESS)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                TRAFCAM_MGR_OBJECT_NAME": cannot remove DSRL from pool (%s)",
                OSAL.pacGetReturnCodeName(eReturnCode)
                    );
        }

        vDSRLRelease(psDSRLArg->hDSRL);
        hDSRL = psDSRLArg->hDSRL;
        psDSRLArg->hDSRL = DSRL_INVALID_OBJECT;

    } while (FALSE);

    vReleaseAppFacingObject(psAppObj);

    if (hDSRL != DSRL_INVALID_OBJECT)
    {
        BOOLEAN bChanged = FALSE, bSuccess = FALSE;

        bSuccess = bRemoveFromGlobalFilter(psObj,
            hDSRL, &bChanged);
        if (bSuccess == FALSE)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                TRAFCAM_MGR_OBJECT_NAME": failed to update global filter");
        }
        else if (bChanged == TRUE)
        {
            bSuccess = GsTrafcamIntf.bSetFilter(psObj->hInterface,
                psObj->hFilter);
            if (bSuccess == FALSE)
            {
                SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    TRAFCAM_MGR_OBJECT_NAME": failed update protocol filter");
            }
        }
    }

    return;
}

/*****************************************************************************
*
*   vReleaseGlobalFilterItem
*
*****************************************************************************/
static void vReleaseGlobalFilterItem (
    TRAFCAM_BSA_FILTER_STRUCT *psFilterItem
        )
{
    if (psFilterItem != NULL)
    {
        if (psFilterItem->hDSRLList != OSAL_INVALID_OBJECT_HDL)
        {
            OSAL_RETURN_CODE_ENUM eReturnCode = OSAL_ERROR;

            eReturnCode = OSAL.eLinkedListRemoveAll(psFilterItem->hDSRLList,
                NULL);
            if (eReturnCode != OSAL_SUCCESS)
            {
                SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    TRAFCAM_MGR_OBJECT_NAME": failed to remove filter DSRLList"
                    " entries (%s)", OSAL.pacGetReturnCodeName(eReturnCode)
                        );
            }

            eReturnCode = OSAL.eLinkedListDelete(psFilterItem->hDSRLList);
            if (eReturnCode != OSAL_SUCCESS)
            {
                SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    TRAFCAM_MGR_OBJECT_NAME": failed to delete list (%s)",
                    OSAL.pacGetReturnCodeName(eReturnCode)
                        );
            }
        }

        OSAL.vLinkedListMemoryFree(psFilterItem);
    }
}

/*****************************************************************************
*
*   bIterateToRemoveFromGlobal
*
*****************************************************************************/
static BOOLEAN bIterateToRemoveFromGlobal(
    TRAFCAM_BSA_FILTER_STRUCT *psFilterItem,
    TRAFCAM_FILTER_ITERATOR_STRUCT *psIteratorArg
        )
{
    BOOLEAN bContinue = FALSE;

    do
    {
        OSAL_RETURN_CODE_ENUM eReturnCode = OSAL_ERROR;
        OSAL_LINKED_LIST_ENTRY hEntry = OSAL_INVALID_LINKED_LIST_ENTRY;
        UN32 un32Items = 0;

        if ((psFilterItem == NULL) || (psIteratorArg == NULL))
        {
            break;
        }

        eReturnCode = OSAL.eLinkedListSearch(psFilterItem->hDSRLList,
            &hEntry, psIteratorArg->hDSRL);
        if (eReturnCode == OSAL_OBJECT_NOT_FOUND)
        {
            bContinue = TRUE;
            break;
        }
        else if (eReturnCode != OSAL_SUCCESS)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                TRAFCAM_MGR_OBJECT_NAME": failed to search in list (%s)",
                OSAL.pacGetReturnCodeName(eReturnCode)
                    );
            break;
        }

        eReturnCode = OSAL.eLinkedListSearch(psIteratorArg->hFilter,
            &psIteratorArg->hEntry, psFilterItem);
        if (eReturnCode == OSAL_SUCCESS)
        {
            bContinue = TRUE;
            break;
        }
        else if (eReturnCode != OSAL_OBJECT_NOT_FOUND)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                TRAFCAM_MGR_OBJECT_NAME": failed to search in list (%s)",
                OSAL.pacGetReturnCodeName(eReturnCode)
                    );
            break;
        }

        // ok, if we here then we need to remove DSRL LL entry from DSRLs LL
        eReturnCode = OSAL.eLinkedListRemove(hEntry);
        if (eReturnCode != OSAL_SUCCESS)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                TRAFCAM_MGR_OBJECT_NAME": failed to remove item from list (%s)",
                OSAL.pacGetReturnCodeName(eReturnCode)
                    );
            break;
        }

        // now lets check number of DSRLs in list
        eReturnCode = OSAL.eLinkedListItems(psFilterItem->hDSRLList,
            &un32Items);
        if (eReturnCode != OSAL_SUCCESS)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                TRAFCAM_MGR_OBJECT_NAME": failed to get number of items (%s)",
                OSAL.pacGetReturnCodeName(eReturnCode)
                    );
            break;
        }

        if (un32Items == 0)
        {
            // lets remove the item
            eReturnCode = OSAL.eLinkedListRemove(psFilterItem->hEntry);
            if (eReturnCode != OSAL_SUCCESS)
            {
                SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    TRAFCAM_MGR_OBJECT_NAME": failed to remove item"
                    " from list (%s)", OSAL.pacGetReturnCodeName(eReturnCode)
                        );
                break;
            }

            vReleaseGlobalFilterItem(psFilterItem);
            psIteratorArg->bChanged = TRUE;
        }

        bContinue = TRUE;
    } while (FALSE);

    return bContinue;
}

/*****************************************************************************
*
*   bIterateToAddToGlobal
*
*****************************************************************************/
static BOOLEAN bIterateToAddToGlobal(
    TRAFCAM_BSA_FILTER_STRUCT *psFilterItem,
    TRAFCAM_FILTER_ITERATOR_STRUCT *psIteratorArg
        )
{
    BOOLEAN bContinue = FALSE;

    do
    {
        OSAL_RETURN_CODE_ENUM eReturnCode = OSAL_ERROR;

        if ((psFilterItem == NULL) || (psIteratorArg == NULL))
        {
            break;
        }

        // removing the item from our linked list
        eReturnCode = OSAL.eLinkedListRemove(psFilterItem->hEntry);
        if (eReturnCode != OSAL_SUCCESS)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                TRAFCAM_MGR_OBJECT_NAME": failed to remove filter item (%s)",
                OSAL.pacGetReturnCodeName(eReturnCode)
                    );
            break;
        }

        psFilterItem->hEntry = OSAL_INVALID_LINKED_LIST_ENTRY;

        // looking for our filter item in global filter
        eReturnCode = OSAL.eLinkedListSearch(psIteratorArg->hFilter,
            &psIteratorArg->hEntry, psFilterItem);
        if (eReturnCode == OSAL_SUCCESS)
        {
            TRAFCAM_BSA_FILTER_STRUCT *psGlobalFilterItem = NULL;

            psGlobalFilterItem =
                (TRAFCAM_BSA_FILTER_STRUCT *)OSAL.pvLinkedListThis(
                psIteratorArg->hEntry);
            eReturnCode = OSAL.eLinkedListAdd(psGlobalFilterItem->hDSRLList,
                OSAL_INVALID_LINKED_LIST_ENTRY_PTR, psIteratorArg->hDSRL);
            if ((eReturnCode != OSAL_SUCCESS) &&
                (eReturnCode != OSAL_ERROR_LIST_ITEM_NOT_UNIQUE))
            {
                SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    TRAFCAM_MGR_OBJECT_NAME": failed to add to list (%s)",
                    OSAL.pacGetReturnCodeName(eReturnCode)
                        );
                break;
            }

            OSAL.vLinkedListMemoryFree(psFilterItem);
            bContinue = TRUE;
            break;
        }
        else if (eReturnCode != OSAL_OBJECT_NOT_FOUND)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                TRAFCAM_MGR_OBJECT_NAME": failed to search in list (%s)",
                OSAL.pacGetReturnCodeName(eReturnCode)
                    );
            OSAL.vLinkedListMemoryFree(psFilterItem);
            break;
        }

        //and now we add this item to global filter
        eReturnCode = OSAL.eLinkedListAdd(psIteratorArg->hFilter,
            &psFilterItem->hEntry, psFilterItem);
        if (eReturnCode != OSAL_SUCCESS)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                TRAFCAM_MGR_OBJECT_NAME": failed to add item to list (%s)",
                OSAL.pacGetReturnCodeName(eReturnCode)
                    );
            vReleaseGlobalFilterItem(psFilterItem);
            break;
        }

        eReturnCode = OSAL.eLinkedListCreate(&psFilterItem->hDSRLList,
            TRAFCAM_MGR_OBJECT_NAME":DSRLList", n16CompareHandles,
            OSAL_LL_OPTION_UNIQUE);
        if (eReturnCode != OSAL_SUCCESS)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                TRAFCAM_MGR_OBJECT_NAME": failed to create list (%s)",
                OSAL.pacGetReturnCodeName(eReturnCode)
                    );

            if (psFilterItem->hEntry != OSAL_INVALID_LINKED_LIST_ENTRY)
            {
                eReturnCode = OSAL.eLinkedListRemove(psFilterItem->hEntry);
                if (eReturnCode != OSAL_SUCCESS)
                {
                    SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                        TRAFCAM_MGR_OBJECT_NAME": failed to remove from"
                        " list (%s)", OSAL.pacGetReturnCodeName(eReturnCode)
                            );
                    break;
                }

                psFilterItem->hEntry = OSAL_INVALID_LINKED_LIST_ENTRY;
            }

            vReleaseGlobalFilterItem(psFilterItem);
            break;
        }

        // need to inform the caller that global filter has been changed
        psIteratorArg->bChanged = TRUE;

        // adding DSRL handle to list of DSRLs related to this filter item
        eReturnCode = OSAL.eLinkedListAdd(psFilterItem->hDSRLList,
            OSAL_INVALID_LINKED_LIST_ENTRY_PTR, psIteratorArg->hDSRL);
        if ((eReturnCode != OSAL_SUCCESS) &&
            (eReturnCode != OSAL_ERROR_LIST_ITEM_NOT_UNIQUE))
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                TRAFCAM_MGR_OBJECT_NAME": failed to add to list (%s)",
                OSAL.pacGetReturnCodeName(eReturnCode)
                    );
            break;
        }

        bContinue = TRUE;
    } while (FALSE);

    return bContinue;
}

/*****************************************************************************
*
*   bUpdateGlobalFilter
*
*****************************************************************************/
static BOOLEAN bUpdateGlobalFilter (
    TRAFCAM_MGR_OBJECT_STRUCT *psObj,
    DSRL_OBJECT hDSRL
        )
{
    BOOLEAN bResult = FALSE;

    do
    {
        OSAL_RETURN_CODE_ENUM eReturnCode = OSAL_ERROR;
        TRAFCAM_FILTER_ITERATOR_STRUCT sIterator;

        OSAL.bMemSet(&sIterator, 0, sizeof(sIterator));

        sIterator.hDSRL = hDSRL;
        sIterator.hFilter = psObj->psAppObj->hNewFilter;

        eReturnCode = OSAL.eLinkedListIterate(psObj->hFilter,
            (OSAL_LL_ITERATOR_HANDLER)bIterateToRemoveFromGlobal, &sIterator);
        if ((eReturnCode != OSAL_NO_OBJECTS) && (eReturnCode != OSAL_SUCCESS))
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                TRAFCAM_MGR_OBJECT_NAME": failed to update global filter (%s)",
                OSAL.pacGetReturnCodeName(eReturnCode)
                    );
            break;
        }

        sIterator.hEntry = OSAL_INVALID_LINKED_LIST_ENTRY;
        sIterator.hFilter = psObj->hFilter;

        eReturnCode = OSAL.eLinkedListIterate(psObj->psAppObj->hNewFilter,
            (OSAL_LL_ITERATOR_HANDLER)bIterateToAddToGlobal, &sIterator);
        if ((eReturnCode != OSAL_NO_OBJECTS) && (eReturnCode != OSAL_SUCCESS))
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                TRAFCAM_MGR_OBJECT_NAME": failed to update global filter (%s)",
                OSAL.pacGetReturnCodeName(eReturnCode)
                    );
            break;
        }

        if (sIterator.bChanged == TRUE)
        {
            bResult =
                GsTrafcamIntf.bSetFilter(psObj->hInterface, psObj->hFilter);
        }
        else
        {
            bResult = TRUE;
        }
    } while (FALSE);

    return bResult;
}

/*****************************************************************************
*
*   bCheckForTimeout
*
*****************************************************************************/
static BOOLEAN bCheckForTimeout(
    TRAFFIC_MSG_OBJECT hTrafficMsg,
    TRAFCAM_MESSAGE_TIMEOUT_ITERATOR *psIterator
        )
{
    BOOLEAN bContinue = FALSE;

    if ((hTrafficMsg != TRAFFIC_MSG_INVALID_OBJECT) && (psIterator != NULL))
    {
        UN32 un32TimeToExpire = 0;

        un32TimeToExpire = TRAFFIC_MSG_IMAGE_un32TimeToExpire(hTrafficMsg);
        if (psIterator->un32TimeStamp > un32TimeToExpire)
        {
            TRAFCAM_IMAGE_HASH tHash = TRAFCAM_IMAGE_INVALID_HASH;

            // removing the traffic msg as expired
            bEntryRemoveFromDSRLs((DSRL_ENTRY_OBJECT)hTrafficMsg);

            // reporting expired hash to trafcam interface
            tHash = TRAFFIC_MSG_IMAGE_tHash(hTrafficMsg);
            GsTrafcamIntf.bReportExpiredHash(psIterator->hInterface, tHash);

            // removing the entry from the pool and releasing memory
            vEntryRemove((DSRL_ENTRY_OBJECT)hTrafficMsg);
            psIterator->bChanged = TRUE;
        }

        bContinue = TRUE;
    }

    return bContinue;
}

/*****************************************************************************
*
*   vProcessTimeout
*
*****************************************************************************/
static void vProcessTimeout(
    TRAFCAM_MGR_OBJECT_STRUCT *psObj
        )
{
    TRAFCAM_APP_OBJECT_STRUCT *psAppObj = NULL;

    psAppObj = psGetAppFacingObject((TRAFFIC_CAMERAS_SERVICE_OBJECT)psObj);
    if (psAppObj != NULL)
    {
        OSAL_RETURN_CODE_ENUM eReturnCode = OSAL_ERROR;
        TRAFCAM_MESSAGE_TIMEOUT_ITERATOR sIterator;

        OSAL.vTimeUp(&sIterator.un32TimeStamp, NULL);
        sIterator.hInterface = psObj->hInterface;

        eReturnCode = OSAL.eLinkedListIterate(
            psAppObj->hEntriesList, (OSAL_LL_ITERATOR_HANDLER)bCheckForTimeout,
            &sIterator);
        if (eReturnCode == OSAL_SUCCESS)
        {
            UN32 un32Items = 0;

            if (sIterator.bChanged == TRUE)
            {
                eReturnCode = OSAL.eLinkedListIterate(psAppObj->hDSRLList,
                    (OSAL_LL_ITERATOR_HANDLER)bDSRLChangeState,
                    (void *)(size_t)DSRL_STATE_READY);
                if ((eReturnCode != OSAL_SUCCESS) &&
                    (eReturnCode != OSAL_NO_OBJECTS))
                {
                    SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                        TRAFCAM_MGR_OBJECT_NAME": failed to iterate DSRLs (%s)",
                        OSAL.pacGetReturnCodeName(eReturnCode));
                }
            }

            eReturnCode = OSAL.eLinkedListItems(psAppObj->hEntriesList,
                &un32Items);
            if (eReturnCode != OSAL_SUCCESS)
            {
                SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    TRAFCAM_MGR_OBJECT_NAME": failed to get number of entries"
                    " (%s)", OSAL.pacGetReturnCodeName(eReturnCode));
            }
            else if (un32Items > 0)
            {
                vStartAgeoutTimer(psObj);
            }

        }
        else if (eReturnCode != OSAL_NO_OBJECTS)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                TRAFCAM_MGR_OBJECT_NAME": failed to iterate entries (%s)",
                OSAL.pacGetReturnCodeName(eReturnCode)
                    );
        }

        vReleaseAppFacingObject(psAppObj);
    }

    return;
}

/*****************************************************************************
*
*   vEventHandler
*
*   This function runs in the context of an SMS resource which has been
*   assigned to this service.
*
*****************************************************************************/
static void vEventHandler (
    DATASERVICE_MGR_OBJECT hDataService,
    DATASERVICE_EVENT_MASK tCurrentEvent,
    void *pvEventArg,
    void *pvEventCallbackArg
        )
{
    TRAFCAM_MGR_OBJECT_STRUCT *psObj = NULL;
    BOOLEAN bValid = FALSE, bStopEvent = FALSE;
    SMSAPI_EVENT_MASK tEventMask = DATASERVICE_EVENT_NONE;

    // Get our traffic cameras mgr handle from the callback argument
    psObj = (TRAFCAM_MGR_OBJECT_STRUCT *)pvEventCallbackArg;

    // Is this object valid?
    bValid = DATASERVICE_IMPL_bValid((DATASERVICE_IMPL_HDL)psObj);

    // Only handle events for valid objects...
    if (bValid == FALSE)
    {
        return;
    }

    switch (tCurrentEvent)
    {
        // Handle Traffic Cameras Service events here...

        // State has changed
        case DATASERVICE_EVENT_STATE:
        {
            BOOLEAN bStateChanged;
            DATASERVICE_STATE_CHANGE_STRUCT const *psStateChange =
                (DATASERVICE_STATE_CHANGE_STRUCT const *)pvEventArg;

            // Process the state transition
            bStateChanged = DATASERVICE_IMPL_bStateFSM(
                (DATASERVICE_IMPL_HDL)psObj,
                psStateChange,
                &GsTrafcamStateHandlers,
                (void *)psObj);

            if (bStateChanged == TRUE)
            {
                // The state has been updated
                tEventMask |= DATASERVICE_EVENT_STATE;
                // Is the service stopped now?
                if (psStateChange->eCurrentState ==
                    DATASERVICE_STATE_STOPPED)
                {
                    bStopEvent = TRUE;
                }
            }
        }
        break;

        // This service has a message to process
        case DATASERVICE_EVENT_NEW_DATA:
        {
            OSAL_BUFFER_HDL hPayload = (OSAL_BUFFER_HDL)pvEventArg;
            BOOLEAN bSuccess = TRUE;

            // Ensure the payload handle is valid
            // If it isn't, there's a problem with
            // SMS
            if (hPayload == OSAL_INVALID_BUFFER_HDL)
            {
                // Set the error
                vSetError(psObj,
                    DATASERVICE_ERROR_CODE_GENERAL
                        );
            }
            // Only process this message if we are not stopping
            else
            {
                printf(TRAFCAM_MGR_OBJECT_NAME": payload received (%u)\n",
                    OSAL.tBufferGetSize(hPayload)
                        );

                // Process this message now
                bSuccess = bProcessMessage(psObj, hPayload);

                if (bSuccess == TRUE)
                {
                    puts(TRAFCAM_MGR_OBJECT_NAME": Message Payload "
                        " Processed Ok");
                }
                else
                {
                    SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                        TRAFCAM_MGR_OBJECT_NAME": failed to process payload"
                            );
                    DATASERVICE_IMPL_vLog(TRAFCAM_MGR_OBJECT_NAME
                        ": failed to process payload\n"
                            );
                }
            }
        }
        break;

        // We need to do some work with our DSRLs
        case DATASERVICE_INTERNAL_EVENT_DSRL:
        {
            BOOLEAN bSuccess = TRUE;
            DSRL_ARG_STRUCT *psDSRLArg =
                (DSRL_ARG_STRUCT *)pvEventArg;

            if ((psDSRLArg == NULL) ||
                (psDSRLArg->hDSRL == DSRL_INVALID_OBJECT))
            {
                SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    TRAFCAM_MGR_OBJECT_NAME": DSRL_EVENT has no data or DSRL"
                        );
                vSetError(psObj, DATASERVICE_ERROR_CODE_GENERAL);
                break;
            }

            switch (psDSRLArg->eAction)
            {
                case DSRL_ACTION_ADD:
                {
                    // Create a new list
                    bSuccess = bHandleCreateList(
                        psObj, psDSRLArg);
                }
                break;

                case DSRL_ACTION_MODIFY:
                {
                    // Modify a pre-existing list
                    bSuccess = bHandleModifyList(
                        psObj, psDSRLArg);
                }
                break;

                case DSRL_ACTION_REFRESH:
                {
                    // Refresh a pre-existing list
                    bSuccess = bHandleRefreshList(
                        psObj, psDSRLArg);
                }
                break;

                case DSRL_ACTION_REMOVE:
                {
                    // Handle the deletion
                    vHandleDeleteList(psObj, psDSRLArg);
                    bSuccess = TRUE;
                }
                break;

                case DSRL_ACTION_INVALID:
                default:
                {
                    bSuccess = FALSE;
                }
                break;
            }

            if (bSuccess == FALSE)
            {
                // If an error occurred, indicate a state change
                // to the application
                tEventMask |= DATASERVICE_EVENT_STATE;

                // Indicate an error
                vSetError(psObj, DATASERVICE_ERROR_CODE_GENERAL);

                DATASERVICE_IMPL_vLog(TRAFCAM_MGR_OBJECT_NAME
                    ": failed to process DSRL event\n"
                        );
            }
        }
        break;

        case DATASERVICE_EVENT_TIMEOUT:
        {
            puts(TRAFCAM_MGR_OBJECT_NAME" Timeout event occurred.");
            vProcessTimeout(psObj);
        }
        break;

        default:
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                TRAFCAM_MGR_OBJECT_NAME": Got unknown event (%u)",
                tCurrentEvent
                    );
        }
        break;
    }

    if (bStopEvent == TRUE)
    {
        // Uninitialize the object, but leave
        // enough to be useful to the callback
        vUninitObject(psObj, FALSE);
    }

    // Update event mask with any relevant events which have occurred
    SMSU_tUpdate(&psObj->sEvent, tEventMask);

    // Notify of any change via any registered callback which may be present
    SMSU_bNotify(&psObj->sEvent);

    if (bStopEvent == TRUE)
    {
        // Filter out all further weather manager updates
        SMSU_tFilter(&psObj->sEvent, DATASERVICE_EVENT_ALL);

        vUninitObject(psObj, TRUE);
    }

    return;
}

/*****************************************************************************
*
*   vUninitObject
*
*****************************************************************************/
static void vUninitObject(
    TRAFCAM_MGR_OBJECT_STRUCT *psObj,
    BOOLEAN bFullDelete
        )
{
    if (bFullDelete == TRUE)
    {
        // Destroy the SMS Update Event object
        SMSU_vDestroy(&psObj->sEvent);
    }
    else
    {
        psObj->hExpireEvent = DATASERVICE_TIMED_EVENT_INVALID_HDL;

        // Uninit the app object
        vUninitAppFacingObject(psObj);

        // Remove any broadcast-specific parsing data
        if (psObj->hInterface != TRAFCAM_INTERFACE_INVALID_OBJECT)
        {
            GsTrafcamIntf.vUnInit(psObj->hInterface);
            psObj->hInterface = TRAFCAM_INTERFACE_INVALID_OBJECT;
        }

        // Destroy persistent storage pathnames
        if (psObj->pacRefDatabaseDirPath != NULL)
        {
            SMSO_vDestroy((SMS_OBJECT)psObj->pacRefDatabaseDirPath);
            psObj->pacRefDatabaseDirPath = NULL;
        }

        // Destroy the file paths
        if (psObj->pcFilesPath != NULL)
        {
            SMSO_vDestroy((SMS_OBJECT)psObj->pcFilesPath);
            psObj->pcFilesPath = NULL;
        }

        if (psObj->pcFileNameBuffer != NULL)
        {
            SMSO_vDestroy((SMS_OBJECT)psObj->pcFileNameBuffer);
            psObj->pcFileNameBuffer = NULL;
        }

        if (psObj->hFilter != OSAL_INVALID_OBJECT_HDL)
        {
            OSAL_RETURN_CODE_ENUM eReturnCode = OSAL_ERROR;

            eReturnCode = OSAL.eLinkedListRemoveAll(psObj->hFilter,
                (OSAL_LL_RELEASE_HANDLER)vReleaseGlobalFilterItem
                    );
            if (eReturnCode != OSAL_SUCCESS)
            {
                SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    TRAFCAM_MGR_OBJECT_NAME": Failed to release"
                    " filter items (%s)", OSAL.pacGetReturnCodeName(eReturnCode)
                        );
            }

            eReturnCode = OSAL.eLinkedListDelete(psObj->hFilter);
            if (eReturnCode != OSAL_SUCCESS)
            {
                SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    TRAFCAM_MGR_OBJECT_NAME": Failed to delete filter"
                    " list (%s)", OSAL.pacGetReturnCodeName(eReturnCode)
                        );
            }

            psObj->hFilter = OSAL_INVALID_OBJECT_HDL;
        }
    }

    return;
}
