/******************************************************************************/
/*                    Copyright (c) Sirius XM Radio, Inc.                     */
/*                            All Rights Reserved                             */
/*         Licensed Materials - Property of Sirius XM Radio, Inc.             */
/******************************************************************************/
/*******************************************************************************
 *
 * DESCRIPTION
 *
 *  This module contains the Object:STRING implementation for the
 *  Sirius Module Services (SMS)
 *
 ******************************************************************************/
#include <string.h>
#include <stdio.h>
#include <ctype.h>

#include "standard.h"
#include "osal.h"

#include "sms_version.h"
#include "sms_api.h"
#include "sms_obj.h"
#include "string_obj.h"
#include "_string_obj.h"

#include "sms_api_debug.h"
static const char *gpacThisFile = __FILE__;

/*****************************************************************************
                             PUBLIC FUNCTIONS
*****************************************************************************/

/*****************************************************************************
*
*   hCreate
*
* Allows the caller to create a new STRING object which represents the
* provided C null terminated string.
*
* Inputs:
*
*   pacString - A pointer to an array of null terminated characters which
*       contains the C string this STRING object will represent.
*   tMinimumLength - The minimum number of bytes to allocate for this
*       STRING. This is useful when the caller can anticipate the string can
*       or will grow when it is modified, thus not requiring a reallocation
*       of memory.
*
* Outputs:
*
*   A valid STRING_OBJECT on success, otherwise STRING_INVALID_OBJECT
*
*****************************************************************************/
static STRING_OBJECT hCreate (
    const char *pacString,
    size_t tMinimumLength
        )
{
    STRING_OBJECT hString;
    STRING_OBJECT_STRUCT *psObj;

    // Check input
    if(pacString == NULL)
    {
        // Error!
        return STRING_INVALID_OBJECT;
    }

    hString = STRING_hCreate( SMS_INVALID_OBJECT, pacString,
                              strlen(pacString), tMinimumLength );
    if(hString != STRING_INVALID_OBJECT)
    {
        // Reference string object
        psObj = (STRING_OBJECT_STRUCT *)hString;

        // Change creator to application
        psObj->eCreator = STRING_CREATOR_APPLICATION;
    }

    return hString;
}

/*****************************************************************************
*
*   tCopy
*
* This method is used to copy the contents of a STRING object into
* another STRING object provided by the caller. The entire contents of the
* source STRING object will be copied to the destination.
*
* Inputs:
*
*   hSrcString - The source STRING_OBJECT to copy from.
*   phDstString - The destination STRING_OBJECT handle pointer to copy into.
*
* Outputs:
*
*   The number of bytes copied to the destination object.
*
*****************************************************************************/
static size_t tCopy (
    STRING_OBJECT hSrcString,
    STRING_OBJECT hDstString
        )
{
    return tCopyLocal(hDstString, hSrcString, TRUE); // Overwrite
}

/*****************************************************************************
*
*   tCopyToCStr
*
* This method is used to copy the contents of a STRING object into
* a C null terminated string provided by the destination pointer pacDst.
* The contents of the STRING object will be copied up to and including
* tDstSize bytes (including NULL).
*
* The tCopyToCStr() function copies no more than tDstSize characters from the
* STRING object (hString) into the array pointed to by pacDst.
* If the STRING object is shorter than tDstSize characters, null characters
* are appended to the copy in the array pointed to by pacDst, until tDstSize
* characters in all have been written. If the STRING object is longer than
* tDstSize characters, then the result isn't terminated by a null character.
*
* Inputs:
*
*   hString - The source STRING_OBJECT to copy from
*   pacDst - A pointer to an array of characters where to copy the STRING
*   object contents to.
*   tDstSize - The maximum number of characters that you want to copy.
*
* Outputs:
*
*   The number of bytes copied to the destination pointer.
*
*****************************************************************************/
static size_t tCopyToCStr (
    STRING_OBJECT hString,
    char *pacDst,
    size_t tDstSize
    )
{
    size_t tCopied = 0, tIndex = 0;
    STRING_OBJECT_STRUCT *psObj =
        (STRING_OBJECT_STRUCT *)hString;
    BOOLEAN bOwner;
    const STRING_BLOCK_STRUCT *psBlock;

    // Verify inputs. Destination pointer must be valid and a size
    // provided greater than 0.
    if((pacDst == NULL) || (tDstSize == 0))
    {
        return 0;
    }

    // Destination is valid. Initialize it. Having this line allows callers
    // to STRING.tCopyToCStr() pass STRING_INVALID_OBJECT to this API and still
    // at least have a returned (although empty) NULL terminated string.
    // Without having to include excessive checks everywhere.
    pacDst[0] = '\0';

    // Determine if the caller owns this resource (i.e. valid)
    bOwner = SMSO_bOwner((SMS_OBJECT)hString);
    if(bOwner == FALSE)
    {
        // We either don't own the resource or we provided an invalid
        // STRING object handle. Either way, fill it!
        psBlock = &gsEmptyBlock;
    }
    else
    {
        // We're ready to rumble.
        psBlock = &psObj->sBlock;
    }

    // Loop through each block and copy up to the length provided
    do
    {
        // Copy character if there is something to copy, otherwise
        // fill with null.
        if(tIndex < psBlock->tSize)
        {
            if (psBlock->pacString[tIndex] != '\0')
            {
                // increment for non-NULL character copied
                tCopied++;
            }

            // Copy character
            *pacDst++ = psBlock->pacString[tIndex++];
            tDstSize--;
        }
        else
        {
            // Move on to next block
            psBlock = psBlock->psNext;
            if(psBlock == NULL)
            {
                // Copy in null terminator
                *pacDst++ = '\0';

                tDstSize--;
                // Use dummy block to allow null filling
                psBlock = &gsEmptyBlock;
            }

            tIndex = 0; // Start at the beginning
        }

    } while(tDstSize > 1);

    // Write NULL
    *pacDst = '\0';

    // increment count by 1 for copied NULL
    tCopied++;

    return tCopied;
}

/*****************************************************************************
*
*   pacCStr
*
* This method is used to retrieve the contents of a STRING object by
* using it similarly to a regular C-string. To do so the contents of the
* string must be already in a contiguous block of memory or the method
* will convert it into one that is. The utility of this method is that the
* contents of a string may be easily accessed (typically for printing) as
* a direct C-string pointer to a NULL terminated array of characters.
*
* Inputs:
*
*   hString - The source STRING_OBJECT to extract as a C-String.
*
* Outputs:
*
*   A pointer to a constant character array where the contents of the STRING
*   object can be found.
*
*****************************************************************************/
static const char *pacCStr (
    STRING_OBJECT hString
        )
{
    BOOLEAN bOwner, bRebuild = FALSE;
    STRING_OBJECT_STRUCT *psObj =
        (STRING_OBJECT_STRUCT *)hString;
    const char *pacCStringNotOwner = "error! not owner",
          *pacCStringNotValid = "n/a",
          *pacCStringError = "error!",
          *pacCString = pacCStringError; // default

    // Check if a non-invalid handle was provided
    if(hString == STRING_INVALID_OBJECT)
    {
        // An invalid handle was provided, but this might
        // have been intentional, so just report our default string.
        return pacCStringNotValid;
    }
    else
    {
        // Verify inputs. Object handle must be owned.
        bOwner = SMSO_bOwner((SMS_OBJECT)hString);
        if(bOwner == FALSE)
        {
            BOOLEAN bValid;

            // Not owner, but is it valid? We mean a valid STRING_OBJECT
            bValid = SMSO_bIsValid((SMS_OBJECT)hString);
            if(bValid == FALSE)
            {
                // Error! Not owned because it is garbage(invalid)
                return pacCStringError;
            }
            else
            {
                // Error! Object is valid, but not owned
                return pacCStringNotOwner;
            }
        }
    }

    // Next check if this string contains only one block. If so
    // then there is really nothing else to do. We just need to make
    // sure we can write a phantom null-terminator there.
    if(psObj->sBlock.psNext == NULL)
    {
        // Determine if the source object is a constant string or
        // a regular string. If it is a constant string then we can use
        // the pointer directly and it is ready to go. Otherwise we need to
        // do some more work.
        if(psObj->sBlock.pacString == (char *)(psObj + 1))
        {
            // It's not a constant string
            // Enough room?
            if(psObj->sBlock.tCapacity > psObj->sBlock.tSize)
            {
                // Yep! So write a phantom null-terminator and we're done/
                psObj->sBlock.pacString[psObj->sBlock.tSize] = '\0';

                // Don't need to rebuild, use as is
                pacCString = psObj->sBlock.pacString;
            }
            else
            {
                // Sorry, need to re-build string
                bRebuild = TRUE;
            }
        }
        else
        {
            // Don't need to rebuild, use as is
            pacCString = psObj->sBlock.pacString;
        }
    }
    else
    {
        // Sorry, need to re-build string
        bRebuild = TRUE;
    }

    // Do we need to re-build the string?
    if(bRebuild == TRUE)
    {
        size_t tStringSize;
        STRING_BLOCK_STRUCT *psBlock;

        // To use the STRING object as a C-String we need it to reside in a
        // contiguous section of memory. The easiest way to do this and still
        // keep the same STRING object handle is to allocate a single new block
        // for the entire string.

        // Determine how large the entire STRING is
        vStatistics(&psObj->sBlock, &tStringSize, NULL, NULL);

        // Allocate the block we need.
        psBlock = (struct string_block_struct *)
            SMSO_hCreate(
                STRING_OBJECT_NAME":Block",
                sizeof(STRING_BLOCK_STRUCT) + tStringSize + 1, // Add null terminator
                (SMS_OBJECT)psObj,
                FALSE
                    );
        if(psBlock != NULL)
        {
            // Initialize this block, copy entire string into it
            psBlock->tCapacity = tStringSize + 1;
            psBlock->pacString = (char *)(psBlock + 1);
            psBlock->tSize =
                tCopyToCStr(hString, psBlock->pacString, psBlock->tCapacity) - 1;
            psBlock->psNext = NULL;

            // This block will now replace any this string has. To do so first
            // we destroy all blocks we have making up this string object
            vDestroyBlocks(psObj, NULL);

            // Skip over 'fixed block' and point to the next block
            psObj->sBlock.psNext = psBlock;

            // Re-compute hash code
            psObj->n32HashCode = n32ComputeHashCode(&psObj->sBlock);

            // String is now ready
            pacCString = psBlock->pacString;
        }
    }

    return pacCString;
}

/*****************************************************************************
*
*   tSize
*
* Retrieves the number of bytes this STRING object represents. This is
* effectively the string size (length) contained in the STRING object
* without including the terminating NULL.
*
*****************************************************************************/
static size_t tSize (
    STRING_OBJECT hString
    )
{
    STRING_OBJECT_STRUCT *psObj =
        (STRING_OBJECT_STRUCT *)hString;
    BOOLEAN bOwner;
    size_t tStringSize = 0;

    // Verify inputs. Object handle must be valid.
    bOwner = SMSO_bOwner((SMS_OBJECT)hString);
    if(bOwner == TRUE)
    {
        // Use statistics
        vStatistics(&psObj->sBlock, &tStringSize, NULL, NULL);
    }

    return tStringSize;
}

/*****************************************************************************
*
*   n16Compare
*
* Compares two STRING objects in relation to one another. Based on the
* input parameter bBinary, either a binary compare (go, no-go) or a
* relationship compare (lt, gt, eq) is performed. The latter is obviously
* more computation intensive because the relationship between the two must
* be discerned. Binary compares are more efficient when only knowledge of
* equality is required.
*
*       Outputs:
*               0   - STRINGs have the same value (equal)
*               > 0 - hString1 is greater than (after) hString2
*               < 0 - hString1 is less than (before) hString2 (or error)
*
*****************************************************************************/
static N16 n16Compare (
    STRING_OBJECT hString1,
    STRING_OBJECT hString2,
    BOOLEAN bBinary
    )
{
    return n16LocalStringCompare( hString1, hString2, bBinary, TRUE );
}

/*****************************************************************************
*
*   n16CaseCompare
*
* Compares two STRING objects in relation to one another, case insensitive.
* Based on the input parameter bBinary, either a binary compare (go, no-go)
* or a relationship compare (lt, gt, eq) is performed. The latter is obviously
* more computation intensive because the relationship between the two must
* be discerned. Binary compares are more efficient when only knowledge of
* equality is required.
*
*       Outputs:
*               0   - STRINGs have the same value (equal)
*               > 0 - hString1 is greater than (after) hString2
*               < 0 - hString1 is less than (before) hString2 (or error)
*
*****************************************************************************/
static N16 n16CaseCompare (
    STRING_OBJECT hString1,
    STRING_OBJECT hString2
    )
{
    return n16LocalStringCompare( hString1, hString2, FALSE, FALSE );
}


/*****************************************************************************
*
*   n16CompareCStr
*
* n16CompareCStr a null terminated C-string to a STRING object in relation to one
* another up to a given length
*
* Inputs:
*
*   pacString - A null terminated C-style string to compare against.
*   tLength - The length of the string to compare to. A value of 0 indicates
*   we are to compare the entire string. Note this behavior is slightly
*   different than standard ANSI-C strncmp().
*   hString - The STRING object to compare to.
*
* Outputs:
*   0   - Objects have the same value (equal)
*   > 0 - pacString is greater than (after) hString
*   < 0 - pacString is less than (before) hString (or error)
*
*****************************************************************************/
static N16 n16CompareCStr (
    const char *pacString,
    size_t tInputLength,
    STRING_OBJECT hString
        )
{
    N16 n16Result;
    BOOLEAN bOwner = FALSE;
    STRING_OBJECT_STRUCT *psObj = (STRING_OBJECT_STRUCT *)hString;
    const STRING_BLOCK_STRUCT *psBlock;
    N32 n32Length = tInputLength;     // make it signed for easier calculations

    // Determine if the caller owns this resource of the string object
    bOwner = SMSO_bOwner((SMS_OBJECT)hString);

    // Verify inputs.
    if((pacString == NULL) || (bOwner == FALSE))
    {
        // Error!
        return N16_MIN;
    }

    // Compare STRING object's blocks to provided CStr
    if(n32Length == 0)
    {
        n32Length = strlen(pacString); // Length to compare to
    }

    psBlock = &psObj->sBlock; // Start block

    do
    {
        // Use C-Library String compare (somewhat CPU intensive when
        // a large number of strings are being compared, but it does provide
        // the ability to determine relative order of 2 strings).
        n16Result = (N16)strncmp(
            pacString, psBlock->pacString, psBlock->tSize);
        if(n16Result != 0)
        {
            // Strings are different
            break;
        }

        // Increment string pointer to compare next
        pacString += psBlock->tSize;

        // Decrement number of bytes compared
        n32Length -= psBlock->tSize;

        // Move to next block
        psBlock = psBlock->psNext;

        if( psBlock == NULL )
        {
            // Break here is any case, because there are no more blocks to check
            if( n32Length > 0 )
            {
                // If this was the last block and we have more to compare
                // then the result is > 0 since the Cstr is larger than the
                // STRING object being compared to.
                n16Result = 1;
            }
            break;
        }
    } while (TRUE);

    return n16Result;
}

/*****************************************************************************
*
*   n16SubstringCStr
*
* Searching a position of null terminated C-substring in a STRING object
*
* Inputs:
*
*   pacSubString - A null terminated C-style string to compare against.
*   hOrigString - The STRING object to compare to.
*   bCaseInsensitive - case sensitivity flag
*
* Outputs:
*   >= 0 - position of Substring in a STRING object
*   <  0 - Substring was not found in a STRING object
*
*****************************************************************************/
static N16 n16SubstringCStr (
        STRING_OBJECT hOrigString,
        const char *pacSubString,
        BOOLEAN bCaseInsensitive
        )
{
    const char *pacOrigin = NULL;
    const char *pacResult = NULL;
    size_t tLength = 0;
    N16 n16Offset = N16_MIN;
    BOOLEAN bOwner = FALSE;

    // Determine if the caller owns this resource of the string object
    bOwner = SMSO_bOwner((SMS_OBJECT)hOrigString);

    // Verify inputs.
    if((pacSubString == NULL) || (bOwner == FALSE))
    {
        // Error!
        return N16_MIN;
    }

    tLength = strlen(pacSubString); // Length to compare to

    if(tLength == 0)
    {
        return N16_MIN;
    }

    tLength = tSize(hOrigString);

    if(tLength == 0)
    {
        return N16_MIN;
    }

    pacOrigin = pacCStr(hOrigString);
    if(bCaseInsensitive == TRUE)
    {
        char *pcUpperOrigStr;
        char *pcUpperSubStr;
        size_t tStrLen;
        size_t i;

        //Origin string to upper
        tStrLen = strlen(pacOrigin);
        pcUpperOrigStr = OSAL.pvMemoryAllocate(
                STRING_OBJECT_NAME":TempString",
                tStrLen + 1,
                FALSE);

        for(i = 0; i < tStrLen; i++)
        {
            pcUpperOrigStr[i] = (char)toupper(pacOrigin[i]);
        }
        pcUpperOrigStr[tStrLen] = '\0';

        //Substring to upper
        tStrLen = strlen(pacSubString);
        pcUpperSubStr = OSAL.pvMemoryAllocate(
                STRING_OBJECT_NAME":TempString",
                tStrLen + 1,
                FALSE);

        for(i = 0; i < tStrLen; i++)
        {
            pcUpperSubStr[i] = (char)toupper(pacSubString[i]);
        }
        pcUpperSubStr[tStrLen] = '\0';

        pacResult = strstr(pcUpperOrigStr, pcUpperSubStr);
        if( pacResult != NULL )
        {
            // Calculate offset relative to the string start
            n16Offset = (N16)(pacResult - pcUpperOrigStr);
        }

        OSAL.vMemoryFree(pcUpperOrigStr);
        OSAL.vMemoryFree(pcUpperSubStr);
    }
    else
    {
        pacResult = strstr(pacOrigin, pacSubString);
        if( pacResult != NULL )
        {
            // Calculate offset relative to the string start
            n16Offset = (N16)(pacResult - pacOrigin);
        }
    }

    return n16Offset;
}

/*****************************************************************************
*
*   hDuplicate
*
* This object interface method is used to duplicate an existing STRING.
* The source STRING may be one created by SMS or an Application. Either way
* a deep-copy is performed and thus a duplicate of the original is made.
* This new STRING may be used by the application independent of SMS however
* it sees fit. Once the caller is done with the STRING however it must
* release it by calling STRING.vDestroy() at some point.
*
* Inputs:
*
*   hString - The STRING the caller wishes to duplicate.
*
* Outputs:
*
*   A new STRING_OBJECT on success. Otherwise STRING_INVALID_OBJECT.
*
*****************************************************************************/
static STRING_OBJECT hDuplicate (
    STRING_OBJECT hString
        )
{
    STRING_OBJECT hNewString;

    hNewString = STRING_hDuplicate(SMS_INVALID_OBJECT, hString);
    if(hNewString != STRING_INVALID_OBJECT)
    {
        // Reference string object
        STRING_OBJECT_STRUCT *psObj = (STRING_OBJECT_STRUCT *)hNewString;

        // Change creator to application
        psObj->eCreator = STRING_CREATOR_APPLICATION;
    }

    return hNewString;
}

/*****************************************************************************
*
*   n32FWrite
*
* This object interface method is used to write the contents of a STRING
* (performing a deep write) to a device specified by psFile. The intent of
* this method is to effectively store the contents of a STRING for later
* retrieval (perhaps after a power cycle). The entire STRING contents are
* written to the device, with enough information to recreate the exact STRING
* when the hFRead() method is called.
*
* Inputs:
*
*   hSTRING - The STRING handle the caller wishes to write.
*   psFile - The device to write the STRING contents to.
*
* Outputs:
*
*   The number of characters written or EOF on error.
*
*****************************************************************************/
static N32 n32FWrite (
    STRING_OBJECT hString,
    FILE *psFile
        )
{
    N32 n32Return = 0;

    n32Return = STRING_n32FWrite(hString, psFile, FALSE);

    return n32Return;
}

/*****************************************************************************
*
*   hFRead
*
* This object interface method is used to read from a specified device psFile
* and generate a STRING from that information. The data read from the device
* must have been previously written by the n32FWrite method. Upon
* successful execution of this method a new STRING is created (created by the
* caller) which may be used to register for events or presented to the
* UI for display, etc. This method allows the caller to effectively read
* a previously stored STRING regenerating the original STRING written(saved) at
* an earlier time.
*
* Inputs:
*
*   psFile - The device to read the STRING contents from.
*
* Outputs:
*
*   A new STRING on success, otherwise STRING_INVALID_OBJECT on failure.
*
*****************************************************************************/
static STRING_OBJECT hFRead (
    FILE *psFile
        )
{
    STRING_OBJECT hNewString;

    hNewString = STRING_hFRead(SMS_INVALID_OBJECT, psFile);
    if(hNewString != STRING_INVALID_OBJECT)
    {
        // Reference string object
        STRING_OBJECT_STRUCT *psObj = (STRING_OBJECT_STRUCT *)hNewString;

        // Change creator to application
        psObj->eCreator = STRING_CREATOR_APPLICATION;
    }

    return hNewString;
}

/*****************************************************************************
*
*   n32FPrintf
*
* This object interface method is used by the caller to send formatted
* output of a STRING's contents to a specified file or device.
* This is mainly helpful during debugging of STRING's but could be used by
* an application for any reason. This API is different than the n32FWrite()
* method which instead writes the contents of a STRING to a file for the
* purposes of later re-generating the STRING (for storage of the object).
* This API instead sends the STRING as a verbose formatted output version.
*
* Inputs:
*
*   hString - The STRING handle the caller wishes to write.
*   psFile - The device to write the STRING contents to.
*
* Outputs:
*
*   The number of characters written or EOF on error.
*
*****************************************************************************/
static N32 n32FPrintf (
    STRING_OBJECT hString,
    FILE *psFile
        )
{
    STRING_OBJECT_STRUCT *psObj =
        (STRING_OBJECT_STRUCT *)hString;
    BOOLEAN bOwner = FALSE;
    N32 n32Return = EOF;

    // Determine if the caller owns this resource
    bOwner = SMSO_bOwner((SMS_OBJECT)hString);

    // Verify inputs. Object handle must be valid as well as the file handle.
    if((bOwner != FALSE) && (psFile != NULL))
    {
        size_t tSize, tCapacity, tNumBlocks;
        const STRING_BLOCK_STRUCT *psBlock = &psObj->sBlock;
        n32Return = 0;

        // Compute STRING size, capacity and blocks used
        vStatistics(&psObj->sBlock, &tSize, &tCapacity, &tNumBlocks);

        // Print common STRING info
        n32Return += fprintf(psFile, "STRING: hString = 0x%p\t(%s)\n",
            psObj, pacCreatorText(psObj->eCreator));
        n32Return += fprintf(psFile, "\tn32HashCode = %d\n",
            psObj->n32HashCode);
        n32Return += fprintf(psFile,
            "\ttCapacity = %u, tSize = %u, tNumBlocks = %u\n",
            tCapacity, tSize, tNumBlocks);
        n32Return += fprintf(psFile, "\tpacString = '");
        do
        {
            // Dump contents
            n32Return += fprintf(psFile, "%.*s",
                psBlock->tSize, psBlock->pacString);

            // Proceed to next block
            psBlock = psBlock->psNext;

        } while(psBlock != NULL);

        n32Return += fprintf(psFile, "'\n");
    }

    return n32Return;
}

/*****************************************************************************
*
*   bModify
*
* Modify an existing STRING object. The STRING must exist already and the
* caller must have exclusive access to the object. The caller may provide new
* contents for which to update the STRING with. If a replacement STRING
* is created to accommodate the new data, one is created and the old one
* will be destroyed. The resulting STRING is always returned via the provided
* pointer to a STRING handle in either case.
*
* Inputs:
*
*   phString - A pointer to a valid STRING handle which represents the STRING
*   to update. It also is used as a return value in case the STRING provided
*   is replaced with another.
*
*   pacSrc - A pointer to a NULL terminated string for which to update this
*       STRING object with.
*
* Outputs:
*
*   TRUE on success and if the STRING was updated (something changed), otherwise
*   FALSE is returned indicating nothing has changed.
*
*****************************************************************************/
static BOOLEAN bModifyCStr (
    STRING_OBJECT hString,
    const char *pacSrc
    )
{
    return bCopyCStrLocal(hString, pacSrc, NULL, TRUE); // Overwrite
}

/*****************************************************************************
*
*   tAppendString
*
* Append a STRING_OBJECT onto an existing STRING object. Both STRING_OBJECTs
* must exist already and the caller must have exclusive access to them.
*
* Inputs:
*
*   hSrcString - A valid STRING handle which represents the source STRING
*       which is to be appended onto hDstString.
*   hDstString - A valid STRING handle which represents the STRING which is
*       to be appended onto.
*
* Outputs:
*
*   The number of bytes copied to the destination object.
*
*****************************************************************************/
static size_t tAppendString (
    STRING_OBJECT hSrcString,
    STRING_OBJECT hDstString
        )
{
    return tCopyLocal(hDstString, hSrcString, FALSE); // Append
}

/*****************************************************************************
*
*   tAppendCStr
*
* Append to an existing STRING object. The STRING must exist already and
* the caller must have exclusive access to the object. The caller may provide
* new contents for which to Append the STRING with. If a replacement STRING
* is created to accommodate the new data, one is created and the old one
* will be destroyed. The resulting STRING is always returned via the provided
* pointer to a STRING handle in either case.
*
* Inputs:
*
*   phString - A pointer to a valid STRING handle which represents the STRING
*   to append to. It also is used as a return value in case the STRING provided
*   is replaced with another.
*
*   pacSrc - A pointer to a NULL terminated string for which to append to this
*       STRING object with.
*
* Outputs:
*
*   The number of bytes appended to the destination object..
*
*****************************************************************************/
static size_t tAppendCStr (
    STRING_OBJECT hString,
    const char *pacSrc
        )
{
    size_t tNumAppended = 0;
    bCopyCStrLocal(hString, pacSrc, &tNumAppended, FALSE); // Append
    return tNumAppended;
}

/*****************************************************************************
*
*   vDestroy
*
* This method is used by anyone who used STRING.hDuplicate() or STRING.hFRead()
* to create a new STRING. If the STRING was not created this way it will not
* be destroyed. Only external (Applications) should make a call to this API.
* Internal SMS calls should instead always use STRING_vDestroy().
*
*****************************************************************************/
static void vDestroy (
    STRING_OBJECT hString
        )
{
    STRING_OBJECT_STRUCT *psObj =
        (STRING_OBJECT_STRUCT *)hString;
    BOOLEAN bOwner;

    // Verify inputs. Object handle must be valid.
    bOwner = SMSO_bOwner((SMS_OBJECT)hString);
    if(bOwner == FALSE)
    {
        // Error!
        return;
    }

    // First check the ownership, if it is not application it cannot be
    // destroyed this way.
    if(psObj->eCreator != STRING_CREATOR_APPLICATION)
    {
        // Error!
        return;
    }

    // Initialize string object
    psObj->eCreator = STRING_CREATOR_SMS; // by default
    psObj->n32HashCode = 0;

    // Destroy all blocks
    vDestroyBlocks(psObj, NULL);

    // Free object instance
    SMSO_vDestroy((SMS_OBJECT)hString);

    return;
}

/*****************************************************************************
                             FRIEND FUNCTIONS
*****************************************************************************/

/*****************************************************************************
*
*   STRING_hCreate
*
* Allows the caller to create a new STRING object which represents the
* provided C null terminated string.
*
* Inputs:
*
*   hSMS - An SMS_OBJECT which is to be this object's parent. The
*       caller may provide SMS_INVALID_OBJECT is this STRING object has no
*       child relationship.
*   pacString - A pointer to an array of characters which
*       contains the C string this STRING object will represent.
*   tSize - The size of the input string.
*   tMinimumLength - The minimum number of bytes to allocate for this
*       STRING. This is useful when the caller can anticipate the string can
*       or will grow when it is modified, thus not requiring a reallocation
*       of memory.
*
* Outputs:
*
*   A valid STRING_OBJECT on success, otherwise STRING_INVALID_OBJECT
*
*****************************************************************************/
STRING_OBJECT STRING_hCreate (
    SMS_OBJECT hSMS,
    const char *pacString,
    size_t tSize,
    size_t tMinimumLength
    )
{
    STRING_OBJECT_STRUCT *psObj =
        (STRING_OBJECT_STRUCT *)STRING_INVALID_OBJECT;
    size_t tCreateSize;

    // Verify inputs. String pointer must be valid.
    if(pacString == NULL)
    {
        // Error!
        return STRING_INVALID_OBJECT;
    }

    // Compute the amount of memory required using the initial string size
    // to allocate for this object as well as the object structure itself.
    tCreateSize = sizeof(STRING_OBJECT_STRUCT) +
        (tSize < tMinimumLength ? tMinimumLength : tSize);

    // Create an instance of this object
    psObj = (STRING_OBJECT_STRUCT *)
        SMSO_hCreate(
            STRING_OBJECT_NAME,
            tCreateSize,
            hSMS, // Child of hSMS (if provided, otherwise it is a parent)
            FALSE); // No Lock (ignored if hSMS is valid (parent provided))
    if(psObj == NULL)
    {
        // Error!
        return STRING_INVALID_OBJECT;
    }

    // Initialize string object
    psObj->eCreator = STRING_CREATOR_SMS; // by default
    psObj->sBlock.tCapacity = tCreateSize - sizeof(STRING_OBJECT_STRUCT);
    psObj->sBlock.tSize = tSize;
    psObj->sBlock.psNext = NULL;
    psObj->sBlock.pacString = (char *)(psObj + 1);
    strncpy(psObj->sBlock.pacString, pacString, tSize);
    psObj->n32HashCode = n32ComputeHashCode(&psObj->sBlock);

    return (STRING_OBJECT)psObj;
}

/*****************************************************************************
*
*   STRING_hCreateFromBuffer
*
*   Note: THIS IS DISABLED UNTIL IT CAN BE PROPERLY TESTED
*
*****************************************************************************/
STRING_OBJECT STRING_hCreateFromBuffer (
    SMS_OBJECT hSMS,
    OSAL_BUFFER_HDL hBuffer,
    UN8 un8CharBitSize,
    size_t tMaxNumCharsWithNull
        )
{
#if 0
    // First, create a string object to write into
    STRING_OBJECT hString;

    // Create a string with a single block large enough
    // for this read
    hString = STRING_hCreate(
        hSMS, NULL,
        tMaxNumCharsWithNull,
        tMaxNumCharsWithNull);
    if (hString != STRING_INVALID_OBJECT)
    {
        STRING_OBJECT_STRUCT *psObj = (STRING_OBJECT_STRUCT *)hString;
        STRING_BLOCK_STRUCT *psBlock = &psObj->sBlock;
        size_t tIndex = 0,
               tBitsRead,
               tBitsRemaining = tMaxNumCharsWithNull * un8CharBitSize;

        do
        {
            BOOLEAN bSuccess = TRUE;

            if (psBlock == NULL)
            {
                break;
            }

            if (psBlock->tCapacity != tMaxNumCharsWithNull)
            {
                break;
            }

            // Stay within our bounds
            while (tBitsRemaining > 0)
            {
                // Read the next byte into the block
                tBitsRead = OSAL.tBufferReadHeadBits(
                    hBuffer,
                    &psBlock->pacString[tIndex], 0,
                    un8CharBitSize );

                if (tBitsRead != un8CharBitSize)
                {
                    bSuccess = FALSE;
                    break;
                }

                if (psBlock->pacString[tIndex] == '\0')
                {
                    // All done!
                    break;
                }

                // Reduce the number of bits remaining in this read
                tBitsRemaining -= un8CharBitSize;

                // Move to the next read position
                tIndex++;
            }

            if (bSuccess == FALSE)
            {
                // Stop here
                break;
            }

            // Store the string size now
            psBlock->tSize = tIndex;

            // Compute the hash now
            psObj->n32HashCode = n32ComputeHashCode(&psObj->sBlock);

            return hString;
        } while (FALSE);

        STRING_vDestroy(hString);
    }
#endif

    return STRING_INVALID_OBJECT;
}

/*****************************************************************************
*
*   STRING_hCreateConst
*
* This is a special STRING object create method with allows the creation
* of a constant string which has the same interface as other strings.
* This is useful if one needs to create a STRING object which will not and
* CANNOT be modified. The STRING object itself can be destroyed but the
* string contents itself cannot be modified.
*
* NOTE: This also means that one cannot modify the provided string as well.
* Doing so will have unpredictable results.
*
* Inputs:
*
*   pacString - A pointer to an array of null terminated characters which
*       contains the C string this STRING object will represent. This pointer
*       must point to a persistent (or static) memory location containing the
*       string as it will not be copied, but instead used directly as the
*       contents of the STRING object.
*   tSize - The size of the incoming string without null terminator
*
* Outputs:
*
*   A valid STRING_OBJECT on success, otherwise STRING_INVALID_OBJECT
*
*****************************************************************************/
STRING_OBJECT STRING_hCreateConst (
    const char *pacString,
    size_t tSize
        )
{
    STRING_OBJECT_STRUCT *psObj =
        (STRING_OBJECT_STRUCT *)STRING_INVALID_OBJECT;

    // Verify inputs. String pointer must be valid.
    if(pacString == NULL)
    {
        // Error!
        return STRING_INVALID_OBJECT;
    }

    // Create an instance of this object plus a place to store the string ptr
    psObj = (STRING_OBJECT_STRUCT *)
        SMSO_hCreate(
            STRING_OBJECT_NAME,
            sizeof(STRING_OBJECT_STRUCT),
            SMS_INVALID_OBJECT, // Never a parent for constant strings
            FALSE); // No Lock feature
    if(psObj == NULL)
    {
        // Error!
        return STRING_INVALID_OBJECT;
    }

    // Initialize string object
    psObj->eCreator = STRING_CREATOR_SMS; // by default
    psObj->sBlock.tCapacity = tSize;
    psObj->sBlock.tSize = tSize;
    psObj->sBlock.psNext = NULL;
    psObj->sBlock.pacString = (char *)pacString;
    psObj->n32HashCode = n32ComputeHashCode(&psObj->sBlock);

    return (STRING_OBJECT)psObj;
}

/*****************************************************************************
*
*   STRING_vDestroy
*
* This function is used to directly destroy any STRING Object.
* It can only be called from within SMS and thus works typically on
* STRINGs which have been created by SMS.
*
*****************************************************************************/
void STRING_vDestroy (
    STRING_OBJECT hString
    )
{
    STRING_OBJECT_STRUCT *psObj =
        (STRING_OBJECT_STRUCT *)hString;
    BOOLEAN bOwner;

    // Verify inputs. Object handle must be valid.
    bOwner = SMSO_bOwner((SMS_OBJECT)hString);
    if(bOwner == FALSE)
    {
        // Error!
        return;
    }

    // Any STRING which is to be destroyed using this entry point
    // the 'friend interface', can be immediately changed to app
    // creator so that the public vDestroy succeeds.
    psObj->eCreator = STRING_CREATOR_APPLICATION;

    // Now destroy it
    vDestroy(hString);

    return;
}

/*****************************************************************************
*
*   STRING_hDuplicate
*
* This object interface method is used to duplicate an existing STRING.
* The source STRING may be one created by SMS or an Application. Either way
* a deep-copy is performed and thus a duplicate of the original is made.
* This new STRING may be used by the application independent of SMS however
* it sees fit. Once the caller is done with the STRING however it must
* release it by calling STRING.vDestroy() at some point.
*
* Inputs:
*
*   hParent - A handle to a parent SMS object the caller wishes to associate
*       this new STRING with. If the caller wants this to be independent
*       then they simply need to provide this as SMS_INVALID_OBJECT.
*
*   hSrcString - The STRING the caller wishes to duplicate.
*
* Outputs:
*
*   A new STRING_OBJECT on success. Otherwise STRING_INVALID_OBJECT.
*
*****************************************************************************/
STRING_OBJECT STRING_hDuplicate (
    SMS_OBJECT hParent,
    STRING_OBJECT hSrcString
        )
{
    STRING_OBJECT hNewString = STRING_INVALID_OBJECT;
    STRING_OBJECT_STRUCT *psSrcObj =
        (STRING_OBJECT_STRUCT *)hSrcString;
    BOOLEAN bOwner;

    // Determine if the caller owns this resource and it is valid
    bOwner = SMSO_bOwner((SMS_OBJECT)hSrcString);

    // Check ownership
    if(bOwner == FALSE)
    {
        // Error!
        return STRING_INVALID_OBJECT;
    }

    // Determine if the source object is a constant string or
    // a regular string. If it is a constant string then we can use
    // that method. Otherwise create a regular string.
    if(psSrcObj->sBlock.pacString != (char *)(psSrcObj + 1))
    {
        // This is a constant string
        // Create a new string and return it to caller
        hNewString = STRING_hCreateConst(
            psSrcObj->sBlock.pacString, psSrcObj->sBlock.tSize);
    }
    else
    {
        size_t tSrcStringSize;

        // This is a regular string with potentially multiple blocks
        // of varying size. Create a new string and return it to caller.

        // First determine the size of the source string by using the
        // STRING object statistics.
        vStatistics(&psSrcObj->sBlock, &tSrcStringSize, NULL, NULL);

        // Create a new empty string which can at a minimum contain
        // the source STRING's contents as one block.
        hNewString =
            STRING_hCreate(hParent, "", 0, tSrcStringSize);
        if(hNewString != STRING_INVALID_OBJECT)
        {
            STRING_OBJECT_STRUCT *psNewObj =
                (STRING_OBJECT_STRUCT *)hNewString;
            const STRING_BLOCK_STRUCT *psSrcBlock =
                &psSrcObj->sBlock;
            char *pacDst = psNewObj->sBlock.pacString;

            // Copy source string into destination
            do
            {
                // Copy source block into destination
                strncpy(pacDst, psSrcBlock->pacString, psSrcBlock->tSize);

                // Increment string pointer
                pacDst += psSrcBlock->tSize;

                // Increment size of this block
                psNewObj->sBlock.tSize += psSrcBlock->tSize;

                // Proceed to next block
                psSrcBlock = psSrcBlock->psNext;

            } while(psSrcBlock != NULL);

            // Compute hash-code
            psNewObj->n32HashCode = n32ComputeHashCode(&psNewObj->sBlock);
        }
    }

    return hNewString;
}

/*****************************************************************************
*
*   STRING_bDecodeAndModifyCStr
*
* Modify an existing STRING object. The STRING must exist already and the
* caller must have exclusive access to the object. The caller may provide new
* contents for which to update the STRING with. If a replacement STRING
* is created to accommodate the new data, one is created and the old one
* will be destroyed. The resulting STRING is always returned via the provided
* pointer to a STRING handle in either case. In case if input string has some
* special symbols, symbols will be decoded.
*
* Inputs:
*
*   phString - A pointer to a valid STRING handle which represents the STRING
*   to update. It also is used as a return value in case the STRING provided
*   is replaced with another.
*
*   pacSrc - A pointer to a NULL terminated string for which to update this
*       STRING object with.
*
* Outputs:
*
*   TRUE on success and if the STRING was updated (something changed), otherwise
*   FALSE is returned indicating nothing has changed.
*
*****************************************************************************/
BOOLEAN STRING_bDecodeAndModifyCStr (
    STRING_OBJECT hString,
    const char *pacSrc
        )
{
    BOOLEAN bResult = FALSE;
    const char *pacRealSrc = pacSrc;
    char *pacDecodedSrc = NULL;

    do
    {
        BOOLEAN bOk = TRUE;

        if (pacSrc != NULL)
        {
            const char *pcSrcSymbol = NULL;
            size_t tSrcLen = 0;
            size_t tStrPartLen = 0;
            char cDecSymbol = 0;

            tSrcLen = strlen(pacSrc);
            pacDecodedSrc = (char*) SMSO_hCreate(STRING_OBJECT_NAME":DecTmpBuf",
                    tSrcLen + 1, SMS_INVALID_OBJECT, FALSE
                        );
            if (pacDecodedSrc == NULL)
            {
                SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                        STRING_OBJECT_NAME": failed to create temp decoder buffer");
                break;
            }

            while (tSrcLen > 0)
            {
                // searching for symbol
                pcSrcSymbol = strchr(pacSrc, STRING_ENCODED_SEQUENCE_MARK);
                if (pcSrcSymbol != NULL)
                {
                    size_t sRem;

                    // if symbol has been found, lets copy part of string before symbol
                    // 1. calculating string part length
                    tStrPartLen = (pcSrcSymbol - pacSrc) / sizeof(char);
                    // 2. copying
                    strncat(pacDecodedSrc, pacSrc, tStrPartLen);

                    // 3. decoding symbol right after encoded sequence mark
                    cDecSymbol = cDecodeSymbol(pcSrcSymbol + 1);
                    if (cDecSymbol == '\0')
                    {
                        // No symbols decoded, so just insert the text 
                        // into target string as is
                        strncat(pacDecodedSrc, pcSrcSymbol, STRING_ENCODED_SEQUENCE_LEN + 1);
                    }
                    else
                    {
                        // adding just one decoded symbol to string
                        strncat(pacDecodedSrc, &cDecSymbol, 1);
                    }
                    // moving pacSrc pointer
                    sRem = strlen(pcSrcSymbol);
                    pacSrc = pcSrcSymbol + 
                        ((sRem < (STRING_ENCODED_SEQUENCE_LEN + 1)) ? 
                            sRem : 
                            (STRING_ENCODED_SEQUENCE_LEN + 1));
                }
                else
                {
                    // if no symbol, append string to buffer and exit
                    strcat(pacDecodedSrc, pacSrc);
                    break;
                }
                tSrcLen = strlen(pacSrc);
            }

            pacRealSrc = pacDecodedSrc;

        }

        if (bOk == FALSE)
        {
            // something went wrong, don't do copy
            break;
        }

        // Do data copy
        bResult = bCopyCStrLocal(hString, pacRealSrc, NULL, TRUE);

    } while (FALSE);

    if (pacDecodedSrc != NULL)
    {
        SMSO_vDestroy((SMS_OBJECT)pacDecodedSrc);
    }

    return bResult;
}

/*****************************************************************************
*
*   STRING_bCapitalize
*
* Modify an existing STRING object by capitalizing its first character.
*
* Inputs:
*
*   phString - A pointer to a valid STRING handle which represents the STRING
*   to update. It also is used as a return value in case the STRING provided
*   is replaced with another.
*
* Outputs:
*
*   TRUE on success and if the STRING was updated (something changed), otherwise
*   FALSE is returned indicating nothing has changed.
*
*****************************************************************************/
BOOLEAN STRING_bCapitalize (
    STRING_OBJECT hString
        )
{
    BOOLEAN bResult = FALSE, bOwner = FALSE;
    STRING_OBJECT_STRUCT *psObj = (STRING_OBJECT_STRUCT*)hString;

    // Determine if the caller owns this resource
    bOwner = SMSO_bOwner((SMS_OBJECT)hString);
    if(bOwner == TRUE)
    {
        // Verify we can actually modify this destination STRING object (i.e. it
        // is not a constant string). If it is a constant string the string pointer
        // would not be part of the STRING object itself.
        if(psObj->sBlock.pacString == (char*)(psObj + 1))
        {
            // Verify if there are any characters in the object.
            // If the first block's size is 0, then there is nothing to capitalize
            if(psObj->sBlock.tSize > 0)
            {
                char cLetter = (char)toupper(psObj->sBlock.pacString[0]);

                // Mark string as modified only if it was really changed
                if(cLetter != psObj->sBlock.pacString[0])
                {
                    psObj->sBlock.pacString[0] = cLetter;
                    psObj->n32HashCode = n32ComputeHashCode(&psObj->sBlock);
                    bResult = TRUE;
                }
            }
        }
    }
    return bResult;
}

/*****************************************************************************
*
*   STRING_bToUpper
*
* Modify an existing STRING object by capitalizing its characters.
*
* Inputs:
*
*   phString - A pointer to a valid STRING handle which represents the STRING
*   to update. It also is used as a return value in case the STRING provided
*   is replaced with another.
*
* Outputs:
*
*   TRUE on success and if the STRING was updated (something changed), otherwise
*   FALSE is returned indicating nothing has changed.
*
*****************************************************************************/

BOOLEAN STRING_bToUpper (
    STRING_OBJECT hString
        )
{
    BOOLEAN bResult = FALSE, bOwner = FALSE;
    STRING_OBJECT_STRUCT *psObj = (STRING_OBJECT_STRUCT*)hString;

    // Determine if the caller owns this resource
    bOwner = SMSO_bOwner((SMS_OBJECT)hString);
    if(bOwner == TRUE)
    {
        STRING_BLOCK_STRUCT *pCurrentBlock = &psObj->sBlock;

        // Verify we can actually modify this destination STRING object (i.e. it
        // is not a constant string). If it is a constant string the string pointer
        // would not be part of the STRING object itself.
        if(psObj->sBlock.pacString == (char*)(psObj + 1))
        {
            while (NULL != pCurrentBlock )
            {
                // Verify if there are any characters in the object.
                // If the block's size is 0, then there is nothing to capitalize
                if(pCurrentBlock->tSize > 0)
                {
                    char cLetter;
                    size_t i = 0;

                    for( i = 0; i < pCurrentBlock->tSize; i++)
                    {
                        cLetter = (char)toupper(pCurrentBlock->pacString[i]);

                        // Mark string as modified only if it was really changed
                        if(cLetter != pCurrentBlock->pacString[i])
                        {
                            pCurrentBlock->pacString[i] = cLetter;
                            bResult = TRUE;
                        }
                    }
                }
                pCurrentBlock = pCurrentBlock->psNext;
            }
        }

        if (FALSE != bResult)
        {
            //Compute hash code only if we have changes
            psObj->n32HashCode = n32ComputeHashCode(&psObj->sBlock);
        }
    }

    return bResult;
}

/*****************************************************************************
*
*   STRING_n32FWrite
*
* This object interface method is used to write the contents of a STRING
* (performing a deep write) to a device specified by psFile. The intent of
* this method is to effectively store the contents of a STRING for later
* retrieval (perhaps after a power cycle). The entire STRING contents are
* written to the device, with enough information to recreate the exact STRING
* when the hFRead() method is called.
*
* Inputs:
*
*   hSTRING - The STRING handle the caller wishes to write.
*   psFile - The device to write the STRING contents to.
*   bDoEncoding - defines encoding mechanizm for the string written to the device.
*
* Outputs:
*
*   The number of characters written or EOF on error.
*
*****************************************************************************/
N32 STRING_n32FWrite (
    STRING_OBJECT hString,
    FILE *psFile,
    BOOLEAN bDoEncoding
        )
{
    STRING_OBJECT_STRUCT *psObj =
        (STRING_OBJECT_STRUCT *)hString;
    N32 n32Return = EOF;
    BOOLEAN bOwner;

    // Determine if the caller owns this resource
    bOwner = SMSO_bOwner((SMS_OBJECT)hString);

    // Verify inputs. Object handle must be valid as well as the file handle.
    if((bOwner != FALSE) && (psFile != NULL))
    {
        const STRING_BLOCK_STRUCT *psBlock = &psObj->sBlock;

        n32Return = 0;
        do
        {
            BOOLEAN bNeedEncoding = bDoEncoding;
            N32 n32TmpReturn;

            if (bNeedEncoding == TRUE)
            {
                // Check that we really need to encode the string.
                bNeedEncoding = bHasSpecialSymbols(psBlock);
            }

            if (bNeedEncoding == FALSE)
            {
                // Write contents at once
                n32TmpReturn = fwrite(
                    psBlock->pacString, sizeof(char), psBlock->tSize, psFile);
                if (n32TmpReturn >= 0)
                {
                    n32Return += n32TmpReturn;
                }
                else
                {
                    return n32TmpReturn;
                }
            }
            else
            {
                size_t tIndex;
                const char *pcSymbol = psBlock->pacString;
                for (tIndex = 0; tIndex < psBlock->tSize; ++tIndex, ++pcSymbol)
                {
                    const char cSymbol = *pcSymbol;
                    BOOLEAN bIsSpecialSymbol = STRING_IS_SPECIAL_SYMBOL(cSymbol);
                    if (bIsSpecialSymbol == TRUE)
                    {
                        // Write content symbol encoded
                        n32TmpReturn = fprintf(psFile, "%c%02X",
                                STRING_ENCODED_SEQUENCE_MARK, (int)cSymbol);
                    }
                    else
                    {
                        // Write content symbol as-is
                        n32TmpReturn = fwrite(pcSymbol, sizeof(char), 1, psFile);
                    }

                    if (n32TmpReturn >= 0)
                    {
                        n32Return += n32TmpReturn;
                    }
                    else
                    {
                        return n32TmpReturn;
                    }
                }
            }

            // Proceed to next block
            psBlock = psBlock->psNext;

        } while(psBlock != NULL);
    }

    return n32Return;
}

/*****************************************************************************
*
*   STRING_hFRead
*
* This object interface method is used to read from a specified device psFile
* and generate a STRING from that information. The data read from the device
* must have been previously written by the n32FWrite method. Upon
* successful execution of this method a new STRING is created (created by the
* caller) which may be used to register for events or presented to the
* UI for display, etc. This method allows the caller to effectively read
* a previously stored STRING regenerating the original STRING written(saved) at
* an earlier time.
*
* Inputs:
*
*   psFile - The device to read the STRING contents from.
*
* Outputs:
*
*   A new STRING on success, otherwise STRING_INVALID_OBJECT on failure.
*
*****************************************************************************/
STRING_OBJECT STRING_hFRead (
    SMS_OBJECT hParent,
    FILE *psFile
        )
{
    STRING_OBJECT hString = STRING_INVALID_OBJECT;
    char acBuffer[32]; // Just a decent sized buffer to incrementally
                       // read in a string.
    UN8 un8Index = 0;
    size_t tNum;
    BOOLEAN bError = FALSE;

    // Verify input
    if(psFile == NULL)
    {
        // Error!
        return STRING_INVALID_OBJECT;
    }

    // Place a null terminator in the last element of our buffer and
    // leave it there for processing purposes later.
    acBuffer[sizeof(acBuffer) - 1] = '\0';

    // Read a string from the file provide byte by byte until a '\0' is found.
    // A '\0' is used to determine the end of a STRING entity.
    do
    {
        // Read a character from the file
        tNum = fread(&acBuffer[un8Index], sizeof(acBuffer[0]), 1, psFile);
        if(tNum != 1)
        {
            if ( FALSE != OSAL.iFeof(psFile))
            {
                // End of file
                acBuffer[un8Index] = '\0';
            }
            else
            {
                // Error! File error occured.
                bError = TRUE;
                break;
            }
        }

        // If we have found either the end of string or we are at the
        // end of our read buffer we need to dump off what we have.
        if((acBuffer[un8Index] == '\0') ||
            (un8Index == sizeof(acBuffer) - 2)) // acount for last character
                                                // which must be '/0' and zero
                                                // origination.
        {
            // Create or append to string
            if(hString == STRING_INVALID_OBJECT)
            {
                // Create a STRING object (first time)
                hString =
                    STRING_hCreate(hParent, &acBuffer[0], strlen(acBuffer), 0);
                if(hString == STRING_INVALID_OBJECT)
                {
                    // Error! Something went wrong.
                    bError = TRUE;
                    break;
                }
            }
            else
            {
                size_t tNumAppended;

                // Append to the STRING object (add to it)
                tNumAppended = STRING.tAppendCStr(hString, &acBuffer[0]);
                if(tNumAppended != strlen(&acBuffer[0]))
                {
                    // Error! Something went wrong.
                    bError = TRUE;
                    break;
                }
            }

            // Check if we're done.
            if(acBuffer[un8Index] == '\0')
            {
               // We're done!
               break;
            }
            else
            {
               // Reset index to beginning of buffer
               un8Index = 0;
            }
        }
        else
        {
            // Increment to next position
            un8Index++;
        }

    } while(TRUE);

    // Check if there was an error. If so destroy anything we created
    // and return an invalid handle.
    if(bError == TRUE)
    {
        if(hString != STRING_INVALID_OBJECT)
        {
            // Destroy STRING object we created
            STRING.vDestroy(hString);
            hString = STRING_INVALID_OBJECT;
        }
    }

    return hString;
}

/*****************************************************************************
                             PRIVATE FUNCTIONS
*****************************************************************************/

/*****************************************************************************
*
*   n16LocalStrNCaseCmp()
*
* Compares two strings in relation to one another. Stolen from libc source.
*
*       Outputs:
*               0   - STRINGs have the same value (equal)
*               > 0 - s1 is greater than (after) s2
*               < 0 - s1 is less than (before) s2 (or error)
*
*****************************************************************************/
static N16 n16LocalStrNCaseCmp(const char *pcString1, const char *pcString2, unsigned int uiLength)
{
    if (uiLength == 0)
        return 0;

    while ((uiLength-- != 0) &&
           (tolower(*(unsigned char *) pcString1) == tolower(*(unsigned char *) pcString2)))
    {
        if (uiLength == 0 || *pcString1 == '\0' || *pcString2 == '\0')
        {
            return 0;
        }
        pcString1++;
        pcString2++;
    }
    return (N16)(tolower(*(unsigned char *) pcString1) - tolower(*(unsigned char *) pcString2));
}

/*****************************************************************************
*
*   n16LocalStringCompare
*
* Compares two STRING objects in relation to one another. Based on the
* input parameter bBinary, either a binary compare (go, no-go) or a
* relationship compare (lt, gt, eq) is performed. The latter is obviously
* more computation intensive because the relationship between the two must
* be discerned. Binary compares are more efficient when only knowledge of
* equality is required.
*
*       Outputs:
*               0   - STRINGs have the same value (equal)
*               > 0 - hString1 is greater than (after) hString2
*               < 0 - hString1 is less than (before) hString2 (or error)
*
*****************************************************************************/
static N16 n16LocalStringCompare(
    STRING_OBJECT hString1,
    STRING_OBJECT hString2,
    BOOLEAN bBinary,
    BOOLEAN bCaseSensitive
        )
{
    N16 n16Result = N16_MIN;
    BOOLEAN bOwner = FALSE;
    STRING_OBJECT_STRUCT *psObj1 = (STRING_OBJECT_STRUCT *)hString1,
                         *psObj2 = (STRING_OBJECT_STRUCT *)hString2;

    // Determine if the caller owns this resource of both strings
    bOwner = SMSO_bOwner((SMS_OBJECT)hString1);
    if(bOwner == TRUE)
    {
        bOwner = SMSO_bOwner((SMS_OBJECT)hString2);
    }
    if(bOwner == FALSE)
    {
        // Error!
        return N16_MIN;
    }

    // Check if the caller wants a full compare or if they want a binary
    // compare only. Full compare takes more time, binary compare is go, no-go.
    if( (bCaseSensitive == FALSE) ||
        (bBinary == FALSE) ||
        (psObj1->n32HashCode == psObj2->n32HashCode))
    {
        size_t tOffset1 = 0, tOffset2 = 0, tSize;
        const STRING_BLOCK_STRUCT
            *psBlock1 = &psObj1->sBlock, *psBlock2 = &psObj2->sBlock;

        // Full compare is desired, or both hash codes are equivalent
        // Either way, more investigation must be done.
        // Dig further and determine if they are 'exactly' the same.

        do
        {
            // Compute the length to compare. Compare the smaller of
            // the two block sizes.
            tSize = psBlock1->tSize - tOffset1 < psBlock2->tSize - tOffset2
                ? psBlock1->tSize - tOffset1 : psBlock2->tSize - tOffset2;

            if ( bCaseSensitive == TRUE )
            {
                n16Result = (N16)strncmp( &psBlock1->pacString[tOffset1],
                    &psBlock2->pacString[tOffset2], tSize);
            }
            else
            {
                n16Result = n16LocalStrNCaseCmp( &psBlock1->pacString[tOffset1],
                    &psBlock2->pacString[tOffset2], tSize);
            }


            if(n16Result == 0) // Equal
            {
                // Bump block 1 & block 2 offsets
                tOffset1 += tSize;
                tOffset2 += tSize;

                // Block 1 need bumped?
                if(psBlock1->tSize == tOffset1)
                {
                    // Next block
                    psBlock1 = psBlock1->psNext;
                    tOffset1 = 0;
                }

                // Block 2 need bumped?
                if(psBlock2->tSize == tOffset2)
                {
                    // Next block
                    psBlock2 = psBlock2->psNext;
                    tOffset2 = 0;
                }

                // Check block's edge conditions
                if (psBlock1 == NULL && psBlock2 != NULL)
                {
                    n16Result = - psBlock2->pacString[tOffset2];
                }
                else if (psBlock1 != NULL && psBlock2 == NULL)
                {
                    n16Result = psBlock1->pacString[tOffset1];
                }
            }

        } while((n16Result == 0) &&
                 (psBlock1 != NULL) && (psBlock2 != NULL));

        // Re-condition the result if the binary result is expected.
        if((bBinary == TRUE) && (n16Result != 0))
        {
            // Not equal
            n16Result = 1;
        }
    }
    else
    {
        // No chance of being equal and all the caller wants to know
        // is if they are equal or not so just return 1 to indicate they are
        // not equal.
        n16Result = 1;
    }

    return n16Result;
}

/*****************************************************************************
*
*    cDecodeSymbol
*
*    This utility function decodes two hex symbols into one char
*
* Inputs:
*
*   pcSrcStr - String which contains two hex symbols
*
* Outputs:
*
*   Char symbol. '\0' will be returned in case of any error
*
*****************************************************************************/
static char cDecodeSymbol(
    const char *pcSrcStr
        )
{
    size_t tStrLen = 0;
    char cResult = '\0';

    tStrLen = strlen(pcSrcStr);
    if (tStrLen >= STRING_ENCODED_SEQUENCE_LEN)
    {
        signed char tMajorSym = '\0';
        signed char tMinorSym = '\0';

        // Decode single symbols
        tMajorSym = cHexDigitToChar(*(pcSrcStr));
        tMinorSym = cHexDigitToChar(*(pcSrcStr + 1));

        if ((tMajorSym >= 0) && (tMinorSym >= 0))
        {
            cResult = (tMajorSym << 4) | tMinorSym;
        }
        else
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                STRING_OBJECT_NAME": incorrect encoded sequence hex digits"
                    );
        }
    }

    return cResult;
}

/*****************************************************************************
*
*       pacCreatorText
*
* This is a local function which simply maps an enumerated type to
* a textual representation for formatting the enumerated type.
*
*****************************************************************************/
static const char *pacCreatorText(
    STRING_CREATOR_ENUM eCreator
        )
{
    const char *pacReturnString;

    switch (eCreator)
    {
        case STRING_CREATOR_SMS:
            pacReturnString = MACRO_TO_STRING(STRING_CREATOR_SMS);
        break;

        case STRING_CREATOR_APPLICATION:
            pacReturnString = MACRO_TO_STRING(STRING_CREATOR_APPLICATION);
        break;

        default:
            pacReturnString = "UKNOWN";
        break;
    }

    return pacReturnString;
}

/*****************************************************************************
*
*   tComputeHashCode
*
*   This functions computes an integer based hash code for a string of
*   characters. A hash code is a way of computing a small (32-bit) digest
*   numeric key from an array of bytes. The numeric key itself is meaningless
*   and the hash code functions for computing them can look a bit strange.
*   However, when you go to look for something, you can do the same digest
*   calculation on the long alphabetic key you are looking for, and no matter
*   how bizarre an algorithm you used, you will calculate the same hash code,
*   and will be able to look up numerically with it. Of course there is
*   always the possibility two different arrays will have the same digest
*   hash code. However, even then, all is not lost; as the two strings can
*   be explicitly compared. The end result is it greatly narrows down the
*   search possibilities, hence speeding it up.
*
*****************************************************************************/
static N32 n32ComputeHashCode (
    const STRING_BLOCK_STRUCT *psBlock
        )
{
    N32 n32Result = 0;
    size_t tIndex;

    // Verify inputs, compute if provided
    if(psBlock != NULL)
    {
        // Compute hash code.
        // The hash code h of a string s of length n is calculated as
        // h = s[0]*31^n-1 + s[1]*31^n-2 + ... + s[n-1]
        //
        // Simple implementation would look like this:
        //
        //  n = strlen(pacString);
        //  for (i = 0; i < n; i++) h = 31 * h + pacString[i];
        //
        // But a more efficient implementation looks like this:
        do
        {
            char *pSymbol = psBlock->pacString;

            for (tIndex = 0; tIndex < psBlock->tSize; ++tIndex, ++pSymbol)
            {
                n32Result = STRING_HASH_MULTIPLIER * n32Result + (*pSymbol);
            }

            // Move on to the next block
            psBlock = psBlock->psNext;

        } while (psBlock != NULL);
    }

    return n32Result;
}

/*****************************************************************************
*
*   vStatistics
*
* Iterates the blocks belonging to a STRING object computing various
* statistics returning them to the caller.
*
*****************************************************************************/
static void vStatistics (
    const STRING_BLOCK_STRUCT *psBlock,
    size_t *ptSize,
    size_t *ptCapacity,
    size_t *ptNumBlocks
        )
{
    size_t tSize, tCapacity, tNumBlocks;

    // Load result pointers as needed/requested
    if(ptSize == NULL) // Do they want it?
    {
        // No, use local
        ptSize = &tSize;
    }
    if(ptCapacity == NULL) // Do they want it?
    {
        // No, use local
        ptCapacity = &tCapacity;
    }
    if(ptNumBlocks == NULL) // Do they want it?
    {
        // No, use local
        ptNumBlocks = &tNumBlocks;
    }

    // Initialize statistics
    *ptSize = 0;
    *ptCapacity = 0;
    *ptNumBlocks = 0;

    // Loop through each block and sum statistics
    do
    {
        // Compute statistics
        *ptSize += psBlock->tSize;
        *ptCapacity += psBlock->tCapacity;
        (*ptNumBlocks)++;

        // Proceed to next block
        psBlock = psBlock->psNext;

    } while(psBlock != NULL);

    return;
}

/*****************************************************************************
*
*   tCopyCStrLocal
*
* Copies a null-terminated CStr into a STRING's blocks expanding the
* destination STRING object as necessary. If bOverwirte = TRUE we overwrite
* the existing string and return the number of chars written. If FALSE we
* append to the existing string and return the number of chars added to
* the string.
*
* The return BOOLEAN indicates TRUE if the original string was modified
* and FALSE indicates the string was not modified or an error occurred.
*
*****************************************************************************/
static BOOLEAN bCopyCStrLocal (
    STRING_OBJECT hString,
    const char *pacSrc,
    size_t *ptCopied,
    BOOLEAN bOverwrite
        )
{
    STRING_OBJECT_STRUCT *psObj =
        (STRING_OBJECT_STRUCT *)hString;
    STRING_BLOCK_STRUCT *psBlock;
    BOOLEAN bOwner;
    size_t tSrcLen, tOffset, tCopied;

    // Verify inputs. Source pointer must all be valid.
    if(pacSrc == NULL)
    {
        // Error!
        return FALSE;
    }

    // If a return value pointer not provided, use local one
    if(ptCopied == NULL)
    {
        ptCopied = &tCopied;
    }

    // Initialize number of chars copied
    *ptCopied = 0;

    // Determine if the caller owns this resource
    bOwner = SMSO_bOwner((SMS_OBJECT)hString);
    if(bOwner == FALSE)
    {
        // Error!
        return FALSE;
    }

    // Verify we can actually modify this STRING object (i.e. it is not
    // a constant string). If it is a constant string the string pointer
    // would not be part of the STRING object itself.
    if(psObj->sBlock.pacString != (char*)(psObj + 1))
    {
        // Error!
        return FALSE;
    }

    // Always start at the first block and character
    psBlock = &psObj->sBlock;
    tOffset = 0;

    // Determine how many characters need to be copied in.
    tSrcLen = strlen(pacSrc);

    // What type of copy? Overwrite or Append?
    if(bOverwrite == TRUE)
    {
        // Check if input string is different than target string
        // If not, do nothing.
        if(n16CompareCStr(pacSrc, 0, hString) == 0)
        {
            // Nothing changed, do nothing
            return FALSE;
        }

        // Copy CStr into destination STRING starting from first
        // block, first char.
    }
    else // Append
    {
        // Appending anything to a STRING always results in a change
        // as long as the source length is > 0
        if(tSrcLen > 0)
        {
            // We need to find the place we should append to.
            do
            {
                // Is this the last block?
                if(psBlock->psNext == NULL)
                {
                    if(psBlock->tSize >= psBlock->tCapacity) // block is full
                    {
                        // Block is full, but no more blocks exist. That's ok
                        // we'll allocate more later. Indicate we should
                        // append after this block.
                        tOffset = psBlock->tCapacity;
                    }
                    else
                    {
                        // Append after this block and it's last character
                        tOffset = psBlock->tSize;
                    }
                    break;
                }
                else
                {
                    // Block is not the last, point to next block
                    psBlock = psBlock->psNext;
                }

            } while(TRUE);
        }
        else
        {
            // There is nothing to do!
            return FALSE;
        }
    }

    // If we get here we are always copying something into this string

    // Copy string into the object...overwriting/appending the existing
    // contents and expanding as necessary. Any remaining (unused) blocks
    // will be left alone, just marked as unused.
    do
    {
        // Determine how much needs to be written and set the block size
        psBlock->tSize = (tSrcLen > (psBlock->tCapacity - tOffset)) ?
            psBlock->tCapacity : (tSrcLen + tOffset);

        // Copy source character array into destination block either up
        // to the capacity of this block or the incoming character array size.
        strncpy(psBlock->pacString + tOffset, pacSrc,
            psBlock->tSize - tOffset);

        // Increment total number of characters copied
        *ptCopied += (psBlock->tSize - tOffset);

        // Increment the source pointer indicating what has been consumed.
        pacSrc += (psBlock->tSize - tOffset);

        // Decrement source length left to write
        tSrcLen -= (psBlock->tSize - tOffset);

        // After the first round, tOffset is always zero
        tOffset = 0;

        // Do we have more to write? If not we simply iterate the remaining
        // blocks marking them as unused (tSize = 0).
        if(tSrcLen > 0)
        {
            // We still have more to write.
            // Check if we have another destination block to use
            if(psBlock->psNext == NULL)
            {
                // Nope! No more blocks, so we need to allocate another.
                // Allocate the block we need.
                psBlock->psNext = (struct string_block_struct *)
                    SMSO_hCreate(
                        STRING_OBJECT_NAME":Block",
                        sizeof(STRING_BLOCK_STRUCT) + tSrcLen,
                        (SMS_OBJECT)psObj,
                        FALSE
                            );
                if(psBlock->psNext != NULL)
                {
                    // Initialize this block
                    psBlock->psNext->tCapacity = tSrcLen;
                    psBlock->psNext->tSize = 0;
                    psBlock->psNext->pacString = (char *)(psBlock->psNext + 1);
                    psBlock->psNext->psNext = NULL;
                }
            }

            // Proceed to the next block
            psBlock = psBlock->psNext;
        }
        else
        {
            // no more to copy

            // Prune remaining blocks (if any exist)
            if(psBlock->psNext != NULL)
            {
                vDestroyBlocks(psObj, psBlock->psNext);
                psBlock->psNext = NULL;
            }
            break;
        }

    } while(psBlock != NULL);

    // Compute hash code of copied/appended string
    psObj->n32HashCode = n32ComputeHashCode(&psObj->sBlock);

    return TRUE;
}

/*****************************************************************************
*
*   tCopyLocal
*
* Copies a STRING object into another STRING object expanding the destination
* STRING object as necessary. If bOverwirte = TRUE we overwrite the existing
* string, if FALSE we append to the existing string.
*
*****************************************************************************/
static size_t tCopyLocal (
    STRING_OBJECT hDstString,
    STRING_OBJECT hSrcString,
    BOOLEAN bOverwrite
        )
{
    STRING_OBJECT_STRUCT *psSrcObj =
        (STRING_OBJECT_STRUCT *)hSrcString;
    STRING_OBJECT_STRUCT *psDstObj =
        (STRING_OBJECT_STRUCT *)hDstString;
    STRING_BLOCK_STRUCT *psSrcBlock, *psDstBlock;
    size_t tSrcIndex, tDstIndex, tCopied = 0;
    BOOLEAN bOwner;

    // Determine if the caller owns this resource
    bOwner = SMSO_bOwner((SMS_OBJECT)hSrcString);
    if(bOwner == FALSE)
    {
        // Error!
        return 0;
    }

    // Determine if the caller owns this resource
    bOwner = SMSO_bOwner((SMS_OBJECT)hDstString);
    if(bOwner == FALSE)
    {
        // Error!
        return 0;
    }

    // Verify we can actually modify this destination STRING object (i.e. it
    // is not a constant string). If it is a constant string the string pointer
    // would not be part of the STRING object itself.
    if(psDstObj->sBlock.pacString != (char*)(psDstObj + 1))
    {
        // Error!
        return 0;
    }

    // Initialize the starting point for both
    // overwrite & append (append may adjust this)
    // Start copy from beginning of source STRING and
    // beginning of destination STRING
    // (first block, first char) of each
    psSrcBlock = &psSrcObj->sBlock;
    tSrcIndex = 0;
    psDstBlock = &psDstObj->sBlock;
    tDstIndex = 0;

    // Are we appending?
    if(bOverwrite == FALSE)
    {
        // Yes!
        size_t tSrcSize;

        // Get the size of the source string
        // (we'll base any new allocations on this size)
        vStatistics(psSrcBlock, &tSrcSize, NULL, NULL);

        // Now, find the destination block onto
        // which we want to append the source string

        // We need to find the place we should append to.
        do
        {
            // Is this the last block?
            if(psDstBlock->psNext == NULL)
            {
                if(psDstBlock->tSize >= psDstBlock->tCapacity) // block is full
                {
                    // We need to allocate a new block
                    // big enough to store the source string
                    psDstBlock->psNext = (struct string_block_struct *)
                        SMSO_hCreate(
                            STRING_OBJECT_NAME":Block",
                            sizeof(STRING_BLOCK_STRUCT) + tSrcSize,
                            (SMS_OBJECT)psDstObj,
                            FALSE
                                );
                    if(psDstBlock->psNext != NULL)
                    {
                        // Initialize this block
                        psDstBlock->psNext->tCapacity = tSrcSize;
                        psDstBlock->psNext->tSize = 0;
                        psDstBlock->psNext->pacString =
                            (char *)(psDstBlock->psNext + 1);
                        psDstBlock->psNext->psNext = NULL;

                        // Point to next block
                        psDstBlock = psDstBlock->psNext;
                    }
                    else
                    {
                        return 0;
                    }
                }
                else // We have space in this block
                {
                    // Start from the end of the
                    // data in the block
                    tDstIndex = psDstBlock->tSize;
                }
                break;
            }
            else
            {
                // Block is not the last, point to next block
                psDstBlock = psDstBlock->psNext;
            }

        } while(TRUE);

    }

    // Copy the object into another...overwriting/appending the existing
    // contents and expanding as necessary. Any remaining (unused) blocks will
    // be left alone, just marked as unused.
    do
    {
        // Determine how much needs to be written and set the block size
        psDstBlock->tSize =
            ((psSrcBlock->tSize - tSrcIndex) >
                (psDstBlock->tCapacity - tDstIndex)) ?
                    psDstBlock->tCapacity : psSrcBlock->tSize - tSrcIndex + tDstIndex;

        // Copy source characters into destination block either up
        // to the capacity of this block or the size of the source block.
        strncpy(psDstBlock->pacString + tDstIndex,
            psSrcBlock->pacString + tSrcIndex,
            psDstBlock->tSize - tDstIndex);

        // Increment total number of characters copied
        tCopied += (psDstBlock->tSize - tDstIndex);

        // Increment source index (consumed)
        tSrcIndex += (psDstBlock->tSize - tDstIndex);

        // Are we at the end of the source block?
        if(psSrcBlock->tSize == tSrcIndex)
        {
            // Next source block
            psSrcBlock = psSrcBlock->psNext;
            tSrcIndex = 0;
        }

        // After the first round, tDstOffset is always zero
        tDstIndex = 0;

        // Do we have more to write? If not we simply iterate the remaining
        // blocks marking them as unused (tSize = 0).
        if(psSrcBlock != NULL)
        {
            if((psSrcBlock->tSize - tSrcIndex) > 0)
            {
                // Compute the remaining amount we have to copy.

                // We still have more to write.
                // Check if we have another destination block to use
                if(psDstBlock->psNext == NULL)
                {
                    size_t tSizeToCopy, tAdditionalSize;

                    // Yep! We need to allocate a new one. We'll allocate
                    // it with the size we need to complete the copy.

                    // Determine how much more we need. Use existing functions
                    // to determine this.

                    // How much do we need to copy?
                    tSizeToCopy = tSize(hSrcString);

                    // What's left?
                    tAdditionalSize = tSizeToCopy - tCopied;

                    // Nope! No more blocks, so we need to allocate another.
                    // Allocate the block we need.
                    psDstBlock->psNext = (struct string_block_struct *)
                        SMSO_hCreate(
                            STRING_OBJECT_NAME":Block",
                            sizeof(STRING_BLOCK_STRUCT) + tAdditionalSize,
                            (SMS_OBJECT)psDstObj,
                            FALSE
                                );
                    if(psDstBlock->psNext != NULL)
                    {
                        // Initialize this block
                        psDstBlock->psNext->tCapacity = tAdditionalSize;
                        psDstBlock->psNext->tSize = 0;
                        psDstBlock->psNext->pacString =
                            (char *)(psDstBlock->psNext + 1);
                        psDstBlock->psNext->psNext = NULL;
                    }
                }
            }

            // Proceed to the next destination block
            psDstBlock = psDstBlock->psNext;
        }
        else
        {
            // no more to copy

            // Prune remaining blocks (if any exist)
            if(psDstBlock->psNext != NULL)
            {
                vDestroyBlocks(psDstObj, psDstBlock->psNext);
                psDstBlock->psNext = NULL;
            }
            break;
        }

    } while((psDstBlock != NULL) && (psSrcBlock != NULL));

    // Compute hash-code for copied string, should match original if we
    // performed an overwrite.
    psDstObj->n32HashCode = n32ComputeHashCode(&psDstObj->sBlock);
    if((bOverwrite == TRUE) && (psDstObj->n32HashCode != psSrcObj->n32HashCode))
    {
        // Error! Something bad happened.
        SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
            STRING_OBJECT_NAME": Hash-code does not match original.");

        // Just make destination zero-length & re-compute hash code.
        psDstObj->sBlock.tSize = 0;
        psDstObj->n32HashCode = n32ComputeHashCode(&psDstObj->sBlock);
    }

    return tCopied;
}

/*****************************************************************************
*
*   bHasSpecialSymbols
*
*****************************************************************************/
static BOOLEAN bHasSpecialSymbols (
    const STRING_BLOCK_STRUCT *psBlock
        )
{
    BOOLEAN bResult = FALSE;

    if (psBlock != NULL)
    {
        size_t tIndex;
        const char *pcSymbol = psBlock->pacString;
        for (tIndex = 0;
             (tIndex < psBlock->tSize) && (bResult == FALSE);
             ++tIndex, ++pcSymbol)
        {
            bResult = STRING_IS_SPECIAL_SYMBOL(*pcSymbol);
        }
    }

    return bResult;
}

/*****************************************************************************
*
*   cHexDigitToChar
*
*   Inputs:
*      cHexDigit - hex digit from 0 to 9, from a(A) to f(F)
*
*   Outputs:
*      Integer hex digit representation or -1 if passed symbol is not digit
*
*****************************************************************************/
static signed char cHexDigitToChar (
    const char cHexDigit
        )
{
    signed char tResult = -1;

    if (isxdigit(cHexDigit))
    {
        if ((cHexDigit >= '0') && (cHexDigit <= '9'))
        {
            tResult = cHexDigit - '0';
        }
        else if ((cHexDigit >= 'a') && (cHexDigit <= 'f'))
        {
            tResult = 0x0A + cHexDigit - 'a';
        }
        else if ((cHexDigit >= 'A') && (cHexDigit <= 'F'))
        {
            tResult = 0x0A + cHexDigit - 'A';
        }
    }

    return tResult;
}

/*****************************************************************************
*
*   vDestroyBlocks
*
*****************************************************************************/
static void vDestroyBlocks (
    STRING_OBJECT_STRUCT *psObj,
    STRING_BLOCK_STRUCT *psStartBlock
        )
{
    BOOLEAN bDestroyBlock;
    STRING_BLOCK_STRUCT *psBlock = psStartBlock;

    // If start block is NULL just use the first one
    if(psBlock == NULL)
    {
        // Use first from the object
        psBlock = &psObj->sBlock;
    }

    // Loop through and destroy blocks as needed
    do
    {
        STRING_BLOCK_STRUCT *psNextBlock;

        // Is this not part of the STRING object?
        bDestroyBlock =
            (&psObj->sBlock == psBlock ? FALSE : TRUE);

        // Remember next block, re-init this one
        psNextBlock = psBlock->psNext;
        psBlock->psNext = NULL;

        // Initialize block size
        psBlock->tSize = 0;

        // Destroy it?
        if(bDestroyBlock == TRUE)
        {
            // Clear capacity
            psBlock->tCapacity = 0;

            // Clear pointer
            psBlock->pacString = NULL;

            // Destroy this block
            SMSO_vDestroy((SMS_OBJECT)psBlock);
        }

        // Point to next block
        psBlock = psNextBlock;

    } while(psBlock != NULL);

    return;
}

#ifdef SUPPORT_CUNIT
#include <string_obj.cunit>
#endif
