/******************************************************************************/
/*                    Copyright (c) Sirius XM Radio, Inc.                     */
/*                            All Rights Reserved                             */
/*            Licensed Materials - Property of Sirius XM Radio, Inc.          */
/******************************************************************************/
/*******************************************************************************
 *
 * DESCRIPTION
 *
 *  This module contains the Object:STOCK_TICKER_MGR implementation for the
 *  Simple Module Services (SMS)
 *
 ******************************************************************************/

#include "sms_api.h"
#include "sms_obj.h"
#include "dsrl_entry_obj.h"

#include "_stock_ticker_mgr_obj.h"
#include "stock_ticker_mgr_obj.h"
#include "stock_ticker_db_constants.h"
#include "dataservice_mgr_impl.h"

#include "sms_api_debug.h"
static const char *gpacThisFile = __FILE__;

/*****************************************************************************
 *                           PUBLIC FUNCTIONS
 *****************************************************************************/
/*****************************************************************************
*
*   hStart
*
*   This function will create all the basics needed for this service to
*   operate.  However, all initial processing to actually get this service
*   running is done at a later time.
*
*****************************************************************************/
static STOCK_TICKER_SERVICE_OBJECT hStart (
    const char *pacSRHDriverName,
    DATASERVICE_EVENT_MASK tEventRequestMask,
    DATASERVICE_EVENT_CALLBACK vEventCallback,
    void *pvAppEventCallbackArg,
    DATASERVICE_OPTIONS_STRUCT const *psOptions
        )
{
    STOCK_TICKER_MGR_OBJECT_STRUCT *psObj =
            (STOCK_TICKER_MGR_OBJECT_STRUCT *)STOCK_TICKER_SERVICE_INVALID_OBJECT;
    BOOLEAN bOk = FALSE;
    DATASERVICE_CREATE_STRUCT sCreate;
    STOCK_TICKER_RETURN_CODE_ENUM eReturnCode = STOCK_TICKER_RETURN_CODE_ERROR;

    printf(STOCK_TICKER_MGR_OBJECT_NAME": Starting service\n");

    do
    {
        // Populate our data service creation structure
        DATASERVICE_IMPL_vInitCreateStruct(&sCreate);
        sCreate.pacSRHDriverName = pacSRHDriverName;
        sCreate.pacServiceObjectName = STOCK_TICKER_MGR_OBJECT_NAME;
        sCreate.tServiceObjectSize = sizeof(STOCK_TICKER_MGR_OBJECT_STRUCT);
        sCreate.tDataID = (DATASERVICE_ID)GsStockTickerIntf.tDSI;

        // Suggest an OTA buffer size
        sCreate.tSuggestedOTABufferByteSize = GsStockTickerIntf.tOTABufferByteSize;

        // Configure the data service's static event attributes
        sCreate.vEventCallback = vEventHandler;
        sCreate.tEventRequestMask = (
            DATASERVICE_EVENT_ALL |
            DATASERVICE_INTERNAL_EVENT_DSRL );

        // Ask the data service manager controller to
        // create our manager object and do everything
        // necessary to create the underlying objects required
        // in order to support this service
        psObj = (STOCK_TICKER_MGR_OBJECT_STRUCT *)
            DATASERVICE_IMPL_hCreateNewService( &sCreate );
        if (psObj == NULL)
        {
            // Can't create the service, fail out!
            break;
        }

        // Init parser data
        psObj->hParser = STOCK_TICKER_INTERFACE_INVALID_OBJECT;

        // Initialize asynchronous update configuration
        SMSU_vInitialize(
            &psObj->sEvent,
            psObj,
            DATASERVICE_EVENT_ALL,
            tEventRequestMask,
            (SMSAPI_OBJECT_EVENT_CALLBACK)vEventCallback,
            pvAppEventCallbackArg
                );

        // Initialize facing objects
        eReturnCode = eInitAppFacingObject(psObj);
        if (eReturnCode != STOCK_TICKER_RETURN_CODE_SUCCESS)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                   STOCK_TICKER_MGR_OBJECT_NAME
                   ": failed to initialize facing objects (%s)",
                   GsStockTickerMgrIntf.pacGetReturnCodeName(eReturnCode)
                    );
            break;
        }

        // Prepare options structure
        OSAL.bMemSet(&psObj->psAppObj->sOptionValues, 0,
                     sizeof(psObj->psAppObj->sOptionValues));
        // grab variable arguments (pop)
        bOk = DATASERVICE_IMPL_bProcessOptions(
            STOCK_TICKER_SUPPORTED_OPTIONS,
            psOptions, &psObj->psAppObj->sOptionValues
                );
        if (bOk == FALSE)
        {
            // Bad options!
            break;
        }

        // The service may now start
        bOk = DATASERVICE_IMPL_bStart((DATASERVICE_IMPL_HDL)psObj);
        if (bOk != TRUE)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    STOCK_TICKER_MGR_OBJECT_NAME": unable to create service");
            break;
        }

        return (STOCK_TICKER_SERVICE_OBJECT)psObj;
    } while (FALSE);

    // Error!
    if (psObj != NULL)
    {
        vUninitObject( psObj, TRUE );
        DATASERVICE_IMPL_vDestroy((DATASERVICE_IMPL_HDL)psObj);
    }

    return STOCK_TICKER_SERVICE_INVALID_OBJECT;
}

/*****************************************************************************
*
*   bStockSymbolsUpdated
*
*****************************************************************************/
static BOOLEAN bStockSymbolsUpdated(
    STOCK_TICKER_SERVICE_OBJECT hStocksService,
    UN32* pun32ContentVersion
        )
{
    BOOLEAN bResult = FALSE;
    STOCK_TICKER_APP_OBJECT_STRUCT *psAppObj = NULL;

    psAppObj = psGetAppFacingObject(hStocksService);
    if (psAppObj != NULL)
    {
        bResult = psAppObj->sDB.bSymbolsUpdated;

        // Extract version caller asked it
        if (pun32ContentVersion != NULL)
        {
            *pun32ContentVersion = psAppObj->sDB.un32ContentVersion;
        }

        SMSO_vUnlock((SMS_OBJECT) psAppObj);
    }

    return bResult;
}

/*****************************************************************************
*
*   eIterateStockSymbols
*
*****************************************************************************/
static SMSAPI_RETURN_CODE_ENUM eIterateStockSymbols (
    STOCK_TICKER_SERVICE_OBJECT hStocksService,
    STOCK_TICKER_SYMBOL_ITERATOR bIterator,
    void *pvIteratorArg
       )
{
    SMSAPI_RETURN_CODE_ENUM eReturnCode = SMSAPI_RETURN_CODE_ERROR;
    STOCK_TICKER_APP_OBJECT_STRUCT *psAppObj = NULL;
    STOCK_TICKER_DB_QUERY_RESULT_STRUCT sArg;
    BOOLEAN bOk = FALSE;

    do
    {
        // Check input
        if (bIterator == NULL)
        {
            eReturnCode = SMSAPI_RETURN_CODE_INVALID_INPUT;
            break;
        }

        // Lock AppFacing object
        psAppObj = psGetAppFacingObject(hStocksService);
        if (psAppObj == NULL)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    STOCK_TICKER_MGR_OBJECT_NAME": failed to get AppFacing object");
            eReturnCode = SMSAPI_RETURN_CODE_NOT_OWNER;
            break;
        }

        ///////////////////////////////////////////////////////////////////////
        // Init argument
        OSAL.bMemSet(&sArg, 0, sizeof(sArg));
        sArg.sInput.bIterator = bIterator;
        sArg.sInput.pvIteratorArg = (void*)pvIteratorArg;
        sArg.sInput.hService = pvIteratorArg;
        sArg.sInput.psAppObj = psAppObj;
#if SMS_DEBUG == 1
        sArg.sOutput.tRowCount = 0;
#endif

        // Perform the SQL query and process the result
        // (it will provide us with a data row)
        bOk = SQL_INTERFACE.bQuery(
            psAppObj->sDB.hSQLConnection,
            STOCK_TICKER_SELECT_DB_ALL_SYMBOLS,
            bStockMsgDescIterateDBProcessResult,
            (void*) &sArg
                );
        if (bOk != TRUE)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                STOCK_TICKER_MGR_OBJECT_NAME": failed to make query %s",
                STOCK_TICKER_SELECT_DB_ALL_SYMBOLS
                    );
            break;
        }

#if SMS_DEBUG == 1
        printf(STOCK_TICKER_MGR_OBJECT_NAME
            ": processed %d record(s) as Stock Symbols\n",
            sArg.sOutput.tRowCount
                );
#endif // SMS_DEBUG==1

        // Say that there is no updates.
        psAppObj->sDB.bSymbolsUpdated = FALSE;

        eReturnCode = SMSAPI_RETURN_CODE_SUCCESS;

    } while (FALSE);

    if (psAppObj != NULL)
    {
        SMSO_vUnlock((SMS_OBJECT)psAppObj);
    }

    return eReturnCode;
}

/*****************************************************************************
*
*   hDataProviderInfo
*
*****************************************************************************/
static STRING_OBJECT hDataProviderInfo (
    STOCK_TICKER_SERVICE_OBJECT hStocksService
       )
{
    STRING_OBJECT hResult = STRING_INVALID_OBJECT;
    STOCK_TICKER_APP_OBJECT_STRUCT *psAppObj;

    // Lock AppFacing object
    psAppObj = psGetAppFacingObject(hStocksService);
    if (psAppObj != NULL)
    {
        // Populate the info string
        hResult = psAppObj->sProviderInfoCtrl.hDataProviderInfoText;

        SMSO_vUnlock((SMS_OBJECT)psAppObj);
    }
    else
    {
        SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                STOCK_TICKER_MGR_OBJECT_NAME": failed to get AppFacing objectpbject");
    }

    return hResult;
}

/*****************************************************************************
*
*   ePayloadProcessStarted
*
*****************************************************************************/
static STOCK_TICKER_RETURN_CODE_ENUM ePayloadProcessStarted (
    STOCK_TICKER_SERVICE_OBJECT hService
       )
{
    STOCK_TICKER_RETURN_CODE_ENUM eResult = STOCK_TICKER_RETURN_CODE_SUCCESS;
    STOCK_TICKER_APP_OBJECT_STRUCT *psAppObj;

    puts(STOCK_TICKER_MGR_OBJECT_NAME": data processing START requested");

    psAppObj = psOwnedAppFacingObject(hService);
    if (psAppObj == NULL)
    {
        SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                STOCK_TICKER_MGR_OBJECT_NAME": incorrect calling context"
                    );
        eResult = STOCK_TICKER_RETURN_CODE_NOT_OWNER;
    }

    DATASERVICE_IMPL_vLog(STOCK_TICKER_MGR_OBJECT_NAME
        ": Data processing started with code %s\n",
        GsStockTickerMgrIntf.pacGetReturnCodeName(eResult)
            );

    return eResult;
}

/*****************************************************************************
*
*   ePayloadProcessCommited
*
*****************************************************************************/
static STOCK_TICKER_RETURN_CODE_ENUM ePayloadProcessCommited (
    STOCK_TICKER_SERVICE_OBJECT hService
       )
{
    STOCK_TICKER_RETURN_CODE_ENUM eResult = STOCK_TICKER_RETURN_CODE_ERROR;
    STOCK_TICKER_APP_OBJECT_STRUCT *psAppObj = NULL;

    DATASERVICE_IMPL_vLog(STOCK_TICKER_MGR_OBJECT_NAME": data processing COMMIT\n");
    printf(STOCK_TICKER_MGR_OBJECT_NAME": data processing COMMIT requested\n");

    // Check owner
    psAppObj = psOwnedAppFacingObject(hService);
    if (psAppObj == NULL)
    {
        eResult = STOCK_TICKER_RETURN_CODE_NOT_OWNER;
    }
    else
    {
        eResult = eProcessServiceData(psAppObj);
        if (eResult != STOCK_TICKER_RETURN_CODE_SUCCESS)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                STOCK_TICKER_MGR_OBJECT_NAME": failed to process service data (%S)",
                GsStockTickerMgrIntf.pacGetReturnCodeName(eResult)
                    );
        }

        // Let's start or stop the timer
        vExpirationTimerUpdate(psAppObj, STOCK_TICKER_TIMER_PERIOD);

        // Simple end transaction if it was opened
        eDBEndTransaction(psAppObj);
    }

    DATASERVICE_IMPL_vLog(STOCK_TICKER_MGR_OBJECT_NAME": Completed COMMIT with code %s\n",
        GsStockTickerMgrIntf.pacGetReturnCodeName(eResult)
            );

    return eResult;
}

/*****************************************************************************
*
*   eUpdate
*
*****************************************************************************/
static STOCK_TICKER_RETURN_CODE_ENUM eUpdate (
    STOCK_TICKER_SERVICE_OBJECT hService,
    STOCK_TICKER_PLUGIN_INTERFACE_ARG_STRUCT *psArg
        )
{
    STOCK_TICKER_RETURN_CODE_ENUM eResult = STOCK_TICKER_RETURN_CODE_ERROR;
    STOCK_TICKER_APP_OBJECT_STRUCT *psAppObj = NULL;

    puts(STOCK_TICKER_MGR_OBJECT_NAME": Update requested from Protocol");

    do
    {
        // Check input
        if (psArg == NULL)
        {
            eResult = STOCK_TICKER_RETURN_CODE_BAD_INPUT;
            break;
        }

        psAppObj = psOwnedAppFacingObject(hService);
        if (psAppObj == NULL)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    STOCK_TICKER_MGR_OBJECT_NAME": incorrect calling context"
                        );
            eResult = STOCK_TICKER_RETURN_CODE_NOT_OWNER;
            break;
        }

        // Initial default value for return code
        eResult = STOCK_TICKER_RETURN_CODE_SUCCESS;

        // Process data by types
        switch (psArg->eType)
        {
            case STOCK_TICKER_MSG_TYPE_STOCK_SYMBOL:
            {
                eResult = eUpdateSymbol(psAppObj,
                                &psArg->uData.sSymbolMsg
                                    );
            }
            break;
            case STOCK_TICKER_MSG_TYPE_STOCK_VALUE:
            case STOCK_TICKER_MSG_TYPE_STOCK_VALUE_EX:
            {
                eResult = eUpdateSymbolValue(psAppObj,
                                &psArg->uData.sValueMsg,
                                NULL, TRUE, psArg->bDoTouch
                                    );
            }
            break;
            case STOCK_TICKER_MSG_TYPE_DATA_PROVIDER_MSG:
            {
                eResult = eUpdateProviderData(psAppObj,
                                &psArg->uData.sProviderInfo
                                    );
            }
            break;
            case STOCK_TICKER_MSG_TYPE_INVALID:
            case STOCK_TICKER_MSG_TYPE_MAX:
            {
                eResult = STOCK_TICKER_RETURN_CODE_UNKNOWN_MSG_TYPE;
            }
            break;
        }
    } while (FALSE);

    if (eResult != STOCK_TICKER_RETURN_CODE_SUCCESS)
    {
        DATASERVICE_IMPL_vLog(STOCK_TICKER_MGR_OBJECT_NAME": failed to perform update with code %s\n",
            GsStockTickerMgrIntf.pacGetReturnCodeName(eResult)
                );
    }

    return eResult;
}

/*****************************************************************************
*
*   eIsPayloadProcessNeeded
*
*****************************************************************************/
static STOCK_TICKER_RETURN_CODE_ENUM eIsPayloadProcessNeeded (
    STOCK_TICKER_SERVICE_OBJECT hService,
    STOCK_TICKER_MSG_TYPE_ENUM eType,
    OSAL_CRC_RESULT tHash
        )
{
    STOCK_TICKER_RETURN_CODE_ENUM eResult;
    STOCK_TICKER_APP_OBJECT_STRUCT *psAppObj;

    psAppObj = psOwnedAppFacingObject(hService);
    if (psAppObj == NULL)
    {
        SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                STOCK_TICKER_MGR_OBJECT_NAME": incorrect calling context"
                    );
        eResult = STOCK_TICKER_RETURN_CODE_NOT_OWNER;
    }
    else
    {
        // Update hash data
        eResult = eStockMsgHashCacheUpdate(psAppObj, eType, tHash);
        if (eResult == STOCK_TICKER_RETURN_CODE_NOTHING_TO_UPDATE)
        {
            if ((eType == STOCK_TICKER_MSG_TYPE_STOCK_VALUE) ||
                (eType == STOCK_TICKER_MSG_TYPE_STOCK_VALUE_EX))
            {
                // Looks like the data we've got already populated to
                // the application, but we need to touch to keep it in the
                // list at least util next timeout case
                eResult = STOCK_TICKER_RETURN_CODE_NEED_TO_TOUCH;
            }
            else
            {
                eResult = STOCK_TICKER_RETURN_CODE_FALSE;
            }
        }
        else if (eResult != STOCK_TICKER_RETURN_CODE_SUCCESS)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                STOCK_TICKER_MGR_OBJECT_NAME
                ": failed update hash (%s)",
                GsStockTickerMgrIntf.pacGetReturnCodeName(eResult)
                    );
        }
    }

    return eResult;
}

/*****************************************************************************
*
*   eIsMessageNeeded
*
*****************************************************************************/
static STOCK_TICKER_RETURN_CODE_ENUM eIsMessageNeeded (
    STOCK_TICKER_SERVICE_OBJECT hService,
    STOCK_TICKER_MSG_TYPE_ENUM eType,
    STOCK_TICKER_PLUGIN_PRECHECK_INTERFACE_ARG_STRUCT const *puData
        )
{
    STOCK_TICKER_RETURN_CODE_ENUM eResult = STOCK_TICKER_RETURN_CODE_ERROR;
    STOCK_TICKER_APP_OBJECT_STRUCT *psAppObj = NULL;

    do
    {
        psAppObj = psOwnedAppFacingObject(hService);
        if (psAppObj == NULL)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                STOCK_TICKER_MGR_OBJECT_NAME": incorrect calling context"
                );
            eResult = STOCK_TICKER_RETURN_CODE_NOT_OWNER;
            break;
        }

        // Check the DSRL existence since some types of
        // messages don't make any since in case of no-DSRL per
        // current requirements.
        if ((eType == STOCK_TICKER_MSG_TYPE_STOCK_SYMBOL) ||
            (eType == STOCK_TICKER_MSG_TYPE_DATA_PROVIDER_MSG))
        {
            // So, we need to process this regardless DSRLs
            eResult = STOCK_TICKER_RETURN_CODE_SUCCESS;
        }
        else if ((eType == STOCK_TICKER_MSG_TYPE_STOCK_VALUE) ||
                 (eType == STOCK_TICKER_MSG_TYPE_STOCK_VALUE_EX))
        {
            // Check DSRLs
            eResult = eDSRLDescHasToUpdate(psAppObj);
            if (eResult == STOCK_TICKER_RETURN_CODE_FALSE)
            {
                puts(STOCK_TICKER_MGR_OBJECT_NAME": no DSRLs to update");
                break;
            }
            else if (eResult != STOCK_TICKER_RETURN_CODE_SUCCESS)
            {
                SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    STOCK_TICKER_MGR_OBJECT_NAME": failed to get DSRL status (%s)",
                    GsStockTickerMgrIntf.pacGetReturnCodeName(eResult)
                        );
                break;
            }

            // Let's check the values against monitored symbols by ID if provided
            if ((eType == STOCK_TICKER_MSG_TYPE_STOCK_VALUE) &&
                (puData != NULL))
            {
                eResult =
                    eDSRLDescHasMessagesToUpdateByValues(psAppObj,
                        (STOCK_ID)puData->sValues.un16StartIndex,
                        (STOCK_ID)(puData->sValues.un16StartIndex +
                                   puData->sValues.un8Count - 1));
                if ((eResult != STOCK_TICKER_RETURN_CODE_SUCCESS) &&
                    (eResult != STOCK_TICKER_RETURN_CODE_FALSE))
                {
                    SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                        STOCK_TICKER_MGR_OBJECT_NAME
                        ": failed to check pool (%s)",
                        GsStockTickerMgrIntf.pacGetReturnCodeName(eResult)
                            );
                    break;
                }
            }
        }
        else
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                STOCK_TICKER_MGR_OBJECT_NAME
                ":don't know how to handle message type %s",
                GsStockTickerMgrIntf.pacGetMessageTypeName(eType)
                    );
            eResult = STOCK_TICKER_RETURN_CODE_UNKNOWN_MSG_TYPE;
        }
    } while (FALSE);

    return eResult;
}

/*****************************************************************************
*
*   pacGetReturnCodeName
*
*****************************************************************************/
static const char *pacGetReturnCodeName (
    STOCK_TICKER_RETURN_CODE_ENUM eCode
        )
{
    const char *pacResult = "UNKNOWN";

    switch (eCode)
    {
        case STOCK_TICKER_RETURN_CODE_SUCCESS:
            pacResult = MACRO_TO_STRING(STOCK_TICKER_RETURN_CODE_SUCCESS);
            break;
        case STOCK_TICKER_RETURN_CODE_BAD_INPUT:
            pacResult = MACRO_TO_STRING(STOCK_TICKER_RETURN_CODE_BAD_INPUT);
            break;
        case STOCK_TICKER_RETURN_CODE_FALSE:
            pacResult = MACRO_TO_STRING(STOCK_TICKER_RETURN_CODE_FALSE);
            break;
        case STOCK_TICKER_RETURN_CODE_NOT_SUPPORTED:
            pacResult = MACRO_TO_STRING(STOCK_TICKER_RETURN_CODE_NOT_SUPPORTED);
            break;
        case STOCK_TICKER_RETURN_CODE_NOT_KNOWN_HANDLE:
            pacResult = MACRO_TO_STRING(STOCK_TICKER_RETURN_CODE_NOT_KNOWN_HANDLE);
            break;
        case STOCK_TICKER_RETURN_CODE_UNKNOWN_MSG_TYPE:
            pacResult = MACRO_TO_STRING(STOCK_TICKER_RETURN_CODE_UNKNOWN_MSG_TYPE);
            break;
        case STOCK_TICKER_RETURN_CODE_NOT_ENOUGH_BUFFER:
            pacResult = MACRO_TO_STRING(STOCK_TICKER_RETURN_CODE_NOT_ENOUGH_BUFFER);
            break;
        case STOCK_TICKER_RETURN_CODE_BAD_DATA:
            pacResult = MACRO_TO_STRING(STOCK_TICKER_RETURN_CODE_BAD_DATA);
            break;
        case STOCK_TICKER_RETURN_CODE_BAD_XMAPP_HDR:
            pacResult = MACRO_TO_STRING(STOCK_TICKER_RETURN_CODE_BAD_XMAPP_HDR);
            break;
        case STOCK_TICKER_RETURN_CODE_NOT_UNIQUE:
            pacResult = MACRO_TO_STRING(STOCK_TICKER_RETURN_CODE_NOT_UNIQUE);
            break;
        case STOCK_TICKER_RETURN_CODE_ERROR:
            pacResult = MACRO_TO_STRING(STOCK_TICKER_RETURN_CODE_ERROR);
            break;
        case STOCK_TICKER_RETURN_CODE_DB_ERROR:
            pacResult = MACRO_TO_STRING(STOCK_TICKER_RETURN_CODE_DB_ERROR);
            break;
        case STOCK_TICKER_RETURN_CODE_BAD_STATE:
            pacResult = MACRO_TO_STRING(STOCK_TICKER_RETURN_CODE_BAD_STATE);
            break;
        case STOCK_TICKER_RETURN_CODE_IO_ERROR:
            pacResult = MACRO_TO_STRING(STOCK_TICKER_RETURN_CODE_IO_ERROR);
            break;
        case STOCK_TICKER_RETURN_CODE_MEMORY_ERROR:
            pacResult = MACRO_TO_STRING(STOCK_TICKER_RETURN_CODE_MEMORY_ERROR);
            break;
        case STOCK_TICKER_RETURN_CODE_NOTHING_TO_UPDATE:
            pacResult = MACRO_TO_STRING(STOCK_TICKER_RETURN_CODE_NOTHING_TO_UPDATE);
            break;
        case STOCK_TICKER_RETURN_CODE_NEED_TO_UPDATE:
            pacResult = MACRO_TO_STRING(STOCK_TICKER_RETURN_CODE_NEED_TO_UPDATE);
            break;
        case STOCK_TICKER_RETURN_CODE_NEED_TO_TOUCH:
            pacResult = MACRO_TO_STRING(STOCK_TICKER_RETURN_CODE_NEED_TO_TOUCH);
            break;
        case STOCK_TICKER_RETURN_CODE_NOT_OWNER:
            pacResult = MACRO_TO_STRING(STOCK_TICKER_RETURN_CODE_NOT_OWNER);
            break;
        case STOCK_TICKER_RETURN_CODE_BAD_SYNC_SEQ:
            pacResult = MACRO_TO_STRING(STOCK_TICKER_RETURN_CODE_BAD_SYNC_SEQ);
            break;
    }

    return pacResult;
}

/*******************************************************************************
*
*   pacGetMessageTypeName
*
*******************************************************************************/
static const char *pacGetMessageTypeName(
    STOCK_TICKER_MSG_TYPE_ENUM eType
        )
{
    const char *pacResult = "";

    switch (eType)
    {
        case STOCK_TICKER_MSG_TYPE_INVALID:
        {
            pacResult = MACRO_TO_STRING(STOCK_TICKER_MSG_TYPE_INVALID);
        }
        break;
        case STOCK_TICKER_MSG_TYPE_STOCK_SYMBOL:
        {
            pacResult = MACRO_TO_STRING(STOCK_TICKER_MSG_TYPE_STOCK_SYMBOL);
        }
        break;
        case STOCK_TICKER_MSG_TYPE_STOCK_VALUE:
        {
            pacResult = MACRO_TO_STRING(STOCK_TICKER_MSG_TYPE_STOCK_VALUE);
        }
        break;
        case STOCK_TICKER_MSG_TYPE_STOCK_VALUE_EX:
        {
            pacResult = MACRO_TO_STRING(STOCK_TICKER_MSG_TYPE_STOCK_VALUE_EX);
        }
        break;
        case STOCK_TICKER_MSG_TYPE_DATA_PROVIDER_MSG:
        {
            pacResult = MACRO_TO_STRING(STOCK_TICKER_MSG_TYPE_DATA_PROVIDER_MSG);
        }
        break;
        case STOCK_TICKER_MSG_TYPE_MAX:
        {
            pacResult = MACRO_TO_STRING(STOCK_TICKER_MSG_TYPE_MAX);
        }
        break;
    }

    return pacResult;
}

/*****************************************************************************
 *                           FRIEND FUNCTIONS
 *****************************************************************************/

/*****************************************************************************
 *                           PRIVATE FUNCTIONS
 *****************************************************************************/

/*****************************************************************************
*
*   eInitAppFacingObject
*
*****************************************************************************/
static STOCK_TICKER_RETURN_CODE_ENUM eInitAppFacingObject(
    STOCK_TICKER_MGR_OBJECT_STRUCT *psObj
        )
{
    STOCK_TICKER_RETURN_CODE_ENUM eReturnCode = STOCK_TICKER_RETURN_CODE_ERROR;

    printf(STOCK_TICKER_MGR_OBJECT_NAME": Initializing Facing object\n");

    do
    {
        DATASERVICE_DSRL_CONFIG_STRUCT sDSRLConfig;

        // Begin DSRL config
        DATASERVICE_IMPL_vInitializeDSRLConfig(&sDSRLConfig);

        // Set service type
        sDSRLConfig.eServiceType = DATASERVICE_TYPE_STOCK_TICKER;

        // Set the parent object size
        sDSRLConfig.tParentObjectSize = sizeof(STOCK_TICKER_APP_OBJECT_STRUCT);

        // Set the DSRL descriptor size
        sDSRLConfig.tServiceDataSize = sizeof(STOCK_TICKER_DSRL_DESC_STRUCT);

        // Configure the favorites feature
        sDSRLConfig.bFavoritesEnabled = TRUE;
        sDSRLConfig.hCreateTargetFromTag = STOCK_SYMBOL_hCreateTargetFromTag;
        sDSRLConfig.hGetTagForTarget = STOCK_SYMBOL_hGetTargetTag;

        // Create it
        psObj->psAppObj = (STOCK_TICKER_APP_OBJECT_STRUCT *)
            DATASERVICE_IMPL_hConfigureDSRL(
                (DATASERVICE_IMPL_HDL)psObj, &sDSRLConfig);
        if (psObj->psAppObj == NULL)
        {
            break;
        }

        // The app object is now available for use
        OSAL.bMemSet(psObj->psAppObj, 0, sizeof(*psObj->psAppObj));
        psObj->psAppObj->bInitialized = FALSE;
        psObj->psAppObj->sDB.hSQLConnection = SQL_INTERFACE_INVALID_OBJECT;
        psObj->psAppObj->sDB.pacDBFilePath = NULL;
        psObj->psAppObj->sDB.bInTransaction = FALSE;
        psObj->psAppObj->sDB.bSymbolsUpdated = TRUE;
        psObj->psAppObj->sDB.un32ContentVersion = 0;
        psObj->psAppObj->hSymbolMsgList = OSAL_INVALID_OBJECT_HDL;
        psObj->psAppObj->hSymbolMsgCacheList = OSAL_INVALID_OBJECT_HDL;
        psObj->psAppObj->hTargetsMonitorList = OSAL_INVALID_OBJECT_HDL;
        psObj->psAppObj->sHashCtrl.hHashCacheList = OSAL_INVALID_OBJECT_HDL;
        psObj->psAppObj->sHashCtrl.un32NextIndex = 0;
        psObj->psAppObj->sProviderInfoCtrl.hDataProviderInfoText = STRING_INVALID_OBJECT;
        psObj->psAppObj->sProviderInfoCtrl.eState = STOCK_DATA_STATE_INITIAL;
        psObj->psAppObj->sProviderInfoCtrl.un8SequenceNumber = 0;
        psObj->psAppObj->sTimer.hExpirationEvent = DATASERVICE_TIMED_EVENT_INVALID_HDL;

        /////////////////////////////////////////
        // Init message descriptors
        eReturnCode = eStockMsgDescInit(psObj);
        if (eReturnCode != STOCK_TICKER_RETURN_CODE_SUCCESS)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    STOCK_TICKER_MGR_OBJECT_NAME": failed to init message descriptors (%s)",
                    GsStockTickerMgrIntf.pacGetReturnCodeName(eReturnCode)
                        );
            break;
        }

        /////////////////////////////////////////
        // Init cache
        eReturnCode = eStockMsgCacheInit(psObj);
        if (eReturnCode != STOCK_TICKER_RETURN_CODE_SUCCESS)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    STOCK_TICKER_MGR_OBJECT_NAME": failed to init message cache (%s)",
                    GsStockTickerMgrIntf.pacGetReturnCodeName(eReturnCode)
                        );
            break;
        }

        /////////////////////////////////////////
        // Init hash
        eReturnCode = eStockMsgHashInit(psObj->psAppObj);
        if (eReturnCode != STOCK_TICKER_RETURN_CODE_SUCCESS)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    STOCK_TICKER_MGR_OBJECT_NAME": failed to init message hash (%s)",
                    GsStockTickerMgrIntf.pacGetReturnCodeName(eReturnCode)
                        );
            break;
        }

        /////////////////////////////////////////
        // Init Data Provider info
        eReturnCode = eStockDataProviderInfoInit(psObj->psAppObj);
        if (eReturnCode != STOCK_TICKER_RETURN_CODE_SUCCESS)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    STOCK_TICKER_MGR_OBJECT_NAME": failed to init Data Provider info (%s)",
                    GsStockTickerMgrIntf.pacGetReturnCodeName(eReturnCode)
                        );
            break;
        }

        /////////////////////////////////////////
        // Init DSRL
        eReturnCode = eDSRLInit(psObj);
        if (eReturnCode != STOCK_TICKER_RETURN_CODE_SUCCESS)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    STOCK_TICKER_MGR_OBJECT_NAME": failed to init DSRL (%s)",
                    GsStockTickerMgrIntf.pacGetReturnCodeName(eReturnCode)
                        );
            break;
        }

        /////////////////////////////////////////
        // Init DSRL Targets Monitor
        eReturnCode = eDSRLTargetMonitorInit(psObj->psAppObj);
        if (eReturnCode != STOCK_TICKER_RETURN_CODE_SUCCESS)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    STOCK_TICKER_MGR_OBJECT_NAME": failed to init DSRL Targets Monitor (%s)",
                    GsStockTickerMgrIntf.pacGetReturnCodeName(eReturnCode)
                        );
            break;
        }

        /////////////////////////////////////////
        // Create Dummy Stock Ticker Msg object for search
        eReturnCode = STOCK_MSG_eCreateDummy(&psObj->psAppObj->hDummyMsg,
            (SMS_OBJECT)psObj->psAppObj);
        if (eReturnCode != STOCK_TICKER_RETURN_CODE_SUCCESS)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                STOCK_TICKER_MGR_OBJECT_NAME": failed to create dummy msg (%s)",
                GsStockTickerMgrIntf.pacGetReturnCodeName(eReturnCode)
                    );
            break;
        }

        eReturnCode = STOCK_TICKER_RETURN_CODE_SUCCESS;
    } while (FALSE);

    if (psObj->psAppObj != NULL)
    {
        // Relinquish ownership
        SMSO_vUnlock((SMS_OBJECT) psObj->psAppObj);
    }

    return eReturnCode;
}

/*****************************************************************************
*
*   vUninitAppFacingObject
*
*****************************************************************************/
static void vUninitAppFacingObject(
    STOCK_TICKER_MGR_OBJECT_STRUCT *psObj
        )
{
    BOOLEAN bLocked;
    STOCK_TICKER_APP_OBJECT_STRUCT *psAppObj;

    printf(STOCK_TICKER_MGR_OBJECT_NAME": Uninitializing Facing object\n");

    // Lock the app object
    bLocked = SMSO_bLock((SMS_OBJECT)psObj->psAppObj, OSAL_OBJ_TIMEOUT_INFINITE);
    if (bLocked == FALSE)
    {
        SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                STOCK_TICKER_MGR_OBJECT_NAME": Unable to lock app object");
        return;
    }

    printf(STOCK_TICKER_MGR_OBJECT_NAME
           ": attempting to uninit Facing Objects\n");

    psAppObj = psObj->psAppObj;

    vUninitDB(psObj);

    eExpirationTimerUnInit(psAppObj);

    eDSRLTargetMonitorUnInit(psAppObj);

    eDSRLUnInit(psAppObj);

    eStockDataProviderInfoUnInit(psObj);

    eStockMsgHashUninit(psAppObj);

    eStockMsgCacheUninit(psObj);

    eStockMsgDescUnInit(psObj);

    STOCK_MSG_vDestroy(psObj->psAppObj->hDummyMsg);
    psObj->psAppObj->hDummyMsg = STOCK_MSG_INVALID_OBJECT;

    // Destroy the app object
    DATASERVICE_IMPL_vDestroyDSRLParent((DATASERVICE_IMPL_HDL)psObj, (SMS_OBJECT)psAppObj);
    psObj->psAppObj = NULL;

    return;
}

/*****************************************************************************
*
*   psGetAppFacingObject
*
*****************************************************************************/
static STOCK_TICKER_APP_OBJECT_STRUCT *psGetAppFacingObject(
    STOCK_TICKER_SERVICE_OBJECT hStocksService
        )
{
    STOCK_TICKER_MGR_OBJECT_STRUCT *psObj =
        (STOCK_TICKER_MGR_OBJECT_STRUCT *)hStocksService;
    BOOLEAN bValid, bLocked;

    do
    {
        bValid = DATASERVICE_IMPL_bValid((DATASERVICE_IMPL_HDL)hStocksService);
        if (bValid == FALSE)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    STOCK_TICKER_MGR_OBJECT_NAME
                   ": failed to validate Facing object");
            break;
        }

        bLocked = SMSO_bLock((SMS_OBJECT)psObj->psAppObj,
            OSAL_OBJ_TIMEOUT_INFINITE);

        if (bLocked == FALSE)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    STOCK_TICKER_MGR_OBJECT_NAME
                   ": failed to lock Facing object");
            break;
        }

        return psObj->psAppObj;

    } while (FALSE);

    return NULL;
}

/*****************************************************************************
*
*   psOwnerAppFacingObject
*
*****************************************************************************/
static STOCK_TICKER_APP_OBJECT_STRUCT *psOwnedAppFacingObject(
    STOCK_TICKER_SERVICE_OBJECT hStocksService
        )
{
    STOCK_TICKER_MGR_OBJECT_STRUCT *psObj =
        (STOCK_TICKER_MGR_OBJECT_STRUCT *)hStocksService;
    BOOLEAN bValid = FALSE, bOwner = FALSE;

    do
    {
        bValid = DATASERVICE_IMPL_bValid((DATASERVICE_IMPL_HDL)hStocksService);

        if (bValid == FALSE)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    STOCK_TICKER_MGR_OBJECT_NAME
                   ": failed to validate Facing object");
            break;
        }

        bOwner = SMSO_bOwner((SMS_OBJECT)psObj->psAppObj);
        if (bOwner == FALSE)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    STOCK_TICKER_MGR_OBJECT_NAME
                   ": Facing object is not owned by the caller thread");
            break;
        }

        return psObj->psAppObj;

    } while (FALSE);

    return NULL;
}

/*****************************************************************************
*
*   eUpdateProviderData
*
*   If the function finished with success the passed string object controlled
*   by the Manager. In opposite case the caller should take care about
*   string object
*
*****************************************************************************/
static STOCK_TICKER_RETURN_CODE_ENUM eUpdateProviderData (
    STOCK_TICKER_APP_OBJECT_STRUCT *psAppObj,
    STOCK_TICKER_PLUGIN_PROVIDER_INFO_ARG_STRUCT *psProviderInfo
        )
{
    STOCK_TICKER_RETURN_CODE_ENUM eResult = STOCK_TICKER_RETURN_CODE_ERROR;

    do
    {
        // Check input
        if ((psProviderInfo == NULL) || (psAppObj == NULL) ||
            (psProviderInfo->hInfo == STRING_INVALID_OBJECT))
        {
            eResult = STOCK_TICKER_RETURN_CODE_BAD_INPUT;
            break;
        }

        // Update text and notify application if needed
        if ((psAppObj->sProviderInfoCtrl.un8SequenceNumber != psProviderInfo->un8SeqNumber) ||
             (psAppObj->sProviderInfoCtrl.hDataProviderInfoText == STRING_INVALID_OBJECT))
        {
            if (psAppObj->sProviderInfoCtrl.hDataProviderInfoText != STRING_INVALID_OBJECT)
            {
                // Destroy old string to clean up resources
                STRING_vDestroy(psAppObj->sProviderInfoCtrl.hDataProviderInfoText);
                psAppObj->sProviderInfoCtrl.hDataProviderInfoText = STRING_INVALID_OBJECT;
            }

            // Do we need to update the string and notify application about it?
            psAppObj->sProviderInfoCtrl.hDataProviderInfoText = psProviderInfo->hInfo;
            if (psAppObj->sProviderInfoCtrl.eState == STOCK_DATA_STATE_INITIAL)
            {
                psAppObj->sProviderInfoCtrl.eState = STOCK_DATA_STATE_CREATED;
            }
            else
            {
                // Keep previous seq-id to update DB
                psAppObj->sProviderInfoCtrl.un8PrevSequenceNumber =
                    psAppObj->sProviderInfoCtrl.un8SequenceNumber;
                psAppObj->sProviderInfoCtrl.eState = STOCK_DATA_STATE_UPDATED;
            }
            psAppObj->sProviderInfoCtrl.un8SequenceNumber = psProviderInfo->un8SeqNumber;

            // Say to caller do not manage the passed string object since now
            psProviderInfo->hInfo = STRING_INVALID_OBJECT;
        }

        eResult = STOCK_TICKER_RETURN_CODE_SUCCESS;
    } while (FALSE);

    return eResult;
}

/*****************************************************************************
*
*   eUpdateSymbolValue
*
*****************************************************************************/
static STOCK_TICKER_RETURN_CODE_ENUM eUpdateSymbolValue (
    STOCK_TICKER_APP_OBJECT_STRUCT *psAppObj,
    STOCK_TICKER_PLUGIN_VALUES_ARG_STRUCT *psValues,
    STOCK_MSG_DESC_STRUCT *psKnownMsgDesc,
    BOOLEAN bUpdateTimeStamp,
    BOOLEAN bDoTouch
        )
{
    STOCK_TICKER_RETURN_CODE_ENUM eResult;

    do
    {
        // Check input
        if ((psAppObj == NULL) || (psValues == NULL))
        {
            eResult = STOCK_TICKER_RETURN_CODE_BAD_INPUT;
            break;
        }

        // Per current requirements we don't need to keep values
        // persistently so the value can be skipped in case if there
        // are not DSRLs
        eResult = eDSRLDescHasToUpdate(psAppObj);
        if (eResult == STOCK_TICKER_RETURN_CODE_SUCCESS)
        {
            STOCK_MSG_DESC_STRUCT *psMsgDesc;

            if (psKnownMsgDesc == NULL)
            {
                // Try to find out the symbol existence
                psMsgDesc = psStockMsgDescGet(psAppObj, psValues->tIndex);
                if (psMsgDesc == NULL)
                {
                    // Absence of the index means that the stock symbol
                    // is not being monitored thus the values for the index
                    // shall be ignored. For caller let say that everything
                    // is ok.
                    eResult = STOCK_TICKER_RETURN_CODE_SUCCESS;
                    break;
                }
            }
            else
            {
                // Use passed one
                psMsgDesc = psKnownMsgDesc;
            }

            if (bDoTouch == FALSE)
            {
                STOCK_MSG_DATA_STRUCT sData;
                STOCK_TICKER_RETURN_CODE_ENUM eCacheResult;

                // Update structure
                OSAL.bMemSet(&sData, 0, sizeof(sData));
                sData.tMask = STOCK_MSG_DATA_VALUES;
                sData.sValues.eState = psValues->eState;
                sData.sValues.sQuote = psValues->sQuote;

                // Need to update the data
                eResult = STOCK_MSG_eUpdate(psMsgDesc->hMessage, &sData);
                if (eResult == STOCK_TICKER_RETURN_CODE_SUCCESS)
                {
                    if (bUpdateTimeStamp == TRUE)
                    {
                        // Update the timestamp in this case also
                        OSAL.vTimeUp(&psMsgDesc->un32LastUpdateTimeStamp, NULL);
                    }

                    // Now say that the time is updated
                    psMsgDesc->tFieldMask |= STOCK_TICKER_MSG_FIELD_MASK_VALUES;
                    vStockMsgDescSetState(psMsgDesc, STOCK_DATA_STATE_UPDATED);
                }
                else if (eResult == STOCK_TICKER_RETURN_CODE_NOTHING_TO_UPDATE)
                {
                    // Need to update timestamp since we receive the data even
                    // if it is the same as it was.
                    if (bUpdateTimeStamp == TRUE)
                    {
                        // Update the timestamp in this case also
                        OSAL.vTimeUp(&psMsgDesc->un32LastUpdateTimeStamp, NULL);
                    }
                }
                else
                {
                    SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                            STOCK_TICKER_MGR_OBJECT_NAME
                            ": failed to update symbol values via MsgDesc 0x%p (%s)",
                            psMsgDesc, GsStockTickerMgrIntf.pacGetReturnCodeName(eResult)
                                );
                    break;
                }

                // Put the message to the cache to make it processed later
                // per STOP processing request
                eCacheResult = eStockMsgCacheAdd(psAppObj, psMsgDesc);
                if (eCacheResult != STOCK_TICKER_RETURN_CODE_SUCCESS)
                {
                    SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                            STOCK_TICKER_MGR_OBJECT_NAME
                            ": failed to add MsgDesc 0x%p for symbol #%d to cache (%s)",
                            psMsgDesc, psValues->tIndex,
                            GsStockTickerMgrIntf.pacGetReturnCodeName(eCacheResult)
                                );
                    eResult = eCacheResult;
                    break;
                }
            }
            else
            {
                // Update the timestamp only in this case
                OSAL.vTimeUp(&psMsgDesc->un32LastUpdateTimeStamp, NULL);
            }
        }
        // STOCK_TICKER_RETURN_CODE_FALSE means that symbol is just the
        // same which we already have, so it is considered as success.
        // For other errors we are breaking out here.
        else if (eResult != STOCK_TICKER_RETURN_CODE_FALSE)
        {
            break;
        }

        // If we are here - we did all good regardless current error code
        eResult = STOCK_TICKER_RETURN_CODE_SUCCESS;
    } while (FALSE);

    return eResult;
}

/*****************************************************************************
*
*   eUpdateSymbol
*
*****************************************************************************/
static STOCK_TICKER_RETURN_CODE_ENUM eUpdateSymbol (
    STOCK_TICKER_APP_OBJECT_STRUCT *psAppObj,
    STOCK_TICKER_PLUGIN_SYMBOL_ARG_STRUCT *psSymbol
        )
{
    STOCK_TICKER_RETURN_CODE_ENUM eResult = STOCK_TICKER_RETURN_CODE_ERROR;
    STOCK_TICKER_RETURN_CODE_ENUM eCacheResult;
    STOCK_MSG_DESC_STRUCT *psMsgDesc;
    STOCK_MSG_DATA_STRUCT sData;

    do
    {
        // Check input
        if ((psAppObj == NULL) || (psSymbol == NULL))
        {
            eResult = STOCK_TICKER_RETURN_CODE_BAD_INPUT;
            break;
        }

        // Fill up message data structure to process
        OSAL.bMemSet(&sData, 0, sizeof(sData));
        sData.tMask = STOCK_MSG_DATA_NAME_CSTR;
        sData.uName.sCStr.pacValue = psSymbol->pacName;
        sData.uName.sCStr.tValueLength = psSymbol->tNameLength;

        // Try to find out the symbol existence
        psMsgDesc = psStockMsgDescGet(psAppObj, psSymbol->tIndex);
        if (psMsgDesc != NULL)
        {
            // Need to update the data
            eResult = STOCK_MSG_eUpdate(psMsgDesc->hMessage, &sData);
            if (eResult == STOCK_TICKER_RETURN_CODE_SUCCESS)
            {
                psMsgDesc->tFieldMask |= STOCK_TICKER_MSG_FIELD_MASK_NAME;
                // Update hash for the new name
                vStockMsgDescSetState(psMsgDesc, STOCK_DATA_STATE_UPDATED);
            }
            else if (eResult != STOCK_TICKER_RETURN_CODE_NOTHING_TO_UPDATE)
            {
                SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                        STOCK_TICKER_MGR_OBJECT_NAME
                        ": failed to update symbol name via MsgDesc 0x%p (%s)",
                        psMsgDesc, GsStockTickerMgrIntf.pacGetReturnCodeName(eResult)
                            );
                break;
            }
        }
        else
        {
            STRING_OBJECT hName;

            // Correct message data to provide index for creation
            sData.tMask |= STOCK_MSG_DATA_INDEX;
            sData.tIndex = psSymbol->tIndex;

            // Create new descriptor since we know there is no such
            // descriptor in the manager
            eResult = eStockMsgDescCreate(&psMsgDesc, psAppObj, &sData);
            if (eResult != STOCK_TICKER_RETURN_CODE_SUCCESS)
            {
                SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                        STOCK_TICKER_MGR_OBJECT_NAME
                        ": failed to create MsgDesc 0x%p for symbol #%d (%s)",
                        psMsgDesc, psSymbol->tIndex, GsStockTickerMgrIntf.pacGetReturnCodeName(eResult)
                            );
                break;
            }

            // This item has been created from the OTA so we need to
            // make sure we have it in the persistent storage.
            psMsgDesc->tFieldMask |= STOCK_TICKER_MSG_FIELD_MASK_NAME;

            // Let's check the DB for this index
            hName = STOCK_MSG.hSymbolName(psMsgDesc->hMessage);
            eResult = eStockMsgDescGetStateInDB(psAppObj, psSymbol->tIndex, hName);
            if (eResult == STOCK_TICKER_RETURN_CODE_NOTHING_TO_UPDATE)
            {
                // All data in the DB
                vStockMsgDescSetState(psMsgDesc, STOCK_DATA_STATE_STABLE);
            }
            else if (eResult == STOCK_TICKER_RETURN_CODE_NEED_TO_UPDATE)
            {
                // Need to update name in the DB
                vStockMsgDescSetState(psMsgDesc, STOCK_DATA_STATE_UPDATED);
            }
            else if (eResult == STOCK_TICKER_RETURN_CODE_FALSE)
            {
                // There is no item in the DB. Need to add it there
                vStockMsgDescSetState(psMsgDesc, STOCK_DATA_STATE_CREATED);
            }
            else
            {
                SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                        STOCK_TICKER_MGR_OBJECT_NAME
                        ": failed to check stock #%s state in DB (%s)",
                        psSymbol->tIndex, GsStockTickerMgrIntf.pacGetReturnCodeName(eResult)
                            );
                break;
            }
        }

        // Put the message to the cache to make it processed later
        // per STOP processing request
        eCacheResult = eStockMsgCacheAdd(psAppObj, psMsgDesc);
        if (eCacheResult != STOCK_TICKER_RETURN_CODE_SUCCESS)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    STOCK_TICKER_MGR_OBJECT_NAME
                    ": failed to add MsgDesc 0x%p for symbol #%d to cache (%s)",
                    psMsgDesc, psSymbol->tIndex,
                    GsStockTickerMgrIntf.pacGetReturnCodeName(eResult)
                        );
            eResult = eCacheResult;
            break;
        }

        // If we are here - we did all good
        eResult = STOCK_TICKER_RETURN_CODE_SUCCESS;
    } while (FALSE);

    return eResult;
}

/*****************************************************************************
*
*   eInitDB
*
*****************************************************************************/
static STOCK_TICKER_RETURN_CODE_ENUM eInitDB(
    STOCK_TICKER_MGR_OBJECT_STRUCT *psObj
        )
{
    BOOLEAN bOk = FALSE;
    STOCK_TICKER_RETURN_CODE_ENUM eReturn = STOCK_TICKER_RETURN_CODE_ERROR;
    STOCK_TICKER_APP_OBJECT_STRUCT *psAppObj = NULL;
    DATASERVICE_ERROR_CODE_ENUM eSqlErrorCode = DATASERVICE_ERROR_UNKNOWN;
    STOCK_TICKER_DB_CONTROL_STRUCT *psDB;

    printf(STOCK_TICKER_MGR_OBJECT_NAME": Initializing DB\n");

    do
    {
        // Check input
        if (psObj == NULL)
        {
            eReturn = STOCK_TICKER_RETURN_CODE_BAD_INPUT;
            break;
        }

        psAppObj = psOwnedAppFacingObject((STOCK_TICKER_SERVICE_OBJECT)psObj);
        if (psAppObj == NULL)
        {
            eReturn = STOCK_TICKER_RETURN_CODE_NOT_OWNER;
            break;
        }

        // Get quick reference
        psDB = &psAppObj->sDB;

        bOk = DB_UTIL_bCreateFilePath(
            psObj->psAppObj->sOptionValues.pcRefDBPath,
            STOCK_TICKER_DATABASE_FOLDER,
            STOCK_TICKER_REF_DATABASE_FILENAME,
            &psDB->pacDBFilePath
                );
        if (bOk == FALSE)
        {
            // Error!  Couldn't build database path
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    STOCK_TICKER_MGR_OBJECT_NAME": failed to build BD path");
            vSetError(psObj, DATASERVICE_ERROR_CODE_GENERAL);
            break;
        }

        printf(STOCK_TICKER_MGR_OBJECT_NAME": Reference DataBase located at %s\n",
                psDB->pacDBFilePath);

        // Connect to the database
        psDB->hSQLConnection =
            DB_UTIL_hConnectToReference(
                &psDB->pacDBFilePath[0],
                bDBVerifyDBVersion,
                NULL,
                &eSqlErrorCode,
                SQL_INTERFACE_OPTIONS_NONE
                    );
        // If the connection succeeded, continue with
        // initial processing
        if (psDB->hSQLConnection == SQL_INTERFACE_INVALID_OBJECT)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    STOCK_TICKER_MGR_OBJECT_NAME
                    ": failed to start SQLite connection on %s reference DB",
                    psDB->pacDBFilePath
                        );

            vSetError(psObj, eSqlErrorCode);
            break;
        }

        // Prepare statements
        psDB->hSelectSymbolByIdStmt =
            SQL_INTERFACE.hCreatePreparedStatement(
                psDB->hSQLConnection,
                STOCK_TICKER_SELECT_DB_SYMBOL_BY_ID);

        if (SQL_PREPARED_STATEMENT_INVALID_HANDLE ==
            psDB->hSelectSymbolByIdStmt)
        {
            vSetError(psObj, eSqlErrorCode);
            break;
        }

        psDB->hSelectDataVersionStmt =
            SQL_INTERFACE.hCreatePreparedStatement(
                psDB->hSQLConnection,
                STOCK_TICKER_SELECT_DATA_VERSION);

        if (SQL_PREPARED_STATEMENT_INVALID_HANDLE ==
            psDB->hSelectDataVersionStmt)
        {
            vSetError(psObj, eSqlErrorCode);
            break;
        }

        psDB->hUpdateSymbolStmt =
            SQL_INTERFACE.hCreatePreparedStatement(
                psDB->hSQLConnection,
                STOCK_TICKER_UPDATE_DB_SYMBOL);

        if (SQL_PREPARED_STATEMENT_INVALID_HANDLE ==
            psDB->hUpdateSymbolStmt)
        {
            vSetError(psObj, eSqlErrorCode);
            break;
        }

        psDB->hInsertSymbolStmt =
            SQL_INTERFACE.hCreatePreparedStatement(
                psDB->hSQLConnection,
                STOCK_TICKER_INSERT_DB_SYMBOL);

        if (SQL_PREPARED_STATEMENT_INVALID_HANDLE ==
            psDB->hInsertSymbolStmt)
        {
            vSetError(psObj, eSqlErrorCode);
            break;
        }

        // Read current version content
        psAppObj->sDB.un32ContentVersion = un32DBExtractDataVersion(psAppObj);

        eReturn = STOCK_TICKER_RETURN_CODE_SUCCESS;
    } while (FALSE);

    return eReturn;
}

/*****************************************************************************
*
*   eLoadDB
*
*****************************************************************************/
static STOCK_TICKER_RETURN_CODE_ENUM eLoadDB(
    STOCK_TICKER_MGR_OBJECT_STRUCT *psObj
        )
{
    STOCK_TICKER_APP_OBJECT_STRUCT *psAppObj = NULL;
    STOCK_TICKER_RETURN_CODE_ENUM eReturnCode = STOCK_TICKER_RETURN_CODE_ERROR;

    printf(STOCK_TICKER_MGR_OBJECT_NAME": Loading DB\n");

    do
    {
        // Check input
        if (psObj == NULL)
        {
            eReturnCode = STOCK_TICKER_RETURN_CODE_BAD_INPUT;
            break;
        }

        // Make sure that the App facing object is owned here
        psAppObj = psOwnedAppFacingObject((STOCK_TICKER_SERVICE_OBJECT) psObj);
        if (psAppObj == NULL)
        {
            eReturnCode = STOCK_TICKER_RETURN_CODE_NOT_OWNER;
            break;
        }

        ///////////////////////////////////////////////////////////////////////
        // Loading HASH database
        eReturnCode = eStockMsgHashLoadDB(psAppObj);
        if (eReturnCode != STOCK_TICKER_RETURN_CODE_SUCCESS)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    STOCK_TICKER_MGR_OBJECT_NAME": failed to load hash DB (%s)",
                    GsStockTickerMgrIntf.pacGetReturnCodeName(eReturnCode)
                        );
            break;
        }

        ///////////////////////////////////////////////////////////////////////
        // Loading Data Provider Info database
        eReturnCode = eStockDataProviderInfoLoadDB(psAppObj);
        if (eReturnCode != STOCK_TICKER_RETURN_CODE_SUCCESS)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    STOCK_TICKER_MGR_OBJECT_NAME": failed to load Data Provider Info DB (%s)",
                    GsStockTickerMgrIntf.pacGetReturnCodeName(eReturnCode)
                        );
            break;
        }

    } while (FALSE);

    return eReturnCode;
}

/*****************************************************************************
*
*   eFlushDB
*
*****************************************************************************/
static STOCK_TICKER_RETURN_CODE_ENUM eFlushDB(
    STOCK_TICKER_MGR_OBJECT_STRUCT *psObj,
    BOOLEAN bFullData
        )
{
    STOCK_TICKER_RETURN_CODE_ENUM eResult = STOCK_TICKER_RETURN_CODE_ERROR;
    STOCK_TICKER_APP_OBJECT_STRUCT *psAppObj = NULL;

    printf(STOCK_TICKER_MGR_OBJECT_NAME": flushing DB\n");

    do
    {
        // Check input
        psAppObj = psOwnedAppFacingObject((STOCK_TICKER_SERVICE_OBJECT) psObj);
        if (psAppObj == NULL)
        {
            eResult = STOCK_TICKER_RETURN_CODE_NOT_OWNER;
            break;
        }

        // Start transaction
        eDBStartTransaction(psAppObj);

        ////////////////////////////////////////
        // Flush Data Provider Info data
        eResult = eStockDataProviderInfoFlushDB(psAppObj);
        if (eResult != STOCK_TICKER_RETURN_CODE_SUCCESS)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    STOCK_TICKER_MGR_OBJECT_NAME": failed to flush DataProviderInfor (%s)",
                    GsStockTickerMgrIntf.pacGetReturnCodeName(eResult)
                        );
        }

        ////////////////////////////////////////
        // Flush Message Desc data
        eResult = eStockMsgDescFlushDB(psAppObj, bFullData);
        if (eResult != STOCK_TICKER_RETURN_CODE_SUCCESS)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    STOCK_TICKER_MGR_OBJECT_NAME": failed to flush MsgDesc list (%s)",
                    GsStockTickerMgrIntf.pacGetReturnCodeName(eResult)
                        );
        }

        ////////////////////////////////////////
        // Flush Hash Data
        eResult = eStockMsgHashCacheFlushDB(psAppObj);
        if (eResult != STOCK_TICKER_RETURN_CODE_SUCCESS)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    STOCK_TICKER_MGR_OBJECT_NAME": failed to flush MsgHash list (%s)",
                    GsStockTickerMgrIntf.pacGetReturnCodeName(eResult)
                        );
        }

        eResult = STOCK_TICKER_RETURN_CODE_SUCCESS;
    } while (FALSE);

    if (psAppObj != NULL)
    {
        // End transaction
        eDBEndTransaction(psAppObj);
    }

    return eResult;
}

/*****************************************************************************
*
*   eDBStartTransaction
*
*****************************************************************************/
static STOCK_TICKER_RETURN_CODE_ENUM eDBStartTransaction(
    STOCK_TICKER_APP_OBJECT_STRUCT *psAppObj
        )
{
    BOOLEAN bStarted = FALSE;
    STOCK_TICKER_RETURN_CODE_ENUM eReturnCode = STOCK_TICKER_RETURN_CODE_ERROR;

    do
    {
        // Check input
        if (psAppObj == NULL)
        {
            eReturnCode = STOCK_TICKER_RETURN_CODE_BAD_INPUT;
            break;
        }

        // Lets start the transaction if it hasn't been stated yet and
        // proper connection exists
        if ((psAppObj->sDB.bInTransaction == FALSE) &&
            (psAppObj->sDB.hSQLConnection != SQL_INTERFACE_INVALID_OBJECT))
        {
            bStarted = SQL_INTERFACE.bStartTransaction(psAppObj->sDB.hSQLConnection);
            if (bStarted == FALSE)
            {
                SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                        STOCK_TICKER_MGR_OBJECT_NAME": failed to start SQL transaction");
                eReturnCode = STOCK_TICKER_RETURN_CODE_DB_ERROR;
                break;
            }

            puts(STOCK_TICKER_MGR_OBJECT_NAME": DB Transaction has been started");
        }

        psAppObj->sDB.bInTransaction = bStarted;
        eReturnCode = STOCK_TICKER_RETURN_CODE_SUCCESS;
    } while (FALSE);

    return eReturnCode;
}

/*****************************************************************************
*
*   eDBEndTransaction
*
*****************************************************************************/
static STOCK_TICKER_RETURN_CODE_ENUM eDBEndTransaction(
    STOCK_TICKER_APP_OBJECT_STRUCT *psAppObj
        )
{
    BOOLEAN bEnded = FALSE;
    STOCK_TICKER_RETURN_CODE_ENUM eReturnCode = STOCK_TICKER_RETURN_CODE_ERROR;

    do
    {
        // Check input
        if (psAppObj == NULL)
        {
            eReturnCode = STOCK_TICKER_RETURN_CODE_BAD_INPUT;
            break;
        }

        if ((psAppObj->sDB.bInTransaction == TRUE) &&
            (psAppObj->sDB.hSQLConnection != SQL_INTERFACE_INVALID_OBJECT))
        {
            bEnded = SQL_INTERFACE.bEndTransaction(psAppObj->sDB.hSQLConnection);
            if (bEnded == FALSE)
            {
                SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                        STOCK_TICKER_MGR_OBJECT_NAME": failed to end SQL transaction");
                eReturnCode = STOCK_TICKER_RETURN_CODE_DB_ERROR;
                break;
            }

            puts(STOCK_TICKER_MGR_OBJECT_NAME": DB Transaction has been ended");
        }

        psAppObj->sDB.bInTransaction = FALSE;
        eReturnCode = STOCK_TICKER_RETURN_CODE_SUCCESS;
    } while (FALSE);

    return eReturnCode;
}

/*******************************************************************************
*
*   bDBProcessSelectDataVersionResult
*
*******************************************************************************/
static BOOLEAN bDBProcessSelectDataVersionResult (
    SQL_QUERY_COLUMN_STRUCT *psColumns,
    N32 n32NumberOfColumns,
    STOCK_TICKER_DB_QUERY_RESULT_STRUCT *psResult
        )
{
    STOCK_TICKER_DATA_VERSION_FIELDS_ENUM eCurrentField =
        (STOCK_TICKER_DATA_VERSION_FIELDS_ENUM)0;

    // Verify input
    if (psResult == NULL)
    {
        return FALSE;
    }

    // If there are the correct
    // number of columns, then we have good results
    if (n32NumberOfColumns == STOCK_TICKER_DATA_VERSION_MAX_FIELDS)
    {
        psResult->sOutput.bResultantRows = TRUE;
    }
    else
    {
        psResult->sOutput.bResultantRows = FALSE;

        return FALSE;
    }

    // We have just performed an SQL query in order to
    // determine the data version in the database.
    // Process the results (should be only one row)
    do
    {
        // Decode the current field based on it's type
        switch (eCurrentField)
        {
            case STOCK_TICKER_DATA_VERSION_FIELD_VERSION:
            {
                psResult->sOutput.uDbRow.sDataVersion.un32Version =
                    (N16)psColumns[eCurrentField].uData.sUN32.un32Data;
            }
            break;

            default: // Shouldn't happen
            break;
        }
    } while (++eCurrentField < STOCK_TICKER_DATA_VERSION_MAX_FIELDS);

    return FALSE;
}

/*****************************************************************************
*
*   un32DBExtractDataVersion
*
*****************************************************************************/
static UN32 un32DBExtractDataVersion (
    STOCK_TICKER_APP_OBJECT_STRUCT *psAppObj
        )
{
    BOOLEAN bOk;
    STOCK_TICKER_DB_QUERY_RESULT_STRUCT sResult;
    UN32 un32Result = 0;

    if (psAppObj != NULL)
    {
        // Initialize the result structure
        OSAL.bMemSet(&sResult, 0, sizeof(sResult));
        sResult.sOutput.bResultantRows = FALSE;

        // Perform the SQL query and process the result
        // (it will provide us with a data row)
        bOk = SQL_INTERFACE.bExecutePreparedStatement(
            psAppObj->sDB.hSQLConnection,
            psAppObj->sDB.hSelectDataVersionStmt,
            (SQL_QUERY_RESULT_HANDLER)bDBProcessSelectDataVersionResult,
            &sResult,
            0);

        if ((bOk == TRUE) && (sResult.sOutput.bResultantRows == TRUE))
        {
            un32Result = sResult.sOutput.uDbRow.sDataVersion.un32Version;
        }
    }

    return un32Result;
}

/*****************************************************************************
*
*   vUninitDB
*
*****************************************************************************/
static void vUninitDB (
    STOCK_TICKER_MGR_OBJECT_STRUCT *psObj
        )
{
    if (psObj != NULL)
    {
        STOCK_TICKER_APP_OBJECT_STRUCT *psAppObj = psObj->psAppObj;
        STOCK_TICKER_RETURN_CODE_ENUM eResult = STOCK_TICKER_RETURN_CODE_ERROR;

        printf(STOCK_TICKER_MGR_OBJECT_NAME": Uninitializing DB\n");

        // Disconnect from the database application
        if (psAppObj->sDB.hSQLConnection != SQL_INTERFACE_INVALID_OBJECT)
        {
            // Flush data to DB if needed
            eResult = eFlushDB(psObj, FALSE);
            if (eResult != STOCK_TICKER_RETURN_CODE_SUCCESS)
            {
                SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                        STOCK_TICKER_MGR_OBJECT_NAME": failed to flush DB at uninit (%s)",
                        GsStockTickerMgrIntf.pacGetReturnCodeName(eResult)
                            );
            }

            // Destroy prepared statements
            SQL_INTERFACE.vDestroyPreparedStatement(
                psAppObj->sDB.hSQLConnection,
                psAppObj->sDB.hSelectSymbolByIdStmt);

            psAppObj->sDB.hSelectSymbolByIdStmt =
                SQL_PREPARED_STATEMENT_INVALID_HANDLE;

            SQL_INTERFACE.vDestroyPreparedStatement(
                psAppObj->sDB.hSQLConnection,
                psAppObj->sDB.hSelectDataVersionStmt);

            psAppObj->sDB.hSelectDataVersionStmt =
                SQL_PREPARED_STATEMENT_INVALID_HANDLE;

            SQL_INTERFACE.vDestroyPreparedStatement(
                psAppObj->sDB.hSQLConnection,
                psAppObj->sDB.hUpdateSymbolStmt);

            psAppObj->sDB.hUpdateSymbolStmt =
                SQL_PREPARED_STATEMENT_INVALID_HANDLE;

            SQL_INTERFACE.vDestroyPreparedStatement(
                psAppObj->sDB.hSQLConnection,
                psAppObj->sDB.hInsertSymbolStmt);

            psAppObj->sDB.hInsertSymbolStmt =
                SQL_PREPARED_STATEMENT_INVALID_HANDLE;

            SQL_INTERFACE.vDisconnect(psAppObj->sDB.hSQLConnection);
            psAppObj->sDB.hSQLConnection = SQL_INTERFACE_INVALID_OBJECT;
        }

        if (psAppObj->sDB.pacDBFilePath != NULL)
        {
            SMSO_vDestroy((SMS_OBJECT) psAppObj->sDB.pacDBFilePath);
            psAppObj->sDB.pacDBFilePath = NULL;
        }
    }

    return;
}

/*****************************************************************************
*
*   eDBUpdateDataVersion
*
*   This function updates the data version by incrementing current value by
*   1 since the STOCK TICKER stream doesn't provide any other way.
*****************************************************************************/
static STOCK_TICKER_RETURN_CODE_ENUM eDBUpdateDataVersion(
    STOCK_TICKER_APP_OBJECT_STRUCT *psAppObj
        )
{
    STOCK_TICKER_RETURN_CODE_ENUM eReturnCode ;
    BOOLEAN bOk;
    do
    {
        // Check input
        if (psAppObj == NULL)
        {
            eReturnCode = STOCK_TICKER_RETURN_CODE_BAD_INPUT;
            break;
        }

        // Update version in the DB
        bOk = SQL_INTERFACE.bExecuteCommand(psAppObj->sDB.hSQLConnection,
                            STOCK_TICKER_UPDATE_DATA_VERSION
                                );
        if (bOk == FALSE)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                STOCK_TICKER_MGR_OBJECT_NAME": failed to execute query (%s)",
                STOCK_TICKER_UPDATE_DATA_VERSION);
            eReturnCode = STOCK_TICKER_RETURN_CODE_DB_ERROR;
            break;
        }

        // Read updated version
        psAppObj->sDB.un32ContentVersion = un32DBExtractDataVersion(psAppObj);

        eReturnCode = STOCK_TICKER_RETURN_CODE_SUCCESS;
    } while (FALSE);

    return eReturnCode;
}

/*****************************************************************************
*
*   bDBVerifyDBVersion
*
*****************************************************************************/
static BOOLEAN bDBVerifyDBVersion (
    SQL_INTERFACE_OBJECT hSQLConnection,
    void *pvArg
        )
{
    STOCK_TICKER_DB_QUERY_RESULT_STRUCT sQueryResult;
    BOOLEAN bVersionMatched = FALSE, bOk;

    puts(STOCK_TICKER_MGR_OBJECT_NAME": Start verifying DB");

    // Initialize the result structure
    sQueryResult.sOutput.bResultantRows = FALSE;

    // Perform the SQL query and process the result
    // (it will provide us with a data row)
    bOk = SQL_INTERFACE.bQuery(hSQLConnection,
        STOCK_TICKER_SELECT_DB_VERSION,
        bProcessSelectDBVersionResult, &sQueryResult
            );

    if ((bOk == TRUE) && (sQueryResult.sOutput.bResultantRows == TRUE))
    {
        STOCK_TICKER_VERSION_ROW_STRUCT *psVersionRow =
            &sQueryResult.sOutput.uDbRow.sVersion;

            // Verify all the fields, except DSI since we don't
            // care about DSI at the moment
        if (psVersionRow->un8DBVer == STOCK_TICKER_DATABASE_FILE_VERSION)
        {
            // Version match!
            printf(STOCK_TICKER_MGR_OBJECT_NAME": Supported version #%d detected\n",
                    psVersionRow->un8DBVer
                        );
            bVersionMatched = TRUE;
        }
        else
        {
            // Version mismatch!
            printf(STOCK_TICKER_MGR_OBJECT_NAME": Unsupported version #%d detected (should be #%d)\n",
                    psVersionRow->un8DBVer,
                    STOCK_TICKER_DATABASE_FILE_VERSION
                        );
        }
    }

    return bVersionMatched;
}

/*******************************************************************************
*
*   bProcessSelectDBVersionResult
*
*   This function is provided to the SQL_INTERFACE_OBJECT in order to process
*   a "select all" query made on the database version relation.  It populates a
*   pointer to a STOCKS_VERSION_ROW_STRUCT with the information found
*   in the database.
*
*******************************************************************************/
static BOOLEAN bProcessSelectDBVersionResult (
    SQL_QUERY_COLUMN_STRUCT *psColumns,
    N32 n32NumberOfColumns,
    void *pvArg
        )
{
    STOCK_TICKER_DB_VERSION_FIELDS_ENUM eCurrentField =
        (STOCK_TICKER_DB_VERSION_FIELDS_ENUM)0;
    STOCK_TICKER_DB_QUERY_RESULT_STRUCT *psResult =
        (STOCK_TICKER_DB_QUERY_RESULT_STRUCT *)pvArg;
    STOCK_TICKER_VERSION_ROW_STRUCT *psVersionRow = NULL;
    SQL_QUERY_COLUMN_STRUCT *psCurrentCol = NULL;

    // Verify input
    if (psResult == NULL)
    {
        return FALSE;
    }

    // If there are the correct
    // number of columns, then we have good results
    if (n32NumberOfColumns == STOCK_TICKER_DB_VERSION_MAX_FIELDS)
    {
        psResult->sOutput.bResultantRows = TRUE;
    }
    else
    {
        psResult->sOutput.bResultantRows = FALSE;

        return FALSE;
    }

    // Get a more convenient pointer
    psVersionRow = &psResult->sOutput.uDbRow.sVersion;

    // Clear the version structure
    OSAL.bMemSet(psVersionRow, 0, sizeof(STOCK_TICKER_VERSION_ROW_STRUCT));

    // We have just performed an SQL query in order to
    // determine the version of the database.
    // Process the results (should be only one row)
    do
    {
        // Grab the current column data
        psCurrentCol = &psColumns[(UN8)eCurrentField];

        // Decode the current field based on it's type
        switch (eCurrentField)
        {
            case STOCK_TICKER_DB_VERSION_FIELD_DB_VER:
            {
                psVersionRow->un8DBVer =
                        (UN8)psCurrentCol->uData.sUN32.un32Data;
            }
            break;

            case STOCK_TICKER_DB_VERSION_FIELD_DSI:
            {
                psVersionRow->tDSI =
                        (UN8)psCurrentCol->uData.sUN32.un32Data;
            }
            break;

            default:
            {
                 // Shouldn't happen
            }
            break;
        }
    } while ( ++eCurrentField < STOCK_TICKER_DB_VERSION_MAX_FIELDS);

    if (eCurrentField == STOCK_TICKER_DB_VERSION_MAX_FIELDS)
    {
        return TRUE;
    }

    return FALSE;
}

/*****************************************************************************
*
*   bHandleServiceReady
*
*   This function is called when SMS is ready for the service to startup.
*   At this time, the service has a context in which to operate (SMS
*   assigned a resource to us). When this call is made, this service will
*   perform all of its time-intensive startup procedures.
*
*****************************************************************************/
static BOOLEAN bHandleServiceReady (
    STOCK_TICKER_MGR_OBJECT_STRUCT *psObj
        )
{
    STOCK_TICKER_RETURN_CODE_ENUM eReturnCode = STOCK_TICKER_RETURN_CODE_ERROR;
    STOCK_TICKER_APP_OBJECT_STRUCT *psAppObj = NULL;

    puts(STOCK_TICKER_MGR_OBJECT_NAME": handling Ready state");

    do
    {
        // Validate the service handle, then get a pointer to the
        // locked app-facing object
        psAppObj = psGetAppFacingObject((STOCK_TICKER_SERVICE_OBJECT) psObj);
        if (psAppObj == NULL)
        {
            // Error, not sure why this would happen
            vSetError( psObj, DATASERVICE_ERROR_CODE_GENERAL );
            eReturnCode = STOCK_TICKER_RETURN_CODE_NOT_OWNER;
            break;
        }

        /////////////////////////////////////////
        // Init Timer
        eReturnCode = eExpirationTimerInit(psObj);
        if (eReturnCode != STOCK_TICKER_RETURN_CODE_SUCCESS)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    STOCK_TICKER_MGR_OBJECT_NAME": failed to init Expiration Timer (%s)",
                    GsStockTickerMgrIntf.pacGetReturnCodeName(eReturnCode));
            vSetError( psObj, DATASERVICE_ERROR_CODE_GENERAL );
            break;
        }

        // Does the service already initialized
        if (psAppObj->bInitialized == TRUE)
        {
            eReturnCode = STOCK_TICKER_RETURN_CODE_SUCCESS;
            break;
        }

        // Initialize DB
        eReturnCode = eInitDB(psObj);
        if (eReturnCode != STOCK_TICKER_RETURN_CODE_SUCCESS)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    STOCK_TICKER_MGR_OBJECT_NAME
                   ": failed to initialize SQLite connection (%s)",
                   GsStockTickerMgrIntf.pacGetReturnCodeName(eReturnCode)
                          );
            break;
        }

        // Load DB
        eReturnCode = eLoadDB(psObj);
        if (eReturnCode != STOCK_TICKER_RETURN_CODE_SUCCESS)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    STOCK_TICKER_MGR_OBJECT_NAME
                   ": failed to load data from DB (%s)",
                   GsStockTickerMgrIntf.pacGetReturnCodeName(eReturnCode)
                          );
            break;
        }

        // Make the call to init the broadcast-specific parser
        psObj->hParser = GsStockTickerIntf.hInit(
            (STOCK_TICKER_SERVICE_OBJECT)psObj,
            DATASERVICE_IMPL_hSMSObj((DATASERVICE_IMPL_HDL)psObj));
        if (psObj->hParser == STOCK_TICKER_INTERFACE_INVALID_OBJECT)
        {
            eReturnCode = STOCK_TICKER_RETURN_CODE_ERROR;
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    STOCK_TICKER_MGR_OBJECT_NAME
                   ": failed to initialize Low-Level Protocol Parser");
            break;
        }

        // ok, we're initialized
        psAppObj->bInitialized = TRUE;

        // Success result
        eReturnCode = STOCK_TICKER_RETURN_CODE_SUCCESS;

    } while(FALSE);

    if (psAppObj != NULL)
    {
        SMSO_vUnlock((SMS_OBJECT)psAppObj);
    }

    if (eReturnCode != STOCK_TICKER_RETURN_CODE_SUCCESS)
    {
        SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
            STOCK_TICKER_MGR_OBJECT_NAME": failed to process READY state (%s)",
            pacGetReturnCodeName(eReturnCode));

        return FALSE;
    }

    return TRUE;
}

/*****************************************************************************
*
*   bHandleServiceError
*
*****************************************************************************/
static BOOLEAN bHandleServiceError (
    STOCK_TICKER_MGR_OBJECT_STRUCT *psObj
        )
{
    // Stop any ageout timer we have now 'cause
    // who knows what's going on up there
    vExpirationTimerStop(psObj->psAppObj);

    return TRUE;
}

/*****************************************************************************
*
*   eProcessServiceData
*
*****************************************************************************/
static STOCK_TICKER_RETURN_CODE_ENUM eProcessServiceData(
    STOCK_TICKER_APP_OBJECT_STRUCT *psAppObj
        )
{
    STOCK_TICKER_RETURN_CODE_ENUM eReturnCode = STOCK_TICKER_RETURN_CODE_ERROR;
    BOOLEAN bDoBreak = FALSE;

    puts(STOCK_TICKER_MGR_OBJECT_NAME": processing service data");

    do
    {
        // Check input
        if (psAppObj == NULL)
        {
            eReturnCode = STOCK_TICKER_RETURN_CODE_BAD_INPUT;
            break;
        }

#if SMS_DEBUG==1
        {
            UN32 un32Items = 0;
            OSAL.eLinkedListItems(psAppObj->hSymbolMsgList, &un32Items);
            printf(STOCK_TICKER_MGR_OBJECT_NAME
                    ": the service has %d quote(s) to process\n",
                    un32Items);
            DATASERVICE_IMPL_vLog(STOCK_TICKER_MGR_OBJECT_NAME
                ": the service has %d quote(s) to process\n",
                un32Items);
        }
#endif

        // Start transaction
        eDBStartTransaction(psAppObj);

        ///////////////////////////////////////////////////
        // Process messages in the cache to update DB and DSRL if required
        eReturnCode = eStockMsgCacheProcess(psAppObj);
        if (eReturnCode != STOCK_TICKER_RETURN_CODE_SUCCESS)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    STOCK_TICKER_MGR_OBJECT_NAME": failed to process message cache (%s)",
                    GsStockTickerMgrIntf.pacGetReturnCodeName(eReturnCode)
                        );
            bDoBreak = TRUE;
        }

        // Clean up the cache even if previous operation failed
        eReturnCode = eStockMsgCacheClean(psAppObj);
        if (eReturnCode != STOCK_TICKER_RETURN_CODE_SUCCESS)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    STOCK_TICKER_MGR_OBJECT_NAME": failed to clean cache (%s)",
                    GsStockTickerMgrIntf.pacGetReturnCodeName(eReturnCode)
                        );
            bDoBreak = TRUE;
        }

        // Move all DSRLs in UPDATING state into READY even if previous operations
        // are failed.
        eReturnCode = eDSRLDescStateTransition(psAppObj,
                        DSRL_STATE_UPDATING, DSRL_STATE_READY
                            );
        if (eReturnCode != STOCK_TICKER_RETURN_CODE_SUCCESS)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    STOCK_TICKER_MGR_OBJECT_NAME": failed to update states for DSRLs (%s)",
                    GsStockTickerMgrIntf.pacGetReturnCodeName(eReturnCode)
                        );
            break;
        }
        else if (bDoBreak == TRUE)
        {
            // Break from here since something above failed
            break;
        }

        ///////////////////////////////////////////////////
        // Flush hashes to the DB if required
        bDoBreak = FALSE;
        eReturnCode = eStockMsgHashCacheFlushDB(psAppObj);
        if (eReturnCode != STOCK_TICKER_RETURN_CODE_SUCCESS)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    STOCK_TICKER_MGR_OBJECT_NAME": failed to flush messages hash (%s)",
                    GsStockTickerMgrIntf.pacGetReturnCodeName(eReturnCode)
                        );
            bDoBreak = TRUE;
        }

        // Clean up the hash cache event id previous operation failed
        eReturnCode = eStockMsgHashCacheClean(psAppObj);
        if (eReturnCode != STOCK_TICKER_RETURN_CODE_SUCCESS)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    STOCK_TICKER_MGR_OBJECT_NAME": failed to clean hash cache (%s)",
                    GsStockTickerMgrIntf.pacGetReturnCodeName(eReturnCode)
                        );
            break;
        }
        else if (bDoBreak == TRUE)
        {
            break;
        }

        ///////////////////////////////////////////////////
        // Flush DataProvider info if required
        eReturnCode = eStockDataProviderInfoFlushDB(psAppObj);
        if ((eReturnCode != STOCK_TICKER_RETURN_CODE_SUCCESS) &&
            (eReturnCode != STOCK_TICKER_RETURN_CODE_NOTHING_TO_UPDATE))
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    STOCK_TICKER_MGR_OBJECT_NAME
                    ": failed flush DataProvider info to DB (%s)",
                    GsStockTickerMgrIntf.pacGetReturnCodeName(eReturnCode)
                        );
            break;
        }

#if SMS_DEBUG==1
        ///////////////////////////////////////////////////
        // Information print
        {
            UN32 un32Items = 0;
            OSAL.eLinkedListItems(psAppObj->hSymbolMsgList, &un32Items);
            printf(STOCK_TICKER_MGR_OBJECT_NAME
                    ": the service remains %d quote(s) after processing\n",
                    un32Items);
            DATASERVICE_IMPL_vLog(STOCK_TICKER_MGR_OBJECT_NAME
                ": %d quote(s) remain after processing\n",
                un32Items);
        }
#endif

        eReturnCode = STOCK_TICKER_RETURN_CODE_SUCCESS;
    } while (FALSE);

    eDBEndTransaction(psAppObj);

    DATASERVICE_IMPL_vLog(STOCK_TICKER_MGR_OBJECT_NAME
        ": Processed service data with code %s\n",
        GsStockTickerMgrIntf.pacGetReturnCodeName(eReturnCode));

    return eReturnCode;
}

/*****************************************************************************
*
*   eExpirationTimerInit
*
*****************************************************************************/
static STOCK_TICKER_RETURN_CODE_ENUM eExpirationTimerInit(
    STOCK_TICKER_MGR_OBJECT_STRUCT *psObj
        )
{
    STOCK_TICKER_RETURN_CODE_ENUM eReturnCode = STOCK_TICKER_RETURN_CODE_BAD_INPUT;

    puts(STOCK_TICKER_MGR_OBJECT_NAME": Initializing Expiration Timer");

    // Check input
    if (psObj != NULL)
    {
        STOCK_TICKER_APP_OBJECT_STRUCT *psAppObj;

        // Get quick reference
        psAppObj = psObj->psAppObj;

        // Register for a timed event
        psAppObj->sTimer.hExpirationEvent =
            DATASERVICE_IMPL_hRegisterTimedEvent(
                (DATASERVICE_IMPL_HDL)psObj, (void *)NULL);

        if(psAppObj->sTimer.hExpirationEvent
                    == DATASERVICE_TIMED_EVENT_INVALID_HDL)
        {
            // Error!
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                STOCK_TICKER_MGR_OBJECT_NAME": failed to allocate event");
            eReturnCode = STOCK_TICKER_RETURN_CODE_MEMORY_ERROR;
        }
        else
        {
            eReturnCode = STOCK_TICKER_RETURN_CODE_SUCCESS;
        }
    }

    return eReturnCode;
}

/*****************************************************************************
*
*   eExpirationTimerUnInit
*
*****************************************************************************/
static STOCK_TICKER_RETURN_CODE_ENUM eExpirationTimerUnInit(
    STOCK_TICKER_APP_OBJECT_STRUCT *psAppObj
        )
{
    STOCK_TICKER_RETURN_CODE_ENUM eReturnCode =
                                        STOCK_TICKER_RETURN_CODE_BAD_INPUT;

    puts(STOCK_TICKER_MGR_OBJECT_NAME": Uninitializing Expiration Timer");

    // Check input
    if (psAppObj != NULL)
    {
        vExpirationTimerStop(psAppObj);

        // Clear the timed event handle, we're done with it
        psAppObj->sTimer.hExpirationEvent =
            DATASERVICE_TIMED_EVENT_INVALID_HDL;

        eReturnCode = STOCK_TICKER_RETURN_CODE_SUCCESS;
    }

    return eReturnCode;
}

/*****************************************************************************
*
*   vExpirationTimerStart
*
*****************************************************************************/
static void vExpirationTimerStart (
    STOCK_TICKER_APP_OBJECT_STRUCT *psAppObj,
    UN32 un32PeriodInSec
        )
{
    BOOLEAN bOk;

    printf(STOCK_TICKER_MGR_OBJECT_NAME
            ": starting Expiration timer for %d sec(s)\n",
            un32PeriodInSec);

    // Kick off the event
    bOk = DATASERVICE_IMPL_bSetTimedEvent(
        psAppObj->sTimer.hExpirationEvent,
        FALSE, FALSE,
        un32PeriodInSec);

    if (bOk == FALSE)
    {
        SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
            STOCK_TICKER_MGR_OBJECT_NAME": failed to start timer");
    }
    else
    {
        puts(STOCK_TICKER_MGR_OBJECT_NAME": Starting Expiration timer");
    }

    return;
}

/*****************************************************************************
*
*   vExpirationTimerStop
*
*****************************************************************************/
static void vExpirationTimerStop (
    STOCK_TICKER_APP_OBJECT_STRUCT *psAppObj
        )
{
    do
    {
        BOOLEAN bOk;

        if ((psAppObj == NULL) ||
            (psAppObj->sTimer.hExpirationEvent == DATASERVICE_TIMED_EVENT_INVALID_HDL))
        {
            break;
        }

        puts(STOCK_TICKER_MGR_OBJECT_NAME": Stopping expiration timer");

        // Stop the expiration event from occurring
        bOk = DATASERVICE_IMPL_bStopTimedEvent(psAppObj->sTimer.hExpirationEvent);

        if (bOk != TRUE)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                STOCK_TICKER_MGR_OBJECT_NAME": failed to stop timer");
            break;
        }

    } while (FALSE);

    return;
}

/*****************************************************************************
*
*   vExpirationTimerUpdate
*
*****************************************************************************/
static void vExpirationTimerUpdate (
    STOCK_TICKER_APP_OBJECT_STRUCT *psAppObj,
    UN32 un32PeriodInSec
        )
{
    do
    {
        OSAL_RETURN_CODE_ENUM eOsalReturnCode;
        UN32 un32Items;

        puts(STOCK_TICKER_MGR_OBJECT_NAME": Updating Expiration timer state");

        if ((psAppObj == NULL) ||
            (psAppObj->sTimer.hExpirationEvent == DATASERVICE_TIMED_EVENT_INVALID_HDL))
        {
            break;
        }

        eOsalReturnCode = OSAL.eLinkedListItems(psAppObj->hSymbolMsgList, &un32Items);
        if (eOsalReturnCode != OSAL_SUCCESS)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                STOCK_TICKER_MGR_OBJECT_NAME": failed to get number of active messages (%s)",
                OSAL.pacGetReturnCodeName(eOsalReturnCode)
                    );
            break;
        }

        if (un32Items > 0)
        {
            vExpirationTimerStart(psAppObj, un32PeriodInSec);
        }
        else
        {
            vExpirationTimerStop(psAppObj);
        }

    } while (FALSE);

    return;
}

/*****************************************************************************
*
*   eStockMsgHashInit
*
*****************************************************************************/
static STOCK_TICKER_RETURN_CODE_ENUM eStockMsgHashInit (
    STOCK_TICKER_APP_OBJECT_STRUCT *psAppObj
        )
{
    STOCK_TICKER_RETURN_CODE_ENUM eResult = STOCK_TICKER_RETURN_CODE_ERROR;
    STOCK_TICKER_MSG_TYPE_ENUM eIndex;
    OSAL_RETURN_CODE_ENUM eOsalReturnCode = OSAL_ERROR;

    puts(STOCK_TICKER_MGR_OBJECT_NAME": Initializing Stock Msg Hash");

    do
    {
        // Check input
        if (psAppObj == NULL)
        {
            eResult = STOCK_TICKER_RETURN_CODE_BAD_INPUT;
            break;
        }

        // Init fields and create hash list for each message type
        psAppObj->sHashCtrl.un32NextIndex = 0;
        for (eIndex = STOCK_TICKER_MSG_TYPE_START;
             eIndex < STOCK_TICKER_MSG_TYPE_MAX; ++eIndex)
        {
            OSAL_OBJECT_HDL hList = OSAL_INVALID_OBJECT_HDL;

            // Create list
            eOsalReturnCode = OSAL.eLinkedListCreate(&hList,
                                        STOCK_TICKER_MGR_OBJECT_NAME":HashList",
                                        (OSAL_LL_COMPARE_HANDLER) n16StockMsgHashCompare,
                                        OSAL_LL_OPTION_UNIQUE |
                                        OSAL_LL_OPTION_BINARY_SEARCH |
                                        OSAL_LL_OPTION_USE_PRE_ALLOCATED_ELEMENTS
                                            );

            psAppObj->sHashCtrl.hHashList[eIndex] = hList;

            if (eOsalReturnCode != OSAL_SUCCESS)
            {
                SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    STOCK_TICKER_MGR_OBJECT_NAME": failed to create list (%s)",
                    OSAL.pacGetReturnCodeName(eOsalReturnCode)
                        );
                break;
            }
        }

        // Check previous routine result
        if (eOsalReturnCode != OSAL_SUCCESS)
        {
            break;
        }

        // Create cache list
        eOsalReturnCode = OSAL.eLinkedListCreate(
                                &psAppObj->sHashCtrl.hHashCacheList,
                                STOCK_TICKER_MGR_OBJECT_NAME":HashChacheList",
                                NULL,
                                OSAL_LL_OPTION_NONE
                                    );
        if (eOsalReturnCode != OSAL_SUCCESS)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    STOCK_TICKER_MGR_OBJECT_NAME": failed to create cache list (%s)",
                    OSAL.pacGetReturnCodeName(eOsalReturnCode)
                        );
            break;
        }

        // We're ok
        eResult = STOCK_TICKER_RETURN_CODE_SUCCESS;
    } while (FALSE);

    return eResult;
}

/*****************************************************************************
*
*   eStockMsgHashItemCreate
*
*****************************************************************************/
static STOCK_TICKER_RETURN_CODE_ENUM eStockMsgHashItemCreate (
    STOCK_TICKER_APP_OBJECT_STRUCT *psAppObj,
    UN32 un32Id,
    STOCK_TICKER_MSG_TYPE_ENUM eType,
    OSAL_CRC_RESULT tHash,
    STOCK_TICKER_MSG_HASH_STRUCT **ppsResult
        )
{
    STOCK_TICKER_RETURN_CODE_ENUM eReturn = STOCK_TICKER_RETURN_CODE_ERROR;
    OSAL_RETURN_CODE_ENUM eOsalReturnCode = OSAL_ERROR;
    STOCK_TICKER_MSG_HASH_STRUCT *psObj = NULL;

#if ((SMS_DEBUG == 1) && (STOCK_TICKER_EXTRA_DEBUG == 1))
    printf(STOCK_TICKER_MGR_OBJECT_NAME": creating hash id:%d, type=%s\n",
            un32Id, GsStockTickerMgrIntf.pacGetMessageTypeName(eType)
                );
#endif

    do
    {
        // Check input
        if ((psAppObj == NULL) || (ppsResult == NULL))
        {
            break;
        }

        // Initialize output
        *ppsResult = NULL;

        // Create object
        psObj = (STOCK_TICKER_MSG_HASH_STRUCT*)
                    OSAL.pvLinkedListMemoryAllocate(
                            STOCK_TICKER_MGR_OBJECT_NAME":HashItm",
                            sizeof(STOCK_TICKER_MSG_HASH_STRUCT),
                            FALSE);
        if (psObj == NULL)
        {
            eReturn = STOCK_TICKER_RETURN_CODE_MEMORY_ERROR;
            break;
        }

        // Fill in object data
        psObj->un32Id = un32Id;
        psObj->eType = eType;
        psObj->tHash = tHash;
        psObj->hEntry = OSAL_INVALID_LINKED_LIST_ENTRY;
        psObj->eState = STOCK_DATA_STATE_CREATED;

        // Add item to the list
        eOsalReturnCode = OSAL.eLinkedListAdd(psAppObj->sHashCtrl.hHashList[eType],
                                    &psObj->hEntry, (void*) psObj
                                        );
        if (eOsalReturnCode != OSAL_SUCCESS)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    STOCK_TICKER_MGR_OBJECT_NAME": failed to add has item 0x%p to the list (%s)",
                    psObj, OSAL.pacGetReturnCodeName(eOsalReturnCode)
                        );
            break;
        }

        *ppsResult = psObj;

        eReturn = STOCK_TICKER_RETURN_CODE_SUCCESS;
    } while (FALSE);

    if ((eReturn != STOCK_TICKER_RETURN_CODE_SUCCESS) && (psObj != NULL))
    {
        vStockMsgHashItemDestroy(psObj, TRUE);
    }

    return eReturn;
}

/*****************************************************************************
*
*   vStockMsgHashItemDestroy
*
*****************************************************************************/
static void vStockMsgHashItemDestroy (
    STOCK_TICKER_MSG_HASH_STRUCT *psObj,
    BOOLEAN bFullDelete
        )
{
    OSAL_RETURN_CODE_ENUM eOsalReturnCode = OSAL_ERROR;

    if (psObj != NULL)
    {
        printf(STOCK_TICKER_MGR_OBJECT_NAME
            ": attempting to destroy StockMsgHashItem %p\n", psObj
                );

        if ((bFullDelete == TRUE) && (psObj->hEntry != OSAL_INVALID_LINKED_LIST_ENTRY))
        {
            eOsalReturnCode = OSAL.eLinkedListRemove(psObj->hEntry);
            if (eOsalReturnCode != OSAL_SUCCESS)
            {
                SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                        STOCK_TICKER_MGR_OBJECT_NAME": failed to remove record from the list (%s)",
                        OSAL.pacGetReturnCodeName(eOsalReturnCode)
                            );
            }

            psObj->hEntry = OSAL_INVALID_LINKED_LIST_ENTRY;
        }

        OSAL.vLinkedListMemoryFree((void*) psObj);
    }

    return;
}

/*****************************************************************************
*
*   vStockMsgHashItemDestroyNotFromList
*
*****************************************************************************/
static void vStockMsgHashItemDestroyNotFromList(
    STOCK_TICKER_MSG_HASH_STRUCT *psDesc
        )
{
    vStockMsgHashItemDestroy(psDesc, FALSE);

    return;
}

/*****************************************************************************
*
*   bStockMsgHashLoadDBProcessResult
*
*****************************************************************************/
static BOOLEAN bStockMsgHashLoadDBProcessResult (
    SQL_QUERY_COLUMN_STRUCT *psColumns,
    N32 n32NumberOfColumns,
    void *pvArg
       )
{
    STOCK_TICKER_RETURN_CODE_ENUM eResult = STOCK_TICKER_RETURN_CODE_ERROR;
    STOCK_TICKER_DB_QUERY_RESULT_STRUCT *psArg =
            (STOCK_TICKER_DB_QUERY_RESULT_STRUCT*) pvArg;
    SQL_QUERY_COLUMN_STRUCT *psIdCol = NULL;
    SQL_QUERY_COLUMN_STRUCT *psTypeCol = NULL;
    SQL_QUERY_COLUMN_STRUCT *psHashCol = NULL;
    STOCK_TICKER_MSG_HASH_STRUCT *psHashItem = NULL;
    UN32 un32Id = 0;
    STOCK_TICKER_MSG_TYPE_ENUM eMsgType = STOCK_TICKER_MSG_TYPE_INVALID;

#if ((SMS_DEBUG == 1) && (STOCK_TICKER_EXTRA_DEBUG == 1))
    printf(STOCK_TICKER_MGR_OBJECT_NAME": processing DB HASH record\n");
#endif

    do
    {
        psArg->sOutput.bResultantRows = FALSE;

        // If there are the correct
        // number of columns, then we have good results
        if (n32NumberOfColumns != STOCK_TICKER_DB_MSG_HASH_MAX_FIELDS)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    STOCK_TICKER_MGR_OBJECT_NAME
                    ": incorrect query for the table (no. of fields is %d)",
                    n32NumberOfColumns
                        );
            break;
        }

        // Grab the current column data
        psIdCol = &psColumns[(UN8)STOCK_TICKER_DB_MSG_HASH_FIELD_ID];
        psTypeCol = &psColumns[(UN8)STOCK_TICKER_DB_MSG_HASH_FIELD_TYPE];
        psHashCol = &psColumns[(UN8)STOCK_TICKER_DB_MSG_HASH_FIELD_HASH];

        // Extract type
        eMsgType = (STOCK_TICKER_MSG_TYPE_ENUM) psTypeCol->uData.sUN32.un32Data;
        un32Id = psIdCol->uData.sUN32.un32Data;

        // Check the type caching necessity
        eResult = eStockMsgHashIsPersistentStorageRequired(eMsgType);
        if (eResult == STOCK_TICKER_RETURN_CODE_SUCCESS)
        {
            // Create instance
            eResult = eStockMsgHashItemCreate(psArg->sInput.psAppObj,
                                un32Id,
                                eMsgType,
                                psHashCol->uData.sUN32.un32Data,
                                &psHashItem
                                    );
            if (eResult != STOCK_TICKER_RETURN_CODE_SUCCESS)
            {
                SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                        STOCK_TICKER_MGR_OBJECT_NAME": failed to create hash item for %s (%s)",
                        GsStockTickerMgrIntf.pacGetMessageTypeName(eMsgType),
                        GsStockTickerMgrIntf.pacGetReturnCodeName(eResult)
                            );
                psArg->sOutput.bResultantRows = FALSE;
                break;
            }
        }
        else if (eResult != STOCK_TICKER_RETURN_CODE_FALSE)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    STOCK_TICKER_MGR_OBJECT_NAME": failed to check message type %s caching necessity (%s)",
                    GsStockTickerMgrIntf.pacGetMessageTypeName(eMsgType),
                    GsStockTickerMgrIntf.pacGetReturnCodeName(eResult)
                        );
            break;
        }
        else
        {
            printf(STOCK_TICKER_MGR_OBJECT_NAME": for the message type %s cashing is not required\n",
                    GsStockTickerMgrIntf.pacGetMessageTypeName(eMsgType)
                        );
        }

        // Update hash id provides by the biggest value
        if (psArg->sOutput.un32HashId < un32Id)
        {
            psArg->sOutput.un32HashId = un32Id;
        }

        // Update record state to STABLe since it is created based on data
        // from the DB
        if (psHashItem != NULL)
        {
            psHashItem->eState = STOCK_DATA_STATE_STABLE;
        }

#if SMS_DEBUG == 1
        ++psArg->sOutput.tRowCount;
#endif

        psArg->sOutput.bResultantRows = TRUE;
    } while (FALSE);

    return psArg->sOutput.bResultantRows;
}

/*****************************************************************************
*
*   eStockMsgHashLoadDB
*
*****************************************************************************/
static STOCK_TICKER_RETURN_CODE_ENUM eStockMsgHashLoadDB (
    STOCK_TICKER_APP_OBJECT_STRUCT *psAppObj
        )
{
    STOCK_TICKER_RETURN_CODE_ENUM eResult = STOCK_TICKER_RETURN_CODE_ERROR;
    BOOLEAN bOk = FALSE;
    STOCK_TICKER_DB_QUERY_RESULT_STRUCT sArg;

    printf(STOCK_TICKER_MGR_OBJECT_NAME": Loading Hash Cache from DB\n");

    do
    {
        // Check input
        if (psAppObj == NULL)
        {
            eResult = STOCK_TICKER_RETURN_CODE_BAD_INPUT;
            break;
        }

        // Init argument
        OSAL.bMemSet(&sArg, 0, sizeof(sArg));
        sArg.sInput.psAppObj = psAppObj;
#if SMS_DEBUG == 1
        sArg.sOutput.tRowCount = 0;
#endif
        sArg.sOutput.bResultantRows = TRUE;

        // Perform the SQL query and process the result
        // (it will provide us with a data row)
        bOk = SQL_INTERFACE.bQuery(
                psAppObj->sDB.hSQLConnection, STOCK_TICKER_SELECT_DB_ALL_HASH,
                bStockMsgHashLoadDBProcessResult,
                (void*)&sArg
                    );
        if (bOk != TRUE)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    STOCK_TICKER_MGR_OBJECT_NAME": failed to make query %s",
                    STOCK_TICKER_SELECT_DB_ALL_HASH
                        );
            break;
        }
        if (sArg.sOutput.bResultantRows == FALSE)
        {
            break;
        }

#if SMS_DEBUG == 1
        printf(STOCK_TICKER_MGR_OBJECT_NAME": loaded %d record(s) for Hash\n\tNext id is %d\n",
                            sArg.sOutput.tRowCount,
                            sArg.sOutput.un32HashId + 1
                                );
#endif

        psAppObj->sHashCtrl.un32NextIndex = sArg.sOutput.un32HashId + 1;

        eResult = STOCK_TICKER_RETURN_CODE_SUCCESS;
    } while (FALSE);

    return eResult;
}

/*******************************************************************************
*
*  bStockMsgHashCachePrepareRow
*
*******************************************************************************/
static BOOLEAN bStockMsgHashCachePrepareRow(
    SQL_COLUMN_INDEX tIndex,
    SQL_BIND_TYPE_ENUM *peType,
    size_t *ptDataSize,
    void **ppvData,
    STOCK_TICKER_MSG_HASH_STRUCT *psHashItem
        )
{
    BOOLEAN bSuccess = TRUE;

    switch (tIndex)
    {
        case STOCK_TICKER_DB_MSG_HASH_FIELD_ID:
        {
            *peType = SQL_BIND_TYPE_UN32;
            *ppvData = (void *)(UN32)psHashItem->un32Id;
        }
        break;

        case STOCK_TICKER_DB_MSG_HASH_FIELD_TYPE:
        {
            *peType = SQL_BIND_TYPE_UN32;
            *ppvData = (void *)(UN32)psHashItem->eType;
        }
        break;

        case STOCK_TICKER_DB_MSG_HASH_FIELD_HASH:
        {
            *peType = SQL_BIND_TYPE_UN32;
            *ppvData = (void *)(UN32)psHashItem->tHash;
        }
        break;

        case STOCK_TICKER_DB_MSG_HASH_MAX_FIELDS:
        default:
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    STOCK_TICKER_MGR_OBJECT_NAME": incorrect field index %d",
                    tIndex
                        );
            bSuccess = FALSE;
        }
        break;
    }

    return bSuccess;
}

/*****************************************************************************
*
*   eStockMsgHashIsPermanentStorageRequired
*
*****************************************************************************/
static STOCK_TICKER_RETURN_CODE_ENUM eStockMsgHashIsPersistentStorageRequired (
    STOCK_TICKER_MSG_TYPE_ENUM eType
        )
{
    STOCK_TICKER_RETURN_CODE_ENUM eReturnCode = STOCK_TICKER_RETURN_CODE_ERROR;

    switch (eType)
    {
        case STOCK_TICKER_MSG_TYPE_STOCK_SYMBOL:
        {
            eReturnCode = STOCK_TICKER_RETURN_CODE_SUCCESS;
        }
        break;
        case STOCK_TICKER_MSG_TYPE_STOCK_VALUE:
        case STOCK_TICKER_MSG_TYPE_STOCK_VALUE_EX:
        case STOCK_TICKER_MSG_TYPE_DATA_PROVIDER_MSG:
        {
            eReturnCode = STOCK_TICKER_RETURN_CODE_FALSE;
        }
        break;
        case STOCK_TICKER_MSG_TYPE_INVALID:
        case STOCK_TICKER_MSG_TYPE_MAX:
        {
            // Nothing to do
        }
        break;
    }

    return eReturnCode;
}

/*****************************************************************************
*
*   eStockMsgHashCacheProcessForDB
*
*****************************************************************************/
static STOCK_TICKER_RETURN_CODE_ENUM eStockMsgHashCacheProcessForDB (
    STOCK_TICKER_APP_OBJECT_STRUCT *psAppObj,
    STOCK_TICKER_MSG_HASH_STRUCT *psHashItem
        )
{
    STOCK_TICKER_RETURN_CODE_ENUM eResult = STOCK_TICKER_RETURN_CODE_ERROR;

    do
    {
        // Check input
        if ((psAppObj == NULL) || (psHashItem == NULL))
        {
            eResult = STOCK_TICKER_RETURN_CODE_BAD_INPUT;
            break;
        }

        // Check states for processing
        if ((psHashItem->eState != STOCK_DATA_STATE_UPDATED) &&
            (psHashItem->eState != STOCK_DATA_STATE_CREATED))
        {
            eResult = STOCK_TICKER_RETURN_CODE_NOTHING_TO_UPDATE;
            break;
        }

        // Check the hash caching necessity
        eResult = eStockMsgHashIsPersistentStorageRequired(psHashItem->eType);
        if (eResult == STOCK_TICKER_RETURN_CODE_FALSE)
        {
#if ((SMS_DEBUG == 1) && (STOCK_TICKER_EXTRA_DEBUG == 1))
            printf(STOCK_TICKER_MGR_OBJECT_NAME": skip processing hash item #%d for %s since not required in DB\n",
                    psHashItem->un32Id,
                    GsStockTickerMgrIntf.pacGetMessageTypeName(psHashItem->eType)
                        );
#endif
            eResult = STOCK_TICKER_RETURN_CODE_NOTHING_TO_UPDATE;
            break;
        }
        else if (eResult != STOCK_TICKER_RETURN_CODE_SUCCESS)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    STOCK_TICKER_MGR_OBJECT_NAME": failed to check message type %s caching necessity (%s)",
                    GsStockTickerMgrIntf.pacGetMessageTypeName(psHashItem->eType),
                    GsStockTickerMgrIntf.pacGetReturnCodeName(eResult)
                        );
            break;
        }

        // Update DB based on state
        if (psHashItem->eState == STOCK_DATA_STATE_UPDATED)
        {
            BOOLEAN bUpdated;

            printf(STOCK_TICKER_MGR_OBJECT_NAME": updating record for #%d MsgHash 0x%p\n",
                     psHashItem->un32Id, psHashItem
                        );

            snprintf(psAppObj->acBuffer, STOCK_TICKER_SHARED_BUFFER_LEN,
                    STOCK_TICKER_UPDATE_INSTEAD_OF_OLD_DB_MSG_HASH, psHashItem->un32OldId
                        );

            bUpdated = SQL_INTERFACE.bExecutePreparedCommand(
                            psAppObj->sDB.hSQLConnection,
                            psAppObj->acBuffer,
                            STOCK_TICKER_DB_STOCK_SYMBOL_MAX_FIELDS,
                            (PREPARED_QUERY_COLUMN_CALLBACK)bStockMsgHashCachePrepareRow,
                            (void *) psHashItem
                                );

            if (bUpdated == FALSE)
            {
                SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    STOCK_TICKER_MGR_OBJECT_NAME": Failed to update hash data (%s)",
                    psAppObj->acBuffer);
                eResult = STOCK_TICKER_RETURN_CODE_DB_ERROR;
                break;
            }
        }
        else if (psHashItem->eState == STOCK_DATA_STATE_CREATED)
        {
            BOOLEAN bInserted;

            printf(STOCK_TICKER_MGR_OBJECT_NAME": creating record for #%d MsgHash 0x%p\n",
                    psHashItem->un32Id, psHashItem
                        );

            bInserted = SQL_INTERFACE.bExecutePreparedCommand(
                            psAppObj->sDB.hSQLConnection,
                            STOCK_TICKER_INSERT_DB_MSG_HASH,
                            STOCK_TICKER_DB_MSG_HASH_MAX_FIELDS,
                            (PREPARED_QUERY_COLUMN_CALLBACK)bStockMsgHashCachePrepareRow,
                            (void *) psHashItem
                                );

            if (bInserted == FALSE)
            {
                SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    STOCK_TICKER_MGR_OBJECT_NAME": Failed to insert message hash data %s",
                    STOCK_TICKER_INSERT_DB_MSG_HASH);
                eResult = STOCK_TICKER_RETURN_CODE_DB_ERROR;
                break;
            }
        }
#if ((SMS_DEBUG == 1) && (STOCK_TICKER_EXTRA_DEBUG == 1))
        else
        {
            printf(STOCK_TICKER_MGR_OBJECT_NAME": skip since no DB action required for #%d\n", psHashItem->un32Id);
        }
#endif

        eResult = STOCK_TICKER_RETURN_CODE_SUCCESS;
    } while (FALSE);

    return eResult;
}

/*****************************************************************************
*
*   bStockMsgHashCacheFlushDBIterator
*
*****************************************************************************/
static BOOLEAN bStockMsgHashCacheFlushDBIterator (
    STOCK_TICKER_MSG_HASH_STRUCT *psHashItem,
    STOCK_TICKER_APP_OBJECT_STRUCT *psAppObj
        )
{
    STOCK_TICKER_RETURN_CODE_ENUM eReturnCode;

    eReturnCode = eStockMsgHashCacheProcessForDB(psAppObj, psHashItem);

    if ((eReturnCode != STOCK_TICKER_RETURN_CODE_SUCCESS) &&
        (eReturnCode != STOCK_TICKER_RETURN_CODE_NOTHING_TO_UPDATE))
    {
        SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                STOCK_TICKER_MGR_OBJECT_NAME": failed to process MsgHash 0x%p (%s)",
                psHashItem, GsStockTickerMgrIntf.pacGetReturnCodeName(eReturnCode)
                    );
        if (eReturnCode != STOCK_TICKER_RETURN_CODE_BAD_INPUT)
        {
            psHashItem->eState = STOCK_DATA_STATE_ERROR;
        }
        return FALSE;
    }

    psHashItem->eState = STOCK_DATA_STATE_STABLE;

    return TRUE;
}

/*****************************************************************************
*
*   eStockMsgHashCacheFlushDB
*
*****************************************************************************/
static STOCK_TICKER_RETURN_CODE_ENUM eStockMsgHashCacheFlushDB (
    STOCK_TICKER_APP_OBJECT_STRUCT *psAppObj
        )
{
    STOCK_TICKER_RETURN_CODE_ENUM eResult = STOCK_TICKER_RETURN_CODE_ERROR;
    OSAL_RETURN_CODE_ENUM eOsalReturnCode;

    puts(STOCK_TICKER_MGR_OBJECT_NAME": flushing hash values to the DB");

    do
    {
        // Check input
        if (psAppObj == NULL)
        {
            eResult = STOCK_TICKER_RETURN_CODE_BAD_INPUT;
            break;
        }

        // Iterate through all hash to flush them to the DB if needed
        eOsalReturnCode = OSAL.eLinkedListIterate(psAppObj->sHashCtrl.hHashCacheList,
                                    (OSAL_LL_ITERATOR_HANDLER) bStockMsgHashCacheFlushDBIterator,
                                    (void*)psAppObj
                                        );
        if ((eOsalReturnCode != OSAL_SUCCESS) && (eOsalReturnCode != OSAL_NO_OBJECTS))
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    STOCK_TICKER_MGR_OBJECT_NAME": failed to iterate through hash cache list (%s)",
                    OSAL.pacGetReturnCodeName(eOsalReturnCode)
                        );
            break;
        }

        eResult = STOCK_TICKER_RETURN_CODE_SUCCESS;
    } while (FALSE);

    return eResult;
}

/*****************************************************************************
*
*   eStockMsgHashCacheUpdate
*
*   This function tries to find the hash item with specified value and message
*   type and in case if item exists returns
*   STOCK_TICKER_RETURN_CODE_NOTHING_TO_UPDATE. If item doesn't exists return
*   STOCK_TICKER_RETURN_CODE_SUCCESS or other coder in case of error
*
*****************************************************************************/
static STOCK_TICKER_RETURN_CODE_ENUM eStockMsgHashCacheUpdate (
    STOCK_TICKER_APP_OBJECT_STRUCT *psAppObj,
    STOCK_TICKER_MSG_TYPE_ENUM eType,
    OSAL_CRC_RESULT tHash
        )
{
    STOCK_TICKER_RETURN_CODE_ENUM eResult = STOCK_TICKER_RETURN_CODE_ERROR;
    OSAL_RETURN_CODE_ENUM eOsalReturnCode = OSAL_ERROR;
    STOCK_TICKER_MSG_HASH_ITERATOR_DATA_STRUCT sIteratorData;
    STOCK_TICKER_MSG_HASH_STRUCT *psHashItem = NULL;
    OSAL_OBJECT_HDL hList = OSAL_INVALID_OBJECT_HDL;
    UN32 un32Items = 0;

    do
    {
        printf(STOCK_TICKER_MGR_OBJECT_NAME
                ": updating hash storage by data for %s and value 0x%08X\n",
                GsStockTickerMgrIntf.pacGetMessageTypeName(eType),
                tHash
                    );

        // Check input
        if (psAppObj == NULL)
        {
            eResult = STOCK_TICKER_RETURN_CODE_BAD_INPUT;
            break;
        }

        // Init search data
        OSAL.bMemSet(&sIteratorData, 0, sizeof(sIteratorData));
        sIteratorData.sInput.tHash = tHash;
        sIteratorData.sOutput.psFoundEntry = NULL;
        sIteratorData.sOutput.psOldestEntry = NULL;
        hList = psAppObj->sHashCtrl.hHashList[eType];

        // Find out number of hashes in the list
        eOsalReturnCode = OSAL.eLinkedListItems(hList, &un32Items);
        if (eOsalReturnCode != OSAL_SUCCESS)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                STOCK_TICKER_MGR_OBJECT_NAME
                ": failed to get number of hashed %s messages (%s)",
                GsStockTickerMgrIntf.pacGetMessageTypeName(eType),
                OSAL.pacGetReturnCodeName(eOsalReturnCode)
                    );
            break;
        }

        // Perform the search iterating
        eOsalReturnCode = OSAL.eLinkedListIterate(hList,
                                    bStockMsgHashIterator,
                                    (void*)&sIteratorData
                                        );
        if (eOsalReturnCode == OSAL_SUCCESS)
        {
            if (sIteratorData.sOutput.psFoundEntry != NULL)
            {
                // Since we've found the record just say it to the caller.
                eResult = STOCK_TICKER_RETURN_CODE_NOTHING_TO_UPDATE;
                break;
            }
        }
        else if (eOsalReturnCode != OSAL_NO_OBJECTS)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    STOCK_TICKER_MGR_OBJECT_NAME
                    ": failed to search hash for type %s and value 0x%08X (%s)",
                    pacGetMessageTypeName(eType),
                    tHash, OSAL.pacGetReturnCodeName(eOsalReturnCode)
                        );
            break;
        }

        // Generate next id for the next hash record
        ++psAppObj->sHashCtrl.un32NextIndex;

        // Process searching result
        if (un32Items >= STOCK_TICKER_MSG_HASH_COUNT_MAX)
        {
            // So, we exceeded limit of hashes use the oldest one for the hash

            psHashItem = sIteratorData.sOutput.psOldestEntry;
            if (psHashItem == NULL)
            {
                SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                        STOCK_TICKER_MGR_OBJECT_NAME
                        ": Oops, cannot be so..."
                            );
                break;
            }

            printf(STOCK_TICKER_MGR_OBJECT_NAME
                    ": Hash limit exceeded so reuse [#%d => #%d]\n",
                    psHashItem->un32Id, psAppObj->sHashCtrl.un32NextIndex
                        );

            // Update the entry
            psHashItem->un32OldId = psHashItem->un32Id;
            psHashItem->un32Id = psAppObj->sHashCtrl.un32NextIndex;
            psHashItem->tHash = tHash;
            psHashItem->eState = STOCK_DATA_STATE_UPDATED;
            eResult = STOCK_TICKER_RETURN_CODE_SUCCESS;
        }
        else
        {
            printf(STOCK_TICKER_MGR_OBJECT_NAME
                    ": Hash limit is not exceeded so create new hash handle for #%d\n",
                    psAppObj->sHashCtrl.un32NextIndex
                        );

            eResult = eStockMsgHashItemCreate(psAppObj,
                                    psAppObj->sHashCtrl.un32NextIndex,
                                    eType, tHash,
                                    &psHashItem
                                        );
        }

        // Check the hash item existence and add it to the hash processing
        // if exists
        if (psHashItem != NULL)
        {
            // Put the hash value to the hash cache list
            eOsalReturnCode = OSAL.eLinkedListAdd(psAppObj->sHashCtrl.hHashCacheList,
                                    OSAL_INVALID_LINKED_LIST_ENTRY_PTR,
                                    (void*)psHashItem
                                        );
            if (eOsalReturnCode != OSAL_SUCCESS)
            {
                SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                        STOCK_TICKER_MGR_OBJECT_NAME
                        ": failed to add the HashItem 0x%p to the cache list (%s)",
                        psHashItem, OSAL.pacGetReturnCodeName(eOsalReturnCode)
                            );
                eResult = STOCK_TICKER_RETURN_CODE_ERROR;
                break;
            }
        }
    } while (FALSE);

    return eResult;
}

/*****************************************************************************
*
*   eStockMsgHashCacheClean
*
*****************************************************************************/
static STOCK_TICKER_RETURN_CODE_ENUM eStockMsgHashCacheClean (
    STOCK_TICKER_APP_OBJECT_STRUCT *psAppObj
        )
{
    STOCK_TICKER_RETURN_CODE_ENUM eReturnCode = STOCK_TICKER_RETURN_CODE_ERROR;
    OSAL_RETURN_CODE_ENUM eOsalReturnCode = OSAL_ERROR;

    puts(STOCK_TICKER_MGR_OBJECT_NAME": cleaning messages hash cache");

    do
    {
        // Check input
        if (psAppObj == NULL)
        {
            eReturnCode = STOCK_TICKER_RETURN_CODE_BAD_INPUT;
            break;
        }

        eOsalReturnCode = OSAL.eLinkedListRemoveAll(
                                    psAppObj->sHashCtrl.hHashCacheList,
                                    NULL);
        if (eOsalReturnCode != OSAL_SUCCESS)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    STOCK_TICKER_MGR_OBJECT_NAME
                    ": failed to clean the hash cache (%s)",
                    OSAL.pacGetReturnCodeName(eOsalReturnCode)
                        );
            break;
        }

        eReturnCode = STOCK_TICKER_RETURN_CODE_SUCCESS;
    } while (FALSE);

    return eReturnCode;
}

/*****************************************************************************
*
*   eStockMsgHashClean
*
*   Remove all hash records related to some particular message type
*
*****************************************************************************/
static STOCK_TICKER_RETURN_CODE_ENUM eStockMsgHashClean (
    STOCK_TICKER_APP_OBJECT_STRUCT *psAppObj,
    STOCK_TICKER_MSG_TYPE_ENUM eType
        )
{
    STOCK_TICKER_RETURN_CODE_ENUM eReturnCode = STOCK_TICKER_RETURN_CODE_ERROR;
    OSAL_OBJECT_HDL hList = OSAL_INVALID_OBJECT_HDL;
    OSAL_RETURN_CODE_ENUM eOsalReturnCode = OSAL_ERROR;

    do
    {
        printf(STOCK_TICKER_MGR_OBJECT_NAME": clean-up hashes for %s\n",
            GsStockTickerMgrIntf.pacGetMessageTypeName(eType));

        // Get an appropriate list
        hList = psAppObj->sHashCtrl.hHashList[eType];
        if (hList == OSAL_INVALID_OBJECT_HDL)
        {
            eReturnCode = STOCK_TICKER_RETURN_CODE_BAD_INPUT;
            break;
        }

        // Remove all entries
        eOsalReturnCode = OSAL.eLinkedListRemoveAll(hList,
                                    (OSAL_LL_RELEASE_HANDLER)vStockMsgHashItemDestroyNotFromList
                                        );
        if (eOsalReturnCode != OSAL_SUCCESS)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                STOCK_TICKER_MGR_OBJECT_NAME": failed to clean up hash list %s (%s)",
                OSAL.pacGetReturnCodeName(eOsalReturnCode),
                GsStockTickerMgrIntf.pacGetMessageTypeName(eType)
                    );
            break;
        }

        eReturnCode = STOCK_TICKER_RETURN_CODE_SUCCESS;
    } while (FALSE);

    return eReturnCode;
}

/*****************************************************************************
*
*   eStockMsgHashValuesClean
*
*   Remove all hash records related to VALUE and VALUE_EX messages
*
*****************************************************************************/
static STOCK_TICKER_RETURN_CODE_ENUM eStockMsgHashValuesClean (
    STOCK_TICKER_APP_OBJECT_STRUCT *psAppObj
        )
{
    STOCK_TICKER_RETURN_CODE_ENUM eReturnCode;

    printf(STOCK_TICKER_MGR_OBJECT_NAME": clean-up value and value-ex hashes\n");

    eReturnCode = eStockMsgHashClean(psAppObj, STOCK_TICKER_MSG_TYPE_STOCK_VALUE);
    if (eReturnCode != STOCK_TICKER_RETURN_CODE_SUCCESS)
    {
        return eReturnCode;
    }

    eReturnCode = eStockMsgHashClean(psAppObj, STOCK_TICKER_MSG_TYPE_STOCK_VALUE_EX);
    if (eReturnCode != STOCK_TICKER_RETURN_CODE_SUCCESS)
    {
        return eReturnCode;
    }

    return STOCK_TICKER_RETURN_CODE_SUCCESS;
}

/*****************************************************************************
*
*   eStockMsgHashUninit
*
*****************************************************************************/
static STOCK_TICKER_RETURN_CODE_ENUM eStockMsgHashUninit (
    STOCK_TICKER_APP_OBJECT_STRUCT *psAppObj
        )
{
    STOCK_TICKER_RETURN_CODE_ENUM eResult = STOCK_TICKER_RETURN_CODE_ERROR;
    STOCK_TICKER_MSG_TYPE_ENUM eIndex;
    OSAL_RETURN_CODE_ENUM eOsalReturnCode = OSAL_ERROR;

    printf(STOCK_TICKER_MGR_OBJECT_NAME": Uninitializing Stock Messages Hash\n");

    do
    {
        // Check input
        if (psAppObj == NULL)
        {
            eResult = STOCK_TICKER_RETURN_CODE_BAD_INPUT;
            break;
        }

        // Destroy the hashes cache list
        if (psAppObj->sHashCtrl.hHashCacheList != OSAL_INVALID_OBJECT_HDL)
        {
            eOsalReturnCode = OSAL.eLinkedListRemoveAll(
                                    psAppObj->sHashCtrl.hHashCacheList,
                                    NULL);
            if (eOsalReturnCode != OSAL_SUCCESS)
            {
                SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                        STOCK_TICKER_MGR_OBJECT_NAME": failed to clean up cache list of hashes (%s)",
                        OSAL.pacGetReturnCodeName(eOsalReturnCode)
                            );
            }

            eOsalReturnCode = OSAL.eLinkedListDelete(psAppObj->sHashCtrl.hHashCacheList);
            if (eOsalReturnCode != OSAL_SUCCESS)
            {
                SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                        STOCK_TICKER_MGR_OBJECT_NAME": failed to destroy cache list of hashes (%s)",
                        OSAL.pacGetReturnCodeName(eOsalReturnCode)
                            );
            }
        }

        // Destroy the hash lists
        for (eIndex = STOCK_TICKER_MSG_TYPE_START; eIndex != STOCK_TICKER_MSG_TYPE_MAX; ++eIndex)
        {
            OSAL_OBJECT_HDL hList = psAppObj->sHashCtrl.hHashList[eIndex];
            if (hList != OSAL_INVALID_OBJECT_HDL)
            {
                eOsalReturnCode = OSAL.eLinkedListRemoveAll(
                    hList,
                    (OSAL_LL_RELEASE_HANDLER)vStockMsgHashItemDestroyNotFromList
                        );
                if (eOsalReturnCode != OSAL_SUCCESS)
                {
                    SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                        STOCK_TICKER_MGR_OBJECT_NAME": failed to clean up list of hashes (%s)",
                        OSAL.pacGetReturnCodeName(eOsalReturnCode)
                            );
                }

                eOsalReturnCode = OSAL.eLinkedListDelete(hList);
                if (eOsalReturnCode != OSAL_SUCCESS)
                {
                    SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                        STOCK_TICKER_MGR_OBJECT_NAME": failed to destroy list of hashes (%s)",
                        OSAL.pacGetReturnCodeName(eOsalReturnCode)
                            );
                }

                psAppObj->sHashCtrl.hHashList[eIndex] = OSAL_INVALID_OBJECT_HDL;
            }
        }

        psAppObj->sHashCtrl.hHashCacheList = OSAL_INVALID_OBJECT_HDL;
        psAppObj->sHashCtrl.un32NextIndex = 0;

        eResult = STOCK_TICKER_RETURN_CODE_SUCCESS;
    } while (FALSE);

    return eResult;
}

/*****************************************************************************
*
*   n16StockMsgHashCompare
*
*****************************************************************************/
static N16 n16StockMsgHashCompare (
    STOCK_TICKER_MSG_HASH_STRUCT *psObj1,
    STOCK_TICKER_MSG_HASH_STRUCT *psObj2
        )
{
    return COMPARE( psObj1->un32Id, psObj2->un32Id );
}

/*****************************************************************************
*
*   bStockMsgHashIterator
*
*****************************************************************************/
static BOOLEAN bStockMsgHashIterator (
    void *pvData,
    void *pvArg
        )
{
    STOCK_TICKER_MSG_HASH_STRUCT *psObj =
            (STOCK_TICKER_MSG_HASH_STRUCT*)pvData;
    STOCK_TICKER_MSG_HASH_ITERATOR_DATA_STRUCT *psIteratorData =
            (STOCK_TICKER_MSG_HASH_ITERATOR_DATA_STRUCT*)pvArg;

    if (psObj->tHash == psIteratorData->sInput.tHash)
    {
        // Since we found the same hash we don't need to
        // look at rest elements.
        psIteratorData->sOutput.psFoundEntry = psObj;
        return FALSE;
    }

    if ((psIteratorData->sOutput.psOldestEntry == NULL) ||
        (psIteratorData->sOutput.psOldestEntry->un32Id > psObj->un32Id))
    {
        psIteratorData->sOutput.psOldestEntry = psObj;
    }

    return TRUE;
}

/*****************************************************************************
*
*   n16StockMsgDescIndexCompareHandler
*
*****************************************************************************/
static N16 n16StockMsgDescIndexCompareHandler (
    STOCK_MSG_DESC_STRUCT *psObj1,
    STOCK_MSG_DESC_STRUCT *psObj2
        )
{
    STOCK_ID tStockIndex1;
    STOCK_ID tStockIndex2;

    tStockIndex1 = STOCK_MSG_tIndex(psObj1->hMessage);
    tStockIndex2 = STOCK_MSG_tIndex(psObj2->hMessage);

    return COMPARE(tStockIndex1, tStockIndex2);
}

/*****************************************************************************
*
*   eStockMsgDescInit
*
*****************************************************************************/
static STOCK_TICKER_RETURN_CODE_ENUM eStockMsgDescInit (
    STOCK_TICKER_MGR_OBJECT_STRUCT *psObj
        )
{
    STOCK_TICKER_RETURN_CODE_ENUM eResult = STOCK_TICKER_RETURN_CODE_ERROR;
    OSAL_RETURN_CODE_ENUM eOsalReturnCode;
    do
    {
        // Check input
        if (psObj == NULL)
        {
            eResult = STOCK_TICKER_RETURN_CODE_BAD_INPUT;
            break;
        }

        // Create symbols list
        eOsalReturnCode = OSAL.eLinkedListCreate(
                            &psObj->psAppObj->hSymbolMsgList,
                            STOCK_TICKER_MGR_OBJECT_NAME":MsgList",
                            (OSAL_LL_COMPARE_HANDLER)
                                n16StockMsgDescIndexCompareHandler,
                            OSAL_LL_OPTION_LINEAR |
                            OSAL_LL_OPTION_BINARY_SEARCH
                                );
        if (eOsalReturnCode != OSAL_SUCCESS)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    STOCK_TICKER_MGR_OBJECT_NAME": failed to create stock messages list (%s)",
                    OSAL.pacGetReturnCodeName(eOsalReturnCode)
                        );
            break;
        }

        // Create symbols pool list
        eOsalReturnCode = OSAL.eLinkedListCreate(
                            &psObj->psAppObj->hSymbolMsgPoolList,
                            STOCK_TICKER_MGR_OBJECT_NAME":MsgPoolList",
                            NULL, OSAL_LL_OPTION_LINEAR
                                );
        if (eOsalReturnCode != OSAL_SUCCESS)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    STOCK_TICKER_MGR_OBJECT_NAME": failed to create stock messages pool list (%s)",
                    OSAL.pacGetReturnCodeName(eOsalReturnCode)
                        );
            break;
        }

        eResult = STOCK_TICKER_RETURN_CODE_SUCCESS;
    } while (FALSE);

    return eResult;
}

/*******************************************************************************
*
*   bStockMsgDescIterateDBProcessResult
*
*   Processing the DB select request for the symbols iteration.
*
*   In case of the caller provided the callback function the data will be
*   populated with the caller only.
*
*******************************************************************************/
static BOOLEAN bStockMsgDescIterateDBProcessResult (
    SQL_QUERY_COLUMN_STRUCT *psColumns,
    N32 n32NumberOfColumns,
    void *pvArg
        )
{
    STOCK_TICKER_DB_QUERY_RESULT_STRUCT *psArg =
            (STOCK_TICKER_DB_QUERY_RESULT_STRUCT *)pvArg;
    STOCK_TICKER_APP_OBJECT_STRUCT *psAppObj= psArg->sInput.psAppObj;
    const char *pacName = NULL;
    BOOLEAN bOk = FALSE;
    SQL_QUERY_COLUMN_STRUCT *psNameCol = NULL;

#if ((SMS_DEBUG == 1) && (STOCK_TICKER_EXTRA_DEBUG == 1))
    printf(STOCK_TICKER_MGR_OBJECT_NAME": processing DB MSG record\n");
#endif

    do
    {
        if (psAppObj == NULL)
        {
            break;
        }

        // If there are the correct
        // number of columns, then we have good results
        if (n32NumberOfColumns != STOCK_TICKER_DB_STOCK_SYMBOL_MAX_FIELDS)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    STOCK_TICKER_MGR_OBJECT_NAME
                    ": incorrect query for the table (no. of fields is %d)",
                    n32NumberOfColumns
                        );
            psArg->sOutput.bResultantRows = FALSE;
            break;
        }

        psArg->sOutput.bResultantRows = TRUE;

        // Grab the current column data
        psNameCol = &psColumns[(UN8)STOCK_TICKER_DB_STOCK_SYMBOL_FIELD_NAME];

        // Extract the data from the fields
        pacName = (const char*)psNameCol->uData.sCString.pcData;

        // Notify the application
        bOk = psArg->sInput.bIterator(
                            psArg->sInput.hService,
                            pacName,
                            psArg->sInput.pvIteratorArg
                                );

#if SMS_DEBUG == 1
        ++psArg->sOutput.tRowCount;
#endif
    } while (FALSE);

    return bOk;
}

/*****************************************************************************
*
*   bStockMsgDescFlushDBIterator
*
*****************************************************************************/
static BOOLEAN bStockMsgDescFlushDBIterator(
    STOCK_MSG_DESC_STRUCT *psMsgDesc,
    STOCK_TICKER_APP_OBJECT_STRUCT *psAppObj
        )
{
    STOCK_TICKER_RETURN_CODE_ENUM eReturnCode = STOCK_TICKER_RETURN_CODE_ERROR;

    // Do some action only in case if the message has NAME as updated field.
    if ((psMsgDesc->tFieldMask & STOCK_TICKER_MSG_FIELD_MASK_NAME) == STOCK_TICKER_MSG_FIELD_MASK_NAME)
    {
        eReturnCode = eStockMsgDescProcessForDB(psAppObj, psMsgDesc);

        if ((eReturnCode != STOCK_TICKER_RETURN_CODE_SUCCESS) &&
            (eReturnCode != STOCK_TICKER_RETURN_CODE_NOTHING_TO_UPDATE))
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    STOCK_TICKER_MGR_OBJECT_NAME": failed to process MsdDesc 0x%p (%s)",
                    psMsgDesc, GsStockTickerMgrIntf.pacGetReturnCodeName(eReturnCode)
                        );
            vStockMsgDescSetState(psMsgDesc, STOCK_DATA_STATE_ERROR);
            return FALSE;
        }

        vStockMsgDescSetState(psMsgDesc, STOCK_DATA_STATE_STABLE);
    }

    return TRUE;
}

/*****************************************************************************
*
*   eStockMsgDescFlushDB
*
*****************************************************************************/
static STOCK_TICKER_RETURN_CODE_ENUM eStockMsgDescFlushDB(
    STOCK_TICKER_APP_OBJECT_STRUCT *psAppObj,
    BOOLEAN bFullData
        )
{
    STOCK_TICKER_RETURN_CODE_ENUM eReturnCode = STOCK_TICKER_RETURN_CODE_ERROR;
    OSAL_RETURN_CODE_ENUM eOsalReturnCode = OSAL_SUCCESS;
    OSAL_OBJECT_HDL hList = OSAL_INVALID_OBJECT_HDL;

    printf(STOCK_TICKER_MGR_OBJECT_NAME": Loading Symbols DB\n");

    do
    {
        // Check input
        if (psAppObj == NULL)
        {
            eReturnCode = STOCK_TICKER_RETURN_CODE_BAD_INPUT;
            break;
        }

        // Iterate through all symbols and update DB based on
        // symbols which not in the "STABLE" state
        if (bFullData == FALSE)
        {
            hList = psAppObj->hSymbolMsgCacheList;
        }
        else
        {
            hList = psAppObj->hSymbolMsgList;
        }

        eOsalReturnCode = OSAL.eLinkedListIterate(
            hList,
            (OSAL_LL_ITERATOR_HANDLER) bStockMsgDescFlushDBIterator,
            (void*)psAppObj
                );
        if ((eOsalReturnCode != OSAL_SUCCESS) &&
            (eOsalReturnCode != OSAL_NO_OBJECTS))
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    STOCK_TICKER_MGR_OBJECT_NAME": failed to iterate message descs list (%s)",
                    OSAL.pacGetReturnCodeName(eOsalReturnCode)
                        );
            break;
        }

        eReturnCode = STOCK_TICKER_RETURN_CODE_SUCCESS;
    } while (FALSE);

    return eReturnCode;
}

/*****************************************************************************
*
*   eStockMsgDescUnInit
*
*****************************************************************************/
static STOCK_TICKER_RETURN_CODE_ENUM eStockMsgDescUnInit (
    STOCK_TICKER_MGR_OBJECT_STRUCT *psObj
        )
{
    STOCK_TICKER_RETURN_CODE_ENUM eResult = STOCK_TICKER_RETURN_CODE_ERROR;
    OSAL_RETURN_CODE_ENUM eOsalReturnCode = OSAL_ERROR;
    STOCK_TICKER_APP_OBJECT_STRUCT *psAppObj = NULL;

    printf(STOCK_TICKER_MGR_OBJECT_NAME": Uninitializing Stock Messages Control Structure \n");

    do
    {
        // Check input
        if (psObj == NULL)
        {
            eResult = STOCK_TICKER_RETURN_CODE_BAD_INPUT;
            break;
        }

        psAppObj = psObj->psAppObj;

        // Clean up active messages list
        if (psAppObj->hSymbolMsgList != OSAL_INVALID_OBJECT_HDL)
        {
            // Just remove all items w/o destroying items
            eOsalReturnCode = OSAL.eLinkedListRemoveAll(
                                psAppObj->hSymbolMsgList, NULL
                                    );
            if (eOsalReturnCode != OSAL_SUCCESS)
            {
                SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                        STOCK_TICKER_MGR_OBJECT_NAME": failed to remove all items from the list (%s)",
                        OSAL.pacGetReturnCodeName(eOsalReturnCode)
                            );
            }

            eOsalReturnCode = OSAL.eLinkedListDelete(
                                psAppObj->hSymbolMsgList
                                    );
            if (eOsalReturnCode != OSAL_SUCCESS)
            {
                SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                        STOCK_TICKER_MGR_OBJECT_NAME": failed to delete the list(%s)",
                        OSAL.pacGetReturnCodeName(eOsalReturnCode)
                            );
            }
            psAppObj->hSymbolMsgList = OSAL_INVALID_OBJECT_HDL;
        }

        // Clean up messages pool
        if (psAppObj->hSymbolMsgPoolList != OSAL_INVALID_OBJECT_HDL)
        {
            eOsalReturnCode = OSAL.eLinkedListRemoveAll(
                                psAppObj->hSymbolMsgPoolList,
                                (OSAL_LL_RELEASE_HANDLER)vStockMsgDescDestroy
                                    );
            if (eOsalReturnCode != OSAL_SUCCESS)
            {
                SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                        STOCK_TICKER_MGR_OBJECT_NAME": failed to remove all items from the pool list (%s)",
                        OSAL.pacGetReturnCodeName(eOsalReturnCode)
                            );
            }

            eOsalReturnCode = OSAL.eLinkedListDelete(
                                psAppObj->hSymbolMsgPoolList
                                    );
            if (eOsalReturnCode != OSAL_SUCCESS)
            {
                SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                        STOCK_TICKER_MGR_OBJECT_NAME": failed to delete the pool list(%s)",
                        OSAL.pacGetReturnCodeName(eOsalReturnCode)
                            );
            }
            psAppObj->hSymbolMsgPoolList = OSAL_INVALID_OBJECT_HDL;
        }

        eResult = STOCK_TICKER_RETURN_CODE_SUCCESS;
    } while (FALSE);

    return eResult;
}

/*****************************************************************************
*
*   eStockMsgDescCreate
*
*****************************************************************************/
static STOCK_TICKER_RETURN_CODE_ENUM eStockMsgDescCreate (
    STOCK_MSG_DESC_STRUCT **ppsDesc,
    STOCK_TICKER_APP_OBJECT_STRUCT *psAppObj,
    STOCK_MSG_DATA_STRUCT *psMsgData
        )
{
    STOCK_TICKER_RETURN_CODE_ENUM eResult = STOCK_TICKER_RETURN_CODE_ERROR;
    OSAL_RETURN_CODE_ENUM eOsalReturnCode = OSAL_ERROR;
    STOCK_MSG_DESC_STRUCT *psDesc = NULL;
    OSAL_LINKED_LIST_ENTRY hPoolEntry = OSAL_INVALID_LINKED_LIST_ENTRY;
    STOCK_MSG_DESC_STRUCT *psPoolDesc = NULL;
    STOCK_MSG_DATA_STRUCT sData;

    do
    {

        // Check input
        if ((psAppObj == NULL) || (ppsDesc == NULL) || (psMsgData == NULL))
        {
            eResult = STOCK_TICKER_RETURN_CODE_BAD_INPUT;
            break;
        }

        sData = *psMsgData;
        *ppsDesc = NULL;

        // Correct the creation structure if price has not been specified
        // because we must be sure that its state is correct
        if ((sData.tMask & STOCK_MSG_DATA_VALUES) != STOCK_MSG_DATA_VALUES)
        {
            sData.tMask |= STOCK_MSG_DATA_VALUES;
            sData.sValues.eState = STOCK_MSG_STATE_PRICE_NOT_AVAILABLE;
            OSAL.bMemSet(&sData.sValues.sQuote, 0, sizeof(sData.sValues.sQuote));
        }

        // Correct the creation structure if name has not been specified
        // because we must be sure that it is correct
        if ((sData.tMask & STOCK_MSG_DATA_NAME_ALL) == STOCK_MSG_DATA_NONE)
        {
            sData.tMask |= STOCK_MSG_DATA_NAME_CSTR;
            sData.uName.sCStr.pacValue = "\0";
            sData.uName.sCStr.tValueLength = 1;
        }

        // Check the pool - probably we have not-used records
        hPoolEntry = OSAL.hLinkedListFirst(psAppObj->hSymbolMsgPoolList, (void**)&psPoolDesc);
        if ((hPoolEntry != OSAL_INVALID_LINKED_LIST_ENTRY) &&
            (psPoolDesc->bIsAllocated == FALSE))
        {
            printf(STOCK_TICKER_MGR_OBJECT_NAME": reuse MsgDesc %p\n", psPoolDesc);

            // Populate this to the caller
            psDesc = psPoolDesc;

            eResult = STOCK_MSG_eUpdate(psPoolDesc->hMessage, &sData);
            if ((eResult != STOCK_TICKER_RETURN_CODE_SUCCESS) &&
                (eResult != STOCK_TICKER_RETURN_CODE_NOTHING_TO_UPDATE))
            {
                SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    STOCK_TICKER_MGR_OBJECT_NAME
                    ": failed to update for STOCK_MSG for #%d (%s)",
                    sData.tIndex,
                    GsStockTickerMgrIntf.pacGetReturnCodeName(eResult)
                        );
                break;
            }
        }
        else
        {
            STOCK_MSG_OBJECT hMsg = STOCK_MSG_INVALID_OBJECT;

            puts(STOCK_TICKER_MGR_OBJECT_NAME": create a new MsgDesc");

            // Create message itself
            eResult = STOCK_MSG_eCreate(&hMsg, (SMS_OBJECT)psAppObj, &sData,
                                            sizeof(STOCK_MSG_DESC_STRUCT)
                                                );
            if (eResult != STOCK_TICKER_RETURN_CODE_SUCCESS)
            {
                SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    STOCK_TICKER_MGR_OBJECT_NAME
                    ": failed to create STOCK_MSG for #%d [mask: %04X] (%s)",
                    sData.tIndex, sData.tMask,
                    GsStockTickerMgrIntf.pacGetReturnCodeName(eResult)
                        );
                eResult = STOCK_TICKER_RETURN_CODE_MEMORY_ERROR;
                break;
            }

            // Get new object descriptor
            psDesc =
                (STOCK_MSG_DESC_STRUCT*)DSRL_ENTRY_pvServiceData((DSRL_ENTRY_OBJECT)hMsg);
            if (psDesc == NULL)
            {
                eResult = STOCK_TICKER_RETURN_CODE_ERROR;
                break;
            }

            // Init values
            OSAL.bMemSet(psDesc, 0, sizeof(STOCK_MSG_DESC_STRUCT));
            psDesc->hPoolEntry = OSAL_INVALID_LINKED_LIST_ENTRY;
            psDesc->hMessage = hMsg;
        }

        // Update message's pool position by moving (adding) it to the bottom
        eResult = eStockMsgDescUpdateInPool(psAppObj, psDesc, TRUE);
        if (eResult != STOCK_TICKER_RETURN_CODE_SUCCESS)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                STOCK_TICKER_MGR_OBJECT_NAME
                ": failed STOCK_MSG in the pool (%s)",
                GsStockTickerMgrIntf.pacGetReturnCodeName(eResult)
                    );
            break;
        }

        // Init values
        psDesc->eState = STOCK_DATA_STATE_INITIAL;
        psDesc->hEntry = OSAL_INVALID_LINKED_LIST_ENTRY;
        psDesc->tFieldMask = STOCK_TICKER_MSG_FIELD_MASK_NONE;
        psDesc->un32Ref = 0;
        psDesc->un32LastUpdateTimeStamp = 0;

        // Add message to the pool
        eOsalReturnCode = OSAL.eLinkedListAdd(psAppObj->hSymbolMsgList,
                            &psDesc->hEntry,
                            (void*) psDesc
                                );
        if (eOsalReturnCode != OSAL_SUCCESS)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    STOCK_TICKER_MGR_OBJECT_NAME": failed to add MsgDesc 0x%p to the list (%s)",
                    psDesc, OSAL.pacGetReturnCodeName(eOsalReturnCode)
                        );
            eResult = STOCK_TICKER_RETURN_CODE_ERROR;
            break;
        }

        // Populate created object
        *ppsDesc = psDesc;

        eResult = STOCK_TICKER_RETURN_CODE_SUCCESS;
    } while (FALSE);

    if ((eResult != STOCK_TICKER_RETURN_CODE_SUCCESS) && (psDesc != NULL))
    {
        vStockMsgDescDestroyFull(psDesc);
    }

    return eResult;
}

/*****************************************************************************
*
*   eStockMsgDescUpdateInPool
*
*****************************************************************************/
static STOCK_TICKER_RETURN_CODE_ENUM eStockMsgDescUpdateInPool(
    STOCK_TICKER_APP_OBJECT_STRUCT *psAppObj,
    STOCK_MSG_DESC_STRUCT *psDesc,
    BOOLEAN bIsAllocated
        )
{
    STOCK_TICKER_RETURN_CODE_ENUM eReturnCode = STOCK_TICKER_RETURN_CODE_ERROR;
    OSAL_RETURN_CODE_ENUM eOsalReturnCode = OSAL_ERROR;
    OSAL_LINKED_LIST_ENTRY hPoolEntry = OSAL_INVALID_LINKED_LIST_ENTRY;

    // Update message's pool position by moving (adding) it to the bottom
    if (psDesc->bIsAllocated != bIsAllocated)
    {
        if (psDesc->hPoolEntry != OSAL_INVALID_LINKED_LIST_ENTRY)
        {
            eOsalReturnCode = OSAL.eLinkedListRemove(psDesc->hPoolEntry);
            psDesc->hPoolEntry = OSAL_INVALID_LINKED_LIST_ENTRY;
            if (eOsalReturnCode != OSAL_SUCCESS)
            {
                SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    STOCK_TICKER_MGR_OBJECT_NAME
                    ": failed to remove MsgDesc 0x%p from the pool (%s)",
                    psDesc, OSAL.pacGetReturnCodeName(eOsalReturnCode)
                    );
                eReturnCode = STOCK_TICKER_RETURN_CODE_ERROR;
                return eReturnCode;
            }
        }

        if (bIsAllocated == TRUE)
        {
            // Allocated item will be added after the last item
            hPoolEntry = OSAL.hLinkedListLast(psAppObj->hSymbolMsgPoolList, NULL);
            if (hPoolEntry == OSAL_INVALID_LINKED_LIST_ENTRY)
            {
                eOsalReturnCode = OSAL.eLinkedListAdd(
                    psAppObj->hSymbolMsgPoolList,
                    &hPoolEntry,
                    psDesc
                        );
            }
            else
            {
                eOsalReturnCode = OSAL.eLinkedListAddAfterEntry(
                    psAppObj->hSymbolMsgPoolList,
                    &hPoolEntry,
                    psDesc
                        );
            }

        }
        else
        {
            // Allocated item will be added before the last item
            hPoolEntry = OSAL.hLinkedListFirst(psAppObj->hSymbolMsgPoolList, NULL);
            if (hPoolEntry == OSAL_INVALID_LINKED_LIST_ENTRY)
            {
                eOsalReturnCode = OSAL.eLinkedListAdd(
                    psAppObj->hSymbolMsgPoolList,
                    &hPoolEntry,
                    psDesc
                        );
            }
            else
            {
                eOsalReturnCode = OSAL.eLinkedListAddBeforeEntry(
                    psAppObj->hSymbolMsgPoolList,
                    &hPoolEntry,
                    psDesc
                        );
            }

        }

        if (eOsalReturnCode != OSAL_SUCCESS)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    STOCK_TICKER_MGR_OBJECT_NAME
                    ": failed to add MsgDesc 0x%p to the list (%s)",
                    psDesc, OSAL.pacGetReturnCodeName(eOsalReturnCode)
                        );
            eReturnCode = STOCK_TICKER_RETURN_CODE_ERROR;
        }
        else
        {
            eReturnCode = STOCK_TICKER_RETURN_CODE_SUCCESS;
        }

        psDesc->bIsAllocated = bIsAllocated;
        psDesc->hPoolEntry = hPoolEntry;
    }
    else
    {
        eReturnCode = STOCK_TICKER_RETURN_CODE_SUCCESS;
    }

    return eReturnCode;
}

/*****************************************************************************
*
*   vStockMsgDescSetState
*
*****************************************************************************/
static void vStockMsgDescSetState(
    STOCK_MSG_DESC_STRUCT *psDesc,
    STOCK_DATA_STATE_ENUM eState
        )
{
    if ((psDesc != NULL) && (psDesc->eState != eState))
    {
#if ((SMS_DEBUG==1) && (STOCK_TICKER_EXTRA_DEBUG==1))
        printf(STOCK_TICKER_MGR_OBJECT_NAME": moved MsgDesc 0x%p [#%d] from %s to %s\n",
                psDesc, STOCK_MSG.tIndex(psDesc->hMessage),
                pacGetStockDataStateName(psDesc->eState),
                pacGetStockDataStateName(eState)
                    );
#endif
        psDesc->eState = eState;
    }

    return;
}

/*****************************************************************************
*
*   bStockMsgDescExistsInDBProcessResult
*
*****************************************************************************/
static BOOLEAN bStockMsgDescExistsInDBProcessResult (
    SQL_QUERY_COLUMN_STRUCT *psColumns,
    N32 n32NumberOfColumns,
    void *pvArg
        )
{
    STOCK_TICKER_DB_QUERY_RESULT_STRUCT *psArg =
            (STOCK_TICKER_DB_QUERY_RESULT_STRUCT *)pvArg;
    size_t tStringSize = 0;

    if ((n32NumberOfColumns != STOCK_TICKER_SELECT_DB_SYMBOL_BY_ID_COLUMN_SIZE) ||
        (psColumns->uData.sCString.pcData == NULL) ||
        (psArg->sOutput.bResultantRows == TRUE)
            )
    {
        return FALSE;
    }

    // Check the current state of the output
     if (psArg->sOutput.uDbRow.sSymbol.hName != STRING_INVALID_OBJECT)
    {
        SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
            STOCK_TICKER_MGR_OBJECT_NAME": name already provided.");
        return FALSE;
    }

    // Calculate the string length
    tStringSize = (size_t)strlen((const char*)psColumns->uData.sCString.pcData);

    // Create the object based on the RAW string
    psArg->sOutput.uDbRow.sSymbol.hName =
            STRING_hCreate(
                    (SMS_OBJECT)psArg->sInput.psAppObj,
                    (const char*)psColumns->uData.sCString.pcData,
                    GsStockTickerIntf.tSymbolNameMaxSize + 1,
                    tStringSize + 1
                        );
    if (psArg->sOutput.uDbRow.sSymbol.hName == STRING_INVALID_OBJECT)
    {
        SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                STOCK_TICKER_MGR_OBJECT_NAME": failed to create string object");
        return FALSE;
    }

    // The only one field is requested and it should have the stock symbol name
    psArg->sOutput.bResultantRows = TRUE;

    return FALSE;
}

/*****************************************************************************
*
*   eStockMsgDescGetNameFromDB
*
*   Output:
*       STOCK_TICKER_RETURN_CODE_SUCCESS
*               - the item in the DB and name is loaded
*       STOCK_TICKER_RETURN_CODE_FALSE
*               - information for the item in the DB completely absent.
*
*****************************************************************************/
static STOCK_TICKER_RETURN_CODE_ENUM eStockMsgDescGetNameFromDB (
    STOCK_TICKER_APP_OBJECT_STRUCT *psAppObj,
    STOCK_ID tIndex,
    STRING_OBJECT *phName
        )
{
    STOCK_TICKER_RETURN_CODE_ENUM eResult = STOCK_TICKER_RETURN_CODE_ERROR;
    STOCK_TICKER_DB_QUERY_RESULT_STRUCT sArg;
    BOOLEAN bOk = FALSE;

#if ((SMS_DEBUG == 1) && (STOCK_TICKER_EXTRA_DEBUG == 1))
    printf(STOCK_TICKER_MGR_OBJECT_NAME
            ": looking for stock #%d in DB to load name\n", tIndex);
#endif

    do
    {
        // Check input
        if ((psAppObj == NULL) || (phName == NULL))
        {
            eResult = STOCK_TICKER_RETURN_CODE_BAD_INPUT;
            break;
        }

        ///////////////////////////////////////////////////////////////////////
        // Init argument
        OSAL.bMemSet(&sArg, 0, sizeof(sArg));
        sArg.sOutput.bResultantRows = FALSE;
        sArg.sOutput.uDbRow.sSymbol.hName = STRING_INVALID_OBJECT;

        // Perform the SQL query and process the result
        // (it will provide us with a data row)
        psAppObj->sDB.asBindParams[STOCK_SELECT_SYMBOL_STMT_INDEX_PARAM].eType =
            SQL_BIND_TYPE_UN32;
        psAppObj->sDB.asBindParams[STOCK_SELECT_SYMBOL_STMT_INDEX_PARAM].pvData =
            (void *)(size_t)tIndex;

        bOk = SQL_INTERFACE.bExecutePreparedStatement(
            psAppObj->sDB.hSQLConnection,
            psAppObj->sDB.hSelectSymbolByIdStmt,
            bStockMsgDescExistsInDBProcessResult,
            (void*)&sArg,
            STOCK_SELECT_SYMBOL_STMT_PARAMS_COUNT,
            psAppObj->sDB.asBindParams);

        if (bOk != TRUE)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    STOCK_TICKER_MGR_OBJECT_NAME": failed to make query");
            eResult = STOCK_TICKER_RETURN_CODE_DB_ERROR;
            break;
        }

        if (sArg.sOutput.bResultantRows == TRUE)
        {
            if (sArg.sOutput.uDbRow.sSymbol.hName == STRING_INVALID_OBJECT)
            {
                SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                        STOCK_TICKER_MGR_OBJECT_NAME": row found but the name is empty. Wierd...");
                eResult = STOCK_TICKER_RETURN_CODE_BAD_INPUT;
                break;
            }

            // Populated the string
            *phName = sArg.sOutput.uDbRow.sSymbol.hName;
            eResult = STOCK_TICKER_RETURN_CODE_SUCCESS;
        }
        else
        {
            // There is not such record in the DB
            *phName = STRING_INVALID_OBJECT;
            eResult = STOCK_TICKER_RETURN_CODE_FALSE;
        }
    } while (FALSE);

    return eResult;
}

/*****************************************************************************
*
*   eStockMsgDescExistsInDB
*
*   Output:
*       STOCK_TICKER_RETURN_CODE_NOTHING_TO_UPDATE
*               - the item in the DB and has the same data
*       STOCK_TICKER_RETURN_CODE_NEED_TO_UPDATE
*               - information in the DB needs to be updated
*       STOCK_TICKER_RETURN_CODE_FALSE
*               - information for the item in the DB completely absent.
*
*****************************************************************************/
static STOCK_TICKER_RETURN_CODE_ENUM eStockMsgDescGetStateInDB (
    STOCK_TICKER_APP_OBJECT_STRUCT *psAppObj,
    STOCK_ID tIndex,
    STRING_OBJECT hName
        )
{
    STOCK_TICKER_RETURN_CODE_ENUM eResult = STOCK_TICKER_RETURN_CODE_ERROR;
    STRING_OBJECT hDBName = STRING_INVALID_OBJECT;

#if ((SMS_DEBUG == 1) && (STOCK_TICKER_EXTRA_DEBUG == 1))
    printf(STOCK_TICKER_MGR_OBJECT_NAME": looking for stock #%d in DB\n",
            tIndex);
#endif

    do
    {
        // Check input
        if ((psAppObj == NULL) || (hName == STRING_INVALID_OBJECT))
        {
            eResult = STOCK_TICKER_RETURN_CODE_BAD_INPUT;
            break;
        }

        // Try to load Name from the DB
        eResult = eStockMsgDescGetNameFromDB(psAppObj, tIndex, &hDBName);
        if (eResult == STOCK_TICKER_RETURN_CODE_FALSE)
        {
            // Nothing to do else. Just report to the caller about
            // FALSE
            break;
        }
        else if (eResult == STOCK_TICKER_RETURN_CODE_SUCCESS)
        {
            N16 n16CompareResult;

            // Compare two strings
            n16CompareResult = STRING.n16Compare(hName, hDBName, TRUE);
            if (n16CompareResult == 0)
            {
                eResult = STOCK_TICKER_RETURN_CODE_NOTHING_TO_UPDATE;
            }
            else
            {
                eResult = STOCK_TICKER_RETURN_CODE_NEED_TO_UPDATE;
            }

            // Clean up the string loaded from DB
            STRING_vDestroy(hDBName);
            hDBName = STRING_INVALID_OBJECT;
        }
        else
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                STOCK_TICKER_MGR_OBJECT_NAME
                ": failed to load Stock's #%d name from DB (%s)",
                tIndex, GsStockTickerMgrIntf.pacGetReturnCodeName(eResult)
                    );
            break;
        }

    } while (FALSE);

    return eResult;
}

/*****************************************************************************
*
*   eStockMsgDescUpdateNameFromDB
*
*****************************************************************************/
static STOCK_TICKER_RETURN_CODE_ENUM eStockMsgDescUpdateNameFromDB(
    STOCK_TICKER_APP_OBJECT_STRUCT *psAppObj,
    STOCK_MSG_DESC_STRUCT *psMsgDesc
        )
{
    STOCK_TICKER_RETURN_CODE_ENUM eResult = STOCK_TICKER_RETURN_CODE_ERROR;
    BOOLEAN bHasName = FALSE;
    STOCK_ID tStockId = STOCK_INVALID_ID;
    STRING_OBJECT hStockName = STRING_INVALID_OBJECT;

    do
    {

        // Check input
        if ((psAppObj == NULL) || (psMsgDesc == NULL))
        {
            eResult = STOCK_TICKER_RETURN_CODE_BAD_INPUT;
            break;
        }

        // Check name existence in the message
        bHasName = STOCK_MSG_bHasName(psMsgDesc->hMessage);
        if (bHasName == TRUE)
        {
            eResult = STOCK_TICKER_RETURN_CODE_NOTHING_TO_UPDATE;
            break;
        }

        // Get stock index
        tStockId = STOCK_MSG_tIndex(psMsgDesc->hMessage);

        // Request name from DB
        eResult = eStockMsgDescGetNameFromDB(psAppObj, tStockId, &hStockName);
        if (eResult == STOCK_TICKER_RETURN_CODE_SUCCESS)
        {
            STOCK_MSG_DATA_STRUCT sData;
            sData.tMask = STOCK_MSG_DATA_NAME;
            sData.uName.hValue = hStockName;

            eResult = STOCK_MSG_eUpdate(psMsgDesc->hMessage, &sData);
            if ((eResult != STOCK_TICKER_RETURN_CODE_SUCCESS) &&
                (eResult != STOCK_TICKER_RETURN_CODE_NOTHING_TO_UPDATE))
            {
                SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    STOCK_TICKER_MGR_OBJECT_NAME": failed to update name (%s)",
                    GsStockTickerMgrIntf.pacGetReturnCodeName(eResult)
                        );
                break;
            }
        }
        else if (eResult == STOCK_TICKER_RETURN_CODE_FALSE)
        {
            // This is really unexpected error code
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                STOCK_TICKER_MGR_OBJECT_NAME": there is no name for stock #%d in DB",
                tStockId
                    );
            eResult = STOCK_TICKER_RETURN_CODE_DB_ERROR;
        }
    } while (FALSE);

    if (hStockName != STRING_INVALID_OBJECT)
    {
        STRING_vDestroy(hStockName);
    }

    return eResult;
}

/*****************************************************************************
*
*   eStockMsgDescProcess
*
*****************************************************************************/
static STOCK_TICKER_RETURN_CODE_ENUM eStockMsgDescProcess(
    STOCK_TICKER_APP_OBJECT_STRUCT *psAppObj,
    STOCK_MSG_DESC_STRUCT *psMsgDesc,
    BOOLEAN *pbIsDBUpdated
        )
{
    STOCK_TICKER_RETURN_CODE_ENUM eResult = STOCK_TICKER_RETURN_CODE_ERROR;
    BOOLEAN bNeedToUpdateDueToName = FALSE;
    BOOLEAN bNeedToUpdateDueToValue = TRUE;

    do
    {
        // Check input
        if ((psAppObj == NULL) || (psMsgDesc == NULL) || (pbIsDBUpdated == NULL))
        {
            eResult = STOCK_TICKER_RETURN_CODE_BAD_INPUT;
            break;
        }

        // If we have any set of flags
        if ((psMsgDesc->tFieldMask & STOCK_TICKER_MSG_FIELD_MASK_ALL)
            != STOCK_TICKER_MSG_FIELD_MASK_NONE)
        {
            // Process new name if we have it and we allowed to update
            // ref-db
            if ((psMsgDesc->tFieldMask & STOCK_TICKER_MSG_FIELD_MASK_NAME) ==
                    STOCK_TICKER_MSG_FIELD_MASK_NAME)
            {
                // Since we did update of the DB we suppose that some of them have been
                // changes and we can process monitoring targets to find out some
                // indexes.
                eResult = eDSRLTargetMonitorProcess(psAppObj, psMsgDesc);
                if ((eResult != STOCK_TICKER_RETURN_CODE_NOTHING_TO_UPDATE) &&
                    (eResult != STOCK_TICKER_RETURN_CODE_SUCCESS))
                {
                    SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                            STOCK_TICKER_MGR_OBJECT_NAME
                            ": failed to process MsgDesc 0x%p for Targets Monitor (%s)",
                            psMsgDesc, GsStockTickerMgrIntf.pacGetReturnCodeName(eResult)
                                );
                    break;
                }

                // Since the name is updated we need to update the DB
                // by new name
                eResult = eStockMsgDescProcessForDB(psAppObj, psMsgDesc);
                if (eResult == STOCK_TICKER_RETURN_CODE_SUCCESS)
                {
                    bNeedToUpdateDueToName = TRUE;

                    *pbIsDBUpdated = TRUE;
                }
                else if (eResult != STOCK_TICKER_RETURN_CODE_NOTHING_TO_UPDATE)
                {
                    SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                            STOCK_TICKER_MGR_OBJECT_NAME
                            ": failed to process MsgDesc 0x%p for the DB (%s)",
                            psMsgDesc, GsStockTickerMgrIntf.pacGetReturnCodeName(eResult)
                                );
                    break;
                }
            }

            // Process any update to notify application via DSRL
            if ((psMsgDesc->tFieldMask & STOCK_TICKER_MSG_FIELD_MASK_VALUES) ==
                    STOCK_TICKER_MSG_FIELD_MASK_VALUES)
            {
                // We have values to populate
                bNeedToUpdateDueToValue = TRUE;
            }

            // Check the flags to find out the necessity of the update
            if ((bNeedToUpdateDueToValue == TRUE) || (bNeedToUpdateDueToName == TRUE))
            {
                BOOLEAN bGoodToPopulate;

                // Check does the message have values to populate.
                bGoodToPopulate = STOCK_MSG_bGoodToPopulate(psMsgDesc->hMessage);

                if (bGoodToPopulate == TRUE)
                {
                    // Check DSRLs existence
                    eResult = eDSRLDescHasToUpdate(psAppObj);
                    if (eResult == STOCK_TICKER_RETURN_CODE_SUCCESS)
                    {
                        // We free to update the DSRL if we have new name or name in case if the
                        // value is correct
                        eResult = eStockMsgDescProcessForDSRLs(psAppObj, psMsgDesc);
                        if (eResult != STOCK_TICKER_RETURN_CODE_SUCCESS)
                        {
                            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                                    STOCK_TICKER_MGR_OBJECT_NAME
                                    ": failed to process MsgDesc 0x%p for the DSRL (%s)",
                                    psMsgDesc, GsStockTickerMgrIntf.pacGetReturnCodeName(eResult)
                                        );
                        }
                    }
                    else if (eResult != STOCK_TICKER_RETURN_CODE_FALSE)
                    {
                        SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                            STOCK_TICKER_MGR_OBJECT_NAME": failed to detect DSRLs existence (%s)",
                            GsStockTickerMgrIntf.pacGetReturnCodeName(eResult)
                                );
                        break;
                    }
                }
            }
        }

        // Reset updating mask
        psMsgDesc->tFieldMask = STOCK_TICKER_MSG_FIELD_MASK_NONE;

        // Move the message to the Stable state
        vStockMsgDescSetState(psMsgDesc, STOCK_DATA_STATE_STABLE);

        eResult = STOCK_TICKER_RETURN_CODE_SUCCESS;
    } while (FALSE);

    return eResult;
}

/*****************************************************************************
*
*   eStockMsgDescProcessForDB
*
*****************************************************************************/
static STOCK_TICKER_RETURN_CODE_ENUM eStockMsgDescProcessForDB (
    STOCK_TICKER_APP_OBJECT_STRUCT *psAppObj,
    STOCK_MSG_DESC_STRUCT *psMsgDesc
        )
{
    STOCK_TICKER_RETURN_CODE_ENUM eResult = STOCK_TICKER_RETURN_CODE_ERROR;
    STOCK_ID tIndex = STOCK_INVALID_ID;

#if  ((SMS_DEBUG == 1) && (STOCK_TICKER_EXTRA_DEBUG == 1))
    printf(STOCK_TICKER_MGR_OBJECT_NAME": processing MsgDesc 0x%p for DB\n", psMsgDesc);
#endif

    do
    {
        // Check input
        if ((psAppObj == NULL) || (psMsgDesc == NULL))
        {
            eResult = STOCK_TICKER_RETURN_CODE_BAD_INPUT;
            break;
        }

        // Process only requests for name update
        if (
            ((psMsgDesc->tFieldMask & STOCK_TICKER_MSG_FIELD_MASK_NAME) != STOCK_TICKER_MSG_FIELD_MASK_NAME) ||
                ((psMsgDesc->eState != STOCK_DATA_STATE_UPDATED) &&
                 (psMsgDesc->eState != STOCK_DATA_STATE_CREATED))
           )
        {
            eResult = STOCK_TICKER_RETURN_CODE_NOTHING_TO_UPDATE;
            break;
        }

        // Get stock index
        tIndex = STOCK_MSG_tIndex(psMsgDesc->hMessage);
        if (psMsgDesc->eState == STOCK_DATA_STATE_UPDATED)
        {
            BOOLEAN bUpdated;

            printf(STOCK_TICKER_MGR_OBJECT_NAME": updating record for #%d MsgDesc 0x%p\n",
                    tIndex, psMsgDesc
                        );

            psAppObj->sDB.asBindParams[STOCK_UPDATE_SYMBOL_STMT_NAME_PARAM].eType =
                SQL_BIND_TYPE_STRING_OBJECT;
            psAppObj->sDB.asBindParams[STOCK_UPDATE_SYMBOL_STMT_NAME_PARAM].pvData =
                (void *)STOCK_MSG.hSymbolName(psMsgDesc->hMessage);

            psAppObj->sDB.asBindParams[STOCK_UPDATE_SYMBOL_STMT_INDEX_PARAM].eType =
                SQL_BIND_TYPE_UN32;
            psAppObj->sDB.asBindParams[STOCK_UPDATE_SYMBOL_STMT_INDEX_PARAM].pvData =
                (void *)(size_t)tIndex;

            bUpdated = SQL_INTERFACE.bExecutePreparedStatement(
                psAppObj->sDB.hSQLConnection,
                psAppObj->sDB.hUpdateSymbolStmt,
                NULL, NULL,
                STOCK_UPDATE_SYMBOL_STMT_PARAMS_COUNT,
                psAppObj->sDB.asBindParams);

            if (bUpdated == FALSE)
            {
                SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    STOCK_TICKER_MGR_OBJECT_NAME": Failed to update symbol data"
                        );
                eResult = STOCK_TICKER_RETURN_CODE_DB_ERROR;
                break;
            }
        }
        else if (psMsgDesc->eState == STOCK_DATA_STATE_CREATED)
        {
            BOOLEAN bInserted;

            printf(STOCK_TICKER_MGR_OBJECT_NAME": creating record for #%d MsgDesc 0x%p\n",
                    tIndex, psMsgDesc
                        );

            psAppObj->sDB.asBindParams[STOCK_INSERT_SYMBOL_STMT_INDEX_PARAM].eType =
                SQL_BIND_TYPE_UN32;
            psAppObj->sDB.asBindParams[STOCK_INSERT_SYMBOL_STMT_INDEX_PARAM].pvData =
                (void *)(size_t)tIndex;

            psAppObj->sDB.asBindParams[STOCK_INSERT_SYMBOL_STMT_NAME_PARAM].eType =
                SQL_BIND_TYPE_STRING_OBJECT;
            psAppObj->sDB.asBindParams[STOCK_INSERT_SYMBOL_STMT_NAME_PARAM].pvData =
                (void *)STOCK_MSG.hSymbolName(psMsgDesc->hMessage);

            bInserted = SQL_INTERFACE.bExecutePreparedStatement(
                psAppObj->sDB.hSQLConnection,
                psAppObj->sDB.hInsertSymbolStmt,
                NULL, NULL,
                STOCK_INSERT_SYMBOL_STMT_PARAMS_COUNT,
                psAppObj->sDB.asBindParams);

            if (bInserted == FALSE)
            {
                SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    STOCK_TICKER_MGR_OBJECT_NAME": Failed to insert symbol data %s"
                    STOCK_TICKER_INSERT_DB_SYMBOL);
                eResult = STOCK_TICKER_RETURN_CODE_DB_ERROR;
                break;
            }
        }
#if ((SMS_DEBUG == 1) && (STOCK_TICKER_EXTRA_DEBUG == 1))
        else
        {
            printf(STOCK_TICKER_MGR_OBJECT_NAME": skip #%d (in %s state)\n",
                    tIndex, pacGetStockDataStateName(psMsgDesc->eState)
                        );
        }
#endif

        eResult = STOCK_TICKER_RETURN_CODE_SUCCESS;
    } while (FALSE);

    return eResult;
}

/*****************************************************************************
*
*   bStockMsgDescProcessForDSRLsIterator
*
*****************************************************************************/
static BOOLEAN bStockMsgDescProcessForDSRLsIterator(
    DSRL_OBJECT hDSRL,
    STOCK_TICKER_MSG_FOR_DSRL_ITERATOR_STRUCT *psArg
        )
{
    STOCK_TICKER_DSRL_DESC_STRUCT *psDSRLDesc = NULL;
    STOCK_TICKER_RETURN_CODE_ENUM eResult;
    BOOLEAN bContinue = TRUE;
    STOCK_MSG_DESC_STRUCT *psMsgDesc = psArg->psMsgDesc;

    // Get the DSRL descriptor from DSRL
    psDSRLDesc = (STOCK_TICKER_DSRL_DESC_STRUCT*)DSRL_pvServiceData(hDSRL);

    eResult = eDSRLDescApplicableForMessage(psArg->psAppObj, psDSRLDesc, psArg->psMsgDesc);
    if (eResult == STOCK_TICKER_RETURN_CODE_SUCCESS)
    {
        OSAL_LINKED_LIST_ENTRY hMsgInDSRLEntry = OSAL_INVALID_LINKED_LIST_ENTRY;

        // Good, we should have this message in the DSRL.
        // Let's check its real existence in the DSRL.
        eResult = eDSRLDescHasEntry(psDSRLDesc, psArg->psMsgDesc, &hMsgInDSRLEntry);
        if (eResult == STOCK_TICKER_RETURN_CODE_SUCCESS)
        {
            DSRL_ADD_REPLACE_RESULT_ENUM eDSRLResult;

            // Move DSRL to the Updating state
            vDSRLDescSetState(hDSRL, DSRL_STATE_UPDATING);

            // Well, the message in the list. So, just say the message is updated
            eDSRLResult = DSRL_eReplaceEntry(hDSRL,
                                (DSRL_ENTRY_OBJECT)psMsgDesc->hMessage,
                                (DSRL_ENTRY_OBJECT)psMsgDesc->hMessage
                                    );
            if (eDSRLResult != DSRL_ADD_REPLACE_OK)
            {
                // Replacing has not been valid so remove the entry then
                eResult = eDSRLDescRemoveEntry(psDSRLDesc,
                                psMsgDesc, hMsgInDSRLEntry, TRUE);
                if (eResult != STOCK_TICKER_RETURN_CODE_SUCCESS)
                {
                    SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                            STOCK_TICKER_MGR_OBJECT_NAME
                            ":failed to remove message 0x%p from the DSRL 0x%p (%s)",
                            psMsgDesc, psDSRLDesc,
                            GsStockTickerMgrIntf.pacGetReturnCodeName(eResult)
                                );
                    bContinue = FALSE;
                }
            }
        }
        else if (eResult == STOCK_TICKER_RETURN_CODE_FALSE)
        {
            // So, we need to add this entry to the DSRL
            (void) eDSRLDescAddEntry(psArg->psAppObj, psDSRLDesc, psMsgDesc);
        }
        else
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    STOCK_TICKER_MGR_OBJECT_NAME
                    ": failed to check message %p existence in the DSRL %p",
                    psMsgDesc, hDSRL);
            bContinue = FALSE;
        }
    }
    else if (eResult != STOCK_TICKER_RETURN_CODE_FALSE)
    {
        SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                STOCK_TICKER_MGR_OBJECT_NAME
                ": failed to check message 0x%p applicability for the DSRL %p",
                psMsgDesc, hDSRL);
        bContinue = FALSE;
    }

    // We're bad...
    if (bContinue == FALSE)
    {
        // Populate the return code, then
        psArg->eResult = eResult;
    }

    return bContinue;
}

/*****************************************************************************
*
*   eStockMsgDescProcessForDSRLs
*
*****************************************************************************/
static STOCK_TICKER_RETURN_CODE_ENUM eStockMsgDescProcessForDSRLs (
    STOCK_TICKER_APP_OBJECT_STRUCT *psAppObj,
    STOCK_MSG_DESC_STRUCT *psMsgDesc
        )
{
    OSAL_RETURN_CODE_ENUM eOsalReturnCode;
    STOCK_TICKER_MSG_FOR_DSRL_ITERATOR_STRUCT sArg;

    // Looking through all DSRL to find out does this updated message should be
    // added to the existing DSRL or not
    sArg.eResult = STOCK_TICKER_RETURN_CODE_SUCCESS;
    sArg.psAppObj = psAppObj;
    sArg.psMsgDesc = psMsgDesc;

    // Check input
    if (psAppObj != NULL)
    {
        eOsalReturnCode = OSAL.eLinkedListIterate(psAppObj->hDSRLList,
                                (OSAL_LL_ITERATOR_HANDLER)bStockMsgDescProcessForDSRLsIterator,
                                (void*)&sArg);
        if ((eOsalReturnCode != OSAL_SUCCESS) && (eOsalReturnCode != OSAL_NO_OBJECTS))
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                STOCK_TICKER_MGR_OBJECT_NAME": failed to do iteration through DSLRs (%s)",
                OSAL.pacGetReturnCodeName(eOsalReturnCode)
                    );
            sArg.eResult = STOCK_TICKER_RETURN_CODE_ERROR;
        }
    }
    else
    {
        sArg.eResult = STOCK_TICKER_RETURN_CODE_BAD_INPUT;
    }

    return sArg.eResult;
}

/*****************************************************************************
*
*   bStockMsgDescInvalidateExpiredIterator
*
*****************************************************************************/
static BOOLEAN bStockMsgDescInvalidateExpiredIterator(
    STOCK_MSG_DESC_STRUCT *psMsgDesc,
    STOCK_TICKIER_MSG_EXPIRATION_ITERATOR_STRUCT *psArg
        )
{
    // Check the timestamp
    const UN32 un32Diff = psArg->un32CurrentTimeStamp - psMsgDesc->un32LastUpdateTimeStamp;
    STOCK_MSG_STATE_ENUM eMsgState = STOCK_MSG_eState(psMsgDesc->hMessage);

    if ((un32Diff >= STOCK_TICKER_VALUES_TIMEOUT) &&
        (eMsgState != psArg->sValues.eState))
    {
        STOCK_TICKER_RETURN_CODE_ENUM eReturnCode;

        // Since the item is expired need to move it to N/A state
        // as required by TRS
        psArg->sValues.tIndex = STOCK_MSG_tIndex(psMsgDesc->hMessage);

        // Do the update of the data in the message
        eReturnCode =
            eUpdateSymbolValue(psArg->psAppObj, &psArg->sValues,
                psMsgDesc, FALSE, FALSE);
        if (eReturnCode != STOCK_TICKER_RETURN_CODE_SUCCESS)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                STOCK_TICKER_MGR_OBJECT_NAME
                ": failed to invalidate message for stock #%d (%s)",
                psArg->sValues.tIndex,
                GsStockTickerMgrIntf.pacGetReturnCodeName(eReturnCode)
                    );
        }
        else
        {
            ++psArg->un32ExpiredMessages;
        }
    }

    return TRUE;
}

/*****************************************************************************
*
*   eStockMsgDescInvalidateExpired
*
*****************************************************************************/
static STOCK_TICKER_RETURN_CODE_ENUM eStockMsgDescInvalidateExpired (
    STOCK_TICKER_APP_OBJECT_STRUCT *psAppObj,
    BOOLEAN *pbExpired
        )
{
    STOCK_TICKER_RETURN_CODE_ENUM eReturnCode = STOCK_TICKER_RETURN_CODE_ERROR;
    OSAL_RETURN_CODE_ENUM eOsalReturnCode = OSAL_ERROR;
    STOCK_TICKIER_MSG_EXPIRATION_ITERATOR_STRUCT sArg;
    UN32 un32AllReferencedMessages = 0;

    do
    {
        // Check input
        if (psAppObj == NULL)
        {
            eReturnCode = STOCK_TICKER_RETURN_CODE_BAD_INPUT;
            break;
        }

        // Init output
        *pbExpired = FALSE;

        // Init iterator
        sArg.psAppObj = psAppObj;
        sArg.un32ExpiredMessages = 0;
        sArg.un32CurrentTimeStamp = 0;
        sArg.sValues.eState = STOCK_MSG_STATE_PRICE_NOT_AVAILABLE;
        sArg.sValues.sQuote.eDir = STOCK_MSG_PRICE_DIRECTION_INVALID;
        sArg.sValues.sQuote.un32Price = 0;
        sArg.sValues.sQuote.un8PriceFrac = 0;
        sArg.sValues.sQuote.un32PriceDelta = 0;
        sArg.sValues.sQuote.un8PriceDeltaFrac = 0;

        // Get current timestamp within the iterator's argument
        OSAL.vTimeUp(&sArg.un32CurrentTimeStamp, (UN16*)NULL);

        // Get number of messages
        eOsalReturnCode = OSAL.eLinkedListItems(psAppObj->hSymbolMsgList,
                                    &un32AllReferencedMessages
                                        );
        if (eOsalReturnCode != OSAL_SUCCESS)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                STOCK_TICKER_MGR_OBJECT_NAME": failed to get number of items (%s)",
                OSAL.pacGetReturnCodeName(eOsalReturnCode)
                    );
            break;
        }

        // Well there is a hint to do not move further with iteration
        if (un32AllReferencedMessages > 0)
        {
            // Looking through referenced messaged to update them if needed
            eOsalReturnCode = OSAL.eLinkedListIterate(psAppObj->hSymbolMsgList,
                                        (OSAL_LL_ITERATOR_HANDLER)bStockMsgDescInvalidateExpiredIterator,
                                        (void*) &sArg);
            if ((eOsalReturnCode != OSAL_SUCCESS) &&
                (eOsalReturnCode != OSAL_NO_OBJECTS))
            {
                SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    STOCK_TICKER_MGR_OBJECT_NAME
                    ": failed to iterate message to invalidate the fact of expiration (%S)",
                    OSAL.pacGetReturnCodeName(eOsalReturnCode)
                        );
            }

            // Let's process the data to notify DSRLs about news if we have something
            if (sArg.un32ExpiredMessages > 0)
            {
                // Populate the knowledge about expiration to the caller
                *pbExpired = TRUE;

                eReturnCode = eProcessServiceData(psAppObj);
                if (eReturnCode != STOCK_TICKER_RETURN_CODE_SUCCESS)
                {
                    SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                        STOCK_TICKER_MGR_OBJECT_NAME": failed to process service data (%S)",
                        GsStockTickerMgrIntf.pacGetReturnCodeName(eReturnCode)
                            );
                    break;
                }
            }

            // Do we still have messages which could be expired shortly?
            if (sArg.un32ExpiredMessages < un32AllReferencedMessages)
            {
                vExpirationTimerStart(psAppObj, STOCK_TICKER_TIMER_PERIOD);
            }
        }

        eReturnCode = STOCK_TICKER_RETURN_CODE_SUCCESS;
    } while (FALSE);

    return eReturnCode;
}

/*****************************************************************************
*
*   psStockMsgDescGet
*
*****************************************************************************/
static STOCK_MSG_DESC_STRUCT *psStockMsgDescGet(
    STOCK_TICKER_APP_OBJECT_STRUCT *psAppObj,
    STOCK_ID tIndex
        )
{
    STOCK_MSG_DESC_STRUCT *psResult = NULL;
    OSAL_RETURN_CODE_ENUM eOsalReturnCode = OSAL_ERROR;
    OSAL_LINKED_LIST_ENTRY hEntry = OSAL_INVALID_LINKED_LIST_ENTRY;

    do
    {
        STOCK_MSG_DESC_STRUCT sSearchCriteria;
        STOCK_TICKER_RETURN_CODE_ENUM eReturnCode;

        // Check input
        if (psAppObj == NULL)
        {
            break;
        }

        printf(STOCK_TICKER_MGR_OBJECT_NAME": looking for stock #%d via Index\n", tIndex);

        // initialize search criteria by input data
        sSearchCriteria.hMessage = psAppObj->hDummyMsg;
        eReturnCode = STOCK_MSG_eSetIndex(psAppObj->hDummyMsg, tIndex);
        if (eReturnCode != STOCK_TICKER_RETURN_CODE_SUCCESS)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                STOCK_TICKER_MGR_OBJECT_NAME
                ": failed to set index #%d for dummy message (%s)",
                tIndex,
                GsStockTickerMgrIntf.pacGetReturnCodeName(eReturnCode)
                    );
            break;
        }

        // Look up the message descriptor by symbol index
        eOsalReturnCode = OSAL.eLinkedListSearch(
                                psAppObj->hSymbolMsgList,
                                &hEntry,
                                &sSearchCriteria
                                    );
        // Check error codes
        if (eOsalReturnCode == OSAL_SUCCESS)
        {
            psResult = (STOCK_MSG_DESC_STRUCT*)OSAL.pvLinkedListThis(hEntry);
        }
        else if (eOsalReturnCode != OSAL_OBJECT_NOT_FOUND)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    STOCK_TICKER_MGR_OBJECT_NAME
                    ": failed to find message (%s)",
                    OSAL.pacGetReturnCodeName(eOsalReturnCode)
                        );
        }

    } while (FALSE);

    return psResult;
}

/*****************************************************************************
*
*   vStockMsgDescDestroy
*
*****************************************************************************/
static void vStockMsgDescDestroy(
    STOCK_MSG_DESC_STRUCT *psDesc
        )
{

#if ((SMS_DEBUG == 1) && (STOCK_TICKER_EXTRA_DEBUG == 1))
    printf(STOCK_TICKER_MGR_OBJECT_NAME": destroying Message Desc 0x%p\n",
            psDesc);
#endif

    if (psDesc != NULL)
    {
        vStockMsgDescSetState(psDesc, STOCK_DATA_STATE_REMOVED);

        // Just erase references
        psDesc->hEntry = OSAL_INVALID_LINKED_LIST_ENTRY;
        psDesc->hPoolEntry = OSAL_INVALID_LINKED_LIST_ENTRY;

        // Remove the object
        STOCK_MSG_vDestroy(psDesc->hMessage);
    }

    return;
}

/*****************************************************************************
*
*   vStockMsgDescDestroyFull
*
*****************************************************************************/
static void vStockMsgDescDestroyFull(
    STOCK_MSG_DESC_STRUCT *psDesc
        )
{
    OSAL_RETURN_CODE_ENUM eOsalReturnCode = OSAL_ERROR;

    if (psDesc != NULL)
    {
#if ((SMS_DEBUG == 1) && (STOCK_TICKER_EXTRA_DEBUG == 1))
        printf(STOCK_TICKER_MGR_OBJECT_NAME": destroying Message Desc 0x%p FULL\n",
                psDesc);
#endif

        // Remove from active messages if referenced
        if (psDesc->hEntry != OSAL_INVALID_LINKED_LIST_ENTRY)
        {
            eOsalReturnCode = OSAL.eLinkedListRemove(psDesc->hEntry);
            if (eOsalReturnCode != OSAL_SUCCESS)
            {
                SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                        STOCK_TICKER_MGR_OBJECT_NAME
                        ": failed to remove MsgDesc 0x%p from the msg list (%s)",
                        psDesc, OSAL.pacGetReturnCodeName(eOsalReturnCode)
                            );
            }

            psDesc->hEntry = OSAL_INVALID_LINKED_LIST_ENTRY;
        }

        // Remove from the pool if referenced
        if (psDesc->hPoolEntry != OSAL_INVALID_LINKED_LIST_ENTRY)
        {
            eOsalReturnCode = OSAL.eLinkedListRemove(psDesc->hPoolEntry);
            if (eOsalReturnCode != OSAL_SUCCESS)
            {
                SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                        STOCK_TICKER_MGR_OBJECT_NAME
                        ": failed to remove MsgDesc 0x%p from the pool (%s)",
                        psDesc, OSAL.pacGetReturnCodeName(eOsalReturnCode)
                            );
            }
        }

        vStockMsgDescDestroy(psDesc);
    }

    return;
}

/*****************************************************************************
*
*   vStockMsgDescRelease
*
*****************************************************************************/
static void vStockMsgDescRelease(
    STOCK_TICKER_APP_OBJECT_STRUCT *psAppObj,
    STOCK_MSG_DESC_STRUCT *psDesc
        )
{
    OSAL_RETURN_CODE_ENUM eOsalReturnCode = OSAL_SUCCESS;
    STOCK_TICKER_RETURN_CODE_ENUM eResult = STOCK_TICKER_RETURN_CODE_ERROR;

#if ((SMS_DEBUG == 1) && (STOCK_TICKER_EXTRA_DEBUG == 1))
    printf(STOCK_TICKER_MGR_OBJECT_NAME": releasing Message Desc 0x%p\n",
            psDesc);
#endif

    if ((psDesc != NULL) && (psDesc->hEntry != OSAL_INVALID_LINKED_LIST_ENTRY))
    {
        eOsalReturnCode = OSAL.eLinkedListRemove(psDesc->hEntry);
        if (eOsalReturnCode != OSAL_SUCCESS)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                STOCK_TICKER_MGR_OBJECT_NAME
                ": failed to remove MsgDesc 0x%p from the msg list (%s)",
                psDesc, OSAL.pacGetReturnCodeName(eOsalReturnCode)
                );
        }

        // Mark as now allocated
        psDesc->hEntry = OSAL_INVALID_LINKED_LIST_ENTRY;

        // Move to the top
        // Update message's pool position by moving (adding) it to the bottom
        eResult = eStockMsgDescUpdateInPool(psAppObj, psDesc, FALSE);
        if (eResult != STOCK_TICKER_RETURN_CODE_SUCCESS)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                STOCK_TICKER_MGR_OBJECT_NAME
                ": failed STOCK_MSG in the pool (%s)",
                GsStockTickerMgrIntf.pacGetReturnCodeName(eResult)
                    );
        }
    }

    return;
}

/*****************************************************************************
*
*   eStockMsgDescApplicableForTarget
*
*****************************************************************************/
static STOCK_TICKER_RETURN_CODE_ENUM eStockMsgDescIsApplicableForTarget(
    STOCK_MSG_DESC_STRUCT *psDesc,
    STOCK_TICKER_DSRL_TARGET_DESC_STRUCT *psTargetDesc
        )
{
    STOCK_TICKER_RETURN_CODE_ENUM eResult = STOCK_TICKER_RETURN_CODE_ERROR;

    do
    {
        // Check input
        if ((psDesc == NULL) || (psTargetDesc == NULL))
        {
            eResult = STOCK_TICKER_RETURN_CODE_BAD_INPUT;
            break;
        }

        // Set default return value since this point
        eResult = STOCK_TICKER_RETURN_CODE_FALSE;

        if (psTargetDesc->tStockId != STOCK_INVALID_ID)
        {
            STOCK_ID tIndex;

            // Get index from the message
            tIndex = STOCK_MSG_tIndex(psDesc->hMessage);

            if (tIndex == psTargetDesc->tStockId)
            {
                eResult = STOCK_TICKER_RETURN_CODE_SUCCESS;
            }
        }
    } while (FALSE);

    return eResult;
}

/*****************************************************************************
*
*   eStockMsgCacheInit
*
*****************************************************************************/
static STOCK_TICKER_RETURN_CODE_ENUM eStockMsgCacheInit (
    STOCK_TICKER_MGR_OBJECT_STRUCT *psObj
        )
{
    STOCK_TICKER_APP_OBJECT_STRUCT *psAppObj = NULL;
    STOCK_TICKER_RETURN_CODE_ENUM eResult = STOCK_TICKER_RETURN_CODE_ERROR;
    OSAL_RETURN_CODE_ENUM eOsalReturnCode = OSAL_ERROR;

    do
    {
        psAppObj = psGetAppFacingObject((STOCK_TICKER_SERVICE_OBJECT) psObj);
        if (psAppObj == NULL)
        {
            eResult = STOCK_TICKER_RETURN_CODE_NOT_OWNER;
            break;
        }

        // Creating cache list
        eOsalReturnCode = OSAL.eLinkedListCreate(&psAppObj->hSymbolMsgCacheList,
                            STOCK_TICKER_MGR_OBJECT_NAME":MsgCacheLst",
                            NULL,
                            OSAL_LL_OPTION_LINEAR | OSAL_LL_OPTION_UNIQUE
                                );
        if (eOsalReturnCode != OSAL_SUCCESS)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    STOCK_TICKER_MGR_OBJECT_NAME": failed to create messages cache list (%s)",
                    OSAL.pacGetReturnCodeName(eOsalReturnCode)
                        );
            break;
        }

        eResult = STOCK_TICKER_RETURN_CODE_SUCCESS;
    } while (FALSE);

    if (psAppObj != NULL)
    {
        SMSO_vUnlock((SMS_OBJECT) psAppObj);
    }

    return eResult;
}

/*****************************************************************************
*
*   eStockMsgCacheCleanIterator
*
*****************************************************************************/
static BOOLEAN eStockMsgCacheCleanIterator(
    STOCK_MSG_DESC_STRUCT *psMsgDesc,
    STOCK_TICKER_APP_OBJECT_STRUCT *psAppObj
        )
{
    if (psMsgDesc->un32Ref == 0)
    {
        // So, no body interested in this quote.
        vStockMsgDescRelease(psAppObj, psMsgDesc);
    }

    // Continue
    return TRUE;
}

/*****************************************************************************
*
*   eStockMsgCacheClean
*
*****************************************************************************/
static STOCK_TICKER_RETURN_CODE_ENUM eStockMsgCacheClean (
    STOCK_TICKER_APP_OBJECT_STRUCT *psAppObj
        )
{
    STOCK_TICKER_RETURN_CODE_ENUM eReturnCode = STOCK_TICKER_RETURN_CODE_ERROR;
    OSAL_RETURN_CODE_ENUM eOsalReturnCode = OSAL_ERROR;

    puts(STOCK_TICKER_MGR_OBJECT_NAME": cleaning messages cache");

    do
    {
        // Check input
        if (psAppObj == NULL)
        {
            eReturnCode = STOCK_TICKER_RETURN_CODE_BAD_INPUT;
            break;
        }

        // Since we have the cache with the updated/create object
        // we can looks at it and remove those of them which are not
        // referenced by any of DSRLs.
        eOsalReturnCode = OSAL.eLinkedListIterate(psAppObj->hSymbolMsgCacheList,
                            (OSAL_LL_ITERATOR_HANDLER)eStockMsgCacheCleanIterator,
                            (void*) psAppObj
                                );
        if ((eOsalReturnCode != OSAL_SUCCESS) && (eOsalReturnCode != OSAL_NO_OBJECTS))
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    STOCK_TICKER_MGR_OBJECT_NAME
                    ": failed to iterate to prune messages (%s)",
                    OSAL.pacGetReturnCodeName(eOsalReturnCode)
                        );
            break;
        }

        // Just clean up the list
        eOsalReturnCode = OSAL.eLinkedListRemoveAll(
                                    psAppObj->hSymbolMsgCacheList,
                                    NULL
                                        );
        if (eOsalReturnCode != OSAL_SUCCESS)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    STOCK_TICKER_MGR_OBJECT_NAME
                    ": failed to clean the cache (%s)",
                    OSAL.pacGetReturnCodeName(eOsalReturnCode)
                        );
            break;
        }

        eReturnCode = STOCK_TICKER_RETURN_CODE_SUCCESS;
    } while (FALSE);

    return eReturnCode;
}

/*****************************************************************************
*
*   bStockMsgCacheProcessIterator
*
*****************************************************************************/
static BOOLEAN bStockMsgCacheProcessIterator (
    STOCK_MSG_DESC_STRUCT *psMsgDesc,
    STOCK_TICKER_MSG_CACHE_LIST_PROCESS_ITERATOR_STRUCT *psArg
        )
{
    BOOLEAN bContinue;

    // Process
    psArg->eReturnCode =
        eStockMsgDescProcess(psArg->psAppObj, psMsgDesc, &psArg->bIsDBUpdated);

    // Continue?
    bContinue = (psArg->eReturnCode != STOCK_TICKER_RETURN_CODE_SUCCESS) ? FALSE : TRUE;

    return bContinue;
}

/*****************************************************************************
*
*   eStockMsgCacheProcess
*
*****************************************************************************/
static STOCK_TICKER_RETURN_CODE_ENUM eStockMsgCacheProcess (
    STOCK_TICKER_APP_OBJECT_STRUCT *psAppObj
        )
{
    STOCK_TICKER_MSG_CACHE_LIST_PROCESS_ITERATOR_STRUCT sArg;
    OSAL_RETURN_CODE_ENUM eOsalReturnCode;

    puts(STOCK_TICKER_MGR_OBJECT_NAME": processing messages cache");

    do
    {
        // Init iterator struct
        sArg.eReturnCode = STOCK_TICKER_RETURN_CODE_SUCCESS;
        sArg.bIsDBUpdated = FALSE;
        sArg.psAppObj = psAppObj;

        // Check input
        if (psAppObj == NULL)
        {
            sArg.eReturnCode = STOCK_TICKER_RETURN_CODE_BAD_INPUT;
            break;
        }

        // Looking through the list of recently updated items
        // to process them from the DSRL and DB points of view.
        eOsalReturnCode = OSAL.eLinkedListIterate(psAppObj->hSymbolMsgCacheList,
                                    (OSAL_LL_ITERATOR_HANDLER)bStockMsgCacheProcessIterator,
                                    (void*) &sArg);
        if ((eOsalReturnCode != OSAL_SUCCESS) && (eOsalReturnCode != OSAL_NO_OBJECTS))
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                STOCK_TICKER_MGR_OBJECT_NAME": failed to iterate through Message Cache (%s)",
               OSAL.pacGetReturnCodeName(eOsalReturnCode)
                   );
            sArg.eReturnCode = STOCK_TICKER_RETURN_CODE_ERROR;
            break;
        }
        else if (sArg.eReturnCode != STOCK_TICKER_RETURN_CODE_SUCCESS)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    STOCK_TICKER_MGR_OBJECT_NAME": failed to process Message Cache (%s)",
                    GsStockTickerMgrIntf.pacGetReturnCodeName(sArg.eReturnCode)
                        );
            break;
        }

        // Has DB updated?
        if (sArg.bIsDBUpdated == TRUE)
        {
            // We have updated symbol and application has to iterate all
            // symbols again to gather up-to-date list.
            psAppObj->sDB.bSymbolsUpdated = TRUE;

            // Update data-version in DB to let application know
            // about it via persistent storage.
            sArg.eReturnCode = eDBUpdateDataVersion(psAppObj);
            if (sArg.eReturnCode != STOCK_TICKER_RETURN_CODE_SUCCESS)
            {
                SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    STOCK_TICKER_MGR_OBJECT_NAME
                    ": failed to update data version (%s)",
                    GsStockTickerMgrIntf.pacGetReturnCodeName(sArg.eReturnCode)
                        );
                break;
            }
        }
    } while (FALSE);

    return sArg.eReturnCode;
}

/*****************************************************************************
*
*   eStockMsgCacheAdd
*
*****************************************************************************/
static STOCK_TICKER_RETURN_CODE_ENUM eStockMsgCacheAdd (
    STOCK_TICKER_APP_OBJECT_STRUCT *psAppObj,
    STOCK_MSG_DESC_STRUCT *psMsgDesc
        )
{
    STOCK_TICKER_RETURN_CODE_ENUM eReturnCode = STOCK_TICKER_RETURN_CODE_ERROR;
    OSAL_RETURN_CODE_ENUM eOsalReturnCode = OSAL_ERROR;

    do
    {
        // Check input
        if ((psAppObj == NULL) || (psMsgDesc == NULL))
        {
            eReturnCode = STOCK_TICKER_RETURN_CODE_BAD_INPUT;
            break;
        }

        // Add tyhe message to the list
        eOsalReturnCode = OSAL.eLinkedListAdd(psAppObj->hSymbolMsgCacheList,
                                    OSAL_INVALID_LINKED_LIST_ENTRY_PTR, psMsgDesc
                                        );
        if ((eOsalReturnCode != OSAL_SUCCESS) &&
            (eOsalReturnCode != OSAL_ERROR_LIST_ITEM_NOT_UNIQUE))
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    STOCK_TICKER_MGR_OBJECT_NAME
                    ": failed to add MsgDesc 0x%p to the cache (%s)",
                    psMsgDesc, OSAL.pacGetReturnCodeName(eOsalReturnCode)
                    );
            break;
        }

        eReturnCode = STOCK_TICKER_RETURN_CODE_SUCCESS;
    } while (FALSE);

    return eReturnCode;
}

/*****************************************************************************
*
*   eStockMsgCacheUninit
*
*****************************************************************************/
static STOCK_TICKER_RETURN_CODE_ENUM eStockMsgCacheUninit (
    STOCK_TICKER_MGR_OBJECT_STRUCT *psObj
        )
{
    STOCK_TICKER_APP_OBJECT_STRUCT *psAppObj = NULL;
    OSAL_RETURN_CODE_ENUM eOsalReturnCode = OSAL_ERROR;
    STOCK_TICKER_RETURN_CODE_ENUM eResult = STOCK_TICKER_RETURN_CODE_ERROR;

    puts(STOCK_TICKER_MGR_OBJECT_NAME": Uninitializing Stock Messages Cache");

    do
    {
        // Check input
        if (psObj == NULL)
        {
            eResult = STOCK_TICKER_RETURN_CODE_BAD_INPUT;
            break;
        }

        psAppObj = psObj->psAppObj;

        // Just destroy list since its content referenced by the main list
        if (psAppObj->hSymbolMsgCacheList != OSAL_INVALID_OBJECT_HDL)
        {
            eOsalReturnCode = OSAL.eLinkedListDelete(psAppObj->hSymbolMsgCacheList);
            if (eOsalReturnCode != OSAL_SUCCESS)
            {
                SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                        STOCK_TICKER_MGR_OBJECT_NAME": failed to destroy messages cache list (%s)",
                        OSAL.pacGetReturnCodeName(eOsalReturnCode)
                            );
            }
            psAppObj->hSymbolMsgCacheList = OSAL_INVALID_OBJECT_HDL;
        }

        eResult = STOCK_TICKER_RETURN_CODE_SUCCESS;
    } while (FALSE);

    return eResult;
}

/*****************************************************************************
*
*   eStockDataProviderInfoInit
*
*****************************************************************************/
static STOCK_TICKER_RETURN_CODE_ENUM eStockDataProviderInfoInit (
    STOCK_TICKER_APP_OBJECT_STRUCT *psAppObj
        )
{
    STOCK_TICKER_RETURN_CODE_ENUM eResult = STOCK_TICKER_RETURN_CODE_ERROR;

    puts(STOCK_TICKER_MGR_OBJECT_NAME": Initializing Data Provider Info");

    do
    {
        // Check input
        if (psAppObj == NULL)
        {
            eResult = STOCK_TICKER_RETURN_CODE_BAD_INPUT;
            break;
        }

        // Init fields
        psAppObj->sProviderInfoCtrl.eState = STOCK_DATA_STATE_INITIAL;
        psAppObj->sProviderInfoCtrl.hDataProviderInfoText = STRING_INVALID_OBJECT;
        psAppObj->sProviderInfoCtrl.un8SequenceNumber = 0;

        eResult = STOCK_TICKER_RETURN_CODE_SUCCESS;
    } while (FALSE);

    return eResult;
}

/*****************************************************************************
*
*   eStockDataProviderInfoLoadDBIterator
*
*****************************************************************************/
static BOOLEAN eStockDataProviderInfoLoadDBIterator (
    SQL_QUERY_COLUMN_STRUCT *psColumns,
    N32 n32NumberOfColumns,
    void *pvArg
       )
{
    STOCK_TICKER_DB_QUERY_RESULT_STRUCT *psArg =
        (STOCK_TICKER_DB_QUERY_RESULT_STRUCT *)pvArg;
    SQL_QUERY_COLUMN_STRUCT *psSeqIdCol = NULL;
    SQL_QUERY_COLUMN_STRUCT *psInfoCol = NULL;
    size_t tStrLen = 0;
    BOOLEAN bResult = TRUE;

    do
    {
        if (psArg->sOutput.bResultantRows == TRUE)
        {
            // Stop iterating since we already have data
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                STOCK_TICKER_MGR_OBJECT_NAME": more that one record found");
            bResult = FALSE;
            break;
        }

        psArg->sOutput.bResultantRows = FALSE;

        // If there are the correct
        // number of columns, then we have good results
        if (n32NumberOfColumns != STOCK_TICKER_DB_DP_INFO_MAX_FIELDS)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                STOCK_TICKER_MGR_OBJECT_NAME
                ": incorrect query for the table (no. of fields is %d)",
                n32NumberOfColumns
                    );
            break;
        }

        // Grab the current column data
        psSeqIdCol = &psColumns[(UN8)STOCK_TICKER_DB_DP_INFO_FIELD_SEQID];
        psInfoCol = &psColumns[(UN8)STOCK_TICKER_DB_DP_INFO_FIELD_INFO];

        tStrLen = strlen((const char *)psInfoCol->uData.sCString.pcData);
        psArg->sOutput.uDbRow.sDataProviderInfo.hInfo =
            STRING.hCreate((const char *)psInfoCol->uData.sCString.pcData, tStrLen);
        if (psArg->sOutput.uDbRow.sDataProviderInfo.hInfo == STRING_INVALID_OBJECT)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                STOCK_TICKER_MGR_OBJECT_NAME": failed to create string object");
            break;
        }
        psArg->sOutput.uDbRow.sDataProviderInfo.un8SeqNumber =
            (UN8)psSeqIdCol->uData.sUN32.un32Data;

        psArg->sOutput.bResultantRows = TRUE;
        bResult = FALSE;
    } while (FALSE);

    return bResult;
}

/*****************************************************************************
*
*   eStockDataProviderInfoLoadDB
*
*****************************************************************************/
static STOCK_TICKER_RETURN_CODE_ENUM eStockDataProviderInfoLoadDB (
    STOCK_TICKER_APP_OBJECT_STRUCT *psAppObj
        )
{
    STOCK_TICKER_DB_QUERY_RESULT_STRUCT sArg;
    STOCK_TICKER_RETURN_CODE_ENUM eReturnCode = STOCK_TICKER_RETURN_CODE_ERROR;
    BOOLEAN bOk = FALSE;

    puts(STOCK_TICKER_MGR_OBJECT_NAME": Loading Data Provider Info");

    do
    {
        // Check input
        if (psAppObj == NULL)
        {
            eReturnCode = STOCK_TICKER_RETURN_CODE_BAD_INPUT;
            break;
        }

        ///////////////////////////////////////////////////////////////////////
        // Init argument
        OSAL.bMemSet(&sArg, 0, sizeof(sArg));
        sArg.sOutput.uDbRow.sDataProviderInfo.hInfo = STRING_INVALID_OBJECT;
        sArg.sOutput.uDbRow.sDataProviderInfo.un8SeqNumber = 0;
        sArg.sOutput.bResultantRows = FALSE;
        sArg.sInput.psAppObj = psAppObj;

        // Perform the SQL query and process the result
        // (it will provide us with a data row)
        bOk = SQL_INTERFACE.bQuery(
                psAppObj->sDB.hSQLConnection,
                STOCK_TICKER_SELECT_DB_DP_ALL,
                eStockDataProviderInfoLoadDBIterator,
                (void*) &sArg
                    );
        if (bOk != TRUE)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    STOCK_TICKER_MGR_OBJECT_NAME": failed to make query %s",
                    STOCK_TICKER_SELECT_DB_DP_ALL
                        );
            eReturnCode = STOCK_TICKER_RETURN_CODE_DB_ERROR;
            break;
        }

        if (sArg.sOutput.bResultantRows == TRUE)
        {
            // So, we have found the record
            psAppObj->sProviderInfoCtrl.hDataProviderInfoText =
                sArg.sOutput.uDbRow.sDataProviderInfo.hInfo;
            psAppObj->sProviderInfoCtrl.un8SequenceNumber =
                sArg.sOutput.uDbRow.sDataProviderInfo.un8SeqNumber;
            psAppObj->sProviderInfoCtrl.eState = STOCK_DATA_STATE_STABLE;
        }
#if SMS_DEBUG == 1
        else
        {
            printf(STOCK_TICKER_MGR_OBJECT_NAME": there is not Data Provider info\n");
        }
#endif

        eReturnCode = STOCK_TICKER_RETURN_CODE_SUCCESS;

    } while (FALSE);

    return eReturnCode;
}

/*******************************************************************************
*
*  bStockDataProviderInfoPrepareRow
*
*******************************************************************************/
static BOOLEAN bStockDataProviderInfoPrepareRow(
    SQL_COLUMN_INDEX tIndex,
    SQL_BIND_TYPE_ENUM *peType,
    size_t *ptDataSize,
    void **ppvData,
    STOCK_TICKER_PROVIDER_INFO_CONTROL_STRUCT *psProvider
        )
{
    BOOLEAN bSuccess = TRUE;

    switch (tIndex)
    {
        case STOCK_TICKER_DB_DP_INFO_FIELD_SEQID:
        {
            *peType = SQL_BIND_TYPE_UN32;
            *ppvData = (void *)(UN32)psProvider->un8SequenceNumber;
        }
        break;

        case STOCK_TICKER_DB_DP_INFO_FIELD_INFO:
        {
            *peType = SQL_BIND_TYPE_STRING_OBJECT;
            if (psProvider->hDataProviderInfoText != STRING_INVALID_OBJECT)
            {
                *ppvData = psProvider->hDataProviderInfoText;
            }
            else
            {
                SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                        STOCK_TICKER_MGR_OBJECT_NAME
                        ": there is no info string to provider"
                            );
            }
        }
        break;

        case STOCK_TICKER_DB_DP_INFO_MAX_FIELDS:
        default:
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    STOCK_TICKER_MGR_OBJECT_NAME": incorrect field index %d",
                    tIndex
                        );
            bSuccess = FALSE;
        }
        break;
    }

    return bSuccess;
}

/*****************************************************************************
*
*   eStockDataProviderInfoFlushDB
*
*****************************************************************************/
static STOCK_TICKER_RETURN_CODE_ENUM eStockDataProviderInfoFlushDB (
    STOCK_TICKER_APP_OBJECT_STRUCT *psAppObj
        )
{
    STOCK_TICKER_RETURN_CODE_ENUM eResult = STOCK_TICKER_RETURN_CODE_ERROR;
    STOCK_TICKER_PROVIDER_INFO_CONTROL_STRUCT *psProvider = NULL;

    do
    {
        // Check input
        if (psAppObj == NULL)
        {
            eResult = STOCK_TICKER_RETURN_CODE_BAD_INPUT;
            break;
        }

        // Get reference for quick access
        psProvider = &psAppObj->sProviderInfoCtrl;

        // Check states for processing
        if ((psProvider->eState != STOCK_DATA_STATE_UPDATED) &&
            (psProvider->eState != STOCK_DATA_STATE_CREATED))
        {
            // Since here nothing to do with DB just say OK
            eResult = STOCK_TICKER_RETURN_CODE_SUCCESS;
            break;
        }

        // Update DB based on state
        if (psProvider->eState == STOCK_DATA_STATE_UPDATED)
        {
            BOOLEAN bUpdated;

            printf(STOCK_TICKER_MGR_OBJECT_NAME": updating data provider info\n"
                        );

            snprintf(psAppObj->acBuffer, STOCK_TICKER_SHARED_BUFFER_LEN,
                    STOCK_TICKER_UPDATE_INSTEAD_OF_OLD,
                    psProvider->un8PrevSequenceNumber
                        );

            bUpdated = SQL_INTERFACE.bExecutePreparedCommand(
                            psAppObj->sDB.hSQLConnection,
                            psAppObj->acBuffer,
                            STOCK_TICKER_DB_STOCK_SYMBOL_MAX_FIELDS,
                            (PREPARED_QUERY_COLUMN_CALLBACK)bStockDataProviderInfoPrepareRow,
                            (void *) psProvider
                                );

            if (bUpdated == FALSE)
            {
                SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    STOCK_TICKER_MGR_OBJECT_NAME": Failed to update data provider (%s)",
                    psAppObj->acBuffer);
                eResult = STOCK_TICKER_RETURN_CODE_DB_ERROR;
                break;
            }
        }
        else if (psProvider->eState == STOCK_DATA_STATE_CREATED)
        {
            BOOLEAN bInserted;

            printf(STOCK_TICKER_MGR_OBJECT_NAME": updating data provider info\n");

            bInserted = SQL_INTERFACE.bExecutePreparedCommand(
                            psAppObj->sDB.hSQLConnection,
                            STOCK_TICKER_INSERT_DB_DP,
                            STOCK_TICKER_DB_STOCK_SYMBOL_MAX_FIELDS,
                            (PREPARED_QUERY_COLUMN_CALLBACK)bStockDataProviderInfoPrepareRow,
                            (void *) psProvider
                                );

            if (bInserted == FALSE)
            {
                SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    STOCK_TICKER_MGR_OBJECT_NAME": Failed to insert provider (%s)",
                    STOCK_TICKER_INSERT_DB_MSG_HASH);
                eResult = STOCK_TICKER_RETURN_CODE_DB_ERROR;
                break;
            }
        }
        else
        {
            printf(STOCK_TICKER_MGR_OBJECT_NAME
                ": skip since no DB action required for Data Provider info with SeqID==#%d\n",
                psProvider->un8SequenceNumber);
        }

        // Move to the stable state anyway
        psProvider->eState = STOCK_DATA_STATE_STABLE;

        eResult = STOCK_TICKER_RETURN_CODE_SUCCESS;
    } while (FALSE);

    return eResult;
}

/*****************************************************************************
*
*   eStockDataProviderInfoUnInit
*
*****************************************************************************/
static STOCK_TICKER_RETURN_CODE_ENUM eStockDataProviderInfoUnInit (
    STOCK_TICKER_MGR_OBJECT_STRUCT *psObj
        )
{
    STOCK_TICKER_APP_OBJECT_STRUCT *psAppObj;
    STOCK_TICKER_RETURN_CODE_ENUM eResult = STOCK_TICKER_RETURN_CODE_BAD_INPUT;

    puts(STOCK_TICKER_MGR_OBJECT_NAME": Uninitializing Data Provider Info");

    do
    {
        // Check input
        if (psObj == NULL)
        {
            break;
        }

        psAppObj = psObj->psAppObj;

        // Remove the provider data info
        if (psAppObj->sProviderInfoCtrl.hDataProviderInfoText != STRING_INVALID_OBJECT)
        {
            STRING_vDestroy(psAppObj->sProviderInfoCtrl.hDataProviderInfoText);
            psAppObj->sProviderInfoCtrl.hDataProviderInfoText = STRING_INVALID_OBJECT;
        }

        psAppObj->sProviderInfoCtrl.eState = STOCK_DATA_STATE_REMOVED;
        psAppObj->sProviderInfoCtrl.un8SequenceNumber = 0;

        eResult = STOCK_TICKER_RETURN_CODE_SUCCESS;
    } while (FALSE);

    return eResult;
}

#if ((SMS_DEBUG==1) && (STOCK_TICKER_EXTRA_DEBUG==1))
/*****************************************************************************
*
*   pacGetStockDataStateName
*
*****************************************************************************/
static const char *pacGetStockDataStateName (
    STOCK_DATA_STATE_ENUM eState
        )
{
    const char *pacResult = "UNKNOWN";

    switch (eState)
    {
        case STOCK_DATA_STATE_INITIAL:
            pacResult = MACRO_TO_STRING(STOCK_DATA_STATE_INITIAL);
            break;
        case STOCK_DATA_STATE_CREATED:
            pacResult = MACRO_TO_STRING(STOCK_DATA_STATE_CREATED);
            break;
        case STOCK_DATA_STATE_UPDATED:
            pacResult = MACRO_TO_STRING(STOCK_DATA_STATE_UPDATED);
            break;
        case STOCK_DATA_STATE_ERROR:
            pacResult = MACRO_TO_STRING(STOCK_DATA_STATE_ERROR);
            break;
        case STOCK_DATA_STATE_STABLE:
            pacResult = MACRO_TO_STRING(STOCK_DATA_STATE_STABLE);
            break;
        case STOCK_DATA_STATE_REMOVED:
            pacResult = MACRO_TO_STRING(STOCK_DATA_STATE_REMOVED);
            break;
    }

    return pacResult;
}
#endif

/*****************************************************************************
*
*   eDSRLInit
*
*****************************************************************************/
static STOCK_TICKER_RETURN_CODE_ENUM eDSRLInit (
    STOCK_TICKER_MGR_OBJECT_STRUCT *psObj
        )
{
    STOCK_TICKER_RETURN_CODE_ENUM eResult = STOCK_TICKER_RETURN_CODE_ERROR;
    OSAL_RETURN_CODE_ENUM eOsalReturnCode = OSAL_ERROR;

    puts(STOCK_TICKER_MGR_OBJECT_NAME": Initializing DSRL");

    do
    {
        // Check input
        if ((psObj == NULL) || (psObj->psAppObj == NULL))
        {
            eResult = STOCK_TICKER_RETURN_CODE_BAD_INPUT;
            break;
        }

        // Create the list
        eOsalReturnCode = OSAL.eLinkedListCreate (
                                    &psObj->psAppObj->hDSRLList,
                                    STOCK_TICKER_MGR_OBJECT_NAME": DSRLList",
                                    NULL, OSAL_LL_OPTION_NONE
                                        );
        if (eOsalReturnCode != OSAL_SUCCESS)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    STOCK_TICKER_MGR_OBJECT_NAME": failed to create list (%s)",
                    OSAL.pacGetReturnCodeName(eOsalReturnCode)
                        );
            break;
        }

        eResult = STOCK_TICKER_RETURN_CODE_SUCCESS;
    } while (FALSE);

    return eResult;
}

/*****************************************************************************
*
*   eDSRLDescHasToUpdate
*
*****************************************************************************/
static STOCK_TICKER_RETURN_CODE_ENUM eDSRLDescHasToUpdate (
    STOCK_TICKER_APP_OBJECT_STRUCT *psAppObj
        )
{
    UN32 un32Items = 0;
    OSAL_RETURN_CODE_ENUM eOsalReturnCode = OSAL_ERROR;
    STOCK_TICKER_RETURN_CODE_ENUM eReturnCode = STOCK_TICKER_RETURN_CODE_ERROR;

    if (psAppObj == NULL)
    {
        return STOCK_TICKER_RETURN_CODE_BAD_INPUT;
    }

    eOsalReturnCode = OSAL.eLinkedListItems(psAppObj->hDSRLList, &un32Items);
    if (eOsalReturnCode != OSAL_SUCCESS)
    {
        SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
            STOCK_TICKER_MGR_OBJECT_NAME": failed to get number of known DSRLs (%s)",
            OSAL.pacGetReturnCodeName(eOsalReturnCode)
                );
    }
    else
    {
        eReturnCode =  (un32Items > 0) ? STOCK_TICKER_RETURN_CODE_SUCCESS :
                                         STOCK_TICKER_RETURN_CODE_FALSE;
    }

    return eReturnCode;
}

/*****************************************************************************
*
*   bDSRLDescHasMessagesToUpdateByValuesIterator
*
*****************************************************************************/
static BOOLEAN bDSRLDescHasMessagesToUpdateByValuesIterator(
    STOCK_MSG_DESC_STRUCT *psMsgDesc,
    STOCK_MSG_INDEX_RANGE_CHECK_ITERATOR_ARG_STRUCT *psArg
        )
{
    STOCK_ID tIndex;
    BOOLEAN bContinue = TRUE;

    tIndex = STOCK_MSG_tIndex(psMsgDesc->hMessage);

    if (psArg->tFirstIndex <= tIndex)
    {
        if (tIndex <= psArg->tLastIndex)
        {
            // The index belongs to the range
            psArg->bFound = TRUE;
        }

        // There is nothing interesting further
        bContinue = FALSE;
    }

    return bContinue;
}

/*****************************************************************************
*
*   eDSRLDescHasMessagesToUpdateByValues
*
*   This function check indexes which are being monitored by the following
*   rules:
*   1) If the un16LastIndex less than the first index in the messages list
*   2) If the un16FirstIndex greater that the last index in the messages list
*   3) Looking through each of indexes inside the message list to find out
*      their presence inside the provided range.
*
*****************************************************************************/
static STOCK_TICKER_RETURN_CODE_ENUM eDSRLDescHasMessagesToUpdateByValues (
    STOCK_TICKER_APP_OBJECT_STRUCT *psAppObj,
    STOCK_ID tFirstIndex,
    STOCK_ID tLastIndex
        )
{
    OSAL_RETURN_CODE_ENUM eOsalReturnCode;
    STOCK_TICKER_RETURN_CODE_ENUM eReturnCode = STOCK_TICKER_RETURN_CODE_FALSE;
    OSAL_LINKED_LIST_ENTRY hEntry;
    STOCK_MSG_DESC_STRUCT *psMsgDesc;
    STOCK_ID tIndex;
    STOCK_MSG_INDEX_RANGE_CHECK_ITERATOR_ARG_STRUCT sArg;

    printf(STOCK_TICKER_MGR_OBJECT_NAME": checking range [#%u : #%u]\n",
        tFirstIndex, tLastIndex);

    do
    {
        ////////////////////////////////////////////////////////////////////
        // Step 1: Check the top entry against the last index
        hEntry = OSAL.hLinkedListFirst(psAppObj->hSymbolMsgList,
                        (void**)&psMsgDesc);
        if (hEntry == OSAL_INVALID_LINKED_LIST_ENTRY)
        {
            puts(STOCK_TICKER_MGR_OBJECT_NAME": looks like no entries in "
                "the messages list based on FIRST entry");
            break;
        }
        tIndex = STOCK_MSG_tIndex(psMsgDesc->hMessage);
        if (tIndex > tLastIndex)
        {
#if (SMS_DEBUG == 1) && (DEBUG_OBJECT == 1)
            STOCK_ID tIndex = STOCK_MSG_tIndex(psMsgDesc->hMessage);

            printf(STOCK_TICKER_MGR_OBJECT_NAME
                ": not needed due to FIRST MSG #%u > LAST #%u\n", tIndex, un16LastIndex);
#endif
            break;
        }
        else if (tIndex == tLastIndex)
        {
            // This one belongs to the range
            eReturnCode = STOCK_TICKER_RETURN_CODE_SUCCESS;
            break;
        }

        ////////////////////////////////////////////////////////////////////
        // Step 2: Check the bottom entry against the first index
        hEntry = OSAL.hLinkedListLast(psAppObj->hSymbolMsgList,
                        (void**)&psMsgDesc);
        if (hEntry == OSAL_INVALID_LINKED_LIST_ENTRY)
        {
            puts(STOCK_TICKER_MGR_OBJECT_NAME": looks like no entries in "
                "the messages list based on LAST entry");
            break;
        }
        tIndex = STOCK_MSG_tIndex(psMsgDesc->hMessage);
        if (tIndex < tFirstIndex)
        {
#if (SMS_DEBUG == 1) && (DEBUG_OBJECT == 1)
            STOCK_ID tIndex = STOCK_MSG_tIndex(psMsgDesc->hMessage);
            printf(STOCK_TICKER_MGR_OBJECT_NAME
                ": not needed due to LAST MSG #%u < FIRST #%u\n", tIndex, un16FirstIndex);
#endif
            break;
        }
        else if (tIndex == tFirstIndex)
        {
            // This one belongs to the range
            eReturnCode = STOCK_TICKER_RETURN_CODE_SUCCESS;
            break;
        }

        ////////////////////////////////////////////////////////////////////
        // Step 3: Iterate to find intersection
        printf(STOCK_TICKER_MGR_OBJECT_NAME
            ": range intersection for FIRST #%u and LAST #%u\n",
                    tFirstIndex, tLastIndex);
        sArg.tFirstIndex = tFirstIndex;
        sArg.tLastIndex = tLastIndex;
        sArg.bFound = FALSE;
        eOsalReturnCode =
            OSAL.eLinkedListIterate(psAppObj->hSymbolMsgList,
                (OSAL_LL_ITERATOR_HANDLER)
                    bDSRLDescHasMessagesToUpdateByValuesIterator,
                (void*)&sArg);
        if (eOsalReturnCode != OSAL_SUCCESS)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                STOCK_TICKER_MGR_OBJECT_NAME": failed to iterate messages (%s)",
                OSAL.pacGetReturnCodeName(eOsalReturnCode)
                    );
            break;
        }
        if (sArg.bFound == TRUE)
        {
            eReturnCode = STOCK_TICKER_RETURN_CODE_SUCCESS;
        }
    } while (FALSE);

    return eReturnCode;
}

/*****************************************************************************
*
*   psDSRLDescCreate
*
*****************************************************************************/
static STOCK_TICKER_DSRL_DESC_STRUCT *psDSRLDescCreate (
    DSRL_ARG_STRUCT *psDSRLArg
        )
{
    STOCK_TICKER_DSRL_DESC_STRUCT *psResult = NULL;
    OSAL_RETURN_CODE_ENUM eOsalReturnCode = OSAL_SUCCESS;
    BOOLEAN bOk = TRUE;
    DSRL_TARGET_TYPE_ENUM eTargetType;
    size_t tIndex;

    do
    {
        // Check input
        if (psDSRLArg == NULL)
        {
            break;
        }

        // Verify all targets are stocks
        for (tIndex = 0; tIndex < psDSRLArg->tNumTargets; ++tIndex)
        {
            // Get the current type
            eTargetType = DSRL_TARGET.eType(psDSRLArg->ahTargetList[tIndex]);
            if (eTargetType != DSRL_TARGET_TYPE_STOCK_SYMBOL)
            {
                // Only allow stock targets in here
                bOk = FALSE;
                break;
            }
        }
        // Did we experience any errors?
        if (bOk == FALSE)
        {
            break;
        }

        // Get the DSRL descriptor Memory
        psResult = (STOCK_TICKER_DSRL_DESC_STRUCT *)
                        DSRL_pvServiceData(psDSRLArg->hDSRL);
        if (psResult == NULL)
        {
            // Error!
            break;
        }

        // Init values
        OSAL.bMemSet(psResult, 0, sizeof(*psResult));
        psResult->hEntry = OSAL_INVALID_LINKED_LIST_ENTRY;
        psResult->hDSRL = psDSRLArg->hDSRL;
        psResult->hEntriesList = OSAL_INVALID_OBJECT_HDL;
        psResult->hTargetList = OSAL_INVALID_OBJECT_HDL;

        // Create a list of targets associated with this DSRL
        eOsalReturnCode = OSAL.eLinkedListCreate(&psResult->hTargetList,
                              STOCK_TICKER_MGR_OBJECT_NAME":DSRLDesc:TargetList",
                              NULL, OSAL_LL_OPTION_USE_PRE_ALLOCATED_ELEMENTS);
        if (eOsalReturnCode != OSAL_SUCCESS)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                      STOCK_TICKER_MGR_OBJECT_NAME
                      ": failed to create DSRL Target List (%s)",
                      OSAL.pacGetReturnCodeName(eOsalReturnCode)
                           );
            break;
        }

        // Create a list of entries  associated with this DSRL by its targets
        eOsalReturnCode = OSAL.eLinkedListCreate(&psResult->hEntriesList,
                              STOCK_TICKER_MGR_OBJECT_NAME":DSRLDesc:EntriesList",
                              NULL, OSAL_LL_OPTION_NONE);
        if (eOsalReturnCode != OSAL_SUCCESS)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                      STOCK_TICKER_MGR_OBJECT_NAME
                      ": failed to create DSRL Entries List (%s)",
                      OSAL.pacGetReturnCodeName(eOsalReturnCode)
                           );
            break;
        }
    } while (FALSE);

    if ((eOsalReturnCode != OSAL_SUCCESS) && (psResult != NULL))
    {
         vDSRLDescDestroy(psResult->hDSRL);
         psResult = NULL;
    }

    return psResult;
}

/*****************************************************************************
*
*   vDSRLDescSetState
*
*****************************************************************************/
static void vDSRLDescSetState (
     DSRL_OBJECT hDSRL,
     DSRL_STATE_ENUM eState
        )
{
    if (hDSRL != DSRL_INVALID_OBJECT)
    {
        DSRL_STATE_ENUM eOldState;

        eOldState = DSRL.eState(hDSRL);
        if (eOldState == DSRL_STATE_ERROR)
        {
            printf(STOCK_TICKER_MGR_OBJECT_NAME": DSRL %p in error state. Skip it.\n",
                hDSRL);
        }
        else
        {
#if (SMS_DEBUG==1)
            if (eOldState != eState)
            {
                printf(STOCK_TICKER_MGR_OBJECT_NAME
                    ": moving DSRL 0x%p from %d to %d\n",
                    hDSRL,
                    eOldState, eState);
            }
#endif
            DSRL_vSetState(hDSRL, eState);
        }
    }

    return;
}

/*****************************************************************************
*
*   bDSRLDescStateTransitionIterator
*
*****************************************************************************/
static BOOLEAN bDSRLDescStateTransitionIterator(
    DSRL_OBJECT hDSRL,
    STOCK_TICKER_DSRL_STATE_CHANGE_ITERATOR_STRUCT const *psArg
        )
{
    DSRL_STATE_ENUM eCurState = DSRL.eState(hDSRL);

    if (eCurState == psArg->eOldState)
    {
        vDSRLDescSetState(hDSRL, psArg->eNewState);
    }

    // Continue
    return TRUE;
}

/*****************************************************************************
*
*   eDSRLDescStateTransition
*
*   Move all DSRLs in specified state to new one
*
*****************************************************************************/
static STOCK_TICKER_RETURN_CODE_ENUM eDSRLDescStateTransition (
     STOCK_TICKER_APP_OBJECT_STRUCT *psAppObj,
     DSRL_STATE_ENUM eOldState,
     DSRL_STATE_ENUM eState
        )
{
    STOCK_TICKER_RETURN_CODE_ENUM eResult = STOCK_TICKER_RETURN_CODE_BAD_INPUT;

    if (psAppObj != NULL)
    {
        OSAL_RETURN_CODE_ENUM eOsalReturnCode;
        STOCK_TICKER_DSRL_STATE_CHANGE_ITERATOR_STRUCT sArg;

        // Init argument
        sArg.eNewState = eState;
        sArg.eOldState = eOldState;

        // Looking through all DSLR to update state
        eOsalReturnCode = OSAL.eLinkedListIterate(psAppObj->hDSRLList,
                                (OSAL_LL_ITERATOR_HANDLER) bDSRLDescStateTransitionIterator,
                                (void*) &sArg);
        if ((eOsalReturnCode != OSAL_SUCCESS) && (eOsalReturnCode != OSAL_NO_OBJECTS))
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                STOCK_TICKER_MGR_OBJECT_NAME": failed to iterate DSRL to change states (%s)",
                OSAL.pacGetReturnCodeName(eOsalReturnCode)
                    );
            eResult = STOCK_TICKER_RETURN_CODE_ERROR;
        }
        else
        {
            eResult = STOCK_TICKER_RETURN_CODE_SUCCESS;
        }
    }

    return eResult;
}

/*****************************************************************************
*
*   vDSRLDescEntryFinalizeFunction
*
*   FIXME: Need to use this function instead of direct removing entry
*          from corresponded DSRL
*
*****************************************************************************/
static void vDSRLDescEntryFinalizeFunction (
    DSRL_OBJECT hDSRL,
    DSRL_ENTRY_OBJECT hDSRLEntry,
    void *pvFinalizeArg
        )
{
    STOCK_TICKER_DSRL_DESC_STRUCT *psDSRLDesc = NULL;
    STOCK_MSG_DESC_STRUCT *psMsgDesc = NULL;
    STOCK_TICKER_RETURN_CODE_ENUM eReturnCode = STOCK_TICKER_RETURN_CODE_ERROR;
    OSAL_LINKED_LIST_ENTRY hEntry = OSAL_INVALID_LINKED_LIST_ENTRY;

    printf(STOCK_TICKER_MGR_OBJECT_NAME
            ": finalizing called for DSRL 0x%p and ENTRY 0x%p\n",
            hDSRL, hDSRLEntry
                );

    do
    {
        // Extract DSRL DEsc reference from the argument
        psDSRLDesc =
            (STOCK_TICKER_DSRL_DESC_STRUCT*) DSRL_pvServiceData(hDSRL);
        if (psDSRLDesc == NULL)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    STOCK_TICKER_MGR_OBJECT_NAME
                    ": failed to get service data from DSLR %p", hDSRL);
            break;
        }

        // Get try to find the message in the DSRL corresponded to the
        // DSRL
        psMsgDesc = psDSRLDescGetEntryByHandle(psDSRLDesc, hDSRLEntry, &hEntry);
        if (psMsgDesc == NULL)
        {
            // Well, looks like the item has been removed already
#if ((SMS_DEBUG == 1) && (STOCK_TICKER_EXTRA_DEBUG == 1))
            printf(STOCK_TICKER_MGR_OBJECT_NAME
                    ": [%d] not found mgs 0x%p in the DSRL 0x%p [ignored]\n",
                    __LINE__, hDSRLEntry, psDSRLDesc);
#endif
            break;
        }

        // Remove the record from the DSRL
        eReturnCode = eDSRLDescRemoveEntry(psDSRLDesc, psMsgDesc, hEntry, FALSE);
        if (eReturnCode != STOCK_TICKER_RETURN_CODE_SUCCESS)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    STOCK_TICKER_MGR_OBJECT_NAME
                    ": failed to remove message 0x%p from the DSRL 0x%p (%s)",
                    hDSRLEntry, hDSRL,
                    GsStockTickerMgrIntf.pacGetReturnCodeName(eReturnCode)
                        );
            break;
        }

    } while (FALSE);

    return;
}

/*****************************************************************************
*
*   bDSRLDescApplicableForMessageIterator
*
*****************************************************************************/
static BOOLEAN bDSRLDescApplicableForMessageIterator(
    STOCK_TICKER_DSRL_TARGET_DESC_STRUCT *psTargetDesc,
    STOCK_TICKER_TARGET_DESC_ITERATOR_STRUCT *psArg
        )
{
    BOOLEAN bContinue = FALSE;

    psArg->eResult = eStockMsgDescIsApplicableForTarget(psArg->psMsgDesc, psTargetDesc);
    if (psArg->eResult == STOCK_TICKER_RETURN_CODE_SUCCESS)
    {
        // Good, the message is applicable for the DSRL targets
    }
    else if (psArg->eResult != STOCK_TICKER_RETURN_CODE_FALSE)
    {
        SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                STOCK_TICKER_MGR_OBJECT_NAME
                ": failed to check message applicability by target (%s)",
                GsStockTickerMgrIntf.pacGetReturnCodeName(psArg->eResult)
                    );
        // We're bad. Stop doing further iteration
    }
    else
    {
        bContinue = TRUE;
    }

    return bContinue;
}

/*****************************************************************************
*
*   eDSRLDescApplicableForMessage
*
*   This function looks through the list of targets to find out does
*   the provided message fit to the DSRL or not regardless real
*   existence this message in the DSRL.
*
*****************************************************************************/
static STOCK_TICKER_RETURN_CODE_ENUM eDSRLDescApplicableForMessage (
    STOCK_TICKER_APP_OBJECT_STRUCT *psAppObj,
    STOCK_TICKER_DSRL_DESC_STRUCT *psDSRLDesc,
    STOCK_MSG_DESC_STRUCT *psMsgDesc
        )
{
    OSAL_RETURN_CODE_ENUM eOsalReturnCode;
    STOCK_TICKER_TARGET_DESC_ITERATOR_STRUCT sArg;

#if 0
    // XXX Turn on this code if you would like to see which symbol
    //     can be used to as targets.
    STRING_OBJECT hStockName;
    STOCK_ID tStockIndex;
    STOCK_MSG_STATE_ENUM eStockState;

    hStockName = STOCK_MSG.hSymbolName(psMsgDesc->hMessage);
    tStockIndex = STOCK_MSG_tIndex(psMsgDesc->hMessage);
    eStockState = STOCK_MSG_eState(psMsgDesc->hMessage);
    if (eStockState != STOCK_MSG_STATE_INVALID)
    {
        printf(STOCK_TICKER_MGR_OBJECT_NAME": *** %5u -> %s\n",
            tStockIndex, STRING.pacCStr(hStockName));
    }
#endif

    // Check input
    if ((psAppObj == NULL) || (psDSRLDesc == NULL) || (psMsgDesc == NULL))
    {
        sArg.eResult = STOCK_TICKER_RETURN_CODE_BAD_INPUT;
    }
    else
    {
        sArg.eResult = STOCK_TICKER_RETURN_CODE_FALSE;
        sArg.psMsgDesc = psMsgDesc;
        sArg.psAppObj = NULL;
        sArg.psDSRLDesc = NULL;

        // Looking through the list of targets to fit provided
        // message
        eOsalReturnCode = OSAL.eLinkedListIterate(psDSRLDesc->hTargetList,
                                (OSAL_LL_ITERATOR_HANDLER)bDSRLDescApplicableForMessageIterator,
                                (void*) &sArg);
        if ((eOsalReturnCode != OSAL_SUCCESS) && (eOsalReturnCode != OSAL_NO_OBJECTS))
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    STOCK_TICKER_MGR_OBJECT_NAME
                    ": failed to iterate target descriptors (%s)",
                    OSAL.pacGetReturnCodeName(eOsalReturnCode)
                        );
        }

#if (SMS_DEBUG == 1)
        if ((sArg.eResult != STOCK_TICKER_RETURN_CODE_SUCCESS) &&
            (sArg.eResult != STOCK_TICKER_RETURN_CODE_FALSE))
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    STOCK_TICKER_MGR_OBJECT_NAME
                    ": failed to check message applicability by target (%s)",
                    GsStockTickerMgrIntf.pacGetReturnCodeName(sArg.eResult)
                        );
        }
#endif
    }

    return sArg.eResult;
}

/*****************************************************************************
*
*   bIterateDSRLDescBuild
*
*****************************************************************************/
static BOOLEAN bDSRLDescBuildIterator(
    STOCK_MSG_DESC_STRUCT *psMsgDesc,
    STOCK_TICKER_MSG_DESCS_ITERATOR_STRUCT *psIteratorData
        )
{
    BOOLEAN bResult = FALSE;

    do
    {
        STOCK_TICKER_RETURN_CODE_ENUM eResult = STOCK_TICKER_RETURN_CODE_ERROR;

        if ((psMsgDesc == NULL) || (psIteratorData == NULL))
        {
            break;
        }

        eResult = eDSRLDescApplicableForMessage(psIteratorData->psAppObj,
            psIteratorData->psDSRLDesc, psMsgDesc
                );

        /**********************************************
         * Message fits, but not sure that it belongs to
         * the DSRL
         */
        if (eResult == STOCK_TICKER_RETURN_CODE_SUCCESS)
        {
            BOOLEAN bMessageExists = FALSE;

            // Good the message fits the DSRL
            printf(STOCK_TICKER_MGR_OBJECT_NAME": message #%d fits to the DSRL\n",
                            STOCK_MSG_tIndex(psMsgDesc->hMessage)
                                );

            if (((psIteratorData->tFlags) & STOCK_DSRL_BUILD_FLAG_CHECK_EXISTENCE) ==
                    STOCK_DSRL_BUILD_FLAG_CHECK_EXISTENCE)
            {
                // Check the items in the list
                eResult =
                    eDSRLDescHasEntry(psIteratorData->psDSRLDesc, psMsgDesc,
                                OSAL_INVALID_LINKED_LIST_ENTRY_PTR);
                if (eResult == STOCK_TICKER_RETURN_CODE_SUCCESS)
                {
                    // Message exists. Don't do anything though.
                    bMessageExists = TRUE;
                }
                else if (eResult != STOCK_TICKER_RETURN_CODE_FALSE)
                {
                    SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                            STOCK_TICKER_MGR_OBJECT_NAME
                            ": failed to check message 0x%p existence in the DSRL 0x%p (%s)",
                            psMsgDesc, psMsgDesc,
                            GsStockTickerMgrIntf.pacGetReturnCodeName(eResult)
                                );
                    break;
                }
            }

            // We need to add this message since it fits to the DSRL
            // but not in it yet.
            if (bMessageExists == FALSE)
            {
                // So, the function has been called
                // to add any message w/o check
                eResult = eDSRLDescAddEntry(psIteratorData->psAppObj,
                                        psIteratorData->psDSRLDesc, psMsgDesc
                                            );
                if (eResult == STOCK_TICKER_RETURN_CODE_FALSE)
                {
                    printf(STOCK_TICKER_MGR_OBJECT_NAME
                        ": looks like the DSRL has some restriction for this item");
                }
                else if (eResult != STOCK_TICKER_RETURN_CODE_SUCCESS)
                {
                    SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                            STOCK_TICKER_MGR_OBJECT_NAME
                            ": failed to add message 0x%p to the DSRL 0x%p (%s)",
                            psMsgDesc, psMsgDesc,
                            GsStockTickerMgrIntf.pacGetReturnCodeName(eResult)
                                );
                    break;
                }
            }
        }
        /**********************************************
         * The message doesn't fit to the DSRL.
         */
        else if (eResult == STOCK_TICKER_RETURN_CODE_FALSE)
        {
            printf(STOCK_TICKER_MGR_OBJECT_NAME": message #%d doesn't fit the DSRL\n",
                            STOCK_MSG_tIndex(psMsgDesc->hMessage)
                                );

            // The caller want us to remove messages which are not
            // fit to the DSRL targets
            if ((psIteratorData->tFlags & STOCK_DSRL_BUILD_FLAG_TRY_TO_REMOVE) ==
                      STOCK_DSRL_BUILD_FLAG_TRY_TO_REMOVE)
            {
                printf(STOCK_TICKER_MGR_OBJECT_NAME": trying to remove the #%d from the DSRL\n",
                                STOCK_MSG_tIndex(psMsgDesc->hMessage)
                                    );

                // So, we suppose that the message doesn't fit to the
                // DSRL, but it could be only after target update
                eResult = eDSRLDescRemoveEntry(psIteratorData->psDSRLDesc, psMsgDesc,
                                    OSAL_INVALID_LINKED_LIST_ENTRY,
                                        TRUE);
                if ((eResult != STOCK_TICKER_RETURN_CODE_SUCCESS) &&
                    (eResult != STOCK_TICKER_RETURN_CODE_FALSE))
                {
                    SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                            STOCK_TICKER_MGR_OBJECT_NAME
                            ": failed to remove message 0x%p from the DSRL 0x%p (%s)",
                            psMsgDesc->hMessage, psIteratorData->psDSRLDesc->hDSRL,
                            GsStockTickerMgrIntf.pacGetReturnCodeName(eResult)
                                );
                    break;
                }
            }
        }
        /**********************************************
         * Some other error code detected
         */
        else
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    STOCK_TICKER_MGR_OBJECT_NAME
                    ": failed to check message applicability by DSRL (%s)",
                    GsStockTickerMgrIntf.pacGetReturnCodeName(eResult)
                        );
            break;
        }

        bResult = TRUE;
    } while (FALSE);

    return bResult;
}


/*****************************************************************************
*
*   eDSRLDescBuild
*
*****************************************************************************/
static STOCK_TICKER_RETURN_CODE_ENUM eDSRLDescBuild (
    STOCK_TICKER_APP_OBJECT_STRUCT *psAppObj,
    STOCK_TICKER_DSRL_DESC_STRUCT *psDSRLDesc,
    STOCK_DSRL_BUILD_FLAGS tFlags
        )
{
    STOCK_TICKER_RETURN_CODE_ENUM eResult = STOCK_TICKER_RETURN_CODE_ERROR;
    OSAL_OBJECT_HDL hList = OSAL_INVALID_OBJECT_HDL;
    printf(STOCK_TICKER_MGR_OBJECT_NAME": building data for DSRL 0x%p\n", psDSRLDesc);

    do
    {
        STOCK_TICKER_MSG_DESCS_ITERATOR_STRUCT sIteratorData;
        OSAL_RETURN_CODE_ENUM eReturnCode = OSAL_ERROR;

        // Check input
        if ((psAppObj == NULL) || (psDSRLDesc == NULL))
        {
            eResult = STOCK_TICKER_RETURN_CODE_BAD_INPUT;
            break;
        }

        OSAL.bMemSet(&sIteratorData, 0, sizeof(sIteratorData));
        sIteratorData.psAppObj = psAppObj;
        sIteratorData.psDSRLDesc = psDSRLDesc;
        sIteratorData.tFlags = tFlags;

        // Looking through the list of messages from one
        // of the supported list based on flag
        if ((tFlags & STOCK_DSRL_BUILD_FLAG_PROCESS_DSRL_LIST) ==
                  STOCK_DSRL_BUILD_FLAG_PROCESS_DSRL_LIST)
        {
            printf(STOCK_TICKER_MGR_OBJECT_NAME": use DSRL entries list\n");
            hList = psDSRLDesc->hEntriesList;
        }
        else
        {
            printf(STOCK_TICKER_MGR_OBJECT_NAME": use full entries list\n");
            hList = psAppObj->hSymbolMsgList;

            // So, lets fill the messages list by messages for
            // all requested targets to make sure that message exists
            // even if stock symbol is known by the service but values
            // for it are absent
            eResult = eDSRLDescBuildBasedOnTargets(psAppObj, psDSRLDesc);
            if (eResult != STOCK_TICKER_RETURN_CODE_SUCCESS)
            {
                break;
            }
        }

        // Build the DSRL based on requested set of messages
        eReturnCode = OSAL.eLinkedListIterate(hList,
            (OSAL_LL_ITERATOR_HANDLER)bDSRLDescBuildIterator, &sIteratorData
                );
        if ((eReturnCode != OSAL_SUCCESS) && (eReturnCode != OSAL_NO_OBJECTS))
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    STOCK_TICKER_MGR_OBJECT_NAME
                    ": failed to iterate through list (%s)",
                    OSAL.pacGetReturnCodeName(eReturnCode)
                        );
            break;
        }

        eResult = STOCK_TICKER_RETURN_CODE_SUCCESS;
    } while (FALSE);

    return eResult;
}

/*****************************************************************************
*
*   eDSRLDescBuildBasedOnTargetsIterator
*
*****************************************************************************/
static BOOLEAN eDSRLDescBuildBasedOnTargetsIterator(
    STOCK_TICKER_DSRL_TARGET_DESC_STRUCT *psTarget,
    STOCK_TICKER_TARGET_DESC_ITERATOR_STRUCT *psArg
        )
{
    BOOLEAN bContinue = TRUE;

    // Is target known by the service?
    if (psTarget->tStockId != STOCK_INVALID_ID)
    {
        STOCK_TICKER_RETURN_CODE_ENUM eIntRetCode;

        eIntRetCode = eDSRLDescHasEntryForTarget(psArg->psDSRLDesc, psTarget);
        if (eIntRetCode == STOCK_TICKER_RETURN_CODE_FALSE)
        {
            // Need to add message to the DSRL, then.
            STOCK_MSG_DESC_STRUCT *psMsgDesc;

            // Do we already have message for the target?
            psMsgDesc = psStockMsgDescGet(psArg->psAppObj, psTarget->tStockId);
            if (psMsgDesc == NULL)
            {
                // Well, we need to create a new one
                STOCK_MSG_DATA_STRUCT sData;
                OSAL.bMemSet(&sData, 0, sizeof(sData));
                sData.tMask = STOCK_MSG_DATA_INDEX | STOCK_MSG_DATA_NAME | STOCK_MSG_DATA_VALUES;
                sData.tIndex = psTarget->tStockId;
                sData.uName.hValue = STOCK_SYMBOL_hName(psTarget->hTarget);
                sData.sValues.eState = STOCK_MSG_STATE_PRICE_NOT_AVAILABLE;

                // Create new descriptor since we know there is no such
                // descriptor in the manager
                psArg->eResult = eStockMsgDescCreate(&psMsgDesc, psArg->psAppObj, &sData);
                if (psArg->eResult != STOCK_TICKER_RETURN_CODE_SUCCESS)
                {
                    SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                            STOCK_TICKER_MGR_OBJECT_NAME
                            ": failed to create MsgDesc 0x%p for symbol #%d (%s)",
                            psMsgDesc, psTarget->tStockId,
                            GsStockTickerMgrIntf.pacGetReturnCodeName(psArg->eResult)
                                );
                    bContinue = FALSE;
                }
            }
        }
        else if (eIntRetCode != STOCK_TICKER_RETURN_CODE_SUCCESS)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                STOCK_TICKER_MGR_OBJECT_NAME
                ": failed to check existence of message for the target (%s)",
                GsStockTickerMgrIntf.pacGetReturnCodeName(eIntRetCode)
                    );
            bContinue = FALSE;
        }
    }

    return bContinue;
}

/*****************************************************************************
*
*   eDSRLDescBuildBasedOnTargets
*
*****************************************************************************/
static STOCK_TICKER_RETURN_CODE_ENUM eDSRLDescBuildBasedOnTargets (
    STOCK_TICKER_APP_OBJECT_STRUCT *psAppObj,
    STOCK_TICKER_DSRL_DESC_STRUCT *psDSRLDesc
        )
{
    STOCK_TICKER_TARGET_DESC_ITERATOR_STRUCT sArg;

    // Check input
    if ((psAppObj == NULL) || (psDSRLDesc == NULL))
    {
        sArg.eResult = STOCK_TICKER_RETURN_CODE_BAD_INPUT;
    }
    else
    {
        OSAL_RETURN_CODE_ENUM eOsalReturnCode;

        // Init argument
        sArg.eResult = STOCK_TICKER_RETURN_CODE_SUCCESS;
        sArg.psAppObj = psAppObj;
        sArg.psDSRLDesc = psDSRLDesc;
        sArg.psMsgDesc = NULL; // Not needed

        // Looking through the targets list
        eOsalReturnCode = OSAL.eLinkedListIterate(psDSRLDesc->hTargetList,
                                (OSAL_LL_ITERATOR_HANDLER)eDSRLDescBuildBasedOnTargetsIterator,
                                (void*) &sArg);
        if ((eOsalReturnCode != OSAL_SUCCESS) && (eOsalReturnCode != OSAL_NO_OBJECTS))
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                STOCK_TICKER_MGR_OBJECT_NAME": failed to iterate targets list (%s)",
                OSAL.pacGetReturnCodeName(eOsalReturnCode)
                    );
            sArg.eResult = STOCK_TICKER_RETURN_CODE_ERROR;
        }
    }

    return sArg.eResult;
}

/*****************************************************************************
*
*   eDSRLDescAddEntry
*
*****************************************************************************/
static STOCK_TICKER_RETURN_CODE_ENUM eDSRLDescAddEntry (
    STOCK_TICKER_APP_OBJECT_STRUCT *psAppObj,
    STOCK_TICKER_DSRL_DESC_STRUCT *psDSRLDesc,
    STOCK_MSG_DESC_STRUCT *psMsgDesc
        )
{
    STOCK_TICKER_RETURN_CODE_ENUM eResult = STOCK_TICKER_RETURN_CODE_ERROR;
    OSAL_RETURN_CODE_ENUM eOsalReturnCode = OSAL_ERROR;
    OSAL_LINKED_LIST_ENTRY hEntry = OSAL_INVALID_LINKED_LIST_ENTRY;
    DSRL_ADD_REPLACE_RESULT_ENUM eDSRLReturnCode = DSRL_ADD_REPLACE_ERROR;

    printf(STOCK_TICKER_MGR_OBJECT_NAME": adding message 0x%p to DSRL 0x%p\n",
                psMsgDesc, psDSRLDesc
                    );

    do
    {
        // Check input
        if ((psAppObj == NULL) || (psDSRLDesc == NULL) || (psMsgDesc == NULL))
        {
            eResult = STOCK_TICKER_RETURN_CODE_BAD_INPUT;
            break;
        }

        // Move to the updating state since we are going to
        // add new entry
        vDSRLDescSetState(psDSRLDesc->hDSRL, DSRL_STATE_UPDATING);

        // So, we are going to add this message to the DSRL but it could be
        // w/o name - let us update it
        eResult = eStockMsgDescUpdateNameFromDB(psAppObj, psMsgDesc);
        if ((eResult != STOCK_TICKER_RETURN_CODE_SUCCESS) &&
            (eResult != STOCK_TICKER_RETURN_CODE_NOTHING_TO_UPDATE))
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                STOCK_TICKER_MGR_OBJECT_NAME": failed to update name for message (%s)",
                GsStockTickerMgrIntf.pacGetReturnCodeName(eResult)
                    );
            break;
        }

        // Add entry to the list
        eDSRLReturnCode = DSRL_eAddEntry(psDSRLDesc->hDSRL,
                                (DSRL_ENTRY_OBJECT)psMsgDesc->hMessage
                                    );
        if (eDSRLReturnCode == DSRL_ADD_REPLACE_ERROR)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    STOCK_TICKER_MGR_OBJECT_NAME": failed to add entry 0x%p to the DSRL 0x%p",
                    psMsgDesc, psDSRLDesc);
            break;
        }
        else if (eDSRLReturnCode == DSRL_ADD_REPLACE_OK)
        {
            // So, the item is used by that DSTL. Increase the reference
            // counter.
            ++psMsgDesc->un32Ref;

            // Register entry
            eOsalReturnCode = OSAL.eLinkedListAdd(
                                        psDSRLDesc->hEntriesList, &hEntry,
                                        psMsgDesc
                                            );
            if (eOsalReturnCode != OSAL_SUCCESS)
            {
                SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                        STOCK_TICKER_MGR_OBJECT_NAME
                        ": failed to register 0x%p to the DSLR 0x%p (%s)",
                        psMsgDesc, psDSRLDesc,
                        OSAL.pacGetReturnCodeName(eOsalReturnCode)
                            );
                break;
            }

            eResult = STOCK_TICKER_RETURN_CODE_SUCCESS;
        }
        else
        {
            printf(STOCK_TICKER_MGR_OBJECT_NAME": the message 0x%p hasn't been added to the DSLR 0x%p",
                    psMsgDesc, psDSRLDesc);

            eResult = STOCK_TICKER_RETURN_CODE_FALSE;
        }

    } while (FALSE);

    return eResult;
}

/*****************************************************************************
*
*   eDSRLDescRemoveEntry
*
*****************************************************************************/
static STOCK_TICKER_RETURN_CODE_ENUM eDSRLDescRemoveEntry (
    STOCK_TICKER_DSRL_DESC_STRUCT *psDSRLDesc,
    STOCK_MSG_DESC_STRUCT *psMsgDesc,
    OSAL_LINKED_LIST_ENTRY hEntry,
    BOOLEAN bProcessAsDSRL
        )
{
    STOCK_TICKER_RETURN_CODE_ENUM eResult = STOCK_TICKER_RETURN_CODE_ERROR;
    OSAL_LINKED_LIST_ENTRY hIntEntry;

    printf(STOCK_TICKER_MGR_OBJECT_NAME": remove message 0x%p from DSRL 0x%p\n",
                psMsgDesc, psDSRLDesc
                    );

    do
    {
        // Check input
        if ((psDSRLDesc == NULL) || (psMsgDesc == NULL))
        {
            eResult = STOCK_TICKER_RETURN_CODE_BAD_INPUT;
            break;
        }

        // Do we have entry for the message in frame of the DSRL?
        if ((hEntry == OSAL_INVALID_LINKED_LIST_ENTRY) && (bProcessAsDSRL == FALSE))
        {
            // So, the caller didn't specify entry - thus try to find it
            eResult = eDSRLDescHasEntry(psDSRLDesc, psMsgDesc, &hIntEntry);
            if (eResult == STOCK_TICKER_RETURN_CODE_FALSE)
            {
                // Just in case
                hIntEntry = OSAL_INVALID_LINKED_LIST_ENTRY;
            }
            else if (eResult != STOCK_TICKER_RETURN_CODE_SUCCESS)
            {
                SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                        STOCK_TICKER_MGR_OBJECT_NAME
                        ": failed to check message 0x%p existence in the DSRL 0x%p (%s)",
                        psMsgDesc, psDSRLDesc,
                        GsStockTickerMgrIntf.pacGetReturnCodeName(eResult)
                            );
                break;
            }
        }
        else
        {
            hIntEntry = hEntry;
        }

        // Do we have entry for the message in frame of the DSRL?
        if ((hIntEntry == OSAL_INVALID_LINKED_LIST_ENTRY) && (bProcessAsDSRL == TRUE))
        {
            // At this point let's leave all work to the DSRL framework
            // via DSRL finalization function. In fact, the same function
            // eDSRLDescRemoveEntry will be called from the DSLR finalization
            // function, but the logic will be moved to another branch.
            DSRL_vRemoveEntry(psDSRLDesc->hDSRL,
                        (DSRL_ENTRY_OBJECT)psMsgDesc->hMessage);

            // We're in any case
            eResult = STOCK_TICKER_RETURN_CODE_SUCCESS;
        }
        else if (hIntEntry != OSAL_INVALID_LINKED_LIST_ENTRY)
        {
            OSAL_RETURN_CODE_ENUM eOsalReturnCode;

            // Remove the record from the DSRL's entries list
            eOsalReturnCode = OSAL.eLinkedListRemove(hIntEntry);
            if (eOsalReturnCode != OSAL_SUCCESS)
            {
                SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    STOCK_TICKER_MGR_OBJECT_NAME
                    ": failed to unreg message 0x%p from the DSRL 0x%p (%s)",
                    psMsgDesc, psDSRLDesc,
                    OSAL.pacGetReturnCodeName(eOsalReturnCode)
                    );

                eResult = STOCK_TICKER_RETURN_CODE_ERROR;
                break;
            }

            // Do we need to take care about this record as DSRL entry?
            if (bProcessAsDSRL == TRUE)
            {
                // Move to the updating state
                vDSRLDescSetState(psDSRLDesc->hDSRL, DSRL_STATE_UPDATING);

                // Remove record from the DSRL
                DSRL_vRemoveEntry(psDSRLDesc->hDSRL,
                            (DSRL_ENTRY_OBJECT)psMsgDesc->hMessage);
            }

            // So, the item is not longer belong to that DSRL and
            // we need to decrease the reference counter
            --psMsgDesc->un32Ref;

            // We're OK
            eResult = STOCK_TICKER_RETURN_CODE_SUCCESS;
        }
        else
        {
            // Nothing to remove
            eResult = STOCK_TICKER_RETURN_CODE_FALSE;
        }

    } while (FALSE);

    return eResult;
}


/*****************************************************************************
*
*   vDSRLDescRemoveAllEntriesNoDSRLUpdateReleaseCallback
*
*****************************************************************************/
static void vDSRLDescRemoveAllEntriesNoDSRLUpdateReleaseCallback (
    STOCK_MSG_DESC_STRUCT *psMsgDesc
        )
{
    --psMsgDesc->un32Ref;

    return;
}

/*****************************************************************************
*
*   eDSRLDescRemoveAllEntries
*
*   Removes all entries referenced by the DSRL from it.
*
*   In case if the second parameter set to TRUE the DSRL will be updated in this
*   case. If FASLE only message reference counter will be decreased.
*
*****************************************************************************/
static STOCK_TICKER_RETURN_CODE_ENUM eDSRLDescRemoveAllEntries (
    STOCK_TICKER_DSRL_DESC_STRUCT *psDSRLDesc,
    BOOLEAN bDoDSRLUpdate
        )
{
    STOCK_TICKER_RETURN_CODE_ENUM eResult = STOCK_TICKER_RETURN_CODE_SUCCESS;

    // Check input
    if (psDSRLDesc == NULL)
    {
        eResult = STOCK_TICKER_RETURN_CODE_BAD_INPUT;
    }
    else if (bDoDSRLUpdate == TRUE)
    {
        STOCK_MSG_DESC_STRUCT *psMsgDesc;
        OSAL_LINKED_LIST_ENTRY hEntry;

        printf(STOCK_TICKER_MGR_OBJECT_NAME": remove all entries from the DSRL %p\n",
            psDSRLDesc->hDSRL
                    );

        // XXX Remove all entries in "old-school" manner since the entry itself
        // doesn't know its own handle within the DSRL's list. Thus to do not degraded
        // performance due to extra searching this code needs to use that way.
        do
        {
            hEntry = OSAL.hLinkedListFirst(psDSRLDesc->hEntriesList, (void**) &psMsgDesc);
            if (hEntry == OSAL_INVALID_LINKED_LIST_ENTRY)
            {
                break;
            }

            // Do real remove of the entry
            eResult = eDSRLDescRemoveEntry(psDSRLDesc, psMsgDesc, hEntry, TRUE);
        } while (eResult == STOCK_TICKER_RETURN_CODE_SUCCESS);
    }
    else
    {
        OSAL_RETURN_CODE_ENUM eOsalReturnCode;

        eOsalReturnCode =
            OSAL.eLinkedListRemoveAll(psDSRLDesc->hEntriesList,
                (OSAL_LL_RELEASE_HANDLER)vDSRLDescRemoveAllEntriesNoDSRLUpdateReleaseCallback
                    );
        if ((eOsalReturnCode != OSAL_SUCCESS) && (eOsalReturnCode != OSAL_NO_OBJECTS))
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                STOCK_TICKER_MGR_OBJECT_NAME": failed to remove all entries for DSRL %p (%s)",
                psDSRLDesc->hDSRL, OSAL.pacGetReturnCodeName(eOsalReturnCode)
                    );
            eResult = STOCK_TICKER_RETURN_CODE_ERROR;
        }
    }

    return eResult;
}

/*****************************************************************************
*
*   eDSRLDescHasEntry
*
*****************************************************************************/
static STOCK_TICKER_RETURN_CODE_ENUM eDSRLDescHasEntry (
    STOCK_TICKER_DSRL_DESC_STRUCT *psDSRLDesc,
    STOCK_MSG_DESC_STRUCT *psMsgDesc,
    OSAL_LINKED_LIST_ENTRY *phEntry
        )
{
    STOCK_TICKER_RETURN_CODE_ENUM eResult = STOCK_TICKER_RETURN_CODE_ERROR;
    OSAL_RETURN_CODE_ENUM eOsalReturnCode = OSAL_ERROR;
    OSAL_LINKED_LIST_ENTRY hEntry = OSAL_INVALID_LINKED_LIST_ENTRY;

    printf(STOCK_TICKER_MGR_OBJECT_NAME": checking Msg 0x%p in the DSRL 0x%p\n",
                psMsgDesc, psDSRLDesc
                    );

    do
    {
        // Check input
        if ((psDSRLDesc == NULL) || (psMsgDesc == NULL))
        {
            eResult = STOCK_TICKER_RETURN_CODE_BAD_INPUT;
            break;
        }

        // Register entry
        eOsalReturnCode = OSAL.eLinkedListSearch(
                                    psDSRLDesc->hEntriesList,
                                    &hEntry,
                                    (void*)psMsgDesc
                                        );
        if (eOsalReturnCode == OSAL_SUCCESS)
        {
            // Good, we found it
            eResult = STOCK_TICKER_RETURN_CODE_SUCCESS;
        }
        else if ((eOsalReturnCode == OSAL_NO_OBJECTS) ||
                 (eOsalReturnCode == OSAL_OBJECT_NOT_FOUND))
        {
            // Nothing similar in the list
            eResult = STOCK_TICKER_RETURN_CODE_FALSE;
        }
        else
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    STOCK_TICKER_MGR_OBJECT_NAME
                    ": failed to search message 0x%p to the DSLR 0x%p",
                    psMsgDesc, psDSRLDesc
                        );
            break;
        }

        if (phEntry != NULL)
        {
            // Populate the handle
            *phEntry = hEntry;
        }

    } while (FALSE);

    return eResult;
}

/*****************************************************************************
*
*   n16DSRLDescHasEntryForTargetSearcher
*
*****************************************************************************/
static N16 n16DSRLDescHasEntryForTargetSearcher(
    STOCK_MSG_DESC_STRUCT *psMsgDesc,
    STRING_OBJECT hTargetName
        )
{
    STRING_OBJECT hMessageName;
    N16 n16ComapreResult;

    // Get name
    hMessageName = STOCK_MSG.hSymbolName(psMsgDesc->hMessage);

    // Do compare
    n16ComapreResult = STRING.n16Compare(hMessageName, hTargetName, TRUE);

    return n16ComapreResult;
}

/*****************************************************************************
*
*   eDSRLDescHasEntryForTarget
*
*****************************************************************************/
static STOCK_TICKER_RETURN_CODE_ENUM eDSRLDescHasEntryForTarget (
    STOCK_TICKER_DSRL_DESC_STRUCT *psDSRLDesc,
    STOCK_TICKER_DSRL_TARGET_DESC_STRUCT *psTarget
        )
{
    STOCK_TICKER_RETURN_CODE_ENUM eResult = STOCK_TICKER_RETURN_CODE_ERROR;
    OSAL_RETURN_CODE_ENUM eOsalReturnCode = OSAL_ERROR;
    OSAL_LINKED_LIST_ENTRY hEntry = OSAL_INVALID_LINKED_LIST_ENTRY;
    STRING_OBJECT hTargetName = STRING_INVALID_OBJECT;

    printf(STOCK_TICKER_MGR_OBJECT_NAME
        ": looking for Msg corresponded to Target 0x%p in the DSRL 0x%p\n",
        psTarget, psDSRLDesc
                    );

    do
    {
        // Check input
        if ((psDSRLDesc == NULL) || (psTarget == NULL))
        {
            eResult = STOCK_TICKER_RETURN_CODE_BAD_INPUT;
            break;
        }

        // Extract name from the target
        hTargetName = STOCK_SYMBOL_hName(psTarget->hTarget);
        if (hTargetName == STRING_INVALID_OBJECT)
        {
            eResult = STOCK_TICKER_RETURN_CODE_BAD_INPUT;
            break;
        }

        // Looking for the message
        eOsalReturnCode = OSAL.eLinkedListLinearSearch(
                            psDSRLDesc->hEntriesList,
                            &hEntry, (OSAL_LL_COMPARE_HANDLER) n16DSRLDescHasEntryForTargetSearcher,
                            (void*) hTargetName
                                );
        if (eOsalReturnCode == OSAL_SUCCESS)
        {
            eResult = STOCK_TICKER_RETURN_CODE_SUCCESS;
        }
        else if ((eOsalReturnCode == OSAL_NO_OBJECTS) ||
                 (eOsalReturnCode == OSAL_OBJECT_NOT_FOUND))
        {
            eResult = STOCK_TICKER_RETURN_CODE_FALSE;
        }
#if SMS_DEBUG==1
        else
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                STOCK_TICKER_MGR_OBJECT_NAME": failed to search in the list (%s)",
                OSAL.pacGetReturnCodeName(eOsalReturnCode)
                    );
        }
#endif
    } while (FALSE);

    return eResult;
}

/*****************************************************************************
*
*   psDSRLDescGetEntryByHandleSearcher
*
*****************************************************************************/
static N16 n16DSRLDescGetEntryByHandleSearcher (
    STOCK_MSG_DESC_STRUCT *psMsgDesc,
    DSRL_ENTRY_OBJECT hDSRLEntry
        )
{
    if (psMsgDesc->hMessage == (STOCK_MSG_OBJECT)hDSRLEntry)
    {
        return 0;
    }

    return -1;
}

/*****************************************************************************
*
*   psDSRLDescGetEntryByHandle
*
*   Looks through all messages from the DSRL to find out descriptor
*   which handles passed DSRL entry. In case of success function
*   provides handle to the list entry in the DSRL entries list and
*   returns reference to the DSRL entry descriptor.
*
*****************************************************************************/
static STOCK_MSG_DESC_STRUCT *psDSRLDescGetEntryByHandle (
    STOCK_TICKER_DSRL_DESC_STRUCT *psDSRLDesc,
    DSRL_ENTRY_OBJECT hDSRLEntry,
    OSAL_LINKED_LIST_ENTRY *phEntry
        )
{
    STOCK_MSG_DESC_STRUCT *psResult = NULL;
    OSAL_RETURN_CODE_ENUM eOsalReturnCode = OSAL_ERROR;
    OSAL_LINKED_LIST_ENTRY hEntry = OSAL_INVALID_LINKED_LIST_ENTRY;

    printf(STOCK_TICKER_MGR_OBJECT_NAME": looking for message 0x%p from DSRL 0x%p\n",
            hDSRLEntry, psDSRLDesc
                    );

    do
    {
        // Check input
        if ((psDSRLDesc == NULL) || (hDSRLEntry == DSRL_ENTRY_INVALID_OBJECT))
        {
            break;
        }

        // Initialize possible return value
        *phEntry = OSAL_INVALID_LINKED_LIST_ENTRY;

        // Register entry
        eOsalReturnCode = OSAL.eLinkedListLinearSearch(
                                    psDSRLDesc->hEntriesList,
                                    &hEntry,
                                    (OSAL_LL_COMPARE_HANDLER) n16DSRLDescGetEntryByHandleSearcher,
                                    (void*) hDSRLEntry
                                        );
        if (eOsalReturnCode == OSAL_SUCCESS)
        {
            // Good, we found it
            psResult =
                    (STOCK_MSG_DESC_STRUCT*)
                            OSAL.pvLinkedListThis(hEntry);

            *phEntry = hEntry;
        }
        else if ((eOsalReturnCode != OSAL_NO_OBJECTS) &&
                 (eOsalReturnCode != OSAL_OBJECT_NOT_FOUND))
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    STOCK_TICKER_MGR_OBJECT_NAME": failed to search message 0x%p to the DSLR 0x%p",
                    hDSRLEntry, psDSRLDesc
                        );
            break;
        }
    } while (FALSE);

    return psResult;
}

/*****************************************************************************
*
*   vDSRLDescDestroy
*
*****************************************************************************/
static void vDSRLDescDestroy (
     DSRL_OBJECT hDSRL
          )
{
    printf(STOCK_TICKER_MGR_OBJECT_NAME
                    ": attempting to destroy DSRLDesc 0x%p\n",
                    hDSRL);

    if (hDSRL != DSRL_INVALID_OBJECT)
    {
        STOCK_TICKER_DSRL_DESC_STRUCT *psDSRLDesc =
           (STOCK_TICKER_DSRL_DESC_STRUCT*) DSRL_pvServiceData(hDSRL);

        if (psDSRLDesc != NULL )
        {
            OSAL_RETURN_CODE_ENUM eOsalReturnCode = OSAL_SUCCESS;
            STOCK_TICKER_RETURN_CODE_ENUM eReturnCode = STOCK_TICKER_RETURN_CODE_ERROR;

            if (psDSRLDesc->hTargetList != OSAL_INVALID_OBJECT_HDL )
            {
                eReturnCode = eDSRLTargetRemoveAll(psDSRLDesc);
                if (eReturnCode != STOCK_TICKER_RETURN_CODE_SUCCESS)
                {
                    SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                        STOCK_TICKER_MGR_OBJECT_NAME": failed to remove all targets (%s)",
                        GsStockTickerMgrIntf.pacGetReturnCodeName(eReturnCode)
                            );
                }

                eOsalReturnCode = OSAL.eLinkedListDelete(
                                                    psDSRLDesc->hTargetList
                                                                 );
                if (eOsalReturnCode != OSAL_SUCCESS)
                {
                    SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                        STOCK_TICKER_MGR_OBJECT_NAME": failed to delete targets list (%s)",
                        OSAL.pacGetReturnCodeName(eOsalReturnCode));
                }
            }

            if (psDSRLDesc->hEntriesList != OSAL_INVALID_OBJECT_HDL )
            {
                eOsalReturnCode = OSAL.eLinkedListRemoveAll(
                          psDSRLDesc->hEntriesList,
                          NULL
                               );
                if (eOsalReturnCode != OSAL_SUCCESS)
                {
                    SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                        STOCK_TICKER_MGR_OBJECT_NAME": failed to remove all entries (%s)",
                        OSAL.pacGetReturnCodeName(eOsalReturnCode));
                }

                eOsalReturnCode = OSAL.eLinkedListDelete(
                          psDSRLDesc->hEntriesList
                               );
                if (eOsalReturnCode != OSAL_SUCCESS)
                {
                    SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                        STOCK_TICKER_MGR_OBJECT_NAME": failed to delete entries list (%s)",
                        OSAL.pacGetReturnCodeName(eOsalReturnCode));
                }

                psDSRLDesc->hEntriesList = OSAL_INVALID_OBJECT_HDL;
            }
        }

        // Destroy the DSRL
        DSRL_vDestroy(hDSRL);
    }

     return;
}

/*****************************************************************************
*
*   eDSRLListModifyAdd
*
*****************************************************************************/
static STOCK_TICKER_RETURN_CODE_ENUM eDSRLListModifyAdd (
    STOCK_TICKER_APP_OBJECT_STRUCT *psAppObj,
    STOCK_TICKER_DSRL_DESC_STRUCT *psDSRLDesc,
    DSRL_ARG_STRUCT *psDSRLArg
        )
{
    STOCK_TICKER_RETURN_CODE_ENUM eReturnCode = STOCK_TICKER_RETURN_CODE_ERROR;

    printf(STOCK_TICKER_MGR_OBJECT_NAME": Processing MODIFY ADD for DSRL 0x%p\n",
            psDSRLArg->hDSRL
                );
    do
    {
        // Check input
        if ((psAppObj == NULL) || (psDSRLDesc == NULL) ||
            (psDSRLArg == NULL))
        {
            eReturnCode = STOCK_TICKER_RETURN_CODE_BAD_INPUT;
            break;
        }

        // Building a target list by adding new received targets
        // to existing ones.
        eReturnCode = eDSRLTargetBuildAll(psAppObj, psDSRLDesc,
                psDSRLArg->ahTargetList, psDSRLArg->tNumTargets
                    );
        if (eReturnCode != STOCK_TICKER_RETURN_CODE_SUCCESS)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    STOCK_TICKER_MGR_OBJECT_NAME
                    ": failed to build targets for DSRL %p",
                    psDSRLDesc->hDSRL
                        );
            break;
        }

        // Refresh the DSRL based on new target since we suppose that
        // data for old data in it we just need to update the list by
        // items corresponded to new target.
        eReturnCode = eDSRLDescBuild(psAppObj, psDSRLDesc,
                  STOCK_DSRL_BUILD_FLAG_CHECK_EXISTENCE
                       );
        if (eReturnCode != STOCK_TICKER_RETURN_CODE_SUCCESS)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    STOCK_TICKER_MGR_OBJECT_NAME
                    ": failed to build DSRL %p",
                    psDSRLDesc->hDSRL
                        );
            break;
        }

    } while (FALSE);

    return eReturnCode;
}

/*****************************************************************************
*
*   eDSRLListModifyReplace
*
*****************************************************************************/
static STOCK_TICKER_RETURN_CODE_ENUM eDSRLListModifyReplace (
    STOCK_TICKER_APP_OBJECT_STRUCT *psAppObj,
    STOCK_TICKER_DSRL_DESC_STRUCT *psDSRLDesc,
    DSRL_ARG_STRUCT *psDSRLArg
        )
{
    STOCK_TICKER_RETURN_CODE_ENUM eReturnCode = STOCK_TICKER_RETURN_CODE_ERROR;

    printf(STOCK_TICKER_MGR_OBJECT_NAME": Processing MODIFY REPLACE for DSRL 0x%p\n",
            psDSRLArg->hDSRL
                );
    do
    {
        // Check input
        if ((psAppObj == NULL) || (psDSRLDesc == NULL) ||
            (psDSRLArg == NULL))
        {
            eReturnCode = STOCK_TICKER_RETURN_CODE_BAD_INPUT;
            break;
        }

        // Clean up current targets
        eReturnCode = eDSRLTargetRemoveAll(psDSRLDesc);
        if (eReturnCode != STOCK_TICKER_RETURN_CODE_SUCCESS)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    STOCK_TICKER_MGR_OBJECT_NAME
                    ": failed to remove all targets for DSRL %p (%s)",
                    psDSRLDesc,
                    GsStockTickerMgrIntf.pacGetReturnCodeName(eReturnCode)
                        );
            break;
        }

        // Build new targets
        eReturnCode = eDSRLTargetBuildAll(psAppObj, psDSRLDesc,
                psDSRLArg->ahTargetList, psDSRLArg->tNumTargets
                    );
        if (eReturnCode != STOCK_TICKER_RETURN_CODE_SUCCESS)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    STOCK_TICKER_MGR_OBJECT_NAME
                    ": failed to build targets for DSRL %p (%s)",
                    psDSRLDesc,
                    GsStockTickerMgrIntf.pacGetReturnCodeName(eReturnCode)
                        );
            break;
        }

        // Refresh the DSRL based on new target
        eReturnCode = eDSRLDescBuild(psAppObj, psDSRLDesc,
                                   STOCK_DSRL_BUILD_FLAG_CHECK_EXISTENCE |
                                   STOCK_DSRL_BUILD_FLAG_TRY_TO_REMOVE
                                        );
        if (eReturnCode != STOCK_TICKER_RETURN_CODE_SUCCESS)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    STOCK_TICKER_MGR_OBJECT_NAME
                    ": failed to build DSRL %p (%s)",
                    psDSRLDesc,
                    GsStockTickerMgrIntf.pacGetReturnCodeName(eReturnCode)
                        );
            break;
        }

    } while (FALSE);

    return eReturnCode;
}

/*****************************************************************************
*
*   eDSRLListModifyRemove
*
*****************************************************************************/
static STOCK_TICKER_RETURN_CODE_ENUM eDSRLListModifyRemove (
    STOCK_TICKER_APP_OBJECT_STRUCT *psAppObj,
    STOCK_TICKER_DSRL_DESC_STRUCT *psDSRLDesc,
    DSRL_ARG_STRUCT *psDSRLArg
        )
{
    STOCK_TICKER_RETURN_CODE_ENUM eReturnCode = STOCK_TICKER_RETURN_CODE_ERROR;
    size_t tIndex = 0;
    BOOLEAN bNeedToRebuild = FALSE;

    printf(STOCK_TICKER_MGR_OBJECT_NAME": Processing MODIFY REMOVE for DSRL 0x%p\n",
            psDSRLArg->hDSRL
                );
    do
    {
        // Check input
        if ((psAppObj == NULL) || (psDSRLDesc == NULL) ||
            (psDSRLArg == NULL))
        {
            eReturnCode = STOCK_TICKER_RETURN_CODE_BAD_INPUT;
            break;
        }

        // Remove targets specified in the list is corresponded
        // is found
        for (tIndex = 0; tIndex < psDSRLArg->tNumTargets; ++tIndex)
        {
            STOCK_TICKER_DSRL_TARGET_DESC_STRUCT *psTarget;
            STOCK_SYMBOL_OBJECT hTarget;
            DSRL_TARGET_TYPE_ENUM eTargetType;

            hTarget = (STOCK_SYMBOL_OBJECT)psDSRLArg->ahTargetList[tIndex];

            // Check target type
            eTargetType = DSRL_TARGET.eType(hTarget);
            if (eTargetType != DSRL_TARGET_TYPE_STOCK_SYMBOL)
            {
                // Continue with next target;
                continue;
            }

            // Try to find the similar target in the DSRL
            psTarget = psDSRLTargetGet(psDSRLDesc, hTarget);
            if (psTarget != NULL)
            {
                // We found the target in the DSRL so need to remove
                // it from the DSLR targets
                eReturnCode = eDSRLTargetRemove(psDSRLDesc, psTarget);
                if (eReturnCode != STOCK_TICKER_RETURN_CODE_SUCCESS)
                {
                    SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                            STOCK_TICKER_MGR_OBJECT_NAME
                            ": failed to remove target %p from DSRL %p (%s)",
                            psTarget, psDSRLDesc,
                            GsStockTickerMgrIntf.pacGetReturnCodeName(eReturnCode)
                                );
                    break;
                }

                // Oh, need to rebuild DSRL since we do remove a target
                bNeedToRebuild = TRUE;
            }
        }

        if (bNeedToRebuild == TRUE)
        {
            // Refresh the DSRL based on new target
            eReturnCode = eDSRLDescBuild(psAppObj, psDSRLDesc,
                                       STOCK_DSRL_BUILD_FLAG_PROCESS_DSRL_LIST |
                                       STOCK_DSRL_BUILD_FLAG_TRY_TO_REMOVE |
                                       STOCK_DSRL_BUILD_FLAG_CHECK_EXISTENCE
                                            );
            if (eReturnCode != STOCK_TICKER_RETURN_CODE_SUCCESS)
            {
                SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                        STOCK_TICKER_MGR_OBJECT_NAME
                        ": failed to build DSRL %p (%s)",
                        psDSRLDesc,
                        GsStockTickerMgrIntf.pacGetReturnCodeName(eReturnCode)
                            );
                break;
            }
        }
        else
        {
            // Just say all done
            eReturnCode = STOCK_TICKER_RETURN_CODE_SUCCESS;
        }

    } while (FALSE);

    return eReturnCode;
}

/*****************************************************************************
*
*   eDSRLListModifyRemoveAll
*
*****************************************************************************/
static STOCK_TICKER_RETURN_CODE_ENUM eDSRLListModifyRemoveAll (
    STOCK_TICKER_APP_OBJECT_STRUCT *psAppObj,
    STOCK_TICKER_DSRL_DESC_STRUCT *psDSRLDesc
        )
{
    STOCK_TICKER_RETURN_CODE_ENUM eReturnCode = STOCK_TICKER_RETURN_CODE_ERROR;

    printf(STOCK_TICKER_MGR_OBJECT_NAME": Processing MODIFY REMOVE for DSRL 0x%p\n",
            psDSRLDesc->hDSRL
                );
    do
    {
        // Check input
        if ((psAppObj == NULL) || (psDSRLDesc == NULL))
        {
            eReturnCode = STOCK_TICKER_RETURN_CODE_BAD_INPUT;
            break;
        }

        // Clean up current targets
        eReturnCode = eDSRLTargetRemoveAll(psDSRLDesc);
        if (eReturnCode != STOCK_TICKER_RETURN_CODE_SUCCESS)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    STOCK_TICKER_MGR_OBJECT_NAME
                    ": failed to remove all targets for DSRL %p (%s)",
                    psDSRLDesc,
                    GsStockTickerMgrIntf.pacGetReturnCodeName(eReturnCode)
                        );
            break;
        }

        // Say to remove all entries from the DSRL
        eReturnCode = eDSRLDescRemoveAllEntries(psDSRLDesc, TRUE);
        if (eReturnCode != STOCK_TICKER_RETURN_CODE_SUCCESS)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    STOCK_TICKER_MGR_OBJECT_NAME
                    ": failed to remove all targets from DSRL %p (%s)",
                    psDSRLDesc,
                    GsStockTickerMgrIntf.pacGetReturnCodeName(eReturnCode)
                        );
            break;
        }

    } while (FALSE);

    return eReturnCode;
}

/*****************************************************************************
*
*   eDSRLUnInit
*
*****************************************************************************/
static STOCK_TICKER_RETURN_CODE_ENUM eDSRLUnInit (
    STOCK_TICKER_APP_OBJECT_STRUCT *psAppObj
        )
{
    STOCK_TICKER_RETURN_CODE_ENUM eResult = STOCK_TICKER_RETURN_CODE_ERROR;
    OSAL_RETURN_CODE_ENUM eOsalReturnCode = OSAL_ERROR;

    printf(STOCK_TICKER_MGR_OBJECT_NAME": Uninitializing DSRL\n");

    do
    {
        // Check input
        if (psAppObj == NULL)
        {
            eResult = STOCK_TICKER_RETURN_CODE_BAD_INPUT;
            break;
        }

        if (psAppObj->hDSRLList != OSAL_INVALID_OBJECT_HDL)
        {
            // Destroy all DSRLs
            eOsalReturnCode = OSAL.eLinkedListRemoveAll(psAppObj->hDSRLList,
                    (OSAL_LL_RELEASE_HANDLER)vDSRLDescDestroy
                        );
            if ((eOsalReturnCode != OSAL_SUCCESS) && (eOsalReturnCode != OSAL_NO_OBJECTS))
            {
                SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                        STOCK_TICKER_MGR_OBJECT_NAME": failed to remove all items from DSRL list");
            }

            eOsalReturnCode = OSAL.eLinkedListDelete(psAppObj->hDSRLList);
            if (eOsalReturnCode != OSAL_SUCCESS)
            {
                SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                        STOCK_TICKER_MGR_OBJECT_NAME": failed to delete DSRL list");
            }

            psAppObj->hDSRLList = OSAL_INVALID_OBJECT_HDL;
        }

        eResult = STOCK_TICKER_RETURN_CODE_SUCCESS;
    } while (FALSE);

    return eResult;
}

/*****************************************************************************
*
*   eDSRLListCreate
*
*****************************************************************************/
static STOCK_TICKER_RETURN_CODE_ENUM eDSRLListCreate (
    STOCK_TICKER_APP_OBJECT_STRUCT *psAppObj,
    DSRL_ARG_STRUCT *psDSRLArg
        )
{
    BOOLEAN bSuccess = FALSE;
    OSAL_RETURN_CODE_ENUM eOsalReturnCode;
    STOCK_TICKER_RETURN_CODE_ENUM eReturnCode = STOCK_TICKER_RETURN_CODE_ERROR;
    STOCK_TICKER_DSRL_DESC_STRUCT *psDSRLDesc = NULL;

    do
    {
        // Create our DSRL descriptor based
        // on the provided arguments
        psDSRLDesc = psDSRLDescCreate(psDSRLArg);
        if (psDSRLDesc == NULL)
        {
            // Error!
            break;
        }

        // Add this to our list of tracked targets for this service
        eOsalReturnCode = OSAL.eLinkedListAdd(psAppObj->hDSRLList,
                            &psDSRLDesc->hEntry,
                            (void*) psDSRLArg->hDSRL
                                );
        if (eOsalReturnCode != OSAL_SUCCESS)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    STOCK_TICKER_MGR_OBJECT_NAME": failed to add new DSRL (%s)\n",
                    OSAL.pacGetReturnCodeName(eOsalReturnCode)
                        );
            break;
        }

        // Set DSRL properties

        // Set finalization function
        bSuccess = DSRL_bSetFinalizeFunction(psDSRLDesc->hDSRL,
                            vDSRLDescEntryFinalizeFunction,
                            NULL
                                );
        if (bSuccess == FALSE)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    STOCK_TICKER_MGR_OBJECT_NAME": failed to set DSRL 0x%p finalize function",
                    psDSRLDesc->hDSRL
                        );
            break;
        }

        // Start populating the list
        eReturnCode = eDSRLTargetBuildAll(psAppObj, psDSRLDesc,
                psDSRLArg->ahTargetList, psDSRLArg->tNumTargets);
        if (eReturnCode != STOCK_TICKER_RETURN_CODE_SUCCESS)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    STOCK_TICKER_MGR_OBJECT_NAME
                    ": failed to build targets for DSRL 0x%p (%s)",
                    psDSRLDesc->hDSRL,
                    GsStockTickerMgrIntf.pacGetReturnCodeName(eReturnCode)
                        );
            break;
        }

        // Build the list based on the targets. Since some of the targets
        // might describe the same entry use the existence check
        eReturnCode = eDSRLDescBuild(psAppObj, psDSRLDesc,
                            STOCK_DSRL_BUILD_FLAG_CHECK_EXISTENCE
                                      );
        if (eReturnCode != STOCK_TICKER_RETURN_CODE_SUCCESS)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    STOCK_TICKER_MGR_OBJECT_NAME": failed to build DSRL 0x%p (%s)",
                    psDSRLDesc->hDSRL,
                    GsStockTickerMgrIntf.pacGetReturnCodeName(eReturnCode)
                        );
            break;
        }

        // Update state allowed for INITIAL/UPDATING states only
        vDSRLDescSetState(psDSRLArg->hDSRL, DSRL_STATE_READY);

    } while(FALSE);

    if (eReturnCode != STOCK_TICKER_RETURN_CODE_SUCCESS)
    {
        eDSRLListDestroy(psAppObj, psDSRLArg);
    }

    return eReturnCode;
}

/*****************************************************************************
*
*   eDSRLListModify
*
*****************************************************************************/
static STOCK_TICKER_RETURN_CODE_ENUM eDSRLListModify (
    STOCK_TICKER_APP_OBJECT_STRUCT *psAppObj,
    DSRL_ARG_STRUCT *psDSRLArg
        )
{
    STOCK_TICKER_RETURN_CODE_ENUM eReturnCode = STOCK_TICKER_RETURN_CODE_ERROR;
    STOCK_TICKER_DSRL_DESC_STRUCT *psDSRLDesc;

    printf(STOCK_TICKER_MGR_OBJECT_NAME": processing MODIFY request for DSRL 0x%p\n",
            psDSRLArg->hDSRL
                );

    do
    {
        // Check input
        if ((psAppObj == NULL) || (psDSRLArg == NULL))
        {
            eReturnCode = STOCK_TICKER_RETURN_CODE_BAD_INPUT;
            break;
        }

        // Get the DSRL descriptor
        psDSRLDesc = (STOCK_TICKER_DSRL_DESC_STRUCT*) DSRL_pvServiceData(psDSRLArg->hDSRL);
        if (psDSRLDesc == NULL)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    STOCK_TICKER_MGR_OBJECT_NAME": unknown DSRL 0x%p",
                        psDSRLArg->hDSRL
                            );
            eReturnCode = STOCK_TICKER_RETURN_CODE_NOT_KNOWN_HANDLE;
            break;
        }

        // Enforce DSRL updating if required
        if (psDSRLArg->uAction.sModify.bForceDSRLStateChange == TRUE)
        {
            // Move to ready state in any case as other services doing.
            vDSRLDescSetState(psDSRLArg->hDSRL, DSRL_STATE_UPDATING);
        }

        // Process the request based on the type
        switch (psDSRLArg->uAction.sModify.eModifyType)
        {
            case DSRL_MODIFY_OPERATION_ADD:
            {
                eReturnCode =
                        eDSRLListModifyAdd(psAppObj, psDSRLDesc, psDSRLArg);
                if (eReturnCode != STOCK_TICKER_RETURN_CODE_SUCCESS)
                {
                    SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                            STOCK_TICKER_MGR_OBJECT_NAME
                            ": failed to process ADD for DSRL 0x%p",
                            psDSRLDesc
                                );
                    break;
                }
            }
            break;
            case DSRL_MODIFY_OPERATION_REPLACE:
            {
                eReturnCode =
                        eDSRLListModifyReplace(psAppObj, psDSRLDesc, psDSRLArg);
                if (eReturnCode != STOCK_TICKER_RETURN_CODE_SUCCESS)
                {
                    SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                            STOCK_TICKER_MGR_OBJECT_NAME
                            ": failed to process REPLACE for DSRL 0x%p",
                            psDSRLDesc
                                );
                    break;
                }
            }
            break;
            case DSRL_MODIFY_OPERATION_REMOVE:
            {
                eReturnCode =
                        eDSRLListModifyRemove(psAppObj, psDSRLDesc, psDSRLArg);
                if (eReturnCode != STOCK_TICKER_RETURN_CODE_SUCCESS)
                {
                    SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                            STOCK_TICKER_MGR_OBJECT_NAME
                            ": failed to process REMOVE for DSRL 0x%p",
                            psDSRLDesc
                                );
                    break;
                }
            }
            break;
            case DSRL_MODIFY_OPERATION_REMOVEALL:
            {
                eReturnCode =
                        eDSRLListModifyRemoveAll(psAppObj, psDSRLDesc);
                if (eReturnCode != STOCK_TICKER_RETURN_CODE_SUCCESS)
                {
                    SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                            STOCK_TICKER_MGR_OBJECT_NAME
                            ": failed to process REMOVE_ALL for DSRL 0x%p",
                            psDSRLDesc
                                );
                    break;
                }
            }
            break;
            case DSRL_MODIFY_OPERATION_INVALID:
            {
                eReturnCode = STOCK_TICKER_RETURN_CODE_NOT_SUPPORTED;
            }
            break;
        }

        // Update state allowed for INITIAL/UPDATING states only
        vDSRLDescSetState(psDSRLArg->hDSRL, DSRL_STATE_READY);

    } while (FALSE);

    return eReturnCode;
}

/*****************************************************************************
*
*   eDSRLListDelete
*
*****************************************************************************/
static STOCK_TICKER_RETURN_CODE_ENUM eDSRLListDestroy (
    STOCK_TICKER_APP_OBJECT_STRUCT *psAppObj,
    DSRL_ARG_STRUCT *psDSRLArg
        )
{
    STOCK_TICKER_RETURN_CODE_ENUM eResult = STOCK_TICKER_RETURN_CODE_ERROR;
    OSAL_RETURN_CODE_ENUM eOsalReturnCode = OSAL_ERROR;
    STOCK_TICKER_DSRL_DESC_STRUCT *psDSRLDesc = NULL;

    do
    {
        // Check input
        if ((psAppObj == NULL) || (psDSRLArg == NULL))
        {
            eResult = STOCK_TICKER_RETURN_CODE_BAD_INPUT;
            break;
        }

        // Find the appropriate descriptor
        psDSRLDesc =
            (STOCK_TICKER_DSRL_DESC_STRUCT *) DSRL_pvServiceData(psDSRLArg->hDSRL);
        if (psDSRLDesc == NULL)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    STOCK_TICKER_MGR_OBJECT_NAME": unknown DSRL 0x%p",
                    psDSRLArg->hDSRL);
            eResult = STOCK_TICKER_RETURN_CODE_NOT_KNOWN_HANDLE;
            break;
        }

        // Reset finalize function
        DSRL_bSetFinalizeFunction(psDSRLArg->hDSRL, NULL, NULL);

        // XXX: In case of message removal in some moment this message
        //      might have no referenced DSRLs. So, don't worry about
        //      this here because during next update this message will
        //      be pruned any way.

        // Say to remove all entries from the DSRL
        eResult = eDSRLDescRemoveAllEntries(psDSRLDesc, FALSE);
        if (eResult != STOCK_TICKER_RETURN_CODE_SUCCESS)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    STOCK_TICKER_MGR_OBJECT_NAME
                    ": failed to remove all targets from DSRL %p (%s)",
                    psDSRLDesc,
                    GsStockTickerMgrIntf.pacGetReturnCodeName(eResult)
                        );
        }

        // Remove this from our list of tracked targets for this service
        eOsalReturnCode = OSAL.eLinkedListRemove(psDSRLDesc->hEntry);
        if (eOsalReturnCode != OSAL_SUCCESS)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    STOCK_TICKER_MGR_OBJECT_NAME
                    ": failed to remove DSRLDesc 0x%p entry from the list (%s)",
                    psDSRLDesc,
                    OSAL.pacGetReturnCodeName(eOsalReturnCode)
                        );
        }

        // Destroy this list
        vDSRLDescDestroy(psDSRLArg->hDSRL);

        // Success
        eResult = STOCK_TICKER_RETURN_CODE_SUCCESS;
    } while (FALSE);

    return eResult;
 }

/*****************************************************************************
*
*   eDSRLListRefresh
*
*****************************************************************************/
static STOCK_TICKER_RETURN_CODE_ENUM eDSRLListRefresh (
    STOCK_TICKER_APP_OBJECT_STRUCT *psAppObj,
    DSRL_ARG_STRUCT *psDSRLArg
        )
{
    STOCK_TICKER_RETURN_CODE_ENUM eReturnCode = STOCK_TICKER_RETURN_CODE_ERROR;
    STOCK_TICKER_DSRL_DESC_STRUCT *psDSRLDesc;

    do
    {
        // Check input
        if ((psAppObj == NULL) || (psDSRLArg == NULL))
        {
            eReturnCode = STOCK_TICKER_RETURN_CODE_BAD_INPUT;
            break;
        }

        // Get the appropriate descriptor
        psDSRLDesc = (STOCK_TICKER_DSRL_DESC_STRUCT*)
                        DSRL_pvServiceData(psDSRLArg->hDSRL);
        if (psDSRLDesc == NULL)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    STOCK_TICKER_MGR_OBJECT_NAME": unknown DSRL 0x%p",
                    psDSRLArg->hDSRL);
            eReturnCode = STOCK_TICKER_RETURN_CODE_NOT_KNOWN_HANDLE;
            break;
        }

        eReturnCode = eDSRLDescBuild(psAppObj, psDSRLDesc,
                                   STOCK_DSRL_BUILD_FLAG_CHECK_EXISTENCE
                                        );
        if (eReturnCode != STOCK_TICKER_RETURN_CODE_SUCCESS)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    STOCK_TICKER_MGR_OBJECT_NAME
                    ": failed to build DSRL %p",
                    psDSRLDesc->hDSRL
                        );
            break;
        }

        // Update state allowed for INITIAL/UPDATING states only
        vDSRLDescSetState(psDSRLArg->hDSRL, DSRL_STATE_READY);

        // We're OK
        eReturnCode = STOCK_TICKER_RETURN_CODE_SUCCESS;
    } while (FALSE);

    return eReturnCode;
}

/*****************************************************************************
*
*   bDSRLTargetUpdateIndexFromDBProcessResult
*
*****************************************************************************/
static BOOLEAN bDSRLTargetUpdateIndexFromDBProcessResult (
    SQL_QUERY_COLUMN_STRUCT *psColumns,
    N32 n32NumberOfColumns,
    void *pvArg
        )
{
    STOCK_TICKER_DB_QUERY_RESULT_STRUCT *psArg =
            (STOCK_TICKER_DB_QUERY_RESULT_STRUCT *)pvArg;

    if ((n32NumberOfColumns != STOCK_TICKER_SELECT_DB_SYMBOL_BY_NAME_COLUMN_SIZE) ||
        (psArg->sOutput.bResultantRows == TRUE))
    {
        return FALSE;
    }

    // Check the current state of the output
    if (psArg->sOutput.uDbRow.sSymbol.tIndex != STOCK_INVALID_ID)
    {
        SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
            STOCK_TICKER_MGR_OBJECT_NAME": index already provided.");
        return FALSE;
    }

    psArg->sOutput.uDbRow.sSymbol.tIndex = (STOCK_ID)psColumns->uData.sUN32.un32Data;

    // The only one field is requested and it should have the stock symbol name
    psArg->sOutput.bResultantRows = TRUE;

    return FALSE;
}

/*****************************************************************************
*
*   eDSRLTargetUpdateIndexFromDB
*
*****************************************************************************/
static STOCK_TICKER_RETURN_CODE_ENUM eDSRLTargetUpdateIndexFromDB(
    STOCK_TICKER_APP_OBJECT_STRUCT *psAppObj,
    STOCK_TICKER_DSRL_TARGET_DESC_STRUCT *psTarget
        )
{
    STOCK_TICKER_RETURN_CODE_ENUM eReturnCode = STOCK_TICKER_RETURN_CODE_ERROR;
    BOOLEAN bOk = FALSE;
    STOCK_TICKER_DB_QUERY_RESULT_STRUCT sArg;
    STRING_OBJECT hSymbolName = STRING_INVALID_OBJECT;
    const char *pacSymbolName = NULL;

    do
    {
        // Check input
        if ((psAppObj == NULL) || (psTarget == NULL))
        {
            eReturnCode = STOCK_TICKER_RETURN_CODE_BAD_INPUT;
            break;
        }

        // Extract Symbol name as C-string
        hSymbolName = STOCK_SYMBOL_hName(psTarget->hTarget);
        if (hSymbolName == STRING_INVALID_OBJECT)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                STOCK_TICKER_MGR_OBJECT_NAME": failed to get name from STOCK_SYMBOL 0x%p",
                psTarget->hTarget
                    );
            break;
        }

        // Extract C-string
        pacSymbolName = STRING.pacCStr(hSymbolName);

        // Prepare query
        snprintf(psAppObj->acBuffer, STOCK_TICKER_SHARED_BUFFER_LEN,
                    STOCK_TICKER_SELECT_DB_SYMBOL_BY_NAME, pacSymbolName
                        );

        // Init argument
        OSAL.bMemSet(&sArg, 0, sizeof(sArg));
        sArg.sOutput.bResultantRows = FALSE;
        sArg.sOutput.uDbRow.sSymbol.tIndex = STOCK_INVALID_ID;

        // Make query to gather the index of the Stock Symbol from DB
        bOk = SQL_INTERFACE.bQuery(
                        psAppObj->sDB.hSQLConnection,
                        psAppObj->acBuffer,
                        bDSRLTargetUpdateIndexFromDBProcessResult,
                        (void*) &sArg
                            );
        if (bOk == FALSE)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                STOCK_TICKER_MGR_OBJECT_NAME": failed to execute query %s",
                psAppObj->acBuffer
                    );
            eReturnCode = STOCK_TICKER_RETURN_CODE_DB_ERROR;
            break;
        }

        // Process Query results
        if (sArg.sOutput.bResultantRows == TRUE)
        {
            psTarget->tStockId = sArg.sOutput.uDbRow.sSymbol.tIndex;
            eReturnCode = STOCK_TICKER_RETURN_CODE_SUCCESS;
        }
        else
        {
            psTarget->tStockId = STOCK_INVALID_ID;
            eReturnCode = STOCK_TICKER_RETURN_CODE_FALSE;
        }

    } while (FALSE);

    return eReturnCode;
}

/*****************************************************************************
*
*   eDSRLTargetMonitorInit
*
*****************************************************************************/
static STOCK_TICKER_RETURN_CODE_ENUM eDSRLTargetMonitorInit(
    STOCK_TICKER_APP_OBJECT_STRUCT *psAppObj
        )
{
    STOCK_TICKER_RETURN_CODE_ENUM eReturnCore = STOCK_TICKER_RETURN_CODE_ERROR;
    OSAL_RETURN_CODE_ENUM eOsalReturnCode = OSAL_ERROR;

    printf(STOCK_TICKER_MGR_OBJECT_NAME": Initializing DSRL Targets Monitor\n");

    do
    {
        // Check input
        if (psAppObj == NULL)
        {
            eReturnCore = STOCK_TICKER_RETURN_CODE_BAD_INPUT;
            break;
        }

        // Create list
        eOsalReturnCode = OSAL.eLinkedListCreate(&psAppObj->hTargetsMonitorList,
                                    STOCK_TICKER_MGR_OBJECT_NAME":TrgMntrList",
                                    NULL,
                                    OSAL_LL_OPTION_LINEAR | OSAL_LL_OPTION_UNIQUE);
        if (eOsalReturnCode != OSAL_SUCCESS)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                STOCK_TICKER_MGR_OBJECT_NAME": failed to create list (%s)",
                OSAL.pacGetReturnCodeName(eOsalReturnCode)
                    );
            break;
        }

        eReturnCore = STOCK_TICKER_RETURN_CODE_SUCCESS;
    } while (FALSE);

    return eReturnCore;
}

/*****************************************************************************
*
*   vDSRLTargetMonitorUnInitIterator
*
*****************************************************************************/
static void vDSRLTargetMonitorUnInitIterator(
    STOCK_TICKER_DSRL_TARGET_DESC_STRUCT *psObj
        )
{
    psObj->hMonitorListEntry = OSAL_INVALID_LINKED_LIST_ENTRY;

    return;
}

/*****************************************************************************
*
*   eDSRLTargetMonitorUnInit
*
*****************************************************************************/
static STOCK_TICKER_RETURN_CODE_ENUM eDSRLTargetMonitorUnInit(
    STOCK_TICKER_APP_OBJECT_STRUCT *psAppObj
        )
{
    STOCK_TICKER_RETURN_CODE_ENUM eReturnCore = STOCK_TICKER_RETURN_CODE_ERROR;
    OSAL_RETURN_CODE_ENUM eOsalReturnCode = OSAL_ERROR;

    printf(STOCK_TICKER_MGR_OBJECT_NAME": Uninitializing DSRL Targets Monitor\n");

    do
    {
        // Check input
        if (psAppObj == NULL)
        {
            eReturnCore = STOCK_TICKER_RETURN_CODE_BAD_INPUT;
            break;
        }

        if (psAppObj->hTargetsMonitorList != OSAL_INVALID_OBJECT_HDL)
        {
            // Clean up the list
            eOsalReturnCode = OSAL.eLinkedListRemoveAll(
                                    psAppObj->hTargetsMonitorList,
                                    (OSAL_LL_RELEASE_HANDLER)vDSRLTargetMonitorUnInitIterator
                                        );
            if (eOsalReturnCode != OSAL_SUCCESS)
            {
                SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    STOCK_TICKER_MGR_OBJECT_NAME": failed to remve all items from list (%s)",
                    OSAL.pacGetReturnCodeName(eOsalReturnCode)
                        );
            }

            // Destroy the list
            eOsalReturnCode = OSAL.eLinkedListDelete(psAppObj->hTargetsMonitorList);
            if (eOsalReturnCode != OSAL_SUCCESS)
            {
                SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    STOCK_TICKER_MGR_OBJECT_NAME": failed to delete list (%s)",
                    OSAL.pacGetReturnCodeName(eOsalReturnCode)
                        );
            }

            psAppObj->hTargetsMonitorList = OSAL_INVALID_OBJECT_HDL;
        }

        eReturnCore = STOCK_TICKER_RETURN_CODE_SUCCESS;
    } while (FALSE);

    return eReturnCore;
}

/*****************************************************************************
*
*   eDSRLTargetMonitorRegister
*
*****************************************************************************/
static STOCK_TICKER_RETURN_CODE_ENUM eDSRLTargetMonitorRegister(
    STOCK_TICKER_APP_OBJECT_STRUCT *psAppObj,
    STOCK_TICKER_DSRL_TARGET_DESC_STRUCT *psTarget
        )
{
    STOCK_TICKER_RETURN_CODE_ENUM eReturnCode = STOCK_TICKER_RETURN_CODE_ERROR;

    do
    {
        // Check input
        if ((psAppObj == NULL) || (psTarget == NULL))
        {
            eReturnCode = STOCK_TICKER_RETURN_CODE_BAD_INPUT;
            break;
        }

        if (psTarget->hMonitorListEntry == OSAL_INVALID_LINKED_LIST_ENTRY)
        {
            OSAL_RETURN_CODE_ENUM eOsalReturnCode;

            printf(STOCK_TICKER_MGR_OBJECT_NAME": start monitoring target %p for %s\n",
                psTarget, STRING.pacCStr(STOCK_SYMBOL_hName(psTarget->hTarget))
                    );

            eOsalReturnCode = OSAL.eLinkedListAdd(psAppObj->hTargetsMonitorList,
                                        &psTarget->hMonitorListEntry, (void*) psTarget
                                            );
            if (eOsalReturnCode != OSAL_SUCCESS)
            {
                SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    STOCK_TICKER_MGR_OBJECT_NAME": failed to reg target 0x%p (%s)",
                    psTarget, OSAL.pacGetReturnCodeName(eOsalReturnCode)
                        );
                break;
            }
        }

        eReturnCode = STOCK_TICKER_RETURN_CODE_SUCCESS;
    } while (FALSE);

    return eReturnCode;
}

/*****************************************************************************
*
*   bDSRLTargetMonitorProcessIterator
*
*****************************************************************************/
static BOOLEAN bDSRLTargetMonitorProcessIterator (
    STOCK_TICKER_DSRL_TARGET_DESC_STRUCT *psTargetDesc,
    STOCK_TICKER_TARGET_MONITOR_LIST_ITERATOR_STRUCT *psArg
        )
{
    BOOLEAN bContinue = TRUE;
    N16 n16CompareResult;
    STRING_OBJECT hStockSymbolName;

    // Get symbol name from the target
    hStockSymbolName = STOCK_SYMBOL_hName(psTargetDesc->hTarget);

    // Compare names
    n16CompareResult = STRING.n16Compare(hStockSymbolName,
                                psArg->hSymbolNameFromMessage, TRUE);
    if (n16CompareResult == 0)
    {
        // Good, the message has index for the target. Extract it.
        psTargetDesc->tStockId = STOCK_MSG_tIndex(psArg->psMsgDesc->hMessage);

        printf(STOCK_TICKER_MGR_OBJECT_NAME": found index #%d for symbol %s\n",
                    psTargetDesc->tStockId,
                    STRING.pacCStr(STOCK_SYMBOL_hName(psTargetDesc->hTarget))
                        );

        // Stop monitoring this target
        psArg->eReturnCode = eDSRLTargetMonitorUnregister(psTargetDesc);
        if (psArg->eReturnCode != STOCK_TICKER_RETURN_CODE_SUCCESS)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                STOCK_TICKER_MGR_OBJECT_NAME": failed to unreg target %s (%s)",
                STRING.pacCStr(hStockSymbolName),
                GsStockTickerMgrIntf.pacGetReturnCodeName(psArg->eReturnCode)
                    );
            bContinue = FALSE;
        }
    }

    return bContinue;
}

/*****************************************************************************
*
*   eDSRLTargetMonitorProcess
*
*****************************************************************************/
static STOCK_TICKER_RETURN_CODE_ENUM eDSRLTargetMonitorProcess(
    STOCK_TICKER_APP_OBJECT_STRUCT *psAppObj,
    STOCK_MSG_DESC_STRUCT *psMsgDesc
        )
{
    STOCK_TICKER_TARGET_MONITOR_LIST_ITERATOR_STRUCT sArg;

    // Check input
    if ((psAppObj == NULL) || (psMsgDesc == NULL))
    {
        sArg.eReturnCode = STOCK_TICKER_RETURN_CODE_BAD_INPUT;
    }
    else
    {
        OSAL_RETURN_CODE_ENUM eOsalReturnCode;

        // Init iterator's argument
        sArg.hSymbolNameFromMessage = STOCK_MSG.hSymbolName(psMsgDesc->hMessage);
        sArg.psMsgDesc = psMsgDesc;
        sArg.eReturnCode = STOCK_TICKER_RETURN_CODE_SUCCESS;

        // Look through all monitoring targets to find out their ids from
        // passed message
        eOsalReturnCode = OSAL.eLinkedListIterate(psAppObj->hTargetsMonitorList,
                            (OSAL_LL_ITERATOR_HANDLER) bDSRLTargetMonitorProcessIterator,
                            (void*) &sArg);
        if (eOsalReturnCode != OSAL_SUCCESS)
        {
            if (eOsalReturnCode == OSAL_NO_OBJECTS)
            {
                puts(STOCK_TICKER_MGR_OBJECT_NAME": no targets under monitor. Skip...");
                sArg.eReturnCode = STOCK_TICKER_RETURN_CODE_NOTHING_TO_UPDATE;
            }
            else
            {
                SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    STOCK_TICKER_MGR_OBJECT_NAME": failed to iterate monitoring targets (%s)",
                    OSAL.pacGetReturnCodeName(eOsalReturnCode));
                sArg.eReturnCode = STOCK_TICKER_RETURN_CODE_ERROR;
            }
        }
        else if (sArg.eReturnCode != STOCK_TICKER_RETURN_CODE_SUCCESS)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                STOCK_TICKER_MGR_OBJECT_NAME": problem found during iteration of monitoring targets (%s)",
                GsStockTickerMgrIntf.pacGetReturnCodeName(sArg.eReturnCode)
                    );
        }
    } while (FALSE);

    return sArg.eReturnCode;
}

/*****************************************************************************
*
*   eDSRLTargetMonitorUnregister
*
*****************************************************************************/
static STOCK_TICKER_RETURN_CODE_ENUM eDSRLTargetMonitorUnregister(
    STOCK_TICKER_DSRL_TARGET_DESC_STRUCT *psTarget
        )
{
    STOCK_TICKER_RETURN_CODE_ENUM eReturnCode = STOCK_TICKER_RETURN_CODE_BAD_INPUT;

    do
    {
        // Check input
        if (psTarget == NULL)
        {
            break;
        }

        if (psTarget->hMonitorListEntry != OSAL_INVALID_LINKED_LIST_ENTRY)
        {
            OSAL_RETURN_CODE_ENUM eOsalReturnCode;

            printf(STOCK_TICKER_MGR_OBJECT_NAME": stop monitoring target %p for %s\n",
                psTarget, STRING.pacCStr(STOCK_SYMBOL_hName(psTarget->hTarget))
                    );

            eOsalReturnCode = OSAL.eLinkedListRemove(psTarget->hMonitorListEntry);
            if (eOsalReturnCode != OSAL_SUCCESS)
            {
                SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    STOCK_TICKER_MGR_OBJECT_NAME": failed to unreg target 0x%p (%s)",
                    psTarget, OSAL.pacGetReturnCodeName(eOsalReturnCode)
                        );
            }

            psTarget->hMonitorListEntry = OSAL_INVALID_LINKED_LIST_ENTRY;
        }

        eReturnCode = STOCK_TICKER_RETURN_CODE_SUCCESS;
    } while (FALSE);

    return eReturnCode;
}

/*****************************************************************************
*
*   psDSRLTargetCreate
*
*****************************************************************************/
static STOCK_TICKER_DSRL_TARGET_DESC_STRUCT *psDSRLTargetCreate (
    STOCK_TICKER_APP_OBJECT_STRUCT *psAppObj,
    STOCK_TICKER_DSRL_DESC_STRUCT *psDSRLDesc,
    STOCK_SYMBOL_OBJECT *phTarget
        )
{
    STOCK_TICKER_DSRL_TARGET_DESC_STRUCT *psTargetDesc = NULL;
    STOCK_TICKER_RETURN_CODE_ENUM eReturnCode = STOCK_TICKER_RETURN_CODE_ERROR;

    do
    {
        // Check input
        if ((psAppObj == NULL) || (phTarget == NULL) ||
            (psDSRLDesc == NULL) || (*phTarget == STOCK_SYMBOL_INVALID_OBJECT))
        {
            break;
        }

#if SMS_DEBUG==1
        // Prepare name if needed
        snprintf(psAppObj->acBuffer, sizeof(psAppObj->acBuffer),
            STOCK_TICKER_MGR_OBJECT_NAME":TD:%s",
            STRING.pacCStr(STOCK_SYMBOL_hName(*phTarget)));
#endif

        // Create the target descriptor
        psTargetDesc = (STOCK_TICKER_DSRL_TARGET_DESC_STRUCT*)
            OSAL.pvLinkedListMemoryAllocate(
#if SMS_DEBUG == 1
                psAppObj->acBuffer,
#else
                STOCK_TICKER_MGR_OBJECT_NAME":TargetDesc",
#endif
                sizeof(STOCK_TICKER_DSRL_TARGET_DESC_STRUCT),
                FALSE);
        if (psTargetDesc == NULL)
        {
            // Error!
            break;
        }

        // Init fields
        psTargetDesc->hTarget = *phTarget;
        psTargetDesc->hEntry = OSAL_INVALID_LINKED_LIST_ENTRY;
        psTargetDesc->hMonitorListEntry = OSAL_INVALID_LINKED_LIST_ENTRY;
        psTargetDesc->tStockId = STOCK_INVALID_ID;

        // Say that the target desc is taking care about the target object
        *phTarget = STOCK_SYMBOL_INVALID_OBJECT;

        // Try to find out the index for the name through the DB
        eReturnCode = eDSRLTargetUpdateIndexFromDB(psAppObj, psTargetDesc);
        if (eReturnCode == STOCK_TICKER_RETURN_CODE_FALSE)
        {
            // Well, we need to monitor the target to find out the index by the name
            eReturnCode = eDSRLTargetMonitorRegister(psAppObj, psTargetDesc);
            if (eReturnCode != STOCK_TICKER_RETURN_CODE_SUCCESS)
            {
                SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    STOCK_TICKER_MGR_OBJECT_NAME": failed to register target in target monitor (%s)",
                    GsStockTickerMgrIntf.pacGetReturnCodeName(eReturnCode)
                        );
                break;
            }
        }
#if SMS_DEBUG==1
        else if (eReturnCode != STOCK_TICKER_RETURN_CODE_SUCCESS)
        {
            STRING_OBJECT hSymbolName = STOCK_SYMBOL_hName(*phTarget);
            const char *pacSymbolName = STRING.pacCStr(hSymbolName);

            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                STOCK_TICKER_MGR_OBJECT_NAME": failed to request index for %s from DB (%s)",
                pacSymbolName,
                GsStockTickerMgrIntf.pacGetReturnCodeName(eReturnCode)
                    );
            break;
        }
#endif
    } while (FALSE);

    if ((eReturnCode != STOCK_TICKER_RETURN_CODE_SUCCESS) &&
        (psTargetDesc != NULL))
    {
        // Return back the referenced object just to be fair in that case
        *phTarget = psTargetDesc->hTarget;
        psTargetDesc->hTarget = STOCK_SYMBOL_INVALID_OBJECT;

        // Do release
        vDSRLTargetRelease(psTargetDesc);
        psTargetDesc = NULL;
    }

    // Populate the result
    return psTargetDesc;
}

/*****************************************************************************
*
*   eDSRLTargetBuildAll
*
*****************************************************************************/
static STOCK_TICKER_RETURN_CODE_ENUM eDSRLTargetBuildAll (
    STOCK_TICKER_APP_OBJECT_STRUCT *psAppObj,
    STOCK_TICKER_DSRL_DESC_STRUCT *psDSRLDesc,
    DSRL_TARGET_OBJECT ahTargetList[],
    size_t tNumTargets
        )
{
    STOCK_TICKER_RETURN_CODE_ENUM eReturnCode =
            STOCK_TICKER_RETURN_CODE_SUCCESS;
    STOCK_TICKER_DSRL_TARGET_DESC_STRUCT *psCurTargetDesc = NULL;
    OSAL_RETURN_CODE_ENUM eOsalReturnCode;
    size_t tIndex;

    printf(STOCK_TICKER_MGR_OBJECT_NAME
            ": building all targets for DSRL 0x%p (%u item(s))\n",
            psDSRLDesc->hDSRL, tNumTargets
                );

    // Iterate through our target list
    for (tIndex = 0; tIndex < tNumTargets; ++tIndex)
    {
        DSRL_TARGET_TYPE_ENUM eTargetType;
        STOCK_SYMBOL_OBJECT *phTarget = (STOCK_SYMBOL_OBJECT*)&ahTargetList[tIndex];

        // Check target type
        eTargetType = DSRL_TARGET.eType(*phTarget);
        if (eTargetType != DSRL_TARGET_TYPE_STOCK_SYMBOL)
        {
            // Continue with next target;
            continue;
        }

        // Try to find out does the target already reflected
        // in the list
        psCurTargetDesc = psDSRLTargetGet(psDSRLDesc, *phTarget);
        if (psCurTargetDesc != NULL)
        {
            // Well, we have such target. Let's look at the next one.
            continue;
        }

        psCurTargetDesc = psDSRLTargetCreate(psAppObj, psDSRLDesc, phTarget);
        if (psCurTargetDesc == NULL)
        {
            eReturnCode = STOCK_TICKER_RETURN_CODE_MEMORY_ERROR;
            break;
        }

        // Add this to our list of tracked targets for this service
        eOsalReturnCode = OSAL.eLinkedListAdd(
                            psDSRLDesc->hTargetList,
                            &psCurTargetDesc->hEntry,
                            (void*) psCurTargetDesc
                                );
        if (eOsalReturnCode != OSAL_SUCCESS)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    STOCK_TICKER_MGR_OBJECT_NAME": failed to add target to the list (%s)",
                    OSAL.pacGetReturnCodeName(eOsalReturnCode)
                        );
            eReturnCode = STOCK_TICKER_RETURN_CODE_ERROR;
            break;
        }
    }

    return eReturnCode;
}

/*****************************************************************************
*
*   n16DSRLTargetGetSearcher
*
*****************************************************************************/
static N16 n16DSRLTargetGetSearcher(
    STOCK_TICKER_DSRL_TARGET_DESC_STRUCT *psTarget,
    STOCK_SYMBOL_OBJECT hTarget
        )
{
    N16 n16Result;

    n16Result = STOCK_SYMBOL_n16Compare(psTarget->hTarget, hTarget);

    return n16Result;
}

/*****************************************************************************
*
*   psDSRLTargetGet
*
*****************************************************************************/
static STOCK_TICKER_DSRL_TARGET_DESC_STRUCT *psDSRLTargetGet (
    STOCK_TICKER_DSRL_DESC_STRUCT *psDSRLDesc,
    STOCK_SYMBOL_OBJECT hTarget
        )
{
    STOCK_TICKER_DSRL_TARGET_DESC_STRUCT *psResult = NULL;
    OSAL_RETURN_CODE_ENUM eOsalReturnCode = OSAL_ERROR;
    OSAL_LINKED_LIST_ENTRY hEntry = OSAL_INVALID_LINKED_LIST_ENTRY;

    do
    {
        // Check input
        if ((psDSRLDesc == NULL) || (hTarget == STOCK_SYMBOL_INVALID_OBJECT))
        {
            break;
        }

        // Looking for the target descriptor by its handle
        eOsalReturnCode = OSAL.eLinkedListLinearSearch(
                                psDSRLDesc->hTargetList,
                                &hEntry,
                                (OSAL_LL_COMPARE_HANDLER) n16DSRLTargetGetSearcher,
                                (void*) hTarget);
        if (eOsalReturnCode == OSAL_SUCCESS)
        {
            psResult =
                    (STOCK_TICKER_DSRL_TARGET_DESC_STRUCT*)
                            OSAL.pvLinkedListThis(hEntry);
        }
        else if ((eOsalReturnCode != OSAL_NO_OBJECTS) &&
                 (eOsalReturnCode != OSAL_OBJECT_NOT_FOUND))
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    STOCK_TICKER_MGR_OBJECT_NAME
                    ": failed to search target 0x%p in DSRL 0x%p (%s)",
                    hTarget, psDSRLDesc,
                    OSAL.pacGetReturnCodeName(eOsalReturnCode)
                        );
            break;
        }
    } while (FALSE);

    return psResult;
}

/*****************************************************************************
*
*   eDSRLTargetRemoveAll
*
*   Removes all current targets, but w/o removing corresponded entries
*   is their exist. To refresh the DSRL content need to build it.
*
*****************************************************************************/
static STOCK_TICKER_RETURN_CODE_ENUM eDSRLTargetRemoveAll (
    STOCK_TICKER_DSRL_DESC_STRUCT *psDSRLDesc
        )
{
    STOCK_TICKER_RETURN_CODE_ENUM eReturnCode = STOCK_TICKER_RETURN_CODE_ERROR;
    OSAL_RETURN_CODE_ENUM eOsalReturnCode = OSAL_ERROR;

    printf(STOCK_TICKER_MGR_OBJECT_NAME
            ": removing all targets for DSRL 0x%p\n",
            psDSRLDesc);

    do
    {
        // Check input
        if (psDSRLDesc == NULL)
        {
            eReturnCode = STOCK_TICKER_RETURN_CODE_BAD_INPUT;
            break;
        }

        // Remove all targets
        eOsalReturnCode = OSAL.eLinkedListRemoveAll(
                                    psDSRLDesc->hTargetList,
                                    (OSAL_LL_RELEASE_HANDLER)vDSRLTargetRelease
                                        );
        if (eOsalReturnCode != OSAL_SUCCESS)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    STOCK_TICKER_MGR_OBJECT_NAME
                    ": failed to remove targets for DSRL 0x%p (%s)",
                    psDSRLDesc,
                    OSAL.pacGetReturnCodeName(eOsalReturnCode)
                        );
            break;
        }

        eReturnCode = STOCK_TICKER_RETURN_CODE_SUCCESS;
    } while (FALSE);

    return eReturnCode;
}

/*****************************************************************************
*
*   eDSRLTargetRemove
*
*****************************************************************************/
static STOCK_TICKER_RETURN_CODE_ENUM eDSRLTargetRemove (
    STOCK_TICKER_DSRL_DESC_STRUCT *psDSRLDesc,
    STOCK_TICKER_DSRL_TARGET_DESC_STRUCT *psTarget
        )
{
    STOCK_TICKER_RETURN_CODE_ENUM eReturnCode = STOCK_TICKER_RETURN_CODE_ERROR;
    OSAL_RETURN_CODE_ENUM eOsalReturnCode = OSAL_ERROR;

    do
    {
        // Check input
        if ((psTarget == NULL) || (psDSRLDesc == NULL))
        {
            eReturnCode = STOCK_TICKER_RETURN_CODE_BAD_INPUT;
            break;
        }

        eOsalReturnCode = OSAL.eLinkedListRemove(psTarget->hEntry);
        if (eOsalReturnCode != OSAL_SUCCESS)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    STOCK_TICKER_MGR_OBJECT_NAME
                    ": failed to remove target 0x%p (%s)",
                    psTarget,
                    OSAL.pacGetReturnCodeName(eOsalReturnCode)
                        );
            break;
        }

        // Release target resources
        vDSRLTargetRelease(psTarget);

        eReturnCode = STOCK_TICKER_RETURN_CODE_SUCCESS;
    } while (FALSE);

    return eReturnCode;
}

/*****************************************************************************
*
*   vDSRLTargetRelease
*
*****************************************************************************/
static void vDSRLTargetRelease (
    STOCK_TICKER_DSRL_TARGET_DESC_STRUCT *psTargetDesc
        )
{
    printf(STOCK_TICKER_MGR_OBJECT_NAME
            ": attempting to destroy target desc %p\n",
            psTargetDesc);

    if (psTargetDesc != NULL)
    {
        if (psTargetDesc->hTarget != STOCK_SYMBOL_INVALID_OBJECT)
        {
            STOCK_SYMBOL.vDestroy(psTargetDesc->hTarget);
            psTargetDesc->hTarget = STOCK_SYMBOL_INVALID_OBJECT;
        }

        psTargetDesc->hEntry = OSAL_INVALID_LINKED_LIST_ENTRY;

        eDSRLTargetMonitorUnregister(psTargetDesc);

        OSAL.vLinkedListMemoryFree((void*) psTargetDesc);
    }

    return;
}

/*****************************************************************************
*
*   vEventHandler
*
*   This function runs in the context of an SMS resource which has been
*   assigned to this service.
*
*****************************************************************************/
static void vEventHandler (
    DATASERVICE_MGR_OBJECT hDataService,
    DATASERVICE_EVENT_MASK tCurrentEvent,
    void *pvEventArg,
    void *pvEventCallbackArg
        )
{
    STOCK_TICKER_MGR_OBJECT_STRUCT *psObj;
    BOOLEAN bValid, bStopEvent = FALSE;
    SMSAPI_EVENT_MASK tEventMask = DATASERVICE_EVENT_NONE;

    // Get our Stocks handle from the callback argument
    psObj = (STOCK_TICKER_MGR_OBJECT_STRUCT *)pvEventCallbackArg;

    // Is this object valid?
    bValid = DATASERVICE_IMPL_bValid((DATASERVICE_IMPL_HDL)psObj);

    // Only handle events for valid objects...
    if (bValid == TRUE)
    {
        switch( tCurrentEvent )
        {
            // Handle Stocks Service events here...

            // State has changed
            case DATASERVICE_EVENT_STATE:
            {
                BOOLEAN bStateChanged;
                DATASERVICE_STATE_CHANGE_STRUCT const *psStateChange =
                    (DATASERVICE_STATE_CHANGE_STRUCT const *)pvEventArg;

                printf(STOCK_TICKER_MGR_OBJECT_NAME": event %s\n",
                    MACRO_TO_STRING(DATASERVICE_EVENT_STATE)
                        );

                // Process the state transition
                bStateChanged = DATASERVICE_IMPL_bStateFSM(
                    (DATASERVICE_IMPL_HDL)psObj,
                    psStateChange,
                    &GsStocksStateHandlers,
                    (void *)psObj);

                if (bStateChanged == TRUE)
                {
                    // The state has been updated
                    tEventMask |= DATASERVICE_EVENT_STATE;

                    // Is the service stopped now?
                    if (psStateChange->eCurrentState ==
                            DATASERVICE_STATE_STOPPED)
                    {
                        bStopEvent = TRUE;
                    }
                }
            }
            break;

            // This service has a tile to process
            case DATASERVICE_EVENT_NEW_DATA:
            {
                OSAL_BUFFER_HDL hPayload = (OSAL_BUFFER_HDL)pvEventArg;
                STOCK_TICKER_RETURN_CODE_ENUM eProcessResult =
                        STOCK_TICKER_RETURN_CODE_ERROR;

                printf(STOCK_TICKER_MGR_OBJECT_NAME": event %s\n",
                    MACRO_TO_STRING(DATASERVICE_EVENT_NEW_DATA)
                        );

                // Ensure the payload handle is valid
                // If it isn't, there's a problem with
                // SMS
                if (hPayload == OSAL_INVALID_BUFFER_HDL)
                {
                    eProcessResult = STOCK_TICKER_RETURN_CODE_ERROR;
                }
                else
                {
                    printf(STOCK_TICKER_MGR_OBJECT_NAME": Payload Received (%u)\n",
                        OSAL.tBufferGetSize(hPayload)
                            );

                    eProcessResult = eProcessPayload(psObj, hPayload);

                    printf(STOCK_TICKER_MGR_OBJECT_NAME
                                ": Message Payload Processed is %s\n\n",
                                GsStockTickerMgrIntf.pacGetReturnCodeName(eProcessResult)
                                    );
                }

                if ((eProcessResult != STOCK_TICKER_RETURN_CODE_SUCCESS) &&
                    (eProcessResult != STOCK_TICKER_RETURN_CODE_BAD_STATE))
                {
                    // If an error occurred, indicate a state change
                    // to the application
                    tEventMask |= DATASERVICE_EVENT_STATE;

                    DATASERVICE_IMPL_vLog(STOCK_TICKER_MGR_OBJECT_NAME
                        ":failed to process payload (%d)\n", eProcessResult
                            );

                    // Indicate an error
                    vSetError(psObj, DATASERVICE_ERROR_CODE_GENERAL);
                }
            }
            break;

            // We need to do some work with our DSRLs
            case DATASERVICE_INTERNAL_EVENT_DSRL:
            {
                DSRL_ARG_STRUCT *psDSRLArg =
                    (DSRL_ARG_STRUCT *)pvEventArg;
                STOCK_TICKER_APP_OBJECT_STRUCT *psAppObj = NULL;
                STOCK_TICKER_RETURN_CODE_ENUM eReturnCode =
                        STOCK_TICKER_RETURN_CODE_SUCCESS;
                BOOLEAN bCleanUpHash = FALSE;

                printf(STOCK_TICKER_MGR_OBJECT_NAME": event %s\n",
                    MACRO_TO_STRING(DATASERVICE_INTERNAL_EVENT_DSRL)
                        );

                do
                {
                    psAppObj = psGetAppFacingObject((AGW_SERVICE_OBJECT)psObj);
                    if (psAppObj == NULL)
                    {
                        eReturnCode = STOCK_TICKER_RETURN_CODE_NOT_OWNER;
                        break;
                    }

                    switch (psDSRLArg->eAction)
                    {
                        case DSRL_ACTION_ADD:
                        {
                            puts(STOCK_TICKER_MGR_OBJECT_NAME": DSRL List creating");
                            eReturnCode = eDSRLListCreate(psObj->psAppObj, psDSRLArg);
                            printf(STOCK_TICKER_MGR_OBJECT_NAME": DSRL List creation is %s\n",
                                    GsStockTickerMgrIntf.pacGetReturnCodeName(eReturnCode)
                                        );
                            bCleanUpHash = TRUE;
                        }
                        break;
                        case DSRL_ACTION_MODIFY:
                        {
                            puts(STOCK_TICKER_MGR_OBJECT_NAME": DSRL List modifying");
                            eReturnCode = eDSRLListModify(psObj->psAppObj, psDSRLArg);
                            printf(STOCK_TICKER_MGR_OBJECT_NAME": DSRL List modification is %s\n",
                                    GsStockTickerMgrIntf.pacGetReturnCodeName(eReturnCode)
                                        );
                            bCleanUpHash = TRUE;
                        }
                        break;
                        case DSRL_ACTION_REMOVE:
                        {
                            puts(STOCK_TICKER_MGR_OBJECT_NAME": DSRL List removing");
                            eReturnCode = eDSRLListDestroy(psObj->psAppObj, psDSRLArg);
                            printf(STOCK_TICKER_MGR_OBJECT_NAME": DSRL List removal is %s\n",
                                    GsStockTickerMgrIntf.pacGetReturnCodeName(eReturnCode)
                                        );
                            bCleanUpHash = FALSE;
                        }
                        break;
                        case DSRL_ACTION_REFRESH:
                        {
                            puts(STOCK_TICKER_MGR_OBJECT_NAME": DSRL List refreshing");
                            eReturnCode = eDSRLListRefresh(psObj->psAppObj, psDSRLArg);
                            printf(STOCK_TICKER_MGR_OBJECT_NAME": DSRL List refresh is %s\n",
                                    GsStockTickerMgrIntf.pacGetReturnCodeName(eReturnCode)
                                        );
                            bCleanUpHash = FALSE;
                        }
                        break;
                        case DSRL_ACTION_INVALID:
                        default:
                        {
                            // Nothing to do
                            bCleanUpHash = FALSE;
                        }
                        break;
                    }

                    // Let's start or stop the timer
                    vExpirationTimerUpdate(psAppObj, STOCK_TICKER_TIMER_PERIOD);

                } while (FALSE);

                // Go to the error state if the create/modify failed
                // or if we aren't ready
                if (eReturnCode != STOCK_TICKER_RETURN_CODE_SUCCESS)
                {
                    BOOLEAN bDSRLLocked = FALSE;

                    DATASERVICE_IMPL_vLog(STOCK_TICKER_MGR_OBJECT_NAME
                        ": failed to process DSRL event (%d)\n", eReturnCode
                            );

                    if (psAppObj == NULL)
                    {
                        // We need to lock DSRL since we cannot set state w/o
                        // this and DSRL mightn't be locked via AppFacing objects
                        bDSRLLocked = SMSO_bLock((SMS_OBJECT) psDSRLArg->hDSRL, OSAL_OBJ_TIMEOUT_INFINITE);
                    }

                    if ((bDSRLLocked == TRUE) || (psAppObj != NULL))
                    {
                        DSRL_vSetState(psDSRLArg->hDSRL, DSRL_STATE_ERROR);

                        if (bDSRLLocked == TRUE)
                        {
                            // Unlock DSRL if it has been locked directly
                            SMSO_vUnlock((SMS_OBJECT) psDSRLArg->hDSRL);
                        }
                    }
                    else
                    {
                        SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                            STOCK_TICKER_MGR_OBJECT_NAME": failed to lock DSRL to update state");
                    }

                    // If an error occurred, indicate a state change
                    // to the application
                    tEventMask |= DATASERVICE_EVENT_STATE;

                    // Indicate an error
                    vSetError(psObj, DATASERVICE_ERROR_CODE_GENERAL);
                }
                else if (bCleanUpHash == TRUE)
                {
                    eReturnCode = eStockMsgHashValuesClean(psAppObj);
                    if (eReturnCode != STOCK_TICKER_RETURN_CODE_SUCCESS)
                    {
                        SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                            STOCK_TICKER_MGR_OBJECT_NAME": failed to clean-up values hashes (%s)",
                            GsStockTickerMgrIntf.pacGetReturnCodeName(eReturnCode)
                                );
                    }
                }

                if (psAppObj != NULL)
                {
                    // Unlock the app facing object
                    SMSO_vUnlock((SMS_OBJECT)psAppObj);
                }
            }
            break;

            case DATASERVICE_EVENT_TIMEOUT:
            {
                STOCK_TICKER_APP_OBJECT_STRUCT *psAppObj = NULL;
                BOOLEAN bExpired = FALSE;
                STOCK_TICKER_RETURN_CODE_ENUM eReturnCode = STOCK_TICKER_RETURN_CODE_ERROR;

                printf(STOCK_TICKER_MGR_OBJECT_NAME": event %s\n",
                    MACRO_TO_STRING(DATASERVICE_EVENT_TIMEOUT)
                        );

                // Need to look through all monitoring symbols to make sure
                // that all of the expired ones moved to N/A state and
                // application is notified about this.
                do
                {
                    psAppObj = psGetAppFacingObject((AGW_SERVICE_OBJECT)psObj);
                    if (psAppObj == NULL)
                    {
                        eReturnCode = STOCK_TICKER_RETURN_CODE_NOT_OWNER;
                        break;
                    }

                    eReturnCode = eStockMsgDescInvalidateExpired(psAppObj, &bExpired);
                    if (eReturnCode != STOCK_TICKER_RETURN_CODE_SUCCESS)
                    {
                        SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                            STOCK_TICKER_MGR_OBJECT_NAME": failed to invalidate message values (%s)",
                            GsStockTickerMgrIntf.pacGetReturnCodeName(eReturnCode)
                            );
                    }

                    if (bExpired == TRUE)
                    {
                        eReturnCode = eStockMsgHashValuesClean(psAppObj);
#if SMS_DEBUG==1
                        if (eReturnCode != STOCK_TICKER_RETURN_CODE_SUCCESS)
                        {
                            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                                STOCK_TICKER_MGR_OBJECT_NAME": failed to clean-up values hashes (%s)",
                                GsStockTickerMgrIntf.pacGetReturnCodeName(eReturnCode)
                                    );
                        }
#endif
                    }
#if SMS_DEBUG==1
                    else
                    {
                        printf(STOCK_TICKER_MGR_OBJECT_NAME
                            ": skip values clean-up since no one value expired\n");
                    }
#endif
                } while (FALSE);

                if (eReturnCode != STOCK_TICKER_RETURN_CODE_SUCCESS)
                {
                    DATASERVICE_IMPL_vLog(STOCK_TICKER_MGR_OBJECT_NAME
                        ": failed to clean timed out data (%d)\n", eReturnCode);
                    SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                        STOCK_TICKER_MGR_OBJECT_NAME": failed to process timer (%s)",
                        GsStockTickerMgrIntf.pacGetReturnCodeName(eReturnCode)
                            );
                }

                if (psAppObj != NULL)
                {
                    SMSO_vUnlock((SMS_OBJECT) psAppObj);
                }
            }
            break;

            default:
            {
                SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    STOCK_TICKER_MGR_OBJECT_NAME": Got unknown event (%u)",
                    tCurrentEvent);
                DATASERVICE_IMPL_vLog(STOCK_TICKER_MGR_OBJECT_NAME": Got unknown event (%u)\n",
                    tCurrentEvent);
            }
            break;
        }

        if (bStopEvent == TRUE)
        {
            // Uninitialize the object, but leave
            // enough to be useful to the callback
            vUninitObject( psObj, FALSE );
        }

        // Update event mask with any relevant events which have occurred
        SMSU_tUpdate(&psObj->sEvent, tEventMask);

        // Notify of any change via any registered callback which may be present
        SMSU_bNotify(&psObj->sEvent);

        if (bStopEvent == TRUE)
        {
            // Filter out all further stocks manager updates
            SMSU_tFilter(&psObj->sEvent, DATASERVICE_EVENT_ALL);

            vDestroyObject(psObj);
        }
    }

    return;
}

/*****************************************************************************
*
*   eProcessPayload
*
*****************************************************************************/
static STOCK_TICKER_RETURN_CODE_ENUM eProcessPayload (
    STOCK_TICKER_MGR_OBJECT_STRUCT *psObj,
    OSAL_BUFFER_HDL hPayload
        )
{
    STOCK_TICKER_RETURN_CODE_ENUM eReturnCode = STOCK_TICKER_RETURN_CODE_ERROR;
    STOCK_TICKER_APP_OBJECT_STRUCT *psAppObj = NULL;

    psAppObj = psGetAppFacingObject((STOCK_TICKER_SERVICE_OBJECT) psObj);
    if (psAppObj != NULL)
    {
        eReturnCode =
            GsStockTickerIntf.eProcessMessage(psObj->hParser, &hPayload);

        // If we still have a payload handle, free it
        if (hPayload != OSAL_INVALID_BUFFER_HDL)
        {
            DATASERVICE_IMPL_bFreeDataPayload(hPayload);
        }

        SMSO_vUnlock((SMS_OBJECT) psAppObj);

        if (eReturnCode != STOCK_TICKER_RETURN_CODE_SUCCESS)
        {
            printf(STOCK_TICKER_MGR_OBJECT_NAME
                    ": [%d] failed to process payload via Parser (%s)",
                    __LINE__, GsStockTickerMgrIntf.pacGetReturnCodeName(eReturnCode)
                        );
        }
    }
    else
    {
        eReturnCode = STOCK_TICKER_RETURN_CODE_NOT_OWNER;
    }

    return eReturnCode;
}

/*****************************************************************************
*
*   vUninitObject
*
*****************************************************************************/
static void vUninitObject (
    STOCK_TICKER_MGR_OBJECT_STRUCT *psObj,
    BOOLEAN bFullDelete
        )
{
    STOCK_TICKER_APP_OBJECT_STRUCT *psAppObj = psObj->psAppObj;
    BOOLEAN bLocked;

    bLocked = SMSO_bLock((SMS_OBJECT)psAppObj, OSAL_OBJ_TIMEOUT_INFINITE);
    if (bLocked == TRUE)
    {
        SMSO_vUnlock((SMS_OBJECT)psAppObj);
    }
    else
    {
        SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                STOCK_TICKER_MGR_OBJECT_NAME
                ": failed to lock Facing object during Un-initialization");
    }

    // Uninit the app object
    vUninitAppFacingObject(psObj);

    // Remove any broadcast-specific parsing data
    if (psObj->hParser != STOCK_TICKER_INTERFACE_INVALID_OBJECT)
    {
        bLocked = DATASERVICE_IMPL_bLock((DATASERVICE_IMPL_HDL)psObj);
        if (bLocked == TRUE)
        {
            // Make the call to uninit the broadcast-specific parser?
            GsStockTickerIntf.vUnInit(psObj->hParser);
            psObj->hParser = STOCK_TICKER_INTERFACE_INVALID_OBJECT;

            DATASERVICE_IMPL_vUnlock((DATASERVICE_IMPL_HDL)psObj);
        }
        else
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    STOCK_TICKER_MGR_OBJECT_NAME
                    ": failed to lock Parsing object during Un-initialization"
                    );
        }
        psObj->hParser = STOCK_TICKER_INTERFACE_INVALID_OBJECT;
    }

    if (bFullDelete == TRUE)
    {
        vDestroyObject(psObj);
    }

    return;
}

/*****************************************************************************
*
*   vDestroyObject
*
*****************************************************************************/
static void vDestroyObject (
    STOCK_TICKER_MGR_OBJECT_STRUCT *psObj
        )
{
    // Destroy the SMS Update Event object
    SMSU_vDestroy(&psObj->sEvent);

    return;
}

/*****************************************************************************
*
*   vSetError
*
*****************************************************************************/
static void vSetError (
    STOCK_TICKER_MGR_OBJECT_STRUCT *psObj,
    DATASERVICE_ERROR_CODE_ENUM eErrorCode
        )
{
    // Tell the DSM about it
    DATASERVICE_IMPL_vError((DATASERVICE_IMPL_HDL)psObj, eErrorCode);

    return;
}

#ifdef SUPPORT_CUNIT
#include <stock_ticker_mgr.cunit>
#endif
