/******************************************************************************/
/*                    Copyright (c) Sirius XM Radio, Inc.                     */
/*                            All Rights Reserved                             */
/*            Licensed Materials - Property of Sirius XM Radio, Inc.          */
/******************************************************************************/
/*******************************************************************************
 *
 * DESCRIPTION
 *
 * This is the Stocks Interface header file.
 *
 ******************************************************************************/

#ifndef STOCK_TICKER_INTERFACE_H_
#define STOCK_TICKER_INTERFACE_H_

#include "standard.h"
#include "osal.h"
#include "sms_api.h"

  /**********************/
 /** GLOBAL CONSTANTS **/
/**********************/

  /*********************/
 /** GLOBAL TYPEDEFS **/
/*********************/

// Stock unique identifier
typedef UN16 STOCK_ID;
#define STOCK_INVALID_ID ((STOCK_ID)UN16_MAX)

// Interface handle
typedef struct stock_ticker_interface_object * STOCK_TICKER_INTERFACE_OBJECT;
#define STOCK_TICKER_INTERFACE_INVALID_OBJECT \
            (STOCK_TICKER_INTERFACE_OBJECT)NULL

// Defines hash type
typedef OSAL_CRC_RESULT STOCK_TICKER_HASH;
#define STOCK_TICKER_INVALID_HASH ((STOCK_TICKER_HASH)UN32_MAX)

typedef enum stock_msg_state_enum
{
    STOCK_MSG_STATE_INVALID = 0,
    STOCK_MSG_STATE_PRICE_AVAILABLE,
    STOCK_MSG_STATE_PRICE_NOT_AVAILABLE,
    STOCK_MSG_STATE_DELISTED
} STOCK_MSG_STATE_ENUM;

// Supported message types
typedef enum stock_ticker_msg_type_enum
{
    STOCK_TICKER_MSG_TYPE_START = 0,
    STOCK_TICKER_MSG_TYPE_STOCK_SYMBOL =
        STOCK_TICKER_MSG_TYPE_START,
    STOCK_TICKER_MSG_TYPE_STOCK_VALUE,
    STOCK_TICKER_MSG_TYPE_STOCK_VALUE_EX,
    STOCK_TICKER_MSG_TYPE_DATA_PROVIDER_MSG,
    STOCK_TICKER_MSG_TYPE_MAX,
    STOCK_TICKER_MSG_TYPE_INVALID
} STOCK_TICKER_MSG_TYPE_ENUM;

// Defines available return codes generated by the STOCKS
typedef enum stock_ticker_return_code_enum
{
    STOCK_TICKER_RETURN_CODE_SUCCESS = 0, // Success case
    STOCK_TICKER_RETURN_CODE_NOTHING_TO_UPDATE,
    STOCK_TICKER_RETURN_CODE_NEED_TO_UPDATE,
    STOCK_TICKER_RETURN_CODE_NEED_TO_TOUCH,
    STOCK_TICKER_RETURN_CODE_FALSE,
    STOCK_TICKER_RETURN_CODE_BAD_INPUT,
    STOCK_TICKER_RETURN_CODE_BAD_STATE,
    STOCK_TICKER_RETURN_CODE_BAD_SYNC_SEQ,
    STOCK_TICKER_RETURN_CODE_BAD_DATA,
    STOCK_TICKER_RETURN_CODE_BAD_XMAPP_HDR,
    STOCK_TICKER_RETURN_CODE_NOT_OWNER,
    STOCK_TICKER_RETURN_CODE_NOT_UNIQUE,
    STOCK_TICKER_RETURN_CODE_NOT_ENOUGH_BUFFER,
    STOCK_TICKER_RETURN_CODE_NOT_SUPPORTED,
    STOCK_TICKER_RETURN_CODE_NOT_KNOWN_HANDLE,
    STOCK_TICKER_RETURN_CODE_UNKNOWN_MSG_TYPE,
    STOCK_TICKER_RETURN_CODE_IO_ERROR,
    STOCK_TICKER_RETURN_CODE_MEMORY_ERROR,
    STOCK_TICKER_RETURN_CODE_DB_ERROR,
    STOCK_TICKER_RETURN_CODE_ERROR // General error
} STOCK_TICKER_RETURN_CODE_ENUM;

  /*****************************/
 /** GLOBAL STRUCT TEMPLATES **/
/*****************************/
typedef struct stock_ticker_quote_struct
{
    // Full part of the current price
    UN32 un32Price;

    // 100th parts of the current price
    UN8 un8PriceFrac;

    // Full part of the price delta
    UN32 un32PriceDelta;

    // 100th parts of the price delta
    UN8 un8PriceDeltaFrac;

    // Value change direction
    STOCK_MSG_PRICE_DIRECTION_ENUM eDir;
} STOCK_TICKER_QUOTE_STRUCT;

typedef struct stock_ticker_plugin_values_arg_struct
{
    // Stock index
    STOCK_ID tIndex;

    // Value state
    STOCK_MSG_STATE_ENUM eState;

    // Quote
    STOCK_TICKER_QUOTE_STRUCT sQuote;
} STOCK_TICKER_PLUGIN_VALUES_ARG_STRUCT;

typedef struct stock_ticker_plugin_symbol_arg_struct
{
    // Stock index
    STOCK_ID tIndex;

    // Stock name
    const char *pacName;
    size_t tNameLength;
} STOCK_TICKER_PLUGIN_SYMBOL_ARG_STRUCT;

typedef struct stock_ticker_plugin_provider_info_arg_struct
{
    // Provider info as a string object
    STRING_OBJECT hInfo;

    // Sequence number
    UN8 un8SeqNumber;
} STOCK_TICKER_PLUGIN_PROVIDER_INFO_ARG_STRUCT;

typedef struct stock_ticker_plugin_interface_arg_struct
{
    // Message type
    STOCK_TICKER_MSG_TYPE_ENUM eType;

    // Request only symple message touch.
    BOOLEAN bDoTouch;

    union
    {
        // Symbol message data
        STOCK_TICKER_PLUGIN_SYMBOL_ARG_STRUCT sSymbolMsg;
        // Values message data
        STOCK_TICKER_PLUGIN_VALUES_ARG_STRUCT sValueMsg;
        // Stock Ticker data provider
        STOCK_TICKER_PLUGIN_PROVIDER_INFO_ARG_STRUCT sProviderInfo;
    } uData;
} STOCK_TICKER_PLUGIN_INTERFACE_ARG_STRUCT;

// Keep extra information provided by parser for payload validation
typedef union stock_ticker_plugin_precheck_interface_arg_union
{
    // Keeps data for SYMBOLS message
    struct
    {
        // Number of symbol names
        UN8 un8SymbolCount;
        // Symbol size in characters
        UN8 un8SymbolSize;
    } sSymbol;

    // Keeps data for VALUES message
    struct
    {
        // Number of values addressed by currently collecting paylaod
        UN8 un8Count;
        // Index of the first addressed index
        UN16 un16StartIndex;
    } sValues;

    // Keeps data for VALUES EX message
    struct
    {
        // Number of values addressed by currently collecting paylaod
        UN8 un8Count;
    } sValuesEx;

    // Keeps data for DATA PROVIDER message
    struct
    {
        // Number of bytes utilized by the provider info
        UN8 un8TextSize;
    } sProviderInfo;

} STOCK_TICKER_PLUGIN_PRECHECK_INTERFACE_ARG_STRUCT;

// The stock ticker interface APIs
typedef struct stock_ticker_plugin_interface_struct
{
    const DSI tDSI;

    const size_t tOTABufferByteSize;

    const size_t tSymbolNameMaxSize;

    STOCK_TICKER_INTERFACE_OBJECT (*hInit) (
        STOCK_TICKER_SERVICE_OBJECT hService,
        SMS_OBJECT hParent
            );

    void (*vUnInit) (
        STOCK_TICKER_INTERFACE_OBJECT hInterface
            );

    STOCK_TICKER_RETURN_CODE_ENUM (*eProcessMessage) (
        STOCK_TICKER_INTERFACE_OBJECT hInterface,
        OSAL_BUFFER_HDL *phPayload
            );

    STOCK_TICKER_RETURN_CODE_ENUM (*eState) (
        STOCK_TICKER_INTERFACE_OBJECT hInterface
            );

} STOCK_TICKER_PLUGIN_INTERFACE_STRUCT;

// The stock ticker manager interface
typedef struct stock_ticker_service_mgr_interface_struct
{
    STOCK_TICKER_RETURN_CODE_ENUM (*ePayloadProcessStarted) (
        STOCK_TICKER_SERVICE_OBJECT hService
           );

    STOCK_TICKER_RETURN_CODE_ENUM (*ePayloadProcessCommited) (
        STOCK_TICKER_SERVICE_OBJECT hService
           );

    STOCK_TICKER_RETURN_CODE_ENUM (*eUpdate) (
        STOCK_TICKER_SERVICE_OBJECT hService,
        STOCK_TICKER_PLUGIN_INTERFACE_ARG_STRUCT *psArg
           );

    STOCK_TICKER_RETURN_CODE_ENUM (*eIsPayloadProcessNeeded) (
        STOCK_TICKER_SERVICE_OBJECT hService,
        STOCK_TICKER_MSG_TYPE_ENUM eType,
        OSAL_CRC_RESULT tHash
            );

    STOCK_TICKER_RETURN_CODE_ENUM (*eIsMessageNeeded) (
        STOCK_TICKER_SERVICE_OBJECT hService,
        STOCK_TICKER_MSG_TYPE_ENUM eType,
        STOCK_TICKER_PLUGIN_PRECHECK_INTERFACE_ARG_STRUCT const *puData
            );

    const char* (*pacGetReturnCodeName) (
        STOCK_TICKER_RETURN_CODE_ENUM eCode
            );

    const char* (*pacGetMessageTypeName) (
        STOCK_TICKER_MSG_TYPE_ENUM eType
            );

} STOCK_TICKER_SERVICE_MGR_INTERFACE_STRUCT;

  /****************************/
 /** GLOBAL UNION TEMPLATES **/
/****************************/

  /**********************/
 /** GLOBAL VARIABLES **/
/**********************/

// Interface into the Stock Ticker plug-ins
extern const STOCK_TICKER_PLUGIN_INTERFACE_STRUCT GsStockTickerIntf;

// Interface into the Stock Ticker manager
extern const STOCK_TICKER_SERVICE_MGR_INTERFACE_STRUCT GsStockTickerMgrIntf;

  /*******************/
 /** GLOBAL MACROS **/
/*******************/

  /***********************/
 /** GLOBAL PROTOTYPES **/
/***********************/

#endif /* STOCK_TICKER_INTERFACE_H_ */
