/******************************************************************************/
/*                    Copyright (c) Sirius XM Radio, Inc.                     */
/*                            All Rights Reserved                             */
/*         Licensed Materials - Property of Sirius XM Radio, Inc.             */
/******************************************************************************/
/*******************************************************************************
 *
 * DESCRIPTION
 *
 *  This module contains the Object:SRM implementation for the
 *  Sirius Module Services (SMS)
 *
 ******************************************************************************/
#include <time.h>
#include <string.h>
#include <stdarg.h>

#include "standard.h"
#include "osal.h"

#include "sms_api.h"
#include "sms_version.h"

#include "sms.h"
#include "sms_obj.h"
#include "sms_task.h"
#include "sms_event.h"
#include "module_obj.h"
#include "decoder_obj.h"

#include "dataservice_mgr_obj.h"
#include "tag_obj.h"
#include "srm_obj.h"
#include "_srm_obj.h"

#include "sms_api_debug.h"
static const char *gpacThisFile = __FILE__;

/*****************************************************************************
                             PUBLIC FUNCTIONS
*****************************************************************************/

/*****************************************************************************
*
*   hGet
*
*   This API is used to get access to the physical satellite receiver module
*   (SRM) and its software driver through an SRM_OBJECT handle. A satellite
*   receiver module may contain one or more modules and decoders as described
*   in Section 8.4. Getting an SRM will power the SRM and properly power
*   its antenna if they have not already been powered by SMS previously.
*   The application must get an SRM before any modules can be started as
*   they are derived from the associated SRM handle returned when this API
*   executes successfully. If a valid SRH driver name and SRM name is
*   provided and the API is able to get the SRM, a handle to this SRM
*   is returned. This handle is used for all subsequent APIs which require
*   an SRM handle. If the call to this API is unsuccessful an invalid
*   handle is returned. When the application uses this API it is making a
*   request to get access to this receiver module for operational use,
*   however the SRM should not be considered "started" until it transitions
*   to the READY state.  This can be determined by the application by either
*   configuring to receive the SRM event (SRM_OBJECT_EVENT_STATE) or simply
*   polling its state.
*
*   Where:
*       pacDriverName - A constant string which represents the exact name of
*       the SRH driver to start. Typically this will be of the form "srh:".
*       pacName - A constant string which represents the exact name of the SRM
*       to start. Typically this will be of the form "srm[x]: where [x] is some
*       enumerated value representing the SRM device (e.g. "srm0:").  This
*       naming convention is a suggestion.  All SRM names must be unique
*       within the SRH driver.
*       tEventRequestMask - A bit mask indicating which events the caller
*       wishes to receive.
*       vEventCallback - A callback function provided by the caller which
*       will be called whenever one or more of the selected events occur.
*       pvEventCallbackArg - The argument provided by the application when
*       the callback function for these events was registered. This is a
*       void pointer which must be cast appropriately within the callback
*       function to the application specific data type it represents.
*
*   Return:
*       A valid SRM_OBJECT handle is returned on success. Otherwise
*       SRM_INVALID_OBJECT is returned on error.
*
*****************************************************************************/
static SRM_OBJECT hGet (
    const char *pacDriverName,
    const char *pacName,
    SRM_EVENT_MASK tEventRequestMask,
    SRM_OBJECT_EVENT_CALLBACK vEventCallback,
    void *pvEventCallbackArg
        )
{
    SRM_OBJECT hSRM = SRM_INVALID_OBJECT;
    BOOLEAN bResult = FALSE;
    SRM_OBJECT_STRUCT *psObj = NULL;

    SMSAPI_DEBUG_vPrint(SRM_OBJECT_NAME, 4,
        "SRM.%s(pacDriverName: %s, pacName: %s, tEventRequestMask: 0x%X, "
        "vEventCallback: %p, pvEventCallbackArg: %p)\n",
        __FUNCTION__, pacDriverName, pacName, tEventRequestMask,
        vEventCallback, pvEventCallbackArg);

    // Verify inputs
    if (NULL == pacDriverName)
    {
        // Error!
        SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
            SRM_OBJECT_NAME": pacDriverName is not specified.");
        return SRM_INVALID_OBJECT;
    }

    if (NULL == pacName)
    {
        // Error!
        SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
            SRM_OBJECT_NAME": pacName is not specified.");
        return SRM_INVALID_OBJECT;
    }

    // Is the library initialized?
    bResult = SMS_bIsInitialized();
    if (FALSE == bResult)
    {
        // Error!
        SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
            SRM_OBJECT_NAME": SMS is not initialized.");
        return SRM_INVALID_OBJECT;
    }

    // To do this, we must lock SMS first
    bResult = SMS_bLock();
    if (FALSE == bResult)
    {
        // Error!
        SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
            SRM_OBJECT_NAME": Unable to lock SMS.");
        return SRM_INVALID_OBJECT;
    }

    do
    {
        // Has a corresponding SRM object already been created?
        hSRM = SMS_hSRM(pacDriverName, pacName);
        if (SRM_INVALID_OBJECT != hSRM)
        {
            // Error! We can only create one SRM with these attributes
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                SRM_OBJECT_NAME": SRM already exists.");
            hSRM = SRM_INVALID_OBJECT;
            break;
        }

        // Create a new SRM object.
        psObj = psCreateObject(
            pacDriverName, pacName,
            tEventRequestMask,
            vEventCallback,
            pvEventCallbackArg);
        if (NULL == psObj)
        {
            // Error! We can only create one SRM with these attributes
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                SRM_OBJECT_NAME": Unable to create SRM object.");
            break;
        }

        // Set initial reference counter
        psObj->un16RefCounter = 1;

        // If at this point all goes well, then we can add ourself
        // to the SMS's management of SRMs.
        bResult = SMS_bAddSRM((SRM_OBJECT)psObj);
        if (FALSE == bResult)
        {
            // Error!
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                SRM_OBJECT_NAME": SRM could not be added to SMS.");

            // Destroy object
            vDestroyObject(psObj);

            break;
        }

        // Success! Added to SMS's management.
        SMSAPI_DEBUG_vPrint(SRM_OBJECT_NAME, 4,
            "%s: Added to SMS.\n", psObj->pacObjectName);

        // Use this one, it's all good.
        hSRM = (SRM_OBJECT)psObj;

        SMSAPI_DEBUG_vPrint(SRM_OBJECT_NAME, 4,
            "%s: Created SRM: %p.\n", psObj->pacObjectName, hSRM);

    } while (FALSE);

    // Unlock SMS. We don't need it anymore
    SMS_vUnLock();

    return hSRM;
}

/*****************************************************************************
*
*   vRelease
*
*   This API is used to release access to an existing SRM object. This API
*   will perform any un-initialization or shutdown required and releases all
*   resources the SRM consumed if SMS is not using this SRM for other tasks.
*   This means it will also destroy any derivative objects which still exist
*   in this case. Upon successful completion, this object will no longer be
*   valid.
*
*   Where:
*       hSRM - A handle to a valid SRM Object returned from a previous
*       successful call to SRM.hGet(). This is the handle of the object the
*       caller wishes to stop.
*
*   Return:
*       None.
*
*****************************************************************************/
static void vRelease (
    SRM_OBJECT hSRM
        )
{
    BOOLEAN bResult;

    SMSAPI_DEBUG_vPrint(SRM_OBJECT_NAME, 4,
        "SRM.%s(hSRM: %p)\n", __FUNCTION__, hSRM);

    // Verify SMS Object
    bResult = SMSO_bValid((SMS_OBJECT)hSRM);
    if (TRUE == bResult)
    {
        // Start application-initiated object release
        bInitiateObjectRelease((SRM_OBJECT_STRUCT *)hSRM,
            SMS_OBJECT_RELEASE_BY_APP);
    }
    else
    {
        // Error!
        SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
            SRM_OBJECT_NAME": Invalid SRM object.");
    }

    return;
}

/*****************************************************************************
*
*   eState
*
*   This API is used to query the current object state. This may be either
*   from a callback function in response to an event, or simply polled by the
*   application. The returned value is the current state of the object.
*
*       Where:
*           hSRM - A handle to a valid SRM Object for which the caller wishes
*           to learn the current state of.
*
*       Return:
*           The current object state as a value of SRM_STATE_ENUM type.
*
*****************************************************************************/
static SRM_STATE_ENUM eState (
    SRM_OBJECT hSRM
        )
{
    SRM_STATE_ENUM eState;

    eState = SRM_eState(hSRM);

    return eState;
}

/*****************************************************************************
*
*   eErrorCode
*
*   This API is used to query the current object's error code. This may be
*   either from a callback function in response to an event, or simply polled
*   by the application. The returned value is the current error code of the
*   object. Typically this is required when an object transitions into the
*   ERROR state and the application wishes to know more about why the object
*   is in error.
*
*       Where:
*           hSRM - A handle to a valid SRM Object for which the caller wishes
*           to learn the error code of.
*
*       Return:
*           The current object error code as a value of SRM_ERROR_CODE_ENUM
*           type.
*
*****************************************************************************/
static SRM_ERROR_CODE_ENUM eErrorCode (
    SRM_OBJECT hSRM
        )
{
    SRM_ERROR_CODE_ENUM eErrorCode =
        SRM_ERROR_CODE_INVALID_OBJECT;

    // Verify and lock SMS Object
    BOOLEAN bLocked =
        SMSO_bLock((SMS_OBJECT)hSRM, OSAL_OBJ_TIMEOUT_INFINITE);
    if(bLocked == TRUE)
    {
        // De-reference object
        SRM_OBJECT_STRUCT *psObj =
            (SRM_OBJECT_STRUCT *)hSRM;

        // Extract object error code
        eErrorCode = psObj->eErrorCode;

        // Unlock object
        SMSO_vUnlock((SMS_OBJECT)hSRM);
    }
    else
    {
        // Error!
        SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
            SRM_OBJECT_NAME": Unable to lock SRM.");
    }

    return eErrorCode;
}

/*****************************************************************************
*
*   tEventMask
*
*   This API is used to query or poll the object's current update mask
*   (pending event mask) . This is typically used when the application needs
*   to poll the current pending events which have not been handled since the
*   last call to this API. Normally events are handled in a callback function
*   with the currently active events reported to the callback through an input
*   parameter. However if the application has not defined a callback function,
*   this API must be used to poll for events otherwise the API is not needed.
*   Events returned using this call will automatically be cleared after it
*   returns any active events.
*
*   Note: The events reported by this API are not affected by the event
*   callback. In other words if the application receives events via callback,
*   those events are still reported when polled. This API provides a copy of
*   and therefore independent indication of any pending events.
*
*   Where:
*       hSRM - A handle to a valid SRM Object for which the caller wishes to
*       poll for events which have occurred.
*
*   Return:
*       A bit mask representing any events which have occurred since the
*       last poll regardless if they were handled by an event callback or
*       not.
*
*****************************************************************************/
static SRM_EVENT_MASK tEventMask (
    SRM_OBJECT hSRM
        )
{
    SRM_EVENT_MASK tEventMask =
        SRM_OBJECT_EVENT_NONE;

    // Verify and lock SMS Object
    BOOLEAN bLocked =
        SMSO_bLock((SMS_OBJECT)hSRM, OSAL_OBJ_TIMEOUT_INFINITE);
    if(bLocked == TRUE)
    {
        // De-reference object
        SRM_OBJECT_STRUCT *psObj =
            (SRM_OBJECT_STRUCT *)hSRM;

        tEventMask = SMSU_tMask(&psObj->sEvent);

        // Unlock object
        SMSO_vUnlock((SMS_OBJECT)hSRM);
    }
    else
    {
        // Error!
        SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
            SRM_OBJECT_NAME": Unable to lock SRM.");
    }

    return tEventMask;
}

/*****************************************************************************
*
*   eModifyRegisteredEventMask
*
*   This API is used to change the event mask that was set with the SRM's
*   hGet API.  Any events that were pending will be provided to the event
*   callback using the newly modified event mask.
*
*   Where:
*       hSRM - A handle to a valid SRM Object for which the caller wishes to
*       modify the event mask of.
*       tEventMask - A bit mask specifying what changes should be made to the
*       registered event mask.  This is used in conjunction with the action
*       described by eModification.
*       eModification - An enumeration that tells the SRM object
*       how to use tEventMask to modify the current event mask.
*       These operations included enabling events, disabling events, and
*       replacing the event mask with the one specified in tEventMask.
*       The SMSAPI_MODIFY_EVENT_MASK_ENUM is described in section 10.2.2.6
*
*   Return:
*       SMSAPI_RETURN_CODE_SUCCESS on success, SMSAPI_RETURN_CODE_ERROR
*       on error
*
*****************************************************************************/
static SMSAPI_RETURN_CODE_ENUM eModifyRegisteredEventMask (
    SRM_OBJECT hSRM,
    SRM_EVENT_MASK tEventMask,
    SMSAPI_MODIFY_EVENT_MASK_ENUM eModification
        )
{
    BOOLEAN bValid;
    SMSAPI_RETURN_CODE_ENUM eReturnCode = SMSAPI_RETURN_CODE_ERROR;

    if ( eModification < SMSAPI_MODIFY_EVENT_MASK_START ||
         eModification >= SMSAPI_MODIFY_EVENT_MASK_INVALID )
    {
        return SMSAPI_RETURN_CODE_INVALID_INPUT;
    }

    // Verify SMS Object is valid
    bValid = SMSO_bValid((SMS_OBJECT)hSRM);
    if(bValid == TRUE)
    {
        SMS_EVENT_HDL hEvent;
        SMS_EVENT_DATA_UNION *puEventData;
        SRM_OBJECT_STRUCT *psObj;

        psObj = (SRM_OBJECT_STRUCT *)hSRM;

        // Allocate an event
        hEvent = SMSE_hAllocateEvent(psObj->hEventHdlr,
                    SMS_EVENT_MODIFY_EVENT_MASK,
                    &puEventData,
                    SMS_EVENT_OPTION_NONE);
        if(hEvent != SMS_INVALID_EVENT_HDL)
        {
            BOOLEAN bPosted;

            // Extract event information and populate it
            puEventData->uSRM.sSrmEventMask.eModification =
                    eModification;
            puEventData->uSRM.sSrmEventMask.tEventMask =
                    tEventMask;

            //post event and wait for return
            bPosted = SMSE_bPostEvent(hEvent);
            if (bPosted == FALSE)
            {
                //Error!!
                SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    SRM_OBJECT_NAME
                    ": Unable to post modify srm event mask.");
            }
            else
            {
                eReturnCode = SMSAPI_RETURN_CODE_SUCCESS;
            }
        }
    }
    else
    {
        // Error!
        SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
            SRM_OBJECT_NAME": Invalid SRM object.");
    }

    return eReturnCode;
}

/*****************************************************************************
*
*   bUpdate
*
*   This API is used to force an update (trigger all events specified).
*   In response to this API, the object will force the call of the registered
*   application provided callback function. When called, this callback will
*   indicate all events which have been specified (regardless if they actually
*   occurred or not). The application provided callback is then able to
*   process all these events, effectively responding to an "update" of all
*   registered events.
*
*   Generally this API is only required during development or as a means to
*   test an object. Since all events registered for asynchronously notify the
*   application the application should always be aware of all events which
*   have occurred. However this method provides a way to force an update of
*   specifically requested events.
*
*   Where:
*       hSRM - A handle to a valid SRM Object for which the caller wishes to
*       force an update (event callback) of all requested events.
*       tMask - An event mask which indicates which events are to be triggered.
*
*   Return:
*       TRUE on success, otherwise FALSE on error.
*
*****************************************************************************/
static BOOLEAN bUpdate (
    SRM_OBJECT hSRM,
    SRM_EVENT_MASK tMask
        )
{
    BOOLEAN bPosted = FALSE, bValid;

    // Verify SMS Object is valid
    bValid = SMSO_bValid((SMS_OBJECT)hSRM);
    if(bValid == TRUE)
    {
        // De-reference object
        SRM_OBJECT_STRUCT *psObj =
            (SRM_OBJECT_STRUCT *)hSRM;
        SMS_EVENT_HDL hEvent;
        SMS_EVENT_DATA_UNION *puEventData;

        // Allocate an event
        hEvent =
            SMSE_hAllocateEvent( psObj->hEventHdlr,
                SMS_EVENT_UPDATE_OBJECT, &puEventData, SMS_EVENT_OPTION_NONE );
        if(hEvent != SMS_INVALID_EVENT_HDL)
        {
            // Extract event information and populate it
            puEventData->uSRM.sUpdate.tEventMask = tMask;
            bPosted = SMSE_bPostEvent(hEvent);
            if(bPosted == FALSE)
            {
                // Error!
                SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    SRM_OBJECT_NAME": Unable to update this SRM.");
            }
        }
        else
        {
            // Error!
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                SRM_OBJECT_NAME": Unable to allocate SRM event.");
        }
    }
    else
    {
        // Error!
        SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
            SRM_OBJECT_NAME": Invalid SRM object.");
    }

    return bPosted;
}

/*****************************************************************************
                             FRIEND FUNCTIONS
*****************************************************************************/

/*******************************************************************************
*
*   SRM_bRelease
*
*   Release SMS internal access to this SRM
*   This may cause object destruction
*
*******************************************************************************/
BOOLEAN SRM_bRelease (
    SRM_OBJECT hSRM,
    SMS_OBJECT_RELEASE_INITIATOR_ENUM eInitiator
        )
{
    BOOLEAN bResult = FALSE;
    SRM_OBJECT_STRUCT *psObj = (SRM_OBJECT_STRUCT *)hSRM;

    SMSAPI_DEBUG_vPrint(SRM_OBJECT_NAME, 4,
        "%s(hSRM: %p, eInitiator: %d)\n",
        __FUNCTION__, hSRM, eInitiator);

    do
    {
        // Verify input
        if (SMS_OBJECT_RELEASE_BY_APP == eInitiator)
        {
            // Error!
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                SRM_OBJECT_NAME": Incorrect release request.");
            break;
        }

        // Verify SMS Object
        bResult = SMSO_bValid((SMS_OBJECT)hSRM);
        if (FALSE == bResult)
        {
            // Error!
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                SRM_OBJECT_NAME": Invalid SRM object.");
            break;
        }

        // Initiate object release
        bResult = bInitiateObjectRelease(psObj, eInitiator);
        if (FALSE == bResult)
        {
            break;
        }

    } while (FALSE);

    return bResult;
}

/*****************************************************************************
*
*   SRM_eState
*
*   This API is used to query the current object state. This may be either
*   from a callback function in response to an event, or simply polled by the
*   application. The returned value is the current state of the object.
*
*       Where:
*           hSRM - A handle to a valid SRM Object for which the caller wishes
*           to learn the current state of.
*
*       Return:
*           The current object state as a value of SRM_STATE_ENUM type.
*
*****************************************************************************/
SRM_STATE_ENUM SRM_eState (
    SRM_OBJECT hSRM
        )
{
   SRM_STATE_ENUM eState = SRM_STATE_INVALID;
   BOOLEAN bLocked;

    // Verify and lock SMS Object
    bLocked =
        SMSO_bLock((SMS_OBJECT)hSRM, OSAL_OBJ_TIMEOUT_INFINITE);
    if(bLocked == TRUE)
    {
        // De-reference object
        SRM_OBJECT_STRUCT *psObj =
            (SRM_OBJECT_STRUCT *)hSRM;

        // Extract object reported state
        eState = psObj->eState;

        // Unlock object
        SMSO_vUnlock((SMS_OBJECT)hSRM);
    }
    else
    {
        // Error!
        SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
            SRM_OBJECT_NAME": Unable to lock SRM.");
    }

    return eState;
}

/*****************************************************************************
*
*   SRM_bAddModule
*
*   This function is used by the RADIO implementation to cause the SRM to add
*   the specified MODULE to it's management list. When a MODULE is added to
*   an SRM, it is acknowledged by the SRM subsequently making the MODULE
*   READY by transitioning the MODULE to the READY state.
*
*****************************************************************************/
BOOLEAN SRM_bAddModule (
    SRM_OBJECT hSRM,
    MODULE_OBJECT hModule,
    MODULE_ID tId
        )
{
    BOOLEAN bResult = FALSE;

    SMSAPI_DEBUG_vPrint(SRM_OBJECT_NAME, 4,
        "%s(hSRM: %p, hModule: %p, tId: %d)\n",
        __FUNCTION__, hSRM, hModule, tId);

    // Lock the SRM object
    bResult = SMSO_bLock((SMS_OBJECT)hSRM, OSAL_OBJ_TIMEOUT_INFINITE);
    if (FALSE == bResult)
    {
        // Error!
        SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
            SRM_OBJECT_NAME": Unable to lock SRM.");
        return FALSE;
    }

    do
    {
        SRM_OBJECT_STRUCT *psObj = (SRM_OBJECT_STRUCT *)hSRM;
        OSAL_RETURN_CODE_ENUM eReturnCode = OSAL_ERROR;
        PSRM_MODULE_ENTRY_STRUCT psModule;
        OSAL_LINKED_LIST_ENTRY hEntry = OSAL_INVALID_LINKED_LIST_ENTRY;

        // Check validity of the object being added
        bResult = SMSO_bValid((SMS_OBJECT)hModule);
        if (FALSE == bResult)
        {
            // Error!
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                SRM_OBJECT_NAME": Invalid MODULE object.");
            break;
        }

        psModule = (PSRM_MODULE_ENTRY_STRUCT)SMSO_hCreate(
            SRM_OBJECT_NAME":Module Entry",
            sizeof(SRM_MODULE_ENTRY_STRUCT), (SMS_OBJECT)psObj, FALSE);
        if (NULL == psModule)
        {
            // Error!
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                SRM_OBJECT_NAME": Could not allocate MODULE entry.\n");
            bResult = FALSE;
            break;
        }

        // Populate entry
        psModule->bValid = TRUE;
        psModule->hModule = hModule;
        psModule->tId = tId;

        // Add this MODULE to the list. This list allows only unique
        // entries. If one exists this means this MODULE already exists.
        eReturnCode = OSAL.eLinkedListAdd(psObj->hModuleList,
            &hEntry, (void *)psModule);
        if (OSAL_SUCCESS != eReturnCode)
        {
            // Error!
            SMSO_vDestroy((SMS_OBJECT)psModule);

            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                SRM_OBJECT_NAME": Could not add MODULE entry: %d (%s).",
                eReturnCode, OSAL.pacGetReturnCodeName(eReturnCode));

            bResult = FALSE;
            break;
        }

        SMSAPI_DEBUG_vPrint(SRM_OBJECT_NAME, 4,
            "%s: Added Module entry.\n", psObj->pacObjectName);

        bResult = bAssociateWithModule(psObj, hModule);
        if (FALSE == bResult)
        {
            // Error!
            break;
        }

    } while (FALSE);

    // Unlock SRM
    SMSO_vUnlock((SMS_OBJECT)hSRM);

    return bResult;
}

/*****************************************************************************
*
*   SRM_bRemoveModule
*
*   This function is called by the Module implementation to indicate to the
*   SRM this MODULE specified is to be removed from it's management list.
*
*****************************************************************************/
BOOLEAN SRM_bRemoveModule (
    SRM_OBJECT hSRM,
    MODULE_OBJECT hModule
        )
{
    BOOLEAN bResult = FALSE;
    SRM_OBJECT_STRUCT *psObj = (SRM_OBJECT_STRUCT *)hSRM;
    SMS_EVENT_HDL hEvent;
    SMS_EVENT_DATA_UNION *puEventData;

    SMSAPI_DEBUG_vPrint(SRM_OBJECT_NAME, 4,
        "%s(hSRM: %p, hModule: %p\n",
        __FUNCTION__, hSRM, hModule);

    do
    {
        // Verify SMS Object
        bResult = SMSO_bValid((SMS_OBJECT)hSRM);
        if (FALSE == bResult)
        {
            // Error!
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                SRM_OBJECT_NAME": Invalid SRM object.");
            break;
        }

        // Allocate an event
        hEvent = SMSE_hAllocateEvent( psObj->hEventHdlr,
            SMS_EVENT_REMOVE_MODULE, &puEventData,
            SMS_EVENT_OPTION_NONE);
        if (SMS_INVALID_EVENT_HDL == hEvent)
        {
            // Error!
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                SRM_OBJECT_NAME": Unable to allocate event.");
            bResult = FALSE;
            break;
        }

        // Populate event information
        puEventData->uSRM.sRemoveModule.hModule = hModule;

        // Post event
        bResult = SMSE_bPostEvent(hEvent);
        if (FALSE == bResult)
        {
            // Error!
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                SRM_OBJECT_NAME": Unable to post event.");
            break;
        }

        SMSAPI_DEBUG_vPrint(SRM_OBJECT_NAME, 4,
            "%s: REMOVE_MODULE event is posted.\n",
            psObj->pacObjectName);

    } while (FALSE);

    return bResult;
}

/*****************************************************************************
*
*   SRM_bSetError
*
*****************************************************************************/
BOOLEAN SRM_bSetError (
    SRM_OBJECT hSRM,
    SRM_ERROR_CODE_ENUM eErrorCode
        )
{
    BOOLEAN bValid, bPosted = FALSE;

    // Verify SMS Object is valid
    bValid = SMSO_bValid((SMS_OBJECT)hSRM);
    if(bValid == TRUE)
    {
        SRM_OBJECT_STRUCT *psObj =
            (SRM_OBJECT_STRUCT *)hSRM;
        SMS_EVENT_HDL hEvent;
        SMS_EVENT_DATA_UNION *puEventData;

        // Allocate an event
        hEvent =
            SMSE_hAllocateEvent( psObj->hEventHdlr,
                SMS_EVENT_ERROR, &puEventData,
                SMS_EVENT_OPTION_NONE
                    );
        if(hEvent != SMS_INVALID_EVENT_HDL)
        {
            // Extract event information and populate it
            puEventData->uSRM.sError.eErrorCode = eErrorCode;

            // TODO: BREAKPOINT
            bPosted = SMSE_bPostEvent(hEvent);
        }
    }

    return bPosted;
}

/*****************************************************************************
*
*   SRM_bPostRadioEvent
*
*   This API is called by a non-SRM context to tell us to post an event
*   to the RADIO specific SRM event handler
*
*****************************************************************************/
BOOLEAN SRM_bPostRadioEvent (
    SRM_OBJECT hSRM,
    SMS_EVENT_TYPE_ENUM eEvent,
    void *pvArg
        )
{
    BOOLEAN bValid, bPosted = FALSE;

    // Verify SMS Object is valid
    bValid = SMSO_bValid((SMS_OBJECT)hSRM);
    if(bValid == TRUE)
    {
        SRM_OBJECT_STRUCT *psObj =
            (SRM_OBJECT_STRUCT *)hSRM;
        SMS_EVENT_HDL hEvent;
        SMS_EVENT_DATA_UNION *puEventData;

        // Allocate an event
        hEvent =
            SMSE_hAllocateEvent( psObj->hEventHdlr,
                eEvent,
                &puEventData,
                SMS_EVENT_OPTION_NONE );
        if(hEvent != SMS_INVALID_EVENT_HDL)
        {
            bPosted = RADIO_bPostSrmEvent(hEvent, pvArg);
        }
    }

    return bPosted;
}

/*******************************************************************************
*
*   SRM_hModule
*
*   This function is used to obtain a MODULE handle for a specified Module Id.
*   If the MODULE exists, and is being managed by SMS (and this
*   SRM) then a valid MODULE_OBJECT handle is returned. Otherwise an invalid
*   object handle is returned (indicating one does not already exist).
*
*******************************************************************************/
MODULE_OBJECT SRM_hModule (
    SRM_OBJECT hSRM,
    MODULE_ID tId
        )
{
    BOOLEAN bLocked;
    MODULE_OBJECT hModule = MODULE_INVALID_OBJECT;

    // Lock the SRM object
    bLocked = SMSO_bLock((SMS_OBJECT)hSRM, OSAL_OBJ_TIMEOUT_INFINITE);
    if(bLocked == TRUE)
    {
        OSAL_RETURN_CODE_ENUM eReturnCode;
        SRM_OBJECT_STRUCT *psObj = (SRM_OBJECT_STRUCT *)hSRM;
        OSAL_LINKED_LIST_ENTRY hEntry = OSAL_INVALID_LINKED_LIST_ENTRY;

        // Locate the list node which matches this MODULE's id
        // This list is kept in order of increasing MODULE handle value.
        // So this search is always linear from the first element.
        eReturnCode = OSAL.eLinkedListLinearSearch(psObj->hModuleList, &hEntry,
            n16CompareModuleId, (void *)&tId);
        if(eReturnCode == OSAL_SUCCESS) // found
        {
            BOOLEAN bLocked;
            PSRM_MODULE_ENTRY_STRUCT psModule;

            // extract and return the MODULE entry
            psModule = (PSRM_MODULE_ENTRY_STRUCT)
                            OSAL.pvLinkedListThis(hEntry);
            if(NULL == psModule)
            {
                hModule = MODULE_INVALID_OBJECT;
            }
            else
            {
                hModule = psModule->hModule;
            }

            bLocked = SMSO_bLock((SMS_OBJECT)hModule,
                OSAL_OBJ_TIMEOUT_INFINITE);
            if(bLocked == FALSE)
            {
                // Couldn't lock the module.
                // We'll create one possibly then that we can lock
                hModule = MODULE_INVALID_OBJECT;
            }
        }

        SMSO_vUnlock((SMS_OBJECT)hSRM);
    }
    else
    {
        // Error!
        SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
            SRM_OBJECT_NAME": Cannot lock SRM.");
    }

    // Return the module handle
    return hModule;
}

/*******************************************************************************
*
*   SRM_pacDriverName
*
*   This function simply returns the constant C-style null terminated character
*   array which contains the SRM's driver name as provided by the SRM object
*   handle.
*
*******************************************************************************/
const char *SRM_pacDriverName (
    SRM_OBJECT hSRM
        )
{
    BOOLEAN bValid;

    bValid = SMSO_bValid((SMS_OBJECT)hSRM);
    if (bValid == TRUE)
    {
        SRM_OBJECT_STRUCT *psObj =
            (SRM_OBJECT_STRUCT *)hSRM;

        return psObj->pacDriverName;
    }

    return NULL;
}

/*****************************************************************************
*
*   SRM_n16CompareName
*
*   This comparator function is used by SMS to compare the name in a
*   given SRM handle with the name provided in pvObj2. Since SRM names
*   are simply c-style null terminated strings a simple strcmp() is
*   performed and the result returned.
*
*****************************************************************************/
N16 SRM_n16CompareName (
    SRM_OBJECT hSRM,
    const char *pacDriverName,
    const char *pacName
        )
{
    BOOLEAN bValid;
    N16 n16CompareResult = N16_MIN;

    // Check the input value
    if(pacName == NULL)
    {
        return N16_MIN;
    }

    // Validate SRM Object
    bValid = SMSO_bValid((SMS_OBJECT)hSRM);
    if(bValid == TRUE)
    {
        SRM_OBJECT_STRUCT *psObj = (SRM_OBJECT_STRUCT *)hSRM;
        int iStrCmpResult;

        // Compare the driver name provided by argument with
        // the name of this SRM's driver
        iStrCmpResult = strcmp( psObj->pacDriverName, pacDriverName );

        // Only need to compare the name if these are different
        if (iStrCmpResult != 0)
        {
            // Compare the name provided by argument with the name
            // of this SRM object
            iStrCmpResult = strcmp( psObj->pacSrmName, pacName );
        }

        if (iStrCmpResult < 0)
        {
            n16CompareResult = -1;
        }
        else if (iStrCmpResult > 0)
        {
            n16CompareResult = 1;
        }
        else
        {
            n16CompareResult = 0;
        }
    }

    return n16CompareResult;
}

/*****************************************************************************
*
*   SRM_bRadioReady
*
*****************************************************************************/
BOOLEAN SRM_bRadioReady (
    SRM_OBJECT hSRM,
    FILE *psDevice
        )
{
    BOOLEAN bResult = FALSE;
    SRM_OBJECT_STRUCT *psObj = (SRM_OBJECT_STRUCT *)hSRM;
    SMS_EVENT_HDL hEvent;
    SMS_EVENT_DATA_UNION *puEventData;

    do
    {
        // Allocate an event
        hEvent =
            SMSE_hAllocateEvent( psObj->hEventHdlr,
                SMS_EVENT_RADIO_READY, &puEventData,
                SMS_EVENT_OPTION_URGENT);
        if (SMS_INVALID_EVENT_HDL == hEvent)
        {
            // Error!
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                SRM_OBJECT_NAME": Unable to allocate event.");
            break;
        }

        // Populate event information
        puEventData->uSRM.sRadioReady.psDevice = psDevice;

        bResult = SMSE_bPostEvent(hEvent);
        if (TRUE == bResult)
        {
            SMSAPI_DEBUG_vPrint(SRM_OBJECT_NAME, 4,
                "%s: RADIO_READY is posted.\n", psObj->pacObjectName);
        }
        else
        {
            // Error!
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                SRM_OBJECT_NAME": Unable to post event.");
            break;
        }

    } while (FALSE);

    return bResult;
}

/*****************************************************************************
*
*   SRM_bSetRadioSpecificData
*
*****************************************************************************/
BOOLEAN SRM_bSetRadioSpecificData (
    SRM_OBJECT hSRM,
    RADIO_PRIVATE_DATA_OBJECT hData
        )
{
    BOOLEAN bOwner = FALSE;

    // Verify SMS Object
    bOwner = SMSO_bOwner((SMS_OBJECT)hSRM);
    if(bOwner == TRUE)
    {
        // De-reference object
        SRM_OBJECT_STRUCT *psObj =
            (SRM_OBJECT_STRUCT *)hSRM;

        // Set radio specific data object
        psObj->hRadioData = hData;
    }

    return bOwner;
}

/*****************************************************************************
*
*   SRM_hGetRadioSpecificData
*
*****************************************************************************/
RADIO_PRIVATE_DATA_OBJECT SRM_hGetRadioSpecificData (
    SRM_OBJECT hSRM
        )
{
    BOOLEAN bOwner;
    RADIO_PRIVATE_DATA_OBJECT hData =
        RADIO_PRIVATE_DATA_INVALID_OBJECT;

    // Verify SMS Object and pointer validity
    bOwner = SMSO_bOwner((SMS_OBJECT)hSRM);
    if(bOwner == TRUE)
    {
        // De-reference object
        SRM_OBJECT_STRUCT *psObj =
            (SRM_OBJECT_STRUCT *)hSRM;

        // Get radio specific data object
        hData = psObj->hRadioData;
    }

    return hData;
}

/*****************************************************************************
*
*   SRM_hGetTag
*
*****************************************************************************/
TAG_OBJECT SRM_hGetTag(
    SRM_OBJECT hSRM
        )
{
    BOOLEAN bValid;
    TAG_OBJECT hTag = TAG_INVALID_OBJECT;
    SRM_OBJECT_STRUCT *psObj = (SRM_OBJECT_STRUCT *)hSRM;

    // Verify
    bValid = SMSO_bValid( (SMS_OBJECT)hSRM );
    if (bValid == TRUE)
    {
        hTag = psObj->hTag;
    }

    return hTag;
}

/*****************************************************************************
*
*   SRM_bReset
*
*****************************************************************************/
BOOLEAN SRM_bReset (
    SRM_OBJECT hSRM
        )
{
    BOOLEAN bResult = FALSE;

    SMSAPI_DEBUG_vPrint(SRM_OBJECT_NAME, 4,
        "%s(hSRM: %p)\n", __FUNCTION__, hSRM);

    bResult = SRM_bResetModule(hSRM, MODULE_INVALID_OBJECT);

    return bResult;
}

/*****************************************************************************
*
*   SRM_bResetModule
*
*****************************************************************************/
BOOLEAN SRM_bResetModule (
    SRM_OBJECT hSRM,
    MODULE_OBJECT hModule
        )
{
    BOOLEAN bValid, bPosted = FALSE;

    SMSAPI_DEBUG_vPrint(SRM_OBJECT_NAME, 4,
        "%s(hSRM: %p, hModule: %p\n",
        __FUNCTION__, hSRM, hModule);

    // Verify SMS Object is valid
    bValid = SMSO_bValid((SMS_OBJECT)hSRM);
    if(bValid == TRUE)
    {
        SRM_OBJECT_STRUCT *psObj =
            (SRM_OBJECT_STRUCT *)hSRM;
        SMS_EVENT_HDL hEvent;
        SMS_EVENT_DATA_UNION *puEventData;

        // Allocate an event
        hEvent =
            SMSE_hAllocateEvent( psObj->hEventHdlr,
                SMS_EVENT_RESET, &puEventData,
                SMS_EVENT_OPTION_URGENT
                    );
        if(hEvent != SMS_INVALID_EVENT_HDL)
        {
            // Extract event information and populate it
            puEventData->uSRM.sReset.hModule = hModule;

            bPosted = SMSE_bPostEvent(hEvent);
        }
    }

    return bPosted;
}

/*****************************************************************************
                             PRIVATE FUNCTIONS
*****************************************************************************/

/*****************************************************************************
*
*  psCreateObject
*
*****************************************************************************/
static SRM_OBJECT_STRUCT *psCreateObject (
    const char *pacDriverName,
    const char *pacName,
    SRM_EVENT_MASK tEventRequestMask,
    SRM_OBJECT_EVENT_CALLBACK vEventCallback,
    void *pvEventCallbackArg
        )
{
    SRM_OBJECT_STRUCT *psObj = NULL;
    char *pacTempText;
    int iNum;

    // Compute length of the required name
    iNum = strlen(SRM_OBJECT_NAME) + 1 + strlen(pacName) + 1
                    + strlen(pacDriverName) + 1;

    // Allocate enough memory for this name
    pacTempText = (char *)OSAL.pvMemoryAllocate(
        SRM_OBJECT_NAME":Text", iNum, FALSE);
    if(pacTempText != NULL)
    {
        // Now actually construct the name
        snprintf( pacTempText, iNum,
            SRM_OBJECT_NAME".%s%c%s", pacName, '\0',
            pacDriverName );

        // Create an instance of this object
        psObj = (SRM_OBJECT_STRUCT *)
            SMSO_hCreate(
                pacTempText,
                sizeof(SRM_OBJECT_STRUCT) + iNum,
                SMS_INVALID_OBJECT, // Parent
                TRUE); // Lock
        if(psObj != NULL)
        {
            TAG_OBJECT hParentTag;
            SMSAPI_RETURN_CODE_ENUM eResult;

            // Initialize object defaults
            *psObj = gsObjectDefaults;

            // Keep a copy of the object, SRM and driver name.
            OSAL.bMemCpy((char *)(psObj + 1), pacTempText, iNum);

            // Point to the specific object, SRM and driver names.
            psObj->pacObjectName = (const char *)(psObj + 1);
            psObj->pacSrmName =
                &psObj->pacObjectName
                    [strcspn(&psObj->pacObjectName[0], ".") + 1];
            psObj->pacDriverName = psObj->pacSrmName +
                strlen(psObj->pacSrmName) + 1;

            // Acquire the parent tag
            hParentTag = SMS_hGetTag();

            // Acquire the SRM tag
            eResult = TAG_eGet(SRM_OBJECT_NAME, hParentTag, &psObj->hTag,
                psObj->pacSrmName, TRUE);
            if((eResult == SMSAPI_RETURN_CODE_SUCCESS) ||
                            (eResult == SMSAPI_RETURN_CODE_CFG_NO_PARENT))
            {
                SMS_TASK_CONFIGURATION_STRUCT sConfig =
                    sSrmTaskConfiguration;

                // Construct the name to be displayed by the SMST
                sConfig.pacName = psObj->pacObjectName;

                // Initialize asynchronous update configuration and save
                // inputs
                SMSU_vInitialize(
                    &psObj->sEvent,
                    psObj,
                    SRM_OBJECT_EVENT_ALL,
                    tEventRequestMask,
                    (SMSAPI_OBJECT_EVENT_CALLBACK)vEventCallback,
                    pvEventCallbackArg
                        );

                // Note, successful installation of this task will cause
                // the provided SMS object to be owned by the SMS task.
                psObj->hServicesTask =
                    SMST_hInstall(
                        (SMS_OBJECT)psObj,
                        &sConfig,
                        (SMS_OBJECT_EVENT_HANDLER_PROTOTYPE)
                        vEventHandler,
                        &psObj->hEventHdlr
                            );
                if(psObj->hServicesTask != SMS_INVALID_TASK_HANDLE)
                {
                    // All is well.
                    SMSAPI_DEBUG_vPrint(SRM_OBJECT_NAME, 4,
                        "%s: Task installed.\n", psObj->pacObjectName);
                }
                else
                {
                    // Error!
                    SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                        SRM_OBJECT_NAME": SRM task could not be installed.");

                    // Destroy object
                    vDestroyObject(psObj);
                    psObj = NULL;
                }
            }
            else
            {
                // Error!
                SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    SRM_OBJECT_NAME": Cannot get TAG handle.");

                // Destroy object
                vDestroyObject(psObj);
                psObj = NULL;
            }
        }
        else
        {
            // Error!
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                SRM_OBJECT_NAME": Cannot allocate memory.");
        }

        // Don't need this memory anymore
        OSAL.vMemoryFree(pacTempText);
    }
    else
    {
        // Error!
        SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
            SRM_OBJECT_NAME": Cannot allocate memory.");
    }

    return psObj;
}

/*****************************************************************************
*
*  vDestroyObject
*
*****************************************************************************/
static  void vDestroyObject (
    SRM_OBJECT_STRUCT *psObj
        )
{
    BOOLEAN bOwner;

    // Check if this object is owned by the caller.
    bOwner = SMSO_bOwner((SMS_OBJECT)psObj);
    if(bOwner == TRUE)
    {
        // Destroy the async update configuration
        SMSU_vDestroy(&psObj->sEvent);

        // Destroy the  object itself
        SMSO_vDestroy((SMS_OBJECT)psObj);
    }
    else
    {
        // Error!
        SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
            SRM_OBJECT_NAME": Not owned object cannot be destroyed.");
    }

    return;
}

/*******************************************************************************
*
*   n16CompareModules
*
*   This function compares two SRM_MODULE_DESCRIPTOR_STRUCTs and orders them
*   such that descriptors with valid MODULE_OBJECT handles always
*   come first, and then ensures all descriptors are sorted by the
*   capabilities their associated modules support.
*
*   Inputs:
*       pvArg1 (in list), pvArg2 (compare against)
*
*   Outputs:
*       0   - Objects have the same value (equal, error)
*       > 0 - Object1(in list) is greater than (after) Object2
*       < 0 - Object1(in list) is less than (before) Object2
*
*******************************************************************************/
static N16 n16CompareModules (
    void *pvObj1,
    void *pvObj2
        )
{
    PSRM_MODULE_ENTRY_STRUCT psModule1 =
                    (PSRM_MODULE_ENTRY_STRUCT) pvObj1;
    PSRM_MODULE_ENTRY_STRUCT psModule2 =
        (PSRM_MODULE_ENTRY_STRUCT) pvObj2;
    BOOLEAN bModule1Valid;
    BOOLEAN bModule2Valid;
    N16 n16Result = N16_MIN;

    if ((psModule1 == NULL) || (psModule2 == NULL))
    {
        return N16_MIN;
    }

    // Determine the validity of the module handles
    bModule1Valid = SMSO_bIsValid((SMS_OBJECT)psModule1->hModule);
    bModule2Valid = SMSO_bIsValid((SMS_OBJECT)psModule2->hModule);

    // First attempt to sort by instantiation
    if (bModule1Valid != bModule2Valid)
    {
        // If module 1 isn't valid, it comes after module 2
        if (bModule1Valid == FALSE)
        {
            n16Result = 1;
        }
        else
        {
            // Otherwise, module 2 isn't valid, and it comes after module 1
            n16Result = -1;
        }
    }
    else
    {
        n16Result = COMPARE(psModule1->hModule, psModule2->hModule);
    }

    return n16Result;
}

/*******************************************************************************
*
*   n16CompareModuleId
*
*******************************************************************************/
static N16 n16CompareModuleId ( void *pvArg1, void *pvArg2 )
{
    PSRM_MODULE_ENTRY_STRUCT psModule =
                    (PSRM_MODULE_ENTRY_STRUCT) pvArg1;
    MODULE_ID *ptId2 = (MODULE_ID *)pvArg2;
    N16 n16Result = N16_MIN;

    if(ptId2 != NULL)
    {
        MODULE_ID tId1;

        tId1 =  psModule->tId;
        if(tId1 >= 0)
        {
            n16Result = tId1 - *ptId2;
            if(n16Result != 0)
            {
                n16Result = -1;
            }
        }
    }

    return n16Result;
}

/*****************************************************************************
*
*   bResetModuleIterator
*
*****************************************************************************/
static BOOLEAN bResetModuleIterator (
    void *pvData,
    void *pvArg
        )
{
    BOOLEAN bSuccess;
    PSRM_MODULE_ENTRY_STRUCT psModule =
                    (PSRM_MODULE_ENTRY_STRUCT)pvData;

    // Reset Module...
    bSuccess = MODULE_bReset(psModule->hModule);
    if(bSuccess == TRUE)
    {
        // Mark this one as in RESET
        psModule->bValid = FALSE;
    }
    else
    {
        // Error!
        SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
            SRM_OBJECT_NAME": Unable to reset this Module.");
    }

    // Keep iteration going
    return TRUE;
}

/*****************************************************************************
*
*   bAllInResetIterator
*
*****************************************************************************/
static BOOLEAN bAllInResetIterator (
    void *pvData,
    void *pvArg
        )
{
    PSRM_MODULE_ENTRY_STRUCT psModule =
                    (PSRM_MODULE_ENTRY_STRUCT)pvData;
    PSRM_MODULE_IN_RESET_STRUCT psInReset =
                    (PSRM_MODULE_IN_RESET_STRUCT)pvArg;

    // Is the MODULE provided the one we are iterating?
    if(psModule->hModule == psInReset->hModule)
    {
        // Set new state
        psModule->bValid = TRUE;
    }

    // If MODULE is in the valid state it is (or has been RESET)
    if(psModule->bValid == FALSE)
    {
        // Then all MODULEs are certainly not in RESET
        psInReset->bAllInReset = FALSE;
    }

    // Keep iteration going
    return TRUE;
}

/*******************************************************************************
*
*   bReleaseModuleIterator
*
*******************************************************************************/
static BOOLEAN bReleaseModuleIterator (
    void *pvData,
    void *pvArg
        )
{
    PSRM_MODULE_ENTRY_STRUCT psModule = (PSRM_MODULE_ENTRY_STRUCT)pvData;

    MODULE_bRelease(psModule->hModule, SMS_OBJECT_RELEASE_BY_PARENT);

    // Keep iteration going
    return TRUE;
}

/*****************************************************************************
*
*   bInitializeObject
*
*****************************************************************************/
static BOOLEAN bInitializeObject (
    SRM_OBJECT_STRUCT *psObj
        )
{
    OSAL_RETURN_CODE_ENUM eReturnCode;

    // (Re)Initialize object...

    // Initialize error code
    psObj->eErrorCode = SRM_ERROR_CODE_NONE;

    // Read mask, thereby clearing it
    SMSU_tMask(&psObj->sEvent);

    // Create MODULE descriptor list (if needed)
    if(psObj->hModuleList == OSAL_INVALID_OBJECT_HDL)
    {
        char acName[OSAL_MAX_OBJECT_NAME_LENGTH_WITH_NULL];

        snprintf(&acName[0], sizeof(acName),
            "%s:ModuleDescList", psObj->pacObjectName);

        eReturnCode =
            OSAL.eLinkedListCreate(
                &psObj->hModuleList,
                &acName[0],
                n16CompareModules,
                OSAL_LL_OPTION_CIRCULAR | OSAL_LL_OPTION_UNIQUE
                    );
        if (eReturnCode != OSAL_SUCCESS)
        {
            // Error!
            return FALSE;
        }
    }

    return TRUE;
}

/*****************************************************************************
*
*   vUninitObject
*
*****************************************************************************/
static void vUninitObject (
    SRM_OBJECT_STRUCT *psObj
        )
{
    SMSAPI_DEBUG_vPrint(SRM_OBJECT_NAME, 4,
        "%s: Object uninitialization (%s)...\n",
        psObj->pacObjectName,
        psObj->bUninitialized == FALSE ? "currently initialized":
        "already uninitialized");

    // only do if we haven't already uninitialized
    if (psObj->bUninitialized == FALSE)
    {
        // don't call this function again, unless re-initialized
        psObj->bUninitialized = TRUE;

        // Destroy module descriptor list if one exists
        if(psObj->hModuleList != OSAL_INVALID_OBJECT_HDL)
        {
            OSAL_RETURN_CODE_ENUM eReturnCode;

            // Remove all entries in the linked list
            OSAL.eLinkedListRemoveAll( psObj->hModuleList,
                (OSAL_LL_RELEASE_HANDLER)SMSO_vDestroy);

            // Delete linked list itself
            eReturnCode =
                OSAL.eLinkedListDelete (
                    psObj->hModuleList );
            if(eReturnCode == OSAL_SUCCESS)
            {
                psObj->hModuleList = OSAL_INVALID_OBJECT_HDL;
            }
        }

        // Destory the async update configuration
        SMSU_vDestroy(&psObj->sEvent);

        // Uninstall Sirius Module Services
        if(psObj->hServicesTask != SMS_INVALID_TASK_HANDLE)
        {
            SMS_TASK_HANDLE hServicesTask;

            SMSAPI_DEBUG_vPrint(SRM_OBJECT_NAME, 4, "Destroy SRM Task...\n");

            // Task shutdown itself will destroy the object (psObj)
            // so there is no need to handle that here. This function
            // is called from the SRM task context, so it will
            // request the SMS-Task to be deleted (shutdown). Once the
            // task is shutdown the last thing it does is destroy the
            // SMS-Object provided at the time of SMS-task creation.
            // This means you must not use psObj anymore after this call!
            hServicesTask = psObj->hServicesTask;
            psObj->hServicesTask = SMS_INVALID_TASK_HANDLE;
            SMST_vUninstall(hServicesTask);

            // Note: Since the thread which is calling this is the same
            // thread being shutdown, it is impossible for psObj to be
            // destroyed before the next event is handled (shutdown).
            // So we are free to return to the caller, handle more
            // events and then finally shutdown.
        }

        SMSAPI_DEBUG_vPrint(SRM_OBJECT_NAME, 4, "SRM is destroyed.\n");
    }

    return;
}

/*****************************************************************************
*
*   vUpdateState
*
*****************************************************************************/
static void vUpdateState (
    SRM_OBJECT_STRUCT *psObj,
    SRM_STATE_ENUM eNextState
        )
{
    SRM_STATE_ENUM eCurrentState;
    // Event to be handled by multiple modules
    SMS_DESCRIPTOR_DISPATCH_STRUCT sDispatcher;

    // Populate with new state information...
    eCurrentState = psObj->eState;

    // Initialize dispatcher
    sDispatcher.eType = SMS_EVENT_INVALID;
    sDispatcher.pvEventArg = NULL;
    sDispatcher.bDispatch = bDispatchToAll;

    switch(eCurrentState)
    {
        case SRM_STATE_INITIAL:
        {
            switch(eNextState)
            {
                case SRM_STATE_INITIAL:
                    eNextState =
                        eTransitionToInitialState(psObj, &sDispatcher);
                break;

                case SRM_STATE_READY:
                    eNextState =
                        eTransitionToReadyState(psObj, &sDispatcher);
                break;

                case SRM_STATE_STOPPED:
                    eNextState =
                        eTransitionToStoppedState(psObj, &sDispatcher);
                break;

                case SRM_STATE_ERROR:
                    eNextState =
                        eTransitionToErrorState(
                            psObj, &sDispatcher, psObj->eErrorCode);
                break;

                case SRM_STATE_INVALID:
                default:
                    // Ignore. Can't go there from here
                    eNextState = eCurrentState;
                break;
            }
        }
        break;

        case SRM_STATE_READY:
        {
            switch(eNextState)
            {
                case SRM_STATE_INITIAL:
                    eNextState =
                        eTransitionToInitialState(psObj, &sDispatcher);
                break;

                case SRM_STATE_READY:
                    // Do nothing.
                break;

                case SRM_STATE_STOPPED:
                    eNextState =
                        eTransitionToStoppedState(psObj, &sDispatcher);
                break;

                case SRM_STATE_ERROR:
                    eNextState =
                        eTransitionToErrorState(
                            psObj, &sDispatcher, psObj->eErrorCode);
                break;

                case SRM_STATE_INVALID:
                default:
                    // Ignore. Can't go there from here
                    eNextState = eCurrentState;
                break;
            }
        }
        break;

        case SRM_STATE_STOPPED:
        {
            // Ignore. Can't go anywhere from here
            eNextState = eCurrentState;
        }
        break;

        case SRM_STATE_ERROR:
        {
            switch(eNextState)
            {
                case SRM_STATE_INITIAL:
                    eNextState =
                        eTransitionToInitialState(psObj, &sDispatcher);
                break;

                case SRM_STATE_STOPPED:
                    eNextState =
                        eTransitionToStoppedState(psObj, &sDispatcher);
                break;

                case SRM_STATE_ERROR:
                    // Do nothing.
                break;

                case SRM_STATE_READY:
                case SRM_STATE_INVALID:
                default:
                    // Ignore. Can't go there from here
                    eNextState = eCurrentState;
                break;
            }
        }
        break;

        case SRM_STATE_INVALID:
        default:
            // Do nothing.
        break;
    }

    // Update internal state
    psObj->eState = eNextState;

    // Anything to dispatch?
    if(sDispatcher.eType != SMS_EVENT_INVALID)
    {
        // Allow every attached/active MODULE to handle this event...
        OSAL.eLinkedListIterate(
            psObj->hModuleList, bModuleIterator, &sDispatcher);
    }

    // Check if next state is different than the current state
    if(eCurrentState != eNextState)
    {
        // Set event mask to indicate state change
        SMSU_tUpdate(&psObj->sEvent, SRM_OBJECT_EVENT_STATE);
    }

    return;
}

/*****************************************************************************
*
*   eTransitionToErrorState
*
*****************************************************************************/
static SRM_STATE_ENUM eTransitionToErrorState (
    SRM_OBJECT_STRUCT *psObj,
    SMS_DESCRIPTOR_DISPATCH_STRUCT *psDispatcher,
    SRM_ERROR_CODE_ENUM eErrorCode
        )
{
    SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
        SRM_OBJECT_NAME": '%s' transitioned to %s due to error %s(%u).",
        psObj->pacSrmName,
        MACRO_TO_STRING(SRM_STATE_ERROR),
        SMSAPI_DEBUG_pacSRMErrorCodeText(eErrorCode),
        eErrorCode
            );

    // Tell all modules we're in ERROR
    if(psDispatcher != NULL)
    {
        psDispatcher->eType = SMS_EVENT_ERROR;
        psDispatcher->pvEventArg = (void *)eErrorCode;
    }

    // Set local error code
    psObj->eErrorCode = eErrorCode;

    // Set current state as ERROR
    // TODO: BREAKPOINT
    return SRM_STATE_ERROR;
}

/*****************************************************************************
*
*   eTransitionToInitialState
*
*****************************************************************************/
static SRM_STATE_ENUM eTransitionToInitialState(
    SRM_OBJECT_STRUCT *psObj,
    SMS_DESCRIPTOR_DISPATCH_STRUCT *psDispatcher
        )
{
    SRM_STATE_ENUM eNextState = SRM_STATE_INITIAL;
    SMS_EVENT_HDL hEvent;
    SMS_EVENT_DATA_UNION *puEventData;

    SMSAPI_DEBUG_vPrint(SRM_OBJECT_NAME, 4, "%s: Transition to %s\n",
        psObj->pacObjectName, MACRO_TO_STRING(SRM_STATE_INITIAL));

    if (TRUE == psObj->bRadioInitialized)
    {
        // Run specific RADIO uninitialization
        RADIO_vUninitializeSrm((SRM_OBJECT)psObj);
        psObj->bRadioInitialized = FALSE;
    }

    // Allocate an event to post initialization
    hEvent =
        SMSE_hAllocateEvent( psObj->hEventHdlr,
            SMS_EVENT_INITIALIZE, &puEventData, SMS_EVENT_OPTION_NONE);
    if (hEvent != SMS_INVALID_EVENT_HDL)
    {
        BOOLEAN bPosted;

        // Post this event
        bPosted = SMSE_bPostEvent(hEvent);
        if(bPosted == FALSE)
        {
            eNextState =
                eTransitionToErrorState(
                    psObj, psDispatcher, SRM_ERROR_CODE_EVENT_POST_ERROR);
        }
    }
    else
    {
        eNextState =
            eTransitionToErrorState(
                psObj, psDispatcher, SRM_ERROR_CODE_EVENT_ALLOCATION_ERROR);
    }

    return eNextState;
}

/*****************************************************************************
*
*   eTransitionToReadyState
*
*****************************************************************************/
static SRM_STATE_ENUM eTransitionToReadyState(
    SRM_OBJECT_STRUCT *psObj,
    SMS_DESCRIPTOR_DISPATCH_STRUCT *psDispatcher
        )
{
    SRM_STATE_ENUM eNextState = SRM_STATE_READY;

    SMSAPI_DEBUG_vPrint(SRM_OBJECT_NAME, 4, "%s: Transition to %s\n",
        psObj->pacObjectName, MACRO_TO_STRING(SRM_STATE_READY));

    // Tell all MODULEs we're ready to associate with them.
    if(psDispatcher != NULL)
    {
        psDispatcher->eType = SMS_EVENT_ASSOCIATE;
        psDispatcher->pvEventArg = (void *)psObj;
    }
    else
    {
        eNextState =
            eTransitionToErrorState(
                psObj, psDispatcher, SRM_ERROR_CODE_EVENT_POST_ERROR);
    }

    return eNextState;
}

/*****************************************************************************
*
*   eTransitionToStoppedState
*
*****************************************************************************/
static SRM_STATE_ENUM eTransitionToStoppedState(
    SRM_OBJECT_STRUCT *psObj,
    SMS_DESCRIPTOR_DISPATCH_STRUCT *psDispatcher
        )
{
    BOOLEAN bLocked;
    SRM_STATE_ENUM eNextState = SRM_STATE_STOPPED;

    SMSAPI_DEBUG_vPrint(SRM_OBJECT_NAME, 4, "%s: Transition to %s\n",
        psObj->pacObjectName, MACRO_TO_STRING(SRM_STATE_STOPPED));

    // Tell owning SMS that we are releasing and instruct it to
    // remove this SRM from it's management.

    // Lock SMS object and remove SRM
    bLocked = SMS_bLock();
    if(bLocked == TRUE)
    {
        BOOLEAN bRemoved;

        // Remove this SRM (immediate)
        bRemoved = SMS_bRemoveSRM((SRM_OBJECT)psObj);
        if(bRemoved == FALSE)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                SRM_OBJECT_NAME": Cannot remove SRM %s.",
                MACRO_TO_STRING(SRM_ERROR_CODE_UNABLE_TO_REMOVE_SRM));
        }

        // Unlock SMS
        SMS_vUnLock();
    }
    else
    {
        // Transition to error state
        vSetError(psObj, SRM_ERROR_CODE_UNABLE_TO_LOCK_SMS);
    }

    return eNextState;
}

/*******************************************************************************
*
*   vEventHandler
*
*******************************************************************************/
static void vEventHandler (
    SRM_OBJECT hSRM,
    SMS_EVENT_HDL hEvent
        )
{
    SRM_OBJECT_STRUCT *psObj = (SRM_OBJECT_STRUCT *)hSRM;
    SMS_EVENT_TYPE_ENUM eEventType;
    SMS_EVENT_DATA_UNION const *puEventData;
    BOOLEAN bLocked;

    // Lock object for exclusive access
    bLocked = SMSO_bLock((SMS_OBJECT)hSRM, OSAL_OBJ_TIMEOUT_INFINITE);
    if(bLocked == FALSE)
    {
        // Error!
        SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
            SRM_OBJECT_NAME": Unable to lock SRM.");
        return;
    }

    // Extract event information
    // Caution! Do not pass puEventData by reference to any other function
    // unless you are absolutely sure you do not end up re-allocating the
    // current event, modify the event data and continue using data from the
    // original event. If you need to do this, first copy what is needed
    // to a local variable and use that instead. This will prevent any
    // issues with potential re-allocation/overwriting of the current event.
    eEventType = SMSE_eGetEvent(hEvent, &puEventData);

    // Process event...
    switch(eEventType)
    {
        // Handle SRM events here...

        // The INITIALIZE event occurs one time only when the task is started
        // and completes before returning to the caller.
        case SMS_EVENT_INITIALIZE:
        {
            vHandleInitEvent(psObj);
        }
        break;

        // This event is received when the Radio-SRM itself becomes ready.
        case SMS_EVENT_RADIO_READY:
        {
            vHandleRadioReadyEvent(psObj,
                puEventData->uSRM.sRadioReady.psDevice);
        }
        break;

        // Someone has requested this object release.
        // Only application or the upper level object (SMS)
        // can actually release the SRM.
        case SMS_EVENT_RELEASE:
        {
            vHandleReleaseEvent(psObj,
                puEventData->uSRM.sRelease.eInitiator);
        }
        break;

        // This is FINAL STOP event
        case SMS_EVENT_STOP:
        {
            vHandleStopEvent(psObj);
        }
        break;

        // Someone requested the SRM to RESET or one of it's MODULEs is in
        // RESET. If an SRM is to RESET then all modes are forcibly RESET as
        // well. If an SRM's MODULE is in RESET then we simply make it READY
        // if this SRM itself is READY.
        case SMS_EVENT_RESET:
        {
            vHandleResetEvent(psObj, puEventData->uSRM.sReset.hModule);
        }
        break;

        // Someone wishes to remove a MODULE from this SRM object's
        // management of it.
        case SMS_EVENT_REMOVE_MODULE:
        {
            vHandleRemoveModuleEvent(psObj,
                puEventData->uSRM.sRemoveModule.hModule);
        }
        break;

        case SMS_EVENT_UPDATE_OBJECT:
        {
            SMSAPI_DEBUG_vPrint(SRM_OBJECT_NAME, 4,
                "%s: SMS_EVENT_UPDATE_OBJECT: EventMask: 0x%X\n",
                psObj->pacObjectName, puEventData->uSRM.sUpdate.tEventMask);

            // Force an update of all that was requested
            SMSU_tUpdate(&psObj->sEvent, puEventData->uSRM.sUpdate.tEventMask);
        }
        break;

        case SMS_EVENT_ERROR:
        {
            SMSAPI_DEBUG_vPrint(SRM_OBJECT_NAME, 4,
                "%s: SMS_EVENT_ERROR: ErrorCode: %d\n",
                psObj->pacObjectName, puEventData->uSRM.sError.eErrorCode);

            // Error!
            vSetError(psObj, puEventData->uSRM.sError.eErrorCode);
        }
        break;

        case SMS_EVENT_MODIFY_EVENT_MASK:
        {
            const SMS_EVENT_SRM_EVENT_MASK_STRUCT sSrmEventMask =
                puEventData->uSRM.sSrmEventMask;
            BOOLEAN bSuccess;

            SMSAPI_DEBUG_vPrint(SRM_OBJECT_NAME, 4,
                "%s: SMS_EVENT_MODIFY_EVENT_MASK: "
                "tEventMask: 0x%X, eModification: %d\n",
                psObj->pacObjectName, sSrmEventMask.tEventMask,
                sSrmEventMask.eModification);

            bSuccess = bProcessSrmEventMaskChange(psObj, &sSrmEventMask);
            if (bSuccess == FALSE)
            {
                // Error!
                SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    SRM_OBJECT_NAME": Could not process decoder"
                    " event mask change:");

                // Error!
                vSetError(psObj, SRM_ERROR_CODE_GENERAL);
            }
        }
        break;

        default:
        {
            // Un-handled event, pass on to RADIO specific handler
            RADIO_bSrmEventHandler(hSRM, eEventType, puEventData);
        }
        break;
    }

    // Notify of any change via any registered callback which may be present
    SMSU_bNotify(&psObj->sEvent);

    // Check if we need to destroy the object. This means not only
    // have we been told to STOP but also no existing modes of operation exist.
    if(psObj->eState == SRM_STATE_STOPPED)
    {
        // Uninitialize Object
        vUninitObject(psObj);
    }

    // Unlock object, any modifications to be made to this object
    // have already been made.
    SMSO_vUnlock((SMS_OBJECT)hSRM);

    // Event handled, simply return
    return;
}

/*******************************************************************************
*
*   bModuleIterator
*
*******************************************************************************/
static BOOLEAN bModuleIterator (
    void *pvData,
    void *pvArg
        )
{
    BOOLEAN bContinue = FALSE;
    PSRM_MODULE_ENTRY_STRUCT psModule =
                    (PSRM_MODULE_ENTRY_STRUCT)pvData;
    SMS_DESCRIPTOR_DISPATCH_STRUCT *psDispatcher =
        (SMS_DESCRIPTOR_DISPATCH_STRUCT *)pvArg;

    // Verify inputs (only the psDispatcher needs to be valid)
    if(psDispatcher != NULL)
    {
        // Check if dispatch function exists
        if(psDispatcher->bDispatch != NULL)
        {
            MODULE_OBJECT hModule = MODULE_INVALID_OBJECT;

            if (psModule != NULL)
            {
                if (psModule->bValid == TRUE)
                {
                    hModule = psModule->hModule;
                }
            }

            // Run dispatch function
            bContinue = psDispatcher->bDispatch(
                hModule, psDispatcher->eType,
                psDispatcher->pvEventArg
                    );
        }
    }

    return bContinue;
}

/*******************************************************************************
*
*   bDispatchToAll
*
*******************************************************************************/
static BOOLEAN bDispatchToAll (
    void *pvDescriptor,
    SMS_EVENT_TYPE_ENUM eType,
    void *pvEventArg
        )
{
    BOOLEAN bResult = TRUE;
    MODULE_OBJECT hModule = (MODULE_OBJECT)pvDescriptor;

    // Verify inputs
    if (hModule == MODULE_INVALID_OBJECT)
    {
        // Error!
        return FALSE;
    }

    // Post event to this MODULE...
    switch(eType)
    {
        case SMS_EVENT_ASSOCIATE:
        {
            bResult =
                bAssociateWithModule((SRM_OBJECT_STRUCT *)pvEventArg, hModule);
        }
        break;

        case SMS_EVENT_ERROR:
        {
            // We can retrieve the error code from the SRM, but since
            // there is no intersection of SRM error codes and Module
            // error codes, this really doesn't do us any good.
            // But maybe in the future it will. For now we simply pass
            // along MODULE_ERROR_CODE_SRM_ERROR at let them check the
            // SRM error.
            // SRM_ERROR_CODE_ENUM eErrorCode = (SRM_ERROR_CODE_ENUM)pvEventArg;

            // Inform Module of an error
            MODULE_bSetError(hModule, MODULE_ERROR_CODE_SRM_ERROR);
        }
        break;

        default:
            // Unhandled event (ok)
        break;
    }

    return bResult;
}

/*******************************************************************************
*
*   vSetError
*
*******************************************************************************/
static void vSetError(
    SRM_OBJECT_STRUCT *psObj,
    SRM_ERROR_CODE_ENUM eErrorCode
        )
{
    // Set local object error code
    psObj->eErrorCode = eErrorCode;

    // Transition to error state
    vUpdateState(psObj, SRM_STATE_ERROR);

    return;
}

/*****************************************************************************
 *
 *  bProcessSrmEventMaskChange
 *
 *****************************************************************************/
static BOOLEAN bProcessSrmEventMaskChange(
    SRM_OBJECT_STRUCT *psObj,
    SMS_EVENT_SRM_EVENT_MASK_STRUCT const *psSrmEventMask
        )
{

    BOOLEAN bValid, bSuccess = FALSE;

    // Verify SMS Object is valid
    bValid = SMSO_bValid((SMS_OBJECT)psObj);
    if(bValid == TRUE)
    {

        // Make the change to the event request mask
        // and see if it worked
        bSuccess = SMSU_bModifyRequestMask(
            &psObj->sEvent,
            NULL,
            NULL,
            psSrmEventMask->tEventMask,
            psSrmEventMask->eModification);
    }

    return bSuccess;
}

/*****************************************************************************
 *
 *   bAssociateWithModule
 *
 *****************************************************************************/
static BOOLEAN bAssociateWithModule (
    SRM_OBJECT_STRUCT *psObj,
    MODULE_OBJECT hModule
        )
{
    BOOLEAN bResult = TRUE;

    // If not ready now, ASSOCIATE event will be posted
    // when SRM becomes ready.
    if (SRM_STATE_READY == psObj->eState)
    {
        // Tell Module we are ready to associate with it
        bResult = MODULE_bAssociate(hModule, (SRM_OBJECT)psObj,
            psObj->pacSrmName, psObj->psDevice);
        if (TRUE == bResult)
        {
            // Now we assume the SRM's resources are utilized by MODULE.
            // Increment reference counter here. To decrement reference counter
            // MODULE shall call SRM_bRelease() to tell the SRM
            // that the MODULE is not utilizing the SRM anymore.

            SMSAPI_DEBUG_vPrint(SRM_OBJECT_NAME, 4,
                "%s: ASSOCIATE request is posted to MODULE.\n",
                psObj->pacObjectName);

            psObj->un16RefCounter++;
        }
        else
        {
            // Error!
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                SRM_OBJECT_NAME": Unable to associate with MODULE.");

            // Transition to error state
            vSetError(psObj, SRM_ERROR_CODE_EVENT_POST_ERROR);
        }
    }

    return bResult;
}

/* Shutdown management */

/*****************************************************************************
 *
 *   bObjectBusy
 *
 *****************************************************************************/
static BOOLEAN bObjectBusy (
    SRM_OBJECT_STRUCT *psObj
        )
{
    OSAL_RETURN_CODE_ENUM eReturnCode;
    UN32 un32NumItems = 0;

    // Check if this object is currently being released
    if (0 == (psObj->tCurrentModes & SMS_MODE_STOPPING))
    {
        SMSAPI_DEBUG_vPrint(SRM_OBJECT_NAME, 4,
            "%s: BUSY: Not being released.\n", psObj->pacObjectName);
        return TRUE;
    }

    //  Check reference counter
    if (0 != psObj->un16RefCounter)
    {
        SMSAPI_DEBUG_vPrint(SRM_OBJECT_NAME, 4,
            "%s: BUSY: Reference Counter: %u\n", psObj->pacObjectName,
            psObj->un16RefCounter);
        return TRUE;
    }

    // Check if we have any MODULEs attached
    eReturnCode = OSAL.eLinkedListItems(psObj->hModuleList, &un32NumItems);
    if (OSAL_SUCCESS != eReturnCode)
    {
        SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
            SRM_OBJECT_NAME": Could not get number of MODULEs: %d (%s).",
            eReturnCode, OSAL.pacGetReturnCodeName(eReturnCode));

        // Transition to ERROR state
        vSetError(psObj, SRM_ERROR_CODE_MODULE_LIST_ERROR);
        return TRUE;
    }
    else if (0 != un32NumItems)
    {
        SMSAPI_DEBUG_vPrint(SRM_OBJECT_NAME, 4,
            "%s: BUSY: Still has %u MODULE(s) attached. "
            "Waiting for REMOVE event.\n",
            psObj->pacObjectName, un32NumItems);
        return TRUE;
    }

    SMSAPI_DEBUG_vPrint(SRM_OBJECT_NAME, 4,
        "%s: SRM is ready for shutdown.\n", psObj->pacObjectName);

    return FALSE;
}

/*****************************************************************************
*
*   bTryStop
*
*****************************************************************************/
static BOOLEAN bTryStop (
    SRM_OBJECT_STRUCT *psObj
        )
{
    BOOLEAN bResult;

    // Check if this object is still utilized
    bResult = bObjectBusy(psObj);
    if (TRUE == bResult)
    {
        // Cannot stop now.
        // Waiting for shutdown completion.
        return FALSE;
    }

    // Now this object is ready for shutdown.
    // But we should dispatch all remaining events.
    // by posting of special event which shall be dispatched
    // last from the queue.
    bResult = bPostFinalStop(psObj);
    if (FALSE == bResult)
    {
        // Error!
        vSetError(psObj, SRM_ERROR_CODE_EVENT_POST_ERROR);
        return FALSE;
    }

    // Now waiting for FINAL STOP event.

    return TRUE;
}

/*****************************************************************************
*
*   bInitiateRelease
*
*****************************************************************************/
static BOOLEAN bInitiateObjectRelease (
    SRM_OBJECT_STRUCT *psObj,
    SMS_OBJECT_RELEASE_INITIATOR_ENUM eInitiator
        )
{
    BOOLEAN bResult = FALSE;
    SMS_EVENT_HDL hEvent;
    SMS_EVENT_DATA_UNION *puEventData;

    do
    {
        // Allocate an event
        hEvent = SMSE_hAllocateEvent(psObj->hEventHdlr, SMS_EVENT_RELEASE,
            &puEventData, SMS_EVENT_OPTION_NONE);
        if (SMS_INVALID_EVENT_HDL == hEvent)
        {
            // Error!
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                SRM_OBJECT_NAME": Cannot allocate event.");
            break;
        }

        // Populate event with caller's parameters
        puEventData->uSRM.sRelease.eInitiator = eInitiator;

        // Post event
        bResult = SMSE_bPostEvent(hEvent);
        if (FALSE == bResult)
        {
            // Error!
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                SRM_OBJECT_NAME": Cannot post event.");
            break;
        }

        SMSAPI_DEBUG_vPrint(SRM_OBJECT_NAME, 4,
            "%s: RELEASE event is posted.\n",
            psObj->pacObjectName);

    } while (FALSE);

    return bResult;
}

/*****************************************************************************
*
*   bReleaseModules
*
*****************************************************************************/
static BOOLEAN bReleaseModules(
    SRM_OBJECT_STRUCT *psObj
        )
{
    OSAL_RETURN_CODE_ENUM eReturnCode;

    // Tell all MODULEs to release...
    eReturnCode =
        OSAL.eLinkedListIterate(
            psObj->hModuleList,
            bReleaseModuleIterator,
            (void *)NULL);
    if (OSAL_SUCCESS == eReturnCode)
    {
        SMSAPI_DEBUG_vPrint(SRM_OBJECT_NAME, 4,
            "%s: MODULE(s) release is requested.\n",
            psObj->pacObjectName);
    }
    else if (OSAL_NO_OBJECTS == eReturnCode)
    {
        SMSAPI_DEBUG_vPrint(SRM_OBJECT_NAME, 4,
            "%s: There are no MODULEs to release.\n",
            psObj->pacObjectName);
    }
    else
    {
        // Error!
        SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
            SRM_OBJECT_NAME": MODULEs iteration failed: %d (%s).\n",
            eReturnCode, OSAL.pacGetReturnCodeName(eReturnCode));

        return FALSE;
    }

    return TRUE;
}

/*****************************************************************************
*
*   bPostFinalStop
*
*****************************************************************************/
static BOOLEAN bPostFinalStop(
    SRM_OBJECT_STRUCT *psObj
        )
{
    BOOLEAN bResult = FALSE;

    bResult = SMSE_bPostSignal(psObj->hEventHdlr,
        SMS_EVENT_STOP, SMS_EVENT_OPTION_DEFERRED);
    if (TRUE == bResult)
    {
        SMSAPI_DEBUG_vPrint(SRM_OBJECT_NAME, 4,
            "%s: FINAL STOP event is posted.\n", psObj->pacObjectName);
    }
    else
    {
        // Error!
        SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
            SRM_OBJECT_NAME": Unable to post FINAL STOP event.");
    }

    return bResult;
}

/* Event handlers */

/*****************************************************************************
*
*   vHandleInitEvent
*
*****************************************************************************/
static void vHandleInitEvent(
    SRM_OBJECT_STRUCT *psObj
        )
{
    BOOLEAN bOk;

    SMSAPI_DEBUG_vPrint(SRM_OBJECT_NAME, 4,
        "%s: SMS_EVENT_INITIALIZE\n", psObj->pacObjectName);

    do
    {
        // Check stopping flag indicating
        // the object is currently being released
        if (0 != (psObj->tCurrentModes & SMS_MODE_STOPPING))
        {
            SMSAPI_DEBUG_vPrint(SRM_OBJECT_NAME, 4,
                "%s: Object is currently being released. Ignore event.\n",
                psObj->pacObjectName);
            break;
        }

        // Initialize the SRM object
        bOk = bInitializeObject(psObj);
        if (FALSE == bOk)
        {
            // Transition to error state
            vSetError(psObj, SRM_ERROR_CODE_INITIALIZATION_ERROR);
            break;
        }

        SMSAPI_DEBUG_vPrint(SRM_OBJECT_NAME, 4,
            "%s: Object is initialized.\n",
            psObj->pacObjectName);

        // Has the RADIO already been initialized?
        if (FALSE == psObj->bRadioInitialized)
        {
            // Perform RADIO specific initializations.
            psObj->bRadioInitialized =
                RADIO_bInitializeSrm(
                    (SRM_OBJECT)psObj,
                    psObj->pacSrmName,
                    psObj->pacDriverName,
                    &psObj->tCapabilities,
                    psObj->hEventHdlr);
            if (FALSE == psObj->bRadioInitialized)
            {
                // Transition to error state
                vSetError(psObj, SRM_ERROR_CODE_INITIALIZATION_ERROR);
                break;
            }
        }

        SMSAPI_DEBUG_vPrint(SRM_OBJECT_NAME, 4,
            "%s: RADIO initialization is requested. "
                "Waiting for RADIO_READY event.\n",
                psObj->pacObjectName);

        // Clear RESET mode
        psObj->tCurrentModes &= ~SMS_MODE_RESET;

    } while (FALSE);

    return;
}

/*****************************************************************************
*
*   vHandleRadioReadyEvent
*
*****************************************************************************/
static void vHandleRadioReadyEvent(
    SRM_OBJECT_STRUCT *psObj,
    FILE *psDevice
        )
{
    SMSAPI_DEBUG_vPrint(SRM_OBJECT_NAME, 4,
        "%s: SMS_EVENT_RADIO_READY: psDevice: %p\n",
        psObj->pacObjectName, psDevice);

    do
    {
        // Try to stop if the object is currently
        // being released and ready for shutdown.
        if (0 != (psObj->tCurrentModes & SMS_MODE_STOPPING))
        {
            SMSAPI_DEBUG_vPrint(SRM_OBJECT_NAME, 4,
                "%s: Object is currently being released."
                " Try to initiate object stop.\n",
                psObj->pacObjectName);

            bTryStop(psObj);
            break;
        }

        // Record device handle
        psObj->psDevice = psDevice;

        // Transition to the READY state
        // All attached Modules will be associated
        vUpdateState(psObj, SRM_STATE_READY);

    } while (FALSE);

    return;
}

/*****************************************************************************
*
*   vHandleReleaseEvent
*
*****************************************************************************/
static void vHandleReleaseEvent(
    SRM_OBJECT_STRUCT *psObj,
    SMS_OBJECT_RELEASE_INITIATOR_ENUM eInitiator
        )
{
    SMSAPI_DEBUG_vPrint(SRM_OBJECT_NAME, 4,
        "%s: SMS_EVENT_RELEASE: eInitiator: %d\n",
        psObj->pacObjectName, eInitiator);

    if ((SMS_OBJECT_RELEASE_BY_APP == eInitiator) ||
        (SMS_OBJECT_RELEASE_BY_PARENT == eInitiator))
    {
        SMSAPI_DEBUG_vPrint(SRM_OBJECT_NAME, 4,
            "%s: Handle APPLICATION initiated release...\n",
            psObj->pacObjectName);

        vHandleAppInitiatedRelease(psObj, eInitiator);
    }
    else
    {
        SMSAPI_DEBUG_vPrint(SRM_OBJECT_NAME, 4,
            "%s: Handle GENERAL release...\n",
            psObj->pacObjectName);

        vHandleGeneralRelease(psObj);
    }

    return;
}

/*****************************************************************************
*
*   vHandleAppInitiatedRelease
*
*****************************************************************************/
static void vHandleAppInitiatedRelease(
    SRM_OBJECT_STRUCT *psObj,
    SMS_OBJECT_RELEASE_INITIATOR_ENUM eInitiator
        )
{
    // This is release request initiated by the application
    // directly or by the parent object release.

    // Check stopping flag indicating the object is currently being released
    if (0 == (psObj->tCurrentModes & SMS_MODE_STOPPING))
    {
        // This is the first release request initiated
        // by the Application or by the Parent object (MODULE)
        psObj->tCurrentModes |= SMS_MODE_STOPPING;
    }
    else
    {
        if (SMS_OBJECT_RELEASE_BY_APP == eInitiator)
        {
            // This never must happen!
            // This means incorrect and dangerous API usage!
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                SRM_OBJECT_NAME": OBJECT IS ALREADY BEING RELEASED "
                "DIRECTLY BY THE APPLICATION OR BY THE PARENT OBJECT!");
        }
        else if (SMS_OBJECT_RELEASE_BY_PARENT == eInitiator)
        {
            // This is Parent initiated release occurred after
            // application initiated release, so just ignore it.

            SMSAPI_DEBUG_vPrint(SRM_OBJECT_NAME, 4,
                "%s: Object is already being released. "
                "PARENT initiated release is ignored.\n",
                psObj->pacObjectName);
        }
        else
        {
            // Error!
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                SRM_OBJECT_NAME": Incorrect release request.");
        }

        return;
    }

    // Initiate attached MODULEs release.
    // It's Ok if there are no MODULEs attached.
    // Associated MODULEs shall not be released
    // until the SRM unassociates them later.
    // If fail, will hope that MODULEs
    // are released directly by the Application.
    bReleaseModules(psObj);

    // Perform common release routine
    vHandleGeneralRelease(psObj);

    return;
}

/*****************************************************************************
*
*   vHandleGeneralRelease
*
*****************************************************************************/
static void vHandleGeneralRelease(
    SRM_OBJECT_STRUCT *psObj
        )
{
    // Check for unnecessary release request
    if (0 == psObj->un16RefCounter)
    {
        // Error! This means incorrect logic somewhere inside the SMS.
        SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
            SRM_OBJECT_NAME": Unnecessary release!");
        return;
    }

    // Now reference counter can be decremented
    psObj->un16RefCounter--;

    // Try to stop if the object is currently
    // being released and ready for shutdown.
    bTryStop(psObj);

    return;
}

/*****************************************************************************
*
*   vHandleStopEvent
*
*****************************************************************************/
static void vHandleStopEvent(
    SRM_OBJECT_STRUCT *psObj
        )
{
    SMSAPI_DEBUG_vPrint(SRM_OBJECT_NAME, 4,
        "%s: SMS_EVENT_STOP\n", psObj->pacObjectName);

    // Now everything is uninitialized.
    // So the object can be safely destroyed.

    // Run specific RADIO un-initialization
    if (TRUE == psObj->bRadioInitialized)
    {
        RADIO_vUninitializeSrm((SRM_OBJECT)psObj);
        psObj->bRadioInitialized = FALSE;
    }

    // Stop the SRM
    vUpdateState(psObj, SRM_STATE_STOPPED);

    return;
}

/*****************************************************************************
*
*   vHandleRemoveModuleEvent
*
*****************************************************************************/
static void vHandleRemoveModuleEvent(
    SRM_OBJECT_STRUCT *psObj,
    MODULE_OBJECT hModule
        )
{
    OSAL_RETURN_CODE_ENUM eReturnCode;
    SRM_MODULE_ENTRY_STRUCT sSearchCriteria;
    BOOLEAN bResult = FALSE;
    OSAL_LINKED_LIST_ENTRY hEntry = OSAL_INVALID_LINKED_LIST_ENTRY;
    PSRM_MODULE_ENTRY_STRUCT psModule = NULL;

    SMSAPI_DEBUG_vPrint(SRM_OBJECT_NAME, 4,
        "%s: SMS_EVENT_REMOVE_MODULE: Module: %p\n",
        psObj->pacObjectName, hModule);

    do
    {
        // fill the search criteria struct with input data
        sSearchCriteria.hModule = hModule;

        // Find this Module's entry (if it exists)
        eReturnCode = OSAL.eLinkedListSearch(
            psObj->hModuleList, &hEntry, &sSearchCriteria);
        if (OSAL_SUCCESS != eReturnCode)
        {
            // Error!
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                SRM_OBJECT_NAME": Unable to locate MODULE entry: %d (%s).",
                eReturnCode, OSAL.pacGetReturnCodeName(eReturnCode));
            break;
        }

        psModule = (PSRM_MODULE_ENTRY_STRUCT)OSAL.pvLinkedListThis(hEntry);

        // Remove this MODULE from the list
        eReturnCode = OSAL.eLinkedListRemove(hEntry);
        if (OSAL_SUCCESS != eReturnCode)
        {
            // Error!
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                SRM_OBJECT_NAME": Could not remove Module entry: %d (%s).",
                eReturnCode, OSAL.pacGetReturnCodeName(eReturnCode));

            // Transition to ERROR state
            vSetError(psObj, SRM_ERROR_CODE_MODULE_LIST_ERROR);
            break;
        }

        // Destroy this entry
        SMSO_vDestroy((SMS_OBJECT)psModule);

        SMSAPI_DEBUG_vPrint(SRM_OBJECT_NAME, 4,
            "%s: Removed Module entry from list.\n",
            psObj->pacObjectName);

        // Post UNASSOCIATE event to the MODULE
        // to confirm that the MODULE is removed.
        bResult = MODULE_bUnassociate(hModule);
        if (FALSE == bResult)
        {
            // Transition to ERROR state
            vSetError(psObj, SRM_ERROR_CODE_EVENT_POST_ERROR);
            break;
        }

        SMSAPI_DEBUG_vPrint(SRM_OBJECT_NAME, 4,
            "%s: UNASSOCIATE is posted to MODULE.\n",
            psObj->pacObjectName);

    } while (FALSE);

    // Try to stop if the object is currently
    // being released and ready for shutdown
    bTryStop(psObj);

    return;
}

/*****************************************************************************
*
*   vHandleResetEvent
*
*****************************************************************************/
static void vHandleResetEvent(
    SRM_OBJECT_STRUCT *psObj,
    MODULE_OBJECT hModule
        )
{
    OSAL_RETURN_CODE_ENUM eReturnCode;

    SMSAPI_DEBUG_vPrint(SRM_OBJECT_NAME, 4,
        "%s: SMS_EVENT_RESET: hModule: %p\n",
        psObj->pacObjectName, hModule);

    // Check stopping flag indicating
    // the object is currently being released
    if (0 != (psObj->tCurrentModes & SMS_MODE_STOPPING))
    {
        SMSAPI_DEBUG_vPrint(SRM_OBJECT_NAME, 4,
            "%s: Object is currently being released. Ignore event.\n",
            psObj->pacObjectName);
        return;
    }

    // If the provided MODULE is non-valid, then it is this SRM
    // which is being instructed to reset.
    if (MODULE_INVALID_OBJECT == hModule)
    {
        // Indicate we have been instructed to RESET
        psObj->tCurrentModes |= SMS_MODE_RESET;

        SMSAPI_DEBUG_vPrint(SRM_OBJECT_NAME, 4,
            "%s: SRM Reset. Updated SRM modes: 0x%X. "
            "Request all attached MODULEs to reset.\n",
            psObj->pacObjectName, psObj->tCurrentModes);

        // Tell all attached MODULEs to RESET...
        // We cannot reset ourself until all attached MODULEs
        // are in RESET themselves. So we now tell all attached MODULEs
        // to RESET themselves. They will let us know when they
        // have completed the reset (back to INITIAL_STATE)
        eReturnCode =
            OSAL.eLinkedListIterate(
                psObj->hModuleList,
                bResetModuleIterator,
                (void *)NULL);
        if (eReturnCode == OSAL_NO_OBJECTS)
        {
            SMSAPI_DEBUG_vPrint(SRM_OBJECT_NAME, 4,
                "%s: No MODULEs attached. Go to INITIAL state.\n",
                psObj->pacObjectName);

            // Transition to INITIAL now (no Modules)
            vUpdateState(psObj, SRM_STATE_INITIAL);
        }
        else if (eReturnCode != OSAL_SUCCESS)
        {
            // Error!
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                SRM_OBJECT_NAME": MODULEs iteration failed: %d (%s).\n",
                eReturnCode, OSAL.pacGetReturnCodeName(eReturnCode));

            // Transition to ERROR state
            vSetError(psObj, SRM_ERROR_CODE_MODULE_LIST_ERROR);
        }
    }
    // We have been told a specific MODULE is in RESET.
    // If we ourself are not already resetting then we can
    // just simply let the specified MODULE know we are READY.
    else if ((psObj->tCurrentModes & SMS_MODE_RESET) != SMS_MODE_RESET)
    {
        bAssociateWithModule(psObj, hModule);
    }
    // Ok, so we have been told a specific MODULE is in RESET
    // but we are resetting ourself. This means we need to wait
    // until ALL MODULEs are RESET themselves (in the INITIAL
    // state) before we can RESET ourself which will have the effect
    // making all these MODULEs READY once we are READY.
    else
    {
        SRM_MODULE_IN_RESET_STRUCT sInReset;

        SMSAPI_DEBUG_vPrint(SRM_OBJECT_NAME, 4,
            "%s: SRM already has RESET mode. "
            "Determine if all modules are valid.\n",
            psObj->pacObjectName);

        // Populate iteration struct
        sInReset.hModule = hModule;
        sInReset.bAllInReset = TRUE;

        // Check if all attached MODULEs are in the INITIAL
        // state (i.e. they have been and are in RESET).
        eReturnCode =
            OSAL.eLinkedListIterate(
                psObj->hModuleList,
                bAllInResetIterator,
                (void *)&sInReset
                    );
        if((eReturnCode == OSAL_NO_OBJECTS) ||
            ((eReturnCode == OSAL_SUCCESS) &&
                (sInReset.bAllInReset == TRUE)))
        {
            SMSAPI_DEBUG_vPrint(SRM_OBJECT_NAME, 4,
                "%s: Modules iterated. ReturnCode: %d, "
                "all valid: %d. Go to INITIAL state.\n",
                psObj->pacObjectName, eReturnCode,
                sInReset.bAllInReset);

            // Transition to INITIAL now (no MODULEs exist
            // or all MODULEs are currently in RESET).
            vUpdateState(psObj, SRM_STATE_INITIAL);
        }
        else if(eReturnCode != OSAL_SUCCESS)
        {
            // Error!
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                SRM_OBJECT_NAME": MODULEs iteration failed: %d (%s).\n",
                eReturnCode, OSAL.pacGetReturnCodeName(eReturnCode));

            // Transition to error state
            vSetError(psObj, SRM_ERROR_CODE_MODULE_LIST_ERROR);
        }
    }

    // Don't worry if we were told a MODULE was reset but we are
    // not READY then when we become READY we will inform all
    // attached MODULEs anyway.

    return;
}
