/******************************************************************************/
/*                    Copyright (c) Sirius XM Radio, Inc.                     */
/*                            All Rights Reserved                             */
/*         Licensed Materials - Property of Sirius XM Radio, Inc.             */
/******************************************************************************/
/*******************************************************************************
 *
 * DESCRIPTION
 *
 * This is the Sports Service Interface header file.
 *
 ******************************************************************************/

#ifndef SPORTS_SERVICE_INTERFACE_H_
#define SPORTS_SERVICE_INTERFACE_H_

#include "sms_api.h"
#include "sports_service_db_constants.h"
#include "sql_interface_obj.h"


  /**********************/
 /** GLOBAL CONSTANTS **/
/**********************/

#define SS_MAX_SPORT_NAME_SYMBOL_COUNT (32)
#define SS_MAX_LABEL_TEXT_SYMBOL_COUNT (32)
#define SS_MAX_TABLE_TITLE_SYMBOL_COUNT (32)
#define SS_MAX_AFFILIATION_NAME_SYMBOL_COUNT (64)

#define SPORTS_SERVICE_OVERRIDE_LABELSIZE_BITLEN (5)
#define SPORTS_SERVICE_OVERRIDE_LABELID_BITLEN (10)

// These two must match the protocol. 0 integer, 1 text
#define SS_LABEL_DATATYPE_INTEGER (0)
#define SS_LABEL_DATATYPE_TEXT (1)

  /*********************/
 /** GLOBAL TYPEDEFS **/
/*********************/

/* Used for populating database */
typedef struct sports_service_data_instance_struct
{
    // The following fields are needed to be able to add an instance of data
    // to the database:
    UN16 un16TableId;
    UN8 un8TableVer;
    UN32 un32TdefKey;
    UN8 un8Tclass;
    UN16 un16AffiliateId;
    UN16 un16Epoch;
    BOOLEAN bInSeason;
    char InstanceTitle[SS_MAX_TABLE_TITLE_SYMBOL_COUNT];
    UN32 un32Crc;
    N64 n64InstanceId;
    UN8    un8RowCount;
    UN32 un32ColumnCount;
    UN32 un32UTCsec;
    BOOLEAN bTableComplete;

}SPORTS_SERVICE_DATA_INSTANCE_STRUCT;

typedef struct sports_service_highband_label_info_struct
{
    UN8 un8LabelId;

    UN8 un8Dtype;    // data type of field as received (0=int, 1=baudot text)

    UN8 un8LbType;    // data context

    BOOLEAN    bIsDmodPresent;    //
    UN8 un8Dmod;        // data modifier

    UN8 un8LPriority;    // label priority

    UN16 un16LabelSize;    // Label data size: bits+1 for int,
                        // max of (LBSIZE+1)*5 bits for baudot

    BOOLEAN bIsLabelTxtPresent;
    char LabelText[SS_MAX_LABEL_TEXT_SYMBOL_COUNT];

    BOOLEAN bIsExtDataPresent;
    OSAL_BUFFER_HDL ExtData;// extra data must be allocated
                            // its own new block if present

}SPORTS_SERVICE_HIGHBAND_LABEL_INFO_STRUCT;

typedef struct sports_service_highband_tdef_info_struct
{
    // Each entry is referenced by its index (i.e. TableID) in the table

    // byte available for status flags
    UN8 IsValid;

    // Table definition version number
    UN8  un8TableVer;

    // Table class information
    UN8     un8TableClass;

    // count of labels in this table definition
    UN8  un8LabelCount;

    // for comparing and detecting changes between OTA with stored
    UN32 CRC;

    // handle to the label defnitions linked list
    // of SPORTS_SERVICE_HIGHBAND_LABEL_INFO_STRUCTs
    OSAL_OBJECT_HDL hLabelInfoList;

} SPORTS_SERVICE_HIGHBAND_TDEF_INFO_STRUCT;

typedef struct sports_service_table_transaction_struct
{
    SQL_PREPARED_STATEMENT_HANDLE hStatement;
    SQL_BIND_PARAMETER_STRUCT *psParams;
    UN32 un32Crc;
    UN32 un32ColumnCount;
}SPORTS_SERVICE_TABLE_TRANSACTION_STRUCT;

typedef struct sports_service_label_override_struct
{
    BOOLEAN bIsOverridden;
    UN8 un8OverriddenSize;
} SPORTS_SERVICE_LABEL_OVERRIDE_STRUCT;

typedef struct sports_service_tdef_add_struct
{
    UN32 un32TdefId;
    SPORTS_SERVICE_HIGHBAND_TDEF_INFO_STRUCT sTdefInfo;

} SPORTS_SERVICE_TDEF_ADD_STRUCT;

// Interface handle
typedef void * SPORTS_SERVICE_INTERFACE_OBJECT;
#define SPORTS_SERVICE_INTERFACE_INVALID_OBJECT \
            (SPORTS_SERVICE_INTERFACE_OBJECT)NULL

  /*****************************/
 /** GLOBAL STRUCT TEMPLATES **/
/*****************************/

// The sports service manager interface APIs
typedef struct sports_service_mgr_interface_struct
{
    BOOLEAN (*bBeginTableUpdate)(
        SPORTS_SERVICE_OBJECT hSportsService,
        SPORTS_SERVICE_DATA_INSTANCE_STRUCT *psTableInstance,
        SPORTS_SERVICE_TABLE_TRANSACTION_STRUCT *psTransaction,
        BOOLEAN bIsPrimaryTable,
        char *acSavePointName
            );

    BOOLEAN (*bEndTableUpdate)(
        SPORTS_SERVICE_OBJECT hSportsService,
        SPORTS_SERVICE_DATA_INSTANCE_STRUCT *psTableInstance,
        SPORTS_SERVICE_TABLE_TRANSACTION_STRUCT *psTransaction
            );

    BOOLEAN (*bBeginRowUpdate)(
        SPORTS_SERVICE_OBJECT hSportsService,
        SPORTS_SERVICE_TABLE_TRANSACTION_STRUCT *psTransaction
            );

    BOOLEAN (*bEndRowUpdate)(
        SPORTS_SERVICE_OBJECT hSportsService,
        SPORTS_SERVICE_TABLE_TRANSACTION_STRUCT *psTransaction
            );

    BOOLEAN (*bGetLastRowId)(
            SPORTS_SERVICE_OBJECT hSportsService,
            N64 *n64RowId
            );

    BOOLEAN (*bUpdateColumnInt)(
        SPORTS_SERVICE_OBJECT hSportsService,
        SPORTS_SERVICE_TABLE_TRANSACTION_STRUCT *psTransaction,
        int iColumn,
        UN8 un8LPriority,
        UN8 un8LType,
        int iValue
            );

    BOOLEAN (*bUpdateColumnString)(
        SPORTS_SERVICE_OBJECT hSportsService,
        SPORTS_SERVICE_TABLE_TRANSACTION_STRUCT *psTransaction,
        int iColumn,
        UN8 un8LPriority,
        UN8 un8LType,
        STRING_OBJECT hString
            );

    BOOLEAN (*bDbgDumpTable)(
        SPORTS_SERVICE_OBJECT hSportsService,
        char *TableName
            );

    BOOLEAN (*bDbgExecuteSqlCmd)(
        SPORTS_SERVICE_OBJECT hSportsService,
        char *SqlStatement
            );

    BOOLEAN (*bDbgDumpDefs)(
        SPORTS_SERVICE_OBJECT hSportsService,
        char *DefInfo
            );

    BOOLEAN (*bStartServiceConfigUpdate)(
            SPORTS_SERVICE_OBJECT hSportsService,
            UN8 un8CfgVer,
            UN32 un32Crc
            );

    BOOLEAN (*bEndServiceConfigUpdate)(
            SPORTS_SERVICE_OBJECT hSportsService,
            BOOLEAN bUpdateOk,
            UN8 un8CfgVer,
            UN32 un32Crc
            );

    BOOLEAN (*bIsCorrectServiceConfigVersion)(
            SPORTS_SERVICE_OBJECT hSportsService,
            UN8 un8CfgVer
            );

    BOOLEAN (*bAddSportsNameToDB)(
            SPORTS_SERVICE_OBJECT hSportsService,
            STRING_OBJECT hSportsName,
            UN8 un8SportId
            );

    BOOLEAN (*bAddTdefSportMappingToDB)(
            SPORTS_SERVICE_OBJECT hSportsService,
            UN16 un16TableID,
            UN8 un8SportId
            );

    BOOLEAN (*bShouldKeepTdef)(
            SPORTS_SERVICE_OBJECT hSportsService,
            UN16 un16TableID,
            UN8 un8TdefClass
            );

    BOOLEAN (*bAddTableDefToDB)(
            SPORTS_SERVICE_OBJECT hSportsService,
            SPORTS_SERVICE_HIGHBAND_TDEF_INFO_STRUCT *TdefInfo,
            UN16 un16TableID
            );

    BOOLEAN (*bDeleteTableDefFromDB)(
            SPORTS_SERVICE_OBJECT hSportsService,
            UN16 un16TableID,
            UN8 un8TabVer,
            BOOLEAN bEraseUnderlyingData
            );

    BOOLEAN (*bDbgDumpAllTableNames)(
        SPORTS_SERVICE_OBJECT hSportsService
            );

    void (*vDebugSetServiceConfigVersion)(
            SPORTS_SERVICE_OBJECT hSportsService,
            UN8 un8CfgVer
            );

    BOOLEAN(*bStartSportAffiliationUpdate)(
              SPORTS_SERVICE_OBJECT hSportsService,
             UN8 un8AdVer,
             UN32 un32CRC
             );

    BOOLEAN (*bAddSportsAffiliationToDB)(
            SPORTS_SERVICE_OBJECT hSportsService,
            STRING_OBJECT hSportAffiliation,
            UN16 un16AffiliateId,
            BOOLEAN bPresBit,
            UN8 un8GDRef,
            UN8 un8SportID
            );

    BOOLEAN(*bEndSportAffiliationUpdate)(
            SPORTS_SERVICE_OBJECT hSportsService,
            BOOLEAN bUpdateOk,
            UN8 un8AdVer,
            UN32 un32CRC
            );

    BOOLEAN (*bUpdateTrefRelationships)(
            SPORTS_SERVICE_OBJECT hSportsService,
            UN32 un32TdefKey,
            UN8 un8LabelId,
            TABLE_ID tChildInstanceId,
            N64 n64RowId,
            TABLE_ID tParentInstanceId,
            UN8 un8LPriority
            );

    void (*vEndDataTableInstance) (
            SPORTS_SERVICE_OBJECT hSportsService,
            const char *acSavePointName,
            BOOLEAN bSuccess,
            SPORTS_SERVICE_DATA_INSTANCE_STRUCT *psTableInstance
            );

    void (*vReleasePayload)(
            OSAL_BUFFER_HDL hPayload
            );

    BOOLEAN (*bUnsafeLoadTdefs)(
            SPORTS_SERVICE_OBJECT hSportsService,
            SPORTS_SERVICE_INTERFACE_OBJECT hSportsHbInterface
            );

    BOOLEAN (*bAddLabelOverride)(
            SPORTS_SERVICE_OBJECT hSportsService,
            TABLE_ID tTableId,
            UN32 un32LabelId,
            UN32 un32Override
            );

} SPORTS_SERVICE_MGR_INTERFACE_STRUCT;

// The sports service plugin interface APIs
typedef struct sports_service_plugin_interface_struct
{
    const DSI tDSI;

    const size_t tOTABufferByteSize;

    SPORTS_SERVICE_INTERFACE_OBJECT (*hInit) (
        SPORTS_SERVICE_OBJECT hSportsDataService,
        SMS_OBJECT hParent
            );

    void (*vUnInit) (
        SPORTS_SERVICE_INTERFACE_OBJECT hInterface
            );

    DATASERVICE_ERROR_CODE_ENUM (*eProcessMessage) (
        SPORTS_SERVICE_INTERFACE_OBJECT hInterface,
        OSAL_BUFFER_HDL *phPayload
            );

    BOOLEAN (*bUnsafeAddTdef) (
            SPORTS_SERVICE_INTERFACE_OBJECT hInterface,
            SPORTS_SERVICE_TDEF_ADD_STRUCT *psTdefAdd
            );

} SPORTS_SERVICE_PLUGIN_INTERFACE_STRUCT;

  /****************************/
 /** GLOBAL UNION TEMPLATES **/
/****************************/

  /**********************/
 /** GLOBAL VARIABLES **/
/**********************/

extern const SPORTS_SERVICE_MGR_INTERFACE_STRUCT GsSportsServiceMgrIntf;
extern const SPORTS_SERVICE_PLUGIN_INTERFACE_STRUCT GsSportsServiceIntf;

  /*******************/
 /** GLOBAL MACROS **/
/*******************/

  /***********************/
 /** GLOBAL PROTOTYPES **/
/***********************/

/* Object Friend Prototypes */

#endif /* SPORTS_SERVICE_INTERFACE_H_ */
