/******************************************************************************/
/*                    Copyright (c) Sirius XM Radio, Inc.                     */
/*                            All Rights Reserved                             */
/*         Licensed Materials - Property of Sirius XM Radio, Inc.             */
/******************************************************************************/
/*******************************************************************************
 *
 * DESCRIPTION
 *
 * This file defines the constants used when utilizing the sports service
 * database file
 *
 ******************************************************************************/

  /*********************************/
 /** PREVENT REDUNDANT INCLUSION **/
/*********************************/
#ifndef SPORTS_SERVICE_DB_CONSTANTS_H_
#define SPORTS_SERVICE_DB_CONSTANTS_H_

  /**************/
 /** INCLUDES **/
/**************/

#include "standard.h"

  /**********************/
 /** GLOBAL CONSTANTS **/
/**********************/

#define SS_QUOTE(s) #s
#define SS_EXPAND_THEN_QUOTE(s) SS_QUOTE(s)

#define SPORTS_SERVICE_DATABASE_INVALID_VERSION 0
#define SPORTS_SERVICE_DATABASE_VERSION 3
#define SPORTS_SERVICE_AGGREGATE_SPORT_ID 0

/* Database File Name */
#define SPORTS_SERVICE_DATABASE_FOLDER "sportsservice"
#define SPORTS_SERVICE_DATABASE_FILENAME "sportsservice.db"

/* Database Table */

// Enumeration specifying all the available columns in the
// SS_TABLE_SC_SPORTS_NAMES
typedef enum ss_table_sc_sports_names_column_enum {
    SS_TABLE_SPORTS_NAMES_SPORT_NAME = 0,
    SS_TABLE_SPORTS_NAMES_SPORT_ID,
    SS_TABLE_SPORTS_NAMES_MAX_FIELDS
} SS_TABLE_SC_SPORTS_NAMES_COLUMN_ENUM;

// Enumeration specifying all the available columns in the
// SS_TABLE_SA_AFFILIATIONS
typedef enum ss_table_sa_affiliations_column_enum {
    SS_TABLE_SA_AFFILIATIONS_AFFILIATE_NAME = 0,
    SS_TABLE_SA_AFFILIATIONS_AFFILIATE_ID,
    SS_TABLE_SA_AFFILIATIONS_LEAGUE_ID,
    SS_TABLE_SA_AFFILIATIONS_SPORT_ID,
    SS_TABLE_SA_AFFILIATIONS_PARENT_AFFILIATE,
    SS_TABLE_SA_AFFILIATIONS_MAX_FIELDS
} SS_TABLE_SA_AFFILIATIONS_COLUMN_ENUM;

// All fixed table names
#define SS_TABLE_SC_CONFIG "SC_Config"
#define SS_TABLE_AF_CONFIG "AF_Config"
#define SS_TABLE_DB_CONFIG "DB_Config"
#define SS_TABLE_SC_SPORTS_NAMES "SC_SportsNames"
#define SS_TABLE_SC_TABLEINFO "SC_TableInfo"
#define SS_TABLE_TD_INSTANCES "TD_Info"
#define SS_TABLE_TD_LABEL_INFO "TD_LabelInfo"
#define SS_TABLE_TABLE_INSTANCES "DI_Info"
#define SS_TABLE_SA_AFFILIATIONS "SA_Affiliations"
#define SS_TABLE_INSTANCE_OVERRIDES "DI_Override"

// Used to update affiliate table.
#define SS_TABLE_SA_AFFILIATIONS_UPDATE "SA_Affiliations_Update"

// Transient tables naming convention. sprintf("DI_%08x", TDefKey);
#define SS_TABLE_MAX_TD_TABLE_NAME_SIZE (11 + 1)

/* Database Queries */
#define SS_MAKE_INSTANCE_TABLE_NAME \
        "DI_%08x"

#define SELECT_ALL_FROM_TABLE \
    "SELECT * FROM %s"

#define GET_TABLE_NAMES_FROM_MASTER_TABLE \
    "SELECT name FROM sqlite_master WHERE TYPE='table' ORDER BY NAME"

// Get the sport list.
#define SS_SELECT_GET_SPORT_LIST \
    "SELECT *" \
    " FROM "SS_TABLE_SC_SPORTS_NAMES""

// Get the root affiliate list.
#define SS_SELECT_GET_ROOT_AFFILIATE_LIST \
    "SELECT AfName,SportID,AfID,LeagueID" \
    " FROM "SS_TABLE_SA_AFFILIATIONS \
    " WHERE ParentAf IS NULL"

// Get the children of an affiliate.
#define SS_SELECT_GET_AFFILIATE_CHILDREN \
    "SELECT AfName,SportID,AfID,LeagueID" \
    " FROM "SS_TABLE_SA_AFFILIATIONS \
    " WHERE ParentAf=%u"

// Get an affiliate.
#define SS_SELECT_GET_AFFILIATE \
    "SELECT AfName,SportID,AfID,LeagueID" \
    " FROM "SS_TABLE_SA_AFFILIATIONS \
    " WHERE AfID=%u"

// Enumeration specifying all the available result columns
typedef enum ss_select_get_affiliate_list_column_enum {
    SS_SELECT_GET_AFFILIATE_LIST_AFFILIATE_NAME = 0,
    SS_SELECT_GET_AFFILIATE_LIST_SPORT_ID,
    SS_SELECT_GET_AFFILIATE_LIST_AFFILIATE_ID,
    SS_SELECT_GET_AFFILIATE_LIST_LEAGUE_ID,
    SS_SELECT_GET_AFFILIATE_LIST_MAX_FIELDS
} SS_SELECT_GET_AFFILIATE_LIST_COLUMN_ENUM;

// Get the available info classes of an affiliate.
#define SS_SELECT_GET_AFFILIATE_INFO_CLASS_LIST \
    "SELECT DISTINCT TableClass" \
    " FROM "SS_TABLE_TABLE_INSTANCES \
    " WHERE AfID=%u AND Parent IS NULL"

// Enumeration specifying all the available result columns
typedef enum ss_select_get_affiliate_info_class_list_column_enum {
    SS_SELECT_GET_AFFILIATE_INFO_CLASS_LIST_TABLE_CLASS = 0,
    SS_SELECT_GET_AFFILIATE_INFO_CLASS_LIST_MAX_FIELDS
} SS_SELECT_GET_AFFILIATE_INFO_CLASS_LIST_COLUMN_ENUM;

// Get the tables of an affiliate.
#define SS_SELECT_GET_AFFILIATE_TABLE_LIST \
    "SELECT InstID,Title,Season,Epoch,ReceiptTime,TDefKey" \
    " FROM "SS_TABLE_TABLE_INSTANCES \
    " WHERE AfID=%u" \
    " AND Parent IS NULL" \
    " AND TableClass=%u" \
    " ORDER BY Epoch ASC, InstID ASC"

// Enumeration specifying all the available result columns
typedef enum ss_select_get_affiliate_table_list_column_enum {
    SS_SELECT_GET_AFFILIATE_TABLE_LIST_INSTANCE_ID = 0,
    SS_SELECT_GET_AFFILIATE_TABLE_LIST_ITITLE,
    SS_SELECT_GET_AFFILIATE_TABLE_LIST_SEASON_STATUS,
    SS_SELECT_GET_AFFILIATE_TABLE_LIST_EPOCH,
    SS_SELECT_GET_AFFILIATE_TABLE_LIST_RECEIPT_TIME,
    SS_SELECT_GET_AFFILIATE_TABLE_LIST_TDEFKEY,
    SS_SELECT_GET_AFFILIATE_TABLE_LIST_MAX_FIELDS
} SS_SELECT_GET_AFFILIATE_TABLE_LIST_COLUMN_ENUM;

// Get the non column portion of a table header.
#define SS_SELECT_GET_TABLE_HEADER_FIXED \
    "SELECT Title,TableClass,Season,Epoch,TDefKey,ReceiptTime" \
    " FROM "SS_TABLE_TABLE_INSTANCES \
    " WHERE InstID=%llu" \
    " AND AfID=%u" \
    " AND Parent IS NULL"

// Get the non column portion of a referenced table header.
#define SS_SELECT_GET_REFERENCED_TABLE_HEADER_FIXED \
    "SELECT Title,TableClass,Season,Epoch,TDefKey,ReceiptTime" \
    " FROM "SS_TABLE_TABLE_INSTANCES \
    " WHERE InstID=%llu" \
    " AND Parent IS NOT NULL"

typedef enum ss_select_get_table_header_fixed_column_enum {
    SS_SELECT_GET_TABLE_HEADER_FIXED_ITITLE = 0,
    SS_SELECT_GET_TABLE_HEADER_FIXED_TABLE_CLASS,
    SS_SELECT_GET_TABLE_HEADER_FIXED_SEASON_STATUS,
    SS_SELECT_GET_TABLE_HEADER_FIXED_EPOCH,
    SS_SELECT_GET_TABLE_HEADER_FIXED_TDEFKEY,
    SS_SELECT_GET_TABLE_HEADER_FIXED_RECEIPT_TIME,
    SS_SELECT_GET_TABLE_HEADER_FIXED_MAX_FIELDS
} SS_SELECT_GET_TABLE_HEADER_FIXED_COLUMN_ENUM;

// Get the column portion of a table header.
#define SS_SELECT_GET_TABLE_HEADER_VARIABLE \
    "SELECT OrderIndex,LType,LTxt,DMod,LPriority,LSize" \
    " FROM "SS_TABLE_TD_LABEL_INFO \
    " WHERE TDefKey=%u" \
    " ORDER BY OrderIndex ASC"

typedef enum ss_select_get_table_header_variable_column_enum {
    SS_SELECT_GET_TABLE_HEADER_VARIABLE_ORDER_INDEX = 0,
    SS_SELECT_GET_TABLE_HEADER_VARIABLE_LBTYPE,
    SS_SELECT_GET_TABLE_HEADER_VARIABLE_LBTXT,
    SS_SELECT_GET_TABLE_HEADER_VARIABLE_DMOD,
    SS_SELECT_GET_TABLE_HEADER_VARIABLE_LPRIORITY,
    SS_SELECT_GET_TABLE_HEADER_VARIABLE_LSIZE,
    SS_SELECT_GET_TABLE_HEADER_VARIABLE_MAX_FIELDS
} SS_SELECT_GET_TABLE_HEADER_VARIABLE_COLUMN_ENUM;

#define SS_SELECT_TABLE_TDEF \
    "SELECT TDefKey" \
    " FROM "SS_TABLE_TABLE_INSTANCES \
    " WHERE InstID=%llu" \
    " AND AfID=%u" \
    " AND Parent IS NULL"

// Get the name of a table definition table for a referenced table.
#define SS_SELECT_REFERENCED_TABLE_TDEF \
    "SELECT TDefKey" \
    " FROM "SS_TABLE_TABLE_INSTANCES \
    " WHERE InstID=%llu" \
    " AND Parent IS NOT NULL"

// Get the rows of a table given its instance id.
// Limit of -1 means no upper bound.
#define SS_SELECT_TABLE \
    "SELECT *" \
    " FROM DI_%08X" \
    " WHERE InstID=%llu" \
    " ORDER BY ROWID LIMIT -1 OFFSET %u"

// Get the rows of a table by column and equality match given its instance id.
// Limit of -1 means no upper bound.
#define SS_SELECT_TABLE_BY_COLUMN_MATCH \
    "SELECT *" \
    " FROM DI_%08X" \
    " WHERE InstID=%llu" \
    " AND c%u=\"%s\"" \
    " ORDER BY ROWID LIMIT -1 OFFSET %u"

// Get the rows of a table by column and range.
// Limit of -1 means no upper bound.
#define SS_SELECT_TABLE_BY_COLUMN_RANGE \
    "SELECT *" \
    " FROM DI_%08X" \
    " WHERE InstID=%llu" \
    " AND c%u BETWEEN %u AND %u" \
    " ORDER BY c%u %s, ROWID ASC LIMIT -1 OFFSET %u"

// Get the rows of a table based on column and team id logic.
// Limit of -1 means no upper bound.
#define SS_SELECT_TABLE_BY_TEAM_PRESENCE \
    "SELECT *" \
    " FROM DI_%08X" \
    " WHERE InstID=%llu" \
    " AND (c%u=%u OR c%u=%u)" \
    " ORDER BY ROWID LIMIT -1 OFFSET %u"

// Table Data manipulation and handling
//
// Adding live data:
// if TableDef/Ver/Affiliate exists matching incoming data
//   remove any matching instance from D_{def}_{ver}
// else
//   add instance rows to TableInstance
// endif
//
// Add data to DI_{def}{ver}
// update CRC in TableInstance
//
#define SS_INSERT_DATA_INFO_ROW \
    "insert into "SS_TABLE_TABLE_INSTANCES \
                 "(Epoch, AfID, Season, Title, TDefKey, TableClass, TDefID)" \
        " values (%u,%u,%u,?,%u,%u,%u);"

#define SS_SELECT_GET_TABLE_KEY_AND_CLASS \
    "SELECT TDefKey,TableClass" \
    " FROM "SS_TABLE_TD_INSTANCES \
    " WHERE TDefID=%d" \
    " AND TDefVer=%d"

typedef enum ss_select_get_table_key_and_class_column_enum {
    SS_SELECT_GET_TABLE_KEY_AND_CLASS_KEY,
    SS_SELECT_GET_TABLE_KEY_AND_CLASS_CLASS,
    SS_SELECT_GET_TABLE_KEY_AND_CLASS_MAX_FIELDS
} SS_SELECT_GET_TABLE_KEY_AND_CLASS_COLUMN_ENUM;

#define SS_SELECT_GET_INSTANCE_ID_AND_CRC \
    "SELECT InstId,Crc,TDefKey" \
    " FROM "SS_TABLE_TABLE_INSTANCES \
    " WHERE TDefID=%d" \
    " AND AfId=%d" \
    " AND Epoch=%d" \
    " AND Parent IS NULL"

typedef enum ss_select_get_instance_id_and_crc_column_enum {
    SS_SELECT_GET_INSTANCE_ID_AND_CRC_ID,
    SS_SELECT_GET_INSTANCE_ID_AND_CRC_CRC,
    SS_SELECT_GET_INSTANCE_ID_AND_CRC_TDEFKEY,
    SS_SELECT_GET_INSTANCE_ID_AND_CRC_MAX_FIELDS
} SS_SELECT_GET_INSTANCE_ID_AND_CRC_COLUMN_ENUM;

typedef enum ss_select_table_column_enum {
    SS_SELECT_SELECT_TABLE_ROWID,
    SS_SELECT_SELECT_TABLE_INSTID,
    SS_SELECT_SELECT_TABLE_MIN_FIELDS
} SS_SELECT_SELECT_TABLE_COLUMN_ENUM;

// The rest of this gets built by code generating templates of
// a prepared statement.
#define SS_INSERT_TABLE_ROW_BEGIN \
    "INSERT INTO DI_%04x%04x VALUES ( NULL, %llu"

#define SS_DELETE_OLD_INSTANCE_DATA_BY_TDEFKEY \
    "DELETE from DI_%08x" \
    " WHERE InstId=%llu;"

#define SS_GET_DI_TABLE_COUNT \
    "SELECT COUNT(TableName) FROM TD_Info"

#define SS_GET_TREF_REMOVE_INFO \
    "SELECT TDefKey,InstId,AfID" \
    " FROM "SS_TABLE_TABLE_INSTANCES \
    " WHERE Parent=%llu"

#define SS_DELETE_TREF_INSTANCES \
    "DELETE" \
    " FROM "SS_TABLE_TABLE_INSTANCES \
    " WHERE Parent=%llu;"

#define SS_GET_TABLE_AND_TREF_REMOVE_INFO \
    "SELECT TDefKey,InstId,AfID" \
    " FROM "SS_TABLE_TABLE_INSTANCES \
    " WHERE Parent=%llu OR InstID=%llu"

#define SS_DELETE_TABLE_AND_TREF_INSTANCES \
    "DELETE" \
    " FROM "SS_TABLE_TABLE_INSTANCES \
    " WHERE Parent=%llu OR InstID=%llu;"

typedef enum ss_get_tref_remove_info_column_enum {
    SS_GET_TREF_REMOVE_INFO_TDEFKEY,
    SS_GET_TREF_REMOVE_INFO_INSTID,
    SS_GET_TREF_REMOVE_INFO_AFID,
    SS_GET_TREF_REMOVE_INFO_MAX_FIELDS
} SS_GET_TREF_REMOVE_INFO_COLUMN_ENUM;

#define SS_GET_TABLENAME_FROM_TDINFO \
        "SELECT TableName FROM TD_Info"

#define SS_CREATE_DATA_INSTANCE_TABLE_START \
        "CREATE TABLE IF NOT EXISTS DI_%08X " \
        "('ROWID' INTEGER PRIMARY KEY AUTOINCREMENT, 'InstID' INTEGER"

#define SS_CREATE_DATA_INSTANCE_TABLE_INDEX \
        "CREATE INDEX IF NOT EXISTS DI_%08X_idx " \
        " ON DI_%08X (InstID);"

#define SS_DROP_DATA_INSTANCE_TABLE_INDEX \
        "DROP INDEX IF EXISTS DI_%08X_idx;"

#define SS_DELETE_OLD_LABEL_INFO \
        "DELETE FROM TD_LabelInfo WHERE TdefKey=%d;"

#define SS_DELETE_OLD_TABLE_DEF_INFO \
        "DELETE from TD_Info WHERE TdefKey=%d;"

#define SS_DELETE_ALL_DATA_FROM_TABLE \
        "DELETE FROM %s;"

//  Label info entries are made up of optional fields
//  so it has to be built in sections
#define SS_CREATE_LABEL_INFO_ENTRY_START \
        "INSERT INTO TD_LabelInfo " \
        "(OrderIndex,LID,Ltype,"

#define SS_CREATE_LABEL_INFO_ENTRY_DMOD \
        "Dmod,"

#define SS_CREATE_LABEL_INFO_ENTRY_LINFO \
        "LPriority,LSize,LTxt,"

#define SS_CREATE_LABEL_INFO_ENTRY_EXTDATA \
        "ExtData,"

#define SS_CREATE_LABEL_INFO_ENTRY_END \
        "DataType,TdefKey) VALUES " \
        "(%d,%d,%d,"

#define SS_CREATE_LABEL_INFO_VALUES_DMOD \
        "%d,"

#define SS_CREATE_LABEL_INFO_VALUES_LINFO \
        "%d,%d,'%s',"

#define SS_CREATE_LABEL_INFO_VALUES_EXTDATA \
        "'%s',"

#define SS_CREATE_LABEL_INFO_VALUES_END \
        "%d,%d); "

#define SS_DROP_TABLE_BY_TDEFKEY \
        "DROP TABLE IF EXISTS DI_%08X;"

#define SS_UPDATE_TABLE_CRC_AND_RECEIPT_TIME \
    "UPDATE "SS_TABLE_TABLE_INSTANCES \
    " SET Crc=%u,ReceiptTime=%u,TdefKey=%u,TableClass=%u" \
    " WHERE InstID=%llu;"

#define SS_UPDATE_TABLE_DATA_INFO \
    "INSERT INTO TD_Info " \
    "(TdefId,TdefVer,TableName,TableClass,LabelCount,TdefKey,Crc) " \
    "VALUES (%d,%d,'DI_%08X',%d,%d,%d,%d);"

// Constant used to size buffers used to perform
// DB queries.
// TODO: Size this based on defined query strings.  Take into account
// the variable input parameters.
#define SPORTS_SERVICE_MAX_SQL_STRING_LENGTH (512)

#define SS_TABLE_GET_SC_CONFIG_VER \
        "SELECT SC_Ver,CRC FROM "SS_TABLE_SC_CONFIG

typedef enum ss_select_get_sc_config_ver_column_enum {
    SS_SELECT_GET_SC_CONFIG_VER_VER,
    SS_SELECT_GET_SC_CONFIG_VER_CRC,
    SS_SELECT_GET_SC_CONFIG_VER_MAX_FIELDS
} SS_SELECT_GET_SC_CONFIG_VER_COLUMN_ENUM;

#define SS_TABLE_UPDATE_SC_CONFIG_VALUES \
"INSERT INTO "SS_TABLE_SC_CONFIG" (SC_Ver,Crc) VALUES (%d,%d);"

#define SS_TABLE_ADD_TO_SC_SPORTS_NAMES \
        "INSERT INTO "SS_TABLE_SC_SPORTS_NAMES \
        " (SportName,SportID) VALUES ('%s',%d);"

#define SS_TABLE_ADD_TDEF_SPORTID_MAPPING \
        "INSERT INTO "SS_TABLE_SC_TABLEINFO \
        " (TDefID,SportID) VALUES (%u,%u);"

#define SS_TABLE_GET_AF_CONFIG_VER \
        "SELECT AF_Ver,CRC FROM "SS_TABLE_AF_CONFIG

typedef enum ss_select_get_af_config_ver_column_enum {
    SS_SELECT_GET_AF_CONFIG_VER_VER,
    SS_SELECT_GET_AF_CONFIG_VER_CRC,
    SS_SELECT_GET_AF_CONFIG_VER_MAX_FIELDS
} SS_SELECT_GET_AF_CONFIG_VER_COLUMN_ENUM;

// Referenced Table Updates
#define SS_UPDATE_PRIMARY_TABLE_TREF \
    "UPDATE DI_%08X" \
    " SET c%d=%llu"    \
    " WHERE ROWID=%llu;"

#define SS_UPDATE_TREF_PARENT \
    "UPDATE "SS_TABLE_TABLE_INSTANCES \
     " SET Parent=%llu" \
     " WHERE InstID=%llu;"

// Sport Affiliation management strings
#define SS_TABLE_CREATE_AFFILIATIONS_UPDATE \
    "CREATE TABLE "SS_TABLE_SA_AFFILIATIONS_UPDATE" (" \
        "\"AfName\" TEXT," \
        "\"AfID\" INTEGER PRIMARY KEY,"\
        "\"LeagueID\" INTEGER,"\
        "\"SportID\" INTEGER,"\
        "\"ParentAf\" INTEGER"\
        ");"

#define SS_TABLE_DROP_AFFILIATIONS \
    "DROP TABLE "SS_TABLE_SA_AFFILIATIONS";"

#define SS_TABLE_REPLACE_AFFILIATIONS \
    "ALTER TABLE "SS_TABLE_SA_AFFILIATIONS_UPDATE  \
    " RENAME TO "SS_TABLE_SA_AFFILIATIONS";"

#define SS_GET_AFFILIATE_REMOVE_INFO \
    "SELECT TdefKey,InstId,AfID" \
    " FROM "SS_TABLE_TABLE_INSTANCES \
    " WHERE AfID=%llu"

#define SS_DELETE_AFFILIATE_INSTANCES \
    "DELETE" \
    " FROM "SS_TABLE_TABLE_INSTANCES \
    " WHERE AfID=%llu;"

#define SS_SELECT_GET_ALL_AF_IDS \
        "SELECT AfID FROM " \
        SS_TABLE_SA_AFFILIATIONS

#define SS_TABLE_GET_NAME_CHANGE_AFFILIATE \
    "SELECT AfName FROM "SS_TABLE_SA_AFFILIATIONS \
    " WHERE AfID=%u"

#define SS_TABLE_UPDATE_AF_CONFIG_VALUES \
    "INSERT INTO "SS_TABLE_AF_CONFIG" (AF_Ver,Crc) VALUES (%d,%d);"

#define SS_TABLE_GET_FIRST_AFID_FROM_AFNAME \
    "SELECT AfId FROM "SS_TABLE_SA_AFFILIATIONS_UPDATE \
    " WHERE AfName='%s' AND" \
    " (SportId=%d OR SportId=" \
    SS_EXPAND_THEN_QUOTE(SPORTS_SERVICE_AGGREGATE_SPORT_ID)")"

#define SS_TABLE_SET_SPORTS_AFFILIATE \
    "INSERT INTO "SS_TABLE_SA_AFFILIATIONS_UPDATE" (AfId,AfName,SportId"

#define SS_TABLE_ADD_PARENT_AFFILIATE \
        ",ParentAf"

#define SS_TABLE_ADD_AFFILIATE_LEAGUE_ID \
    ",LeagueID"

#define SS_TABLE_SET_SPORTS_AFFILIATE_VALUES \
    ") VALUES (%d,'%s',%d"

#define SS_TABLE_ADD_PARENT_AFFILIATE_VALUES \
    ",%d"

#define SS_TABLE_ADD_AFFILIATE_LEAGUE_ID_VALUES \
    ",%d"

#define SS_TABLE_SET_SPORTS_AFFILIATE_END \
    ");"

#define SS_SELECT_LAST_INSERT_ROWID \
    "SELECT last_insert_rowid()"

#define SS_PRAGMA_GET_PAGE_SIZE \
    "PRAGMA page_size"

#define SS_PRAGMA_SET_PAGE_SIZE \
    "PRAGMA page_size=%u"

#define SS_SELECT_GET_DB_VERSION \
    "SELECT DB_Ver FROM "SS_TABLE_DB_CONFIG

// Macros to set/clear/use rollback points
#define SS_SET_ROLLBACK_POINT \
    "SAVEPOINT %s;"

#define SS_CLEAR_ROLLBACK_POINT \
    "RELEASE %s;"

#define SS_USE_ROLLBACK_POINT \
    "ROLLBACK TO %s;"

#define SS_ROLLBACK_POINT_AF_UPDATE \
    "Af_Update_Rollback"

#define SS_ROLLBACK_POINT_SC_UPDATE \
    "SC_Update_Rollback"

#define SS_ROLLBACK_POINT_TD_UPDATE \
    "TD_Update_Rollback"

#define SS_ROLLBACK_POINT_DI_UPDATE \
    "DI_Update_Rollback"

#define SS_ROLLBACK_POINT_AGE_OUT \
    "Age_Out_Rollback"

#define SS_JOURNAL_MODE_STRING_MAXSIZE 16
#define SS_JOURNAL_MODE_MEMORY "memory"
#define SS_JOURNAL_MODE_TEMP_DB "temp."
#define SS_JOURNAL_MODE_MAIN_DB "main."
#define SS_JOURNAL_MODE_ENTIRE_DB ""

#define SS_PRAGMA_SET_JOURNAL_MODE \
    "PRAGMA %sjournal_mode = %s"

#define SS_SELECT_TDEF_DESC \
    "SELECT TDefID,TDefVer,TableClass,LabelCount,Crc" \
    " FROM "SS_TABLE_TD_INSTANCES \
    " ORDER BY TDefID ASC, ROWID DESC"

// Enumeration specifying all the available result columns
typedef enum ss_select_tdef_desc_column_enum {
    SS_SELECT_TDEF_DESC_TDEFID = 0,
    SS_SELECT_TDEF_DESC_TDEFVER,
    SS_SELECT_TDEF_DESC_TABLECLASS,
    SS_SELECT_TDEF_DESC_LABELCOUNT,
    SS_SELECT_TDEF_DESC_CRC,
    SS_SELECT_TDEF_DESC_MAX_FIELDS
} SS_SELECT_TDEF_DESC_COLUMN_ENUM;

#define SS_SELECT_GET_TDEF_LABELS \
    "SELECT LID,LType,DMod,LPriority,LSize,LTxt,DataType" \
    " FROM "SS_TABLE_TD_LABEL_INFO \
    " WHERE TDefKey=%u" \
    " ORDER BY OrderIndex ASC"

// Enumeration specifying all the available result columns
typedef enum ss_select_get_labels_column_enum {
    SS_SELECT_GET_TDEF_LABELS_LID = 0,
    SS_SELECT_GET_TDEF_LABELS_LTYPE,
    SS_SELECT_GET_TDEF_LABELS_DMOD,
    SS_SELECT_GET_TDEF_LABELS_LPRIORITY,
    SS_SELECT_GET_TDEF_LABELS_LSIZE,
    SS_SELECT_GET_TDEF_LABELS_LTXT,
    SS_SELECT_GET_TDEF_LABELS_DATATYPE,
    SS_SELECT_GET_TDEF_LABELS_MAX_FIELDS
} SS_SELECT_GET_TDEF_LABELS_COLUMN_ENUM;

#define SS_GET_DI_INFO_BY_TDEFKEY \
    "SELECT TDefKey" \
    " FROM "SS_TABLE_TABLE_INSTANCES \
    " WHERE TDefKey=%u"

// Enumeration specifying all the available result columns
typedef enum ss_select_tclass_info_column_enum {
    SS_SELECT_TCLASS_INFO_TABLECLASS = 0,
    SS_SELECT_TCLASS_INFO_SAVEDUR,
    SS_SELECT_TCLASS_INFO_SAVERULE,
    SS_SELECT_TCLASS_INFO_MAX_FIELDS
} SS_SELECT_TCLASS_INFO_COLUMN_ENUM;

#define SS_SAVE_RULES_REMOVE_COUNT \
    "SELECT COUNT(*)" \
    " FROM "SS_TABLE_TABLE_INSTANCES \
    " WHERE TableClass=%u" \
    " AND (Epoch<%u OR ReceiptTime<%u)" \
    " AND Parent IS NULL"

#define SS_SAVE_RULES_GET_TABLES_TO_REMOVE \
    "SELECT TdefKey,InstId,AfID" \
    " FROM "SS_TABLE_TABLE_INSTANCES \
    " WHERE TableClass=%u" \
    " AND (Epoch<%u OR ReceiptTime<%u)" \
    " AND Parent IS NULL"

#define SS_SHUTDOWN_RULES_REMOVE_COUNT \
    "SELECT COUNT(*)" \
    " FROM "SS_TABLE_TABLE_INSTANCES \
    " WHERE TableClass=%u AND Epoch=%u AND Parent IS NULL"

#define SS_SHUTDOWN_RULES_GET_TABLES_TO_REMOVE \
    "SELECT TdefKey,InstId,AfID" \
    " FROM "SS_TABLE_TABLE_INSTANCES \
    " WHERE TableClass=%u AND Epoch=%u AND Parent IS NULL"

#define SS_REMOVE_TABLE_FROM_DI_INFO \
    "DELETE" \
    " FROM "SS_TABLE_TABLE_INSTANCES \
    " WHERE InstID=%llu;"

#define SS_GET_REPLACED_TABLES \
    "SELECT TdefKey,InstId,AfID" \
    " FROM "SS_TABLE_TABLE_INSTANCES \
    " WHERE InstId!=%llu" \
    " AND TDefID=%d" \
    " AND AfId=%d" \
    " AND Parent IS NULL"

// Enumeration specifying all the available result columns
typedef enum ss_get_replaced_table_column_enum {
    SS_GET_REPLACED_TABLE_INSTID = 0,
    SS_GET_REPLACED_TABLE_TDEFKEY,
    SS_GET_REPLACED_TABLE_MAX_FIELDS
} SS_GET_REPLACED_TABLE_ENUM;

#define SS_GET_SPORTID_FOR_TDEF \
    "SELECT SportID" \
    " FROM "SS_TABLE_SC_TABLEINFO \
    " WHERE TDefID=%u"

#define SS_GET_COLUMN_PRIORITY \
    "SELECT LPriority,LType" \
    " FROM "SS_TABLE_TD_LABEL_INFO \
    " WHERE TDefKey=%u" \
    " ORDER BY OrderIndex ASC"

// Enumeration specifying all the available result columns
typedef enum ss_get_column_priority_column_enum {
    SS_GET_COLUMN_PRIORITY_PRIORITY = 0,
    SS_GET_COLUMN_PRIORITY_LTYPE,
    SS_GET_COLUMN_PRIORITY_MAX_FIELDS
} SS_GET_COLUMN_PRIORITY_ENUM;

#define SS_UPDATE_TABLE_RECEIPT_TIME \
    "UPDATE "SS_TABLE_TABLE_INSTANCES \
    " SET ReceiptTime=%u" \
    " WHERE InstID=%llu OR Parent=%llu;"

#define SS_TABLE_ADD_LABEL_OVERRIDE \
        "INSERT INTO "SS_TABLE_INSTANCE_OVERRIDES \
        " (InstID,LID,LSize)" \
        " VALUES (%llu,%u,%u);"

#define SS_DELETE_LABEL_OVERRIDES \
        "DELETE FROM "SS_TABLE_INSTANCE_OVERRIDES \
        " WHERE InstID=%llu;"

#define SS_GET_LABEL_OVERRIDES \
    "SELECT "SS_TABLE_TD_LABEL_INFO".OrderIndex," \
    SS_TABLE_TD_LABEL_INFO".LType," \
    SS_TABLE_TD_LABEL_INFO".LPriority," \
    SS_TABLE_INSTANCE_OVERRIDES".LSize" \
    " FROM "SS_TABLE_INSTANCE_OVERRIDES \
    " INNER JOIN "SS_TABLE_TD_LABEL_INFO \
    " ON "SS_TABLE_INSTANCE_OVERRIDES".InstID=%llu" \
    " AND "SS_TABLE_TD_LABEL_INFO".TDefKey=%u" \
    " AND "SS_TABLE_INSTANCE_OVERRIDES".LID="SS_TABLE_TD_LABEL_INFO".LID" \
    " ORDER BY "SS_TABLE_TD_LABEL_INFO".OrderIndex ASC"

// Enumeration specifying all the available result columns.
typedef enum ss_get_label_overrides_enum {
    SS_GET_LABEL_OVERRRIDES_ORDERINDEX = 0,
    SS_GET_LABEL_OVERRRIDES_LTYPE,
    SS_GET_LABEL_OVERRRIDES_LPRIORITY,
    SS_GET_LABEL_OVERRRIDES_LSIZE,
    SS_GET_LABEL_OVERRRIDES_MAX_FIELDS
} SS_GET_LABEL_OVERRIDES_ENUM;

#define SS_GET_ALL_TABLEDEF_KEYS \
    "SELECT DISTINCT TDefKey" \
    " FROM "SS_TABLE_TD_INSTANCES

// Enumeration specifying all the available result columns.
typedef enum ss_get_all_tabledef_keys_enum {
    SS_GET_ALL_TABLEDEF_KEYS_TDEFKEY = 0,
    SS_GET_ALL_TABLEDEF_KEYS_MAX_FIELDS
} SS_GET_ALL_TABLEDEF_KEYS_ENUM;

#define SS_GET_TABLEDEF_INSTANCE_REMOVE_INFO \
    "SELECT TdefKey,InstId,AfID" \
    " FROM "SS_TABLE_TABLE_INSTANCES \
    " WHERE TdefKey=%u"

#define SS_CLEAR_AF_CONFIG_TABLE \
        "DELETE FROM "SS_TABLE_AF_CONFIG";"

#define SS_CREATE_AF_ID_INDEX \
    "CREATE INDEX IF NOT EXISTS " \
    SS_TABLE_SA_AFFILIATIONS_UPDATE"_%u_%u_afid_idx" \
    " ON "SS_TABLE_SA_AFFILIATIONS_UPDATE" (AfID);"

#define SS_CREATE_AF_PARENT_INDEX \
    "CREATE INDEX IF NOT EXISTS " \
    SS_TABLE_SA_AFFILIATIONS_UPDATE"_%u_%u_parentaf_idx" \
    " ON "SS_TABLE_SA_AFFILIATIONS_UPDATE" (ParentAf);"

#define SS_DB_ANALYZE \
    "ANALYZE;"

//
// Strings to build the db.
//

#define SS_BUILD_DB_STATEMENT_COUNT 16

#define SS_BUILD_DB_CREATE_TABLE_1 \
        "CREATE TABLE "SS_TABLE_AF_CONFIG" (" \
        "'AF_Ver' INTEGER," \
        "'Crc' INTEGER);"

#define SS_BUILD_DB_CREATE_TABLE_2 \
        "CREATE TABLE "SS_TABLE_DB_CONFIG" (" \
        "'DB_Ver' INTEGER);"

#define SS_BUILD_DB_CREATE_TABLE_3 \
        "CREATE TABLE "SS_TABLE_TABLE_INSTANCES" (" \
        "'Parent' INTEGER," \
        "'Crc' INTEGER," \
        "'InstID' INTEGER PRIMARY KEY NOT NULL UNIQUE," \
        "'Epoch' INTEGER," \
        "'AfID' INTEGER," \
        "'Season' INTEGER," \
        "'Title' TEXT," \
        "'TDefKey' INTEGER," \
        "'TableClass' INTEGER," \
        "'TDefID' INTEGER," \
        "'ReceiptTime' INTEGER);"

#define SS_BUILD_DB_CREATE_TABLE_4 \
        "CREATE TABLE "SS_TABLE_SA_AFFILIATIONS" (" \
        "'AfName' TEXT," \
        "'AfID' INTEGER PRIMARY KEY," \
        "'LeagueID' INTEGER," \
        "'SportID' INTEGER," \
        "'ParentAf' INTEGER);"

#define SS_BUILD_DB_CREATE_TABLE_5 \
        "CREATE TABLE "SS_TABLE_SC_CONFIG" (" \
        "'SC_Ver' INTEGER," \
        "'Crc' INTEGER);"

#define SS_BUILD_DB_CREATE_TABLE_6 \
        "CREATE TABLE "SS_TABLE_SC_SPORTS_NAMES" (" \
        "'SportName' TEXT," \
        "'SportID' INTEGER PRIMARY KEY NOT NULL UNIQUE);"

#define SS_BUILD_DB_CREATE_TABLE_7 \
        "CREATE TABLE "SS_TABLE_SC_TABLEINFO" (" \
        "'TDefID' INTEGER PRIMARY KEY," \
        "'SportID' INTEGER);"

#define SS_BUILD_DB_CREATE_TABLE_8 \
        "CREATE TABLE "SS_TABLE_TD_INSTANCES" (" \
        "'TDefID' INTEGER," \
        "'TDefVer' INTEGER," \
        "'TableName' TEXT," \
        "'TableClass' INTEGER," \
        "'LabelCount' INTEGER," \
        "'TDefKey' INTEGER," \
        "'Crc' INTEGER," \
        "'ROWID' INTEGER PRIMARY KEY AUTOINCREMENT);"

#define SS_BUILD_DB_CREATE_TABLE_9 \
        "CREATE TABLE "SS_TABLE_TD_LABEL_INFO" (" \
        "'OrderIndex' INTEGER," \
        "'LID' INTEGER," \
        "'LType' INTEGER," \
        "'DMod' INTEGER," \
        "'LPriority' INTEGER," \
        "'LSize' INTEGER," \
        "'LTxt' TEXT," \
        "'ExtData' BLOB," \
        "'DataType' INTEGER," \
        "'TDefKey' INTEGER);"

#define SS_BUILD_DB_CREATE_TABLE_10 \
        "CREATE TABLE "SS_TABLE_INSTANCE_OVERRIDES" (" \
        "'InstID' INTEGER," \
        "'LID' INTEGER," \
        "'LSize' INTEGER);"

#define SS_BUILD_DB_SET_VERSION \
        "INSERT INTO "SS_TABLE_DB_CONFIG \
        " (DB_Ver) VALUES ("SS_EXPAND_THEN_QUOTE(SPORTS_SERVICE_DATABASE_VERSION)");"

#define SS_BUILD_DB_CREATE_INDEX_1 \
        "CREATE INDEX IF NOT EXISTS "SS_TABLE_TABLE_INSTANCES"_Parent_idx" \
         " ON "SS_TABLE_TABLE_INSTANCES" (Parent);"

#define SS_BUILD_DB_CREATE_INDEX_2 \
        "CREATE INDEX IF NOT EXISTS "SS_TABLE_TABLE_INSTANCES"_AfID_idx" \
         " ON "SS_TABLE_TABLE_INSTANCES" (AfID);"

#define SS_BUILD_DB_CREATE_INDEX_3 \
        "CREATE INDEX IF NOT EXISTS "SS_TABLE_TD_LABEL_INFO"_TDefKey_idx" \
         " ON "SS_TABLE_TD_LABEL_INFO" (TDefKey);"

#define SS_BUILD_DB_CREATE_INDEX_4 \
        "CREATE INDEX IF NOT EXISTS "SS_TABLE_SA_AFFILIATIONS"_afid_idx" \
         " ON "SS_TABLE_SA_AFFILIATIONS" (AfID);"

#define SS_BUILD_DB_CREATE_INDEX_5 \
        "CREATE INDEX IF NOT EXISTS "SS_TABLE_SA_AFFILIATIONS"_parentaf_idx" \
         " ON "SS_TABLE_SA_AFFILIATIONS" (ParentAf);"

#endif // SPORTS_SERVICE_DB_CONSTANTS_H_
