/******************************************************************************/
/*                    Copyright (c) Sirius XM Radio, Inc.                     */
/*                            All Rights Reserved                             */
/*         Licensed Materials - Property of Sirius XM Radio, Inc.             */
/******************************************************************************/

#include "standard.h"
#include "osal.h"
#include "sms_api.h"
#include "sms_obj.h"
#include "sports_monitor_obj.h"
#include "_sports_monitor_obj.h"
#include "sports_service_mgr_obj.h"

#include "sms_api_debug.h"
static const char *gpacThisFile = __FILE__;

#ifndef DEBUG_PRINT
    #define DEBUG_PRINT 0
#endif // DEBUG_PRINT
#if DEBUG_PRINT
//#define DPRINTF(...) do{ printf(__VA_ARGS__); }while(0)
//#define DPRINTF(...) do{ SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__, __VA_ARGS__ ); }while(0)
SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__, __VA_ARGS__ ); }while(0)
#define DPRINTF(...) do{
SMSAPI_DEBUG_vPrintErrorFull(NULL,__LINE__,__VA_ARGS__ ); }while(0)
#else
    #if 0//__STDC_VERSION__ >= 199901L
        #define DPRINTF(...) do{}while(0)
    #else
        #define DPRINTF if(1);else printf
    #endif
#endif // DEBUG_PRINT

/*****************************************************************************
                             PUBLIC FUNCTIONS
*****************************************************************************/

/*******************************************************************************
*
*   hCreate
*
*******************************************************************************/
static SPORTS_MONITOR_OBJECT hCreate (
    SPORTS_SERVICE_OBJECT hService,
    SPORTS_MONITOR_CALLBACK vCallback,
    void *pvCallbackArg )
{
    return SPORTS_SERVICE_MGR_hMonitorCreate (
                hService,
                vCallback,
                pvCallbackArg );
}

/*******************************************************************************
*
*   eGetEventMask
*
*******************************************************************************/
static SMSAPI_RETURN_CODE_ENUM eGetEventMask (
    SPORTS_MONITOR_OBJECT hMonitor,
    SPORTS_MONITOR_EVENT_MASK *ptEventMask )
{
    BOOLEAN bOk;
    SPORTS_MONITOR_OBJECT_STRUCT *psMonObj =
                            (SPORTS_MONITOR_OBJECT_STRUCT *)hMonitor;

    // Check input.
    bOk = SMSO_bValid( (SMS_OBJECT)hMonitor );
    if( (bOk == FALSE) || (ptEventMask == NULL) )
    {
        return SMSAPI_RETURN_CODE_INVALID_INPUT;
    }

    return SPORTS_SERVICE_MGR_eMonitorGetEventMask (
            psMonObj->hSportsService,
            hMonitor,
            ptEventMask    );
}

/*******************************************************************************
*
*   eSetEventMask
*
*******************************************************************************/
static SMSAPI_RETURN_CODE_ENUM eSetEventMask (
    SPORTS_MONITOR_OBJECT hMonitor,
    SPORTS_MONITOR_EVENT_MASK tEventMask )
{
    BOOLEAN bOk;
    SPORTS_MONITOR_OBJECT_STRUCT *psMonObj =
                            (SPORTS_MONITOR_OBJECT_STRUCT *)hMonitor;

    // Check input.
    bOk = SMSO_bValid( (SMS_OBJECT)hMonitor );
    if( bOk == FALSE )
    {
        return SMSAPI_RETURN_CODE_INVALID_INPUT;
    }

    return SPORTS_SERVICE_MGR_eMonitorSetEventMask (
            psMonObj->hSportsService,
            hMonitor,
            tEventMask     );
}

/*******************************************************************************
*
*   vDestroy
*
*******************************************************************************/
static void vDestroy (
    SPORTS_SERVICE_OBJECT hService,
    SPORTS_MONITOR_OBJECT hMonitor )
{
    SPORTS_SERVICE_MGR_vMonitorDestroy ( hService, hMonitor );

    return;
}

/*******************************************************************************
*
*   eSetAffiliateMask
*
*******************************************************************************/
static SMSAPI_RETURN_CODE_ENUM eSetAffiliateMask (
    SPORTS_MONITOR_OBJECT hMonitor,
    AFFILIATE_ID tAffiliateId,
    SPORTS_MONITOR_EVENT_MASK tEventMask )
{
    BOOLEAN bOk;
    SPORTS_MONITOR_OBJECT_STRUCT *psMonObj =
                            (SPORTS_MONITOR_OBJECT_STRUCT *)hMonitor;

    // Check input.
    bOk = SMSO_bValid( (SMS_OBJECT)hMonitor );
    if( bOk == FALSE )
    {
        return SMSAPI_RETURN_CODE_INVALID_INPUT;
    }

    return SPORTS_SERVICE_MGR_eMonitorSetAffiliateMask (
        psMonObj->hSportsService,
        hMonitor,
        tAffiliateId,
        tEventMask );
}

/*******************************************************************************
*
*   eGetAffiliateMask
*
*******************************************************************************/
static SMSAPI_RETURN_CODE_ENUM eGetAffiliateMask (
    SPORTS_MONITOR_OBJECT hMonitor,
    AFFILIATE_ID tAffiliateId,
    SPORTS_MONITOR_EVENT_MASK *ptEventMask )
{
    BOOLEAN bOk;
    SPORTS_MONITOR_OBJECT_STRUCT *psMonObj =
                            (SPORTS_MONITOR_OBJECT_STRUCT *)hMonitor;

    // Check input.
    bOk = SMSO_bValid( (SMS_OBJECT)hMonitor );
    if( (bOk == FALSE) || (ptEventMask == NULL) )
    {
        return SMSAPI_RETURN_CODE_INVALID_INPUT;
    }

    return SPORTS_SERVICE_MGR_eMonitorGetAffiliateMask (
        psMonObj->hSportsService,
        hMonitor,
        tAffiliateId,
        ptEventMask    );
}

/*******************************************************************************
*
*   eRemoveAllAffiliates
*
*******************************************************************************/
static SMSAPI_RETURN_CODE_ENUM eRemoveAllAffiliates (
    SPORTS_MONITOR_OBJECT hMonitor )
{
    BOOLEAN bOk;
    SPORTS_MONITOR_OBJECT_STRUCT *psMonObj =
                            (SPORTS_MONITOR_OBJECT_STRUCT *)hMonitor;

    // Check input.
    bOk = SMSO_bValid( (SMS_OBJECT)hMonitor );
    if( bOk == FALSE )
    {
        return SMSAPI_RETURN_CODE_INVALID_INPUT;
    }

    return SPORTS_SERVICE_MGR_eMonitorRemoveAllAffiliates (
        psMonObj->hSportsService,
        hMonitor );
}

/*******************************************************************************
*
*   eIterateChangedAffiliates
*
*******************************************************************************/
static SMSAPI_RETURN_CODE_ENUM eIterateChangedAffiliates (
    SPORTS_MONITOR_OBJECT hMonitor,
    SPORT_MONITOR_ITERATOR bIterator,
    void *pvIteratorArg )
{
    BOOLEAN bOk;
    SPORTS_MONITOR_OBJECT_STRUCT *psMonObj =
                            (SPORTS_MONITOR_OBJECT_STRUCT *)hMonitor;

    // Check input.
    bOk = SMSO_bValid( (SMS_OBJECT)hMonitor );
    if( (bOk == FALSE) || (bIterator == NULL) )
    {
        return SMSAPI_RETURN_CODE_INVALID_INPUT;
    }

    return SPORTS_SERVICE_MGR_eIterateChangedAffiliates (
            psMonObj->hSportsService,
            hMonitor,
            bIterator,
            pvIteratorArg );
}

/*****************************************************************************
                             FRIEND FUNCTIONS
*****************************************************************************/

/*******************************************************************************
*
*   SPORTS_MONITOR_hCreate
*
*******************************************************************************/
SPORTS_MONITOR_OBJECT SPORTS_MONITOR_hCreate (
    SPORTS_SERVICE_OBJECT hService,
    SPORTS_MONITOR_CALLBACK vCallback,
    void *pvCallbackArg
        )
{
    SPORTS_MONITOR_OBJECT_STRUCT *psMonObj = NULL;
    OSAL_RETURN_CODE_ENUM eOsalCode;
    char acName[OSAL_MAX_OBJECT_NAME_LENGTH_WITH_NULL];
    N32 n32Written;

    // We will only increment on successful creation.  That way if
    // a snap shot of the objects is examined and the naming is not
    // continuous we know there must have been a vDestroy action.
    static UN32 un32Instance = 0;

    // Validate the input callback
    if (vCallback == NULL)
    {
        return SPORTS_MONITOR_INVALID_OBJECT;
    }

    // Create object name.
    n32Written = snprintf(&acName[0], sizeof(acName),
        SPORTS_MONITOR_OBJECT_NAME":%u",
        un32Instance );

    // We're ensuring n32Written is non-negative,
    // so the cast to UN32 should be safe.
    if( ( n32Written < 0 ) ||
        ( (UN32)n32Written >= sizeof(acName) ) )
    {
        return SPORTS_MONITOR_INVALID_OBJECT;
    }

    // Create an instance with no lock or owner.
    psMonObj = (SPORTS_MONITOR_OBJECT_STRUCT *) SMSO_hCreate(
        &acName[0],
        sizeof(SPORTS_MONITOR_OBJECT_STRUCT),
        SMS_INVALID_OBJECT,
        FALSE);

    if(psMonObj == NULL)
    {
        return SPORTS_MONITOR_INVALID_OBJECT;
    }

    do
    {
        // Create the affiliate list name.
        n32Written = snprintf(&acName[n32Written],
                sizeof(acName) - n32Written,
            ":EntryList");

        // We're ensuring n32Written is non-negative,
        // so the cast to UN32 should be safe.
        if( ( n32Written < 0 ) ||
            ( (UN32)n32Written >= sizeof(acName) ) )
        {
            break;
        }

        // Create the affiliate list.
        eOsalCode = OSAL.eLinkedListCreate (
            &psMonObj->hAffiliateList,
            &acName[0],
            NULL,
            OSAL_LL_OPTION_LINEAR );

        if ( eOsalCode != OSAL_SUCCESS )
        {
            break;
        }

        // Initialize the structure.
        psMonObj->vCallback = vCallback;
        psMonObj->pvCallbackArg = pvCallbackArg;
        psMonObj->tUserMask = 0;
        psMonObj->tEventBits = 0;
        psMonObj->hSportsService = hService;

        un32Instance++;
        return (SPORTS_MONITOR_OBJECT)psMonObj;
    }
    while( FALSE );

    SMSO_vDestroy((SMS_OBJECT)psMonObj);
    return SPORTS_MONITOR_INVALID_OBJECT;
}

/*******************************************************************************
*
*   SPORTS_MONITOR_eGetEventMask
*
*******************************************************************************/
SMSAPI_RETURN_CODE_ENUM SPORTS_MONITOR_eGetEventMask (
    SPORTS_MONITOR_OBJECT hMonitor,
    SPORTS_MONITOR_EVENT_MASK *ptEventMask
        )
{
    SPORTS_MONITOR_OBJECT_STRUCT *psMonObj =
                            (SPORTS_MONITOR_OBJECT_STRUCT *)hMonitor;
    *ptEventMask = psMonObj->tUserMask;
    return SMSAPI_RETURN_CODE_SUCCESS;
}

/*******************************************************************************
*
*   SPORTS_MONITOR_eSetEventMask
*
*******************************************************************************/
SMSAPI_RETURN_CODE_ENUM SPORTS_MONITOR_eSetEventMask (
    SPORTS_MONITOR_OBJECT hMonitor,
    SPORTS_MONITOR_EVENT_MASK tEventMask
        )
{
    SPORTS_MONITOR_OBJECT_STRUCT *psMonObj =
                            (SPORTS_MONITOR_OBJECT_STRUCT *)hMonitor;
    psMonObj->tUserMask = tEventMask;
    return SMSAPI_RETURN_CODE_SUCCESS;
}

/*******************************************************************************
*
*   SPORTS_MONITOR_Destroy
*
*******************************************************************************/
void SPORTS_MONITOR_Destroy (
    SPORTS_MONITOR_OBJECT hMonitor
        )
{
    BOOLEAN bOk;
    OSAL_RETURN_CODE_ENUM eOsalCode;
    SPORTS_MONITOR_OBJECT_STRUCT *psMonObj =
                        (SPORTS_MONITOR_OBJECT_STRUCT *)hMonitor;

    // Validate the inputs.
    bOk = SMSO_bValid( (SMS_OBJECT)hMonitor );
    if( bOk == TRUE )
    {
        // Free the affiliate list and the monitor itself.
        eOsalCode = OSAL.eLinkedListRemoveAll(
                psMonObj->hAffiliateList,
                (OSAL_LL_RELEASE_HANDLER)SMSO_vDestroy );

        if ( eOsalCode == OSAL_SUCCESS )
        {
            (void)OSAL.eLinkedListDelete( psMonObj->hAffiliateList );
        }
        SMSO_vDestroy((SMS_OBJECT)psMonObj);
    }
    return;
}

/*******************************************************************************
*
*   SPORTS_MONITOR_eSetAffiliate
*
*******************************************************************************/
SMSAPI_RETURN_CODE_ENUM SPORTS_MONITOR_eSetAffiliateMask (
    SPORTS_MONITOR_OBJECT hMonitor,
    AFFILIATE_ID tAffiliateId,
    SPORTS_MONITOR_EVENT_MASK tEventMask
        )
{
    SMSAPI_RETURN_CODE_ENUM eResult = SMSAPI_RETURN_CODE_SUCCESS;
    OSAL_RETURN_CODE_ENUM eOsalCode;
    AFFILIATE_MONITOR_STRUCT *psAffiliateMonitor;
    OSAL_LINKED_LIST_ENTRY hEntry = OSAL_INVALID_LINKED_LIST_ENTRY;
    SPORTS_MONITOR_OBJECT_STRUCT *psMonObj =
                            (SPORTS_MONITOR_OBJECT_STRUCT *)hMonitor;

    // Find the affiliate in the list.
    eOsalCode = OSAL.eLinkedListLinearSearch(
        psMonObj->hAffiliateList,
        &hEntry,
        (OSAL_LL_COMPARE_HANDLER)n16FindByAffiliateId,
        (void*)&tAffiliateId );

    if (eOsalCode == OSAL_OBJECT_NOT_FOUND )
    {
        // Construct the new affiliate list entry.
        psAffiliateMonitor = (AFFILIATE_MONITOR_STRUCT *)SMSO_hCreate(
            SPORTS_MONITOR_OBJECT_NAME":AFMon",
            sizeof(AFFILIATE_MONITOR_STRUCT),
            SMS_INVALID_OBJECT,
            FALSE );

        if( psAffiliateMonitor == NULL )
        {
            eResult = SMSAPI_RETURN_CODE_ERROR;
        }
        else
        {
            // Initialize the new affiliate monitor.
            vAffiliateMonitorInitialize (
                psAffiliateMonitor,
                tAffiliateId,
                tEventMask );

            // Add to the list.
            eOsalCode = OSAL.eLinkedListAdd(
                psMonObj->hAffiliateList,
                OSAL_INVALID_LINKED_LIST_ENTRY_PTR,
                (void*)psAffiliateMonitor );

            if ( eOsalCode != OSAL_SUCCESS )
            {
                SMSO_vDestroy( (SMS_OBJECT)psAffiliateMonitor );
                eResult = SMSAPI_RETURN_CODE_ERROR;
            }
        }
    }
    else if ( eOsalCode != OSAL_SUCCESS )
    {
        eResult = SMSAPI_RETURN_CODE_ERROR;
    }
    else
    {
        // Replace existing entry.
        AFFILIATE_MONITOR_STRUCT *psAffiliateMonitor =
                                            OSAL.pvLinkedListThis( hEntry );
        psAffiliateMonitor->tAffiliateId = tAffiliateId;
        psAffiliateMonitor->tUserMask = tEventMask;
    }

    return eResult;
}

/*******************************************************************************
*
*   SPORTS_MONITOR_eGetAffiliateMask
*
*******************************************************************************/
SMSAPI_RETURN_CODE_ENUM SPORTS_MONITOR_eGetAffiliateMask (
    SPORTS_MONITOR_OBJECT hMonitor,
    AFFILIATE_ID tAffiliateId,
    SPORTS_MONITOR_EVENT_MASK *ptEventMask
        )
{
    SMSAPI_RETURN_CODE_ENUM eResult = SMSAPI_RETURN_CODE_SUCCESS;
    OSAL_RETURN_CODE_ENUM eOsalCode;
    OSAL_LINKED_LIST_ENTRY hEntry = OSAL_INVALID_LINKED_LIST_ENTRY;
    SPORTS_MONITOR_OBJECT_STRUCT *psMonObj =
                            (SPORTS_MONITOR_OBJECT_STRUCT *)hMonitor;

    // Find the affiliate in the list.
    eOsalCode = OSAL.eLinkedListLinearSearch(
        psMonObj->hAffiliateList,
        &hEntry,
        (OSAL_LL_COMPARE_HANDLER)n16FindByAffiliateId,
        (void*)&tAffiliateId );

    if (eOsalCode == OSAL_OBJECT_NOT_FOUND )
    {
        eResult = SMSAPI_RETURN_CODE_NOT_FOUND;
    }
    else if ( eOsalCode != OSAL_SUCCESS )
    {
        eResult = SMSAPI_RETURN_CODE_ERROR;
    }
    else
    {
        // Get the mask.
        AFFILIATE_MONITOR_STRUCT *psAffiliateMonitor =
            (AFFILIATE_MONITOR_STRUCT*)OSAL.pvLinkedListThis( hEntry );
        *ptEventMask = psAffiliateMonitor->tUserMask;
    }

    return eResult;
}

/*******************************************************************************
*
*   SPORTS_MONITOR_eRemoveAllAffiliates
*
*******************************************************************************/
SMSAPI_RETURN_CODE_ENUM SPORTS_MONITOR_eRemoveAllAffiliates (
    SPORTS_MONITOR_OBJECT hMonitor
        )
{
    OSAL_RETURN_CODE_ENUM eOsalCode;
    SPORTS_MONITOR_OBJECT_STRUCT *psMonObj =
                            (SPORTS_MONITOR_OBJECT_STRUCT *)hMonitor;

    eOsalCode = OSAL.eLinkedListRemoveAll(
            psMonObj->hAffiliateList,
            (OSAL_LL_RELEASE_HANDLER)SMSO_vDestroy );

    if ( eOsalCode != OSAL_SUCCESS )
    {
        SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
            SPORTS_MONITOR_OBJECT_NAME
            ": eRemoveAllAffiliates() Failed to delete list (%s)\n",
            OSAL.pacGetReturnCodeName(eOsalCode) );
        return SMSAPI_RETURN_CODE_ERROR;
    }

    return SMSAPI_RETURN_CODE_SUCCESS;
}

/*******************************************************************************
*
*   SPORTS_MONITOR_eIterateChangedAffiliates
*
*******************************************************************************/
SMSAPI_RETURN_CODE_ENUM SPORTS_MONITOR_eIterateChangedAffiliates (
    SPORTS_MONITOR_OBJECT hMonitor,
    SPORT_MONITOR_ITERATOR bIterator,
    void *pvIteratorArg
        )
{
    OSAL_RETURN_CODE_ENUM eOsalCode;
    AFFILIATE_ITERATOR_STRUCT sIterateArg;
    SPORTS_MONITOR_OBJECT_STRUCT *psMonObj =
                            (SPORTS_MONITOR_OBJECT_STRUCT *)hMonitor;

    // Set up the iterate helping structure.
    sIterateArg.psMonObj = psMonObj;
    sIterateArg.bIterator = bIterator;
    sIterateArg.pvIteratorArg = pvIteratorArg;

    eOsalCode = OSAL.eLinkedListIterate (
        psMonObj->hAffiliateList,
        (OSAL_LL_ITERATOR_HANDLER) bIterateChangedAffiliatesHandler,
        (void*)&sIterateArg );

    // Clear the monitor level affiliate change indicator.
    psMonObj->tEventBits &= ~SPORTS_MONITOR_EVENT_AFFILIATE_INDIVIDUAL;

    if( eOsalCode == OSAL_NO_OBJECTS )
    {
        return SMSAPI_RETURN_CODE_NO_OBJECTS;
    }
    else if( eOsalCode != OSAL_SUCCESS )
    {
        return SMSAPI_RETURN_CODE_ERROR;
    }
    return SMSAPI_RETURN_CODE_SUCCESS;
}

/*******************************************************************************
*
*   SPORTS_MONITOR_bNotifySportListChange
*
*******************************************************************************/
void SPORTS_MONITOR_vNotifySportListChange (
    SPORTS_MONITOR_OBJECT hMonitor
        )
{
    SPORTS_MONITOR_OBJECT_STRUCT *psMonObj =
                            (SPORTS_MONITOR_OBJECT_STRUCT *)hMonitor;

    if( SMSO_bValid( (SMS_OBJECT)hMonitor ) )
    {
        if( psMonObj->tUserMask & SPORTS_MONITOR_EVENT_SPORTLIST )
        {
            psMonObj->tEventBits |= SPORTS_MONITOR_EVENT_SPORTLIST;
        }
    }
    return;
}

/*******************************************************************************
*
*   SPORTS_MONITOR_vNotifyAffiliateExistanceChange
*
*******************************************************************************/
void SPORTS_MONITOR_vNotifyAffiliateExistanceChange (
    SPORTS_MONITOR_OBJECT hMonitor
        )
{
    SPORTS_MONITOR_OBJECT_STRUCT *psMonObj =
                            (SPORTS_MONITOR_OBJECT_STRUCT *)hMonitor;

    if( SMSO_bValid( (SMS_OBJECT)hMonitor ) )
    {
        if( psMonObj->tUserMask & SPORTS_MONITOR_EVENT_AFFILIATE_EXISTANCE )
        {
            psMonObj->tEventBits |=    SPORTS_MONITOR_EVENT_AFFILIATE_EXISTANCE;
        }
    }
    return;
}

/*******************************************************************************
*
*   SPORTS_MONITOR_vNotifyAffiliateChange
*
*******************************************************************************/
void SPORTS_MONITOR_vNotifyAffiliateChange (
    SPORTS_MONITOR_OBJECT hMonitor,
    AFFILIATE_ID tAffiliateId,
    SPORTS_MONITOR_EVENT_MASK tChangeBits
        )
{
    OSAL_RETURN_CODE_ENUM eOsalCode;
    OSAL_LINKED_LIST_ENTRY hEntry = OSAL_INVALID_LINKED_LIST_ENTRY;
    AFFILIATE_MONITOR_STRUCT *psAffiliateMonitor;
    SPORTS_MONITOR_OBJECT_STRUCT *psMonObj =
                            (SPORTS_MONITOR_OBJECT_STRUCT *)hMonitor;

    if( !SMSO_bValid( (SMS_OBJECT)hMonitor ) )
    {
        return;
    }

    DPRINTF( "SM_AfChange: af=%d monmask=0x%x monbits=0x%x"
             " chbits=x0%x\n",
            tAffiliateId,
            psMonObj->tUserMask,
            psMonObj->tEventBits,
            tChangeBits );

    // The user is not interested in affiliate changes.
    if( !(psMonObj->tUserMask & SPORTS_MONITOR_EVENT_AFFILIATE_INDIVIDUAL) )
    {
        return;
    }

    // Find the affiliate in the list.
    eOsalCode = OSAL.eLinkedListLinearSearch(
        psMonObj->hAffiliateList,
        &hEntry,
        (OSAL_LL_COMPARE_HANDLER)n16FindByAffiliateId,
        (void*)&tAffiliateId );

    if (eOsalCode == OSAL_OBJECT_NOT_FOUND )
    {
        DPRINTF( "SM_AfChange: af=%d not found.\n", tAffiliateId );
        // Nothing to do if the affiliate is not in the list.
        return;
    }
    else if ( eOsalCode != OSAL_SUCCESS )
    {
        return;
    }

    psAffiliateMonitor =
            (AFFILIATE_MONITOR_STRUCT*)OSAL.pvLinkedListThis( hEntry );

    DPRINTF( "SM_AfChange: af=%d found."
            " afmask=0x%x afbits=0x%x monbits=0x%x\n",
            tAffiliateId,
            psAffiliateMonitor->tUserMask,
            psAffiliateMonitor->tEventBits,
            psMonObj->tEventBits );

    // Only consider what the user is interested in.
    tChangeBits &= psAffiliateMonitor->tUserMask;
    if( tChangeBits )
    {
        psAffiliateMonitor->tEventBits |= tChangeBits;
        // Record affiliate change at the monitor level.
        psMonObj->tEventBits |= SPORTS_MONITOR_EVENT_AFFILIATE_INDIVIDUAL;
        DPRINTF( "SM_AfChange: af=%d change set."
                " afbits=0x%x monbits=0x%x\n",
                tAffiliateId,
                psAffiliateMonitor->tEventBits,
                psMonObj->tEventBits );
    }

    return;
}

/*******************************************************************************
*
*   SPORTS_MONITOR_vProcessChange
*
*******************************************************************************/
void SPORTS_MONITOR_vProcessChange (
    SPORTS_MONITOR_OBJECT hMonitor
        )
{
    SPORTS_MONITOR_EVENT_MASK tBits;
    SPORTS_MONITOR_OBJECT_STRUCT *psMonObj =
                            (SPORTS_MONITOR_OBJECT_STRUCT *)hMonitor;

    if( !SMSO_bValid( (SMS_OBJECT)hMonitor ) )
    {
        return;
    }

    // User interested bits must be set.
    DPRINTF( "SM_ProcChange: mon=0x%x"
                    " mask=0x%x bits=0x%x\n",
                    (UN32)hMonitor,
                    psMonObj->tUserMask,
                    psMonObj->tEventBits );

    tBits = psMonObj->tUserMask & psMonObj->tEventBits;
    if( tBits )
    {
        psMonObj->vCallback (
            hMonitor,
            tBits,
            psMonObj->pvCallbackArg );

        // Clear the event bits.
        psMonObj->tEventBits = 0;
    }
}

/*****************************************************************************
                             PRIVATE FUNCTIONS
*****************************************************************************/
/*****************************************************************************
*
*   n16FindByAffiliateId
*
*****************************************************************************/
static N16 n16FindByAffiliateId (
    AFFILIATE_MONITOR_STRUCT *psAffiliateMonitor,
    AFFILIATE_ID *ptDesiredAffiliateId
        )
{
    if( (psAffiliateMonitor == NULL) ||
        (ptDesiredAffiliateId == NULL))
    {
        return N16_MIN;
    }

    if ( psAffiliateMonitor->tAffiliateId == *ptDesiredAffiliateId )
    {
        // Found it!
        return 0;
    }

    // Keep looking
    return 1;
}

/*****************************************************************************
*
*   bIterateChangedAffiliatesHandler
*
*****************************************************************************/
static BOOLEAN bIterateChangedAffiliatesHandler (
    AFFILIATE_MONITOR_STRUCT *psAffiliateMonitor,
    void *pvArg
        )
{
    BOOLEAN bContinue;
    SPORTS_MONITOR_EVENT_MASK tBits;
    AFFILIATE_ITERATOR_STRUCT *psIterateHelper =
        (AFFILIATE_ITERATOR_STRUCT *)pvArg;

    if( (psAffiliateMonitor == NULL) || (psIterateHelper == NULL) )
    {
        return FALSE;
    }

    // Always continue.  Only user indication can make us stop iterating
    // the list.
    bContinue = TRUE;

    DPRINTF( "SM_IterAfChange: mon=0x%x"
                    " monmask=0x%x monbits=0x%x"
                    " amask=0x%x abits=0x%x",
                    (UN32)psIterateHelper->psMonObj,
                    psIterateHelper->psMonObj->tUserMask,
                    psIterateHelper->psMonObj->tEventBits,
                    psAffiliateMonitor->tUserMask,
                    psAffiliateMonitor->tEventBits );

    // See if the user has interest in affiliate level event processing and
    // an affiliate level event happened.
    tBits = psIterateHelper->psMonObj->tUserMask &
        psIterateHelper->psMonObj->tEventBits;
    if( tBits & SPORTS_MONITOR_EVENT_AFFILIATE_INDIVIDUAL )
    {
        // See if any affiliate events the user is interested in happened.
        tBits = psAffiliateMonitor->tEventBits & psAffiliateMonitor->tUserMask;
        if( tBits )
        {
            // Let the user know the bits that changed.
            bContinue = psIterateHelper->bIterator(
                (SPORTS_MONITOR_OBJECT) psIterateHelper->psMonObj,
                psAffiliateMonitor->tAffiliateId,
                tBits,
                psIterateHelper->pvIteratorArg );

            // Clear the affiliate bits.
            psAffiliateMonitor->tEventBits = 0;
        }
    }

    return bContinue;
}
