/******************************************************************************/
/*                    Copyright (c) Sirius XM Radio, Inc.                     */
/*                            All Rights Reserved                             */
/*         Licensed Materials - Property of Sirius XM Radio, Inc.             */
/******************************************************************************/
/*******************************************************************************
 *
 * DESCRIPTION
 *
 *  This module contains the Object:SONGLIST implementation for the
 *  Sirius Module Services (SMS)
 *
 ******************************************************************************/
#include <string.h>

#include "standard.h"
#include "osal.h"

#include "sms_version.h"
#include "radio.h"
#include "sms_obj.h"
#include "sms_update.h"

#include "playback_obj.h"
#include "song_obj.h"
#include "songlist_obj.h"
#include "_songlist_obj.h"

#include "sms_api_debug.h"
static const char *gpacThisFile = __FILE__;

#if SMS_DEBUG != 1
#define SONG_vPrint(a,b)
#endif /* SMS_DEBUG == 1 */
/*****************************************************************************
                             PUBLIC FUNCTIONS
*****************************************************************************/

/*****************************************************************************
*
*   hCreate
*
*   This API is used to create a SONGLIST object. If the API is able to
*   create the list, a handle to this SONGLIST is returned. This handle
*   is used for all subsequent APIs which require a SONGLIST handle. If
*   the call to this API is unsuccessful an invalid handle is returned.
*   The call of this API owns the created SONGLIST and is responsible
*   for eventually destroying it when it is no longer needed, the associated
*   DECODER is released or the system is shutdown.
*
*   Note: the variable argument list will be defined in the future. It will
*   contain items for file based songlists such as file directory
*
*****************************************************************************/
static SONGLIST_OBJECT hCreate (
    PLAYBACK_OBJECT hPlayback,
    SONGLIST_TYPE_ENUM eSongListType,
    UN16 un16Before,
    UN16 un16After,
    SONG_EVENT_MASK tEventRequestMask,
    SONGLIST_OBJECT_EVENT_CALLBACK vEventCallback,
    void *pvEventCallbackArg,
    ...
        )
{
    SONGLIST_OBJECT_STRUCT *psObj =
        (SONGLIST_OBJECT_STRUCT *)SONGLIST_INVALID_OBJECT;

    SMSAPI_DEBUG_vPrint(SONGLIST_OBJECT_NAME, 5, "%s(%p,%d,%u,%u,%d,%p,%p)", __FUNCTION__,
        hPlayback, eSongListType, un16Before, un16After, tEventRequestMask,
        vEventCallback, pvEventCallbackArg);

    do
    {
        static UN32 un32Instance = 0;
        BOOLEAN bOk;
        OSAL_RETURN_CODE_ENUM eReturnCode;
        char acName[OSAL_MAX_OBJECT_NAME_LENGTH_WITH_NULL];

        // Verify inputs
        bOk = SMSO_bValid((SMS_OBJECT)hPlayback);
        if ((bOk == FALSE) || (eSongListType >= SONGLIST_TYPE_INVALID))
        {
            // Error!
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                             SONGLIST_OBJECT_NAME
                             ": Unable to create songlist object.");
            break;
        }

        // Construct a unique name for this channel list.
        snprintf(&acName[0],
                sizeof(acName),
                SONGLIST_OBJECT_NAME":%u",
                un32Instance++);

        // Create an instance of this object. SONGLIST objects are owned
        // by the creator and are the parent object and have a lock associated
        // with them. The lock is required because song lists may be updated
        // in the context of the DECODER via SONG object callbacks.
        psObj = (SONGLIST_OBJECT_STRUCT *)
            SMSO_hCreate(
                &acName[0],
                sizeof(SONGLIST_OBJECT_STRUCT),
                SMS_INVALID_OBJECT, // Parent
                TRUE ); // Lock
        if(psObj == NULL)
        {
            // Error!
            break;
        }

        SMSAPI_DEBUG_vPrint(SONGLIST_OBJECT_NAME, 5, "Creating Songlist %p", psObj);

        // Copy object name into SONGLIST object
        strncpy(&psObj->acName[0], &acName[0], sizeof(psObj->acName));

        // Initialize object with inputs
        psObj->hPlayback = hPlayback;
        psObj->eSongListType = eSongListType;
        psObj->un16Before = 0;
        psObj->un16After = 0;
        psObj->un16BeforeRequest = un16Before;
        psObj->un16AfterRequest = un16After;

        SMSAPI_DEBUG_vPrint(SONGLIST_OBJECT_NAME, 5, "Initializing asynchronous update configuration");

        // Initialize asynchronous update configuration
        SMSU_vInitialize(
            &psObj->sEvent,
            psObj,
            SONGLIST_OBJECT_EVENT_NONE,
            tEventRequestMask,
            (SMSAPI_OBJECT_EVENT_CALLBACK)vEventCallback,
            pvEventCallbackArg);

        SMSAPI_DEBUG_vPrint(SONGLIST_OBJECT_NAME, 5, "Initialized vCallback=%p, pvCallbackArg=%p",
            &vEventCallback, pvEventCallbackArg);

        // Initialize song list handle
        psObj->hSongList = OSAL_INVALID_OBJECT_HDL;

        // Create a linked list
        eReturnCode = OSAL.eLinkedListCreate(
            &psObj->hSongList,
            &psObj->acName[0],
            NULL,
            OSAL_LL_OPTION_CIRCULAR
                );
        if(eReturnCode != OSAL_SUCCESS)
        {
            // Error!
            break;
        }

        SMSAPI_DEBUG_vPrint(SONGLIST_OBJECT_NAME, 5, "Posting songlist creation event");
        // post an event to the decoder. the actual list construction
        // will occur there
        PLAYBACK_vPostCreateSongList(hPlayback, (SONGLIST_OBJECT)psObj);

        SMSAPI_DEBUG_vPrint(SONGLIST_OBJECT_NAME, 5, "Songlist creation event is posted");

        // Relinquish control of this object
        SMSO_vUnlock((SMS_OBJECT)psObj);

        SMSAPI_DEBUG_vPrint(SONGLIST_OBJECT_NAME, 5, "Created Songlist %p", psObj);

        return (SONGLIST_OBJECT)psObj;

    } while (FALSE);

    if (psObj != NULL)
    {
        // Destroy object
        vDestroy((SONGLIST_OBJECT)psObj);
    }

    return SONGLIST_INVALID_OBJECT;
}

/*****************************************************************************
*
*   vDestroy
*
*   This API is used to destroy an existing SONGLIST object. This API
*   will perform any removal of SONG objects from the list as required
*   and releases all resources it consumed. When this API returns to the
*   caller it should be assumed the SONGLIST object handle is no longer
*   valid. Only the original creator/owner of the SONGLIST may destroy
*   a SONGLIST.
*
*****************************************************************************/
static void vDestroy (
    SONGLIST_OBJECT hSonglist
        )
{
    PLAYBACK_OBJECT hPlayback = PLAYBACK_INVALID_OBJECT;
    BOOLEAN bLocked;

    // Verify and lock SMS Object
    bLocked =
        SMSO_bLock((SMS_OBJECT)hSonglist, OSAL_OBJ_TIMEOUT_INFINITE);
    if(bLocked == TRUE)
    {
        // De-reference object
        SONGLIST_OBJECT_STRUCT *psObj =
            (SONGLIST_OBJECT_STRUCT *)hSonglist;

        hPlayback = psObj->hPlayback;

        // Unlock object
        SMSO_vUnlock((SMS_OBJECT)hSonglist);

        // post an event to the decoder. the actual list destruction
        // will occur there
        PLAYBACK_vPostDestroySongList(hPlayback, hSonglist);
    }

    return;
}

/*****************************************************************************
*
*   tEventMask
*
*****************************************************************************/
static SONGLIST_OBJECT_EVENT_MASK tEventMask (
    SONGLIST_OBJECT hSonglist
        )
{
    SONGLIST_OBJECT_EVENT_MASK tEventMask =
        SONGLIST_OBJECT_EVENT_NONE;
    BOOLEAN bLocked;

    // Verify and lock SMS Object
    bLocked =
        SMSO_bLock((SMS_OBJECT)hSonglist, OSAL_OBJ_TIMEOUT_INFINITE);
    if(bLocked == TRUE)
    {
        // De-reference object
        SONGLIST_OBJECT_STRUCT *psObj =
            (SONGLIST_OBJECT_STRUCT *)hSonglist;

        tEventMask = SMSU_tMask(&psObj->sEvent);

        // Unlock object
        SMSO_vUnlock((SMS_OBJECT)hSonglist);
    }

    return tEventMask;
}

/*****************************************************************************
*
*   hSong
*
*   This API is used to index into a SONGLIST much like an array, using
*   a provided signed offset with respect to the reference song. This
*   offset represents the direction and index into the array or list of
*   songs this SONGLIST object possesses. The offset provided is
*   signed and will wrap around the list if the magnitude of the offset is
*   larger than the list itself. This API may only be called from the context
*   of a DECODER object (e.g. SONGLIST callback function). It cannot be
*   called outside of a DECODER.
*
*   NOTE: THIS API CAN ONLY BE DIRECTLY CALLED FROM THE CONTEXT
*   OF A DECODER OBJECT!
*
*****************************************************************************/
static SONG_OBJECT hSong (
    SONGLIST_OBJECT hSonglist,
    N16 n16Offset
        )
{
    SONGLIST_OBJECT_STRUCT *psObj = (SONGLIST_OBJECT_STRUCT *)hSonglist;
    SONG_OBJECT hSong = SONG_INVALID_OBJECT;
    OSAL_LINKED_LIST_ENTRY hCurrentEntry;
    BOOLEAN bOk;
    UN32 un32NumItems;
    int iOffset = (int)n16Offset;

    // Verify inputs
    bOk = SMSO_bValid((SMS_OBJECT)hSonglist);
    if(bOk == FALSE)
    {
        // Error!
        return SONG_INVALID_OBJECT;
    }

    // Check caller ownership. Caller may only call this
    // API if it is in the context of the PLAYBACK or it has the PLAYBACK mutex.
    bOk = SMSO_bOwner((SMS_OBJECT)psObj->hPlayback);
    if(bOk == FALSE)
    {
        // Error!
        return SONG_INVALID_OBJECT;
    }
    // Determine how big our list is and adjust offset...
    un32NumItems = 0;
    OSAL.eLinkedListItems(psObj->hSongList, &un32NumItems);
    if(un32NumItems == 0)
    {
        iOffset = 0;
    }
    else
    {
        N8 n8Sign;
        n8Sign = iOffset < 0 ? -1 : 1;
        iOffset = abs(iOffset) % un32NumItems;
        iOffset *= n8Sign;
    }

    // Assign current entry handle to the reference entry
    hCurrentEntry = psObj->hReferenceSongEntry;
    hSong = (SONG_OBJECT)OSAL.pvLinkedListThis(hCurrentEntry);

    // Based on offset provided, find the song
    do
    {
        if(iOffset > 0)
        {
            hCurrentEntry =
                OSAL.hLinkedListNext(hCurrentEntry, (void *)((void *)&hSong));
            iOffset--;
        }
        else if(iOffset < 0)
        {
            hCurrentEntry =
                OSAL.hLinkedListPrev(hCurrentEntry, (void *)((void *)&hSong));
            iOffset++;
        }

    } while(iOffset != 0);

    return hSong;
}

/*****************************************************************************
*
*   vCleanList
*
*****************************************************************************/
static void vCleanList(
    SONGLIST_OBJECT_STRUCT *psObj
        )
{
    OSAL_RETURN_CODE_ENUM eReturnCode;

    // Remove all notifications from all the entries in the linked list
    eReturnCode = OSAL.eLinkedListIterate(
        psObj->hSongList,
        bRemoveSong,
        psObj);
    if ((eReturnCode != OSAL_SUCCESS) && (eReturnCode != OSAL_NO_OBJECTS))
    {
        SMSAPI_DEBUG_vPrint(SONGLIST_OBJECT_NAME, 5, "Error iterating SongList");
    }

    // Remove all entries in the linked list
    eReturnCode = OSAL.eLinkedListRemoveAll(
        psObj->hSongList, NULL);
    if (eReturnCode != OSAL_SUCCESS)
    {
        SMSAPI_DEBUG_vPrint(SONGLIST_OBJECT_NAME, 5, "Error cleaning SongList");
    }
}

/*****************************************************************************
*
*   eBrowseList
*
*****************************************************************************/
static SMSAPI_RETURN_CODE_ENUM eBrowseList (
    SONGLIST_OBJECT hSonglist,
    N16 n16Offset
        )
{
    SMSAPI_RETURN_CODE_ENUM eReturnCode = SMSAPI_RETURN_CODE_ERROR;
    BOOLEAN bLocked, bValid;
    N16 n16ReferenceSongOffset;
    N16 n16NumSongsInSCache;
    SCACHE_OBJECT hSCache;
    PLAYBACK_OBJECT hPlayback;
    SONG_OBJECT hReferenceSong;
    SONGLIST_OBJECT_STRUCT *psObj = (SONGLIST_OBJECT_STRUCT *)hSonglist;

    // Verify inputs
    bValid = SMSO_bValid((SMS_OBJECT)hSonglist);
    if(bValid == FALSE)
    {
        // Error!
        return SMSAPI_RETURN_CODE_ERROR;
    }

    // Lock the associated PLAYBACK when using it to manipulate
    // SONG information
    hPlayback = psObj->hPlayback;
    bLocked = SMSO_bLock((SMS_OBJECT)hPlayback, OSAL_OBJ_TIMEOUT_INFINITE);
    if(bLocked == FALSE)
    {
        // Error!
        return SMSAPI_RETURN_CODE_ERROR;
    }

    hSCache = PLAYBACK_hSCache(hPlayback);
    hReferenceSong =
        (SONG_OBJECT)OSAL.pvLinkedListThis(psObj->hReferenceSongEntry);

    // get the reference song's absolute offset in the scache
    n16ReferenceSongOffset = SONG_n16Offset(hReferenceSong);

    if (n16Offset < 0)
    {
        if (n16ReferenceSongOffset == 0)
        {
            // cannot browse down any further, we're already at the limit
        }
        else
        {
            // browse down

            // Cleanup the list
            vCleanList(psObj);

            if (n16ReferenceSongOffset+n16Offset < 0)
            {
                // the requested browse offset is beyond the limit of the
                // scache - clip to the first entry
                vPopulateSongList(psObj, hSCache, 0);
            }
            else
            {
                // repopulate list
                vPopulateSongList(psObj,
                                  hSCache,
                                  (N16)(n16ReferenceSongOffset+n16Offset));
            }
        }
    } // if (n16Offset < 0)
    else if (n16Offset > 0)
    {
        n16NumSongsInSCache = SCACHE_n16NumberOfSongs(hSCache);
        if (n16ReferenceSongOffset == (n16NumSongsInSCache-1))
        {
            // cannot browse up any further, we're already at the limit
        }
        else
        {
            // browse up

            // remove all songs from the lsit
            vCleanList(psObj);

            // repopulate list
            if (n16ReferenceSongOffset+n16Offset > (n16NumSongsInSCache-1))
            {
                // the requested browse offset is beyond the limit of the
                // scache - clip to the last entry
                vPopulateSongList(psObj, hSCache, (N16)(n16NumSongsInSCache-1));
            }
            else
            {
                vPopulateSongList(psObj,
                                  hSCache,
                                  (N16)(n16ReferenceSongOffset+n16Offset));
            }
        }
    } // if (n16Offset > 0)
    else
    {
        // no browsing required...
    }

    // Set event mask
    SMSU_tUpdate(&psObj->sEvent, SONGLIST_OBJECT_EVENT_UPDATE);

    // call any registerd callback for this SONGLIST
    SONGLIST_vUpdate((SONGLIST_OBJECT)psObj);

    // Unlock PLAYBACK
    SMSO_vUnlock((SMS_OBJECT)hPlayback);
    eReturnCode = SMSAPI_RETURN_CODE_SUCCESS;

    return eReturnCode;
}

/*****************************************************************************
*
*   eSize
*
*****************************************************************************/
static SMSAPI_RETURN_CODE_ENUM eSize (
    SONGLIST_OBJECT hSonglist,
    size_t *ptSize,
    UN16 *pun16Before,
    UN16 *pun16After
        )
{
    SMSAPI_RETURN_CODE_ENUM eReturnCode = SMSAPI_RETURN_CODE_ERROR;
    BOOLEAN bLocked;

    // Verify and lock the SONGLIST object
    bLocked = SMSO_bLock((SMS_OBJECT)hSonglist, OSAL_OBJ_TIMEOUT_INFINITE);
    if(bLocked == TRUE)
    {
        SONGLIST_OBJECT_STRUCT *psObj =
            (SONGLIST_OBJECT_STRUCT *)hSonglist;
        UN32 un32Items = 0;

        OSAL.eLinkedListItems(
            psObj->hSongList, &un32Items);

        if(ptSize != NULL)
        {
            *ptSize = (size_t)un32Items;
        }

        if(pun16Before != NULL)
        {
            *pun16Before = psObj->un16Before;
        }

        if(pun16After != NULL)
        {
            *pun16After = psObj->un16After;
        }

        // Unlock SONGLIST
        SMSO_vUnlock((SMS_OBJECT)hSonglist);
        eReturnCode = SMSAPI_RETURN_CODE_SUCCESS;
    }

    return eReturnCode;
}

/*****************************************************************************
                             FRIEND FUNCTIONS
*****************************************************************************/


/*****************************************************************************
*
*   SONGLIST_vUpdate
*
*   This API exists so that it may be called exclusively by the DECODER
*   object. The purpose is to trigger a SONGLIST event when any event
*   change has occured. SONGLIST events are always called from the
*   context of the DECODER object.
*
*   NOTE: THIS API MAY BE CALLED ONLY FROM THE CONTEXT OF A DECODER OBJECT!
*
*****************************************************************************/
void SONGLIST_vUpdate (
    SONGLIST_OBJECT hSonglist
        )
{
    SONGLIST_OBJECT_STRUCT *psObj =
        (SONGLIST_OBJECT_STRUCT *)hSonglist;
    BOOLEAN bLocked;

    // Lock SONGLIST object since this call is always made from
    // the context of the DECODER.
    bLocked = SMSO_bLock((SMS_OBJECT)hSonglist, OSAL_OBJ_TIMEOUT_INFINITE);
    if(bLocked == TRUE)
    {
            // Notify callback if registered and change occured
            SMSU_bNotify(&psObj->sEvent);

        // Unlock SONGLIST
        SMSO_vUnlock((SMS_OBJECT)hSonglist);
    }

    return;
}


/*****************************************************************************
*
*   SONGLIST_vCreate
*
*   This API exists so that it may be called exclusively by the DECODER
*   object. The purpose is to create a song list
*
*   NOTE: THIS API MAY BE CALLED ONLY FROM THE CONTEXT OF A DECODER OBJECT!
*
*****************************************************************************/
void SONGLIST_vCreate(SONGLIST_OBJECT hSonglist, PLAYBACK_OBJECT hPlayback)
{
    BOOLEAN bValid, bOk = FALSE;
    SCACHE_OBJECT hSCache;
    SONGLIST_OBJECT_STRUCT *psObj =
        (SONGLIST_OBJECT_STRUCT *)hSonglist;

    bValid = SMSO_bValid((SMS_OBJECT)hSonglist);

    if (bValid == TRUE)
    {
        // Create a list...
        bOk = bCreateList(hSonglist);

        if(bOk == FALSE)
        {
            // Error!
            vDestroy(hSonglist);
            return;
        }

        hSCache = PLAYBACK_hSCache(hPlayback);

        // we need to get notified of scache events (songs added / removed etc)
        bOk = SCACHE_bRegisterNotification (
            hSCache,
            SCACHE_OBJECT_EVENT_ALL,
            vSCacheEventCallback,
            psObj
                );

        if(bOk == FALSE)
        {
            // Error!
            vDestroy(hSonglist);
            return;
        }

        // Set event mask
        SMSU_tUpdate(&psObj->sEvent, SONGLIST_OBJECT_EVENT_UPDATE);

        // call any registerd callback for this SONGLIST
        SONGLIST_vUpdate((SONGLIST_OBJECT)psObj);
    }
    return;
}

/*****************************************************************************
*
*   SONGLIST_vDestroy
*
*   This API is used to destroy an existing SONGLIST object. This API
*   will perform any removal of SONG objects from the list as required
*   and releases all resources it consumed. When this API returns to the
*   caller it should be assumed the SONGLIST object handle is no longer
*   valid. Only the original creator/owner of the SONGLIST may destroy
*   a SONGLIST.
*
*****************************************************************************/
void SONGLIST_vDestroy (
    SONGLIST_OBJECT hSonglist
        )
{
    SONGLIST_OBJECT_STRUCT *psObj =
        (SONGLIST_OBJECT_STRUCT *)hSonglist;
    OSAL_RETURN_CODE_ENUM eReturnCode;
    BOOLEAN bLocked, bValid;
    SCACHE_OBJECT hSCache;
    PLAYBACK_OBJECT hPlayback = PLAYBACK_INVALID_OBJECT;

    SMSAPI_DEBUG_vPrint(SONGLIST_OBJECT_NAME, 5, "Destroying Songlist %p", psObj);
    // Validate the songlist object
    bValid = SMSO_bValid( (SMS_OBJECT)hSonglist );
    if (bValid == TRUE)
    {
        // Grab the playback handle
        hPlayback = psObj->hPlayback;
    }

    // Filter out all further song updates
    SMSU_tFilter(&psObj->sEvent, SONG_OBJECT_EVENT_NONE);

    // Verify and lock SMS object
    bLocked = SMSO_bLock((SMS_OBJECT)hPlayback, OSAL_OBJ_TIMEOUT_INFINITE);
    if(bLocked == FALSE)
    {
        return;
    }

    // Verify and lock SMS object
    bLocked = SMSO_bLock((SMS_OBJECT)hSonglist, OSAL_OBJ_TIMEOUT_INFINITE);
    if(bLocked == FALSE)
    {
        // Error!
        SMSO_vUnlock((SMS_OBJECT)psObj->hPlayback);
        return;
    }

    hSCache = PLAYBACK_hSCache(psObj->hPlayback);
    SCACHE_vUnregisterNotification(hSCache, vSCacheEventCallback, psObj);

    // Initialize song list parameters...
    psObj->hPlayback = PLAYBACK_INVALID_OBJECT;
    psObj->un16Before = 0;
    psObj->un16After = 0;
    psObj->un16BeforeRequest = 0;
    psObj->un16AfterRequest = 0;

    // Initialize song entry info
    psObj->hReferenceSongEntry = OSAL_INVALID_LINKED_LIST_ENTRY;
    psObj->hTopSongEntry = OSAL_INVALID_LINKED_LIST_ENTRY;
    psObj->hBottomSongEntry = OSAL_INVALID_LINKED_LIST_ENTRY;

    // Destroy song list if one exists
    if(psObj->hSongList != OSAL_INVALID_OBJECT_HDL)
    {
        // Remove all entries in the linked list
        vCleanList(psObj);

        // Delete linked list itself
        eReturnCode = OSAL.eLinkedListDelete(
            psObj->hSongList);
        if(eReturnCode == OSAL_SUCCESS)
        {
            psObj->hSongList = OSAL_INVALID_OBJECT_HDL;
        }
    }

    // Destory the async update configuration
    SMSU_vDestroy(&psObj->sEvent);

    // Free object instance
    SMSO_vDestroy((SMS_OBJECT)hSonglist);

    SMSO_vUnlock((SMS_OBJECT)hPlayback);

    SMSAPI_DEBUG_vPrint(SONGLIST_OBJECT_NAME, 5, "Destroyed Songlist %p", hSonglist);

    return;
}

/*****************************************************************************
                             PRIVATE FUNCTIONS
*****************************************************************************/

/*****************************************************************************
*
*   bAddSong
*
*   The purpose of this function is to add a song to the songlist
*
*
*****************************************************************************/
static BOOLEAN bAddSong(
    SONGLIST_OBJECT hSonglist,
    SONG_OBJECT hSong,
    BOOLEAN bBefore
        )
{
    OSAL_LINKED_LIST_ENTRY hSongEntry;
    OSAL_RETURN_CODE_ENUM eReturnCode;

    BOOLEAN bValid, bReturn = FALSE;

    // Verify and lock the SONGLIST object
    bValid = SMSO_bValid((SMS_OBJECT)hSonglist);
    if(bValid == TRUE)
    {
        SONGLIST_OBJECT_STRUCT *psObj =
            (SONGLIST_OBJECT_STRUCT *)hSonglist;

        if (bBefore == TRUE)
        {
            // Add this song to the song list
            hSongEntry = psObj->hTopSongEntry;
            if (hSongEntry != OSAL_INVALID_LINKED_LIST_ENTRY)
            {
               SONG_OBJECT hTopSong = (SONG_OBJECT)OSAL.pvLinkedListThis(hSongEntry);
               SCACHE_OBJECT hSCache = PLAYBACK_hSCache(psObj->hPlayback);
               SONG_OBJECT hTailSong = SCACHE_hGetTail(hSCache);
               N16 n16TopSongOffset = SONG_n16Offset(hTopSong);
               N16 n16SCacheTailOffset = SONG_n16Offset(hTailSong);
               if (n16TopSongOffset != n16SCacheTailOffset)
               {
                    // if not "live"
                    eReturnCode = OSAL.eLinkedListAddBeforeEntry(
                                      psObj->hSongList,
                                      &hSongEntry,
                                      hSong);
                    psObj->hTopSongEntry = hSongEntry;
                    psObj->un16Before++;
               }
               else
               {
                   // else live,
                   if (psObj->un16Before < psObj->un16BeforeRequest)
                   {
                       //so add after the live entry
                       eReturnCode = OSAL.eLinkedListAddAfterEntry(
                                        psObj->hSongList,
                                        &hSongEntry,
                                        hSong);

                       psObj->un16Before++;
                   }
                   else
                   {
                       // Replace entry
                       SONG_OBJECT hOldSong =
                           (SONG_OBJECT)OSAL.pvLinkedListThis(hSongEntry);
                       vUnregisterNotification(psObj, hOldSong);

                       eReturnCode = OSAL.eLinkedListReplaceEntry(
                           psObj->hSongList,
                           hSongEntry,
                           hSong);
                   }
               }
            }
            else
            {
                eReturnCode = OSAL.eLinkedListAdd(
                      psObj->hSongList,
                      &hSongEntry,
                      hSong);

                // might want to check for successful add before setting
                psObj->hTopSongEntry = hSongEntry;
                psObj->un16Before++;

            }

        } // if (hSongEntry != OSAL_INVALID_LINKED_LIST_ENTRY)
        else
        {
            // Add this song to the song list

            hSongEntry = psObj->hReferenceSongEntry;
            //so add after the live entry
            eReturnCode = OSAL.eLinkedListAddAfterEntry(
                              psObj->hSongList,
                              &hSongEntry,
                              hSong);

            if (psObj->un16After == 0)
            {
                // there weren't any after, so set this to the bottom and
                // incrrment after count

                // might want to check for successful add before setting
                psObj->hBottomSongEntry = hSongEntry;
                psObj->un16After++;
            }
            else if (psObj->un16After <  psObj->un16AfterRequest)
            {
                // leave the bottom, just stick this one in
                psObj->un16After++;
            }
            else
            {
                OSAL_LINKED_LIST_ENTRY hOldBottom, hNewBottom;
                SONG_OBJECT hBottomSong;

                hOldBottom = psObj->hBottomSongEntry;
                hBottomSong = (SONG_OBJECT)OSAL.pvLinkedListThis(hOldBottom);
                vUnregisterNotification(psObj, hBottomSong);

                if (hOldBottom != OSAL_INVALID_LINKED_LIST_ENTRY)
                {
                    hNewBottom = OSAL.hLinkedListPrev(
                                   hOldBottom,
                                   NULL
                                       );

                    OSAL.eLinkedListRemove(hOldBottom);
                    //vDeleteSong
                    psObj->hBottomSongEntry = hNewBottom;
                }
            }
        }

        if(eReturnCode == OSAL_SUCCESS)
        {
            bRegisterNotification(psObj, hSong);
            bReturn =  TRUE;
        }
    } // if(bValid == TRUE)

    return bReturn;
}

/*****************************************************************************
*
*   vRemoveSong
*
*   The purpose of this function is to remove a song from the songlist
*
*****************************************************************************/
static BOOLEAN bRemoveSong(
    void *pvData,
    void *pvArg)
{
    SONG_OBJECT hSong = (SONG_OBJECT)pvData;
    SONG_vUnregisterNotification(hSong, vSongEventCallback, pvArg);
    SMSAPI_DEBUG_vPrint("SONG", 5, "bRemoveSong(%p, %p, %p)", hSong, vSongEventCallback, pvArg);

    return TRUE;
}


/*****************************************************************************
*
*   bCreateList
*
*****************************************************************************/
static BOOLEAN bCreateList (
    SONGLIST_OBJECT hSonglist
        )
{
    SONGLIST_OBJECT_STRUCT *psObj =
        (SONGLIST_OBJECT_STRUCT *)hSonglist;
    SONG_OBJECT hSong;
    SCACHE_OBJECT hSCache;
    BOOLEAN bValid;
    N16 n16ReferenceSongOffset;

    // Verify inputs
    bValid = SMSO_bValid((SMS_OBJECT)hSonglist);
    if(bValid == FALSE)
    {
        // Error!
        return FALSE;
    }

    // Determine which type of list this is and configure list create
    // parameters based on that.
    if(psObj->eSongListType == SONGLIST_TYPE_TIME_SHIFTED_CONTENT)
    {
        // nothing to configure yet
    }
    else
    {
        // Error! Unknown type
        return FALSE;
    }

    // From the PLAYBACK handle, determine the song cache handle
    hSCache = PLAYBACK_hSCache(psObj->hPlayback);

    // From the scache, get the current song offset.
    // this is used as the reference
    hSong = SCACHE_hCurrentSong(hSCache);
    if(hSong != SONG_INVALID_OBJECT)
    {
        n16ReferenceSongOffset = SCACHE_n16CurrentSongOffset(hSCache);

        SMSAPI_DEBUG_vPrint(SONGLIST_OBJECT_NAME, 5, "Reference song %d", 
            n16ReferenceSongOffset);

        // Now, generate the requested song list...
        vPopulateSongList(
            psObj, hSCache, n16ReferenceSongOffset);

        // Indicate the song list has been modified
        SMSU_tUpdate(&psObj->sEvent, SONGLIST_OBJECT_EVENT_ALL);
    }

#if DEBUG_OBJECT == 1

    // DEBUG
    {
        N16 n16AbsoluteOffset;
        SONG_OBJECT hSong;
        OSAL_LINKED_LIST_ENTRY hSongEntry =
            psObj->hReferenceSongEntry;

        printf("\nSong List:\n\n");
        hSong = (SONG_OBJECT)
            OSAL.pvLinkedListThis(psObj->hTopSongEntry);
        n16AbsoluteOffset = SONG_n16Offset(hSong);
        printf("Top Song: \t\t%u\n", n16AbsoluteOffset);
        hSong = (SONG_OBJECT)
            OSAL.pvLinkedListThis(psObj->hReferenceSongEntry);
        n16AbsoluteOffset = SONG_n16Offset(hSong);
        printf("Reference Song: \t%u\n", n16AbsoluteOffset);
        hSong = (SONG_OBJECT)
            OSAL.pvLinkedListThis(psObj->hBottomSongEntry);
        n16AbsoluteOffset = SONG_n16Offset(hSong);
        printf("Bottom Song: \t\t%u\n\n", n16AbsoluteOffset);

        printf("\n");
        hSongEntry = psObj->hTopSongEntry;

        do
        {

            hSong = (SONG_OBJECT)
                OSAL.pvLinkedListThis(hSongEntry);
            n16AbsoluteOffset = SONG_n16Offset(hSong);
            if(hSongEntry == psObj->hReferenceSongEntry)
            {
                printf("*Song#%u\n", n16AbsoluteOffset);
            }
            else
            {
                printf("Song #%u\n", n16AbsoluteOffset);
            }

            hSongEntry =
                OSAL.hLinkedListNext(hSongEntry, (void**)&hSong);

        } while(hSongEntry != psObj->hTopSongEntry);

        printf("\n");
    }

#endif /* DEBUG_OBJECT == 1 */

    // Done!
    return TRUE;
}

/*****************************************************************************
*
*   vPopulateSongList
*
*****************************************************************************/
static void vPopulateSongList (
    SONGLIST_OBJECT_STRUCT *psObj,
    SCACHE_OBJECT hSCache,
    N16 n16ReferenceSongOffset
        )
{
    OSAL_RETURN_CODE_ENUM eReturnCode;
    SONG_OBJECT hSong;
    OSAL_LINKED_LIST_ENTRY hSongEntry;
    UN16 un16Before, un16After;
    N16 n16Offset;
    BOOLEAN bSuccess;
#if DEBUG_OBJECT
    STRING_OBJECT hSongName;
#endif

    // Initialize
    un16Before = 0;
    un16After = 0;

    // Get song handle from cache, pulling from the pool of
    // songs available in the cache. Here we are selecting to
    // retrieve the nth song in the list of all available songs
    hSong = SCACHE_hSong(hSCache, n16ReferenceSongOffset);

#if DEBUG_OBJECT
    hSongName = SONG.hTitle(hSong);
    SMSAPI_DEBUG_vPrint(SONGLIST_OBJECT_NAME, 5, "Reference song (%d) is %s",
        n16ReferenceSongOffset, STRING.pacCStr(hSongName));
#endif

    // Check that we have a valid song handle. If not, bail
    // and try to get the next one. Not sure what else we can do.
    if(hSong == SONG_INVALID_OBJECT)
    {
        // Error!
        return;
    }

    bSuccess = bRegisterNotification(psObj, hSong);
    if(bSuccess == FALSE)
    {
        // Error!
        return;
    }

    // Add the reference song to the song list
    hSongEntry = OSAL_INVALID_LINKED_LIST_ENTRY;
    eReturnCode = OSAL.eLinkedListAdd(
                      psObj->hSongList,
                      &hSongEntry,
                      hSong);
    if(eReturnCode != OSAL_SUCCESS)
    {
        // Error!
        vUnregisterNotification(psObj, hSong);
        return;
    }

    SMSAPI_DEBUG_vPrint(SONGLIST_OBJECT_NAME, 2, "Setting hReferenceSongEntry");

    // Set reference, top and bottom channel entries
    psObj->hTopSongEntry =
            psObj->hBottomSongEntry =
                psObj->hReferenceSongEntry =
                    hSongEntry;

    // now we will populate list with songs "before" the reference song
    if (psObj->un16BeforeRequest > 0)
    {
        while(un16Before < psObj->un16BeforeRequest)
        {
            // compute the offset of the next "before" song
            n16Offset =  n16ReferenceSongOffset+((N16)un16Before+1);
            // try to get the "before" song
            hSong = SCACHE_hSong(hSCache, n16Offset);

            if(hSong == SONG_INVALID_OBJECT)
            {
                // there is not a song in the scache at that offset,
                // so that means we have gotten all the "before" songs we could
                break;
            }

            // we want to get the events for this song, so register for them
            bSuccess = bRegisterNotification(psObj, hSong);
            if(bSuccess == FALSE)
            {
                // Error!
                return;
            }

#if DEBUG_OBJECT
            hSongName = SONG.hTitle(hSong);
            SMSAPI_DEBUG_vPrint(SONGLIST_OBJECT_NAME, 5, "Adding song before %d: %s",
                n16Offset, STRING.pacCStr(hSongName));
#endif

            // Add this song to the song list, before the last one added
            eReturnCode = OSAL.eLinkedListAddBeforeEntry(
                              psObj->hSongList,
                              &hSongEntry,
                              hSong);

            if(eReturnCode != OSAL_SUCCESS)
            {
                // Error!
                vUnregisterNotification(psObj, hSong);
                return;
            }
            // now this is the top of the list
            psObj->hTopSongEntry = hSongEntry;
            // increment our before count
            un16Before++;
        } // while(un16Before < psObj->un16BeforeRequest)
    } // if (psObj->un16BeforeRequest > 0)

    // okay, we are done with the "before" songs

    // reset this to the reference song
    hSongEntry = psObj->hReferenceSongEntry;

    // now we'll do the "after" ones
    if (psObj->un16AfterRequest > 0)
    {
        while(un16After < psObj->un16AfterRequest)
        {
            // compute the offset of the next "after" song
            n16Offset =  n16ReferenceSongOffset-((N16)un16After+1);

            // try to get the "after" song
            hSong = SCACHE_hSong(hSCache, n16Offset);

           if(hSong == SONG_INVALID_OBJECT)
           {
              // there is not a song in the scache at that offset,
              // so that means we have gotten all the "before" songs we could
              break;
           }

            // we want to get the events for this song, so register for them
            bSuccess = bRegisterNotification(psObj, hSong);
            if(bSuccess == FALSE)
            {
                // Error!
                return;
            }

#if DEBUG_OBJECT
            hSongName = SONG.hTitle(hSong);
            SMSAPI_DEBUG_vPrint(SONGLIST_OBJECT_NAME, 5, "Adding song after %d: %s",
                n16Offset, STRING.pacCStr(hSongName));
#endif

            // Add this song to the song list, after the last one
            eReturnCode = OSAL.eLinkedListAddAfterEntry(
                              psObj->hSongList,
                              &hSongEntry,
                              hSong);

            if(eReturnCode != OSAL_SUCCESS)
            {
                // Error!
                vUnregisterNotification(psObj, hSong);
                return;
            }

            // now this is the bottom of the list
            psObj->hBottomSongEntry = hSongEntry;
            // increment our after count
            un16After++;
        } // while(un16After < psObj->un16AfterRequest)
    } // if (psObj->un16AfterRequest > 0)

    // Set new values of before & after (may be truncated)
    psObj->un16Before = un16Before;
    psObj->un16After = un16After;

    // Done
    return;
}

/*****************************************************************************
*
*   bRegisterNotification
*
*****************************************************************************/
static BOOLEAN bRegisterNotification(
    SONGLIST_OBJECT_STRUCT *psObj,
    SONG_OBJECT hSong
        )
{
    // Assign an update callback to the song
    return
        SONG_bRegisterNotification(
            hSong,
            SONG_OBJECT_EVENT_ALL,
            vSongEventCallback,
            psObj);
}

/*****************************************************************************
*
*   vUnregisterNotification
*
*****************************************************************************/
static void vUnregisterNotification(
    SONGLIST_OBJECT_STRUCT *psObj,
    SONG_OBJECT hSong
        )
{
    bRemoveSong(hSong, psObj);
    return;
}

/*******************************************************************************
*
*   vSongEventCallback
*
*   This callback function is used to register with SONG objects which
*   belong to a specific SONGLIST. The purpose of this function is to
*   simply apply any SONG object event to the entire SONGLIST object.
*   when this occurs a SONGLIST object event will occur as well.
*
*   THIS FUNCTION IS EXCLUSIVELY CALLED IN THE CONTEXT OF A DECODER
*
*****************************************************************************/
static void vSongEventCallback (
    SONG_OBJECT hSong,
    SONG_EVENT_MASK tEventMask,
    void *pvEventCallbackArg
        )
{
    SONGLIST_OBJECT_STRUCT *psObj =
        (SONGLIST_OBJECT_STRUCT *)pvEventCallbackArg;
    SONGLIST_OBJECT hSonglist =
        (SONGLIST_OBJECT)pvEventCallbackArg;
    BOOLEAN bLocked;

    // Lock SONGLIST object
    bLocked = SMSO_bLock((SMS_OBJECT)hSonglist, OSAL_OBJ_TIMEOUT_INFINITE);
    if(bLocked == TRUE)
    {
        //    set event
        SMSU_tUpdate(&psObj->sEvent, SONGLIST_OBJECT_EVENT_SONG);

        // call any registered callback for this SONGLIST
        SONGLIST_vUpdate((SONGLIST_OBJECT)psObj);

        // Unlock SONGLIST
        SMSO_vUnlock((SMS_OBJECT)hSonglist);
    }

    return;
}

/*******************************************************************************
*
*   vSCacheEventCallback
*
*   This callback function is used to registered with the SCACHE object which
*   is associated with a specific SONGLIST. The purpose of this function is to
*   simply apply any SCACHE object event to the entire SONGLIST object..
*
*   THIS FUNCTION IS EXCLUSIVELY CALLED IN THE CONTEXT OF A DECODER
*
*****************************************************************************/
static void vSCacheEventCallback (
    SCACHE_OBJECT hSCache,
    SCACHE_EVENT_MASK tEventMask,
    void *pvEventCallbackArg
        )
{
    SONGLIST_OBJECT_STRUCT *psObj =
        (SONGLIST_OBJECT_STRUCT *)pvEventCallbackArg;
    SONGLIST_OBJECT hSonglist =
        (SONGLIST_OBJECT)pvEventCallbackArg;
    BOOLEAN bLocked;

    // Lock SONGLIST object
    bLocked = SMSO_bLock((SMS_OBJECT)hSonglist, OSAL_OBJ_TIMEOUT_INFINITE);
    if(bLocked == TRUE)
    {
        vSCacheInternalEventHandler(psObj, hSCache, tEventMask);

        // Unlock SONGLIST
        SMSO_vUnlock((SMS_OBJECT)hSonglist);
    }
    return;
}

/*******************************************************************************
*
*   vSCacheInternalEventHandler
*
*   This callback function is used to handle events sent by the SCACHE.
*   The SCACHE is an internal object, so we handle these events internally
*   and they do not get passed to the application.
*
*   THIS FUNCTION IS EXCLUSIVELY CALLED IN THE CONTEXT OF A DECODER
*
*****************************************************************************/
static void vSCacheInternalEventHandler(SONGLIST_OBJECT_STRUCT *psObj,
                                        SCACHE_OBJECT hSCache,
                                        SCACHE_EVENT_MASK tEventMask)
{
    N16 n16RefOffset;

    if (tEventMask & SCACHE_OBJECT_EVENT_FLUSHED)
    {
        SMSAPI_DEBUG_vPrint(SONGLIST_OBJECT_NAME, 2, "SCACHE_OBJECT_EVENT_FLUSHED");

        // Remove all entries in the linked list
        vCleanList(psObj);

        // reset
        psObj->un16Before = 0;
        psObj->un16After = 0;
        psObj->hReferenceSongEntry = OSAL_INVALID_LINKED_LIST_ENTRY;
        psObj->hTopSongEntry = OSAL_INVALID_LINKED_LIST_ENTRY;
        psObj->hBottomSongEntry = OSAL_INVALID_LINKED_LIST_ENTRY;

        SMSU_tUpdate(&psObj->sEvent, SONGLIST_OBJECT_EVENT_UPDATE);

        // call any registered callback for this SONGLIST
        SONGLIST_vUpdate((SONGLIST_OBJECT)psObj);
    } // if (tEventMask & SCACHE_OBJECT_EVENT_FLUSHED)

    // 'Ready' event is intended to notify the client that all songs
    // have been received and offset of the current song is in range. 
    // Also it is re-used to notify the client, that it should 
    // update the reference song position.
    if (tEventMask & SCACHE_OBJECT_EVENT_READY)
    {
        SMSAPI_DEBUG_vPrint(SONGLIST_OBJECT_NAME, 2, "SCACHE_OBJECT_EVENT_READY");

        // we need to remember what our reference is
        n16RefOffset = SCACHE_n16CurrentSongOffset(hSCache);

        // rebuild
        vPopulateSongList(psObj, hSCache, n16RefOffset);

        // set event
        SMSU_tUpdate(&psObj->sEvent, SONGLIST_OBJECT_EVENT_UPDATE);

        // call any registered callback for this SONGLIST
        SONGLIST_vUpdate((SONGLIST_OBJECT)psObj);

    } // if (tEventMask & SCACHE_OBJECT_EVENT_READY)

    if (tEventMask & SCACHE_OBJECT_EVENT_NEW_SONG)
    {
        // a new song was added

        // determine if the new song is in our "window"
        SONG_OBJECT hSong = SONG_INVALID_OBJECT;

        SMSAPI_DEBUG_vPrint(SONGLIST_OBJECT_NAME, 2, "SCACHE_OBJECT_EVENT_NEW_SONG");

        if (psObj->hReferenceSongEntry != OSAL_INVALID_LINKED_LIST_ENTRY)
        {
            hSong = (SONG_OBJECT)
                       OSAL.pvLinkedListThis(psObj->hReferenceSongEntry);
            n16RefOffset = SONG_n16Offset(hSong);

            // Remove all entries in the linked list
            vCleanList(psObj);

            // reset
            psObj->un16Before = 0;
            psObj->un16After = 0;
            psObj->hReferenceSongEntry = OSAL_INVALID_LINKED_LIST_ENTRY;
            psObj->hTopSongEntry = OSAL_INVALID_LINKED_LIST_ENTRY;
            psObj->hBottomSongEntry = OSAL_INVALID_LINKED_LIST_ENTRY;

            SMSAPI_DEBUG_vPrint(SONGLIST_OBJECT_NAME, 5, 
                "Rebuilding songlist around ref song %d", n16RefOffset);

            // rebuilt, using previous reference (if applicable)
            vPopulateSongList(psObj, hSCache, n16RefOffset);

            //    set event
            SMSU_tUpdate(&psObj->sEvent, SONGLIST_OBJECT_EVENT_UPDATE);

            SMSAPI_DEBUG_vPrint(SONGLIST_OBJECT_NAME, 2, "SONGLIST_OBJECT_EVENT_UPDATE");

            // call any registered callback for this SONGLIST
            SONGLIST_vUpdate((SONGLIST_OBJECT)psObj);
        }

    } // if (tEventMask & SCACHE_OBJECT_EVENT_NEW_SONG)

    if (tEventMask & SCACHE_OBJECT_EVENT_REMOVE_SONG)
    {
        BOOLEAN bReferenceUpdated = TRUE;

       // a song was purged from the scache
       // determine if the removed song is in our "window"

        N16 n16NumSongs = SCACHE_n16NumberOfSongs(hSCache);

       // get the reference song and its absolute offset in the scache
       SONG_OBJECT hSong = (SONG_OBJECT)
            OSAL.pvLinkedListThis(psObj->hReferenceSongEntry);

       SMSAPI_DEBUG_vPrint(SONGLIST_OBJECT_NAME, 2, "SCACHE_OBJECT_EVENT_REMOVE_SONG");

       n16RefOffset = SONG_n16Offset(hSong);

       // songs removed from the head of the scache (absolute index = 0)
       if ( (n16RefOffset - psObj->un16After) <= 0 )
       {
           // the song is in our window

           SONG_OBJECT hNewBottomSong;

           // get the bottom song
           OSAL_LINKED_LIST_ENTRY hOldBottomEntry = psObj->hBottomSongEntry;

            if (psObj->hBottomSongEntry == OSAL_INVALID_LINKED_LIST_ENTRY)
            {
                // Perfect! Looks like some crazy things happen in Module.
                // This could happen when not all IRRecordMetadataInd were
                // reported. Let's dump the SCache stats and bail.
                // Note: consider to drive the oldest song removal under the
                // IRRecordMetadataInd directly.

                N16 n16CurrentSongOffset;

                n16CurrentSongOffset = SCACHE_n16CurrentSongOffset(hSCache);

                SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    SONGLIST_OBJECT_NAME": SCache remove song has arrived for empty list!: "
                    "Songs: %d, Current: %d", n16NumSongs, n16CurrentSongOffset);
                return;
            }

           // find the new bottom entry
           psObj->hBottomSongEntry = OSAL.hLinkedListPrev(
                                         hOldBottomEntry,
                                          (void **)((void *)&hNewBottomSong)
                                             );
            if (hOldBottomEntry == psObj->hBottomSongEntry)
            {
                // this is a cirular ll and we just wrapped around
                bReferenceUpdated = FALSE;
            }

           // was the old bottom the reference song?
           if (hOldBottomEntry == psObj->hReferenceSongEntry)
           {
                // yes, the old bottom song is the reference.
                // change the reference to the new bottom song entry
               psObj->hReferenceSongEntry = psObj->hBottomSongEntry;

               // since the reference song changed,
               // see if we should add a song before.
               n16RefOffset = SONG_n16Offset(hNewBottomSong);

               // this song is now the reference, but it used to be a "before"
               if (psObj->un16Before > 0)
               {
                   psObj->un16Before--;
               }

               if ((n16RefOffset + psObj->un16BeforeRequest) <= (n16NumSongs-1))
               {
                   // get the next song in the scache
                   hSong = SCACHE_hSong(
                               hSCache,
                               (N16)(n16RefOffset+psObj->un16Before+1));

                   if (hSong != SONG_INVALID_OBJECT)
                   {
                       //    add it to the song list
                       bAddSong((SONGLIST_OBJECT)psObj, hSong, bReferenceUpdated);

                        if (bReferenceUpdated == FALSE)
                        {
                            // find the new bottom entry, now that we added a song
                            psObj->hBottomSongEntry =
                                OSAL.hLinkedListPrev(
                                    hOldBottomEntry,
                                    (void **)((void *)&hNewBottomSong)
                               );

                            // change the reference to the new bottom song entry
                            psObj->hReferenceSongEntry = psObj->hBottomSongEntry;
                        }
                   }
               }
           } // if (hOldBottomEntry == psObj->hReferenceSongEntry)

           // remove the song
           hSong = (SONG_OBJECT)OSAL.pvLinkedListThis(hOldBottomEntry);
           bRemoveSong(hSong, psObj);
           OSAL.eLinkedListRemove(hOldBottomEntry);
           if (psObj->un16After > 0)
           {
               psObj->un16After--;
           }

           //    set event
           SMSU_tUpdate(&psObj->sEvent, SONGLIST_OBJECT_EVENT_UPDATE);

           // call any registerd callback for this SONGLIST
           SONGLIST_vUpdate((SONGLIST_OBJECT)psObj);
        } // if ( (n16RefOffset - psObj->un16After) <= 0 )
       else
       {
           //     don't need to do anything! it's not in the window
       }
    } // if (tEventMask & SCACHE_OBJECT_EVENT_REMOVE_SONG)

    return;
}

