/******************************************************************************/
/*                    Copyright (c) Sirius XM Radio, Inc.                     */
/*                            All Rights Reserved                             */
/*         Licensed Materials - Property of Sirius XM Radio, Inc.             */
/******************************************************************************/
/*******************************************************************************
 *
 * DESCRIPTION
 *
 *  This module contains the Song Tag Service implementation for the
 *  Sirius Module Services (SMS)
 *
******************************************************************************/
#include "standard.h"
#include "osal.h"

#include "sms_version.h"
#include "decoder_obj.h"
#include "sms.h"
#include "song_tag_service.h"
#include "_song_tag_service.h"

#include "sms_api_debug.h"
static const char *gpacThisFile = __FILE__;

/*****************************************************************************
                             PUBLIC FUNCTIONS
*****************************************************************************/

/*****************************************************************************
*
*   SONGTAG_hCreate
*
*****************************************************************************/
SONG_TAG_OBJECT SONGTAG_hCreate(
    TAG_OBJECT hParentTag
        )
{
    BOOLEAN bRestored;
    SONG_TAG_SERVICE_STRUCT *psStruct;

    // Create an instance of this object
    psStruct = (SONG_TAG_SERVICE_STRUCT *)
        SMSO_hCreate(
            "SONG_TAG_SERVICE",
            sizeof(SONG_TAG_SERVICE_STRUCT),
            SMS_INVALID_OBJECT, // Parent
            TRUE ); // Lock
    if(psStruct == NULL)
    {
        // Error!
        SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
            "SONG_TAG_SERVICE"
            ": Unable to allocate memory or object already exists.");
        return SONG_TAG_INVALID_OBJECT;
    }

    bRestored = bRestoreFromConfig(hParentTag, psStruct);
    if(bRestored == FALSE)
    {
        SMSO_vDestroy((SMS_OBJECT)psStruct);
        psStruct = NULL;
    }
    else
    {
        SMSO_vUnlock((SMS_OBJECT)psStruct);
    }

    return (SONG_TAG_OBJECT)psStruct;
}

/*****************************************************************************
*
*   SONGTAG_vDestroy
*
*****************************************************************************/
void SONGTAG_vDestroy(
    SONG_TAG_OBJECT hSongTag
        )
{
    BOOLEAN bLocked;

    bLocked = SMSO_bLock(
        (SMS_OBJECT)hSongTag, OSAL_OBJ_TIMEOUT_INFINITE);
    if(bLocked == TRUE)
    {
        SONG_TAG_SERVICE_STRUCT *psStruct =
            (SONG_TAG_SERVICE_STRUCT *)hSongTag;

        if (psStruct->hITunesURL != STRING_INVALID_OBJECT)
        {
            STRING.vDestroy(psStruct->hITunesURL);
            psStruct->hITunesURL = STRING_INVALID_OBJECT;
        }

        SMSO_vDestroy((SMS_OBJECT)psStruct);
    }

    return;
}

/*****************************************************************************
*
*   hURL
*
* Retrieves the specified Song Tag Service's URL.
*
*****************************************************************************/
static STRING_OBJECT hURL (
    SONG_TAG_SERVICE_ENUM eService
        )
{
    STRING_OBJECT *phURL = NULL;
    STRING_OBJECT hURL = STRING_INVALID_OBJECT;

    if (eService < SONG_TAG_SERVICE_INVALID)
    {
        SONG_TAG_OBJECT hSongTag;

        hSongTag = SMS_hSongTagService();
        if(hSongTag != SONG_TAG_INVALID_OBJECT)
        {
            BOOLEAN bLocked;

            bLocked = SMSO_bLock(
                (SMS_OBJECT)hSongTag, OSAL_OBJ_TIMEOUT_INFINITE);
            if(bLocked == TRUE)
            {
                SONG_TAG_SERVICE_STRUCT *psStruct =
                    (SONG_TAG_SERVICE_STRUCT *)hSongTag;

                bGetURL(psStruct, eService, &phURL);
                if( phURL != NULL )
                {
                    hURL = *phURL;
                }

                SMSO_vUnlock((SMS_OBJECT)hSongTag);
            }
        }
    }

    return hURL;
}

/*****************************************************************************
*
*   bIterate
*
* Iterate the song tag services.
*
* Inputs:
*
*   bIteratorCallback
*           A function to call for each song tag service
*   pvContentIteratorArg
*           An anonymous pointer which is caller specific and provided
*           each time bIteratorCallback is called.
*
*****************************************************************************/
static BOOLEAN bIterate (
    SONG_TAG_SERVICE_ITERATOR_CALLBACK bIteratorCallback,
    void *pvContentIteratorArg
        )
{
    BOOLEAN bReturn = FALSE;

    // Did caller provide a valid callback?
    if (bIteratorCallback != NULL)
    {
        SONG_TAG_OBJECT hSongTag;

        hSongTag = SMS_hSongTagService();
        if(hSongTag != SONG_TAG_INVALID_OBJECT)
        {
            BOOLEAN bLocked;

            bLocked = SMSO_bLock(
                (SMS_OBJECT)hSongTag, OSAL_OBJ_TIMEOUT_INFINITE);
            if(bLocked == TRUE)
            {
                SONG_TAG_SERVICE_STRUCT *psStruct =
                    (SONG_TAG_SERVICE_STRUCT *)hSongTag;

                vIterateServices(
                    psStruct,
                    bIteratorCallback,
                    pvContentIteratorArg
                        );

                bReturn = TRUE;
                SMSO_vUnlock((SMS_OBJECT)hSongTag);
            }
        }
    }
    return bReturn;
}

/*****************************************************************************
*
*   eGetITunesTagValue
*
* Retrieves the specified iTunes Tag value.
*
*****************************************************************************/
static SMSAPI_RETURN_CODE_ENUM eGetITunesTagValue (
    CID_OBJECT hTag,
    UN32 *pun32TagValue
        )
{
    SMSAPI_RETURN_CODE_ENUM eReturn = SMSAPI_RETURN_CODE_INVALID_INPUT;
    CID_ENUM eType;

    eType = CID_eType(hTag);
    if ((eType == CID_SONG_TAG_ID) && (pun32TagValue != NULL))
    {
        N32 n32NumWritten;
        void *pvVoid;

        pvVoid = pun32TagValue;
        n32NumWritten = CID_n32GetValue(hTag, (void **)&pvVoid);
        if (n32NumWritten == EOF)
        {
            eReturn = SMSAPI_RETURN_CODE_ERROR;
        }
        else
        {
            eReturn = SMSAPI_RETURN_CODE_SUCCESS;
        }
    }

    return eReturn;
}

/*****************************************************************************
*
*   hCreateITunesTag
*
* Creates a CID from an iTunes Tag value.
*
*****************************************************************************/
static CID_OBJECT hCreateITunesTag (
    UN32 un32ITunesTagValue
        )
{
    CID_OBJECT hTag;

    hTag = CID_hCreateConst(CID_SONG_TAG_ID, &un32ITunesTagValue);

    return hTag;
}

/*****************************************************************************
 *
 *  SONGTAG_vSongTagServiceURLUpdate
 *
 *****************************************************************************/
void SONGTAG_vSongTagServiceURLUpdate(
    SONG_TAG_SERVICE_ENUM eService,
    char const *pacURL
        )
{
    STRING_OBJECT *phURL = NULL;
    BOOLEAN bValid = FALSE;
    SONG_TAG_OBJECT hSongTag;

    hSongTag = SMS_hSongTagService();
    if(hSongTag != SONG_TAG_INVALID_OBJECT)
    {
        BOOLEAN bLocked;

        bLocked = SMSO_bLock(
            (SMS_OBJECT)hSongTag, OSAL_OBJ_TIMEOUT_INFINITE);
        if(bLocked == TRUE)
        {
            SONG_TAG_SERVICE_STRUCT *psStruct =
                (SONG_TAG_SERVICE_STRUCT *)hSongTag;

            // map the service to the handle
            bValid = bGetURL(psStruct, eService, &phURL);

            // Was the mapping successful?
            if ((bValid == TRUE) && (phURL != NULL))
            {
                BOOLEAN bDifferent = FALSE;

                // Do we have an existing value for this URL?
                if (*phURL != NULL)
                {
                    // Yes we have a value.
                    // Is it different?
                    if (STRING.n16CompareCStr(pacURL, 0, *phURL) != 0)
                    {
                        // It's different, so copy in the new value.
                        STRING.bModifyCStr(*phURL, pacURL);
                        bDifferent = TRUE;
                    }
                }
                else
                {
                    // Need to create a new string.
                    *phURL = STRING.hCreate(pacURL, 0);
                    bDifferent = TRUE;
                }

                // Did our URL change?
                if (bDifferent == TRUE)
                {
                    // Write out the URL to CFG
                    bSaveToConfig(eService, psStruct->hTag, *phURL);
                }
            }

            SMSO_vUnlock((SMS_OBJECT)hSongTag);
        }
    }
    return;
}

/*****************************************************************************
*
*   bRestoreFromConfig
*
*****************************************************************************/
static BOOLEAN bRestoreFromConfig(
    TAG_OBJECT hParentTag,
    SONG_TAG_SERVICE_STRUCT *psStruct
        )
{
    SMSAPI_RETURN_CODE_ENUM eReturn;
    BOOLEAN bReturn = FALSE;
    TAG_OBJECT hSongTagTag;

    // first initialize the song tagging tag to invalid
    psStruct->hTag = TAG_INVALID_OBJECT;

    // next initialize all service url handles to invalid
    psStruct->hITunesURL = STRING_INVALID_OBJECT;

    // now we'll try to get the actual tag and URLs

    // first get the song tag service
    eReturn = TAG_eGet(
        SONG_TAG_SERVICE_NAME,
        hParentTag,
        &hSongTagTag,
        NULL,
        TRUE
            );

   if (eReturn == SMSAPI_RETURN_CODE_SUCCESS)
    {
        // we found or created a tag. save that handle.
        psStruct->hTag = hSongTagTag;

        // iterate the handle, looking for child tags. the child tags
        // will be the service URLs. We'll process any we find as we iterate
        eReturn = TAG_eIterateChildren(
            psStruct->hTag,
            bRestoreServiceIterator,
            (void *)psStruct
                );
    }

    if (eReturn == SMSAPI_RETURN_CODE_SUCCESS)
    {
        // everything went ok
        bReturn = TRUE;
    }

    return bReturn;
}

/*****************************************************************************
 *
 *   bGetSongTagURL
 *
 *****************************************************************************/
static BOOLEAN bGetURL(
    SONG_TAG_SERVICE_STRUCT *psStruct, SONG_TAG_SERVICE_ENUM eService,
    STRING_OBJECT **pphURL
        )
{
    BOOLEAN bValid = FALSE;

    // map the service to the handle
    switch (eService)
    {
        case SONG_TAG_SERVICE_ITUNES:
        {
            *pphURL = &psStruct->hITunesURL;
            bValid = TRUE;
        }
        break;

        case SONG_TAG_SERVICE_INVALID:
        default:
           // do nothing
        break;
    }

    return bValid;
}

/*****************************************************************************
*
*   pacSongTagServiceText
*
*****************************************************************************/
static const char *pacSongTagServiceText( SONG_TAG_SERVICE_ENUM eService )
{
    const char *pacReturnString;

    // based on the seek type, get the text that describes it
    switch (eService)
    {
        case SONG_TAG_SERVICE_ITUNES:
            pacReturnString = "iTunes";
        break;

        case SONG_TAG_SERVICE_INVALID:
        default:
            pacReturnString = "Unknown";
        break;
    }

    return pacReturnString;
}

/*****************************************************************************
*
*   bRestoreServiceIterator
*
*****************************************************************************/
static BOOLEAN bRestoreServiceIterator(
    TAG_OBJECT hTag,
    void *pvArg
        )
{
    STRING_OBJECT hInstanceName;
    SONG_TAG_MAPPING_STUCT sStruct;
    SONG_TAG_SERVICE_STRUCT *psStruct =
        (SONG_TAG_SERVICE_STRUCT *)pvArg;

    // get the instance name.
    hInstanceName = TAG_hTagInstanceName(hTag);

    // init the iterator struct
    sStruct.hInstance = hInstanceName;
    sStruct.eService = SONG_TAG_SERVICE_INVALID;

    // iterate the services to find the service enum that matches the
    // instance name
    vIterateServices(psStruct, bMapping, (void *)&sStruct);

    if (sStruct.eService != SONG_TAG_SERVICE_INVALID)
    {
        // found a match, now get the value and set it.
        SMSAPI_RETURN_CODE_ENUM eReturn;
        STRING_OBJECT hValueString = STRING_INVALID_OBJECT, *phString;
        size_t tSize;

        tSize = sizeof(STRING_OBJECT);
        phString = &hValueString;

        // get URL
        eReturn = TAG_eGetTagValue(
            hTag,
            TAG_TYPE_STRING,
            (void **)&phString,
            &tSize);


        if (eReturn == SMSAPI_RETURN_CODE_SUCCESS)
        {
            STRING_OBJECT *phURL = NULL;

            // get the location to store this service's URL
            bGetURL(
                psStruct, sStruct.eService, &phURL);
            if( phURL != NULL )
            {
                // store the URL
                *phURL = hValueString;
            }
        }
    }

    // keep iterating
    return TRUE;
}

/*****************************************************************************
*
*   bMapping
*
*****************************************************************************/
static BOOLEAN bMapping(
    SONG_TAG_SERVICE_ENUM eService,
    STRING_OBJECT hURL,
    void *pvIteratorArg
        )
{
    SONG_TAG_MAPPING_STUCT *psStruct = (SONG_TAG_MAPPING_STUCT *)pvIteratorArg;
    char *pacServiceText;

    // Get the textual name for this service
    pacServiceText = (char *)pacSongTagServiceText(eService);

    // See if this matches the instance name
    if (STRING.n16CompareCStr( pacServiceText, 0, psStruct->hInstance) == 0)
    {
        // Yup, this is the service we're looking for.
        psStruct->eService = eService;

        // done
        return FALSE;
    }

    // Not the one we're looking for.
    // Keep iterating.
    return TRUE;
}

/*****************************************************************************
*
*   bSaveToConfig
*
*****************************************************************************/
static BOOLEAN bSaveToConfig(
    SONG_TAG_SERVICE_ENUM eService,
    TAG_OBJECT hParentTag,
    STRING_OBJECT hNewURL
        )
{
    char *pacService;
    TAG_OBJECT hTag;
    SMSAPI_RETURN_CODE_ENUM eReturn;
    BOOLEAN bReturn = FALSE;

    // get the text for this service
    pacService = (char *)pacSongTagServiceText(eService);

    eReturn = TAG_eGet(
                  SONG_TAG_URL_NAME,
                  hParentTag,
                  &hTag,
                  pacService,
                  TRUE);

    if (eReturn == SMSAPI_RETURN_CODE_SUCCESS)
    {
        eReturn = TAG_eSetTagValue(
                      hTag,
                      TAG_TYPE_STRING,
                      &hNewURL,
                      sizeof(STRING_OBJECT),
                      TRUE);

        if (eReturn == SMSAPI_RETURN_CODE_SUCCESS)
        {
            bReturn = TRUE;
        }
    }

    return bReturn;
}

/*****************************************************************************
*
*   vIterateServices
*
*****************************************************************************/
static void vIterateServices(
    SONG_TAG_SERVICE_STRUCT *psStruct,
    SONG_TAG_SERVICE_ITERATOR_CALLBACK bIteratorCallback,
    void *pvContentIteratorArg
        )
{
    size_t tCurrentService;
    BOOLEAN bOk;
    STRING_OBJECT *phURL;

    // Cycle through the services
    for (tCurrentService = 0; tCurrentService < SONG_TAG_SERVICE_MAX;
         tCurrentService++)
    {
        // Get the URL
        bOk = bGetURL(psStruct,
                   (SONG_TAG_SERVICE_ENUM)tCurrentService, &phURL);
        if (bOk == TRUE)
        {
            BOOLEAN bKeepIterating;

            // Call the provided callback
            bKeepIterating = bIteratorCallback(
                (SONG_TAG_SERVICE_ENUM)tCurrentService,
                *phURL, pvContentIteratorArg);

            // Should we keep iterating?
            if (bKeepIterating == FALSE)
            {
                // Nope. We're done
                break;
            }
        }
    }

    return;
}
