/******************************************************************************/
/*                    Copyright (c) Sirius XM Radio, Inc.                     */
/*                            All Rights Reserved                             */
/*         Licensed Materials - Property of Sirius XM Radio, Inc.             */
/******************************************************************************/
/*******************************************************************************
 *
 * DESCRIPTION
 *
 *  This module contains the Object:SONG implementation for the
 *  Sirius Module Services (SMS)
 *
 ******************************************************************************/
#include <string.h>
#include <time.h>

#include "standard.h"
#include "osal.h"

#include "sms_version.h"
#include "radio.h"
#include "sms_obj.h"
#include "sms_update.h"
#include "string_obj.h"
#include "scache.h"

#include "song_obj.h"
#include "_song_obj.h"

/*****************************************************************************
                             PUBLIC FUNCTIONS
*****************************************************************************/

/*****************************************************************************
*
*   tEventMask
*
*****************************************************************************/
static SONG_EVENT_MASK tEventMask (
    SONG_OBJECT hSong
        )
{
    BOOLEAN bOwner;
    SONG_OBJECT_STRUCT *psObj =
        (SONG_OBJECT_STRUCT *)hSong;

    // Verify inputs
    bOwner = SMSO_bOwner((SMS_OBJECT)hSong);
    if(bOwner == FALSE)
    {
        // Error!
        return SONG_OBJECT_EVENT_NONE;
    }

    return SMSU_tMask(&psObj->sEvent);
}

/*****************************************************************************
*
*   hArtist
*
*****************************************************************************/
static STRING_OBJECT hArtist (
    SONG_OBJECT hSong
        )
{
    STRING_OBJECT hString = STRING_INVALID_OBJECT;
    BOOLEAN bOwner;

    // Verify inputs
    bOwner = SMSO_bOwner((SMS_OBJECT)hSong);

    if(bOwner == TRUE)
    {
        SONG_OBJECT_STRUCT *psObj =
            (SONG_OBJECT_STRUCT *)hSong;
        hString = psObj->hArtist;
    }

	return hString;
}

/*****************************************************************************
*
*   hTitle
*
*****************************************************************************/
static STRING_OBJECT hTitle (
    SONG_OBJECT hSong
        )
{
    STRING_OBJECT hString = STRING_INVALID_OBJECT;
    BOOLEAN bOwner;

    // Verify inputs
    bOwner = SMSO_bOwner((SMS_OBJECT)hSong);

    if(bOwner == TRUE)
    {
        SONG_OBJECT_STRUCT *psObj =
            (SONG_OBJECT_STRUCT *)hSong;
        hString = psObj->hTitle;
    }

	return hString;
}

/*****************************************************************************
*
*   eTimeStamp
*
*****************************************************************************/
static SMSAPI_RETURN_CODE_ENUM eTimeStamp (
    SONG_OBJECT hSong,
	UN32 *pun32TimeStamp
        )
{
    SMSAPI_RETURN_CODE_ENUM eReturnCode =
        SMSAPI_RETURN_CODE_ERROR;

    if (pun32TimeStamp !=NULL)
	{
	    // non-NULL pointer passed in. so we can copy the value
	    BOOLEAN bOwner;

        // Verify inputs
        bOwner = SMSO_bOwner((SMS_OBJECT)hSong);
        if(bOwner == TRUE)
        {
            SONG_OBJECT_STRUCT *psObj =
                (SONG_OBJECT_STRUCT *)hSong;

            *pun32TimeStamp = psObj->un32TimeStamp;
            eReturnCode = SMSAPI_RETURN_CODE_SUCCESS;
        }
	}

	return eReturnCode;
}

/*****************************************************************************
*
*   eDuration
*
*****************************************************************************/
static SMSAPI_RETURN_CODE_ENUM eDuration (
    SONG_OBJECT hSong,
	UN32 *pun32Duration
        )
{
    SMSAPI_RETURN_CODE_ENUM eReturnCode =
        SMSAPI_RETURN_CODE_ERROR;

    if (pun32Duration !=NULL)
	{
	    // non-NULL pointer passed in. so we can copy the value
	    BOOLEAN bOwner;

        // Verify inputs
        bOwner = SMSO_bOwner((SMS_OBJECT)hSong);
        if(bOwner == TRUE)
        {
            SONG_OBJECT_STRUCT *psObj =
                (SONG_OBJECT_STRUCT *)hSong;

            *pun32Duration = psObj->un32Duration;
            eReturnCode = SMSAPI_RETURN_CODE_SUCCESS;
        }
	}

	return eReturnCode;
}

/*****************************************************************************
*
*   eChannelId
*
*****************************************************************************/
static SMSAPI_RETURN_CODE_ENUM eChannelId (
    SONG_OBJECT hSong,
	CHANNEL_ID *ptChannelId
        )
{
    SMSAPI_RETURN_CODE_ENUM eReturnCode =
        SMSAPI_RETURN_CODE_ERROR;

    if (ptChannelId !=NULL)
	{
	    // non-NULL pointer passed in. so we can copy the value
	    BOOLEAN bOwner;

        // Verify inputs
        bOwner = SMSO_bOwner((SMS_OBJECT)hSong);
        if(bOwner == TRUE)
        {
            SONG_OBJECT_STRUCT *psObj =
                (SONG_OBJECT_STRUCT *)hSong;

            *ptChannelId = psObj->tChannelId;
            eReturnCode = SMSAPI_RETURN_CODE_SUCCESS;
        }
	}

	return eReturnCode;
}

/*****************************************************************************
*
*   eStatus
*
*****************************************************************************/
static SMSAPI_RETURN_CODE_ENUM eStatus (
    SONG_OBJECT hSong,
	SMSAPI_SONG_STATUS_ENUM *peStatus
        )
{
    SMSAPI_RETURN_CODE_ENUM eReturnCode = SMSAPI_RETURN_CODE_INVALID_INPUT;

    if (peStatus !=NULL)
	{
	    // non-NULL pointer passed in. so we can copy the value
	    BOOLEAN bOwner;

        // Verify inputs
        bOwner = SMSO_bOwner((SMS_OBJECT)hSong);
        if(bOwner == TRUE)
        {
            SONG_OBJECT_STRUCT *psObj =
                (SONG_OBJECT_STRUCT *)hSong;

            *peStatus = psObj->eStatus;
        }
        else
        {
            eReturnCode = SMSAPI_RETURN_CODE_ERROR;
        }
	}

	return eReturnCode;
}

/*****************************************************************************
                             FRIEND FUNCTIONS
*****************************************************************************/

/*****************************************************************************
*
*   SONG_hCreate
*
*****************************************************************************/
SONG_OBJECT SONG_hCreate (
    SCACHE_OBJECT hSCache,
	N16 n16Offset
	    )
{
   SONG_OBJECT_STRUCT *psObj =
        (SONG_OBJECT_STRUCT *)SONG_INVALID_OBJECT;
    char acName[OSAL_MAX_OBJECT_NAME_LENGTH_WITH_NULL];
    BOOLEAN bValid;
	static UN32 un32Instance = 0;

    // Verify input
    bValid = SMSO_bValid((SMS_OBJECT)hSCache);
    if(bValid == FALSE)
    {
        // Error!
        return SONG_INVALID_OBJECT;
    }

    // Construct a unique name for this song.
    snprintf(&acName[0],
            sizeof(acName),
            SONG_OBJECT_NAME":%u",
            un32Instance++);

    // Create an instance of this object
    psObj = (SONG_OBJECT_STRUCT *)
        SMSO_hCreate(
            SONG_OBJECT_NAME,
            sizeof(SONG_OBJECT_STRUCT),
            (SMS_OBJECT)hSCache, // Child
            FALSE); // No Lock (ignored)
    if(psObj == NULL)
    {
        // Error!
        return SONG_INVALID_OBJECT;
    }

    // Initialize object...

    // Initialize song object info
    psObj->n16Offset = n16Offset;
    psObj->hTitle = STRING_INVALID_OBJECT;
    psObj->hArtist = STRING_INVALID_OBJECT;
    psObj->tChannelId = CHANNEL_INVALID_ID;
    psObj->un32BaseTime = 0;
    psObj->un32Duration = 0;
	psObj->un32TimeStamp = 0;
    psObj->tId = SONG_INVALID_ID;

    // Initialize asynchronous update configuration
    SMSU_vInitialize(
        &psObj->sEvent,
        psObj,
        SMS_OBJECT_EVENT_NONE,
        SMS_OBJECT_EVENT_NONE,
        (SMSAPI_OBJECT_EVENT_CALLBACK)NULL,
        NULL);

    return (SONG_OBJECT)psObj;
}

/*****************************************************************************
*
*   SONG_vDestroy
*
*****************************************************************************/
void SONG_vDestroy (
    SONG_OBJECT hSong
        )
{
    BOOLEAN bValid;
    SONG_OBJECT_STRUCT *psObj =
        (SONG_OBJECT_STRUCT *)hSong;

    // Verify inputs
    bValid = SMSO_bValid((SMS_OBJECT)hSong);
    if(bValid == TRUE)
    {
        // Filter out all further song updates
        SMSU_tFilter(&psObj->sEvent, SONG_OBJECT_EVENT_ALL);

        // Un-Initialize parameters...
        psObj->n16Offset = 0;
        psObj->un32BaseTime = 0;
        psObj->un32Duration = 0;
	    psObj->un32TimeStamp = 0;
        psObj->tChannelId = CHANNEL_INVALID_ID;

        // Destroy string objects (if they exist)...

        if(psObj->hTitle != STRING_INVALID_OBJECT)
        {
            STRING_vDestroy(psObj->hTitle);
            psObj->hTitle = STRING_INVALID_OBJECT;
        }

        if(psObj->hArtist != STRING_INVALID_OBJECT)
        {
            STRING_vDestroy(psObj->hArtist);
            psObj->hArtist = STRING_INVALID_OBJECT;
        }

        // Destory the async update configuration
        SMSU_vDestroy(&psObj->sEvent);

        // Free object instance
        SMSO_vDestroy((SMS_OBJECT)hSong);
    } // if(bValid == TRUE)
    return;
}

/*****************************************************************************
*
*   SONG_bUpdateChannelId
*
*****************************************************************************/
BOOLEAN SONG_bUpdateChannelId (
    SONG_OBJECT hSong,
	CHANNEL_ID tId
        )
{
	BOOLEAN bValid;

    // validate object
    bValid = SMSO_bValid((SMS_OBJECT)hSong);
    if(bValid == TRUE)
    {
        SONG_OBJECT_STRUCT *psObj =
            (SONG_OBJECT_STRUCT *)hSong;

        psObj->tChannelId = tId;

	    return TRUE;
	}
	return FALSE;
}

/*****************************************************************************
*
*   SONG_bUpdateOffset
*
*****************************************************************************/
BOOLEAN SONG_bUpdateOffset (
    SONG_OBJECT hSong,
	N16 n16Offset
        )
{
	BOOLEAN bValid;

    // validate object
    bValid = SMSO_bValid((SMS_OBJECT)hSong);
    if(bValid == TRUE)
    {
        SONG_OBJECT_STRUCT *psObj =
            (SONG_OBJECT_STRUCT *)hSong;

        psObj->n16Offset = n16Offset;

	    return TRUE;
    }

    return FALSE;
}

/*****************************************************************************
*
*   SONG_bSetTimestamp
*
*****************************************************************************/
BOOLEAN SONG_bSetTimestamp (
    SONG_OBJECT hSong
        )
{
	BOOLEAN bValid;

    // validate object
    bValid = SMSO_bValid((SMS_OBJECT)hSong);
    if(bValid == TRUE)
    {
        SONG_OBJECT_STRUCT *psObj =
            (SONG_OBJECT_STRUCT *)hSong;

	    UN32 un32Time;
	    OSAL_RETURN_CODE_ENUM eReturn;

        eReturn = OSAL.eTimeGetLocal(&un32Time);
	    if (eReturn == OSAL_SUCCESS)
	    {
	        psObj->un32TimeStamp = un32Time;
 	    }
	    else
	    {
            printf("Result: %s\n",
                   OSAL.pacGetReturnCodeName(eReturn));
            psObj->un32TimeStamp = 0;
	    }

        // Set event mask
        SMSU_tUpdate(&psObj->sEvent, SONG_OBJECT_EVENT_TIME_STAMP);

        // Notify callback if registered and change occured
        SMSU_bNotify(&psObj->sEvent);

	    return TRUE;
    }

    return FALSE;
}

/*****************************************************************************
*
*   SONG_bSetId
*
*****************************************************************************/
BOOLEAN SONG_bSetId (
    SONG_OBJECT hSong,
    SONG_ID tId
        )
{
	BOOLEAN bValid;

    // validate object
    bValid = SMSO_bValid((SMS_OBJECT)hSong);
    if(bValid == TRUE)
    {
        SONG_OBJECT_STRUCT *psObj =
            (SONG_OBJECT_STRUCT *)hSong;

        psObj->tId = tId;

	    return TRUE;
    }

    return FALSE;
}

/*****************************************************************************
*
*   SONG_tId
*
*****************************************************************************/
SONG_ID SONG_tId (
    SONG_OBJECT hSong
        )
{
	BOOLEAN bValid;
    SONG_ID tId;

    // validate object
    bValid = SMSO_bValid((SMS_OBJECT)hSong);
    if(bValid == TRUE)
    {
        SONG_OBJECT_STRUCT *psObj =
            (SONG_OBJECT_STRUCT *)hSong;

        tId = psObj->tId;
    }
    else
    {
        tId = SONG_INVALID_ID;
    }

    return tId;
}

/*****************************************************************************
*
*   SONG_bSetDuration
*
*****************************************************************************/
BOOLEAN SONG_bSetDuration (
    SONG_OBJECT hSong
        )
{
	BOOLEAN bValid, bReturn = FALSE;

    // validate object
    bValid = SMSO_bValid((SMS_OBJECT)hSong);
    if(bValid == TRUE)
    {
        SONG_OBJECT_STRUCT *psObj =
            (SONG_OBJECT_STRUCT *)hSong;
	    UN32 un32Time;

	    OSAL.vTimeUp(&un32Time, NULL);
	    psObj->un32Duration = un32Time - psObj->un32BaseTime;

        // Set event mask
        SMSU_tUpdate(&psObj->sEvent, SONG_OBJECT_EVENT_DURATION);

        // Notify callback if registered and change occured
        SMSU_bNotify(&psObj->sEvent);

	    bReturn = TRUE;
	}

	return bReturn;
}

/*****************************************************************************
*
*   SONG_bSetDurationValue
*
*****************************************************************************/
BOOLEAN SONG_bSetDurationValue (
    SONG_OBJECT hSong,
    UN32 un32Duration
        )
{
	BOOLEAN bValid, bReturn = FALSE;

    // validate object
    bValid = SMSO_bValid((SMS_OBJECT)hSong);
    if(bValid == TRUE)
    {
        SONG_OBJECT_STRUCT *psObj =
            (SONG_OBJECT_STRUCT *)hSong;

	    psObj->un32Duration = un32Duration;

        // Set event mask
        SMSU_tUpdate(&psObj->sEvent, SONG_OBJECT_EVENT_DURATION);

        // Notify callback if registered and change occured
        SMSU_bNotify(&psObj->sEvent);

	    bReturn = TRUE;
	}

	return bReturn;
}

/*****************************************************************************
*
*   SONG_bSetStatus
*
*****************************************************************************/
BOOLEAN SONG_bSetStatus (
    SONG_OBJECT hSong,
    SMSAPI_SONG_STATUS_ENUM eStatus
        )
{
	BOOLEAN bValid, bReturn = FALSE;

    // validate object
    bValid = SMSO_bValid((SMS_OBJECT)hSong);
    if(bValid == TRUE)
    {
        SONG_OBJECT_STRUCT *psObj =
            (SONG_OBJECT_STRUCT *)hSong;

        printf( SONG_OBJECT_NAME": Updating Song: %s (#%d) Status: %s -> %s\n",
            STRING.pacCStr(psObj->hTitle),
            psObj->n16Offset,
            pcSongStatusToString(psObj->eStatus),
            pcSongStatusToString(eStatus)
                );

	    psObj->eStatus = eStatus;

        // Set event mask
        SMSU_tUpdate(&psObj->sEvent, SONG_OBJECT_EVENT_STATUS);

        // Notify callback if registered and change occured
        SMSU_bNotify(&psObj->sEvent);

	    bReturn = TRUE;
	}

	return bReturn;
}

/*****************************************************************************
*
*   SONG_bUpdateSong
*
*****************************************************************************/
BOOLEAN SONG_bUpdateSong (
    SONG_OBJECT hSong,
    const char *pacArtist,
	const char *pacTitle,
	N16 n16Offset
        )
{
    SONG_OBJECT_STRUCT *psObj =
        (SONG_OBJECT_STRUCT *)hSong;
    SMSAPI_EVENT_MASK tEventMask =
        SMS_OBJECT_EVENT_NONE;
    BOOLEAN bUpdate = FALSE, bValid;

    // validate object
    bValid = SMSO_bValid((SMS_OBJECT)hSong);
    if( bValid == FALSE )
    {
        // Error!
        return FALSE;
    }

    if(psObj->hArtist == STRING_INVALID_OBJECT)
    {
        // we don't currently have a string object, we need to create it
        psObj->hArtist = STRING_hCreate(
                             (SMS_OBJECT)hSong,
                             pacArtist,
                             strlen(pacArtist),
                             GsRadio.sChannel.un16ArtistMaxLen
                                 );

        if (psObj->hArtist != STRING_INVALID_OBJECT)
        {
            // Set event mask
            tEventMask |= SMSU_tUpdate(&psObj->sEvent, SONG_OBJECT_EVENT_ARTIST);
        }
    } // if(psObj->hArtist == STRING_INVALID_OBJECT)
    else
    {
        // we already have a string object, just update it
        bUpdate = STRING.bModifyCStr(
				      psObj->hArtist,
                      pacArtist
                          );
        if(bUpdate == TRUE)
        {
            // Set event mask
            tEventMask |= SMSU_tUpdate(
						      &psObj->sEvent,
							  SONG_OBJECT_EVENT_ARTIST);
        }
    }

    if(psObj->hTitle == STRING_INVALID_OBJECT)
    {
        // we don't currently have a string object, we need to create it
        psObj->hTitle = STRING_hCreate(
                            (SMS_OBJECT)hSong,
                            pacTitle,
                            strlen(pacTitle),
                            GsRadio.sChannel.un16TitleMaxLen
                                );

        if (psObj->hTitle != STRING_INVALID_OBJECT)
        {
            // Set event mask
            tEventMask |= SMSU_tUpdate(&psObj->sEvent, SONG_OBJECT_EVENT_TITLE);
        }
    } // if(psObj->hTitle == STRING_INVALID_OBJECT)
    else
    {
        bUpdate = STRING.bModifyCStr(
				      psObj->hTitle,
                      pacTitle
					      );
        if(bUpdate == TRUE)
        {
            // Set event mask
            tEventMask |= SMSU_tUpdate(
						      &psObj->sEvent,
							  SONG_OBJECT_EVENT_TITLE);
        }
    }

	if (psObj->n16Offset != n16Offset)
	{
        // offset changed, so update it
	    psObj->n16Offset = n16Offset;
        bUpdate = TRUE;
	}

    // If any updates occured, run notification
    if(tEventMask != SMS_OBJECT_EVENT_NONE)
    {
        // Notify callback if registered and change occured
        SMSU_bNotify(&psObj->sEvent);
    }
    return bUpdate;
}

/*****************************************************************************
*
*   SONG_bUpdateArtist
*
*****************************************************************************/
BOOLEAN SONG_bUpdateArtist (
    SONG_OBJECT hSong,
    const char *pacArtistText
        )
{
    SONG_OBJECT_STRUCT *psObj =
        (SONG_OBJECT_STRUCT *)hSong;
	SMSAPI_EVENT_MASK tEventMask =
        SMS_OBJECT_EVENT_NONE;
	BOOLEAN bValid;

    // validate object & input
    bValid = SMSO_bValid((SMS_OBJECT)hSong);
    if((bValid == TRUE) && (pacArtistText != NULL))
    {
        if(psObj->hArtist == STRING_INVALID_OBJECT)
        {
            // don't have a valid Artist string yet,
            // so we need to create
            psObj->hArtist =
                STRING_hCreate(
                    (SMS_OBJECT)hSong,
                    pacArtistText,
                    strlen(pacArtistText),
                    GsRadio.sChannel.un16LongNameMaxLen
                        );
            if (psObj->hArtist != STRING_INVALID_OBJECT)
            {
                // Set event mask
                tEventMask |=
                    SMSU_tUpdate(
                        &psObj->sEvent,
                        SONG_OBJECT_EVENT_ARTIST
                            );
            }
        }
        else
        {
            BOOLEAN bUpdate;

            // have a valid Artist string yet,
            // so we need to just update it
            bUpdate =
                STRING.bModifyCStr(
                    psObj->hArtist,
                    pacArtistText
                        );
            if(bUpdate == TRUE)
            {
                // Set event mask
                tEventMask |=
                    SMSU_tUpdate(
                        &psObj->sEvent,
                        SONG_OBJECT_EVENT_ARTIST
                            );
            }
        }
    }

    // If any updates occurred, run notification
    if(tEventMask != SMS_OBJECT_EVENT_NONE)
    {
        // Notify callback if registered and change occured
        SMSU_bNotify(&psObj->sEvent);
        return TRUE;
    }

    return FALSE;
}

/*****************************************************************************
*
*   SONG_bUpdateTitle
*
*****************************************************************************/
BOOLEAN SONG_bUpdateTitle (
    SONG_OBJECT hSong,
    const char *pacTitleText
        )
{
    SONG_OBJECT_STRUCT *psObj =
        (SONG_OBJECT_STRUCT *)hSong;
    SMSAPI_EVENT_MASK tEventMask =
        SMS_OBJECT_EVENT_NONE;
    BOOLEAN bValid;

    // validate object & input
    bValid = SMSO_bValid((SMS_OBJECT)hSong);
    if((bValid == TRUE) && (pacTitleText != NULL))
    {
        if(psObj->hTitle == STRING_INVALID_OBJECT)
        {
            // don't have a valid Title string yet,
            // so we need to create
            psObj->hTitle =
                STRING_hCreate(
                    (SMS_OBJECT)hSong,
                    pacTitleText,
                    strlen(pacTitleText),
                    GsRadio.sChannel.un16LongNameMaxLen
                        );
            if (psObj->hTitle != STRING_INVALID_OBJECT)
            {
                // Set event mask
                tEventMask |=
                    SMSU_tUpdate(
                        &psObj->sEvent,
                        SONG_OBJECT_EVENT_TITLE
                            );
            }
        }
        else
        {
            BOOLEAN bUpdate;

            // have a valid Title string yet,
            // so we need to just update it
            bUpdate =
                STRING.bModifyCStr(
                    psObj->hTitle,
                    pacTitleText
                        );
            if(bUpdate == TRUE)
            {
                // Set event mask
                tEventMask |=
                    SMSU_tUpdate(
                        &psObj->sEvent,
                        SONG_OBJECT_EVENT_TITLE
                            );
            }
        }
    }

    // If any updates occurred, run notification
    if(tEventMask != SMS_OBJECT_EVENT_NONE)
    {
        // Notify callback if registered and change occured
        SMSU_bNotify(&psObj->sEvent);
        return TRUE;
    }

    return FALSE;
}

/*****************************************************************************
*
*   SONG_bCompare
*
*   This function is used to compare a Song's Artist and Title Info
*
*   Inputs:
*
*       hSong - A handle to a valid SONG object to compare to.
*       pacArtistText - A C-style null terminated string to compare artist
*       text info against. May be null to skip compare.
*       pacTitleText - A C-style null terminated string to compare title
*       text info against. May be null to skip compare.
*
*   Outputs:
*       TRUE if different, FALSE if same
*
*****************************************************************************/
BOOLEAN SONG_bCompare (
    SONG_OBJECT hSong,
    const char *pacArtistText,
    const char *pacTitleText
        )
{
	BOOLEAN bValid, bDifferent = FALSE;

    // validate object
    bValid = SMSO_bValid((SMS_OBJECT)hSong);
    if(bValid == FALSE)
    {
        // Error!
        return FALSE;
    }

    // Check artist?
    if(pacArtistText != NULL)
    {
        N16 n16Result;
        STRING_OBJECT hArtist = SONG.hArtist(hSong);

        n16Result = STRING.n16CompareCStr(pacArtistText, 0, hArtist);
        if(n16Result != 0)
        {
            // these are not the same
            bDifferent = TRUE;
            printf("Artist %s doesn't match ", pacArtistText);
            STRING.n32FWrite(hArtist, stdout);
            printf("\n");
        }
    }

    // Check title?
    if(pacTitleText != NULL)
    {
        N16 n16Result;
        STRING_OBJECT hTitle = SONG.hTitle(hSong);

        n16Result = STRING.n16CompareCStr(pacTitleText, 0, hTitle);
        if(n16Result != 0)
        {
            // these are not the same
            bDifferent = TRUE;
            printf("Title %s doesn't match ", pacTitleText);
            STRING.n32FWrite(hTitle, stdout);
            printf("\n");
        }
    }

    return bDifferent;
}

/*****************************************************************************
*
*   SONG_n16Offset
*   This function get's this song's absolute offset in the scache
*
*****************************************************************************/
N16 SONG_n16Offset (
    SONG_OBJECT hSong
        )
{
    BOOLEAN bValid;

    // Verify inputs
    bValid = SMSO_bValid((SMS_OBJECT)hSong);

    if(bValid == TRUE)
    {
        SONG_OBJECT_STRUCT *psObj =
            (SONG_OBJECT_STRUCT *)hSong;
        return psObj->n16Offset;
    }

    // Error!
    return -1;
}

/*****************************************************************************
*
*   SONG_bRegisterNotification
*
*****************************************************************************/
BOOLEAN SONG_bRegisterNotification (
    SONG_OBJECT hSong,
    SONG_EVENT_MASK tEventRequestMask,
    SONG_OBJECT_EVENT_CALLBACK vEventCallback,
    void *pvEventCallbackArg
        )
{
    BOOLEAN bValid;

    // Verify inputs
    bValid = SMSO_bValid((SMS_OBJECT)hSong);

    if(bValid == TRUE)
    {
        SONG_OBJECT_STRUCT *psObj =
            (SONG_OBJECT_STRUCT *)hSong;
        BOOLEAN bResult;

        // Register a callback in the
        // asynchronous update configuration
        bResult = SMSU_bRegisterCallback(
            &psObj->sEvent,
            tEventRequestMask,
            (SMSAPI_OBJECT_EVENT_CALLBACK)vEventCallback,
            pvEventCallbackArg,
            TRUE);

        SMSAPI_DEBUG_vPrint(SONG_OBJECT_NAME, 5, "Registered callback %p(%p), ret=%d",
            vEventCallback, pvEventCallbackArg, bResult);

        return bResult;
    }

    return FALSE;
}

/*****************************************************************************
*
*   SONG_vUnregisterNotification
*
*****************************************************************************/
void SONG_vUnregisterNotification (
    SONG_OBJECT hSong,
    SONG_OBJECT_EVENT_CALLBACK vEventCallback,
    void *pvEventCallbackArg
        )
{
    BOOLEAN bValid;

    SMSAPI_DEBUG_vPrint(SONG_OBJECT_NAME, 5, "%s(%p, %p, %p)", __FUNCTION__, hSong, vEventCallback, pvEventCallbackArg);

    // Verify inputs
    bValid = SMSO_bValid((SMS_OBJECT)hSong);

    if(bValid == TRUE)
    {
        SONG_OBJECT_STRUCT *psObj =
            (SONG_OBJECT_STRUCT *)hSong;
        BOOLEAN bResult;

        // Remove this callback from the
        // asynchronous update configuration
        bResult = SMSU_bUnregisterCallback(
            &psObj->sEvent,
            (SMSAPI_OBJECT_EVENT_CALLBACK) vEventCallback,
            pvEventCallbackArg
                );
        SMSAPI_DEBUG_vPrint(SONG_OBJECT_NAME, 5, "Unregistered callback %p(%p), ret=%d", 
            vEventCallback, pvEventCallbackArg, bResult);
    }

    return;
}

/*****************************************************************************
*
*   SONG_n16FindSongByOffset
*       This is a Linked List search function
*
*****************************************************************************/
N16 SONG_n16FindSongByOffset( void *pvArg1, void *pvArg2 )
{
	SONG_OBJECT_STRUCT *psSong = (SONG_OBJECT_STRUCT *)pvArg1;
	N16 n16Offset = (N16)(size_t)pvArg2;

    return (psSong->n16Offset - n16Offset);
}

/*****************************************************************************
*
*   SONG_n16SongIdEqual
*
*   This function is used to determine if a SONG's ID is equal to a given Id.
*
*   Inputs:
*       *pvArg1 - pointer to one of the songs being compared
*       *pvArg2 - pointer to the song id being compared
*
*   Outputs:
*       n16Result - The comparison between the song id passed in
*           and the song id currently being searched.
*         0 - Objects have the same value (equal )
*       > 0 - Object1 is greater than (after) Object2
*       < 0 - Object1 is less than (before) Object2 (or error)
*
*****************************************************************************/
N16 SONG_n16SongIdEqual(
    void *pvArg1,
    void *pvArg2
        )
{
    N16 n16Result = N16_MIN;
    SONG_OBJECT_STRUCT *psObj1 = (SONG_OBJECT_STRUCT *)pvArg1;
    SONG_ID tSongId2 = (SONG_ID)(size_t)pvArg2;

    // Check pointers
    if(psObj1 != NULL)
    {
        if ( psObj1->tId == tSongId2 )
        {
            n16Result = 0;
        }
    }

    return n16Result;
}

/*****************************************************************************
                             PRIVATE FUNCTIONS
*****************************************************************************/

/*****************************************************************************
*
*   pcSongStatusToString
*
*****************************************************************************/
static char *pcSongStatusToString (
    SMSAPI_SONG_STATUS_ENUM eStatus
        )
{
    char *pcStatus = "Unknown";

    switch ( eStatus )
    {
        case SMSAPI_SONG_STATUS_INCOMPLETE:
        {
            pcStatus = "Incomplete";
        }
        break;

        case SMSAPI_SONG_STATUS_COMPLETE:
        {
            pcStatus = "Complete";
        }
        break;

        case SMSAPI_SONG_STATUS_UNKNOWN:
        {
            // Already unknown. Just to eliminate warning
        }
        break;
    };

    return pcStatus;
}
