/******************************************************************************/
/*                    Copyright (c) Sirius XM Radio, Inc.                     */
/*                            All Rights Reserved                             */
/*         Licensed Materials - Property of Sirius XM Radio, Inc.             */
/******************************************************************************/
/*******************************************************************************
*
* DESCRIPTION
*
*  This module contains various APIs and functions which implement the
*  Sirius Module Services Event Handler.
*
******************************************************************************/
#include <time.h>
#include <stdio.h>
#include <stddef.h>

#include "standard.h"
#include "osal.h"

#include "sms_version.h"
#include "sms_obj.h"
#include "sms_event.h"
#include "_sms_event.h"

#include "sms_api_debug.h"
static const char *gpacThisFile = __FILE__;

/*****************************************************************************
*
*   SMSE_bEventHdlrInit
*
* This is the generic SMS event handler initialization function. It is called
* by an SMS thread when it starts to initialize the event handler and begin
* processing events.
*
* Inputs:
*
*       const char *pacName - An ASCII null terminated string representing a
*           unique name for this event handler.
*       SMS_OBJECT hSMS -
*           A handle to the SMS object which processes events.
*       SMS_EVENT_HANDLER *phEventHdlr - A pointer to an event control
*           structure owed by the SMS object which subscribes to events.
*       UN32 un32EventQueueSize - The number of events (messages) which can
*           be allocated at one time.
*       SMS_OBJECT_EVENT_HANDLER_PROTOTYPE vEventHandler - An event handler
*           function which is called when an event for that handler is received
*           by the generic event handler.
*       void **ppvEventHdlrArg - An argument to provide to the event
*           handler when it is called.
*
* Return:
*
*       BOOLEAN - TRUE if initialization completed, otherwise FALSE.
*
*****************************************************************************/
SMS_EVENT_HANDLER_INIT(SMSE_bEventHdlrInit)
{
    SMS_EVENT_HANDLER_STRUCT *psEventHdlr;
    BOOLEAN bPosted, bSuccess;
    OSAL_TASK_ID tThisTaskId;
#if SMS_LOGGING == 1
    OSAL_RETURN_CODE_ENUM eReturnCode;
    char acName[OSAL_MAX_OBJECT_NAME_LENGTH_WITH_NULL];
    UN8 un8Index;
#endif

    // Get our task id
    bSuccess = OSAL.bTaskGetId(&tThisTaskId);
    if (bSuccess == FALSE)
    {
        // Error!
        return FALSE;
    }

    if (phEventHdlr == NULL)
    {
        return FALSE;
    }

    // Verify only valid options were provided
    if( (un32Options & ~SMS_EVENT_HANDLER_OPTION_ALL)
        != SMS_EVENT_HANDLER_OPTION_NONE )
    {
        return FALSE;
    }

    printf("%s: SMSE_bEventHdlrInit()\n", pacName);

    // Allocate memory for the event handler structure
    psEventHdlr =
        (SMS_EVENT_HANDLER_STRUCT *)
            SMSO_hCreate(
                pacName,
                sizeof(SMS_EVENT_HANDLER_STRUCT),
                hSMS, // Child
                FALSE); // No Lock (ignored)
    if(psEventHdlr == NULL)
    {
        // Error!
        return FALSE;
    }

    // Initialize event handler object and function provided by the caller.
    psEventHdlr->hSMS = hSMS;
    psEventHdlr->tTaskId = tThisTaskId;
    psEventHdlr->vEventHandler = vEventHandler;
    psEventHdlr->hQueueData = SMS_INVALID_OBJECT;

    if ((un32Options & SMS_EVENT_HANDLER_OPTION_DUAL_Q)
            == SMS_EVENT_HANDLER_OPTION_DUAL_Q){
        psEventHdlr->psQueueInterface = &sDualQueue;
    }
    else
    {
        psEventHdlr->psQueueInterface = &sSingleQueue;
    }

    // Initialize event control structure...

    // Currently not handling any event
    psEventHdlr->sCtrl.psCurrentEvent = NULL;

    // Create Event Queue
    psEventHdlr->hQueueData =
        psEventHdlr->psQueueInterface->hCreate(
            (SMS_OBJECT)psEventHdlr,
            pacName, un32EventQueueSize);

    if (psEventHdlr->hQueueData == SMS_INVALID_OBJECT)
    {
        // Error!
        SMSE_vEventHdlrUninit(psEventHdlr);
        return FALSE;
    }

#if SMS_LOGGING == 1

    // Construct a unique name for the 'event logger'
    snprintf(&acName[0], sizeof(acName), "%s", pacName);

    eReturnCode = OSAL.eLogCreate(
        &psEventHdlr->sCtrl.sLogging.hLog, pacName,
        SMS_EVENT_LOG_SIZE, OSAL_LOG_OPTION_FORMAT);
    if(eReturnCode != OSAL_SUCCESS)
    {
        // Error!
        SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
            SMS_EVENT_NAME": Cannot create event log '%s' (%s).",
            pacName,
            OSAL.pacGetReturnCodeName(eReturnCode)
                );
        SMSE_vEventHdlrUninit(psEventHdlr);
        return FALSE;
    }

    // Initialize number of events
    psEventHdlr->sCtrl.sLogging.un32NumberOfEvents = 0;

    // Initialize maximum execution time
    psEventHdlr->sCtrl.sLogging.un32MaxExecutionMsec = 0;

    // Initialize the last queue dump time to 0
    psEventHdlr->sCtrl.sLogging.un32LastQueueLogDump = 0;

    // Initialize the last message drop counts
    for (un8Index = 0;
         un8Index < SMSE_EVENT_LOGGING_MAX_QUEUE_TRACKING;
         un8Index++)
    {
        psEventHdlr->sCtrl.sLogging.aun32LastMessageDropCount[un8Index] = 0;
    }

#endif /* SMS_LOGGING == 1 */

    // Always allocate and post the first event (INITIALIZE)
    // So it is the first event handled when the event handler runs
    bPosted = SMSE_bPostSignal(
        (SMS_EVENT_HANDLER)psEventHdlr,
        SMS_EVENT_INITIALIZE, SMS_EVENT_OPTION_NONE);
    if(bPosted == FALSE)
    {
        // Error!
        SMSE_vEventHdlrUninit(psEventHdlr);
        return FALSE;
    }

    // Pass back to caller the event handler object to use
    *phEventHdlr = (SMS_EVENT_HANDLER)psEventHdlr;

    return TRUE;
}

/*****************************************************************************
*
*   SMSE_vEventHdlrUninit
*
* This function is called to uninitialize the generic event handler for
* an SMS thread. It is called by the SMS thread when it shutsdown.
*
* Inputs:
*
*         void *pvEventHdlrArg - The event handler argument provided when
*           the event handler was initialized.
*
* Return:
*         None.
*
*****************************************************************************/
SMS_EVENT_HANDLER_UNINIT(SMSE_vEventHdlrUninit)
{
    SMS_EVENT_HANDLER_STRUCT *psEventHdlr =
        (SMS_EVENT_HANDLER_STRUCT *)hEventHdlr;
#if SMS_LOGGING == 1
    OSAL_RETURN_CODE_ENUM eReturnCode;
#endif

    puts("SMSE_vEventHdlrUninit()");

#if SMS_LOGGING == 1
    // If the Log hasn't been destroyed yet, destroy it now
    if(psEventHdlr->sCtrl.sLogging.hLog != OSAL_INVALID_OBJECT_HDL)
    {
        eReturnCode =  OSAL.eLogDelete(psEventHdlr->sCtrl.sLogging.hLog);
        if(eReturnCode == OSAL_SUCCESS)
        {
            // Invalidate log handle
            psEventHdlr->sCtrl.sLogging.hLog = OSAL_INVALID_OBJECT_HDL;
        }
    }
#endif /* SMS_LOGGING == 1 */

    // Destroy the queue data if it exists
    if (psEventHdlr->hQueueData != SMS_INVALID_OBJECT)
    {
        psEventHdlr->psQueueInterface->vDestroy(psEventHdlr->hQueueData);
        psEventHdlr->hQueueData = SMS_INVALID_OBJECT;
    }

    // Free control structure
    SMSO_vDestroy((SMS_OBJECT)psEventHdlr);

    return;
}

/*****************************************************************************
*
*   SMSE_hAllocateEvent
*
* The API is used to allocate an event from an event processor. Each
* event processor is tied directly to an SMS object which handles events
* sent to it. To send an event it is necessary to allocate a new event or
* re-allocate an existing event. Once (re)allocated the event data may be
* populated and subsequently posted. There is only one event processor
* for which an event can be posted to and handled by. It is the even processor
* indicated by the provided event control handle. This event control handle
* is associated with the SMS object which will handle the events.
*
* Upon successful allocation, the event data (a union of event data) is passed
* back to the caller via the provided ppuData pointer. The pointer may be
* de-referenced how the caller sees fit based on the type of data they are
* posting.
*
* Inputs:
*
*   SMS_EVENT_HANDLER hEventHdlr - An event handler handle held by
*       the SMS object for which this event will be posted to.
*   SMS_EVENT_TYPE_ENUM eEvent - An enumeration indicating the event type
*       (aka message or signal identifier) which will be attached to this
*       event. Note that the event type is not modifiable once allocated. It
*       can be modified however by re-allocating an event.
*   SMS_EVENT_DATA_UNION **ppuData - A pointer to a pointer to an instance of
*       the event union data structure which can be de-referenced and
*       populated by the caller prior to posting. Providing a value is
*       optional and may not be required when pre-allocating events for
*       others to use.
*   EVENT_OPTIONS_TYPE tOptions - A mask of event allocation options which
*       can be provided by the caller. They control aspects of how the event
*       is allocated and various properties. These options can be NONE,
*       URGENT, SYNCHRONOUS and/or STATIC.
*
* An event which is allocated as URGENT will be posted at the head
* of the event queue. Note if all events were allocated as URGENT this
* would implement a LIFO queue.
*
* An event which is allocated as SYNCHRONOUS will be allocated as any
* other event, but once posted will wait for the event processor to receive
* the event. This effectively blocks the caller until the event is received.
* Note that you may not post a SYNCHRONOUS event within the context of the
* receiving entity or deadlock will result.
*
* An event which is allocated as STATIC has the property of being handled
* as any other event by the event processor, but will never be consumed
* or recycled by the even processor. Typically this option is used to
* pre-allocate or reserve events for specific and critical tasks which
* cannot or should not go through the allocation/post process and simply
* should just post. This might be due to the fact that an allocation may
* either block or fail based on event availability and the event to be
* Signaled would be missed. STATIC events are essentially always allocated
* and may be re-posted at will. However each STATIC event must be posted and
* subsequently handled (even processor must run and dequeue it) before it
* can be successfully posted again.
*
* Essentially the event processor with its separate allocate/post mechanism
* has the characteristic of basically guaranteeing every post succeeds, but
* allocation of an event may block or fail based on system availability.
*
* Return:
*
*   SMS_EVENR_HDL A valid SMS_EVENT_HDL on success, otherwise
*   SMS_INVALID_EVENT_HDL
*
*****************************************************************************/
SMS_EVENT_HDL SMSE_hAllocateEvent (
    SMS_EVENT_HANDLER hEventHdlr,
    SMS_EVENT_TYPE_ENUM eEvent,
    SMS_EVENT_DATA_UNION **ppuData,
    EVENT_OPTIONS_TYPE tOptions
        )
{
    SMS_EVENT_STRUCT *psEvent = NULL;
    SMS_EVENT_HANDLER_STRUCT *psEventHdlr =
        (SMS_EVENT_HANDLER_STRUCT *)hEventHdlr;
    UN32 un32Flags = OSAL_QUEUE_FLAG_NONE;
    OSAL_OBJECT_HDL hEventQueue;
    BOOLEAN bValid;

    // Check input
    bValid = SMSO_bValid((SMS_OBJECT)hEventHdlr);
    if (bValid == FALSE)
    {
        // Error!
        return SMS_INVALID_EVENT_HDL;
    }

    // Grab the queue handle from which we
    // will allocate a message
    hEventQueue =
        psEventHdlr->psQueueInterface->hGetQueue(
            psEventHdlr->hQueueData, tOptions);
    if (hEventQueue == OSAL_INVALID_OBJECT_HDL)
    {
        // Error!
        return SMS_INVALID_EVENT_HDL;
    }

    // Determine if the URGENT flag is set
    if (tOptions & SMS_EVENT_OPTION_URGENT)
    {
        // Apply URGENT flag to event allocation
        un32Flags |= OSAL_QUEUE_FLAG_URGENT;
    }

    // Determine if the DEFERRED flag is set
    if (tOptions & SMS_EVENT_OPTION_DEFERRED)
    {
        // Apply DEFERRED flag to event allocation
        un32Flags |= OSAL_QUEUE_FLAG_DEFERRED;
    }

    // Determine if the NONBLOCK flag is set
    if (tOptions & SMS_EVENT_OPTION_NONBLOCK)
    {
        // Add non-blocking flag
        un32Flags |= OSAL_QUEUE_FLAG_NONBLOCK;
    }
    else
    {
        // Add blocking flag
        un32Flags |= OSAL_QUEUE_FLAG_BLOCK;
    }

    // Check if any special flags are needed, which cannot
    // be addressed by re-allocation. At this point only
    // URGENT and DEFERRED requests cannot be addressed
    // by re-allocation.
    if(0 == (un32Flags & OSAL_QUEUE_FLAG_URGENT) &&
       0 == (un32Flags & OSAL_QUEUE_FLAG_DEFERRED))
    {
        // Attempt re-allocation of an existing event
        psEvent = (SMS_EVENT_STRUCT *)hReAllocateEvent(
            psEventHdlr,
            eEvent,
            ppuData
                );
    }

    // Check if re-allocation was attempted and unsuccessful
    if(NULL == psEvent)
    {
        OSAL_RETURN_CODE_ENUM eReturnCode;

        // Re-allocation was unsuccessful or not attemtped.
        // We need to try another way.

        // Allocate a message from Event Queue
        eReturnCode =
            OSAL.eMessageAllocate (
            hEventQueue,
            (void *)((void *)&psEvent),
            un32Flags );
        if(eReturnCode != OSAL_SUCCESS)
        {
            if (eReturnCode != OSAL_MESSAGE_NOT_AVAILABLE)
            {
                // Error!
                SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    SMS_EVENT_NAME": Cannot allocate from event queue.");
            }
        }
        else
        {
            // Initialize internal event flags
            psEvent->tFlags = SMSE_EVENT_FLAGS_NONE;
        }
    }

    // Once we have a valid event, it either has been newly allocated
    // or it has been re-allocated. Either way we are free now to
    // manipulate it.
    if(NULL != psEvent)
    {
        // Fill in event type with provided enumeration
        psEvent->eType = eEvent;

        // Check if STATIC option selected
        if(tOptions & SMS_EVENT_OPTION_STATIC)
        {
            // Mark event as static (cannot be freed)
            psEvent->tFlags |= SMSE_EVENT_FLAGS_STATIC;
        }

        // Check if the SYNCHRONOUS option is set
        if (tOptions & SMS_EVENT_OPTION_SYNCHRONOUS)
        {
            OSAL_RETURN_CODE_ENUM eReturnCode;
            char acName[OSAL_MAX_OBJECT_NAME_LENGTH_WITH_NULL];

            // Allocate a synchronous event mutex

            // Construct a unique name for the 'event:syncmutex'
            snprintf(&acName[0], sizeof(acName), SMS_EVENT_NAME
                ":SyncMutex[%p]", (void *)psEvent);

            // Create Synchronous Event Mutex
            eReturnCode =
                OSAL.eSemCreate(
                    &psEvent->hSynchronousEventMutex,   // Mutex handle
                    &acName[0],                         // Mutex name
                    0,                                  // Initial value
                    1,                                  // Number of resources
                    OSAL_SEM_OPTION_NONE                // Options
                        );
            if(eReturnCode != OSAL_SUCCESS)
            {
                // Error!
                SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    SMS_EVENT_NAME": Cannot create synchronous event.");

                vFreeEvent(psEvent);
                return SMS_INVALID_EVENT_HDL;
            }
        }
        else // Not synchronous
        {
            psEvent->hSynchronousEventMutex = OSAL_INVALID_OBJECT_HDL;
        }

        // Remember event handler which owns this event
        psEvent->psEventHdlr = psEventHdlr;

        // Check if they want the event data returned
        if(ppuData != NULL)
        {
            *ppuData = &psEvent->uData;
        }
    }

    // Give them the event handle
    return (SMS_EVENT_HDL)psEvent;
}

/*****************************************************************************
*
*   SMSE_vReleaseStaticEvent
*
* This function allows a previously allocated static event to be returned
* to the event handler for general use.
*
* Input:
*
*   SMS_EVENT_HDL hEvent - An event to release back to the handler
*
* Return:
*
*   None.
*
*****************************************************************************/
void SMSE_vReleaseStaticEvent (
    SMS_EVENT_HDL hEvent
        )
{
    do
    {
        SMS_EVENT_STRUCT *psEvent = (SMS_EVENT_STRUCT *)hEvent;
        BOOLEAN bOwner;

        if (hEvent == SMS_INVALID_EVENT_HDL)
        {
            break;
        }

        // Verify we were provided a static event
        if ((psEvent->tFlags & SMSE_EVENT_FLAGS_STATIC) != SMSE_EVENT_FLAGS_STATIC)
        {
            break;
        }

        // Ensure we own the object and have exclusive access here
        bOwner = SMSO_bOwner((SMS_OBJECT)psEvent->psEventHdlr);
        if (bOwner == FALSE)
        {
            break;
        }

        // Remove the static flag.  We want this event to be a regular
        // event now so that it gets freed after it is serviced
        psEvent->tFlags &= ~SMSE_EVENT_FLAGS_STATIC;

        // Is the event we're attempting to free the current event?
        if (psEvent->psEventHdlr->sCtrl.psCurrentEvent == psEvent)
        {
            // Yes it is.  It'll be freed after this event completes,
            // so we don't have any more work to do here
            break;
        }

        // We just want this event to be freed.
        psEvent->tFlags |= SMSE_EVENT_FLAGS_FREE;

        // This event has to be in the queue in order to ensure
        // it gets freed by the handler.  So, post it -- since
        // it has already been allocated this post won't cause
        // a deadlock.
        SMSE_bPostEvent(hEvent);

    } while (FALSE);

    return;
}

/*****************************************************************************
*
*   SMSE_bPostEvent
*
* This function posts a previously allocated (or re-allocated) event onto
* the target SMS object's queue. Typically any valid event provided will
* always result in a successful post since the act of allocating and or
* re-allocating it always assures a place for it to be posted.
*
* Note that an event may not be posted more than once without the event
* processor first dequeuing the event before re-posting. An event need not
* be re-allocated to re-post.
*
* Inputs:
*
* SMS_EVENT_HDL hEvent - A valid event handle representing the event to
*   post.
*
* Return:
*
*   BOOLEAN - TRUE on successful post. Otherwise FALSE on error. Note that
*       if an event was attempted to post but was unsuccessful that
*       event will be automatically freed before returning. Not doing so
*       would cause an event-leak since that event would not have been
*       posted but was successfully allocated. By doing this automatically
*       within this API there is no need for anyone to ever free an event
*       and instead the generic event processor handles all event dequeues
*       and event recycling without any external interaction.
*
*****************************************************************************/
BOOLEAN SMSE_bPostEvent (
    SMS_EVENT_HDL hEvent
        )
{
    BOOLEAN bPosted = FALSE;

    // Check input
    if(hEvent != SMS_INVALID_EVENT_HDL)
    {
        SMS_EVENT_STRUCT *psEvent = (SMS_EVENT_STRUCT *)hEvent;
        BOOLEAN bAlreadyPosted = FALSE;
        BOOLEAN bSynchronousEvent =
            (psEvent->hSynchronousEventMutex != OSAL_INVALID_OBJECT_HDL) ? TRUE : FALSE;

        // Check if provided event structure has valid reference
        // to bPost method
        if( (psEvent->psEventHdlr != NULL) &&
            (psEvent->psEventHdlr->psQueueInterface != NULL) &&
            (psEvent->psEventHdlr->psQueueInterface->bPost != NULL) )
        {
            // Post onto the destination queue
            bPosted = psEvent->psEventHdlr->psQueueInterface->bPost(
                psEvent->psEventHdlr->hQueueData, hEvent,
                sizeof(SMS_EVENT_STRUCT), &bAlreadyPosted);
            if(bPosted == TRUE)
            {
                // If this event is synchronous, the event poster wants to be
                // informed when the event has been handled.
                if(bSynchronousEvent == TRUE)
                {
                    OSAL_RETURN_CODE_ENUM eSynchronousMutex;

                    /*
                    At this point the handler is handling the synchronous event (or
                    it will soon). In anycase we must wait for it to get done.

                    Attempt to get the synchronous event mutex
                    again.  This of course relies on the fact that
                    somebody posted to it (that somebody being the
                    event handler task).
                    */
                    eSynchronousMutex = OSAL.eSemTake(
                        psEvent->hSynchronousEventMutex,
                        OSAL_OBJ_TIMEOUT_INFINITE);
                    if(eSynchronousMutex == OSAL_SUCCESS)
                    {
                        // Destroy synchronous event
                        OSAL.eSemDelete( psEvent->hSynchronousEventMutex );
                        psEvent->hSynchronousEventMutex =
                            OSAL_INVALID_OBJECT_HDL;

                        // Free event now since handler did not.
                        vFreeEvent(psEvent);
                    }
                    else
                    {
                        // Error! Something is wrong.
                        SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                            SMS_EVENT_NAME ": Cannot obtain "
                            "synchronization mutex (%s).",
                            OSAL.pacGetReturnCodeName(eSynchronousMutex)
                            );
                    }
                }
            }
            else if(bAlreadyPosted == TRUE)
            {
                // Indicate event was posted(already)
                bPosted = TRUE;
            }
            else // Failed post
            {
                // Error! Something is wrong.
                SMSAPI_DEBUG_vPrintErrorFull(
                    gpacThisFile, __LINE__, SMS_EVENT_NAME
                    ": Failed to queue(post) event.");

                // If we were trying to synchronize we need to give back the mutex
                if(psEvent->hSynchronousEventMutex != OSAL_INVALID_OBJECT_HDL)
                {
                    // Destroy synchronous event
                    OSAL.eSemDelete( psEvent->hSynchronousEventMutex );
                    psEvent->hSynchronousEventMutex =
                        OSAL_INVALID_OBJECT_HDL;
                }

                // Free event, since we could not post it, something is wrong
                vFreeEvent(psEvent);
            }
        }
        else
        {
            // Error! Bad event object
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                SMS_EVENT_NAME": Incorrect event object provided. Cannot post");
        }
    }

    return bPosted;
}

/*****************************************************************************
*
*   SMSE_bPostSignal
*
* This API is used to post a signal onto an event queue. This is provided as
* an all-in-one allocation of an event, with a post followed up. It is
* not possible to send data along with an event using this method. If this
* is required the combination hAllocateEvent(), populate and bPostEvent()
* sequence must be used. This is simply a helper function which provides
* the ability to allocate and post in one simple call when only the posting of
* a signal is required (i.e. no data).
*
* Inputs:
*   hEventCtrl - An event control handle associated with the event queue
*       to allocate from and post a messaged onto.
*   eEvent - The signal code/enumeration to post.
*   tOptions - An event allocation option set.
*
* Return:
*   BOOLEAN - TRUE if signal was posted, otherwise FALSE.
*
*****************************************************************************/
BOOLEAN SMSE_bPostSignal (
    SMS_EVENT_HANDLER hEventHdlr,
    SMS_EVENT_TYPE_ENUM eEvent,
    EVENT_OPTIONS_TYPE tOptions
        )
{
    BOOLEAN bPosted = FALSE;
    SMS_EVENT_HDL hEvent;
    BOOLEAN bValid;

    // Check input
    bValid = SMSO_bValid((SMS_OBJECT)hEventHdlr);
    if (bValid == FALSE)
    {
        // Error!
        return FALSE;
    }

    // Allocate an event
    hEvent = SMSE_hAllocateEvent(hEventHdlr, eEvent, NULL, tOptions);
    if(hEvent != SMS_INVALID_EVENT_HDL)
    {
        // Post the event
        bPosted = SMSE_bPostEvent(hEvent);
    }

    return bPosted;
}

/*****************************************************************************
*
*   SMSE_eDispatchEvent
*
* This function runs in the context of the SMS thread. It is the top level
* generic event handler which dispatches specific events to the attached
* event handler belonging to the SMS object these events are targeted to.
* This API waits on an event queue for n32Timout msec and then processes
* or dispatches any event received. This API is to be
* called only by the SMS thread (SMST_n32Task) routine. Nobody else should
* ever call this!
*
* Inputs:
*   hEventCtrl - An event control handle associated with the event queue
*       to allocate from and post a messaged onto.
*   N32 n32Timeout - A timeout to pend in msec.
*
* Return:
*
*       SMS_EVENT_RESULT_ENUM - Typically this value is SMS_EVENT_RESULT_OK
*           but may be SMS_EVENT_RESULT_SHUTDOWN if the SMS thread has been
*           shutdown or may be SMS_EVENT_RESULT_ERROR if an error occurred.
*
*****************************************************************************/
SMS_EVENT_RESULT_ENUM SMSE_eDispatchEvent (
    SMS_EVENT_HANDLER hEventHdlr,
    N32 n32Timeout
        )
{
   SMS_EVENT_HANDLER_STRUCT *psEventHdlr =
        (SMS_EVENT_HANDLER_STRUCT *)hEventHdlr;
    SMS_EVENT_RESULT_ENUM eEventResult = SMS_EVENT_RESULT_OK;
    SMS_EVENT_STRUCT *psEvent = NULL;
    BOOLEAN bValid;

    // Check input
    bValid = SMSO_bValid((SMS_OBJECT)hEventHdlr);
    if (bValid == FALSE)
    {
        // Error!
        return SMS_EVENT_RESULT_ERROR;
    }

    // Get a message from the queue
    psEvent = (SMS_EVENT_STRUCT *)
        psEventHdlr->psQueueInterface->pvGetMsg(
            psEventHdlr->hQueueData, n32Timeout);
    if (psEvent != NULL)
    {

#if SMS_LOGGING == 1
        // Log start of event...
        vLogStartOfEvent(psEventHdlr, psEvent);
#endif /* SMS_LOGGING == 1 */

        // Events marked as free are only freed.  All others
        // get processed normally
        if ((psEvent->tFlags & SMSE_EVENT_FLAGS_FREE) != SMSE_EVENT_FLAGS_FREE)
        {
            // Record current event being handled
            psEventHdlr->sCtrl.psCurrentEvent = psEvent;

            // Call Event Handler and process event
            eEventResult =
                eEventHdlr(psEventHdlr, psEvent);

            // Indicate no current event handled
            psEventHdlr->sCtrl.psCurrentEvent = NULL;
        }

        // If this event is synchronous, the event poster wants to be
        // informed when the event has been handled.
        if (psEvent->hSynchronousEventMutex != OSAL_INVALID_OBJECT_HDL)
        {
            // Inform poster that we handled this event.
            // The event posting logic (waiter) will handle releasing
            // this.
            OSAL.eSemGive(psEvent->hSynchronousEventMutex);
        }
        else
        {
            // Free up event if we are finished with it
            vFreeEvent(psEvent);
        }

#if SMS_LOGGING == 1
        // Log end of event...
        vLogEndOfEvent(psEventHdlr);
#endif /* SMS_LOGGING == 1 */
    }

    // Return a handle to the event we received
    return eEventResult;
}

/*****************************************************************************
*
*   SMSE_eGetEvent
*
* This is a simple helper function which takes a valid event and extracts
* the event type (signal or message enumeration) as well as the data union
* from a received event. Note that the returned data union is constant
* which prevents the data from being modified by the caller. The only
* way to modify event data is to re-allocate it first.
*
* Note you cannot "get" event information of a posted event, only one which
* has been posted and de-queued (currently being handled). It is also not
* permissible to modify the contents of any event you "get". To do so you
* must first re-allocate it.
*
*****************************************************************************/
SMS_EVENT_TYPE_ENUM SMSE_eGetEvent (
    SMS_EVENT_HDL hEvent,
    SMS_EVENT_DATA_UNION const **ppuData
        )
{
    SMS_EVENT_TYPE_ENUM eType = SMS_EVENT_INVALID;

    // Check for validity
    if(hEvent != SMS_INVALID_EVENT_HDL)
    {
        SMS_EVENT_STRUCT *psEvent = (SMS_EVENT_STRUCT *)hEvent;

        // Extract the event type
        eType = psEvent->eType;

        // Check if they want the event data returned
        if(ppuData != NULL)
        {
            *ppuData = &psEvent->uData;
        }
    }

    return eType;
}

#if SMS_LOGGING == 1
/*****************************************************************************
*
*   SMSE_vLog
*
* This is a simple interface into the SMS Event Logging Utility.
* It works just like printf() for adding log information as the event
* handler executes. Use of the log is compiled out automatically if
* a Release build is used, otherwise logging is always enabled.
*
*****************************************************************************/
void SMSE_vLog (
    SMS_EVENT_HANDLER hEventHdlr,
    const char *pcFormat,
    ...
        )
{
    va_list tArgList;

    // grab variable arguments from the stack
    va_start(tArgList, pcFormat);

    // Call the log using list function
    SMSE_vLogUsingList(
        hEventHdlr, pcFormat, &tArgList );

    // restore stack
    va_end(tArgList);

    return;
}

/*****************************************************************************
*
*   SMSE_vLogUsingList
*
* This is a simple interface into the SMS Event Logging Utility which allows
* the caller to provide a va_list * instead of directly utilizing var args.
* It works just like printf() for adding log information as the event
* handler executes. Use of the log is compiled out automatically if
* a Release build is used, otherwise logging is always enabled.
*
*****************************************************************************/
void SMSE_vLogUsingList (
    SMS_EVENT_HANDLER hEventHdlr,
    const char *pcFormat,
    va_list *ptArgList
        )
{
    SMS_EVENT_HANDLER_STRUCT *psEventHdlr =
        (SMS_EVENT_HANDLER_STRUCT *)hEventHdlr;

    // Check input
    if (hEventHdlr != SMS_INVALID_EVENT_HANDLER)
    {
        char *pcPrepend = " ";

        // Is this the first message for this event?
        if (psEventHdlr->sCtrl.sLogging.bEventMessagePresent == FALSE)
        {
            pcPrepend = ", ";

            // There is now an event message present for the log
            psEventHdlr->sCtrl.sLogging.bEventMessagePresent = TRUE;
        }

        // Add some formatting
        OSAL.n32LogWrite(
            psEventHdlr->sCtrl.sLogging.hLog, pcPrepend);

        // Just write stuff...
        OSAL.n32VLogWrite(
            psEventHdlr->sCtrl.sLogging.hLog, pcFormat, ptArgList
            );
    }

    return;
}

#elif __STDC_VERSION__ < 199901L
void SMSE_vLogNothing (
                       SMS_EVENT_HANDLER hEventHdlr,
                       const char *pcFormat,
                       ...
                       )
{
    return;
}

void SMSE_vLogUsingListNothing (
                                SMS_EVENT_HANDLER hEventHdlr,
                                const char *pcFormat,
                                va_list *ptArgList
                                )
{
    return;
}
#endif

/*****************************************************************************
*
*   eEventHdlr
*
*   This event handler specifically handles the SHUTDOWN, SLEEP and WAKEUP
*   events generically for all SMS threads/objects. Any other event is passed
*   on to the specific event handler registered at initialization.
*
* Inputs:
*
*       void *pvEventHdlrArg - The event handler argument provided when
*           the event handler was initialized.
*       SMS_EVENT_HDL hEvent - A handle to an event (provided by the generic
*           event handler) which is being processed.
*
* Return:
*
*       SMS_EVENT_RESULT_ENUM - Typically this value is SMS_EVENT_RESULT_OK
*           but may be SMS_EVENT_RESULT_SHUTDOWN if the SMS thread has been
*           shutdown or may be SMS_EVENT_RESULT_ERROR if an error occurred.
*
*****************************************************************************/
static SMS_EVENT_RESULT_ENUM eEventHdlr (
    SMS_EVENT_HANDLER_STRUCT *psEventHdlr,
    SMS_EVENT_STRUCT *psEvent
        )
{
    SMS_EVENT_RESULT_ENUM eResult = SMS_EVENT_RESULT_OK;

    // Process Events...
    switch(psEvent->eType)
    {
        // Generic events to be processed...

        case SMS_EVENT_SHUTDOWN:
        {
            // Event handled, simply return
            eResult = SMS_EVENT_RESULT_SHUTDOWN;
        }
        break;

        case SMS_EVENT_SLEEP:
        case SMS_EVENT_WAKEUP:
        {
            // Event handled, simply return
        }
        break;

        default:
        {
            // Allow the specific object to handle the event...
            psEventHdlr->vEventHandler(psEventHdlr->hSMS, (SMS_EVENT_HDL)psEvent);
        }
        break;
    }

    return eResult;
}

/*****************************************************************************
*
*   vFreeEvent
*
* This function simply frees an event if it is not STATIC and has not
* Subsequently been re-posted since it was de-queued. A freed event
* is returned back into the available pool of events.
*
* Input:
*
*   SMS_EVENT_STRUCT *psEvent - An event pointer for which to free
*
* Return:
*
*   None.
*
*****************************************************************************/
static void vFreeEvent (
    SMS_EVENT_STRUCT *psEvent
        )
{
    BOOLEAN bFree = TRUE;

    // Check that this event is neither STATIC or REALLOCATED
    if(psEvent->tFlags & (SMSE_EVENT_FLAGS_STATIC | SMSE_EVENT_FLAGS_REALLOCATED))
    {
        // Clear REALLOCATED flag.  It was provided here to stop
        // the free below and it is no longer needed
        psEvent->tFlags &= ~SMSE_EVENT_FLAGS_REALLOCATED;

        // Do not free it.
        bFree = FALSE;
    }

    // Are we synchronizing?
    // If the synchronous event semaphore hasn't been destroyed yet,
    // destroy it now. We destroy the synchronous event first because
    // we want to make sure we have the event available before doing so.
    if (psEvent->hSynchronousEventMutex != OSAL_INVALID_OBJECT_HDL)
    {
        // Release syncronization mutex.
        OSAL.eSemGive(psEvent->hSynchronousEventMutex);

        // Event will be freed when synchronization is complete.
        bFree = FALSE;
    }

    // Can we free the event?
    if(bFree == TRUE)
    {
        // Free this message
        OSAL.eMessageFree((const void *)psEvent);
    }

    return;
}

/*****************************************************************************
*
*   hReAllocateEvent
*
* This function is similar in purpose as the earlier allocate function but
* requires an already existing event before one can be reallocated. This
* function avoids the potential of an event allocation block or failure
* by re-using or re-cycling an existing event (typically the one currently
* being handled). As long as the event is valid and has not been previously
* re-posted this API call will be successful. Note that using the API
* is the only way to change an existing event's type (eEvent). Also
* this is the only way to obtain a pointer to and modify an event's
* data union for subsequent re-posting.
*
* Inputs:
*
*   SMS_EVENT_HDL hEvent - An existing event which has been previously allocated
*       or has been provided (currently being handled) by the event
*       processor.
*   SMS_EVENT_TYPE_ENUM eEvent - The new event type (signal/msg type)
*   SMS_EVENT_DATA_UNION **ppuData - An optional pointer-to a pointer to
*       an instance of the event's data union which may be de-refferenced
*       and populated as needed by the caller.
*
* Return:
*
*   SMS_EVENR_HDL A valid SMS_EVENT_HDL on success, otherwise
*   SMS_INVALID_EVENT_HDL
*
*****************************************************************************/
static SMS_EVENT_HDL hReAllocateEvent (
    SMS_EVENT_HANDLER_STRUCT *psEventHdlr,
    SMS_EVENT_TYPE_ENUM eEvent,
    SMS_EVENT_DATA_UNION **ppuData
        )
{
    SMS_EVENT_HDL hNewEvent = SMS_INVALID_EVENT_HDL;

    // Verify we have a valid event handler
    if(psEventHdlr != NULL)
    {
        BOOLEAN bIsOwner;

        // Re-Allocate an event from ourself since we own the event
        // handler object (and queue). This prevents us from potential
        // problems with attempting to allocate an event from our
        // own full queue. Plus since we are already processing an
        // event, we might as well re-use it, which guarantees we
        // have an event to populate.

        // Check for ownership of this object and ensure we're in the
        // correct context
        bIsOwner = SMSO_bTaskOwner(
            (SMS_OBJECT)psEventHdlr, psEventHdlr->tTaskId);
        if(bIsOwner == TRUE)
        {
            // Does an event exist?
            if(psEventHdlr->sCtrl.psCurrentEvent != NULL)
            {
                // Can it be re-allocated? Must have no flags and
                // must not be synchronous.
                if((psEventHdlr->sCtrl.psCurrentEvent->tFlags
                        == SMSE_EVENT_FLAGS_NONE) &&
                   (psEventHdlr->sCtrl.psCurrentEvent->hSynchronousEventMutex ==
                        OSAL_INVALID_OBJECT_HDL)
                            )
                {
                    // We can re-allocate this event.
                    // Fill in event type with provided enumeration
                    psEventHdlr->sCtrl.psCurrentEvent->eType = eEvent;

                    // Prevent this event from being freed
                    psEventHdlr->sCtrl.psCurrentEvent->tFlags =
                        SMSE_EVENT_FLAGS_REALLOCATED;

                    // Check if they want the event data returned
                    if(ppuData != NULL)
                    {
                        *ppuData = &psEventHdlr->sCtrl.psCurrentEvent->uData;
                    }

                    // Return this as a new event
                    hNewEvent = (SMS_EVENT_HDL)
                        psEventHdlr->sCtrl.psCurrentEvent;
                }
            }
        }
    }

    return hNewEvent;
}

#if SMS_LOGGING == 1
/*****************************************************************************
*
*   vLogStartOfEvent
*
* This function simply writes an event to its log.
*
*****************************************************************************/
static void vLogStartOfEvent (
    SMS_EVENT_HANDLER_STRUCT *psEventHdlr,
    SMS_EVENT_STRUCT *psEvent
        )
{
    SMS_EVENT_LOGGING_STRUCT *psLogging = &psEventHdlr->sCtrl.sLogging;

    // First log? Print header
    if(psLogging->un32NumberOfEvents == 0)
    {
        OSAL.n32LogWrite(psLogging->hLog,
            "Event#,Timestamp,Type,Type Value,Flags,Event "
            "Messages,Duration(msec)\n"
            );
    }

    // Increment event number
    ++psLogging->un32NumberOfEvents;

    // Time stamp start
    OSAL.vTimeUp(&psLogging->sStart.un32Seconds,
        &psLogging->sStart.un16MilliSeconds);

    // Write start of log info...
    OSAL.n32LogWrite(psLogging->hLog,
        "%u,%u,%s,%u,%#x",
        psLogging->un32NumberOfEvents,
        psLogging->sStart.un32Seconds * 1000 +
        psLogging->sStart.un16MilliSeconds, // msec
        pacEventTypeText(psEvent->eType),
        psEvent->eType,
        psEvent->tFlags
        );

    return;
}

/*****************************************************************************
*
*   vLogEndOfEvent
*
* This function simply writes an event to its log.
*
*****************************************************************************/
static void vLogEndOfEvent (
    SMS_EVENT_HANDLER_STRUCT *psEventHdlr
        )
{
    UN32 un32DiffMsec;
    SMS_EVENT_LOGGING_STRUCT *psLogging = &psEventHdlr->sCtrl.sLogging;

    // Time stamp end
    OSAL.vTimeUp(&psLogging->sEnd.un32Seconds,
        &psLogging->sEnd.un16MilliSeconds);

    // Compute time difference (event duration)
    un32DiffMsec =
        ((1000 * (psLogging->sEnd.un32Seconds - psLogging->sStart.un32Seconds)) +
        psLogging->sEnd.un16MilliSeconds) - psLogging->sStart.un16MilliSeconds;

    // Check if a new maximum occurred.
    if(un32DiffMsec > psLogging->un32MaxExecutionMsec)
    {
        // Set new maximum.
        psLogging->un32MaxExecutionMsec = un32DiffMsec;
    }

    // Check if any messages were added to the event
    if (psEventHdlr->sCtrl.sLogging.bEventMessagePresent == FALSE)
    {
        // Indicate no messages present
        OSAL.n32LogWrite(psLogging->hLog,",<NONE>");
    }

    // Reset the event message flag
    psEventHdlr->sCtrl.sLogging.bEventMessagePresent = FALSE;

    // Write end of log info...
    OSAL.n32LogWrite(psLogging->hLog,",%u\n",un32DiffMsec);

    // Is it time to print the Event Queue statistics?
    if ((psLogging->sEnd.un32Seconds - psLogging->un32LastQueueLogDump)
        >= SMSE_EVENT_LOGGING_RATE_SECS)
    {
        // Update last log time
        psLogging->un32LastQueueLogDump = psLogging->sEnd.un32Seconds;

        OSAL.n32LogWrite(psLogging->hLog, SMSE_EVENT_LOGGING_BANNER);

        // Iterate the queues to print their data
        if ((SMSE_ITERATE_QUEUES)NULL !=
             psEventHdlr->psQueueInterface->vIterateQueues)
        {
            psEventHdlr->psQueueInterface->vIterateQueues(
                psEventHdlr->hQueueData,
                (SMSE_ITERATE_QUEUE_CALLBACK)bIterateQueuesForLog,
                (void *)psLogging);
        }

        OSAL.n32LogWrite(psLogging->hLog, SMSE_EVENT_LOGGING_END);
    }

    return;
}

/*****************************************************************************
*
*   bIterateQueuesForLog
*
*****************************************************************************/
static BOOLEAN bIterateQueuesForLog (
    OSAL_OBJECT_HDL hEventQueue,
    const char *pacQueueName,
    UN8 un8QueueIndex,
    SMS_EVENT_LOGGING_STRUCT *psLogging
        )
{
    OSAL_QUEUE_INFO_STRUCT sQueueInfo;
    OSAL_RETURN_CODE_ENUM eReturnCode;

    // Validate inputs
    if (((SMS_EVENT_LOGGING_STRUCT *)NULL == psLogging) ||
        (OSAL_INVALID_OBJECT_HDL == hEventQueue) ||
        (OSAL_INVALID_OBJECT_HDL == psLogging->hLog) ||
        ((const char *)NULL == pacQueueName))
    {
        // Stop here
        return FALSE;
    }

    // Get the info on this queue
    eReturnCode = OSAL.eQueueGetInfo(hEventQueue, &sQueueInfo);

    // Did that work?
    if (OSAL_SUCCESS == eReturnCode)
    {
        UN32 un32PercentDropped = 0, un32NewDropped = 0;

        // Only calculate the drop percentage if we have a valid
        // denominator
        if (0 != sQueueInfo.sAllocSemInfo.un32TakeAttemptCount)
        {
            // Report a simple rounded value for this
            // percentage
            un32PercentDropped =
                (sQueueInfo.un32FailedAllocations * 100)
                / sQueueInfo.sAllocSemInfo.un32TakeAttemptCount;
        }

        if (un8QueueIndex < SMSE_EVENT_LOGGING_MAX_QUEUE_TRACKING)
        {
            // Calculate how many have been dropped since the last report
            un32NewDropped = sQueueInfo.un32FailedAllocations
                - psLogging->aun32LastMessageDropCount[un8QueueIndex];

            // Update the last drop count now
            psLogging->aun32LastMessageDropCount[un8QueueIndex] =
                sQueueInfo.un32FailedAllocations;
        }

        // Log now
        OSAL.n32LogWrite(psLogging->hLog,
            "%s (Depth: %u)\n"
            "\t  Messages Queued: %u\n"
            "\t  Messages Available: %u\n"
            "\t  Messages Taken: %u\n"
            "\t  Maximum Messages Taken: %u\n"
            "\t  Dropped Messages Since Last Report: %u\n"
            "\t  Total Dropped Messages: %u\n"
            "\t  Message Drop Rate: %u%%\n",
            pacQueueName,
            sQueueInfo.un32QueueDepth,
            sQueueInfo.un32MessagesQueued,
            sQueueInfo.un32MessagesAvailable,
            sQueueInfo.un32MessagesTaken,
            sQueueInfo.sAllocSemInfo.un32MaxCount,
            un32NewDropped,
            sQueueInfo.un32FailedAllocations,
            un32PercentDropped
                );
    }

    // Keep iteratin'
    return TRUE;
}

/*****************************************************************************
*
*   pacEventTypeText
*
* This is a local function which simply maps an enumerated type to
* a textual representation for formatting the enumerated type.
*
*****************************************************************************/
static const char *pacEventTypeText(
    SMS_EVENT_TYPE_ENUM eType
        )
{
    const char *pacReturnString = NULL;

    switch (eType)
    {
        case SMS_EVENT_INITIALIZE:
            pacReturnString = MACRO_TO_STRING(SMS_EVENT_INITIALIZE);
            break;

        case SMS_EVENT_SHUTDOWN:
            pacReturnString = MACRO_TO_STRING(SMS_EVENT_SHUTDOWN);
            break;

        case SMS_EVENT_SLEEP:
            pacReturnString = MACRO_TO_STRING(SMS_EVENT_SLEEP);
            break;

        case SMS_EVENT_WAKEUP:
            pacReturnString = MACRO_TO_STRING(SMS_EVENT_WAKEUP);
            break;

        case SMS_EVENT_RESET:
            pacReturnString = MACRO_TO_STRING(SMS_EVENT_RESET);
            break;

        case SMS_EVENT_STOP:
            pacReturnString = MACRO_TO_STRING(SMS_EVENT_STOP)"(RELEASE)";
            break;

        case SMS_EVENT_UPDATE_OBJECT:
            pacReturnString = MACRO_TO_STRING(SMS_EVENT_UPDATE_OBJECT);
            break;

        case SMS_EVENT_RELEASE:
            pacReturnString = MACRO_TO_STRING(SMS_EVENT_RELEASE);
            break;

        case SMS_EVENT_REMOVE_DECODER:
            pacReturnString = MACRO_TO_STRING(SMS_EVENT_REMOVE_DECODER);
            break;

        case SMS_EVENT_REMOVE_MODULE:
            pacReturnString = MACRO_TO_STRING(SMS_EVENT_REMOVE_MODULE);
            break;

        case SMS_EVENT_ERROR:
            pacReturnString = MACRO_TO_STRING(SMS_EVENT_ERROR);
            break;

        case SMS_EVENT_TIMEOUT:
            pacReturnString = MACRO_TO_STRING(SMS_EVENT_TIMEOUT);
            break;

        case SMS_EVENT_CCACHE:
            pacReturnString = MACRO_TO_STRING(SMS_EVENT_CCACHE);
            break;

        case SMS_EVENT_CHANNEL:
            pacReturnString = MACRO_TO_STRING(SMS_EVENT_CHANNEL);
            break;

        case SMS_EVENT_INVALID:
            pacReturnString = MACRO_TO_STRING(SMS_EVENT_INVALID);
            break;

        case SMS_EVENT_ASSOCIATE:
            pacReturnString = MACRO_TO_STRING(SMS_EVENT_ASSOCIATE);
            break;

        case SMS_EVENT_UNASSOCIATE:
            pacReturnString = MACRO_TO_STRING(SMS_EVENT_UNASSOCIATE);
            break;

        case SMS_EVENT_UPDATE_TIME:
            pacReturnString = MACRO_TO_STRING(SMS_EVENT_UPDATE_TIME);
            break;

        case SMS_EVENT_SYNCHRONIZE_TIME:
            pacReturnString = MACRO_TO_STRING(SMS_EVENT_SYNCHRONIZE_TIME);
            break;

        case SMS_EVENT_SAVE_TIME_PARAMS:
            pacReturnString = MACRO_TO_STRING(SMS_EVENT_SAVE_TIME_PARAMS);
            break;

        case SMS_EVENT_UPDATE_SIGNAL:
            pacReturnString = MACRO_TO_STRING(SMS_EVENT_UPDATE_SIGNAL);
            break;

        case SMS_EVENT_UPDATE_ANTENNA:
            pacReturnString = MACRO_TO_STRING(SMS_EVENT_UPDATE_ANTENNA);
            break;

        case SMS_EVENT_PLAY:
            pacReturnString = MACRO_TO_STRING(SMS_EVENT_PLAY);
            break;

        case SMS_EVENT_PAUSE:
            pacReturnString = MACRO_TO_STRING(SMS_EVENT_PAUSE);
            break;

        case SMS_EVENT_SEEK_TIME:
            pacReturnString = MACRO_TO_STRING(SMS_EVENT_SEEK_TIME);
            break;

        case SMS_EVENT_SEEK_SONG:
            pacReturnString = MACRO_TO_STRING(SMS_EVENT_SEEK_SONG);
            break;

        case SMS_EVENT_SEEK_NEXT:
            pacReturnString = MACRO_TO_STRING(SMS_EVENT_SEEK_NEXT);
            break;

        case SMS_EVENT_SEEK_PREVIOUS:
            pacReturnString = MACRO_TO_STRING(SMS_EVENT_SEEK_PREVIOUS);
            break;

        case SMS_EVENT_PRINT_SCACHE:
            pacReturnString = MACRO_TO_STRING(SMS_EVENT_PRINT_SCACHE);
            break;

        case SMS_EVENT_CREATE_SONGLIST:
            pacReturnString = MACRO_TO_STRING(SMS_EVENT_CREATE_SONGLIST);
            break;

        case SMS_EVENT_DATASERVICE:
            pacReturnString = MACRO_TO_STRING(SMS_EVENT_DATASERVICE);
            break;

        case SMS_EVENT_RFD_INTERFACE:
            pacReturnString = MACRO_TO_STRING(SMS_EVENT_RFD_INTERFACE);
            break;

        case SMS_EVENT_DISPATCH:
            pacReturnString = MACRO_TO_STRING(SMS_EVENT_DISPATCH);
            break;

        case SMS_EVENT_RADIO_READY:
            pacReturnString = MACRO_TO_STRING(SMS_EVENT_RADIO_READY);
            break;

        case SMS_EVENT_RADIO_RELEASED:
            pacReturnString = MACRO_TO_STRING(SMS_EVENT_RADIO_RELEASED);
            break;

        case SMS_EVENT_ALL_CHAN_CAT_NOTIFY:
            pacReturnString = MACRO_TO_STRING(SMS_EVENT_ALL_CHAN_CAT_NOTIFY);
            break;

        case SMS_EVENT_DESTROY_SONGLIST:
            pacReturnString = MACRO_TO_STRING(SMS_EVENT_DESTROY_SONGLIST);
            break;

        case SMS_EVENT_MODIFY_EVENT_MASK:
            pacReturnString = MACRO_TO_STRING(SMS_EVENT_MODIFY_EVENT_MASK);
            break;

        case SMS_EVENT_MODIFY_CHANLIST_EVENT_MASK:
            pacReturnString = MACRO_TO_STRING(SMS_EVENT_MODIFY_CHANLIST_EVENT_MASK);
            break;

        case SMS_EVENT_TONE_GENERATION:
            pacReturnString = MACRO_TO_STRING(SMS_EVENT_TONE_GENERATION);
            break;

        case SMS_EVENT_METADATA_WAIT_TIMEOUT:
            pacReturnString = MACRO_TO_STRING(SMS_EVENT_METADATA_WAIT_TIMEOUT);
            break;

        case SMS_EVENT_METADATA_CHANGED:
            pacReturnString = MACRO_TO_STRING(SMS_EVENT_METADATA_CHANGED);
            break;

        case SMS_EVENT_TEXT_REPLACEMENT_TIMEOUT:
            pacReturnString = MACRO_TO_STRING(SMS_EVENT_TEXT_REPLACEMENT_TIMEOUT);
            break;

        case SMS_EVENT_FEATURED_FAVORITES:
            pacReturnString = MACRO_TO_STRING(SMS_EVENT_FEATURED_FAVORITES);
            break;

        case SMS_EVENT_SMART_FAVORITES:
            pacReturnString = MACRO_TO_STRING(SMS_EVENT_SMART_FAVORITES);
            break;

        case SMS_EVENT_TUNE_SCAN:
            pacReturnString = MACRO_TO_STRING(SMS_EVENT_TUNE_SCAN);
            break;

        case SMS_EVENT_AUDIO_CONTROL:
            pacReturnString = MACRO_TO_STRING(SMS_EVENT_AUDIO_CONTROL);
            break;

        case SMS_EVENT_MODIFY_ENGINEERING_DATA_STATE:
            pacReturnString = MACRO_TO_STRING(SMS_EVENT_MODIFY_ENGINEERING_DATA_STATE);
            break;

        case SMS_EVENT_ART:
            pacReturnString = MACRO_TO_STRING(SMS_EVENT_ART);
            break;

        case SMS_EVENT_EPG:
            pacReturnString = MACRO_TO_STRING(SMS_EVENT_EPG);
            break;

        case SMS_EVENT_BROWSE:
            pacReturnString = MACRO_TO_STRING(SMS_EVENT_BROWSE);
            break;

        case SMS_EVENT_TUNE:
            pacReturnString = MACRO_TO_STRING(SMS_EVENT_TUNE);
            break;

        case SMS_EVENT_TUNEMIX_CONFIGURE:
            pacReturnString = MACRO_TO_STRING(SMS_EVENT_TUNEMIX_CONFIGURE);
            break;

        case SMS_EVENT_SPORTS_FLASH:
            pacReturnString = MACRO_TO_STRING(SMS_EVENT_SPORTS_FLASH);
            break;

        case SMS_EVENT_TW_NOW:
            pacReturnString = MACRO_TO_STRING(SMS_EVENT_TW_NOW);
            break;

        case SMS_EVENT_RADIO:
            // This will be handled among radio-specific events
            // Do not do anything here
            break;

        /***************************************************************
         * Intentionally avoiding "default" case in this switch to have
         * compiler warning generated once any new event is added into
         * enum but not handled in this function
         ***************************************************************/
    }

    if(pacReturnString == NULL)
    {
        // Try radio-specific events
        pacReturnString = RADIO_pacEventTypeText(eType);
        if(pacReturnString == NULL)
        {
            pacReturnString = "UNKNOWN";
        }
    }

    return pacReturnString;
}

#endif /* SMS_LOGGING == 1 */

/*  INTERFACE FUNCTIONS FOR SINGLE QUEUE*/

/*****************************************************************************
*
*   hCreateQueue
*
* This is an interface function which creates a single-queue object.
*
*****************************************************************************/
static SMS_OBJECT hCreateQueue (
    SMS_OBJECT hParent,
    const char *pacName,
    UN32 un32EventQueueSize
        )
{
    SMS_EVENT_SINGLE_QUEUE_STRUCT *psSingle =
        (SMS_EVENT_SINGLE_QUEUE_STRUCT *)NULL;
    do
    {

        OSAL_RETURN_CODE_ENUM eReturnCode;
        char acName[OSAL_MAX_OBJECT_NAME_LENGTH_WITH_NULL];

        // Construct a unique name for the 'event:queue'
        snprintf(&acName[0], sizeof(acName), "%s:SingleQueue", pacName);

        // Allocate memory for the queue structure
        psSingle = (SMS_EVENT_SINGLE_QUEUE_STRUCT *)SMSO_hCreate(
            &acName[0],
            sizeof(SMS_EVENT_SINGLE_QUEUE_STRUCT),
            hParent, FALSE);

        if ( psSingle == (SMS_EVENT_SINGLE_QUEUE_STRUCT *)NULL)
        {
            break;
        }

        // Construct a unique name for the 'event:queue'
        snprintf(&acName[0], sizeof(acName), "%s:EventQueue", pacName);

        // Create Event Queue
        eReturnCode =
            OSAL.eQueueCreate(
                &psSingle->hEventQueue,
                &acName[0],
                un32EventQueueSize,
                sizeof(SMS_EVENT_STRUCT),
                OSAL_QUEUE_OPTION_FIXED_SIZE
                    );
        if(eReturnCode != OSAL_SUCCESS)
        {
            break;
        }

        return (SMS_OBJECT)psSingle;

    } while (FALSE);

    if (psSingle != (SMS_EVENT_SINGLE_QUEUE_STRUCT *)NULL)
    {
        vDestroyQueue((SMS_OBJECT)psSingle);
    }

    return SMS_INVALID_OBJECT;
}

/*****************************************************************************
*
*   vDestroyQueue
*
* This is an interface function which destroys a single-queue object.
*
*****************************************************************************/
static void vDestroyQueue (
    SMS_OBJECT hQueueData
        )
{
    BOOLEAN bOwner;

    bOwner = SMSO_bOwner(hQueueData);
    if (bOwner == TRUE)
    {
        SMS_EVENT_SINGLE_QUEUE_STRUCT *psSingle =
            (SMS_EVENT_SINGLE_QUEUE_STRUCT *)hQueueData;
        OSAL_RETURN_CODE_ENUM eReturnCode;

        // If the Queue hasn't been destroyed yet, destroy it now
        if(psSingle->hEventQueue != OSAL_INVALID_OBJECT_HDL)
        {
            eReturnCode = OSAL.eQueueDelete(psSingle->hEventQueue);
            if(eReturnCode == OSAL_SUCCESS)
            {
                // Invalidate queue handle
                psSingle->hEventQueue = OSAL_INVALID_OBJECT_HDL;

                // Destroy the interface data
                SMSO_vDestroy((SMS_OBJECT)psSingle);
            }
        }
    }

    return;
}

/*****************************************************************************
*
*   hQueueForEvent
*
* This is an interface function which provides a handle to the queue
* the interface object wishes SMSE to use when allocating an event.
*
*****************************************************************************/
static OSAL_OBJECT_HDL hQueueForEvent (
    SMS_OBJECT hQueueData,
    EVENT_OPTIONS_TYPE tEventOptions
        )
{
    BOOLEAN bValid;

    bValid = SMSO_bValid(hQueueData);
    if (bValid == TRUE)
    {
        SMS_EVENT_SINGLE_QUEUE_STRUCT *psSingle =
            (SMS_EVENT_SINGLE_QUEUE_STRUCT *)hQueueData;

        return psSingle->hEventQueue;
    }

    return OSAL_INVALID_OBJECT_HDL;
}

/*****************************************************************************
*
*   pvGetMsg
*
* This is an interface function which blocks on the queue while
* waiting for a message to become available.
*
*****************************************************************************/
static void *pvGetMsg (
    SMS_OBJECT hQueueData,
    N32 n32Timeout
        )
{
    BOOLEAN bValid;

    bValid = SMSO_bValid(hQueueData);
    if (bValid == TRUE)
    {
        SMS_EVENT_SINGLE_QUEUE_STRUCT *psSingle =
            (SMS_EVENT_SINGLE_QUEUE_STRUCT *)hQueueData;
        OSAL_RETURN_CODE_ENUM eReturnCode;
        void *pvMessage = NULL;
        UN32 un32MessageSize;

        // Get a message from the Event Queue
        eReturnCode =
            OSAL.eQueueGet (
            psSingle->hEventQueue,
            &pvMessage, &un32MessageSize,
            n32Timeout);
        if (eReturnCode == OSAL_SUCCESS)
        {
            return pvMessage;
        }
    }

    return NULL;
}

/*****************************************************************************
*
*   bPostEvent
*
* This is an interface function which posts a new event to the queue
*
*****************************************************************************/
static BOOLEAN bPostEvent (
    SMS_OBJECT hQueueData,
    const void *pvMessage,
    UN32 un32MessageSize,
    BOOLEAN *pbAlreadyPresent
        )
{
    OSAL_RETURN_CODE_ENUM eReturnCode;

    *pbAlreadyPresent = FALSE;

    eReturnCode =
        OSAL.eQueuePut(pvMessage, un32MessageSize);

    // Report success if the put worked
    if (eReturnCode == OSAL_SUCCESS)
    {
        return TRUE;
    }
    //  If the data was already there then report that as well
    else if (eReturnCode == OSAL_QUEUE_MESSAGE_ALREADY_QUEUED)
    {
        *pbAlreadyPresent = TRUE;
    }

    return FALSE;
}

/*****************************************************************************
*
*   vIterateQueue
*
* This is an interface function which "iterates" the single queue
* used by this interface
*
*****************************************************************************/
static void vIterateQueue (
    SMS_OBJECT hQueueData,
    SMSE_ITERATE_QUEUE_CALLBACK bCallback,
    void *pvCallbackArg
        )
{
    BOOLEAN bValid;

    // Validate function pointer
    if ((SMSE_ITERATE_QUEUE_CALLBACK)NULL == bCallback)
    {
        return;
    }

    bValid = SMSO_bValid(hQueueData);
    if (TRUE == bValid)
    {
        SMS_EVENT_SINGLE_QUEUE_STRUCT *psSingle =
            (SMS_EVENT_SINGLE_QUEUE_STRUCT *)hQueueData;

        // Provide them with our only queue (index 0)
        bCallback(psSingle->hEventQueue, "Queue", 0, pvCallbackArg);
    }

    return;
}

/* INTERFACE FUNCTIONS FOR DUAL QUEUE */

/*****************************************************************************
*
*   hCreateDualQueue
*
* This is an interface function which creates a dual-queue object.
*
*****************************************************************************/
static SMS_OBJECT hCreateDualQueue (
    SMS_OBJECT hParent,
    const char *pacName,
    UN32 un32EventQueueSize
        )
{
    SMS_EVENT_DUAL_QUEUE_STRUCT *psDual =
        (SMS_EVENT_DUAL_QUEUE_STRUCT *)NULL;
    do
    {

        OSAL_RETURN_CODE_ENUM eReturnCode;
        char acName[OSAL_MAX_OBJECT_NAME_LENGTH_WITH_NULL];

        // Construct a unique name for the 'event:queue'
        snprintf(&acName[0], sizeof(acName), "%s:DualQueue", pacName);

        // Allocate memory for the queue structure
        psDual = (SMS_EVENT_DUAL_QUEUE_STRUCT *)SMSO_hCreate(
            &acName[0],
            sizeof(SMS_EVENT_DUAL_QUEUE_STRUCT),
            hParent, FALSE);

        if ( psDual == (SMS_EVENT_DUAL_QUEUE_STRUCT *)NULL)
        {
            break;
        }

        // Construct a unique name for the signal semaphore
        snprintf(&acName[0], sizeof(acName), "%s:EventQueue1", pacName);

        // Create signal semaphore. This is used as a latch (count-up)
        // rather than as a protection semaphore (count down)
        eReturnCode = OSAL.eSemCreate(
            &psDual->hSignalSem,
            &acName[0],
            0, un32EventQueueSize * 2,
            OSAL_SEM_OPTION_NONE);
        if(eReturnCode != OSAL_SUCCESS)
        {
            break;
        }

        // Construct a unique name for queue 1
        snprintf(&acName[0], sizeof(acName), "%s:EventQueue1", pacName);

        // Create Event Queue 1
        eReturnCode =
            OSAL.eQueueCreate(
                &psDual->hEventQueue1,
                &acName[0],
                un32EventQueueSize,
                sizeof(SMS_EVENT_STRUCT),
                OSAL_QUEUE_OPTION_FIXED_SIZE
                    );
        if(eReturnCode != OSAL_SUCCESS)
        {
            break;
        }

        // Construct a unique name for queue 2
        snprintf(&acName[0], sizeof(acName), "%s:EventQueue2", pacName);

        // Create Event Queue 2
        eReturnCode =
            OSAL.eQueueCreate(
                &psDual->hEventQueue2,
                &acName[0],
                un32EventQueueSize,
                sizeof(SMS_EVENT_STRUCT),
                OSAL_QUEUE_OPTION_FIXED_SIZE
                    );
        if(eReturnCode != OSAL_SUCCESS)
        {
            break;
        }

        return (SMS_OBJECT)psDual;

    } while (FALSE);

    if (psDual != (SMS_EVENT_DUAL_QUEUE_STRUCT *)NULL)
    {
        vDestroyQueue((SMS_OBJECT)psDual);
    }

    return SMS_INVALID_OBJECT;
}

/*****************************************************************************
*
*   vDestroyDualQueue
*
* This is an interface function which destroys a dual-queue object.
*
*****************************************************************************/
static void vDestroyDualQueue (
    SMS_OBJECT hQueueData
        )
{
    do
    {
        BOOLEAN bOwner;
        SMS_EVENT_DUAL_QUEUE_STRUCT *psDual =
            (SMS_EVENT_DUAL_QUEUE_STRUCT *)hQueueData;
        OSAL_RETURN_CODE_ENUM eReturnCode;

        bOwner = SMSO_bOwner(hQueueData);
        if (bOwner == FALSE)
        {
            break;
        }

        // This semaphore controls the rest of the
        // attributes of this structure, so if it isn't
        // there then we have nothing do do with the
        // OSAL queues
        if (psDual->hSignalSem != OSAL_INVALID_OBJECT_HDL)
        {
            // Now, destroy it
            eReturnCode = OSAL.eSemDelete(psDual->hSignalSem);
            if (eReturnCode != OSAL_SUCCESS)
            {
                break;
            }

            // Clear the handle
            psDual->hSignalSem = OSAL_INVALID_OBJECT_HDL;

            // Destroy queue 1
            if (psDual->hEventQueue1 != OSAL_INVALID_OBJECT_HDL)
            {
                eReturnCode = OSAL.eQueueDelete(psDual->hEventQueue1);
                if (eReturnCode != OSAL_SUCCESS)
                {
                    break;
                }

                psDual->hEventQueue1 = OSAL_INVALID_OBJECT_HDL;
            }

            // Destroy queue 2
            if (psDual->hEventQueue2 != OSAL_INVALID_OBJECT_HDL)
            {
                eReturnCode = OSAL.eQueueDelete(psDual->hEventQueue2);
                if (eReturnCode != OSAL_SUCCESS)
                {
                    break;
                }

                psDual->hEventQueue2 = OSAL_INVALID_OBJECT_HDL;
            }
        }

        // Destroy the object now
        SMSO_vDestroy(hQueueData);

    } while (FALSE);

    return;
}

/*****************************************************************************
*
*   hQueueForEventDualQueue
*
* This is an interface function which provides a handle to the queue
* the interface object wishes SMSE to use when allocating an event.
*
*****************************************************************************/
static OSAL_OBJECT_HDL hQueueForEventDualQueue (
    SMS_OBJECT hQueueData,
    EVENT_OPTIONS_TYPE tEventOptions
        )
{
    BOOLEAN bValid;

    bValid = SMSO_bValid(hQueueData);
    if (bValid == TRUE)
    {
        SMS_EVENT_DUAL_QUEUE_STRUCT *psDual =
            (SMS_EVENT_DUAL_QUEUE_STRUCT *)hQueueData;

        // All non-blocking events go on the second queue
        if ((tEventOptions & SMS_EVENT_OPTION_NONBLOCK) == SMS_EVENT_OPTION_NONBLOCK)
        {
            return psDual->hEventQueue2;
        }

        return psDual->hEventQueue1;
    }

    return OSAL_INVALID_OBJECT_HDL;
}

/*****************************************************************************
*
*   pvGetMsgDualQueue
*
* This is an interface function which blocks on the queues while
* waiting for a message to become available.
*
*****************************************************************************/
static void *pvGetMsgDualQueue (
    SMS_OBJECT hQueueData,
    N32 n32Timeout
        )
{

    void *pvMessage = NULL;

    do
    {
        SMS_EVENT_DUAL_QUEUE_STRUCT *psDual =
            (SMS_EVENT_DUAL_QUEUE_STRUCT *)hQueueData;
        BOOLEAN bValid;
        OSAL_RETURN_CODE_ENUM eReturnCode;
        UN32 un32MessageSize;

        bValid = SMSO_bValid(hQueueData);
        if (bValid == FALSE)
        {
            break;
        }

        // Wait on the signal semaphore first
        eReturnCode = OSAL.eSemTake(psDual->hSignalSem, n32Timeout);

        if (eReturnCode != OSAL_SUCCESS)
        {
            break;
        }

        // Try to get a message out of queue 1 (no waiting, 'cause we already did that)
        eReturnCode =
            OSAL.eQueueGet (
            psDual->hEventQueue1,
            &pvMessage, &un32MessageSize, 0);

        if (eReturnCode == OSAL_QUEUE_EMPTY)
        {
            // Nothing in the first queue, now look at the second

            // Try to get a message out of queue 2 (no waiting, 'cause we already did that)
            eReturnCode =
                OSAL.eQueueGet (
                psDual->hEventQueue2,
                &pvMessage, &un32MessageSize,0);
        }

        if (eReturnCode != OSAL_SUCCESS)
        {
            break;
        }

        return pvMessage;
    } while (FALSE);

    return NULL;
}

/*****************************************************************************
*
*   bPostEventDualQueue
*
* This is an interface function which posts a new event to the queue and
* signals any waiting task via the signal semaphore
*
*****************************************************************************/
static BOOLEAN bPostEventDualQueue (
    SMS_OBJECT hQueueData,
    const void *pvMessage,
    UN32 un32MessageSize,
    BOOLEAN *pbAlreadyPresent
        )
{
    BOOLEAN bValid;
    SMS_EVENT_DUAL_QUEUE_STRUCT *psDual =
            (SMS_EVENT_DUAL_QUEUE_STRUCT *)hQueueData;
    OSAL_RETURN_CODE_ENUM eReturnCode;

    *pbAlreadyPresent = FALSE;

    bValid = SMSO_bValid(hQueueData);
    if (bValid == FALSE)
    {
        return FALSE;
    }

    *pbAlreadyPresent = FALSE;

    // Add the message to the queue
    eReturnCode =
        OSAL.eQueuePut(pvMessage, un32MessageSize);

    // Report success if the put worked
    if (eReturnCode == OSAL_SUCCESS)
    {
        // Now, post to the signal semaphore that
        // something is available now

        // Don't check the return code here because we
        // can't really do anything to respond to a failure
        OSAL.eSemGive(psDual->hSignalSem);

        return TRUE;
    }
    //  If the data was already there then report that as well
    else if (eReturnCode == OSAL_QUEUE_MESSAGE_ALREADY_QUEUED)
    {
        *pbAlreadyPresent = TRUE;
    }

    return FALSE;
}

/*****************************************************************************
*
*   vIterateDualQueue
*
* This is an interface function which iterates the two queues
* used by this interface
*
*****************************************************************************/
static void vIterateDualQueue (
    SMS_OBJECT hQueueData,
    SMSE_ITERATE_QUEUE_CALLBACK bCallback,
    void *pvCallbackArg
        )
{
    BOOLEAN bValid;

    // Validate function pointer
    if ((SMSE_ITERATE_QUEUE_CALLBACK)NULL == bCallback)
    {
        return;
    }

    bValid = SMSO_bValid(hQueueData);
    if (TRUE == bValid)
    {
        BOOLEAN bContinue;
        SMS_EVENT_DUAL_QUEUE_STRUCT *psDual =
            (SMS_EVENT_DUAL_QUEUE_STRUCT *)hQueueData;

        // Provide them with the Main Queue (index 0)
        bContinue = bCallback(psDual->hEventQueue1,
            "Main Queue", 0, pvCallbackArg);

        if (TRUE == bContinue)
        {
            // Provide them with the Secondary Queue (index 1)
            bContinue = bCallback(psDual->hEventQueue2,
                "Secondary Queue", 1, pvCallbackArg);
        }
    }

    return;
}
