/******************************************************************************/
/*                    Copyright (c) Sirius XM Radio, Inc.                     */
/*                            All Rights Reserved                             */
/*         Licensed Materials - Property of Sirius XM Radio, Inc.             */
/******************************************************************************/
/*******************************************************************************
 *
 * DESCRIPTION
 *
 *  This module contains the Object:SKI_ATTRIBUTES implementation for the
 *  Sirius Module Services (SMS)
 *
 ******************************************************************************/
#include "standard.h"
#include "osal.h"

#include "sms_api.h"
#include "sms_obj.h"
#include "_ski_attributes_obj.h"
#include "ski_attributes_obj.h"

#include "sms_api_debug.h"

/*****************************************************************************
                             PUBLIC FUNCTIONS
*****************************************************************************/

/*****************************************************************************
*
*   eNumOfLifts
*
*****************************************************************************/
static SMSAPI_RETURN_CODE_ENUM eNumOfLifts (
    SKI_ATTRIBUTES_OBJECT hSkiAttrs,
    UN8 *pun8NumOfLifts
        )
{
    BOOLEAN bOwner = FALSE;
    SMSAPI_RETURN_CODE_ENUM eResult = SMSAPI_RETURN_CODE_ERROR;
    SKI_ATTRIBUTES_OBJECT_STRUCT *psObj =
        (SKI_ATTRIBUTES_OBJECT_STRUCT *)hSkiAttrs;

    do
    {
        if (pun8NumOfLifts == NULL)
        {
            eResult = SMSAPI_RETURN_CODE_INVALID_INPUT;
            break;
        }

        bOwner = SMSO_bOwner((SMS_OBJECT)hSkiAttrs);
        if (bOwner == FALSE)
        {
            eResult = SMSAPI_RETURN_CODE_NOT_OWNER;
            break;
        }

        if (psObj->bLiftsAvailable == TRUE)
        {
            eResult = SMSAPI_RETURN_CODE_SUCCESS;
            *pun8NumOfLifts = psObj->un8Lifts;
        }
        else
        {
            eResult = SMSAPI_RETURN_CODE_NOT_FOUND;
        }
    } while (FALSE);

    return eResult;
}

/*****************************************************************************
*
*   eNumOfTrailsRange
*
*****************************************************************************/
static SMSAPI_RETURN_CODE_ENUM eNumOfTrailsRange (
    SKI_ATTRIBUTES_OBJECT hSkiAttrs,
    UN8 *pun8NumOfTrailsRangeLow,
    UN8 *pun8NumOfTrailsRangeHigh
        )
{
    BOOLEAN bOwner = FALSE;
    SMSAPI_RETURN_CODE_ENUM eResult = SMSAPI_RETURN_CODE_ERROR;
    SKI_ATTRIBUTES_OBJECT_STRUCT *psObj = (SKI_ATTRIBUTES_OBJECT_STRUCT *)hSkiAttrs;

    do
    {
        if ((pun8NumOfTrailsRangeLow == NULL) || (pun8NumOfTrailsRangeHigh == NULL))
        {
            eResult = SMSAPI_RETURN_CODE_INVALID_INPUT;
            break;
        }

        bOwner = SMSO_bOwner((SMS_OBJECT)hSkiAttrs);
        if (bOwner == FALSE)
        {
            eResult = SMSAPI_RETURN_CODE_NOT_OWNER;
            break;
        }

        if (psObj->bTrailsAvailable == TRUE)
        {
            eResult = SMSAPI_RETURN_CODE_SUCCESS;
            *pun8NumOfTrailsRangeLow = psObj->sTrailData.un8Low;
            *pun8NumOfTrailsRangeHigh = psObj->sTrailData.un8High;
        }
        else
        {
            eResult = SMSAPI_RETURN_CODE_NOT_FOUND;
        }
    } while (FALSE);

    return eResult;
}

/*****************************************************************************
*
*   eAmenities
*
*****************************************************************************/
SMSAPI_RETURN_CODE_ENUM eAmenities (
    SKI_ATTRIBUTES_OBJECT hSkiAttrs,
    size_t tNumAmenities,
    AMENITY_STRUCT *pasSkiAmenities
      )
{
    BOOLEAN bOwner;

    if (tNumAmenities < SKI_NUM_AMENITIES)
    {
        return SMSAPI_RETURN_CODE_INVALID_INPUT;
    }
    else if (pasSkiAmenities == NULL)
    {
        return SMSAPI_RETURN_CODE_INVALID_INPUT;
    }

    // Check ownership of the object
    bOwner = SMSO_bOwner((SMS_OBJECT)hSkiAttrs);
    if (bOwner == FALSE)
    {
        return SMSAPI_RETURN_CODE_NOT_OWNER;
    }
    else
    {
        SKI_ATTRIBUTES_OBJECT_STRUCT *psObj =
            (SKI_ATTRIBUTES_OBJECT_STRUCT *)hSkiAttrs;
        UN8 un8AmenityOffset = 0;
        SMSAPI_RETURN_CODE_ENUM eReturnCode =
            SMSAPI_RETURN_CODE_SUCCESS;
        SKI_AMENITY_ENUM eCurAmenity = SKI_ATTRIBUTES_BASE_AMEN;

        do
        {
            switch (eCurAmenity)
            {
                case SKI_AMENITY_SKI_LIFTS:
                {
                    pasSkiAmenities[un8AmenityOffset].uAmenity.eSkiAmenity =
                        SKI_AMENITY_SKI_LIFTS;

                    if (psObj->bLiftsAvailable == TRUE)
                    {
                        pasSkiAmenities[un8AmenityOffset].eStatus = AMENITY_STATUS_AVAILABLE;
                    }
                    else
                    {
                        pasSkiAmenities[un8AmenityOffset].eStatus = AMENITY_STATUS_UNAVAILABLE;
                    }

                    un8AmenityOffset++;
                }
                break;

                case SKI_AMENITY_SKI_TRAILS:
                {
                   pasSkiAmenities[un8AmenityOffset].uAmenity.eSkiAmenity =
                       SKI_AMENITY_SKI_TRAILS;

                   if (psObj->bTrailsAvailable == TRUE)
                   {
                       pasSkiAmenities[un8AmenityOffset].eStatus = AMENITY_STATUS_AVAILABLE;
                   }
                   else
                   {
                       pasSkiAmenities[un8AmenityOffset].eStatus = AMENITY_STATUS_UNAVAILABLE;
                   }

                   un8AmenityOffset++;
                }
                break;

                case SKI_AMENITY_SNOW_MADE_LAST_24HRS:
                {
                    if ((psObj->eMake == WEATHER_FLAG_TRUE) ||
                        (psObj->eMake == WEATHER_FLAG_FALSE))
                    {
                        pasSkiAmenities[un8AmenityOffset].uAmenity.eSkiAmenity =
                            SKI_AMENITY_SNOW_MADE_LAST_24HRS;

                        if (psObj->eMake == WEATHER_FLAG_TRUE)
                        {
                            pasSkiAmenities[un8AmenityOffset].eStatus = AMENITY_STATUS_AVAILABLE;
                        }
                        else
                        {
                            pasSkiAmenities[un8AmenityOffset].eStatus = AMENITY_STATUS_UNAVAILABLE;
                        }

                        un8AmenityOffset++;
                    }
                }
                break;

                case SKI_AMENITY_SNOW_GROOMED_LAST_24HRS:
                {
                    if ((psObj->eGroom == WEATHER_FLAG_TRUE) ||
                        (psObj->eGroom == WEATHER_FLAG_FALSE))
                    {
                        pasSkiAmenities[un8AmenityOffset].uAmenity.eSkiAmenity =
                            SKI_AMENITY_SNOW_GROOMED_LAST_24HRS;

                        if (psObj->eGroom == WEATHER_FLAG_TRUE)
                        {
                            pasSkiAmenities[un8AmenityOffset].eStatus = AMENITY_STATUS_AVAILABLE;
                        }
                        else
                        {
                            pasSkiAmenities[un8AmenityOffset].eStatus = AMENITY_STATUS_UNAVAILABLE;
                        }

                        un8AmenityOffset++;
                    }
                }
                break;

                case SKI_AMENITY_NIGHT_SKIING_PERMITTED:
                {
                    if ((psObj->eNight == WEATHER_FLAG_TRUE) ||
                        (psObj->eNight == WEATHER_FLAG_FALSE))
                    {
                        pasSkiAmenities[un8AmenityOffset].uAmenity.eSkiAmenity =
                            SKI_AMENITY_NIGHT_SKIING_PERMITTED;

                        if (psObj->eNight == WEATHER_FLAG_TRUE)
                        {
                            pasSkiAmenities[un8AmenityOffset].eStatus = AMENITY_STATUS_AVAILABLE;
                        }
                        else
                        {
                            pasSkiAmenities[un8AmenityOffset].eStatus = AMENITY_STATUS_UNAVAILABLE;
                        }

                        un8AmenityOffset++;
                    }
                }
                break;

                case SKI_AMENITY_SNOWBOARDING:
                {
                    if ((psObj->eBoard == WEATHER_FLAG_TRUE) ||
                        (psObj->eBoard == WEATHER_FLAG_FALSE))
                    {
                        pasSkiAmenities[un8AmenityOffset].uAmenity.eSkiAmenity =
                            SKI_AMENITY_SNOWBOARDING;

                        if (psObj->eBoard == WEATHER_FLAG_TRUE)
                        {
                            pasSkiAmenities[un8AmenityOffset].eStatus = AMENITY_STATUS_AVAILABLE;
                        }
                        else
                        {
                            pasSkiAmenities[un8AmenityOffset].eStatus = AMENITY_STATUS_UNAVAILABLE;
                        }

                        un8AmenityOffset++;
                    }
                }
                break;

                default:
                {
                    // We can't identify these amenities
                    eReturnCode = SMSAPI_RETURN_CODE_ERROR;
                    break;
                }
            }
        } while (++eCurAmenity <= SKI_ATTRIBUTES_MAX_AMEN);

        if ((eReturnCode == SMSAPI_RETURN_CODE_SUCCESS) &&
            (un8AmenityOffset == 0))
        {
            // We didn't find any amenities, let the caller know
            eReturnCode = SMSAPI_RETURN_CODE_NO_OBJECTS;
        }

        return eReturnCode;
    }
}

/*****************************************************************************
*
*   n32FPrintf
*
*****************************************************************************/
static N32 n32FPrintf (
    SKI_ATTRIBUTES_OBJECT hSkiAttrs,
    FILE *psFile
        )
{
    N32 n32Return = 0;
    SKI_ATTRIBUTES_OBJECT_STRUCT *psObj = (SKI_ATTRIBUTES_OBJECT_STRUCT *)hSkiAttrs;
    BOOLEAN bOwner;

    // Determine if the handle is valid
    bOwner = SMSO_bOwner((SMS_OBJECT)hSkiAttrs);
    if ((bOwner == FALSE) || (psFile == NULL))
    {
        return EOF;
    }

    n32Return += fprintf(psFile, "\nSki Attributes (Object:0x%X):\n",
                         psObj);

    if (psObj->bLiftsAvailable == TRUE)
    {
        n32Return += fprintf(psFile, "\tNumber of Lifts: %u\n", psObj->un8Lifts);
    }

    if (psObj->bTrailsAvailable == TRUE)
    {
        n32Return += fprintf(psFile, "\tNumber of Trails: From %u To %u\n",
                                psObj->sTrailData.un8Low,
                             psObj->sTrailData.un8High);
    }

    if (psObj->eBoard != WEATHER_FLAG_NOT_AVAILABLE)
    {
        n32Return += fprintf(psFile,
            "\tSnowboarding Available: %s\n",
            (psObj->eBoard == WEATHER_FLAG_TRUE)?"Yes":"No");
    }

    if (psObj->eGroom != WEATHER_FLAG_NOT_AVAILABLE)
    {
        n32Return += fprintf(psFile,
            "\tGroomed in past 24 hours: %s\n",
               (psObj->eGroom == WEATHER_FLAG_TRUE)?"Yes":"No");
    }

    if (psObj->eMake != WEATHER_FLAG_NOT_AVAILABLE)
    {
        n32Return += fprintf(psFile, "\tSnow made in past 24 hours: %s\n",
               (psObj->eMake == WEATHER_FLAG_TRUE)?"Yes":"No");
    }

    if (psObj->eNight != WEATHER_FLAG_NOT_AVAILABLE)
    {
        n32Return += fprintf(psFile, "\tNight skiing available: %s\n",
               (psObj->eNight == WEATHER_FLAG_TRUE)?"Yes":"No");
    }

    return n32Return;
}

/*****************************************************************************
                             FRIEND FUNCTIONS
*****************************************************************************/

/*****************************************************************************
*
*   SKI_ATTRIBUTES_hCreate
*
*****************************************************************************/
SKI_ATTRIBUTES_OBJECT SKI_ATTRIBUTES_hCreate(
    SMS_OBJECT hParent,
    WEATHER_FLAG_ENUM eBoard,
    WEATHER_FLAG_ENUM eGroom,
    WEATHER_FLAG_ENUM eMake,
    WEATHER_FLAG_ENUM eNight,
    BOOLEAN bLiftsAvailable,
    UN8 un8Lift,
    BOOLEAN bTrailsAvailable,
    TRAILS_UNPARSED_DATA tTrailData
        )
{
    SKI_ATTRIBUTES_OBJECT_STRUCT *psObj;

    // Create an instance of the HUMIDITY object
    psObj = (SKI_ATTRIBUTES_OBJECT_STRUCT *)
        SMSO_hCreate(
            SKI_ATTRIBUTES_OBJECT_NAME,
            sizeof(SKI_ATTRIBUTES_OBJECT_STRUCT),
            hParent,
            FALSE
                );

    if(psObj == NULL)
    {
        // Error!
        return SKI_ATTRIBUTES_INVALID_OBJECT;
    }

    // Initialize object per inputs
    psObj->eBoard = eBoard;
    psObj->eGroom = eGroom;
    psObj->eMake = eMake;
    psObj->eNight = eNight;
    psObj->bTrailsAvailable = bTrailsAvailable;
    psObj->bLiftsAvailable = bLiftsAvailable;
    psObj->un8Lifts = un8Lift;

    if (bTrailsAvailable == TRUE)
    {
        psObj->bTrailsAvailable = GsWeatherIntf.bProcessTrailsData(tTrailData, &psObj->sTrailData);
    }

    return (SKI_ATTRIBUTES_OBJECT)psObj;
}

/*****************************************************************************
*
*   SKI_ATTRIBUTES_vDestroy
*
*****************************************************************************/
void SKI_ATTRIBUTES_vDestroy (
    SKI_ATTRIBUTES_OBJECT hSkiAttrs
        )
{
    BOOLEAN bOwner;

    bOwner = SMSO_bOwner((SMS_OBJECT)hSkiAttrs);
    if(bOwner == TRUE)
    {
        // Free object instance
        SMSO_vDestroy((SMS_OBJECT)hSkiAttrs);
    }
    return;
}

/*****************************************************************************
                             PRIVATE FUNCTIONS
*****************************************************************************/
