/******************************************************************************/
/*                    Copyright (c) Sirius XM Radio, Inc.                     */
/*                            All Rights Reserved                             */
/*         Licensed Materials - Property of Sirius XM Radio, Inc.             */
/******************************************************************************/
/*******************************************************************************
 *
 * DESCRIPTION
 *
 *  This module contains the Object:SHAPE implementation for the
 *  Simple Module Services (SMS)
 *
 ******************************************************************************/

#include "standard.h"
#include "osal.h"

#include "sms_api.h"
#include "sms_obj.h"
#include "sms.h"

#include "location_obj.h"
#include "shape_obj.h"
#include "_shape_obj.h"

#include "sms_api_debug.h"
static const char *gpacThisFile = __FILE__;

/*****************************************************************************
 *                             PUBLIC FUNCTIONS
 *****************************************************************************/
/*****************************************************************************
*
*   eShapeType
*
*****************************************************************************/
static SHAPE_TYPE_ENUM eShapeType (
    SHAPE_OBJECT hShape
        )
{
    SHAPE_TYPE_ENUM eResult = SHAPE_TYPE_INVALID;
    BOOLEAN bOwner;

    // Verify ownership of this object
    bOwner = SMSO_bOwner((SMS_OBJECT)hShape);
    if (bOwner == TRUE)
    {
        SHAPE_OBJECT_STRUCT *psObj =
                (SHAPE_OBJECT_STRUCT*) hShape;
        if (hShape != SHAPE_INVALID_OBJECT)
        {
            eResult = psObj->eType;
        }
    }

    return eResult;
}

/*****************************************************************************
*
*   eIteratePoints
*
*****************************************************************************/
static SMSAPI_RETURN_CODE_ENUM eIteratePoints (
    SHAPE_OBJECT hShape,
    SHAPE_POINTS_ITERATOR_CALLBACK bCallback,
    void *pvCallbackArg
        )
{
    SMSAPI_RETURN_CODE_ENUM eReturnCode = SMSAPI_RETURN_CODE_ERROR;
    BOOLEAN bOwner = FALSE, bOk = FALSE;
    SHAPE_OBJECT_STRUCT *psObj = (SHAPE_OBJECT_STRUCT*) hShape;
    GEO_POINT_STRUCT *psPoint = NULL;
    UN16 un16Point;

    do
    {
        // Check inputs
        if ((hShape == SHAPE_INVALID_OBJECT) || (bCallback == NULL))
        {
            eReturnCode = SMSAPI_RETURN_CODE_INVALID_INPUT;
            break;
        }

        bOwner = SMSO_bOwner((SMS_OBJECT)hShape);
        if (bOwner != TRUE)
        {
            eReturnCode = SMSAPI_RETURN_CODE_NOT_OWNER;
            break;
        }

        if (psObj->un16Points > 0)
        {
            LOCATION_OBJECT hLocationObj = LOCATION_INVALID_OBJECT;

            psPoint = psObj->pasPoints;
            bOk = TRUE;
            eReturnCode = SMSAPI_RETURN_CODE_SUCCESS;

            for (un16Point = 0;
                 (un16Point < psObj->un16Points) && (bOk == TRUE);
                 ++un16Point, ++psPoint)
            {
                if ((psPoint->hLat != OSAL_FIXED_INVALID_OBJECT) &&
                    (psPoint->hLon != OSAL_FIXED_INVALID_OBJECT))
                {
                    if(hLocationObj == LOCATION_INVALID_OBJECT)
                    {
                        hLocationObj = LOCATION.hCreateForRadius(
                                    psPoint->hLat, psPoint->hLon,
                                    DISTANCE_INVALID_OBJECT
                                        );
                        if (hLocationObj == LOCATION_INVALID_OBJECT)
                        {
                            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                                SHAPE_OBJECT_NAME": failed to create location"
                                    );
                            eReturnCode = SMSAPI_RETURN_CODE_OUT_OF_MEMORY;
                            break;
                        }
                    }
                    else
                    {
                        bOk = LOCATION_bUpdateCoordinatesByFixed(
                                    hLocationObj, psPoint->hLat, psPoint->hLon
                                        );
                        if (bOk == FALSE)
                        {
                            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                                SHAPE_OBJECT_NAME": failed to update location");
                            eReturnCode = SMSAPI_RETURN_CODE_ERROR;
                            break;
                        }
                    }

                    // Do call
                    bOk = bCallback(hLocationObj, pvCallbackArg);
                }
                else
                {
                    SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                            SHAPE_OBJECT_NAME": incorrect GEO point"
                                );
                    eReturnCode = SMSAPI_RETURN_CODE_ERROR;
                }
            }

            if (hLocationObj != LOCATION_INVALID_OBJECT)
            {
                LOCATION.vDestroy(hLocationObj);
            }
        }

    } while (FALSE);

    return eReturnCode;
}

/*****************************************************************************
*
*   n32FPrintf
*
*****************************************************************************/
static N32 n32FPrintf (
    SHAPE_OBJECT hShape,
    FILE *psFile,
    SMSAPI_OUTPUT_OPTION_ENUM eOutputOption
        )
{
    BOOLEAN bOwner;
    N32 n32Return = 0;
    SHAPE_OBJECT_STRUCT *psObj = (SHAPE_OBJECT_STRUCT *)NULL;
    GEO_POINT_STRUCT *psPoint = (GEO_POINT_STRUCT *)NULL;
    UN16 un16Point = 0;

    bOwner = SMSO_bOwner((SMS_OBJECT)hShape);
    if ((bOwner == FALSE) || (psFile == NULL) ||
          ((eOutputOption != SMS_OUTPUT_OPTION_TERSE) &&
           (eOutputOption != SMS_OUTPUT_OPTION_VERBOSE) &&
           (eOutputOption != SMS_OUTPUT_OPTION_GROSS)))
    {
        return EOF;
    }

    psObj = (SHAPE_OBJECT_STRUCT*) hShape;

    psPoint = psObj->pasPoints;

    if ((eOutputOption == SMS_OUTPUT_OPTION_TERSE) ||
        (eOutputOption == SMS_OUTPUT_OPTION_VERBOSE) )
    {
        n32Return += fprintf(psFile, "%d points total\n", psObj->un16Points);
    }
    else
    {
        for (un16Point = 0; un16Point < psObj->un16Points;
             ++un16Point, ++psPoint)
        {
            n32Return += fprintf(psFile, "#%03d: ", un16Point);
            n32Return += n32GeoPointFPrintf(psPoint,
                                           psFile, eOutputOption
                                               );
        }
    }

    return n32Return;
}

/*****************************************************************************
 *                              FRIEND FUNCTIONS
 *****************************************************************************/
/*****************************************************************************
 *
 *   SHAPE_hCreate
 *
 *****************************************************************************/
SHAPE_OBJECT SHAPE_hCreate(
    SMS_OBJECT hParent
        )
{
    SHAPE_OBJECT_STRUCT *psObj =
            (SHAPE_OBJECT_STRUCT*) SHAPE_INVALID_OBJECT;

    do
    {
        // Check inputs
        if (hParent == SMS_INVALID_OBJECT)
        {
            break;
        }

        // Create an instance of the AGW Msg object
        psObj = (SHAPE_OBJECT_STRUCT *)
            SMSO_hCreate(
                SHAPE_OBJECT_NAME": Obj",
                sizeof(SHAPE_OBJECT_STRUCT),
                hParent,
                FALSE);

        if (psObj == (SHAPE_OBJECT_STRUCT*)DSRL_ENTRY_INVALID_OBJECT)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    SHAPE_OBJECT_NAME": failed to create object\n"
                        );
            break;
        }

        // Init members
        OSAL.bMemSet(psObj, 0, sizeof(SHAPE_OBJECT_STRUCT));
        psObj->eType = SHAPE_TYPE_CLOSED;
        psObj->un16Points = 0;
        psObj->pasPoints = (GEO_POINT_STRUCT*)NULL;

    } while (FALSE);

    return (SHAPE_OBJECT)psObj;
}

/*****************************************************************************
 *
 *   SHAPE_vDestroy
 *
 *****************************************************************************/
void SHAPE_vDestroy(
   SHAPE_OBJECT hShape
        )
{
    BOOLEAN bOwner;
    SHAPE_OBJECT_STRUCT *psObj = (SHAPE_OBJECT_STRUCT*)hShape;

    printf(SHAPE_OBJECT_NAME
            ": attempting to destroy object 0x%p\n",
            hShape);

    // Verify ownership of this object
    bOwner = SMSO_bOwner((SMS_OBJECT)hShape);
    if (bOwner == FALSE)
    {
        return;
    }

    vPointsArrayDestroy(psObj);

    SMSO_vDestroy((SMS_OBJECT)hShape);

    return;
}

/*****************************************************************************
 *
 *   SHAPE_bPreparePoints
 *
 *   Prepares object for collecting points
 *
 *****************************************************************************/
BOOLEAN SHAPE_bPreparePoints(
    SHAPE_OBJECT hShape,
    UN16 un16Points,
    SHAPE_TYPE_ENUM eType
        )
{
    BOOLEAN bOk = FALSE, bOwner = FALSE;
    SHAPE_OBJECT_STRUCT *psObj = (SHAPE_OBJECT_STRUCT*)hShape;

    do
    {
        // Check input
        if (hShape == SHAPE_INVALID_OBJECT)
        {
            break;
        }

        // Verify ownership of this object
        bOwner = SMSO_bOwner((SMS_OBJECT)hShape);
        if (bOwner == FALSE)
        {
            break;
        }

        vPointsArrayDestroy(psObj);
        psObj->eType = eType;
        psObj->un16Points = un16Points;
        bOk = bPointsArrayCreate(psObj);
        if (bOk == FALSE)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    SHAPE_OBJECT_NAME
                    ": failed to create GEO-points array"
                        );
        }

    } while (FALSE);

    return bOk;
}

/*****************************************************************************
 *
 *   SHAPE_bSetPoint
 *
 *****************************************************************************/
BOOLEAN SHAPE_bSetPoint(
    SHAPE_OBJECT hShape,
    UN16 un16Index,
    N32 n32Lon,
    N32 n32Lat,
    UN8 un8FracBits
        )
{
    BOOLEAN bOk = FALSE, bOwner = FALSE;
    SHAPE_OBJECT_STRUCT *psObj = (SHAPE_OBJECT_STRUCT*)hShape;
    GEO_POINT_STRUCT *psGeoPoint = NULL;

    do
    {
        // Check inputs
        if ((hShape == SHAPE_INVALID_OBJECT) ||
            (un16Index >= psObj->un16Points) || (psObj->pasPoints == NULL))
        {
            break;
        }

        // Verify ownership of this object
        bOwner = SMSO_bOwner((SMS_OBJECT)hShape);
        if (bOwner == FALSE)
        {
            break;
        }

        // Check for point existence
        psGeoPoint = &psObj->pasPoints[un16Index];
        if ((psGeoPoint->hLat != OSAL_FIXED_INVALID_OBJECT) ||
            (psGeoPoint->hLon != OSAL_FIXED_INVALID_OBJECT))
        {
            printf(SHAPE_OBJECT_NAME": point for index %d already set",
                    un16Index);
            break;
        }

        // Create GEO-point in the memory
        psGeoPoint = psGeoPointCreateInMemory(n32Lat, n32Lon,
                         psGeoPoint, un8FracBits
                             );
        if (psGeoPoint == NULL)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    SHAPE_OBJECT_NAME
                    ": failed to create GEO-point in memory"
                        );
            break;
        }

        bOk = TRUE;
    } while (FALSE);

    return bOk;
}

/*****************************************************************************
 *
 *   SHAPE_eIterateLines
 *
 *****************************************************************************/
SMSAPI_RETURN_CODE_ENUM SHAPE_eIterateLines (
    SHAPE_OBJECT hShape,
    SHAPE_LINES_ITERATOR_CALLBACK bCallback,
    void *pvCallbackArg
        )
{
    SMSAPI_RETURN_CODE_ENUM eReturnCode = SMSAPI_RETURN_CODE_ERROR;
    BOOLEAN bOwner = FALSE, bOk = FALSE;
    SHAPE_OBJECT_STRUCT *psObj = (SHAPE_OBJECT_STRUCT*) hShape;
    UN16 un16Point;

    do
    {
        // Check inputs
        if ((hShape == SHAPE_INVALID_OBJECT) || (bCallback == NULL))
        {
            eReturnCode = SMSAPI_RETURN_CODE_INVALID_INPUT;
            break;
        }

        bOwner = SMSO_bOwner((SMS_OBJECT)hShape);
        if (bOwner != TRUE)
        {
            eReturnCode = SMSAPI_RETURN_CODE_NOT_OWNER;
            break;
        }

        eReturnCode = SMSAPI_RETURN_CODE_SUCCESS;
        if (psObj->un16Points > 0)
        {
            GEO_POINT_STRUCT *psPoint1 = NULL;
            GEO_POINT_STRUCT *psPoint2 = NULL;
            psPoint1 = psObj->pasPoints;
            psPoint2 = psObj->pasPoints + 1;
            bOk = TRUE;

            for (un16Point = 0;
                 (un16Point < psObj->un16Points - 1) && (bOk == TRUE);
                 ++un16Point, ++psPoint1, ++psPoint2)
            {
                if ((psPoint1->hLat == OSAL_FIXED_INVALID_OBJECT) ||
                    (psPoint1->hLon == OSAL_FIXED_INVALID_OBJECT) ||
                    (psPoint2->hLat == OSAL_FIXED_INVALID_OBJECT) ||
                    (psPoint2->hLon == OSAL_FIXED_INVALID_OBJECT))
                {
                    break;
                }
                else
                {
                    // Do call
                    bOk = bCallback(
                            psPoint1->hLat,
                            psPoint1->hLon,
                            psPoint2->hLat,
                            psPoint2->hLon,
                            pvCallbackArg
                                );
                }
            }
        }

    } while (FALSE);

    return eReturnCode;
}

/*****************************************************************************
 *                              PRIVATE FUNCTIONS
 *****************************************************************************/

/*****************************************************************************
 *
 *   bPointsArrayCreate
 *
 *****************************************************************************/
static BOOLEAN bPointsArrayCreate (
    SHAPE_OBJECT_STRUCT *psObj
        )
{
    BOOLEAN bOk = FALSE;

    do
    {
        // Check input
        if ((psObj == NULL) || (psObj->pasPoints != NULL))
        {
            break;
        }

        if (psObj->un16Points > 0)
        {
            psObj->pasPoints = (GEO_POINT_STRUCT*)SMSO_hCreate(
                                    SHAPE_OBJECT_NAME": GeoPointsArray",
                                    sizeof(GEO_POINT_STRUCT) * psObj->un16Points,
                                    (SMS_OBJECT)psObj,
                                    FALSE
                                        );
            if (psObj->pasPoints == NULL)
            {
                SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    SHAPE_OBJECT_NAME
                    ": failed to create GEO-point array"
                        );
                break;
            }

            OSAL.bMemSet(psObj->pasPoints, 0,
                sizeof(GEO_POINT_STRUCT) * psObj->un16Points
                    );
        }
        bOk = TRUE;

    } while (FALSE);

    return bOk;
}


/*****************************************************************************
 *
 *   vPointsArrayDestroy
 *
 *****************************************************************************/
static void vPointsArrayDestroy (
    SHAPE_OBJECT_STRUCT *psObj
        )
{
    if ((psObj != NULL) && (psObj->pasPoints != NULL))
    {
        SMSO_vDestroy((SMS_OBJECT)psObj->pasPoints);
        psObj->pasPoints = NULL;
    }

    return;
}


/*****************************************************************************
*
*   psGeoPointCreateInMemory
*
*****************************************************************************/
static GEO_POINT_STRUCT *psGeoPointCreateInMemory (
    N32 n32Lat,
    N32 n32Lon,
    GEO_POINT_STRUCT *psGeoPoint,
    UN8 un8FracBits
        )
{
    BOOLEAN bOk = FALSE;

    do
    {
        // Check inputs
        if (psGeoPoint == NULL)
        {
            break;
        }

        psGeoPoint->hLat =
            OSAL_FIXED.hCreateInMemory(n32Lat, un8FracBits, psGeoPoint->aLatData);
        if (psGeoPoint->hLat == OSAL_FIXED_INVALID_OBJECT)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                SHAPE_OBJECT_NAME": failed to OSAL_FIXED for Lat location"
                    );
            break;
        }

        psGeoPoint->hLon =
            OSAL_FIXED.hCreateInMemory(n32Lon, un8FracBits, psGeoPoint->aLonData);
        if (psGeoPoint->hLon == OSAL_FIXED_INVALID_OBJECT)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                SHAPE_OBJECT_NAME": failed to OSAL_FIXED for Lon location"
                    );
            break;
        }

        bOk = TRUE;
    } while (FALSE);

    return (bOk == FALSE) ? NULL : psGeoPoint;
}


/*****************************************************************************
*
*   n32GeoPointFPrintf
*
*****************************************************************************/
static N32 n32GeoPointFPrintf (
    GEO_POINT_STRUCT *psObj,
    FILE *psFile,
    SMSAPI_OUTPUT_OPTION_ENUM eOutput
        )
{
    N32 n32Result = 0;
    BOOLEAN bIsTerse;

    if ( eOutput == SMS_OUTPUT_OPTION_TERSE )
    {
        bIsTerse = TRUE;
    }
    else
    {
        bIsTerse = FALSE;
    }

    n32Result += fprintf(psFile, SHAPE_OBJECT_NAME"_GEO_POINT:\nLAT: ");
    if (psObj->hLat != OSAL_FIXED_INVALID_OBJECT)
    {
        n32Result += OSAL_FIXED.n32FPrintf(psObj->hLat, psFile, bIsTerse);
    }
    else
    {
        n32Result += fprintf(psFile, " INVALID\n");
    }

    n32Result += fprintf(psFile, "LON: ");
    if (psObj->hLon != OSAL_FIXED_INVALID_OBJECT)
    {
        n32Result += OSAL_FIXED.n32FPrintf(psObj->hLon, psFile, bIsTerse);
    }
    else
    {
        n32Result += fprintf(psFile, " INVALID\n");
    }

    return n32Result;
}

