/******************************************************************************/
/*                    Copyright (c) Sirius XM Radio, Inc.                     */
/*                            All Rights Reserved                             */
/*         Licensed Materials - Property of Sirius XM Radio, Inc.             */
/******************************************************************************/
/*******************************************************************************
 *
 * DESCRIPTION
 *
 *  This module contains the Object:SEEK_CONTENT implementation for the
 *  Sirius Module Services (SMS)
 *
 ******************************************************************************/

#include "standard.h"
#include "osal.h"

#include "sms_version.h"
#include "sms_api.h"
#include "sms_obj.h"

#include "string_obj.h"

#include "cm.h"
#include "seek.h"
#include "at_seek_content.h"
#include "cal_content.h"
#include "seek_content.h"
#include "_seek_content.h"

/*****************************************************************************
                             PUBLIC FUNCTIONS
*****************************************************************************/

/*****************************************************************************
*
*   eService
*
* This API is used to retrieve the type of Seek Service this Seek Content
* belongs to.
*
* Inputs:
*
*   hSeekContent    A handle to a valid Seek Content for which the
*                   caller wishes to query the service to which it belongs
*
* Outputs:
*
*   SEEK_SERVICE_ENUM indicating the type of Seek Service this Seek Content
*   belongs to
*
*****************************************************************************/
static SEEK_SERVICE_ENUM eService (
    SEEK_CONTENT_OBJECT hSeekContent
        )
{
    BOOLEAN bOwner;
    SEEK_CONTENT_REGISTERED_ITEM_STRUCT *psRegisteredItem;

    // verify input
    bOwner =
        SMSO_bOwner((SMS_OBJECT)hSeekContent);
    if(bOwner == FALSE)
    {
        // not valid content
        return SEEK_SERVICE_INVALID;
    }

    // get the registered item structure from this Seek Content
    psRegisteredItem = SEEK_CONTENT_psRegisteredItem(hSeekContent);


    if (psRegisteredItem == NULL)
    {
        // something went wrong
        return SEEK_SERVICE_INVALID;
    }

    // ask the Seek service what type it is and return the result
    return SEEK_eService(psRegisteredItem->hSeekService);
}

/*****************************************************************************
*
*   eRemove
*
*
*   This API is used to remove a Seek Content obj from its Seek Service.
*
*   Inputs:
*
*   hSeekContent - A handle to a valid seek content.
*
*   Outputs:
*
*   SMSAPI_RETURN_CODE_SUCCESS on success, SMSAPI_RETURN_CODE_ERROR on error
*
*****************************************************************************/
static SMSAPI_RETURN_CODE_ENUM eRemove (
    SEEK_CONTENT_OBJECT hSeekContent
        )
{
    BOOLEAN bOwner;
    SMSAPI_RETURN_CODE_ENUM eReturn;
    SEEK_CONTENT_REGISTERED_ITEM_STRUCT *psRegisteredItem;

    // validate input
    bOwner =
        SMSO_bOwner((SMS_OBJECT)hSeekContent);
    if(bOwner == FALSE)
    {
        // invalid obj
        return SMSAPI_RETURN_CODE_API_NOT_ALLOWED;
    }

    // extract from the CAL_CONTENT object the registered item
    psRegisteredItem = SEEK_CONTENT_psRegisteredItem(hSeekContent);

	TAG_eRemove(psRegisteredItem->hTag, TRUE);

    // remove the currently iterated item from the CAL
    // since a SEEK_CONTENT is really a CAL_CONTENT we can use the
    // CAL_CONTENT's API
    eReturn = CAL_CONTENT.eRemove((CAL_CONTENT_OBJECT)hSeekContent);

    return eReturn;
}

/*****************************************************************************
*
*   eEnableItem
*
*   This API is used to enable the currently iterated item from a seek service.
*   This API may only be called from with a seek iterator callback function
*
*   Inputs:
*
*   hSeekContent - A handle to a valid seek content.
*
*   Outputs:
*
*   SMSAPI_RETURN_CODE_SUCCESS on success, SMSAPI_RETURN_CODE_ERROR on error
*
*****************************************************************************/
static SMSAPI_RETURN_CODE_ENUM eEnable (
    SEEK_CONTENT_OBJECT hSeekContent
        )
{
    BOOLEAN bOwner;
    SEEK_CONTENT_REGISTERED_ITEM_STRUCT *psRegisteredItem;
    SMSAPI_RETURN_CODE_ENUM eReturn = SMSAPI_RETURN_CODE_ERROR;

    // validate input
    bOwner =
        SMSO_bOwner((SMS_OBJECT)hSeekContent);
    if(bOwner == FALSE)
    {
        // Not owner of the provided object
        return SMSAPI_RETURN_CODE_API_NOT_ALLOWED;
    }

    // extract the registered item information
    psRegisteredItem = SEEK_CONTENT_psRegisteredItem(hSeekContent);

    if (psRegisteredItem != NULL)
    {
        if (psRegisteredItem->bEnabled == TRUE)
        {
            // this item is already enabled. no further work needs to be done
            eReturn = SMSAPI_RETURN_CODE_SUCCESS;
        }
        else
        {

        // enable this item
        psRegisteredItem->bEnabled = TRUE;

        // update the tag
        eReturn = SEEK_CONTENT_eUpdateEnabledTag(psRegisteredItem, TRUE);

        // see if the specific service has a function to determine if currently
        // occurring sontent should be searched for matches
        if (psRegisteredItem->psInterface->bSearchForCurrentAfterEnable != NULL)
        {
            // there is a service speciifc interface function to call
            BOOLEAN bSearch;

            // call it
            bSearch =
                psRegisteredItem->psInterface->bSearchForCurrentAfterEnable(
                    &psRegisteredItem->uServiceSpecific
                        );
            // does the specific service want to search for currently occurring
            // content following the enable?
            if (bSearch == TRUE)
            {
                // yes they do
                CAL_CONTENT.eSearchForCurrent(hSeekContent);
            }
        }
    }
    }

    return eReturn;
}

/*****************************************************************************
*
*   eDisableItem
*
*   This API is used to disable the currently iterated item from a seek service.
*   This API may only be called from with a seek iterator callback function
*
*   Inputs:
*
*   hSeekContent - A handle to a valid seek content.
*
*   Outputs:
*
*   SMSAPI_RETURN_CODE_SUCCESS on success, SMSAPI_RETURN_CODE_ERROR on error
*
*****************************************************************************/
static SMSAPI_RETURN_CODE_ENUM eDisable (
    SEEK_CONTENT_OBJECT hSeekContent
        )
{
    BOOLEAN bOwner;
    SEEK_CONTENT_REGISTERED_ITEM_STRUCT *psRegisteredItem;
    SMSAPI_RETURN_CODE_ENUM eReturn = SMSAPI_RETURN_CODE_ERROR;

    // validate input
    bOwner =
        SMSO_bOwner((SMS_OBJECT)hSeekContent);
    if(bOwner == FALSE)
    {
        // Not owner of the provided object
        return SMSAPI_RETURN_CODE_API_NOT_ALLOWED;
    }

    // extract the registered item information
    psRegisteredItem = SEEK_CONTENT_psRegisteredItem(hSeekContent);

    if (psRegisteredItem != NULL)
    {
        if (psRegisteredItem->bEnabled == FALSE)
        {
            // this item is already disabled. no further work needs to be done
            eReturn = SMSAPI_RETURN_CODE_SUCCESS;
        }
        else
        {
        // disable this item
        psRegisteredItem->bEnabled = FALSE;

        // update the tag
        eReturn = SEEK_CONTENT_eUpdateEnabledTag(psRegisteredItem, TRUE);
        }
    }

    return eReturn;
}

/*****************************************************************************
*
*   bEnabled
*
*   This API is used to retrieve the enabled status of the currently iterated
*   item from a seek service. This API may only be called from with a seek
*   iterator callback function
*
*   Inputs:
*
*   hSeekContent - A handle to a valid seek content.
*
*   Outputs:
*
*   TRUE if Enabled. FALSE if Disabled or error
*
*****************************************************************************/
static BOOLEAN bEnabled (
    SEEK_CONTENT_OBJECT hSeekContent
        )
{
    BOOLEAN bOwner;
    SEEK_CONTENT_REGISTERED_ITEM_STRUCT *psRegisteredItem;

    // verify input
    bOwner =
        SMSO_bOwner((SMS_OBJECT)hSeekContent);
    if(bOwner == FALSE)
    {
        // not the owner
        return FALSE;
    }

    // extract the registered item information
    psRegisteredItem = SEEK_CONTENT_psRegisteredItem(hSeekContent);


    if (psRegisteredItem == NULL)
    {
        // something went wrong
        return FALSE;
    }

    // return the enabled status
    return psRegisteredItem->bEnabled;
}

/*****************************************************************************
*
*   eUseSeekContent
*
*   This API is used to perform any type of action of
*   a SEEK_CONTNET handle. It allows a caller to provide a function and
*   argument which allows the direct manipulation of SEEK_CONTENT.
*   Manipulation of the SEEK_CONTENT object is performed
*   while safely observing exclusive access rules.
*
*   Inputs:
*       hSeekContent - A handle to a valid SEEK_CONTENT_OBJECT.
*       vSeekContentAccessCallback - The callback which will be
*           invoked if the requested image type is located within the
*           provided art object.
*       *pvCallbackArg - The caller-provided argument which will be
*       given as a paramter to vSeekContentAccessCallback
*
*   Outputs:
*       SMSAPI_RETURN_CODE_SUCCESS on success or
*       SMSAPI_RETURN_CODE_INVALID_INPUT if any parameter check fails.
*
*****************************************************************************/
static SMSAPI_RETURN_CODE_ENUM eUseSeekContent (
    SEEK_CONTENT_OBJECT hSeekContent,
    SEEK_CONTENT_ACCESS_CALLBACK vSeekContentAccessCallback,
    void *pvCallbackArg
        )
{
    BOOLEAN bLocked;

    // Verify input pointers
    if (vSeekContentAccessCallback == NULL)
    {
        return SMSAPI_RETURN_CODE_INVALID_INPUT;
    }

    // Lock object for exclusive access
    bLocked = SMSO_bLock( (SMS_OBJECT)hSeekContent, OSAL_OBJ_TIMEOUT_INFINITE );

    if ( bLocked == FALSE )
    {
        // We were provided a bad seek content handle
        return SMSAPI_RETURN_CODE_INVALID_INPUT;
    }

    // Call function
    vSeekContentAccessCallback( hSeekContent, pvCallbackArg );

    // Unlock and return success
    SMSO_vUnlock( (SMS_OBJECT)hSeekContent );

    return SMSAPI_RETURN_CODE_SUCCESS;
}

/*****************************************************************************
                             FRIEND FUNCTIONS
*****************************************************************************/

/*****************************************************************************
*
*   SEEK_CONTENT_psCreateRegisteredItem
*
*****************************************************************************/
SEEK_CONTENT_REGISTERED_ITEM_STRUCT *SEEK_CONTENT_psCreateRegisteredItem(
    SEEK_SERVICE_OBJECT hSeekService,
    BOOLEAN bInitiallyEnabled
        )
{
    SEEK_CONTENT_REGISTERED_ITEM_STRUCT *psStruct;
    SEEK_SERVICE_ENUM eService;

    eService = SEEK_eService(hSeekService);
    if (eService == SEEK_SERVICE_INVALID)
    {
        return NULL;
    }

    // create a new SEEK_CONTENT_REGISTERED_ITEM_STRUCT obj
    psStruct = (SEEK_CONTENT_REGISTERED_ITEM_STRUCT *)
                   SMSO_hCreate(
                       SEEK_CONTENT_REGISTERED_ITEM_NAME,
                       sizeof(SEEK_CONTENT_REGISTERED_ITEM_STRUCT),
                       (SMS_OBJECT)hSeekService,
                       FALSE
                           );

    if (psStruct != NULL)
    {
        // initialize
        psStruct->hSeekService = hSeekService;
        // initially enabled
        psStruct->bEnabled = bInitiallyEnabled;

        // set the service specific content interface
        psStruct->psInterface =
            SEEK_psServiceSpecificContentInterface(hSeekService);
    }

    return psStruct;
}


/*****************************************************************************
*
*   SEEK_CONTENT_psRegisteredItem
*
*****************************************************************************/
SEEK_CONTENT_REGISTERED_ITEM_STRUCT *SEEK_CONTENT_psRegisteredItem (
    SEEK_CONTENT_OBJECT hSeekContent
        )
{
    // The Seek Content Registered Item is stored as the CAL_CONTENT Content Arg
    // (and the SEEK_CONTENT is really a CAL_CONTENT)
    return (SEEK_CONTENT_REGISTERED_ITEM_STRUCT *)
               CAL_CONTENT.pvContentArg((CAL_CONTENT_OBJECT)hSeekContent);
}

/*****************************************************************************
*
*   SEEK_CONTENT_vDestroySeekContent
*
*****************************************************************************/
void SEEK_CONTENT_vDestroySeekContent(
    SEEK_CONTENT_OBJECT hSeekContent
        )
{
    BOOLEAN bValid;
    SEEK_CONTENT_REGISTERED_ITEM_STRUCT *psRegisteredItem;

    // verify input
    bValid =
        SMSO_bValid((SMS_OBJECT)hSeekContent);
    if(bValid == FALSE)
    {
        // not valid
        return;
    }

    // extract from the CAL_CONTENT object the registered item
    psRegisteredItem = SEEK_CONTENT_psRegisteredItem(hSeekContent);
    // Make sure the content  doesn't refer it any more
    CAL_CONTENT_eUpdateContentArg((CAL_CONTENT_OBJECT)hSeekContent, NULL);
    // destroy the registered item
    SEEK_CONTENT_vDestroyRegisteredItem(psRegisteredItem);

}

/*****************************************************************************
*
*   SEEK_CONTENT_vDestroyRegisteredItem
*
*****************************************************************************/
void SEEK_CONTENT_vDestroyRegisteredItem(
    SEEK_CONTENT_REGISTERED_ITEM_STRUCT *psRegItem
        )
{
    BOOLEAN bValid;

    // check input
    bValid = SMSO_bValid((SMS_OBJECT)psRegItem);
    if (bValid ==TRUE)
    {
        // call service specific destroy function here (if it exists)
        if (psRegItem->psInterface->vUnInit != NULL)
        {
            // there is a service specific function to uninit the registered
            // content
            psRegItem->psInterface->vUnInit(&psRegItem->uServiceSpecific);
        }

        // uninitialize
        psRegItem->bEnabled = FALSE;
        psRegItem->hSeekService = SEEK_SERVICE_INVALID_OBJECT;

        // free the object we created when we registered this item
        SMSO_vDestroy((SMS_OBJECT)psRegItem);
    }
}

/*****************************************************************************
*
*   SEEK_CONTENT_eUpdateEnabledTag
*
* This function is used to set the seek content enabled tag
*
*****************************************************************************/
SMSAPI_RETURN_CODE_ENUM SEEK_CONTENT_eUpdateEnabledTag(
    SEEK_CONTENT_REGISTERED_ITEM_STRUCT *psRegItem,
    BOOLEAN bCommit
        )
{
    BOOLEAN bValid;
    SMSAPI_RETURN_CODE_ENUM eResult = SMSAPI_RETURN_CODE_ERROR;

    // check input
    bValid = SMSO_bValid((SMS_OBJECT)psRegItem);
    if (bValid == TRUE)
    {
        TAG_OBJECT hTag;

        eResult = TAG_eGet(
                      SEEK_CONTENT_ENABLED,
                      psRegItem->hTag,
                      &hTag,
                      NULL,
                      TRUE
                          );

        if (eResult == SMSAPI_RETURN_CODE_SUCCESS)
        {
            STRING_OBJECT hValueString;

            if (psRegItem->bEnabled == TRUE)
            {
                hValueString = CM_hTrueString();
            }
            else
            {
                hValueString = CM_hFalseString();
            }

            // set the tag value
            eResult = TAG_eSetTagValue(
                          hTag,
                          TAG_TYPE_STRING,
                          &hValueString,
                          sizeof(STRING_OBJECT),
                          bCommit
                              );
        }
        else if (eResult == SMSAPI_RETURN_CODE_CFG_NO_PARENT)
        {
            // must mean there is no file system
            eResult = SMSAPI_RETURN_CODE_SUCCESS;
        }
    }

    return eResult;
}

/*****************************************************************************
                             PRIVATE FUNCTIONS
*****************************************************************************/

/*****************************************************************************
*
*   n32FPrintf
*
*    This function is used to print out to a file human
*    readable information about the content registered to the seek service
*
*****************************************************************************/
static N32 n32FPrintf (
    SEEK_CONTENT_OBJECT hSeekContent,
    FILE *psFile,
    SMSAPI_OUTPUT_OPTION_ENUM eOutputOption
        )
{
    SEEK_CONTENT_REGISTERED_ITEM_STRUCT *psItem;
    STRING_OBJECT hArtistString, hTitleString;
    CID_OBJECT hRegisteredCID;
    N32 n32NumWritten = 0, n32Temp;

    // get the item
    psItem = SEEK_CONTENT_psRegisteredItem(hSeekContent);
    if ((psItem == NULL) || (psFile == NULL))
    {
        return EOF;
    }

    // get the registered Artist Text which is stored in CAL_CONTENT objects
    // and the SEEK_CONTENT is really a CAL_CONTENT
    hArtistString =
            CAL_CONTENT.hArtistText((CAL_CONTENT_OBJECT)hSeekContent);
    // get the registered Title Text which is stored in CAL_CONTENT objects
    // and the SEEK_CONTENT is really a CAL_CONTENT
    hTitleString = CAL_CONTENT.hTitleText((CAL_CONTENT_OBJECT)hSeekContent);

    // print - based on the desired option
    switch(eOutputOption)
    {
        case SMS_OUTPUT_OPTION_TERSE:
        {
            // print out the registered Artist Text
            if (hArtistString != STRING_INVALID_OBJECT)
            {
                n32NumWritten += fprintf(psFile, "\"");
                n32Temp = STRING.n32FWrite(hArtistString, stdout);
                if(n32Temp > 0)
                {
                    n32NumWritten += n32Temp;
                }
                n32NumWritten += fprintf(psFile, "\"\\");
            }

            // print out the registered Title Text
            if (hTitleString != STRING_INVALID_OBJECT)
            {
                n32NumWritten += fprintf(psFile, "\"");
                n32Temp = STRING.n32FWrite(hTitleString, psFile);
                if(n32Temp > 0)
                {
                    n32NumWritten += n32Temp;
                }
                n32NumWritten += fprintf(psFile, "\" ");
            }

            // print out whether the item is enabled or disabled
            n32NumWritten += fprintf(psFile, "%s ",
                                     psItem->bEnabled == TRUE ?
                                     "ENABLED" : "DISABLED");

            // based on the type, print out service specific information.
            // (if there is a service specific function)
            if (psItem->psInterface->n32FPrintf != NULL)
            {
                n32NumWritten +=
                        psItem->psInterface->n32FPrintf(
                            &psItem->uServiceSpecific,
                            psFile,
                            eOutputOption
                                );
            }

            n32NumWritten += fprintf(psFile, "\n");

        }
        break;

        case SMS_OUTPUT_OPTION_VERBOSE:
        case SMS_OUTPUT_OPTION_GROSS:
        default:
        {
            // print out whether the item is enabled or disabled
            n32NumWritten += fprintf(psFile, "%s\n",
                                     psItem->bEnabled == TRUE ?
                                     "ENABLED" : "DISABLED");

            // print out the Artist Text
            if (hArtistString != STRING_INVALID_OBJECT)
            {
                n32NumWritten += fprintf(psFile, "ArtistText:\t");
                n32Temp = STRING.n32FWrite(hArtistString, stdout);
                if(n32Temp > 0)
                {
                    n32NumWritten += n32Temp;
                }
                n32NumWritten += fprintf(psFile, "\n");
            }

            // print out the Title Text
            if (hTitleString != STRING_INVALID_OBJECT)
            {
                n32NumWritten += fprintf(psFile, "TitleText:\t");
                n32Temp = STRING.n32FWrite(hTitleString, psFile);
                if(n32Temp > 0)
                {
                    n32NumWritten += n32Temp;
                }
                n32NumWritten += fprintf(psFile, "\n");
            }

            // based on the type, print out service specific information.
            // (if there is a service specific function)
            if (psItem->psInterface->n32FPrintf != NULL)
            {
                n32NumWritten +=
                        psItem->psInterface->n32FPrintf(
                            &psItem->uServiceSpecific,
                                    psFile,
                                    eOutputOption
                                );
            }

            hRegisteredCID =
                    CAL_CONTENT.hCID((CAL_CONTENT_OBJECT)hSeekContent);
            n32NumWritten += fprintf(psFile, "Registered ");
            n32NumWritten += CID.n32FPrintf(hRegisteredCID, psFile);
        }
        break;
    }

    return n32NumWritten;
}
