/******************************************************************************/
/*                    Copyright (c) Sirius XM Radio, Inc.                     */
/*                            All Rights Reserved                             */
/*         Licensed Materials - Property of Sirius XM Radio, Inc.             */
/******************************************************************************/
/*******************************************************************************
 *
 * DESCRIPTION
 *
 *  This module contains the Object:SCache implementation for the
 *  Sirius Module Services (SMS)
 *
 ******************************************************************************/
#include <string.h>
#include <stdio.h>

#include "standard.h"
#include "osal.h"

#include "sms_version.h"
#include "sms_api.h"
#include "sms_obj.h"
#include "radio.h"
#include "song_obj.h"

#include "scache.h"
#include "_scache.h"

#include "sms_api_debug.h"
static const char *gpacThisFile = __FILE__;

/*****************************************************************************
                             PUBLIC FUNCTIONS
*****************************************************************************/

/*****************************************************************************
                             FRIEND FUNCTIONS
*****************************************************************************/
/*****************************************************************************
*
*   SCACHE_hCreate
*
*****************************************************************************/
SCACHE_OBJECT SCACHE_hCreate (
    PLAYBACK_OBJECT hPlayback
        )
{
    static UN32 un32Instance = 0;
    OSAL_RETURN_CODE_ENUM eReturnCode;
    SCACHE_OBJECT_STRUCT *psObj =
        (SCACHE_OBJECT_STRUCT *)SCACHE_INVALID_OBJECT;
    char acName[OSAL_MAX_OBJECT_NAME_LENGTH_WITH_NULL];
    BOOLEAN bValid;

    // Verify inputs
    bValid = SMSO_bValid((SMS_OBJECT)hPlayback);
    if(bValid == FALSE)
    {
        // Error!
        return SCACHE_INVALID_OBJECT;
    }

    // Construct a unique name for this song cache.
    snprintf(&acName[0],
            sizeof(acName),
            SCACHE_OBJECT_NAME":%u",
            un32Instance++);

    // Create an instance of this object (child of the specified DECODER)
    psObj = (SCACHE_OBJECT_STRUCT *)
        SMSO_hCreate(
            SCACHE_OBJECT_NAME,
            sizeof(SCACHE_OBJECT_STRUCT),
            (SMS_OBJECT)hPlayback, // Child
            FALSE); // No Lock (ignored)
    if(psObj == NULL)
    {
        // Error!
        return SCACHE_INVALID_OBJECT;
    }

    // Copy object name into SCACHE object
    strncpy(&psObj->acName[0], &acName[0], sizeof(psObj->acName));

    // Initialize object...

    psObj->hPlayback = hPlayback;

    // Create song list
    eReturnCode = OSAL.eLinkedListCreate(
        &psObj->hSongList,
        &psObj->acName[0],
        (OSAL_LL_COMPARE_HANDLER)n16SongListCompareHandler,
        OSAL_LL_OPTION_UNIQUE
            );
    if(eReturnCode != OSAL_SUCCESS)
    {
        // Error!
        SCACHE_vDestroy((SCACHE_OBJECT)psObj);
        return SCACHE_INVALID_OBJECT;
    }
    psObj->hTail = OSAL_INVALID_LINKED_LIST_ENTRY;

    psObj->tCurrentSongID = SONG_INVALID_ID;

        // Initialize asynchronous update configuration
    SMSU_vInitialize(
        &psObj->sEvent,
        psObj,
        SCACHE_OBJECT_EVENT_NONE,
        SMS_OBJECT_EVENT_NONE,
        (SMSAPI_OBJECT_EVENT_CALLBACK)NULL,
        NULL);

    return (SCACHE_OBJECT)psObj;
}

/*****************************************************************************
*
*   SCACHE_vDestroy
*
*****************************************************************************/
void SCACHE_vDestroy (
    SCACHE_OBJECT hSCache
        )
{
    BOOLEAN bValid;

    bValid = SMSO_bValid((SMS_OBJECT)hSCache);

    if (bValid == TRUE)
    {
        SCACHE_OBJECT_STRUCT *psObj =
            (SCACHE_OBJECT_STRUCT *)hSCache;

        // Filter out all further scache updates
        SMSU_tFilter(&psObj->sEvent, SCACHE_OBJECT_EVENT_ALL);

        psObj->hPlayback = PLAYBACK_INVALID_OBJECT;

        psObj->tCurrentSongID = SONG_INVALID_ID;

        // Destroy song list if one exists
        if(psObj->hSongList != OSAL_INVALID_OBJECT_HDL)
        {
            OSAL_RETURN_CODE_ENUM eReturnCode;
            // Remove all entries in the linked list and destroy each song
            OSAL.eLinkedListRemoveAll(psObj->hSongList, vDeleteSong);

            // Delete linked list itself
            eReturnCode = OSAL.eLinkedListDelete(psObj->hSongList);
            if(eReturnCode == OSAL_SUCCESS)
            {
                psObj->hSongList = OSAL_INVALID_OBJECT_HDL;
            }
        }
        psObj->hTail = OSAL_INVALID_LINKED_LIST_ENTRY;

        // Destroy the async update configuration
        SMSU_vDestroy(&psObj->sEvent);

        // Free object instance
        SMSO_vDestroy((SMS_OBJECT)hSCache);
    } // if (bValid == TRUE)
    return;
}

/*****************************************************************************
*
*   SCACHE_hSong
*
*   This retrieves a SONG from the SCACHE at a specified absolute offset
*
*****************************************************************************/
SONG_OBJECT SCACHE_hSong (
    SCACHE_OBJECT hSCache,
    N16 n16Offset
    )
{
    SCACHE_OBJECT_STRUCT *psObj =
        (SCACHE_OBJECT_STRUCT *)hSCache;
    SONG_OBJECT hSong = SONG_INVALID_OBJECT;
    BOOLEAN bValid;
    OSAL_RETURN_CODE_ENUM eReturnCode;
    OSAL_LINKED_LIST_ENTRY hLLEntry = OSAL_INVALID_LINKED_LIST_ENTRY;

    // Verify inputs
    bValid = SMSO_bValid((SMS_OBJECT)hSCache);
    if((bValid == FALSE) || (n16Offset < 0) )
    {
        // Error!
        return SONG_INVALID_OBJECT;
    }

    eReturnCode = OSAL.eLinkedListLinearSearch(
        psObj->hSongList,
        &hLLEntry,
        (OSAL_LL_COMPARE_HANDLER)SONG_n16FindSongByOffset,
        (void*)(size_t)n16Offset);

    // Return song handle found
    if (eReturnCode != OSAL_SUCCESS)
    {
        return SONG_INVALID_OBJECT;
    }

    // Channel exists, extract the data from this entry
    hSong = (SONG_OBJECT)OSAL.pvLinkedListThis(hLLEntry);

    return hSong;
}

/*****************************************************************************
*
*   SCACHE_hCurrentSong
*
*   This retrieves the current SONG from the SCACHE
*
*****************************************************************************/
SONG_OBJECT SCACHE_hCurrentSong (
    SCACHE_OBJECT hSCache
        )
{
    BOOLEAN bValid;
    SONG_OBJECT hSong = SONG_INVALID_OBJECT;

    // Validate object
    bValid = SMSO_bValid((SMS_OBJECT)hSCache);

    if (bValid == TRUE)
    {
        SCACHE_OBJECT_STRUCT *psObj =
            (SCACHE_OBJECT_STRUCT *)hSCache;

        hSong = hSongByID(psObj, psObj->tCurrentSongID);
    }

    return hSong;
}

/*****************************************************************************
*
*   SCACHE_n16NumberOfSongs
*
*****************************************************************************/
N16 SCACHE_n16NumberOfSongs (
    SCACHE_OBJECT hSCache
        )
{
    BOOLEAN bValid;
    N16 n16NumSongs = 0;
    OSAL_RETURN_CODE_ENUM eResult;

    // Validate object
    bValid = SMSO_bValid((SMS_OBJECT)hSCache);

    if (bValid == TRUE)
    {
        UN32 un32NumSongs;
        SCACHE_OBJECT_STRUCT *psObj =
            (SCACHE_OBJECT_STRUCT *)hSCache;

        eResult = OSAL.eLinkedListItems(psObj->hSongList, &un32NumSongs);
        if (eResult == OSAL_SUCCESS)
        {
            n16NumSongs = (N16)un32NumSongs;
        }
    }

    return n16NumSongs;
}

/*****************************************************************************
*
*   SCACHE_n16CurrentSongOffset
*
*****************************************************************************/
N16 SCACHE_n16CurrentSongOffset (
    SCACHE_OBJECT hSCache
        )
{
    BOOLEAN bValid;
    SONG_OBJECT hSong;
    SONG_ID tSongId;
    N16 n16CurrentOffset = 0;

    // Validate object
    bValid = SMSO_bValid((SMS_OBJECT)hSCache);

    if (bValid == TRUE)
    {
        OSAL_RETURN_CODE_ENUM eReturn;
        SCACHE_OBJECT_STRUCT *psObj =
            (SCACHE_OBJECT_STRUCT *)hSCache;

        // If we are looking for <<LIVE>>, it is always the last one
        if (psObj->tCurrentSongID == SONG_LIVE_ID)
        {
            UN32 un32Count = 0;

            eReturn = OSAL.eLinkedListItems(psObj->hSongList, &un32Count);
            if (OSAL_SUCCESS != eReturn)
            {
                SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    "Song List does not exist");
            }
            else
            {
                // Offset is 0-based
                n16CurrentOffset = (un32Count & 0xFFFF) - 1;
            }
            
        }
        else
        {
            OSAL_LINKED_LIST_ENTRY hEntry;

            hEntry = psObj->hTail;
            n16CurrentOffset = 0;
            hSong = (SONG_OBJECT)OSAL.pvLinkedListThis(hEntry);

            do
            {
                tSongId = SONG_tId(hSong);
                if (tSongId == psObj->tCurrentSongID)
                {
                    break;
                }
                hEntry = OSAL.hLinkedListNext(hEntry, (void**)(size_t)&hSong);
                n16CurrentOffset++;

            } while (hEntry != psObj->hTail);
        }
    }

    return n16CurrentOffset;
}


/*****************************************************************************
*
*   SCACHE_tCurrentSongID
*
*****************************************************************************/
SONG_ID SCACHE_tCurrentSongID (
    SCACHE_OBJECT hSCache
        )
{
    BOOLEAN bValid;
    SONG_ID tCurrentSongID = SONG_INVALID_ID;

    // Validate object
    bValid = SMSO_bValid((SMS_OBJECT)hSCache);

    if (bValid == TRUE)
    {
        SCACHE_OBJECT_STRUCT *psObj =
            (SCACHE_OBJECT_STRUCT *)hSCache;
        tCurrentSongID =  psObj->tCurrentSongID;
    }
    return tCurrentSongID;
}

/*****************************************************************************
*
*   SCACHE_vChangeCurrentSongOffset
*
*   This function will change the current song offset (an absolute offset)
*   in the scache by a relative offset.
*
*   Inputs:
*       hSCache - A valid SCache handle
*       n16RelOffset - A relative offset value from the current song offset
*           (an absolute offset)
*
*   Outputs:
*       A boolean value, representing in-range (TRUE) / out-range (FALSE)
*           offset was provided
*
*****************************************************************************/
BOOLEAN SCACHE_bChangeCurrentSongID (
    SCACHE_OBJECT hSCache,
    SONG_ID tCurrentSongID
        )
{
    BOOLEAN bValid, bInRange = FALSE;

    bValid = SMSO_bValid((SMS_OBJECT)hSCache);

    SMSAPI_DEBUG_vPrint(SCACHE_OBJECT_NAME, 5,
        "%s( %d )", __FUNCTION__, tCurrentSongID);

    if (bValid == TRUE)
    {
        SCACHE_OBJECT_STRUCT *psObj =
            (SCACHE_OBJECT_STRUCT *)hSCache;
        OSAL_RETURN_CODE_ENUM eReturn;
        OSAL_LINKED_LIST_ENTRY hEntry = OSAL_INVALID_LINKED_LIST_ENTRY;

        // LIVE ID can be assigned unconditionally
        if (SONG_LIVE_ID == tCurrentSongID)
        {
            psObj->tCurrentSongID = tCurrentSongID;
            bInRange = TRUE;
        }
        else
        {
            // Search the list for required ID
            eReturn = OSAL.eLinkedListLinearSearch(psObj->hSongList,
                &hEntry,
                (OSAL_LL_COMPARE_HANDLER)SONG_n16SongIdEqual,
                (void*)(size_t)tCurrentSongID);

            if (OSAL_SUCCESS == eReturn)
            {
                // Got it!
                psObj->tCurrentSongID = tCurrentSongID;
                bInRange = TRUE;
            }
            else
            {
                SMSAPI_DEBUG_vPrint(SCACHE_OBJECT_NAME, 5,
                    "Song ID %d is not found", tCurrentSongID);
            }
        }

        SMSAPI_DEBUG_vPrint(SCACHE_OBJECT_NAME, 5,
		    "Current Song offset: %d, inRange = %s",
            psObj->tCurrentSongID, bInRange ? "YES" : "NO");
    }

    return bInRange;
}

/*****************************************************************************
*
*   SCACHE_hGetSong
*
* Create a new SONG with this tId and add it to the SCACHE in
* the appropriate slot but if a SONG already exists in the SCACHE with the
* given tId, return its handle instead
*
*****************************************************************************/
SONG_OBJECT SCACHE_hGetSong(
    SCACHE_OBJECT hSCache,
    SONG_ID tId
        )
{
    BOOLEAN bOwner;
    SONG_OBJECT hNewSong = SONG_INVALID_OBJECT;
    OSAL_RETURN_CODE_ENUM eResult;
    OSAL_LINKED_LIST_ENTRY hNewEntry = OSAL_INVALID_LINKED_LIST_ENTRY;

    SMSAPI_DEBUG_vPrint(SCACHE_OBJECT_NAME, 5, "%s(tId=%d)", __FUNCTION__, tId);

    bOwner = SMSO_bOwner((SMS_OBJECT)hSCache);

    if (bOwner == TRUE)
    {
        SCACHE_OBJECT_STRUCT *psObj =
            (SCACHE_OBJECT_STRUCT *)hSCache;

        // create a new song. we'll just set the offset to 0 for now
        // then  we'll update it after we decide what it should be
        hNewSong = SONG_hCreate(hSCache, 0);

        if (hNewSong != SONG_INVALID_OBJECT)
        {
            N16 n16NumSongs;
            N16 n16Offset;
            OSAL_LINKED_LIST_ENTRY hEntry;
            SONG_OBJECT hSong;

            // song was created successfully.
            SMSAPI_DEBUG_vPrint(SCACHE_OBJECT_NAME, 5, "Created song Id %d", tId);

            // set the id
            SONG_bSetId(hNewSong, tId);

            //now add it to the scache
            eResult = OSAL.eLinkedListAdd(
                psObj->hSongList,
                &hNewEntry,
                hNewSong );
            
            if (OSAL_ERROR_LIST_ITEM_NOT_UNIQUE == eResult)
            {
                // don't need this
                SONG_vDestroy(hNewSong);

                // give them the song obj they wanted
                hNewSong = (SONG_OBJECT)OSAL.pvLinkedListThis(hNewEntry);
                SMSAPI_DEBUG_vPrint(SCACHE_OBJECT_NAME, 5, 
                    ":: Found existing %p (entry %p)", 
                    hNewSong, hNewEntry);

                // Returning immediately. We do not need to do anything else
                return hNewSong;
            }
            else if (OSAL_SUCCESS != eResult)
            {
                SMSAPI_DEBUG_vPrint(SCACHE_OBJECT_NAME, 5, "Cannot add song (%s)",
                    OSAL.pacGetReturnCodeName(eResult));

                // Error. Don't need this anyway
                SONG_vDestroy(hNewSong);

                // Returning
                return SONG_INVALID_OBJECT;
            }

            SMSAPI_DEBUG_vPrint(SCACHE_OBJECT_NAME, 5, 
                "Song added into the list (%p, entry %p)",
                hNewSong, hNewEntry);

            n16NumSongs = SCACHE_n16NumberOfSongs(hSCache);
            // new entry successfully added to the list
            if (n16NumSongs == 1)
            {
                // add entry for "live" only if we are not playing TuneMix
                SONG_OBJECT hLive;
                OSAL_LINKED_LIST_ENTRY hLiveEntry =
                    OSAL_INVALID_LINKED_LIST_ENTRY;

                SMSAPI_DEBUG_vPrint(SCACHE_OBJECT_NAME, 5, ":: Inserting <<LIVE>>");

                hLive = SONG_hCreate(hSCache, 1);
                if (hLive != SONG_INVALID_OBJECT)
                {
                    // populate the artist and title fields
                    SONG_bUpdateSong(
                        hLive,
                        SCACHE_LIVE_SONG_ARTIST_STRING,
                        SCACHE_LIVE_SONG_TITLE_STRING,
                        0 );

                    SONG_bSetId(hLive, SONG_LIVE_ID);

                    // add to the tail
                    eResult = OSAL.eLinkedListAdd(
                        psObj->hSongList,
                        &hLiveEntry,
                        hLive );

                    if (eResult != OSAL_SUCCESS)
                    {
                        // for some reason the live song was not added to
                        // the scache
                        // remove the new song,
                        OSAL.eLinkedListRemove(hNewEntry);
                        // destroy both
                        SONG_vDestroy(hNewSong);
                        SONG_vDestroy(hLive);

                        // Reset tail back to NULL
                        psObj->hTail = OSAL_INVALID_LINKED_LIST_ENTRY;
                        return SONG_INVALID_OBJECT;
                    }
                    else
                    {
                        // save list tail pointer
                        psObj->hTail = hNewEntry;
                    }
                } //(hLive != SONG_INVALID_OBJECT)
            } // if (psObj->n16NumSongs == 1)

            // Tail song is zero offset always
            hEntry = psObj->hTail;
            hSong = (SONG_OBJECT)OSAL.pvLinkedListThis(psObj->hTail);
            n16Offset = 0;

            // now traverse the list, updating offsets....
            do
            {
                // update the song's new offset and increment the offset
                SONG_bUpdateOffset(hSong, n16Offset++);
                hEntry = OSAL.hLinkedListNext(hEntry, (void**)&hSong);

            } while (hEntry != psObj->hTail); 

            // Set Time Stamp for a new Song
            SONG_bSetTimestamp(hNewSong);

            // Set event mask
            SMSU_tUpdate(&psObj->sEvent, SCACHE_OBJECT_EVENT_NEW_SONG);

            // Tell PLAYBACK to call any registered callback for this SCACHE
            PLAYBACK_vUpdateSCache(psObj->hPlayback);
        } // if (hNewSong != SONG_INVALID_OBJECT)
    } // if (bOwner == TRUE)

    return hNewSong;
}

/*****************************************************************************
*
*   SCACHE_hSongLLEntry
*
*****************************************************************************/
OSAL_LINKED_LIST_ENTRY SCACHE_hSongLLEntry(
    SCACHE_OBJECT hSCache,
    SONG_ID tSongId
        )
{
    BOOLEAN bOwner;
    OSAL_RETURN_CODE_ENUM eResult;
    OSAL_LINKED_LIST_ENTRY hEntry = OSAL_INVALID_LINKED_LIST_ENTRY;

    bOwner = SMSO_bOwner((SMS_OBJECT)hSCache);

    if (bOwner == TRUE)
    {
        SCACHE_OBJECT_STRUCT *psObj =
            (SCACHE_OBJECT_STRUCT *)hSCache;



        eResult = OSAL.eLinkedListLinearSearch(
                      psObj->hSongList,
                      &hEntry,
                      (OSAL_LL_COMPARE_HANDLER)SONG_n16SongIdEqual,
                      (void *)(size_t)tSongId
                          );

        if ( (eResult != OSAL_SUCCESS) && (eResult != OSAL_OBJECT_NOT_FOUND) )
        {
            printf("ERROR: SCACHE_hFindSong: failed LL search (%u)\n", eResult);
        }

    } // if (bOwner == TRUE)

    return hEntry;
}

/*****************************************************************************
*
*   SCACHE_vRemoveSongFromSCache
*
*****************************************************************************/
void SCACHE_vRemoveSongFromSCache(
    SCACHE_OBJECT hSCache,
    SONG_ID tId
        )
{
    SCACHE_OBJECT_STRUCT *psObj =
            (SCACHE_OBJECT_STRUCT *)hSCache;
    BOOLEAN bOwner;

    bOwner = SMSO_bOwner((SMS_OBJECT)hSCache);

    if (bOwner == TRUE)
    {
        SONG_OBJECT hSong;
        OSAL_LINKED_LIST_ENTRY hEntryToRemove, hEntry;

        hEntryToRemove = SCACHE_hSongLLEntry(hSCache, tId);

        hSong = SCACHE_hSongFromLLEntry(hEntryToRemove);
        if (hSong != SONG_INVALID_OBJECT)
        {
            OSAL_RETURN_CODE_ENUM eResult;
            SONG_OBJECT hSong2;
            
            hEntry = OSAL.hLinkedListNext(hEntryToRemove, (void**)&hSong2);
            
            if (hEntry == hEntryToRemove)
            {
                // There is only one entry left and it is the one to remove
                psObj->hTail = OSAL_INVALID_LINKED_LIST_ENTRY;
                psObj->tCurrentSongID = SONG_INVALID_ID;
            }
            else
            {
                // if it was current song ID -> move to next
                if (psObj->tCurrentSongID == tId)
                {
                    psObj->tCurrentSongID = SONG_tId(hSong2);
                }

                // if it was the tail - move to next
                if (psObj->hTail == hEntryToRemove)
                {
                    psObj->hTail = hEntry;
                }
            }

            eResult = OSAL.eLinkedListRemove(hEntryToRemove);
            if (eResult != OSAL_SUCCESS)
            {
                puts(SCACHE_OBJECT_NAME": Failed to remove song from scache");
            }

            // Set event mask
            SMSU_tUpdate(&psObj->sEvent, SCACHE_OBJECT_EVENT_REMOVE_SONG);

            // Tell PLAYBACK to call any registered callback for this SCACHE
            PLAYBACK_vUpdateSCache(psObj->hPlayback);

            // Now remove the song
            SONG_vDestroy(hSong);
        }

    } // if (bOwner == TRUE)

    return;
}


/*****************************************************************************
*
*   SCACHE_vFlush
*   This function is ued to remove all songs from the cache and destroy them.
*
*   input:
*       hSCache
*   output:
*       none
*****************************************************************************/
void SCACHE_vFlush(
    SCACHE_OBJECT hSCache
        )
{
    BOOLEAN bOwner;

    bOwner = SMSO_bOwner((SMS_OBJECT)hSCache);

    if (bOwner == TRUE)
    {
        SCACHE_OBJECT_STRUCT *psObj =
            (SCACHE_OBJECT_STRUCT *)hSCache;

        puts( SCACHE_OBJECT_NAME": Flushing SCache" );

        // remove the songs from the list and destroy them
        OSAL.eLinkedListRemoveAll(
            psObj->hSongList,
            vDeleteSong
                );
        // reset parameters
        psObj->hTail = OSAL_INVALID_LINKED_LIST_ENTRY;
        psObj->tCurrentSongID = SONG_INVALID_ID;

        // Set event mask
        SMSU_tUpdate(&psObj->sEvent, SCACHE_OBJECT_EVENT_FLUSHED);

        // Tell PLAYBACK to call any registered callback for this SCACHE
        PLAYBACK_vUpdateSCache(psObj->hPlayback);
    }

    return;
}

/*****************************************************************************
*
*   SCACHE_hGetTail
*       this function is used to find the song just before the tail.
*       this is a special song because it is the song currently being saved
*
*****************************************************************************/
SONG_OBJECT SCACHE_hGetTailMinusOne (
    SCACHE_OBJECT hSCache
        )
{
    BOOLEAN bValid;
    SONG_OBJECT hSong = SONG_INVALID_OBJECT;

    // Verify inputs
    bValid = SMSO_bValid((SMS_OBJECT)hSCache);

    if (bValid == TRUE)
    {
        SCACHE_OBJECT_STRUCT *psObj =
            (SCACHE_OBJECT_STRUCT *)hSCache;

        OSAL_LINKED_LIST_ENTRY hTail = psObj->hTail;

        OSAL.hLinkedListPrev(hTail, (void **)(&hSong));
    }
    return hSong;
}

/*****************************************************************************
*
*   SCACHE_hGetTail
*       this function is used to find the tail song.
*       this is a special song because it is the song representing live audio
*
*****************************************************************************/
SONG_OBJECT SCACHE_hGetTail (
    SCACHE_OBJECT hSCache
        )
{
    BOOLEAN bValid;
    SONG_OBJECT hSong = SONG_INVALID_OBJECT;

    // Verify inputs
    bValid = SMSO_bValid((SMS_OBJECT)hSCache);

    if (bValid == TRUE)
    {
        SCACHE_OBJECT_STRUCT *psObj =
            (SCACHE_OBJECT_STRUCT *)hSCache;

        hSong = (SONG_OBJECT)OSAL.pvLinkedListThis(psObj->hTail);
    }
    return hSong;
}

/*****************************************************************************
*
*   SCACHE_bRegisterNotification
*
*****************************************************************************/
BOOLEAN SCACHE_bRegisterNotification (
    SCACHE_OBJECT hSCache,
    SCACHE_EVENT_MASK tEventRequestMask,
    SCACHE_OBJECT_EVENT_CALLBACK vEventCallback,
    void *pvEventCallbackArg
        )
{
    BOOLEAN bValid, bReturn = FALSE;

    // Verify inputs
    bValid = SMSO_bValid((SMS_OBJECT)hSCache);

    if (bValid == TRUE)
    {
        SCACHE_OBJECT_STRUCT *psObj =
            (SCACHE_OBJECT_STRUCT *)hSCache;

        // Register a callback in the
        // asynchronous update configuration
        SMSU_bRegisterCallback(
            &psObj->sEvent,
            tEventRequestMask,
            (SMSAPI_OBJECT_EVENT_CALLBACK)vEventCallback,
            pvEventCallbackArg,
            TRUE);

        bReturn = TRUE;
    }
    return bReturn;
}

/*****************************************************************************
*
*   SCACHE_vUnregisterNotification
*
*****************************************************************************/
void SCACHE_vUnregisterNotification (
    SCACHE_OBJECT hSCache,
    SCACHE_OBJECT_EVENT_CALLBACK vEventCallback,
    void *pvEventCallbackArg
        )
{
    BOOLEAN bValid;

    bValid = SMSO_bValid((SMS_OBJECT)hSCache);

    if (bValid == TRUE)
    {
        SCACHE_OBJECT_STRUCT *psObj =
            (SCACHE_OBJECT_STRUCT *)hSCache;

        // Remove this callback from the
        // asynchronous update configuration
        SMSU_bUnregisterCallback(
            &psObj->sEvent,
            (SMSAPI_OBJECT_EVENT_CALLBACK)vEventCallback,
            pvEventCallbackArg
                );
    }
    return;
}

/*****************************************************************************
*
*   SCACHE_vUpdate
*
*   This API exists so that it may be called exclusively by the DECODER
*   object. The purpose is to trigger a SCACHE event when any event
*   change has occured. SCACHE events are always called from the
*   context of the DECODER object.
*
*   NOTE: THIS API MAY BE CALLED ONLY FROM THE CONTEXT OF A DECODER OBJECT!
*
*****************************************************************************/
void SCACHE_vUpdate (
    SCACHE_OBJECT hSCache
        )
{
    SCACHE_OBJECT_STRUCT *psObj =
        (SCACHE_OBJECT_STRUCT *)hSCache;
    BOOLEAN bValid;

    // Lock SCACHE object since this call is always made from
    // the context of the DECODER.
    bValid = SMSO_bValid((SMS_OBJECT)hSCache);
    if(bValid == TRUE)
    {
        // Notify callback if registered and change occured
        SMSU_bNotify(&psObj->sEvent);
    }

    return;
}

/*****************************************************************************
*
*   SCACHE_hSongFromLLEntry
*
*****************************************************************************/
SONG_OBJECT SCACHE_hSongFromLLEntry(OSAL_LINKED_LIST_ENTRY hLLEntry)
{
    SONG_OBJECT hSong = SONG_INVALID_OBJECT;

    if (hLLEntry != OSAL_INVALID_LINKED_LIST_ENTRY)
    {
        hSong = (SONG_OBJECT)OSAL.pvLinkedListThis(hLLEntry);
    }

    return hSong;
}

/*****************************************************************************
*
*   SCACHE_n16CompareSongOrder
*
*****************************************************************************/
N16 SCACHE_n16CompareSongOrder(
   SONG_ID tId1,
   SONG_ID tId2
       )
{
    N16 n16Return = N16_MIN;

    if (tId1 == tId2)
    {
        // these are the same
        return 0;
    }

    // compare the ids. In general the higher the id the newer the song
    // but the ids rollover, so we have to take that into account to....

    if (tId1 > tId2)
    {
        // check if rollover occurred
        // should we check for invalid or live????
        if ((tId1-tId2) > SONG_ID_ROLLOVER_MARGIN)
        {
            // rollover occurred
            // tId1 is for a song older than tId2
            n16Return = -1;
        }
        else
        {
            // no rollover occurred
            // tId1 is for a song newer than tId2
            n16Return = 1;
        }
    }
    else // (tId1 < tId2)
    {
         // check if rollover occurred
        // should we check for invalid or live????
        if ((tId2-tId1) > SONG_ID_ROLLOVER_MARGIN)
        {
            // rollover occurred
            // tId1 is for a song newer than tId2
            n16Return = 1;
        }
        else
        {
            // no rollover occurred
            // tId1 is for a song newer than tId2
            n16Return = -1;
        }
    }

    return n16Return;
}

/*****************************************************************************
*
*   SCACHE_vReady
*
*****************************************************************************/
void SCACHE_vReady(
    SCACHE_OBJECT hSCache
        )
{
    SCACHE_OBJECT_STRUCT *psObj = (SCACHE_OBJECT_STRUCT *)hSCache;

    // Set event mask
    SMSU_tUpdate(&psObj->sEvent, SCACHE_OBJECT_EVENT_READY);

    // Tell PLAYBACK to call any registered callback for this SCACHE
    PLAYBACK_vUpdateSCache(psObj->hPlayback);
    return;
}

#if SMS_DEBUG == 1

/*****************************************************************************
*
*   SCACHE_bIterateSongs
*
*   This function is used to iterate all the songs in the cache one
*   by one. Based on the iterator function provided iteration starts from
*   the beginning of the cache to the end, if the provider iterator continues
*   to return TRUE. Otherwise, iteration is stopped when this function returns
*   FALSE.
*
*   Inputs:
*       hSCache - A Song Cache handle.
*       bIterator - An iterator function callback which is called for each
*           song in the cache.
*       pvArg - A caller defined argument to supply in the iterator callback
*           for each song in the cache.
*
*   Returns:
*       TRUE if the cache was iterated. FALSE if the cache could not be
*       iterated.
*
*****************************************************************************/
BOOLEAN SCACHE_bIterateSongs (
    SCACHE_OBJECT hSCache,
    SCACHE_SONG_ITERATOR_HANDLER bIterator,
    void *pvArg
        )
{
    BOOLEAN bOwner, bReturn;
    SCACHE_OBJECT_STRUCT *psObj = (SCACHE_OBJECT_STRUCT *)hSCache;
    SONG_OBJECT hSong;
    OSAL_LINKED_LIST_ENTRY hEntry;

    // Verify inputs. Must already be a scache owner and they must provide
    // some type of iterator, otherwise what's the point?
    bOwner = SMSO_bOwner((SMS_OBJECT)hSCache);
    if((bOwner == FALSE) || (bIterator == NULL))
    {
        // Error!
        return FALSE;
    }

    // Do we have any items in the list?
    hEntry = psObj->hTail;
    if(hEntry == OSAL_INVALID_LINKED_LIST_ENTRY)
    {
        // Error!
        return FALSE;
    }

    // Iterate all songs in the cache...
    do 
    {
        hEntry = OSAL.hLinkedListPrev(hEntry, (void**)&hSong);

        // Call user callback function
        bReturn = bIterator(hSong, pvArg);

    } while ((bReturn == TRUE) && (hEntry != psObj->hTail));
    
    return TRUE;
}

#endif

/*****************************************************************************
                             PRIVATE FUNCTIONS
*****************************************************************************/

/*****************************************************************************
*
*   vDeleteSong
*
*****************************************************************************/
static void vDeleteSong(
    void *pvData
        )
{
    SONG_vDestroy((SONG_OBJECT)pvData);
    return;
}

/*****************************************************************************
*
*   n16SongListCompareHandler
*
*****************************************************************************/
static N16 n16SongListCompareHandler (SONG_OBJECT hSong1, SONG_OBJECT hSong2)
{
    N16 n16Return = 0;

    // If both are NULL consider them equal
    if ((SONG_INVALID_OBJECT != hSong1) || (SONG_INVALID_OBJECT != hSong2))
    {
        if (SONG_INVALID_OBJECT == hSong1)
        {
            // Song 1 is NULL and Song 2 is not
            n16Return = 1;
        }
        else if (SONG_INVALID_OBJECT == hSong2)
        {
            // Song 2 is NULL and Song 1 is not
            n16Return = -1;
        }
        else
        {
            SONG_ID tId1 = SONG_tId(hSong1), 
                tId2 = SONG_tId(hSong2);

            n16Return = SCACHE_n16CompareSongOrder(tId1, tId2);
        }
    }

    return n16Return;
}

/*****************************************************************************
*
*   hSongByID
*
*   This retrieves a SONG from the SCACHE by a specified Song ID
*
*****************************************************************************/
static SONG_OBJECT hSongByID (
    SCACHE_OBJECT_STRUCT *psObj,
    SONG_ID tSongID
    )
{
    SONG_OBJECT hSong = SONG_INVALID_OBJECT;
    OSAL_RETURN_CODE_ENUM eReturnCode;
    OSAL_LINKED_LIST_ENTRY hLLEntry = OSAL_INVALID_LINKED_LIST_ENTRY;

    // Verify inputs
    if(tSongID > SONG_LIVE_ID)
    {
        // Error!
        return SONG_INVALID_OBJECT;
    }

    eReturnCode = OSAL.eLinkedListLinearSearch(
        psObj->hSongList,
        &hLLEntry,
        (OSAL_LL_COMPARE_HANDLER)SONG_n16SongIdEqual,
        (void*)(size_t)tSongID
        );

    // Return song handle found
    if (eReturnCode != OSAL_SUCCESS)
    {
        return SONG_INVALID_OBJECT;
    }

    // Channel exists, extract the data from this entry
    hSong = (SONG_OBJECT)OSAL.pvLinkedListThis(hLLEntry);

    return hSong;
}

#if SMS_DEBUG == 1

/*****************************************************************************
*
*   bSongIterator
*
*   This is a simple linked list iterator shim function for the song
*   cache. It just simply allows the linked list iterator to apply
*   a caller defined callback with associated data.
*
*****************************************************************************/
static BOOLEAN bSongIterator (
    void *pvData,
    void *pvArg
        )
{
    BOOLEAN bRetVal;
    SONG_OBJECT hSong = (SONG_OBJECT)pvData;
    SCACHE_SONG_ITERATOR_HANDLER_STRUCT *psIteratorShim =
        (SCACHE_SONG_ITERATOR_HANDLER_STRUCT *)pvArg;

    // Call iterator provided by the caller
    bRetVal = psIteratorShim->bIterator(hSong, psIteratorShim->pvArg);

    return bRetVal;
}

#endif
