/******************************************************************************/
/*                    Copyright (c) Sirius XM Radio, Inc.                     */
/*                            All Rights Reserved                             */
/*            Licensed Materials - Property of Sirius XM Radio, Inc.          */
/******************************************************************************/
/*******************************************************************************
 *
 * DESCRIPTION
 *
 *  This module contains the Object:SAFECAM_SUBLOCATION implementation for the
 *  Sirius Module Services (SMS)
 *
 ******************************************************************************/
#include "standard.h"
#include "osal.h"

#include "sms_api.h"
#include "sms_obj.h"
#include "_safecam_sublocation_obj.h"
#include "safecam_sublocation_obj.h"
#include "location_obj.h"

#include "sms_api_debug.h"

/*****************************************************************************
                             PUBLIC FUNCTIONS
*****************************************************************************/

/*****************************************************************************
*
*   hLocation
*
*****************************************************************************/
static LOCATION_OBJECT hLocation (
    SAFETY_CAMERAS_SUBLOCATION_OBJECT hSafeCamSubLoc
        )
{
    LOCATION_OBJECT hResult = LOCATION_INVALID_OBJECT;

    do
    {
        SAFECAM_SUBLOCATION_OBJECT_STRUCT *psObj =
            (SAFECAM_SUBLOCATION_OBJECT_STRUCT*)hSafeCamSubLoc;
        BOOLEAN bOwner = FALSE;

        if (hSafeCamSubLoc == SAFETY_CAMERAS_SUBLOCATION_INVALID_OBJECT)

        {
            break;
        }

        bOwner = SMSO_bOwner((SMS_OBJECT)hSafeCamSubLoc);
        if (bOwner == FALSE)
        {
            break;
        }

        hResult = psObj->hLocation;
    } while (FALSE);

    return hResult;
}

/*****************************************************************************
*
*   eDirection
*
*****************************************************************************/
static SMSAPI_RETURN_CODE_ENUM eDirection (
    SAFETY_CAMERAS_SUBLOCATION_OBJECT hSafeCamSubLoc,
    UN16* pun16Direction
        )
{
    SMSAPI_RETURN_CODE_ENUM eResult = SMSAPI_RETURN_CODE_ERROR;

    do
    {
        SAFECAM_SUBLOCATION_OBJECT_STRUCT *psObj =
            (SAFECAM_SUBLOCATION_OBJECT_STRUCT*)hSafeCamSubLoc;
        BOOLEAN bOwner = FALSE;

        if ((hSafeCamSubLoc == SAFETY_CAMERAS_SUBLOCATION_INVALID_OBJECT) ||
            (pun16Direction == NULL))
        {
            eResult = SMSAPI_RETURN_CODE_INVALID_INPUT;
            break;
        }

        bOwner = SMSO_bOwner((SMS_OBJECT)hSafeCamSubLoc);
        if (bOwner == FALSE)
        {
            eResult = SMSAPI_RETURN_CODE_NOT_OWNER;
            break;
        }

        *pun16Direction = psObj->un16Direction;
        eResult = SMSAPI_RETURN_CODE_SUCCESS;
    } while (FALSE);

    return eResult;
}

/*****************************************************************************
*
*   eHighSpeedLimit
*
*****************************************************************************/
static SMSAPI_RETURN_CODE_ENUM eHighSpeedLimit (
    SAFETY_CAMERAS_SUBLOCATION_OBJECT hSafeCamSubLoc,
    OSAL_FIXED_OBJECT hHighSpeedLimit
        )
{
    SMSAPI_RETURN_CODE_ENUM eResult = SMSAPI_RETURN_CODE_ERROR;
    OSAL_FIXED_OBJECT hKmToMiles = OSAL_FIXED_INVALID_OBJECT;

    do
    {
        SAFECAM_SUBLOCATION_OBJECT_STRUCT *psObj =
            (SAFECAM_SUBLOCATION_OBJECT_STRUCT*)hSafeCamSubLoc;
        BOOLEAN bOwner = FALSE, bSuccess = FALSE;
        OSAL_FIXED_OBJECT hTempSpeed = OSAL_FIXED_INVALID_OBJECT;
        OSAL_FIXED_OBJECT_DATA atSpeedFixedData[OSAL_FIXED_OBJECT_SIZE];

        if ((hHighSpeedLimit == OSAL_FIXED_INVALID_OBJECT) ||
            (hSafeCamSubLoc == SAFETY_CAMERAS_SUBLOCATION_INVALID_OBJECT))

        {
            eResult = SMSAPI_RETURN_CODE_INVALID_INPUT;
            break;
        }

        bOwner = SMSO_bOwner((SMS_OBJECT)hSafeCamSubLoc);
        if (bOwner == FALSE)
        {
            eResult = SMSAPI_RETURN_CODE_NOT_OWNER;
            break;
        }

        if (psObj->un16Speed1 == 0)
        {
            eResult = SMSAPI_RETURN_CODE_NOT_FOUND;
            break;
        }


        hTempSpeed = OSAL_FIXED.hCreateInMemory(psObj->un16Speed1,
            0, atSpeedFixedData
                );
        if (hTempSpeed == OSAL_FIXED_INVALID_OBJECT)
        {
            break;
        }

        if (psObj->bSpeedInMiles != TRUE)
        {
            OSAL_RETURN_CODE_ENUM eReturnCode = OSAL_ERROR;

            hKmToMiles = OSAL_FIXED.hCreate(SAFECAM_SUBLOCATION_KM_TO_MILES_WHOLE,
                SAFECAM_SUBLOCATION_KM_TO_MILES_FRAC, SAFECAM_SUBLOCATION_KM_TO_MILES_POW10
                    );
            if (hKmToMiles == OSAL_FIXED_INVALID_OBJECT)
            {
                break;
            }

            eReturnCode = OSAL_FIXED.eMultiply(hTempSpeed, hKmToMiles, hTempSpeed);
            if (eReturnCode != OSAL_SUCCESS)
            {
                break;
            }
        }

        bSuccess = OSAL_FIXED.bCopyToMemory(hTempSpeed, hHighSpeedLimit);
        if (bSuccess == FALSE)
        {
            break;
        }

        eResult = SMSAPI_RETURN_CODE_SUCCESS;
    } while (FALSE);

    if (hKmToMiles != OSAL_FIXED_INVALID_OBJECT)
    {
        OSAL_FIXED.vDestroy(hKmToMiles);
    }

    return eResult;
}

/*****************************************************************************
*
*   eLowSpeedLimit
*
*****************************************************************************/
static SMSAPI_RETURN_CODE_ENUM eLowSpeedLimit (
    SAFETY_CAMERAS_SUBLOCATION_OBJECT hSafeCamSubLoc,
    OSAL_FIXED_OBJECT hLowSpeedLimit
        )
{
    SMSAPI_RETURN_CODE_ENUM eResult = SMSAPI_RETURN_CODE_ERROR;
    OSAL_FIXED_OBJECT hKmToMiles = OSAL_FIXED_INVALID_OBJECT;

    do
    {
        SAFECAM_SUBLOCATION_OBJECT_STRUCT *psObj =
            (SAFECAM_SUBLOCATION_OBJECT_STRUCT*)hSafeCamSubLoc;
        BOOLEAN bOwner = FALSE, bSuccess = FALSE;
        OSAL_FIXED_OBJECT hTempSpeed = OSAL_FIXED_INVALID_OBJECT;
        OSAL_FIXED_OBJECT_DATA atSpeedFixedData[OSAL_FIXED_OBJECT_SIZE];

        if ((hLowSpeedLimit == OSAL_FIXED_INVALID_OBJECT) ||
            (hSafeCamSubLoc == SAFETY_CAMERAS_SUBLOCATION_INVALID_OBJECT))

        {
            eResult = SMSAPI_RETURN_CODE_INVALID_INPUT;
            break;
        }

        bOwner = SMSO_bOwner((SMS_OBJECT)hSafeCamSubLoc);
        if (bOwner == FALSE)
        {
            eResult = SMSAPI_RETURN_CODE_NOT_OWNER;
            break;
        }

        if (psObj->un16Speed2 == 0)
        {
            eResult = SMSAPI_RETURN_CODE_NOT_FOUND;
            break;
        }


        hTempSpeed = OSAL_FIXED.hCreateInMemory(psObj->un16Speed2,
            0, atSpeedFixedData
                );
        if (hTempSpeed == OSAL_FIXED_INVALID_OBJECT)
        {
            break;
        }

        if (psObj->bSpeedInMiles != TRUE)
        {
            OSAL_RETURN_CODE_ENUM eReturnCode = OSAL_ERROR;

            hKmToMiles = OSAL_FIXED.hCreate(SAFECAM_SUBLOCATION_KM_TO_MILES_WHOLE,
                SAFECAM_SUBLOCATION_KM_TO_MILES_FRAC, SAFECAM_SUBLOCATION_KM_TO_MILES_POW10
                    );
            if (hKmToMiles == OSAL_FIXED_INVALID_OBJECT)
            {
                break;
            }

            eReturnCode = OSAL_FIXED.eMultiply(hTempSpeed, hKmToMiles, hTempSpeed);
            if (eReturnCode != OSAL_SUCCESS)
            {
                break;
            }
        }

        bSuccess = OSAL_FIXED.bCopyToMemory(hTempSpeed, hLowSpeedLimit);
        if (bSuccess == FALSE)
        {
            break;
        }

        eResult = SMSAPI_RETURN_CODE_SUCCESS;
    } while (FALSE);

    if (hKmToMiles != OSAL_FIXED_INVALID_OBJECT)
    {
        OSAL_FIXED.vDestroy(hKmToMiles);
    }

    return eResult;
}

/*****************************************************************************
*
*   hAlertType
*
*****************************************************************************/
static SAFETY_CAMERAS_ALERT_ID tAlertType (
    SAFETY_CAMERAS_SUBLOCATION_OBJECT hSafeCamSubLoc
        )
{
    SAFETY_CAMERAS_ALERT_ID tResult = SAFETY_CAMERAS_ALERT_INVALID_ID;

    do
    {
        SAFECAM_SUBLOCATION_OBJECT_STRUCT *psObj =
            (SAFECAM_SUBLOCATION_OBJECT_STRUCT*)hSafeCamSubLoc;
        BOOLEAN bOwner = FALSE;

        if (hSafeCamSubLoc == SAFETY_CAMERAS_SUBLOCATION_INVALID_OBJECT)

        {
            break;
        }

        bOwner = SMSO_bOwner((SMS_OBJECT)hSafeCamSubLoc);
        if (bOwner == FALSE)
        {
            break;
        }

        tResult = psObj->tAlertID;
    } while (FALSE);

    return tResult;
}

/*****************************************************************************
*
*   n32FPrintf
*
*****************************************************************************/
static N32 n32FPrintf (
    SAFETY_CAMERAS_SUBLOCATION_OBJECT hSafeCamSubLoc,
    FILE *psFile,
    SMSAPI_OUTPUT_OPTION_ENUM eOutputOption
        )
{
    N32 n32Return = EOF;
    BOOLEAN bOwner = FALSE;

    // Determine if the handle is valid
    bOwner = SMSO_bOwner((SMS_OBJECT)hSafeCamSubLoc);

    if ((bOwner == TRUE) && (psFile != NULL))
    {
        SAFECAM_SUBLOCATION_OBJECT_STRUCT *psObj =
            (SAFECAM_SUBLOCATION_OBJECT_STRUCT *)hSafeCamSubLoc;
        OSAL_FIXED_OBJECT hLat = OSAL_FIXED_INVALID_OBJECT,
            hLon = OSAL_FIXED_INVALID_OBJECT;
        N32 n32Temp = 0;

        // Print SAFECAM_SUBLOCATION information header
        n32Return = fprintf(psFile, "SAFECAM_SUBLOCATION:"
            "\n\thSafeCamSubLoc = %p\n", psObj
                );
        n32Return += fprintf(psFile, "\tun8Id = %d\n",
            psObj->un8Id
                );
        n32Return += fprintf(psFile, "\tun16Direction = %d\n",
            psObj->un16Direction
                );
        n32Return += fprintf(psFile, "\tCoordinates are:\n");

        hLat = LOCATION.hLat(psObj->hLocation);
        if (hLat != OSAL_FIXED_INVALID_OBJECT)
        {
            n32Return += fprintf(psFile, "\tLat:\n");
            n32Temp = OSAL_FIXED.n32FPrintf(hLat, psFile, FALSE);
            n32Return += (n32Temp>0)? n32Temp : 0;
        }

        hLon = LOCATION.hLon(psObj->hLocation);
        if (hLon != OSAL_FIXED_INVALID_OBJECT)
        {
            n32Return += fprintf(psFile, "\n\tLon:\n");
            n32Temp = OSAL_FIXED.n32FPrintf(hLon, psFile, FALSE);
            n32Return += (n32Temp>0)? n32Temp : 0;
        }

        n32Return += fprintf(psFile, "\n\tSpeed in Miles: %s\n",
            ((psObj->bSpeedInMiles == 1) ? "YES" : "NO")
                );
        n32Return += fprintf(psFile, "\tSpeed1: %d\n", psObj->un16Speed1);
        n32Return += fprintf(psFile, "\tSpeed2: %d\n", psObj->un16Speed2);
        n32Return += fprintf(psFile, "\tAlert Type: %d\n", psObj->tAlertID);
    }

    return n32Return;
}

/*****************************************************************************
                             FRIEND FUNCTIONS
*****************************************************************************/

/*****************************************************************************
*
*   SAFECAM_SUBLOCATION_hCreate
*
*****************************************************************************/
SAFETY_CAMERAS_SUBLOCATION_OBJECT SAFECAM_SUBLOCATION_hCreate (
    SMS_OBJECT hParent,
    SAFECAM_SUB_LOCATIONS_ROW_STRUCT *psSubLocationRow,
    BOOLEAN bSpeedInMiles,
    SAFETY_CAMERAS_ALERT_ID tAlertType
        )
{
    SAFECAM_SUBLOCATION_OBJECT_STRUCT *psObj =
        (SAFECAM_SUBLOCATION_OBJECT_STRUCT *)
            SAFETY_CAMERAS_SUBLOCATION_INVALID_OBJECT;

    do
    {
        if (psSubLocationRow == NULL)
        {
            break;
        }

        // Create an instance of the SAFECAM_SUBLOCATION object
        psObj = (SAFECAM_SUBLOCATION_OBJECT_STRUCT *)
            SMSO_hCreate(SAFECAM_SUBLOCATION_OBJECT_NAME":Data",
            sizeof(SAFECAM_SUBLOCATION_OBJECT_STRUCT), hParent, FALSE
                );

        if(psObj == NULL)
        {
            // Error!
            break;
        }

        psObj->tAlertID = tAlertType;
        psObj->un16Direction = psSubLocationRow->un16Dir;
        psObj->un8Id = psSubLocationRow->un8SubID;
        psObj->hLocation = LOCATION_hCreate(hParent, LOCID_INVALID_OBJECT,
            psSubLocationRow->n32Lat, LOCATION_BINPOINT,
            psSubLocationRow->n32Lon, LOCATION_BINPOINT,
            DISTANCE_INVALID_OBJECT, (LOCATION_ATTRIBUTE_STRUCT*)NULL, FALSE
                );
        psObj->bSpeedInMiles = bSpeedInMiles;
        psObj->un16Speed1 = psSubLocationRow->un16Speed1;
        psObj->un16Speed2 = psSubLocationRow->un16Speed2;

        return (SAFETY_CAMERAS_SUBLOCATION_OBJECT)psObj;

    } while (FALSE);

    vDestroyObject(psObj);

    return SAFETY_CAMERAS_SUBLOCATION_INVALID_OBJECT;
}

/*****************************************************************************
*
*   SAFECAM_SUBLOCATION_vDestroy
*
*****************************************************************************/
void SAFECAM_SUBLOCATION_vDestroy (
    SAFETY_CAMERAS_SUBLOCATION_OBJECT hSafeCamSubLoc
        )
{
    BOOLEAN bOwner = FALSE;

    bOwner = SMSO_bOwner((SMS_OBJECT)hSafeCamSubLoc);
    if(bOwner == TRUE)
    {
        SAFECAM_SUBLOCATION_OBJECT_STRUCT *psObj =
            (SAFECAM_SUBLOCATION_OBJECT_STRUCT*)hSafeCamSubLoc;
        vDestroyObject(psObj);
    }

    return;
}

/*****************************************************************************
*
*   SAFECAM_SUBLOCATION_un8ID
*
*****************************************************************************/
UN8 SAFECAM_SUBLOCATION_un8ID(
    SAFETY_CAMERAS_SUBLOCATION_OBJECT hSafeCamSubLoc
        )
{
    BOOLEAN bOwner = FALSE;
    UN8 un8Result = 0;

    bOwner = SMSO_bOwner((SMS_OBJECT)hSafeCamSubLoc);
    if(bOwner == TRUE)
    {
        SAFECAM_SUBLOCATION_OBJECT_STRUCT *psObj =
            (SAFECAM_SUBLOCATION_OBJECT_STRUCT*)hSafeCamSubLoc;
        un8Result = psObj->un8Id;
    }

    return un8Result;
}

/*****************************************************************************
                             PRIVATE FUNCTIONS
*****************************************************************************/

/*****************************************************************************
*
*   vDestroyObject
*
*****************************************************************************/
static void vDestroyObject(SAFECAM_SUBLOCATION_OBJECT_STRUCT *psObj)
{
    if (psObj != NULL)
    {
        if (psObj->hLocation != LOCATION_INVALID_OBJECT)
        {
            LOCATION.vDestroy(psObj->hLocation);
        }
        // Free object instance itself
        SMSO_vDestroy((SMS_OBJECT)psObj);
    }

    return;
}
