/******************************************************************************/
/*                    Copyright (c) Sirius XM Radio, Inc.                     */
/*                            All Rights Reserved                             */
/*            Licensed Materials - Property of Sirius XM Radio, Inc.          */
/******************************************************************************/
/*******************************************************************************
 *
 * DESCRIPTION
 *
 *  This module contains the Object:SAFECAM_LOCATION implementation for the
 *  Sirius Module Services (SMS)
 *
 ******************************************************************************/
#include "standard.h"
#include "osal.h"

#include "sms_api.h"
#include "sms_obj.h"
#include "_safecam_location_obj.h"
#include "safecam_location_obj.h"
#include "dsrl_entry_obj.h"
#include "string_obj.h"
#include "location_obj.h"
#include "locid_obj.h"
#include "safecam_sublocation_obj.h"
#include "safecam_mgr_obj.h"

#include "sms_api_debug.h"
static const char *gpacThisFile = __FILE__;

/*****************************************************************************
                             PUBLIC FUNCTIONS
*****************************************************************************/

/*****************************************************************************
*
*   eGetPoint
*
*****************************************************************************/
static LOCATION_OBJECT hLocation (
    SAFETY_CAMERAS_LOCATION_OBJECT hSafeCamLoc
        )
{
    LOCATION_OBJECT hResult = LOCATION_INVALID_OBJECT;

    do
    {
        SAFECAM_LOCATION_OBJECT_STRUCT *psObj =
            (SAFECAM_LOCATION_OBJECT_STRUCT*)hSafeCamLoc;
        BOOLEAN bOwner = FALSE;

        if (hSafeCamLoc == SAFETY_CAMERAS_LOCATION_INVALID_OBJECT)

        {
            break;
        }

        bOwner = DSRL_ENTRY_bOwner((DSRL_ENTRY_OBJECT)hSafeCamLoc);
        if (bOwner == FALSE)
        {
            break;
        }

        hResult = psObj->hLocation;
    } while (FALSE);

    return hResult;
}

/*****************************************************************************
*
*   eIterateSubLocations
*
*****************************************************************************/
static SMSAPI_RETURN_CODE_ENUM eIterateSubLocations (
    SAFETY_CAMERAS_LOCATION_OBJECT hSafeCamLoc,
    SAFETY_CAMERAS_SUBLOCATIONS_ITERATOR_CALLBACK bCallback,
    void *pvCallbackArg
        )
{
    SMSAPI_RETURN_CODE_ENUM eResult = SMSAPI_RETURN_CODE_ERROR;

    do
    {
        SAFECAM_LOCATION_OBJECT_STRUCT *psObj =
            (SAFECAM_LOCATION_OBJECT_STRUCT*)hSafeCamLoc;
        BOOLEAN bOwner = FALSE;
        OSAL_RETURN_CODE_ENUM eReturnCode = OSAL_ERROR;

        if ((bCallback == NULL) ||
            (hSafeCamLoc == SAFETY_CAMERAS_LOCATION_INVALID_OBJECT))

        {
            eResult = SMSAPI_RETURN_CODE_INVALID_INPUT;
            break;
        }

        bOwner = DSRL_ENTRY_bOwner((DSRL_ENTRY_OBJECT)hSafeCamLoc);
        if (bOwner == FALSE)
        {
            eResult = SMSAPI_RETURN_CODE_NOT_OWNER;
            break;
        }

        if (psObj->hSubLocations == OSAL_INVALID_OBJECT_HDL)
        {
            eResult = SMSAPI_RETURN_CODE_NO_OBJECTS;
            break;
        }

        eReturnCode = OSAL.eLinkedListIterate(psObj->hSubLocations,
            (OSAL_LL_ITERATOR_HANDLER)bCallback, pvCallbackArg
                );
        if (eReturnCode == OSAL_NO_OBJECTS)
        {
            eResult = SMSAPI_RETURN_CODE_NO_OBJECTS;
            break;
        }
        else if (eReturnCode != OSAL_SUCCESS)
        {
            break;
        }

        eResult = SMSAPI_RETURN_CODE_SUCCESS;
    } while (FALSE);

    return eResult;
}

/*****************************************************************************
*
*   n32FPrintf
*
*****************************************************************************/
static N32 n32FPrintf (
    SAFETY_CAMERAS_LOCATION_OBJECT hSafeCamLoc,
    FILE *psFile,
    SMSAPI_OUTPUT_OPTION_ENUM eOutputOption
        )
{
    N32 n32Return = EOF;
    BOOLEAN bOwner = FALSE;

    // Determine if the handle is valid
    bOwner = DSRL_ENTRY_bOwner((DSRL_ENTRY_OBJECT)hSafeCamLoc);

    if ((bOwner == TRUE) && (psFile != NULL))
    {
        SAFECAM_LOCATION_OBJECT_STRUCT *psObj =
            (SAFECAM_LOCATION_OBJECT_STRUCT*)hSafeCamLoc;
        N32 n32Temp = 0;

        // Print SAFECAM_LOCATION information header
        n32Return = fprintf(psFile, "SAFECAM_LOCATION:\n\thSafeCamLoc = %p\n",
            psObj);

        n32Return += fprintf(psFile, "\tun32UID = %x\n",
            psObj->un32UID);

        if (psObj->hLocation != LOCATION_INVALID_OBJECT)
        {
            STRING_OBJECT hDescription = STRING_INVALID_OBJECT;
            OSAL_FIXED_OBJECT hTempLat = OSAL_FIXED_INVALID_OBJECT,
                hTempLon = OSAL_FIXED_INVALID_OBJECT;
            LOCID_OBJECT hLocID;

            n32Return += fprintf(psFile, "\tDescription is ");

            hDescription = LOCATION.hDescription(psObj->hLocation);

            n32Temp = STRING.n32FWrite(hDescription, psFile);
            n32Return += (n32Temp>0)? n32Temp : 0;
            n32Return += fprintf(psFile, "\n");

            n32Return += fprintf(psFile, "\tCoordinates are:\n");
            hTempLat = LOCATION.hLat(psObj->hLocation);
            if (hTempLat != OSAL_FIXED_INVALID_OBJECT)
            {
                n32Return += fprintf(psFile, "\tLat:\n");
                n32Temp = OSAL_FIXED.n32FPrintf(hTempLat, psFile, FALSE);
                n32Return += (n32Temp>0)? n32Temp : 0;
            }

            hTempLon = LOCATION.hLon(psObj->hLocation);
            if (hTempLon != OSAL_FIXED_INVALID_OBJECT)
            {
                n32Return += fprintf(psFile, "\n\tLon:\n");
                n32Temp = OSAL_FIXED.n32FPrintf(hTempLon, psFile, FALSE);
                n32Return += (n32Temp>0)? n32Temp : 0;
            }

            hLocID = LOCATION.hLocID(psObj->hLocation);
            if (hLocID != LOCID_INVALID_OBJECT)
            {
                n32Return += fprintf(psFile, "\n\tLocID:\n");
                n32Temp = LOCID.n32FPrintf(hLocID, psFile);
                n32Return += (n32Temp>0)? n32Temp : 0;
            }
        }

        n32Return += fprintf(psFile, "\n\t %d sub Locations:\n",
            psObj->un8SubNum
                );

        if (psObj->hSubLocations != OSAL_INVALID_OBJECT_HDL)
        {
            OSAL_LINKED_LIST_ENTRY hEntry = OSAL_INVALID_LINKED_LIST_ENTRY;
            SAFETY_CAMERAS_SUBLOCATION_OBJECT hSubLocation =
                SAFETY_CAMERAS_SUBLOCATION_INVALID_OBJECT;
            N32 n32Temp = 0;

            hEntry = OSAL.hLinkedListFirst(psObj->hSubLocations,
                (void**)&hSubLocation
                    );
            while ((hEntry != OSAL_INVALID_LINKED_LIST_ENTRY) &&
                (hSubLocation != SAFETY_CAMERAS_SUBLOCATION_INVALID_OBJECT))
            {
                n32Temp = SAFETY_CAMERAS_SUBLOCATION.n32FPrintf(hSubLocation,
                    psFile, eOutputOption
                        );
                if (n32Temp > 0)
                {
                    n32Return += n32Temp;
                }

                hEntry = OSAL.hLinkedListNext(hEntry, (void**)&hSubLocation);
            }
        }
    }

    return n32Return;
}

/*****************************************************************************
                             FRIEND FUNCTIONS
*****************************************************************************/

/*****************************************************************************
*
*   SAFECAM_LOCATION_hCreate
*
*****************************************************************************/
SAFETY_CAMERAS_LOCATION_OBJECT SAFECAM_LOCATION_hCreate (
    SMS_OBJECT hParent,
    SAFECAM_ALERT_LOCATIONS_ROW_STRUCT *psAlertLocationRow,
    size_t tDescriptorSize
        )
{
    SAFECAM_LOCATION_OBJECT_STRUCT *psObj =
        (SAFECAM_LOCATION_OBJECT_STRUCT*)SAFETY_CAMERAS_LOCATION_INVALID_OBJECT;

    do
    {
        LOCATION_ATTRIBUTE_STRUCT sLocAttrs;
        LOCID_OBJECT hLocId;

        if (psAlertLocationRow == NULL)
        {
            break;
        }

        OSAL.bMemSet(&sLocAttrs, 0, sizeof(sLocAttrs));

        // Create an instance of the SAFECAM_LOCATION object
        psObj = (SAFECAM_LOCATION_OBJECT_STRUCT*)
            DSRL_ENTRY_hCreate(
                SAFECAM_LOCATION_OBJECT_NAME":Data",
                DSRL_ENTRY_TYPE_SAFETY_CAMERAS_LOCATION,
                sizeof(SAFECAM_LOCATION_OBJECT_STRUCT),
                tDescriptorSize, hParent, TRUE);

        if(psObj == NULL)
        {
            // Error!
            break;
        }

        // Set our attributes
        psObj->un8SubNum = psAlertLocationRow->un8NoSub;
        psObj->un32UID = psAlertLocationRow->un32UID;
        psObj->bSpeedInMiles = psAlertLocationRow->bMilesUnits;

        // Create a loc id for this location
        hLocId = LOCID_hCreate((LOC_ID)psObj->un32UID,
            LOCID_TYPE_SAFETY_CAMERAS,
            (const LOCID_INTERFACE_STRUCT *)NULL);

        if (hLocId == LOCID_INVALID_OBJECT)
        {
            // Error!
            break;
        }

        // Set the description appropriately
        sLocAttrs.hDescription = psAlertLocationRow->hInfo;
        psObj->hLocation = LOCATION_hCreate(hParent, hLocId,
            psAlertLocationRow->n32Lat, LOCATION_BINPOINT,
            psAlertLocationRow->n32Lon, LOCATION_BINPOINT,
            DISTANCE_INVALID_OBJECT, &sLocAttrs, FALSE
                );

        if (psObj->hLocation == LOCATION_INVALID_OBJECT)
        {
            LOCID.vDestroy(hLocId);
            break;
        }

        // cleanup the string handle to prevent the caller from
        // destroying this string
        psAlertLocationRow->hInfo = STRING_INVALID_OBJECT;

        return (SAFETY_CAMERAS_LOCATION_OBJECT)psObj;
    } while (FALSE);

    vDestroyObject(psObj);

    return SAFETY_CAMERAS_LOCATION_INVALID_OBJECT;
}

/*****************************************************************************
*
*   SAFECAM_LOCATION_hCreateDummy
*
*****************************************************************************/
SAFETY_CAMERAS_LOCATION_OBJECT SAFECAM_LOCATION_hCreateDummy (
    SMS_OBJECT hParent
        )
{
    SAFECAM_LOCATION_OBJECT_STRUCT *psObj =
        (SAFECAM_LOCATION_OBJECT_STRUCT*)SAFETY_CAMERAS_LOCATION_INVALID_OBJECT;

    // Create an instance of the SAFECAM_LOCATION object
    psObj = (SAFECAM_LOCATION_OBJECT_STRUCT *)
        DSRL_ENTRY_hCreate(
        SAFECAM_LOCATION_OBJECT_NAME":Data",
        DSRL_ENTRY_TYPE_SAFETY_CAMERAS_LOCATION,
        sizeof(SAFECAM_LOCATION_OBJECT_STRUCT),
        0,
        hParent,
        TRUE);

    return (SAFETY_CAMERAS_LOCATION_OBJECT)psObj;
}

/*****************************************************************************
*
*   SAFECAM_LOCATION_vDestroy
*
*****************************************************************************/
void SAFECAM_LOCATION_vDestroy (
    SAFETY_CAMERAS_LOCATION_OBJECT hSafeCamLoc
        )
{
    BOOLEAN bOwner = FALSE;

    bOwner = DSRL_ENTRY_bOwner((DSRL_ENTRY_OBJECT)hSafeCamLoc);
    if(bOwner == TRUE)
    {
        SAFECAM_LOCATION_OBJECT_STRUCT *psObj =
            (SAFECAM_LOCATION_OBJECT_STRUCT*)hSafeCamLoc;
        vDestroyObject(psObj);
    }

    return;
}

/*****************************************************************************
*
*   SAFECAM_LOCATION_bSetUID
*
*****************************************************************************/
BOOLEAN SAFECAM_LOCATION_bSetUID (
    SAFETY_CAMERAS_LOCATION_OBJECT hSafeCamLoc,
    UN32 un32UID
        )
{
    BOOLEAN bOwner = FALSE;

    bOwner = DSRL_ENTRY_bOwner((DSRL_ENTRY_OBJECT)hSafeCamLoc);
    if(bOwner == TRUE)
    {
        SAFECAM_LOCATION_OBJECT_STRUCT *psObj =
            (SAFECAM_LOCATION_OBJECT_STRUCT *)hSafeCamLoc;

        psObj->un32UID = un32UID;
    }

    return bOwner;
}

/*****************************************************************************
*
*   SAFECAM_LOCATION_bSetSubLocation
*
*****************************************************************************/
BOOLEAN SAFECAM_LOCATION_bSetSubLocation(
    SAFETY_CAMERAS_LOCATION_OBJECT hSafeCamLoc,
    SAFETY_CAMERAS_SUBLOCATION_OBJECT hSafeCamSubLoc
        )
{
    BOOLEAN bResult = FALSE;

    do
    {
        BOOLEAN bOwner = FALSE;
        SAFECAM_LOCATION_OBJECT_STRUCT *psObj =
            (SAFECAM_LOCATION_OBJECT_STRUCT*) hSafeCamLoc;
        OSAL_RETURN_CODE_ENUM eReturnCode = OSAL_ERROR;

        bOwner = DSRL_ENTRY_bOwner((DSRL_ENTRY_OBJECT)hSafeCamLoc);

        if ((bOwner == FALSE) ||
            (hSafeCamSubLoc == SAFETY_CAMERAS_SUBLOCATION_INVALID_OBJECT))
        {
            break;
        }

        if (psObj->hSubLocations == OSAL_INVALID_OBJECT_HDL)
        {
            eReturnCode = OSAL.eLinkedListCreate(&psObj->hSubLocations,
                SAFECAM_LOCATION_OBJECT_NAME"SubLocsList",
                (OSAL_LL_COMPARE_HANDLER)NULL,
                OSAL_LL_OPTION_LINEAR|OSAL_LL_OPTION_UNIQUE
                    );
            if (eReturnCode != OSAL_SUCCESS)
            {
                SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    SAFECAM_LOCATION_OBJECT_NAME": Unable to create list (%s)",
                    OSAL.pacGetReturnCodeName(eReturnCode)
                        );
                break;
            }
        }

        eReturnCode = OSAL.eLinkedListAdd(psObj->hSubLocations,
            OSAL_INVALID_LINKED_LIST_ENTRY_PTR, hSafeCamSubLoc
                );
        if (eReturnCode != OSAL_SUCCESS)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                SAFECAM_LOCATION_OBJECT_NAME": Unable to add item "
                "to list (%s)", OSAL.pacGetReturnCodeName(eReturnCode)
                    );
            break;
        }

        bResult = TRUE;
    } while (FALSE);

    return bResult;
}

/*****************************************************************************
*
*   SAFECAM_LOCATION_un32UID
*
*****************************************************************************/
UN32 SAFECAM_LOCATION_un32UID (
    SAFETY_CAMERAS_LOCATION_OBJECT hSafeCamLoc
        )
{
    BOOLEAN bOwner = FALSE;
    UN32 un32UID = 0;

    bOwner = DSRL_ENTRY_bOwner((DSRL_ENTRY_OBJECT)hSafeCamLoc);
    if(bOwner == TRUE)
    {
        SAFECAM_LOCATION_OBJECT_STRUCT *psObj =
            (SAFECAM_LOCATION_OBJECT_STRUCT*)hSafeCamLoc;

        un32UID = psObj->un32UID;
    }

    return un32UID;
}

/*****************************************************************************
*
*   SAFECAM_LOCATION_bInitCompleted
*
*****************************************************************************/
BOOLEAN SAFECAM_LOCATION_bInitCompleted (
    SAFETY_CAMERAS_LOCATION_OBJECT hSafeCamLoc
        )
{
    BOOLEAN bInitCompleted = FALSE, bOwner = FALSE;


    bOwner = DSRL_ENTRY_bOwner((DSRL_ENTRY_OBJECT)hSafeCamLoc);
    if(bOwner == TRUE)
    {
        SAFECAM_LOCATION_OBJECT_STRUCT *psObj =
            (SAFECAM_LOCATION_OBJECT_STRUCT*)hSafeCamLoc;

        bInitCompleted = psObj->bInitCompleted;
    }

    return bInitCompleted;
}

/*****************************************************************************
*
*   SAFECAM_LOCATION_vSetInitCompleted
*
*****************************************************************************/
void SAFECAM_LOCATION_vSetInitCompleted (
    SAFETY_CAMERAS_LOCATION_OBJECT hSafeCamLoc
        )
{
    BOOLEAN bOwner = FALSE;

    bOwner = DSRL_ENTRY_bOwner((DSRL_ENTRY_OBJECT)hSafeCamLoc);
    if(bOwner == TRUE)
    {
        SAFECAM_LOCATION_OBJECT_STRUCT *psObj =
            (SAFECAM_LOCATION_OBJECT_STRUCT*)hSafeCamLoc;

        psObj->bInitCompleted = TRUE;
    }

    return;
}

/*****************************************************************************
*
*   SAFECAM_LOCATION_bSpeedInMiles
*
*****************************************************************************/
BOOLEAN SAFECAM_LOCATION_bSpeedInMiles (
    SAFETY_CAMERAS_LOCATION_OBJECT hSafeCamLoc
        )
{
    BOOLEAN bSpeedInMiles = FALSE, bOwner = FALSE;


    bOwner = DSRL_ENTRY_bOwner((DSRL_ENTRY_OBJECT)hSafeCamLoc);
    if(bOwner == TRUE)
    {
        SAFECAM_LOCATION_OBJECT_STRUCT *psObj =
            (SAFECAM_LOCATION_OBJECT_STRUCT*)hSafeCamLoc;

        bSpeedInMiles = psObj->bSpeedInMiles;
    }

    return bSpeedInMiles;
}

/*****************************************************************************
                             PRIVATE FUNCTIONS
*****************************************************************************/

/*****************************************************************************
*
*   vDestroyObject
*
*****************************************************************************/
static void vDestroyObject(SAFECAM_LOCATION_OBJECT_STRUCT *psObj)
{
    if (psObj != NULL)
    {
        if (psObj->hLocation != LOCATION_INVALID_OBJECT)
        {
            LOCATION.vDestroy(psObj->hLocation);
            psObj->hLocation = LOCATION_INVALID_OBJECT;
        }

        if (psObj->hSubLocations != OSAL_INVALID_OBJECT_HDL)
        {
            OSAL_RETURN_CODE_ENUM eReturnCode = OSAL_ERROR;

            eReturnCode = OSAL.eLinkedListRemoveAll(psObj->hSubLocations,
                (OSAL_LL_RELEASE_HANDLER)SAFECAM_SUBLOCATION_vDestroy
                    );
            if (eReturnCode != OSAL_SUCCESS)
            {
                SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    SAFECAM_LOCATION_OBJECT_NAME": Unable to remove items "
                    "from list (%s)", OSAL.pacGetReturnCodeName(eReturnCode)
                        );
            }

            eReturnCode = OSAL.eLinkedListDelete(psObj->hSubLocations);
            if (eReturnCode != OSAL_SUCCESS)
            {
                SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    SAFECAM_LOCATION_OBJECT_NAME": Unable to delete list (%s)",
                    OSAL.pacGetReturnCodeName(eReturnCode)
                        );
            }

            psObj->hSubLocations = OSAL_INVALID_OBJECT_HDL;
        }

        // Free object instance itself
        DSRL_ENTRY_vDestroy((DSRL_ENTRY_OBJECT)psObj);
    }

    return;
}
