/******************************************************************************/
/*                    Copyright (c) Sirius XM Radio, Inc.                     */
/*                            All Rights Reserved                             */
/*            Licensed Materials - Property of Sirius XM Radio, Inc.          */
/******************************************************************************/

/*******************************************************************************
 *
 * DESCRIPTION
 *
 * This file defines the constants used when utilizing the Safety Camera
 * database file
 *
 ******************************************************************************/

  /*********************************/
 /** PREVENT REDUNDANT INCLUSION **/
/*********************************/

#ifndef SAFECAM_DB_CONSTANTS_H_
#define SAFECAM_DB_CONSTANTS_H_


  /**************/
 /** INCLUDES **/
/**************/

#include "sms_api.h"


  /**********************/
 /** GLOBAL CONSTANTS **/
/**********************/

/* Database File Name */
#define SAFECAM_DATABASE_FOLDER "safecam"
#define SAFECAM_REF_DATABASE_FILENAMEA "safecam_ref.db"
#define SAFECAM_REF_DATABASE_FILENAMEB "safecam_ref.b.db"

/* Database Table Attributes */
#define SAFECAM_DATABASE_FILE_VERSION (4)

/* Database Table Names */
#define DATABASE_VERSION_TABLE_NAME "db_version"
#define PARITIONS_TABLE_NAME "partitions"
#define ALERT_LOCATIONS_TABLE_NAME "alert_locations"
#define SUB_LOCATIONS_TABLE_NAME "sub_locations"
#define ALERT_TYPES_TABLE_NAME "alert_types"
#define LOCATIONS_RTREE_TABLE_NAME "locations_rtree"

/*  set fixed_constant [expr 1 << $binpoint]
    set rtree_range [expr .000015 * $fixed_constant] */
#define SAFECAM_RTREE_RANGE      ((N32)62)

// Defines speed fields binpoint placement
#define SAFECAM_SPEED_BINPOINT (1)

#define SAFECAM_MAX_SQL_STRING_LENGTH (620)

/* Database Queries */

// Select alert_locations db schema version
#define SAFECAM_SELECT_DB_SCHEMA_VERSION \
    "select version from "DATABASE_VERSION_TABLE_NAME" where type=0;"

// Select alert_locations db schema version and dsi
#define SAFECAM_SELECT_DB_SCHEMA_VERSION_DSI \
    "select version, dsi from "DATABASE_VERSION_TABLE_NAME" where type=0;"

// Select alert_locations db data version
#define SAFECAM_SELECT_DB_DATA_VERSION \
    "select version from "DATABASE_VERSION_TABLE_NAME" where type=1;"

// Select alert_locations db update number
#define SAFECAM_SELECT_UPDATE_NUMBER \
    "select version from "DATABASE_VERSION_TABLE_NAME" where type=2;"

// Select alert_locations db next update number
#define SAFECAM_SELECT_NEXT_UPDATE_NUMBER \
    "select version from "DATABASE_VERSION_TABLE_NAME" where type=3;"

// Select alert locations by coordinates
#define SAFECAM_SELECT_ALERT_LOCATIONS_BY_COORDINATES \
    "select * from "ALERT_LOCATIONS_TABLE_NAME" where uid in " \
    "(select uid from "LOCATIONS_RTREE_TABLE_NAME" where " \
    LOCATIONS_RTREE_TABLE_NAME".minLat>=%d and " \
    LOCATIONS_RTREE_TABLE_NAME".maxLat<=%d and " \
    LOCATIONS_RTREE_TABLE_NAME".minLon>=%d and " \
    LOCATIONS_RTREE_TABLE_NAME".maxLon<=%d) order by uid;"

// Select sub locations coordinates by UID
#define SAFECAM_SELECT_SUB_LOCATIONS_COORDS_BY_UID \
    "select lat, lon from "SUB_LOCATIONS_TABLE_NAME" where uid=%u;"

// Select sub locations by UID
#define SAFECAM_SELECT_SUBLOC_STMT_UID_PARAM    0
#define SAFECAM_SELECT_SUBLOC_STMT_PARAMS_COUNT 1

#define SAFECAM_SELECT_SUB_LOCATIONS_BY_UID \
    "select * from "SUB_LOCATIONS_TABLE_NAME" where uid=?;"

// Insert alert location record to alert locations table
#define SAFECAM_INSERT_ALERT_LOCATION \
    "insert or replace into "ALERT_LOCATIONS_TABLE_NAME \
    " (uid, lat, lon, info, nosub, unit) values (?, ?, ?,"\
    " ?, ?, ?);"

// Insert alert type record to alert types table
#define SAFECAM_INSERT_ALERT_TYPE \
    "insert or replace into "ALERT_TYPES_TABLE_NAME \
    " (type, desc) values (?, ?);"

// Insert sub location record
#define SAFECAM_INSERT_SUB_LOCATION \
    "insert or replace into "SUB_LOCATIONS_TABLE_NAME \
    " (sub_id, uid, type, lat, lon, dir, speed1, speed2) values (?, ?, ?,"\
    " ?, ?, ?, ?, ?);"

// Insert rtree for alert location
#define SAFECAM_INSERT_LOCATION_RTREE \
    "insert or replace into "LOCATIONS_RTREE_TABLE_NAME \
    " values(%u, %i, %i, %i, %i);"

// Remove alert type record with some type
#define SAFECAM_DELETE_ALERT_TYPE \
    "delete from "ALERT_TYPES_TABLE_NAME" where type=%u;"

// Remove sub location records with some uid
#define SAFECAM_DELETE_ALL_SUB_LOCATIONS \
    "delete from "SUB_LOCATIONS_TABLE_NAME" where uid=%u;"

// Remove alert location record with some uid
#define SAFECAM_DELETE_ALERT_LOCATION \
    "delete from "ALERT_LOCATIONS_TABLE_NAME" where uid=%u;"

// Remove alert location record from rtree table
#define SAFECAM_DELETE_ALERT_LOCATION_RTREE \
    "delete from "LOCATIONS_RTREE_TABLE_NAME" where uid=%u;"

// Select alert location record by uid
#define SAFECAM_SELECT_ALERT_LOCATION_BY_UID \
    "select * from "ALERT_LOCATIONS_TABLE_NAME" where uid=%u;"

// Select alert location record by uid
#define SAFECAM_SELECT_ALERT_LOC_STMT_MIN_LAT_PARAM 0
#define SAFECAM_SELECT_ALERT_LOC_STMT_MAX_LAT_PARAM 1
#define SAFECAM_SELECT_ALERT_LOC_STMT_MIN_LON_PARAM 2
#define SAFECAM_SELECT_ALERT_LOC_STMT_MAX_LON_PARAM 3
#define SAFECAM_SELECT_ALERT_LOC_STMT_PARAMS_COUNT  4

#define SAFECAM_SELECT_ALERT_LOCATIONS_BY_LOCATION \
	"select * from "ALERT_LOCATIONS_TABLE_NAME" where uid in " \
	"(select uid from "LOCATIONS_RTREE_TABLE_NAME" where " \
	LOCATIONS_RTREE_TABLE_NAME".min_lat >= ? and " \
	LOCATIONS_RTREE_TABLE_NAME".max_lat <= ? and " \
	LOCATIONS_RTREE_TABLE_NAME".min_lon >= ? and " \
	LOCATIONS_RTREE_TABLE_NAME".max_lon <= ?) order by uid;"

// Select alert type by type id
#define SAFECAM_SELECT_ALERT_TYPE_BY_TYPE \
	"select * from "ALERT_TYPES_TABLE_NAME" where type=%u;"

// Select all alert types
#define SAFECAM_SELECT_ALL_ALERT_TYPES \
	"select * from "ALERT_TYPES_TABLE_NAME";"


// Select sub location record by uid and sub id
#define SAFECAM_SELECT_SUB_LOCATION_BY_UID_SUB_ID \
    "select * from "SUB_LOCATIONS_TABLE_NAME" where uid=%u and"\
    " sub_id=%u;"

// Remove sublocation record by id and uid
#define SAFECAM_DELETE_SUBLOCATION_BY_ID_UID \
	"delete from "SUB_LOCATIONS_TABLE_NAME" where sub_id=%u and"\
	" uid=%u;"

// Get count of alert_types with some type
#define SAFECAM_GET_COUNT_FOR_ALERT_TYPE \
    "select count(type) from "ALERT_TYPES_TABLE_NAME" where" \
    " type=%u;"

// Update version field for data
#define SAFECAM_UPDATE_DB_DATA_VERSION \
    "update "DATABASE_VERSION_TABLE_NAME" set version=%u where type=1;"

// Change update number field
#define SAFECAM_SET_UPDATE_NUMBER \
    "update "DATABASE_VERSION_TABLE_NAME" set version=%d where type=2;"

// Change next update number field
#define SAFECAM_SET_NEXT_UPDATE_NUMBER \
    "update "DATABASE_VERSION_TABLE_NAME" set version=%d where type=3;"


  /*********************/
 /** GLOBAL TYPEDEFS **/
/*********************/

typedef enum safecam_db_version_type_enum {
    SAFECAM_DB_VERSION_TYPE_SCHEMA = 0,
    SAFECAM_DB_VERSION_TYPE_DATA
} SAFECAM_DB_VERSION_TYPE_ENUM;

// Enumeration specifying all the available fields
// in the db_version table
typedef enum safecam_db_version_fields_enum {
    SAFECAM_DB_VERSION_FIELD_DSI = 0,
    SAFECAM_DB_VERSION_FIELD_VERSION,
    SAFECAM_DB_VERSION_MAX_FIELDS
} SAFECAM_DB_VERSION_FIELDS_ENUM;

// Enumeration specifying all the available fields
// in the alert locations table
typedef enum safecam_alert_locations_fields_enum {
    SAFECAM_ALERT_LOCATIONS_FIELD_UID = 0,
    SAFECAM_ALERT_LOCATIONS_FIELD_LATITUDE,
    SAFECAM_ALERT_LOCATIONS_FIELD_LONGITUDE,
    SAFECAM_ALERT_LOCATIONS_FIELD_INFO,
    SAFECAM_ALERT_LOCATIONS_FIELD_NOSUB,
    SAFECAM_ALERT_LOCATIONS_FIELD_UNIT,
    SAFECAM_ALERT_LOCATIONS_MAX_FIELDS
} SAFECAM_ALERT_LOCATIONS_FIELDS_ENUM;

// Enumeration specifying all the available fields
// in the sub locations table
typedef enum safecam_sub_locations_fields_enum {
    SAFECAM_SUB_LOCATIONS_FIELD_SUB_ID = 0,
    SAFECAM_SUB_LOCATIONS_FIELD_UID,
    SAFECAM_SUB_LOCATIONS_FIELD_TYPE,
    SAFECAM_SUB_LOCATIONS_FIELD_LATITUDE,
    SAFECAM_SUB_LOCATIONS_FIELD_LONGITUDE,
    SAFECAM_SUB_LOCATIONS_FIELD_DIR,
    SAFECAM_SUB_LOCATIONS_FIELD_SPEED1,
    SAFECAM_SUB_LOCATIONS_FIELD_SPEED2,
    SAFECAM_SUB_LOCATIONS_MAX_FIELDS
} SAFECAM_SUB_LOCATIONS_FIELDS_ENUM;

// Enumeration specifying all the available fields
// in the alert types table
typedef enum safecam_alert_types_fields_enum {
    SAFECAM_ALERT_TYPES_FIELD_TYPE = 0,
    SAFECAM_ALERT_TYPES_FIELD_DESC,
    SAFECAM_ALERT_TYPES_MAX_FIELDS
} SAFECAM_ALERT_TYPES_FIELDS_ENUM;


  /*****************************/
 /** GLOBAL STRUCT TEMPLATES **/
/*****************************/

// Structure which specifies all attributes
// of a single database version row that is
// available within the safecam database
// These are the raw values that are retrieved from the database
typedef struct safecam_db_version_row_struct
{
    UN8 un8Type;
    UN32 un32Version;
    DSI tDSI;
} SAFECAM_DB_VERSION_ROW_STRUCT;

// Structure which specifies all attributes
// of a single alert location row that is
// available within the safecam database
// These are the raw values that are retrieved from the database
typedef struct safecam_alert_locations_row_struct
{
    UN32 un32UID;           // UID (Region + ID)
    N32 n32Lat;
    N32 n32Lon;
    STRING_OBJECT hInfo;
    UN8 un8NoSub;           // Sublocations number
    BOOLEAN bMilesUnits;    // TRUE in case if speed data will be sent in miles
} SAFECAM_ALERT_LOCATIONS_ROW_STRUCT;

// Structure which specifies all attributes
// of a single sub location row that is
// available within the safecam database
// These are the raw values that are retrieved from the database
typedef struct safecam_sub_locations_row_struct
{
    UN8 un8SubID;        // Sub location ID
    UN32 un32UID;        // UID (related alert location UID)
    UN8 un8Type;         // Alert Type ID
    N32 n32Lat;
    N32 n32Lon;
    UN16 un16Dir;       // direction in degrees
    UN16 un16Speed1;
    UN16 un16Speed2;
} SAFECAM_SUB_LOCATIONS_ROW_STRUCT;

// Structure which specifies all attributes
// of a single alert type row that is
// available within the safecam database
// These are the raw values that are retrieved from the database
typedef struct safecam_alert_types_row_struct
{
    UN8 un8Type;            // Alert Type ID
    STRING_OBJECT hDesc;    // Alert Type Description
} SAFECAM_ALERT_TYPES_ROW_STRUCT;


  /****************************/
 /** GLOBAL UNION TEMPLATES **/
/****************************/

// Union to group all the row definitions
// in a convenient manner
typedef union safecam_db_row_union
{
    SAFECAM_DB_VERSION_ROW_STRUCT *psDBVersion;
    SAFECAM_ALERT_LOCATIONS_ROW_STRUCT *psAlertLocations;
    SAFECAM_SUB_LOCATIONS_ROW_STRUCT *psSubLocations;
    SAFECAM_ALERT_TYPES_ROW_STRUCT *psAlertTypes;
} SAFECAM_DB_ROW_UNION;


  /**********************/
 /** GLOBAL VARIABLES **/
/**********************/

  /*******************/
 /** GLOBAL MACROS **/
/*******************/

  /***********************/
 /** GLOBAL PROTOTYPES **/
/***********************/

#endif /* SAFECAM_DB_CONSTANTS_H_ */
