/******************************************************************************/
/*                    Copyright (c) Sirius XM Radio, Inc.                     */
/*                            All Rights Reserved                             */
/*         Licensed Materials - Property of Sirius XM Radio, Inc.             */
/******************************************************************************/
/*******************************************************************************
 *
 * DESCRIPTION
 *
 * This is the Object:PRESETS header file.
 *
 ******************************************************************************/

#ifndef PRESETS_OBJ_H_
#define PRESETS_OBJ_H_

#include "standard.h"
#include "sms_api.h"
#include "tag_obj.h"

  /**********************/
 /** GLOBAL CONSTANTS **/
/**********************/

// Channel preset handle
typedef void * CHANNEL_PRESET_HDL;
#define CHANNEL_PRESET_INVALID_HDL (CHANNEL_PRESET_HDL)NULL

// Featured Favorites Band Order (6-bit field)
typedef UN8 FAVORITES_BAND_ORDER;

#define FAVORITES_BAND_ORDER_MIN                (0)
#define FAVORITES_BAND_ORDER_INVALID            (UN8_MAX)

// Featured Favorites Band ID (6-bit field)
typedef UN8 FAVORITES_BAND_ID;

#define FAVORITES_BAND_ID_MIN                   (0)
#define FAVORITES_BAND_ID_INVALID               (UN8_MAX)

// Featured Favorites Band sequence (4-bit field)
typedef UN8 FAVORITES_BAND_SEQUENCE;

#define FAVORITES_BAND_SEQUENCE_MIN             (0)
#define FAVORITES_BAND_SEQUENCE_INVALID         (UN8_MAX)

  /*********************/
 /** GLOBAL TYPEDEFS **/
/*********************/

typedef enum preset_band_state_enum
{
    PRESET_BAND_STATE_NEW,
    PRESET_BAND_STATE_EXISTING,
    PRESET_BAND_STATE_MODIFIED,
    PRESET_BAND_STATE_REJECTED

} PRESET_BAND_STATE_ENUM;

typedef struct preset_band_marker_struct
{
    BOOLEAN bRemoved;
    FAVORITES_BAND_ID tBankId;

} PRESET_BAND_MARKER_STRUCT;

typedef struct preset_band_search_for_tag_iterator_struct
{
    FAVORITES_BAND_ID tId;
    FAVORITES_BAND_SEQUENCE tSeq;
    size_t tCurrentBandId;
    PRESET_BAND_OBJECT hBand;
    OSAL_LINKED_LIST_ENTRY hEntry;
    PRESETS_OBJECT hPresets;
    PRESET_BAND_STATE_ENUM eState;
    UN32 un32BandsCount;
    UN8 un8ArrIndex;
    UN32 un32Removed;
    FAVORITES_BAND_CAPACITY tCapacity;
    FAVORITES_BAND_NUMBER tBandsNumber;

} PRESET_BAND_SEARCH_FOR_TAG_ITERATOR_STRUCT;

// Preset update type enum used
// for setting preset attributes
typedef enum preset_update_type_enum
{
    PRESET_UPDATE_TYPE_CHAN,
    PRESET_UPDATE_TYPE_NAME

} PRESET_UPDATE_TYPE_ENUM;

// Defines preset band private attributes
typedef struct featured_band_private_attrs_struct
{
    // Signifying the relative ordering of the
    // Featured Favorites banks provided by a service
    // Unique index for the received featured favorites
    FAVORITES_BAND_ORDER tOrder;

    // Unique ID
    FAVORITES_BAND_ID tId;

    // cycles from 0 ... 15, 0 ... each time an existing
    // bank is modified (identified by Bank ID)
    FAVORITES_BAND_SEQUENCE tSequence;

    // Indicates a purpose of a bank
    FAVORITES_BAND_PURPOSE_MASK tPurpose;

    // State of Band
    BOOLEAN bRemoved;

} FEATURED_BAND_PRIVATE_ATTRS_STRUCT;

// Defines preset band private attributes
typedef struct preset_band_private_attrs_struct
{
    // The band's name
    STRING_OBJECT hName;

    // The band's long name
    STRING_OBJECT hLongName;

    // The band's verbose name
    STRING_OBJECT hVerboseName;

    // The band's description
    STRING_OBJECT hDescription;

    // Indicate that this band was loaded from broadcast
    PRESET_BAND_TYPE_ENUM eType;

    // THe band's number of presets
    size_t tCapacity;

    // Featured Favorites private information
    FEATURED_BAND_PRIVATE_ATTRS_STRUCT sFeatured;

} PRESET_BAND_PRIVATE_ATTRS_STRUCT;

// Featured Favorites receiver targeting information
typedef struct favorites_receiver_info_struct
{
    // Band capacity
    FAVORITES_BAND_CAPACITY tCapacity;

    // Featured Band Purpose
    FAVORITES_BAND_PURPOSE_MASK tPurpose;

    // Featured Bands Limit
    FAVORITES_BAND_NUMBER tBandsNumber;

} FAVORITES_RECEIVER_INFO_STRUCT;

// Featured Favorities enable
typedef struct sms_event_featured_favorities_enable_struct
{
    FAVORITES_RECEIVER_INFO_STRUCT sCapabilities;

} SMS_EVENT_FEATURED_FAVORITIES_ENABLE_STRUCT;

typedef struct sms_event_featured_favorities_disable_struct
{
    // Remove bands or not
    BOOLEAN bRemoveBands;

} SMS_EVENT_FEATURED_FAVORITIES_DISABLE_STRUCT;

// Featured Favorities union
typedef struct sms_event_featured_favorites_union
{
    SMS_EVENT_FEATURED_FAVORITIES_ENABLE_STRUCT sEnable;
    SMS_EVENT_FEATURED_FAVORITIES_DISABLE_STRUCT sDisable;

} SMS_EVENT_FEATURED_FAVORITES_UNION;

// Featured Favorites events
typedef struct sms_event_featured_favorites_struct
{
    BOOLEAN bEnable;
    SMS_EVENT_FEATURED_FAVORITES_UNION uEvent;

} SMS_EVENT_FEATURED_FAVORITES_STRUCT;

  /*****************************/
 /** GLOBAL STRUCT TEMPLATES **/
/*****************************/

  /****************************/
 /** GLOBAL UNION TEMPLATES **/
/****************************/

  /**********************/
 /** GLOBAL VARIABLES **/
/**********************/

  /*******************/
 /** GLOBAL MACROS **/
/*******************/

  /***********************/
 /** GLOBAL PROTOTYPES **/
/***********************/

/* Object Friend Prototypes */

SMSAPI_RETURN_CODE_ENUM PRESETS_eCreate (
    DECODER_OBJECT hDecoder,
    PRESETS_OBJECT *phPresets,
    const char *pacLongName,
    const char *pacShortName
        );

OSAL_LINKED_LIST_ENTRY PRESETS_hAddBand (
    PRESETS_OBJECT hPresets,
    PRESET_BAND_OBJECT hBand,
    size_t tBandCapacity
        );

BOOLEAN PRESETS_bRemoveBand(
    PRESETS_OBJECT hPresets,
    PRESET_BAND_OBJECT hBand,
    OSAL_LINKED_LIST_ENTRY hBandEntry,
    BOOLEAN bUpdate
        );

SMSAPI_RETURN_CODE_ENUM PRESETS_eSetPreset(
    PRESET_BAND_OBJECT hBand,
    OSAL_LINKED_LIST_ENTRY hBandEntry,
    size_t tPresetIndex,
    PRESET_UPDATE_TYPE_ENUM eUpdateType,
    SERVICE_ID tServiceId,
    const char *pacName,
    BOOLEAN bUpdateSmartFavorites
        );

BOOLEAN PRESETS_bGetPreset (
    PRESET_BAND_OBJECT hBand,
    OSAL_LINKED_LIST_ENTRY hBandEntry,
    size_t tPresetIndex,
    SERVICE_ID *ptServiceId,
    STRING_OBJECT *phName
        );

BOOLEAN PRESETS_bClearAllPresets(
    PRESET_BAND_OBJECT hBand,
    BOOLEAN bClearNames,
    BOOLEAN bUpdateSmartFavorites
        );

SMSAPI_RETURN_CODE_ENUM PRESETS_eSaveBandPresets (
    OSAL_LINKED_LIST_ENTRY hBandEntry
        );

BOOLEAN PRESETS_bSetAllFeaturedBanksRemovedFlag (
    PRESETS_OBJECT hPresets,
    BOOLEAN bRemoved
        );

size_t PRESETS_tGetCurrentBandId (
    PRESETS_OBJECT hPresets
        );

BOOLEAN PRESETS_bRemoveAllSelectedFFBanks (
    PRESETS_OBJECT hPresets
        );

BOOLEAN PRESETS_bRemoveFirstSelectedFFBank (
    PRESETS_OBJECT hPresets
        );

BOOLEAN PRESETS_bResetFeaturedFavoritesId (
    PRESETS_OBJECT hPresets
        );

BOOLEAN PRESETS_bSetSelectedFavoritesBankRemoved (
    PRESETS_OBJECT hPresets,
    PRESET_BAND_MARKER_STRUCT sMarkerStruct
        );

BOOLEAN PRESETS_bSortBands (
    PRESETS_OBJECT hPresets
        );

SMSAPI_RETURN_CODE_ENUM PRESETS_eTune (
    DECODER_OBJECT hDecoder,
    PRESET_BAND_OBJECT hBand,
    OSAL_LINKED_LIST_ENTRY hBandEntry,
    size_t tPresetIndex,
    BOOLEAN bLockOverride
        );

SMSAPI_RETURN_CODE_ENUM PRESETS_eIterateBand (
    PRESET_BAND_OBJECT hBand,
    PRESETS_ITERATOR_HANDLER n16Iterator,
    void *pvIterateArg
        );

N32 PRESETS_n32PrintBand (
    PRESET_BAND_OBJECT hBand,
    OSAL_LINKED_LIST_ENTRY hBandEntry,
    FILE *psFile
        );

BOOLEAN PRESETS_bThisPreset (
    CHANNEL_OBJECT hChannel,
    CHANNEL_PRESET_HDL hPresetHdl,
    PRESET_BAND_OBJECT *phBand,
    STRING_OBJECT *phPresetName,
    size_t *ptPresetIndex
        );

void PRESETS_vUpdateHdl (
    PRESETS_OBJECT hPresets,
    CHANNEL_OBJECT hChannel,
    size_t tHdlIndex
        );

TAG_OBJECT PRESETS_hGetTag (
    PRESETS_OBJECT hPresets
        );

DECODER_OBJECT PRESETS_hDecoder (
    PRESETS_OBJECT hPresets
        );

PRESET_BAND_STATE_ENUM PRESETS_eGetPresetBandAndState (
    PRESETS_OBJECT hPresets,
    PRESET_BAND_SEARCH_FOR_TAG_ITERATOR_STRUCT *psIteratorStruct,
    FAVORITES_BAND_CAPACITY tBandsLimit
        );

OSAL_RETURN_CODE_ENUM PRESETS_bAddToID (
    PRESETS_OBJECT hPresets,
    FAVORITES_BAND_ID tId
        );

UN32 PRESETS_un32BankIDCount (
    PRESETS_OBJECT hPresets
        );

void PRESETS_vUpdateBankDescriptorOrder (
    OSAL_LINKED_LIST_ENTRY hEntry,
    FAVORITES_BAND_ORDER tOrder
        );

BOOLEAN PRESETS_bRemoveAllFeaturedBands (
    PRESETS_OBJECT hPresets
        );

BOOLEAN PRESETS_bGetCapabilities (
	PRESETS_OBJECT hPresets,
    FAVORITES_RECEIVER_INFO_STRUCT *psInfo
		);

BOOLEAN PRESETS_bUpdate (
    PRESETS_OBJECT hPresets,
    PRESET_BAND_OBJECT hBand,
    PRESETS_EVENT_MASK tUpdateMask
        );

SMSAPI_RETURN_CODE_ENUM PRESETS_eSetBandSyncCategory (
    PRESETS_OBJECT hPresets,
    CATEGORY_ID tCategoryId
        );

BOOLEAN PRESETS_bCurrentBandSync (
    PRESETS_OBJECT hPresets,
    PRESET_BAND_OBJECT hBand
        );

OSAL_LINKED_LIST_ENTRY PRESETS_hCurrentBandEntry(
    PRESETS_OBJECT hPresets
        );

void PRESETS_vBlockNotifications (
    PRESETS_OBJECT hPresets,
    BOOLEAN bBlock
        );

void PRESETS_vFavoritesStart (
    PRESETS_OBJECT hPresets,
    SMS_EVENT_FEATURED_FAVORITIES_ENABLE_STRUCT const *psEnable
        );

void PRESETS_vFavoritesStop (
    PRESETS_OBJECT hPresets,
    SMS_EVENT_FEATURED_FAVORITIES_DISABLE_STRUCT const *psDisable
        );

#endif /* PRESETS_OBJ_H_ */

