/******************************************************************************/
/*                    Copyright (c) Sirius XM Radio, Inc.                     */
/*                            All Rights Reserved                             */
/*         Licensed Materials - Property of Sirius XM Radio, Inc.             */
/******************************************************************************/
/*******************************************************************************
 *
 * DESCRIPTION
 *
 *  This module contains the Object:PHONETICS_SERVICE implementation for the
 *  Simple Module Services (SMS)
 *
 ******************************************************************************/
#include <stdio.h>
#include <string.h>
#include "standard.h"
#include "osal.h"

#include "sms_api.h"
#include "sms_obj.h"
#include "sms.h"
#include "sms_version.h"
#include "league_obj.h"
#include "report_obj.h"
#include "sql_interface_obj.h"
#include "sms_update.h"
#include "sms_event.h"
#include "dataservice_mgr_impl.h"
#include "string_obj.h"
#include "db_util.h"
#include "_phonetics_mgr_obj.h"
#include "phonetics_interface.h"

#include "sms_api_debug.h"
static const char *gpacThisFile = __FILE__;

/*****************************************************************************
                             PUBLIC FUNCTIONS
*****************************************************************************/

/*****************************************************************************
*
*   hStart
*
*   This function will create all the basics needed for this service to
*   operate.  However, all initial processing to actually get this service
*   running is done at a later time.
*
*****************************************************************************/
static PHONETICS_SERVICE_OBJECT hStart (
    const char *pacSRHDriverName,
    const char *pacOutputDir,
    DATASERVICE_EVENT_MASK tEventRequestMask,
    DATASERVICE_EVENT_CALLBACK vEventCallback,
    void *pvEventCallbackArg,
    PHONETICS_FILE_UPDATE_CALLBACK vFileCallback,
    void *pvFileCallbackArg,
    DATASERVICE_OPTIONS_STRUCT const *psOptions
        )
{
    PHONETICS_MGR_OBJECT_STRUCT *psObj;
    DATASERVICE_CREATE_STRUCT sCreate;
    BOOLEAN bOk;

    if (pacOutputDir == NULL)
    {
        // Phonetics needs this info!
        return PHONETICS_SERVICE_INVALID_OBJECT;
    }

    // Populate our data service creation structure
    DATASERVICE_IMPL_vInitCreateStruct(&sCreate);
    sCreate.pacSRHDriverName = pacSRHDriverName;
    sCreate.pacServiceObjectName = PHONETICS_MGR_OBJECT_NAME;
    sCreate.tServiceObjectSize = sizeof(PHONETICS_MGR_OBJECT_STRUCT);
    sCreate.tDataID = (DATASERVICE_ID)GsPhoneticsIntf.tDSI;

    // Suggest an OTA buffer size
    sCreate.tSuggestedOTABufferByteSize = GsPhoneticsIntf.tOTABufferByteSize;

    // This service doesn't care about the time
    sCreate.bTimeRequired = FALSE;

    // Configure the data service's static event attributes
    sCreate.vEventCallback = vEventHandler;
    sCreate.tEventRequestMask = DATASERVICE_EVENT_STATE |
                                DATASERVICE_EVENT_NEW_DATA;

    // Ask the data service manager controller to
    // create our manager object and do everything
    // necessary to create the underlying objects required
    // in order to support this service
    psObj = (PHONETICS_MGR_OBJECT_STRUCT *)
        DATASERVICE_IMPL_hCreateNewService( &sCreate );
    if (psObj == NULL)
    {
        // Can't create the service, fail out!
        return PHONETICS_SERVICE_INVALID_OBJECT;
    }

    do
    {
        DATASERVICE_OPTION_VALUES_STRUCT sOptionValues;

        bOk = DATASERVICE_IMPL_bProcessOptions(
            PHONETICS_SUPPORTED_OPTIONS, psOptions, &sOptionValues);
        // restore stack (push)
        if (bOk == FALSE)
        {
            // Bad options!
            break;
        }

        // Validate path given, create directories as needed,
        // store resultant output paths
        bOk = bPrepareOutputDirectories(psObj, pacOutputDir);
        if (bOk == FALSE)
        {
            break;
        }

        // We don't have work units now
        psObj->psWorkUnit = NULL;

        // Update the path for the reference db
        psObj->pacRefDatabaseDirPath = sOptionValues.pcRefDBPath;

        // Save file notification attributes
        psObj->vFileCallback = vFileCallback;
        psObj->pvFileCallbackArg = pvFileCallbackArg;

        // Initialize asynchronous update configuration
        SMSU_vInitialize(
            &psObj->sEvent,
            psObj,
            DATASERVICE_EVENT_NONE,
            tEventRequestMask,
            (SMSAPI_OBJECT_EVENT_CALLBACK)vEventCallback,
            pvEventCallbackArg);

        // The service may now start
        bOk = DATASERVICE_IMPL_bStart((DATASERVICE_IMPL_HDL)psObj);

        if (bOk == FALSE)
        {
            break;
        }

        return (PHONETICS_SERVICE_OBJECT)psObj;
    } while (FALSE);

    // Something bad happened
    vUninitObject( psObj, TRUE );
    DATASERVICE_IMPL_vDestroy((DATASERVICE_IMPL_HDL)psObj);

    return PHONETICS_SERVICE_INVALID_OBJECT;
}

/*****************************************************************************
*
*   bGenerateDistributionFiles
*
*****************************************************************************/
static BOOLEAN bGenerateDistributionFiles (
    const char *pacTTSOutputDir,
    const char *pacRecOutputDir,
    const char *pacDBPath
        )
{
    BOOLEAN bSuccess = FALSE;
    PHONETICS_WORK_UNIT_STRUCT *psWorkUnit = NULL;
    SQL_INTERFACE_OBJECT hRefDBConnection =
        SQL_INTERFACE_INVALID_OBJECT;
    STRING_OBJECT hTTSOutputDir = STRING_INVALID_OBJECT,
                  hRecOutputDir = STRING_INVALID_OBJECT;

    do
    {
        PHONETICS_VER_CTRL_STRUCT sTableVersions;
        DATASERVICE_ERROR_CODE_ENUM eErrorCode =
            DATASERVICE_ERROR_CODE_NONE;
        PHONETICS_FILE_TYPE_ENUM eFileToProcess;

        // Create the output directory string objects
        hTTSOutputDir = hCreateOutputDir (pacTTSOutputDir);
        hRecOutputDir = hCreateOutputDir (pacRecOutputDir);
        if ((hTTSOutputDir == STRING_INVALID_OBJECT) ||
            (hRecOutputDir == STRING_INVALID_OBJECT))
        {
            break;
        }

        // Create/init the work unit
        psWorkUnit = psCreateWorkUnit(
            NULL, 0, 0,
            PHONETICS_FILE_TYPE_CHANNELS, OSAL_INVALID_OBJECT_HDL);

        if (psWorkUnit == (PHONETICS_WORK_UNIT_STRUCT *)NULL)
        {
            break;
        }

#ifdef PHONETICS_TEST
        // Insert test data
        bSuccess = bInsertTestData(&psWorkUnit->hListData);
        if (bSuccess == FALSE)
        {
            break;
        }
#endif

        // Connect to the read-only ref database
        // never perform an integrity check
        hRefDBConnection =
            DB_UTIL_hConnectToReference(
                &pacDBPath[0],
                (DB_UTIL_CHECK_VERSION_HANDLER)bVerifyAndLoadDBVersionFields,
                (void *)&sTableVersions,
                &eErrorCode,
                SQL_INTERFACE_OPTIONS_READONLY |
                SQL_INTERFACE_OPTIONS_SKIP_CORRUPTION_CHECK);
        if (hRefDBConnection == SQL_INTERFACE_INVALID_OBJECT)
        {
            break;
        }

        // Generate all the phonetic tables we have
        for (eFileToProcess = PHONETICS_FILE_TYPE_CHANNELS;
             eFileToProcess < PHONETICS_FILE_INVALID_TYPE;
             eFileToProcess++)
        {
            // We are now processing this file type
            psWorkUnit->eType = eFileToProcess;

            // Report a new list is ready
            bSuccess = bNewListLocal(
                PHONETICS_SERVICE_INVALID_OBJECT,
                psWorkUnit,
                hTTSOutputDir,
                hRecOutputDir);

            if (bSuccess == TRUE)
            {
                // Process the work unit now
                bSuccess = bProcessNewPhoneticsList(
                    hRefDBConnection,
                    SQL_INTERFACE_INVALID_OBJECT,
                    psWorkUnit );
            }

            // Clean up
            vClearWorkUnit(psWorkUnit);

            if (bSuccess == FALSE)
            {
                break;
            }
        }
    } while (FALSE);

    if (hRefDBConnection != SQL_INTERFACE_INVALID_OBJECT)
    {
        SQL_INTERFACE.vDisconnect(hRefDBConnection);
    }

    if (hTTSOutputDir != STRING_INVALID_OBJECT)
    {
        STRING_vDestroy(hTTSOutputDir);
    }

    if (hRecOutputDir != STRING_INVALID_OBJECT)
    {
        STRING_vDestroy(hRecOutputDir);
    }

    if (NULL != psWorkUnit)
    {
#ifdef PHONETICS_TEST
        vClearWorkUnit(psWorkUnit);
#endif
        SMSO_vDestroy((SMS_OBJECT)psWorkUnit);
    }

    return bSuccess;
}

/*****************************************************************************
*
*   tReadServiceIdFromDictionary
*
*****************************************************************************/
SERVICE_ID tReadServiceIdFromDictionary (
    FILE *psDictionaryFile
        )
{
    BOOLEAN bRead;
    UN32 un32RawId = 0;

    // Read this ID from the file
    bRead = bReadIdFromFile(&un32RawId, psDictionaryFile);
    if (bRead == TRUE)
    {
        return (SERVICE_ID)un32RawId;
    }

    return SERVICE_INVALID_ID;
}

/*****************************************************************************
*
*   tReadCategoryIdFromDictionary
*
*****************************************************************************/
CATEGORY_ID tReadCategoryIdFromDictionary (
    FILE *psDictionaryFile
        )
{
    BOOLEAN bRead;
    UN32 un32RawId = 0;

    // Read this ID from the file
    bRead = bReadIdFromFile(&un32RawId, psDictionaryFile);
    if (bRead == TRUE)
    {
        return (CATEGORY_ID)un32RawId;
    }

    return CATEGORY_INVALID_ID;
}

/*****************************************************************************
                             FRIEND FUNCTIONS
*****************************************************************************/

/*****************************************************************************
                             PRIVATE FUNCTIONS
*****************************************************************************/

/*****************************************************************************
*
*   bNewList
*
*   This function is called by the interface when it has a new phonetics
*   list ready for the manager to consume.
*
*   It prepares a work unit to later be used for the generation of new
*   output files
*
*****************************************************************************/
static BOOLEAN bNewList (
    PHONETICS_SERVICE_OBJECT hPhoneticsService,
    PHONETICS_FILE_TYPE_ENUM eType,
    UN8 un8PhoneticsFileVer,
    UN16 un16PhoneticsContentVer,
    OSAL_OBJECT_HDL hPhoneticsList
        )
{
    BOOLEAN bOwner;

    // Check object ownership
    bOwner = DATASERVICE_IMPL_bOwner((DATASERVICE_IMPL_HDL)hPhoneticsService);

    if (bOwner == TRUE)
    {
        PHONETICS_MGR_OBJECT_STRUCT *psObj =
            (PHONETICS_MGR_OBJECT_STRUCT *)hPhoneticsService;
        PHONETICS_WORK_UNIT_STRUCT *psWorkUnit = NULL;

        DATASERVICE_IMPL_vLog(PHONETICS_MGR_OBJECT_NAME
            ": New Phonetics Table Ready (Type: %s Ver: %u Content Ver: %u)\n",
            pacFileType(eType), un8PhoneticsFileVer, un16PhoneticsContentVer);

        do
        {
            BOOLEAN bSuccess;

            // Create/init the work unit
            psWorkUnit = psCreateWorkUnit(
                psObj, un8PhoneticsFileVer, un16PhoneticsContentVer, eType, hPhoneticsList);

            if (psWorkUnit == (PHONETICS_WORK_UNIT_STRUCT *)NULL)
            {
                break;
            }

            // Report a new list is ready
            bSuccess = bNewListLocal(
                hPhoneticsService,
                psWorkUnit,
                psObj->hTTSOutputDir,
                psObj->hRecOutputDir);
            if (bSuccess == FALSE)
            {
                break;
            }

            // Use this work unit now
            psObj->psWorkUnit = psWorkUnit;

            return TRUE;

        } while (FALSE);

        if (psWorkUnit != (PHONETICS_WORK_UNIT_STRUCT *)NULL)
        {
            SMSO_vDestroy((SMS_OBJECT)psWorkUnit);
        }
    }

    return FALSE;
}

/*****************************************************************************
*
*   eTypeForId
*
*   This function is called by the interface when it wants to know what
*   file data its working on
*
*****************************************************************************/
static PHONETICS_FILE_TYPE_ENUM eTypeForId (
    UN8 un8TableId
        )
{
    PHONETICS_FILE_TYPE_ENUM eFileType;

    switch (un8TableId)
    {
        case PHONETICS_CHAN_TABLE_ID:
        {
            eFileType = PHONETICS_FILE_TYPE_CHANNELS;
        }
        break;

        case PHONETICS_CAT_TABLE_ID:
        {
            eFileType = PHONETICS_FILE_TYPE_CATEGORIES;
        }
        break;

        case PHONETICS_MARKET_TABLE_ID:
        {
            eFileType = PHONETICS_FILE_TYPE_MARKETS;
        }
        break;

        case PHONETICS_TEAM_TABLE_ID:
        {
            eFileType = PHONETICS_FILE_TYPE_TEAMS;
        }
        break;

        case PHONETICS_LEAGUE_TABLE_ID:
        {
            eFileType = PHONETICS_FILE_TYPE_LEAGUES;
        }
        break;

        default:
        {
            eFileType = PHONETICS_FILE_INVALID_TYPE;
        }
    }

    return eFileType;
}

/*****************************************************************************
*
*   hCreatePhoneticsList
*   This function is called by the interface when it wants to create a new
*   list of phonetics
*
*****************************************************************************/
static OSAL_OBJECT_HDL hCreatePhoneticsList (
    PHONETICS_FILE_TYPE_ENUM eType
        )
{
    OSAL_RETURN_CODE_ENUM eReturnCode;
    OSAL_OBJECT_HDL hList = OSAL_INVALID_OBJECT_HDL;
    char acName[OSAL_MAX_OBJECT_NAME_LENGTH_WITH_NULL];

    // Name this list
    snprintf( &acName[0], sizeof(acName),
        PHONETICS_MGR_OBJECT_NAME
        ":PhoneticList (%s)",
        pacFileType(eType));

    // Make it linear & preallocated
    eReturnCode = OSAL.eLinkedListCreate(
        &hList,
        &acName[0],
        (OSAL_LL_COMPARE_HANDLER)n16ComparePhoneticNode,
        OSAL_LL_OPTION_LINEAR |
        OSAL_LL_OPTION_USE_PRE_ALLOCATED_ELEMENTS );
    if (eReturnCode != OSAL_SUCCESS)
    {
        return OSAL_INVALID_OBJECT_HDL;
    }

    DATASERVICE_IMPL_vLog(PHONETICS_MGR_OBJECT_NAME
        ": New %s table created\n",
        GsPhoneticsMgrIntf.pacFileType(eType));

    return hList;
}

/*****************************************************************************
*
*   bConcatPhoneticData
*   This function is called by the interface when it wants to put two string
*   objects which contain phonetic data together
*
*****************************************************************************/
static BOOLEAN bConcatPhoneticData (
    STRING_OBJECT hAggregate,
    STRING_OBJECT hNewPhonetics
        )
{
    do
    {
        size_t tAppended;

        // Append the delimiter to the aggregate
        tAppended = STRING.tAppendCStr(
            hAggregate, PHONETICS_DELIMITER_STRING);
        if (0 == tAppended)
        {
            break;
        }

        // STRING.tAppendString is (src, dst)
        tAppended = STRING.tAppendString(
            hNewPhonetics, hAggregate);
        if (0 == tAppended)
        {
            break;
        }

        return TRUE;

    } while (FALSE);

    return FALSE;
}

/*******************************************************************************
*
*   n16ComparePhoneticNode
*
*   This function compares two PHONETICS_LIST_NODE_STRUCT and orders
*   them by the primary id and secondary id.
*
*   Inputs:
*       psNode1 (in list), psNode2 (compare against)
*
*   Outputs:
*       0   - Objects have the same value (equal, error)
*       > 0 - Object1(in list) is greater than (after) Object2
*       < 0 - Object1(in list) is less than (before) Object2
*
*******************************************************************************/
static N16 n16ComparePhoneticNode (
    PHONETICS_LIST_NODE_STRUCT *psNode1,
    PHONETICS_LIST_NODE_STRUCT *psNode2
        )
{
    // Sort by primary id (sufficient for all types except teams)
    if (psNode1->sID.un32Id != psNode2->sID.un32Id)
    {
        if (psNode1->sID.un32Id < psNode2->sID.un32Id)
        {
            return -1;
        }

        return 1;
    }
    else if (psNode1->sID.un8NumLeagues == 0)
    {
        // Stop here with a match if we're not talking about teams
        return 0;
    }

    // If both have league data, then compare 'em.  Here's the issue with
    // comparing teams:  A team entry in phonetics is uniquely identified
    // by matching any one of the leagues listed.  You don't have to
    // match all of the leagues for the entries to be considered equal.
    // You just have to match one.
    if ((psNode1->sID.un8NumLeagues != 0) &&
        (psNode2->sID.un8NumLeagues != 0))
    {
        UN8 un8Index;
        UN8 un8LeaguesToCheck = psNode1->sID.un8NumLeagues;

        // Only attempt to match the number of leagues which
        // are available in both
        if (psNode2->sID.un8NumLeagues < un8LeaguesToCheck)
        {
            un8LeaguesToCheck = psNode2->sID.un8NumLeagues;
        }

        // First attempt to match any of these leagues
        for (un8Index = 0; un8Index < un8LeaguesToCheck; un8Index++)
        {
            if (psNode1->sID.aun16Leagues[un8Index] == psNode2->sID.aun16Leagues[un8Index])
            {
                // Match!
                return 0;
            }
        }

         // If we got here then the league list is totally different,
        // so compare those (compare all three in this case, so we can
        // get the difference between all league entries)
        for (un8Index = 0; un8Index < PHONETICS_MAX_LEAGUES; un8Index++)
        {
            if (psNode1->sID.aun16Leagues[un8Index] < psNode2->sID.aun16Leagues[un8Index])
            {
                return -1;
            }

            if (psNode1->sID.aun16Leagues[un8Index] > psNode2->sID.aun16Leagues[un8Index])
            {
                return 1;
            }

        }
    }

    // Error!
    return N16_MIN;
}

/*****************************************************************************
*
*   bRemovePhoneticNode
*
*****************************************************************************/
static BOOLEAN bRemovePhoneticNode (
    PHONETICS_LIST_NODE_STRUCT *psNode,
    void *pvUnused
        )
{
    if (psNode != NULL)
    {
        UN8 un8Index;

        // Clear the identifier(s)
        psNode->sID.un32Id = 0;
        psNode->sID.un8NumLeagues = 0;

        for (un8Index = 0;
             un8Index < PHONETICS_MAX_LANGUAGES;
             un8Index++)
        {
            // Destroy all associated phonetics
            if (psNode->ahPhonetics[un8Index] != STRING_INVALID_OBJECT)
            {
                STRING_vDestroy(psNode->ahPhonetics[un8Index]);
                psNode->ahPhonetics[un8Index] = STRING_INVALID_OBJECT;
            }
        }

        if (psNode->hEntry != OSAL_INVALID_LINKED_LIST_ENTRY)
        {
            OSAL_RETURN_CODE_ENUM eReturnCode;
            eReturnCode = OSAL.eLinkedListRemove(psNode->hEntry);

            if (eReturnCode == OSAL_SUCCESS)
            {
                psNode->hEntry = OSAL_INVALID_LINKED_LIST_ENTRY;
                OSAL.vLinkedListMemoryFree(psNode);
            }
        }
    }

    return TRUE;
}

/*****************************************************************************
*
*   pacFileType
*
*****************************************************************************/
static char const *pacFileType (
    PHONETICS_FILE_TYPE_ENUM eType
        )
{
    char const *pacType = "Unknown";

    switch (eType)
    {
        case PHONETICS_FILE_TYPE_CHANNELS:
        {
            pacType = "Channels";
        }
        break;

        case PHONETICS_FILE_TYPE_CATEGORIES:
        {
            pacType = "Categories";
        }
        break;

        case PHONETICS_FILE_TYPE_MARKETS:
        {
            pacType = "Markets";
        }
        break;

        case PHONETICS_FILE_TYPE_TEAMS:
        {
            pacType = "Teams";
        }
        break;

        case PHONETICS_FILE_TYPE_LEAGUES:
        {
            pacType = "Leagues";
        }
        break;

        default:
        break;
    }

    return pacType;
}

/*****************************************************************************
*
*   vEventHandler
*
*   This function runs in the context of an SMS resource which has been
*   assigned to this service.
*
*****************************************************************************/
static void vEventHandler (
    DATASERVICE_MGR_OBJECT hDataService,
    DATASERVICE_EVENT_MASK tCurrentEvent,
    void *pvEventArg,
    void *pvEventCallbackArg
        )
{
    PHONETICS_MGR_OBJECT_STRUCT *psObj;
    BOOLEAN bValid, bStopEvent = FALSE;
    SMSAPI_EVENT_MASK tEventMask = DATASERVICE_EVENT_NONE;

    // Get our phonetics handle from the callback argument
    psObj = (PHONETICS_MGR_OBJECT_STRUCT *)pvEventCallbackArg;

    // Is this object valid?
    bValid = DATASERVICE_IMPL_bValid((DATASERVICE_IMPL_HDL)psObj);

    // Only handle events for valid objects...
    if (bValid == FALSE)
    {
        return;
    }

    switch( tCurrentEvent )
    {
        // Handle Phonetics Service events here...

        // State has changed
        case DATASERVICE_EVENT_STATE:
        {
            BOOLEAN bStateChanged;
            DATASERVICE_STATE_CHANGE_STRUCT const *psStateChange =
                (DATASERVICE_STATE_CHANGE_STRUCT const *)pvEventArg;

            // Process the state transition
            bStateChanged = DATASERVICE_IMPL_bStateFSM(
                (DATASERVICE_IMPL_HDL)psObj,
                psStateChange,
                &GsPhoneticsStateHandlers,
                (void *)psObj);

            if (bStateChanged == TRUE)
            {
                // The state has been updated
                tEventMask |= DATASERVICE_EVENT_STATE;

                // Is the service stopped now?
                if (psStateChange->eCurrentState ==
                        DATASERVICE_STATE_STOPPED)
                {
                    bStopEvent = TRUE;
                }
            }
        }
        break;

        // This service has a message to process
        case DATASERVICE_EVENT_NEW_DATA:
        {
            OSAL_BUFFER_HDL hPayload = (OSAL_BUFFER_HDL)pvEventArg;
            BOOLEAN bOk = FALSE;

            // Ensure the payload handle is valid
            // If it isn't, there's a problem with
            // SMS
            if (hPayload != OSAL_INVALID_BUFFER_HDL)
            {
                printf(PHONETICS_MGR_OBJECT_NAME": Payload Received (%u)\n",
                    OSAL.tBufferGetSize(hPayload));

                // Process this message now
                bOk = GsPhoneticsIntf.bProcessMessage(psObj->hInterface, hPayload );

                if (bOk == TRUE)
                {
                    puts(PHONETICS_MGR_OBJECT_NAME": Message Payload Processed Ok");

                    // Do we have something to process now
                    if (psObj->psWorkUnit != NULL)
                    {
                        // Yes! Process it
                        bOk = bHandleNewPhoneticsList(psObj);
                    }
                }
                else
                {
                    DATASERVICE_IMPL_vLog(
                        PHONETICS_MGR_OBJECT_NAME": Failed to process message\n");
                }
            }

            // We're all done with this payload
            DATASERVICE_IMPL_bFreeDataPayload(hPayload);
        }
        break;

        default:
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                PHONETICS_MGR_OBJECT_NAME": Got unknown event (%u)",
                tCurrentEvent);
        }
        break;
    }

    if (bStopEvent == TRUE)
    {
        // Uninitialize the object, but leave
        // enough to be useful to the callback
        vUninitObject( psObj, FALSE );
    }

    // Update event mask with any relevant events which have occurred
    SMSU_tUpdate(&psObj->sEvent, tEventMask);

    // Notify of any change via any registered callback which may be present
    SMSU_bNotify(&psObj->sEvent);

    if (bStopEvent == TRUE)
    {
        // Filter out all further manager updates
        SMSU_tFilter(&psObj->sEvent, DATASERVICE_EVENT_ALL);

        vDestroyObject(psObj);
    }

    return;
}

/*****************************************************************************
*
*   bHandleServiceReady
*
*   This function is called when SMS is ready for the service to startup.
*   At this time, the service has a context in which to operate (SMS
*   assigned a resource to us). When this call is made, this service will
*   perform all of its time-intensive startup procedures.
*
*****************************************************************************/
static BOOLEAN bHandleServiceReady (
    PHONETICS_MGR_OBJECT_STRUCT *psObj
        )
{
    DATASERVICE_ERROR_CODE_ENUM eErrorCode =
        DATASERVICE_ERROR_CODE_NONE;

    do
    {
        BOOLEAN bOk;
        PHONETICS_VER_CTRL_STRUCT sTableVersions;
        char *pacDatabaseFilePath;

        // Construct the path needed for the reference database
        bOk = DB_UTIL_bCreateFilePath(
            psObj->pacRefDatabaseDirPath,
            PHONETICS_DATABASE_FOLDER,
            PHONETICS_REF_DATABASE_FILENAME,
            &pacDatabaseFilePath);

        if (bOk == FALSE)
        {
            // Error!  Couldn't build database path
            eErrorCode = DATASERVICE_ERROR_CODE_GENERAL;
            break;
        }

        // Connect to the read-only ref database;
        // never perform an integrity check
        psObj->hSQLRefConnection =
            DB_UTIL_hConnectToReference(
                &pacDatabaseFilePath[0],
                (DB_UTIL_CHECK_VERSION_HANDLER)bVerifyAndLoadDBVersionFields,
                (void *)&sTableVersions,
                &eErrorCode,
                SQL_INTERFACE_OPTIONS_READONLY |
                SQL_INTERFACE_OPTIONS_SKIP_CORRUPTION_CHECK);

        // We don't need the database file path anymore
        SMSO_vDestroy((SMS_OBJECT)pacDatabaseFilePath);

        // If the connection failed, indicate this
        if (psObj->hSQLRefConnection == SQL_INTERFACE_INVALID_OBJECT)
        {
            break;
        }

        // Build the path to the persitent storage DB
        bOk = DB_UTIL_bCreateFilePath(
            NULL, // We don't know the base path
            PHONETICS_DATABASE_FOLDER,
            PHONETICS_PERSIST_DATABASE_FILENAME,
            &pacDatabaseFilePath);

        if (bOk == FALSE)
        {
            // Error!  Couldn't build database path
            eErrorCode = DATASERVICE_ERROR_CODE_GENERAL;
            break;
        }

        // Connect to the persitent storage database
        psObj->hSQLPersistConnection =
            DB_UTIL_hConnectToPersistent(
                &pacDatabaseFilePath[0],
                (DB_UTIL_CREATE_DB_HANDLER)bCreatePersistDBTables,
                (void *)&sTableVersions,
                (DB_UTIL_CHECK_VERSION_HANDLER)bVerifyAndLoadDBVersionFields,
                (void *)&sTableVersions,
                &eErrorCode);

        // We don't need the database file path anymore
        SMSO_vDestroy((SMS_OBJECT)pacDatabaseFilePath);

        // If the connection failed, indicate this
        if (psObj->hSQLPersistConnection == SQL_INTERFACE_INVALID_OBJECT)
        {
            break;
        }

        // Now, connect to the high-band interface and
        // provide our table data
        bOk = bConnectToInterface(
            psObj, PHONETICS_NUM_TABLES, &sTableVersions);
        if (bOk == FALSE)
        {
            break;
        }

        // All is well
        return TRUE;

    } while (FALSE);

    // Set the error we experienced
    vSetError( psObj, eErrorCode );

    // Indicate failure
    return FALSE;
}

/*****************************************************************************
*
*   bConnectToInterface
*
*****************************************************************************/
static BOOLEAN bConnectToInterface (
    PHONETICS_MGR_OBJECT_STRUCT *psObj,
    size_t tNumTables,
    PHONETICS_VER_CTRL_STRUCT const *psTableVersions
        )
{
    BOOLEAN bSuccess = FALSE;

    // Connect to the interface
    psObj->hInterface = GsPhoneticsIntf.hInit(
        (PHONETICS_SERVICE_OBJECT)psObj,
        DATASERVICE_IMPL_hSMSObj((DATASERVICE_IMPL_HDL)psObj));
    if (psObj->hInterface != PHONETICS_INTERFACE_INVALID_OBJECT)
    {
        size_t tIndex;
        PHONETICS_FILE_TYPE_ENUM eFileType =
            PHONETICS_FILE_TYPE_CHANNELS;

        for (tIndex = 0, eFileType = PHONETICS_FILE_TYPE_CHANNELS;
             tIndex < tNumTables;
             tIndex++, eFileType++)
        {
            // Add all of our tables
            bSuccess = GsPhoneticsIntf.bAddTable(
                psObj->hInterface, eFileType, (UN8)tIndex,
                psTableVersions->asTableVer[tIndex].un8TableDefVer,
                psTableVersions->asTableVer[tIndex].un16TableContentVer);

            if (bSuccess == FALSE)
            {
                break;
            }
        }
    }

    return bSuccess;
}

/*****************************************************************************
*
*   vUninitObject
*
*****************************************************************************/
static void vUninitObject (
    PHONETICS_MGR_OBJECT_STRUCT *psObj,
    BOOLEAN bFullDelete
        )
{
    // Disconnect from the ref database
    if (psObj->hSQLRefConnection != SQL_INTERFACE_INVALID_OBJECT)
    {
        SQL_INTERFACE.vDisconnect( psObj->hSQLRefConnection );
        psObj->hSQLRefConnection = SQL_INTERFACE_INVALID_OBJECT;
    }

    // Disconnect from the persistent database
    if (psObj->hSQLPersistConnection != SQL_INTERFACE_INVALID_OBJECT)
    {
        SQL_INTERFACE.vDisconnect( psObj->hSQLPersistConnection );
        psObj->hSQLPersistConnection = SQL_INTERFACE_INVALID_OBJECT;
    }

    // Delete output directories
    if (psObj->hTTSOutputDir != STRING_INVALID_OBJECT)
    {
        STRING_vDestroy(psObj->hTTSOutputDir);
        psObj->hTTSOutputDir = STRING_INVALID_OBJECT;
    }

    if (psObj->hRecOutputDir != STRING_INVALID_OBJECT)
    {
        STRING_vDestroy(psObj->hRecOutputDir);
        psObj->hRecOutputDir = STRING_INVALID_OBJECT;
    }

    // Stop the phonetics interface
    GsPhoneticsIntf.vUninit(psObj->hInterface);
    psObj->hInterface = PHONETICS_INTERFACE_INVALID_OBJECT;

    if (bFullDelete == TRUE)
    {
        vDestroyObject(psObj);
    }

    return;
}

/*****************************************************************************
*
*   vDestroyObject
*
*****************************************************************************/
static void vDestroyObject(
    PHONETICS_MGR_OBJECT_STRUCT *psObj
        )
{
    // Destroy the SMS Update Event object
    SMSU_vDestroy(&psObj->sEvent);

    return;
}

/*****************************************************************************
*
*   bPrepareOutputDirectories
*
*   Validate directory provided, create subdirectories as necessary,
*   create string objects to store directory paths
*
*****************************************************************************/
static BOOLEAN bPrepareOutputDirectories (
    PHONETICS_MGR_OBJECT_STRUCT *psObj,
    const char *pacOutputDir
        )
{
    BOOLEAN bSuccess = FALSE;

    do
    {
        size_t tSize;
        UN8 un8FileAttributes = 0;

        // Create the output directory string objects
        psObj->hTTSOutputDir = hCreateOutputDir(pacOutputDir);
        psObj->hRecOutputDir = hCreateOutputDir(pacOutputDir);
        if ((psObj->hTTSOutputDir == STRING_INVALID_OBJECT) ||
            (psObj->hRecOutputDir == STRING_INVALID_OBJECT))
        {
            break;
        }

        // Add the directory names
        tSize = STRING.tAppendCStr(psObj->hTTSOutputDir, PHONETICS_TTS_DIR_NAME);
        if (tSize == 0)
        {
            break;
        }

        tSize = STRING.tAppendCStr(psObj->hRecOutputDir, PHONETICS_REC_DIR_NAME);
        if (tSize == 0)
        {
            break;
        }

        // Now, append the delimiter to complete these paths
        tSize = STRING.tAppendCStr(psObj->hTTSOutputDir, PHONETICS_DEFAULT_FS_DELIM);
        if (tSize == 0)
        {
            break;
        }

        tSize = STRING.tAppendCStr(psObj->hRecOutputDir, PHONETICS_DEFAULT_FS_DELIM);
        if (tSize == 0)
        {
            break;
        }

        /***********************************/
        /*   Create/validate directories   */
        /***********************************/

        // Create / validate the TTS directory
        un8FileAttributes = 0;
        bSuccess = OSAL.bFileSystemGetFileAttributes(
            STRING.pacCStr(psObj->hTTSOutputDir),
            &un8FileAttributes);
        if (bSuccess == TRUE)
        {
            // Somethere is here -- what is it?
            if ((un8FileAttributes & OSAL_FILE_ATTR_DIRECTORY)
                != OSAL_FILE_ATTR_DIRECTORY)
            {
                // Can't use a file
                bSuccess = FALSE;
            }
        }
        else
        {
            // Create the directory
            bSuccess = OSAL.bFileSystemMakeDir(STRING.pacCStr(psObj->hTTSOutputDir));
        }

        if (bSuccess == FALSE)
        {
            break;
        }

        // Create / validate the voice rec directory
        un8FileAttributes = 0;
        bSuccess = OSAL.bFileSystemGetFileAttributes(
            STRING.pacCStr(psObj->hRecOutputDir),
            &un8FileAttributes);
        if (bSuccess == TRUE)
        {
            // Somethere is here -- what is it?
            if ((un8FileAttributes & OSAL_FILE_ATTR_DIRECTORY)
                != OSAL_FILE_ATTR_DIRECTORY)
            {
                // Can't use a file
                bSuccess = FALSE;
            }
        }
        else
        {
            // Create the directory
            bSuccess = OSAL.bFileSystemMakeDir(STRING.pacCStr(psObj->hRecOutputDir));
        }

        if (bSuccess == FALSE)
        {
            break;
        }

    } while (FALSE);

    return bSuccess;
}

/*****************************************************************************
*
*   hCreateOutputDir
*
*   This function checks output dir's path and places it
*   into a STRING object
*
*****************************************************************************/
static STRING_OBJECT hCreateOutputDir (
    const char *pacOutputDir
        )
{
    STRING_OBJECT hOutputDir = STRING_INVALID_OBJECT;
    BOOLEAN bSuccess = FALSE;

    do
    {
        size_t tSize;
        UN8 un8FileAttributes = 0;

        /***********************************/
        /*      Validate path provided     */
        /***********************************/

        // Validate path pointer
        if (pacOutputDir == NULL)
        {
            break;
        }

        // Validate length of given path
        tSize = strlen(pacOutputDir);
        if (tSize == 0)
        {
            break;
        }

        // Check on the status of the output directory
        // create it if it doesn't exist
        bSuccess = OSAL.bFileSystemGetFileAttributes(
            pacOutputDir, &un8FileAttributes);
        if (bSuccess == TRUE)
        {
            // Somethere is here -- what is it?
            if ((un8FileAttributes & OSAL_FILE_ATTR_DIRECTORY)
                != OSAL_FILE_ATTR_DIRECTORY)
            {
                // Can't use a file
                bSuccess = FALSE;
            }
        }
        else
        {
            // Create the directory
            bSuccess = OSAL.bFileSystemMakeDir(pacOutputDir);
        }

        if (bSuccess == FALSE)
        {
            break;
        }

        /***********************************/
        /*       Create output paths       */
        /***********************************/

        // Create the output directory string object
        hOutputDir = STRING.hCreate(pacOutputDir, tSize);
        if (hOutputDir == STRING_INVALID_OBJECT)
        {
            break;
        }

        // If the path given didn't end with a delimiter
        // we need to add one
        if ((pacOutputDir[tSize - 1] != '/') &&
            (pacOutputDir[tSize - 1] != '\\'))
        {
            // Append delimiter
            tSize = STRING.tAppendCStr(hOutputDir, PHONETICS_DEFAULT_FS_DELIM);
            if (tSize == 0)
            {
                bSuccess = FALSE;
                break;
            }
        }

    } while (FALSE);

    if ((bSuccess == FALSE) &&
        (hOutputDir != STRING_INVALID_OBJECT))
    {
        STRING.vDestroy(hOutputDir);
        hOutputDir = STRING_INVALID_OBJECT;
    }

    return hOutputDir;
}

/*****************************************************************************
*
*   bReadIdFromFile
*
*   Performs all the common work of reading a category / service id from
*   a phonetics dictionary file.
*
*****************************************************************************/
static BOOLEAN bReadIdFromFile (
    UN32 *pun32Id,
    FILE *psFile
        )
{
    char acBuffer[PHONETICS_MAX_ID_CHAR_LEN + 1];
    UN8 un8Index = 0;
    BOOLEAN bIdRead = FALSE;
    size_t tLen;

    // Null terminate
    acBuffer[PHONETICS_MAX_ID_CHAR_LEN] = '\0';

    for (un8Index = 0; un8Index < PHONETICS_MAX_ID_CHAR_LEN; un8Index++)
    {
        // Read another character
        tLen = fread(&acBuffer[un8Index], sizeof(char), 1, psFile);
        if(tLen != 1)
        {
            // End of file or error occurred
            break;
        }

        // Did we find the end of the number-string?
        if (acBuffer[un8Index] == '\0')
        {
            // Yes! Did we collect any characters though?
            if (un8Index > 0)
            {
                // Yes, at least one
                bIdRead = TRUE;
            }

            break;
        }
    }

    if (bIdRead == TRUE)
    {
        // Read this ID an a integer (decimal)
        *pun32Id = (UN32)strtoul(&acBuffer[0], NULL, 10);
    }

    return bIdRead;
}

/*****************************************************************************
*
*   bHandleNewPhoneticsList
*
*   This function is called when the manager detects there is a new list of
*   phonetics which needs to be processed.
*
*****************************************************************************/
static BOOLEAN bHandleNewPhoneticsList (
    PHONETICS_MGR_OBJECT_STRUCT *psObj
        )
{
    BOOLEAN bSuccess;

    // Process the new list
    bSuccess = bProcessNewPhoneticsList(
        psObj->hSQLRefConnection,
        psObj->hSQLPersistConnection,
        psObj->psWorkUnit);

    if (bSuccess == TRUE)
    {
        PHONETICS_FILE_OUTPUT_STRUCT *psFileOutput;
        UN8 un8Index;

        for (un8Index = 0; un8Index < PHONETICS_MAX_LANGUAGES; un8Index++)
        {
            // Grab the current file output structure
            psFileOutput = &psObj->psWorkUnit->asFiles[un8Index];

            if (psFileOutput->bDataPresentInFiles == TRUE)
            {
                if (NULL != psObj->vFileCallback)
                {
                    // Notify the application now
                    psObj->vFileCallback((PHONETICS_SERVICE_OBJECT)psObj,
                        psObj->psWorkUnit->eType,
                        psFileOutput->eLanguage,
                        STRING.pacCStr(psFileOutput->hTTSFilePath),
                        STRING.pacCStr(psFileOutput->hRecFilePath),
                        psObj->pvFileCallbackArg);
                }
            }
            else
            {
                const char *pacFileName;

                // Get rid of the voice rec file
                pacFileName = STRING.pacCStr(psFileOutput->hRecFilePath);

                if (pacFileName != NULL)
                {
                    remove(pacFileName);
                }

                // Get rid of the TTS file
                pacFileName = STRING.pacCStr(psFileOutput->hTTSFilePath);

                if (pacFileName != NULL)
                {
                    remove(pacFileName);
                }
            }
        }
    }

    // Whatever happened, it is time to destroy the work unit
    vClearWorkUnit(psObj->psWorkUnit);
    SMSO_vDestroy((SMS_OBJECT)psObj->psWorkUnit);

    psObj->psWorkUnit = (PHONETICS_WORK_UNIT_STRUCT *)NULL;

    return bSuccess;
}

/*****************************************************************************
*
*   bNewListLocal
*
*   This function handles all initial processing necessary when a new
*   phonetics list is available.
*
*****************************************************************************/
static BOOLEAN bNewListLocal (
    PHONETICS_SERVICE_OBJECT hPhoneticsService,
    PHONETICS_WORK_UNIT_STRUCT *psWorkUnit,
    STRING_OBJECT hTTSOutputDir,
    STRING_OBJECT hRecOutputDir
        )
{
    // The caller has performed all validation required
    const char *pacFileName = NULL;
    BOOLEAN bCreatePool = FALSE;

    switch (psWorkUnit->eType)
    {
        case PHONETICS_FILE_TYPE_CHANNELS:
        {
            pacFileName = PHONETICS_CHANNEL_FILENAME;
            psWorkUnit->pacSQLQuery = PHONETICS_SELECT_CHANNELS;
            psWorkUnit->eDefVersion = PHONETICS_DB_VERSION_TYPE_CHAN_DEF;
            psWorkUnit->eContentVersion = PHONETICS_DB_VERSION_TYPE_CHAN_CONTENT;
        }
        break;

        case PHONETICS_FILE_TYPE_CATEGORIES:
        {
            pacFileName = PHONETICS_CATEGORY_FILENAME;
            psWorkUnit->pacSQLQuery = PHONETICS_SELECT_CATEGORIES;
            psWorkUnit->eDefVersion = PHONETICS_DB_VERSION_TYPE_CAT_DEF;
            psWorkUnit->eContentVersion = PHONETICS_DB_VERSION_TYPE_CAT_CONTENT;
        }
        break;

        case PHONETICS_FILE_TYPE_MARKETS:
        {
            pacFileName = PHONETICS_MARKET_FILENAME;
            psWorkUnit->pacSQLQuery = PHONETICS_SELECT_MARKETS;
            psWorkUnit->eDefVersion = PHONETICS_DB_VERSION_TYPE_MARKET_DEF;
            psWorkUnit->eContentVersion = PHONETICS_DB_VERSION_TYPE_MARKET_CONTENT;
            psWorkUnit->uCidCreator.hMarketCreator = REPORT_hCreateMarketCid();
            bCreatePool = TRUE;
        }
        break;

        case PHONETICS_FILE_TYPE_TEAMS:
        {
            pacFileName = PHONETICS_TEAM_FILENAME;
            psWorkUnit->pacSQLQuery = PHONETICS_SELECT_TEAMS;
            psWorkUnit->eDefVersion = PHONETICS_DB_VERSION_TYPE_TEAM_DEF;
            psWorkUnit->eContentVersion = PHONETICS_DB_VERSION_TYPE_TEAM_CONTENT;
            bCreatePool = TRUE;
        }
        break;

        case PHONETICS_FILE_TYPE_LEAGUES:
        {
            pacFileName = PHONETICS_LEAGUE_FILENAME;
            psWorkUnit->pacSQLQuery = PHONETICS_SELECT_LEAGUES;
            psWorkUnit->eDefVersion = PHONETICS_DB_VERSION_TYPE_LEAGUE_DEF;
            psWorkUnit->eContentVersion = PHONETICS_DB_VERSION_TYPE_LEAGUE_CONTENT;
            bCreatePool = TRUE;
        }
        break;

        default:
        {
            // We don't understand this file type!
            return FALSE;
        }
    }

    do
    {
        BOOLEAN bOk = TRUE;
        UN8 un8Index;

        // Create a cid pool if necessary
        if (bCreatePool == TRUE)
        {
            psWorkUnit->hCidPool = CID_hCreatePool(SMS_INVALID_OBJECT);
            if (psWorkUnit->hCidPool == CID_POOL_INVALID_OBJECT)
            {
                break;
            }
        }
        else
        {
            // Make sure this is cleared
            psWorkUnit->hCidPool = CID_POOL_INVALID_OBJECT;
        }

        for (un8Index = 0; un8Index < PHONETICS_MAX_LANGUAGES; un8Index++)
        {
            // Create the file paths
            bOk = bCreateFilePaths(
                hPhoneticsService,
                pacFileName,
                hTTSOutputDir, hRecOutputDir,
                &psWorkUnit->asFiles[un8Index]);

            if (bOk == FALSE)
            {
                break;
            }
        }

        if (bOk == FALSE)
        {
            break;
        }

        return TRUE;
    } while (FALSE);

    vClearWorkUnit(psWorkUnit);

    return FALSE;
}

/*****************************************************************************
*
*   bCreateFilePaths
*
*   This function is used to cross-reference new phonetics table data
*   with the information in the reference database.
*
*****************************************************************************/
static BOOLEAN bCreateFilePaths (
    PHONETICS_SERVICE_OBJECT hPhoneticsService,
    const char *pacFilename,
    STRING_OBJECT hTTSOutputDir,
    STRING_OBJECT hRecOutputDir,
    PHONETICS_FILE_OUTPUT_STRUCT *psFiles
        )
{
    const char *pacLangIndicator = NULL;

    switch (psFiles->eLanguage)
    {
        case SMS_LANGUAGE_ENGLISH:
        {
            pacLangIndicator = PHONETICS_ENGLISH_FILE;
        }
        break;

        case SMS_LANGUAGE_SPANISH:
        {
            pacLangIndicator = PHONETICS_SPANISH_FILE;
        }
        break;

        case SMS_LANGUAGE_FRENCH:
        {
            pacLangIndicator = PHONETICS_FRENCH_FILE;
        }
        break;

        default:
        {
            return FALSE;
        }
    }

    do
    {
        size_t tAppended;

        /* Create the TTS File Path */

        // Create the string object now
        psFiles->hTTSFilePath =
            STRING_hCreate(
                (SMS_OBJECT)hPhoneticsService,
                STRING.pacCStr(hTTSOutputDir),
                STRING.tSize(hTTSOutputDir), 0);
        if (psFiles->hTTSFilePath == STRING_INVALID_OBJECT)
        {
            break;
        }

        // Append the file name
        tAppended = STRING.tAppendCStr(psFiles->hTTSFilePath, pacFilename);
        if (tAppended == 0)
        {
            break;
        }

        // Append the language indication
        tAppended = STRING.tAppendCStr(psFiles->hTTSFilePath, pacLangIndicator);
        if (tAppended == 0)
        {
            break;
        }

        // Append the file extension
        tAppended = STRING.tAppendCStr(psFiles->hTTSFilePath, PHONETICS_FILE_EXTENSION);
        if (tAppended == 0)
        {
            break;
        }

        /* Create the Voice Rec File Path */

        // Create the string object now
        psFiles->hRecFilePath =
            STRING_hCreate(
                (SMS_OBJECT)hPhoneticsService,
                STRING.pacCStr(hRecOutputDir),
                STRING.tSize(hRecOutputDir), 0);
        if (psFiles->hRecFilePath == STRING_INVALID_OBJECT)
        {
            break;
        }

        // Append the file name
        tAppended = STRING.tAppendCStr(psFiles->hRecFilePath, pacFilename);
        if (tAppended == 0)
        {
            break;
        }

        // Append the language indication
        tAppended = STRING.tAppendCStr(psFiles->hRecFilePath, pacLangIndicator);
        if (tAppended == 0)
        {
            break;
        }

        // Append the file extension
        tAppended = STRING.tAppendCStr(psFiles->hRecFilePath, PHONETICS_FILE_EXTENSION);
        if (tAppended == 0)
        {
            break;
        }

        return TRUE;
    } while (FALSE);

    // Clean up any memory we left behind
    vDestroyFilePaths(psFiles);

    return FALSE;
}

/*****************************************************************************
*
*   vDestroyFilePaths
*
*****************************************************************************/
static void vDestroyFilePaths (
    PHONETICS_FILE_OUTPUT_STRUCT *psFiles
        )
{
    STRING_vDestroy(psFiles->hRecFilePath);
    psFiles->hRecFilePath = STRING_INVALID_OBJECT;

    STRING_vDestroy(psFiles->hTTSFilePath);
    psFiles->hTTSFilePath = STRING_INVALID_OBJECT;

    return;
}

/*****************************************************************************
*
*   bOpenOutputFiles
*
*****************************************************************************/
static BOOLEAN bOpenOutputFiles (
    PHONETICS_FILE_OUTPUT_STRUCT *psFiles
        )
{
    do
    {
        const char *pacFilename;

        // Get the TTS filename
        pacFilename = STRING.pacCStr(psFiles->hTTSFilePath);

        if (pacFilename == NULL)
        {
            break;
        }

        // Open the TTS output file now
        psFiles->psTTSFile = fopen(
            pacFilename, PHONETICS_FILE_MODE);

        if (psFiles->psTTSFile == (FILE *)NULL)
        {
            // Nope!
            return FALSE;
        }

        // Get the voice rec filename
        pacFilename = STRING.pacCStr(psFiles->hRecFilePath);

        if (pacFilename == NULL)
        {
            break;
        }


        // Open the voice rec file
        psFiles->psRecFile = fopen(
            pacFilename, PHONETICS_FILE_MODE);

        if (psFiles->psRecFile == (FILE *)NULL)
        {
            // Nope!
            return FALSE;
        }

        return TRUE;
    } while (FALSE);

    return FALSE;
}

/*****************************************************************************
*
*   vCloseOutputFiles
*
*****************************************************************************/
static void vCloseOutputFiles (
    PHONETICS_FILE_OUTPUT_STRUCT *psFiles
        )
{
    if (psFiles->psTTSFile != (FILE *)NULL)
    {
        fclose(psFiles->psTTSFile);
        psFiles->psTTSFile = (FILE *)NULL;
    }

    if (psFiles->psRecFile != (FILE *)NULL)
    {
        fclose(psFiles->psRecFile);
        psFiles->psRecFile = (FILE *)NULL;
    }

    return;
}

/*****************************************************************************
*
*   bProcessNewPhoneticsList
*
*   This function is used to cross-reference new phonetics table data
*   with the information in the reference database.
*
*****************************************************************************/
static BOOLEAN bProcessNewPhoneticsList (
    SQL_INTERFACE_OBJECT hRefDBConnection,
    SQL_INTERFACE_OBJECT hPersistDBConnection,
    PHONETICS_WORK_UNIT_STRUCT *psWorkUnit
        )
{
    BOOLEAN bSuccess = FALSE,
            bInTransaction = FALSE;
    PHONETICS_DB_PROCESS_STRUCT sProcess;
    UN8 un8Index;

    // Initialize processing structure
    sProcess.psWorkUnit = psWorkUnit;
    sProcess.bSuccess = FALSE;
    sProcess.psEntry = (PHONETICS_LIST_NODE_STRUCT *)NULL;
    sProcess.hEntry = OSAL.hLinkedListFirst(
        psWorkUnit->hListData, (void **)&sProcess.psEntry);

    do
    {
        BOOLEAN bFilesOpened = FALSE;

        for (un8Index = 0; un8Index < PHONETICS_MAX_LANGUAGES; un8Index++)
        {
            // Open files for all languages
            bFilesOpened = bOpenOutputFiles(&psWorkUnit->asFiles[un8Index]);
            if (bFilesOpened == FALSE)
            {
                break;
            }
        }

        if (bFilesOpened == FALSE)
        {
            break;
        }

        // Now, select the appropriate phonetics table
        // from the reference DB and generate output files
        bSuccess = SQL_INTERFACE.bQuery(
            hRefDBConnection,
            psWorkUnit->pacSQLQuery,
            (SQL_QUERY_RESULT_HANDLER)bProcessSelectPhonetics,
            (void *)&sProcess);
        if (bSuccess == FALSE)
        {
            break;
        }

        // We cross referenced all OTA data with the DB data, but
        // we may have more to do with the OTA data, so do it now
        if (sProcess.hEntry != OSAL_INVALID_LINKED_LIST_ENTRY)
        {
            while ((sProcess.hEntry != OSAL_INVALID_LINKED_LIST_ENTRY) &&
                   (sProcess.psEntry != (PHONETICS_LIST_NODE_STRUCT *)NULL))
            {
                // Write this data to our output files
                bSuccess = bWritePhoneticsRowByLanguage(
                    psWorkUnit,
                    &sProcess.psEntry->sID,
                    sProcess.psEntry,
                    NULL);

                if (bSuccess == FALSE)
                {
                    break;
                }

                // Get the next entry
                sProcess.hEntry = OSAL.hLinkedListNext(
                    sProcess.hEntry, (void **)&sProcess.psEntry);
            }

            if (bSuccess == FALSE)
            {
                break;
            }
        }

        // Close our files now
        for (un8Index = 0; un8Index < PHONETICS_MAX_LANGUAGES; un8Index++)
        {
            vCloseOutputFiles(&psWorkUnit->asFiles[un8Index]);
        }

        // Update the persistent db with this new table version
        // if we have a connection to it
        if (hPersistDBConnection != SQL_INTERFACE_INVALID_OBJECT)
        {
            char acVersion[PHONETICS_SQL_COMMAND_MAX_LEN];

            // Start a transaction
            bInTransaction = SQL_INTERFACE.bStartTransaction(hPersistDBConnection);
            if (bInTransaction == FALSE)
            {
                break;
            }

            // Update the table definition version
            snprintf(&acVersion[0], PHONETICS_SQL_COMMAND_MAX_LEN,
                PHONETICS_INSERT_UPDATE_VERSION_ROW,
                psWorkUnit->eDefVersion,
                psWorkUnit->un8PhoneticsFileVer);

            bSuccess = SQL_INTERFACE.bExecuteCommand(hPersistDBConnection, &acVersion[0]);
            if (bSuccess == FALSE)
            {
                break;
            }

            // Update the table content version
            snprintf(&acVersion[0], PHONETICS_SQL_COMMAND_MAX_LEN,
                PHONETICS_INSERT_UPDATE_VERSION_ROW,
                psWorkUnit->eContentVersion,
                psWorkUnit->un16PhoneticsContentVer);

            bSuccess = SQL_INTERFACE.bExecuteCommand(hPersistDBConnection, &acVersion[0]);
            if (bSuccess == FALSE)
            {
                break;
            }

            bSuccess = SQL_INTERFACE.bEndTransaction(hPersistDBConnection);
            if (bSuccess == FALSE)
            {
                break;
            }

            // We are no longer in a transaction
            bInTransaction = FALSE;
        }

    } while (FALSE);

    // Are we still in the middle of a transaction
    if (bInTransaction == TRUE)
    {
        // Try to clear it, but don't worry about the return
        // code since we're already in error
        SQL_INTERFACE.bEndTransaction(hPersistDBConnection);
    }

    // Close our files now
    for (un8Index = 0; un8Index < PHONETICS_MAX_LANGUAGES; un8Index++)
    {
        vCloseOutputFiles(&psWorkUnit->asFiles[un8Index]);
    }

    return bSuccess;
}

/*****************************************************************************
*
*   psCreateWorkUnit
*
*   Creates / initializes new work unit
*
*****************************************************************************/
static PHONETICS_WORK_UNIT_STRUCT *psCreateWorkUnit (
    PHONETICS_MGR_OBJECT_STRUCT *psObj,
    UN8 un8PhoneticsFileVer,
    UN16 un16PhoneticsContentVer,
    PHONETICS_FILE_TYPE_ENUM eType,
    OSAL_OBJECT_HDL hPhoneticsList
        )
{
    PHONETICS_WORK_UNIT_STRUCT *psWorkUnit;

    // Allocate the work unit
    psWorkUnit = (PHONETICS_WORK_UNIT_STRUCT *)
        SMSO_hCreate(PHONETICS_MGR_OBJECT_NAME":Work Unit",
        sizeof(PHONETICS_WORK_UNIT_STRUCT),
        DATASERVICE_IMPL_hSMSObj((DATASERVICE_IMPL_HDL)psObj),
        FALSE);
    if (psWorkUnit != (PHONETICS_WORK_UNIT_STRUCT *)NULL)
    {
        UN8 un8Index;

        // Initialize the work unit
        psWorkUnit->un8PhoneticsFileVer = un8PhoneticsFileVer;
        psWorkUnit->un16PhoneticsContentVer = un16PhoneticsContentVer;
        psWorkUnit->eType = eType;
        psWorkUnit->hListData = hPhoneticsList;

        for (un8Index = 0; un8Index < PHONETICS_MAX_LANGUAGES; un8Index++)
        {
            // Indicate there is nothing in these files yet
            psWorkUnit->asFiles[un8Index].bDataPresentInFiles = FALSE;

            // Set the language
            psWorkUnit->asFiles[un8Index].eLanguage =
                (SMS_LANGUAGE_ENUM)((SMS_LANGUAGE_ENGLISH + un8Index));
        }
    }

    return psWorkUnit;
}

/*****************************************************************************
*
*   vClearWorkUnit
*
*   Free all memory associated with this work unit, but don't free the
*   work unit itself since it may have been allocated on the stack
*
*****************************************************************************/
static void vClearWorkUnit (
    PHONETICS_WORK_UNIT_STRUCT *psWorkUnit
        )
{
    UN8 un8Index;

    if (psWorkUnit == NULL)
    {
        return;
    }

    if (psWorkUnit->hCidPool != CID_POOL_INVALID_OBJECT)
    {
        CID_vDestroyPool(psWorkUnit->hCidPool);
        psWorkUnit->hCidPool = CID_POOL_INVALID_OBJECT;
    }

    if (psWorkUnit->hListData != OSAL_INVALID_OBJECT_HDL)
    {
        OSAL_RETURN_CODE_ENUM eReturnCode;

        eReturnCode = OSAL.eLinkedListIterate(
            psWorkUnit->hListData,
            (OSAL_LL_ITERATOR_HANDLER)bRemovePhoneticNode,
            NULL);

        if (eReturnCode == OSAL_SUCCESS)
        {
            OSAL.eLinkedListDelete(psWorkUnit->hListData);
            psWorkUnit->hListData = OSAL_INVALID_OBJECT_HDL;
        }
    }

    // Destroy all file paths we created
    for (un8Index = 0; un8Index < PHONETICS_MAX_LANGUAGES; un8Index++)
    {
        vDestroyFilePaths(&psWorkUnit->asFiles[un8Index]);
    }

    // Clear these attributes as well
    psWorkUnit->eType = PHONETICS_FILE_INVALID_TYPE;
    psWorkUnit->pacSQLQuery = NULL;
    psWorkUnit->eDefVersion = PHONETICS_DB_VERSION_MAX_TYPES;
    psWorkUnit->eContentVersion = PHONETICS_DB_VERSION_MAX_TYPES;
    psWorkUnit->un8PhoneticsFileVer = 0;
    psWorkUnit->un16PhoneticsContentVer = 0;

    return;
}

/*****************************************************************************
*
*   n16CompareRowToNode
*
*****************************************************************************/
static N16 n16CompareRowToNode (
    PHONETICS_PHONETIC_ROW_STRUCT *psDBRow,
    PHONETICS_LIST_NODE_STRUCT *psNode
        )
{
    N16 n16Return;
    PHONETICS_LIST_NODE_STRUCT sNodeFromRow;

    // Copy the id information
    sNodeFromRow.sID = psDBRow->sID;

    // Compare using the DB as the baseline
    n16Return = n16ComparePhoneticNode(&sNodeFromRow, psNode);

    return n16Return;
}

/*****************************************************************************
*
*   bWritePhoneticsRowByLanguage
*
*****************************************************************************/
static BOOLEAN bWritePhoneticsRowByLanguage (
    PHONETICS_WORK_UNIT_STRUCT *psWorkUnit,
    PHONETICS_PHONETIC_ID_STRUCT *psID,
    PHONETICS_LIST_NODE_STRUCT *psOTAPhonetics,
    PHONETICS_PHONETIC_ROW_STRUCT *psDBPhonetics
        )
{
    UN8 un8Index;
    STRING_OBJECT hOTAPhonetics = STRING_INVALID_OBJECT;
    const char *pacDBPhonetics = (const char *)NULL;
    BOOLEAN bSuccess = FALSE;
    BOOLEAN bWriteToFile, bCheckDBPhoneticsLen;
    size_t tDBPhoneticsLen;

    for (un8Index = 0;
         un8Index < PHONETICS_MAX_LANGUAGES;
         un8Index++)
    {
        // Plan on writing to the file
        bWriteToFile = TRUE;

        // Don't plan on checking the length of the DB phonetics
        bCheckDBPhoneticsLen = FALSE;

        if (psOTAPhonetics != NULL)
        {
            // Grab the new phonetics from OTA if available
            hOTAPhonetics = psOTAPhonetics->ahPhonetics[un8Index];

            // If OTA is telling us there are no phonetics
            // for this language/entry combo then we're
            // not going to write this entry to this file
            if (hOTAPhonetics == STRING_INVALID_OBJECT)
            {
                bWriteToFile = FALSE;
            }
        }
        else
        {
            // Nothing from OTA about this entry, see
            // if the DB has anything for us
            bCheckDBPhoneticsLen = TRUE;
        }

        if (psDBPhonetics != NULL)
        {
            // Grab the DB phonetics if available
            pacDBPhonetics = psDBPhonetics->pacPhonetics[un8Index];

            // Do we still need to determine if we're gonna write to this file?
            if (bCheckDBPhoneticsLen == TRUE)
            {
                // Yes.  See if we have anything for this language
                // from the database
                tDBPhoneticsLen = strlen(pacDBPhonetics);
                if (tDBPhoneticsLen == 0)
                {
                    bWriteToFile = FALSE;
                }
            }
        }

        if (bWriteToFile == FALSE)
        {
            // Not a problem, but we're done with this language
            bSuccess = TRUE;
            continue;
        }

        // We will write this entry to the file if:
        // the OTA phonetics are present, or
        // if the DB phonetics

        // Write this language's data now
        bSuccess = bWritePhoneticsRow(
            psWorkUnit, psID,
            pacDBPhonetics, hOTAPhonetics,
            &psWorkUnit->asFiles[un8Index]);

        if (bSuccess == FALSE)
        {
            break;
        }

        // There is now data in these files
        psWorkUnit->asFiles[un8Index].bDataPresentInFiles = TRUE;
    }

    return bSuccess;
}

/*****************************************************************************
*
*   bWritePhoneticsRow
*
*****************************************************************************/
static BOOLEAN bWritePhoneticsRow (
    PHONETICS_WORK_UNIT_STRUCT *psWorkUnit,
    PHONETICS_PHONETIC_ID_STRUCT *psID,
    const char *pacDBPhonetics,
    STRING_OBJECT hNewPhonetics,
    PHONETICS_FILE_OUTPUT_STRUCT *psOutputFiles
        )
{
    BOOLEAN bSuccess = FALSE;

    // Write ID (type - specific)
    switch (psWorkUnit->eType)
    {
        case PHONETICS_FILE_TYPE_CHANNELS:
        case PHONETICS_FILE_TYPE_CATEGORIES:
        {
            BOOLEAN bCategory = (psWorkUnit->eType == PHONETICS_FILE_TYPE_CATEGORIES);

            // Write Service ID / Category ID
            bSuccess = bWriteSvcCatID(
                psID->un32Id,
                bCategory,
                psOutputFiles);
        }
        break;

        case PHONETICS_FILE_TYPE_MARKETS:
        {
            bSuccess = bWriteMarketID(
                psWorkUnit->hCidPool,
                psWorkUnit->uCidCreator.hMarketCreator,
                psID->un32Id,
                psOutputFiles);
        }
        break;

        case PHONETICS_FILE_TYPE_TEAMS:
        {
            bSuccess = bWriteTeamID (
                psWorkUnit->hCidPool,
                psID->un32Id,
                psID->un8NumLeagues,
                &psID->aun16Leagues[0],
                psOutputFiles);
        }
        break;

        case PHONETICS_FILE_TYPE_LEAGUES:
        {
            bSuccess = bWriteLeagueID(
                psWorkUnit->hCidPool,
                psID->un32Id,
                psOutputFiles);
        }
        break;

        default:
        {
            // That's not good
            bSuccess = FALSE;
        }
        break;
    }

    if (bSuccess == TRUE)
    {
        // Write the c-string portion of this entry now
        bSuccess = bWritePhoneticsString(
            hNewPhonetics, pacDBPhonetics,
            psOutputFiles);
    }

    return bSuccess;
}

/*****************************************************************************
*
*   bWriteSvcCatID
*
*****************************************************************************/
static BOOLEAN bWriteSvcCatID (
    UN32 un32Id,
    BOOLEAN bCategory,
    PHONETICS_FILE_OUTPUT_STRUCT *psOutputFiles
        )
{
    if (bCategory == FALSE)
    {
        SERVICE_ID tServiceId = (SERVICE_ID)un32Id;

        // Write this id to both files
        fprintf(psOutputFiles->psTTSFile, "%u", tServiceId);
        fprintf(psOutputFiles->psRecFile, "%u", tServiceId);
    }
    else
    {
        CATEGORY_ID tCategoryId = (CATEGORY_ID)un32Id;

        fprintf(psOutputFiles->psTTSFile, "%u", tCategoryId);
        fprintf(psOutputFiles->psRecFile, "%u", tCategoryId);
    }

    // Add null terminator to identify the end of the string
    fwrite("\0", 1, 1, psOutputFiles->psTTSFile);
    fwrite("\0", 1, 1, psOutputFiles->psRecFile);

    return TRUE;
}

/*****************************************************************************
*
*   bWriteMarketID
*
*****************************************************************************/
static BOOLEAN bWriteMarketID (
    CID_POOL hCidPool,
    MARKET_CID_CREATOR hMarketCreator,
    UN32 un32Id,
    PHONETICS_FILE_OUTPUT_STRUCT *psOutputFiles
        )
{
    N32 n32NumWritten = 0;
    CID_OBJECT hCID;

    hCID = hMarketCreator(hCidPool, un32Id);
    if (hCID != CID_INVALID_OBJECT)
    {
        n32NumWritten = CID.n32FWrite(hCID, psOutputFiles->psTTSFile);

        if (n32NumWritten > 0)
        {
            n32NumWritten = CID.n32FWrite(hCID, psOutputFiles->psRecFile);
        }
    }

    if (hCID != CID_INVALID_OBJECT)
    {
        // Back to the pool!
        CID_vDestroy(hCID);
    }

    return (n32NumWritten > 0);
}

/*****************************************************************************
*
*   bWriteLeagueID
*
*****************************************************************************/
static BOOLEAN bWriteLeagueID (
    CID_POOL hCidPool,
    UN32 un32Id,
    PHONETICS_FILE_OUTPUT_STRUCT *psOutputFiles
        )
{
    N32 n32NumWritten = 0;
    CID_OBJECT hCID;

    hCID = LEAGUE_hCreateCid(hCidPool, &un32Id);
    if (hCID != CID_INVALID_OBJECT)
    {
        n32NumWritten = CID.n32FWrite(hCID, psOutputFiles->psTTSFile);

        if (n32NumWritten > 0)
        {
            n32NumWritten = CID.n32FWrite(hCID, psOutputFiles->psRecFile);
        }
    }

    if (hCID != CID_INVALID_OBJECT)
    {
        // Back to the pool!
        CID_vDestroy(hCID);
    }

    return (n32NumWritten > 0);
}

/*****************************************************************************
*
*   bWriteTeamID
*
*****************************************************************************/
static BOOLEAN bWriteTeamID (
    CID_POOL hCidPool,
    UN32 un32TeamId,
    UN8 un8NumLeagues,
    UN16 *pun16Leagues,
    PHONETICS_FILE_OUTPUT_STRUCT *psOutputFiles
        )
{
    N32 n32NumWritten = 0;
    BOOLEAN bSuccess = TRUE;
    CID_OBJECT hCID, hEquivCID;

    // Create the initial team cid using the first league
    hCID = TEAM_hCreateCid(hCidPool, &un32TeamId);
    if (hCID != CID_INVALID_OBJECT)
    {
        UN8 un8Index;

        // Skip the first league since we already used it
        for (un8Index = 1; un8Index < un8NumLeagues; un8Index++)
        {
            // Create the next cid
            hEquivCID = TEAM_hCreateCid(
                hCidPool,
                &un32TeamId);

            if (hEquivCID == CID_INVALID_OBJECT)
            {
                // Stop here
                break;
            }

            // The original CID should take this one as
            // an equivalent cid
            bSuccess = CID.bSetEquivalent(hCID, hEquivCID);

            // Always destroy it (send back to pool)
            CID_vDestroy(hEquivCID);

            if (bSuccess == FALSE)
            {
                break;
            }
        }

        // Did that go well?
        if (bSuccess == TRUE)
        {
            // Yes! Write the cid to the files now
            n32NumWritten = CID.n32FWrite(hCID, psOutputFiles->psTTSFile);

            if (n32NumWritten > 0)
            {
                n32NumWritten = CID.n32FWrite(hCID, psOutputFiles->psRecFile);
            }

            if (n32NumWritten <= 0)
            {
                bSuccess = FALSE;
            }
        }
    }

    if (hCID != CID_INVALID_OBJECT)
    {
        // Back to the pool!
        CID_vDestroy(hCID);
    }

    return bSuccess;
}

/*****************************************************************************
*
*   bWritePhoneticsString
*
*****************************************************************************/
static BOOLEAN bWritePhoneticsString (
    STRING_OBJECT hNewPhonetics,
    const char *pacDBPhonetics,
    PHONETICS_FILE_OUTPUT_STRUCT *psOutputFiles
        )
{
    // Initialize with the phonetics from the database
    const char *pacSource = pacDBPhonetics;
    const char *pacEndFirstPhonetics;
    ptrdiff_t tFirstPhoneticsLen;
    size_t tPhoneticsLen;

    // Did we get an update to the DB phonetics?
    if (hNewPhonetics != STRING_INVALID_OBJECT)
    {
        // Yeah, use that
        pacSource = STRING.pacCStr(hNewPhonetics);
    }

    // Make sure we have something here!
    if (pacSource == NULL)
    {
        return FALSE;
    }

    // Determine how long this is
    tPhoneticsLen = strlen(pacSource);

    // Find the location of the first delimiter
    pacEndFirstPhonetics = strstr(pacSource, PHONETICS_DELIMITER_STRING);

    if (pacEndFirstPhonetics == NULL)
    {
        // No delimiter found -- so we only have one
        // piece of phonetics here already
        tFirstPhoneticsLen = tPhoneticsLen;
    }
    else
    {
        // Compute the length of the first entry
        tFirstPhoneticsLen = pacEndFirstPhonetics - pacSource;
    }

    // Write the delimiter
    fwrite((void *)PHONETICS_DELIMITER_STRING, sizeof(char), 1,
        psOutputFiles->psTTSFile);
    fwrite((void *)PHONETICS_DELIMITER_STRING, sizeof(char), 1,
        psOutputFiles->psRecFile);

    // Write the data to the tts file & rec file
    fwrite((void *)&pacSource[0],sizeof(char),tFirstPhoneticsLen,
        psOutputFiles->psTTSFile);
    fwrite((void *)&pacSource[0],sizeof(char),tPhoneticsLen,
        psOutputFiles->psRecFile);

    fprintf(psOutputFiles->psTTSFile, "\n");
    fprintf(psOutputFiles->psRecFile, "\n");

    return TRUE;
}

/*****************************************************************************
*
*   bCreatePersistDBTables
*
*****************************************************************************/
static BOOLEAN bCreatePersistDBTables (
    SQL_INTERFACE_OBJECT hSQLConnection,
    PHONETICS_VER_CTRL_STRUCT *psVerCtrl
        )
{
    BOOLEAN bSuccess = FALSE,
            bTransactionStarted = FALSE;

    do
    {
        size_t tIndex;
        char acVersion[PHONETICS_SQL_COMMAND_MAX_LEN];
        PHONETICS_DB_VERSION_TYPES_ENUM eVersionType =
            PHONETICS_DB_VERSION_TYPE_DB_SCHEMA;

        if (psVerCtrl == NULL)
        {
            break;
        }

        bTransactionStarted = SQL_INTERFACE.bStartTransaction(hSQLConnection);

        if (bTransactionStarted == FALSE)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                PHONETICS_MGR_OBJECT_NAME
                ": failed to start transaction");
            break;
        }

        // Create the version table
        bSuccess = SQL_INTERFACE.bExecuteCommand(hSQLConnection,
            PHONETICS_CREATE_VERSION_TABLE);

        if (bSuccess == FALSE)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                PHONETICS_MGR_OBJECT_NAME
                ": failed to create prices table in persistant storage");
            break;
        }

        // Insert version row data for the database schema
        // Build the version string
        snprintf(&acVersion[0], PHONETICS_SQL_COMMAND_MAX_LEN,
            PHONETICS_INSERT_UPDATE_VERSION_ROW,
            eVersionType++,
            PHONETICS_DATABASE_FILE_VERSION);

        // Insert this row
        bSuccess = SQL_INTERFACE.bExecuteCommand(
            hSQLConnection, &acVersion[0]);

        if (bSuccess == FALSE)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                PHONETICS_MGR_OBJECT_NAME
                ": failed to set db version in persistent storage");
            break;
        }

        // Insert version row data for the phonetics tables
        for (tIndex = 0; tIndex < PHONETICS_NUM_TABLES; tIndex++)
        {
            // Build the version string
            snprintf(&acVersion[0], PHONETICS_SQL_COMMAND_MAX_LEN,
                PHONETICS_INSERT_UPDATE_VERSION_ROW,
                eVersionType++,
                psVerCtrl->asTableVer[tIndex].un8TableDefVer);

            // Insert this row
            bSuccess = SQL_INTERFACE.bExecuteCommand(
                hSQLConnection, &acVersion[0]);

            if (bSuccess == TRUE)
            {
                // Build the version string
                snprintf(&acVersion[0], PHONETICS_SQL_COMMAND_MAX_LEN,
                    PHONETICS_INSERT_UPDATE_VERSION_ROW,
                    eVersionType++,
                    psVerCtrl->asTableVer[tIndex].un16TableContentVer);

                // Insert this row
                bSuccess = SQL_INTERFACE.bExecuteCommand(
                    hSQLConnection, &acVersion[0]);
            }

            if (bSuccess == FALSE)
            {
                break;
            }
        }
    } while (FALSE);

    if (bTransactionStarted == TRUE)
    {
        BOOLEAN bTransactionEnded;

        // End transaction
        bTransactionEnded = SQL_INTERFACE.bEndTransaction(hSQLConnection);

        if (bTransactionEnded == FALSE)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                PHONETICS_MGR_OBJECT_NAME
                ": failed to end transaction");
            bSuccess = FALSE;
        }
    }

    return bSuccess;
}

/*****************************************************************************
*
*   bVerifyAndLoadDBVersionFields
*
*****************************************************************************/
static BOOLEAN bVerifyAndLoadDBVersionFields (
    SQL_INTERFACE_OBJECT hSQLConnection,
    PHONETICS_VER_CTRL_STRUCT *psVerCtrl
        )
{
    BOOLEAN bVersionMatched = FALSE, bOk;
    PHONETICS_DB_VERSION_RESULT_STRUCT sVersion;

    // Initialize the result structure
    sVersion.bSuccess = FALSE;
    sVersion.psVerCtrl = psVerCtrl;

    // Perform the SQL query and process the result
    // (it will provide us with a data row)
    bOk = SQL_INTERFACE.bQuery(
             hSQLConnection, PHONETICS_SELECT_ALL_DB_VERSION,
            (SQL_QUERY_RESULT_HANDLER)bProcessSelectVersion,
            &sVersion ) ;

    if (bOk == TRUE)
    {
        bVersionMatched = sVersion.bSuccess;
    }

    return bVersionMatched;
}

/*******************************************************************************
*
*   bProcessSelectVersion
*
*   This function is provided to the SQL_INTERFACE_OBJECT in order to process
*   a "select all" query made on the database version relation.  It populates
*   a pointer to a PHONETICS_VERSION_ROW_STRUCT with the information found
*   in the database as well as performs version verification.
*
*******************************************************************************/
static BOOLEAN bProcessSelectVersion (
    SQL_QUERY_COLUMN_STRUCT *psColumn,
    N32 n32NumberOfColumns,
    PHONETICS_DB_VERSION_RESULT_STRUCT *psResult
        )
{
    PHONETICS_DB_VERSION_FIELDS_ENUM eCurrentField =
        (PHONETICS_DB_VERSION_FIELDS_ENUM)0;
    PHONETICS_VERSION_ROW_STRUCT sCurrentRow;

    // Verify input
    if (psResult == NULL)
    {
        return FALSE;
    }

    if (n32NumberOfColumns == PHONETICS_DB_VERSION_MAX_FIELDS)
    {
        // Start off by marking this a success
        psResult->bSuccess = TRUE;
    }
    else
    {
        // This is definitely not a match
        psResult->bSuccess = FALSE;
        return FALSE;
    }

    // Initialize row structure
    sCurrentRow.eType = PHONETICS_DB_VERSION_MAX_TYPES;
    sCurrentRow.n32Version = -1;

    do
    {
        switch (eCurrentField)
        {
            case PHONETICS_DB_VERSION_TYPE:
            {
                sCurrentRow.eType = (PHONETICS_DB_VERSION_TYPES_ENUM)
                    psColumn[eCurrentField].uData.sUN32.un32Data;
            }
            break;

            case PHONETICS_DB_VERSION_VER:
            {
                sCurrentRow.n32Version = (N32)
                    psColumn[eCurrentField].uData.sUN32.un32Data;
            }
            break;

            default:
            {
                SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    PHONETICS_MGR_OBJECT_NAME": bad field in ver table");
                sCurrentRow.eType = PHONETICS_DB_VERSION_MAX_TYPES;
                psResult->bSuccess = FALSE;
            }
            break;
        }
    } while ( ++eCurrentField < PHONETICS_DB_VERSION_MAX_FIELDS);

    // Now process the version data
    switch(sCurrentRow.eType)
    {
        case PHONETICS_DB_VERSION_TYPE_DB_SCHEMA:
        {
            if (sCurrentRow.n32Version != (N32)PHONETICS_DATABASE_FILE_VERSION)
            {
                printf(PHONETICS_MGR_OBJECT_NAME": DB Version Fail\n");
                psResult->bSuccess = FALSE;
            }
        }
        break;

        case PHONETICS_DB_VERSION_TYPE_CHAN_DEF:
        {
            psResult->psVerCtrl->asTableVer[PHONETICS_CHAN_TABLE_ID].
                un8TableDefVer = (UN8)sCurrentRow.n32Version;
        }
        break;

        case PHONETICS_DB_VERSION_TYPE_CHAN_CONTENT:
        {
            psResult->psVerCtrl->asTableVer[PHONETICS_CHAN_TABLE_ID].
                un16TableContentVer = (UN16)sCurrentRow.n32Version;
        }
        break;

        case PHONETICS_DB_VERSION_TYPE_CAT_DEF:
        {
            psResult->psVerCtrl->asTableVer[PHONETICS_CAT_TABLE_ID].
                un8TableDefVer = (UN8)sCurrentRow.n32Version;
        }
        break;

        case PHONETICS_DB_VERSION_TYPE_CAT_CONTENT:
        {
            psResult->psVerCtrl->asTableVer[PHONETICS_CAT_TABLE_ID].
                un16TableContentVer = (UN16)sCurrentRow.n32Version;
        }
        break;

        case PHONETICS_DB_VERSION_TYPE_MARKET_DEF:
        {
            psResult->psVerCtrl->asTableVer[PHONETICS_MARKET_TABLE_ID].
                un8TableDefVer = (UN8)sCurrentRow.n32Version;
        }
        break;

        case PHONETICS_DB_VERSION_TYPE_MARKET_CONTENT:
        {
            psResult->psVerCtrl->asTableVer[PHONETICS_MARKET_TABLE_ID].
                un16TableContentVer = (UN16)sCurrentRow.n32Version;
        }
        break;

        case PHONETICS_DB_VERSION_TYPE_TEAM_DEF:
        {
            psResult->psVerCtrl->asTableVer[PHONETICS_TEAM_TABLE_ID].
                un8TableDefVer = (UN8)sCurrentRow.n32Version;
        }
        break;

        case PHONETICS_DB_VERSION_TYPE_TEAM_CONTENT:
        {
            psResult->psVerCtrl->asTableVer[PHONETICS_TEAM_TABLE_ID].
                un16TableContentVer = (UN16)sCurrentRow.n32Version;
        }
        break;

        case PHONETICS_DB_VERSION_TYPE_LEAGUE_DEF:
        {
            psResult->psVerCtrl->asTableVer[PHONETICS_LEAGUE_TABLE_ID].
                un8TableDefVer = (UN8)sCurrentRow.n32Version;
        }
        break;

        case PHONETICS_DB_VERSION_TYPE_LEAGUE_CONTENT:
        {
            psResult->psVerCtrl->asTableVer[PHONETICS_LEAGUE_TABLE_ID].
                un16TableContentVer = (UN16)sCurrentRow.n32Version;
        }
        break;

        case PHONETICS_DB_VERSION_MAX_TYPES:
        default:
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    PHONETICS_MGR_OBJECT_NAME": bad field in ver table");
            psResult->bSuccess = FALSE;
        }
    }

    // FALSE will stop iteration; TRUE will keep going
    return psResult->bSuccess;
}

/*******************************************************************************
*
*   bProcessSelectPhonetics
*
*   This function is provided to the SQL_INTERFACE_OBJECT in order to process
*   a "select all" query made on the database phonetics relation.  It populates
*   a pointer to a PHONETICS_PHONETIC_ID_STRUCT with the information found
*   in the database and then cross references any data which we may be
*   processing that is a delta from what is found in the DB.
*
*******************************************************************************/
static BOOLEAN bProcessSelectPhonetics (
    SQL_QUERY_COLUMN_STRUCT *psColumn,
    N32 n32NumberOfColumns,
    PHONETICS_DB_PROCESS_STRUCT *psProcess
        )
{
    PHONETICS_PHONETIC_ROW_STRUCT sCurrentRow;
    BOOLEAN bWriteEntryToFile = TRUE;
    UN32 un32IDCol, un32PhoneticsCol;
    N16 n16CompareResult;
    PHONETICS_PHONETIC_ID_STRUCT *psIDToWrite = &sCurrentRow.sID;
    UN8 un8Index;

    // psProcess input
    if (psProcess == NULL)
    {
        return FALSE;
    }

    // Verify the data we got back
    if (psProcess->psWorkUnit->eType == PHONETICS_FILE_TYPE_TEAMS)
    {
        // Working on the teams table
        if (n32NumberOfColumns != PHONETICS_DB_TEAM_PHONETIC_MAX_FIELDS)
        {
            // This is definitely not a match
            psProcess->bSuccess = FALSE;
            return FALSE;
        }

        // Use the appropriate columns
        un32IDCol = (UN32)PHONETICS_DB_TEAM_PHONETIC_ID;
        un32PhoneticsCol = (UN32)PHONETICS_DB_TEAM_PHONETIC_ENG;
    }
    else
    {
        // This is not the teams table
        if (n32NumberOfColumns != PHONETICS_DB_BASE_PHONETIC_MAX_FIELDS)
        {
            // This is definitely not a match
            psProcess->bSuccess = FALSE;
            return FALSE;
        }

        // Use the appropriate columns
        un32IDCol = (UN32)PHONETICS_DB_BASE_PHONETIC_ID;
        un32PhoneticsCol = (UN32)PHONETICS_DB_BASE_PHONETICS_ENG;
    }

    // Start off by marking this a success
    psProcess->bSuccess = TRUE;

    // Grab the id, clear the league count
    sCurrentRow.sID.un32Id = psColumn[un32IDCol].uData.sUN32.un32Data;
    sCurrentRow.sID.un8NumLeagues = 0;

    // grab the phonetic data
    for (un8Index = 0;
         un8Index < PHONETICS_MAX_LANGUAGES;
         un8Index++)
    {
        sCurrentRow.pacPhonetics[un8Index] =
            (const char *)psColumn[un32PhoneticsCol+un8Index].uData.sCString.pcData;
    }

    // And the league info
    if (psProcess->psWorkUnit->eType == PHONETICS_FILE_TYPE_TEAMS)
    {
        // Get the number of leagues
        sCurrentRow.sID.un8NumLeagues = (UN8)
            psColumn[PHONETICS_DB_TEAM_PHONETIC_NUM_LEAGUES].uData.sUN32.un32Data;

        // Grab the league ids into a nice, convenient array
        sCurrentRow.sID.aun16Leagues[0] = (UN16)
            psColumn[PHONETICS_DB_TEAM_LEAGUE0].uData.sUN32.un32Data;
        sCurrentRow.sID.aun16Leagues[1] = (UN16)
            psColumn[PHONETICS_DB_TEAM_LEAGUE1].uData.sUN32.un32Data;
        sCurrentRow.sID.aun16Leagues[2] = (UN16)
            psColumn[PHONETICS_DB_TEAM_LEAGUE2].uData.sUN32.un32Data;
    }

    /*
        Cross reference this data with what came in from OTA, and:

        A Phonetics entry from the DB is overwritten by an
        entry from OTA.

        Only write an entry to the output files if that entry
        has phonetic data
    */

    // Do we have something to cross-reference?
    if (psProcess->psEntry != NULL)
    {
        // Get the comparison result
        n16CompareResult = n16CompareRowToNode(&sCurrentRow, psProcess->psEntry);

        // Is the DB entry ID greater than the OTA entry ID?
        if (n16CompareResult > 0)
        {
            // Yes -- time to catch up the OTA entries.
            // These are all new entries
            // so we are guaranteed that each of these has
            // actual phonetics strings to write out
            do
            {
                // Write the entry info
                psProcess->bSuccess = bWritePhoneticsRowByLanguage(
                    psProcess->psWorkUnit,
                    &psProcess->psEntry->sID,
                    psProcess->psEntry,
                    NULL // No DB Phonetics
                        );

                if (psProcess->bSuccess == FALSE)
                {
                    // Error!
                    break;
                }

                // Grab the next entry
                psProcess->hEntry =
                    OSAL.hLinkedListNext(
                        psProcess->hEntry,
                        (void **)&psProcess->psEntry);

                if ((psProcess->hEntry == OSAL_INVALID_LINKED_LIST_ENTRY) ||
                    (psProcess->psEntry == (PHONETICS_LIST_NODE_STRUCT *)NULL ))
                {
                    // Stop if there's no more data
                    break;
                }

                n16CompareResult = n16CompareRowToNode(&sCurrentRow, psProcess->psEntry);

                // Also stop when we're all caught up
            } while (n16CompareResult > 0);
        }
    }

    // Only continue if we didn't have an error
    if (psProcess->bSuccess == TRUE)
    {
        PHONETICS_LIST_NODE_STRUCT *psNewPhonetics =
            (PHONETICS_LIST_NODE_STRUCT *)NULL;

        // Do we still have something to cross-reference?
        if (psProcess->psEntry != NULL)
        {
            // Compare them
            n16CompareResult = n16CompareRowToNode(&sCurrentRow, psProcess->psEntry);

            // Is the OTA entry ID equal to the DB entry ID?
            if (n16CompareResult == 0)
            {
                // Yeah, these match
                if (psProcess->psEntry->bPhoneticsPresent == FALSE)
                {
                    // We aren't writing anything for this entry!
                    bWriteEntryToFile = FALSE;
                }
                else
                {
                    // We're going to write these phonetics to our output files
                    psNewPhonetics = psProcess->psEntry;

                    // Write out the ID for this row that we have in
                    // this node (because of the team / league deal)
                    psIDToWrite = &psProcess->psEntry->sID;
                }

                // Move the OTA entry along now
                psProcess->hEntry =
                    OSAL.hLinkedListNext(
                        psProcess->hEntry, (void **)&psProcess->psEntry);
            }
        }

        if (bWriteEntryToFile == TRUE)
        {
            // Write the entry info for what we have here
            psProcess->bSuccess = bWritePhoneticsRowByLanguage(
                psProcess->psWorkUnit,
                psIDToWrite,
                psNewPhonetics,
                &sCurrentRow);
        }
    }

    // Keep iterating while things are looking up
    return psProcess->bSuccess;
}

/*****************************************************************************
*
*   vSetError
*
*****************************************************************************/
static void vSetError (
    PHONETICS_MGR_OBJECT_STRUCT *psObj,
    DATASERVICE_ERROR_CODE_ENUM eErrorCode
        )
{
    // Tell the world about it
    SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
        PHONETICS_MGR_OBJECT_NAME": vSetError()");

    // Tell the DSM about it
    DATASERVICE_IMPL_vError((DATASERVICE_IMPL_HDL)psObj,eErrorCode);

    return;
}

#ifdef PHONETICS_TEST
/*****************************************************************************
*
*   Creates test data for generating files.  Will be removed
*   when we have actual OTA data to work with
*
*****************************************************************************/
static BOOLEAN bInsertTestData (
    OSAL_OBJECT_HDL *phListHandle
        )
{
    OSAL_RETURN_CODE_ENUM eReturnCode;
    PHONETICS_LIST_NODE_STRUCT *psTest =
        (PHONETICS_LIST_NODE_STRUCT *)NULL;

    do
    {
        eReturnCode = OSAL.eLinkedListCreate(
            phListHandle,
            "test list",
            (OSAL_LL_COMPARE_HANDLER)n16ComparePhoneticNode,
            OSAL_LL_OPTION_LINEAR |
            OSAL_LL_OPTION_USE_PRE_ALLOCATED_ELEMENTS );
        if (eReturnCode != OSAL_SUCCESS)
        {
            // Ensure the handle is invalid
            *phListHandle = OSAL_INVALID_OBJECT_HDL;
            break;
        }

        // test channels
        psTest = (PHONETICS_LIST_NODE_STRUCT *)
                OSAL.pvLinkedListMemoryAllocate(
                    "Test:ListEntry",
                    sizeof(PHONETICS_LIST_NODE_STRUCT),
                    TRUE );
        if (psTest == NULL)
        {
            break;
        }

        // Populate this test entry
        psTest->hPhonetics = STRING_INVALID_OBJECT;
        psTest->sID.un32Id = 5;

        eReturnCode = OSAL.eLinkedListAdd(
            *phListHandle, OSAL_INVALID_LINKED_LIST_ENTRY_PTR, psTest);
        if (eReturnCode != OSAL_SUCCESS)
        {
            // Ensure the handle is invalid
            break;
        }

        psTest = (PHONETICS_LIST_NODE_STRUCT *)
                OSAL.pvLinkedListMemoryAllocate(
                    "Test:ListEntry",
                    sizeof(PHONETICS_LIST_NODE_STRUCT),
                    TRUE );

        if (psTest == NULL)
        {
            break;
        }

        psTest->hPhonetics = STRING_hCreateConst("phonetics6", strlen("phonetics7"));
        psTest->sID.un32Id = 6;

        eReturnCode = OSAL.eLinkedListAdd(
            *phListHandle, OSAL_INVALID_LINKED_LIST_ENTRY_PTR, psTest);
        if (eReturnCode != OSAL_SUCCESS)
        {
            // Ensure the handle is invalid
            break;
        }

        psTest = (PHONETICS_LIST_NODE_STRUCT *)
                OSAL.pvLinkedListMemoryAllocate(
                    "Test:ListEntry",
                    sizeof(PHONETICS_LIST_NODE_STRUCT),
                    TRUE );

        if (psTest == NULL)
        {
            break;
        }

        psTest->hPhonetics = STRING_hCreateConst("phonetics7", strlen("phonetics7"));
        psTest->sID.un32Id = 7;

        eReturnCode = OSAL.eLinkedListAdd(
            *phListHandle, OSAL_INVALID_LINKED_LIST_ENTRY_PTR, psTest);
        if (eReturnCode != OSAL_SUCCESS)
        {
            // Ensure the handle is invalid
            break;
        }

        psTest = (PHONETICS_LIST_NODE_STRUCT *)
                OSAL.pvLinkedListMemoryAllocate(
                    "Test:ListEntry",
                    sizeof(PHONETICS_LIST_NODE_STRUCT),
                    TRUE );

        if (psTest == NULL)
        {
            break;
        }

        psTest->hPhonetics = STRING_hCreateConst("phonetics8", strlen("phonetics8"));
        psTest->sID.un32Id = 8;

        eReturnCode = OSAL.eLinkedListAdd(
            *phListHandle, OSAL_INVALID_LINKED_LIST_ENTRY_PTR, psTest);
        if (eReturnCode != OSAL_SUCCESS)
        {
            // Ensure the handle is invalid
            break;
        }

        psTest = (PHONETICS_LIST_NODE_STRUCT *)
                OSAL.pvLinkedListMemoryAllocate(
                    "Test:ListEntry",
                    sizeof(PHONETICS_LIST_NODE_STRUCT),
                    TRUE );

        if (psTest == NULL)
        {
            break;
        }

        psTest->hPhonetics = STRING_hCreateConst("phonetics9|phonetics10", strlen("phonetics9|phonetics10"));
        psTest->sID.un32Id = 9;

        eReturnCode = OSAL.eLinkedListAdd(
            *phListHandle, OSAL_INVALID_LINKED_LIST_ENTRY_PTR, psTest);
        if (eReturnCode != OSAL_SUCCESS)
        {
            // Ensure the handle is invalid
            break;
        }

        // test teams
        /*
        psTest = (PHONETICS_LIST_NODE_STRUCT *)
                OSAL.pvLinkedListMemoryAllocate(
                    "Test:ListEntry",
                    sizeof(PHONETICS_LIST_NODE_STRUCT),
                    TRUE );

        if (psTest == NULL)
        {
            break;
        }

        psTest->hPhonetics = STRING_hCreateConst("purdue|boilers|purdue's the best", strlen("purdue|boilers|purdue's the best"));
        psTest->sID.un32Id = 236;
        psTest->sID.un8NumLeagues = 2;
        psTest->sID.aun16Leagues[0] = 4;
        psTest->sID.aun16Leagues[1] = 5;
        OSAL.eLinkedListAdd(psWorkUnit->hListData, OSAL_INVALID_LINKED_LIST_ENTRY_PTR, psTest);*/

        return TRUE;
    } while (FALSE);

    if (psTest != NULL)
    {
        OSAL.vLinkedListMemoryFree(psTest);
    }

    OSAL.eLinkedListRemoveAll(*phListHandle,
        (OSAL_LL_RELEASE_HANDLER)vClearPhoneticNode);

    return FALSE;
}
#endif
