/******************************************************************************/
/*                    Copyright (c) Sirius XM Radio, Inc.                     */
/*                            All Rights Reserved                             */
/*         Licensed Materials - Property of Sirius XM Radio, Inc.             */
/******************************************************************************/
/*******************************************************************************
 *
 * DESCRIPTION
 *
 *  This module contains the Object:MUSIC implementation for the
 *  Sirius Module Services (SMS)
 *
******************************************************************************/
#include "standard.h"
#include "osal.h"

#include "sms_version.h"
#include "sms_api.h"
#include "sms_obj.h"
#include "cdo_obj.h"
#include "cid_obj.h"
#include "cme.h"

#include "music_obj.h"
#include "_music_obj.h"

/*****************************************************************************
                             PUBLIC FUNCTIONS
*****************************************************************************/

/*****************************************************************************
*
*   eSubType
*
* Retrievs the enumerated value representing the MUSIC object type.
*
*****************************************************************************/
static MUSIC_ENUM eSubType (
    CD_OBJECT hCDO
        )
{
    CDO_TYPE_ENUM eType;
    MUSIC_OBJECT_STRUCT *psObj;
    MUSIC_ENUM eSubType = MUSIC_UNKNOWN;

    // Verify CDO type is correct for this API
    eType = CDO.eType(hCDO);
    if(eType == CDO_MUSIC)
    {
        // Go get CDO structure from provided CDO
        psObj = (MUSIC_OBJECT_STRUCT *)CDO_pvContentData(hCDO);
        if(psObj != NULL)
        {
            eSubType = psObj->eSubType;
        }
    }

    return eSubType;
}

/*****************************************************************************
*
*   hArtistId
*
* Retrieves the CID associated with this CDO given that it is a MUSIC type.
* The returned CID is the unique identifier for the artist this CDO describes.
*
*****************************************************************************/
static CID_OBJECT hArtistId (
    CD_OBJECT hCDO
        )
{
    CDO_TYPE_ENUM eType;
    MUSIC_OBJECT_STRUCT *psObj;
    CID_OBJECT hArtistId = CID_INVALID_OBJECT;

    // Verify CDO type is correct for this API
    eType = CDO.eType(hCDO);
    if(eType == CDO_MUSIC)
    {
        // Go get CDO structure from provided CDO
        psObj = (MUSIC_OBJECT_STRUCT *)CDO_pvContentData(hCDO);
        if(psObj != NULL)
        {
            hArtistId = psObj->hArtistId;
        }
    }

    return hArtistId;
}

/*****************************************************************************
*
*   hSongId
*
*   Retrieves the CID associated with this CDO given that it is a MUSIC type.
*   The returned CID is the unique identifier for the song this CDO describes.
*
*****************************************************************************/
static CID_OBJECT hSongId (
    CD_OBJECT hCDO
        )
{
    CDO_TYPE_ENUM eType;
    MUSIC_OBJECT_STRUCT *psObj;
    CID_OBJECT hSongId = CID_INVALID_OBJECT;

    // Verify CDO type is correct for this API
    eType = CDO.eType(hCDO);
    if(eType == CDO_MUSIC)
    {
        // Go get CDO structure from provided CDO
        psObj = (MUSIC_OBJECT_STRUCT *)CDO_pvContentData(hCDO);
        if(psObj != NULL)
        {
            hSongId = psObj->hSongId;
        }
    }

    return hSongId;
}

/*****************************************************************************
*
*   hSongTag
*
*   Retrieves the CID associated with this CDO given that it is a MUSIC type.
*   The returned CID is the unique identifier for the song tag this CDO
*   describes.
*
*****************************************************************************/
static CID_OBJECT hSongTag (
    CD_OBJECT hCDO,
    SONG_TAG_SERVICE_ENUM eService
        )
{
    CDO_TYPE_ENUM eType;
    MUSIC_OBJECT_STRUCT *psObj;
    CID_OBJECT hSongTag = CID_INVALID_OBJECT;

    // Verify CDO type is correct for this API
    eType = CDO.eType(hCDO);
    if(eType == CDO_MUSIC)
    {
        // Go get CDO structure from provided CDO
        psObj = (MUSIC_OBJECT_STRUCT *)CDO_pvContentData(hCDO);
        if(psObj != NULL)
        {
            switch (eService)
            {
                case SONG_TAG_SERVICE_ITUNES:
                {
                    hSongTag = psObj->sSongTag.hITunesTagId;
                }
                break;

                case SONG_TAG_SERVICE_INVALID:
                default:
                    // do nothing
                break;
            }
        }
    }

    return hSongTag;
}

/*****************************************************************************
                             FRIEND FUNCTIONS
*****************************************************************************/

/*****************************************************************************
*
*   MUSIC_bValidSubId
*
* Local function which checks an enumerated type to make sure it is
* one of the valid MUSIC_ENUM types.
*
*****************************************************************************/
BOOLEAN MUSIC_bValidSubId (
    MUSIC_ENUM eType
        )
{
    switch (eType)
    {
        case MUSIC_SHOW:
        case MUSIC_MIX:
        case MUSIC_OTHER:
            return TRUE;

        default:
            return FALSE;
    }
}

/*****************************************************************************
                             PRIVATE FUNCTIONS
*****************************************************************************/

/*****************************************************************************
*
*   vUnInit
*
* CDO Un-initialize method. Called when a MUSIC CDO is replaced by
* another type, requiring any resources to be released.
*
*****************************************************************************/
static void vUnInit (
    MUSIC_OBJECT_STRUCT *psObj
        )
{
    // Check inputs.
    if(psObj == NULL)
    {
        // Error!
        return;
    }

    // Make sure we free any CIDs we have.

    // Un-init MUSIC CDO structure
    psObj->eSubType = MUSIC_UNKNOWN;

    // Free ArtistId
    if(psObj->hArtistId != CID_INVALID_OBJECT)
    {
        CD_OBJECT hCDO = CDO_hCDO(psObj);

        // Inform CME of the content change
        CME_eDestroyContent(hCDO, &psObj->hArtistId);
    }

    // Free SongId
    if(psObj->hSongId != CID_INVALID_OBJECT)
    {
        CD_OBJECT hCDO = CDO_hCDO(psObj);

        // Inform CME of the content change
        CME_eDestroyContent(hCDO, &psObj->hSongId);
    }

    // Free SongTags
    if(psObj->sSongTag.hITunesTagId != CID_INVALID_OBJECT)
    {
        CD_OBJECT hCDO = CDO_hCDO(psObj);

        // Inform CME of the content change
        CME_eDestroyContent(hCDO, &psObj->sSongTag.hITunesTagId);
    }

    // Default object now
    *psObj = gsDefaultMusic;

    return;
}

/*****************************************************************************
*
*   n16Equal
*
*       This is the function used to compare MUSIC CDOs. Specifically
*       this function is used to perform a go, no-go comparison
*       or binary comparison of the two CDOs. Two MUSIC CDOs are considered
*       equal if both the ArtistId and SongId are the same.
*
*       Outputs:
*               0   - CDOs have the same value (equal)
*               -1  - CDOs are not equal (or error)
*
*****************************************************************************/
static N16 n16Equal (
    MUSIC_OBJECT_STRUCT *psObj1,
    MUSIC_OBJECT_STRUCT *psObj2
        )
{
    N16 n16Result, n16ReturnResult = -1;

    // Verify inputs
    if( (psObj1 == NULL) || (psObj2 == NULL) )
    {
        // Error
        return -1;
    }

    // Both ARTIST and SONG id must match
    n16Result = CID.n16Equal(
        psObj1->hArtistId, psObj2->hArtistId);
    if(n16Result == 0)
    {
        n16Result = CID.n16Equal(
            psObj1->hSongId, psObj2->hSongId);
        if(n16Result == 0)
        {
            // Indicate the CDO's are equal
            n16ReturnResult = 0;
        }
    }

   return n16ReturnResult;
}

/*****************************************************************************
*
*   n16Compare
*
*       This is the function used to compare MUSIC CDOs. Specifically
*       this function is used to keep CDOs in order for
*       comparison/sorting and thus a relational compare is performed.
*       This function keeps CDOs in ARTIST then SONG Id order.
*
*       Outputs:
*               0   - CDOs have the same value (equal)
*               > 0 - CDO1 is greater than (after) CDO2
*               < 0 - CDO1 is less than (before) CDO2 or error
*
*****************************************************************************/
static N16 n16Compare (
    MUSIC_OBJECT_STRUCT *psObj1,
    MUSIC_OBJECT_STRUCT *psObj2
        )
{
    N16 n16Result;

    // Verify inputs
    if( (psObj1 == NULL) || (psObj2 == NULL) )
    {
        // Error
        return N16_MIN;
    }

    if ((psObj1->hArtistId == CID_INVALID_OBJECT) &&
        (psObj2->hArtistId == CID_INVALID_OBJECT))
    {
        return 0;
    }

    // Sort by ARTIST and then SONG Id
    n16Result = CID.n16Compare(
        psObj1->hArtistId, psObj2->hArtistId);
    if(n16Result == 0)
    {
        if ((psObj1->hSongId == CID_INVALID_OBJECT) &&
            (psObj2->hSongId == CID_INVALID_OBJECT))
        {
            return 0;
        }

        n16Result = CID.n16Compare(
            psObj1->hSongId, psObj2->hSongId);
    }

    return n16Result;
}

/*****************************************************************************
*
*   n32FPrintf
*
* This method is used by the caller to send formatted
* output of a CDO's contents to a specified file or device.
* This is mainly helpful during debugging of CDO's but could be used by
* a caller for any reason. This API is different than the n32FWrite()
* method which instead writes the contents of a CDO to a file for the
* purposes of later re-generating the CDO (for storage of the object).
* This API instead sends the CDO as a verbose formatted output version.
*
* Inputs:
*
*   psObj - The CDO structure the caller wishes to print.
*   psFile - The device to write the CDO contents to.
*
*****************************************************************************/
static N32 n32FPrintf (
    const MUSIC_OBJECT_STRUCT *psObj,
    FILE *psFile
        )
{
    N32 n32Return = EOF;
    N32 n32Temp   = 0;

    // Check inputs.
    if((psObj != NULL) && (psFile != NULL))
    {
        UN32 un32Tag;
        SMSAPI_RETURN_CODE_ENUM eReturn;

        n32Return = 0;

        // Print CDO information...
        n32Return += fprintf(psFile,
            "eSubType: %s\n\n", pacTypeText(psObj->eSubType));

        n32Return += fprintf(psFile, "hArtistId:\n");
        n32Temp = CID.n32FPrintf(psObj->hArtistId, psFile);
        if (n32Temp > 0)
        {
            n32Return += n32Temp;
        }
        n32Return += fprintf(psFile, "\n");

        n32Return += fprintf(psFile, "hSongId:\n");
        n32Temp = CID.n32FPrintf(psObj->hSongId, psFile);
        if (n32Temp > 0)
        {
            n32Return += n32Temp;
        }
        n32Return += fprintf(psFile, "\n");

        eReturn =
            SONG_TAG_SERVICE.eGetITunesTagValue(psObj->sSongTag.hITunesTagId, &un32Tag);
        if (eReturn == SMSAPI_RETURN_CODE_SUCCESS)
        {
            n32Return += fprintf(psFile, "iTunesSongTag = %u\n", un32Tag);
        }
        n32Return += fprintf(psFile, "\n");
    }

    return n32Return;
}

/*****************************************************************************
*
*   bHasId
*
* This method is used to examine a CDO sub-type for a specific Content-ID (CID)
* provided by the caller. If this sub-type has this id, then TRUE is returned
* otherwise, FALSE is returned if it does not contain it. Keep in mind
* this functions checks the entire sub-type for any matches.
*
* Inputs:
*   psObj - A pointer to an object for which to examine and determine if
*       the provided CID is contained within it.
*   hId - A valid CID to look for within the object.
*
* Returns:
*   TRUE of the CID was found anywhere within the object, otherwise FALSE
*   is returned if it could not be found.
*
*****************************************************************************/
static BOOLEAN bHasId (
    const MUSIC_OBJECT_STRUCT *psObj,
    CID_OBJECT hId
        )
{
    BOOLEAN bHasId = FALSE;

    // We can look to see if this CID matches one we have in this object.
    // Any hit is enough to cause us to fall out and return TRUE.
    do
    {
        UN16 n16Equal;

        // hId == hArtistId?
        n16Equal = CID.n16Equal(hId, psObj->hArtistId);
        if(n16Equal == 0)
        {
            // Equal, so they match
            bHasId = TRUE;
            break;
        }

        // hId == hSongId?
        n16Equal = CID.n16Equal(hId, psObj->hSongId);
        if(n16Equal == 0)
        {
            // Equal, so they match
            bHasId = TRUE;
            break;
        }

    } while(FALSE);

    return bHasId;
}

/*****************************************************************************
                             PRIVATE FUNCTIONS
*****************************************************************************/

/*****************************************************************************
*
*   pacTypeText
*
* This local function translates a provided MUSIC_ENUM into a text
* string representation.
*
*****************************************************************************/
static const char *pacTypeText(
    MUSIC_ENUM eType
        )
{
    const char *pacReturnString;

    switch (eType)
    {
        case MUSIC_SHOW:
            pacReturnString = MACRO_TO_STRING(MUSIC_SHOW);
        break;

        case MUSIC_MIX:
            pacReturnString = MACRO_TO_STRING(MUSIC_MIX);
        break;

        case MUSIC_OTHER:
            pacReturnString = MACRO_TO_STRING(MUSIC_OTHER);
        break;

        case MUSIC_UNKNOWN:
        default:
            pacReturnString = MACRO_TO_STRING(MUSIC_UNKNOWN);
        break;
    }

    return pacReturnString;
}
