/******************************************************************************/
/*                    Copyright (c) Sirius XM Radio, Inc.                     */
/*                            All Rights Reserved                             */
/*         Licensed Materials - Property of Sirius XM Radio, Inc.             */
/******************************************************************************/
/*******************************************************************************
 *
 * DESCRIPTION
 *
 *  This module contains the Object:MOVIES_MGR implementation for the
 *  Simple Module Services (SMS)
 *
 ******************************************************************************/
#include <stdio.h>
#include <string.h>
#include "standard.h"
#include "osal.h"

#include "sms_api.h"
#include "sms_obj.h"
#include "sms.h"
#include "sql_interface_obj.h"
#include "srm_obj.h"
#include "sms_update.h"
#include "dataservice_mgr_impl.h"
#include "string_obj.h"
#include "location_obj.h"
#include "dsrl_obj.h"
#include "dsrl_entry_obj.h"
#include "dsrl_target_obj.h"
#include "baudot.h"
#include "db_util.h"

#include "movies_db_constants.h"

#include "rfd_interface_obj.h"

#include "movie_obj.h"
#include "theater_times_obj.h"
#include "movies_mgr_obj.h"
#include "_movies_mgr_obj.h"



#include "sms_api_debug.h"
static const char *gpacThisFile = __FILE__;

/*****************************************************************************
                             PUBLIC FUNCTIONS
*****************************************************************************/

/*****************************************************************************
*
*   hStart
*
*   This function will create all the basics needed for this service to
*   operate.  However, all initial processing to actually get this service
*   running is done at a later time.
*
*****************************************************************************/
static MOVIES_SERVICE_OBJECT hStart (
    const char *pacSRHDriverName,
    DATASERVICE_EVENT_MASK tEventRequestMask,
    DATASERVICE_EVENT_CALLBACK vEventCallback,
    void *pvAppEventCallbackArg,
    DATASERVICE_OPTIONS_STRUCT const *psOptions
        )
{
    BOOLEAN bServiceStarted, bSuccess;
    MOVIES_MGR_OBJECT_STRUCT *psObj = (MOVIES_MGR_OBJECT_STRUCT *)
        MOVIES_SERVICE_INVALID_OBJECT;
    OSAL_RETURN_CODE_ENUM eReturnCode;
    DATASERVICE_CREATE_STRUCT sCreate;
    DATASERVICE_OPTION_VALUES_STRUCT sOptionValues;

    bSuccess = DATASERVICE_IMPL_bProcessOptions(
        MOVIES_SUPPORTED_OPTIONS, psOptions, &sOptionValues);
    if (bSuccess == FALSE)
    {
        // Bad options!
        return MOVIES_SERVICE_INVALID_OBJECT;
    }

    // Populate our data service creation structure
    DATASERVICE_IMPL_vInitCreateStruct(&sCreate);
    sCreate.pacSRHDriverName = pacSRHDriverName;
    sCreate.pacServiceObjectName = MOVIES_MGR_OBJECT_NAME;
    sCreate.tServiceObjectSize = sizeof(MOVIES_MGR_OBJECT_STRUCT);
    sCreate.tDataID = (DATASERVICE_ID)GsMoviesIntf.tDSI;

    // Get the interface to tell us how much memory
    // this service needs at a minimum for the given
    // startup options
    sCreate.tSuggestedOTABufferByteSize =
        GsMoviesIntf.tMinimumOTABufferByteSize(sOptionValues.bUpdateRefDB);

    // Configure the data service's static event attributes
    sCreate.vEventCallback = vEventHandler;
    sCreate.tEventRequestMask = (
        DATASERVICE_EVENT_STATE |
        DATASERVICE_EVENT_NEW_DATA |
        DATASERVICE_EVENT_TIMEOUT |
        DATASERVICE_INTERNAL_EVENT_DSRL );

    // Ask the data service manager controller to
    // create our manager object and do everything
    // necessary to create the underlying objects required
    // in order to support this service
    psObj = (MOVIES_MGR_OBJECT_STRUCT *)
        DATASERVICE_IMPL_hCreateNewService( &sCreate );
    if (psObj == NULL)
    {
        // Free options memory
        DATASERVICE_IMPL_vFreeOptions(&sOptionValues);

        return MOVIES_SERVICE_INVALID_OBJECT;
    }

    do
    {
        // Update the flag & path for the reference db
        psObj->bRefDBUpdatesEnabled = sOptionValues.bUpdateRefDB;
        psObj->pacRefDatabaseDirPath = sOptionValues.pcRefDBPath;

        // Init the baseline theater version value
        psObj->tBaselineVersion = MOVIE_INVALID_VERSION;

        // Initialize asynchronous update configuration
        SMSU_vInitialize(
            &psObj->sEvent,
            psObj,
            DATASERVICE_EVENT_NONE,
            tEventRequestMask,
            (SMSAPI_OBJECT_EVENT_CALLBACK)vEventCallback,
            pvAppEventCallbackArg);

        // Create our Movie Structures
        psObj->psInProgressMovies = psBuildMovieDescStruct(
            DATASERVICE_IMPL_hSMSObj((DATASERVICE_IMPL_HDL)psObj));

        if (psObj->psInProgressMovies == NULL)
        {
            // Error!
            break;
        }

        // Create our Times Structures
        psObj->psInProgressTimes = psBuildMovieTimesStruct(
            DATASERVICE_IMPL_hSMSObj((DATASERVICE_IMPL_HDL)psObj));

        if (psObj->psInProgressTimes == NULL)
        {
            // Error!
            break;
        }

        // Create our DSRL list
        eReturnCode = OSAL.eLinkedListCreate(
                   &psObj->hDSRLList,
                   MOVIES_MGR_OBJECT_NAME":DSRLs",
                   (OSAL_LL_COMPARE_HANDLER)NULL,
                   (OSAL_LL_OPTION_LINEAR | OSAL_LL_OPTION_UNIQUE)
                       );

        if(eReturnCode != OSAL_SUCCESS)
        {
            // Error!
            break;
        }

        bSuccess = bInitAppFacingObject(psObj);

        if (bSuccess == FALSE)
        {
            break;
        }

        // The service may now start
        bServiceStarted = DATASERVICE_IMPL_bStart((DATASERVICE_IMPL_HDL)psObj);

        if (bServiceStarted == FALSE)
        {
            break;
        }

        return (MOVIES_SERVICE_OBJECT)psObj;

    } while (FALSE);

    // Error!
    vUninitObject( psObj, TRUE );
    DATASERVICE_IMPL_vDestroy((DATASERVICE_IMPL_HDL)psObj);

    return MOVIES_SERVICE_INVALID_OBJECT;
}

/*****************************************************************************
*
*   eIterate
*
*****************************************************************************/
static SMSAPI_RETURN_CODE_ENUM eIterate (
      MOVIES_SERVICE_OBJECT hMoviesService,
      MOVIES_ITERATOR bIterator,
      void *pvIteratorArg
        )
{
    MOVIES_APP_OBJECT_STRUCT *psAppObj = NULL;
    SMSAPI_RETURN_CODE_ENUM eReturnCode = SMSAPI_RETURN_CODE_ERROR;
    BOOLEAN bValid;

    // Verify service object
    bValid =
        DATASERVICE_IMPL_bValid((DATASERVICE_IMPL_HDL)hMoviesService);
    if (FALSE == bValid)
    {
        return SMSAPI_RETURN_CODE_INVALID_INPUT;
    }

    // Get the app facing object
    psAppObj = psGetAppFacingObject(hMoviesService);

    if (psAppObj != NULL)
    {
        if (psAppObj->psCurrentMovies->eState == MOVIE_OTA_UPDATE_STATE_STABLE)
        {
            OSAL_RETURN_CODE_ENUM eOSALReturnCode;

            eOSALReturnCode = OSAL.eLinkedListIterate(
                psAppObj->psCurrentMovies->hDescriptions,
                (OSAL_LL_ITERATOR_HANDLER)bIterator,
                pvIteratorArg);

            if (eOSALReturnCode == OSAL_SUCCESS)
            {
                eReturnCode = SMSAPI_RETURN_CODE_SUCCESS;
            }
        }
        else
        {
            // We don't have a stable movie list, inform the caller
            eReturnCode = SMSAPI_RETURN_CODE_NO_OBJECTS;
        }

        SMSO_vUnlock((SMS_OBJECT)psAppObj);
    }

    return eReturnCode;
}

/*****************************************************************************
*
*   eGetReferenceDataVersion
*
*****************************************************************************/
DATASERVICE_ERROR_CODE_ENUM eGetReferenceDataVersion (
    const char *pcContainingDirectoryPath,
    DATASERVICE_REF_DATA_VER *ptCurrentRefDataVer,
    DATASERVICE_REF_DATA_VER *ptNextRefDataVer
        )
{
    DATASERVICE_ERROR_CODE_ENUM eReturnCode = DATASERVICE_ERROR_UNKNOWN;
    BOOLEAN bOk;
    char *pacDatabaseFilePathA = NULL,
         *pacDatabaseFilePathB = NULL;

    if (ptCurrentRefDataVer == NULL)
    {
        return DATASERVICE_ERROR_CODE_GENERAL;
    }

    do
    {
        bOk = DB_UTIL_bCreateFilePath(
            pcContainingDirectoryPath,
            MOVIES_DATABASE_FOLDER,
            MOVIES_REF_DATABASE_FILENAMEA,
            &pacDatabaseFilePathA);
        if (bOk != TRUE)
        {
            break;
        }

        bOk = DB_UTIL_bCreateFilePath(
            pcContainingDirectoryPath,
            MOVIES_DATABASE_FOLDER,
            MOVIES_REF_DATABASE_FILENAMEB,
            &pacDatabaseFilePathB);
        if (bOk != TRUE)
        {
            break;
        }

        // Connect to the movies ref database bank
        eReturnCode =
            DB_UTIL_eCheckReferenceBanks (
                &pacDatabaseFilePathA[0],
                &pacDatabaseFilePathB[0],
                n32ExtractDataVersion, NULL,
                GsMoviesIntf.tMaxVersionBitlen,
                ptCurrentRefDataVer, ptNextRefDataVer );

    } while (FALSE);


    // Remove database path resources
    if (pacDatabaseFilePathA != NULL)
    {
        SMSO_vDestroy(
            (SMS_OBJECT)pacDatabaseFilePathA);
    }

    if (pacDatabaseFilePathB != NULL)
    {
        SMSO_vDestroy(
            (SMS_OBJECT)pacDatabaseFilePathB);
    }

    return eReturnCode;
}

/*****************************************************************************
                             FRIEND FUNCTIONS
*****************************************************************************/

/*****************************************************************************
*
*   MOVIES_MGR_hGetMovieForID
*
*****************************************************************************/
MOVIE_OBJECT MOVIES_MGR_hGetMovieForID (
    SMS_OBJECT hParent,
    MOVIE_ID tMovieID
        )
{
    MOVIE_OBJECT hMovie = MOVIE_INVALID_OBJECT;
    MOVIES_APP_OBJECT_STRUCT *psAppObj = 
        (MOVIES_APP_OBJECT_STRUCT *)hParent;
    BOOLEAN bOwner;

    // Ensure we own the parent
    bOwner = SMSO_bOwner(hParent);

    if (TRUE == bOwner)
    {
        BOOLEAN bUpdated;

        // Update the dummy object with this id
        bUpdated = MOVIE_bUpdateDummyID(psAppObj->hDummyMovie, tMovieID);

        if (TRUE == bUpdated)
        {
            OSAL_LINKED_LIST_ENTRY hEntry = 
                OSAL_INVALID_LINKED_LIST_ENTRY;
            OSAL_RETURN_CODE_ENUM eReturnCode;

            // Find this movie
            eReturnCode = OSAL.eLinkedListSearch(
                psAppObj->psCurrentMovies->hDescriptions,
                &hEntry,
                (void *)psAppObj->hDummyMovie);

            if (OSAL_SUCCESS == eReturnCode)
            {
                hMovie = (MOVIE_OBJECT)OSAL.pvLinkedListThis(hEntry);
            }
        }
    }

    return hMovie;
}

/*****************************************************************************
*
*   MOVIES_MGR_bGetRatingInfo
*
*****************************************************************************/
BOOLEAN MOVIES_MGR_bGetRatingInfo (
    MOVIES_SERVICE_OBJECT hMovieMgr,
    MOVIE_RATING_SYS_CODE tSysCode,
    MOVIE_RATING_CODE tRatingCode,
    MOVIE_RATING_STRUCT *psRating
        )
{
    MOVIES_APP_OBJECT_STRUCT *psAppObj;
    BOOLEAN bFound = FALSE;

    // Nothing to do since we were given invalid inputs
    if ((tRatingCode == MOVIE_RATING_INVALID_CODE) ||
        (psRating == (MOVIE_RATING_STRUCT *)NULL))
    {
        return FALSE;
    }

    // Get and lock the app facing object that contains the ratings list
    psAppObj = psGetAppFacingObject(hMovieMgr);

    if (psAppObj != NULL)
    {
        OSAL_LINKED_LIST_ENTRY hEntry = OSAL_INVALID_LINKED_LIST_ENTRY;
        OSAL_RETURN_CODE_ENUM eReturnCode;
        MOVIE_RATINGS_ENTRY_STRUCT sRatingToFind;

        sRatingToFind.tSysCode = tSysCode;
        sRatingToFind.tRatingCode = tRatingCode;

        eReturnCode =
            OSAL.eLinkedListSearch(
                psAppObj->hRatingsList,
                &hEntry,
                (void *)&sRatingToFind);

        if (eReturnCode == OSAL_SUCCESS)
        {
            MOVIE_RATINGS_ENTRY_STRUCT *psEntryFound =
                (MOVIE_RATINGS_ENTRY_STRUCT *)OSAL.pvLinkedListThis(hEntry);

            // Extract data
            psRating->hRatingSystemText = psEntryFound->hSystemText;
            psRating->hRatingText = psEntryFound->hRatingText;
            psRating->eRating = psEntryFound->eRating;
            psRating->eRatingSystem =  psEntryFound->eSystem;
            psRating->tRatingSys = tSysCode;
            psRating->tRating = tRatingCode;

            // Found!
            bFound = TRUE;
        }

        SMSO_vUnlock((SMS_OBJECT)psAppObj);
    }

    if (bFound == FALSE)
    {
        // Rating is unknown
        psRating->hRatingSystemText = STRING_INVALID_OBJECT;
        psRating->hRatingText = STRING_INVALID_OBJECT;
        psRating->eRating = MOVIE_RATING_UNKNOWN;
        psRating->eRatingSystem =  MOVIE_RATING_SYSTEM_UNKNOWN;
    }

    return bFound;
}

/*****************************************************************************
*
*   MOVIES_MGR_bShouldProcessTimes
*
*****************************************************************************/
BOOLEAN MOVIES_MGR_bShouldProcessTimes (
    MOVIES_SERVICE_OBJECT hMovieMgr,
    MOVIE_STATE_ID tStateID
        )
{
    MOVIES_MGR_OBJECT_STRUCT *psObj = (MOVIES_MGR_OBJECT_STRUCT *)hMovieMgr;
    BOOLEAN bOwner, bShouldProcess = FALSE;

    // Validate ownership
    bOwner = DATASERVICE_IMPL_bOwner((DATASERVICE_IMPL_HDL)hMovieMgr);
    if (TRUE == bOwner)
    {
        // Is this a state we care about?
        bShouldProcess = bIsStateInDSRLs(psObj, tStateID);
    }
    
    return bShouldProcess;
}

/*****************************************************************************
*
*   MOVIES_MGR_bPrepareForRefDBUpdate
*
*****************************************************************************/
BOOLEAN MOVIES_MGR_bPrepareForRefDBUpdate (
    MOVIES_SERVICE_OBJECT hMovieMgr,
    STRING_OBJECT *phInUseDB,
    STRING_OBJECT *phNextDB
        )
{
    BOOLEAN bLocked, bSuccess = FALSE;

    if ((phInUseDB == NULL) || (phNextDB == NULL))
    {
        return FALSE;
    }

    // Initialize inputs
    *phInUseDB = *phNextDB = STRING_INVALID_OBJECT;

    // Verify and lock service object
    bLocked = DATASERVICE_IMPL_bLock((DATASERVICE_IMPL_HDL)hMovieMgr);

    if (bLocked == TRUE)
    {
        MOVIES_MGR_OBJECT_STRUCT *psObj =
            (MOVIES_MGR_OBJECT_STRUCT *)hMovieMgr;

        do
        {
            // Provide the caller with the path info
            *phInUseDB = STRING.hCreate(
                psObj->pacCurRefDatabaseFilePath,
                strlen(psObj->pacCurRefDatabaseFilePath));

            if (*phInUseDB == STRING_INVALID_OBJECT)
            {
                // Error! Stop here
                break;
            }

            *phNextDB = STRING.hCreate(
                psObj->pacNextRefDatabaseFilePath,
                strlen(psObj->pacNextRefDatabaseFilePath));

            if (*phNextDB == STRING_INVALID_OBJECT)
            {
                // Clear the other string now
                STRING_vDestroy(*phInUseDB);
                *phInUseDB = STRING_INVALID_OBJECT;

                break;
            }

            // All went well
            bSuccess = TRUE;
        } while (FALSE);

        DATASERVICE_IMPL_vUnlock((DATASERVICE_IMPL_HDL)hMovieMgr);
    }

    return bSuccess;
}

/*****************************************************************************
*
*   MOVIES_MGR_bStartDBUpdate
*
*****************************************************************************/
BOOLEAN MOVIES_MGR_bStartDBUpdate (
    SQL_INTERFACE_OBJECT hConnection,
    char *pcSQLCommandBuffer,
    size_t tBufferSize
        )
{
    BOOLEAN bOk;

    // Generate the SQL command to update the version row
    // in the database to 0
    snprintf( &pcSQLCommandBuffer[0], tBufferSize,
        MOVIES_UPDATE_BASELINE_VERSION,
        DB_UTIL_DB_UNDER_CONSTRUCTION_VER,
        MOVIES_DATABASE_FILE_VERSION );

    // Update the row
    bOk = SQL_INTERFACE.bExecuteCommand(
        hConnection, &pcSQLCommandBuffer[0] );

    return bOk;
}


/*******************************************************************************
*
*   MOVIES_MGR_bEndDBUpdate
*
*******************************************************************************/
BOOLEAN MOVIES_MGR_bEndDBUpdate (
    SQL_INTERFACE_OBJECT hConnection,
    char *pacBuffer,
    size_t tBufferSize,
    MOVIE_VERSION tNewBaselineVersion
        )
{
    BOOLEAN bResult = FALSE;

    do
    {
        N32 n32Result;

        // While the DB_UTIL function will print its own error, we also print
        // one here to make it easier to trace the DB failure back to a
        // particular service.
        bResult = DB_UTIL_bUpdateTimestamp( hConnection, pacBuffer, tBufferSize );
        if ( bResult == FALSE )
        {
            SMSAPI_DEBUG_vPrintErrorFull( gpacThisFile, __LINE__,
                    MOVIES_MGR_OBJECT_NAME": Unable to set timestamp.");
        }

        n32Result = snprintf( pacBuffer, tBufferSize,
                            MOVIES_UPDATE_BASELINE_VERSION,
                            tNewBaselineVersion,
                            MOVIES_DATABASE_FILE_VERSION );
        if ( n32Result <= 0 )
        {
            bResult = FALSE;
            break;
        }

        // Updating the baseline version comes last, as this effectively stamps
        // the DB as "ready."
        bResult = SQL_INTERFACE.bExecuteCommand( hConnection, pacBuffer );
        if ( bResult == FALSE )
        {
            SMSAPI_DEBUG_vPrintErrorFull( gpacThisFile, __LINE__,
                    MOVIES_MGR_OBJECT_NAME": Unable to set baseline version.");
            break;
        }

    } while ( FALSE );

    return bResult;
}

/*****************************************************************************
*
*   MOVIES_MGR_bRemoveTheater
*
*****************************************************************************/
BOOLEAN MOVIES_MGR_bRemoveTheater (
    SQL_INTERFACE_OBJECT hSQLConnection,
    char *pcSQLCommandBuffer,
    size_t tBufferSize,
    THEATER_ID tID
        )
{
    BOOLEAN bSuccess;

    bSuccess = bDeleteTheaterFromDB(
        hSQLConnection, pcSQLCommandBuffer,tBufferSize, tID);

    return bSuccess;
}

/*****************************************************************************
*
*   MOVIES_MGR_bInsertTheater
*
*****************************************************************************/
BOOLEAN MOVIES_MGR_bInsertTheater (
    SQL_INTERFACE_OBJECT hSQLConnection,
    char *pcSQLCommandBuffer,
    size_t tBufferSize,
    THEATER_ROW_STRUCT *psTheaterRow
        )
{
    BOOLEAN bSuccess;

    // add theater to database
    bSuccess = bInsertTheaterIntoDB(
        hSQLConnection, pcSQLCommandBuffer,
        tBufferSize, psTheaterRow);

    // Failed to create the theater, so free the strings
    vFreeTheaterRowObjects(psTheaterRow);

    return bSuccess;
}

/*****************************************************************************
*
*   MOVIES_MGR_bUpdateTheater
*
*****************************************************************************/
BOOLEAN MOVIES_MGR_bUpdateTheater (
    SQL_INTERFACE_OBJECT hSQLConnection,
    char *pcSQLCommandBuffer,
    size_t tBufferSize,
    THEATER_ROW_STRUCT *psTheaterRow,
    BOOLEAN bAmenitiesUpdated
        )
{
    BOOLEAN bSuccess;

    // update theater in database
    bSuccess = bUpdateTheaterInDB(
        hSQLConnection, pcSQLCommandBuffer,
        tBufferSize, psTheaterRow, bAmenitiesUpdated);

    vFreeTheaterRowObjects(psTheaterRow);

    return bSuccess;
}

/*****************************************************************************
*
*   MOVIES_MGR_bClearRatings
*
*****************************************************************************/
BOOLEAN MOVIES_MGR_bClearRatings (
    SQL_INTERFACE_OBJECT hSQLConnection
        )
{
    BOOLEAN bSuccess;

    // Clear the ratings table
    bSuccess = SQL_INTERFACE.bExecuteCommand(
        hSQLConnection, MOVIES_CLEAR_DB_RATINGS);

    return bSuccess;
}

/*****************************************************************************
*
*   MOVIES_MGR_bInsertRatings
*
*****************************************************************************/
BOOLEAN MOVIES_MGR_bInsertRatings (
    SQL_INTERFACE_OBJECT hSQLConnection,
    MOVIE_RATINGS_ROW_STRUCT *psRatingsRow
        )
{
    BOOLEAN bSuccess;

    // Insert this row into the database now
    bSuccess = SQL_INTERFACE.bExecutePreparedCommand(
        hSQLConnection,
        MOVIES_INSERT_DB_RATINGS,
        DB_RATINGS_MAX_FIELDS,
        (PREPARED_QUERY_COLUMN_CALLBACK)bPrepareRatingsColumn,
        (void *)psRatingsRow);

    return bSuccess;
}

/*****************************************************************************
                             PRIVATE FUNCTIONS
*****************************************************************************/

/*****************************************************************************
*
*   vEventHandler
*
*   This function runs in the context of an SMS resource which has been
*   assigned to this service.
*
*****************************************************************************/
static void vEventHandler (
    DATASERVICE_MGR_OBJECT hDataService,
    DATASERVICE_EVENT_MASK tCurrentEvent,
    void *pvEventArg,
    void *pvEventCallbackArg
        )
{
    MOVIES_MGR_OBJECT_STRUCT *psObj;
    BOOLEAN bValid, bStopEvent = FALSE;
    SMSAPI_EVENT_MASK tEventMask = DATASERVICE_EVENT_NONE;

    // Get our movies handle from the data service manager
    psObj = (MOVIES_MGR_OBJECT_STRUCT *)pvEventCallbackArg;

    // Is this object valid?
    bValid = DATASERVICE_IMPL_bValid((DATASERVICE_IMPL_HDL)psObj);

    // Only handle events for valid objects...
    if (bValid == TRUE)
    {
        switch( tCurrentEvent )
        {
            // Handle Movies Service events here...

            // State has changed
            case DATASERVICE_EVENT_STATE:
            {
                BOOLEAN bStateChanged;
                DATASERVICE_STATE_CHANGE_STRUCT const *psStateChange =
                    (DATASERVICE_STATE_CHANGE_STRUCT const *)pvEventArg;

                // Process the state transition
                bStateChanged = DATASERVICE_IMPL_bStateFSM(
                    (DATASERVICE_IMPL_HDL)psObj,
                    psStateChange,
                    &GsMoviesStateHandlers,
                    (void *)psObj);

                if (bStateChanged == TRUE)
                {
                    // The state has been updated
                    tEventMask |= DATASERVICE_EVENT_STATE;

                    // Is the service stopped now?
                    if (psStateChange->eCurrentState ==
                            DATASERVICE_STATE_STOPPED)
                    {
                        bStopEvent = TRUE;
                    }
                }
            }
            break;

            // This service has a message to process
            case DATASERVICE_EVENT_NEW_DATA:
            {
                OSAL_BUFFER_HDL hPayload = (OSAL_BUFFER_HDL)pvEventArg;
                BOOLEAN bOk = TRUE;

                // Ensure the payload handle is valid
                // If it isn't, there's a problem with
                // SMS
                if (hPayload == OSAL_INVALID_BUFFER_HDL)
                {
                    bOk = FALSE;

                    // Set the error
                    vSetError( psObj,
                               DATASERVICE_ERROR_CODE_GENERAL);
                }
                else
                {
                    BOOLEAN bPayloadOk;

                    // Process this payload now
                    bPayloadOk = bProcessPayload(psObj, hPayload);

                    if (bPayloadOk == TRUE)
                    {
                        puts(MOVIES_MGR_OBJECT_NAME": Message Payload Processed Ok");
                    }
                    else
                    {
                        puts(MOVIES_MGR_OBJECT_NAME": Failed to process message");
                        DATASERVICE_IMPL_vLog(
                            MOVIES_MGR_OBJECT_NAME": Failed to process message\n");
                    }
                }

                if (bOk != TRUE)
                {
                    // If an error occurred, indicate a state change
                    // to the application
                    tEventMask |= DATASERVICE_EVENT_STATE;
                }
            }
            break;

            // We just experienced a movies data timeout
            case DATASERVICE_EVENT_TIMEOUT:
            {
                MOVIES_APP_OBJECT_STRUCT *psAppObj;

                psAppObj = psGetAppFacingObject((MOVIES_SERVICE_OBJECT)psObj);
        
                if (psAppObj != NULL)
                {
                    // Iterate all the theaters to remove movies
                    vRemoveMoviesAsExpired(psObj, psAppObj);

                    SMSO_vUnlock((SMS_OBJECT)psAppObj);
                }
            }
            break;

            // We need to do some work with our DSRLs
            case DATASERVICE_INTERNAL_EVENT_DSRL:
            {
                DSRL_ARG_STRUCT *psDSRLArg =
                    (DSRL_ARG_STRUCT *)pvEventArg;
                MOVIES_APP_OBJECT_STRUCT *psAppObj;
                BOOLEAN bSuccess = TRUE;

                psAppObj = psGetAppFacingObject((MOVIES_SERVICE_OBJECT)psObj);

                if (psAppObj != NULL)
                {
                    switch (psDSRLArg->eAction)
                    {
                        case DSRL_ACTION_ADD:
                        {
                            // Create a new list
                            bSuccess = bHandleCreateList(
                                psObj, psDSRLArg);
                        }
                        break;

                        case DSRL_ACTION_MODIFY:
                        {
                            // Modify a pre-existing list
                            bSuccess = bHandleModifyList (
                                psObj, psDSRLArg);
                        }
                        break;

                        case DSRL_ACTION_REFRESH:
                        {
                            // Refresh a pre-existing list
                            bSuccess = bHandleRefreshList (
                                psObj, psDSRLArg);
                        }
                        break;

                        case DSRL_ACTION_REMOVE:
                        {
                            // Handle the deletion
                            vHandleDeleteList(psObj, psDSRLArg);
                        }
                        break;

                        case DSRL_ACTION_INVALID:
                        default:
                        {
                            bSuccess = FALSE;
                        }
                        break;
                    }

                    // Check for un-used theaters after a DSRL update
                    vPruneTheaters(psObj);

                    SMSO_vUnlock((SMS_OBJECT)psAppObj);

                    if (bSuccess == FALSE)
                    {
                        SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                            MOVIES_MGR_OBJECT_NAME
                            ": DSRL Modify Action failed");
                        break;
                    }
                }
            }
            break;

            default:
            {
                SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    MOVIES_MGR_OBJECT_NAME": Got unknown event (%u)",
                    tCurrentEvent);
            }
            break;
        }

        if (bStopEvent == TRUE)
        {
            // Uninitialize the object, but leave
            // enough to be useful to the callback
            vUninitObject( psObj, FALSE );
        }

        // Update event mask with any relevant events which have occurred
        SMSU_tUpdate(&psObj->sEvent, tEventMask);

        // Notify of any change via any registered callback which may be present
        SMSU_bNotify(&psObj->sEvent);

        if (bStopEvent == TRUE)
        {
            // Filter out all further movies manager updates
            SMSU_tFilter(&psObj->sEvent, DATASERVICE_EVENT_ALL);

            vDestroyObject(psObj);
        }


    } // bValid == TRUE

    return;
}

/*****************************************************************************
*
*   psGetAppFacingObject
*
*****************************************************************************/
static MOVIES_APP_OBJECT_STRUCT *psGetAppFacingObject(
    MOVIES_SERVICE_OBJECT hMoviesService
        )
{
    MOVIES_MGR_OBJECT_STRUCT *psObj =
        (MOVIES_MGR_OBJECT_STRUCT *)hMoviesService;

    do
    {
        BOOLEAN bValid, bLocked;

        bValid = DATASERVICE_IMPL_bValid((DATASERVICE_IMPL_HDL)hMoviesService);

        if (bValid == FALSE)
        {
            break;
        }

        bLocked = SMSO_bLock((SMS_OBJECT)psObj->psAppObj,
            OSAL_OBJ_TIMEOUT_INFINITE);

        if (bLocked == FALSE)
        {
            break;
        }

        return psObj->psAppObj;

    } while (FALSE);

    return NULL;
}

/*****************************************************************************
*
*   bHandleServiceReady
*
*   This function is called when SMS is ready for the service to startup.
*   At this time, the service has a context in which to operate (SMS
*   assigned a resource to us). When this call is made, this service will
*   perform all of its time-intensive startup procedures.
*
*****************************************************************************/
static BOOLEAN bHandleServiceReady (
    MOVIES_MGR_OBJECT_STRUCT *psObj
        )
{
    BOOLEAN bOk = TRUE, bServiceUpdated = FALSE;
    DATASERVICE_ERROR_CODE_ENUM eErrorCode = DATASERVICE_ERROR_CODE_NONE;
    char *pacDatabaseFilePath,
         *pacDatabaseFilePathB;
    do
    {
        // Register for a timed event
        psObj->hDataExpireEvent =
            DATASERVICE_IMPL_hRegisterTimedEvent(
                (DATASERVICE_IMPL_HDL)psObj, NULL);

        if (psObj->hDataExpireEvent ==
            DATASERVICE_TIMED_EVENT_INVALID_HDL)
        {
            // Error!
            eErrorCode = DATASERVICE_ERROR_CODE_GENERAL;
            break;
        }

        // Construct the paths needed by the databases
        bOk = DB_UTIL_bCreateFilePath(
            psObj->pacRefDatabaseDirPath,
            MOVIES_DATABASE_FOLDER,
            MOVIES_REF_DATABASE_FILENAMEA,
            &pacDatabaseFilePath);

        if (bOk == FALSE)
        {
            // Error!  Couldn't build database path
            vSetError( psObj, DATASERVICE_ERROR_CODE_GENERAL );
            break;
        }

        // Construct the paths needed by the databases
        bOk = DB_UTIL_bCreateFilePath(
            psObj->pacRefDatabaseDirPath,
            MOVIES_DATABASE_FOLDER,
            MOVIES_REF_DATABASE_FILENAMEB,
            &pacDatabaseFilePathB);

        if (bOk == FALSE)
        {
            // Error!  Couldn't build database path
            vSetError( psObj, DATASERVICE_ERROR_CODE_GENERAL );
            break;
        }

        // Connect to the ref database bank, but
        // never perform an integrity check
        psObj->hSQLRefConnection =
            DB_UTIL_hConnectToReferenceBank (
                &pacDatabaseFilePath[0],
                &pacDatabaseFilePathB[0],
                &psObj->pacCurRefDatabaseFilePath,
                n32ExtractDataVersion, NULL,
                TRUE, &bServiceUpdated,
                GsMoviesIntf.tMaxVersionBitlen,
                (DB_UTIL_CHECK_VERSION_HANDLER)bVerifyDBVersion,
                (void *)(size_t)&psObj->tBaselineVersion,
                &eErrorCode,
                SQL_INTERFACE_OPTIONS_READONLY |
                SQL_INTERFACE_OPTIONS_SKIP_CORRUPTION_CHECK);

        if (psObj->pacCurRefDatabaseFilePath == pacDatabaseFilePath)
        {
            psObj->pacNextRefDatabaseFilePath = pacDatabaseFilePathB;
        }
        else
        {
            psObj->pacNextRefDatabaseFilePath = pacDatabaseFilePath;
        }

        // If the connection failed, indicate this
        if (psObj->hSQLRefConnection == SQL_INTERFACE_INVALID_OBJECT)
        {
            vSetError( psObj, eErrorCode );
            break;
        }

        // Load our ratings table now that we have a connection
        // to the refrence database
        bOk = bLoadRatingsTable( psObj );

        if (bOk == FALSE)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                MOVIES_MGR_OBJECT_NAME": Error reading the ratings table from the DB");
            vSetError(
                psObj,
                DATASERVICE_ERROR_CODE_DATABASE_ACCESS_FAILURE);
            break;
        }

        // initialize the interface-specific object, if we need to
        if (GsMoviesIntf.hInit != MOVIES_INTERFACE_INVALID_OBJECT)
        {
            // Make the call to init the broadcast-specific parser
            psObj->hMoviesInterfaceData = GsMoviesIntf.hInit(
                (MOVIES_SERVICE_OBJECT)psObj,
                DATASERVICE_IMPL_hSMSObj((DATASERVICE_IMPL_HDL)psObj),
                psObj->bRefDBUpdatesEnabled,
                (UN8)psObj->tBaselineVersion);

            if (psObj->hMoviesInterfaceData == MOVIES_INTERFACE_INVALID_OBJECT)
            {
                SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    MOVIES_MGR_OBJECT_NAME": Error initializing Movie Protocol Parser");

                // Error, not sure why this would happen
                vSetError( psObj, DATASERVICE_ERROR_CODE_GENERAL );
                break;
            }
        }

        // Build the path to the persitant storage DB
        bOk = DB_UTIL_bCreateFilePath(
            NULL, // We don't know the base path
            MOVIES_DATABASE_FOLDER,
            MOVIES_PERSIST_DATABASE_FILENAME,
            &pacDatabaseFilePath);

        if (bOk == FALSE)
        {
            // Error!  Couldn't build database path
            vSetError( psObj, DATASERVICE_ERROR_CODE_GENERAL );
            break;
        }

        if (bServiceUpdated == TRUE)
        {
            // We just updated our reference DB -- so
            // now we trash the persistent DB
            remove(&pacDatabaseFilePath[0]);
        }

        // Connect to the persitent storage database
        psObj->hSQLPersistConnection =
            DB_UTIL_hConnectToPersistent(
                &pacDatabaseFilePath[0],
                bCreatePersistDBTables,
                (void *)(size_t)GsMoviesIntf.tDSI,
                (DB_UTIL_CHECK_VERSION_HANDLER)bVerifyDBVersion,
                (void *)NULL,
                &eErrorCode);

        // Done with that
        SMSO_vDestroy((SMS_OBJECT)pacDatabaseFilePath);

        if (psObj->hSQLPersistConnection == SQL_INTERFACE_INVALID_OBJECT)
        {
            vSetError( psObj, eErrorCode );

            break;
        }

        psObj->hInsertTheaterTimesStmt =
            SQL_INTERFACE.hCreatePreparedStatement(
                psObj->hSQLPersistConnection,
                MOVIES_INSERT_TIMES_FOR_THEATER);

        if (psObj->hInsertTheaterTimesStmt ==
            SQL_PREPARED_STATEMENT_INVALID_HANDLE)
        {
            vSetError(psObj, DATASERVICE_ERROR_CODE_GENERAL);
            break;
        }

        psObj->hInsertDescriptionStmt =
            SQL_INTERFACE.hCreatePreparedStatement(
                psObj->hSQLPersistConnection,
                MOVIES_INSERT_DESCRIPTION);

        if (psObj->hInsertDescriptionStmt ==
            SQL_PREPARED_STATEMENT_INVALID_HANDLE)
        {
            vSetError(psObj, DATASERVICE_ERROR_CODE_GENERAL);
            break;
        }

        psObj->hClearMoviesStmt = 
            SQL_INTERFACE.hCreatePreparedStatement(
                psObj->hSQLPersistConnection,
                MOVIES_CLEAR_DESCRIPTIONS);

        if (psObj->hClearMoviesStmt ==
            SQL_PREPARED_STATEMENT_INVALID_HANDLE)
        {
            vSetError(psObj, DATASERVICE_ERROR_CODE_GENERAL);
            break;
        }

        // Load any persistent data we have
        bOk = bLoadPersistentData( psObj );
        if (bOk == FALSE)
        {
            vSetError(psObj, DATASERVICE_ERROR_CODE_DATABASE_CORRUPT);

            break;
        }

        return TRUE;

    } while (FALSE);

    return FALSE;
}

/*****************************************************************************
*
*   bHandleServiceError
*
*****************************************************************************/
static BOOLEAN bHandleServiceError (
    MOVIES_MGR_OBJECT_STRUCT *psObj
        )
{
    // Stop any timers we have now 'cause
    // who knows what's going on up there
    vStopAgeoutTimer(psObj);

    return TRUE;
}

/*****************************************************************************
*
*   psBuildMovieDescStruct
*
*****************************************************************************/
static MOVIE_DESCRIPTION_CONTROL_STRUCT *psBuildMovieDescStruct (
    SMS_OBJECT hOwner
        )
{
    OSAL_RETURN_CODE_ENUM eReturnCode;
    MOVIE_DESCRIPTION_CONTROL_STRUCT *psDesc =
        (MOVIE_DESCRIPTION_CONTROL_STRUCT *)SMSO_hCreate(
            MOVIES_MGR_OBJECT_NAME":MvDesc",
            sizeof(MOVIE_DESCRIPTION_CONTROL_STRUCT),
            (SMS_OBJECT)hOwner,
            FALSE);

    do
    {
        if (psDesc == NULL)
        {
            break;
        }

        // Initialize the member variables
        psDesc->tVersion = MOVIE_INVALID_VERSION;
        psDesc->eState = MOVIE_OTA_UPDATE_STATE_INITIAL;

        // Build the list for the movies
        eReturnCode = OSAL.eLinkedListCreate(
            &psDesc->hDescriptions,
            MOVIES_MGR_OBJECT_NAME":MvDescs",
            (OSAL_LL_COMPARE_HANDLER)MOVIE_n16CompareID,
            OSAL_LL_OPTION_NONE);

        if (eReturnCode != OSAL_SUCCESS)
        {
            break;
        }

        // All done
        return psDesc;

    } while (FALSE);

    if (NULL != psDesc)
    {
        SMSO_vDestroy((SMS_OBJECT)psDesc);
    }

    return NULL;
}

/*****************************************************************************
*
*   vFreeMovieDescStruct
*
*****************************************************************************/
static void vFreeMovieDescStruct (
    MOVIE_DESCRIPTION_CONTROL_STRUCT *psDesc
        )
{
    OSAL_RETURN_CODE_ENUM eReturnCode;

    if (psDesc != NULL)
    {
        eReturnCode = OSAL.eLinkedListRemoveAll(psDesc->hDescriptions,
            (OSAL_LL_RELEASE_HANDLER)MOVIE_vDestroy);

        if (eReturnCode == OSAL_SUCCESS)
        {
            eReturnCode = OSAL.eLinkedListDelete(psDesc->hDescriptions);

            if (eReturnCode == OSAL_SUCCESS)
            {
                psDesc->hDescriptions = OSAL_INVALID_OBJECT_HDL;
            }
        }

        psDesc->tVersion = MOVIE_INVALID_VERSION;

        SMSO_vDestroy((SMS_OBJECT)psDesc);
    }

    return;
}


/*****************************************************************************
*
*   psBuildMovieTimesStruct
*
*****************************************************************************/
static MOVIE_TIMES_CONTROL_STRUCT *psBuildMovieTimesStruct (
    SMS_OBJECT hOwner
        )
{
    OSAL_RETURN_CODE_ENUM eReturnCode;
    MOVIE_TIMES_CONTROL_STRUCT *psTimes =
        (MOVIE_TIMES_CONTROL_STRUCT *)SMSO_hCreate(
            MOVIES_MGR_OBJECT_NAME":MvTimes",
            sizeof(MOVIE_TIMES_CONTROL_STRUCT),
            (SMS_OBJECT)hOwner,
            FALSE);

    do
    {
        if (psTimes == NULL)
        {
            break;
        }

        // Initilize member variables
        psTimes->tVersion = MOVIE_INVALID_VERSION;
        psTimes->eState = MOVIE_OTA_UPDATE_STATE_INITIAL;

        // Build the list for the theater times objects
        eReturnCode = OSAL.eLinkedListCreate(
            &psTimes->hTimes,
            MOVIES_MGR_OBJECT_NAME":MvTimesList",
            (OSAL_LL_COMPARE_HANDLER)n16CompareTheaterTimes,
            OSAL_LL_OPTION_CIRCULAR);

        if (eReturnCode != OSAL_SUCCESS)
        {
            break;
        }

        // All done
        return psTimes;

    } while (FALSE);

    if (NULL != psTimes)
    {
        SMSO_vDestroy((SMS_OBJECT)psTimes);
    }

    return NULL;
}

/*****************************************************************************
*
*   vFreeMovieTimesStruct
*
*****************************************************************************/
static void vFreeMovieTimesStruct (
    MOVIE_TIMES_CONTROL_STRUCT *psTimes
        )
{
    OSAL_RETURN_CODE_ENUM eReturnCode;

    if (psTimes != NULL)
    {
        eReturnCode = OSAL.eLinkedListRemoveAll(psTimes->hTimes,
            (OSAL_LL_RELEASE_HANDLER)vDestroyTheaterTimesEntry);

        if (eReturnCode == OSAL_SUCCESS)
        {
            eReturnCode = OSAL.eLinkedListDelete(psTimes->hTimes);

            if (eReturnCode == OSAL_SUCCESS)
            {
                psTimes->hTimes = OSAL_INVALID_OBJECT_HDL;
            }
        }

        if (psTimes->hScratchTimes != THEATER_TIMES_INVALID_OBJECT)
        {
            THEATER_TIMES_vDestroy(psTimes->hScratchTimes);

            psTimes->hScratchTimes = THEATER_TIMES_INVALID_OBJECT;
        }

        psTimes->tVersion = MOVIE_INVALID_VERSION;

        SMSO_vDestroy((SMS_OBJECT)psTimes);
    }

    return;
}

/*****************************************************************************
*
*   vDestroyTheaterTimesEntry
*
*****************************************************************************/
static void vDestroyTheaterTimesEntry (
    MOVIE_THEATER_TIMES_ENTRY_STRUCT *psTimesEntry
        )
{
    if (psTimesEntry != NULL)
    {
        if (psTimesEntry->hShowTimes != THEATER_TIMES_INVALID_OBJECT)
        {
            THEATER_TIMES_vDestroy(psTimesEntry->hShowTimes);
            psTimesEntry->hShowTimes = NULL;
        }

        psTimesEntry->tTheaterID = THEATER_INVALID_ID;

        SMSO_vDestroy((SMS_OBJECT)psTimesEntry);
    }

    return;
}

/*****************************************************************************
*
*   bInitAppFacingObject
*
*****************************************************************************/
static BOOLEAN bInitAppFacingObject (
    MOVIES_MGR_OBJECT_STRUCT *psObj
        )
{
    OSAL_RETURN_CODE_ENUM eReturnCode;
    DATASERVICE_DSRL_CONFIG_STRUCT sDSRLConfig;

    // Begin DSRL config
    DATASERVICE_IMPL_vInitializeDSRLConfig(&sDSRLConfig);

    // Set service type
    sDSRLConfig.eServiceType = DATASERVICE_TYPE_MOVIES;

    // Set the parent object size
    sDSRLConfig.tParentObjectSize = sizeof(MOVIES_APP_OBJECT_STRUCT);

    // Set the DSRL descriptor size
    sDSRLConfig.tServiceDataSize = sizeof(MOVIES_DSRL_DESC_STRUCT);

    // Configure the favorites feature
    sDSRLConfig.bFavoritesEnabled = TRUE;
    sDSRLConfig.hCreateTargetFromTag = LOCATION_hCreateTargetFromTag;
    sDSRLConfig.hGetTagForTarget = LOCATION_hGetTargetTag;

    // Configure the device feature
    sDSRLConfig.bDeviceEnabled = TRUE;
    sDSRLConfig.un32DeviceNotifyDistance = MOVIES_DEVICE_DISTANCE_THRESHOLD;
    sDSRLConfig.eDeviceNotifyUnits = DISTANCE_UNIT_TYPE_MILES;

    // Create it
    psObj->psAppObj = (MOVIES_APP_OBJECT_STRUCT *)
        DATASERVICE_IMPL_hConfigureDSRL(
            (DATASERVICE_IMPL_HDL)psObj, &sDSRLConfig);

    if (psObj->psAppObj == NULL)
    {
        return FALSE;
    }

    psObj->psAppObj->psCurrentMovies = psBuildMovieDescStruct(
        (SMS_OBJECT)psObj->psAppObj);

    if (psObj->psAppObj->psCurrentMovies == NULL)
    {
        // Error!
        vUninitAppFacingObject(psObj);
        return FALSE;
    }

    psObj->psAppObj->psCurrentTimes = psBuildMovieTimesStruct(
        (SMS_OBJECT)psObj->psAppObj);

    if (psObj->psAppObj->psCurrentTimes == NULL)
    {
        // Error!
        vUninitAppFacingObject(psObj);
        return FALSE;
    }

    // Create our ratings list
    eReturnCode = OSAL.eLinkedListCreate(
               &psObj->psAppObj->hRatingsList,
               MOVIES_MGR_OBJECT_NAME":RatingsList",
               (OSAL_LL_COMPARE_HANDLER)n16CompareRatingsEntry,
               (OSAL_LL_OPTION_LINEAR | OSAL_LL_OPTION_UNIQUE)
                   );

    if(eReturnCode != OSAL_SUCCESS)
    {
        // Error!
        vUninitAppFacingObject(psObj);
        return FALSE;
    }

    // Create our theater list
    eReturnCode = OSAL.eLinkedListCreate(
               &psObj->psAppObj->hTheaterList,
               MOVIES_MGR_OBJECT_NAME":TheaterList",
               (OSAL_LL_COMPARE_HANDLER)n16CompareTheaterEntry,
               OSAL_LL_OPTION_CIRCULAR
                   );

    if(eReturnCode != OSAL_SUCCESS)
    {
        // Error!
        vUninitAppFacingObject(psObj);
        return FALSE;
    }

    // Create our dummy movie object (for quick searching)
    psObj->psAppObj->hDummyMovie = 
        MOVIE_hCreateDummy((SMS_OBJECT)psObj->psAppObj);
    if (MOVIE_INVALID_OBJECT == psObj->psAppObj->hDummyMovie)
    {
        // Error!
        vUninitAppFacingObject(psObj);
        return FALSE;
    }

    // Relinquish ownership
    SMSO_vUnlock((SMS_OBJECT) psObj->psAppObj);

    return TRUE;
}

/*****************************************************************************
*
*   vUninitAppFacingObject
*
*****************************************************************************/
static void vUninitAppFacingObject(
    MOVIES_MGR_OBJECT_STRUCT *psObj
        )
{
    if ((psObj != NULL) && (psObj->psAppObj != NULL))
    {
        OSAL_RETURN_CODE_ENUM eReturnCode;

        if (MOVIE_INVALID_OBJECT != psObj->psAppObj->hDummyMovie)
        {
            MOVIE_vDestroy(psObj->psAppObj->hDummyMovie);
            psObj->psAppObj->hDummyMovie = MOVIE_INVALID_OBJECT;
        }

        vFreeMovieDescStruct(psObj->psAppObj->psCurrentMovies);

        vFreeMovieTimesStruct(psObj->psAppObj->psCurrentTimes);

        // Destroy the theater list
        eReturnCode = OSAL.eLinkedListRemoveAll(
           psObj->psAppObj->hTheaterList,
           (OSAL_LL_RELEASE_HANDLER)vDestroyTheaterEntry);

        if (eReturnCode == OSAL_SUCCESS)
        {
           OSAL.eLinkedListDelete(psObj->psAppObj->hTheaterList);
           psObj->psAppObj->hTheaterList = OSAL_INVALID_OBJECT_HDL;
        }

        // Destroy the ratings list
        eReturnCode = OSAL.eLinkedListRemoveAll(
           psObj->psAppObj->hRatingsList, (OSAL_LL_RELEASE_HANDLER)vDestroyRatingsEntry);

        if (eReturnCode == OSAL_SUCCESS)
        {
           OSAL.eLinkedListDelete(psObj->psAppObj->hRatingsList);
           psObj->psAppObj->hRatingsList = OSAL_INVALID_OBJECT_HDL;
        }

        // Destroy the app object
        DATASERVICE_IMPL_vDestroyDSRLParent((DATASERVICE_IMPL_HDL)psObj,(SMS_OBJECT)psObj->psAppObj);
        psObj->psAppObj = NULL;
    }

    return;
}

/*****************************************************************************
*
*   vUninitObject
*
*****************************************************************************/
static void vUninitObject (
    MOVIES_MGR_OBJECT_STRUCT *psObj,
    BOOLEAN bFullDelete
        )
{
    OSAL_RETURN_CODE_ENUM eReturnCode;

    // Get the app facing object
    psGetAppFacingObject((MOVIES_SERVICE_OBJECT)psObj);

    // Clear the timed event handle
    psObj->hDataExpireEvent = DATASERVICE_TIMED_EVENT_INVALID_HDL;

    // Remove any broadcast-specific parsing data
    if (GsMoviesIntf.vUninit != NULL)
    {
        // Make the call to uninit the broadcast-specific parser?
        GsMoviesIntf.vUninit(psObj->hMoviesInterfaceData);
        psObj->hMoviesInterfaceData = MOVIES_INTERFACE_INVALID_OBJECT;
    }

    // Disconnect from the reference database
    if (psObj->hSQLRefConnection != SQL_INTERFACE_INVALID_OBJECT)
    {
        SQL_INTERFACE.vDisconnect( psObj->hSQLRefConnection );
        psObj->hSQLRefConnection = SQL_INTERFACE_INVALID_OBJECT;
    }

    if (psObj->pacCurRefDatabaseFilePath != NULL)
    {
        SMSO_vDestroy((SMS_OBJECT)psObj->pacCurRefDatabaseFilePath);
        psObj->pacCurRefDatabaseFilePath = NULL;
    }

    if (psObj->pacNextRefDatabaseFilePath != NULL)
    {
        SMSO_vDestroy((SMS_OBJECT)psObj->pacNextRefDatabaseFilePath);
        psObj->pacNextRefDatabaseFilePath = NULL;
    }

    if (psObj->pacRefDatabaseDirPath != NULL)
    {
        SMSO_vDestroy((SMS_OBJECT)psObj->pacRefDatabaseDirPath);
        psObj->pacRefDatabaseDirPath = NULL;
    }

    // Disconnect from the persistent database
    if (psObj->hSQLPersistConnection != SQL_INTERFACE_INVALID_OBJECT)
    {
        SQL_INTERFACE.vDestroyPreparedStatement(
            psObj->hSQLPersistConnection,
            psObj->hInsertTheaterTimesStmt);

        psObj->hInsertTheaterTimesStmt =
            SQL_PREPARED_STATEMENT_INVALID_HANDLE;

        SQL_INTERFACE.vDestroyPreparedStatement(
            psObj->hSQLPersistConnection,
            psObj->hInsertDescriptionStmt);

        psObj->hInsertDescriptionStmt =
            SQL_PREPARED_STATEMENT_INVALID_HANDLE;

        SQL_INTERFACE.vDestroyPreparedStatement(
            psObj->hSQLPersistConnection,
            psObj->hClearMoviesStmt);

        psObj->hClearMoviesStmt =
            SQL_PREPARED_STATEMENT_INVALID_HANDLE;

        SQL_INTERFACE.vDisconnect( psObj->hSQLPersistConnection );
        psObj->hSQLPersistConnection = SQL_INTERFACE_INVALID_OBJECT;
    }

    // Destroy our Movie Desc Structures
    vFreeMovieDescStruct(psObj->psInProgressMovies);

    // Destroy our Movie Times Structures
    vFreeMovieTimesStruct(psObj->psInProgressTimes);

    // Destroy the target list
    eReturnCode = OSAL.eLinkedListRemoveAll(
        psObj->hDSRLList, (OSAL_LL_RELEASE_HANDLER)vReleaseDSRLDesc);

    if (eReturnCode == OSAL_SUCCESS)
    {
        OSAL.eLinkedListDelete(psObj->hDSRLList);
        psObj->hDSRLList = OSAL_INVALID_OBJECT_HDL;
    }

    vUninitAppFacingObject(psObj);

    if (bFullDelete == TRUE)
    {
        vDestroyObject(psObj);
    }

    return;
}

/*****************************************************************************
*
*   vRemoveMoviesAsExpired
*
*****************************************************************************/
static void vRemoveMoviesAsExpired (
    MOVIES_MGR_OBJECT_STRUCT *psObj,
    MOVIES_APP_OBJECT_STRUCT *psAppObj
        )
{
    OSAL_RETURN_CODE_ENUM eReturnCode;
    BOOLEAN bSuccess = FALSE;

    do
    {
        // Clear the current movies list
        eReturnCode = OSAL.eLinkedListRemoveAll(
            psAppObj->psCurrentMovies->hDescriptions,
            (OSAL_LL_RELEASE_HANDLER)MOVIE_vDestroy);

        if (eReturnCode != OSAL_SUCCESS)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                MOVIES_MGR_OBJECT_NAME
                ": vRemoveMoviesAsExpired() Error clearing movie list (%s)",
                OSAL.pacGetReturnCodeName(eReturnCode));
            break;
        }

        // Clear the current times list
        eReturnCode = OSAL.eLinkedListRemoveAll(
            psAppObj->psCurrentTimes->hTimes,
            (OSAL_LL_RELEASE_HANDLER)vDestroyTheaterTimesEntry);

        if (eReturnCode != OSAL_SUCCESS)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                MOVIES_MGR_OBJECT_NAME
                ": vRemoveMoviesAsExpired() Error clearing times list (%s)",
                OSAL.pacGetReturnCodeName(eReturnCode));
            break;
        }

        // Update times in DSRLs
        bSuccess = bUpdateTheatersWithTimes(psObj);

        if (bSuccess == FALSE)
        {
            break;
        }

        vUpdateAllDSRLStates(psObj, DSRL_STATE_READY);

        // Clear the current times and movies structures
        psAppObj->psCurrentTimes->eState = MOVIE_OTA_UPDATE_STATE_INITIAL;
        psAppObj->psCurrentTimes->tVersion = MOVIE_INVALID_VERSION;
        psAppObj->psCurrentTimes->un32UTCsec = 0;

        psAppObj->psCurrentMovies->eState = MOVIE_OTA_UPDATE_STATE_INITIAL;
        psAppObj->psCurrentMovies->tVersion = MOVIE_INVALID_VERSION;
        psAppObj->psCurrentMovies->un32UTCsec = 0;

    } while (FALSE);

    return;
}

/*****************************************************************************
*
*   vDestroyObject
*
*****************************************************************************/
static void vDestroyObject (
    MOVIES_MGR_OBJECT_STRUCT *psObj
        )
{
    // Destroy the SMS Update Event object
    SMSU_vDestroy(&psObj->sEvent);

    return;
}

/*****************************************************************************
*
*   bHandleCreateList
*
*****************************************************************************/
static BOOLEAN bHandleCreateList (
    MOVIES_MGR_OBJECT_STRUCT *psObj,
    DSRL_ARG_STRUCT *psDSRLArg
        )
{
    BOOLEAN bSuccess = FALSE;
    OSAL_RETURN_CODE_ENUM eReturnCode;
    MOVIES_DSRL_DESC_STRUCT *psDSRLDesc;

    do
    {
        // Create our DSRL descriptor based
        // on the provided arguments
        psDSRLDesc = psCreateDSRLDesc(psObj, psDSRLArg);
        if (psDSRLDesc == NULL)
        {
            // Error!
            break;
        }


        // Add this to our list of tracked targets for this service
        eReturnCode = OSAL.eLinkedListAdd(
            psObj->hDSRLList, &psDSRLDesc->hEntry, psDSRLDesc);

        if (eReturnCode != OSAL_SUCCESS)
        {
            vReleaseDSRLDesc(psDSRLDesc);
            break;
        }

        // Set the default sort function
        bSuccess = DSRL_bSetDefaultSortFunction(
            psDSRLDesc->hDSRL,
            (DSRL_SORT_FUNCTION)n16SortDSRLByDistance,
            (void *)psDSRLDesc);
        if (FALSE == bSuccess)
        {
            break;
        }

        // Set the finalize function
        bSuccess = DSRL_bSetFinalizeFunction(psDSRLDesc->hDSRL,
            (DSRL_FINALIZE_FUNCTION)vFinalizeDSRLEntry, psObj);
        if (FALSE == bSuccess)
        {
            break;
        }

        // Build our target descriptions
        bSuccess = bBuildTargetsForDSRL(psObj, psDSRLDesc, psDSRLArg);

        if (bSuccess == FALSE)
        {
            break;
        }

        // Start populating the list (and move it to ready)
        bSuccess = bBuildDSRL(psObj, psDSRLDesc, TRUE, FALSE);

        if (bSuccess == TRUE)
        {
            DSRL_STATE_ENUM eDSRLState;

            eDSRLState = DSRL.eState(psDSRLDesc->hDSRL);

            // Clear out the tracked payloads for theater times
            // so that we can re-aquire for the new target if we aren't ready
            if (eDSRLState != DSRL_STATE_READY)
            {
                GsMoviesIntf.bResetTimesPayloadTracking(
                    psObj->hMoviesInterfaceData);
            }
        }

    } while(FALSE);

    return bSuccess;
}

/*****************************************************************************
*
*   bHandleModifyList
*
*****************************************************************************/
static BOOLEAN bHandleModifyList (
    MOVIES_MGR_OBJECT_STRUCT *psObj,
    DSRL_ARG_STRUCT *psDSRLArg
        )
{
    BOOLEAN bSuccess = FALSE;
    MOVIES_DSRL_DESC_STRUCT *psDSRLDesc = NULL;
    DSRL_STATE_ENUM eNextState = DSRL_STATE_READY;

    do
    {
        // Extract the DSRL descriptor
        psDSRLDesc = (MOVIES_DSRL_DESC_STRUCT *)
            DSRL_pvServiceData(psDSRLArg->hDSRL);
        if (psDSRLDesc == NULL)
        {
            // Error!
            break;
        }

        // This operation tells us explicitly if
        // we should do this or not
        if (psDSRLArg->uAction.sModify.bForceDSRLStateChange == TRUE)
        {
            // Set the DSRL state to updating state
            DSRL_vSetState(psDSRLDesc->hDSRL,
                DSRL_STATE_UPDATING);
        }

        // Modify the DSRL as the user specifed
        switch (psDSRLArg->uAction.sModify.eModifyType)
        {
            case DSRL_MODIFY_OPERATION_ADD:
            {
                bSuccess = bHandleAddTargetList(
                    psObj, psDSRLDesc, psDSRLArg);
            }
            break;

            case DSRL_MODIFY_OPERATION_REPLACE:
            {
                // Handle this in another function
                bSuccess = bHandleReplaceTargetList(
                    psObj, psDSRLDesc, psDSRLArg);
            }
            break;

            case DSRL_MODIFY_OPERATION_REMOVE:
            {
                // Remove the specified targets
                bSuccess = bHandleRemoveTargets(
                    psObj, psDSRLArg, psDSRLDesc);
            }
            break;

            case DSRL_MODIFY_OPERATION_REMOVEALL:
            {
                OSAL_RETURN_CODE_ENUM eReturnCode;

                // Remove all targets
                DSRL_vRemoveAllEntries(psDSRLDesc->hDSRL);

                OSAL.eLinkedListIterate(
                    psObj->psAppObj->hTheaterList,
                    (OSAL_LL_ITERATOR_HANDLER)bRemoveDSRLDescFromTheaterEntry,
                    psDSRLDesc);

                eReturnCode = OSAL.eLinkedListRemoveAll(
                    psDSRLDesc->hTargetList,
                    (OSAL_LL_RELEASE_HANDLER)vReleaseDSRLTargetDesc);

                if (eReturnCode == OSAL_SUCCESS)
                {
                    bSuccess = TRUE;
                }
            }
            break;

            case DSRL_MODIFY_OPERATION_INVALID:
            default:
                break;

        }

    } while (FALSE);

    if (bSuccess == FALSE)
    {
        eNextState = DSRL_STATE_ERROR;
    }

    if (psDSRLDesc != NULL)
    {
        // Set the next state for the DSRL
        DSRL_vSetState(psDSRLDesc->hDSRL, eNextState);
    }

    return bSuccess;
}

/*****************************************************************************
*
*   bHandleAddTargetList
*
*****************************************************************************/
static BOOLEAN bHandleAddTargetList (
    MOVIES_MGR_OBJECT_STRUCT *psObj,
    MOVIES_DSRL_DESC_STRUCT *psDSRLDesc,
    DSRL_ARG_STRUCT *psDSRLArg
        )
{
    BOOLEAN bSuccess;

    bSuccess = bBuildTargetsForDSRL(psObj, psDSRLDesc, psDSRLArg);

    if (bSuccess == TRUE)
    {
        // Build the dsrl, but don't manipulate DSRL state
        bSuccess = bBuildDSRL(psObj, psDSRLDesc, FALSE, FALSE);
    }

    return bSuccess;
}

/*****************************************************************************
*
*   bHandleReplaceTargetList
*
*****************************************************************************/
static BOOLEAN bHandleReplaceTargetList (
    MOVIES_MGR_OBJECT_STRUCT *psObj,
    MOVIES_DSRL_DESC_STRUCT *psDSRLDesc,
    DSRL_ARG_STRUCT *psDSRLArg
        )
{
    BOOLEAN bSuccess = FALSE;

    do
    {
        OSAL_RETURN_CODE_ENUM eReturnCode;
        BOOLEAN bBuilt, bSortOk;

        // Iterate all the theaters to prepare all DSRL entries
        // associated with this DSRL for a replace/refresh
        eReturnCode = OSAL.eLinkedListIterate(
            psObj->psAppObj->hTheaterList,
            (OSAL_LL_ITERATOR_HANDLER)bPrepareTheaterEntriesForReplace,
            psDSRLDesc);

        if ((eReturnCode != OSAL_SUCCESS) && (eReturnCode != OSAL_NO_OBJECTS))
        {
            break;
        }

        // Remove all the current targets and clear their lists
        eReturnCode = OSAL.eLinkedListRemoveAll(
            psDSRLDesc->hTargetList,
            (OSAL_LL_RELEASE_HANDLER)vReleaseDSRLTargetDesc);

        if (eReturnCode != OSAL_SUCCESS)
        {
            break;
        }

        // Build the new targets now
        bBuilt = bBuildTargetsForDSRL(psObj, psDSRLDesc, psDSRLArg);
        if (FALSE == bBuilt)
        {
            break;
        }

        // Our default sort uses the target list
        // as the sort criteria...it just changed so
        // it's time for a re-sort
        bSortOk = DSRL_bSort(psDSRLDesc->hDSRL);
        if (FALSE == bSortOk)
        {
            // Error!
            break;
        }

        // Build the DSRL and load theaters, but don't
        // let this function determine when to move
        // this DSRL to ready
        bBuilt = bBuildDSRL(psObj, psDSRLDesc, FALSE, FALSE);
        if (FALSE == bBuilt)
        {
            break;
        }

        // Clear out any entries which no longer make the cut
        eReturnCode = OSAL.eLinkedListIterate(
            psObj->psAppObj->hTheaterList,
            (OSAL_LL_ITERATOR_HANDLER)bPostProcessTheaterEntriesForReplace,
            psDSRLDesc);

        if ((eReturnCode != OSAL_SUCCESS) && (eReturnCode != OSAL_NO_OBJECTS))
        {
            break;
        }

        // All good now
        bSuccess = TRUE;

    } while (FALSE);

    if (bSuccess == TRUE)
    {
        DSRL_vSetState(psDSRLDesc->hDSRL, DSRL_STATE_READY);
    }
    else
    {
        DSRL_vSetState(psDSRLDesc->hDSRL, DSRL_STATE_ERROR);
    }

    return bSuccess;
}
/*****************************************************************************
*
*   bHandleRefreshList
*
*****************************************************************************/
static BOOLEAN bHandleRefreshList (
    MOVIES_MGR_OBJECT_STRUCT *psObj,
    DSRL_ARG_STRUCT *psDSRLArg
        )
{
    BOOLEAN bSuccess = FALSE, bBuilt = FALSE;
    MOVIES_DSRL_DESC_STRUCT *psDSRLDesc = NULL;
    OSAL_RETURN_CODE_ENUM eReturnCode;
    DSRL_STATE_ENUM eNextState = DSRL_STATE_READY;

    do
    {
        // Extract the DSRL descriptor
        psDSRLDesc = (MOVIES_DSRL_DESC_STRUCT *)
            DSRL_pvServiceData(psDSRLArg->hDSRL);
        if (psDSRLDesc == NULL)
        {
            // Error!
            break;
        }

        // Set the DSRL state to updating state
        DSRL_vSetState(psDSRLDesc->hDSRL, DSRL_STATE_UPDATING);

        // Iterate all the theaters to prepare all DSRL entries
        // associated with this DSRL for a replace/refresh
        eReturnCode = OSAL.eLinkedListIterate(
            psObj->psAppObj->hTheaterList,
            (OSAL_LL_ITERATOR_HANDLER)bPrepareTheaterEntriesForReplace,
            psDSRLDesc);

        if ((eReturnCode != OSAL_SUCCESS) && (eReturnCode != OSAL_NO_OBJECTS))
        {
            break;
        }

        // Iterate the targets in order to clear
        // their state & theater lists
        eReturnCode = OSAL.eLinkedListIterate(
            psDSRLDesc->hTargetList,
            (OSAL_LL_ITERATOR_HANDLER)bIterateTargetsForRefresh,
            NULL );
        if (eReturnCode != OSAL_SUCCESS)
        {
            break;
        }

        // Build the DSRL and load theaters, but don't
        // let this function determine when to move
        // this DSRL to ready
        bBuilt = bBuildDSRL(psObj, psDSRLDesc, FALSE, TRUE);
        if (bBuilt == FALSE)
        {
            break;
        }

        // Clear out any entries which no longer make the cut
        eReturnCode = OSAL.eLinkedListIterate(
            psObj->psAppObj->hTheaterList,
            (OSAL_LL_ITERATOR_HANDLER)bPostProcessTheaterEntriesForReplace,
            psDSRLDesc);

        if ((eReturnCode != OSAL_SUCCESS) && (eReturnCode != OSAL_NO_OBJECTS))
        {
            break;
        }

        bSuccess = TRUE;

    } while (FALSE);

    if (bSuccess == FALSE)
    {
        eNextState = DSRL_STATE_ERROR;
    }

    if (psDSRLDesc != NULL)
    {
        // Bring the DSRL to its next state
        DSRL_vSetState(psDSRLDesc->hDSRL, eNextState);
    }

    return bSuccess;
}

/*****************************************************************************
*
*   bHandleRemoveTargets
*
*****************************************************************************/
static BOOLEAN bHandleRemoveTargets (
    MOVIES_MGR_OBJECT_STRUCT *psObj,
    DSRL_ARG_STRUCT *psDSRLArg,
    MOVIES_DSRL_DESC_STRUCT *psDSRLDesc
        )
{
    size_t tIndex;
    MOVIES_DSRL_TARGET_DESC_STRUCT sTargetDescSearch;
    MOVIES_DSRL_TARGET_DESC_STRUCT *psCurTargetDesc;
    MOVIE_THEATER_ENTRY_STRUCT *psCurTheaterEntry;
    OSAL_RETURN_CODE_ENUM eReturnCode;
    BOOLEAN bSuccess = TRUE, bSortNeeded = FALSE;
    OSAL_LINKED_LIST_ENTRY hEntry = OSAL_INVALID_LINKED_LIST_ENTRY,
                           hEntryToRemove = OSAL_INVALID_LINKED_LIST_ENTRY,
                           hFirstTarget;

    // Get the first target entry in the descriptor.  If we end up removing
    // this target it means we need to re-sort the DSRL (the default sort is
    // by the first target's location)
    hFirstTarget = OSAL.hLinkedListFirst(
        psDSRLDesc->hTargetList, (void **)NULL);

    // Iterate through all the targets in this argument
    for (tIndex = 0; tIndex < psDSRLArg->tNumTargets; tIndex++)
    {
        // Find the current target in our list
        sTargetDescSearch.hLocation =
            psDSRLArg->ahTargetList[tIndex];

        // Locate this target in the list
        eReturnCode = OSAL.eLinkedListLinearSearch(
            psDSRLDesc->hTargetList, &hEntry,
            (OSAL_LL_COMPARE_HANDLER)n16CompareTargetDesc,
            &sTargetDescSearch);

        // We are done searching for this target, so destroy it
        // Do this before we check the error code so that we don't leave any
        // memory lying around
        DSRL_TARGET_vDestroyByType(psDSRLArg->ahTargetList[tIndex]);
        psDSRLArg->ahTargetList[tIndex] = DSRL_TARGET_INVALID_OBJECT;

        // If this target isn't in our list, continue
        if (eReturnCode != OSAL_SUCCESS)
        {
            continue;
        }

        // Extract target descriptor
        psCurTargetDesc = (MOVIES_DSRL_TARGET_DESC_STRUCT *)
            OSAL.pvLinkedListThis(hEntry);

        // Is this our first target?
        if (hEntry == hFirstTarget)
        {
            // Yes, we need to re-sort the DSRL
            // when we're done processing this request
            bSortNeeded = TRUE;
        }

        // Remove the target entry from our list of targets for this DSRL
        OSAL.eLinkedListRemove(hEntry);
        hEntry = OSAL_INVALID_LINKED_LIST_ENTRY;

        // Iterate the theaters and unlink them from the DSRL
        hEntryToRemove = OSAL.hLinkedListFirst(
            psCurTargetDesc->hTheaterList,
            (void**)&psCurTheaterEntry);
        while (hEntryToRemove != OSAL_INVALID_LINKED_LIST_ENTRY)
        {
            // Remove the entry from the list
            OSAL.eLinkedListRemove(hEntryToRemove);

            // Is this theater tracking this DSRL?
            // Remove it from the DSRL and also remove the
            // dsrl entry from the theater
            bClearDSRLEntryInTheaterEntry(psDSRLDesc, psCurTheaterEntry);

            hEntryToRemove = OSAL.hLinkedListFirst(
                psCurTargetDesc->hTheaterList,
                (void**)&psCurTheaterEntry);
        }

        vReleaseDSRLTargetDesc(psCurTargetDesc);
    }

    if (TRUE == bSortNeeded)
    {
        // It's time for a re-sort
        bSuccess = DSRL_bSort(psDSRLDesc->hDSRL);
    }

    return bSuccess;
}

/*****************************************************************************
*
*   bClearDSRLEntryInTheaterEntry
*
*****************************************************************************/
static BOOLEAN bClearDSRLEntryInTheaterEntry(
    MOVIES_DSRL_DESC_STRUCT *psDSRLDesc,
    MOVIE_THEATER_ENTRY_STRUCT *psTheaterEntry
        )
{
    BOOLEAN bSuccess = FALSE;

    do
    {
        OSAL_RETURN_CODE_ENUM eReturnCode;
        MOVIES_DSRL_REMOVE_TARGET_ITERATOR_STRUCT sIterator;
        MOVIES_THEATER_DSRL_ENTRY_STRUCT *psDSRLEntry;

        // We're looking to see if this entry is linked to
        // this DSRL, and if so what that link is
        sIterator.hTargetDSRL = psDSRLDesc->hDSRL;
        sIterator.hFoundEntry = OSAL_INVALID_LINKED_LIST_ENTRY;

        eReturnCode = OSAL.eLinkedListIterate(
            psTheaterEntry->hDSRLList,
            (OSAL_LL_ITERATOR_HANDLER)bFindTheaterEntryInTargetDesc,
            (void *)&sIterator );
        if ((eReturnCode != OSAL_SUCCESS) && (eReturnCode != OSAL_NO_OBJECTS))
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                MOVIES_MGR_OBJECT_NAME
                ": bClearDSRLEntryInTheaterEntry() Unable to remove DSRL entry from list (%s)",
                OSAL.pacGetReturnCodeName(eReturnCode));
            break;
        }

        // Did we find a corresponding entry?
        if (sIterator.hFoundEntry == OSAL_INVALID_LINKED_LIST_ENTRY)
        {
            // Nope, so we can stop here without error
            bSuccess = TRUE;
            break;
        }

        // Extract the entry we found
        psDSRLEntry = (MOVIES_THEATER_DSRL_ENTRY_STRUCT *)
            OSAL.pvLinkedListThis(sIterator.hFoundEntry);

        // Is this in the DSRL
        if (psDSRLEntry->bInDSRL == TRUE)
        {
            // This theater is in the DSRL -- remove it
            DSRL_vRemoveEntry(psDSRLDesc->hDSRL,
                (DSRL_ENTRY_OBJECT)psTheaterEntry->hTheater);
        }

        // Remove the entry from the list now
        eReturnCode = OSAL.eLinkedListRemove(sIterator.hFoundEntry);
        if (eReturnCode != OSAL_SUCCESS)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                MOVIES_MGR_OBJECT_NAME
                ": vFinalizeDSRLEntry() Unable to remove DSRL entry from list (%s)",
                OSAL.pacGetReturnCodeName(eReturnCode));
            break;
        }

        // Clear the entry
        psDSRLEntry->hEntry = OSAL_INVALID_LINKED_LIST_ENTRY;
        psDSRLEntry->psDesc = NULL;

        // Destroy it
        SMSO_vDestroy((SMS_OBJECT)psDSRLEntry);

        bSuccess = TRUE;
    } while (FALSE);

    return bSuccess;
}

/*****************************************************************************
*
*   bFindTheaterEntryInTargetDesc
*
*****************************************************************************/
static BOOLEAN bFindTheaterEntryInTargetDesc(
    MOVIES_THEATER_DSRL_ENTRY_STRUCT *psDSRLEntry,
    MOVIES_DSRL_REMOVE_TARGET_ITERATOR_STRUCT *psIterator
        )
{
    // Is the the DSRL we're looking for?
    if (psDSRLEntry->psDesc->hDSRL == psIterator->hTargetDSRL)
    {
        psIterator->hFoundEntry = psDSRLEntry->hEntry;

        // Stop looking since we found what we were looking for
        return FALSE;
    }

    // Keep looking
    return TRUE;
}

/*****************************************************************************
*
*   vHandleDeleteList
*
*****************************************************************************/
static void vHandleDeleteList (
    MOVIES_MGR_OBJECT_STRUCT *psObj,
    DSRL_ARG_STRUCT *psDSRLArg
        )
{
    MOVIES_DSRL_DESC_STRUCT *psDSRLDesc;
    
    // Extract the dsrl descriptor
    psDSRLDesc = (MOVIES_DSRL_DESC_STRUCT *)
        DSRL_pvServiceData(psDSRLArg->hDSRL);

    if (psDSRLDesc != NULL)
    {
        BOOLEAN bRelease = TRUE;

        // Remove all the entries
        DSRL_vRemoveAllEntries(psDSRLDesc->hDSRL);

        // Iterate all the theaters to remove this DSRL
        OSAL.eLinkedListIterate(
            psObj->psAppObj->hTheaterList,
            (OSAL_LL_ITERATOR_HANDLER)bRemoveDSRLDescFromTheaterEntry,
            psDSRLDesc);

        // Is this descriptor in the list?
        if (psDSRLDesc->hEntry != OSAL_INVALID_LINKED_LIST_ENTRY)
        {
            OSAL_RETURN_CODE_ENUM eReturnCode;

            // Yes, remove this descriptor from the list
            eReturnCode = OSAL.eLinkedListRemove(psDSRLDesc->hEntry);
            if (eReturnCode != OSAL_SUCCESS)
            {
                // If we can't remove it from the list we 
                // can't free this entry
                bRelease = FALSE;
            }
        }

        if (bRelease == TRUE)
        {
            // Destroy this target
            vReleaseDSRLDesc(psDSRLDesc);
        }
    }

    return;
}

/*****************************************************************************
*
*   bPrepareTheaterEntriesForReplace
*
*****************************************************************************/
static BOOLEAN bPrepareTheaterEntriesForReplace (
    MOVIE_THEATER_ENTRY_STRUCT *psTheaterEntry,
    MOVIES_DSRL_DESC_STRUCT *psDSRLDesc
        )
{
    // Prepare all entries from this theater which match psDSRLDesc
    OSAL.eLinkedListIterate(
        psTheaterEntry->hDSRLList,
        (OSAL_LL_ITERATOR_HANDLER)bPrepareDSRLEntryForReplace,
        psDSRLDesc);

    return TRUE;
}

/*****************************************************************************
*
*   bPostProcessTheaterEntriesForReplace
*
*****************************************************************************/
static BOOLEAN bPostProcessTheaterEntriesForReplace (
    MOVIE_THEATER_ENTRY_STRUCT *psTheaterEntry,
    MOVIES_DSRL_DESC_STRUCT *psDSRLDesc
        )
{
    MOVIES_REPLACE_DSRL_ITERATOR_STRUCT sReplace;

    sReplace.hTheater = psTheaterEntry->hTheater;
    sReplace.psDSRLDesc = psDSRLDesc;
    sReplace.bDSRLUpdated = FALSE;

    // Post process all entries from this theater which match psDSRLDesc
    OSAL.eLinkedListIterate(
        psTheaterEntry->hDSRLList,
        (OSAL_LL_ITERATOR_HANDLER)bPostProcessDSRLEntryForReplace,
        (void *)&sReplace);

    if (sReplace.bDSRLUpdated == TRUE)
    {
        DSRL_vSetState(psDSRLDesc->hDSRL, DSRL_STATE_UPDATING);
    }

    return TRUE;
}

/*****************************************************************************
*
*   bIterateTheaterEntryForDescriptionUpdate
*
*****************************************************************************/
static BOOLEAN bIterateTheaterEntryForDescriptionUpdate (
    MOVIE_THEATER_ENTRY_STRUCT *psTheaterEntry,
    MOVIE_OBJECT hUpdatedMovie
        )
{
    SMSAPI_RETURN_CODE_ENUM eReturnCode;
    MOVIES_DESC_UPDATE_ITERATOR_STRUCT sUpdate;

    sUpdate.bInTheater = FALSE;
    sUpdate.hUpdatedMovie = hUpdatedMovie;

    // Iterate the movies at this theater to see if
    // the updated movie is available here
    eReturnCode = THEATER.eIterateMovies(
        psTheaterEntry->hTheater,
        (THEATER_MOVIES_ITERATOR)bIterateMoviesForDescriptionUpdate,
        &sUpdate);
    if ((eReturnCode != SMSAPI_RETURN_CODE_SUCCESS) &&
        (eReturnCode != SMSAPI_RETURN_CODE_NO_OBJECTS))
    {
        SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
            MOVIES_MGR_OBJECT_NAME
            ": bIterateTheaterEntryForDescriptionUpdate()"
            "Error iterating movies in theater (%u)",
            eReturnCode);

        return FALSE;
    }

    if (sUpdate.bInTheater == TRUE)
    {
        OSAL_RETURN_CODE_ENUM eOsalReturnCode;

        // Iterate the list of DSRLs and replace if necessary
        eOsalReturnCode = OSAL.eLinkedListIterate(
            psTheaterEntry->hDSRLList,
            (OSAL_LL_ITERATOR_HANDLER)
                bIterateDSRLEntryForDescriptionUpdate,
            psTheaterEntry->hTheater );
        if ((eOsalReturnCode != OSAL_SUCCESS) &&
            (eOsalReturnCode != OSAL_NO_OBJECTS))
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                MOVIES_MGR_OBJECT_NAME
                ": bIterateTheaterEntryForDescriptionUpdate()"
                "Unable to iterate theater's DSRL list (%s)",
                OSAL.pacGetReturnCodeName(eOsalReturnCode));

            return FALSE;
        }
    }

    return TRUE;
}

/*****************************************************************************
*
*   bIterateMoviesForDescriptionUpdate
*
*****************************************************************************/
static BOOLEAN bIterateMoviesForDescriptionUpdate (
    THEATER_OBJECT hTheater,
    MOVIE_OBJECT hMovie,
    MOVIES_DESC_UPDATE_ITERATOR_STRUCT *psUpdate
        )
{
    N16 n16CompareMovies;

    // Ask the movie objects to compare themselves
    n16CompareMovies = MOVIE_n16CompareID(
        hMovie, psUpdate->hUpdatedMovie);
    if (n16CompareMovies == 0)
    {
        // We found it so stop here
        psUpdate->bInTheater = TRUE;
        return FALSE;
    }

    return TRUE;
}

/*****************************************************************************
*
*   bIterateDSRLEntryForDescriptionUpdate
*
*****************************************************************************/
static BOOLEAN bIterateDSRLEntryForDescriptionUpdate (
    MOVIES_THEATER_DSRL_ENTRY_STRUCT *psDSRLEntry,
    THEATER_OBJECT hTheater
        )
{
    if (psDSRLEntry->bInDSRL == TRUE)
    {
        DSRL_ADD_REPLACE_RESULT_ENUM eResult;

        eResult = DSRL_eReplaceEntry(
            psDSRLEntry->psDesc->hDSRL,
            (DSRL_ENTRY_OBJECT)hTheater,
            (DSRL_ENTRY_OBJECT)hTheater);
        if (eResult != DSRL_ADD_REPLACE_OK)
        {
            // We can't replace this entry, so it must not meet criteria
            // any longer. Remove the theater from this DSRL
            DSRL_vRemoveEntry(psDSRLEntry->psDesc->hDSRL,
                (DSRL_ENTRY_OBJECT)hTheater);
        }
    }

    return TRUE;
}

/*****************************************************************************
*
*   bRemoveDSRLDescFromTheaterEntry
*
*****************************************************************************/
static BOOLEAN bRemoveDSRLDescFromTheaterEntry (
    MOVIE_THEATER_ENTRY_STRUCT *psTheaterEntry,
    MOVIES_DSRL_DESC_STRUCT *psDSRLDesc
        )
{
    // Remove all entries from this theater which match psDSRLDesc
    OSAL.eLinkedListIterate(
        psTheaterEntry->hDSRLList,
        (OSAL_LL_ITERATOR_HANDLER)bRemoveDSRLEntryFromTheaterEntry,
        psDSRLDesc);

    return TRUE;

}

/*****************************************************************************
*
*   vRemoveAllDSRLEntriesFromTheaterEntry
*
*****************************************************************************/
static void vRemoveAllDSRLEntriesFromTheaterEntry (
    MOVIES_THEATER_DSRL_ENTRY_STRUCT *psDSRLEntry
        )
{
    psDSRLEntry->hEntry = OSAL_INVALID_LINKED_LIST_ENTRY;
    psDSRLEntry->bInDSRL = FALSE;
    psDSRLEntry->psDesc = NULL;

    SMSO_vDestroy((SMS_OBJECT)psDSRLEntry);

    return;
}

/*****************************************************************************
*
*   bRemoveDSRLEntryFromTheaterEntry
*
*****************************************************************************/
static BOOLEAN bRemoveDSRLEntryFromTheaterEntry (
    MOVIES_THEATER_DSRL_ENTRY_STRUCT *psDSRLEntry,
    MOVIES_DSRL_DESC_STRUCT *psDSRLDesc
        )
{
    if (psDSRLEntry->psDesc == psDSRLDesc)
    {
        OSAL.eLinkedListRemove(psDSRLEntry->hEntry);
        vRemoveAllDSRLEntriesFromTheaterEntry(psDSRLEntry);
    }

    return TRUE;
}

/*****************************************************************************
*
*   bPrepareDSRLEntryForReplace
*
*****************************************************************************/
static BOOLEAN bPrepareDSRLEntryForReplace (
    MOVIES_THEATER_DSRL_ENTRY_STRUCT *psDSRLEntry,
    MOVIES_DSRL_DESC_STRUCT *psDSRLDesc
        )
{
    if (psDSRLEntry->psDesc == psDSRLDesc)
    {
        psDSRLEntry->bMayDestroy = TRUE;
    }

    return TRUE;
}

/*****************************************************************************
*
*   bPostProcessDSRLEntryForReplace
*
*****************************************************************************/
static BOOLEAN bPostProcessDSRLEntryForReplace (
    MOVIES_THEATER_DSRL_ENTRY_STRUCT *psDSRLEntry,
    MOVIES_REPLACE_DSRL_ITERATOR_STRUCT *psReplace
        )
{
    // If we have a matching DSRL descriptor and it's
    // okay to destroy this entry then let's do so
    if ((psDSRLEntry->psDesc == psReplace->psDSRLDesc) &&
        (psDSRLEntry->bMayDestroy == TRUE))
    {
        if (psDSRLEntry->bInDSRL == TRUE)
        {
            // We no longer want this in our DSRL
            DSRL_vRemoveEntry(psDSRLEntry->psDesc->hDSRL,
                (DSRL_ENTRY_OBJECT)psReplace->hTheater);

            // This DSRL has been updated
            psReplace->bDSRLUpdated = TRUE;
        }

        OSAL.eLinkedListRemove(psDSRLEntry->hEntry);
        vRemoveAllDSRLEntriesFromTheaterEntry(psDSRLEntry);
    }

    return TRUE;
}

/*****************************************************************************
*
*   psCreateDSRLDesc
*
*****************************************************************************/
static MOVIES_DSRL_DESC_STRUCT *psCreateDSRLDesc (
    MOVIES_MGR_OBJECT_STRUCT *psObj,
    DSRL_ARG_STRUCT *psDSRLArg
        )
{
    MOVIES_DSRL_DESC_STRUCT *psDSRLDesc = NULL;

    do
    {
        BOOLEAN bOk = TRUE;
        DSRL_TARGET_TYPE_ENUM eTargetType;
        OSAL_RETURN_CODE_ENUM eReturnCode;
        size_t tIndex;

        // Verify all targets are locations
        for (tIndex = 0; tIndex < psDSRLArg->tNumTargets; tIndex++)
        {
            // Get the current type
            eTargetType = DSRL_TARGET.eType(psDSRLArg->ahTargetList[tIndex]);
            if (eTargetType != DSRL_TARGET_TYPE_LOCATION)
            {
                // Only allow location targets in here
                bOk = FALSE;
                break;
            }
        }

        // Did we experience any errors?
        if (bOk == FALSE)
        {
            break;
        }

        // Get the target descriptor from the DSRL
        psDSRLDesc = (MOVIES_DSRL_DESC_STRUCT *)
            DSRL_pvServiceData(psDSRLArg->hDSRL);

        if (psDSRLDesc == NULL)
        {
            // Error!
            break;
        }

        // Initialize the entry handle
        psDSRLDesc->hEntry = OSAL_INVALID_LINKED_LIST_ENTRY;

        // Create a list of targets associated with this DSRL
        eReturnCode = OSAL.eLinkedListCreate(
            &psDSRLDesc->hTargetList,
            MOVIES_MGR_OBJECT_NAME":DSRLDesc:TargetList",
            (OSAL_LL_COMPARE_HANDLER)NULL,
            OSAL_LL_OPTION_UNIQUE);

        if (eReturnCode != OSAL_SUCCESS)
        {
            // Error!
            break;
        }

        psDSRLDesc->hDSRL = psDSRLArg->hDSRL;

        // Provide the caller with the
        // target we've created
        return psDSRLDesc;

    } while (FALSE);

    // Destroy this DSRL Desc now
    vReleaseDSRLDesc(psDSRLDesc);

    return NULL;
}

/*****************************************************************************
*
*   vReleaseDSRLDesc
*
*****************************************************************************/
static void vReleaseDSRLDesc (
    MOVIES_DSRL_DESC_STRUCT *psDSRLDesc
        )
{
    OSAL_RETURN_CODE_ENUM eReturnCode;

    if (psDSRLDesc == NULL)
    {
        return;
    }

    printf(MOVIES_MGR_OBJECT_NAME
        ": Attempting to destroy DSRL Desc 0x%p\n",
        psDSRLDesc);

    // Clear the DSRL
    DSRL_vRemoveAllEntries(psDSRLDesc->hDSRL);

    // Do we have any targets?
    if (psDSRLDesc->hTargetList != OSAL_INVALID_OBJECT_HDL)
    {
        // Yes, clean them up too
        eReturnCode =
            OSAL.eLinkedListRemoveAll(
                psDSRLDesc->hTargetList,
                (OSAL_LL_RELEASE_HANDLER)vReleaseDSRLTargetDesc);

        if (eReturnCode == OSAL_SUCCESS)
        {
            eReturnCode =
                OSAL.eLinkedListDelete(psDSRLDesc->hTargetList);

            if (eReturnCode == OSAL_SUCCESS)
            {
                psDSRLDesc->hTargetList = OSAL_INVALID_OBJECT_HDL;
            }
        }
    }

    // Free memory (DSRL & descriptor)
    DSRL_vDestroy(psDSRLDesc->hDSRL);

    return;
}

/*****************************************************************************
*
*   vReleaseDSRLTargetDesc
*
*****************************************************************************/
static void vReleaseDSRLTargetDesc (
    MOVIES_DSRL_TARGET_DESC_STRUCT *psTargetDesc
        )
{
    OSAL_RETURN_CODE_ENUM eReturnCode;

    if (psTargetDesc == NULL)
    {
        return;
    }

    // Free state list
    if (psTargetDesc->hStateList != OSAL_INVALID_OBJECT_HDL)
    {
        eReturnCode =
            OSAL.eLinkedListRemoveAll(
                psTargetDesc->hStateList, NULL);

        if (eReturnCode == OSAL_SUCCESS)
        {
            OSAL.eLinkedListDelete(psTargetDesc->hStateList);
            psTargetDesc->hStateList = OSAL_INVALID_OBJECT_HDL;
        }
    }

    // Free THID list
    if (psTargetDesc->hTheaterList != OSAL_INVALID_OBJECT_HDL)
    {
        eReturnCode =
            OSAL.eLinkedListRemoveAll(
                psTargetDesc->hTheaterList, NULL);

        if (eReturnCode == OSAL_SUCCESS)
        {
            OSAL.eLinkedListDelete(psTargetDesc->hTheaterList);
            psTargetDesc->hTheaterList = OSAL_INVALID_OBJECT_HDL;
        }
    }

    LOCATION.vDestroy(psTargetDesc->hLocation);

    SMSO_vDestroy((SMS_OBJECT)psTargetDesc);

    return;
}

/*****************************************************************************
*
*   bBuildTargetsForDSRL
*
*****************************************************************************/
static BOOLEAN bBuildTargetsForDSRL (
    MOVIES_MGR_OBJECT_STRUCT *psObj,
    MOVIES_DSRL_DESC_STRUCT *psDSRLDesc,
    DSRL_ARG_STRUCT *psDSRLArg
        )
{
    BOOLEAN bSuccess = TRUE;
    size_t tIndex;
    MOVIES_DSRL_TARGET_DESC_STRUCT *psCurTargetDesc;
    OSAL_RETURN_CODE_ENUM eReturnCode;

    // iterate through our target list building the target descriptions
    for (tIndex = 0; tIndex < psDSRLArg->tNumTargets; tIndex++)
    {
        psCurTargetDesc = psBuildTargetDesc(
            psDSRLDesc, psDSRLArg->ahTargetList[tIndex]);

        if (psCurTargetDesc == NULL)
        {
            bSuccess = FALSE;
            break;
        }

        // We own this handle now
        psDSRLArg->ahTargetList[tIndex] = DSRL_TARGET_INVALID_OBJECT;

        // Add the target description to our DSRL
        // Add this to our list of tracked targets for this service
        eReturnCode = OSAL.eLinkedListAdd(
            psDSRLDesc->hTargetList,
            OSAL_INVALID_LINKED_LIST_ENTRY_PTR, 
            psCurTargetDesc);

        if (eReturnCode != OSAL_SUCCESS)
        {
            bSuccess = FALSE;
            break;
        }
    }

    if (bSuccess == FALSE)
    {
        DSRL_vSetState(psDSRLDesc->hDSRL, DSRL_STATE_ERROR);
    }

    return bSuccess;
}

/*****************************************************************************
*
*   bBuildDSRL
*
*****************************************************************************/
static BOOLEAN bBuildDSRL (
    MOVIES_MGR_OBJECT_STRUCT *psObj,
    MOVIES_DSRL_DESC_STRUCT *psDSRLDesc,
    BOOLEAN bMoveToReady,
    BOOLEAN bRefresh
        )
{
    OSAL_RETURN_CODE_ENUM eReturnCode;
    MOVIES_BUILD_DSRL_ITERATOR_STRUCT sBuildDSRL;
    BOOLEAN bSuccess;

    sBuildDSRL.psMgr = psObj;
    sBuildDSRL.bSuccess = FALSE;
    sBuildDSRL.bRefresh = bRefresh;
    sBuildDSRL.bDSRLUpdated = FALSE;
    sBuildDSRL.psDSRLDesc = psDSRLDesc;

    do
    {
        // iterate through our target list, adding theaters for that location
        eReturnCode = OSAL.eLinkedListIterate(
            psDSRLDesc->hTargetList,
            (OSAL_LL_ITERATOR_HANDLER)bLoadTheatersForDSRL,
            (void *)&sBuildDSRL);

        // If we had a non-empty list that didn't error out while iterating
        // move to the error state to indicate the problem
        if ( (eReturnCode != OSAL_NO_OBJECTS) &&
             (eReturnCode != OSAL_SUCCESS) )
        {
            DSRL_vSetState(psDSRLDesc->hDSRL, DSRL_STATE_ERROR);
            break;
        }

        if (eReturnCode != OSAL_NO_OBJECTS)
        {
            // Link the times & theaters if we actually did some work
            bSuccess = bUpdateTheatersWithTimes(psObj);
            if (bSuccess == FALSE)
            {
                break;
            }

            if (sBuildDSRL.bDSRLUpdated == TRUE)
            {
                DSRL_vSetState(psDSRLDesc->hDSRL, DSRL_STATE_UPDATING);
            }
        }
        else
        {
            /* an empty list is not an error */
            sBuildDSRL.bSuccess = TRUE;
        }

        if (bMoveToReady == TRUE)
        {
            DSRL_vSetState(psDSRLDesc->hDSRL, DSRL_STATE_READY);
        }

    } while (FALSE);

    return sBuildDSRL.bSuccess;

}

/*****************************************************************************
*
*   psBuildTargetDesc
*
*****************************************************************************/
static MOVIES_DSRL_TARGET_DESC_STRUCT *psBuildTargetDesc (
    MOVIES_DSRL_DESC_STRUCT *psDSRLDesc,
    LOCATION_OBJECT hLocation
        )
{
    MOVIES_DSRL_TARGET_DESC_STRUCT *psTargetDesc = NULL;
    OSAL_RETURN_CODE_ENUM eReturnCode;

    do
    {
        // Create the target descriptor
        psTargetDesc = (MOVIES_DSRL_TARGET_DESC_STRUCT *)
            SMSO_hCreate(
                MOVIES_MGR_OBJECT_NAME":TargetDesc",
                sizeof(MOVIES_DSRL_TARGET_DESC_STRUCT),
                (SMS_OBJECT)psDSRLDesc,
                FALSE);

        if (psTargetDesc == NULL)
        {
            // Error!
            break;
        }

        // Create a list of states associated with this target
        eReturnCode = OSAL.eLinkedListCreate(
            &psTargetDesc->hStateList,
            MOVIES_MGR_OBJECT_NAME":TargetDesc:StateList",
            (OSAL_LL_COMPARE_HANDLER)NULL,
            OSAL_LL_OPTION_UNIQUE);

        if (eReturnCode != OSAL_SUCCESS)
        {
            // Error!
            break;
        }

        // Create a list of theaters associated with this target
        eReturnCode = OSAL.eLinkedListCreate(
            &psTargetDesc->hTheaterList,
            MOVIES_MGR_OBJECT_NAME":TargetDesc:TheaterList",
            (OSAL_LL_COMPARE_HANDLER)NULL,
            OSAL_LL_OPTION_UNIQUE);

        if (eReturnCode != OSAL_SUCCESS)
        {
            // Error!
            break;
        }

        // Copy the handle
        // This is our own object to use, the DSRL
        // already handled duplicating the target for us
        psTargetDesc->hLocation = hLocation;

        return psTargetDesc;

    } while (FALSE);

    vReleaseDSRLTargetDesc(psTargetDesc);

    return NULL;
}

/*****************************************************************************
*
*   bIterateTargetsForRefresh
*
*****************************************************************************/
static BOOLEAN bIterateTargetsForRefresh (
    MOVIES_DSRL_TARGET_DESC_STRUCT *psTarget,
    void *pvUnused
        )
{
    OSAL.eLinkedListRemoveAll(psTarget->hStateList, NULL);
    OSAL.eLinkedListRemoveAll(psTarget->hTheaterList, NULL);

    return TRUE;
}

/*****************************************************************************
*
*   bLoadTheatersForDSRL
*
*   Pull theaters from the DB into memory for a provided target
*
*****************************************************************************/
static BOOLEAN bLoadTheatersForDSRL (
    MOVIES_DSRL_TARGET_DESC_STRUCT *psTarget,
    MOVIES_BUILD_DSRL_ITERATOR_STRUCT *psBuildDSRL
        )
{
    BOOLEAN bOk = FALSE;

    do
    {
        LOCID_OBJECT hLocId;
        MOVIES_DB_QUERY_BUILD_STATION_STRUCT sBuild;

        hLocId = LOCATION.hLocID(psTarget->hLocation);

        // Is there a valid hLocId associated with it?
        if (hLocId != LOCID_INVALID_OBJECT)
        {
            THEATER_ID tTheaterID = (THEATER_ID)LOCID.tID(hLocId);

            // Specify a search box within the target area
            snprintf( &psBuildDSRL->psMgr->acBuffer[0],
                MOVIES_MAX_SQL_STRING_LENGTH,
                MOVIES_SELECT_THEATERS_BY_ID, tTheaterID);
        }
        // Build a query for a specific region
        else
        {
            OSAL_FIXED_OBJECT hTopRightLat,
                              hTopRightLon,
                              hBottomLeftLat,
                              hBottomLeftLon,
                              hCenterLat,
                              hCenterLon;

            OSAL_FIXED_OBJECT_DATA atFixedData[OSAL_FIXED_OBJECT_SIZE * 4];
            UN8 un8NumFixed = 0;    // Keep track of the fixed objects

            N32 n32FixedTopRightLat,
                n32FixedTopRightLon,
                n32FixedBottomLeftLat,
                n32FixedhBottomLeftLon;

            // Check the lat/lon provided
            hCenterLat = LOCATION.hLat(psTarget->hLocation);
            hCenterLon = LOCATION.hLon(psTarget->hLocation);

            // Is the target location populated?
            if ((hCenterLat == OSAL_FIXED_INVALID_OBJECT) ||
                (hCenterLon == OSAL_FIXED_INVALID_OBJECT))
            {
                // No -- but that's okay...just hold off for now
                bOk = TRUE;
                break;
            }

            // Create the objects.  If this fails, LOCATION_bTopRight
            // will result in an error
            hTopRightLat = OSAL_FIXED.hCreateInMemory(0, 0,
                &atFixedData[OSAL_FIXED_OBJECT_SIZE*(un8NumFixed++)]);
            hTopRightLon = OSAL_FIXED.hCreateInMemory(0, 0,
                &atFixedData[OSAL_FIXED_OBJECT_SIZE*(un8NumFixed++)]);

            // Get the top right lat, lon
            bOk = LOCATION_bTopRight(psTarget->hLocation,
                hTopRightLat, hTopRightLon);

            if (bOk == FALSE)
            {
                break;
            }

            // Create the objects.  If this fails, LOCATION_bBottomLeft
            // will result in an error
            hBottomLeftLat = OSAL_FIXED.hCreateInMemory(0, 0,
                &atFixedData[OSAL_FIXED_OBJECT_SIZE*(un8NumFixed++)]);
            hBottomLeftLon = OSAL_FIXED.hCreateInMemory(0, 0,
                &atFixedData[OSAL_FIXED_OBJECT_SIZE*(un8NumFixed++)]);

            // Get the bottom left lat, lon
            bOk = LOCATION_bBottomLeft(psTarget->hLocation,
                hBottomLeftLat, hBottomLeftLon);

            if (bOk == FALSE)
            {
                break;
            }

            n32FixedTopRightLat =
                OSAL_FIXED.n32ScaledValue(hTopRightLat, LOCATION_BINPOINT);
            n32FixedTopRightLon =
                OSAL_FIXED.n32ScaledValue(hTopRightLon, LOCATION_BINPOINT);

            n32FixedBottomLeftLat =
                OSAL_FIXED.n32ScaledValue(hBottomLeftLat, LOCATION_BINPOINT);
            n32FixedhBottomLeftLon =
                OSAL_FIXED.n32ScaledValue(hBottomLeftLon, LOCATION_BINPOINT);

            // Specify a search box within the target area
            snprintf( &psBuildDSRL->psMgr->acBuffer[0],
                MOVIES_MAX_SQL_STRING_LENGTH,
                MOVIES_SELECT_THEATERS_BY_LOCATION,
                n32FixedBottomLeftLat, n32FixedTopRightLat, // lat min / max
                n32FixedhBottomLeftLon, n32FixedTopRightLon // lon min /max
                );
        }

        // Initialize the result struct
        sBuild.psMgr = psBuildDSRL->psMgr;
        sBuild.psTarget = psTarget;
        sBuild.tCurrentStateID = STATE_INVALID_ID;
        sBuild.pbDSRLUpdated = &psBuildDSRL->bDSRLUpdated;
        sBuild.bRefresh = psBuildDSRL->bRefresh;
        sBuild.psDSRLDesc = psBuildDSRL->psDSRLDesc;

        // Perform the SQL query and process the result
        bOk = SQL_INTERFACE.bQuery(
                psBuildDSRL->psMgr->hSQLRefConnection,
                &psBuildDSRL->psMgr->acBuffer[0],
                (SQL_QUERY_RESULT_HANDLER)bProcessSelectTheatersAndBuildObjects,
                &sBuild ) ;

    } while (FALSE);

    psBuildDSRL->bSuccess = bOk;

    return bOk;
}

/*******************************************************************************
*
*   bProcessSelectTheatersAndBuildObjects
*
*******************************************************************************/
static BOOLEAN bProcessSelectTheatersAndBuildObjects (
    SQL_QUERY_COLUMN_STRUCT *psColumn,
    N32 n32NumberOfColumns,
    MOVIES_DB_QUERY_BUILD_STATION_STRUCT *psBuild
        )
{
    THEATER_ROW_STRUCT sTheaterRow = {
        THEATER_INVALID_ID,        // tID
        STRING_INVALID_OBJECT,     // hName
        STRING_INVALID_OBJECT,     // hAddr
        STRING_INVALID_OBJECT,     // hCity
        STATE_INVALID_ID,          // tStateID
        STRING_INVALID_OBJECT,     // hZIP
        STRING_INVALID_OBJECT,     // hPhone
        { 0, 0 },                  // atLatFixedData
        OSAL_FIXED_INVALID_OBJECT, // hLat
        { 0, 0 },                  // atLonFixedData
        OSAL_FIXED_INVALID_OBJECT, // hLon
        0                          // un16Amenities
    };
    BOOLEAN bTheaterInUse,
            bOk = TRUE,
            bTheaterOwnsStrings = FALSE;
    THEATER_OBJECT hTheater = THEATER_INVALID_OBJECT;
    OSAL_LINKED_LIST_ENTRY hEntry = OSAL_INVALID_LINKED_LIST_ENTRY;
    BOOLEAN bDSRLUpdated = FALSE;

    // Verify input
    if (psBuild == NULL)
    {
        return FALSE;
    }

    // If there is at least one data row and the correct
    // number of columns (just make sure we have enough),
    // then we have good results
    if (n32NumberOfColumns < DB_THEATER_MAX_FIELDS )
    {
        return FALSE;
    }

    // Fetch the state ID and make sure it is in our target's
    // state filter
    sTheaterRow.tStateID = (STATE_ID)
        psColumn[DB_THEATER_STATE].uData.sUN32.un32Data;

    if (sTheaterRow.tStateID != psBuild->tCurrentStateID)
    {
        // Since we haven't seen this state yet during our build, add to the list
        // added this condition here to limit the number of times we search
        // the target's state list for uniquess
        OSAL.eLinkedListAdd(psBuild->psTarget->hStateList,
            OSAL_INVALID_LINKED_LIST_ENTRY_PTR,
            (void *)(size_t)sTheaterRow.tStateID);
        psBuild->tCurrentStateID = sTheaterRow.tStateID;
    }

    // Get the theater ID
    sTheaterRow.tID = psColumn[DB_THEATER_ID].uData.sUN32.un32Data;

    // See if we already have this Theater available from
    // other DSRLs
    bTheaterInUse = bIsTHIDInList(psBuild->psMgr,
        sTheaterRow.tID, &hEntry);

    printf(MOVIES_MGR_OBJECT_NAME": THID: %u in DB.  InUse=%d\n",sTheaterRow.tID, bTheaterInUse);
    if (bTheaterInUse == TRUE)
    {
        // Already present in the main control structure
        // but still needs to be added to DSRL
        bAddTheater(
            psBuild->psMgr, 
            psBuild->psDSRLDesc, psBuild->psTarget,
            THEATER_INVALID_OBJECT, &hEntry,
            psBuild->bRefresh, &bDSRLUpdated);

        if (bDSRLUpdated == TRUE)
        {
            *psBuild->pbDSRLUpdated = TRUE;
        }

        return TRUE;
    }

    // We got past the THID filter -- grab the
    // entire theater record and create a theater object
    bOk = bReadTheaterFromDB(&sTheaterRow, psColumn);

    if (bOk == TRUE)
    {
        // Now, create a theater object and take ownership
        // of the objects in the row
        hTheater = THEATER_hCreate(
            (SMS_OBJECT)psBuild->psMgr->psAppObj, &sTheaterRow,
            sizeof(MOVIE_THEATER_ENTRY_STRUCT));
        if (hTheater != THEATER_INVALID_OBJECT)
        {
            // The theater object owns the STRING objects now
            bTheaterOwnsStrings = TRUE;

            // Add the theater to the DSRL and our main theater list
            bOk = bAddTheater(
                psBuild->psMgr, 
                psBuild->psDSRLDesc, psBuild->psTarget,
                hTheater, &hEntry, psBuild->bRefresh,
                &bDSRLUpdated);

            if (bDSRLUpdated == TRUE)
            {
                *psBuild->pbDSRLUpdated = TRUE;
            }
        }
    }

    if (bOk == FALSE)
    {
        // We need to destroy all strings directly
        if (bTheaterOwnsStrings == FALSE)
        {
            vFreeTheaterRowObjects(&sTheaterRow);
        }
    }

    // FALSE will stop iteration; TRUE will keep going
    return TRUE;
}

/*****************************************************************************
*
*   vFreeTheaterRowObjects
*
*****************************************************************************/
static void vFreeTheaterRowObjects(
    THEATER_ROW_STRUCT *psTheaterRow
        )
{
    if (psTheaterRow->hAddr != STRING_INVALID_OBJECT)
    {
        STRING_vDestroy(psTheaterRow->hAddr);
        psTheaterRow->hAddr = STRING_INVALID_OBJECT;
    }

    if (psTheaterRow->hCity != STRING_INVALID_OBJECT)
    {
        STRING_vDestroy(psTheaterRow->hCity);
        psTheaterRow->hCity = STRING_INVALID_OBJECT;
    }

    if (psTheaterRow->hName != STRING_INVALID_OBJECT)
    {
        STRING_vDestroy(psTheaterRow->hName);
        psTheaterRow->hName = STRING_INVALID_OBJECT;
    }

    if (psTheaterRow->hPhone != STRING_INVALID_OBJECT)
    {
        STRING_vDestroy(psTheaterRow->hPhone);
        psTheaterRow->hPhone = STRING_INVALID_OBJECT;
    }

    if (psTheaterRow->hZIP != STRING_INVALID_OBJECT)
    {
        STRING_vDestroy(psTheaterRow->hZIP);
        psTheaterRow->hZIP = STRING_INVALID_OBJECT;
    }

    return;
}

/*****************************************************************************
*
*   psFindTheaterTimes
*
*****************************************************************************/
static MOVIE_THEATER_TIMES_ENTRY_STRUCT *psFindTheaterTimes (
    MOVIES_APP_OBJECT_STRUCT *psAppObj,
    OSAL_OBJECT_HDL hTimesList,
    THEATER_ID tID
        )
{
    MOVIE_THEATER_TIMES_ENTRY_STRUCT *psResult = 
        (MOVIE_THEATER_TIMES_ENTRY_STRUCT *)NULL;
    OSAL_LINKED_LIST_ENTRY hEntry = OSAL_INVALID_LINKED_LIST_ENTRY;
    OSAL_RETURN_CODE_ENUM eReturnCode;
    MOVIE_THEATER_TIMES_ENTRY_STRUCT sSearch;

    // We're searching for this entry
    sSearch.tTheaterID = tID;

    // Search for this entry now
    eReturnCode = OSAL.eLinkedListSearch(
        hTimesList, &hEntry,
        (void *)&sSearch);
    if (OSAL_SUCCESS == eReturnCode)
    {
        // Extract the entry now
        psResult = (MOVIE_THEATER_TIMES_ENTRY_STRUCT *)
            OSAL.pvLinkedListThis(hEntry);
    }
    else if (OSAL_OBJECT_NOT_FOUND != eReturnCode)
    {
        SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
            MOVIES_MGR_OBJECT_NAME
            ": psFindTheaterTimes() unable to search times list (%s)",
            OSAL.pacGetReturnCodeName(eReturnCode));
    }

    return psResult;
}

/*****************************************************************************
*
*   bAddTheater
*
*****************************************************************************/
static BOOLEAN bAddTheater (
    MOVIES_MGR_OBJECT_STRUCT *psObj,
    MOVIES_DSRL_DESC_STRUCT *psDSRLDesc,
    MOVIES_DSRL_TARGET_DESC_STRUCT *psTarget,
    THEATER_OBJECT hTheater,
    OSAL_LINKED_LIST_ENTRY *phEntry,
    BOOLEAN bRefresh,
    BOOLEAN *pbAddedToDSRL
        )
{
    BOOLEAN bNewEntryCreated = FALSE;
    OSAL_RETURN_CODE_ENUM eReturnCode;
    MOVIE_THEATER_ENTRY_STRUCT *psTheaterEntry = NULL;
    OSAL_LINKED_LIST_ENTRY hNewTheaterLLEntry = OSAL_INVALID_LINKED_LIST_ENTRY;
    OSAL_LINKED_LIST_ENTRY hOldDSRLLLEntry = OSAL_INVALID_LINKED_LIST_ENTRY;
    MOVIES_THEATER_DSRL_ENTRY_STRUCT *psDSRLEntry = NULL;
    DSRL_ADD_REPLACE_RESULT_ENUM eAddResult;

    if (pbAddedToDSRL != NULL)
    {
        *pbAddedToDSRL = FALSE;
    }

    do
    {
        if ( (hTheater == THEATER_INVALID_OBJECT) &&
             (*phEntry == OSAL_INVALID_LINKED_LIST_ENTRY)
            )
        {
            // We need to have at least one of these
            // be valid handles.  If they both are
            // invalid, we can't do much
            break;
        }

        if (*phEntry != OSAL_INVALID_LINKED_LIST_ENTRY)
        {
            // We already have a valid MOVIE_THEATER_ENTRY_STRUCT
            // in the main hTheaterList, we just need to do the DSRL update
            bNewEntryCreated = FALSE;

            psTheaterEntry = (MOVIE_THEATER_ENTRY_STRUCT *)
                OSAL.pvLinkedListThis(*phEntry);
            if (psTheaterEntry == NULL)
            {
                break;
            }

            printf(MOVIES_MGR_OBJECT_NAME
                ": Updating Theater Entry 0x%p with new target 0x%p, %u\n",
                psTheaterEntry, psTarget,
                LOCID.tID(LOCATION.hLocID(THEATER.hLocation(psTheaterEntry->hTheater))));
        }
        else
        {
            // Get theater location information
            LOCATION_OBJECT hLocation = THEATER.hLocation(hTheater);
            LOCID_OBJECT hLocID = LOCATION.hLocID(hLocation);
            THEATER_ID tTheaterID = (THEATER_ID)LOCID.tID(hLocID);

            // Get our entry
            psTheaterEntry = (MOVIE_THEATER_ENTRY_STRUCT *)
                DSRL_ENTRY_pvServiceData((DSRL_ENTRY_OBJECT)hTheater);

            if (psTheaterEntry == NULL)
            {
                break;
            }

            // Set our flag in case we error out and need
            // to free up some memory
            bNewEntryCreated = TRUE;

            // init the entry members
            psTheaterEntry->hTheater = hTheater;
            psTheaterEntry->tID = tTheaterID;

            // Create our list of MOVIES_DSRL_DESC_STRUCT that
            // contain a reference to this theater
            eReturnCode = OSAL.eLinkedListCreate(
                &psTheaterEntry->hDSRLList,
                MOVIES_MGR_OBJECT_NAME":MvTheaterEntry:DSRLList",
                (OSAL_LL_COMPARE_HANDLER)NULL,
                (OSAL_LL_OPTION_LINEAR)
                    );

            if (eReturnCode != OSAL_SUCCESS)
            {
                SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    MOVIES_MGR_OBJECT_NAME
                    ": bAddTheater() Error creating Theater Entry Target List (%s)",
                    OSAL.pacGetReturnCodeName(eReturnCode));
                break;
            }

            // Add the theater entry to our main data store
            eReturnCode = OSAL.eLinkedListAdd(
                psObj->psAppObj->hTheaterList, &psTheaterEntry->hEntry, (void *)psTheaterEntry);

            if (eReturnCode != OSAL_SUCCESS)
            {
                SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    MOVIES_MGR_OBJECT_NAME
                    ": bAddTheater() Error adding Theater Entry to List (%s)",
                    OSAL.pacGetReturnCodeName(eReturnCode));
                break;
            }

            *phEntry = psTheaterEntry->hEntry;

            printf(MOVIES_MGR_OBJECT_NAME
                ": Created new Theater Entry 0x%p with new target 0x%p, %u\n",
                psTheaterEntry, psTarget,
                LOCID.tID(LOCATION.hLocID(THEATER.hLocation(psTheaterEntry->hTheater))));
        }

        // Add the theater entry to our target desc's list of Theater Entries
        eReturnCode = OSAL.eLinkedListAdd(psTarget->hTheaterList,
            &hNewTheaterLLEntry, (void *)psTheaterEntry);

        if ( (eReturnCode != OSAL_SUCCESS) &&
             (eReturnCode != OSAL_ERROR_LIST_ITEM_NOT_UNIQUE) )
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                MOVIES_MGR_OBJECT_NAME
                ": bAddTheater() Can't add theater ID %d to target's THID list (%s)",
                psTheaterEntry->tID, OSAL.pacGetReturnCodeName(eReturnCode));
            break;
        }

        // Is this DSRL Entry already in the list?
        eReturnCode = OSAL.eLinkedListLinearSearch(
            psTheaterEntry->hDSRLList, &hOldDSRLLLEntry,
            (OSAL_LL_COMPARE_HANDLER)n16FindDSRLInTheater,
            psDSRLDesc->hDSRL);

        if (eReturnCode == OSAL_SUCCESS)
        {
            psDSRLEntry = (MOVIES_THEATER_DSRL_ENTRY_STRUCT *)
                OSAL.pvLinkedListThis(hOldDSRLLLEntry);
        }
        else if (eReturnCode == OSAL_OBJECT_NOT_FOUND)
        {
            // Create the DSRL entry for this theater's DSRL list
            psDSRLEntry = (MOVIES_THEATER_DSRL_ENTRY_STRUCT *)
                    SMSO_hCreate(MOVIES_MGR_OBJECT_NAME":DSRLEntry",
                        sizeof(MOVIES_THEATER_DSRL_ENTRY_STRUCT),
                        SMS_INVALID_OBJECT, FALSE);


            if (psDSRLEntry == NULL)
            {
                SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    MOVIES_MGR_OBJECT_NAME
                    ": bAddTheater() Can't create entry for theater's DSRL entry list");
                break;
            }

            // Initialize values
            psDSRLEntry->bInDSRL = FALSE;
            psDSRLEntry->psDesc = psDSRLDesc;
            psDSRLEntry->hEntry = OSAL_INVALID_LINKED_LIST_ENTRY;
        }
        else
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                MOVIES_MGR_OBJECT_NAME
                ": bAddTheater() Can't search theater's DSRL entry list (%s)",
                OSAL.pacGetReturnCodeName(eReturnCode));

            break;
        }

        if (psDSRLEntry == NULL)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                MOVIES_MGR_OBJECT_NAME
                ": bAddTheater() Theater's DSRL entry list corrupted");

            break;
        }

        // If this entry was just created or if it is being re-used we
        // may no longer destroy it
        psDSRLEntry->bMayDestroy = FALSE;

        if (psDSRLEntry->hEntry == OSAL_INVALID_LINKED_LIST_ENTRY)
        {
            // Add the DSRL entry to the in-use list
            eReturnCode = OSAL.eLinkedListAdd(psTheaterEntry->hDSRLList,
                &psDSRLEntry->hEntry, (void *)psDSRLEntry);

            if (eReturnCode != OSAL_SUCCESS)
            {
                printf(MOVIES_MGR_OBJECT_NAME
                    ":Can't add DSRL to theater entry's DSRL list -- %d\n",
                    eReturnCode);
                break;
            }
        }
        else if (bRefresh == FALSE)
        {
            // This entry has already been vetted by the DSRL
            // it is servicing -- so we're done with this one

            // If we're refreshing the DSRL it means we want to
            // try adding this theater again if it isn't already in there
            return TRUE;
        }

        // Check to see if we have stable theater times to link
        // to this theater
        if (psObj->psAppObj->psCurrentTimes->eState
              == MOVIE_OTA_UPDATE_STATE_STABLE)
        {
            MOVIE_THEATER_TIMES_ENTRY_STRUCT *psTheaterTimesEntry;
            
            // Find this entry now
            psTheaterTimesEntry = psFindTheaterTimes(
                psObj->psAppObj, 
                psObj->psAppObj->psCurrentTimes->hTimes, 
                psTheaterEntry->tID);

            // Did we find it?
            if ((MOVIE_THEATER_TIMES_ENTRY_STRUCT *)NULL != psTheaterTimesEntry)
            {
                THEATER_bSetShowTimes(
                    psTheaterEntry->hTheater,
                    psTheaterTimesEntry->hShowTimes,
                    NULL);

                printf(MOVIES_MGR_OBJECT_NAME
                    ": bAddTheater: THEATER_bSetShowTimes called for theater %p and showtimes %p\n",
                    psTheaterEntry->hTheater, psTheaterTimesEntry->hShowTimes);
            }
        }

        if (psDSRLEntry->bInDSRL == FALSE)
        {
            // Add this theater to the DSRL
            eAddResult = DSRL_eAddEntry(psDSRLDesc->hDSRL,
                (DSRL_ENTRY_OBJECT)psTheaterEntry->hTheater);
            if (eAddResult == DSRL_ADD_REPLACE_OK)
            {
                psDSRLEntry->bInDSRL = TRUE;

                if (pbAddedToDSRL != NULL)
                {
                    *pbAddedToDSRL = TRUE;
                }
            }
        }

        return TRUE;

    } while (FALSE);

    // If we created some memory in this function call
    // and failed, clean up after ourselves
    if (bNewEntryCreated == TRUE)
    {
        if (psTheaterEntry->hEntry != OSAL_INVALID_LINKED_LIST_ENTRY)
        {
            OSAL.eLinkedListRemove(psTheaterEntry->hEntry);
        }

        vDestroyTheaterEntry(psTheaterEntry);
    }

    return FALSE;
}


/*****************************************************************************
*
*   vFinalizeDSRLEntry
*
*****************************************************************************/
static void vFinalizeDSRLEntry (
    DSRL_OBJECT hDSRL,
    DSRL_ENTRY_OBJECT hEntry,
    MOVIES_MGR_OBJECT_STRUCT *psObj
        )
{
    do
    {
        OSAL_LINKED_LIST_ENTRY hLLEntry = OSAL_INVALID_LINKED_LIST_ENTRY;
        OSAL_RETURN_CODE_ENUM eReturnCode;
        MOVIES_THEATER_DSRL_ENTRY_STRUCT *psDSRLEntry;
        MOVIE_THEATER_ENTRY_STRUCT *psTheaterEntry;

        // Get the theater entry struct
        psTheaterEntry = (MOVIE_THEATER_ENTRY_STRUCT *)
            DSRL_ENTRY_pvServiceData(hEntry);
        if (psTheaterEntry == NULL)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                MOVIES_MGR_OBJECT_NAME
                ": vFinalizeDSRLEntry() No DSRL_ENTRY service data");
            break;
        }

        // Search for our DSRL in the target list
        eReturnCode =
            OSAL.eLinkedListLinearSearch(
                psTheaterEntry->hDSRLList,
                &hLLEntry,
                (OSAL_LL_COMPARE_HANDLER)n16FindDSRLInTheater,
                (void *)hDSRL);

        if (eReturnCode != OSAL_SUCCESS)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                MOVIES_MGR_OBJECT_NAME
                ": vFinalizeDSRLEntry() Can't find target(%u) for DSRL 0x%p in theater entry 0x%p",
                eReturnCode, hDSRL, psTheaterEntry);
            break;
        }

        // Extract the entry
        psDSRLEntry = (MOVIES_THEATER_DSRL_ENTRY_STRUCT *)
            OSAL.pvLinkedListThis(hLLEntry);

        if (psDSRLEntry == NULL)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                MOVIES_MGR_OBJECT_NAME
                ": vFinalizeDSRLEntry() Theater's DSRL list corrupted");
            break;
        }

        // This entry is no longer in the DSRL
        psDSRLEntry->bInDSRL = FALSE;

    } while (FALSE);

    return;
}

/*****************************************************************************
*
*   vDestroyTheaterEntry
*
*****************************************************************************/
static void vDestroyTheaterEntry (
    MOVIE_THEATER_ENTRY_STRUCT *psTheaterEntry
        )
{
    if (psTheaterEntry != NULL)
    {
        printf(MOVIES_MGR_OBJECT_NAME": Attempting to delete theater entry 0x%p,%d\n",
            psTheaterEntry,
            LOCID.tID(LOCATION.hLocID(THEATER.hLocation(psTheaterEntry->hTheater))));

        if (psTheaterEntry->hDSRLList != OSAL_INVALID_OBJECT_HDL)
        {
            OSAL_RETURN_CODE_ENUM eReturnCode;

            // Empty the target list and free any DSRL entries which are left
            eReturnCode = OSAL.eLinkedListRemoveAll(
                psTheaterEntry->hDSRLList,
                (OSAL_LL_RELEASE_HANDLER)vRemoveAllDSRLEntriesFromTheaterEntry);

            if (eReturnCode == OSAL_SUCCESS)
            {
                eReturnCode = OSAL.eLinkedListDelete(
                    psTheaterEntry->hDSRLList);

                if (eReturnCode == OSAL_SUCCESS)
                {
                    psTheaterEntry->hDSRLList = OSAL_INVALID_OBJECT_HDL;
                }
            }
        }

        if (psTheaterEntry->hTheater != THEATER_INVALID_OBJECT)
        {
            // Destroy the theater and the entry
            THEATER_vDestroy(psTheaterEntry->hTheater);
        }

    }

    return;
}

/*****************************************************************************
*
*   vPruneTheaters
*
*****************************************************************************/
static void vPruneTheaters(
    MOVIES_MGR_OBJECT_STRUCT *psObj
        )
{
    // Iterate through the theater list looking for theaters
    // that aren't being used by a target
    OSAL.eLinkedListIterate(psObj->psAppObj->hTheaterList,
        (OSAL_LL_ITERATOR_HANDLER)bPruneTheatersIterator, NULL);

    return;

}

/*****************************************************************************
*
*   bPruneTheatersIterator
*
*****************************************************************************/
static BOOLEAN bPruneTheatersIterator (
    MOVIE_THEATER_ENTRY_STRUCT *psTheaterEntry,
    void *pvArg
        )
{
    if (psTheaterEntry != NULL)
    {
        UN32 un32NumDSRLs = 0;
        OSAL_RETURN_CODE_ENUM eReturnCode =
            OSAL.eLinkedListItems(
                psTheaterEntry->hDSRLList,  &un32NumDSRLs);

        printf("vPruneTheaters(): un32NumDSRLs == %u\n", un32NumDSRLs);

        if ( (eReturnCode == OSAL_SUCCESS) &&
             (un32NumDSRLs == 0)
           )
        {
            eReturnCode =
                OSAL.eLinkedListRemove(psTheaterEntry->hEntry);

            if (eReturnCode == OSAL_SUCCESS)
            {
                vDestroyTheaterEntry(psTheaterEntry);
            }
        }
    }

    return TRUE;
}

/*****************************************************************************
*
*   bProcessPayload
*
*****************************************************************************/
static BOOLEAN bProcessPayload (
    MOVIES_MGR_OBJECT_STRUCT *psObj,
    OSAL_BUFFER_HDL hPayload
        )
{
    BOOLEAN bSuccess = FALSE;

    do
    {
        MOVIE_MESSAGE_TYPE_ENUM eMsgType;
        MOVIE_VERSION tVersion = 0;

        // Figure out what this payload is
        eMsgType = GsMoviesIntf.eParseMessageHeader(
            psObj->hMoviesInterfaceData, hPayload, &tVersion);

        switch (eMsgType)
        {
            case MOVIE_MESSAGE_TYPE_DESCRIPTION:
            {
                bSuccess =
                    bProcessDescription(psObj, hPayload, tVersion);
            }
            break;

            case MOVIE_MESSAGE_TYPE_TIMES:
            {
                bSuccess =
                    bProcessTimes(psObj, hPayload, tVersion);
            }
            break;

            case MOVIE_MESSAGE_TYPE_RFD:
            {
                // Payload processed by RFD, invalidate it
                hPayload = OSAL_INVALID_BUFFER_HDL;

                // Fall through
            }

            case MOVIE_MESSAGE_TYPE_IGNORE:
            {
                // Not a problem
                bSuccess = TRUE;
            }
            break;

            case MOVIE_MESSAGE_TYPE_ERROR:
            {
                // We don't know how to handle this message
                DATASERVICE_IMPL_vLog(
                    MOVIES_MGR_OBJECT_NAME": MOVIE_MESSAGE_TYPE_ERROR encountered\n");
            }
            break;

            default:
            {
                // We don't know how to handle this message
                DATASERVICE_IMPL_vLog(
                    MOVIES_MGR_OBJECT_NAME": Unknown message type (%u) encountered\n",
                    eMsgType);
            }
            break;
        }

    } while (FALSE);

    // If we still have a payload handle, free it
    if (hPayload != OSAL_INVALID_BUFFER_HDL)
    {
        DATASERVICE_IMPL_bFreeDataPayload(hPayload);
    }

    return bSuccess;
}

/*****************************************************************************
*
*   bProcessDescription
*
*****************************************************************************/
static BOOLEAN bProcessDescription (
    MOVIES_MGR_OBJECT_STRUCT *psObj,
    OSAL_BUFFER_HDL hPayload,
    MOVIE_VERSION tVersion
        )
{
    BOOLEAN bSuccess = TRUE;
    MOVIES_APP_OBJECT_STRUCT *psAppObj;

    // Get the app facing object
    psAppObj = psGetAppFacingObject((MOVIES_SERVICE_OBJECT)psObj);

    if (psAppObj == NULL)
    {
        SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
            MOVIES_MGR_OBJECT_NAME
            ": Couldn't get app facing object");
        
        return FALSE;
    }

    do
    {
        OSAL_RETURN_CODE_ENUM eReturnCode;
        BOOLEAN bCarouselComplete;
        MOVIE_OBJECT hMovie;
        // Initialize this to process payloads into our in-progress list
        OSAL_OBJECT_HDL hMovieList = psObj->psInProgressMovies->hDescriptions;
        BOOLEAN bNotifyDSRLs = FALSE,
                bMoreData,
                bNewDataFound = FALSE;

        puts(MOVIES_MGR_OBJECT_NAME": Starting to process description payload");

        // Is this an AU for the current version?  Only process a payload
        // into the current table if our in-progress table isn't being updated
        // or looking for an update
        if ( (tVersion == psObj->psAppObj->psCurrentMovies->tVersion) &&
             (psObj->psInProgressMovies->eState == MOVIE_OTA_UPDATE_STATE_INITIAL) )
        {
            // This is when we get a desc message for the current version
            // regardless of the state of the current bank and when the in-progess movies bank
            // has been reset (we most likely just did a bank swap)
            // We must have missed this AU, process the payload
            // into the current movies list
            hMovieList = psObj->psAppObj->psCurrentMovies->hDescriptions;
            bNotifyDSRLs = TRUE;
        }
        // Is this an AU for a new version?
        else if (tVersion != psObj->psInProgressMovies->tVersion)
        {
            // We have a brand new version,
            // prepare the in-progress structure
            DATASERVICE_IMPL_vLog(
                MOVIES_MGR_OBJECT_NAME
                ": Processing new version (%d) of description carousel\n",
                tVersion);

            // Clear the in progress list
            eReturnCode = OSAL.eLinkedListRemoveAll(
                psObj->psInProgressMovies->hDescriptions,
                (OSAL_LL_RELEASE_HANDLER)MOVIE_vDestroy);

            if (eReturnCode != OSAL_SUCCESS)
            {
                SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    MOVIES_MGR_OBJECT_NAME
                    ": bProcessDescription() Error clearing in-progress movie list (%s)",
                    OSAL.pacGetReturnCodeName(eReturnCode));
                bSuccess = FALSE;
                break;
            }

            // set the in-progress version and update our state
            psObj->psInProgressMovies->tVersion = tVersion;
            psObj->psInProgressMovies->eState =
                MOVIE_OTA_UPDATE_STATE_IN_PROGRESS;

            // Mark the in-progress times data as "unstable"
            if ( (psObj->psInProgressTimes->eState != MOVIE_OTA_UPDATE_STATE_IN_PROGRESS) &&
                 (psObj->psInProgressTimes->eState != MOVIE_OTA_UPDATE_STATE_COMPLETE) )
            {
                psObj->psInProgressTimes->tVersion = MOVIE_INVALID_VERSION;
                psObj->psInProgressTimes->eState =
                    MOVIE_OTA_UPDATE_STATE_LOOKING_FOR_UPDATE;

                bSuccess =
                    GsMoviesIntf.bResetTimesPayloadTracking(psObj->hMoviesInterfaceData);

                if (bSuccess == FALSE)
                {
                    SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                        MOVIES_MGR_OBJECT_NAME
                        ": sMoviesIntf.bResetTimesPayloadTracking failed");
                    break;
                }
            }
        }

        do
        {
            // Get a movie from the payload
            hMovie = GsMoviesIntf.hParseDescription(
                psObj->hMoviesInterfaceData,
                (SMS_OBJECT)psObj->psAppObj,
                hPayload, &bMoreData);

            if (hMovie == MOVIE_INVALID_OBJECT)
            {
                SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    MOVIES_MGR_OBJECT_NAME
                    ": sMoviesIntf.hParseDescription failed");
                bSuccess = FALSE;
                break;
            }
            else
            {
                // Add our movie to this list
                eReturnCode =
                    OSAL.eLinkedListAdd(hMovieList,
                        OSAL_INVALID_LINKED_LIST_ENTRY_PTR, hMovie);
                if (eReturnCode != OSAL_SUCCESS)
                {
                    // Some sort of error, stop processing
                    SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                        MOVIES_MGR_OBJECT_NAME
                        ": bProcessDescription() Error adding movie to list (%s)",
                        OSAL.pacGetReturnCodeName(eReturnCode));
                    bSuccess = FALSE;
                    break;
                }

                if (bNotifyDSRLs == TRUE)
                {
                    // Tell DSRLs about this change
                    OSAL.eLinkedListIterate(
                        psObj->psAppObj->hTheaterList,
                        (OSAL_LL_ITERATOR_HANDLER)
                            bIterateTheaterEntryForDescriptionUpdate,
                        hMovie);
                }

                // Set our flag to indicate that we found at least one item of interest
                bNewDataFound = TRUE;
            }

        } while (hMovie != MOVIE_INVALID_OBJECT && bMoreData == TRUE);

        if (bSuccess == FALSE)
        {
            // Had an error reading descriptions
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                MOVIES_MGR_OBJECT_NAME
                ": bProcessDescription() Error adding movie descriptions");
            break;
        }

        if ( (bNotifyDSRLs == TRUE) && (bNewDataFound == TRUE) )
        {
            // Update the database with our new descriptions
            bSuccess = bSaveDescToDatabase(psObj);
            if (bSuccess == FALSE)
            {
                break;
            }

            puts(MOVIES_MGR_OBJECT_NAME": New movie descriptions found after"
                "AU was marked as complete");

            // All affected DSRLs have been notified already --
            // time to make them READY again
            vUpdateAllDSRLStates(psObj, DSRL_STATE_READY);
        }

        // Tell the interface we have processed this AU
        bSuccess = GsMoviesIntf.bMarkDescriptionPayloadAsParsed(
            psObj->hMoviesInterfaceData, &bCarouselComplete);

        if (bSuccess == FALSE)
        {
            // Can't mark this AU as complete?  Error out
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                MOVIES_MGR_OBJECT_NAME
                ": GsMoviesIntf.bMarkDescriptionPayloadAsParsed failed");
            break;
        }

        // What's our status now?
        if ( (tVersion == psObj->psInProgressMovies->tVersion) &&
             (bCarouselComplete == TRUE))
        {
            // Looks like our carousel is complete.  So says the AR&R
            psObj->psInProgressMovies->eState =
                MOVIE_OTA_UPDATE_STATE_COMPLETE;
            OSAL.eTimeGet(&psObj->psInProgressMovies->un32UTCsec);

            DATASERVICE_IMPL_vLog(
                MOVIES_MGR_OBJECT_NAME": Description carousel complete (version: %d)\n",
                psObj->psInProgressMovies->tVersion);

            // Move our in-progress control structs (times and movies)
            // to our current control structs if both are ready to transfer
            vAttemptToMoveToCurrent(psObj);

            puts(MOVIES_MGR_OBJECT_NAME": Movie description carousel is complete");
        }

    } while (FALSE);

    if (bSuccess == FALSE)
    {
        // If we left any DSRLs in UPDATING send them to ERROR
        vSendAllUpdatingDSRLsToError(psObj);
    }

    SMSO_vUnlock((SMS_OBJECT)psAppObj);

    return bSuccess;
}

/*****************************************************************************
*
*   bProcessTimes
*
*****************************************************************************/
static BOOLEAN bProcessTimes (
    MOVIES_MGR_OBJECT_STRUCT *psObj,
    OSAL_BUFFER_HDL hPayload,
    MOVIE_VERSION tVersion
        )
{
    BOOLEAN bSuccess = TRUE, bMoreData, bCarouselComplete;
    MOVIES_APP_OBJECT_STRUCT *psAppObj;

    // Get the app facing object
    psAppObj = psGetAppFacingObject((MOVIES_SERVICE_OBJECT)psObj);

    if (psAppObj == NULL)
    {
        SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
            MOVIES_MGR_OBJECT_NAME
            ": Couldn't get app facing object");
        
        return FALSE;
    }

    do
    {
        // Initialize this to process payloads into our in-progress list
        MOVIE_TIMES_CONTROL_STRUCT *psTimes = psObj->psInProgressTimes;
        BOOLEAN bNotifyDSRLs = FALSE, 
                bNewDataFound = FALSE;
        THEATER_ID tTHID;
        OSAL_RETURN_CODE_ENUM eReturnCode;

        // Is this an AU for the current version?  Only process a payload
        // into the current table if our in-progress table isn't being updated
        // or looking for an update
        if ( (tVersion == psObj->psAppObj->psCurrentTimes->tVersion) &&
             (psObj->psInProgressMovies->eState == MOVIE_OTA_UPDATE_STATE_INITIAL) )
        {
            // Process this into our current times
            psTimes = psObj->psAppObj->psCurrentTimes;

            // Set our update DSRL flag to true.  We got new data AFTER
            // we got a complete carousel, so we may need it.  How did we
            // get new data?  Well, we don't process the entire times
            // carousel.  We only process what we need for the DSRL
            // configuration at the time we receive times data.  Our
            // DSRLs change over time, and thus our need for times
            // data changes over time
            bNotifyDSRLs = TRUE;
        }
        else if (tVersion != psObj->psInProgressTimes->tVersion)
        {
            // We have a brand new version
            // prepare the in-progress structure
            printf(
                MOVIES_MGR_OBJECT_NAME
                ": Processing new version (%d) of times carousel\n",
                tVersion);
            DATASERVICE_IMPL_vLog(
                MOVIES_MGR_OBJECT_NAME
                ": Processing new version (%d) of times carousel\n",
                tVersion);

            // Clear the in progress list
            eReturnCode = OSAL.eLinkedListRemoveAll(
                psObj->psInProgressTimes->hTimes,
                (OSAL_LL_RELEASE_HANDLER)vDestroyTheaterTimesEntry);

            if (eReturnCode != OSAL_SUCCESS)
            {
                SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    MOVIES_MGR_OBJECT_NAME
                    ": bProcessTimes() Error clearing in-progress times list (%s)",
                    OSAL.pacGetReturnCodeName(eReturnCode));
                bSuccess = FALSE;
                break;
            }

            // set the in-progress version and state
            psObj->psInProgressTimes->tVersion = tVersion;
            psObj->psInProgressTimes->eState = MOVIE_OTA_UPDATE_STATE_IN_PROGRESS;

            // Mark the in-progress movies as needing an update if it isn't already
            // in the middle of an update
            if ( (psObj->psInProgressMovies->eState != MOVIE_OTA_UPDATE_STATE_IN_PROGRESS) &&
                 (psObj->psInProgressMovies->eState != MOVIE_OTA_UPDATE_STATE_COMPLETE) )
            {
                psObj->psInProgressMovies->tVersion = MOVIE_INVALID_VERSION;
                psObj->psInProgressMovies->eState =
                    MOVIE_OTA_UPDATE_STATE_LOOKING_FOR_UPDATE;

                bSuccess =
                    GsMoviesIntf.bResetDescriptionPayloadTracking(psObj->hMoviesInterfaceData);

                if (bSuccess == FALSE)
                {
                    SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                        MOVIES_MGR_OBJECT_NAME
                        ": GsMoviesIntf.bResetDescriptionPayloadTracking failed");
                    break;
                }
            }
        }

        if ((MOVIE_TIMES_CONTROL_STRUCT *)NULL == psTimes)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                        MOVIES_MGR_OBJECT_NAME
                        ": bProcessTimes: cache corrupted");
            bSuccess = FALSE;
            break;
        }

        do
        {
            // Get a theater times object from the payload
            bSuccess = GsMoviesIntf.bParseTimes(
                psObj->hMoviesInterfaceData, (SMS_OBJECT)psObj->psAppObj,
                hPayload, &psTimes->hScratchTimes, &tTHID, &bMoreData);

            if (FALSE == bSuccess)
            {
                // Some sort of error, stop processing
                SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    MOVIES_MGR_OBJECT_NAME
                    ": GsMoviesIntf.bParseTimes failed");
                break;
            }

            // Add the times to our internal structure.  They'll be
            // linked with theaters once we have a complete set of data
            bSuccess = bAddShowTimes(
                psObj, psTimes->hTimes, psTimes->hScratchTimes, tTHID);

            if (bSuccess == FALSE)
            {
                // Some sort of error, stop processing
                SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    MOVIES_MGR_OBJECT_NAME
                    ": bProcessDescriptions() Error adding times to list");
                break;
            }

            // Set our flag to indicate that we found at least one item of interest
            bNewDataFound = TRUE;

            // Mark the scratch object as null since
            // it is now in use.  The interface will
            // create a new object for us next time around
            psTimes->hScratchTimes = THEATER_TIMES_INVALID_OBJECT;

        } while ((bSuccess == TRUE) && (bMoreData == TRUE));

        if (bSuccess == FALSE)
        {
            // Some sort of issue reading out the times data
            break;
        }

        if ( (bNotifyDSRLs == TRUE) && (bNewDataFound == TRUE) )
        {
            // Update the theaters with the new data.  This function
            // will handle setting DSRLs to updating state if need be
            bSuccess = bUpdateTheatersWithTimes(psObj);
            if (bSuccess == FALSE)
            {
                break;
            }

            // Update the database with our new times
            bSuccess = bSaveTimesToDatabase(psObj);
            if (bSuccess == FALSE)
            {
                break;
            }

            // Update all the DSRLs to READY.  Only the DSRLs that
            // aren't currently READY will move to ready
            vUpdateAllDSRLStates(psObj, DSRL_STATE_READY);
        }

        // Tell the interface we have processed this AU
        bSuccess = GsMoviesIntf.bMarkTimesPayloadAsParsed(
            psObj->hMoviesInterfaceData, &bCarouselComplete);

        if (bSuccess == FALSE)
        {
            // Can't mark this AU as complete?  Error out
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                MOVIES_MGR_OBJECT_NAME
                ": GsMoviesIntf.bMarkTimesPayloadAsParsed failed");
            break;
        }

        if ( (tVersion == psObj->psInProgressTimes->tVersion) &&
             (bCarouselComplete == TRUE))
        {
            // Looks like our carousel is complete.  So says the AR&R
            psObj->psInProgressTimes->eState = MOVIE_OTA_UPDATE_STATE_COMPLETE;
            OSAL.eTimeGet(&psObj->psInProgressTimes->un32UTCsec);

            DATASERVICE_IMPL_vLog(
                MOVIES_MGR_OBJECT_NAME": Times carousel complete (version: %d)\n",
                psObj->psInProgressTimes->tVersion);

            // Move our in-progress control structs (times and movies)
            // to our current control structs if both are ready to transfer
            vAttemptToMoveToCurrent(psObj);

            puts(MOVIES_MGR_OBJECT_NAME": Times Carousel is complete");
        }

    } while (FALSE);

    if (bSuccess == FALSE)
    {
        // If we left any DSRLs in UPDATING send them to ERROR
        vSendAllUpdatingDSRLsToError(psObj);
    }

    SMSO_vUnlock((SMS_OBJECT)psAppObj);

    return bSuccess;
}

/*****************************************************************************
*
*   bAddShowTimes
*
*****************************************************************************/
static BOOLEAN bAddShowTimes(
    MOVIES_MGR_OBJECT_STRUCT *psObj,
    OSAL_OBJECT_HDL hTimesList,
    THEATER_TIMES_OBJECT hShowTimes,
    THEATER_ID tTHID
        )
{
    MOVIE_THEATER_TIMES_ENTRY_STRUCT *psTheaterTimesEntry;
    OSAL_RETURN_CODE_ENUM eReturnCode;

    do
    {
        // Allocate a new times entry
        psTheaterTimesEntry = (MOVIE_THEATER_TIMES_ENTRY_STRUCT *)
            SMSO_hCreate(MOVIES_MGR_OBJECT_NAME":MvTheaterTimesEntry",
                sizeof(MOVIE_THEATER_TIMES_ENTRY_STRUCT),
                (SMS_OBJECT)psObj->psAppObj, FALSE);

        if (psTheaterTimesEntry == NULL)
        {
            break;
        }

        // Populate the theater entry
        psTheaterTimesEntry->tTheaterID = tTHID;
        psTheaterTimesEntry->hShowTimes = hShowTimes;

        // Add it to our times list now
        eReturnCode = OSAL.eLinkedListAdd(
            hTimesList,
            OSAL_INVALID_LINKED_LIST_ENTRY_PTR, 
            (void *)psTheaterTimesEntry);

        if (eReturnCode != OSAL_SUCCESS)
        {
            break;
        }

        return TRUE;

    } while (FALSE);

    return FALSE;
}

/*****************************************************************************
*
*   vAttemptToMoveToCurrent
*
*****************************************************************************/
static void vAttemptToMoveToCurrent (
    MOVIES_MGR_OBJECT_STRUCT *psObj
        )
{
    if ( (psObj->psInProgressMovies->eState ==
            MOVIE_OTA_UPDATE_STATE_COMPLETE) &&
         (psObj->psInProgressTimes->eState ==
             MOVIE_OTA_UPDATE_STATE_COMPLETE))
    {
        vMoveOTADataToComplete(psObj);
        puts(MOVIES_MGR_OBJECT_NAME": OTA data now current");
    }
    else
    {
        puts(MOVIES_MGR_OBJECT_NAME" OTA data not yet current:");
        printf("\tDescriptions %s complete\n",
            (psObj->psInProgressMovies->eState == MOVIE_OTA_UPDATE_STATE_COMPLETE)?"":"not");
        printf("\tTimes %s complete\n",
            (psObj->psInProgressTimes->eState == MOVIE_OTA_UPDATE_STATE_COMPLETE)?"":"not");
    }

    return;
}

/*****************************************************************************
*
*   vMoveOTADataToComplete
*
*****************************************************************************/
static void vMoveOTADataToComplete (
    MOVIES_MGR_OBJECT_STRUCT *psObj
        )
{
    BOOLEAN bSuccess;
    MOVIE_DESCRIPTION_CONTROL_STRUCT *psNewCurrentMovies =
        psObj->psInProgressMovies;
    MOVIE_TIMES_CONTROL_STRUCT *psNewCurrentTimes =
        psObj->psInProgressTimes;

    // Swap the control structure for the movie descriptions
    psObj->psInProgressMovies = psObj->psAppObj->psCurrentMovies;
    psObj->psAppObj->psCurrentMovies = psNewCurrentMovies;

    // Swap the control structure for the theater showtimes
    psObj->psInProgressTimes = psObj->psAppObj->psCurrentTimes;
    psObj->psAppObj->psCurrentTimes = psNewCurrentTimes;

    // Set state, version and time back to initial for both control structures
    psObj->psInProgressTimes->eState = MOVIE_OTA_UPDATE_STATE_INITIAL;
    psObj->psInProgressTimes->tVersion = MOVIE_INVALID_VERSION;
    psObj->psInProgressTimes->un32UTCsec = 0;

    // Set state, version and time back to initial for in progress movies 
    psObj->psInProgressMovies->eState = MOVIE_OTA_UPDATE_STATE_INITIAL;
    psObj->psInProgressMovies->tVersion = MOVIE_INVALID_VERSION;
    psObj->psInProgressMovies->un32UTCsec = 0;

    // Iterate theaters, updating them with the new time data
    bSuccess = bUpdateTheatersWithTimes(psObj);

    if (bSuccess == TRUE)
    {
        // Mark our OTA data as stable
        psObj->psAppObj->psCurrentMovies->eState = MOVIE_OTA_UPDATE_STATE_STABLE;
        psObj->psAppObj->psCurrentTimes->eState = MOVIE_OTA_UPDATE_STATE_STABLE;

        bSuccess = bSaveTimesToDatabase(psObj);
        bSuccess &= bSaveDescToDatabase(psObj);

        if (bSuccess == TRUE)
        {
            vSaveControlDataToDatabase(psObj);
        }

        // All our DSRLs are now in a ready state since we have a complete
        // set of data
        vUpdateAllDSRLStates(psObj, DSRL_STATE_READY);

        DATASERVICE_IMPL_vLog(MOVIES_MGR_OBJECT_NAME": Moved to new OTA data set (Movies Ver:%d Times Ver:%d)\n",
            psObj->psAppObj->psCurrentMovies->tVersion,
            psObj->psAppObj->psCurrentTimes->tVersion);

        puts(MOVIES_MGR_OBJECT_NAME": Moved to new OTA dataset");
    }

    return;
}

/*****************************************************************************
*
*   bUpdateTheatersWithShowTimes
*
*****************************************************************************/
static BOOLEAN bUpdateTheatersWithTimes (
    MOVIES_MGR_OBJECT_STRUCT *psObj
        )
{
    OSAL_RETURN_CODE_ENUM eReturnCode;

    // Link new times with theaters
    eReturnCode = OSAL.eLinkedListIterate(
        psObj->psAppObj->hTheaterList,
        (OSAL_LL_ITERATOR_HANDLER)bLinkTimesWithTheaters, (void*)psObj);

    if ((eReturnCode != OSAL_SUCCESS) && (eReturnCode != OSAL_NO_OBJECTS))
    {
        SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
            MOVIES_MGR_OBJECT_NAME
            ": bUpdateTheatersWithTimes() unable to iterate theater list (%s)",
            OSAL.pacGetReturnCodeName(eReturnCode));

        return FALSE;
    }

    return TRUE;
}

/*****************************************************************************
*
*   bLinkTimesWithTheaters
*
*   This function is a linked list iterator callback and thus always
*   returns true
*
*****************************************************************************/
static BOOLEAN bLinkTimesWithTheaters (
    MOVIE_THEATER_ENTRY_STRUCT *psTheaterEntry,
    MOVIES_MGR_OBJECT_STRUCT *psObj
        )
{
    do
    {
        MOVIE_THEATER_TIMES_ENTRY_STRUCT *psTimesEntry = NULL;
        BOOLEAN bSuccess, bTimesChanged = FALSE;

        if (psTheaterEntry == NULL)
        {
            break;
        }
   
        // Find if we have any showtimes for this theater
        psTimesEntry = psFindTheaterTimes(
            psObj->psAppObj, 
            psObj->psAppObj->psCurrentTimes->hTimes, 
            psTheaterEntry->tID);

        if (psTimesEntry == NULL)
        {
            // No times for this theater, so clear them out
            bSuccess = THEATER_bSetShowTimes(
                psTheaterEntry->hTheater,
                THEATER_TIMES_INVALID_OBJECT,
                &bTimesChanged);

            printf(MOVIES_MGR_OBJECT_NAME
                ": bLinkTimesWithTheaters: THEATER_bSetShowTimes called for theater %p and showtimes %p (changed:%s)\n",
                psTheaterEntry->hTheater, NULL,
                (bTimesChanged == TRUE)?"yes":"no"
                );

            if (bSuccess == FALSE)
            {
                SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    MOVIES_MGR_OBJECT_NAME
                    ": bLinkTimesWithTheaters() THEATER_bSetShowTimes() failed");
                break;
            }
        }
        else
        {
            bSuccess = THEATER_bSetShowTimes(
                psTheaterEntry->hTheater,
                psTimesEntry->hShowTimes,
                &bTimesChanged);

            printf(MOVIES_MGR_OBJECT_NAME
                ": bLinkTimesWithTheaters: THEATER_bSetShowTimes called for theater %p and showtimes %p (changed:%s)\n",
                psTheaterEntry->hTheater, psTimesEntry->hShowTimes,
                (bTimesChanged == TRUE)?"yes":"no"
                );
            if (bSuccess == FALSE)
            {
                SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    MOVIES_MGR_OBJECT_NAME
                    ": bLinkTimesWithTheaters() THEATER_bSetShowTimes() failed");
                break;
            }
        }

        if (bTimesChanged == TRUE)
        {
            OSAL_RETURN_CODE_ENUM eReturnCode;

            // Update this entry on all DSRLs
            eReturnCode = OSAL.eLinkedListIterate(
                psTheaterEntry->hDSRLList,
                (OSAL_LL_ITERATOR_HANDLER)bUpdateTheaterEntryInDSRL,
                (void*)psTheaterEntry);
            if ((eReturnCode != OSAL_SUCCESS) && (eReturnCode != OSAL_NO_OBJECTS))
            {
                SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    MOVIES_MGR_OBJECT_NAME
                    ": bLinkTimesWithTheaters() unable to iterate theater's dsrl list (%s)",
                    OSAL.pacGetReturnCodeName(eReturnCode));
            }
        }

    } while (FALSE);

    return TRUE;
}

/*****************************************************************************
*
*   bUpdateTheaterEntryInDSRL
*
*****************************************************************************/
static BOOLEAN bUpdateTheaterEntryInDSRL (
    MOVIES_THEATER_DSRL_ENTRY_STRUCT *psDSRLEntry,
    MOVIE_THEATER_ENTRY_STRUCT *psTheaterEntry
        )
{
    if (psDSRLEntry == NULL)
    {
        SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
            MOVIES_MGR_OBJECT_NAME
            ": bUpdateTheaterEntryInDSRL() DSRL list corrupted");
    }
    else
    {
        BOOLEAN bDSRLUpdated = FALSE;
        DSRL_ADD_REPLACE_RESULT_ENUM eResult;

        if (psDSRLEntry->bInDSRL == TRUE)
        {
            eResult =
                DSRL_eReplaceEntry(psDSRLEntry->psDesc->hDSRL,
                    (DSRL_ENTRY_OBJECT)psTheaterEntry->hTheater,
                    (DSRL_ENTRY_OBJECT)psTheaterEntry->hTheater);

            if (eResult != DSRL_ADD_REPLACE_OK)
            {
                // We can't replace this entry, so it must not meet criteria
                // any longer. Remove the theater from this DSRL
                DSRL_vRemoveEntry(psDSRLEntry->psDesc->hDSRL,
                    (DSRL_ENTRY_OBJECT)psTheaterEntry->hTheater);
            }

            bDSRLUpdated = TRUE;
        }
        else
        {
            // This entry is not in a DSRL at this time -- try to add it
            eResult = DSRL_eAddEntry(psDSRLEntry->psDesc->hDSRL,
                (DSRL_ENTRY_OBJECT)psTheaterEntry->hTheater);
            if (eResult == DSRL_ADD_REPLACE_OK)
            {
                psDSRLEntry->bInDSRL = TRUE;
                bDSRLUpdated = TRUE;
            }
        }

        if (bDSRLUpdated == TRUE)
        {
            DSRL_STATE_ENUM eCurState;

            eCurState = DSRL.eState(psDSRLEntry->psDesc->hDSRL);
            if (eCurState != DSRL_STATE_ERROR)
            {
                DSRL_vSetState(psDSRLEntry->psDesc->hDSRL, DSRL_STATE_UPDATING);
            }
        }
    }

    return TRUE;
}

/*****************************************************************************
*
*   bIsStateInDSRLs
*
*****************************************************************************/
static BOOLEAN bIsStateInDSRLs (
    MOVIES_MGR_OBJECT_STRUCT *psObj,
    STATE_ID tStateID
        )
{
    MOVIES_TARGET_ITERATOR_STRUCT sIterator;

    // Iterate our targets to find this state ID
    // We stop iterating when we find one occurrence of the state ID
    sIterator.bFound = FALSE;
    sIterator.tStateID = tStateID;

    // Iterate our DSRLs to see if any cares about this state
    OSAL.eLinkedListIterate(
        psObj->hDSRLList,
        (OSAL_LL_ITERATOR_HANDLER)bIsStateInDSRLIterator,
        (void *)&sIterator);

    // The iterator and search function will set this value
    // regardless of the return code
    return sIterator.bFound;
}

/*****************************************************************************
*
*   bIsStateInDSRLIterator
*
*****************************************************************************/
static BOOLEAN bIsStateInDSRLIterator (
    MOVIES_DSRL_DESC_STRUCT *psDSRLDesc,
    MOVIES_TARGET_ITERATOR_STRUCT *psIterator
        )
{
    BOOLEAN bKeepIterating = TRUE;

    // Iterate the targets to see if any care about the state
    OSAL.eLinkedListIterate(
        psDSRLDesc->hTargetList,
        (OSAL_LL_ITERATOR_HANDLER)bIsStateInTargetIterator,
        (void *)psIterator);

    if (psIterator->bFound == TRUE)
    {
        // Stopping looking at the states since we
        // found what we were looking for
        bKeepIterating = FALSE;
    }

    return bKeepIterating;
}

/*****************************************************************************
*
*   bIsStateInTargetIterator
*
*****************************************************************************/
static BOOLEAN bIsStateInTargetIterator (
    MOVIES_DSRL_TARGET_DESC_STRUCT *psTarget,
    MOVIES_TARGET_ITERATOR_STRUCT *psIterator
        )
{
    OSAL_RETURN_CODE_ENUM eReturnCode;
    OSAL_LINKED_LIST_ENTRY hEntry = OSAL_INVALID_LINKED_LIST_ENTRY;

    if (psTarget->hStateList != OSAL_INVALID_OBJECT_HDL &&
        psIterator != NULL)
    {
        // Search the target's states for 
        // the state id in question
        eReturnCode = OSAL.eLinkedListSearch(
            psTarget->hStateList, &hEntry,
            (void *)(size_t)psIterator->tStateID);

        if (eReturnCode == OSAL_SUCCESS)
        {
            psIterator->bFound = TRUE;

            // Stop looking at the states since we
            // found what we were looking for
            return FALSE;
        }
    }

    // Keep looking
    return TRUE;
}

/*****************************************************************************
*
*   bIsTHIDInList
*
*****************************************************************************/
static BOOLEAN bIsTHIDInList (
    MOVIES_MGR_OBJECT_STRUCT *psObj,
    THEATER_ID tTheaterIDToFind,
    OSAL_LINKED_LIST_ENTRY *phEntry
        )
{
    OSAL_RETURN_CODE_ENUM eReturnCode;
    MOVIE_THEATER_ENTRY_STRUCT sEntryToFind;
    BOOLEAN bFound = FALSE;
    OSAL_LINKED_LIST_ENTRY hEntry = OSAL_INVALID_LINKED_LIST_ENTRY;

    if (phEntry == NULL)
    {
        phEntry = &hEntry;
    }

    // create the search entry
    sEntryToFind.tID = tTheaterIDToFind;

    eReturnCode = OSAL.eLinkedListSearch(
        psObj->psAppObj->hTheaterList,
        phEntry, &sEntryToFind);

    if (eReturnCode == OSAL_SUCCESS)
    {
        bFound = TRUE;
    }

    return bFound;
}

/*****************************************************************************
*
*   bCreatePersistDBTables
*
*****************************************************************************/
static BOOLEAN bCreatePersistDBTables (
    SQL_INTERFACE_OBJECT hSQLConnection,
    void *pvArg
        )
{
    BOOLEAN bSuccess = FALSE, bTransactionStarted = FALSE;
    DSI tDSI = (DSI)(size_t)pvArg;
    char acBuffer[MOVIES_MAX_SQL_STRING_LENGTH];

    do
    {
        bTransactionStarted = SQL_INTERFACE.bStartTransaction(hSQLConnection);

        if (bTransactionStarted == FALSE)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                MOVIES_MGR_OBJECT_NAME
                ": failed to start transaction to recreate persistent storage");
            break;
        }

        bSuccess = SQL_INTERFACE.bExecuteCommand(hSQLConnection,
            MOVIES_CREATE_CONTROL_TABLE);

        if (bSuccess == FALSE)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                MOVIES_MGR_OBJECT_NAME
                ": failed to create movies control table");
            break;
        }

        bSuccess = SQL_INTERFACE.bExecuteCommand(hSQLConnection,
            MOVIES_INSERT_DEFAULT_CONTROL_ROW);

        if (bSuccess == FALSE)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                MOVIES_MGR_OBJECT_NAME
                ": failed to set initial control table versions");
            break;
        }

        bSuccess = SQL_INTERFACE.bExecuteCommand(hSQLConnection,
            MOVIES_CREATE_MOVIES_TABLE);

        if (bSuccess == FALSE)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                MOVIES_MGR_OBJECT_NAME
                ": failed to create movies table");
            break;
        }

        bSuccess = SQL_INTERFACE.bExecuteCommand(hSQLConnection,
            MOVIES_CREATE_TIMES_TABLE);

        if (bSuccess == FALSE)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                MOVIES_MGR_OBJECT_NAME
                ": failed to create times table");
            break;
        }

        bSuccess = SQL_INTERFACE.bExecuteCommand(hSQLConnection,
            MOVIES_CREATE_VERSION_TABLE);

        if (bSuccess == FALSE)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                MOVIES_MGR_OBJECT_NAME
                ": failed to create db_version in persistent storage");
            break;
        }

        // Build the version string
        snprintf(&acBuffer[0], sizeof(acBuffer),
            MOVIES_INSERT_VERSION_ROW,
            MOVIES_DATABASE_FILE_VERSION, tDSI);

        bSuccess = SQL_INTERFACE.bExecuteCommand(hSQLConnection, &acBuffer[0]);

        if (bSuccess == FALSE)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                MOVIES_MGR_OBJECT_NAME
                ": failed to set db version in persistant storage");
            break;
        }
    }
    while (FALSE);

    if (bTransactionStarted == TRUE)
    {
        BOOLEAN bTransactionEnded;

        bTransactionEnded = SQL_INTERFACE.bEndTransaction(hSQLConnection);

        if (bTransactionEnded == FALSE)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                MOVIES_MGR_OBJECT_NAME
                ": failed to end transaction");

            bSuccess = FALSE;
        }
    }

    return bSuccess;
}

/*****************************************************************************
*
*   n32ExtractDataVersion
*
*****************************************************************************/
static N32 n32ExtractDataVersion (
    SQL_INTERFACE_OBJECT hConnection,
    void *pvArg
        )
{
    N32 n32Result = DB_UTIL_DB_UNDER_CONSTRUCTION_VER;
    MOVIE_VERSION tBaselineVersion;
    BOOLEAN bVerified;

    if (hConnection == SQL_INTERFACE_INVALID_OBJECT)
    {
        return DB_UTIL_DB_UNDER_CONSTRUCTION_VER;
    }

    // Get the version field
    bVerified = bVerifyDBVersion(
        hConnection, &tBaselineVersion);
    if (bVerified == TRUE)
    {
        n32Result = (N32)tBaselineVersion;
    }

    return n32Result;
}

/*****************************************************************************
*
*   bVerifyDBVersion
*
*****************************************************************************/
static BOOLEAN bVerifyDBVersion (
    SQL_INTERFACE_OBJECT hSQLConnection,
    MOVIE_VERSION *ptBaselineVersion
        )
{
    MOVIES_DB_QUERY_RESULT_STRUCT sQueryResult;
    BOOLEAN bOk;

    OSAL.bMemSet(&sQueryResult, 0, sizeof(MOVIES_DB_QUERY_RESULT_STRUCT));

    // Perform the SQL query and process the result
    // (it will provide us with a data row)
    bOk = SQL_INTERFACE.bQuery(
            hSQLConnection, MOVIES_SELECT_DB_VERSION,
            bProcessSelectDBVersionResult, &sQueryResult ) ;

    if (bOk == TRUE)
    {
        if (sQueryResult.bResultantRows == TRUE)
        {
            MOVIES_VERSION_ROW_STRUCT *psVersionRow =
                &sQueryResult.uDbRow.sVersion;

            // Verify that the db schema version matched
            if ( (psVersionRow->un8DBVer == MOVIES_DATABASE_FILE_VERSION) &&
                 (psVersionRow->tDSI == GsMoviesIntf.tDSI) )
            {
                // Version match!
                if (ptBaselineVersion != NULL)
                {
                    // Save the baseline version if requested
                    *ptBaselineVersion = psVersionRow->tBaselineVersion;
                }

                return TRUE;
            }
            else
            {
                // Version mismatch!
                return FALSE;
            }
        }
    }

    return FALSE;
}

/*******************************************************************************
*
*   vProcessSelectDBVersionResult
*
*******************************************************************************/
static BOOLEAN bProcessSelectDBVersionResult (
    SQL_QUERY_COLUMN_STRUCT *psColumns,
    N32 n32NumberOfColumns,
    void *pvArg
       )
{
    SQL_QUERY_COLUMN_STRUCT *psCurrentCol;
    DB_VERSION_FIELDS_ENUM eCurrentField = (DB_VERSION_FIELDS_ENUM)0;
    MOVIES_DB_QUERY_RESULT_STRUCT *psQueryResult =
        (MOVIES_DB_QUERY_RESULT_STRUCT *)pvArg;

    if ( psQueryResult == NULL )
    {
        return FALSE;
    }

    do
    {
        psQueryResult->bResultantRows = TRUE;

        // Grab the current column data
        psCurrentCol = &psColumns[(UN8)eCurrentField];

        // Decode the current field based on it's type
        switch (eCurrentField)
        {
            case DB_VERSION_FIELD_DB_VER:
            {
                psQueryResult->uDbRow.sVersion.un8DBVer =
                    (UN8)psCurrentCol->uData.sUN32.un32Data;
            }
            break;

            case DB_VERSION_FIELD_DSI:
            {
                psQueryResult->uDbRow.sVersion.tDSI =
                    (DSI)psCurrentCol->uData.sUN32.un32Data;
            }
            break;

            case DB_VERSION_BASELINE:
            {
                psQueryResult->uDbRow.sVersion.tBaselineVersion =
                    (MOVIE_VERSION)psCurrentCol->uData.sUN32.un32Data;
            }
            break;

            default: // Shouldn't happen
            break;
        }

        ++eCurrentField;

    } while ( ( eCurrentField < DB_VERSION_MAX_FIELDS )   &&
              ( (N32)eCurrentField < n32NumberOfColumns ) );

    // Return false since we don't need to iterate through the table
    // should only be one row
    return FALSE;
}

/*******************************************************************************
*
*   bLoadRatingsTable
*
*******************************************************************************/
static BOOLEAN bLoadRatingsTable (
    MOVIES_MGR_OBJECT_STRUCT *psObj
        )
{
    BOOLEAN bSuccess = FALSE;
    MOVIES_APP_OBJECT_STRUCT *psAppObj;

    psAppObj = psGetAppFacingObject((MOVIES_SERVICE_OBJECT)psObj);

    if (psAppObj != NULL)
    {
        // Perform the SQL query and process the result
        // (it will provide us with a data row)
        bSuccess =  SQL_INTERFACE.bQuery(
            psObj->hSQLRefConnection,
            MOVIES_SELECT_DB_RATINGS,
            (SQL_QUERY_RESULT_HANDLER)bProcessSelectRatingsTableResult,
            (void *)psObj) ;

        SMSO_vUnlock((SMS_OBJECT)psAppObj);

    }

    return bSuccess;
}

/*******************************************************************************
*
*   bProcessSelectRatingsTableResult
*
*******************************************************************************/
static BOOLEAN bProcessSelectRatingsTableResult (
    SQL_QUERY_COLUMN_STRUCT *psColumns,
    N32 n32NumberOfColumns,
    void *pvArg
       )
{
    SQL_QUERY_COLUMN_STRUCT *psCurrentCol;
    DB_RATINGS_FIELDS_ENUM eCurrentField = (DB_RATINGS_FIELDS_ENUM)0;
    MOVIES_MGR_OBJECT_STRUCT *psObj =
        (MOVIES_MGR_OBJECT_STRUCT *)pvArg;
    MOVIE_RATINGS_ROW_STRUCT sRatingsRow;
    BOOLEAN bOk = TRUE;

    if (psObj == NULL)
    {
        return FALSE;
    }

    do
    {

        // Blank out our row
        sRatingsRow.tSysCode = 0;
        sRatingsRow.hSysName = STRING_INVALID_OBJECT;
        sRatingsRow.tRatingCode = 0;
        sRatingsRow.hRatingText = STRING_INVALID_OBJECT;

        // If there is not the correct number of columns,
        // then we have an error
        if (n32NumberOfColumns != DB_RATINGS_MAX_FIELDS )
        {
            break;
        }

        do
        {
            // Grab the current column data
            psCurrentCol = &psColumns[(UN8)eCurrentField];

            // Decode the current field based on it's type
            switch (eCurrentField)
            {
                case DB_RATINGS_SCODE:
                {
                    sRatingsRow.tSysCode =
                        (MOVIE_RATING_SYS_CODE)psCurrentCol->uData.sUN32.un32Data;
                }
                break;

                case DB_RATINGS_STEXT:
                {
                    sRatingsRow.hSysName = STRING_hCreate(
                        (SMS_OBJECT)psObj->psAppObj,
                        (const char *)psCurrentCol->uData.sCString.pcData,
                        strlen((const char *)psCurrentCol->uData.sCString.pcData),0);
                    if (sRatingsRow.hSysName == STRING_INVALID_OBJECT)
                    {
                        SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                            MOVIES_MGR_OBJECT_NAME
                            ": bProcessSelectRatingsTableResult() Error getting system name from DB");
                        bOk = FALSE;
                    }
                }
                break;

                case DB_RATINGS_RCODE:
                {
                    sRatingsRow.tRatingCode =
                        (MOVIE_RATING_CODE)psCurrentCol->uData.sUN32.un32Data;
                }
                break;

                case DB_RATINGS_RTEXT:
                {
                    sRatingsRow.hRatingText = STRING_hCreate(
                        (SMS_OBJECT)psObj->psAppObj,
                        (const char *)psCurrentCol->uData.sCString.pcData,
                        strlen((const char *)psCurrentCol->uData.sCString.pcData),0);
                    if (sRatingsRow.hRatingText == STRING_INVALID_OBJECT)
                    {
                        SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                            MOVIES_MGR_OBJECT_NAME
                            ": bProcessSelectRatingsTableResult() Error getting rating text from DB");
                        bOk = FALSE;
                    }
                }
                break;

                default: // Shouldn't happen
                break;
            }

        } while ( (++eCurrentField < DB_RATINGS_MAX_FIELDS) && (bOk == TRUE));

        // Attempt to add the ratings to our list
        bOk = bAddRatingsEntry(psObj, &sRatingsRow);

        if (bOk == FALSE)
        {
            // Failed to add the entry, so clean up our string objects
            STRING_vDestroy(sRatingsRow.hSysName);
            STRING_vDestroy(sRatingsRow.hRatingText);
        }

    } while (FALSE);

    return TRUE;
}

/*******************************************************************************
*
*   bLoadPersistentData
*
*******************************************************************************/
static BOOLEAN bLoadPersistentData (
    MOVIES_MGR_OBJECT_STRUCT *psObj
        )
{

    BOOLEAN bSuccess = FALSE;
    MOVIES_APP_OBJECT_STRUCT *psAppObj;

    psAppObj = psGetAppFacingObject((MOVIES_SERVICE_OBJECT)psObj);
    if (psAppObj == NULL)
    {
        return FALSE;
    }

    do
    {
        OSAL_RETURN_CODE_ENUM eReturnCode;
        UN32 un32UTCsec;
        int iReturn;
        eReturnCode = OSAL.eTimeGet(&un32UTCsec);

        if (eReturnCode != OSAL_SUCCESS)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                MOVIES_MGR_OBJECT_NAME
                ": bLoadPersistentData() Error getting time: %s",
                    OSAL.pacGetReturnCodeName(eReturnCode));
            break;
        }

        // We don't want any data that is older than our timeout
        un32UTCsec -= MOVIES_CONTROL_TABLE_TIMEOUT_SECS;

        // Build our string to fetch our control variables
        iReturn = snprintf( &psObj->acBuffer[0], sizeof(psObj->acBuffer),
            MOVIES_FETCH_CONTROL_ROW,
            un32UTCsec, un32UTCsec);

        if (iReturn <= 0)
        {
            break;
        }

        // Fetch the control row from the table
        bSuccess =  SQL_INTERFACE.bQuery(
            psObj->hSQLPersistConnection,
            &psObj->acBuffer[0],
            (SQL_QUERY_RESULT_HANDLER)bProcessControlTableResult,
            (void *)psAppObj) ;

        if (bSuccess == FALSE)
        {
            break;
        }

        // If either set of data is stale, stop where we are at
        if ( (psAppObj->psCurrentTimes->tVersion == MOVIE_INVALID_VERSION) ||
             (psAppObj->psCurrentMovies->tVersion == MOVIE_INVALID_VERSION) )
        {
            // We didn't read any new versions from the db, so break
            // but don't indicate an error
            bSuccess = TRUE;
            break;
        }

        // Process the movie descriptions
        bSuccess =  SQL_INTERFACE.bQuery(
            psObj->hSQLPersistConnection,
            MOVIES_FETCH_DESCRIPTIONS,
            (SQL_QUERY_RESULT_HANDLER)bProcessMovieDescFromDB,
            (void *)psObj);

        if (bSuccess == FALSE)
        {
            break;
        }

        // Indicate we should fetch new data when we get it in case
        // for some crazy reason the stored version matches the current
        // OTA version
        psObj->psInProgressMovies->eState =
                MOVIE_OTA_UPDATE_STATE_LOOKING_FOR_UPDATE;

        // Process the movie times
        bSuccess =  SQL_INTERFACE.bQuery(
            psObj->hSQLPersistConnection,
            MOVIES_FETCH_TIMES,
            (SQL_QUERY_RESULT_HANDLER)bProcessMovieTimesFromDB,
            (void *)psObj) ;

        if (bSuccess == FALSE)
        {
            break;
        }

        // Indicate we should fetch new data when we get it in case
        // for some crazy reason the stored version matches the current
        // OTA version
        psObj->psInProgressTimes->eState =
            MOVIE_OTA_UPDATE_STATE_LOOKING_FOR_UPDATE;

        // Start the ageout timer
        vStartAgeoutTimer(psObj);
        
    } while (FALSE);

    if (bSuccess == FALSE)
    {
        // We failed so reset the current times and movies
        // Clear the current times and movies structures
        psObj->psAppObj->psCurrentTimes->eState = MOVIE_OTA_UPDATE_STATE_INITIAL;
        psObj->psAppObj->psCurrentTimes->tVersion = MOVIE_INVALID_VERSION;
        psObj->psAppObj->psCurrentTimes->un32UTCsec = 0;

        psObj->psAppObj->psCurrentMovies->eState = MOVIE_OTA_UPDATE_STATE_INITIAL;
        psObj->psAppObj->psCurrentMovies->tVersion = MOVIE_INVALID_VERSION;
        psObj->psAppObj->psCurrentMovies->un32UTCsec = 0;
    }

    SMSO_vUnlock((SMS_OBJECT)psAppObj);

    return bSuccess;
}

/*****************************************************************************
*
*   bProcessControlTableResult
*
*****************************************************************************/
static BOOLEAN bProcessControlTableResult (
    SQL_QUERY_COLUMN_STRUCT *psColumns,
    N32 n32NumberOfColumns,
    void *pvArg
       )
{
    SQL_QUERY_COLUMN_STRUCT *psCurrentCol;
    DB_CONTROL_FIELDS_ENUM eCurrentField = (DB_CONTROL_FIELDS_ENUM)0;
    MOVIES_APP_OBJECT_STRUCT *psAppObj =
        (MOVIES_APP_OBJECT_STRUCT *)pvArg;

    if (psAppObj == NULL)
    {
        return FALSE;
    }

    do
    {
        // If there is not the correct number of columns,
        // then we have an error
        if (n32NumberOfColumns != DB_CONTROL_MAX_FIELDS )
        {
            break;
        }

        do
        {
            // Grab the current column data
            psCurrentCol = &psColumns[(UN8)eCurrentField];

            // Decode the current field based on it's type
            switch (eCurrentField)
            {
                case DB_CONTROL_TIMES_VER:
                {
                    psAppObj->psCurrentTimes->tVersion =
                        (MOVIE_VERSION)psCurrentCol->uData.sUN32.un32Data;
                }
                break;

                case DB_CONTROL_TIMES_TIMESTAMP:
                {
                    psAppObj->psCurrentTimes->un32UTCsec =
                        psCurrentCol->uData.sUN32.un32Data;
                }
                break;

                case DB_CONTROL_DESC_VER:
                {
                    psAppObj->psCurrentMovies->tVersion =
                        (MOVIE_VERSION)psCurrentCol->uData.sUN32.un32Data;
                }
                break;

                case DB_CONTROL_DESC_TIMESTAMP:
                {
                    psAppObj->psCurrentMovies->un32UTCsec =
                        psCurrentCol->uData.sUN32.un32Data;
                }
                break;

                default: // Shouldn't happen
                break;
            }

        } while ( ++eCurrentField < DB_CONTROL_MAX_FIELDS);

        // Set both structures as ready
        psAppObj->psCurrentMovies->eState = MOVIE_OTA_UPDATE_STATE_STABLE;
        psAppObj->psCurrentTimes->eState = MOVIE_OTA_UPDATE_STATE_STABLE;

    } while (FALSE);


    return TRUE;
}

/*****************************************************************************
*
*   bProcessMovieDescFromDB
*
*****************************************************************************/
static BOOLEAN bProcessMovieDescFromDB (
    SQL_QUERY_COLUMN_STRUCT *psColumns,
    N32 n32NumberOfColumns,
    void *pvArg
       )
{
    BOOLEAN bOk = TRUE;
    OSAL_RETURN_CODE_ENUM eReturnCode;
    SQL_QUERY_COLUMN_STRUCT *psCurrentCol;
    DB_MOVIE_DESC_FIELDS_ENUM eCurrentField = (DB_MOVIE_DESC_FIELDS_ENUM)0;
    MOVIES_MGR_OBJECT_STRUCT *psObj =
        (MOVIES_MGR_OBJECT_STRUCT *)pvArg;
    MOVIE_OBJECT hMovie = MOVIE_INVALID_OBJECT;
    MOVIE_ID tID = 0;
    UN8 un8RunTime = 0;
    STRING_OBJECT hActors = STRING_INVALID_OBJECT;
    MOVIE_MULTI_LANG_FIELD_STRUCT sName;
    MOVIE_MULTI_LANG_FIELD_STRUCT sSynopsis;
    size_t tNumRatings = 0, tLen;

    if (psObj == NULL)
    {
        return FALSE;
    }

    // Initialize local structures
    OSAL.bMemSet((void*)&sName, 0, sizeof(sName));
    OSAL.bMemSet((void*)&sSynopsis, 0, sizeof(sSynopsis));

    do
    {
        // If there is not the correct number of columns,
        // then we have an error
        if (n32NumberOfColumns != DB_MOVIE_DESC_MAX_FIELDS )
        {
            break;
        }

        do
        {
            // Grab the current column data
            psCurrentCol = &psColumns[(UN8)eCurrentField];

            // Decode the current field based on it's type
            switch (eCurrentField)
            {
                case DB_MOVIE_DESC_MVID:
                {
                    tID =
                        (MOVIE_ID)psCurrentCol->uData.sUN32.un32Data;
                }
                break;

                case DB_MOVIE_DESC_ENG_NAME:
                case DB_MOVIE_DESC_FR_NAME:
                case DB_MOVIE_DESC_SPN_NAME:
                case DB_MOVIE_DESC_ENG_SYNOPSIS:
                case DB_MOVIE_DESC_FR_SYNOPSIS:
                case DB_MOVIE_DESC_SPN_SYNOPSIS:
                {
                    STRING_OBJECT *phStr;

                    if ( DB_MOVIE_DESC_ENG_NAME == eCurrentField )
                    {
                        phStr = &sName.hEnglish;
                    }
                    else if ( DB_MOVIE_DESC_FR_NAME == eCurrentField )
                    {
                        phStr = &sName.hFrench;
                    }
                    else if ( DB_MOVIE_DESC_SPN_NAME == eCurrentField )
                    {
                        phStr = &sName.hSpanish;
                    }
                    else if ( DB_MOVIE_DESC_ENG_SYNOPSIS == eCurrentField )
                    {
                        phStr = &sSynopsis.hEnglish;
                    }
                    else if ( DB_MOVIE_DESC_FR_SYNOPSIS == eCurrentField )
                    {
                        phStr = &sSynopsis.hFrench;
                    }
                    else if ( DB_MOVIE_DESC_SPN_SYNOPSIS == eCurrentField )
                    {
                        phStr = &sSynopsis.hSpanish;
                    }

                    // How long is this string?
                    tLen = strlen((const char *)psCurrentCol->uData.sCString.pcData);

                    // Only create a string object for valid names
                    if (tLen == 0)
                    {
                        *phStr = STRING_INVALID_OBJECT;
                    }
                    else
                    {
                        // Create the string
                        *phStr = STRING_hCreate(
                            (SMS_OBJECT)psObj->psAppObj,
                            (const char *)psCurrentCol->uData.sCString.pcData,
                            tLen, 0);

                        // Did that work?
                        if (*phStr == STRING_INVALID_OBJECT)
                        {
                            bOk = FALSE;
                        }
                    }
                }
                break;

                case DB_MOVIE_DESC_RUNTIME:
                {
                    un8RunTime =
                        (UN8)psCurrentCol->uData.sUN32.un32Data;
                }
                break;

                case DB_MOVIE_DESC_ACTORS:
                {
                    hActors = STRING_hCreate(
                        (SMS_OBJECT)psObj->psAppObj,
                        (const char *)psCurrentCol->uData.sCString.pcData,
                        strlen((const char *)psCurrentCol->uData.sCString.pcData),0);
                    if (hActors == STRING_INVALID_OBJECT)
                    {
                        bOk = FALSE;
                    }
                }
                break;

                case DB_MOVIE_DESC_NUM_RATINGS:
                {
                    tNumRatings = 
                        (size_t)psCurrentCol->uData.sUN32.un32Data;
                }
                break;

                case DB_MOVIE_DESC_RATINGS:
                {
                    size_t tIndex, tExpectedSize = 
                        sizeof(MOVIE_RATING_STRUCT) * tNumRatings;

                    if (tExpectedSize == psCurrentCol->uData.sBLOB.tSize)
                    {
                        OSAL.bMemCpy(&psObj->asRatings[0],
                            psCurrentCol->uData.sBLOB.pvData,
                            tExpectedSize);
                    }
                    else
                    {
                        // Clear the ratings
                        tNumRatings = 0;
                    }

                    // Get the ratings data for each of these now
                    for (tIndex = 0; tIndex < tNumRatings; tIndex++)
                    {
                        MOVIES_MGR_bGetRatingInfo(
                            (MOVIES_SERVICE_OBJECT)psObj,
                            psObj->asRatings[tIndex].tRatingSys,
                            psObj->asRatings[tIndex].tRating,
                            &psObj->asRatings[tIndex]);
                    }
                }
                break;

                default: // Shouldn't happen
                break;
            }

        } while ((++eCurrentField < DB_MOVIE_DESC_MAX_FIELDS) && (bOk == TRUE));

        if (bOk == FALSE)
        {
            break;
        }

        // Create this movie now
        hMovie = MOVIE_hCreate(
            (SMS_OBJECT)psObj->psAppObj,
            tID, &sName, un8RunTime,
            hActors, &sSynopsis, 
            tNumRatings, 
            &psObj->asRatings[0]);

        if (hMovie == MOVIE_INVALID_OBJECT)
        {
            break;
        }

        // Add our movie to this list
        eReturnCode =
            OSAL.eLinkedListAdd(
                psObj->psAppObj->psCurrentMovies->hDescriptions,
                OSAL_INVALID_LINKED_LIST_ENTRY_PTR, hMovie);
        if (eReturnCode != OSAL_SUCCESS)
        {
            // Some sort of error, stop processing
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                MOVIES_MGR_OBJECT_NAME
                ": bProcessMovieDescFromDB() Error adding movie to list from DB");
            break;
        }

        return TRUE;

    } while (FALSE);

    // Free memory since something bad happened
    STRING_vDestroy(sName.hEnglish);
    STRING_vDestroy(sName.hSpanish);
    STRING_vDestroy(sName.hFrench);
    STRING_vDestroy(hActors);
    STRING_vDestroy(sSynopsis.hEnglish);
    STRING_vDestroy(sSynopsis.hSpanish);
    STRING_vDestroy(sSynopsis.hFrench);
    MOVIE_vDestroy(hMovie);

    return TRUE;
}

/*****************************************************************************
*
*   bProcessMovieTimesFromDB
*
*****************************************************************************/
static BOOLEAN bProcessMovieTimesFromDB (
    SQL_QUERY_COLUMN_STRUCT *psColumns,
    N32 n32NumberOfColumns,
    void *pvArg
       )
{
    BOOLEAN bOk = TRUE;
    SQL_QUERY_COLUMN_STRUCT *psCurrentCol;
    DB_MOVIE_TIMES_FIELDS_ENUM eCurrentField = (DB_MOVIE_TIMES_FIELDS_ENUM)0;
    MOVIES_MGR_OBJECT_STRUCT *psObj =
        (MOVIES_MGR_OBJECT_STRUCT *)pvArg;
    THEATER_TIMES_ROW_STRUCT sTimesRow;
    THEATER_TIMES_OBJECT hTheaterTimes = THEATER_TIMES_INVALID_OBJECT;

    OSAL.bMemSet(&sTimesRow, 0, sizeof(sTimesRow));

    // The times blob data will always point to the blob data from the manager
    sTimesRow.pun16BlobData = &psObj->aun16TimesBlobData[0];

    if (psObj == NULL)
    {
        return FALSE;
    }

    do
    {
        // If there is not the correct number of columns,
        // then we have an error
        if (n32NumberOfColumns != DB_MOVIE_TIMES_MAX_FIELDS )
        {
            break;
        }

        do
        {
            // Grab the current column data
            psCurrentCol = &psColumns[(UN8)eCurrentField];

            // Decode the current field based on it's type
            switch (eCurrentField)
            {
                case DB_MOVIE_TIMES_ID:
                {
                    sTimesRow.tID =
                        (THEATER_ID)psCurrentCol->uData.sUN32.un32Data;
                }
                break;

                case DB_MOVIE_TIMES_INITIAL_TIME:
                {
                    sTimesRow.un32InitialTime =
                        psCurrentCol->uData.sUN32.un32Data;
                }
                break;

                case DB_MOVIE_TIMES_NUM_MOVIES:
                {
                    sTimesRow.un8NumMovies =
                        (UN8)psCurrentCol->uData.sUN32.un32Data;
                }
                break;

                case DB_MOVIE_TIMES_NUM_UN16S:
                {
                    sTimesRow.tNumUN16s =
                        (size_t)psCurrentCol->uData.sUN32.un32Data;
                }
                break;

                case DB_MOVIE_TIMES_SHOWTIMES:
                {
                    if (psCurrentCol->uData.sBLOB.tSize ==
                        (sTimesRow.tNumUN16s * sizeof(UN16)))
                    {
                        OSAL.bMemCpy(
                            sTimesRow.pun16BlobData,
                            psCurrentCol->uData.sBLOB.pvData,
                            psCurrentCol->uData.sBLOB.tSize);
                    }
                    else
                    {
                        bOk = FALSE;
                    }
                }
                break;

                default: // Shouldn't happen
                break;
            }

        } while ((++eCurrentField < DB_MOVIE_TIMES_MAX_FIELDS) && (bOk == TRUE));

        if (bOk == FALSE)
        {
            break;
        }

        hTheaterTimes = THEATER_TIMES_hCreateFromDB(
            (MOVIES_SERVICE_OBJECT)psObj,
            (SMS_OBJECT)psObj->psAppObj, &sTimesRow);

        if (hTheaterTimes == THEATER_TIMES_INVALID_OBJECT)
        {
            break;
        }

        // Add our show times to this list
        bOk = bAddShowTimes(
            psObj, psObj->psAppObj->psCurrentTimes->hTimes, 
            hTheaterTimes, sTimesRow.tID);

        if (bOk == FALSE)
        {
            // Some sort of error, stop processing
            break;
        }

        return TRUE;

    } while (FALSE);

    // Free memory since something bad happened
    THEATER_TIMES_vDestroy(hTheaterTimes);

    return TRUE;
}

/*****************************************************************************
*
*   vStartAgeoutTimer
*
*   This function starts the ageout timer based on the oldest
*   of two timestamps - theaters' and descriptions'.
*
*****************************************************************************/
static void vStartAgeoutTimer (
    MOVIES_MGR_OBJECT_STRUCT *psObj
        )
{
    BOOLEAN bOk;
    OSAL_RETURN_CODE_ENUM eReturnCode;
    UN32 un32UTCsec, un32NextExpireTime;

    eReturnCode = OSAL.eTimeGet(&un32UTCsec);

    if (eReturnCode != OSAL_SUCCESS)
    {
        SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
            MOVIES_MGR_OBJECT_NAME
            ": vStartAgeoutTimer() Error getting time: %s",
            OSAL.pacGetReturnCodeName(eReturnCode));
        return;
    }

    // Choose the oldest update and start timer on its time plus 36 hrs 
    if (psObj->psAppObj->psCurrentMovies->un32UTCsec <
        psObj->psAppObj->psCurrentTimes->un32UTCsec)
    {
        un32NextExpireTime = 
            psObj->psAppObj->psCurrentMovies->un32UTCsec + 
            MOVIES_CONTROL_TABLE_TIMEOUT_SECS;
    }
    else
    {
        un32NextExpireTime = 
            psObj->psAppObj->psCurrentTimes->un32UTCsec + 
            MOVIES_CONTROL_TABLE_TIMEOUT_SECS;    
    }

    if (un32NextExpireTime > un32UTCsec)
    {
        un32NextExpireTime -= un32UTCsec;

        // Kick off the event
        bOk = DATASERVICE_IMPL_bSetTimedEvent(
            psObj->hDataExpireEvent, TRUE, FALSE, un32NextExpireTime);

        if (bOk == FALSE)
        {
            puts(MOVIES_MGR_OBJECT_NAME": Unable to start ageout event");
        }
    }

    return;
}

/*****************************************************************************
*
*   vStopAgeoutTimer
*
*****************************************************************************/
static void vStopAgeoutTimer (
    MOVIES_MGR_OBJECT_STRUCT *psObj
        )
{
    // Stop the expire event from occurring
    DATASERVICE_IMPL_bStopTimedEvent(psObj->hDataExpireEvent);

    return;
}

/*****************************************************************************
*
*   bSaveTimesToDatabase
*
*****************************************************************************/
static BOOLEAN bSaveTimesToDatabase (
    MOVIES_MGR_OBJECT_STRUCT *psObj
        )
{
    BOOLEAN bSuccess;
    OSAL_RETURN_CODE_ENUM eReturnCode;
    MOVIES_DB_WRITE_ITERATOR_STRUCT sIteratorArg;

    do
    {
        // Start the transaction
        bSuccess = SQL_INTERFACE.bStartTransaction(psObj->hSQLPersistConnection);

        if (bSuccess == FALSE)
        {
            break;
        }

        // Empty the current times table
        bSuccess = SQL_INTERFACE.bExecuteCommand(
            psObj->hSQLPersistConnection, MOVIES_CLEAR_TIMES);

        if (bSuccess == FALSE)
        {
            break;
        }

        // Prepare iterator argument
        sIteratorArg.bError = FALSE;
        sIteratorArg.psMgr = psObj;

        // Iterate the movies
        eReturnCode = OSAL.eLinkedListIterate(
            psObj->psAppObj->psCurrentTimes->hTimes,
            (OSAL_LL_ITERATOR_HANDLER)bWriteTimesToDB,
            (void *)&sIteratorArg);

        // end the transaction
        bSuccess = SQL_INTERFACE.bEndTransaction(psObj->hSQLPersistConnection);

        if (bSuccess == FALSE)
        {
            break;
        }

        if ((eReturnCode != OSAL_SUCCESS) || (sIteratorArg.bError == TRUE))
        {
            break;
        }

        return TRUE;

    } while (FALSE);

    return FALSE;
}

/*****************************************************************************
*
*   bWriteTimesToDB
*
*****************************************************************************/
static BOOLEAN bWriteTimesToDB (
    MOVIE_THEATER_TIMES_ENTRY_STRUCT *psTimesEntry,
    MOVIES_DB_WRITE_ITERATOR_STRUCT *psIteratorArg
        )
{
    BOOLEAN bSuccess;

    // Insert into DB
    bSuccess = THEATER_TIMES_bInsertIntoDB(psTimesEntry->hShowTimes,
        psIteratorArg->psMgr->hSQLPersistConnection,
        psIteratorArg->psMgr->hInsertTheaterTimesStmt,
        psIteratorArg->psMgr->asBindParams,
        &psIteratorArg->psMgr->aun16TimesBlobData[0],
        sizeof(psIteratorArg->psMgr->aun16TimesBlobData));

    if (bSuccess == FALSE)
    {
        psIteratorArg->bError = TRUE;
    }

    return bSuccess;
}

/*****************************************************************************
*
*   bSaveDescToDatabase
*
*****************************************************************************/
static BOOLEAN bSaveDescToDatabase (
    MOVIES_MGR_OBJECT_STRUCT *psObj
        )
{
    BOOLEAN bSuccess;
    OSAL_RETURN_CODE_ENUM eReturnCode;
    MOVIES_DB_WRITE_ITERATOR_STRUCT sIteratorArg;

    do
    {
        // Start the transaction
        bSuccess = SQL_INTERFACE.bStartTransaction(psObj->hSQLPersistConnection);

        if (bSuccess == FALSE)
        {
            break;
        }

        // Empty the current movies table
        bSuccess = SQL_INTERFACE.bExecutePreparedStatement(
            psObj->hSQLPersistConnection,
            psObj->hClearMoviesStmt,
            NULL, NULL, 0);
        
        if (bSuccess == FALSE)
        {
            break;
        }

        // Prepare iterator argument
        sIteratorArg.bError = FALSE;
        sIteratorArg.psMgr = psObj;

        // Iterate the movies
        eReturnCode = OSAL.eLinkedListIterate(
            psObj->psAppObj->psCurrentMovies->hDescriptions,
            (OSAL_LL_ITERATOR_HANDLER)bWriteMovieDescToDB,
            (void *)&sIteratorArg);

        // end the transaction
        bSuccess = SQL_INTERFACE.bEndTransaction(psObj->hSQLPersistConnection);

        if (bSuccess == FALSE)
        {
            break;
        }

        if (eReturnCode != OSAL_SUCCESS || sIteratorArg.bError == TRUE)
        {
            break;
        }

        return TRUE;

    } while (FALSE);

    return FALSE;
}

/*****************************************************************************
*
*   bWriteMovieDescToDB
*
*****************************************************************************/
static BOOLEAN bWriteMovieDescToDB (
    MOVIE_OBJECT hMovie,
    MOVIES_DB_WRITE_ITERATOR_STRUCT *psIteratorArg
        )
{
    BOOLEAN bSuccess;
    MOVIE_ID tID = MOVIE.tID(hMovie);
    UN8 un8RunTime = MOVIE.un8RunTime(hMovie);
    STRING_OBJECT hActors = MOVIE.hActors(hMovie);
    MOVIE_MULTI_LANG_FIELD_STRUCT *psName, *psSynopsis;
    size_t tNumRatings = MOVIES_MAX_MOVIE_RATINGS;

    // Get the multi-value fields from this movie
    bSuccess = MOVIE_bGetMultiValueFields(
        hMovie, &psName, &psSynopsis, &tNumRatings, 
        &psIteratorArg->psMgr->asRatings[0]);
    if (bSuccess == FALSE)
    {
        // Skip this entry
        psIteratorArg->bError = TRUE;

        return TRUE;
    }

    vSetInsertDescriptionStmtParams(
        tID, psName, un8RunTime, hActors, 
        psSynopsis, tNumRatings, 
        &psIteratorArg->psMgr->asRatings[0],
        psIteratorArg->psMgr->asBindParams);

    bSuccess = SQL_INTERFACE.bExecutePreparedStatement(
        psIteratorArg->psMgr->hSQLPersistConnection,
        psIteratorArg->psMgr->hInsertDescriptionStmt,
        NULL, NULL,
        MOVIES_INSERT_DESC_STMT_PARAMS_COUNT,
        psIteratorArg->psMgr->asBindParams);

    if (bSuccess == FALSE)
    {
        SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
            MOVIES_MGR_OBJECT_NAME": Error writing movie to DB -- MVID: %u",
            tID);

        psIteratorArg->bError = TRUE;
    }

    // Keep going
    return TRUE;
}

/*****************************************************************************
*
*   vSetInsertDescriptionStmtParams
*
*****************************************************************************/
static void vSetInsertDescriptionStmtParams (
    MOVIE_ID tID,
    MOVIE_MULTI_LANG_FIELD_STRUCT *psName,
    UN8 un8RunTime,
    STRING_OBJECT hActors,
    MOVIE_MULTI_LANG_FIELD_STRUCT *psSynopsis,
    size_t tNumRatings,
    MOVIE_RATING_STRUCT *pasRatings,
    SQL_BIND_PARAMETER_STRUCT *psBindParams
        )
{
    psBindParams[MOVIES_INSERT_DESC_STMT_ID_PARAM].eType =
        SQL_BIND_TYPE_UN32;
    psBindParams[MOVIES_INSERT_DESC_STMT_ID_PARAM].pvData =
        (void *)(size_t)tID;

    psBindParams[MOVIES_INSERT_DESC_STMT_ENG_NAME_PARAM].eType =
        SQL_BIND_TYPE_STRING_OBJECT;
    psBindParams[MOVIES_INSERT_DESC_STMT_ENG_NAME_PARAM].pvData =
        (void *)psName->hEnglish;

    psBindParams[MOVIES_INSERT_DESC_STMT_SPN_NAME_PARAM].eType =
        SQL_BIND_TYPE_STRING_OBJECT;
    psBindParams[MOVIES_INSERT_DESC_STMT_SPN_NAME_PARAM].pvData =
        (void *)psName->hSpanish;

    psBindParams[MOVIES_INSERT_DESC_STMT_FR_NAME_PARAM].eType =
        SQL_BIND_TYPE_STRING_OBJECT;
    psBindParams[MOVIES_INSERT_DESC_STMT_FR_NAME_PARAM].pvData =
        (void *)psName->hFrench;
    
    psBindParams[MOVIES_INSERT_DESC_STMT_RUN_TIME_PARAM].eType =
        SQL_BIND_TYPE_UN32;
    psBindParams[MOVIES_INSERT_DESC_STMT_RUN_TIME_PARAM].pvData =
        (void *)(size_t)un8RunTime;

    psBindParams[MOVIES_INSERT_DESC_STMT_ACTORS_PARAM].eType =
        SQL_BIND_TYPE_STRING_OBJECT;
    psBindParams[MOVIES_INSERT_DESC_STMT_ACTORS_PARAM].pvData =
        (void *)hActors;

    psBindParams[MOVIES_INSERT_DESC_STMT_ENG_SYNOPSIS_PARAM].eType =
        SQL_BIND_TYPE_STRING_OBJECT;
    psBindParams[MOVIES_INSERT_DESC_STMT_ENG_SYNOPSIS_PARAM].pvData =
        (void *)psSynopsis->hEnglish;

    psBindParams[MOVIES_INSERT_DESC_STMT_SPN_SYNOPSIS_PARAM].eType =
        SQL_BIND_TYPE_STRING_OBJECT;
    psBindParams[MOVIES_INSERT_DESC_STMT_SPN_SYNOPSIS_PARAM].pvData =
        (void *)psSynopsis->hSpanish;

    psBindParams[MOVIES_INSERT_DESC_STMT_FR_SYNOPSIS_PARAM].eType =
        SQL_BIND_TYPE_STRING_OBJECT;
    psBindParams[MOVIES_INSERT_DESC_STMT_FR_SYNOPSIS_PARAM].pvData =
        (void *)psSynopsis->hFrench;

    psBindParams[MOVIES_INSERT_DESC_STMT_NUMRATING_PARAM].eType =
        SQL_BIND_TYPE_UN32;
    psBindParams[MOVIES_INSERT_DESC_STMT_NUMRATING_PARAM].pvData =
        (void *)tNumRatings;

    psBindParams[MOVIES_INSERT_DESC_STMT_RATING_PARAM].eType =
        SQL_BIND_TYPE_BLOB;
    psBindParams[MOVIES_INSERT_DESC_STMT_RATING_PARAM].pvData =
        (void *)&pasRatings[0];
    psBindParams[MOVIES_INSERT_DESC_STMT_RATING_PARAM].tSize =
        tNumRatings * sizeof(MOVIE_RATING_STRUCT);

    return;
}

/*****************************************************************************
*
*   vSaveControlDataToDatabase
*
*****************************************************************************/
static void vSaveControlDataToDatabase(
    MOVIES_MGR_OBJECT_STRUCT *psObj
        )
{
    BOOLEAN bSuccess;
    int iReturn;

    do
    {
        // Start the transaction
        bSuccess = SQL_INTERFACE.bStartTransaction(psObj->hSQLPersistConnection);

        if (bSuccess == FALSE)
        {
            break;
        }

        // Empty the current control table table
        bSuccess = SQL_INTERFACE.bExecuteCommand(
            psObj->hSQLPersistConnection, MOVIES_CLEAR_CONTROL_TABLE);

        if (bSuccess == FALSE)
        {
            break;
        }

        // Build our string to insert our control variables
        iReturn = snprintf( &psObj->acBuffer[0], sizeof(psObj->acBuffer),
            MOVIES_INSERT_CONTROL_ROW,
            psObj->psAppObj->psCurrentTimes->tVersion,
            psObj->psAppObj->psCurrentTimes->un32UTCsec,
            psObj->psAppObj->psCurrentMovies->tVersion,
            psObj->psAppObj->psCurrentMovies->un32UTCsec);

        if (iReturn <= 0)
        {
            break;
        }

        // Insert the new control table row
        bSuccess = SQL_INTERFACE.bExecuteCommand(
            psObj->hSQLPersistConnection, &psObj->acBuffer[0]);

        if (bSuccess == FALSE)
        {
            break;
        }

        // Start the timer
        vStartAgeoutTimer(psObj);

    } while (FALSE);

    // End the transaction
    SQL_INTERFACE.bEndTransaction(psObj->hSQLPersistConnection);

    return;
}

/*****************************************************************************
*
*   bAddRatingsEntry
*
*****************************************************************************/
static BOOLEAN bAddRatingsEntry (
    MOVIES_MGR_OBJECT_STRUCT *psObj,
    MOVIE_RATINGS_ROW_STRUCT *psRatingsRow
        )
{
    MOVIE_RATINGS_ENTRY_STRUCT *psEntry;
    OSAL_RETURN_CODE_ENUM eReturnCode;

    do
    {
        // Create a new ratings table entry
        psEntry = (MOVIE_RATINGS_ENTRY_STRUCT *)SMSO_hCreate(
            MOVIES_MGR_OBJECT_NAME":RatingsEntry",
            sizeof(MOVIE_RATINGS_ENTRY_STRUCT),
            (SMS_OBJECT)psObj->psAppObj, FALSE);

        if (psEntry == NULL)
        {
            break;
        }

        // Populate it with the values from the DB
        psEntry->tSysCode = psRatingsRow->tSysCode;
        psEntry->hSystemText = psRatingsRow->hSysName;
        psEntry->tRatingCode = psRatingsRow->tRatingCode;
        psEntry->hRatingText = psRatingsRow->hRatingText;

        // Determine the enum now
        switch (psEntry->tSysCode)
        {
            case MOVIE_RSYS_MPAA:
            {
                // Set the system enum now
                psEntry->eSystem = MOVIE_RATING_SYSTEM_US_MPAA;

                // Determine the rating code
                switch (psEntry->tRatingCode)
                {
                    case MOVIE_RCODE_MPAA_UNRATED:
                    {
                        psEntry->eRating = MOVIE_RATING_MPAA_UNRATED;
                    }
                    break;

                    case MOVIE_RCODE_MPAA_G:
                    {
                        psEntry->eRating = MOVIE_RATING_MPAA_G;
                    }
                    break;

                    case MOVIE_RCODE_MPAA_PG:
                    {
                        psEntry->eRating = MOVIE_RATING_MPAA_PG;
                    }
                    break;

                    case MOVIE_RCODE_MPAA_PG13:
                    {
                        psEntry->eRating = MOVIE_RATING_MPAA_PG13;
                    }
                    break;

                    case MOVIE_RCODE_MPAA_R:
                    {
                        psEntry->eRating = MOVIE_RATING_MPAA_R;
                    }
                    break;

                    case MOVIE_RCODE_MPAA_NC17:
                    {
                        psEntry->eRating = MOVIE_RATING_MPAA_NC17;
                    }
                    break;
                }
            }
            break;

            case MOVIE_RSYS_CAN1:
            {
                // Set the system enum now
                psEntry->eSystem = MOVIE_RATING_SYSTEM_CANADA1;

                // Determine the rating code
                switch (psEntry->tRatingCode)
                {
                    case MOVIE_RCODE_CAN1_UNRATED:
                    {
                        psEntry->eRating = MOVIE_RATING_CANADA1_UNRATED;
                    }
                    break;

                    case MOVIE_RCODE_CAN1_G:
                    {
                        psEntry->eRating = MOVIE_RATING_CANADA1_G;
                    }
                    break;

                    case MOVIE_RCODE_CAN1_PG:
                    {
                        psEntry->eRating = MOVIE_RATING_CANADA1_PG;
                    }
                    break;

                    case MOVIE_RCODE_CAN1_14A:
                    {
                        psEntry->eRating = MOVIE_RATING_CANADA1_14A;
                    }
                    break;

                    case MOVIE_RCODE_CAN1_18A:
                    {
                        psEntry->eRating = MOVIE_RATING_CANADA1_18A;
                    }
                    break;

                    case MOVIE_RCODE_CAN1_R:
                    {
                        psEntry->eRating = MOVIE_RATING_CANADA1_R;
                    }
                    break;
                }
            }
            break;

            case MOVIE_RSYS_CAN2:
            {
                // Set the system enum now
                psEntry->eSystem = MOVIE_RATING_SYSTEM_CANADA2;

                // Determine the rating code
                switch (psEntry->tRatingCode)
                {
                    case MOVIE_RCODE_CAN2_UNRATED:
                    {
                        psEntry->eRating = MOVIE_RATING_CANADA2_UNRATED;
                    }
                    break;

                    case MOVIE_RCODE_CAN2_G:
                    {
                        psEntry->eRating = MOVIE_RATING_CANADA2_G;
                    }
                    break;

                    case MOVIE_RCODE_CAN2_PG:
                    {
                        psEntry->eRating = MOVIE_RATING_CANADA2_PG;
                    }
                    break;

                    case MOVIE_RCODE_CAN2_14:
                    {
                        psEntry->eRating = MOVIE_RATING_CANADA2_14;
                    }
                    break;

                    case MOVIE_RCODE_CAN2_18:
                    {
                        psEntry->eRating = MOVIE_RATING_CANADA2_18;
                    }
                    break;

                    case MOVIE_RCODE_CAN2_E:
                    {
                        psEntry->eRating = MOVIE_RATING_CANADA2_E;
                    }
                    break;

                    case MOVIE_RCODE_CAN2_R:
                    {
                        psEntry->eRating = MOVIE_RATING_CANADA2_R;
                    }
                    break;
                }
            }
            break;

            case MOVIE_RSYS_QUEBEC:
            {
                // Set the system enum now
                psEntry->eSystem = MOVIE_RATING_SYSTEM_QUEBEC;

                // Determine the rating code
                switch (psEntry->tRatingCode)
                {
                    case MOVIE_RCODE_QUEBEC_NONCLASS:
                    {
                        psEntry->eRating = MOVIE_RATING_QUEBEC_NONCLASS;
                    }
                    break;

                    case MOVIE_RCODE_QUEBEC_G:
                    {
                        psEntry->eRating = MOVIE_RATING_QUEBEC_G;
                    }
                    break;

                    case MOVIE_RCODE_QUEBEC_13:
                    {
                        psEntry->eRating = MOVIE_RATING_QUEBEC_13PLUS;
                    }
                    break;

                    case MOVIE_RCODE_QUEBEC_16:
                    {
                        psEntry->eRating = MOVIE_RATING_QUEBEC_16PLUS;
                    }
                    break;

                    case MOVIE_RCODE_QUEBEC_18:
                    {
                        psEntry->eRating = MOVIE_RATING_QUEBEC_18PLUS;
                    }
                    break;
                }
            }
            break;

            case MOVIE_RSYS_MEX:
            {
                // Set the system enum now
                psEntry->eSystem = MOVIE_RATING_SYSTEM_MEXICO;

                // Determine the rating code
                switch (psEntry->tRatingCode)
                {
                    case MOVIE_RCODE_MEX_SC:
                    {
                        psEntry->eRating = MOVIE_RATING_MEXICO_SC;
                    }
                    break;

                    case MOVIE_RCODE_MEX_AA:
                    {
                        psEntry->eRating = MOVIE_RATING_MEXICO_AA;
                    }
                    break;

                    case MOVIE_RCODE_MEX_A:
                    {
                        psEntry->eRating = MOVIE_RATING_MEXICO_A;
                    }
                    break;

                    case MOVIE_RCODE_MEX_B:
                    {
                        psEntry->eRating = MOVIE_RATING_MEXICO_B;
                    }
                    break;

                    case MOVIE_RCODE_MEX_B15:
                    {
                        psEntry->eRating = MOVIE_RATING_MEXICO_B15;
                    }
                    break;

                    case MOVIE_RCODE_MEX_C:
                    {
                        psEntry->eRating = MOVIE_RATING_MEXICO_C;
                    }
                    break;

                    case MOVIE_RCODE_MEX_D:
                    {
                        psEntry->eRating = MOVIE_RATING_MEXICO_D;
                    }
                    break;
                }
            }
            break;

            case MOVIE_RATING_SYSTEM_UNKNOWN:
            default:
            {
                // We aren't able to map these
                psEntry->eSystem = MOVIE_RATING_SYSTEM_UNKNOWN;
                psEntry->eRating = MOVIE_RATING_UNKNOWN;
            }
        }

        // Add it to the list of ratings
        eReturnCode = OSAL.eLinkedListAdd(
            psObj->psAppObj->hRatingsList,
            OSAL_INVALID_LINKED_LIST_ENTRY_PTR, (void *)psEntry);

        if (eReturnCode != OSAL_SUCCESS)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                MOVIES_MGR_OBJECT_NAME
                ": bAddRatingsEntry() Error adding rating entry (%s)",
                OSAL.pacGetReturnCodeName(eReturnCode));
            break;
        }

        printf(MOVIES_MGR_OBJECT_NAME
            ": Rating entry (SCODE %d)-(RCODE %d) added: ",
            psEntry->tSysCode, psEntry->tRatingCode);
#if DEBUG_OBJECT == 1
        STRING.n32FWrite(psEntry->hRatingText, stdout);
#endif
        printf("\n");
        return TRUE;

    } while (FALSE);

    if (psEntry != NULL)
    {
        SMSO_vDestroy((SMS_OBJECT)psEntry);
    }

    return FALSE;
}

/*****************************************************************************
*
*   vDestroyRatingsEntry
*
*****************************************************************************/
static void vDestroyRatingsEntry (
    MOVIE_RATINGS_ENTRY_STRUCT *psEntry
        )
{
    if (psEntry != NULL)
    {
        STRING_vDestroy(psEntry->hSystemText);
        STRING_vDestroy(psEntry->hRatingText);
        SMSO_vDestroy((SMS_OBJECT)psEntry);
    }

    return;
}

/*****************************************************************************
*
*   vUpdateAllDSRLStates
*
*****************************************************************************/
static void vUpdateAllDSRLStates (
    MOVIES_MGR_OBJECT_STRUCT *psObj,
    DSRL_STATE_ENUM eState
        )
{
    OSAL_RETURN_CODE_ENUM eReturnCode;

    eReturnCode = OSAL.eLinkedListIterate(
        psObj->hDSRLList,
       (OSAL_LL_ITERATOR_HANDLER)bUpdateAllDSRLStateIterator,
       (void *)(size_t)eState);

    if ( (eReturnCode != OSAL_SUCCESS) &&
         (eReturnCode != OSAL_NO_OBJECTS))
    {
        SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
            MOVIES_MGR_OBJECT_NAME": error while performing vUpdateAllDSRLStates");
    }

    return;
}

/*****************************************************************************
*
*   vSendAllUpdatingDSRLsToError
*
*****************************************************************************/
static void vSendAllUpdatingDSRLsToError (
    MOVIES_MGR_OBJECT_STRUCT *psObj
        )
{
    OSAL_RETURN_CODE_ENUM eReturnCode;

    eReturnCode = OSAL.eLinkedListIterate(
        psObj->hDSRLList,
        (OSAL_LL_ITERATOR_HANDLER)bErrorAllUpdatingDSRLs,
        NULL);

    if ( (eReturnCode != OSAL_SUCCESS) &&
         (eReturnCode != OSAL_NO_OBJECTS))
    {
        SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
            MOVIES_MGR_OBJECT_NAME": error while performing vUpdateAllDSRLStates");
    }

    return;
}

/*****************************************************************************
*
*   bUpdateAllDSRLStateIterator
*
*****************************************************************************/
static BOOLEAN bUpdateAllDSRLStateIterator (
    MOVIES_DSRL_DESC_STRUCT *psDSRLDesc,
    DSRL_STATE_ENUM eState
        )
{
    DSRL_STATE_ENUM eCurState;

    eCurState = DSRL.eState(psDSRLDesc->hDSRL);
    if (eCurState != DSRL_STATE_ERROR)
    {
        DSRL_vSetState(psDSRLDesc->hDSRL, eState);
    }

    return TRUE;
}

/*****************************************************************************
*
*   bErrorAllUpdatingDSRLs
*
*****************************************************************************/
static BOOLEAN bErrorAllUpdatingDSRLs (
    MOVIES_DSRL_DESC_STRUCT *psDSRLDesc,
    void *pvUnused
        )
{
    DSRL_STATE_ENUM eCurState;

    eCurState = DSRL.eState(psDSRLDesc->hDSRL);
    if (eCurState == DSRL_STATE_UPDATING)
    {
        DSRL_vSetState(psDSRLDesc->hDSRL, DSRL_STATE_ERROR);
    }

    return TRUE;
}

/*****************************************************************************
*
*   n16CompareTargetDesc
*
*****************************************************************************/
static N16 n16CompareTargetDesc (
    MOVIES_DSRL_TARGET_DESC_STRUCT *psObj1,
    MOVIES_DSRL_TARGET_DESC_STRUCT *psObj2
        )
{
    N16 n16Result = N16_MIN;

    if (psObj1 != NULL && psObj2 != NULL)
    {
        // Compare the locations
        BOOLEAN bCompare;

        bCompare = LOCATION_bCompare(psObj1->hLocation, psObj2->hLocation);

        if (bCompare == TRUE)
        {
            n16Result = 0;
        }
    }

    return n16Result;
}

/*****************************************************************************
*
*   n16FindDSRLInTheater
*
*****************************************************************************/
static N16 n16FindDSRLInTheater (
    MOVIES_THEATER_DSRL_ENTRY_STRUCT *psDSRLEntry,
    DSRL_OBJECT hDSRL
        )
{
    N16 n16Result = N16_MIN;

    if (psDSRLEntry != NULL)
    {
        if ((psDSRLEntry->bInDSRL == TRUE) && (psDSRLEntry->psDesc != NULL))
        {
            if (psDSRLEntry->psDesc->hDSRL == hDSRL)
            {
                // Found it!
                n16Result = 0;
            }
            else
            {
                // Not it, keep looking
                n16Result = 1;
            }
        }
    }

    // Keep looking
    return n16Result;
}

/*****************************************************************************
*
*   n16CompareTheaterEntry
*
*****************************************************************************/
static N16 n16CompareTheaterEntry (
    MOVIE_THEATER_ENTRY_STRUCT *psObj1,
    MOVIE_THEATER_ENTRY_STRUCT *psObj2
        )
{
    if ( (psObj1 != NULL) &&
         (psObj2 != NULL)
       )
    {
        return COMPARE(psObj1->tID, psObj2->tID);
    }

    return N16_MIN;
}

/*****************************************************************************
*
*   n16CompareTheaterTimes
*
*****************************************************************************/
static N16 n16CompareTheaterTimes (
    MOVIE_THEATER_TIMES_ENTRY_STRUCT *psTimes1,
    MOVIE_THEATER_TIMES_ENTRY_STRUCT *psTimes2
        )
{
    if (((MOVIE_THEATER_TIMES_ENTRY_STRUCT *)NULL == psTimes1) ||
        ((MOVIE_THEATER_TIMES_ENTRY_STRUCT *)NULL == psTimes2))
    {
        return N16_MIN;
    }

    return COMPARE(psTimes1->tTheaterID, psTimes2->tTheaterID);
}

/*****************************************************************************
*
*   n16CompareRatingsEntry
*
*****************************************************************************/
static N16 n16CompareRatingsEntry (
    MOVIE_RATINGS_ENTRY_STRUCT *psEntry1,
    MOVIE_RATINGS_ENTRY_STRUCT *psEntry2
        )
{
    N16 n16Return = N16_MIN;

    if ( ((MOVIE_RATINGS_ENTRY_STRUCT *)NULL != psEntry1) &&
         ((MOVIE_RATINGS_ENTRY_STRUCT *)NULL != psEntry2)
       )
    {
        // Compare the system code now
        n16Return = COMPARE(psEntry1->tSysCode, psEntry2->tSysCode);

        // If it's the same
        if (0 == n16Return)
        {
            // Then compare the rating code as well
            n16Return = COMPARE(psEntry1->tRatingCode, psEntry2->tRatingCode);
        }
    }

    return n16Return;
}

/*****************************************************************************
*
*   n16SortDSRLByDistance
*
*****************************************************************************/
static N16 n16SortDSRLByDistance (
    DSRL_OBJECT hDSRL,
    DSRL_ENTRY_OBJECT hEntry1,
    DSRL_ENTRY_OBJECT hEntry2,
    MOVIES_DSRL_DESC_STRUCT *psDSRLDesc
        )
{
    N16 n16Return = 1;
    MOVIES_DSRL_TARGET_DESC_STRUCT *psTargetDesc =
        (MOVIES_DSRL_TARGET_DESC_STRUCT *)NULL;

    // Fetch the first Target Description
    OSAL.hLinkedListFirst(psDSRLDesc->hTargetList, (void**)&psTargetDesc);

    if (psTargetDesc != NULL)
    {
        THEATER_OBJECT hTheater1, hTheater2;
        LOCATION_OBJECT hLocation1, hLocation2, hClosestLoc;

        // Extract theaters & locations
        hTheater1 = (THEATER_OBJECT)DSRL_ENTRY.hTheater(hEntry1);
        hLocation1 = THEATER.hLocation(hTheater1);
        hTheater2 = (THEATER_OBJECT)DSRL_ENTRY.hTheater(hEntry2);
        hLocation2 = THEATER.hLocation(hTheater2);

        // Determine the closest location
        hClosestLoc = LOCATION.hClosest(psTargetDesc->hLocation, hLocation1, hLocation2);

        if (hClosestLoc != LOCATION_INVALID_OBJECT)
        {
            if (hClosestLoc == hLocation1)
            {
                n16Return = -1;
            }
            else
            {
                n16Return = 1;
            }
        }
    }

    return n16Return;
}

/*****************************************************************************
*
*   bDeleteTheaterFromDB
*
*****************************************************************************/
static BOOLEAN bDeleteTheaterFromDB (
    SQL_INTERFACE_OBJECT hSQLConnection,
    char *pcSQLCommandBuffer,
    size_t tBufferSize,
    THEATER_ID tID
        )
{
    int iReturn;
    BOOLEAN bSuccess = FALSE;

    do
    {
        // Build our string to fetch to remove the theater from the database
        iReturn = snprintf( &pcSQLCommandBuffer[0], tBufferSize,
            MOVIES_DELETE_THEATER, tID);
        if (iReturn <= 0)
        {
            break;
        }

        // Remove from Database
        bSuccess = SQL_INTERFACE.bExecuteCommand(
            hSQLConnection, &pcSQLCommandBuffer[0]);
        if (bSuccess == FALSE)
        {
            break;
        }

        bSuccess = FALSE;

        // Build our string to remove the theater_rtree from the database
        iReturn = snprintf( &pcSQLCommandBuffer[0], tBufferSize,
            MOVIES_DELETE_THEATER_RTREE, tID);
        if (iReturn <= 0)
        {
            break;
        }

        // Remove from Database
        bSuccess = SQL_INTERFACE.bExecuteCommand(
                hSQLConnection, &pcSQLCommandBuffer[0]);


    } while (FALSE);

    return bSuccess;
}

/*****************************************************************************
*
*   bInsertTheaterIntoDB
*
*****************************************************************************/
static BOOLEAN bInsertTheaterIntoDB (
    SQL_INTERFACE_OBJECT hSQLConnection,
    char *pcSQLCommandBuffer,
    size_t tBufferSize,
    THEATER_ROW_STRUCT *psTheaterRow
        )
{
    int iReturn;
    BOOLEAN bSuccess;

    // Delete the theater to make sure it isn't there
    bSuccess = bDeleteTheaterFromDB(
        hSQLConnection,
        pcSQLCommandBuffer,
        tBufferSize, psTheaterRow->tID);

    if (bSuccess == TRUE)
    {
        bSuccess =
            SQL_INTERFACE.bExecutePreparedCommand(
                hSQLConnection,
                MOVIES_INSERT_THEATER,
                DB_THEATER_MAX_FIELDS,
                (PREPARED_QUERY_COLUMN_CALLBACK)bPrepareTheaterColumn,
                (void *)psTheaterRow);

        if (bSuccess == TRUE)
        {
            // Insert record into the R-Tree table
            N32 n32Lat, n32Lon;

            n32Lat = OSAL_FIXED.n32ScaledValue(
                psTheaterRow->hLat, LOCATION_BINPOINT);

            n32Lon = OSAL_FIXED.n32ScaledValue(
                psTheaterRow->hLon, LOCATION_BINPOINT);

            iReturn = snprintf(
                &pcSQLCommandBuffer[0],
                tBufferSize,
                MOVIES_INSERT_THEATER_RTREE,
                psTheaterRow->tID,
                n32Lat - MOVIES_RTREE_RANGE,
                n32Lat + MOVIES_RTREE_RANGE,
                n32Lon - MOVIES_RTREE_RANGE,
                n32Lon + MOVIES_RTREE_RANGE);

            if (iReturn > 0)
            {
                bSuccess = SQL_INTERFACE.bExecuteCommand(
                    hSQLConnection, &pcSQLCommandBuffer[0]);
            }
        }
    }

    return bSuccess;
}

/*****************************************************************************
*
*   bUpdateTheaterInDB
*
*****************************************************************************/
static BOOLEAN bUpdateTheaterInDB (
    SQL_INTERFACE_OBJECT hSQLConnection,
    char *pcSQLCommandBuffer,
    size_t tBufferSize,
    THEATER_ROW_STRUCT *psTheaterRow,
    BOOLEAN bAmenitiesUpdated
        )
{
    int iReturn;
    BOOLEAN bSuccess = FALSE;
    THEATER_ROW_STRUCT sCurTheaterRow;

    OSAL.bMemSet(&sCurTheaterRow, 0, sizeof(sCurTheaterRow));

    // Build our string to fetch to remove the theater from the database
    iReturn = snprintf( &pcSQLCommandBuffer[0], tBufferSize,
        MOVIES_SELECT_THEATERS_BY_ID, psTheaterRow->tID);

    if (iReturn > 0)
    {
        // Is theater in DB?
        sCurTheaterRow.tID = THEATER_INVALID_ID;
        bSuccess = SQL_INTERFACE.bQuery(hSQLConnection,
            &pcSQLCommandBuffer[0],
            (SQL_QUERY_RESULT_HANDLER)bCheckTheaterInDB,
            (void*)&sCurTheaterRow);

        if (bSuccess == TRUE)
        {
            if (sCurTheaterRow.tID != THEATER_INVALID_ID)
            {
                // Update our previous row struct with new values
                bSuccess = bCopyTheaterRowValues(
                    psTheaterRow, &sCurTheaterRow, bAmenitiesUpdated);

                if (bSuccess == TRUE)
                {
                    bSuccess =
                        SQL_INTERFACE.bExecutePreparedCommand(
                            hSQLConnection,
                            MOVIES_UPDATE_THEATER,
                            DB_THEATER_MAX_FIELDS,
                            (PREPARED_QUERY_COLUMN_CALLBACK)bPrepareTheaterColumn,
                            (void *)&sCurTheaterRow);

                    if (bSuccess == TRUE)
                    {
                        // Update R-Tree
                        N32 n32Lat, n32Lon;

                        n32Lat = OSAL_FIXED.n32ScaledValue(
                            sCurTheaterRow.hLat, LOCATION_BINPOINT);

                        n32Lon = OSAL_FIXED.n32ScaledValue(
                            sCurTheaterRow.hLon, LOCATION_BINPOINT);

                        iReturn = snprintf(
                            &pcSQLCommandBuffer[0], tBufferSize,
                            MOVIES_UPDATE_THEATER_RTREE,
                            n32Lat - MOVIES_RTREE_RANGE,
                            n32Lat + MOVIES_RTREE_RANGE,
                            n32Lon - MOVIES_RTREE_RANGE,
                            n32Lon + MOVIES_RTREE_RANGE,
                            sCurTheaterRow.tID);

                        if (iReturn > 0)
                        {
                            bSuccess = SQL_INTERFACE.bExecuteCommand(
                                hSQLConnection, &pcSQLCommandBuffer[0]);
                        }
                    }
                }
            }
            else
            {
                // If theater isn't in db, perform an insert
                bSuccess = bInsertTheaterIntoDB(hSQLConnection,
                    &pcSQLCommandBuffer[0], tBufferSize ,psTheaterRow);
            }
        }
    }

    vFreeTheaterRowObjects(&sCurTheaterRow);

    return bSuccess;
}

/*****************************************************************************
*
*   bCheckTheaterInDB
*
*****************************************************************************/
static BOOLEAN bCheckTheaterInDB (
    SQL_QUERY_COLUMN_STRUCT *psColumn,
    N32 n32NumberOfColumns,
    THEATER_ROW_STRUCT  *psTheaterRow
        )
{
    BOOLEAN bReturn = FALSE;

    // Verify input
    if ((psTheaterRow != NULL) &&
        (n32NumberOfColumns == DB_THEATER_MAX_FIELDS ))
    {
        // Populate data from the table
        bReturn = bReadTheaterFromDB(psTheaterRow, psColumn);
    }

    return bReturn;
}

/*******************************************************************************
*
*   bReadTheaterFromDB
*
*******************************************************************************/
static BOOLEAN bReadTheaterFromDB (
    THEATER_ROW_STRUCT *psTheaterRow,
    SQL_QUERY_COLUMN_STRUCT *psColumn
        )
{
    DB_THEATER_FIELDS_ENUM eCurrentField = (DB_THEATER_FIELDS_ENUM)0;
    BOOLEAN bOk = TRUE;
    N32 n32FixedValue;

    do
    {
        switch (eCurrentField)
        {
            case DB_THEATER_ID:
            {
                psTheaterRow->tID = (THEATER_ID)
                    psColumn[eCurrentField].uData.sUN32.un32Data;
            }
            break;

            case DB_THEATER_NAME:
            {
                psTheaterRow->hName = STRING.hCreate(
                    (const char *)psColumn[eCurrentField].uData.sCString.pcData,
                    strlen((const char *)psColumn[eCurrentField].uData.sCString.pcData));
                if (psTheaterRow->hName == STRING_INVALID_OBJECT)
                {
                    SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                        MOVIES_MGR_OBJECT_NAME
                        ": bReadTheaterFromDB() "
                        "Error getting theater name from DB");
                    bOk = FALSE;
                }
            }
            break;

            case DB_THEATER_STATE:
            {
                psTheaterRow->tStateID = (STATE_ID)
                    psColumn[eCurrentField].uData.sUN32.un32Data;
            }
            break;

            case DB_THEATER_ADDRESS:
            {
                psTheaterRow->hAddr = STRING.hCreate(
                    (const char *)psColumn[eCurrentField].uData.sCString.pcData,
                    strlen((const char *)psColumn[eCurrentField].uData.sCString.pcData));
                if (psTheaterRow->hAddr == STRING_INVALID_OBJECT)
                {
                    SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                        MOVIES_MGR_OBJECT_NAME
                        ": bReadTheaterFromDB() "
                        "Error getting theater address from DB");
                    bOk = FALSE;
                }
            }
            break;

            case DB_THEATER_CITY:
            {
                psTheaterRow->hCity = STRING.hCreate(
                    (const char *)psColumn[eCurrentField].uData.sCString.pcData,
                    strlen((const char *)psColumn[eCurrentField].uData.sCString.pcData));
                if (psTheaterRow->hCity == STRING_INVALID_OBJECT)
                {
                    SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                        MOVIES_MGR_OBJECT_NAME
                        ": bReadTheaterFromDB() "
                        "Error getting theater city from DB");
                    bOk = FALSE;
                }
            }
            break;

            case DB_THEATER_ZIP:
            {
                psTheaterRow->hZIP = STRING.hCreate(
                    (const char *)psColumn[eCurrentField].uData.sCString.pcData,
                    strlen((const char *)psColumn[eCurrentField].uData.sCString.pcData));
                if (psTheaterRow->hZIP == STRING_INVALID_OBJECT)
                {
                    SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                        MOVIES_MGR_OBJECT_NAME
                        ": bReadTheaterFromDB() "
                        "Error getting theater ZIP code from DB");
                    bOk = FALSE;
                }
            }
            break;

            case DB_THEATER_PHONE:
            {
                psTheaterRow->hPhone = STRING.hCreate(
                    (const char *)psColumn[eCurrentField].uData.sCString.pcData,
                    strlen((const char *)psColumn[eCurrentField].uData.sCString.pcData));
                if (psTheaterRow->hPhone == STRING_INVALID_OBJECT)
                {
                    SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                        MOVIES_MGR_OBJECT_NAME
                        ": bReadTheaterFromDB() "
                        "Error getting theater phone # from DB");
                    bOk = FALSE;
                }
            }
            break;

            case DB_THEATER_LAT:
            {
                n32FixedValue = (N32)
                    psColumn[eCurrentField].uData.sUN32.un32Data;

                psTheaterRow->hLat =
                    OSAL_FIXED.hCreateInMemory(n32FixedValue, LOCATION_BINPOINT,
                    &psTheaterRow->atLatFixedData[0]);
            }
            break;

            case DB_THEATER_LON:
            {
                n32FixedValue = (N32)
                    psColumn[eCurrentField].uData.sUN32.un32Data;

                psTheaterRow->hLon =
                    OSAL_FIXED.hCreateInMemory(n32FixedValue, LOCATION_BINPOINT,
                    &psTheaterRow->atLonFixedData[0]);
            }
            break;

            case DB_THEATER_AMENITIES:
            {
                psTheaterRow->un16Amenities = (UN16)
                    psColumn[eCurrentField].uData.sUN32.un32Data;
            }
            break;

            default:
            {
                // Unexpected field. Error!
                bOk = FALSE;
            }
            break;
        }

        // Stop if we experienced an error
        if (bOk == FALSE)
        {
            break;
        }

    } while (++eCurrentField < DB_THEATER_MAX_FIELDS);

    return bOk;
}

/*****************************************************************************
*
*   bCopyTheaterRowValues
*
*****************************************************************************/
static BOOLEAN bCopyTheaterRowValues (
    THEATER_ROW_STRUCT *psFromTheaterRow,
    THEATER_ROW_STRUCT *psToTheaterRow,
    BOOLEAN bAmenitiesUpdated
        )
{
    BOOLEAN bOk = TRUE;

    psToTheaterRow->tID = psFromTheaterRow->tID;
    psToTheaterRow->tStateID = psFromTheaterRow->tStateID;

    do
    {
        // If new data contains hName field...
        if (psFromTheaterRow->hName != STRING_INVALID_OBJECT)
        {
            // .. we check if hName was already present before.
            if (psToTheaterRow->hName != STRING_INVALID_OBJECT)
            {
                // If it is - remove obsolete STRING object,
                STRING.vDestroy(psToTheaterRow->hName);
            }

            // Putting new pointer into structure,
            psToTheaterRow->hName = psFromTheaterRow->hName;
            // Need to clear the pointer in source structure so the
            // object would not be double-freed
            psFromTheaterRow->hName = STRING_INVALID_OBJECT;
        }

        // Repeat the same for all STRING objects.
        if (psFromTheaterRow->hAddr != STRING_INVALID_OBJECT)
        {
            if (psToTheaterRow->hAddr != STRING_INVALID_OBJECT)
            {
                STRING.vDestroy(psToTheaterRow->hAddr);
            }

            psToTheaterRow->hAddr = psFromTheaterRow->hAddr;
            psFromTheaterRow->hAddr = STRING_INVALID_OBJECT;
        }

        if (psFromTheaterRow->hCity != STRING_INVALID_OBJECT)
        {
            if (psToTheaterRow->hCity != STRING_INVALID_OBJECT)
            {
                STRING.vDestroy(psToTheaterRow->hCity);
            }

            psToTheaterRow->hCity = psFromTheaterRow->hCity;
            psFromTheaterRow->hCity = STRING_INVALID_OBJECT;
        }

        if (psFromTheaterRow->hZIP != STRING_INVALID_OBJECT)
        {
            if (psToTheaterRow->hZIP != STRING_INVALID_OBJECT)
            {
                STRING.vDestroy(psToTheaterRow->hZIP);
            }

            psToTheaterRow->hZIP = psFromTheaterRow->hZIP;
            psFromTheaterRow->hZIP = STRING_INVALID_OBJECT;
        }

        if (psFromTheaterRow->hPhone != STRING_INVALID_OBJECT)
        {
            if (psToTheaterRow->hPhone != STRING_INVALID_OBJECT)
            {
                STRING.vDestroy(psToTheaterRow->hPhone);
            }

            psToTheaterRow->hPhone = psFromTheaterRow->hPhone;
            psFromTheaterRow->hPhone = STRING_INVALID_OBJECT;
        }

        if (psFromTheaterRow->hLat != OSAL_FIXED_INVALID_OBJECT)
        {
            bOk = OSAL_FIXED.bCopyToMemory(psFromTheaterRow->hLat,
                &psToTheaterRow->atLatFixedData[0]);
            if (bOk == TRUE)
            {
                psToTheaterRow->hLat =
                    (OSAL_FIXED_OBJECT)&psToTheaterRow->atLatFixedData[0];
            }
            else
            {
                break;
            }
        }

        if (psFromTheaterRow->hLon != OSAL_FIXED_INVALID_OBJECT)
        {
            bOk = OSAL_FIXED.bCopyToMemory(psFromTheaterRow->hLon,
                &psToTheaterRow->atLonFixedData[0]);
            if (bOk == TRUE)
            {
                psToTheaterRow->hLon =
                    (OSAL_FIXED_OBJECT)&psToTheaterRow->atLonFixedData[0];
            }
            else
            {
                break;
            }
        }

        // Were the amenities updated?
        if (TRUE == bAmenitiesUpdated)
        {
            // Yep, copy them over now
            psToTheaterRow->un16Amenities = psFromTheaterRow->un16Amenities;
        }

    } while (FALSE);

    return bOk;
}

/*****************************************************************************
*
*   bPrepareTheaterColumn
*
*****************************************************************************/
static BOOLEAN bPrepareTheaterColumn (
    SQL_COLUMN_INDEX tIndex,
    SQL_BIND_TYPE_ENUM *peType,
    size_t *ptDataSize,
    void **ppvData,
    THEATER_ROW_STRUCT *psTheaterRow
        )
{
    BOOLEAN bSuccess = TRUE;
    N32 n32FixedValue;

    switch (tIndex)
    {
        case DB_THEATER_ID:
        {
            *peType = SQL_BIND_TYPE_UN32;
            *ppvData = (void *)(size_t)psTheaterRow->tID;
        }
        break;

        case DB_THEATER_NAME:
        {
            *peType = SQL_BIND_TYPE_STRING_OBJECT;
            *ppvData = psTheaterRow->hName;
        }
        break;

        case DB_THEATER_STATE:
        {
            *peType = SQL_BIND_TYPE_UN32;
            *ppvData = (void *)(size_t)psTheaterRow->tStateID;
        }
        break;

        case DB_THEATER_ADDRESS:
        {
            *peType = SQL_BIND_TYPE_STRING_OBJECT;
            *ppvData = psTheaterRow->hAddr;
        }
        break;

        case DB_THEATER_CITY:
        {
            *peType = SQL_BIND_TYPE_STRING_OBJECT;
            *ppvData = psTheaterRow->hCity;
        }
        break;

        case DB_THEATER_ZIP:
        {
            *peType = SQL_BIND_TYPE_STRING_OBJECT;
            *ppvData = psTheaterRow->hZIP;
        }
        break;

        case DB_THEATER_PHONE:
        {
            *peType = SQL_BIND_TYPE_STRING_OBJECT;
            *ppvData = psTheaterRow->hPhone;
        }
        break;

        case DB_THEATER_AMENITIES:
        {
            *peType = SQL_BIND_TYPE_UN32;
            *ppvData = (void *)(size_t)psTheaterRow->un16Amenities;
        }
        break;

        case DB_THEATER_LAT:
        {
            n32FixedValue = OSAL_FIXED.n32ScaledValue(
                psTheaterRow->hLat, LOCATION_BINPOINT);

            *peType = SQL_BIND_TYPE_UN32;
            *ppvData = (void *)(size_t)(UN32)n32FixedValue;
        }
        break;

        case DB_THEATER_LON:
        {
            n32FixedValue = OSAL_FIXED.n32ScaledValue(
                psTheaterRow->hLon, LOCATION_BINPOINT);

            *peType = SQL_BIND_TYPE_UN32;
            *ppvData = (void *)(size_t)(UN32)n32FixedValue;
        }
        break;

        case DB_THEATER_MAX_FIELDS:
        default:
            bSuccess = FALSE;
        break;

    }

    return bSuccess;
}

/*****************************************************************************
*
*   bPrepareRatingsColumn
*
*****************************************************************************/
static BOOLEAN bPrepareRatingsColumn (
    SQL_COLUMN_INDEX tIndex,
    SQL_BIND_TYPE_ENUM *peType,
    size_t *ptDataSize,
    void **ppvData,
    MOVIE_RATINGS_ROW_STRUCT *psRatingsRow
        )
{
    BOOLEAN bSuccess = TRUE;

    switch (tIndex)
    {
        case DB_RATINGS_SCODE:
        {
            *peType = SQL_BIND_TYPE_UN32;
            *ppvData = (void *)(size_t)psRatingsRow->tSysCode;
        }
        break;

        case DB_RATINGS_STEXT:
        {
            *peType = SQL_BIND_TYPE_STRING_OBJECT;
            *ppvData = psRatingsRow->hSysName;
        }
        break;

        case DB_RATINGS_RCODE:
        {
            *peType = SQL_BIND_TYPE_UN32;
            *ppvData = (void *)(size_t)psRatingsRow->tRatingCode;
        }
        break;

        case DB_RATINGS_RTEXT:
        {
            *peType = SQL_BIND_TYPE_STRING_OBJECT;
            *ppvData = psRatingsRow->hRatingText;
        }
        break;

        case DB_RATINGS_MAX_FIELDS:
        default:
            bSuccess = FALSE;
        break;

    }

    return bSuccess;
}

/*****************************************************************************
*
*   vSetError
*
*****************************************************************************/
static void vSetError (
    MOVIES_MGR_OBJECT_STRUCT *psObj,
    DATASERVICE_ERROR_CODE_ENUM eErrorCode
        )
{
    // Tell the DSM about it
    DATASERVICE_IMPL_vError((DATASERVICE_IMPL_HDL)psObj, eErrorCode);

    return;
}
