/******************************************************************************/
/*                    Copyright (c) Sirius XM Radio, Inc.                     */
/*                            All Rights Reserved                             */
/*         Licensed Materials - Property of Sirius XM Radio, Inc.             */
/******************************************************************************/
/*******************************************************************************
 *
 * DESCRIPTION
 *
 *  This module contains the Object:MODULE VERSION implementation for the
 *  Sirius Module Services (SMS)
 *
 ******************************************************************************/
#include "sms_api.h"

#include "module_obj.h"
#include "module_version_obj.h"
#include "_module_version_obj.h"

#include "sms_api_debug.h"
static const char *gpacThisFile = __FILE__;

/*****************************************************************************
                             PUBLIC FUNCTIONS
*****************************************************************************/
/*****************************************************************************
*
*   hTypeID
*
*****************************************************************************/
static STRING_OBJECT hTypeID (
    MODULE_OBJECT hModule
        )
{
    STRING_OBJECT hStrReturn = STRING_INVALID_OBJECT;
    MODULE_VERSION_OBJECT_STRUCT *psObj;
    SMSAPI_RETURN_CODE_ENUM eReturnCode;

    eReturnCode = eAccessObjectFromModule(hModule, &psObj);
    if ( eReturnCode != SMSAPI_RETURN_CODE_SUCCESS )
    {
        return hStrReturn;
    }

    hStrReturn = psObj->hTypeID;

    return hStrReturn;
}


/*****************************************************************************
*
*   hHardwareRevision
*
*****************************************************************************/
static STRING_OBJECT hHardwareRevision (
    MODULE_OBJECT hModule
        )
{
    STRING_OBJECT hStrReturn = STRING_INVALID_OBJECT;
    MODULE_VERSION_OBJECT_STRUCT *psObj;
    SMSAPI_RETURN_CODE_ENUM eReturnCode;

    eReturnCode = eAccessObjectFromModule(hModule, &psObj);
    if ( eReturnCode != SMSAPI_RETURN_CODE_SUCCESS )
    {
        return hStrReturn;
    }

    hStrReturn = psObj->hHardwareRevision;

    return hStrReturn;
}

/*****************************************************************************
*
*   hSoftwareRevision
*
*****************************************************************************/
static STRING_OBJECT hSoftwareRevision (
    MODULE_OBJECT hModule
        )
{
    STRING_OBJECT hStrReturn = STRING_INVALID_OBJECT;
    MODULE_VERSION_OBJECT_STRUCT *psObj;
    SMSAPI_RETURN_CODE_ENUM eReturnCode;

    eReturnCode = eAccessObjectFromModule(hModule, &psObj);
    if ( eReturnCode != SMSAPI_RETURN_CODE_SUCCESS )
    {
        return hStrReturn;
    }

    hStrReturn = psObj->hSoftwareRevision;

    return hStrReturn;
}

/*****************************************************************************
*
*   hProtocolRevision
*
*****************************************************************************/
static STRING_OBJECT hProtocolRevision (
    MODULE_OBJECT hModule
        )
{
    STRING_OBJECT hStrReturn = STRING_INVALID_OBJECT;
    MODULE_VERSION_OBJECT_STRUCT *psObj;
    SMSAPI_RETURN_CODE_ENUM eReturnCode;

    eReturnCode = eAccessObjectFromModule(hModule, &psObj);
    if ( eReturnCode != SMSAPI_RETURN_CODE_SUCCESS )
    {
        return hStrReturn;
    }

    hStrReturn = psObj->hProtocolRevision;

    return hStrReturn;
}

/*****************************************************************************
*
*   hDecoderRevision
*
*****************************************************************************/
static STRING_OBJECT hDecoderRevision (
    MODULE_OBJECT hModule
        )
{
    STRING_OBJECT hStrReturn = STRING_INVALID_OBJECT;
    MODULE_VERSION_OBJECT_STRUCT *psObj;
    SMSAPI_RETURN_CODE_ENUM eReturnCode;

    eReturnCode = eAccessObjectFromModule(hModule, &psObj);
    if ( eReturnCode != SMSAPI_RETURN_CODE_SUCCESS )
    {
        return hStrReturn;
    }

    hStrReturn = psObj->hDecoderRevision;

    return hStrReturn;
}


/*****************************************************************************
*
*   hBaseBandRevision
*
*****************************************************************************/
static STRING_OBJECT hBaseBandRevision (
    MODULE_OBJECT hModule
        )
{
    STRING_OBJECT hStrReturn = STRING_INVALID_OBJECT;
    MODULE_VERSION_OBJECT_STRUCT *psObj;
    SMSAPI_RETURN_CODE_ENUM eReturnCode;

    eReturnCode = eAccessObjectFromModule(hModule, &psObj);
    if ( eReturnCode != SMSAPI_RETURN_CODE_SUCCESS )
    {
        return hStrReturn;
    }

    hStrReturn = psObj->hBaseBandRevision;

    return hStrReturn;
}

/*****************************************************************************
*
*   hRFRevision
*
*****************************************************************************/
static STRING_OBJECT hRFRevision (
    MODULE_OBJECT hModule
        )
{
    STRING_OBJECT hStrReturn = STRING_INVALID_OBJECT;
    MODULE_VERSION_OBJECT_STRUCT *psObj;
    SMSAPI_RETURN_CODE_ENUM eReturnCode;

    eReturnCode = eAccessObjectFromModule(hModule, &psObj);
    if ( eReturnCode != SMSAPI_RETURN_CODE_SUCCESS )
    {
        return hStrReturn;
    }

    hStrReturn = psObj->hRFRevision;

    return hStrReturn;
}

/*****************************************************************************
*
*   hSPLRevision
*
*****************************************************************************/
static STRING_OBJECT hSPLRevision (
    MODULE_OBJECT hModule
        )
{
    STRING_OBJECT hStrReturn = STRING_INVALID_OBJECT;
    MODULE_VERSION_OBJECT_STRUCT *psObj;
    SMSAPI_RETURN_CODE_ENUM eReturnCode;

    eReturnCode = eAccessObjectFromModule(hModule, &psObj);
    if ( eReturnCode != SMSAPI_RETURN_CODE_SUCCESS )
    {
        return hStrReturn;
    }

    hStrReturn = psObj->hSPLRevision;

    return hStrReturn;
}

/*****************************************************************************
*
*   hCapability
*
*****************************************************************************/
static STRING_OBJECT hCapability (
    MODULE_OBJECT hModule
        )
{
    STRING_OBJECT hStrReturn = STRING_INVALID_OBJECT;
    MODULE_VERSION_OBJECT_STRUCT *psObj;
    SMSAPI_RETURN_CODE_ENUM eReturnCode;

    eReturnCode = eAccessObjectFromModule(hModule, &psObj);
    if ( eReturnCode != SMSAPI_RETURN_CODE_SUCCESS )
    {
        return hStrReturn;
    }

    hStrReturn = psObj->hCapability;

    return hStrReturn;
}

/*****************************************************************************
*
*   hModuleHwName
*
*****************************************************************************/
static STRING_OBJECT hModuleHwName (
    MODULE_OBJECT hModule
        )
{
    STRING_OBJECT hStrReturn = STRING_INVALID_OBJECT;
    MODULE_VERSION_OBJECT_STRUCT *psObj;
    SMSAPI_RETURN_CODE_ENUM eReturnCode;

    eReturnCode = eAccessObjectFromModule(hModule, &psObj);
    if ( eReturnCode != SMSAPI_RETURN_CODE_SUCCESS )
    {
        return hStrReturn;
    }

    hStrReturn = psObj->hModuleHwName;

    return hStrReturn;
}

/*****************************************************************************
*
*   eFWUpdateFileVersion
*
*****************************************************************************/
static SMSAPI_RETURN_CODE_ENUM eFWUpdateFileVersion(
    const char *pcFile,
    STRING_OBJECT hFileVersion,
    STRING_OBJECT hFileEarliestVersion,
    STRING_OBJECT hFileLatestVersion,
    STRING_OBJECT hFileTypeID
        )
{
    SMSAPI_RETURN_CODE_ENUM eReturnValue;
    FILE *psFirmwareFile = (FILE *)NULL;

    do
    {
        char acVersionString[MODULE_VERSION_STRING_FORMAT_LEN];
        UN32 un32FileVersion=0,
             un32FileEarliestVersion=0,
             un32FileLatestVersion=0,
             un32FileTypeID=0;
        int iCharsWritten;
        BOOLEAN bReturn;

        if (   (hFileVersion == STRING_INVALID_OBJECT)
            || (hFileEarliestVersion == STRING_INVALID_OBJECT)
            || (hFileLatestVersion == STRING_INVALID_OBJECT)
            || (hFileTypeID == STRING_INVALID_OBJECT)
           )
        {
            eReturnValue = SMSAPI_RETURN_CODE_INVALID_INPUT;
            break;
        }

        psFirmwareFile = fopen(pcFile,"rb");
        if (psFirmwareFile == NULL)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    MODULE_VERSION_OBJECT_NAME
                    ": Unable to open file %s",pcFile);

            eReturnValue = SMSAPI_RETURN_CODE_INVALID_INPUT;
            break;
        }

        eReturnValue = RADIO_eGetFirmwareFileVersion( 
                                        psFirmwareFile,
                                        &un32FileVersion,
                                        &un32FileEarliestVersion,
                                        &un32FileLatestVersion,
                                        &un32FileTypeID);
        if (eReturnValue != SMSAPI_RETURN_CODE_SUCCESS)
        {
            break;
        }

        iCharsWritten = snprintf(acVersionString,
             MODULE_VERSION_STRING_FORMAT_LEN,
             MODULE_VERSION_STRING_FORMAT,
             BYTE2(un32FileTypeID),
             BYTE1(un32FileTypeID),
             BYTE0(un32FileTypeID));
        if (iCharsWritten <= 0)
        {
            eReturnValue = SMSAPI_RETURN_CODE_ERROR;
            break;
        }
        bReturn = STRING.bModifyCStr(hFileTypeID, acVersionString);
        if (bReturn == FALSE)
        {
            eReturnValue = SMSAPI_RETURN_CODE_ERROR;
            break;
        }

        iCharsWritten = snprintf(acVersionString,
             MODULE_VERSION_STRING_FORMAT_LEN,
             MODULE_VERSION_STRING_FORMAT,
             BYTE2(un32FileLatestVersion),
             BYTE1(un32FileLatestVersion),
             BYTE0(un32FileLatestVersion));
        if (iCharsWritten <= 0)
        {
            eReturnValue = SMSAPI_RETURN_CODE_ERROR;
            break;
        }
        bReturn = STRING.bModifyCStr(hFileLatestVersion, acVersionString);
        if (bReturn == FALSE)
        {
            eReturnValue = SMSAPI_RETURN_CODE_ERROR;
            break;
        }

        iCharsWritten = snprintf(acVersionString,
             MODULE_VERSION_STRING_FORMAT_LEN,
             MODULE_VERSION_STRING_FORMAT,
             BYTE2(un32FileEarliestVersion),
             BYTE1(un32FileEarliestVersion),
             BYTE0(un32FileEarliestVersion));
        if (iCharsWritten <= 0)
        {
            eReturnValue = SMSAPI_RETURN_CODE_ERROR;
            break;
        }
        bReturn = STRING.bModifyCStr(hFileEarliestVersion, acVersionString);
        if (bReturn == FALSE)
        {
            eReturnValue = SMSAPI_RETURN_CODE_ERROR;
            break;
        }

        iCharsWritten = snprintf(acVersionString,
             MODULE_VERSION_STRING_FORMAT_LEN,
             MODULE_VERSION_STRING_FORMAT,
             BYTE2(un32FileVersion),
             BYTE1(un32FileVersion),
             BYTE0(un32FileVersion));
        if (iCharsWritten <= 0)
        {
            eReturnValue = SMSAPI_RETURN_CODE_ERROR;
            break;
        }
        bReturn = STRING.bModifyCStr(hFileVersion, acVersionString);
        if (bReturn == FALSE)
        {
            eReturnValue = SMSAPI_RETURN_CODE_ERROR;
            break;
        }

    } while (FALSE);

    if (psFirmwareFile != NULL)
    {
        fclose(psFirmwareFile);
    }

    return eReturnValue;
}

/*****************************************************************************
*
*   n32FPrintf
*
*****************************************************************************/
static N32 n32FPrintf (
        MODULE_OBJECT hModule,
        FILE *psFile
            )
{
	N32 n32Return = 0;
    SMSAPI_RETURN_CODE_ENUM eReturnCode = SMSAPI_RETURN_CODE_ERROR;
    MODULE_VERSION_OBJECT_STRUCT *psObj;
    STRING_OBJECT hPrintStr;

    eReturnCode = eAccessObjectFromModule(hModule, &psObj);
    if ((eReturnCode != SMSAPI_RETURN_CODE_SUCCESS ) || (NULL == psFile))
    {
        return EOF;
    }

    hPrintStr = MODULE_VERSION.hTypeID(hModule);
    if ( hPrintStr != STRING_INVALID_OBJECT )
    {
        n32Return += fprintf(psFile, "hTypeID=%s\n",
            STRING.pacCStr(hPrintStr));
    }
    else
    {
        SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    MODULE_VERSION_OBJECT_NAME
                    ": Unable to get typeID");
    }

    hPrintStr = MODULE_VERSION.hHardwareRevision(hModule);
    if ( hPrintStr != STRING_INVALID_OBJECT )
    {
        n32Return += fprintf(psFile, "hHardwareRevision=%s\n",
                   STRING.pacCStr(hPrintStr));
    }
    else
    {
        SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    MODULE_VERSION_OBJECT_NAME
                    ": Unable to get Hardware revision");
    }
    hPrintStr = MODULE_VERSION.hSoftwareRevision(hModule);
    if ( hPrintStr != STRING_INVALID_OBJECT )
    {
        n32Return += fprintf(psFile, "hSoftwareRevision=%s\n",
            STRING.pacCStr(hPrintStr));
    }
    else
    {
        SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    MODULE_VERSION_OBJECT_NAME
                    ": Unable to get software revision");
    }

    hPrintStr = MODULE_VERSION.hProtocolRevision(hModule);
    if ( hPrintStr != STRING_INVALID_OBJECT )
    {
        n32Return += fprintf(psFile, "hProtocolRevision=%s\n",
            STRING.pacCStr(hPrintStr));
    }
    else
    {
        SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    MODULE_VERSION_OBJECT_NAME
                    ": Unable to get protocol revision");
    }

    hPrintStr = MODULE_VERSION.hBaseBandRevision(hModule);
    if ( hPrintStr != STRING_INVALID_OBJECT )
    {
        n32Return += fprintf(psFile, "hBaseBandRevision=%s\n",
            STRING.pacCStr(hPrintStr));
    }
    else
    {
        SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    MODULE_VERSION_OBJECT_NAME
                    ": Unable to get baseband revision");
    }

    hPrintStr = MODULE_VERSION.hDecoderRevision(hModule);
    if ( hPrintStr != STRING_INVALID_OBJECT )
    {
        n32Return += fprintf(psFile, "hDecoderRevision=%s\n",
            STRING.pacCStr(hPrintStr));
    }
    else
    {
        SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    MODULE_VERSION_OBJECT_NAME
                    ": Unable to get decoder revision");
    }

    hPrintStr = MODULE_VERSION.hRFRevision(hModule);
    if ( hPrintStr != STRING_INVALID_OBJECT )
    {
        n32Return += fprintf(psFile, "hRFRevision=%s\n",
            STRING.pacCStr(hPrintStr));
    }
    else
    {
        SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    MODULE_VERSION_OBJECT_NAME
                    ": Unable to get RF revision");
    }

    hPrintStr = MODULE_VERSION.hSPLRevision(hModule);
    if ( hPrintStr != STRING_INVALID_OBJECT )
    {
        n32Return += fprintf(psFile, "hSPLRevision=%s\n",
            STRING.pacCStr(hPrintStr));
    }
    else
    {
        SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    MODULE_VERSION_OBJECT_NAME
                    ": Unable to get SPL revision");
    }

    hPrintStr = MODULE_VERSION.hCapability(hModule);
    if ( hPrintStr != STRING_INVALID_OBJECT )
    {
        n32Return += fprintf(psFile, "hCapability=%s\n",
            STRING.pacCStr(hPrintStr));
    }
    else
    {
        SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    MODULE_VERSION_OBJECT_NAME
                    ": Unable to get capability");
    }

    hPrintStr = MODULE_VERSION.hModuleHwName(hModule);
    if ( hPrintStr != STRING_INVALID_OBJECT )
    {
        n32Return += fprintf(psFile, "Module Hardware Name=%s\n",
            STRING.pacCStr(hPrintStr));
    }
    else
    {
        SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    MODULE_VERSION_OBJECT_NAME
                    ": Unable to get module hw name");
    }

    return n32Return;
}

/*****************************************************************************
                             FRIEND FUNCTIONS
*****************************************************************************/

/*****************************************************************************
*
*   MODULE_VERSION_vUpdate
*
*****************************************************************************/
void MODULE_VERSION_vUpdate(
    MODULE_OBJECT hModule,
    MODULE_VERSION_INFO_STRUCT const *psModuleVerInfo
        )

{
    MODULE_VERSION_OBJECT_STRUCT *psObj;
    int iCharsWritten;
    SMSAPI_RETURN_CODE_ENUM eReturn = SMSAPI_RETURN_CODE_ERROR;
    BOOLEAN bLocked;


    eReturn = eAccessObjectFromModule(hModule, &psObj);
    if ( eReturn != SMSAPI_RETURN_CODE_SUCCESS )
    {
        SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile,__LINE__,
                            MODULE_VERSION_OBJECT_NAME
                            ": Unable to access MODULE_VERSION_OBJECT");
        return;
    }

    bLocked = SMSO_bLock((SMS_OBJECT)psObj, OSAL_OBJ_TIMEOUT_INFINITE);
    if ( bLocked != TRUE )
    {
        SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                            MODULE_VERSION_OBJECT_NAME
                            ": Unable to lock module version struct");
        return;
    }

    //typeID
    iCharsWritten = snprintf(psObj->acTypeID,
             MODULE_VERSION_STRING_FORMAT_LEN,
             MODULE_VERSION_STRING_FORMAT,
             psModuleVerInfo->un8ModuleTypeIDA,
             psModuleVerInfo->un8ModuleTypeIDB,
             psModuleVerInfo->un8ModuleTypeIDC);
    if ( iCharsWritten > 0 )
    {
        if(psObj->hTypeID != STRING_INVALID_OBJECT)
        {
            STRING_vDestroy(psObj->hTypeID);
            psObj->hTypeID = STRING_INVALID_OBJECT;
        }

        psObj->hTypeID = STRING_hCreateConst(
             psObj->acTypeID,
             MODULE_VERSION_STRING_FORMAT_LEN);
    }

    //hardwareRevision
    iCharsWritten = snprintf(psObj->acHardwareRevision,
             MODULE_VERSION_STRING_FORMAT_LEN,
             MODULE_VERSION_STRING_FORMAT,
             psModuleVerInfo->un8ModuleHWRevA,
             psModuleVerInfo->un8ModuleHWRevB,
             psModuleVerInfo->un8ModuleHWRevC);
    if ( iCharsWritten > 0 )
    {
        if(psObj->hHardwareRevision != STRING_INVALID_OBJECT)
        {
            STRING_vDestroy(psObj->hHardwareRevision);
            psObj->hHardwareRevision = STRING_INVALID_OBJECT;
        }

        psObj->hHardwareRevision = STRING_hCreateConst(
             psObj->acHardwareRevision,
             MODULE_VERSION_STRING_FORMAT_LEN);
    }

    //softwareRevision
    iCharsWritten = snprintf(psObj->acSoftwareRevision,
             MODULE_VERSION_STRING_FORMAT_LEN,
             MODULE_VERSION_STRING_FORMAT,
             psModuleVerInfo->un8ModSWRevMajor,
             psModuleVerInfo->un8ModSWRevMinor,
             psModuleVerInfo->un8ModSWRevInc);
    if ( iCharsWritten > 0 )
    {
        if(psObj->hSoftwareRevision != STRING_INVALID_OBJECT)
        {
            STRING_vDestroy(psObj->hSoftwareRevision);
            psObj->hSoftwareRevision = STRING_INVALID_OBJECT;
        }

        psObj->hSoftwareRevision = STRING_hCreateConst(
             psObj->acSoftwareRevision,
             MODULE_VERSION_STRING_FORMAT_LEN);
    }

    //ProtocolRevision
    iCharsWritten = snprintf(psObj->acProtocolRevision,
             MODULE_VERSION_STRING_FORMAT_LEN,
             MODULE_VERSION_STRING_FORMAT,
             psModuleVerInfo->un8ProtocolRevMajor,
             psModuleVerInfo->un8ProtocolRevMinor,
             psModuleVerInfo->un8ProtocolRevInc);
    if ( iCharsWritten > 0 )
    {
        if(psObj->hProtocolRevision != STRING_INVALID_OBJECT)
        {
            STRING_vDestroy(psObj->hProtocolRevision);
            psObj->hProtocolRevision = STRING_INVALID_OBJECT;
        }

        psObj->hProtocolRevision = STRING_hCreateConst(
             psObj->acProtocolRevision,
             MODULE_VERSION_STRING_FORMAT_LEN);
    }

    //BaseBandRevision
    iCharsWritten = snprintf(psObj->acBaseBandRevision,
             MODULE_VERSION_STRING_FORMAT_LEN,
             MODULE_VERSION_STRING_FORMAT,
             psModuleVerInfo->un8BBRevMajor,
             psModuleVerInfo->un8BBRevMinor,
             psModuleVerInfo->un8BBRevInc);
    if ( iCharsWritten > 0 )
    {
        if(psObj->hBaseBandRevision != STRING_INVALID_OBJECT)
        {
            STRING_vDestroy(psObj->hBaseBandRevision);
            psObj->hBaseBandRevision = STRING_INVALID_OBJECT;
        }

       psObj->hBaseBandRevision = STRING_hCreateConst(
             psObj->acBaseBandRevision,
             MODULE_VERSION_STRING_FORMAT_LEN);
    }

    //DecoderRevision
    iCharsWritten = snprintf(psObj->acDecoderRevision,
             MODULE_VERSION_STRING_FORMAT_LEN,
             MODULE_VERSION_STRING_FORMAT,
             psModuleVerInfo->un8HDecRevMajor,
             psModuleVerInfo->un8HDecRevMinor,
             psModuleVerInfo->un8HDecRevInc);
    if ( iCharsWritten > 0 )
    {
        if(psObj->hDecoderRevision != STRING_INVALID_OBJECT)
        {
            STRING_vDestroy(psObj->hDecoderRevision);
            psObj->hDecoderRevision = STRING_INVALID_OBJECT;
        }

        psObj->hDecoderRevision = STRING_hCreateConst(
             psObj->acDecoderRevision,
             MODULE_VERSION_STRING_FORMAT_LEN);
    }

    //RFRevision
    iCharsWritten = snprintf(psObj->acRFRevision,
             MODULE_VERSION_STRING_FORMAT_LEN,
             MODULE_VERSION_STRING_FORMAT,
             psModuleVerInfo->un8RFRevMajor,
             psModuleVerInfo->un8RFRevMinor,
             psModuleVerInfo->un8RFRevInc);
    if ( iCharsWritten > 0 )
    {
        if(psObj->hRFRevision != STRING_INVALID_OBJECT)
        {
            STRING_vDestroy(psObj->hRFRevision);
            psObj->hRFRevision = STRING_INVALID_OBJECT;
        }

        psObj->hRFRevision = STRING_hCreateConst(
             psObj->acRFRevision,
             MODULE_VERSION_STRING_FORMAT_LEN);
    }

    //SPLRevision
    iCharsWritten = snprintf(psObj->acSPLRevision,
             MODULE_VERSION_STRING_FORMAT_LEN,
             MODULE_VERSION_STRING_FORMAT,
             psModuleVerInfo->un8SPLRevMajor,
             psModuleVerInfo->un8SPLRevMinor,
             psModuleVerInfo->un8SPLRevInc);
    if ( iCharsWritten > 0 )
    {
        if(psObj->hSPLRevision != STRING_INVALID_OBJECT)
        {
            STRING_vDestroy(psObj->hSPLRevision);
            psObj->hSPLRevision = STRING_INVALID_OBJECT;
        }

        psObj->hSPLRevision = STRING_hCreateConst(
             psObj->acSPLRevision,
             MODULE_VERSION_STRING_FORMAT_LEN);
    }

    //Capability
    iCharsWritten = snprintf(psObj->acCapability,
             MODULE_VERSION_STRING_FORMAT_CAPABILITY_LEN,
             MODULE_VERSION_STRING_FORMAT_CAPABILITY,
             BYTE3(psModuleVerInfo->un32Capability),
             BYTE2(psModuleVerInfo->un32Capability),
             BYTE1(psModuleVerInfo->un32Capability),
             BYTE0(psModuleVerInfo->un32Capability));
    if ( iCharsWritten > 0 )
    {
        if(psObj->hCapability != STRING_INVALID_OBJECT)
        {
            STRING_vDestroy(psObj->hCapability);
            psObj->hCapability = STRING_INVALID_OBJECT;
        }

        psObj->hCapability = STRING_hCreateConst(
             psObj->acCapability,
             MODULE_VERSION_STRING_FORMAT_CAPABILITY_LEN);
    }

    OSAL.bMemCpy(psObj->acModuleHwName,psModuleVerInfo->pcModuleHwName,
        MODULE_VERSION_STRING_HW_NAME_LEN);

    if(psObj->hModuleHwName != STRING_INVALID_OBJECT)
    {
        STRING_vDestroy(psObj->hModuleHwName);
        psObj->hModuleHwName = STRING_INVALID_OBJECT;
    }

    psObj->hModuleHwName = STRING_hCreateConst(
             psObj->acModuleHwName,
             MODULE_VERSION_STRING_FORMAT_LEN);

    psObj->sModuleVersionData = *psModuleVerInfo;
    psObj->sModuleVersionData.bModuleVersionAvailable = TRUE;

    SMSO_vUnlock((SMS_OBJECT)psObj);
    return;
}

/*****************************************************************************
*
*   MODULE_VERSION_vInitializeObject
*
*****************************************************************************/
void MODULE_VERSION_vInitializeObject(
    MODULE_VERSION_OBJECT hModuleVersion
        )
{
    //Clean up the STRING_OBJETS associated with
    //the module version object
    vReleaseStrings(hModuleVersion);
	return;
}

/*****************************************************************************
*
*   MODULE_VERSION_hCreate
*
*****************************************************************************/
MODULE_VERSION_OBJECT MODULE_VERSION_hCreate(
    MODULE_OBJECT hOwner
        )
{
    MODULE_VERSION_OBJECT_STRUCT *psObj;
    MODULE_VERSION_OBJECT hModuleVersion;
    BOOLEAN bOwner;

    // Verify ownership of parent object
    bOwner = SMSO_bOwner((SMS_OBJECT)hOwner);

    if (bOwner == FALSE)
    {
        return MODULE_VERSION_INVALID_OBJECT;
    }

    //check to make sure there isn't a Module Version object already allocated
    //If one is already allocated just return it

    // Retrieve the MODULES's MODULE_VERSION_OBJECT instance
    //(if one exists)
    psObj = (MODULE_VERSION_OBJECT_STRUCT *)
                MODULE_hModuleVersion(hOwner);

    if( psObj != NULL )
    {
            return (MODULE_VERSION_OBJECT)psObj;
    }

    // Create an instance of this object
    psObj = (MODULE_VERSION_OBJECT_STRUCT *)
         SMSO_hCreate(
            MODULE_VERSION_OBJECT_NAME,
            sizeof(MODULE_VERSION_OBJECT_STRUCT),
            (SMS_OBJECT)hOwner,
            FALSE);

    if( psObj == NULL )
    {
        // Error!
        return MODULE_VERSION_INVALID_OBJECT;
    }


    hModuleVersion = (MODULE_VERSION_OBJECT)psObj;
    MODULE_VERSION_vInitializeObject(hModuleVersion);
    return hModuleVersion;
}


/*****************************************************************************
*
*   MODULE_VERSION_vDestroy
*
*****************************************************************************/
void MODULE_VERSION_vDestroy(
    MODULE_VERSION_OBJECT hModuleVersion
        )

{
    vReleaseStrings(hModuleVersion);
    // Free object instance
    SMSO_vDestroy((SMS_OBJECT)hModuleVersion);

    return;
}

/*****************************************************************************
*
*   MODULE_VERSION_bGetSwVersion
*
*****************************************************************************/
BOOLEAN MODULE_VERSION_bGetSwVersion(
    MODULE_OBJECT hModule,
    UN32 *un32SwVer
        )
{
    MODULE_VERSION_OBJECT_STRUCT *psObj = NULL;
    SMSAPI_RETURN_CODE_ENUM eReturnCode;
    BOOLEAN bReturn;

    if ( un32SwVer == NULL )
    {
        return FALSE;
    }
    eReturnCode = eAccessObjectFromModule(hModule,
                                      &psObj);

    if ( eReturnCode != SMSAPI_RETURN_CODE_SUCCESS )
    {
        return FALSE;
    }

    bReturn = psObj->sModuleVersionData.bModuleVersionAvailable;
    *un32SwVer =
        MODULE_SW_VERSION_PACK(
            psObj->sModuleVersionData.un8ModSWRevMajor,
            psObj->sModuleVersionData.un8ModSWRevMinor,
            psObj->sModuleVersionData.un8ModSWRevInc);
    return bReturn;
}

/*****************************************************************************
*
*   MODULE_VERSION_bGetModuleType
*
*****************************************************************************/
BOOLEAN MODULE_VERSION_bGetModuleType(
    MODULE_OBJECT hModule,
    UN32 *un32ModuleType
        )
{
    MODULE_VERSION_OBJECT_STRUCT *psObj = NULL;
    SMSAPI_RETURN_CODE_ENUM eReturnCode;
    BOOLEAN bReturn;

    if ( un32ModuleType == NULL )
    {
        return FALSE;
    }

    eReturnCode = eAccessObjectFromModule(hModule,
                                      &psObj);

    if ( eReturnCode != SMSAPI_RETURN_CODE_SUCCESS )
    {
        return FALSE;
    }

    bReturn = psObj->sModuleVersionData.bModuleVersionAvailable;
    *un32ModuleType =
        MODULE_MODULE_VERSION_PACK(
            psObj->sModuleVersionData.un8ModuleTypeIDA,
            psObj->sModuleVersionData.un8ModuleTypeIDB,
            psObj->sModuleVersionData.un8ModuleTypeIDC);

    return bReturn;
}

/*****************************************************************************
*
*   MODULE_VERSION_bGetModuleFeatureMask
*
*****************************************************************************/
BOOLEAN MODULE_VERSION_bGetModuleFeatureMask(
    MODULE_OBJECT hModule,
    RADIO_MODULE_FEATURE_MASK *ptFeatureMask
        )
{
    MODULE_VERSION_OBJECT_STRUCT *psObj = NULL;
    SMSAPI_RETURN_CODE_ENUM eReturnCode;
    BOOLEAN bReturn;

    if ( ptFeatureMask == NULL )
    {
        return FALSE;
    }

    eReturnCode = eAccessObjectFromModule(hModule,
                                          &psObj);

    if ( eReturnCode != SMSAPI_RETURN_CODE_SUCCESS )
    {
        return FALSE;
    }

    bReturn = psObj->sModuleVersionData.bModuleVersionAvailable;
    *ptFeatureMask = psObj->sModuleVersionData.tFeatures;

    return bReturn;
}

/*****************************************************************************
*
*   MODULE_VERSION_bGetProtocolVersion
*
*****************************************************************************/
BOOLEAN MODULE_VERSION_bGetProtocolVersion(
    MODULE_OBJECT hModule,
    UN32 *pun32ProtocolVer
        )
{
    MODULE_VERSION_OBJECT_STRUCT *psObj = NULL;
    SMSAPI_RETURN_CODE_ENUM eReturnCode;
    BOOLEAN bReturn;

    if ( pun32ProtocolVer == NULL )
    {
        return FALSE;
    }
    eReturnCode = eAccessObjectFromModule(hModule,
                                      &psObj);

    if ( eReturnCode != SMSAPI_RETURN_CODE_SUCCESS )
    {
        return FALSE;
    }

    bReturn = psObj->sModuleVersionData.bModuleVersionAvailable;
    *pun32ProtocolVer =
        MODULE_PROTOCOL_VERSION_PACK(
            psObj->sModuleVersionData.un8ProtocolRevMajor,
            psObj->sModuleVersionData.un8ProtocolRevMinor,
            psObj->sModuleVersionData.un8ProtocolRevInc);

    return bReturn;
}

/*****************************************************************************
*
*   MODULE_VERSION_bMaxTuneMix
*
*****************************************************************************/
BOOLEAN MODULE_VERSION_bMaxTuneMix(
    MODULE_OBJECT hModule,
    UN8 *pun8MaxTuneMix
        )
{
    BOOLEAN bReturn;

    SMSAPI_RETURN_CODE_ENUM eReturnCode;
    MODULE_VERSION_OBJECT_STRUCT *psObj;

    if (pun8MaxTuneMix == NULL)
    {
        return FALSE;
    }

    eReturnCode = eAccessObjectFromModule(hModule, &psObj);
    if (eReturnCode != SMSAPI_RETURN_CODE_SUCCESS)
    {
        return FALSE;
    }

    bReturn = psObj->sModuleVersionData.bModuleVersionAvailable;
    *pun8MaxTuneMix = psObj->sModuleVersionData.un8MaxTuneMix;

    return bReturn;
}

/*****************************************************************************
*
*   MODULE_VERSION_bMaxSmartFavs
*
*****************************************************************************/
BOOLEAN MODULE_VERSION_bMaxSmartFavs(
    MODULE_OBJECT hModule,
    UN8 *pun8MaxSmartFavs
        )
{
    BOOLEAN bReturn;

    SMSAPI_RETURN_CODE_ENUM eReturnCode;
    MODULE_VERSION_OBJECT_STRUCT *psObj;

    if (pun8MaxSmartFavs == NULL)
    {
        return FALSE;
    }

    eReturnCode = eAccessObjectFromModule(hModule, &psObj);
    if (eReturnCode != SMSAPI_RETURN_CODE_SUCCESS)
    {
        return FALSE;
    }

    bReturn = psObj->sModuleVersionData.bModuleVersionAvailable;
    *pun8MaxSmartFavs = psObj->sModuleVersionData.un8MaxSmartFavorites;

    return bReturn;
}

/*****************************************************************************
*
*   MODULE_VERSION_bMaxSportsFlash
*
*****************************************************************************/
BOOLEAN MODULE_VERSION_bMaxSportsFlash(
    MODULE_OBJECT hModule,
    UN8 *pun8MaxSportsFlash
        )
{
    BOOLEAN bReturn;

    SMSAPI_RETURN_CODE_ENUM eReturnCode;
    MODULE_VERSION_OBJECT_STRUCT *psObj;

    if (pun8MaxSportsFlash == NULL)
    {
        return FALSE;
    }

    eReturnCode = eAccessObjectFromModule(hModule, &psObj);
    if (eReturnCode != SMSAPI_RETURN_CODE_SUCCESS)
    {
        return FALSE;
    }

    bReturn = psObj->sModuleVersionData.bModuleVersionAvailable;
    *pun8MaxSportsFlash = psObj->sModuleVersionData.un8MaxSportsFlash;

    return bReturn;
}


/*****************************************************************************
*
*   MODULE_VERSION_bMaxTWNow
*
*****************************************************************************/
BOOLEAN MODULE_VERSION_bMaxTWNow(
    MODULE_OBJECT hModule,
    UN8 *pun8MaxTWNow
        )
{
    BOOLEAN bReturn;

    SMSAPI_RETURN_CODE_ENUM eReturnCode;
    MODULE_VERSION_OBJECT_STRUCT *psObj;

    if (pun8MaxTWNow == NULL)
    {
        return FALSE;
    }

    eReturnCode = eAccessObjectFromModule(hModule, &psObj);
    if (eReturnCode != SMSAPI_RETURN_CODE_SUCCESS)
    {
        return FALSE;
    }

    bReturn = psObj->sModuleVersionData.bModuleVersionAvailable;
    *pun8MaxTWNow = psObj->sModuleVersionData.un8MaxTWNow;

    return bReturn;
}

/*****************************************************************************
                             PRIVATE FUNCTIONS
*****************************************************************************/


/*****************************************************************************
*
*   eAccessObjectFromModule
*
*****************************************************************************/
static SMSAPI_RETURN_CODE_ENUM eAccessObjectFromModule(
    MODULE_OBJECT hModule,
    MODULE_VERSION_OBJECT_STRUCT **ppsObj
        )
{
    BOOLEAN bValid;
    SMSAPI_RETURN_CODE_ENUM eReturnCode = SMSAPI_RETURN_CODE_INVALID_INPUT;

    bValid = SMSO_bValid((SMS_OBJECT) hModule);
    if( bValid == TRUE )
    {
        // Retrieve the MODULES's MODULE_VERSION_OBJECT instance
        //(if one exists)
        *ppsObj = (MODULE_VERSION_OBJECT_STRUCT *)
                        MODULE_hModuleVersion(hModule);

        if( *ppsObj == NULL )
        {
            eReturnCode = SMSAPI_RETURN_CODE_ERROR;
        }
        else
        {
            eReturnCode = SMSAPI_RETURN_CODE_SUCCESS;
        }
    }
    return eReturnCode;
}

/*****************************************************************************
*
*   vReleaseStrings
*
*****************************************************************************/
static void vReleaseStrings(
    MODULE_VERSION_OBJECT hModuleVersion
        )
{
    MODULE_VERSION_OBJECT_STRUCT *psObj =
            (MODULE_VERSION_OBJECT_STRUCT *)hModuleVersion;

    if(psObj->hTypeID != STRING_INVALID_OBJECT)
    {
        STRING_vDestroy(psObj->hTypeID);
        psObj->hTypeID = STRING_INVALID_OBJECT;
    }

    if(psObj->hHardwareRevision != STRING_INVALID_OBJECT)
    {
        STRING_vDestroy(psObj->hHardwareRevision);
        psObj->hHardwareRevision = STRING_INVALID_OBJECT;
    }

    if(psObj->hSoftwareRevision != STRING_INVALID_OBJECT)
    {
        STRING_vDestroy(psObj->hSoftwareRevision);
        psObj->hSoftwareRevision = STRING_INVALID_OBJECT;
    }

    if(psObj->hProtocolRevision != STRING_INVALID_OBJECT)
    {
        STRING_vDestroy(psObj->hProtocolRevision);
        psObj->hProtocolRevision = STRING_INVALID_OBJECT;
    }

    if(psObj->hDecoderRevision != STRING_INVALID_OBJECT)
    {
        STRING_vDestroy(psObj->hDecoderRevision);
        psObj->hDecoderRevision = STRING_INVALID_OBJECT;
    }

    if(psObj->hRFRevision != STRING_INVALID_OBJECT)
    {
        STRING_vDestroy(psObj->hRFRevision);
        psObj->hRFRevision = STRING_INVALID_OBJECT;
    }

    if(psObj->hSPLRevision != STRING_INVALID_OBJECT)
    {
        STRING_vDestroy(psObj->hSPLRevision);
        psObj->hSPLRevision = STRING_INVALID_OBJECT;
    }

    if(psObj->hCapability != STRING_INVALID_OBJECT)
    {
        STRING_vDestroy(psObj->hCapability);
        psObj->hCapability = STRING_INVALID_OBJECT;
    }

    if(psObj->hBaseBandRevision != STRING_INVALID_OBJECT)
    {
        STRING_vDestroy(psObj->hBaseBandRevision);
        psObj->hBaseBandRevision = STRING_INVALID_OBJECT;
    }

    if(psObj->hModuleHwName != STRING_INVALID_OBJECT)
    {
        STRING_vDestroy(psObj->hModuleHwName);
        psObj->hModuleHwName = STRING_INVALID_OBJECT;
    }

    return;
}
