/******************************************************************************/
/*                    Copyright (c) Sirius XM Radio, Inc.                     */
/*                            All Rights Reserved                             */
/*         Licensed Materials - Property of Sirius XM Radio, Inc.             */
/******************************************************************************/
/*******************************************************************************
 *
 * DESCRIPTION
 *
 *  This module contains the Object:MODULE implementation for the
 *  Sirius Module Services (SMS)
 *
 ******************************************************************************/
#include <string.h>
#include <stdio.h>

#include "standard.h"
#include "osal.h"

#include "sms_version.h"
#include "sms_api.h"
#include "sms_event.h"
#include "sms_task.h"
#include "sms_obj.h"
#include "sms.h"
#include "srh.h"
#include "srm_obj.h"
#include "decoder_obj.h"
#include "tag_obj.h"
#include "cm.h"
#include "module_version_obj.h"

#include "dataservice_mgr_obj.h"

#include "radio.h"

#include "module_obj.h"
#include "_module_obj.h"

#include "sms_api_debug.h"
static const char *gpacThisFile = __FILE__;

/*****************************************************************************
                             PUBLIC FUNCTIONS
*****************************************************************************/

/*****************************************************************************
*
*   hGet
*
*   This API is used to get access to a physical module within an SRM through
*   a MODULE_OBJECT handle. A module may contain one or more signal decoders
*   as described in Section 8.4. Getting a MODULE will initialize the object
*   and establish SSP communications with the module if the module has not
*   already been started by SMS previously. The application must get a MODULE
*   before any decoders can be obtained as they are derived from their
*   associated MODULE handle returned when this API executes successfully.
*   If the API is able to get the module, a handle to this MODULE is returned.
*   This handle is used for all subsequent APIs which require a MODULE handle.
*   If the call to this API is unsuccessful an invalid handle is returned.
*
*   When the application uses this API it is making a request to get access
*   to this module for operational use, however the MODULE should not be
*   considered "started" until it transitions to the READY state. This can be
*   determined by the application by either configuring to receive the MODULE
*   event (MODULE_OBJECT_EVENT_STATE) or simply polling its state.
*
*   Where:
*       hSRM - A valid SRM handle for which this MODULE object will be derived
*       from. The SRM must be in the READY state before starting an associated
*       MODULE.
*       pacDecoderName - An ASCII Null terminated string. A name is limited
*       to OSAL_MAX_OBJECT_NAME_LENGTH characters (not including the NULL).
*       This name must be mapped to a physical decoder device in the SRM
*       device driver file. The Module to which that mapped decoder belongs
*       is the Module that will be started.
*       tEventRequestMask - A bit mask indicating which events the caller
*       wishes to receive.
*       vEventCallback - A callback function provided by the caller which
*       will be called whenever one or more of the selected events occur.
*       pvEventCallbackArg - The argument provided by the application when
*       the callback function for these events was registered. This is a void
*       pointer which must be cast appropriately within the callback function
*       to the application specific data type it represents.
*
*   Return:
*       A valid MODULE_OBJECT handle is returned on success. Otherwise
*       MODULE_INVALID_OBJECT is returned on error.
*
*****************************************************************************/
static MODULE_OBJECT hGet (
    SRM_OBJECT hSRM,
    const char *pacDecoderName,
    MODULE_EVENT_MASK tEventRequestMask,
    MODULE_OBJECT_EVENT_CALLBACK vEventCallback,
    void *pvEventCallbackArg
        )
{
    BOOLEAN bResult;
    MODULE_ID tId;
    MODULE_OBJECT hModule = MODULE_INVALID_OBJECT;
    MODULE_OBJECT_STRUCT *psObj;

    SMSAPI_DEBUG_vPrint(MODULE_OBJECT_NAME, 4,
        "MODULE.%s(hSRM: %p, pacDecoderName: %s, tEventRequestMask: 0x%X, "
        "vEventCallback: %p, pvEventCallbackArg: %p)\n",
        __FUNCTION__, hSRM, pacDecoderName, tEventRequestMask,
        vEventCallback, pvEventCallbackArg);

    do
    {
        // Verify inputs
        bResult = SMSO_bValid((SMS_OBJECT)hSRM);
        if (FALSE == bResult)
        {
            // Error!
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                MODULE_OBJECT_NAME": Invalid SRM object.");
            break;
        }

        // Verify inputs
        if (NULL == pacDecoderName)
        {
            // Error!
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                MODULE_OBJECT_NAME": pacDecoderName is not specified.");
            break;
        }

        // Is the library initialized? This is probably overkill but
        // it doesn't hurt to check.
        bResult = SMS_bIsInitialized();
        if (FALSE == bResult)
        {
            // Error!
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                MODULE_OBJECT_NAME": SMS is not initialized.");
            break;
        }

        // Based on input DECODER name, find the unique MODULE Id
        // Acquire a unique module id for the named decoder provided
        tId = RADIO_tGetUniqueModuleId(hSRM, pacDecoderName);
        if (MODULE_ID_INVALID == tId)
        {
            // Error!
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                MODULE_OBJECT_NAME": No such decoder exists '%s'.",
                pacDecoderName);
            break;
        }

        // Now we have the Module Id, but has a
        // corresponding MODULE object already been created?  Ask the SRM...
        hModule = SRM_hModule(hSRM, tId);
        if (MODULE_INVALID_OBJECT != hModule)
        {
            // Error!
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                MODULE_OBJECT_NAME": MODULE already exists.");

            // Unlock MODULE. We don't need it anymore.
            SMSO_vUnlock((SMS_OBJECT)hModule);
            hModule = MODULE_INVALID_OBJECT;
            break;
        }

        // Create a new MODULE object and start it
        psObj = psCreateObject(hSRM, pacDecoderName, tId,
            tEventRequestMask, vEventCallback, pvEventCallbackArg);
        if (NULL == psObj)
        {
            // Error!
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                MODULE_OBJECT_NAME": Unable to create MODULE object.");
            break;
        }

        // Save owning SRM handle
        psObj->hSRM = hSRM;

        // Set initial reference counter
        psObj->un16RefCounter = 1;

        // If at this point all goes well, then we can add ourself
        // to the SRM's management of Modules.
        bResult = SRM_bAddModule(hSRM, (MODULE_OBJECT)psObj, tId);
        if (FALSE == bResult)
        {
            // Error!
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                MODULE_OBJECT_NAME": Module could not be added to SRM.");

            // Destroy object
            vDestroyObject(psObj);
            break;
        }

        // Set approprite flag
        psObj->tCurrentModes |= SMS_MODE_PARENT_ADDED;

        // Success! Added to SRM's management.
        SMSAPI_DEBUG_vPrint(MODULE_OBJECT_NAME, 4,
            "%s: Added to SRM.\n", psObj->pacObjectName);

        // Use this one, it's all good.
        hModule = (MODULE_OBJECT)psObj;

        SMSAPI_DEBUG_vPrint(MODULE_OBJECT_NAME, 4,
            "%s: Created MODULE: %p.\n", psObj->pacObjectName, hModule);

    } while (FALSE);

    return hModule;
}

/*****************************************************************************
*
*   vRelease
*
*   This API is used to release access to an existing MODULE object. This API
*   will perform any un-initialization or shutdown required and releases all
*   resources the MODULE consumed if SMS is not using this MODULE for other
*   tasks. This means it will also destroy any derivative objects which still
*   exist in this case. Upon successful completion, this object will no longer
*   be valid.
*
*   Where:
*       hModule - A handle to a valid MODULE object returned from a previous
*       successful call to MODULE.hGet(). This is the handle of the object
*       the caller wishes to stop.
*
*   Return:
*       None.
*
*****************************************************************************/
static void vRelease (
    MODULE_OBJECT hModule
        )
{
    BOOLEAN bResult;

    SMSAPI_DEBUG_vPrint(MODULE_OBJECT_NAME, 4,
        "MODULE.%s(hModule: %p)\n", __FUNCTION__, hModule);

    // Verify SMS Object
    bResult = SMSO_bValid((SMS_OBJECT)hModule);
    if (TRUE == bResult)
    {
        // Start application-initiated object release
        bInitiateObjectRelease((MODULE_OBJECT_STRUCT *)hModule,
            SMS_OBJECT_RELEASE_BY_APP);
    }
    else
    {
        // Error!
        SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
            MODULE_OBJECT_NAME": Invalid MODULE object.");
    }

    return;
}

/*****************************************************************************
*
*   eState
*
*   This API is used to query the current object state. This may be either
*   from a callback function in response to an event, or simply polled by the
*   application. The returned value is the current state of the object.
*
*   Where:
*       hModule - A handle to a valid MODULE object for which the caller
*       wishes to learn the current state of.
*
*   Return:
*       The current object state as a value of MODULE_STATE_ENUM type.
*
*****************************************************************************/
static MODULE_STATE_ENUM eState (
    MODULE_OBJECT hModule
        )
{
    MODULE_STATE_ENUM eState;

    eState = MODULE_eState(hModule);

    return eState;
}

/*****************************************************************************
*
*   eErrorCode
*
*   This API is used to query the current object's error code. This may be
*   either from a callback function in response to an event, or simply polled
*   by the application. The returned value is the current error code of the
*   object. Typically this is required when an object transitions into the
*   ERROR state and the application wishes to know more about why the object
*   is in error.
*
*   Where:
*       hModule - A handle to a valid MODULE Object for which the caller
*       wishes to learn the error code of.
*
*   Return:
*       The current object error code as a value of MODULE_ERROR_CODE_ENUM type.
*
*****************************************************************************/
static MODULE_ERROR_CODE_ENUM eErrorCode (
    MODULE_OBJECT hModule
        )
{
    MODULE_ERROR_CODE_ENUM eErrorCode =
        MODULE_ERROR_CODE_INVALID_OBJECT;
    BOOLEAN bLocked;

    // Verify and lock SMS Object
    bLocked =
        SMSO_bLock((SMS_OBJECT)hModule, OSAL_OBJ_TIMEOUT_INFINITE);
    if(bLocked == TRUE)
    {
        // De-reference object
        MODULE_OBJECT_STRUCT *psObj =
            (MODULE_OBJECT_STRUCT *)hModule;

        // Extract object error code
        eErrorCode = psObj->eErrorCode;

        // Unlock object
        SMSO_vUnlock((SMS_OBJECT)hModule);
    }
    else
    {
        // Error!
        SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
            MODULE_OBJECT_NAME": Unable to lock MODULE.");
    }

    return eErrorCode;
}

/*****************************************************************************
*
*   tEventMask
*
*   This API is used to query or poll the object's current update mask
*   (pending event mask) . This is typically used when the application needs
*   to poll the current pending events which have not been handled since
*   the last call to this API. Normally events are handled in a callback
*   function with the currently active events reported to the callback through
*   an input parameter. However if the application has not defined a callback
*   function, this API must be used to poll for events otherwise the API is not
*   needed. Events returned using this call will automatically be cleared after
*   it returns any active events.
*
*   Note: The events reported by this API are not affected by the event
*   callback. In other words if the application receives events via callback,
*   those events are still reported when polled. This API provides a copy of
*   and therefore independent indication of any pending events.
*
*   Where:
*       hModule - A handle to a valid MODULE object for which the caller
*       wishes to poll for events which have occurred.
*
*   Return:
*       A bit mask representing any events which have occurred since the last
*       poll regardless if they were handled by an event callback or not.
*
*****************************************************************************/
static MODULE_EVENT_MASK tEventMask (
    MODULE_OBJECT hModule
        )
{
    MODULE_EVENT_MASK tEventMask =
        MODULE_OBJECT_EVENT_NONE;
    BOOLEAN bLocked;

    // Verify and lock SMS Object
    bLocked =
        SMSO_bLock((SMS_OBJECT)hModule, OSAL_OBJ_TIMEOUT_INFINITE);
    if(bLocked == TRUE)
    {
        // De-reference object
        MODULE_OBJECT_STRUCT *psObj =
            (MODULE_OBJECT_STRUCT *)hModule;

        tEventMask = SMSU_tMask(&psObj->sEvent);

        // Unlock object
        SMSO_vUnlock((SMS_OBJECT)hModule);
    }
    else
    {
        // Error!
        SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
            MODULE_OBJECT_NAME": Unable to lock MODULE.");
    }

    return tEventMask;
}

/*****************************************************************************
*
*   eModifyRegisteredEventMask
*
*   This API is used to change the event mask that was set with the MODULE's
*   hGet API.  Any events that were pending will be provided to the event
*   callback using the newly modified event mask.
*
*   Where:
*       hModule - A handle to a valid MODULE Object for which the caller
*       wishes to modify the event mask of.
*       tEventMask - A bit mask specifying what changes should be made to
*       the registered event mask.  This is used in conjunction with the
*       action described by eModification.
*       eModification - An enumeration that tells the MODULE object how to
*       use tEventMask to modify the current event mask.  These operations
*       included enabling events, disabling events, and replacing the event
*       mask with the one specified in tEventMask. The
*       SMSAPI_MODIFY_EVENT_MASK_ENUM is described in section 10.2.2.6
*
*   Return:
*       SMSAPI_RETURN_CODE_SUCCESS on success, SMSAPI_RETURN_CODE_ERROR
*       on error.
*
*****************************************************************************/
static SMSAPI_RETURN_CODE_ENUM eModifyRegisteredEventMask (
    MODULE_OBJECT hModule,
    MODULE_EVENT_MASK tEventMask,
    SMSAPI_MODIFY_EVENT_MASK_ENUM eModification
        )
{
    BOOLEAN bValid;
    SMSAPI_RETURN_CODE_ENUM eReturnCode = SMSAPI_RETURN_CODE_ERROR;

    if ( eModification < SMSAPI_MODIFY_EVENT_MASK_START ||
         eModification >= SMSAPI_MODIFY_EVENT_MASK_INVALID )
    {
        return SMSAPI_RETURN_CODE_INVALID_INPUT;
    }

    // Verify SMS Object is valid
    bValid = SMSO_bValid((SMS_OBJECT)hModule);
    if(bValid == TRUE)
    {
        SMS_EVENT_HDL hEvent;
        SMS_EVENT_DATA_UNION *puEventData;
        MODULE_OBJECT_STRUCT *psObj;

        psObj = (MODULE_OBJECT_STRUCT *)hModule;

        // Allocate an event
        hEvent = SMSE_hAllocateEvent(psObj->hEventHdlr,
                    SMS_EVENT_MODIFY_EVENT_MASK,
                    &puEventData,
                    SMS_EVENT_OPTION_NONE);
        if(hEvent != SMS_INVALID_EVENT_HDL)
        {
            BOOLEAN bPosted;

            // Extract event information and populate it
            puEventData->uModule.sModuleEventMask.eModification =
                    eModification;
            puEventData->uModule.sModuleEventMask.tEventMask =
                    tEventMask;

            bPosted = SMSE_bPostEvent(hEvent);
            if (bPosted == FALSE)
            {
                //Error!!
                SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    MODULE_OBJECT_NAME
                    ": Unable to post modify module event mask.");
            }
            else
            {
                eReturnCode = SMSAPI_RETURN_CODE_SUCCESS;
            }
        }
    }
    else
    {
        // Error!
        SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
            MODULE_OBJECT_NAME": Invalid MODULE object.");
    }

    return eReturnCode;
}

/*****************************************************************************
*
*   bUpdate
*
*   This API is used to force an update (trigger all events specified).
*   In response to this API, the object will force the call of the registered
*   application provided callback function. When called, this callback will
*   indicate all events which have been specified (regardless if they actually
*   occurred or not). The application provided callback is then able to process
*   all these events, effectively responding to an â€œupdateâ€� of all registered
*   events.
*
*   Generally this API is only required during development or as a means to
*   test an object. Since all events registered for asynchronously notify the
*   application the application should always be aware of all events which
*   have occurred. However this method provides a way to force an update of
*   specifically requested events.
*
*   Where:
*       hModule - A handle to a valid MODULE object for which the
*           caller wishes to force an update (event callback) of all
*           requested events.
*       tMask - An event mask which indicates which events are to be
*           triggered.
*
*   Return:
*       TRUE on success, otherwise FALSE on error.
*
*****************************************************************************/
static BOOLEAN bUpdate (
    MODULE_OBJECT hModule,
    MODULE_EVENT_MASK tMask
        )
{
    BOOLEAN bPosted = FALSE, bValid;

    // Verify SMS Object is valid
    bValid = SMSO_bValid((SMS_OBJECT)hModule);
    if(bValid == TRUE)
    {
        // De-reference object
        MODULE_OBJECT_STRUCT *psObj =
            (MODULE_OBJECT_STRUCT *)hModule;
        SMS_EVENT_HDL hEvent;
        SMS_EVENT_DATA_UNION *puEventData;

        // Allocate an event
        hEvent =
            SMSE_hAllocateEvent( psObj->hEventHdlr,
                SMS_EVENT_UPDATE_OBJECT, &puEventData, SMS_EVENT_OPTION_NONE );
        if(hEvent != SMS_INVALID_EVENT_HDL)
        {
            // Extract event information and populate it
            puEventData->uModule.sUpdate.tEventMask = tMask;

            bPosted = SMSE_bPostEvent(hEvent);
        }
    }
    else
    {
        // Error!
        SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
            MODULE_OBJECT_NAME": Invalid MODULE object.");
    }

    return bPosted;
}

/*****************************************************************************
*
*   n16GmtOffsetGet
*
*   This API is used to retrieve the current GMT offset from the module.
*   The GMT offset is used to set local system time according to the device's
*   local operating area. The default value is -300 minutes (Eastern time zone).
*
*   Where:
*       hModule - A handle to a valid MODULE object returned from a previous
*       successful call to MODULE.hGet(). This is the handle of the object
*       the caller wishes to retrieve the current GMT offset setting for.
*
*   Return:
*       A signed 16-bit integer representing the GMT offset in minutes.
*
*****************************************************************************/
static N16 n16GmtOffsetGet (
    MODULE_OBJECT hModule
        )
{
    N32 n32GMT_Minutes = 0;

    // Just pull this from OSAL
    OSAL.eTimeGetGMToffset(&n32GMT_Minutes);

    return (N16)n32GMT_Minutes;
}

/*****************************************************************************
*
*   eGmtOffsetSet
*
*   This API is used to set the current GMT offset of the module. The GMT
*   offset is used to set local system time according to the device's local
*   operating area. The default value is -300 minutes (Eastern time zone).
*
*   Where:
*       hModule - A handle to a valid MODULE object returned from a previous
*           successful call to MODULE.hGet(). This is the handle of the object
*           the caller wishes to set the current GMT offset setting for.
*       n16GmtOffset - A signed 16-bit integer representing the GMT offset
*           in minutes.
*
*   Return:
*       SMSAPI_RETURN_CODE_SUCCESS on success or SMSAPI_RETURN_CODE_ERROR
*       on error.
*
*****************************************************************************/
static SMSAPI_RETURN_CODE_ENUM eGmtOffsetSet (
    MODULE_OBJECT hModule,
    N16 n16GmtOffset
        )
{
    SMSAPI_RETURN_CODE_ENUM eReturnCode =
        SMSAPI_RETURN_CODE_ERROR;
    OSAL_RETURN_CODE_ENUM eOsalReturnCode;

    // Just put this into OSAL
    eOsalReturnCode = OSAL.eTimeSetGMToffset((N32)n16GmtOffset);
    if(eOsalReturnCode == OSAL_SUCCESS)
    {
        eReturnCode = SMSAPI_RETURN_CODE_SUCCESS;
    }

    return eReturnCode;
}

/*****************************************************************************
*
*   bDaylightSavingsTimeObservedGet
*
*   This API is used to retrieve the current Daylight Savings Time (DST)
*   observance from the module. The DST observance is used to set local
*   system time according to the device's local operating area. The default
*   value is TRUE (observed).
*
*   Where:
*       hModule - A handle to a valid MODULE object returned from a previous
*       successful call to MODULE.hGet(). This is the handle of the object
*       the caller wishes to retrieve the current DST observance setting for.
*
*   Return:
*       TRUE if DST correction is enabled and FALSE if disabled.
*
*****************************************************************************/
static BOOLEAN bDaylightSavingsTimeObservedGet (
    MODULE_OBJECT hModule
        )
{
    // We do not allow module application of DST observance.
    return FALSE;
}

/*****************************************************************************
*
*   eDaylightSavingsTimeObservedSet
*
*   This API is used to set the current Daylight Savings Time (DST) observance
*   of the module. The DST observance is used to set local system time
*   according to the device's local operating area. The default value is
*   TRUE (observed).
*
*   Where:
*       hModule - A handle to a valid MODULE object returned from a
*           previous successful call to MODULE.hGet(). This is the handle of the
*           object the caller wishes to set the current DST observance setting
*           for.
*       bDaylightSavingsTimeObserved - A Boolean value representing if DST in
*       the local area is observed or not.
*
*   Return:
*       SMSAPI_RETURN_CODE_SUCCESS on success or SMSAPI_RETURN_CODE_ERROR
*       on error.
*
*****************************************************************************/
static SMSAPI_RETURN_CODE_ENUM eDaylightSavingsTimeObservedSet (
    MODULE_OBJECT hModule,
    BOOLEAN bDaylightSavingsTimeObserved
        )
{
    // We don't allow anyone to turn this on
    return SMSAPI_RETURN_CODE_UNSUPPORTED_API;
}

/*****************************************************************************
*
*   hESN
*
*   This API is used to retrieve the Sirius-XM Id or ESN number from the
*   physical module this object represents. The ESN is a unique ASCCII string
*   for all modules.
*
*   Where:
*       hModule - A handle to a valid MODULE object returned from a previous
*       successful call to MODULE.hGet(). This is the handle of the object
*       the caller wishes to get the ESN for.
*
*   Return:
*       A valid STRING object handle which represents the ESN string.
*       Otherwise STRING_INVALID_HANDLE is returned on error.
*
*****************************************************************************/
static STRING_OBJECT hESN (
    MODULE_OBJECT hModule
        )
{
    STRING_OBJECT hString =
        STRING_INVALID_OBJECT;
    BOOLEAN bLocked;

    // Verify and lock SMS Object
    bLocked =
        SMSO_bLock((SMS_OBJECT)hModule, OSAL_OBJ_TIMEOUT_INFINITE);
    if(bLocked == TRUE)
    {
        // De-reference object
        MODULE_OBJECT_STRUCT *psObj =
            (MODULE_OBJECT_STRUCT *)hModule;

        // Get string handle
        hString = psObj->sCache.hESN;

        // Unlock object
        SMSO_vUnlock((SMS_OBJECT)hModule);
    }
    else
    {
        // Error!
        SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
            MODULE_OBJECT_NAME": Unable to lock MODULE.");
    }

    return hString;
}

/*****************************************************************************
*
*   eSubscription
*
*****************************************************************************/
static SMSAPI_RETURN_CODE_ENUM eSubscription (
    MODULE_OBJECT hModule,
    MODULE_SUBSTATUS_ENUM *peStatus,
    MODULE_SUBSTATUS_REASON_CODE *ptReasonCode,
    UN32 *pun32UTCSuspendDate,
    STRING_OBJECT hReasonText,
    STRING_OBJECT hPhoneNumber,
    BOOL *pbIsAudioSubscribed
        )
{
    SMSAPI_RETURN_CODE_ENUM eReturnCode = SMSAPI_RETURN_CODE_SUCCESS;
    BOOLEAN bLocked;
    STRING_OBJECT hString;

    // Lock SMS Object
    bLocked =
        SMSO_bLock((SMS_OBJECT)hModule, OSAL_OBJ_TIMEOUT_INFINITE);
    if(bLocked == TRUE)
    {
        if (peStatus != NULL)
        {
            // Get status
            *peStatus = RADIO_eSubStatus(hModule, DECODER_INVALID_OBJECT);
        }

        if (ptReasonCode != NULL)
        {
            // Get reason code
            *ptReasonCode = RADIO_tReasonCode(hModule, DECODER_INVALID_OBJECT);
        }

        if (pun32UTCSuspendDate != NULL)
        {
            // Get the date
            *pun32UTCSuspendDate = RADIO_un32SuspendDate(hModule, DECODER_INVALID_OBJECT);
        }

        if (hReasonText != STRING_INVALID_OBJECT)
        {
            // Get the text
            hString = RADIO_hReasonText(hModule);
            if (hString != STRING_INVALID_OBJECT)
            {
                STRING.tCopy(hString, hReasonText);
            }
        }

        if (hPhoneNumber != STRING_INVALID_OBJECT)
        {
            // Get the number
            hString = RADIO_hPhoneNumber(hModule);
            if (hString != STRING_INVALID_OBJECT)
            {
                STRING.tCopy(hString, hPhoneNumber);
            }
        }

        if (pbIsAudioSubscribed != NULL)
        {
            *pbIsAudioSubscribed = RADIO_bIsAudioSubscribed(hModule);
        }

        // Unlock object
        SMSO_vUnlock((SMS_OBJECT)hModule);
    }
    else
    {
        // Error!
        SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
            MODULE_OBJECT_NAME": Unable to lock MODULE.");

        eReturnCode = SMSAPI_RETURN_CODE_INVALID_INPUT;
    }

    return eReturnCode;
}

/*****************************************************************************
*
*   eFirmwareUpdate
*
*****************************************************************************/
static SMSAPI_RETURN_CODE_ENUM eFirmwareUpdate (
    MODULE_OBJECT hModule,
    const char *pcFirmwareFileName
        )
{
    SMSAPI_RETURN_CODE_ENUM eReturnCode = SMSAPI_RETURN_CODE_SUCCESS;
    BOOLEAN bLocked;

    if ( hModule == MODULE_INVALID_OBJECT )
    {
        return SMSAPI_RETURN_CODE_INVALID_INPUT;
    }

    do
    {
        UN32 un32NumberDecoderItems=0;
        OSAL_RETURN_CODE_ENUM eOsalReturnCode;
        MODULE_OBJECT_STRUCT *psObj = (MODULE_OBJECT_STRUCT *)hModule;

        bLocked = SMSO_bLock((SMS_OBJECT)hModule, OSAL_OBJ_TIMEOUT_INFINITE);
        if ( bLocked != TRUE )
        {
            // Error!
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                MODULE_OBJECT_NAME": Unable to lock MODULE.");

            eReturnCode = SMSAPI_RETURN_CODE_ERROR;
            break;
        }

        // Can only launch firmware update out of ready state.
        if ( psObj->eState == MODULE_STATE_UPDATING )
        {
            // If we are in the updating state,
            // fw update must already be in progress
            eReturnCode =
                SMSAPI_RETURN_CODE_MODULE_FWUPDATE_ALREADY_IN_PROGRESS;
            break;
        }
        else if ( psObj->eState != MODULE_STATE_READY )
        {
            // FW update not allowed from this state
            eReturnCode = SMSAPI_RETURN_CODE_API_NOT_ALLOWED;
            break;
        }

        // Ensure there are no decoders allocated
        eOsalReturnCode = OSAL.eLinkedListItems(psObj->hDecoderList,
                    &un32NumberDecoderItems);
        if (( eOsalReturnCode != OSAL_SUCCESS )
            || (un32NumberDecoderItems >0)
           )
        {
            eReturnCode = SMSAPI_RETURN_CODE_API_NOT_ALLOWED;
            break;
        }
        SMSO_vUnlock((SMS_OBJECT)hModule);
        bLocked = FALSE;

        eReturnCode = RADIO_eFirmwareUpdate(hModule, pcFirmwareFileName);
        if ( eReturnCode == SMSAPI_RETURN_CODE_SUCCESS)
        {
            vUpdateState(psObj, MODULE_STATE_UPDATING);
        }

    } while(FALSE);

    if (bLocked==TRUE)
    {
        SMSO_vUnlock((SMS_OBJECT)hModule);
    }

    return eReturnCode;
}

/*****************************************************************************
*
*   eUpdateProgress
*
*****************************************************************************/
static SMSAPI_RETURN_CODE_ENUM eUpdateProgress (
    MODULE_OBJECT hModule,
    UN8 *pun8UpdateProgress
        )
{
    BOOLEAN bOwner = FALSE;
    SMSAPI_RETURN_CODE_ENUM eReturnCode = SMSAPI_RETURN_CODE_ERROR;
    MODULE_OBJECT_STRUCT *psObj = (MODULE_OBJECT_STRUCT *)hModule;

    if (( hModule == MODULE_INVALID_OBJECT )
        || ( pun8UpdateProgress == NULL ))
    {
        return eReturnCode;
    }
    bOwner = SMSO_bOwner((SMS_OBJECT)hModule);
    if (bOwner == TRUE)
    {
        *pun8UpdateProgress = psObj->un8UpdateProgress;
        eReturnCode = SMSAPI_RETURN_CODE_SUCCESS;
    }

    return eReturnCode;
}

/*****************************************************************************
*
*   eEventData
*
*****************************************************************************/
static SMSAPI_RETURN_CODE_ENUM eEventData (
    MODULE_OBJECT hModule,
    MODULE_EVENT_DATA_STRUCT *psEventData
        )
{

    SMSAPI_RETURN_CODE_ENUM eReturn = SMSAPI_RETURN_CODE_INVALID_INPUT;

    // verify input
    if (psEventData != NULL)
    {
        BOOLEAN bOwner;

        bOwner = SMSO_bOwner((SMS_OBJECT)hModule);
        if (bOwner == TRUE)
        {
            MODULE_OBJECT_STRUCT *psObj = (MODULE_OBJECT_STRUCT *)hModule;

            // copy the info
            *psEventData = psObj->sEventData;
            eReturn = SMSAPI_RETURN_CODE_SUCCESS;

            // since the data was retreived, clear it out
            psObj->sEventData = gsDefaultEventData;
        }
        else
        {
            eReturn = SMSAPI_RETURN_CODE_NOT_OWNER;
        }
    }

    return eReturn;
}

/*****************************************************************************
*
*   ePackage
*
*****************************************************************************/
static SMSAPI_RETURN_CODE_ENUM ePackage (
    MODULE_OBJECT hModule,
    MODULE_PACKGE_CMD_ENUM eCmd,
    UN8 un8PackageIndex,
    UN8 *pun8Resp,
    size_t tRespSize,
    size_t *ptBytesRxd
        )
{
    SMSAPI_RETURN_CODE_ENUM eReturn = SMSAPI_RETURN_CODE_ERROR;
    BOOLEAN bLocked;
    size_t tBytesFromRadio = 0;

    // Check for invalid input parameters
    if ( ((pun8Resp == NULL) && (eCmd == MODULE_PACKGE_CMD_SELECT)) ||
         (eCmd >= MODULE_PACKGE_CMD_INVALID) )
    {
        return SMSAPI_RETURN_CODE_INVALID_INPUT;
    }

    // Did the caller want the num bytes received?
    // If not, just use our own.
    if (ptBytesRxd == NULL)
    {
        // Use ours
        ptBytesRxd = &tBytesFromRadio;
    }

    // Lock SMS Object
    bLocked =
        SMSO_bLock((SMS_OBJECT)hModule, OSAL_OBJ_TIMEOUT_INFINITE);
    if(bLocked == TRUE)
    {
        MODULE_OBJECT_STRUCT *psObj = (MODULE_OBJECT_STRUCT *)hModule;

        // Remove existing package data and prepare for a
        // new response to be popped in there.
        if(psObj->sCache.sPackage.pun8Data != NULL)
        {
            // Free up existing memory, just to be sure.
            OSAL.vMemoryFree((void*)psObj->sCache.sPackage.pun8Data);
            psObj->sCache.sPackage.pun8Data = NULL;
            psObj->sCache.sPackage.tNumBytes = 0;
        }

        // Make sure mutex is taken
        OSAL.eSemTake(psObj->sCache.sPackage.hMutex, OSAL_OBJ_TIMEOUT_NONE);

        // Send package command
        eReturn = RADIO_ePackage(
            hModule,
            eCmd,
            un8PackageIndex
                );

        // Unlock object
        SMSO_vUnlock((SMS_OBJECT)hModule);
    }

    // Did package command get sent successfully?
    if(eReturn == SMSAPI_RETURN_CODE_SUCCESS)
    {
        // Now we wait for the package data and copy it...
        *ptBytesRxd = tCopyPackageData(
            (MODULE_OBJECT_STRUCT *)hModule,
            5000, // 5-second timeout
            pun8Resp, tRespSize
                );

        // We successfully sent the Select cmd, we need to send
        // out a Query. This makes the chipset update its notion of
        // the ArrayHash
        if (eCmd == MODULE_PACKGE_CMD_SELECT)
        {
            // Lock SMS Object
            bLocked =
                SMSO_bLock((SMS_OBJECT)hModule, OSAL_OBJ_TIMEOUT_INFINITE);
            if(bLocked == TRUE)
            {
                MODULE_OBJECT_STRUCT *psObj =
                    (MODULE_OBJECT_STRUCT *)hModule;

                // Make sure mutex is taken
                OSAL.eSemTake(psObj->sCache.sPackage.hMutex,
                    OSAL_OBJ_TIMEOUT_NONE);

                // Send package command
                eReturn = RADIO_ePackage(
                    hModule,
                    MODULE_PACKGE_CMD_QUERY,
                    un8PackageIndex
                        );

                // Unlock object
                SMSO_vUnlock((SMS_OBJECT)hModule);

                // Now we wait for the package indication but do not
                // really copy its data
                (void) tCopyPackageData(
                    (MODULE_OBJECT_STRUCT *)hModule,
                    5000, // 5-second timeout
                    NULL, 0 );
            }
        }
    }

    return eReturn;
}

/*****************************************************************************
                             FRIEND FUNCTIONS
*****************************************************************************/

/*******************************************************************************
*
*   MODULE_bRelease
*
*   Release SMS internal access to this MODULE
*   This may cause object destruction
*
*******************************************************************************/
BOOLEAN MODULE_bRelease (
    MODULE_OBJECT hModule,
    SMS_OBJECT_RELEASE_INITIATOR_ENUM eInitiator
        )
{
    BOOLEAN bResult;
    MODULE_OBJECT_STRUCT *psObj = (MODULE_OBJECT_STRUCT *)hModule;

    SMSAPI_DEBUG_vPrint(MODULE_OBJECT_NAME, 4,
        "%s(hModule: %p, eInitiator: %d)\n",
        __FUNCTION__, hModule, eInitiator);

    // Verify input
    if (SMS_OBJECT_RELEASE_BY_APP == eInitiator)
    {
        // Error!
        SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
            MODULE_OBJECT_NAME": Incorrect release request.");
        return FALSE;
    }

    // Verify SMS Object
    bResult = SMSO_bValid((SMS_OBJECT)hModule);
    if (FALSE == bResult)
    {
        // Error!
        SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
            MODULE_OBJECT_NAME": Invalid MODULE object.");
        return FALSE;
    }

    // Initiate object release
    bResult = bInitiateObjectRelease(psObj, eInitiator);
    if (FALSE == bResult)
    {
        return FALSE;
    }

    return TRUE;
}

/*****************************************************************************
*
*   MODULE_eState
*
*   This API is used to query the current object state. This may be either
*   from a callback function in response to an event, or simply polled by the
*   application. The returned value is the current state of the object.
*
*   Where:
*       hModule - A handle to a valid MODULE object for which the caller
*       wishes to learn the current state of.
*
*   Return:
*       The current object state as a value of MODULE_STATE_ENUM type.
*
*****************************************************************************/
MODULE_STATE_ENUM MODULE_eState (
    MODULE_OBJECT hModule
        )
{
    MODULE_STATE_ENUM eState =
        MODULE_STATE_INVALID;
    BOOLEAN bLocked;

    // Verify and lock SMS Object
    bLocked =
        SMSO_bLock((SMS_OBJECT)hModule, OSAL_OBJ_TIMEOUT_INFINITE);
    if(bLocked == TRUE)
    {
        // De-reference object
        MODULE_OBJECT_STRUCT *psObj =
            (MODULE_OBJECT_STRUCT *)hModule;

        // Pass thru state
        eState = psObj->eState;

        // Unlock object
        SMSO_vUnlock((SMS_OBJECT)hModule);
    }
    else
    {
        // Error!
        SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
            MODULE_OBJECT_NAME": Unable to lock MODULE.");
    }

    return eState;
}

/*****************************************************************************
*
*   MODULE_bAssociate
*
*   This function is called by SMS to indicate the MODULE has been registered
*   and SMS is ready to begin managing this MODULE. In response this will cause
*   the MODULE to initialize itself and transition to the READY state.
*
*****************************************************************************/
BOOLEAN MODULE_bAssociate (
    MODULE_OBJECT hModule,
    SRM_OBJECT hSRM,
    const char *pacName,
    FILE *psDevice
        )
{
    BOOLEAN bResult = FALSE;
    MODULE_OBJECT_STRUCT *psObj = (MODULE_OBJECT_STRUCT *)hModule;
    SMS_EVENT_HDL hEvent;
    SMS_EVENT_DATA_UNION *puEventData;

    SMSAPI_DEBUG_vPrint(MODULE_OBJECT_NAME, 4,
        "%s(hModule: %p, hSRM: %p, pacName: %s, psDevice: %p)\n",
        __FUNCTION__, hModule, hSRM, pacName, psDevice);

    do
    {
        // Verify SMS Object is valid
        bResult = SMSO_bValid((SMS_OBJECT)hModule);
        if (FALSE == bResult)
        {
            // Error!
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                MODULE_OBJECT_NAME": Invalid MODULE object.");
            break;
        }

        // Allocate an event
        hEvent =
            SMSE_hAllocateEvent(psObj->hEventHdlr,
                SMS_EVENT_ASSOCIATE, &puEventData,
                SMS_EVENT_OPTION_URGENT);
        if (SMS_INVALID_EVENT_HDL == hEvent)
        {
            // Error!
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                MODULE_OBJECT_NAME": Unable to allocate event.");
            bResult = FALSE;
            break;
        }

        // Populate event information
        puEventData->uModule.sAssociate.hSRM = hSRM;
        puEventData->uModule.sAssociate.pacName = pacName;
        puEventData->uModule.sAssociate.psDevice = psDevice;

        // Post event
        bResult = SMSE_bPostEvent(hEvent);
        if (FALSE == bResult)
        {
            // Error!
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                MODULE_OBJECT_NAME": Unable to post ASSOCIATE event.");
            break;
        }

        SMSAPI_DEBUG_vPrint(MODULE_OBJECT_NAME, 4,
            "%s: ASSOCIATE event is posted.\n", psObj->pacObjectName);

    } while (FALSE);

    return bResult;
}

/*****************************************************************************
*
*   MODULE_bUnassociate
*
*****************************************************************************/
BOOLEAN MODULE_bUnassociate (
    MODULE_OBJECT hModule
        )
{
    BOOLEAN bResult = FALSE;
    MODULE_OBJECT_STRUCT *psObj = (MODULE_OBJECT_STRUCT *)hModule;

    SMSAPI_DEBUG_vPrint(MODULE_OBJECT_NAME, 4,
        "%s(hModule: %p)\n", __FUNCTION__, hModule);

    do
    {
        // Verify SMS Object is valid
        bResult = SMSO_bValid((SMS_OBJECT)hModule);
        if (FALSE == bResult)
        {
            // Error!
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                MODULE_OBJECT_NAME": Invalid MODULE object.");
            break;
        }

        bResult = SMSE_bPostSignal(psObj->hEventHdlr,
            SMS_EVENT_UNASSOCIATE, SMS_EVENT_OPTION_NONE);
        if (FALSE == bResult)
        {
            // Error!
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                MODULE_OBJECT_NAME": Unable to post UNASSOCIATE event.");
            break;
        }

        SMSAPI_DEBUG_vPrint(MODULE_OBJECT_NAME, 4,
            "%s: UNASSOCIATE event is posted.\n", psObj->pacObjectName);

    } while (FALSE);

    return bResult;
}

/*****************************************************************************
*
*   MODULE_bAddDecoder
*
*   This is called by the DECODER to inform the MODULE
*   it is should be added to the MODULE list.
*
*****************************************************************************/
BOOLEAN MODULE_bAddDecoder (
    MODULE_OBJECT hModule,
    DECODER_OBJECT hDecoder
        )
{
    BOOLEAN bResult = FALSE;

    SMSAPI_DEBUG_vPrint(MODULE_OBJECT_NAME, 4,
        "%s(hModule: %p, hDecoder: %p)\n",
        __FUNCTION__, hModule, hDecoder);

    // Verify the Module object is owned by caller
    bResult = SMSO_bLock((SMS_OBJECT)hModule, OSAL_OBJ_TIMEOUT_INFINITE);
    if (FALSE == bResult)
    {
        // Error!
        SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
            MODULE_OBJECT_NAME": Unable to lock MODULE.");
        return FALSE;
    }

    do
    {
        MODULE_OBJECT_STRUCT *psObj = (MODULE_OBJECT_STRUCT *)hModule;
        OSAL_RETURN_CODE_ENUM eReturnCode = OSAL_ERROR;
        MODULE_DECODER_ENTRY_STRUCT *psDecoder;
        OSAL_LINKED_LIST_ENTRY hEntry = OSAL_INVALID_LINKED_LIST_ENTRY;

        // Check validity of the object being added
        bResult = SMSO_bValid((SMS_OBJECT)hDecoder);
        if (FALSE == bResult)
        {
            // Error!
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                MODULE_OBJECT_NAME": Invalid DECODER object.");
            break;
        }

        // Allocate entry
        psDecoder = (MODULE_DECODER_ENTRY_STRUCT *)SMSO_hCreate(
            MODULE_OBJECT_NAME":Decoder Entry",
            sizeof(MODULE_DECODER_ENTRY_STRUCT), (SMS_OBJECT)psObj, FALSE);
        if (NULL == psDecoder)
        {
            // Error!
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                MODULE_OBJECT_NAME": Could not allocate DECODER entry.\n");
            bResult = FALSE;
            break;
        }

        // Populate entry
        psDecoder->bValid = TRUE;
        psDecoder->hDecoder = hDecoder;
        psDecoder->bEngineeringDataEnabled = FALSE;

        // Add this DECODER to the list. This list allows only unique
        // entries. If one exists this means this DECODER already exists.
        eReturnCode = OSAL.eLinkedListAdd(psObj->hDecoderList,
            &hEntry, (void *)psDecoder);
        if (OSAL_SUCCESS != eReturnCode)
        {
            // Error!
            SMSO_vDestroy((SMS_OBJECT)psDecoder);

            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                MODULE_OBJECT_NAME": Could not add Decoder entry: %d (%s)\n",
                eReturnCode, OSAL.pacGetReturnCodeName(eReturnCode));

            bResult = FALSE;
            break;
        }

        SMSAPI_DEBUG_vPrint(MODULE_OBJECT_NAME, 4,
            "%s: Added DECODER entry.\n", psObj->pacObjectName);

        bResult = bAssociateWithDecoder(psObj, hDecoder);
        if (FALSE == bResult)
        {
            break;
        }

    } while (FALSE);

    // Unlock MODULE
    SMSO_vUnlock((SMS_OBJECT)hModule);

    return bResult;
}

/*****************************************************************************
*
*   MODULE_bRemoveDecoder
*
*   This is called by the DECODER to have it remove itself.
*
*****************************************************************************/
BOOLEAN MODULE_bRemoveDecoder (
    MODULE_OBJECT hModule,
    DECODER_OBJECT hDecoder
        )
{
    BOOLEAN bResult = FALSE;
    MODULE_OBJECT_STRUCT *psObj = (MODULE_OBJECT_STRUCT *)hModule;
    SMS_EVENT_HDL hEvent;
    SMS_EVENT_DATA_UNION *puEventData;

    SMSAPI_DEBUG_vPrint(MODULE_OBJECT_NAME, 4,
        "%s(hModule: %p, hDecoder: %p)\n",
        __FUNCTION__, hModule, hDecoder);

    do
    {
        // Verify SMS Object is valid
        bResult = SMSO_bValid((SMS_OBJECT)hModule);
        if (FALSE == bResult)
        {
            // Error!
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                MODULE_OBJECT_NAME": Invalid MODULE object.");
            break;
        }

        // Allocate an event
        hEvent =
            SMSE_hAllocateEvent( psObj->hEventHdlr,
                SMS_EVENT_REMOVE_DECODER, &puEventData,
                SMS_EVENT_OPTION_NONE);
        if (SMS_INVALID_EVENT_HDL == hEvent)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                MODULE_OBJECT_NAME": Unable to allocate event.");
            bResult = FALSE;
            break;
        }

        // Populate event information
        puEventData->uModule.sRemoveDecoder.hDecoder = hDecoder;

        // Post event
        bResult = SMSE_bPostEvent(hEvent);
        if (FALSE == bResult)
        {
            // Error!
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                MODULE_OBJECT_NAME": Unable to post REMOVE_DECODER event.");
            break;
        }

        SMSAPI_DEBUG_vPrint(MODULE_OBJECT_NAME, 4,
            "%s: REMOVE_DECODER event is posted.\n", psObj->pacObjectName);

    } while (FALSE);

    return bResult;
}

/*****************************************************************************
*
*   MODULE_eModifyEngineeringData
*
*   This is called by the DECODER to have engineering data status modified
*
*****************************************************************************/
SMSAPI_RETURN_CODE_ENUM MODULE_eModifyEngineeringData (
    MODULE_OBJECT hModule,
    DECODER_OBJECT hDecoder,
    SMSAPI_MODIFY_EVENT_MASK_ENUM eModification
        )
{
    BOOLEAN bValid;
    SMSAPI_RETURN_CODE_ENUM eReturnCode = SMSAPI_RETURN_CODE_ERROR;

    // Verify SMS Object is valid
    bValid = SMSO_bValid((SMS_OBJECT)hModule);
    if(bValid == TRUE)
    {
        MODULE_OBJECT_STRUCT *psObj =
            (MODULE_OBJECT_STRUCT *)hModule;
        SMS_EVENT_HDL hEvent;
        SMS_EVENT_DATA_UNION *puEventData;

        // Allocate an event
        hEvent =
            SMSE_hAllocateEvent( psObj->hEventHdlr,
                SMS_EVENT_MODIFY_ENGINEERING_DATA_STATE, &puEventData,
                SMS_EVENT_OPTION_NONE
                    );
        if(hEvent != SMS_INVALID_EVENT_HDL)
        {
            BOOLEAN bPosted;

            // Extract event information and populate it
            puEventData->uModule.sModModifyEngData.hDecoder = hDecoder;
            puEventData->uModule.sModModifyEngData.eModification = eModification;

            bPosted = SMSE_bPostEvent(hEvent);
            if(bPosted == TRUE)
            {
                // All is well.
                eReturnCode = SMSAPI_RETURN_CODE_SUCCESS;
            }
        }

    }

    return eReturnCode;
}

/*****************************************************************************
*
*   MODULE_bSetError
*
*****************************************************************************/
BOOLEAN MODULE_bSetError (
    MODULE_OBJECT hModule,
    MODULE_ERROR_CODE_ENUM eErrorCode
        )
{
    BOOLEAN bValid, bPosted = FALSE;

    // Verify SMS Object is valid
    bValid = SMSO_bValid((SMS_OBJECT)hModule);
    if(bValid == TRUE)
    {
        MODULE_OBJECT_STRUCT *psObj =
            (MODULE_OBJECT_STRUCT *)hModule;
        SMS_EVENT_HDL hEvent;
        SMS_EVENT_DATA_UNION *puEventData;

        // Allocate an event
        hEvent =
            SMSE_hAllocateEvent( psObj->hEventHdlr,
                SMS_EVENT_ERROR, &puEventData,
                SMS_EVENT_OPTION_NONE
                    );
        if(hEvent != SMS_INVALID_EVENT_HDL)
        {
            // Extract event information and populate it
            puEventData->uModule.sError.eErrorCode = eErrorCode;

            // TODO: BREAKPOINT
            bPosted = SMSE_bPostEvent(hEvent);
        }
    }

    return bPosted;
}

/*****************************************************************************
 *
 *   MODULE_hAllocateEvent
 *
 *   This API is called by someone who wants to allocate an event from the
 *   specific MODULE specified.
 *
 *****************************************************************************/
SMS_EVENT_HDL MODULE_hAllocateEvent(
    MODULE_OBJECT hModule,
    SMS_EVENT_TYPE_ENUM eEvent,
    EVENT_OPTIONS_TYPE tOptions,
    SMS_EVENT_DATA_UNION ** ppuEventData
        )
{
    BOOLEAN bValid;
    SMS_EVENT_HDL hEvent = SMS_INVALID_EVENT_HDL;

    // Verify SMS Object is valid
    bValid = SMSO_bValid((SMS_OBJECT)hModule);
    if(bValid == TRUE)
    {
        MODULE_OBJECT_STRUCT *psObj = (MODULE_OBJECT_STRUCT *)hModule;

        // Allocate an event
        hEvent = SMSE_hAllocateEvent(
            psObj->hEventHdlr, eEvent, ppuEventData, tOptions);
    }

    return hEvent;
}

/*****************************************************************************
*
*   MODULE_bDispatchDecoderEvent
*
*   This API is called by a MODULE context to tell the MODULE to post
*   this event to each of this MODULE's DECODERs based on the provided
*   compare function. Note the caller must be already in the MODULE context.
*
*****************************************************************************/
BOOLEAN MODULE_bDispatchDecoderEvent (
    MODULE_OBJECT hModule,
    SMS_EVENT_TYPE_ENUM eType,
    void *pvEventArg,
    DISPATCH_DECODER_EVENT_CALLBACK bDispatch,
    BOOLEAN bOnlyIfEngineeringDataRequested
        )
{
    BOOLEAN bSuccess = TRUE;
    MODULE_OBJECT_STRUCT *psObj = (MODULE_OBJECT_STRUCT *)hModule;
    OSAL_RETURN_CODE_ENUM eReturnCode;
    MODULE_DECODER_DISPATCHER_STRUCT sDispatchRequest;
    SMS_DESCRIPTOR_DISPATCH_STRUCT sDispatcher;

    do
    {
        // Check if we are running in the object's context, if not we cannot
        // do this.
        bSuccess = SMSO_bOwner((SMS_OBJECT)hModule);
        if (FALSE == bSuccess)
        {
            // Error!
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                MODULE_OBJECT_NAME": Not owner.");
            break;
        }

        // Populate dispatch request
        sDispatchRequest.eType = eType;
        sDispatchRequest.pvEventArg = pvEventArg;
        sDispatchRequest.bDispatch = bDispatch;
        sDispatchRequest.bOnlyIfEngineeringDataRequested =
            bOnlyIfEngineeringDataRequested;

        // Populate dispatcher
        sDispatcher.eType = SMS_EVENT_DISPATCH;
        sDispatcher.pvEventArg = (void *)&sDispatchRequest;
        sDispatcher.bDispatch = bDispatchToAll;

        // Allow every attached/active Decoder to handle this event...
        eReturnCode = OSAL.eLinkedListIterate(
            psObj->hDecoderList, bDecoderIterator,
            &sDispatcher
                );
        if (eReturnCode != OSAL_SUCCESS &&
            eReturnCode != OSAL_NO_OBJECTS)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                MODULE_OBJECT_NAME": SMS_EVENT_DISPATCH (%s).",
                OSAL.pacGetReturnCodeName(eReturnCode));

            // Error!
            bSuccess = FALSE;
            break;
        }

    } while (FALSE);

    return bSuccess;
}

/*******************************************************************************
*
*   MODULE_hDecoder
*
*   This function is used to obtain a DECODER handle for a specified DECODER
*   by name. If the DECODER exists, and is being managed by SMS (and this
*   MODULE) then a valid DECODER_OBJECT handle is returned. Otherwise an
*   invalid object handle is returned (indicating one does not already exist).
*
*******************************************************************************/
DECODER_OBJECT MODULE_hDecoder (
    MODULE_OBJECT hModule,
    const char *pacDecoderName
        )
{
    BOOLEAN bOwner;
    DECODER_OBJECT hDecoder = DECODER_INVALID_OBJECT;

    // Verify the MODULE object is owned by caller
    bOwner =
        SMSO_bOwner((SMS_OBJECT)hModule);
    if (bOwner == TRUE)
    {
        OSAL_RETURN_CODE_ENUM eReturnCode;
        MODULE_OBJECT_STRUCT *psObj =
            (MODULE_OBJECT_STRUCT *)hModule;
        OSAL_LINKED_LIST_ENTRY hEntry =
            OSAL_INVALID_LINKED_LIST_ENTRY;

        // Locate the list node which matches this DECODER's name
        // This list is kept in order of increasing DECODER handle value.
        // So this search is always linear from the first element.
        eReturnCode = OSAL.eLinkedListLinearSearch(
                psObj->hDecoderList,
                &hEntry,
                n16CompareDecoderName,
                (void *)pacDecoderName
                    );
        if(eReturnCode == OSAL_SUCCESS) // found
        {
            MODULE_DECODER_ENTRY_STRUCT *psDecoder;

            // extract and return the DECODER entry
            psDecoder = (MODULE_DECODER_ENTRY_STRUCT *)
                            OSAL.pvLinkedListThis(hEntry);
            if(NULL == psDecoder)
            {
                hDecoder = DECODER_INVALID_OBJECT;
            }
            else
            {
                hDecoder = psDecoder->hDecoder;
            }
        }
    }

    return hDecoder;
}

/*******************************************************************************
*
*   MODULE_pacName
*
*   This function simply returns the constant C-style null terminated character
*   array which contains the MODULE's name as provided by the MODULE object
*   handle.
*
*******************************************************************************/
const char *MODULE_pacName (
    MODULE_OBJECT hModule
        )
{
    BOOLEAN bValid;

    bValid = SMSO_bValid((SMS_OBJECT)hModule);
    if (bValid == TRUE)
    {
        MODULE_OBJECT_STRUCT *psObj =
            (MODULE_OBJECT_STRUCT *)hModule;

        return psObj->pacObjectName;
    }

    return NULL;
}

/*****************************************************************************
*
*   MODULE_bRadioReady
*
*****************************************************************************/
BOOLEAN MODULE_bRadioReady (
    MODULE_OBJECT hModule,
    STI_HDL hSTI
        )
{
    BOOLEAN bPosted = FALSE;
    MODULE_OBJECT_STRUCT *psObj = (MODULE_OBJECT_STRUCT *)hModule;
    SMS_EVENT_HDL hEvent;
    SMS_EVENT_DATA_UNION *puEventData;

    // Allocate an event
    hEvent =
        SMSE_hAllocateEvent( psObj->hEventHdlr,
            SMS_EVENT_RADIO_READY, &puEventData,
            SMS_EVENT_OPTION_URGENT);
    if (SMS_INVALID_EVENT_HDL != hEvent)
    {
        // Populate event information
        puEventData->uModule.sRadioReady.hSTI = hSTI;

        bPosted = SMSE_bPostEvent(hEvent);
        if (TRUE == bPosted)
        {
            SMSAPI_DEBUG_vPrint(MODULE_OBJECT_NAME, 4,
                "%s: RADIO_READY is posted.\n",
                psObj->pacObjectName);
        }
        else
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                MODULE_OBJECT_NAME": Unable to post event.");
        }
    }
    else
    {
        SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
            MODULE_OBJECT_NAME": Unable to allocate event.");
    }

    return bPosted;
}

/*****************************************************************************
*
*   MODULE_bRadioReleased
*
*****************************************************************************/
BOOLEAN MODULE_bRadioReleased (
    MODULE_OBJECT hModule
        )
{
    BOOLEAN bPosted = FALSE;
    MODULE_OBJECT_STRUCT *psObj = (MODULE_OBJECT_STRUCT *)hModule;

    bPosted = SMSE_bPostSignal(psObj->hEventHdlr,
        SMS_EVENT_RADIO_RELEASED, SMS_EVENT_OPTION_NONE);
    if (TRUE == bPosted)
    {
        SMSAPI_DEBUG_vPrint(MODULE_OBJECT_NAME, 4,
            "%s: RADIO_RELEASED is posted.\n", psObj->pacObjectName);
    }
    else
    {
        SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
            MODULE_OBJECT_NAME": Unable to post event.");
    }

    return bPosted;
}

/*****************************************************************************
*
*   MODULE_bSetRadioSpecificData
*
*****************************************************************************/
BOOLEAN MODULE_bSetRadioSpecificData (
    MODULE_OBJECT hModule,
    RADIO_PRIVATE_DATA_OBJECT hData
        )
{
    BOOLEAN bOwner = FALSE;

    // Verify SMS Object
    bOwner = SMSO_bOwner((SMS_OBJECT)hModule);
    if(bOwner == TRUE)
    {
        // De-reference object
        MODULE_OBJECT_STRUCT *psObj =
            (MODULE_OBJECT_STRUCT *)hModule;

        // Set radio specific data handle
        psObj->hRadioData = hData;
    }

    return bOwner;
}

/*****************************************************************************
*
*   MODULE_hGetRadioSpecificData
*
*****************************************************************************/
RADIO_PRIVATE_DATA_OBJECT MODULE_hGetRadioSpecificData (
    MODULE_OBJECT hModule
        )
{
    BOOLEAN bOwner;
    RADIO_PRIVATE_DATA_OBJECT hData =
        RADIO_PRIVATE_DATA_INVALID_OBJECT;

    // Verify SMS Object and pointer validity
    bOwner = SMSO_bOwner((SMS_OBJECT)hModule);
    if(bOwner == TRUE)
    {
        // De-reference object
        MODULE_OBJECT_STRUCT *psObj =
            (MODULE_OBJECT_STRUCT *)hModule;

        // Get radio specific data object
        hData = psObj->hRadioData;
    }

    return hData;
}

/*****************************************************************************
*
*   MODULE_hGetTag
*
*****************************************************************************/
TAG_OBJECT MODULE_hGetTag(
    MODULE_OBJECT hModule
        )
{
    BOOLEAN bValid;
    TAG_OBJECT hTag = TAG_INVALID_OBJECT;
    MODULE_OBJECT_STRUCT *psObj = (MODULE_OBJECT_STRUCT *)hModule;

    // Verify the object
    bValid = SMSO_bValid( (SMS_OBJECT)hModule );

    if (bValid == TRUE)
    {
        hTag = psObj->hTag;
    }

    return hTag;
}

/*****************************************************************************
*
*   MODULE_bResetDecoder
*
*****************************************************************************/
BOOLEAN MODULE_bResetDecoder (
    MODULE_OBJECT hModule,
    DECODER_OBJECT hDecoder
        )
{
    BOOLEAN bValid, bPosted = FALSE;

    SMSAPI_DEBUG_vPrint(MODULE_OBJECT_NAME, 4,
        "%s(hModule: %p, hDecoder: %p)\n",
        __FUNCTION__, hModule, hDecoder);

    // Verify SMS Object is valid
    bValid = SMSO_bValid((SMS_OBJECT)hModule);
    if(bValid == TRUE)
    {
        MODULE_OBJECT_STRUCT *psObj =
            (MODULE_OBJECT_STRUCT *)hModule;
        SMS_EVENT_HDL hEvent;
        SMS_EVENT_DATA_UNION *puEventData;

        // Allocate an event
        hEvent =
            SMSE_hAllocateEvent( psObj->hEventHdlr,
                SMS_EVENT_RESET, &puEventData,
                SMS_EVENT_OPTION_URGENT
                    );
        if(hEvent != SMS_INVALID_EVENT_HDL)
        {
            // Extract event information and populate it
            puEventData->uModule.sReset.hDecoder = hDecoder;

            bPosted = SMSE_bPostEvent(hEvent);
        }
    }

    return bPosted;
}

/*****************************************************************************
*
*   MODULE_bReset
*
*****************************************************************************/
BOOLEAN MODULE_bReset (
    MODULE_OBJECT hModule
        )
{
    BOOLEAN bResult = FALSE;

    SMSAPI_DEBUG_vPrint(MODULE_OBJECT_NAME, 4,
        "%s(hModule: %p)\n", __FUNCTION__, hModule);

    bResult = MODULE_bResetDecoder(hModule, DECODER_INVALID_OBJECT);

    return bResult;
}

/*****************************************************************************
*
*   MODULE_eInitialize
*
*****************************************************************************/
MODULE_ERROR_CODE_ENUM MODULE_eInitialize (
    MODULE_OBJECT hModule
        )
{
    MODULE_ERROR_CODE_ENUM eErrorCode;
    BOOLEAN bValid, bPosted = FALSE;

    SMSAPI_DEBUG_vPrint(MODULE_OBJECT_NAME, 4,
        "%s(hModule: %p)\n", __FUNCTION__, hModule);

    // Verify SMS Object is valid
    bValid = SMSO_bValid((SMS_OBJECT)hModule);
    if(bValid == TRUE)
    {
        MODULE_OBJECT_STRUCT *psObj =
            (MODULE_OBJECT_STRUCT *)hModule;

        bPosted = SMSE_bPostSignal(psObj->hEventHdlr, SMS_EVENT_INITIALIZE,
            SMS_EVENT_OPTION_NONE);
        if(bPosted == TRUE)
        {
            // All is well
            eErrorCode = MODULE_ERROR_CODE_NONE;
        }
        else
        {
            // Error!
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                MODULE_OBJECT_NAME": Unable to initialize this Module.");
            eErrorCode = MODULE_ERROR_CODE_EVENT_POST_ERROR;
        }
    }
    else
    {
        // Error!
        SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
            MODULE_OBJECT_NAME": Invalid MODULE.");
        eErrorCode = MODULE_ERROR_CODE_INVALID_OBJECT;
    }

    return eErrorCode;
}

/*****************************************************************************
 *
 *  MODULE_vUpdateSubStatus
 *
 *****************************************************************************/
void MODULE_vUpdateSubStatus(
    MODULE_OBJECT hModule)
{
    BOOLEAN bValid;

    // Verify SMS Object is valid
    bValid = SMSO_bValid((SMS_OBJECT)hModule);
    if(bValid == TRUE)
    {
        MODULE_OBJECT_STRUCT *psObj = (MODULE_OBJECT_STRUCT *)hModule;

        // Set event mask
        SMSU_tUpdate(&psObj->sEvent, MODULE_OBJECT_EVENT_SUBSTATUS);
    }
    return;
}

/*****************************************************************************
 *
 *  MODULE_hGetModuleVersion
 *
 *****************************************************************************/
MODULE_VERSION_OBJECT MODULE_hModuleVersion (
    MODULE_OBJECT hModule
        )
{
    BOOLEAN bValid;
    MODULE_VERSION_OBJECT hModuleVersion = MODULE_VERSION_INVALID_OBJECT;

    bValid = SMSO_bValid((SMS_OBJECT)hModule);
    if (bValid == TRUE)
    {
        MODULE_OBJECT_STRUCT *psObj;

        psObj = (MODULE_OBJECT_STRUCT *)hModule;
        hModuleVersion = psObj->hModuleVersion;
    }
    return hModuleVersion;
}

/*****************************************************************************
*
*  MODULE_vUpdateIRSupport
*
*  The function can be called only from the MODULE context
*
*****************************************************************************/
void MODULE_vUpdateIRSupport(
    MODULE_OBJECT hModule,
    BOOLEAN bValue
        )
{
    BOOLEAN bOwner;

    // Check the object validity
    bOwner = SMSO_bOwner((SMS_OBJECT) hModule);
    if (bOwner == TRUE)
    {
        MODULE_OBJECT_STRUCT *psObj = (MODULE_OBJECT_STRUCT*)hModule;

        // Update the support of IR
        psObj->sSupport.bIRSupported = bValue;
    }

    return;
}

/*****************************************************************************
*
*  MODULE_vUpdateAdvancedIRSupport
*
*  The function can be called only from the MODULE context
*
*****************************************************************************/
void MODULE_vUpdateAdvancedIRSupport(
    MODULE_OBJECT hModule,
    BOOLEAN bValue
        )
{
    BOOLEAN bOwner;

    // Check the object validity
    bOwner = SMSO_bOwner((SMS_OBJECT) hModule);
    if (bOwner == TRUE)
    {
        MODULE_OBJECT_STRUCT *psObj = (MODULE_OBJECT_STRUCT*)hModule;

        // Update the support of Advanced IR
        psObj->sSupport.bAdvancedIRSupported = bValue;
    }

    return;
}

/*****************************************************************************
*
*  MODULE_bIsIRSupported
*
*****************************************************************************/
BOOLEAN MODULE_bIsIRSupported(
    MODULE_OBJECT hModule
        )
{
    BOOLEAN bValid, bResult = FALSE;

    // Check the object validity
    bValid = SMSO_bValid((SMS_OBJECT) hModule);
    if (bValid == TRUE)
    {
        MODULE_OBJECT_STRUCT *psObj = (MODULE_OBJECT_STRUCT*)hModule;
        RADIO_MODULE_FEATURE_MASK tFeatures;
        BOOLEAN bSuccess;

        // Update the value from the internals
        bResult = psObj->sSupport.bIRSupported;

        // Ask the Radio what features we support
        bSuccess = MODULE_VERSION_bGetModuleFeatureMask(hModule, &tFeatures);
        if (bSuccess == TRUE)
        {
            // Find out if Advanced IR is supported via the mask
            if ((tFeatures & RADIO_MODULE_FEATURE_IR)
                != RADIO_MODULE_FEATURE_IR)
            {
                // Make sure the value is FALSE
                bResult = FALSE;
            }
        }
    }

    return bResult;
}

/*****************************************************************************
*
*  MODULE_vUpdateProgress
*
*****************************************************************************/
void MODULE_vUpdateProgress(
    MODULE_OBJECT hModule,
    UN8 un8UpdateProgress,
    SMSAPI_RETURN_CODE_ENUM eFWUpdateFileLoadError
        )
{
    BOOLEAN bOwner = FALSE;
    MODULE_OBJECT_STRUCT *psObj = (MODULE_OBJECT_STRUCT *)hModule;

    SMSAPI_DEBUG_vPrint(MODULE_OBJECT_NAME, 5,
        "%s: Update progress: %u.\n",
        psObj->pacObjectName, un8UpdateProgress);

    if ( hModule == MODULE_INVALID_OBJECT )
    {
        return;
    }

    bOwner = SMSO_bOwner((SMS_OBJECT)hModule);
    if (bOwner == TRUE)
    {
        // load was successfull kicking off new update
        if (   (un8UpdateProgress != psObj->un8UpdateProgress)
            || (eFWUpdateFileLoadError != psObj->eFWUpdateFileLoadError)
           )
        {
            psObj->un8UpdateProgress = un8UpdateProgress;
            psObj->eFWUpdateFileLoadError = eFWUpdateFileLoadError;
            SMSU_tUpdate(&psObj->sEvent, MODULE_OBJECT_EVENT_UPDATE_PROGRESS);
        }
    }
    return;
}

/*****************************************************************************
*
*   MODULE_bSetPkgData
*
*****************************************************************************/
BOOLEAN MODULE_bSetPkgData (
    MODULE_OBJECT hModule,
    size_t tNumBytes,
    UN8 const *pun8Data
        )
{
    BOOLEAN bOk = FALSE;

    // verify input
    if (pun8Data != NULL)
    {
        // Verify SMS Object
        bOk = SMSO_bOwner((SMS_OBJECT)hModule);

        if (bOk == TRUE)
        {
             // De-reference object
            MODULE_OBJECT_STRUCT *psObj =
                (MODULE_OBJECT_STRUCT *)hModule;

            // Copy the data to module maintained struct
            if(psObj->sCache.sPackage.pun8Data != NULL) // old?
            {
                // Get rid of the old one.
                OSAL.vMemoryFree((void *)psObj->sCache.sPackage.pun8Data);
                psObj->sCache.sPackage.tNumBytes = 0;
                psObj->sCache.sPackage.pun8Data = NULL;
            }

            // Put in the new one
            psObj->sCache.sPackage.tNumBytes = tNumBytes;
            psObj->sCache.sPackage.pun8Data = pun8Data;

            // Signal that data is ready
            OSAL.eSemGive(psObj->sCache.sPackage.hMutex);
        }
    }

    return bOk;
}

/*****************************************************************************
*
*   MODULE_bSetEventData
*
*****************************************************************************/
BOOLEAN MODULE_bSetEventData (
    MODULE_OBJECT hModule,
    MODULE_EVENT_DATA_STRUCT *psData
        )
{
    BOOLEAN bOk = FALSE;

    // verify input
    if (psData != NULL)
    {
        // Verify SMS Object
        bOk = SMSO_bOwner((SMS_OBJECT)hModule);

        if (bOk == TRUE)
        {
             // De-reference object
            MODULE_OBJECT_STRUCT *psObj =
                (MODULE_OBJECT_STRUCT *)hModule;

            // Copy the data to module maintained struct
            psObj->sEventData = *psData;
        }
    }

    return bOk;
}

/*****************************************************************************
 *
 *   MODULE_bOverlaySupported
 *
 *****************************************************************************/
BOOLEAN MODULE_bOverlaySupported(
    MODULE_OBJECT hModule
        )
{
    BOOLEAN bReturn = FALSE, bValid;
    RADIO_MODULE_FEATURE_MASK tFeatures;

    // Verify SMS Object is valid
    bValid = SMSO_bValid((SMS_OBJECT)hModule);
    if(bValid == TRUE)
    {
        // Ask the Radio what features we support
        bReturn = MODULE_VERSION_bGetModuleFeatureMask(hModule, &tFeatures);
        if (bReturn == TRUE)
        {
            //Find out if overlay is supported via the mask
            if ((tFeatures & RADIO_MODULE_FEATURE_OVERLAY)
                 == RADIO_MODULE_FEATURE_OVERLAY
               )
            {
                bReturn = TRUE;
            }
            else
            {
                bReturn = FALSE;
            }
        }
        else
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                MODULE_OBJECT_NAME": Unable to determine overlay support.");
        }
    }
    return bReturn;
}

/*****************************************************************************
 *
 *   MODULE_bAdvancedIrSuported
 *
 *****************************************************************************/
BOOLEAN MODULE_bIsAdvancedIrSupported (
    MODULE_OBJECT hModule
        )
{
    BOOLEAN bSupported = FALSE, bValid;
    RADIO_MODULE_FEATURE_MASK tFeatures;

    // Verify SMS object is valid
    bValid = SMSO_bValid((SMS_OBJECT)hModule);
    if (bValid == TRUE)
    {
        BOOLEAN bSuccess;
        MODULE_OBJECT_STRUCT *psObj = (MODULE_OBJECT_STRUCT *)hModule;

        // Update the value from the internals
        bSupported = psObj->sSupport.bAdvancedIRSupported;

        // Ask the Radio what features we support
        bSuccess = MODULE_VERSION_bGetModuleFeatureMask(hModule, &tFeatures);
        if (bSuccess == TRUE)
        {
            // Find out if Advanced IR is supported via the mask
            if ((tFeatures & RADIO_MODULE_FEATURE_ADVANCED_IR)
                    != RADIO_MODULE_FEATURE_ADVANCED_IR)
            {
                bSupported = FALSE;
            }
        }
        else
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                MODULE_OBJECT_NAME": Unable to determine Advanced IR support");
        }
    }

    return bSupported;
}

/*****************************************************************************
 *
 *   MODULE_bSetTime
 *
 *  Note: This function expects time to be given WITH GMT and/or DST
 *  adjustment per configuration.
 *
 *****************************************************************************/
BOOLEAN MODULE_bSetTime (
    MODULE_OBJECT hModule,
    TIME_T tTod,
    UN32 un32AtSeconds,
    UN16 un16AtMsecs
        )
{
    BOOLEAN bSuccess = FALSE;

    // Validate SMS Object
    bSuccess = SMSO_bValid((SMS_OBJECT)hModule);
    if(TRUE == bSuccess)
    {
         // De-reference object
        MODULE_OBJECT_STRUCT *psObj =
            (MODULE_OBJECT_STRUCT *)hModule;

        SMS_EVENT_HDL hEvent;
        SMS_EVENT_DATA_UNION *puEventData;

        // Allocate an event
        hEvent =
            SMSE_hAllocateEvent( psObj->hEventHdlr,
                SMS_EVENT_UPDATE_TIME, &puEventData,
                SMS_EVENT_OPTION_NONE
                    );
        if(hEvent != SMS_INVALID_EVENT_HDL)
        {
            // Extract event information and populate it
            puEventData->uModule.sSetTime.tTod = tTod;
            puEventData->uModule.sSetTime.un32AtSeconds = un32AtSeconds;
            puEventData->uModule.sSetTime.un16AtMsecs = un16AtMsecs;

            // Send event
            bSuccess = SMSE_bPostEvent(hEvent);
            if(FALSE == bSuccess)
            {
                // Error!
                SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    MODULE_OBJECT_NAME": Unable to set time.");
            }
        }
    }

    return bSuccess;
}

/*****************************************************************************
                             PRIVATE FUNCTIONS
*****************************************************************************/

/*****************************************************************************
*
*  bSetTime
*
*****************************************************************************/
static BOOLEAN bSetTime (
    MODULE_OBJECT_STRUCT const *psObj,
    SMS_EVENT_MODULE_SET_TIME_STRUCT const *psSetTime
        )
{
    BOOLEAN bSuccess = FALSE;

    // Time offsets
    UN64 un64OffsetMsec;
    UN32 un32OffsetSeconds;

    // Time as reported by the system (timestamp)
    UN32 un32NowSeconds = 0;
    UN16 un16NowMsecs = 0;

    OSAL_RETURN_CODE_ENUM eReturnCode;

    // Extract current uptime (timestamp)
    // The purpose is to time stamp when we received
    // this message and compensate for any offset
    // induced by the time we actually handle this.
    OSAL.vTimeUp(&un32NowSeconds, &un16NowMsecs);

    // Compute number of milliseconds since this was set
    un64OffsetMsec = (((UN64)un32NowSeconds * 1000) + (UN64)un16NowMsecs) -
        (((UN64)psSetTime->un32AtSeconds * 1000) + (UN64)psSetTime->un16AtMsecs);

    // Compute number of seconds (floor)
    un32OffsetSeconds = (UN32)(un64OffsetMsec / 1000);

    // Add any compensation
    un32NowSeconds = (UN32)psSetTime->tTod + un32OffsetSeconds;

    // Now handle this reported time
    eReturnCode = OSAL.eTimeSet(un32NowSeconds);
    if(OSAL_SUCCESS == eReturnCode)
    {
        // New time has been set
        bSuccess = TRUE;
    }
    else
    {
        // Error!
        SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
            MODULE_OBJECT_NAME": Unable to set time.");
    }

    return bSuccess;
}

/*****************************************************************************
*
*  psCreateObject
*
*****************************************************************************/
static MODULE_OBJECT_STRUCT *psCreateObject (
    SRM_OBJECT hSRM,
    const char *pacDecoderName,
    MODULE_ID tId,
    MODULE_EVENT_MASK tEventRequestMask,
    MODULE_OBJECT_EVENT_CALLBACK vEventCallback,
    void *pvEventCallbackArg
        )
{
    MODULE_OBJECT_STRUCT *psObj = NULL;
    const char *pacDriverName;

    // Acquire the driver name associated with the SRM
    pacDriverName = SRM_pacDriverName(hSRM);
    if(pacDriverName != NULL)
    {
        char *pacTempText;
        int iNum;

        // Compute length of the required name (10 = maximum UNIT32)
        iNum = strlen(MODULE_OBJECT_NAME) + 1 + 10 + 1;

        // Allocate enough memory for this name
        pacTempText = (char *)OSAL.pvMemoryAllocate(
            MODULE_OBJECT_NAME":Text", ++iNum, FALSE);
        if(pacTempText != NULL)
        {
            // Now actually construct the name
            snprintf( pacTempText, iNum,
                MODULE_OBJECT_NAME".%u", tId);

            // Create an instance of this object
            psObj = (MODULE_OBJECT_STRUCT *)
                SMSO_hCreate(
                    pacTempText,
                    sizeof(MODULE_OBJECT_STRUCT) + iNum,
                    SMS_INVALID_OBJECT, // Parent
                    TRUE); // Lock
            if(psObj != NULL)
            {
                TAG_OBJECT hParentTag;
                SMSAPI_RETURN_CODE_ENUM eResult;

                // Initialize object defaults
                *psObj = gsObjectDefaults;

                // Keep a copy of the object, id and decoder name.
                OSAL.bMemCpy((char *)(psObj + 1), pacTempText, iNum);

                // Point to the specific object name.
                psObj->pacObjectName = (const char *)(psObj + 1);

                // Populate module id
                psObj->tId = tId;

                // Initial STI connection
                psObj->hSTI = STI_INVALID_HDL;

                // set event data to default values
                psObj->sEventData = gsDefaultEventData;

                // Acquire parent tag
                hParentTag = SRM_hGetTag(hSRM);

                // Acquire MODULE tag
                eResult = TAG_eGet(
                              MODULE_OBJECT_NAME,
                              hParentTag,
                              &psObj->hTag,
                              psObj->pacObjectName,
                              TRUE); // create if it doesn't exist
                if((eResult == SMSAPI_RETURN_CODE_SUCCESS) ||
                                (eResult == SMSAPI_RETURN_CODE_CFG_NO_PARENT))
                {
                    SMS_TASK_CONFIGURATION_STRUCT sConfig =
                        gsModuleTaskConfiguration;

                    // Configure module name
                    sConfig.pacName = psObj->pacObjectName;

                    // Initialize asynchronous update configuration and
                    // save inputs.
                    SMSU_vInitialize(
                        &psObj->sEvent,
                        psObj,
                        MODULE_OBJECT_EVENT_ALL,
                        tEventRequestMask,
                        (SMSAPI_OBJECT_EVENT_CALLBACK)vEventCallback,
                        pvEventCallbackArg
                            );

                    // Install SMS Task for this module
                    // Note, successful installation of this task will cause the
                    // provided SMS object to be owned by the SMS task.
                    psObj->hServicesTask =
                        SMST_hInstall(
                            (SMS_OBJECT)psObj,
                            &sConfig,
                            (SMS_OBJECT_EVENT_HANDLER_PROTOTYPE)vEventHandler,
                            &psObj->hEventHdlr
                                );
                    if(psObj->hServicesTask != SMS_INVALID_TASK_HANDLE)
                    {
                        // All is well.
                        SMSAPI_DEBUG_vPrint(MODULE_OBJECT_NAME, 4,
                            "%s: Task installed.\n",
                            psObj->pacObjectName);
                    }
                    else
                    {
                        // Error!
                        SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                            MODULE_OBJECT_NAME": MODULE task could not be "
                            "installed.");

                        // Destroy the async update configuration
                        // initialized above.
                        SMSU_vDestroy(&psObj->sEvent);

                        // Kill object
                        SMSO_vDestroy((SMS_OBJECT)psObj);
                        psObj = NULL;
                    }
                }
                else
                {
                    // Error!
                    SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                        MODULE_OBJECT_NAME": Cannot get TAG handle.");
                    SMSO_vDestroy((SMS_OBJECT)psObj);
                    psObj = NULL;
                }
            }
            else
            {
                // Error!
                SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    MODULE_OBJECT_NAME": Cannot allocate memory.");
            }

            // Don't need this memory anymore
            OSAL.vMemoryFree(pacTempText);
        }
        else
        {
            // Error!
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                MODULE_OBJECT_NAME": Cannot allocate memory.");
        }
    }
    else
    {
        // Error!
        SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
            MODULE_OBJECT_NAME": Driver name is NULL.");
    }

    return psObj;
}

/*****************************************************************************
*
*  vDestroyObject
*
*****************************************************************************/
static  void vDestroyObject (
    MODULE_OBJECT_STRUCT *psObj
        )
{
    BOOLEAN bOwner;

    // Check if this object is owned by the caller.
    bOwner = SMSO_bIsOwner((SMS_OBJECT)psObj);
    if(bOwner == TRUE)
    {
        // Destroy the async update configuration
        SMSU_vDestroy(&psObj->sEvent);

        // Destroy the  object itself
        SMSO_vDestroy((SMS_OBJECT)psObj);
    }
    else
    {
        // Error!
        SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
            MODULE_OBJECT_NAME": Not owned object cannot be destroyed.");
    }

    return;
}

/*******************************************************************************
*
*   n16CompareDecoders
*
*   This function compares two MODULE_DECODER_DESCRIPTOR_STRUCT and orders
*   them such that descriptors with valid DECODER_OBJECT handles always
*   come first, and then ensures all descriptors are sorted by the
*   capabilities their associated modules support.
*
*   Inputs:
*       pvArg1 (in list), pvArg2 (compare against)
*
*   Outputs:
*       0   - Objects have the same value (equal, error)
*       > 0 - Object1(in list) is greater than (after) Object2
*       < 0 - Object1(in list) is less than (before) Object2
*
*******************************************************************************/
static N16 n16CompareDecoders (
    void *pvObj1,
    void *pvObj2
        )
{
    MODULE_DECODER_ENTRY_STRUCT *psDecoder1 =
                    (MODULE_DECODER_ENTRY_STRUCT *)pvObj1;
    MODULE_DECODER_ENTRY_STRUCT *psDecoder2 =
        (MODULE_DECODER_ENTRY_STRUCT *)pvObj2;
    BOOLEAN bDecoder1Valid;
    BOOLEAN bDecoder2Valid;
    N16 n16Result = N16_MIN;

    if ((psDecoder1 == NULL) || (psDecoder2 == NULL))
    {
        return N16_MIN;
    }

    // Determine the validity of the decoder handles
    bDecoder1Valid = SMSO_bIsValid((SMS_OBJECT)psDecoder1->hDecoder);
    bDecoder2Valid = SMSO_bIsValid((SMS_OBJECT)psDecoder2->hDecoder);

    // First attempt to sort by instantiation
    if (bDecoder1Valid != bDecoder2Valid)
    {
        // If decoder 1 isn't valid, it comes after decoder 2
        if (bDecoder1Valid == FALSE)
        {
            return 1;
        }

        // Otherwise, decoder 2 isn't valid, and it comes after decoder 1
        return -1;
    }
    else
    {
        n16Result = COMPARE(psDecoder1->hDecoder, psDecoder2->hDecoder);
    }

    return n16Result;
}

/*******************************************************************************
*
*   n16CompareDecoderName
*
*******************************************************************************/
static N16 n16CompareDecoderName (
    void *pvArg1,
    void *pvArg2
        )
{
    MODULE_DECODER_ENTRY_STRUCT *psDecoder =
                    (MODULE_DECODER_ENTRY_STRUCT *)pvArg1;
    DECODER_OBJECT hDecoder = psDecoder->hDecoder;
    const char *pacDecoderName2 = (const char *)pvArg2;
    N16 n16Result = N16_MIN;

    if((hDecoder != DECODER_INVALID_OBJECT) && (pacDecoderName2 != NULL))
    {
        const char *pacDecoderName1;

        // Extract this DECODER's DECODER name
        pacDecoderName1 = DECODER_pacName(hDecoder);

        // Check if we have a valid pointer
        if(pacDecoderName1 != NULL)
        {
            int iStrCmpResult;

            // Check if this DECODER's DECODER name matches
            iStrCmpResult = strcmp(pacDecoderName1, pacDecoderName2);

            if (iStrCmpResult != 0)
            {
                n16Result = -1;
            }
            else
            {
                n16Result = 0;
            }
        }
    }

    return n16Result;
}

/*****************************************************************************
*
*   bResetDecoderIterator
*
*****************************************************************************/
static BOOLEAN bResetDecoderIterator (
    void *pvData,
    void *pvArg
        )
{
    BOOLEAN bSuccess;
    MODULE_DECODER_ENTRY_STRUCT *psDecoder =
                    (MODULE_DECODER_ENTRY_STRUCT *)pvData;

    // Reset Decoder...
    bSuccess = DECODER_bReset(psDecoder->hDecoder);
    if(bSuccess == TRUE)
    {
        // Mark this one as in RESET
        psDecoder->bValid = FALSE;
    }
    else
    {
        // Error!
        SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
            MODULE_OBJECT_NAME": Unable to reset DECODER.");
    }

    // Keep iteration going
    return TRUE;
}

/*****************************************************************************
*
*   bAllInResetIterator
*
*****************************************************************************/
static BOOLEAN bAllInResetIterator (
    void *pvData,
    void *pvArg
        )
{
    MODULE_DECODER_ENTRY_STRUCT *psDecoder =
                    (MODULE_DECODER_ENTRY_STRUCT *)pvData;
    MODULE_DECODER_IN_RESET_STRUCT *psInReset =
                    (MODULE_DECODER_IN_RESET_STRUCT *)pvArg;

    // Is the DECODER provided the one we are iterating?
    if(psDecoder->hDecoder == psInReset->hDecoder)
    {
        // Set new state
        psDecoder->bValid = TRUE;
    }

    // If DECODER is in the valid state it is (or has been RESET)
    if(psDecoder->bValid == FALSE)
    {
        // Then all DECODERs are certainly not in RESET
        psInReset->bAllInReset = FALSE;
    }

    // Keep iteration going
    return TRUE;
}

/*******************************************************************************
*
*   bReleaseDecoderIterator
*
*******************************************************************************/
static BOOLEAN bReleaseDecoderIterator (
    void *pvData,
    void *pvArg
        )
{
    MODULE_DECODER_ENTRY_STRUCT * psDecoder =
        (MODULE_DECODER_ENTRY_STRUCT *)pvData;

    DECODER_bRelease(psDecoder->hDecoder, SMS_OBJECT_RELEASE_BY_PARENT);

    // Keep iteration going
    return TRUE;
}

/*****************************************************************************
*
*   bInitializeObject
*
*****************************************************************************/
static BOOLEAN bInitializeObject (
    MODULE_OBJECT_STRUCT *psObj
        )
{
    OSAL_RETURN_CODE_ENUM eReturnCode;
    char acName[OSAL_MAX_OBJECT_NAME_LENGTH_WITH_NULL];

    // Initialize object...

    // Initialize error code
    psObj->eErrorCode = MODULE_ERROR_CODE_NONE;

    // Check if I have already registered for time updates.
    // If we have we don't need to again.
    if(OSAL_TIME_NOTIFICATION_INVALID_OBJECT ==
        psObj->hTimeNotificationHandle)
    {
        // Let's register for time updates
        eReturnCode = OSAL.eTimeSetRegisterNotification(
            &psObj->hTimeNotificationHandle,
            OSAL_TIME_UPDATE_MASK_GMT,
            (OSAL_TIME_UPDATE_HANDLER)vTimeUpdateNotification,
            (void *)psObj
                );
        if(eReturnCode != OSAL_SUCCESS)
        {
            // Humm, can't get time...
            return FALSE;
        }
    }

    // Initialize package data structure
    if(psObj->sCache.sPackage.hMutex == OSAL_INVALID_OBJECT_HDL)
    {
        snprintf(&acName[0], sizeof(acName),
            "%s:PackageSigal", psObj->pacObjectName);

        // Create mutex initially as unavailable
        eReturnCode = OSAL.eSemCreate(
            &psObj->sCache.sPackage.hMutex,
            &acName[0], 0, 1, OSAL_SEM_OPTION_NONE
                );
        if(eReturnCode != OSAL_SUCCESS)
        {
            // Error!
            return FALSE;
        }
    }
    else
    {
        // We already have a mutex, so just make sure it
        // is not available.
        OSAL.eSemTake(psObj->sCache.sPackage.hMutex, OSAL_OBJ_TIMEOUT_NONE);
    }

    if(psObj->sCache.sPackage.pun8Data != NULL)
    {
        OSAL.vMemoryFree((void *)psObj->sCache.sPackage.pun8Data);
        psObj->sCache.sPackage.pun8Data = NULL;
    }
    psObj->sCache.sPackage.tNumBytes = 0;

    // Read mask, thereby clearing it
    SMSU_tMask(&psObj->sEvent);

    // Create the module version object if it doesn't already exist
    if (psObj->hModuleVersion == MODULE_VERSION_INVALID_OBJECT)
    {
        psObj->hModuleVersion =  MODULE_VERSION_hCreate((MODULE_OBJECT)psObj);
    }
    else
    {
        MODULE_VERSION_vInitializeObject(psObj->hModuleVersion);
    }
    // Create decoder descriptor list
    if(psObj->hDecoderList == OSAL_INVALID_OBJECT_HDL)
    {
        snprintf(&acName[0], sizeof(acName),
            "%s:DecoderDescList", psObj->pacObjectName);

        eReturnCode = OSAL.eLinkedListCreate(
            &psObj->hDecoderList,
            &acName[0],
            n16CompareDecoders,
            OSAL_LL_OPTION_CIRCULAR | OSAL_LL_OPTION_UNIQUE
                );
        if(eReturnCode != OSAL_SUCCESS)
        {
            // Error!
            return FALSE;
        }
    }

    // This will cause the configuration file to be read and
    // subsequently adjust any local time settings accordingly to
    // that configuration via an event post back to the MODULE.
    vGetTimeParams(psObj);

    return TRUE;
}

/*****************************************************************************
*
*   vUninitObject
*
*****************************************************************************/
static void vUninitObject (
    MODULE_OBJECT_STRUCT *psObj
        )
{
    SMSAPI_DEBUG_vPrint(MODULE_OBJECT_NAME, 4,
        "%s: Object uninitialization (%s)...\n",
        psObj->pacObjectName,
        psObj->bUninitialized == FALSE ? "currently initialized":
        "already uninitialized");

    // only do if we haven't already uninitialized
    if (psObj->bUninitialized == FALSE)
    {
        OSAL_RETURN_CODE_ENUM eReturnCode;

        // don't call this function again, unless re-initialized
        psObj->bUninitialized = TRUE;

        // Forget ESN
        psObj->sCache.hESN = STRING_INVALID_OBJECT;

        // Uninitialize Activation Package Response

        if(psObj->sCache.sPackage.hMutex != OSAL_INVALID_OBJECT_HDL)
        {
            eReturnCode = OSAL.eSemDelete(psObj->sCache.sPackage.hMutex);
            if(eReturnCode == OSAL_SUCCESS)
            {
                // Invalidate mutex handle
                psObj->sCache.sPackage.hMutex = OSAL_INVALID_OBJECT_HDL;
            }
        }

        if(psObj->sCache.sPackage.pun8Data != NULL)
        {
            OSAL.vMemoryFree((void *)psObj->sCache.sPackage.pun8Data);
            psObj->sCache.sPackage.pun8Data = NULL;
        }

        psObj->sCache.sPackage.tNumBytes = 0;

        // Destroy Module Version object
        if (psObj->hModuleVersion != MODULE_VERSION_INVALID_OBJECT)
        {
            MODULE_VERSION_vDestroy(psObj->hModuleVersion);
            psObj->hModuleVersion = MODULE_VERSION_INVALID_OBJECT;
        }

        // Unregister for any time updates
        if(OSAL_TIME_NOTIFICATION_INVALID_OBJECT !=
            psObj->hTimeNotificationHandle)
        {
            OSAL_RETURN_CODE_ENUM eReturnCode;

            // Unregister
            eReturnCode = OSAL.eTimeSetUnRegisterNotification(
                psObj->hTimeNotificationHandle);
            if(eReturnCode == OSAL_SUCCESS)
            {
                psObj->hTimeNotificationHandle =
                    OSAL_TIME_NOTIFICATION_INVALID_OBJECT;
            }
        }

        // Destroy decoder descriptor list if one exists
        if(psObj->hDecoderList != OSAL_INVALID_OBJECT_HDL)
        {
            // Remove all entries in the linked list
            OSAL.eLinkedListRemoveAll( psObj->hDecoderList,
                (OSAL_LL_RELEASE_HANDLER)SMSO_vDestroy);

            // Delete linked list itself
            eReturnCode =
                OSAL.eLinkedListDelete(
                    psObj->hDecoderList );
            if(eReturnCode == OSAL_SUCCESS)
            {
                psObj->hDecoderList = OSAL_INVALID_OBJECT_HDL;
            }
        }

        // Destroy the async update configuration
        SMSU_vDestroy(&psObj->sEvent);

        // Uninstall Sirius Module Services
        if(psObj->hServicesTask != SMS_INVALID_TASK_HANDLE)
        {
            SMS_TASK_HANDLE hServicesTask;

            SMSAPI_DEBUG_vPrint(MODULE_OBJECT_NAME, 4,
                "Destroy MODULE Task...\n");
            // Task shutdown itself will destroy the object (psObj)
            // so there is no need to handle that here. This function
            // is called from the MODULE task context, so it will
            // request the SMS-Task to be deleted (shutdown). Once the
            // task is shutdown the last thing it does is destroy the
            // SMS-Object provided at the time of SMS-task creation.
            // This means you must not use psObj anymore after this call!
            hServicesTask = psObj->hServicesTask;
            psObj->hServicesTask = SMS_INVALID_TASK_HANDLE;
            SMST_vUninstall(hServicesTask);

            // Note: Since the thread which is calling this is the same
            // thread being shutdown, it is impossible for psObj to be
            // destroyed before the next event is handled (shutdown).
            // So we are free to return to the caller, handle more
            // events and then finally shutdown.
        }

        SMSAPI_DEBUG_vPrint(MODULE_OBJECT_NAME, 4,
            "MODULE is destroyed.\n");
    }

    return;
}

/*****************************************************************************
*
*   vUpdateState
*
*****************************************************************************/
static void vUpdateState (
    MODULE_OBJECT_STRUCT *psObj,
    MODULE_STATE_ENUM eNextState
        )
{
    MODULE_STATE_ENUM eCurrentState;
    // Event to be handled by multiple decoders
    SMS_DESCRIPTOR_DISPATCH_STRUCT sDispatcher;

    // Populate with new state information...
    eCurrentState = psObj->eState;

    // Initialize dispatcher
    sDispatcher.eType = SMS_EVENT_INVALID;
    sDispatcher.pvEventArg = NULL;
    sDispatcher.bDispatch = bDispatchToAll;

    switch(eCurrentState)
    {
        case MODULE_STATE_INITIAL:
        {
            switch(eNextState)
            {
                case MODULE_STATE_INITIAL:
                    eNextState =
                        eTransitionToInitialState(psObj, &sDispatcher);
                break;

                case MODULE_STATE_UPDATING:
                    // Ignore. Can't go there from here
                    eNextState = eCurrentState;
                break;

                case MODULE_STATE_READY:
                    eNextState =
                        eTransitionToReadyState(psObj, &sDispatcher);
                break;

                case MODULE_STATE_STOPPED:
                    eNextState =
                        eTransitionToStoppedState(psObj, &sDispatcher);
                break;

                case MODULE_STATE_ERROR:
                    eTransitionToErrorState(
                        psObj, &sDispatcher, psObj->eErrorCode);
                break;

                case MODULE_STATE_INVALID:
                default:
                    // Ignore. Can't go there from here
                    eNextState = eCurrentState;
                break;
            }
        }
        break;

        case MODULE_STATE_UPDATING:
        {
            switch(eNextState)
            {
                case MODULE_STATE_INITIAL:
                    eNextState =
                        eTransitionToInitialState(psObj, &sDispatcher);
                break;

                case MODULE_STATE_UPDATING:
                    // Do nothing.
                break;

                case MODULE_STATE_READY:
                    eNextState =
                        eTransitionToReadyState(psObj, &sDispatcher);
                break;

                case MODULE_STATE_STOPPED:
                    eNextState =
                        eTransitionToStoppedState(psObj, &sDispatcher);
                break;

                case MODULE_STATE_ERROR:
                    eNextState =
                        eTransitionToErrorState(
                            psObj, &sDispatcher, psObj->eErrorCode);
                break;

                case MODULE_STATE_INVALID:
                default:
                    // Ignore. Can't go there from here
                    eNextState = eCurrentState;
                break;
            }
        }
        break;

        case MODULE_STATE_READY:
        {
            switch(eNextState)
            {
                case MODULE_STATE_INITIAL:
                    eNextState =
                        eTransitionToInitialState(psObj, &sDispatcher);
                break;

                case MODULE_STATE_UPDATING:
                    eNextState =
                        eTransitionToUpdatingState(psObj, &sDispatcher);
                break;

                case MODULE_STATE_READY:
                    // Do nothing.
                break;

                case MODULE_STATE_STOPPED:
                    eNextState =
                        eTransitionToStoppedState(psObj, &sDispatcher);
                break;

                case MODULE_STATE_ERROR:
                    eNextState =
                        eTransitionToErrorState(
                            psObj, &sDispatcher, psObj->eErrorCode);
                break;

                case MODULE_STATE_INVALID:
                default:
                    // Ignore. Can't go there from here
                    eNextState = eCurrentState;
                break;
            }
        }
        break;

        case MODULE_STATE_STOPPED:
        {
            // Ignore. Can't go anywhere from here
            eNextState = eCurrentState;
        }
        break;

        case MODULE_STATE_ERROR:
        {
            switch(eNextState)
            {
                case MODULE_STATE_INITIAL:
                    eNextState =
                        eTransitionToInitialState(psObj, &sDispatcher);
                break;

                case MODULE_STATE_STOPPED:
                    eNextState =
                        eTransitionToStoppedState(psObj, &sDispatcher);
                break;

                case MODULE_STATE_ERROR:
                    // Do nothing.
                break;

                case MODULE_STATE_UPDATING:
                case MODULE_STATE_READY:
                case MODULE_STATE_INVALID:
                default:
                    // Ignore. Can't go there from here
                    eNextState = eCurrentState;
                break;
            }
        }
        break;

        case MODULE_STATE_INVALID:
        default:
            // Do nothing.
        break;
    }

    // Update internal state
    psObj->eState = eNextState;

    // Anything to dispatch?
    if(sDispatcher.eType != SMS_EVENT_INVALID)
    {
        // Allow every attached/active DECODER to handle this event...
        OSAL.eLinkedListIterate(
            psObj->hDecoderList, bDecoderIterator, &sDispatcher);
    }

    // Check if next state is different than the current state
    if(eCurrentState != eNextState)
    {
        // Set event mask to indicate state change
        SMSU_tUpdate(&psObj->sEvent, MODULE_OBJECT_EVENT_STATE);
    }

    return;
}

/*****************************************************************************
*
*   eTransitionToErrorState
*
*****************************************************************************/
static MODULE_STATE_ENUM eTransitionToErrorState (
    MODULE_OBJECT_STRUCT *psObj,
    SMS_DESCRIPTOR_DISPATCH_STRUCT *psDispatcher,
    MODULE_ERROR_CODE_ENUM eErrorCode
        )
{

    SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
        MODULE_OBJECT_NAME": '%s' transitioned to %s due to error %s(%u).",
        psObj->pacObjectName, MACRO_TO_STRING(MODULE_STATE_ERROR),
        SMSAPI_DEBUG_pacModuleErrorCodeText(eErrorCode), eErrorCode);

    // Tell all decoders we're in ERROR
    if(psDispatcher != NULL)
    {
        psDispatcher->eType = SMS_EVENT_ERROR;
        psDispatcher->pvEventArg = (void *)eErrorCode;
    }

    // Set local error code
    psObj->eErrorCode = eErrorCode;

    // This module is being used by data services -
    // we need to tell dataservices to transition to error
    DATASERVICE_MGR_bPostEvent(
        DATASERVICE_MGR_INVALID_OBJECT,
        DATASERVICE_FW_EVENT_SERVICE_ERROR,
        (void *)DATASERVICE_ERROR_CODE_DEVICE_FAILURE);

    // Set current state as ERROR
    // TODO: BREAKPOINT
    return MODULE_STATE_ERROR;
}

/*****************************************************************************
*
*   eTransitionToInitialState
*
*****************************************************************************/
static MODULE_STATE_ENUM eTransitionToInitialState(
    MODULE_OBJECT_STRUCT *psObj,
    SMS_DESCRIPTOR_DISPATCH_STRUCT *psDispatcher
        )
{
    MODULE_STATE_ENUM eNextState = MODULE_STATE_INITIAL;

    SMSAPI_DEBUG_vPrint(MODULE_OBJECT_NAME, 4, "%s: Transition to %s\n",
        psObj->pacObjectName, MACRO_TO_STRING(MODULE_STATE_INITIAL));

    // Run specific RADIO uninitialization
    if (TRUE == psObj->bRadioInitialized)
    {
        RADIO_vUninitializeModule((MODULE_OBJECT)psObj, TRUE);
        psObj->bRadioInitialized = FALSE;
    }

    return eNextState;
}

/*****************************************************************************
*
*   eTransitionToReadyState
*
*****************************************************************************/
static MODULE_STATE_ENUM eTransitionToReadyState(
    MODULE_OBJECT_STRUCT *psObj,
    SMS_DESCRIPTOR_DISPATCH_STRUCT *psDispatcher
        )
{
    MODULE_STATE_ENUM eNextState = MODULE_STATE_READY;

    SMSAPI_DEBUG_vPrint(MODULE_OBJECT_NAME, 4, "%s: Transition to %s\n",
        psObj->pacObjectName, MACRO_TO_STRING(MODULE_STATE_READY));

    // Time to tell the data services that this module is ready 
    vAssignToDataServices(psObj);

    // Tell all DECODERs we're READY
    if(psDispatcher != NULL)
    {
        psDispatcher->eType = SMS_EVENT_ASSOCIATE;
        psDispatcher->pvEventArg = (void *)psObj;
    }
    else
    {
        eNextState =
            eTransitionToErrorState(
                psObj, psDispatcher, MODULE_ERROR_CODE_INVALID_INPUT);
    }

    return eNextState;
}

/*****************************************************************************
*
*   eTransitionToStoppedState
*
*****************************************************************************/
static MODULE_STATE_ENUM eTransitionToStoppedState(
    MODULE_OBJECT_STRUCT *psObj,
    SMS_DESCRIPTOR_DISPATCH_STRUCT *psDispatcher
        )
{
    MODULE_STATE_ENUM eNextState = MODULE_STATE_STOPPED;

    SMSAPI_DEBUG_vPrint(MODULE_OBJECT_NAME, 4, "%s: Transition to %s\n",
        psObj->pacObjectName, MACRO_TO_STRING(MODULE_STATE_STOPPED));

    return eNextState;
}

/*****************************************************************************
*
*   eTransitionToUpdatingState
*
*****************************************************************************/
static MODULE_STATE_ENUM eTransitionToUpdatingState(
    MODULE_OBJECT_STRUCT *psObj,
    SMS_DESCRIPTOR_DISPATCH_STRUCT *psDispatcher
        )
{
    SMSAPI_DEBUG_vPrint(MODULE_OBJECT_NAME, 4, "%s: Transition to %s\n",
        psObj->pacObjectName, MACRO_TO_STRING(MODULE_STATE_UPDATING));

    // Set current state as UPDATING
    return MODULE_STATE_UPDATING;
}

/*******************************************************************************
*
*   vEventHandler
*
*   This function runs in the context of the Sirius Services Task
*
*******************************************************************************/
static void vEventHandler (
    MODULE_OBJECT hModule,
    SMS_EVENT_HDL hEvent
        )
{
    MODULE_OBJECT_STRUCT *psObj =
        (MODULE_OBJECT_STRUCT *)hModule;
    SMS_EVENT_TYPE_ENUM eEventType;
    SMS_EVENT_DATA_UNION const *puEventData;
    BOOLEAN bLocked;

    // Lock object for exclusive access
    bLocked = SMSO_bLock((SMS_OBJECT)hModule, OSAL_OBJ_TIMEOUT_INFINITE);
    if(bLocked == FALSE)
    {
        // Error!
        SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
            MODULE_OBJECT_NAME": Unable to lock MODULE.");
        return;
    }

    // Extract event information
    // Caution! Do not pass puEventData by reference to any other function
    // unless you are absolutely sure you do not end up re-allocating the
    // current event, modify the event data and continue using data from the
    // original event. If you need to do this, first copy what is needed
    // to a local variable and use that instead. This will prevent any
    // issues with potential re-allocation/overwriting of the current event.
    eEventType = SMSE_eGetEvent(hEvent, &puEventData);

    // Process event...
    switch(eEventType)
    {
        // Handle MODULE events here...

        // The INITIALIZE event occurs one time only when the task is started
        // and completes before returning to the caller.
        // Also this occurs after MODULE reset.
        case SMS_EVENT_INITIALIZE:
        {
            vHandleInitEvent(psObj);
        }
        break;

        // This event is received when the SRM object itself becomes ready
        // or the SRM object is already ready and successfully adds this
        // Module to it's management.
        case SMS_EVENT_ASSOCIATE:
        {
            vHandleAssociateEvent(psObj,
                &(puEventData->uModule.sAssociate));
        }
        break;

        // Event from Parent (SRM) to confirm that MODULE is removed from SRM
        case SMS_EVENT_UNASSOCIATE:
        {
            vHandleUnassociateEvent(psObj);
        }
        break;

        // This event is received when the Radio-MODULE itself becomes ready.
        case SMS_EVENT_RADIO_READY:
        {
            vHandleRadioReadyEvent(psObj,
                puEventData->uModule.sRadioReady.hSTI);
        }
        break;

        // This event is received when the Radio-MODULE itself becomes uninitialized.
        case SMS_EVENT_RADIO_RELEASED:
        {
            vHandleRadioReleasedEvent(psObj);
        }
        break;

        // Someone has requested this object release.
        // Only application or the upper level
        // object (an SRM) can actually remove the Module.
        case SMS_EVENT_RELEASE:
        {
            vHandleReleaseEvent(psObj,
                puEventData->uModule.sRelease.eInitiator);
        }
        break;

        // Someone requested the MODULE to RESET or one of it's DECODERs is in
        // RESET. If a MODULE is to RESET then all modes are forcibly RESET as
        // well. If a MODULE's DECODER is in RESET then we simply make it READY
        // if this MODULE itself is READY.
        case SMS_EVENT_RESET:
        {
            vHandleResetEvent(psObj, puEventData->uModule.sReset.hDecoder);
        }
        break;

        // Final Stop
        case SMS_EVENT_STOP:
        {
            vHandleStopEvent(psObj);
        }
        break;

        case SMS_EVENT_ERROR:
        {
            const MODULE_ERROR_CODE_ENUM eErrorCode =
                puEventData->uModule.sError.eErrorCode;

            SMSAPI_DEBUG_vPrint(MODULE_OBJECT_NAME, 4,
                "%s: SMS_EVENT_ERROR: eErrorCode: %d\n",
                psObj->pacObjectName, eErrorCode);

            // Error!
            vSetError(psObj, eErrorCode);
        }
        break;

        case SMS_EVENT_UPDATE_TIME:
        {
            BOOLEAN bSuccess;
            const SMS_EVENT_MODULE_SET_TIME_STRUCT sSetTime =
                puEventData->uModule.sSetTime;

            SMSAPI_DEBUG_vPrint(MODULE_OBJECT_NAME, 4,
                "%s: SMS_EVENT_UPDATE_TIME.\n",
                psObj->pacObjectName);

            // Apply new time
            bSuccess = bSetTime(psObj, &sSetTime);
            if(FALSE == bSuccess)
            {
                // Error!
                SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    MODULE_OBJECT_NAME": Cannot set time.");
            }
        }
        break;

        case SMS_EVENT_UPDATE_OBJECT:
        {
            const SMSAPI_EVENT_MASK tEventMask =
                puEventData->uModule.sUpdate.tEventMask;

            SMSAPI_DEBUG_vPrint(MODULE_OBJECT_NAME, 4,
                "%s: SMS_EVENT_UPDATE_OBJECT: tEventMask: 0x%X\n",
                psObj->pacObjectName, tEventMask);

            // Force an update of all that was requested
            SMSU_tUpdate(&psObj->sEvent, tEventMask);
        }
        break;

        // DECODER wishes to remove itself from this MODULE.
        // MODULE shell respond with ANUSSOCIATE event
        // to confirm the DECODER is removed.
        case SMS_EVENT_REMOVE_DECODER:
        {
            vHandleRemoveDecoderEvent(psObj,
                puEventData->uModule.sRemoveDecoder.hDecoder);
        }
        break;

        case SMS_EVENT_SAVE_TIME_PARAMS:
        {
            OSAL_RETURN_CODE_ENUM eOsalReturnCode;
            N32 n32GMToffset = 0;

            SMSAPI_DEBUG_vPrint(MODULE_OBJECT_NAME, 4,
                "%s: SMS_EVENT_SAVE_TIME_PARAMS.\n",
                psObj->pacObjectName);

            // Get current GMT offset
            eOsalReturnCode = OSAL.eTimeGetGMToffset(&n32GMToffset);
            if(eOsalReturnCode == OSAL_SUCCESS)
            {
                SMSAPI_RETURN_CODE_ENUM eResult;
                TAG_OBJECT hGmtTag;

                // get tag
                eResult = TAG_eGet(
                              GMT_OFFSET_TAG,
                              psObj->hTag,
                              &hGmtTag,
                              NULL,
                              TRUE); // Create if it doesn't exist
                if ( (eResult == SMSAPI_RETURN_CODE_SUCCESS) &&
                     (hGmtTag != TAG_INVALID_OBJECT) )
                {
                    // got the tag, now set its value
                    TAG_eSetTagValue(
                        hGmtTag,
                        TAG_TYPE_INTEGER,
                        (void *)&n32GMToffset,
                        sizeof(N32),
                        FALSE); // don't commit now
                }
            }
        }
        break;

        case SMS_EVENT_MODIFY_EVENT_MASK:
        {
            const SMS_EVENT_MODULE_EVENT_MASK_STRUCT sModuleEventMask =
                puEventData->uModule.sModuleEventMask;
            BOOLEAN bSuccess;

            SMSAPI_DEBUG_vPrint(MODULE_OBJECT_NAME, 4,
                "SMS_EVENT_MODIFY_MODULE_EVENT_MASK. tEventMask: %d, eModification: %d\n",
                sModuleEventMask.tEventMask, sModuleEventMask.eModification);

            bSuccess = bProcessModuleEventMaskChange(psObj, &sModuleEventMask);
            if(bSuccess == FALSE)
            {
                // Error!
                SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    MODULE_OBJECT_NAME": Could not process module"
                    " event mask change:");

                // Error!
                vSetError(psObj, MODULE_ERROR_CODE_GENERAL);
            }
        }
        break;

        case SMS_EVENT_MODIFY_ENGINEERING_DATA_STATE:
        {
            const SMS_EVENT_MODULE_ENGDATA_STRUCT sModModifyEngData =
                puEventData->uModule.sModModifyEngData;
            SMSAPI_RETURN_CODE_ENUM eReturnCode;

            SMSAPI_DEBUG_vPrint(MODULE_OBJECT_NAME, 4,
                "SMS_EVENT_MODIFY_ENGINEERING_DATA_STATE. hDecoder: %d, eModification: %d\n",
                sModModifyEngData.hDecoder, sModModifyEngData.eModification);

            eReturnCode = eProcessModuleEngineeringDataChange(
                psObj, &sModModifyEngData);
            if(eReturnCode == SMSAPI_RETURN_CODE_ERROR)
            {
                // Error!
                SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    MODULE_OBJECT_NAME": Could not process engineering"
                    " data change: SMSAPI_RETURN_CODE_ERROR.");

                // Error!
                vSetError(psObj, MODULE_ERROR_CODE_RADIO_OBJECT_ERROR);
            }
        }
        break;

        default:
        {
            // Un-handled event, pass on to RADIO specific handler
            RADIO_bModuleEventHandler(hModule, eEventType, puEventData);
        }

    } // switch(eEventType)

    // Notify of any change via any registered callback which may be present
    SMSU_bNotify(&psObj->sEvent);

    // Check if we need to destroy the object. This means not only
    // have we been told to STOP but also no existing modes of operation exist.
    if(psObj->eState == MODULE_STATE_STOPPED)
    {
        // Uninitialize Object
        vUninitObject(psObj);
    }

    // Unlock object, any modifications to be made to this object
    // have already been made.
    SMSO_vUnlock((SMS_OBJECT)hModule);

    // Event handled, simply return
    return;
}

/*******************************************************************************
*
*   bDecoderIterator
*
*******************************************************************************/
static BOOLEAN bDecoderIterator (
    void *pvData,
    void *pvArg
        )
{
    BOOLEAN bContinue = FALSE;
    MODULE_DECODER_ENTRY_STRUCT *psDecoder =
                    (MODULE_DECODER_ENTRY_STRUCT *)pvData;
    SMS_DESCRIPTOR_DISPATCH_STRUCT *psDispatcher =
        (SMS_DESCRIPTOR_DISPATCH_STRUCT *)pvArg;

    // Verify inputs (only the psDispatcher needs to be valid)
    if(psDispatcher != NULL)
    {
        // Check if dispatch function exists
        if(psDispatcher->bDispatch != NULL)
        {
            if (psDecoder != NULL)
            {
                if (psDecoder->bValid == TRUE)
                {
                    // Run dispatch function
                    bContinue = psDispatcher->bDispatch(
                        psDecoder,
                        psDispatcher->eType,
                        psDispatcher->pvEventArg
                            );
                }
            }
        }
    }

    return bContinue;
}

/*******************************************************************************
*
*   bDispatchToAll
*
*******************************************************************************/
static BOOLEAN bDispatchToAll (
    void *pvDescriptor,
    SMS_EVENT_TYPE_ENUM eType,
    void *pvEventArg
        )
{
    BOOLEAN bResult = TRUE;
    MODULE_DECODER_ENTRY_STRUCT *psDecoder =
        (MODULE_DECODER_ENTRY_STRUCT *)pvDescriptor;

    if (NULL == psDecoder)
    {
        // Error!
        return FALSE;
    }

    // Post event to this DECODER...
    switch(eType)
    {
        case SMS_EVENT_ASSOCIATE:
        {
            // Now tell this Decoder we are ready to associate with it.
            bResult =
                bAssociateWithDecoder((MODULE_OBJECT_STRUCT *)pvEventArg,
                    psDecoder->hDecoder);
        }
        break;

        case SMS_EVENT_ERROR:
        {
            // We can retrieve the error code from the Module, but since
            // there is no intersection of Module error codes and Decoder
            // error codes, this really doesn't do us any good.
            // But maybe in the future it will. For now we simply pass
            // along DECODER_ERROR_CODE_MODULE_ERROR at let them check the
            // Module error.
            //MODULE_ERROR_CODE_ENUM eErrorCode =
            //    (MODULE_ERROR_CODE_ENUM)pvEventArg;

            // Inform Module of an error
            DECODER_bSetError(psDecoder->hDecoder,
                DECODER_ERROR_CODE_MODULE_ERROR);
        }
        break;

        case SMS_EVENT_DISPATCH:
        {
            MODULE_DECODER_DISPATCHER_STRUCT *psDispatchRequest =
                (MODULE_DECODER_DISPATCHER_STRUCT *)pvEventArg;

            // Execute dispatch request

            // Verify inputs (only the psDispatchRequest needs to be valid)
            if(psDispatchRequest != NULL)
            {
                // Check if dispatch function exists
                if(psDispatchRequest->bDispatch != NULL)
                {
                    // Dispatch only if engineering data enabled?
                    if((psDispatchRequest->bOnlyIfEngineeringDataRequested == FALSE) ||
                        ((psDispatchRequest->bOnlyIfEngineeringDataRequested == TRUE)
                            & (psDecoder->bEngineeringDataEnabled == TRUE))
                      )
                    {
                        // Run dispatch function
                        bResult = psDispatchRequest->bDispatch(
                            psDecoder->hDecoder,
                            psDispatchRequest->eType,
                            psDispatchRequest->pvEventArg
                                );
                    }
                }
            }
        }
        break;

        default:
            // Unhandled event (ok)
        break;
    }

    return bResult;
}

/*****************************************************************************
 *
 *   vAssignToDataServices
 *
 *   Post an event to the DSM informing it that this MODULE is ready to
 *   be used for data services
 *
 *****************************************************************************/
static void vAssignToDataServices (
    MODULE_OBJECT_STRUCT *psObj
        )
{
    do
    {
        BOOLEAN bPosted = FALSE;

        if (psObj->eState == MODULE_STATE_ERROR)
        {
            // This object is in the error state.
            // Report that to the DSM.
            bPosted = DATASERVICE_MGR_bPostEvent(
                DATASERVICE_MGR_INVALID_OBJECT,
                DATASERVICE_FW_EVENT_SERVICE_ERROR,
                (void *)DATASERVICE_ERROR_CODE_DEVICE_FAILURE);
        }
        else
        {
            const char *pacDriverName;

            // Get the driver name from our SRM
            pacDriverName = SRM_pacDriverName(psObj->hSRM);

            if (pacDriverName != NULL)
            {
                // We may now assign this module to data services
                bPosted = DATASERVICE_MGR_bAssociateModule(
                    pacDriverName, (MODULE_OBJECT)psObj, psObj->hSTI);
                if (TRUE == bPosted)
                {
                    // Now we assume the Module's resources are utilized by DSM.
                    // Increment reference counter. To decrement reference counter
                    // DSM shall call MODULE_bRelease() to tell the Module
                    // that the Module is not utilized anymore.

                    SMSAPI_DEBUG_vPrint(MODULE_OBJECT_NAME, 4,
                        "%s: ASSOCIATE request is posted to DSM.\n",
                        psObj->pacObjectName);

                    psObj->un16RefCounter++;
                }
            }
        }

        if (bPosted == FALSE)
        {
            vSetError(psObj, MODULE_ERROR_CODE_EVENT_POST_ERROR);
            break;
        }

    } while (FALSE);

    return;
}

/*******************************************************************************
*
*   vSetError
*
*******************************************************************************/
static void vSetError(
    MODULE_OBJECT_STRUCT *psObj,
    MODULE_ERROR_CODE_ENUM eErrorCode
        )
{
    // Set local object error code
    psObj->eErrorCode = eErrorCode;

    // Transition to error state
    vUpdateState(psObj, MODULE_STATE_ERROR);

    return;
}

/*******************************************************************************
*
*   vGetTimeParams
*
*******************************************************************************/
static void vGetTimeParams(
    MODULE_OBJECT_STRUCT *psObj
        )
{
    SMSAPI_RETURN_CODE_ENUM eResult;
    TAG_OBJECT hGmtTag = TAG_INVALID_OBJECT;
    N32 n32TagValue = 0;

    // Get tag for gmt offset, but don't bother creating if it doesn't exist
    eResult = TAG_eGet(GMT_OFFSET_TAG, psObj->hTag, &hGmtTag, NULL, FALSE);
    if ( (eResult == SMSAPI_RETURN_CODE_SUCCESS) &&
         (hGmtTag != TAG_INVALID_OBJECT) )
    {
        // got the tag, now get its value
        N32 *pn32TagValue;
        size_t tSize;

        pn32TagValue = &n32TagValue;
        eResult = TAG_eGetTagValue(
                      hGmtTag,
                      TAG_TYPE_INTEGER,
                      (void **)&pn32TagValue,
                      &tSize);

        if (eResult != SMSAPI_RETURN_CODE_SUCCESS)
        {
            // use the default value
            n32TagValue = GMT_OFFSET_DEFAULT;
        }
    }
    else
    {
        // if either we didn't get the tag or didn't get a value

        // use the default value
        n32TagValue = GMT_OFFSET_DEFAULT;
    }

    // Eitherway we have something. Set it accordingly
    eGmtOffsetSet(MODULE_INVALID_OBJECT, (N16)n32TagValue);

    return;
}

/*****************************************************************************
 *
 *   bEnableEngineeringDataOnDecoder
 *
 *****************************************************************************/
static BOOLEAN bEnableEngineeringDataOnDecoder(
    void *pvData,
    void *pvArg
        )
{
    MODULE_DECODER_ENTRY_STRUCT *psDecoder =
                    (MODULE_DECODER_ENTRY_STRUCT *)pvData;
    MODULE_DECODER_ENABLE_ENGINEERING_DATA *psEnableEngineeringData;

    psEnableEngineeringData = (MODULE_DECODER_ENABLE_ENGINEERING_DATA *)pvArg;

    // if a decoder has engineering data enabled count it and then
    // mark it if it's state is changing.
    if (psDecoder->bEngineeringDataEnabled == TRUE)
    {
        psEnableEngineeringData->un8NumEngineeringDataEnabled++;
        // engineering data state changes if this specific decoder has
        // been requested to be disabled, or all have been requested to
        // be disabled
        if (  (    (psEnableEngineeringData->eModification ==
                        SMSAPI_MODIFY_EVENT_MASK_DISABLE)
                && (psDecoder->hDecoder == psEnableEngineeringData->hDecoder)
              )
              ||
              (psEnableEngineeringData->bDisableAll == TRUE)
           )
        {
            psDecoder->bEngineeringDataEnabled = FALSE;
        }
    }
    else
    {
        // a decoder with engineering data disabled is marked as enabled if
        // this specific decoder has been requested to be enabled
        if (  (    (psEnableEngineeringData->eModification ==
                        SMSAPI_MODIFY_EVENT_MASK_ENABLE)
                && (psDecoder->hDecoder == psEnableEngineeringData->hDecoder)
              )
              &&
              (psEnableEngineeringData->bDisableAll != TRUE)
           )
        {
            psDecoder->bEngineeringDataEnabled = TRUE;
        }
    }
    return TRUE;

}

/*****************************************************************************
 *
 *   eProcessModuleEngineeringDataChange
 *
 *****************************************************************************/
SMSAPI_RETURN_CODE_ENUM eProcessModuleEngineeringDataChange(
    MODULE_OBJECT_STRUCT *psObj,
    SMS_EVENT_MODULE_ENGDATA_STRUCT const *psModModifyEngData
        )
{
    SMSAPI_RETURN_CODE_ENUM eReturnCode = SMSAPI_RETURN_CODE_ERROR;
    MODULE_DECODER_ENABLE_ENGINEERING_DATA sEnableEngineeringData;
    OSAL_RETURN_CODE_ENUM eOsalReturnCode;

    sEnableEngineeringData.un8NumEngineeringDataEnabled = 0;
    sEnableEngineeringData.eModification = psModModifyEngData->eModification;
    sEnableEngineeringData.hDecoder = psModModifyEngData->hDecoder;
    sEnableEngineeringData.bDisableAll = FALSE;

    // Allow every attached/active Decoder to handle this event...
    eOsalReturnCode = OSAL.eLinkedListIterate(
        psObj->hDecoderList,
        bEnableEngineeringDataOnDecoder,
        &sEnableEngineeringData
            );
    if ((eOsalReturnCode != OSAL_SUCCESS) &&
        (eOsalReturnCode != OSAL_NO_OBJECTS))
    {
        SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
            MODULE_OBJECT_NAME": Decoder list iterate failed: %s.",
            OSAL.pacGetReturnCodeName(eOsalReturnCode));
    }

    // we only have to issue the command to the device
    // if we are enabling and no previous decoders were enabled.
    // or we are disabling for a decoder and exactly one was previously
    // enabled
    if ((  (sEnableEngineeringData.un8NumEngineeringDataEnabled == 0)
        &&(psModModifyEngData->eModification == SMSAPI_MODIFY_EVENT_MASK_ENABLE)
        )
        ||
        (  (sEnableEngineeringData.un8NumEngineeringDataEnabled == 1)
        &&(psModModifyEngData->eModification == SMSAPI_MODIFY_EVENT_MASK_DISABLE)
        )
       )
    {
        eReturnCode =
            RADIO_eModifyEngineeringData( (MODULE_OBJECT)psObj,
                                   psModModifyEngData->eModification);
    }
    else
    {
        // (the other two cases) not needing to enable engineering data on an already 
        // enabled module or disable on an already disabled module is not an error case.
        eReturnCode = SMSAPI_RETURN_CODE_SUCCESS;
    }

    // if for some reason turning engineering data on on the device didn't work
    // let's do our best to maintain system integrity by doing our best to turn
    // it off on the device and marking all decoders in our list as having
    // engineering data disabled.
    if (eReturnCode != SMSAPI_RETURN_CODE_SUCCESS)
    {
        RADIO_eModifyEngineeringData((MODULE_OBJECT)psObj,
                                      SMSAPI_MODIFY_EVENT_MASK_DISABLE);
        sEnableEngineeringData.bDisableAll = TRUE;
        sEnableEngineeringData.eModification = SMSAPI_MODIFY_EVENT_MASK_DISABLE;
        OSAL.eLinkedListIterate(
            psObj->hDecoderList,
            bEnableEngineeringDataOnDecoder,
            &sEnableEngineeringData
            );
    }
    return eReturnCode;
}

/*****************************************************************************
 *
 *   bProcessModuleEventMaskChange
 *
 *****************************************************************************/
static BOOLEAN bProcessModuleEventMaskChange(
    MODULE_OBJECT_STRUCT *psObj,
    SMS_EVENT_MODULE_EVENT_MASK_STRUCT const *psModuleEventMask
        )
{
    BOOLEAN bValid, bSuccess = FALSE;

    // Verify SMS Object is valid
    bValid = SMSO_bValid((SMS_OBJECT)psObj);
    if(bValid == TRUE)
     {
        bSuccess = SMSU_bModifyRequestMask(
            &psObj->sEvent,
            NULL,
            NULL,
            psModuleEventMask->tEventMask,
            psModuleEventMask->eModification);
    }

    return bSuccess;
}

/*****************************************************************************
 *
 *   tCopyPackageData
 *
 *****************************************************************************/
static size_t tCopyPackageData(
    MODULE_OBJECT_STRUCT *psModule,
    UN32 un32Timeout,
    UN8 *pun8Resp,
    size_t tRespSize
        )
{
    size_t tNumRxBytes = 0;
    OSAL_RETURN_CODE_ENUM eReturnCode;

    // Wait for package data to be received
    eReturnCode = OSAL.eSemTake(
        psModule->sCache.sPackage.hMutex,
        un32Timeout
            );
    if(eReturnCode == OSAL_SUCCESS)
    {
        size_t tNumBytesToCopy;
        BOOLEAN bLocked;

        // We have package data

        // Lock SMS Object
        bLocked =
            SMSO_bLock((SMS_OBJECT)psModule, OSAL_OBJ_TIMEOUT_INFINITE);
        if(bLocked == TRUE)
        {
            // Determine how much data we can copy out
            tNumRxBytes = psModule->sCache.sPackage.tNumBytes;
            if (tNumRxBytes > tRespSize)
            {
                // Copy some
                tNumBytesToCopy = tRespSize;
            }
            else
            {
                // Copy it all
                tNumBytesToCopy = tNumRxBytes;
            }

            // Check if a destination for this response was
            // given, if not they just need to know the size.
            if ((pun8Resp != NULL) &&
                (psModule->sCache.sPackage.pun8Data != NULL))
            {
                BOOLEAN bOk;

                // Copy out package data
                bOk = OSAL.bMemCpy(pun8Resp,
                    psModule->sCache.sPackage.pun8Data,
                    tNumBytesToCopy
                    );
                if (bOk == TRUE)
                {
                    // Free the memory now
                    OSAL.vMemoryFree((void*)psModule->sCache.sPackage.pun8Data);
                    psModule->sCache.sPackage.pun8Data = NULL;
                    psModule->sCache.sPackage.tNumBytes = 0;
                }
                else
                {
                    // Couldn't copy the data
                    tNumRxBytes = 0;
                }
            }

            // Unlock object
            SMSO_vUnlock((SMS_OBJECT)psModule);
        }
    }
    else
    {
        // Humm, this is odd. We should not have timedout
    }

    return tNumRxBytes;
}

/*****************************************************************************
*
*   vTimeUpdateNotification
*
*****************************************************************************/
static void vTimeUpdateNotification (
    OSAL_TIME_UPDATE_MASK tUpdateMask,
    void *pvArg
        )
{
    MODULE_OBJECT_STRUCT *psObj = (MODULE_OBJECT_STRUCT *)pvArg;
    BOOLEAN bValid;

    // Verify SMS Object is valid
    bValid = SMSO_bValid((SMS_OBJECT)psObj);
    if(bValid == TRUE)
    {
        BOOLEAN bPosted;

        bPosted = SMSE_bPostSignal(psObj->hEventHdlr,
            SMS_EVENT_SAVE_TIME_PARAMS, SMS_EVENT_OPTION_NONE);
        if(bPosted == FALSE)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                MODULE_OBJECT_NAME": Unable to post time"
                " available event.");
        }
    }

    return;
}

/*****************************************************************************
 *
 *   bAssociateWithDecoder
 *
 *****************************************************************************/
static BOOLEAN bAssociateWithDecoder (
    MODULE_OBJECT_STRUCT *psObj,
    DECODER_OBJECT hDecoder
        )
{
    BOOLEAN bResult = TRUE;

    // If not ready, ASSOCIATE event will be posted
    // when the MODULE becomes ready.
    if (MODULE_STATE_READY == psObj->eState)
    {
        // Tell Decoder we are ready to associate with it.
        // Now tell this Decoder we are ready to associate with it.
        bResult = DECODER_bAssociate(hDecoder,
            (MODULE_OBJECT)psObj, psObj->hSTI);
        if (TRUE == bResult)
        {
            // Now we assume the MODULE's resources are utilized by DECODER.
            // Increment reference counter. To decrement reference counter
            // DECODER shall call MODULE_bRelease() to tell the MODULE
            // that the DECODER is not utilizing the MODULE anymore.

            SMSAPI_DEBUG_vPrint(MODULE_OBJECT_NAME, 4,
                "%s: ASSOCIATE request is posted to DECODER.\n",
                psObj->pacObjectName);

            psObj->un16RefCounter++;
        }
        else
        {
            // Error!
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                MODULE_OBJECT_NAME": Unable to associate with DECODER.");

            // Transition to error state
            vSetError(psObj, MODULE_ERROR_CODE_EVENT_POST_ERROR);
        }
    }

    return bResult;
}

/*****************************************************************************
 *
 *   bReleaseParentSRM
 *
 *****************************************************************************/
static BOOLEAN bReleaseParentSRM (
    MODULE_OBJECT_STRUCT *psObj
        )
{
    BOOLEAN bResult = TRUE;

    // If this MODULE is associated with SRM, release SRM.
    if (0 != (psObj->tCurrentModes & SMS_MODE_PARENT_ASSOCIATED))
    {
        bResult = SRM_bRelease(psObj->hSRM, SMS_OBJECT_RELEASE_BY_OTHERS);
        if (TRUE == bResult)
        {
            psObj->tCurrentModes &= ~SMS_MODE_PARENT_ASSOCIATED;

            SMSAPI_DEBUG_vPrint(MODULE_OBJECT_NAME, 4,
                "%s: RELEASE request is posted to SRM.\n",
                psObj->pacObjectName);
        }
        else
        {
            // Error!
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                MODULE_OBJECT_NAME": Unable to release SRM.");
        }
    }

    return bResult;
}

/* Shutdown management */

/*****************************************************************************
 *
 *   bObjectBusy
 *
 *****************************************************************************/
static BOOLEAN bObjectBusy (
    MODULE_OBJECT_STRUCT *psObj
        )
{
    OSAL_RETURN_CODE_ENUM eReturnCode;
    UN32 un32NumItems = 0;

    // Check if this object is currently being released
    if (0 == (psObj->tCurrentModes & SMS_MODE_STOPPING))
    {
        SMSAPI_DEBUG_vPrint(MODULE_OBJECT_NAME, 4,
            "%s: BUSY: Not being released.\n", psObj->pacObjectName);
        return TRUE;
    }

    //  Check reference counter
    if (0 != psObj->un16RefCounter)
    {
        SMSAPI_DEBUG_vPrint(MODULE_OBJECT_NAME, 4,
            "%s: BUSY: Reference Counter: %u\n", psObj->pacObjectName,
            psObj->un16RefCounter);
        return TRUE;
    }

    // Check if we have any Child objects attached
    eReturnCode = OSAL.eLinkedListItems(psObj->hDecoderList, &un32NumItems);
    if (OSAL_SUCCESS != eReturnCode)
    {
        SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
            MODULE_OBJECT_NAME": Could not get number of DECODERs: %d (%s).",
            eReturnCode, OSAL.pacGetReturnCodeName(eReturnCode));

        // Transition to ERROR state
        vSetError(psObj, MODULE_ERROR_CODE_DECODER_LIST_ERROR);
        return TRUE;
    }
    else if (0 != un32NumItems)
    {
        SMSAPI_DEBUG_vPrint(MODULE_OBJECT_NAME, 4,
            "%s: BUSY: Still has %u DECODERs attached. "
            "Waiting for REMOVE event.\n", psObj->pacObjectName,
            un32NumItems);
        return TRUE;
    }

    SMSAPI_DEBUG_vPrint(MODULE_OBJECT_NAME, 4,
        "%s: MODULE is ready for shutdown.\n", psObj->pacObjectName);

    return FALSE;
}

/*****************************************************************************
*
*   bTryStop
*
*****************************************************************************/
static BOOLEAN bTryStop (
    MODULE_OBJECT_STRUCT *psObj
        )
{
    BOOLEAN bResult;

    // Check if this object is currently utilized
    bResult = bObjectBusy(psObj);
    if (TRUE == bResult)
    {
        // Cannot stop now.
        // Waiting for shutdown completion.
        return FALSE;
    }

    if ((SRM_INVALID_OBJECT != psObj->hSRM) &&
        (0 != (psObj->tCurrentModes & SMS_MODE_PARENT_ADDED)))
    {
        // Tell owning SRM that it should now remove this MODULE.
        // SRM shall respond with UNASSOCIATE event to confirm
        // that this MODULE is removed.
        bResult = SRM_bRemoveModule(psObj->hSRM, (MODULE_OBJECT)psObj);
        if (TRUE == bResult)
        {
            // Clear flag
            psObj->tCurrentModes &= ~SMS_MODE_PARENT_ADDED;

            SMSAPI_DEBUG_vPrint(MODULE_OBJECT_NAME, 4,
                "%s: REMOVE request is posted to SRM. "
                "Waiting for UNASSOCIATE event.\n",
                psObj->pacObjectName);
        }
        else
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                MODULE_OBJECT_NAME": Cannot remove MODULE (%s).",
                MACRO_TO_STRING(MODULE_ERROR_CODE_UNABLE_TO_REMOVE_MODULE));
        }

        // Now waiting for UNASSOCIATE event

        return bResult;
    }

    // Now this object is ready for shutdown.
    // But we should dispatch all remaining events.
    // by posting of special event which shall be dispatched
    // last from the queue.
    bResult = bPostFinalStop(psObj);
    if (FALSE == bResult)
    {
        // Error!
        vSetError(psObj, MODULE_ERROR_CODE_EVENT_POST_ERROR);
        return FALSE;
    }

    // Now waiting for FINAL STOP event.

    return TRUE;
}

/*****************************************************************************
*
*   bInitiateObjectRelease
*
*****************************************************************************/
static BOOLEAN bInitiateObjectRelease (
    MODULE_OBJECT_STRUCT *psObj,
    SMS_OBJECT_RELEASE_INITIATOR_ENUM eInitiator
        )
{
    BOOLEAN bResult = FALSE;
    SMS_EVENT_HDL hEvent;
    SMS_EVENT_DATA_UNION *puEventData;

    // Allocate an event
    hEvent = SMSE_hAllocateEvent(psObj->hEventHdlr, SMS_EVENT_RELEASE,
        &puEventData, SMS_EVENT_OPTION_NONE);
    if (SMS_INVALID_EVENT_HDL != hEvent)
    {
        // Populate event information
        puEventData->uModule.sRelease.eInitiator = eInitiator;

        // Post event
        bResult = SMSE_bPostEvent(hEvent);
        if (TRUE == bResult)
        {
            SMSAPI_DEBUG_vPrint(MODULE_OBJECT_NAME, 4,
                "%s: RELEASE event is posted.\n",
                psObj->pacObjectName);
        }
        else
        {
            // Error!
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                MODULE_OBJECT_NAME": Cannot post event.");
        }
    }
    else
    {
        // Error!
        SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
            MODULE_OBJECT_NAME": Cannot allocate event.");
    }

    return bResult;
}

/*****************************************************************************
*
*   bReleaseDecoders
*
*****************************************************************************/
static BOOLEAN bReleaseDecoders(
    MODULE_OBJECT_STRUCT *psObj
        )
{
    OSAL_RETURN_CODE_ENUM eReturnCode;

    // Tell all DECODERs to release...
    eReturnCode =
        OSAL.eLinkedListIterate(
            psObj->hDecoderList,
            bReleaseDecoderIterator,
            (void *)NULL);
    if (OSAL_SUCCESS == eReturnCode)
    {
        SMSAPI_DEBUG_vPrint(MODULE_OBJECT_NAME, 4,
            "%s: DECODER(s) release is requested.\n",
            psObj->pacObjectName);
    }
    else if (OSAL_NO_OBJECTS == eReturnCode)
    {
        SMSAPI_DEBUG_vPrint(MODULE_OBJECT_NAME, 4,
            "%s: There are no DECODERs to release.\n",
            psObj->pacObjectName);
    }
    else
    {
        // Error!
        SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
            MODULE_OBJECT_NAME": Unable to initiate DECODER(s) release: %d (%s)",
            eReturnCode, OSAL.pacGetReturnCodeName(eReturnCode));

        // Transition to ERROR state
        vSetError(psObj, MODULE_ERROR_CODE_DECODER_LIST_ERROR);
        return FALSE;
    }

    return TRUE;
}

/*****************************************************************************
*
*   bPostFinalStop
*
*****************************************************************************/
static BOOLEAN bPostFinalStop(
    MODULE_OBJECT_STRUCT *psObj
        )
{
    BOOLEAN bResult = FALSE;

    bResult = SMSE_bPostSignal(psObj->hEventHdlr,
        SMS_EVENT_STOP, SMS_EVENT_OPTION_DEFERRED);
    if (TRUE == bResult)
    {
        SMSAPI_DEBUG_vPrint(MODULE_OBJECT_NAME, 4,
            "%s: FINAL STOP event is posted.\n", psObj->pacObjectName);
    }
    else
    {
        // Error!
        SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
            MODULE_OBJECT_NAME": Unable to post FINAL STOP event.");
    }

    return bResult;
}

/* General events handlers */

/*****************************************************************************
*
*   vHandleInitEvent
*
*****************************************************************************/
static void vHandleInitEvent(
    MODULE_OBJECT_STRUCT *psObj
        )
{
    BOOLEAN bOk;

    SMSAPI_DEBUG_vPrint(MODULE_OBJECT_NAME, 4,
        "%s: SMS_EVENT_INITIALIZE\n", psObj->pacObjectName);

    do
    {
        // Check stopping flag indicating the object is currently being released
        if (0 != (psObj->tCurrentModes & SMS_MODE_STOPPING))
        {
            SMSAPI_DEBUG_vPrint(MODULE_OBJECT_NAME, 4,
                "%s: Object is currently being released. Ignore event.\n",
                psObj->pacObjectName);
            break;
        }

        // Initialize the MODULE object
        bOk = bInitializeObject(psObj);
        if (FALSE == bOk)
        {
            // Transition to error state
            vSetError(psObj, MODULE_ERROR_CODE_INITIALIZATION_ERROR);
            break;
        }

        SMSAPI_DEBUG_vPrint(MODULE_OBJECT_NAME, 4,
            "%s: MODULE object is initialized.\n", psObj->pacObjectName);

        // Check if we were RESET
        if (0 == (psObj->tCurrentModes & SMS_MODE_RESET))
        {
            // Nothing else to do
            break;
        }

        SMSAPI_DEBUG_vPrint(MODULE_OBJECT_NAME, 4,
            "%s: MODULE has been reset.\n", psObj->pacObjectName);

        // Tell owning SRM that we are resetting. Note the above post
        // makes sure the INITIALIZE event gets put on the SRM's queue first
        // while this post must go through the SRM which will make the MODULE
        // ready again once the SRM itself is ready.
        bOk = SRM_bResetModule(psObj->hSRM, (MODULE_OBJECT)psObj);
        if (FALSE == bOk)
        {
            // Transition to ERROR state
            vSetError(psObj, MODULE_ERROR_CODE_UNABLE_TO_RESET_SRM);
            break;
        }

        SMSAPI_DEBUG_vPrint(MODULE_OBJECT_NAME, 4,
            "%s: RESET is posted to SRM.\n", psObj->pacObjectName);

        // Clear mode
        psObj->tCurrentModes &= ~SMS_MODE_RESET;

    } while (FALSE);

    return;
}

/*****************************************************************************
*
*   vHandleRadioReadyEvent
*
*****************************************************************************/
static void vHandleRadioReadyEvent(
    MODULE_OBJECT_STRUCT *psObj,
    STI_HDL hSTI
        )
{
    BOOLEAN bResult;

    SMSAPI_DEBUG_vPrint(MODULE_OBJECT_NAME, 4,
        "%s: SMS_EVENT_RADIO_READY: hSTI: %p\n",
        psObj->pacObjectName, hSTI);

    do
    {
        // Try to stop if the object is currently
        // being released and ready for shutdown.
        if (0 != (psObj->tCurrentModes & SMS_MODE_STOPPING))
        {
            SMSAPI_DEBUG_vPrint(MODULE_OBJECT_NAME, 4,
                "%s: Object is currently being released."
                " Try to initiate object stop.\n",
                psObj->pacObjectName);

            bTryStop(psObj);
            break;
        }

        // Extract provided STI HDL
        psObj->hSTI = hSTI;

        // Get ESN
        bResult = RADIO_bGetESN((MODULE_OBJECT)psObj, &psObj->sCache.hESN);
        if (FALSE == bResult)
        {
            // Error!
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                MODULE_OBJECT_NAME": Unable to get ESN from radio.");

            // Transition to error state
            vSetError(psObj, MODULE_ERROR_CODE_RADIO_OBJECT_ERROR);
            break;
        }

        // Set Time Parameters (always set to GMT=0, DST=FALSE)
        bResult = RADIO_bSetTimeParameters((MODULE_OBJECT)psObj,
            0, // GMT offset minutes
            FALSE // DST-observed
                );
        if (FALSE == bResult)
        {
            // Error!
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                MODULE_OBJECT_NAME": Unable to set time parameters "
                "with radio.");

            // Transition to error state
            vSetError(psObj, MODULE_ERROR_CODE_RADIO_OBJECT_ERROR);
            break;
        }

        // Transition to the READY state
        vUpdateState(psObj, MODULE_STATE_READY);

    } while (FALSE);

    return;
}

/*****************************************************************************
*
*   vHandleRadioReleasedEvent
*
*****************************************************************************/
static void vHandleRadioReleasedEvent(
    MODULE_OBJECT_STRUCT *psObj
        )
{
    SMSAPI_DEBUG_vPrint(MODULE_OBJECT_NAME, 4,
        "%s: SMS_EVENT_RADIO_RELEASED\n", psObj->pacObjectName);

    // This is the latest event in shutdown process.
    // Now everything is uninitialized.
    // So the object can be safely destroyed.

    psObj->bRadioInitialized = FALSE;

    // If this MODULE is associated with SRM,
    // we have to post SRM release now.
    bReleaseParentSRM(psObj);

    // Stop this MODULE now
    vUpdateState(psObj, MODULE_STATE_STOPPED);

    return;
}

/*****************************************************************************
*
*   vHandleAssociateEvent
*
*****************************************************************************/
static void vHandleAssociateEvent(
    MODULE_OBJECT_STRUCT *psObj,
    const SMS_EVENT_MODULE_ASSOCIATE_STRUCT *psAssociate
        )
{
    SMSAPI_DEBUG_vPrint(MODULE_OBJECT_NAME, 4,
        "%s: SMS_EVENT_ASSOCIATE: "
        "hSRM: %p, pacName: %s, psDevice: %p\n",
        psObj->pacObjectName, psAssociate->hSRM,
        psAssociate->pacName, psAssociate->psDevice);

    do
    {
        // Only accept ASSOCIATE from the SRM this MODULE is added to.
        // Otherwise tell the SRM it can be released.
        if (psAssociate->hSRM == psObj->hSRM)
        {
            // Set approprite flag to not forget to post release to SRM.
            // If set, the MODULE shall release owning SRM
            // by calling SRM_bRelease().
            psObj->tCurrentModes |= SMS_MODE_PARENT_ASSOCIATED;
        }
        else
        {
            // Error!
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                MODULE_OBJECT_NAME": ASSOCIATE from unknown SRM.");

            SRM_bRelease(psAssociate->hSRM, SMS_OBJECT_RELEASE_BY_OTHERS);
            break;
        }

        // Check stopping flag indicating the object is currently being released
        if (0 != (psObj->tCurrentModes & SMS_MODE_STOPPING))
        {
            SMSAPI_DEBUG_vPrint(MODULE_OBJECT_NAME, 4,
                "%s: Object is currently being released. Ignore event.\n",
                psObj->pacObjectName);
            break;
        }

        // Copy in object parameters
        psObj->pacSRMName = psAssociate->pacName;
        psObj->psDevice = psAssociate->psDevice;

        // Has the RADIO already been initialized?
        if (TRUE == psObj->bRadioInitialized)
        {
            // Error!
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                MODULE_OBJECT_NAME": RADIO MODULE must not be initialized.");
            break;
        }

        // Perform RADIO specific initializations. Once the RADIO
        // is successfully initialized, a RADIO_READY event will
        // get posted.
        psObj->bRadioInitialized =
            RADIO_bInitializeModule((MODULE_OBJECT)psObj,
                psObj->tId, psObj->hSRM, psObj->pacSRMName,
                psObj->psDevice, psObj->hEventHdlr);
        if (TRUE == psObj->bRadioInitialized)
        {
            SMSAPI_DEBUG_vPrint(MODULE_OBJECT_NAME, 4,
                "%s: RADIO initialization is requested. "
                "Waiting for RADIO_READY event.\n",
                psObj->pacObjectName);
        }
        else
        {
            // Transition to error state
            vSetError(psObj, MODULE_ERROR_CODE_INITIALIZATION_ERROR);
            break;
        }

    } while (FALSE);

    return;
}

/*****************************************************************************
*
*   vHandleUnassociateEvent
*
*****************************************************************************/
static void vHandleUnassociateEvent(
    MODULE_OBJECT_STRUCT *psObj
        )
{
    BOOLEAN bResult = FALSE;

    SMSAPI_DEBUG_vPrint(MODULE_OBJECT_NAME, 4,
        "%s: SMS_EVENT_UNASSOCIATE\n", psObj->pacObjectName);

    // Now this object is ready for shutdown.
    // But we should dispatch all remaining events.
    // by posting of special event which shall be dispatched
    // last from the queue.
    bResult = bPostFinalStop(psObj);
    if (FALSE == bResult)
    {
        // Error!
        vSetError(psObj, MODULE_ERROR_CODE_EVENT_POST_ERROR);
    }

    // Now waiting for FINAL STOP event.

    return;
}

/*****************************************************************************
*
*   vHandleReleaseEvent
*
*****************************************************************************/
static void vHandleReleaseEvent(
    MODULE_OBJECT_STRUCT *psObj,
    SMS_OBJECT_RELEASE_INITIATOR_ENUM eInitiator
        )
{
    SMSAPI_DEBUG_vPrint(MODULE_OBJECT_NAME, 4,
        "%s: SMS_EVENT_RELEASE: eInitiator: %d\n",
        psObj->pacObjectName, eInitiator);

    if ((SMS_OBJECT_RELEASE_BY_APP == eInitiator) ||
        (SMS_OBJECT_RELEASE_BY_PARENT == eInitiator))
    {
        SMSAPI_DEBUG_vPrint(MODULE_OBJECT_NAME, 4,
            "%s: Handle APPLICATION initiated release...\n",
            psObj->pacObjectName);

        vHandleAppInitiatedRelease(psObj, eInitiator);
    }
    else
    {
        SMSAPI_DEBUG_vPrint(MODULE_OBJECT_NAME, 4,
            "%s: Handle GENERAL release...\n",
            psObj->pacObjectName);

        vHandleGeneralRelease(psObj);
    }

    return;
}

/*****************************************************************************
*
*   vHandleAppInitiatedRelease
*
*****************************************************************************/
static void vHandleAppInitiatedRelease(
    MODULE_OBJECT_STRUCT *psObj,
    SMS_OBJECT_RELEASE_INITIATOR_ENUM eInitiator
        )
{
    // This is release request initiated by the application
    // directly or by the parent object release.

    // Check stopping flag indicating the object is currently being released
    if (0 == (psObj->tCurrentModes & SMS_MODE_STOPPING))
    {
        // This is the first release request initiated
        // by the Application or by the Parent object (MODULE)
        psObj->tCurrentModes |= SMS_MODE_STOPPING;
    }
    else
    {
        if (SMS_OBJECT_RELEASE_BY_APP == eInitiator)
        {
            // This never must happen!
            // This means incorrect and dangerous API usage!
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                MODULE_OBJECT_NAME": OBJECT IS ALREADY BEING RELEASED "
                "DIRECTLY BY THE APPLICATION OR BY THE PARENT OBJECT!");
        }
        else if (SMS_OBJECT_RELEASE_BY_PARENT == eInitiator)
        {
            // This is Parent initiated release occurred after
            // application initiated release, so just ignore it.

            SMSAPI_DEBUG_vPrint(MODULE_OBJECT_NAME, 4,
                "%s: Object is already being released. "
                "PARENT initiated release is ignored.\n",
                psObj->pacObjectName);
        }
        else
        {
            // Error!
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                MODULE_OBJECT_NAME": Incorrect release request.");
        }

        return;
    }

    // Initiate associated DECODERs release.
    // It's Ok if there are no DECODER(s) attached.
    // Associated DECODERs shall not be released
    // until the MODULE unassociates them later.
    // If fail, will hope that DECODER(s)
    // are released directly by the Application.
    bReleaseDecoders(psObj);

    // Abort firmware update if in progress
    if ( psObj->eState == MODULE_STATE_UPDATING )
    {
        RADIO_vAbortFirmwareUpdate((MODULE_OBJECT)psObj);
    }

    // Tell the DSM that this MODULE is being released
    DATASERVICE_MGR_vUnassociateModule((MODULE_OBJECT)psObj);

    SMSAPI_DEBUG_vPrint(MODULE_OBJECT_NAME, 4,
        "%s: UNASSOCIATE request is posted to DSM.\n",
        psObj->pacObjectName);

    // Perform common release routine
    vHandleGeneralRelease(psObj);

    return;
}

/*****************************************************************************
*
*   vHandleGeneralRelease
*
*****************************************************************************/
static void vHandleGeneralRelease(
    MODULE_OBJECT_STRUCT *psObj
        )
{
    // Check for unnecessary release request
    if (0 == psObj->un16RefCounter)
    {
        // Error! This means incorrect logic somewhere inside the SMS.
        SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
            MODULE_OBJECT_NAME": Unnecessary release!");
        return;
    }

    // Now reference counter can be decremented
    psObj->un16RefCounter--;

    // Try to stop if the object is currently
    // being released and ready for shutdown.
    bTryStop(psObj);

    return;
}

/*****************************************************************************
*
*   vHandleStopEvent
*
*****************************************************************************/
static void vHandleStopEvent(
    MODULE_OBJECT_STRUCT *psObj
        )
{
    SMSAPI_DEBUG_vPrint(MODULE_OBJECT_NAME, 4,
        "%s: SMS_EVENT_STOP\n", psObj->pacObjectName);

    do
    {
        // Now shutdown process is completed
        // (all events are processed),
        // so we can uninit the Radio (if initialized)
        // or destroy right now.
        if (TRUE == psObj->bRadioInitialized)
        {
            RADIO_vUninitializeModule((MODULE_OBJECT)psObj, FALSE);

            // Waiting for RADIO_RELEASED event
            SMSAPI_DEBUG_vPrint(MODULE_OBJECT_NAME, 4,
                "%s: RADIO uninitialization is requested. "
                "Waiting for RADIO_RELEASED event.\n",
                psObj->pacObjectName);
            break;
        }

        // Now everything is uninitialized.
        // So the object can be safely destroyed.

        // If this MODULE is associated with SRM,
        // we have to release SRM now.
        bReleaseParentSRM(psObj);

        // Stop this MODULE now
        vUpdateState(psObj, MODULE_STATE_STOPPED);

    } while (FALSE);

    return;
}

/*****************************************************************************
*
*   vHandleRemoveDecoderEvent
*
*****************************************************************************/
static void vHandleRemoveDecoderEvent(
    MODULE_OBJECT_STRUCT *psObj,
    DECODER_OBJECT hDecoder
        )
{
    OSAL_RETURN_CODE_ENUM eReturnCode;
    MODULE_DECODER_ENTRY_STRUCT sSearchCriteria;
    BOOLEAN bResult = FALSE;
    OSAL_LINKED_LIST_ENTRY hEntry = OSAL_INVALID_LINKED_LIST_ENTRY;
    MODULE_DECODER_ENTRY_STRUCT *psDecoder;

    SMSAPI_DEBUG_vPrint(MODULE_OBJECT_NAME, 4,
        "%s: SMS_EVENT_REMOVE_DECODER. hDecoder: %p\n",
        psObj->pacObjectName, hDecoder);

    do
    {
        // init search criteria struct with input data
        sSearchCriteria.hDecoder = hDecoder;

        // Find this Decoder's entry (if it exists)
        eReturnCode = OSAL.eLinkedListSearch(
            psObj->hDecoderList, &hEntry, &sSearchCriteria);
        if (OSAL_SUCCESS != eReturnCode)
        {
            // Error!
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                MODULE_OBJECT_NAME": Locating DECODER entry: %d (%s).",
                eReturnCode, OSAL.pacGetReturnCodeName(eReturnCode));
            break;
        }

        psDecoder =
            (MODULE_DECODER_ENTRY_STRUCT *)OSAL.pvLinkedListThis(hEntry);

        // Remove this Decoder from the list
        eReturnCode = OSAL.eLinkedListRemove(hEntry);
        if (OSAL_SUCCESS != eReturnCode)
        {
            // Error!
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                MODULE_OBJECT_NAME": Could not remove DECODER entry: %d (%s).",
                eReturnCode, OSAL.pacGetReturnCodeName(eReturnCode));

            // Transition to ERROR state
            vSetError(psObj, MODULE_ERROR_CODE_DECODER_LIST_ERROR);
            break;
        }

        // Destroy this entry
        SMSO_vDestroy((SMS_OBJECT)psDecoder);

        SMSAPI_DEBUG_vPrint(MODULE_OBJECT_NAME, 4,
            "%s:  Removed DECODER entry from list.\n",
            psObj->pacObjectName);

        // Post UNASSOCIATE event to the DECODER
        // to confirm that the DECODER is removed.
        bResult = DECODER_bUnassociate(hDecoder);
        if (FALSE == bResult)
        {
            // Transition to ERROR state
            vSetError(psObj,
                MODULE_ERROR_CODE_UNABLE_TO_STOP_DECODER);
            break;
        }

        SMSAPI_DEBUG_vPrint(MODULE_OBJECT_NAME, 4,
            "%s: UNASSOCIATE is posted to DECODER.\n",
            psObj->pacObjectName);

    } while (FALSE);

    // Try to stop if the object is currently
    // being released and ready for shutdown.
    bTryStop(psObj);

    return;
}

/*****************************************************************************
*
*   vHandleResetEvent
*
*****************************************************************************/
static void vHandleResetEvent(
    MODULE_OBJECT_STRUCT *psObj,
    DECODER_OBJECT hDecoder
        )
{
    OSAL_RETURN_CODE_ENUM eReturnCode;

    SMSAPI_DEBUG_vPrint(MODULE_OBJECT_NAME, 4,
        "%s: SMS_EVENT_RESET: hDecoder: %p\n",
        psObj->pacObjectName, hDecoder);

    // Check stopping flag indicating the object is currently being released
    if (0 != (psObj->tCurrentModes & SMS_MODE_STOPPING))
    {
        SMSAPI_DEBUG_vPrint(MODULE_OBJECT_NAME, 4,
            "%s: Object is currently being released. Ignore event.\n",
            psObj->pacObjectName);
        return;
    }

    // If the provided DECODER is non-valid, then it is this MODULE
    // which is being instructed to reset.
    if (DECODER_INVALID_OBJECT == hDecoder)
    {
        // Indicate we have been instructed to RESET
        psObj->tCurrentModes |= SMS_MODE_RESET;

        // Tell all attached DECODERs to RESET...
        // We cannot reset ourself until all attached DECODERs
        // are in RESET themselves. So we now tell all
        // attached DECODERs to RESET themselves. They will let us
        // know when they have completed the reset
        // (back to INITIAL_STATE)
        eReturnCode =
            OSAL.eLinkedListIterate(
                psObj->hDecoderList,
                bResetDecoderIterator,
                (void *)NULL);
        if(eReturnCode == OSAL_NO_OBJECTS)
        {
            // Transition to INITIAL now (no DECODERs)
            vUpdateState(psObj, MODULE_STATE_INITIAL);

            // If this MODULE is associated with SRM,
            // we have to release SRM now.
            bReleaseParentSRM(psObj);
        }
        else if(eReturnCode != OSAL_SUCCESS)
        {
            // Transition to error state
            vSetError(psObj, MODULE_ERROR_CODE_DECODER_LIST_ERROR);
        }
    }
    // We have been told a specific DECODER is in reset.
    // If we ourself are not already resetting then we can
    // just simply let the specified DECODER know we are READY.
    else if((psObj->tCurrentModes & SMS_MODE_RESET) != SMS_MODE_RESET)
    {
        bAssociateWithDecoder(psObj, hDecoder);
    }
    // Ok, so we have been told a specific DECODER is in RESET
    // but we are resetting ourself. This means we need to wait
    // until ALL DECODERs are RESET themselves (in the INITIAL
    // state) before we can RESET ourself which will have the effect
    // making all these DECODERs READY once we are READY.
    else
    {
        MODULE_DECODER_IN_RESET_STRUCT sInReset;

        // Populate iteration struct, this DECODER is in INITIAL
        sInReset.hDecoder = hDecoder;
        sInReset.bAllInReset = TRUE;

        // Check if all attached DECODERs are in the INITIAL
        // state (i.e. they have been and are in RESET).
        eReturnCode =
            OSAL.eLinkedListIterate(
                psObj->hDecoderList,
                bAllInResetIterator, (void *)&sInReset
                    );
        if((eReturnCode == OSAL_NO_OBJECTS) ||
            ((eReturnCode == OSAL_SUCCESS) &&
                            (sInReset.bAllInReset == TRUE)))
        {
            // Transition to INITIAL now (no DECODERs exist
            // or all DECODERs are currently in RESET).
            vUpdateState(psObj, MODULE_STATE_INITIAL);

            // If this MODULE is associated with SRM,
            // we have to release SRM now.
            bReleaseParentSRM(psObj);
        }
        else if(eReturnCode != OSAL_SUCCESS)
        {
            // Transition to error state
            vSetError(psObj, MODULE_ERROR_CODE_DECODER_LIST_ERROR);
        }
    }

    // Don't worry if we were told a DECODER was reset but we are
    // not READY then when we become READY we will inform all
    // attached DECODERs anyway.

    return;
}
