/******************************************************************************/
/*                    Copyright (c) Sirius XM Radio, Inc.                     */
/*                            All Rights Reserved                             */
/*         Licensed Materials - Property of Sirius XM Radio, Inc.             */
/******************************************************************************/
/*******************************************************************************
 *
 * DESCRIPTION
 *
 * This is the Object:IMAGE header file.
 *
 ******************************************************************************/

#ifndef IMAGE_OBJ_H_
#define IMAGE_OBJ_H_

#include "standard.h"

#include "sms_api.h"

  /**********************/
 /** GLOBAL CONSTANTS **/
/**********************/

  /*********************/
 /** GLOBAL TYPEDEFS **/
/*********************/

typedef enum image_property_internal_enum
{
    IMAGE_PROPERTY_INTERNAL_WIDTH,
    IMAGE_PROPERTY_INTERNAL_HEIGHT,
    IMAGE_PROPERTY_INTERNAL_BITS_PER_PIXEL,
    IMAGE_PROPERTY_INTERNAL_ID,
    IMAGE_PROPERTY_INTERNAL_VERSION,
    IMAGE_PROPERTY_INTERNAL_BACKGROUND,
    IMAGE_PROPERTY_INTERNAL_CAPTION,
    IMAGE_PROPERTY_INTERNAL_INVALID

} IMAGE_PROPERTY_INTERNAL_ENUM;

typedef enum image_property_internal_type_enum
{
    IMAGE_PROPERTY_INTERNAL_TYPE_UN32,
    IMAGE_PROPERTY_INTERNAL_TYPE_STRING,
    IMAGE_PROPERTY_INTERNAL_TYPE_BACKGROUND

} IMAGE_PROPERTY_INTERNAL_TYPE_ENUM;

typedef struct image_background_struct
{
    // Background Color Info
    BOOLEAN bUseBackgroundColor;
    RGB_STRUCT sBackgroundColor;

    // Tint Info
    BOOLEAN bUseLineBitmap;
    size_t tNumLineBitmapEntries;
    const RGB_STRUCT *psLineBitmap;

} IMAGE_BACKGROUND_STRUCT;

typedef struct image_property_internal_value_struct
{
    IMAGE_PROPERTY_INTERNAL_TYPE_ENUM eType;

    union
    {
        UN32 un32Value;
        STRING_OBJECT hStringValue;
        IMAGE_BACKGROUND_STRUCT *psBackground;
    } uData;

} IMAGE_PROPERTY_INTERNAL_VALUE_STRUCT;

typedef struct image_intf_struct
{
    // This is a size of the extra memory which be allocated
    // in addition to the IMAGE object data.
    const size_t tSpecificDataSize;

    // This function is called by the IMAGE object during object creation
    // to initialize caller specific data
    BOOLEAN (*bInitSpecificData)(
        IMAGE_OBJECT hImage,
        void *pvSpecificData,
        void *pvArg
            );

    // This function is called by the IMAGE object after IMAGE object fully
    // created to let certain implementation update some specific data
    // which based on constructed object. At this point Image file path
    // is available to use.
    BOOLEAN (*bCreationPostStep)(
        IMAGE_OBJECT hImage,
        void *pvSpecificData
            );

    // This function is called by the IMAGE object during object destruction
    // to let owner clean up specific data if needed
    void (*vUninitSpecificData)(
        IMAGE_OBJECT hImage,
        void *pvSpecificData
            );

    // This function shall calculate the image file name length taking into
    // account room for C-string null-terminator.
    BOOLEAN (*bCalculateImageFileNameLen)(
        IMAGE_OBJECT hImage,
        const char *pacFilePath,
        void *pvSpecificData,
        size_t *ptLength
            );

    // This function shall create path for the image file
    BOOLEAN (*bCreateImageFileName)(
        IMAGE_OBJECT hImage,
        const char *pacFilePath,
        void *pvSpecificData,
        char *pacBuffer,
        size_t tBufferSize
            );

    // This function provides property values via certain
    // interface implementation.
    SMSAPI_RETURN_CODE_ENUM (*eProperty)(
        IMAGE_OBJECT hImage,
        IMAGE_PROPERTY_INTERNAL_ENUM eProperty,
        void *pvSpecificData,
        IMAGE_PROPERTY_INTERNAL_VALUE_STRUCT *psValue
            );

    // This function allows to modify property of the object.
    SMSAPI_RETURN_CODE_ENUM (*eUpdateProperty)(
        IMAGE_OBJECT hImage,
        IMAGE_PROPERTY_INTERNAL_ENUM eProperty,
        void *pvSpecificData,
        const IMAGE_PROPERTY_INTERNAL_VALUE_STRUCT *psValue,
        BOOLEAN *pbUpdated
            );

} IMAGE_INTF_STRUCT;

  /*****************************/
 /** GLOBAL STRUCT TEMPLATES **/
/*****************************/

  /****************************/
 /** GLOBAL UNION TEMPLATES **/
/****************************/

  /**********************/
 /** GLOBAL VARIABLES **/
/**********************/

  /*******************/
 /** GLOBAL MACROS **/
/*******************/

  /***********************/
 /** GLOBAL PROTOTYPES **/
/***********************/

/* Object Friend Prototypes */

IMAGE_OBJECT IMAGE_hCreate (
    SMS_OBJECT hOwner,
    const char *pacFilePath,
    IMAGE_FORMAT_ENUM eFormat,
    const IMAGE_INTF_STRUCT *psIntf,
    void *pvArg,
    BOOLEAN bFileExists
        );

IMAGE_OBJECT IMAGE_hCreateCopy (
    IMAGE_OBJECT hSrcImage
        );

SMSAPI_RETURN_CODE_ENUM IMAGE_eProperty (
    IMAGE_OBJECT hImage,
    IMAGE_PROPERTY_INTERNAL_ENUM eProperty,
    IMAGE_PROPERTY_INTERNAL_VALUE_STRUCT *psValue
        );

SMSAPI_RETURN_CODE_ENUM IMAGE_eUpdateProperty (
    IMAGE_OBJECT hImage,
    IMAGE_PROPERTY_INTERNAL_ENUM eProperty,
    const IMAGE_PROPERTY_INTERNAL_VALUE_STRUCT *psValue,
    BOOLEAN *pbUpdated
        );

void *IMAGE_pvData (
    IMAGE_OBJECT hImage
        );

N32 IMAGE_n32FPrintf(
    IMAGE_OBJECT hImage,
    FILE *psFile
        );

void IMAGE_vDestroy (
    IMAGE_OBJECT hImage
        );

const RGB_STRUCT *IMAGE_psGetLineBitmap (
    IMAGE_OBJECT hImage
        );

#endif /* IMAGE_OBJ_H_ */
