/******************************************************************************/
/*                    Copyright (c) Sirius XM Radio, Inc.                     */
/*                            All Rights Reserved                             */
/*         Licensed Materials - Property of Sirius XM Radio, Inc.             */
/******************************************************************************/
/*******************************************************************************
 *
 * DESCRIPTION
 *
 *  This module contains the Object:FORECAST implementation for the
 *  Sirius Module Services (SMS)
 *
 ******************************************************************************/
#include "standard.h"
#include "osal.h"

#include "sms_api.h"
#include "sms_obj.h"
#include "_forecast_obj.h"
#include "forecast_obj.h"
#include "weather_event_obj.h"
#include "temperature_obj.h"
#include "precipitation_obj.h"
#include "wind_obj.h"
#include "humidity_obj.h"
#include "uvinfo_obj.h"
#include "air_obj.h"

#include "sms_api_debug.h"

/*****************************************************************************
                             PUBLIC FUNCTIONS
*****************************************************************************/

/*****************************************************************************
*
*       un32TimeStamp
*
*****************************************************************************/
static UN32 un32TimeStamp (
    FORECAST_OBJECT hForecast
        )
{
    FORECAST_OBJECT_STRUCT *psObj =
        (FORECAST_OBJECT_STRUCT *)hForecast;
    BOOLEAN bOwner;
    UN32 un32TimeStamp = 0;

    bOwner = SMSO_bOwner((SMS_OBJECT)hForecast);
    if (bOwner == TRUE)
    {
        un32TimeStamp = psObj->un32TimeStamp;
    }

    return un32TimeStamp;
}


/*****************************************************************************
*
*   eWeatherEvent
*
*****************************************************************************/
static SMSAPI_RETURN_CODE_ENUM eWeatherEvent (
    FORECAST_OBJECT hForecast,
    WEATHER_EVENT_OBJECT *phWeatherEvent
        )
{
    BOOLEAN bOwner = FALSE;
    SMSAPI_RETURN_CODE_ENUM eResult = SMSAPI_RETURN_CODE_ERROR;
    FORECAST_OBJECT_STRUCT *psObj = (FORECAST_OBJECT_STRUCT *) hForecast;

    do
    {
        if (phWeatherEvent == NULL)
        {
            eResult = SMSAPI_RETURN_CODE_INVALID_INPUT;
            break;
        }

        bOwner = SMSO_bOwner((SMS_OBJECT) hForecast);
        if (bOwner == FALSE)
        {
            eResult = SMSAPI_RETURN_CODE_NOT_OWNER;
            break;
        }

        if (psObj->hWeatherEvent == WEATHER_EVENT_INVALID_OBJECT)
        {
            psObj->hWeatherEvent = WEATHER_EVENT_hCreate(
                    (SMS_OBJECT) hForecast,
                    TRUE,
                    psObj->sData.tEvent);
        }
        if (psObj->hWeatherEvent == WEATHER_EVENT_INVALID_OBJECT)
        {
            break;
        }

        *phWeatherEvent = psObj->hWeatherEvent;
        eResult = SMSAPI_RETURN_CODE_SUCCESS;
    } while (FALSE);

    return eResult;
}


/*****************************************************************************
*
*   eTemperature
*
*****************************************************************************/
static SMSAPI_RETURN_CODE_ENUM eTemperature (
    FORECAST_OBJECT hForecast,
    TEMPERATURE_OBJECT *phTemperature
        )
{
    BOOLEAN bOwner = FALSE;
    SMSAPI_RETURN_CODE_ENUM eResult = SMSAPI_RETURN_CODE_ERROR;
    FORECAST_OBJECT_STRUCT *psObj = (FORECAST_OBJECT_STRUCT *) hForecast;
    BOOLEAN bCurrentFlag = FALSE;
    BOOLEAN bHighFlag = FALSE;
    BOOLEAN bLowFlag = FALSE;

    do
    {
        if (phTemperature == NULL)
        {
            eResult = SMSAPI_RETURN_CODE_INVALID_INPUT;
            break;
        }

        bOwner = SMSO_bOwner((SMS_OBJECT) hForecast);
        if (bOwner == FALSE)
        {
            eResult = SMSAPI_RETURN_CODE_NOT_OWNER;
            break;
        }

        if (psObj->hTemperature == TEMPERATURE_INVALID_OBJECT)
        {
            if (psObj->sData.un16Flags & (1 << FORECAST_AVERAGE_TEMP_FLAG))
            {
                bCurrentFlag = TRUE;
            }
            if (psObj->sData.un16Flags & (1 << FORECAST_HIGH_TEMP_FLAG))
            {
                bHighFlag = TRUE;
            }
            if (psObj->sData.un16Flags & (1 << FORECAST_LOW_TEMP_FLAG))
            {
                bLowFlag = TRUE;
            }


            if ((bCurrentFlag == TRUE) ||
                    (bHighFlag == TRUE) ||
                    (bLowFlag == TRUE))
            {
                psObj->hTemperature = TEMPERATURE_hCreate(
                        (SMS_OBJECT)hForecast,
                        bHighFlag,
                        psObj->sData.n16Tmax,
                        bLowFlag,
                        psObj->sData.n16Tmin,
                        bCurrentFlag,
                        psObj->sData.n16Temp);
            }
            else
            {
                eResult = SMSAPI_RETURN_CODE_NOT_FOUND;
                break;
            }
        }

        if (psObj->hTemperature == TEMPERATURE_INVALID_OBJECT)
        {
            break;
        }
        *phTemperature = psObj->hTemperature;
        eResult = SMSAPI_RETURN_CODE_SUCCESS;
    } while (FALSE);

    return eResult;
}


/*****************************************************************************
*
*   ePrecipitation
*
*****************************************************************************/
static SMSAPI_RETURN_CODE_ENUM ePrecipitation (
    FORECAST_OBJECT hForecast,
    PRECIPITATION_OBJECT *phPrecipitation
        )
{
    BOOLEAN bOwner = FALSE;
    SMSAPI_RETURN_CODE_ENUM eResult = SMSAPI_RETURN_CODE_ERROR;
    FORECAST_OBJECT_STRUCT *psObj = (FORECAST_OBJECT_STRUCT *) hForecast;
    BOOLEAN bPrecipAmountFlag = FALSE;
    BOOLEAN bPrecipChanceFlag = FALSE;

    do
    {
        if (phPrecipitation == NULL)
        {
            eResult = SMSAPI_RETURN_CODE_INVALID_INPUT;
            break;
        }

        bOwner = SMSO_bOwner((SMS_OBJECT) hForecast);
        if (bOwner == FALSE)
        {
            eResult = SMSAPI_RETURN_CODE_NOT_OWNER;
            break;
        }

        if (psObj->hPrecipitation == PRECIPITATION_INVALID_OBJECT)
        {
            if (psObj->sData.un16Flags & (1  << FORECAST_PRECIP_AMOUNT_FLAG))
            {
                bPrecipAmountFlag = TRUE;
            }

            if (psObj->sData.un16Flags & (1  << FORECAST_PRECIP_CHANCE_FLAG))
            {
                bPrecipChanceFlag = TRUE;
            }

            if ((bPrecipAmountFlag == TRUE) ||
                    (bPrecipChanceFlag == TRUE))
            {
                psObj->hPrecipitation = PRECIPITATION_hCreate(
                        (SMS_OBJECT) hForecast,
                        bPrecipChanceFlag,
                        bPrecipAmountFlag,
                        psObj->sData.un8Pop,
                        psObj->sData.tPrecip);
            }
            else
            {
                eResult = SMSAPI_RETURN_CODE_NOT_FOUND;
                break;
            }
        }

        if (psObj->hPrecipitation == PRECIPITATION_INVALID_OBJECT)
        {
            break;
        }
        *phPrecipitation = psObj->hPrecipitation;
        eResult = SMSAPI_RETURN_CODE_SUCCESS;
    } while (FALSE);

    return eResult;
}


/*****************************************************************************
*
*   eWind
*
*****************************************************************************/
static SMSAPI_RETURN_CODE_ENUM eWind (
    FORECAST_OBJECT hForecast,
    WIND_OBJECT *phWind
        )
{
    BOOLEAN bOwner = FALSE;
    SMSAPI_RETURN_CODE_ENUM eResult = SMSAPI_RETURN_CODE_ERROR;
    FORECAST_OBJECT_STRUCT *psObj = (FORECAST_OBJECT_STRUCT *) hForecast;
    BOOLEAN bWindSpeedFlag = FALSE;
    BOOLEAN bWindDirFlag = FALSE;

    do
    {
        if (phWind == NULL)
        {
            eResult = SMSAPI_RETURN_CODE_INVALID_INPUT;
            break;
        }

        bOwner = SMSO_bOwner((SMS_OBJECT)hForecast);
        if (bOwner == FALSE)
        {
            eResult = SMSAPI_RETURN_CODE_NOT_OWNER;
            break;
        }

        if (psObj->hWind == WIND_INVALID_OBJECT)
        {
            if (psObj->sData.un16Flags & (1 << FORECAST_WIND_DIRECTION_FLAG))
            {
                bWindDirFlag = TRUE;
            }

            if (psObj->sData.un16Flags & (1 << FORECAST_WIND_SPEED_FLAG))
            {
                bWindSpeedFlag = TRUE;
            }


            if ((bWindDirFlag == TRUE) ||
                    (bWindSpeedFlag == TRUE))
            {
                psObj->hWind = WIND_hCreate(
                        (SMS_OBJECT) hForecast, TRUE, psObj->sData.tData);
            }
            else
            {
                eResult = SMSAPI_RETURN_CODE_NOT_FOUND;
                break;
            }
        }

        if (psObj->hWind == WIND_INVALID_OBJECT)
        {
            break;
        }
        *phWind = psObj->hWind;
        eResult = SMSAPI_RETURN_CODE_SUCCESS;
    } while (FALSE);

    return eResult;
}


/*****************************************************************************
*
*   eHumidity
*
*****************************************************************************/
static SMSAPI_RETURN_CODE_ENUM eHumidity (
    FORECAST_OBJECT hForecast,
    HUMIDITY_OBJECT *phHumidity
        )
{
    BOOLEAN bOwner = FALSE;
    SMSAPI_RETURN_CODE_ENUM eResult = SMSAPI_RETURN_CODE_ERROR;
    FORECAST_OBJECT_STRUCT *psObj = (FORECAST_OBJECT_STRUCT *) hForecast;
    BOOLEAN bHumidFlag = FALSE;

    do
    {
        if (phHumidity == NULL)
        {
            eResult = SMSAPI_RETURN_CODE_INVALID_INPUT;
            break;
        }

        bOwner = SMSO_bOwner((SMS_OBJECT)hForecast);
        if (bOwner == FALSE)
        {
            eResult = SMSAPI_RETURN_CODE_NOT_OWNER;
            break;
        }

        if (psObj->hHumidity == HUMIDITY_INVALID_OBJECT)
        {
            if (psObj->sData.un16Flags & (1 << FORECAST_HUMIDITY_FLAG))
            {
                bHumidFlag = TRUE;
            }
            if (bHumidFlag == TRUE)
            {
                psObj->hHumidity = HUMIDITY_hCreate(
                        (SMS_OBJECT)hForecast,
                        bHumidFlag,
                        psObj->sData.tHumid);
            }
            else
            {
                eResult = SMSAPI_RETURN_CODE_NOT_FOUND;
                break;
            }
        }
        if (psObj->hHumidity == HUMIDITY_INVALID_OBJECT)
        {
            break;
        }

        *phHumidity = psObj->hHumidity;
        eResult = SMSAPI_RETURN_CODE_SUCCESS;
    } while (FALSE);

    return eResult;
}


/*****************************************************************************
*
*   eUVInfo
*
*****************************************************************************/
static SMSAPI_RETURN_CODE_ENUM eUVInfo (
    FORECAST_OBJECT hForecast,
    UVINFO_OBJECT *phUVInfo
        )
{
    BOOLEAN bOwner = FALSE;
    SMSAPI_RETURN_CODE_ENUM eResult = SMSAPI_RETURN_CODE_ERROR;
    FORECAST_OBJECT_STRUCT *psObj = (FORECAST_OBJECT_STRUCT *) hForecast;
    BOOLEAN bUVFlag = FALSE;

    do
    {
        if (phUVInfo == NULL)
        {
            eResult = SMSAPI_RETURN_CODE_INVALID_INPUT;
            break;
        }

        bOwner = SMSO_bOwner((SMS_OBJECT)hForecast);
        if (bOwner == FALSE)
        {
            eResult = SMSAPI_RETURN_CODE_NOT_OWNER;
            break;
        }

        if (psObj->hUVInfo == UVINFO_INVALID_OBJECT)
        {
            if (psObj->sData.un16Flags & (1 << FORECAST_UV_INDEX_FLAG))
            {
                bUVFlag = TRUE;
            }
            if (bUVFlag == TRUE)
            {
                psObj->hUVInfo = UVINFO_hCreate(
                        (SMS_OBJECT) hForecast,
                        bUVFlag,
                        psObj->sData.tUV);
            }
            else
            {
                eResult = SMSAPI_RETURN_CODE_NOT_FOUND;
                break;
            }
        }
        if (psObj->hUVInfo == UVINFO_INVALID_OBJECT)
        {
            break;
        }

        *phUVInfo = psObj->hUVInfo;
        eResult = SMSAPI_RETURN_CODE_SUCCESS;
    } while (FALSE);

    return eResult;
}


/*****************************************************************************
*
*   eAir
*
*****************************************************************************/
static SMSAPI_RETURN_CODE_ENUM eAir (
    FORECAST_OBJECT hForecast,
    AIR_OBJECT *phAir
        )
{
    BOOLEAN bOwner = FALSE;
    SMSAPI_RETURN_CODE_ENUM eResult = SMSAPI_RETURN_CODE_ERROR;
    FORECAST_OBJECT_STRUCT *psObj = (FORECAST_OBJECT_STRUCT *) hForecast;
    BOOLEAN bAirFlag = FALSE;
    BOOLEAN bPollenFlag = FALSE;

    do
    {
        if (phAir == NULL)
        {
            eResult = SMSAPI_RETURN_CODE_INVALID_INPUT;
            break;
        }

        bOwner = SMSO_bOwner((SMS_OBJECT)hForecast);
        if (bOwner == FALSE)
        {
            eResult = SMSAPI_RETURN_CODE_NOT_OWNER;
            break;
        }

        if (psObj->hAir == AIR_INVALID_OBJECT)
        {
            if (psObj->sData.un16Flags & (1 << FORECAST_AIR_QUALITY_FLAG))
            {
                bAirFlag = TRUE;
            }

            if (psObj->sData.un16Flags & (1 << FORECAST_POLLEN_COUNT_FLAG))
            {
                bPollenFlag = TRUE;
            }

            if ((bAirFlag == TRUE) ||
                    (bPollenFlag == TRUE))
            {
                psObj->hAir = AIR_hCreate(
                        (SMS_OBJECT) hForecast,
                        bAirFlag,
                        psObj->sData.eAirQuality,
                        bPollenFlag,
                        psObj->sData.tPollen);
            }
            else
            {
                eResult = SMSAPI_RETURN_CODE_NOT_FOUND;
                break;
            }
        }
        if (psObj->hAir == AIR_INVALID_OBJECT)
        {
            break;
        }

        *phAir = psObj->hAir;
        eResult = SMSAPI_RETURN_CODE_SUCCESS;
    } while (FALSE);

    return eResult;
}


/*****************************************************************************
*
*   eCloudCover
*
*****************************************************************************/
static SMSAPI_RETURN_CODE_ENUM eCloudCover (
    FORECAST_OBJECT hForecast,
    FORECAST_CLOUD_COVER_ENUM *peCloudCover
        )
{
    BOOLEAN bOwner = FALSE;
    SMSAPI_RETURN_CODE_ENUM eResult = SMSAPI_RETURN_CODE_ERROR;
    FORECAST_OBJECT_STRUCT *psObj = (FORECAST_OBJECT_STRUCT *)hForecast;

    do
    {
        if (peCloudCover == NULL)
        {
            eResult = SMSAPI_RETURN_CODE_INVALID_INPUT;
            break;
        }

        bOwner = SMSO_bOwner((SMS_OBJECT)hForecast);
        if (bOwner == FALSE)
        {
            eResult = SMSAPI_RETURN_CODE_NOT_OWNER;
            break;
        }

        if (psObj->sData.un16Flags & (1 << FORECAST_CLOUD_COVER_FLAG))
        {
            *peCloudCover = psObj->sData.eCloudCover;
            eResult = SMSAPI_RETURN_CODE_SUCCESS;
        }
        else
        {
            eResult = SMSAPI_RETURN_CODE_NOT_FOUND;
        }
    } while (FALSE);

    return eResult;
}


/*****************************************************************************
*
*   n32FPrintf
*
*****************************************************************************/
static N32 n32FPrintf (
    FORECAST_OBJECT hForecast,
    FILE *psFile,
    SMSAPI_OUTPUT_OPTION_ENUM eOutputOption
        )
{
    N32 n32Return = 0, n32Temp;
    FORECAST_OBJECT_STRUCT *psObj =
        (FORECAST_OBJECT_STRUCT *) hForecast;
    BOOLEAN bOwner;
    WEATHER_EVENT_OBJECT hWeatherEvent = WEATHER_EVENT_INVALID_OBJECT;
    TEMPERATURE_OBJECT hTemperature = TEMPERATURE_INVALID_OBJECT;
    SMSAPI_RETURN_CODE_ENUM eResult;

    bOwner = SMSO_bOwner((SMS_OBJECT)hForecast);
    if ((bOwner == FALSE) || (psFile == NULL))
    {
        return EOF;
    }

    // Print Forecast information header
    n32Return = fprintf(psFile, "\nForecast");
    if (eOutputOption == SMS_OUTPUT_OPTION_GROSS)
    {
        n32Return += fprintf(psFile, " (Object: 0x%X)", hForecast);
    }
    n32Return += fprintf(psFile, ":\n");

    // These items are always printed
    eResult = FORECAST.eWeatherEvent(hForecast, &hWeatherEvent);
    if (eResult == SMSAPI_RETURN_CODE_SUCCESS)
    {
        n32Temp = WEATHER_EVENT.n32FPrintf(hWeatherEvent, psFile);
        if (n32Temp > 0)
        {
            n32Return += n32Temp;
        }
    }

    eResult = FORECAST.eTemperature(hForecast, &hTemperature);
    if (eResult == SMSAPI_RETURN_CODE_SUCCESS)
    {
        n32Temp = TEMPERATURE.n32FPrintf(hTemperature, psFile);
        if (n32Temp > 0)
        {
            n32Return += n32Temp;
        }
    }

    // Print these if verbose or gross
    if ((eOutputOption == SMS_OUTPUT_OPTION_VERBOSE) ||
        (eOutputOption == SMS_OUTPUT_OPTION_GROSS))
    {
        WIND_OBJECT hWind;
        PRECIPITATION_OBJECT hPrecipitation;
        HUMIDITY_OBJECT hHumidity;

        eResult = FORECAST.eWind(hForecast, &hWind);
        if (eResult == SMSAPI_RETURN_CODE_SUCCESS)
        {
            n32Temp = WIND.n32FPrintf(hWind, psFile);
            if (n32Temp > 0)
            {
                n32Return += n32Temp;
            }
        }

        eResult = FORECAST.ePrecipitation(hForecast, &hPrecipitation);
        if (eResult == SMSAPI_RETURN_CODE_SUCCESS)
        {
            n32Temp = PRECIPITATION.n32FPrintf(hPrecipitation, psFile);
            if (n32Temp > 0)
            {
                n32Return += n32Temp;
            }
        }

        eResult = FORECAST.eHumidity(hForecast, &hHumidity);
        if (eResult == SMSAPI_RETURN_CODE_SUCCESS)
        {
            n32Temp = HUMIDITY.n32FPrintf(hHumidity, psFile);
            if (n32Temp > 0)
            {
                n32Return += n32Temp;
            }
        }
    }

    // Print this if verbose
    if (eOutputOption == SMS_OUTPUT_OPTION_GROSS)
    {
        UVINFO_OBJECT hUVInfo;
        AIR_OBJECT hAir;
        FORECAST_CLOUD_COVER_ENUM eCloudCover =
            FORECAST_CLOUD_COVER_NO_DATA;
        const char *pacCover;
        UN32 un32TimeStamp;
        struct tm tTimeStruct;
        char acBuffer[OSAL_ASCBUFSIZE];
        TIME_T tTime;

        eResult = FORECAST.eUVInfo(hForecast, &hUVInfo);
        if (eResult == SMSAPI_RETURN_CODE_SUCCESS)
        {
            n32Temp = UVINFO.n32FPrintf(hUVInfo, psFile);
            if (n32Temp > 0)
            {
                n32Return += n32Temp;
            }
        }

        eResult = FORECAST.eAir(hForecast, &hAir);
        if (eResult == SMSAPI_RETURN_CODE_SUCCESS)
        {
            n32Temp = AIR.n32FPrintf(hAir, psFile);
            if (n32Temp > 0)
            {
                n32Return += n32Temp;
            }
        }

        eResult = FORECAST.eCloudCover(hForecast, &eCloudCover);

        pacCover = pacCloudCover(eCloudCover);
        if (eResult == SMSAPI_RETURN_CODE_SUCCESS)
        {
            n32Temp = fprintf(psFile, "\tCloudCover: %s\n",
                                pacCover);
            if (n32Temp > 0)
            {
                n32Return += n32Temp;
            }
        }

        n32Temp = fprintf(psFile, "\nHash Entry: 0x%X\n",
                            psObj->sData.hHashEntry);
        if (n32Temp > 0)
        {
            n32Return += n32Temp;
        }

        // Get timestamp of the object (in UTC)
        un32TimeStamp = FORECAST.un32TimeStamp(hForecast);
        // Convert into local time
        tTime = (TIME_T)(un32TimeStamp);
        OSAL.localtime_r( &tTime, &tTimeStruct );
        // Now convert into text for printout
        OSAL.asctime_r( &tTimeStruct, acBuffer );

        n32Temp = fprintf(psFile, "\tTimeStamp = %s\n", acBuffer);
        if (n32Temp > 0)
        {
            n32Return += n32Temp;
        }
    }

    return n32Return;
}


/*****************************************************************************
                             FRIEND FUNCTIONS
*****************************************************************************/

/*****************************************************************************
*
*   FORECAST_hCreate
*
*****************************************************************************/
FORECAST_OBJECT FORECAST_hCreate(
    SMS_OBJECT hParent,
    FORECAST_DATA_STRUCT *psData,
    UN32 un32TimeStamp
        )
{
    FORECAST_OBJECT_STRUCT *psObj;

    // Create an instance of the object
    psObj = (FORECAST_OBJECT_STRUCT *)
        SMSO_hCreate(
            FORECAST_OBJECT_NAME,
            sizeof(FORECAST_OBJECT_STRUCT),
            hParent,
            FALSE
                );

    if(psObj == NULL)
    {
        // Error!
        return FORECAST_INVALID_OBJECT;
    }

    psObj->hAir = AIR_INVALID_OBJECT;
    psObj->hHumidity = HUMIDITY_INVALID_OBJECT;
    psObj->hPrecipitation = PRECIPITATION_INVALID_OBJECT;
    psObj->hTemperature = TEMPERATURE_INVALID_OBJECT;
    psObj->hUVInfo = UVINFO_INVALID_OBJECT;
    psObj->hWeatherEvent = WEATHER_EVENT_INVALID_OBJECT;
    psObj->hWind = WIND_INVALID_OBJECT;
    OSAL.bMemCpy(&psObj->sData, psData, sizeof(psObj->sData));

    if (un32TimeStamp == 0)
    {
        OSAL.eTimeGet(&psObj->un32TimeStamp);
    }
    else
    {
        psObj->un32TimeStamp = un32TimeStamp;
    }

    gun32ForecastCounter++;

    return (FORECAST_OBJECT)psObj;
}


/*****************************************************************************
*
*   FORECAST_vDestroy
*
*****************************************************************************/
void FORECAST_vDestroy (
    FORECAST_OBJECT hForecast
        )
{
    BOOLEAN bOwner;
    FORECAST_OBJECT_STRUCT *psObj = (FORECAST_OBJECT_STRUCT *)hForecast;

    bOwner = SMSO_bOwner((SMS_OBJECT) hForecast);
    if(bOwner == TRUE)
    {
        vDestroy(psObj);
    }
    return;
}


/*****************************************************************************
*
*       FORECAST_bCompareData
*
*****************************************************************************/
BOOLEAN FORECAST_bCompareData (
    FORECAST_OBJECT hForecast,
    FORECAST_DATA_STRUCT *psForecastData
        )
{
    FORECAST_OBJECT_STRUCT *psObj =
        (FORECAST_OBJECT_STRUCT *)hForecast;
    BOOLEAN bValid, bResult = FALSE;

    do {
        bValid = SMSO_bValid((SMS_OBJECT)hForecast);
        if (bValid == FALSE)
        {
            break;
        }

        bValid = SMSO_bValid((SMS_OBJECT)psForecastData);
        if (bValid == FALSE)
        {
            break;
        }

        if (psObj->sData.un16Flags != psForecastData->un16Flags)
        {
            break;
        }

        if (psObj->sData.tEvent != psForecastData->tEvent)
        {
            break;
        }

        if ((psObj->sData.un16Flags) & (1 << FORECAST_AIR_QUALITY_FLAG))
        {
            if (psObj->sData.eAirQuality != psForecastData->eAirQuality)
            {
                break;
            }
        }

        if ((psObj->sData.un16Flags) & (1 << FORECAST_AVERAGE_TEMP_FLAG))
        {
            if (psObj->sData.n16Temp != psForecastData->n16Temp)
            {
                break;
            }
        }

        if ((psObj->sData.un16Flags) & (1 << FORECAST_CLOUD_COVER_FLAG))
        {
            if (psObj->sData.eCloudCover != psForecastData->eCloudCover)
            {
                break;
            }
        }

        if ((psObj->sData.un16Flags) & (1 << FORECAST_HIGH_TEMP_FLAG))
        {
            if (psObj->sData.n16Tmax != psForecastData->n16Tmax)
            {
                break;
            }
        }

        if ((psObj->sData.un16Flags) & (1 << FORECAST_LOW_TEMP_FLAG))
        {
            if (psObj->sData.n16Tmin != psForecastData->n16Tmin)
            {
                break;
            }
        }

        if ((psObj->sData.un16Flags) & (1 << FORECAST_POLLEN_COUNT_FLAG))
        {
            if (psObj->sData.tPollen != psForecastData->tPollen)
            {
                break;
            }
        }

        if ((psObj->sData.un16Flags) & (1 << FORECAST_HUMIDITY_FLAG))
        {
            if (psObj->sData.tHumid != psForecastData->tHumid)
            {
                break;
            }
        }

        if ((psObj->sData.un16Flags) & (1 << FORECAST_PRECIP_AMOUNT_FLAG))
        {
            if (psObj->sData.tPrecip != psForecastData->tPrecip)
            {
                break;
            }
        }

        if ((psObj->sData.un16Flags) & (1 << FORECAST_PRECIP_CHANCE_FLAG))
        {
            if (psObj->sData.un8Pop != psForecastData->un8Pop)
            {
                break;
            }
        }

        if ((psObj->sData.un16Flags) & (1 << FORECAST_UV_INDEX_FLAG))
        {
            if (psObj->sData.tUV != psForecastData->tUV)
            {
                break;
            }
        }

        if ((psObj->sData.un16Flags) & (1 << FORECAST_WIND_DIRECTION_FLAG))
        {
            if (psObj->sData.tData != psForecastData->tData)
            {
                break;
            }
        }
        //we do not compare ext data because it is not used for now

        bResult = TRUE;
    } while (FALSE);

    return bResult;
}


/*****************************************************************************
*
*       FORECAST_vUpdateTimeStamp
*
*****************************************************************************/
void FORECAST_vUpdateTimeStamp (
    FORECAST_OBJECT hForecast
        )
{
    FORECAST_OBJECT_STRUCT *psObj =
        (FORECAST_OBJECT_STRUCT *)hForecast;
    BOOLEAN bOwner;

    bOwner = SMSO_bOwner((SMS_OBJECT) hForecast);
    if(bOwner == TRUE)
    {
        OSAL.eTimeGet(&(psObj->un32TimeStamp));
    }
    return;
}


/*****************************************************************************
*
*   FORECAST_hHashEntry
*
*****************************************************************************/
WEATHER_HASH_OBJECT FORECAST_hHashEntry (
    FORECAST_OBJECT hForecast
        )
{
    FORECAST_OBJECT_STRUCT *psObj =
        (FORECAST_OBJECT_STRUCT *)hForecast;
    WEATHER_HASH_OBJECT hHashEntry = NULL;
    BOOLEAN bValid;

    bValid = SMSO_bValid((SMS_OBJECT)hForecast);

    // Verify inputs.
    if(bValid == TRUE)
    {
        hHashEntry = psObj->sData.hHashEntry;
    }

    return hHashEntry;
}



/*****************************************************************************
*
*   FORECAST_un32Count
*
*****************************************************************************/
UN32 FORECAST_un32Count ( void )
{
    return gun32ForecastCounter;
}


/*****************************************************************************
*
*   FORECAST_bUpdateHashEntry
*
*****************************************************************************/
BOOLEAN FORECAST_bUpdateHashEntry(
    FORECAST_OBJECT hForecast,
    WEATHER_HASH_OBJECT hNewHashEntry
        )
{
    FORECAST_OBJECT_STRUCT *psObj =
        (FORECAST_OBJECT_STRUCT *)hForecast;
    BOOLEAN bResult = FALSE;
    BOOLEAN bOwner;

    bOwner = SMSO_bOwner((SMS_OBJECT)hForecast);

    // Verify inputs.
    if(bOwner == TRUE)
    {
        psObj->sData.hHashEntry = hNewHashEntry;
        bResult = TRUE;
    }

    return bResult;
}


/*****************************************************************************
                             PRIVATE FUNCTIONS
*****************************************************************************/

/*****************************************************************************
*
*   vDestroy
*
*****************************************************************************/
static void vDestroy(FORECAST_OBJECT_STRUCT *psObj)
{
    if (psObj->hAir != AIR_INVALID_OBJECT)
    {
        AIR_vDestroy(psObj->hAir);
    }

    if (psObj->hHumidity != HUMIDITY_INVALID_OBJECT)
    {
        HUMIDITY_vDestroy(psObj->hHumidity);
    }

    if (psObj->hPrecipitation != PRECIPITATION_INVALID_OBJECT)
    {
        PRECIPITATION_vDestroy(psObj->hPrecipitation);
    }

    if (psObj->hTemperature != TEMPERATURE_INVALID_OBJECT)
    {
        TEMPERATURE_vDestroy(psObj->hTemperature);
    }

    if (psObj->hUVInfo != UVINFO_INVALID_OBJECT)
    {
        UVINFO_vDestroy(psObj->hUVInfo);
    }

    if (psObj->hWeatherEvent != WEATHER_EVENT_INVALID_OBJECT)
    {
        WEATHER_EVENT_vDestroy(psObj->hWeatherEvent);
    }

    if (psObj->hWind != WIND_INVALID_OBJECT)
    {
        WIND_vDestroy(psObj->hWind);
    }

    // Free object instance
    SMSO_vDestroy((SMS_OBJECT)psObj);
    gun32ForecastCounter--;
    return;
}

/*****************************************************************************
*
*   pacCloudCover
*
*****************************************************************************/
static const char *pacCloudCover (
    FORECAST_CLOUD_COVER_ENUM eCloudCover
        )
{
    const char *pacCover = "Unknown";

    switch (eCloudCover)
    {
        case FORECAST_CLOUD_COVER_CLEAR:
        {
            pacCover = "Clear";
        }
        break;

        case FORECAST_CLOUD_COVER_FEW:
        {
            pacCover = "Few";
        }
        break;

        case FORECAST_CLOUD_COVER_SCATTERED:
        {
            pacCover = "Scattered";
        }
        break;

        case FORECAST_CLOUD_COVER_BROKEN:
        {
            pacCover = "Broken";
        }
        break;

        case FORECAST_CLOUD_COVER_OVERCAST:
        {
            pacCover = "Overcast";
        }
        break;

        case FORECAST_CLOUD_COVER_NO_DATA:
        default:
        {
        }
        break;
    }

    return pacCover;
}
