/******************************************************************************/
/*                    Copyright (c) Sirius XM Radio, Inc.                     */
/*                            All Rights Reserved                             */
/*         Licensed Materials - Property of Sirius XM Radio, Inc.             */
/******************************************************************************/
/*******************************************************************************
 *
 * DESCRIPTION
 *
 *  This module contains the Object:EV CHARGING implementation for the
 *  Simple Module Services (SMS)
 *
 ******************************************************************************/
#include <stdio.h>
#include <string.h>
#include "standard.h"
#include "osal.h"

#include "sms_api.h"
#include "sms_obj.h"
#include "sms.h"
#include "sms_version.h"
#include "sql_interface_obj.h"
#include "dataservice_mgr_impl.h"
#include "string_obj.h"
#include "location_obj.h"
#include "fuel_db_constants.h"
#include "locid_obj.h"
#include "db_util.h"
#include "_ev_charging_service.h"
#include "ev_charging_service.h"
#include "fuel_interface.h"
#include "fuel_mgr_obj.h"
#include "fuel_station_obj.h"

#include "sms_api_debug.h"
static const char *gpacThisFile = __FILE__;


/*****************************************************************************
                             PUBLIC FUNCTIONS
*****************************************************************************/

/*****************************************************************************
*
*   hConnect
*
*   Connects to the fuel price database and initializes the fuel price
*   database object
*
*   Inputs:
*       hFuelService - A valid handle to a FUEL_SERVICE_OBJECT representing
*           the central fuel service manager.
*       *pacRefDBDirPath - The path to the reference database
*       *peErrorCode - Reports any error encountered
*
*   Output:
*       An object handle representing this instantiation of the
*       FUEL_DB_INTERFACE_OBJECT
*
*****************************************************************************/
static FUEL_DB_INTERFACE_OBJECT hConnect (
    FUEL_SERVICE_OBJECT hFuelService,
    SMS_OBJECT hParent,
    const char *pacRefDBDirPath,
    const FUEL_DB_INTERFACE_STRUCT *psDBInterface,
    const FUEL_OTA_INTERFACE_STRUCT *psOTAInterface,
    DATASERVICE_ERROR_CODE_ENUM *peErrorCode
        )
{
    EV_CHARGING_DB_OBJ_STRUCT *psObj =
        (EV_CHARGING_DB_OBJ_STRUCT *)NULL;
    BOOLEAN bOwner;
    char *pacBaselineLogoPath = NULL,
         *pacLogoPath = NULL;

    if ((hFuelService == FUEL_SERVICE_INVALID_OBJECT) ||
        (peErrorCode == NULL))
    {
        return FUEL_DB_INTERFACE_INVALID_OBJECT;
    }

    // Initialize
    *peErrorCode = DATASERVICE_ERROR_CODE_NONE;

    // Verify we own service handle
    bOwner = SMSO_bOwner(hParent);
    if (bOwner == FALSE)
    {
        return FUEL_DB_INTERFACE_INVALID_OBJECT;
    }

    // Create an instance of this object
    psObj = (EV_CHARGING_DB_OBJ_STRUCT *)
        SMSO_hCreate(
            EV_CHARGING_NAME,
            sizeof(EV_CHARGING_DB_OBJ_STRUCT),
            hParent, FALSE );

    if (psObj == NULL)
    {
        return FUEL_DB_INTERFACE_INVALID_OBJECT;
    }

    // Save the service handle
    psObj->hFuelService = hFuelService;

    // Save this path data
    psObj->pacRefDatabaseDirPath = pacRefDBDirPath;

    // Not in a transaction right now
    psObj->bInTransaction = FALSE;

    // Initialize version information
    psObj->sVerCtrl.n16DBContentVersion = EV_CHARGING_TEXT_INITIAL_DB_VER;

    do
    {
        BOOLEAN bOk, bServiceUpdated = FALSE;
        char *pacDatabaseFilePath,
             *pacDatabaseFilePathB;

        bOk = bInitializeWellKnownFuelTypes(psObj);
        if (bOk == FALSE)
        {
            // Error!
            *peErrorCode = DATASERVICE_ERROR_CODE_GENERAL;
            break;
        }

        // Construct the path needed for the reference database
        bOk = DB_UTIL_bCreateFilePath(
            psObj->pacRefDatabaseDirPath,
            EV_CHARGING_DATABASE_FOLDER,
            EV_CHARGING_REF_DATABASE_FILENAMEA, &pacDatabaseFilePath);

        if (bOk == FALSE)
        {
            // Error!  Couldn't build database path
            *peErrorCode = DATASERVICE_ERROR_CODE_GENERAL;
            break;
        }

        // Construct the path needed for the reference database
        bOk = DB_UTIL_bCreateFilePath(
            psObj->pacRefDatabaseDirPath,
            EV_CHARGING_DATABASE_FOLDER,
            EV_CHARGING_REF_DATABASE_FILENAMEB, &pacDatabaseFilePathB);

        if (bOk == FALSE)
        {
            // Error!  Couldn't build database path
            *peErrorCode = DATASERVICE_ERROR_CODE_GENERAL;
            break;
        }

        // Connect to the ref database bank, but
        // never perform an integrity check
        psObj->hSQLRefConnection =
            DB_UTIL_hConnectToReferenceBank(
                &pacDatabaseFilePath[0],
                &pacDatabaseFilePathB[0],
                &psObj->pacCurRefDatabaseFilePath,
                n32ExtractDataVersion, NULL,
                TRUE, &bServiceUpdated,
                GsEVChargingOTAIntf.tMaxVersionBitlen,
                (DB_UTIL_CHECK_VERSION_HANDLER)bVerifyAndLoadDBVersionFields,
                (void *)&psObj->sVerCtrl, peErrorCode,
                SQL_INTERFACE_OPTIONS_READONLY |
                SQL_INTERFACE_OPTIONS_SKIP_CORRUPTION_CHECK);

        if (psObj->pacCurRefDatabaseFilePath == pacDatabaseFilePath)
        {
            psObj->pacNextRefDatabaseFilePath = pacDatabaseFilePathB;
        }
        else
        {
            psObj->pacNextRefDatabaseFilePath = pacDatabaseFilePath;
        }

        // Stop here if the connection failed
        if (psObj->hSQLRefConnection == SQL_INTERFACE_INVALID_OBJECT)
        {
            break;
        }

        // Build the path to the persitent storage DB
        bOk = DB_UTIL_bCreateFilePath(
            NULL, // We don't know the base path
            EV_CHARGING_DATABASE_FOLDER,
            EV_CHARGING_PERSIST_DATABASE_FILENAME,
            &pacDatabaseFilePath);

        if (bOk == FALSE)
        {
            // Error!  Couldn't build database path
            *peErrorCode = DATASERVICE_ERROR_CODE_GENERAL;
            break;
        }

        // Connect to the persitent storage database
        psObj->hSQLPersistConnection =
            DB_UTIL_hConnectToPersistent(
                &pacDatabaseFilePath[0],
                (DB_UTIL_CREATE_DB_HANDLER)bCreatePersistDBTables,
                (void *)psObj,
                (DB_UTIL_CHECK_VERSION_HANDLER)bVerifyAndLoadDBVersionFields,
                (void *)&psObj->sVerCtrl,
                peErrorCode);

        // Done with that
        SMSO_vDestroy((SMS_OBJECT)pacDatabaseFilePath);

        if (psObj->hSQLPersistConnection == SQL_INTERFACE_INVALID_OBJECT)
        {
            break;
        }

        // Build the path to the persitent storage DB
        bOk = DB_UTIL_bCreateFilePath(
            psObj->pacRefDatabaseDirPath,
            EV_CHARGING_DATABASE_FOLDER,
            EV_CHARGING_BASELINE_FOLDER,
            &pacBaselineLogoPath);

        if (bOk == FALSE)
        {
            // Error!  Couldn't build database path
            *peErrorCode = DATASERVICE_ERROR_CODE_GENERAL;
            break;
        }

        // Build the path to the persitent storage DB
        bOk = DB_UTIL_bCreateFilePath(
            NULL, // We don't know the base path
            EV_CHARGING_DATABASE_FOLDER,
            NULL,
            &pacLogoPath);

        if (bOk == FALSE)
        {
            // Error!  Couldn't build database path
            *peErrorCode = DATASERVICE_ERROR_CODE_GENERAL;
            break;
        }

        //pacBaselineLogoPath
        // Now it's time to enable logo support
        bOk = GsFuelMgrIntf.bEnableLogoSupport(
            psObj->hFuelService, pacLogoPath, pacBaselineLogoPath);
        if (bOk == FALSE)
        {
            // Error!  Couldn't build database path
            *peErrorCode = DATASERVICE_ERROR_CODE_GENERAL;
            break;
        }

        // Logo support has been added, and
        // these are now owned by the fuel manager
        pacBaselineLogoPath = NULL;
        pacLogoPath = NULL;

        if (bServiceUpdated == TRUE)
        {
            bOk = bFlushPrices(psObj);
        }
        else
        {
            // Age out price entries now
            bOk = bAgeoutPrices(psObj);
        }

        if (bOk == FALSE)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                EV_CHARGING_NAME": Unable to flush prices");
            *peErrorCode = DATASERVICE_ERROR_CODE_GENERAL;
            break;
        }

        // Now, load all other tables that
        // we need to have available at all times
        bOk = bLoadTables( psObj );
        if (bOk == FALSE)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                EV_CHARGING_NAME": Unable to load fuel tables!");
            *peErrorCode = DATASERVICE_ERROR_CODE_DATABASE_ACCESS_FAILURE;
            break;
        }

        return (FUEL_DB_INTERFACE_OBJECT)psObj;

    } while (FALSE);

    if (psObj != (EV_CHARGING_DB_OBJ_STRUCT *)NULL)
    {
        vDisconnect((FUEL_DB_INTERFACE_OBJECT)psObj);
    }

    // Clean these up
    if (NULL != pacBaselineLogoPath)
    {
        SMSO_vDestroy((SMS_OBJECT)pacBaselineLogoPath);
    }

    if (NULL != pacLogoPath)
    {
        SMSO_vDestroy((SMS_OBJECT)pacLogoPath);
    }

    return FUEL_DB_INTERFACE_INVALID_OBJECT;
}

/*****************************************************************************
*
*   vDisconnect
*
*   Uninitializes and disconnect from the fuel price db.
*
*   Inputs:
*       hDBInterface - The interface object handle created with a
*           previous call to hConnect.
*
*****************************************************************************/
static void vDisconnect (
    FUEL_DB_INTERFACE_OBJECT hDBInterface
        )
{
    BOOLEAN bOwner;

    bOwner = SMSO_bOwner((SMS_OBJECT)hDBInterface);
    if (bOwner == TRUE)
    {
        EV_CHARGING_DB_OBJ_STRUCT *psObj =
            (EV_CHARGING_DB_OBJ_STRUCT *)hDBInterface;

        // End any currently-open transaction
        vEndTransaction(psObj);

        if (psObj->pacCurRefDatabaseFilePath != NULL)
        {
            SMSO_vDestroy((SMS_OBJECT)psObj->pacCurRefDatabaseFilePath);
            psObj->pacCurRefDatabaseFilePath = NULL;
        }

        if (psObj->pacNextRefDatabaseFilePath != NULL)
        {
            SMSO_vDestroy((SMS_OBJECT)psObj->pacNextRefDatabaseFilePath);
            psObj->pacNextRefDatabaseFilePath = NULL;
        }

        if (psObj->pacRefDatabaseDirPath != NULL)
        {
            SMSO_vDestroy((SMS_OBJECT)psObj->pacRefDatabaseDirPath);
            psObj->pacRefDatabaseDirPath = NULL;
        }

        // Disconnect from the ref database
        if (psObj->hSQLRefConnection != SQL_INTERFACE_INVALID_OBJECT)
        {
            SQL_INTERFACE.vDisconnect( psObj->hSQLRefConnection );
            psObj->hSQLRefConnection = SQL_INTERFACE_INVALID_OBJECT;
        }

        // Disconnect from the persistent database
        if (psObj->hSQLPersistConnection != SQL_INTERFACE_INVALID_OBJECT)
        {
            SQL_INTERFACE.vDisconnect( psObj->hSQLPersistConnection );
            psObj->hSQLPersistConnection = SQL_INTERFACE_INVALID_OBJECT;
        }

        SMSO_vDestroy((SMS_OBJECT)psObj);
    }

    return;
}

/*****************************************************************************
*
*   bCanProcessUpdate
*
*   This function is called by the manager to determine if we're ready
*   to begin update messages.
*
*****************************************************************************/
static BOOLEAN bCanProcessUpdate (
    FUEL_DB_INTERFACE_OBJECT hDBInterface,
    UN8 un8TextVersion
        )
{
    // We'll take whatever we get...the version field does not
    // prevent us from processing an update
    return TRUE;
}

/*****************************************************************************
*
*   eMatchFuelType
*
*   This function attempts to match a fuel type with a value from
*   the well-known fuel type enumeration
*
*****************************************************************************/
static FUEL_TYPE_ENUM eMatchFuelType (
    UN8 un8TextId
        )
{
    FUEL_TYPE_ENUM eMatchedType = FUEL_TYPE_UNKNOWN;

    switch (un8TextId)
    {
        case LEVEL1_120V_ID:
        {
            eMatchedType = FUEL_TYPE_ELECTRIC_L1_120V;
        }
        break;

        case LEVEL2_240V_ID:
        {
            eMatchedType = FUEL_TYPE_ELECTRIC_L2_240V;
        }
        break;

        case DC_ID:
        {
            eMatchedType = FUEL_TYPE_ELECTRIC_DC;
        }
        break;

        default:
        {
            // Nothing to do
        }
    }

    return eMatchedType;
}

/*****************************************************************************
*
*   tLoadStations
*
*   This function is called by the manager to extract all stations found
*   in a provided area (defined by (lat/lon+radius, region id) or
*   (region id, station id)
*
*****************************************************************************/
static size_t tLoadStations (
    FUEL_DB_INTERFACE_OBJECT hDBInterface,
    LOCATION_OBJECT hLocation,
    FUEL_REGION tRegionID,
    FUEL_STATION_ID tStationID
        )
{
    BOOLEAN bValid;

    bValid = SMSO_bValid((SMS_OBJECT)hDBInterface);
    if (bValid == TRUE)
    {
        EV_CHARGING_DB_OBJ_STRUCT *psObj =
            (EV_CHARGING_DB_OBJ_STRUCT *)hDBInterface;
        EV_CHARGING_DB_LOAD_STATION_STRUCT sLoad;
        BOOLEAN bOk;

        // Build a query for a specific region & station
        if (tStationID != FUEL_INVALID_STATION_ID)
        {
            snprintf( &psObj->acBuffer[0], sizeof(psObj->acBuffer),
                FUEL_SELECT_STATION_DESC,
                tRegionID, tStationID );
        }
        // Build a query for a specific region
        else
        {
            BOOLEAN bBuildQuery;

            // Build the query (it's a process!)
            bBuildQuery = bBuildStationLoadQuery(
                psObj, hLocation, tRegionID);
            if (bBuildQuery == FALSE)
            {
                return FALSE;
            }
        }

        // Initialize the result struct
        sLoad.psObj = psObj;
        sLoad.tRegion = tRegionID;
        sLoad.tStation = tStationID;
        sLoad.bSuccess = TRUE;
        sLoad.tNumObjectsCreated = 0;

        // Perform the SQL query and process the result
        bOk = SQL_INTERFACE.bQuery(
                psObj->hSQLRefConnection, &psObj->acBuffer[0],
                (SQL_QUERY_RESULT_HANDLER)bProcessSelectStations,
                &sLoad ) ;

        if ((bOk == TRUE) && (sLoad.bSuccess == TRUE))
        {
            return sLoad.tNumObjectsCreated;
        }
    }

    return 0;
}

/*****************************************************************************
*
*   bLoadPrices
*
*   This function is called by the manager to extract all price data or
*   price data for a given region and station
*
*****************************************************************************/
static BOOLEAN bLoadPrices (
    FUEL_DB_INTERFACE_OBJECT hDBInterface,
    FUEL_REGION tRegionID,
    FUEL_STATION_ID tStationID
        )
{
    BOOLEAN bValid, bOk = FALSE;

    bValid = SMSO_bValid((SMS_OBJECT)hDBInterface);
    if (bValid == TRUE)
    {
        EV_DB_PRICE_QUERY_RESULT_STRUCT sPriceResult;
        EV_CHARGING_DB_OBJ_STRUCT *psObj =
            (EV_CHARGING_DB_OBJ_STRUCT *)hDBInterface;
        OSAL_RETURN_CODE_ENUM eReturnCode;

        // Get the current time (UTC)
        eReturnCode = OSAL.eTimeGet(&sPriceResult.un32UTCsec);
        if (eReturnCode != OSAL_SUCCESS)
        {
            // We couldn't get the time or it was not set properly
            printf(EV_CHARGING_NAME": eTimeGet Result: %s\n",
                   OSAL.pacGetReturnCodeName(eReturnCode));
            return FALSE;
        }

        if (sPriceResult.un32UTCsec <= FUEL_ZERO_DATE)
        {
            // Time was not set properly
            puts(EV_CHARGING_NAME": bad time from OSAL");
            return FALSE;
        }

        // Compute the age limit
        sPriceResult.un32PriceAgeLimit =
            sPriceResult.un32UTCsec - EV_CHARGING_AGE_MAX_IN_SECS;

        if (tStationID != FUEL_INVALID_STATION_ID)
        {
            // Build our price query for a single station
            snprintf( &psObj->acBuffer[0], sizeof(psObj->acBuffer),
                FUEL_SELECT_PRICES_STATIONID, sPriceResult.un32PriceAgeLimit,
                tRegionID, tStationID);
        }
        else
        {
            // Read all prices
            // Build our price query
            snprintf( &psObj->acBuffer[0], sizeof(psObj->acBuffer),
                FUEL_SELECT_PRICES, sPriceResult.un32PriceAgeLimit);
        }

        // Initialize the result struct
        sPriceResult.bSuccess = TRUE;
        sPriceResult.psObj = psObj;

        // Perform the SQL query and process the result
        bOk = SQL_INTERFACE.bQuery(
                psObj->hSQLPersistConnection, &psObj->acBuffer[0],
                (SQL_QUERY_RESULT_HANDLER)bProcessSelectPrices,
                &sPriceResult ) ;

        if (bOk == TRUE)
        {
            // Just use the success flag
            bOk = sPriceResult.bSuccess;
        }
    }

    return bOk;
}

/*****************************************************************************
*
*   bTextQuery
*
*   This function is used to determine if a newly received text update
*   needs to be processed.
*
*****************************************************************************/
static BOOLEAN bTextQuery (
    FUEL_DB_INTERFACE_OBJECT hDBInterface,
    UN8 un8NewTextVersion
        )
{
    // We don't support this
    return FALSE;
}

/*****************************************************************************
*
*   bUpdateTextTable
*
*   This function is called by the manager to begin or finsh updating the
*   text table.
*
*****************************************************************************/
static BOOLEAN bUpdateTextTable (
    FUEL_DB_INTERFACE_OBJECT hDBInterface,
    BOOLEAN bBegin,
    UN8 un8NewTextVersion
        )
{
    // We don't support this
    return FALSE;
}

/*****************************************************************************
*
*   bUpdateTextEntry
*
*   This function is called by the manager to update a text table entry
*
*   Note: hDBInterface is ignored in this implementation of this
*   interface function
*
*****************************************************************************/
static BOOLEAN bUpdateTextEntry (
    FUEL_DB_INTERFACE_OBJECT hDBInterface,
    SQL_INTERFACE_OBJECT hSQLConnection,
    FUEL_TEXT_ROW_STRUCT *psTextRow
        )
{
    BOOLEAN bUpdated;

    bUpdated = SQL_INTERFACE.bExecutePreparedCommand(
        hSQLConnection,
        EV_CHARGING_INSERT_TEXT_ROW,
        EV_CHARGING_DB_TEXT_MAX_FIELDS,
        (PREPARED_QUERY_COLUMN_CALLBACK)bPrepareTextColumn,
        psTextRow);

    return bUpdated;
}

/*****************************************************************************
*
*   bUpdateLogoEntry
*
*****************************************************************************/
static BOOLEAN bUpdateLogoEntry (
    FUEL_DB_INTERFACE_OBJECT hDBInterface,
    BOOLEAN bDelete,
    FUEL_LOGO_ROW_STRUCT *psLogoRow
        )
{
    BOOLEAN bValid, bSuccess = FALSE;

    bValid = SMSO_bValid((SMS_OBJECT)hDBInterface);
    if (bValid == TRUE)
    {
        EV_CHARGING_DB_OBJ_STRUCT *psObj =
            (EV_CHARGING_DB_OBJ_STRUCT *)hDBInterface;

        if (bDelete == TRUE)
        {
            // Build deletion string
            snprintf(&psObj->acBuffer[0], sizeof(psObj->acBuffer),
                EV_CHARGING_REMOVE_LOGO,
                psLogoRow->un16LogoId,
                psLogoRow->eLogoType);

            // Do it now
            bSuccess = SQL_INTERFACE.bExecuteCommand(
                psObj->hSQLPersistConnection,
                &psObj->acBuffer[0]);
        }
        else
        {
            bSuccess = SQL_INTERFACE.bExecutePreparedCommand(
                psObj->hSQLPersistConnection,
                EV_CHARGING_INSERT_UPDATE_LOGO,
                EV_CHARGING_DB_LOGO_MAX_FIELDS,
                (PREPARED_QUERY_COLUMN_CALLBACK)bPrepareLogoColumn,
                psLogoRow);
        }
    }

    return bSuccess;
}

/*****************************************************************************
*
*   vUpdatePriceTable
*
*   This function is called by the manager to begin or finsh updating the
*   price table.
*
*****************************************************************************/
static void vUpdatePriceTable (
    FUEL_DB_INTERFACE_OBJECT hDBInterface,
    BOOLEAN bBegin
        )
{
    BOOLEAN bValid;

    bValid = SMSO_bValid((SMS_OBJECT)hDBInterface);
    if (bValid == TRUE)
    {
        EV_CHARGING_DB_OBJ_STRUCT *psObj =
            (EV_CHARGING_DB_OBJ_STRUCT *)hDBInterface;

        if (bBegin == TRUE)
        {
            vStartTransaction(psObj);
        }
        else
        {
            vEndTransaction(psObj);
        }
    }

    return;
}

/*****************************************************************************
*
*   bUpdatePriceEntry
*
*   This function is called by the manager to update a price table entry
*
*****************************************************************************/
static BOOLEAN bUpdatePriceEntry (
    FUEL_DB_INTERFACE_OBJECT hDBInterface,
    FUEL_STATION_OBJECT hStation
        )
{
    UN8 un8NumPrices;
    BOOLEAN bValid, bSuccess = TRUE;
    EV_CHARGING_DB_OBJ_STRUCT *psObj =
        (EV_CHARGING_DB_OBJ_STRUCT *)hDBInterface;
    FUEL_DB_UPDATE_PRICE_TABLE_STRUCT sUpdate;
    FUEL_REFUELING_TYPE_STRUCT asTypes[FUEL_REFUELING_TYPES_ARRAY_SIZE];

    bValid = SMSO_bValid((SMS_OBJECT)hDBInterface);
    if (bValid == FALSE)
    {
        return FALSE;
    }

    do
    {
        // Find out how many entries are here
        un8NumPrices = FUEL_STATION.un8NumAvailableFuelTypes(hStation);

        if (un8NumPrices > FUEL_REFUELING_TYPES_ARRAY_SIZE)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                EV_CHARGING_NAME": bUpdatePriceEntry() too many prices reported from station");
            break;
        }

        // Clear the update structure
        OSAL.bMemSet(&sUpdate, 0, sizeof(FUEL_DB_UPDATE_PRICE_TABLE_STRUCT));

        // Clear the types array
        OSAL.bMemSet(&asTypes[0], 0,
            (sizeof(FUEL_REFUELING_TYPE_STRUCT) * FUEL_REFUELING_TYPES_ARRAY_SIZE));

        // Ensure the price list is empty
        sUpdate.sPriceRow.pasPrices = (FUEL_PRICE_ENTRY_STRUCT *)NULL;

        // Make the types array available
        sUpdate.sStationRow.psTypes = &asTypes[0];

        // We need to serialize this entry
        bSuccess = FUEL_STATION_bSerializeData(hStation, &sUpdate);
        if (bSuccess == FALSE)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                EV_CHARGING_NAME": bUpdatePriceDBEntry() FUEL_STATION_bSerializeData failed");
            break;
        }

        // Update the database
        bSuccess = SQL_INTERFACE.bExecutePreparedCommand(
            psObj->hSQLPersistConnection, EV_CHARGING_INSERT_UPDATE_PRICE_ROW,
            EV_CHARGING_DB_PRICE_MAX_FIELDS,
            (PREPARED_QUERY_COLUMN_CALLBACK)bPreparePriceColumn, &sUpdate);

    } while (FALSE);

    return bSuccess;
}

/*****************************************************************************
*
*   un16DBContentVer
*
*   This function is called by the manager to extract the current DB
*   content version number
*
*****************************************************************************/
static UN16 un16DBContentVer (
    FUEL_DB_INTERFACE_OBJECT hDBInterface
        )
{
    BOOLEAN bOwner;
    UN16 un16Ver = 0;

    bOwner = SMSO_bOwner((SMS_OBJECT)hDBInterface);
    if (bOwner == TRUE)
    {
        EV_CHARGING_DB_OBJ_STRUCT *psObj =
            (EV_CHARGING_DB_OBJ_STRUCT *)hDBInterface;

        un16Ver = (UN16)psObj->sVerCtrl.n16DBContentVersion;
    }

    return un16Ver;
}

/*****************************************************************************
*
*   bRefDBBank
*
*****************************************************************************/
static BOOLEAN bRefDBBank (
    FUEL_DB_INTERFACE_OBJECT hDBInterface,
    STRING_OBJECT *phInUseDB,
    STRING_OBJECT *phNextDB
        )
{
    BOOLEAN bLocked, bSuccess = FALSE;

    if ((phInUseDB == NULL) || (phNextDB == NULL))
    {
        return FALSE;
    }

    // Initialize inputs
    *phInUseDB = *phNextDB = STRING_INVALID_OBJECT;

    bLocked = SMSO_bLock(
        (SMS_OBJECT)hDBInterface, OSAL_OBJ_TIMEOUT_INFINITE);
    if (bLocked == TRUE)
    {
        EV_CHARGING_DB_OBJ_STRUCT *psObj =
            (EV_CHARGING_DB_OBJ_STRUCT *)hDBInterface;

        do
        {
            // Provide the caller with the path info
            *phInUseDB = STRING.hCreate(
                psObj->pacCurRefDatabaseFilePath,
                strlen(psObj->pacCurRefDatabaseFilePath));

            if (*phInUseDB == STRING_INVALID_OBJECT)
            {
                // Error! Stop here
                break;
            }

            *phNextDB = STRING.hCreate(
                psObj->pacNextRefDatabaseFilePath,
                strlen(psObj->pacNextRefDatabaseFilePath));

            if (*phNextDB == STRING_INVALID_OBJECT)
            {
                // Clear the other string now
                STRING_vDestroy(*phInUseDB);
                *phInUseDB = STRING_INVALID_OBJECT;

                break;
            }

            // All went well
            bSuccess = TRUE;
        } while (FALSE);

        SMSO_vUnlock((SMS_OBJECT)hDBInterface);
    }

    return bSuccess;
}

/*****************************************************************************
*
*   bDBUpdateBegin
*
*****************************************************************************/
static BOOLEAN bDBUpdateBegin (
   SQL_INTERFACE_OBJECT hSQLConnection,
   char *pcSQLCommandBuffer,
   size_t tBufferSize
        )
{
    BOOLEAN bOk;

    // Generate the SQL command to update the version row
    // in the database to the db_util "under construction" value
    snprintf( &pcSQLCommandBuffer[0], tBufferSize,
        FUEL_UPDATE_DB_VERSION_ROW,
        DB_UTIL_DB_UNDER_CONSTRUCTION_VER,
        EV_CHARGING_DB_VERSION_TYPE_CONTENTS
            );

    // Update the row
    bOk = SQL_INTERFACE.bExecuteCommand(
        hSQLConnection, &pcSQLCommandBuffer[0] );

    return bOk;
}


/*****************************************************************************
*
*   bDBUpdateEnd
*
*   This function is part of the manager interface API and is utilized by
*   the interface to indicate a database update has been completed and
*   a report the new database version number.
*
*****************************************************************************/
static BOOLEAN bDBUpdateEnd (
    SQL_INTERFACE_OBJECT hConnection,
    char *pcSQLCommandBuffer,
    size_t tBufferSize,
    UN16 un16NewDBVersion
        )
{
    BOOLEAN  bResult = FALSE;

    do
    {
        N32 n32Result;

        // While the DB_UTIL function will print its own error, we also print
        // one here to make it easier to trace the DB failure back to a
        // particular service.
        bResult = DB_UTIL_bUpdateTimestamp( hConnection, pcSQLCommandBuffer, tBufferSize );
        if ( bResult == FALSE )
        {
            SMSAPI_DEBUG_vPrintErrorFull( gpacThisFile, __LINE__,
                    EV_CHARGING_NAME": Unable to set timestamp.");
        }

        n32Result = snprintf( pcSQLCommandBuffer, tBufferSize,
                              FUEL_UPDATE_DB_VERSION_ROW,
                              un16NewDBVersion,
                              EV_CHARGING_DB_VERSION_TYPE_CONTENTS );
        if ( n32Result <= 0 )
        {
            SMSAPI_DEBUG_vPrintErrorFull( gpacThisFile, __LINE__,
                    EV_CHARGING_NAME": snprintf failed.");
            break;
        }

        // Updating the baseline version comes last, as this effectively stamps
        // the DB as "ready."
        bResult = SQL_INTERFACE.bExecuteCommand( hConnection, pcSQLCommandBuffer );
        if ( bResult == FALSE )
        {
            SMSAPI_DEBUG_vPrintErrorFull( gpacThisFile, __LINE__,
                    EV_CHARGING_NAME": Unable to set baseline version.");
            break;
        }

    } while ( FALSE );

    return bResult;
}

/*****************************************************************************
*
*   bRegionUpdateBegin
*
*   This function is part of the manager interface API and is utilized by
*   the interface to indicate a region update has been received.
*
*   This API must only be called when already in the
*   context of the fuel manager.
*
*   Inputs:
*       hFuelService - The fuel service object handle provided to
*           the interface.
*       *psNewRegionRow - A pointer to the newly received region information.
*       un8MsgId - The ID of the message bearing this update
*
*   Output: BOOLEAN TRUE on success, FALSE on error or unwanted data
*
*****************************************************************************/
static BOOLEAN bRegionUpdateBegin (
    SQL_INTERFACE_OBJECT hSQLConnection,
    char *pcSQLCommandBuffer,
    size_t tBufferSize,
    FUEL_REGION tRegionID
        )
{
    BOOLEAN bSuccess, bRegionExists = FALSE;

    // Generate the command to query for this region
    snprintf( &pcSQLCommandBuffer[0], tBufferSize,
        FUEL_SELECT_REGION, tRegionID );

    // Start a new transaction
    bSuccess = SQL_INTERFACE.bStartTransaction(hSQLConnection);
    if (bSuccess == FALSE)
    {
        return FALSE;
    }

    // Does this region exist?
    bSuccess = SQL_INTERFACE.bQuery(
        hSQLConnection, &pcSQLCommandBuffer[0],
        (SQL_QUERY_RESULT_HANDLER)bProcessSelectRegionExists,
        (void *)&bRegionExists);
    if (bSuccess == TRUE)
    {
        if (bRegionExists == FALSE)
        {
            // We need to add a region to our DB now

            // Insert the region entry
            bSuccess = SQL_INTERFACE.bExecutePreparedCommand(
                hSQLConnection, FUEL_INSERT_REGION_ROW,
                FUEL_DB_REGION_MAX_FIELDS,
                (PREPARED_QUERY_COLUMN_CALLBACK)bPrepareRegionColumn,
                &tRegionID);
            if (bSuccess == FALSE)
            {
                SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    EV_CHARGING_NAME": Unable to insert the region entry");
            }

            // Generate the command to create the table
            snprintf( &pcSQLCommandBuffer[0], tBufferSize,
                EV_CHARGING_CREATE_STATION_TABLE, tRegionID );

            // Create the new table
            bSuccess = SQL_INTERFACE.bExecuteCommand(
                hSQLConnection, &pcSQLCommandBuffer[0]);
            if (bSuccess == FALSE)
            {
                SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    EV_CHARGING_NAME": Unable to create new stations table");
            }
        }
    }

    return bSuccess;
}

/*****************************************************************************
*
*   bRegionUpdateEnd
*
*   This function is part of the manager interface API and is utilized by
*   the interface to indicate a region update has been completed.
*
*   This API must only be called when already in the
*   context of the fuel manager.
*
*   Inputs:
*       hFuelService - The fuel service object handle provided to
*           the interface.
*       *psRegion - A pointer to the newly received region information.
*       un8MsgId - The ID of the message bearing this update
*
*   Output: BOOLEAN TRUE on success, FALSE on error or unwanted data
*
*****************************************************************************/
static BOOLEAN bRegionUpdateEnd (
    SQL_INTERFACE_OBJECT hSQLConnection
        )
{
    return SQL_INTERFACE.bEndTransaction(hSQLConnection);
}

/*****************************************************************************
*
*   bStationUpdate
*
*   This function is part of the manager interface API and is utilized by
*   the interface to indicate a station update has been received.
*
*****************************************************************************/
static BOOLEAN bStationUpdate (
   SQL_INTERFACE_OBJECT hSQLConnection,
   char **ppcSQLCommandBuffer,
   size_t *ptBufferSize,
   FUEL_STATION_ROW_STRUCT *psStationRow,
   FUEL_STATION_UPDATE_TYPE_ENUM eUpdateType,
   BOOLEAN bAmenitiesUpdated
       )
{
    BOOLEAN bOk;

    if (eUpdateType >= FUEL_STATION_UPDATE_MAX_TYPES)
    {
        SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
            EV_CHARGING_NAME": invalid update type!");

        return FALSE;
    }

    // Update the database
    bOk = bUpdateStationDBEntry(
        hSQLConnection, *ppcSQLCommandBuffer, *ptBufferSize,
        psStationRow, eUpdateType, bAmenitiesUpdated);

    // Free all strings given to us by the interface

    // ...Except don't destroy the brand text
    // since it is persistent
    psStationRow->hBrand = STRING_INVALID_OBJECT;

    STRING_vDestroy(psStationRow->hAddr);
    psStationRow->hAddr = STRING_INVALID_OBJECT;

    STRING_vDestroy(psStationRow->hCity);
    psStationRow->hCity = STRING_INVALID_OBJECT;

    STRING_vDestroy(psStationRow->hName);
    psStationRow->hName = STRING_INVALID_OBJECT;

    STRING_vDestroy(psStationRow->hZIP);
    psStationRow->hZIP = STRING_INVALID_OBJECT;

    STRING_vDestroy(psStationRow->hDesc);
    psStationRow->hDesc = STRING_INVALID_OBJECT;

    return bOk;
}

/*****************************************************************************
                             FRIEND FUNCTIONS
*****************************************************************************/

/*****************************************************************************
*
*   EV_CHARGING_hStart
*
*   This function will create all the basics needed for this service to
*   operate.  However, all initial processing to actually get this service
*   running is done at a later time.
*
*****************************************************************************/
FUEL_SERVICE_OBJECT EV_CHARGING_hStart (
    const char *pacSRHDriverName,
    FUEL_PRICE_SORT_METHOD_ENUM eFuelPriceSortMethod,
    DATASERVICE_EVENT_MASK tEventRequestMask,
    DATASERVICE_EVENT_CALLBACK vEventCallback,
    void *pvEventCallbackArg,
    DATASERVICE_OPTIONS_STRUCT const *psOptions
        )
{
    FUEL_SERVICE_OBJECT hEVChargingService =
        FUEL_SERVICE_INVALID_OBJECT;
    FUEL_SERVICE_OBJECT hFuelService;
    DATASERVICE_CREATE_STRUCT sCreate;

    // Populate our data service creation structure with attributes
    // specific to the fuel price data service (also populate the common items
    // that we can populate given the input arguments
    DATASERVICE_IMPL_vInitCreateStruct(&sCreate);

    // Input arguemnts
    sCreate.pacSRHDriverName = pacSRHDriverName;

    // Fuel prices specific attributes
    sCreate.pacServiceObjectName = EV_CHARGING_NAME;
    sCreate.tDataID = GsEVChargingOTAIntf.tDataID;

    // Start the fuel service using the common
    // underlying hstartext function
    hFuelService = FUEL_MGR_hStart(
        &sCreate, DATASERVICE_TYPE_EV_CHARGING, TRUE,
        &GsEVChargingOTAIntf, &GsEVChargingDBIntf,
        eFuelPriceSortMethod, tEventRequestMask,
        vEventCallback, pvEventCallbackArg,
        psOptions );
    if (hFuelService != FUEL_SERVICE_INVALID_OBJECT)
    {
        // Provide the caller with the fuel service object
        hEVChargingService = (FUEL_SERVICE_OBJECT)hFuelService;
    }

    return hEVChargingService;
}

/*****************************************************************************
*
*   EV_CHARGING_eGetReferenceDataVersion
*
*****************************************************************************/
DATASERVICE_ERROR_CODE_ENUM EV_CHARGING_eGetReferenceDataVersion (
    const char *pcContainingDirectoryPath,
    DATASERVICE_REF_DATA_VER *ptCurrentRefDataVer,
    DATASERVICE_REF_DATA_VER *ptNextRefDataVer
        )
{
    DATASERVICE_ERROR_CODE_ENUM eErrorCode = DATASERVICE_ERROR_CODE_GENERAL;
    BOOLEAN bOk;
    char *pacDatabaseFilePathA = (char *)NULL,
         *pacDatabaseFilePathB = (char *)NULL;

    if (ptCurrentRefDataVer == NULL)
    {
        return DATASERVICE_ERROR_CODE_GENERAL;
    }

    do
    {
        bOk = DB_UTIL_bCreateFilePath(
            pcContainingDirectoryPath,
            EV_CHARGING_DATABASE_FOLDER,
            EV_CHARGING_REF_DATABASE_FILENAMEA,
            &pacDatabaseFilePathA);

        if (bOk != TRUE)
        {
            break;
        }

        bOk = DB_UTIL_bCreateFilePath(
            pcContainingDirectoryPath,
            EV_CHARGING_DATABASE_FOLDER,
            EV_CHARGING_REF_DATABASE_FILENAMEB,
            &pacDatabaseFilePathB);

        if (bOk != TRUE)
        {
            break;
        }

        // Connect to the fuel ref database
        eErrorCode = DB_UTIL_eCheckReferenceBanks(
            &pacDatabaseFilePathA[0],
            &pacDatabaseFilePathB[0],
            n32ExtractDataVersion, NULL,
            GsEVChargingOTAIntf.tMaxVersionBitlen,
            ptCurrentRefDataVer, ptNextRefDataVer );

    } while (FALSE);

    // Remove database path resources
    if (pacDatabaseFilePathA != NULL)
    {
        SMSO_vDestroy(
            (SMS_OBJECT)pacDatabaseFilePathA);
    }

    if (pacDatabaseFilePathB != NULL)
    {
        SMSO_vDestroy(
            (SMS_OBJECT)pacDatabaseFilePathB);
    }

    return eErrorCode;
}

/*****************************************************************************
                             PRIVATE FUNCTIONS
*****************************************************************************/

/*****************************************************************************
*
*   bPriceAgeFilter
*
*   This function is called when price information is extracted from
*   the database in order to determine if the age of the price data meets
*   the application's current age-out criteria.
*
*****************************************************************************/
static BOOLEAN bPriceAgeFilter (
    UN32 un32PriceAgeInEpochSeconds,
    UN32 un32CurrentTime
        )
{
    UN32 un32AgeInEpochSeconds = un32CurrentTime;
    BOOLEAN bMeetsCriteria = FALSE;

    // Only compute the age if the time was set
    // correctly
    if (un32AgeInEpochSeconds > un32PriceAgeInEpochSeconds)
    {
        un32AgeInEpochSeconds -= un32PriceAgeInEpochSeconds;

        if (un32AgeInEpochSeconds < EV_CHARGING_AGE_MAX_IN_SECS)
        {
            bMeetsCriteria = TRUE;
        }
    }

    return bMeetsCriteria;
}

/*****************************************************************************
*
*   bInitializeWellKnownFuelTypes
*
*****************************************************************************/
static BOOLEAN bInitializeWellKnownFuelTypes (
    EV_CHARGING_DB_OBJ_STRUCT *psObj
       )
{
    BOOLEAN bSuccess = FALSE;
    STRING_OBJECT hLongName = STRING_INVALID_OBJECT,
                  hShortName =  STRING_INVALID_OBJECT;

    do
    {
        BOOLEAN bTextAdded;
        FUEL_TEXT_ROW_STRUCT sText;

        // No brand information is being provided
        sText.bBrandText = FALSE;

        // Create the strings for L1
        hLongName =
            STRING_hCreate( SMS_INVALID_OBJECT,
                LEVEL1_120V_LNAME,
                strlen(LEVEL1_120V_LNAME),
                strlen(LEVEL1_120V_LNAME));

        hShortName =
            STRING_hCreate( SMS_INVALID_OBJECT,
                LEVEL1_120V_SNAME,
                strlen(LEVEL1_120V_SNAME),
                strlen(LEVEL1_120V_SNAME));

        if ((hLongName == STRING_INVALID_OBJECT) ||
            (hShortName == STRING_INVALID_OBJECT))
        {
            break;
        }

        // Populate the structure
        sText.hText = hShortName;
        sText.hLongText = hLongName;
        sText.un8TextId = LEVEL1_120V_ID;

        // Provide to the manager
        bTextAdded = GsFuelMgrIntf.bReportTextFromDB(
            psObj->hFuelService, &sText);
        if (bTextAdded == FALSE)
        {
            break;
        }

        // Create the strings for L2
        hLongName =
            STRING_hCreate( SMS_INVALID_OBJECT,
                LEVEL2_240V_LNAME,
                strlen(LEVEL2_240V_LNAME),
                strlen(LEVEL2_240V_LNAME));

        hShortName =
            STRING_hCreate( SMS_INVALID_OBJECT,
                LEVEL2_240V_SNAME,
                strlen(LEVEL2_240V_SNAME),
                strlen(LEVEL2_240V_SNAME));

        if ((hLongName == STRING_INVALID_OBJECT) ||
            (hShortName == STRING_INVALID_OBJECT))
        {
            break;
        }

        // Populate the structure
        sText.hText = hShortName;
        sText.hLongText = hLongName;
        sText.un8TextId = LEVEL2_240V_ID;

        // Provide to the manager
        bTextAdded = GsFuelMgrIntf.bReportTextFromDB(
            psObj->hFuelService, &sText);
        if (bTextAdded == FALSE)
        {
            break;
        }

        // Create the strings for DC
        hLongName =
            STRING_hCreate( SMS_INVALID_OBJECT,
                DC_LNAME,
                strlen(DC_LNAME),
                strlen(DC_LNAME));

        hShortName =
            STRING_hCreate( SMS_INVALID_OBJECT,
                DC_SNAME,
                strlen(DC_SNAME),
                strlen(DC_SNAME));

        if ((hLongName == STRING_INVALID_OBJECT) ||
            (hShortName == STRING_INVALID_OBJECT))
        {
            break;
        }

        // Populate the structure
        sText.hText = hShortName;
        sText.hLongText = hLongName;
        sText.un8TextId = DC_ID;

        // Provide to the manager
        bTextAdded = GsFuelMgrIntf.bReportTextFromDB(
            psObj->hFuelService, &sText);
        if (bTextAdded == FALSE)
        {
            break;
        }

        // Clear these handles since that worked
        hShortName = STRING_INVALID_OBJECT;
        hLongName = STRING_INVALID_OBJECT;

        // All done!
        bSuccess = TRUE;
    } while (FALSE);

    // Clean up anything we left behind
    if (hShortName != STRING_INVALID_OBJECT)
    {
        STRING_vDestroy(hShortName);
    }

    if (hLongName != STRING_INVALID_OBJECT)
    {
        STRING_vDestroy(hLongName);
    }

    return bSuccess;
}

/*****************************************************************************
*
*   bFlushPrices
*
*   Flush all prices out of the database.
*
*****************************************************************************/
static BOOLEAN bFlushPrices (
    EV_CHARGING_DB_OBJ_STRUCT *psObj
        )
{
    BOOLEAN bSuccess;

    printf(EV_CHARGING_NAME": "FUEL_FLUSH_PRICE_ROWS"\n");

    // Perform the flush
    bSuccess = SQL_INTERFACE.bExecuteCommand(
        psObj->hSQLPersistConnection, FUEL_FLUSH_PRICE_ROWS );

    return bSuccess;
}

/*****************************************************************************
*
*   bAgeoutPrices
*
*   This function removes all old price entries from the database
*
*****************************************************************************/
static BOOLEAN bAgeoutPrices (
    EV_CHARGING_DB_OBJ_STRUCT *psObj
        )
{
    OSAL_RETURN_CODE_ENUM eReturnCode;
    BOOLEAN bSuccess = TRUE;
    UN32 un32AgeLimitUTCsec = 0;

    // Get the current time
    eReturnCode = OSAL.eTimeGet(&un32AgeLimitUTCsec);
    if (eReturnCode != OSAL_SUCCESS)
    {
        SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
            EV_CHARGING_NAME
            ": bAgeoutPriceEntries() OSAL.eTimeGet () failure (%s)",
            OSAL.pacGetReturnCodeName(eReturnCode));

        return FALSE;
    }

    // Ensure that the clock is set correctly(-ish)
    if (un32AgeLimitUTCsec <= FUEL_ZERO_DATE)
    {
        SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
            EV_CHARGING_NAME
            ": bHandleAgeout() bad time reported");

        return FALSE;
    }

    // Data which is at least EV_CHARGING_AGE_MAX_IN_SECS
    // old must be aged out now.  Compute that
    // in epoch seconds
    un32AgeLimitUTCsec -= EV_CHARGING_AGE_MAX_IN_SECS;

    // Generate the command
    snprintf( &psObj->acBuffer[0], sizeof(psObj->acBuffer),
            FUEL_AGEOUT_PRICE_ROWS, un32AgeLimitUTCsec);

    bSuccess = SQL_INTERFACE.bExecuteCommand(
        psObj->hSQLPersistConnection, &psObj->acBuffer[0]);

    return bSuccess;
}

/*****************************************************************************
*
*   bUpdateStationDBEntry
*
*****************************************************************************/
static BOOLEAN bUpdateStationDBEntry (
    SQL_INTERFACE_OBJECT hSQLConnection,
    char *pacBuffer,
    size_t tBufferSize,
    FUEL_STATION_ROW_STRUCT *psStation,
    FUEL_STATION_UPDATE_TYPE_ENUM eUpdateType,
    BOOLEAN bAmenitiesUpdated
        )
{
    BOOLEAN bSuccess = TRUE, bUsePreparedStatement = TRUE;
    EV_DB_STATION_UPDATE_STRUCT sUpdate;

    sUpdate.psStationRow = psStation;
    sUpdate.un8NumFields = 0;

    // If we're processing a modify we need to first see if the
    // station listed exists.  If not, we'll treat this as an add
    if (FUEL_STATION_UPDATE_TYPE_MODIFY_GIVEN_ATTRIBS == eUpdateType)
    {
        BOOLEAN bStationExists = FALSE;

        // Generate the command to query for this station
        snprintf( pacBuffer, tBufferSize,
            FUEL_SELECT_STATION_EXISTS,
            psStation->tRegion, psStation->tStationId );

        // Does this station exist?
        bSuccess = SQL_INTERFACE.bQuery(
            hSQLConnection, pacBuffer,
            (SQL_QUERY_RESULT_HANDLER)bProcessSelectStationExists,
            (void *)&bStationExists);

        if (FALSE == bSuccess)
        {
            // Error!
            return FALSE;
        }

        if (FALSE == bStationExists)
        {
            // This station doesn't exist...so treat this
            // modify command as if it were an add
            eUpdateType = FUEL_STATION_UPDATE_TYPE_NEW;
        }
    }

    switch (eUpdateType)
    {
        case FUEL_STATION_UPDATE_TYPE_NEW:
        {
            bSuccess = bPrepareUpdateCommand(
                pacBuffer, tBufferSize, &sUpdate,
                TRUE, bAmenitiesUpdated);
        }
        break;

        case FUEL_STATION_UPDATE_TYPE_MODIFY_GIVEN_ATTRIBS:
        {
            bSuccess = bPrepareUpdateCommand(
                pacBuffer, tBufferSize, &sUpdate,
                FALSE, bAmenitiesUpdated);
        }
        break;

        case FUEL_STATION_UPDATE_TYPE_DELETE:
        {
            // Just generate a delete command
            snprintf(
                &pacBuffer[0], tBufferSize,
                FUEL_DEL_STATION_DESC,
                psStation->tRegion,
                psStation->tStationId);

            // Just a simple command...
            bUsePreparedStatement = FALSE;
        }
        break;

        default:
        {
            bSuccess = FALSE;
        }
    }

    if (bSuccess == TRUE)
    {
        if (bUsePreparedStatement == TRUE)
        {
            bSuccess = SQL_INTERFACE.bExecutePreparedCommand(
                hSQLConnection, &pacBuffer[0],
                sUpdate.un8NumFields,
                (PREPARED_QUERY_COLUMN_CALLBACK)bPrepareStationColumn,
                &sUpdate);
        }
        else
        {
            // Update the database
            bSuccess = SQL_INTERFACE.bExecuteCommand(
                hSQLConnection, &pacBuffer[0] );
        }
    }

    return bSuccess;
}

/*****************************************************************************
*
*   bPrepareUpdateCommand
*
*****************************************************************************/
static BOOLEAN bPrepareUpdateCommand (
    char *pacBuffer,
    size_t tBufferSize,
    EV_DB_STATION_UPDATE_STRUCT *psUpdate,
    BOOLEAN bNewStation,
    BOOLEAN bAmenitiesUpdated
        )
{
    size_t tCharsWritten;
    UN8 un8Index;

    if (bNewStation == TRUE)
    {
        // Insert the command header
        tCharsWritten = snprintf(
            &pacBuffer[0], tBufferSize,
            EV_CHARGING_INSERT_UPDATE_STATION_ROW,
            psUpdate->psStationRow->tRegion);

        // All of the fields are present
        psUpdate->un8NumFields = (UN8)EV_CHARGING_DB_STATION_MAX_FIELDS;
        for (un8Index = 0;
             un8Index < EV_CHARGING_DB_STATION_MAX_FIELDS; un8Index++)
        {
            psUpdate->aeFields[un8Index] = (EV_CHARGING_DB_STATION_FIELDS_ENUM)un8Index;
        }

    }
    else
    {
        EV_CHARGING_DB_STATION_FIELDS_ENUM eCurField;
        BOOLEAN bUseField;

        // Insert the command header
        tCharsWritten = snprintf(
            &pacBuffer[0], tBufferSize,
            FUEL_UPDATE_ROW_BEGIN,
            psUpdate->psStationRow->tRegion);

        // Build the modify command
        for (eCurField = (EV_CHARGING_DB_STATION_FIELDS_ENUM)0;
             eCurField < EV_CHARGING_DB_STATION_MAX_FIELDS;
             eCurField++)
        {
            bUseField = FALSE;

            switch (eCurField)
            {
                case EV_CHARGING_DB_STATION_ID:
                {
                    // Guaranteed to be the first entry
                    tCharsWritten += snprintf(
                        &pacBuffer[tCharsWritten], tBufferSize,
                        "StationID=?");
                    bUseField = TRUE;
                }
                break;

                case EV_CHARGING_DB_STATION_BRAND:
                {
                    if (psUpdate->psStationRow->hBrand != STRING_INVALID_OBJECT)
                    {
                        tCharsWritten += snprintf(
                            &pacBuffer[tCharsWritten], tBufferSize,
                            ",brand=?");

                        bUseField = TRUE;
                    }
                }
                break;

                case EV_CHARGING_DB_STATION_LOGO:
                {
                    if (psUpdate->psStationRow->n16LogoId >= 0)
                    {
                        tCharsWritten += snprintf(
                            &pacBuffer[tCharsWritten], tBufferSize,
                            ",logoID=?");

                        bUseField = TRUE;
                    }
                }
                break;

                case EV_CHARGING_DB_STATION_NAME:
                {
                    if (psUpdate->psStationRow->hName != STRING_INVALID_OBJECT)
                    {
                        tCharsWritten += snprintf(
                            &pacBuffer[tCharsWritten], tBufferSize,
                            ",name=?");

                        bUseField = TRUE;
                    }
                }
                break;

                case EV_CHARGING_DB_STATION_DESC:
                {
                    if (psUpdate->psStationRow->hDesc != STRING_INVALID_OBJECT)
                    {
                        tCharsWritten += snprintf(
                            &pacBuffer[tCharsWritten], tBufferSize,
                            ",description=?");

                        bUseField = TRUE;
                    }
                }
                break;

                case EV_CHARGING_DB_STATION_ADDR:
                {
                    if (psUpdate->psStationRow->hAddr != STRING_INVALID_OBJECT)
                    {
                        tCharsWritten += snprintf(
                            &pacBuffer[tCharsWritten], tBufferSize,
                            ",addr=?");

                        bUseField = TRUE;
                    }
                }
                break;

                case EV_CHARGING_DB_STATION_CITY:
                {
                    if (psUpdate->psStationRow->hCity != STRING_INVALID_OBJECT)
                    {
                        tCharsWritten += snprintf(
                            &pacBuffer[tCharsWritten], tBufferSize,
                            ",city=?");

                        bUseField = TRUE;
                    }
                }
                break;

                case EV_CHARGING_DB_STATION_STATE:
                {
                    if (psUpdate->psStationRow->tStateID != STATE_INVALID_ID)
                    {
                        tCharsWritten += snprintf(
                            &pacBuffer[tCharsWritten], tBufferSize,
                            ",state=?");

                        bUseField = TRUE;
                    }
                }
                break;

                case EV_CHARGING_DB_STATION_ZIP:
                {
                    if (psUpdate->psStationRow->hZIP != STRING_INVALID_OBJECT)
                    {
                        tCharsWritten += snprintf(
                            &pacBuffer[tCharsWritten], tBufferSize,
                            ",zip=?");

                        bUseField = TRUE;
                    }
                }
                break;

                case EV_CHARGING_DB_STATION_PHONE:
                {
                    if (psUpdate->psStationRow->n64Phone > 0)
                    {
                        tCharsWritten += snprintf(
                            &pacBuffer[tCharsWritten], tBufferSize,
                            ",phone=?");

                        bUseField = TRUE;
                    }
                }
                break;

                case EV_CHARGING_DB_STATION_CHARGE_TYPE_COUNT:
                {
                    if (psUpdate->psStationRow->un8NumPermanentTypes != 0)
                    {
                        tCharsWritten += snprintf(
                            &pacBuffer[tCharsWritten], tBufferSize,
                            ",chargeTypeCount=?");

                        bUseField = TRUE;
                    }
                }
                break;

                case EV_CHARGING_DB_STATION_CHARGE_TYPES:
                {
                    if ((psUpdate->psStationRow->un8NumPermanentTypes != 0) 
                        &&
                        (psUpdate->psStationRow->bUpdateTypes == TRUE))
                    {
                        tCharsWritten += snprintf(
                            &pacBuffer[tCharsWritten], tBufferSize,
                            ",chargeTypes=?");

                        bUseField = TRUE;
                    }
                }
                break;

                case EV_CHARGING_DB_STATION_CHARGE_NUMS:
                {
                    if ((psUpdate->psStationRow->un8NumPermanentTypes != 0)
                        &&
                        (psUpdate->psStationRow->bUpdateNumbers == TRUE))
                    {
                        tCharsWritten += snprintf(
                            &pacBuffer[tCharsWritten], tBufferSize,
                            ",chargeNums=?");

                        bUseField = TRUE;
                    }
                }
                break;

                case EV_CHARGING_DB_STATION_LAT:
                {
                    if (psUpdate->psStationRow->n32Lat != -1)
                    {
                        tCharsWritten += snprintf(
                            &pacBuffer[tCharsWritten], tBufferSize,
                            ",lat=?");

                        bUseField = TRUE;
                    }
                }
                break;

                case EV_CHARGING_DB_STATION_LON:
                {
                    if (psUpdate->psStationRow->n32Lon != -1)
                    {
                        tCharsWritten += snprintf(
                            &pacBuffer[tCharsWritten], tBufferSize,
                            ",lon=?");

                        bUseField = TRUE;
                    }
                }
                break;

                case EV_CHARGING_DB_STATION_AMENITIES:
                {
                    if (bAmenitiesUpdated == TRUE)
                    {
                        tCharsWritten += snprintf(
                            &pacBuffer[tCharsWritten], tBufferSize,
                            ",amenities=?");

                        bUseField = TRUE;
                    }
                }
                break;

                case EV_CHARGING_DB_STATION_MAX_FIELDS:
                {
                    // Nothing to do
                }
                break;

            }

            if (bUseField == TRUE)
            {
                psUpdate->aeFields[psUpdate->un8NumFields++] = eCurField;
            }
        }

        // Close the command
        snprintf(
            &pacBuffer[tCharsWritten], tBufferSize,
            " where StationID=%10u;",
            psUpdate->psStationRow->tStationId);
    }

    return TRUE;
}

/*****************************************************************************
*
*   bBuildStationLoadQuery
*
*   Perform all the math necessary to construct a location-based DB
*   query for a station search.
*
*****************************************************************************/
static BOOLEAN bBuildStationLoadQuery (
    EV_CHARGING_DB_OBJ_STRUCT *psObj,
    LOCATION_OBJECT hLocation,
    FUEL_REGION tRegion
        )
{
    BOOLEAN bQueryBuilt = FALSE;
    OSAL_FIXED_OBJECT hTopRightLat,
                      hTopRightLon,
                      hBottomLeftLat,
                      hBottomLeftLon;

    OSAL_FIXED_OBJECT_DATA atFixedData[OSAL_FIXED_OBJECT_SIZE * 4];
    UN8 un8NumFixed = 0;    // Keep track of the fixed objects
    BOOLEAN bOk;
    N32 n32FixedTopRightLat,
        n32FixedTopRightLon,
        n32FixedBottomLeftLat,
        n32FixedBottomLeftLon;

    do
    {
        // Create the objects.  If this fails, LOCATION_bTopRight
        // will result in an error
        hTopRightLat = OSAL_FIXED.hCreateInMemory(0, 0,
            &atFixedData[OSAL_FIXED_OBJECT_SIZE*(un8NumFixed++)]);
        hTopRightLon = OSAL_FIXED.hCreateInMemory(0, 0,
            &atFixedData[OSAL_FIXED_OBJECT_SIZE*(un8NumFixed++)]);

        // Get the top right lat, lon
        bOk = LOCATION_bTopRight(hLocation, hTopRightLat, hTopRightLon);

        if (bOk == FALSE)
        {
            break;
        }

        // Create the objects.  If this fails, LOCATION_bBottomLeft
        // will result in an error
        hBottomLeftLat = OSAL_FIXED.hCreateInMemory(0, 0,
            &atFixedData[OSAL_FIXED_OBJECT_SIZE*(un8NumFixed++)]);
        hBottomLeftLon = OSAL_FIXED.hCreateInMemory(0, 0,
            &atFixedData[OSAL_FIXED_OBJECT_SIZE*(un8NumFixed++)]);

        // Get the bottom left lat, lon
        bOk = LOCATION_bBottomLeft(hLocation, hBottomLeftLat, hBottomLeftLon);

        if (bOk == FALSE)
        {
            break;
        }

        n32FixedTopRightLat =
            OSAL_FIXED.n32ScaledValue(hTopRightLat, LOCATION_BINPOINT);
        n32FixedTopRightLon =
            OSAL_FIXED.n32ScaledValue(hTopRightLon, LOCATION_BINPOINT);

        n32FixedBottomLeftLat =
            OSAL_FIXED.n32ScaledValue(hBottomLeftLat, LOCATION_BINPOINT);
        n32FixedBottomLeftLon =
            OSAL_FIXED.n32ScaledValue(hBottomLeftLon, LOCATION_BINPOINT);

        // Specify a search box within the region
        snprintf( &psObj->acBuffer[0], sizeof(psObj->acBuffer),
            FUEL_SELECT_STATION_DESC_BY_LOCATION, tRegion,
            n32FixedBottomLeftLat, n32FixedTopRightLat, // lat min / max
            n32FixedBottomLeftLon, n32FixedTopRightLon); // lon min /max

        bQueryBuilt = TRUE;
    } while (FALSE);

    return bQueryBuilt;
}

/*****************************************************************************
*
*   bLoadTables
*
*****************************************************************************/
static BOOLEAN bLoadTables (
    EV_CHARGING_DB_OBJ_STRUCT *psObj
        )
{
    BOOLEAN bSuccess;

    do
    {
        // Load the region table
        bSuccess = bLoadRegionTable(psObj);
        if (bSuccess == FALSE)
        {
            break;
        }

        // Load the text table
        bSuccess = bLoadTextTable(psObj);
        if (bSuccess == FALSE)
        {
            break;
        }

        // Load the logo table
        bSuccess = bLoadLogoTable(psObj);
        if (bSuccess == FALSE)
        {
            break;
        }

        return TRUE;

    } while (FALSE);

    return FALSE;
}

/*****************************************************************************
*
*   bLoadTextTable
*
*****************************************************************************/
static BOOLEAN bLoadTextTable (
    EV_CHARGING_DB_OBJ_STRUCT *psObj
        )
{
    EV_CHARGING_DB_RESULT_STRUCT sTextResult;
    BOOLEAN bOk;

    // Initialize the result struct
    sTextResult.bSuccess = TRUE;
    sTextResult.psObj = psObj;

    // Perform the SQL query and process the result
    // (it will provide us with a data row)
    bOk = SQL_INTERFACE.bQuery(
        psObj->hSQLRefConnection,
        FUEL_SELECT_ALL_TEXT,
        (SQL_QUERY_RESULT_HANDLER)bProcessSelectText,
        &sTextResult ) ;

    if (bOk == TRUE)
    {
        bOk = sTextResult.bSuccess;
    }

    return bOk;
}

/*****************************************************************************
*
*   bLoadLogoTable
*
*****************************************************************************/
static BOOLEAN bLoadLogoTable (
    EV_CHARGING_DB_OBJ_STRUCT *psObj
        )
{
    EV_CHARGING_DB_RESULT_STRUCT sLogoResult;
    BOOLEAN bOk;

    do
    {
        // Initialize the result struct
        sLogoResult.bSuccess = TRUE;
        sLogoResult.psObj = psObj;

        // Perform the SQL query on the persist DB and process the result
        bOk = SQL_INTERFACE.bQuery(
            psObj->hSQLPersistConnection,
            EV_CHARGING_SELECT_ALL_LOGOS,
            (SQL_QUERY_RESULT_HANDLER)bProcessSelectLogos,
            &sLogoResult );

        if ((bOk == FALSE) ||
            (sLogoResult.bSuccess == FALSE))
        {
            break;
        }

        return TRUE;
    } while (FALSE);

    return FALSE;
}

/*****************************************************************************
*
*   bLoadRegionTable
*
*****************************************************************************/
static BOOLEAN bLoadRegionTable (
    EV_CHARGING_DB_OBJ_STRUCT *psObj
        )
{
    EV_CHARGING_DB_RESULT_STRUCT sRegionResult;
    BOOLEAN bOk;

    // Initialize the result struct
    sRegionResult.bSuccess = FALSE;
    sRegionResult.psObj = psObj;

    // Perform the SQL query and process the result
    // (it will provide us with a data row)
    bOk = SQL_INTERFACE.bQuery(
            psObj->hSQLRefConnection, FUEL_SELECT_ALL_REGIONS,
            (SQL_QUERY_RESULT_HANDLER)bProcessSelectRegions,
            &sRegionResult ) ;

    if (bOk == TRUE)
    {
        bOk = sRegionResult.bSuccess;
    }

    return bOk;
}

/*****************************************************************************
*
*   vStartTransaction
*
*    This is for starting an SQL transaction
*
*****************************************************************************/
static void vStartTransaction (
    EV_CHARGING_DB_OBJ_STRUCT *psObj
        )
{
    if (psObj->bInTransaction == FALSE)
    {
        // Start a transaction for all the upcoming
        // station updates
        psObj->bInTransaction =
            SQL_INTERFACE.bStartTransaction(psObj->hSQLPersistConnection);
    }

    return;
}

/*****************************************************************************
*
*   vEndTransaction
*
*    This is for ending an SQL transaction
*
*****************************************************************************/
static void vEndTransaction (
    EV_CHARGING_DB_OBJ_STRUCT *psObj
        )
{
    if (psObj->bInTransaction == TRUE)
    {
        // End it now
        psObj->bInTransaction =
            !SQL_INTERFACE.bEndTransaction(psObj->hSQLPersistConnection);
    }

    return;
}

/*****************************************************************************
*
*   bCreatePersistDBTables
*
*   Creates the default tables in the persistent database
*
*****************************************************************************/
static BOOLEAN bCreatePersistDBTables (
    SQL_INTERFACE_OBJECT hSQLConnection,
    EV_CHARGING_DB_OBJ_STRUCT *psObj
        )
{
    BOOLEAN bSuccess = FALSE, bTransactionStarted = FALSE;

    do
    {
        EV_CHARGING_LOGO_COPY_RESULT_STRUCT sCopyResult;

        if (NULL == psObj)
        {
            break;
        }

        // Using direct call instead of vStartTransaction since
        // we don't know we have valid database handle until this
        // call returns
        bTransactionStarted = SQL_INTERFACE.bStartTransaction(hSQLConnection);

        if (FALSE == bTransactionStarted)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                EV_CHARGING_NAME
                ": failed to start transaction");
            break;
        }

        // Create price table
        bSuccess = SQL_INTERFACE.bExecuteCommand(hSQLConnection,
            EV_CHARGING_CREATE_PRICES_TABLE);

        if (bSuccess == FALSE)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                EV_CHARGING_NAME
                ": failed to create prices table in persistent storage");
            break;
        }

        // Create version table
        bSuccess = SQL_INTERFACE.bExecuteCommand(hSQLConnection,
            FUEL_CREATE_VERSION_TABLE);

        if (bSuccess == FALSE)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                EV_CHARGING_NAME
                ": failed to create version table in persistent storage");
            break;
        }

        // Create logo table
        bSuccess = SQL_INTERFACE.bExecuteCommand(hSQLConnection,
            EV_CHARGING_CREATE_LOGO_TABLE);

        if (bSuccess == FALSE)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                EV_CHARGING_NAME
                ": failed to create logo table in persistent storage");
            break;
        }

        // Copy logos from the reference db to the persistent db
        sCopyResult.bSuccess = TRUE;
        sCopyResult.hWriteConnection = hSQLConnection;

        // Perform the query on the reference logo table
        bSuccess = SQL_INTERFACE.bQuery(
            psObj->hSQLRefConnection,
            EV_CHARGING_SELECT_ALL_LOGOS,
            (SQL_QUERY_RESULT_HANDLER)bProcessSelectLogosIntoPersistDB,
            &sCopyResult );

        if ((bSuccess == FALSE) || (sCopyResult.bSuccess == FALSE))
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                EV_CHARGING_NAME
                ": failed to populate logo table in persistent storage");
            break;
        }

        // Build the schema version string
        snprintf(&psObj->acBuffer[0], sizeof(psObj->acBuffer),
            FUEL_INSERT_VERSION_ROW,
            EV_CHARGING_DB_VERSION_TYPE_DB,
            EV_CHARGING_DATABASE_FILE_VERSION);

        bSuccess = SQL_INTERFACE.bExecuteCommand(hSQLConnection, &psObj->acBuffer[0]);

        if (bSuccess == FALSE)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                EV_CHARGING_NAME
                ": failed to set db version in persistent storage");
            break;
        }
    }
    while (FALSE);

    if (bTransactionStarted == TRUE)
    {
        BOOLEAN bTransactionEnded;

        // Using direct call instead of vEndTransaction since
        // we don't know we have valid database handle until this
        // call returns
        bTransactionEnded = SQL_INTERFACE.bEndTransaction(hSQLConnection);

        if (bTransactionEnded == FALSE)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                EV_CHARGING_NAME
                ": failed to end transaction");
            bSuccess = FALSE;
        }
    }

    return bSuccess;
}

/*****************************************************************************
*
*   bPrepareTextColumn
*
*****************************************************************************/
static BOOLEAN bPrepareTextColumn (
    SQL_COLUMN_INDEX tIndex,
    SQL_BIND_TYPE_ENUM *peType,
    size_t *ptDataSize,
    void **ppvData,
    FUEL_TEXT_ROW_STRUCT *psTextRow
        )
{
    BOOLEAN bSuccess = TRUE;
    EV_CHARGING_DB_TEXT_FIELDS_ENUM eCurrentField =
        (EV_CHARGING_DB_TEXT_FIELDS_ENUM)tIndex;

    switch (eCurrentField)
    {
        case EV_CHARGING_DB_TEXT_ID:
        {
            *peType = SQL_BIND_TYPE_UN32;
            *ppvData = (void *)(size_t)psTextRow->un8TextId;
        }
        break;

        case EV_CHARGING_DB_SHORT_TEXT:
        {
            *peType = SQL_BIND_TYPE_STRING_OBJECT;
            *ppvData = (void *)(size_t)psTextRow->hText;
        }
        break;

        case EV_CHARGING_DB_LONG_TEXT:
        {
            *peType = SQL_BIND_TYPE_STRING_OBJECT;
            *ppvData = (void *)(size_t)psTextRow->hLongText;
        }
        break;

        case EV_CHARGING_DB_TEXT_MAX_FIELDS:
        default:
            bSuccess = FALSE;
        break;

    }

    return bSuccess;
}

/*****************************************************************************
*
*   bPrepareLogoColumn
*
*****************************************************************************/
static BOOLEAN bPrepareLogoColumn (
    SQL_COLUMN_INDEX tIndex,
    SQL_BIND_TYPE_ENUM *peType,
    size_t *ptDataSize,
    void **ppvData,
    FUEL_LOGO_ROW_STRUCT *psLogoRow
        )
{
    BOOLEAN bSuccess = TRUE;
    EV_CHARGING_DB_LOGO_FIELDS_ENUM eCurrentField =
        (EV_CHARGING_DB_LOGO_FIELDS_ENUM)tIndex;

    switch (eCurrentField)
    {
        case EV_CHARGING_DB_LOGO_ID:
        {
            *peType = SQL_BIND_TYPE_UN32;
            *ppvData = (void *)(size_t)psLogoRow->un16LogoId;
        }
        break;

        case EV_CHARGING_DB_LOGO_TYPE:
        {
            *peType = SQL_BIND_TYPE_UN32;
            *ppvData = (void *)(size_t)psLogoRow->eLogoType;
        }
        break;

        case EV_CHARGING_DB_LOGO_TABLE_VER:
        {
            *peType = SQL_BIND_TYPE_UN32;
            *ppvData = (void *)(size_t)psLogoRow->un8LogoTableVer;
        }
        break;

        case EV_CHARGING_DB_LOGO_LOGO_VER:
        {
            *peType = SQL_BIND_TYPE_UN32;
            *ppvData = (void *)(size_t)psLogoRow->un8LogoVer;
        }
        break;

        case EV_CHARGING_DB_LOGO_BASELINE:
        {
            *peType = SQL_BIND_TYPE_UN32;
            *ppvData = (void *)(size_t)psLogoRow->bIsBaselineLogo;
        }
        break;

        case EV_CHARGING_DB_LOGO_MAX_FIELDS:
        default:
            bSuccess = FALSE;
        break;

    }

    return bSuccess;
}

/*****************************************************************************
*
*   bPrepareRegionColumn
*
*****************************************************************************/
static BOOLEAN bPrepareRegionColumn (
    SQL_COLUMN_INDEX tIndex,
    SQL_BIND_TYPE_ENUM *peType,
    size_t *ptDataSize,
    void **ppvData,
    FUEL_REGION *ptRegionID
        )
{
    BOOLEAN bSuccess = TRUE;
    FUEL_DB_REGION_FIELDS_ENUM eCurrentField =
        (FUEL_DB_REGION_FIELDS_ENUM)tIndex;

    switch (eCurrentField)
    {
        case FUEL_DB_REGION_ID:
        {
            *peType = SQL_BIND_TYPE_UN32;
            *ppvData = (void *)(size_t)(*ptRegionID);
        }
        break;

        case FUEL_DB_REGION_MAX_FIELDS:
        default:
            bSuccess = FALSE;
        break;

    }

    return bSuccess;
}

/*****************************************************************************
*
*   bPreparePriceColumn
*
*****************************************************************************/
static BOOLEAN bPreparePriceColumn (
    SQL_COLUMN_INDEX tIndex,
    SQL_BIND_TYPE_ENUM *peType,
    size_t *ptDataSize,
    void **ppvData,
    FUEL_DB_UPDATE_PRICE_TABLE_STRUCT *psDBUpdate
        )
{
    BOOLEAN bSuccess = TRUE;
    EV_CHARGING_DB_PRICE_FIELDS_ENUM eCurrentField =
        (EV_CHARGING_DB_PRICE_FIELDS_ENUM)tIndex;

    switch (eCurrentField)
    {
        case EV_CHARGING_DB_PRICE_REGION_ID:
        {
            *peType = SQL_BIND_TYPE_UN32;
            *ppvData = (void *)(size_t)psDBUpdate->sStationRow.tRegion;
        }
        break;

        case EV_CHARGING_DB_PRICE_STATION_ID:
        {
            *peType = SQL_BIND_TYPE_UN32;
            *ppvData = (void *)(size_t)psDBUpdate->sStationRow.tStationId;
        }
        break;

        case EV_CHARGING_DB_PRICE_BRAND:
        {
            *peType = SQL_BIND_TYPE_STRING_OBJECT;
            *ppvData = psDBUpdate->sStationRow.hBrand;
        }
        break;

        case EV_CHARGING_DB_PRICE_LOGO:
        {
            *peType = SQL_BIND_TYPE_UN32;
            *ppvData = (void *)(size_t)psDBUpdate->sStationRow.n16LogoId;
        }
        break;

        case EV_CHARGING_DB_PRICE_NAME:
        {
            *peType = SQL_BIND_TYPE_STRING_OBJECT;
            *ppvData = psDBUpdate->sStationRow.hName;
        }
        break;

        case EV_CHARGING_DB_PRICE_DESC:
        {
            *peType = SQL_BIND_TYPE_STRING_OBJECT;
            *ppvData = psDBUpdate->sStationRow.hDesc;
        }
        break;

        case EV_CHARGING_DB_PRICE_ADDR:
        {
            *peType = SQL_BIND_TYPE_STRING_OBJECT;
            *ppvData = psDBUpdate->sStationRow.hAddr;
        }
        break;

        case EV_CHARGING_DB_PRICE_CITY:
        {
            *peType = SQL_BIND_TYPE_STRING_OBJECT;
            *ppvData = psDBUpdate->sStationRow.hCity;
        }
        break;

        case EV_CHARGING_DB_PRICE_STATE:
        {
            *peType = SQL_BIND_TYPE_UN32;
            *ppvData = (void *)(size_t)psDBUpdate->sStationRow.tStateID;
        }
        break;

        case EV_CHARGING_DB_PRICE_ZIP:
        {
            *peType = SQL_BIND_TYPE_STRING_OBJECT;
            *ppvData = psDBUpdate->sStationRow.hZIP;
        }
        break;

        case EV_CHARGING_DB_PRICE_PHONE:
        {
            *peType = SQL_BIND_TYPE_N64;
            *ppvData = (void *)(&psDBUpdate->sStationRow.n64Phone);
        }
        break;

        case EV_CHARGING_DB_PRICE_CHARGE_TYPE_COUNT:
        {
            *peType = SQL_BIND_TYPE_UN32;
            *ppvData = (void *)(size_t)psDBUpdate->sStationRow.un8NumPermanentTypes;
        }
        break;

        case EV_CHARGING_DB_PRICE_CHARGE_TYPES:
        {
            *peType = SQL_BIND_TYPE_BLOB;

            if ((psDBUpdate->sStationRow.un8NumPermanentTypes == 0) ||
                (psDBUpdate->sStationRow.psTypes == NULL))
            {
                *ppvData = (void *)NULL;
                *ptDataSize = 0;
            }
            else
            {
                UN8 un8Index;

                for (un8Index = 0;
                     un8Index < psDBUpdate->sStationRow.un8NumPermanentTypes;
                     un8Index++)
                {
                    // Read all fuel types
                    psDBUpdate->aun8RawRefuelingTypeData[un8Index] =
                        psDBUpdate->sStationRow.psTypes[un8Index].un8FuelType;
                }

                *ppvData = (void *)&psDBUpdate->aun8RawRefuelingTypeData[0];
                *ptDataSize = psDBUpdate->sStationRow.un8NumPermanentTypes * sizeof(UN8);
            }
        }
        break;

        case EV_CHARGING_DB_PRICE_CHARGE_NUMS:
        {
            *peType = SQL_BIND_TYPE_BLOB;

            if ((psDBUpdate->sStationRow.un8NumPermanentTypes == 0) ||
                (psDBUpdate->sStationRow.psTypes == NULL))
            {
                *ppvData = (void *)NULL;
                *ptDataSize = 0;
            }
            else
            {
                UN8 un8Index;

                for (un8Index = 0;
                     un8Index < psDBUpdate->sStationRow.un8NumPermanentTypes;
                     un8Index++)
                {
                    // Read all position counts
                    psDBUpdate->aun8RawRefuelingTypeData[un8Index] =
                        psDBUpdate->sStationRow.psTypes[un8Index].un8NumPositions;
                }

                *ppvData = (void *)&psDBUpdate->aun8RawRefuelingTypeData[0];
                *ptDataSize = psDBUpdate->sStationRow.un8NumPermanentTypes * sizeof(UN8);
            }
        }
        break;

        case EV_CHARGING_DB_PRICE_LAT:
        {
            *peType = SQL_BIND_TYPE_UN32;
            *ppvData = (void *)(size_t)(UN32)psDBUpdate->sStationRow.n32Lat;
        }
        break;

        case EV_CHARGING_DB_PRICE_LON:
        {
            *peType = SQL_BIND_TYPE_UN32;
            *ppvData = (void *)(size_t)(UN32)psDBUpdate->sStationRow.n32Lon;
        }
        break;

        case EV_CHARGING_DB_PRICE_AMENITIES:
        {
            *peType = SQL_BIND_TYPE_UN32;
            *ppvData = (void *)(size_t)(UN32)psDBUpdate->sStationRow.un32Amenities;
        }
        break;

        case EV_CHARGING_DB_PRICE_AGE_EPOCH_SECONDS:
        {
            *peType = SQL_BIND_TYPE_UN32;
            *ppvData = (void *)(size_t)(UN32)psDBUpdate->sPriceRow.un32YoungestPriceAgeUTCSeconds;
        }
        break;

        case EV_CHARGING_DB_PRICE_MAX_FIELDS:
        default:
            bSuccess = FALSE;
        break;

    }

    return bSuccess;
}

/*****************************************************************************
*
*   bPrepareStationColumn
*
*****************************************************************************/
static BOOLEAN bPrepareStationColumn (
    SQL_COLUMN_INDEX tIndex,
    SQL_BIND_TYPE_ENUM *peType,
    size_t *ptDataSize,
    void **ppvData,
    EV_DB_STATION_UPDATE_STRUCT *psUpdate
        )
{
    BOOLEAN bSuccess = TRUE;
    EV_CHARGING_DB_STATION_FIELDS_ENUM eCurrentField = psUpdate->aeFields[tIndex];

    switch (eCurrentField)
    {
        case EV_CHARGING_DB_STATION_ID:
        {
            *peType = SQL_BIND_TYPE_UN32;
            *ppvData = (void *)(size_t)psUpdate->psStationRow->tStationId;
        }
        break;

        case EV_CHARGING_DB_STATION_BRAND:
        {
            *peType = SQL_BIND_TYPE_STRING_OBJECT;
            *ppvData = psUpdate->psStationRow->hBrand;
        }
        break;

        case EV_CHARGING_DB_STATION_LOGO:
        {
            *peType = SQL_BIND_TYPE_UN32;
            *ppvData = (void *)(size_t)psUpdate->psStationRow->n16LogoId;
        }
        break;

        case EV_CHARGING_DB_STATION_NAME:
        {
            *peType = SQL_BIND_TYPE_STRING_OBJECT;
            *ppvData = psUpdate->psStationRow->hName;
        }
        break;

        case EV_CHARGING_DB_STATION_DESC:
        {
            *peType = SQL_BIND_TYPE_STRING_OBJECT;
            *ppvData = psUpdate->psStationRow->hDesc;
        }
        break;

        case EV_CHARGING_DB_STATION_ADDR:
        {
            *peType = SQL_BIND_TYPE_STRING_OBJECT;
            *ppvData = psUpdate->psStationRow->hAddr;
        }
        break;

        case EV_CHARGING_DB_STATION_CITY:
        {
            *peType = SQL_BIND_TYPE_STRING_OBJECT;
            *ppvData = psUpdate->psStationRow->hCity;
        }
        break;

        case EV_CHARGING_DB_STATION_STATE:
        {
            *peType = SQL_BIND_TYPE_UN32;
            *ppvData = (void *)(size_t)psUpdate->psStationRow->tStateID;
        }
        break;

        case EV_CHARGING_DB_STATION_ZIP:
        {
            *peType = SQL_BIND_TYPE_STRING_OBJECT;
            *ppvData = psUpdate->psStationRow->hZIP;
        }
        break;

        case EV_CHARGING_DB_STATION_PHONE:
        {
            *peType = SQL_BIND_TYPE_N64;
            *ppvData = (void *)(&psUpdate->psStationRow->n64Phone);
        }
        break;

        case EV_CHARGING_DB_STATION_CHARGE_TYPE_COUNT:
        {
            *peType = SQL_BIND_TYPE_UN32;
            *ppvData = (void *)(size_t)psUpdate->psStationRow->un8NumPermanentTypes;
        }
        break;

        case EV_CHARGING_DB_STATION_CHARGE_TYPES:
        {
            *peType = SQL_BIND_TYPE_BLOB;

            if ((psUpdate->psStationRow->un8NumPermanentTypes == 0) ||
                (psUpdate->psStationRow->psTypes == NULL))
            {
                *ppvData = (void *)NULL;
                *ptDataSize = 0;
            }
            else
            {
                UN8 un8Index;

                for (un8Index = 0;
                     un8Index < psUpdate->psStationRow->un8NumPermanentTypes;
                     un8Index++)
                {
                    // Read all fuel types
                    psUpdate->aun8RawRefuelingTypeData[un8Index] =
                        psUpdate->psStationRow->psTypes[un8Index].un8FuelType;
                }

                *ppvData = (void *)&psUpdate->aun8RawRefuelingTypeData[0];
                *ptDataSize = psUpdate->psStationRow->un8NumPermanentTypes * sizeof(UN8);
            }
        }
        break;

        case EV_CHARGING_DB_STATION_CHARGE_NUMS:
        {
            *peType = SQL_BIND_TYPE_BLOB;

            if ((psUpdate->psStationRow->un8NumPermanentTypes == 0) ||
                (psUpdate->psStationRow->psTypes == NULL))
            {
                *ppvData = (void *)NULL;
                *ptDataSize = 0;
            }
            else
            {
                UN8 un8Index;

                for (un8Index = 0;
                     un8Index < psUpdate->psStationRow->un8NumPermanentTypes;
                     un8Index++)
                {
                    // Read all position counts
                    psUpdate->aun8RawRefuelingTypeData[un8Index] =
                        psUpdate->psStationRow->psTypes[un8Index].un8NumPositions;
                }

                *ppvData = (void *)&psUpdate->aun8RawRefuelingTypeData[0];
                *ptDataSize = psUpdate->psStationRow->un8NumPermanentTypes * sizeof(UN8);
            }
        }
        break;

        case EV_CHARGING_DB_STATION_LAT:
        {
            *peType = SQL_BIND_TYPE_UN32;
            *ppvData = (void *)(size_t)(UN32)psUpdate->psStationRow->n32Lat;
        }
        break;

        case EV_CHARGING_DB_STATION_LON:
        {
            *peType = SQL_BIND_TYPE_UN32;
            *ppvData = (void *)(size_t)(UN32)psUpdate->psStationRow->n32Lon;
        }
        break;

        case EV_CHARGING_DB_STATION_AMENITIES:
        {
            *peType = SQL_BIND_TYPE_UN32;
            *ppvData = (void *)(size_t)(UN32)psUpdate->psStationRow->un32Amenities;
        }
        break;

        case EV_CHARGING_DB_STATION_MAX_FIELDS:
        default:
            bSuccess = FALSE;
        break;

    }

    return bSuccess;
}

/*****************************************************************************
*
*   bVerifyAndLoadDBVersionFields
*
*****************************************************************************/
static BOOLEAN bVerifyAndLoadDBVersionFields (
    SQL_INTERFACE_OBJECT hSQLConnection,
    EV_CHARGING_VER_CTRL_STRUCT *psVerCtrl
        )
{
    EV_CHARGING_DB_VERSION_RESULT_STRUCT sVersion;
    BOOLEAN bVersionMatched = FALSE, bOk;

    // Initialize the result structure
    sVersion.bSuccess = FALSE;
    sVersion.psVerCtrl = psVerCtrl;

    // Perform the SQL query and process the result
    // (it will provide us with a data row)
    bOk = SQL_INTERFACE.bQuery(
             hSQLConnection, FUEL_SELECT_ALL_DB_VERSION,
            (SQL_QUERY_RESULT_HANDLER)bProcessSelectVersion,
            &sVersion ) ;

    if (bOk == TRUE)
    {
        bVersionMatched = sVersion.bSuccess;
    }

    return bVersionMatched;
}

/*****************************************************************************
*
*   n32ExtractDataVersion
*
*****************************************************************************/
static N32 n32ExtractDataVersion (
    SQL_INTERFACE_OBJECT hConnection,
    void *pvArg
        )
{
    BOOLEAN bVerified;
    EV_CHARGING_VER_CTRL_STRUCT sVerCtrl;
    N32 n32Result = DB_UTIL_DB_UNDER_CONSTRUCTION_VER;

    if (hConnection == SQL_INTERFACE_INVALID_OBJECT)
    {
        return DB_UTIL_DB_UNDER_CONSTRUCTION_VER;
    }

    // Get the version fields
    bVerified = bVerifyAndLoadDBVersionFields(
        hConnection, &sVerCtrl);
    if (bVerified == TRUE)
    {
        n32Result = (N32)sVerCtrl.n16DBContentVersion;
    }

    return n32Result;
}

/*******************************************************************************
*
*   bProcessSelectVersion
*
*   This function is provided to the SQL_INTERFACE_OBJECT in order to process
*   a "select all" query made on the database version relation.  It populates
*   a pointer to a FUEL_VERSION_ROW_STRUCT with the information found
*   in the database as well as performs version verification.
*
*******************************************************************************/
static BOOLEAN bProcessSelectVersion (
    SQL_QUERY_COLUMN_STRUCT *psColumn,
    N32 n32NumberOfColumns,
    EV_CHARGING_DB_VERSION_RESULT_STRUCT *psResult
        )
{
    EV_CHARGING_DB_VERSION_FIELDS_ENUM eCurrentField =
        (EV_CHARGING_DB_VERSION_FIELDS_ENUM)0;
    EV_CHARGING_VERSION_ROW_STRUCT sCurrentRow;

    // Verify input
    if (psResult == NULL)
    {
        return FALSE;
    }

    // Initialize row structure
    sCurrentRow.eType = EV_CHARGING_DB_VERSION_MAX_TYPES;
    sCurrentRow.n32Version = -1;

    // If we have the correct number of columns, then we have good results
    if ( n32NumberOfColumns == EV_CHARGING_DB_VERSION_MAX_FIELDS )
    {
        // Start off by marking this a success
        psResult->bSuccess = TRUE;
    }
    else
    {
        // This is definitely not a match
        psResult->bSuccess = FALSE;
        return FALSE;
    }

    do
    {
        // Decode the current field based on it's type
        switch (eCurrentField)
        {
            case EV_CHARGING_DB_VERSION_TYPE:
            {
                sCurrentRow.eType = (EV_CHARGING_DB_VERSION_TYPES_ENUM)
                    psColumn[eCurrentField].uData.sUN32.un32Data;
            }
            break;

            case EV_CHARGING_DB_VERSION_VER:
            {
                sCurrentRow.n32Version =(N32)
                    psColumn[eCurrentField].uData.sUN32.un32Data;
            }
            break;

            default:
            {
                SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    EV_CHARGING_NAME": bad field in ver tabl");

                psResult->bSuccess = FALSE;
            }
            break;
        }

    } while ( ++eCurrentField < EV_CHARGING_DB_VERSION_MAX_FIELDS);

    // Now process the version data
    switch(sCurrentRow.eType)
    {
        case EV_CHARGING_DB_VERSION_TYPE_DB:
        {
            if (sCurrentRow.n32Version != (N32)EV_CHARGING_DATABASE_FILE_VERSION)
            {
                printf(EV_CHARGING_NAME": DB version fail");
                psResult->bSuccess = FALSE;
            }
        }
        break;

        case EV_CHARGING_DB_VERSION_TYPE_CONTENTS:
        {
            // Just pull this value in
            psResult->psVerCtrl->n16DBContentVersion =
                (N16)sCurrentRow.n32Version;
        }
        break;

        default:
        {
            psResult->bSuccess = FALSE;
        }
    }

    // FALSE will stop iteration; TRUE will keep going
    return psResult->bSuccess;
}


/*******************************************************************************
*
*   bProcessSelectText
*
*   This function is provided to the SQL_INTERFACE_OBJECT in order to process
*   a "select all" query made on the database text table.  It populates the
*   manager's text table based upon the data found in the database.
*
*******************************************************************************/
static BOOLEAN bProcessSelectText (
    SQL_QUERY_COLUMN_STRUCT *psColumn,
    N32 n32NumberOfColumns,
    EV_CHARGING_DB_RESULT_STRUCT *psResult
        )
{
    EV_CHARGING_DB_TEXT_FIELDS_ENUM eCurrentField =
        (EV_CHARGING_DB_TEXT_FIELDS_ENUM)0;
    FUEL_TEXT_ROW_STRUCT sCurrentRow;

    // Verify input
    if (psResult == NULL)
    {
        return FALSE;
    }

    // Initialize the row structure
    OSAL.bMemSet((void *)&sCurrentRow, 0, sizeof(FUEL_TEXT_ROW_STRUCT));

    do
    {
        // Decode the current field based on it's type
        switch (eCurrentField)
        {
            case EV_CHARGING_DB_TEXT_ID:
            {
                sCurrentRow.un8TextId = (UN8)
                    psColumn[eCurrentField].uData.sUN32.un32Data;
            }
            break;

            case EV_CHARGING_DB_SHORT_TEXT:
            {
                // Create a string object for this
                sCurrentRow.hText = STRING.hCreate(
                    (const char *)psColumn[eCurrentField].uData.sCString.pcData,
                    strlen((const char *)psColumn[eCurrentField].uData.sCString.pcData));
                if (sCurrentRow.hText == STRING_INVALID_OBJECT)
                {
                    psResult->bSuccess = FALSE;
                }
            }
            break;

            case EV_CHARGING_DB_LONG_TEXT:
            {
                // Create a string object for this
                sCurrentRow.hLongText = STRING.hCreate(
                    (const char *)psColumn[eCurrentField].uData.sCString.pcData,
                    strlen((const char *)psColumn[eCurrentField].uData.sCString.pcData));
                if (sCurrentRow.hLongText == STRING_INVALID_OBJECT)
                {
                    psResult->bSuccess = FALSE;
                }
            }
            break;

            default:
            {
                SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    EV_CHARGING_NAME": bad field in text table");

                psResult->bSuccess = FALSE;
            }
            break;
        }

        // Stop it we experienced an error
        if (psResult->bSuccess == FALSE)
        {
            break;
        }

    } while ( ++eCurrentField < EV_CHARGING_DB_TEXT_MAX_FIELDS);

    if (psResult->bSuccess == TRUE)
    {
        // Tell the manager about this text
        psResult->bSuccess = GsFuelMgrIntf.bReportTextFromDB(
            psResult->psObj->hFuelService, &sCurrentRow);
    }

    // Stop if we found an error
    if (psResult->bSuccess == FALSE)
    {
        if (sCurrentRow.hText != STRING_INVALID_OBJECT)
        {
            // Free the string
            STRING_vDestroy(sCurrentRow.hText);
            sCurrentRow.hText = STRING_INVALID_OBJECT;
        }

        // Stop iterating
        return FALSE;
    }

    // Keep going
    return TRUE;
}

/*******************************************************************************
*
*   bProcessSelectLogos
*
*******************************************************************************/
static BOOLEAN bProcessSelectLogos (
    SQL_QUERY_COLUMN_STRUCT *psColumn,
    N32 n32NumberOfColumns,
    EV_CHARGING_DB_RESULT_STRUCT *psResult
        )
{
    FUEL_LOGO_ROW_STRUCT sCurrentRow;

    // Verify input
    if (psResult == NULL)
    {
        return FALSE;
    }

    if (n32NumberOfColumns == EV_CHARGING_DB_LOGO_MAX_FIELDS)
    {
        psResult->bSuccess = TRUE;
    }
    else
    {
        psResult->bSuccess = FALSE;
        return FALSE;
    }

    // Initialize the row structure
    OSAL.bMemSet((void *)&sCurrentRow, 0, sizeof(FUEL_LOGO_ROW_STRUCT));

    // Read the values for this row
    sCurrentRow.un16LogoId = (UN16)
        psColumn[EV_CHARGING_DB_LOGO_ID].uData.sUN32.un32Data;

    sCurrentRow.eLogoType = (FUEL_BRAND_LOGO_IMAGE_TYPE_ENUM)
        psColumn[EV_CHARGING_DB_LOGO_TYPE].uData.sUN32.un32Data;

    sCurrentRow.un8LogoTableVer = (UN8)
        psColumn[EV_CHARGING_DB_LOGO_TABLE_VER].uData.sUN32.un32Data;

    sCurrentRow.un8LogoVer = (UN8)
        psColumn[EV_CHARGING_DB_LOGO_LOGO_VER].uData.sUN32.un32Data;

    sCurrentRow.bIsBaselineLogo = (BOOLEAN)
        psColumn[EV_CHARGING_DB_LOGO_BASELINE].uData.sUN32.un32Data;

    // Tell the manager about this logo
    psResult->bSuccess = GsFuelMgrIntf.bReportLogoFromDB(
        psResult->psObj->hFuelService, &sCurrentRow);

    // Keep going
    return TRUE;
}

/*******************************************************************************
*
*   bProcessSelectLogos
*
*   Only used when we need to copy the logo table from the
*   reference database
*
*******************************************************************************/
static BOOLEAN bProcessSelectLogosIntoPersistDB (
    SQL_QUERY_COLUMN_STRUCT *psColumn,
    N32 n32NumberOfColumns,
    EV_CHARGING_LOGO_COPY_RESULT_STRUCT *psResult
        )
{
    FUEL_LOGO_ROW_STRUCT sCurrentRow;

    // Verify input
    if (psResult == NULL)
    {
        return FALSE;
    }

    if (n32NumberOfColumns == EV_CHARGING_DB_LOGO_MAX_FIELDS)
    {
        psResult->bSuccess = TRUE;
    }
    else
    {
        psResult->bSuccess = FALSE;
        return FALSE;
    }

    // Initialize the row structure
    OSAL.bMemSet((void *)&sCurrentRow, 0, sizeof(FUEL_LOGO_ROW_STRUCT));

    // Read the values for this row
    sCurrentRow.un16LogoId = (UN16)
        psColumn[EV_CHARGING_DB_LOGO_ID].uData.sUN32.un32Data;

    sCurrentRow.eLogoType = (FUEL_BRAND_LOGO_IMAGE_TYPE_ENUM)
        psColumn[EV_CHARGING_DB_LOGO_TYPE].uData.sUN32.un32Data;

    sCurrentRow.un8LogoTableVer = (UN8)
        psColumn[EV_CHARGING_DB_LOGO_TABLE_VER].uData.sUN32.un32Data;

    sCurrentRow.un8LogoVer = (UN8)
        psColumn[EV_CHARGING_DB_LOGO_LOGO_VER].uData.sUN32.un32Data;

    sCurrentRow.bIsBaselineLogo = (BOOLEAN)
        psColumn[EV_CHARGING_DB_LOGO_BASELINE].uData.sUN32.un32Data;

    psResult->bSuccess = SQL_INTERFACE.bExecutePreparedCommand(
        psResult->hWriteConnection,
        EV_CHARGING_INSERT_UPDATE_LOGO,
        EV_CHARGING_DB_LOGO_MAX_FIELDS,
        (PREPARED_QUERY_COLUMN_CALLBACK)bPrepareLogoColumn,
        &sCurrentRow);

    // Keep going if that went well
    return psResult->bSuccess;
}

/*******************************************************************************
*
*   bProcessSelectRegionExists
*
*******************************************************************************/
static BOOLEAN bProcessSelectRegionExists (
    SQL_QUERY_COLUMN_STRUCT *psColumn,
    N32 n32NumberOfColumns,
    BOOLEAN *pbRegionExists
        )
{
    // Verify input
    if (pbRegionExists == NULL)
    {
        // Stop now
        return FALSE;
    }

    // We should only have 1 column here
    if (n32NumberOfColumns == 1 )
    {
        if ((psColumn[0].eType == SQL_COLUMN_TYPE_INTEGER) &&
            (psColumn[0].uData.sUN32.un32Data == 1))
        {
            // We performed a "select count(1)" on this
            // region ID and we got a "1" in response...good
            *pbRegionExists = TRUE;
        }
    }

    return FALSE;
}

/*******************************************************************************
*
*   bProcessSelectRegions
*
*   This function is provided to the SQL_INTERFACE_OBJECT in order to process
*   a "select all" query made on the database region table.  It populates the
*   manager's region list based upon the data found in the database.
*
*******************************************************************************/
static BOOLEAN bProcessSelectRegions (
    SQL_QUERY_COLUMN_STRUCT *psColumn,
    N32 n32NumberOfColumns,
    EV_CHARGING_DB_RESULT_STRUCT *psResult
        )
{
    FUEL_DB_REGION_FIELDS_ENUM eCurrentField =
        (FUEL_DB_REGION_FIELDS_ENUM)0;
    FUEL_REGION tRegion = FUEL_INVALID_REGION;

    // Verify input
    if (psResult == NULL)
    {
        // Stop now
        return FALSE;
    }

    // If we have the correct number of columns, then we have good results
    if (n32NumberOfColumns == FUEL_DB_REGION_MAX_FIELDS )
    {
        // Start off by marking this a success
        psResult->bSuccess = TRUE;
    }
    else
    {
        // This is definitely not a match
        psResult->bSuccess = FALSE;

        // Stop now
        return FALSE;
    }

    do
    {
        // Decode the current field based on it's type
        switch (eCurrentField)
        {
            case FUEL_DB_REGION_ID:
            {
                tRegion = (FUEL_REGION)
                    psColumn[eCurrentField].uData.sUN32.un32Data;
            }
            break;

            default:
            {
                SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    EV_CHARGING_NAME": bad field in region table");
            }
            break;
        }

    } while ( ++eCurrentField < FUEL_DB_REGION_MAX_FIELDS);

    // Only add this region if we extracted it without error
    if (psResult->bSuccess == TRUE)
    {
        // Add this region to our table
        psResult->bSuccess = GsFuelMgrIntf.bReportRegionFromDB(
            psResult->psObj->hFuelService, tRegion);
    }

    // Stop if we encountered an error
    return psResult->bSuccess;
}

/*******************************************************************************
*
*   bProcessSelectStations
*
*******************************************************************************/
static BOOLEAN bProcessSelectStations (
    SQL_QUERY_COLUMN_STRUCT *psColumn,
    N32 n32NumberOfColumns,
    EV_CHARGING_DB_LOAD_STATION_STRUCT *psBuild
        )
{
    EV_CHARGING_DB_STATION_FIELDS_ENUM eCurrentField =
        (EV_CHARGING_DB_STATION_FIELDS_ENUM)0;
    FUEL_STATION_ROW_STRUCT sStationRow;
    FUEL_REFUELING_TYPE_STRUCT asTypes[FUEL_REFUELING_TYPES_ARRAY_SIZE];
    BOOLEAN bStationInUse = FALSE,
            bStationNeeded = TRUE,
            bOk = TRUE,
            bStationOwnsStrings = FALSE,
            bCacheOnly = TRUE;
    FUEL_STATION_OBJECT hStation =
        FUEL_STATION_INVALID_OBJECT;
    OSAL_FIXED_OBJECT_DATA atFixedData[OSAL_FIXED_OBJECT_SIZE * 2];
    OSAL_FIXED_OBJECT hLat = OSAL_FIXED_INVALID_OBJECT,
                      hLon = OSAL_FIXED_INVALID_OBJECT;
    UN8 *pun8PositionFuelTypes = NULL, *pun8PositionsAvailable = NULL;

    // Verify input
    if (psBuild == NULL)
    {
        return FALSE;
    }

    // If there is at least one data row and the correct
    // number of columns (just make sure we have enough),
    // then we have good results
    if (n32NumberOfColumns < EV_CHARGING_DB_STATION_MAX_FIELDS )
    {
        psBuild->bSuccess = FALSE;

        return FALSE;
    }

    // Initialize the row structures
    OSAL.bMemSet(&sStationRow, 0, sizeof(FUEL_STATION_ROW_STRUCT));
    OSAL.bMemSet(&asTypes[0], 0,
        (sizeof(FUEL_REFUELING_TYPE_STRUCT) * FUEL_REFUELING_TYPES_ARRAY_SIZE));

    // Populate the station row's region
    sStationRow.tRegion = psBuild->tRegion;

    // Get the station ID
    sStationRow.tStationId = (FUEL_STATION_ID)
        psColumn[EV_CHARGING_DB_STATION_ID].uData.sUN32.un32Data;

    // Get the lat/lon of this station
    sStationRow.n32Lat = (N32)
            psColumn[EV_CHARGING_DB_STATION_LAT].uData.sUN32.un32Data;
    sStationRow.n32Lon = (N32)
            psColumn[EV_CHARGING_DB_STATION_LON].uData.sUN32.un32Data;

    // Create some fixed objects on the stack if the database
    // has valid values
    if ((sStationRow.n32Lat != -1) && (sStationRow.n32Lon != -1))
    {
        hLat = OSAL_FIXED.hCreateInMemory(
            sStationRow.n32Lat, LOCATION_BINPOINT,
            &atFixedData[0]);
        hLon = OSAL_FIXED.hCreateInMemory(
            sStationRow.n32Lon, LOCATION_BINPOINT,
            &atFixedData[OSAL_FIXED_OBJECT_SIZE]);
    }

    // If this this a not a query on a particular station,
    // then we need to check the location criteria
    if (psBuild->tStation == FUEL_INVALID_STATION_ID)
    {
        BOOLEAN bLocationAccepted;

        // Does the station meet our background location criteria?
        // Note: could we improve performance with r-tree query and remove this call?
        bLocationAccepted = GsFuelMgrIntf.bBackgroundLocationFilter(
            psBuild->psObj->hFuelService, hLat, hLon, &bCacheOnly);

        if (bLocationAccepted == FALSE)
        {
            // We're done now with this station,
            // but keep iterating
            return TRUE;
        }
    }
    else // This is a query for a particular station
    {
        // Is this the one we asked for?
        if (psBuild->tStation != sStationRow.tStationId)
        {
            // This isn't the correct station
            return TRUE;
        }

        // This is a request to place a station into the DSRL
        bCacheOnly = FALSE;
    }

    // Attempt to find this station to see if
    // it's already in service.  We may already be
    // tracking this station, and if so, we can skip
    // the database processing code below
    hStation = GsFuelMgrIntf.hGetStation(
        psBuild->psObj->hFuelService,
        sStationRow.tRegion, sStationRow.tStationId,
        &bStationNeeded);
    if (hStation != FUEL_STATION_INVALID_OBJECT)
    {
        // This station is available...
        // We can skip the database processing below
        // now, so setup the loop to quit immediately
        eCurrentField = EV_CHARGING_DB_STATION_MAX_FIELDS;

        // This station is in use by the manager already
        bStationInUse = TRUE;
    }

    if (bStationNeeded == FALSE)
    {
        // We don't want this station
        // -- keep iterating
        return TRUE;
    }

    // We got past the location filter -- grab the
    // entire station record and create a station object
    do
    {
        switch (eCurrentField)
        {
            case EV_CHARGING_DB_STATION_ID:
            {
                // We already got this
            }
            break;

            case EV_CHARGING_DB_STATION_BRAND:
            {
                sStationRow.hBrand = STRING.hCreate(
                    (const char *)psColumn[eCurrentField].uData.sCString.pcData,
                    strlen((const char *)psColumn[eCurrentField].uData.sCString.pcData));
                if (sStationRow.hBrand == STRING_INVALID_OBJECT)
                {
                    bOk = FALSE;
                }
            }
            break;

            case EV_CHARGING_DB_STATION_LOGO:
            {
                sStationRow.n16LogoId = (N16)
                    psColumn[eCurrentField].uData.sUN32.un32Data;
            }
            break;

            case EV_CHARGING_DB_STATION_NAME:
            {
                sStationRow.hName = STRING.hCreate(
                    (const char *)psColumn[eCurrentField].uData.sCString.pcData,
                    strlen((const char *)psColumn[eCurrentField].uData.sCString.pcData));
                if (sStationRow.hName == STRING_INVALID_OBJECT)
                {
                    bOk = FALSE;
                }
            }
            break;

            case EV_CHARGING_DB_STATION_DESC:
            {
                sStationRow.hDesc = STRING.hCreate(
                    (const char *)psColumn[eCurrentField].uData.sCString.pcData,
                    strlen((const char *)psColumn[eCurrentField].uData.sCString.pcData));
            }
            break;

            case EV_CHARGING_DB_STATION_ADDR:
            {
                sStationRow.hAddr = STRING.hCreate(
                    (const char *)psColumn[eCurrentField].uData.sCString.pcData,
                    strlen((const char *)psColumn[eCurrentField].uData.sCString.pcData));
                if (sStationRow.hAddr == STRING_INVALID_OBJECT)
                {
                    bOk = FALSE;
                }
            }
            break;

            case EV_CHARGING_DB_STATION_CITY:
            {
                sStationRow.hCity = STRING.hCreate(
                    (const char *)psColumn[eCurrentField].uData.sCString.pcData,
                    strlen((const char *)psColumn[eCurrentField].uData.sCString.pcData));
            }
            break;

            case EV_CHARGING_DB_STATION_STATE:
            {
                sStationRow.tStateID = (STATE_ID)
                    psColumn[eCurrentField].uData.sUN32.un32Data;
            }
            break;

            case EV_CHARGING_DB_STATION_ZIP:
            {
                sStationRow.hZIP = STRING.hCreate(
                    (const char *)psColumn[eCurrentField].uData.sCString.pcData,
                    strlen((const char *)psColumn[eCurrentField].uData.sCString.pcData));
            }
            break;

            case EV_CHARGING_DB_STATION_PHONE:
            {
                sStationRow.n64Phone =
                    psColumn[eCurrentField].n64NativeInteger;
            }
            break;

            case EV_CHARGING_DB_STATION_CHARGE_TYPE_COUNT:
            {
                // Grab this value
                sStationRow.un8NumPermanentTypes = (UN8)
                    psColumn[eCurrentField].uData.sUN32.un32Data;
            }
            break;

            case EV_CHARGING_DB_STATION_CHARGE_TYPES:
            {
                pun8PositionFuelTypes = (UN8 *)psColumn[eCurrentField].uData.sBLOB.pvData;

                if (pun8PositionFuelTypes == NULL)
                {
                    // This shouldn't happen
                    bOk = FALSE;
                }
            }
            break;

            case EV_CHARGING_DB_STATION_CHARGE_NUMS:
            {
                pun8PositionsAvailable = (UN8 *)psColumn[eCurrentField].uData.sBLOB.pvData;

                if (pun8PositionsAvailable == NULL)
                {
                    // This shouldn't happen
                    bOk = FALSE;
                }
            }
            break;

            case EV_CHARGING_DB_STATION_LAT:
            case EV_CHARGING_DB_STATION_LON:
            {
                // We already got these
            }
            break;

            case EV_CHARGING_DB_STATION_AMENITIES:
            {
                sStationRow.un32Amenities =
                    psColumn[eCurrentField].uData.sUN32.un32Data;
            }
            break;

            default: // Shouldn't happen
            break;
        }

        // Stop if we experienced an error
        if (bOk == FALSE)
        {
            break;
        }
    } while (++eCurrentField < EV_CHARGING_DB_STATION_MAX_FIELDS);

    if (bOk == TRUE)
    {
        UN8 un8Index;

        // Populate the refueling positions structure
        for (un8Index = 0;
            un8Index < sStationRow.un8NumPermanentTypes;
            un8Index++)
        {
            // Process each entry now
            asTypes[un8Index].un8FuelType = pun8PositionFuelTypes[un8Index];
            asTypes[un8Index].eFuelType = eMatchFuelType(pun8PositionFuelTypes[un8Index]);
            asTypes[un8Index].un8NumPositions = pun8PositionsAvailable[un8Index];
            asTypes[un8Index].hLongFuelName = STRING_INVALID_OBJECT;
            asTypes[un8Index].hShortFuelName = STRING_INVALID_OBJECT;
        }

        // Provide our types array
        sStationRow.psTypes = &asTypes[0];

        if (hStation == FUEL_STATION_INVALID_OBJECT)
        {
            // Now, create a station object and take ownership
            // of the objects in the row
            hStation = GsFuelMgrIntf.hCreateStation(
                psBuild->psObj->hFuelService, &sStationRow);

            // We have created an object now, (well, at least we tried)
            psBuild->tNumObjectsCreated++;
        }

        if (hStation != FUEL_STATION_INVALID_OBJECT)
        {
            // The fuel station object owns the STRING objects now
            bStationOwnsStrings = TRUE;

            // Have the manager put this station into service
            bStationInUse |= GsFuelMgrIntf.bPutStationInService(
                psBuild->psObj->hFuelService,
                hStation, bCacheOnly);

            // Is the station in use now?
            if (FALSE == bStationInUse)
            {
                // We need to destroy this station now
                FUEL_STATION_vDestroy(hStation);
            }
        }
    }
    else
    {
        psBuild->bSuccess = FALSE;

        // We need to destroy all strings directly
        if (bStationOwnsStrings == FALSE)
        {
            if (sStationRow.hAddr != STRING_INVALID_OBJECT)
            {
                STRING_vDestroy(sStationRow.hAddr);
                sStationRow.hAddr = STRING_INVALID_OBJECT;
            }

            if (sStationRow.hBrand != STRING_INVALID_OBJECT)
            {
                STRING_vDestroy(sStationRow.hBrand);
                sStationRow.hBrand = STRING_INVALID_OBJECT;
            }

            if (sStationRow.hCity != STRING_INVALID_OBJECT)
            {
                STRING_vDestroy(sStationRow.hCity);
                sStationRow.hCity = STRING_INVALID_OBJECT;
            }

            if (sStationRow.hDesc != STRING_INVALID_OBJECT)
            {
                STRING_vDestroy(sStationRow.hDesc);
                sStationRow.hDesc = STRING_INVALID_OBJECT;
            }

            if (sStationRow.hName != STRING_INVALID_OBJECT)
            {
                STRING_vDestroy(sStationRow.hName);
                sStationRow.hName = STRING_INVALID_OBJECT;
            }

            if (sStationRow.hZIP != STRING_INVALID_OBJECT)
            {
                STRING_vDestroy(sStationRow.hZIP);
                sStationRow.hZIP = STRING_INVALID_OBJECT;
            }
        }
    }

    // FALSE will stop iteration; TRUE will keep going
    return psBuild->bSuccess;
}

/*******************************************************************************
*
*   bProcessSelectStationExists
*
*******************************************************************************/
static BOOLEAN bProcessSelectStationExists (
    SQL_QUERY_COLUMN_STRUCT *psColumn,
    N32 n32NumberOfColumns,
    BOOLEAN *pbStationExists
        )
{
    // Verify input
    if (NULL == pbStationExists)
    {
        // Stop now
        return FALSE;
    }

    // We should only have 1 column here
    if (1 == n32NumberOfColumns)
    {
        if ((SQL_COLUMN_TYPE_INTEGER == psColumn[0].eType) &&
            (1 == psColumn[0].uData.sUN32.un32Data))
        {
            // We performed a "select count(1)" on this
            // station ID and we got a "1" in response...good
            *pbStationExists = TRUE;
        }
    }

    // Stop processing
    return FALSE;
}

/*******************************************************************************
*
*   bProcessSelectPrices
*
*   This function is provided to the SQL_INTERFACE_OBJECT in order to process
*   a "select" query made on the database price table.  It creates FUEL_STATION
*   objects (if necessary) for a target and populates those stations with prices
*   (again, if necessary).  These stations are then passed to the target's
*   DSRL to determine if they are wanted.
*
*******************************************************************************/
static BOOLEAN bProcessSelectPrices (
    SQL_QUERY_COLUMN_STRUCT *psColumn,
    N32 n32NumberOfColumns,
    EV_DB_PRICE_QUERY_RESULT_STRUCT *psResult
        )
{
    EV_CHARGING_DB_PRICE_FIELDS_ENUM eCurrentField =
        EV_CHARGING_DB_PRICE_BRAND;
    FUEL_STATION_ROW_STRUCT sStationRow;
    FUEL_REFUELING_TYPE_STRUCT asTypes[FUEL_REFUELING_TYPES_ARRAY_SIZE];
    FUEL_PRICE_ROW_STRUCT sPriceRow;
    FUEL_STATION_OBJECT hStation = FUEL_STATION_INVALID_OBJECT;
    OSAL_FIXED_OBJECT_DATA atFixedData[OSAL_FIXED_OBJECT_SIZE * 2];
    LOC_ID tLocId;
    BOOLEAN bOk,
            bPriceAccepted,
            bStationNeeded = TRUE,
            bStationInUse = FALSE;
    OSAL_FIXED_OBJECT hLat = OSAL_FIXED_INVALID_OBJECT,
                      hLon = OSAL_FIXED_INVALID_OBJECT;
    UN8 *pun8PositionFuelTypes = NULL, *pun8PositionsAvailable = NULL;

    // Verify input
    if (psResult == NULL)
    {
        return FALSE;
    }

    // If there is at least one data row and the correct
    // number of columns, then we have good results
    if (n32NumberOfColumns == EV_CHARGING_DB_PRICE_MAX_FIELDS )
    {
        psResult->bSuccess = TRUE;
    }
    else
    {
        psResult->bSuccess = FALSE;
        return FALSE;
    }

    // Initialize the row structures
    OSAL.bMemSet(&sStationRow, 0, sizeof(FUEL_STATION_ROW_STRUCT));
    OSAL.bMemSet(&sPriceRow, 0, sizeof(FUEL_PRICE_ROW_STRUCT));
    OSAL.bMemSet(&asTypes[0], 0,
        (sizeof(FUEL_REFUELING_TYPE_STRUCT) * FUEL_REFUELING_TYPES_ARRAY_SIZE));

    // First, filter by age
    sPriceRow.un32YoungestPriceAgeUTCSeconds = (UN32)
        psColumn[EV_CHARGING_DB_PRICE_AGE_EPOCH_SECONDS].uData.sUN32.un32Data;

    // Can we accept the age of this price data?
    bPriceAccepted = bPriceAgeFilter(
        sPriceRow.un32YoungestPriceAgeUTCSeconds,
        psResult->un32UTCsec );
    if (bPriceAccepted == FALSE)
    {
        // Stop processing this row -- try the next
        return TRUE;
    }

    // Populate the station row's region
    sStationRow.tRegion =
        psColumn[EV_CHARGING_DB_PRICE_REGION_ID].uData.sUN32.un32Data;

    // Get the station ID
    sStationRow.tStationId =
        psColumn[EV_CHARGING_DB_PRICE_STATION_ID].uData.sUN32.un32Data;

    // Get the lat/lon of this station
    sStationRow.n32Lat = (N32)
            psColumn[EV_CHARGING_DB_PRICE_LAT].uData.sUN32.un32Data;
    sStationRow.n32Lon = (N32)
            psColumn[EV_CHARGING_DB_PRICE_LON].uData.sUN32.un32Data;

    // Create some fixed objects on the stack
    hLat = OSAL_FIXED.hCreateInMemory(
        sStationRow.n32Lat, LOCATION_BINPOINT,
        &atFixedData[0]);
    hLon = OSAL_FIXED.hCreateInMemory(
        sStationRow.n32Lon, LOCATION_BINPOINT,
            &atFixedData[OSAL_FIXED_OBJECT_SIZE]);

    // Generate the loc id
    tLocId = FUEL_REG_STATID_TO_LOCID(
        sStationRow.tRegion, sStationRow.tStationId);

    // Now, filter by location
    bOk = GsFuelMgrIntf.bVerifyQueryResultLocation(
        psResult->psObj->hFuelService, tLocId, hLat, hLon);
    if (bOk == FALSE)
    {
        // We don't want this station...but keep looking
        return TRUE;
    }

    // Is this station already present in the cache?
    hStation = GsFuelMgrIntf.hGetStation(
        psResult->psObj->hFuelService,
        sStationRow.tRegion,
        sStationRow.tStationId,
        &bStationNeeded);

    if (bStationNeeded == FALSE)
    {
        // We don't want this station
        return TRUE;
    }

    if (hStation != FUEL_STATION_INVALID_OBJECT)
    {
        // Skip the row processor
        eCurrentField = EV_CHARGING_DB_PRICE_MAX_FIELDS;

        // This station is already in use in the cache
        bStationInUse = TRUE;
    }

    // Grab the entire price record
    do
    {
        switch (eCurrentField)
        {
            case EV_CHARGING_DB_PRICE_REGION_ID:
            case EV_CHARGING_DB_PRICE_STATION_ID:
            case EV_CHARGING_DB_PRICE_AGE_EPOCH_SECONDS:
            case EV_CHARGING_DB_PRICE_LAT:
            case EV_CHARGING_DB_PRICE_LON:
            {
                // Already done
            }
            break;

            case EV_CHARGING_DB_PRICE_BRAND:
            {
                sStationRow.hBrand = STRING.hCreate(
                    (const char *)psColumn[eCurrentField].uData.sCString.pcData,
                    strlen((const char *)psColumn[eCurrentField].uData.sCString.pcData));
                if (sStationRow.hBrand == STRING_INVALID_OBJECT)
                {
                    bOk = FALSE;
                }
            }
            break;

            case EV_CHARGING_DB_PRICE_LOGO:
            {
                sStationRow.n16LogoId = (N16)
                    psColumn[eCurrentField].uData.sUN32.un32Data;
            }
            break;

            case EV_CHARGING_DB_PRICE_NAME:
            {
                sStationRow.hName = STRING.hCreate(
                    (const char *)psColumn[eCurrentField].uData.sCString.pcData,
                    strlen((const char *)psColumn[eCurrentField].uData.sCString.pcData));
                if (sStationRow.hName == STRING_INVALID_OBJECT)
                {
                    bOk = FALSE;
                }
            }
            break;

            case EV_CHARGING_DB_PRICE_DESC:
            {
                sStationRow.hDesc = STRING.hCreate(
                    (const char *)psColumn[eCurrentField].uData.sCString.pcData,
                    strlen((const char *)psColumn[eCurrentField].uData.sCString.pcData));
            }
            break;

            case EV_CHARGING_DB_PRICE_ADDR:
            {
                sStationRow.hAddr = STRING.hCreate(
                    (const char *)psColumn[eCurrentField].uData.sCString.pcData,
                    strlen((const char *)psColumn[eCurrentField].uData.sCString.pcData));
                if (sStationRow.hAddr == STRING_INVALID_OBJECT)
                {
                    bOk = FALSE;
                }
            }
            break;

            case EV_CHARGING_DB_PRICE_CITY:
            {
                sStationRow.hCity = STRING.hCreate(
                    (const char *)psColumn[eCurrentField].uData.sCString.pcData,
                    strlen((const char *)psColumn[eCurrentField].uData.sCString.pcData));
            }
            break;

            case EV_CHARGING_DB_PRICE_STATE:
            {
                sStationRow.tStateID = (STATE_ID)
                    psColumn[eCurrentField].uData.sUN32.un32Data;
            }
            break;

            case EV_CHARGING_DB_PRICE_ZIP:
            {
                sStationRow.hZIP = STRING.hCreate(
                    (const char *)psColumn[eCurrentField].uData.sCString.pcData,
                    strlen((const char *)psColumn[eCurrentField].uData.sCString.pcData));
            }
            break;

            case EV_CHARGING_DB_PRICE_PHONE:
            {
                sStationRow.n64Phone =
                    psColumn[eCurrentField].n64NativeInteger;
            }
            break;

            case EV_CHARGING_DB_PRICE_CHARGE_TYPE_COUNT:
            {
                sStationRow.un8NumPermanentTypes = (UN8)
                    psColumn[eCurrentField].uData.sUN32.un32Data;
            }
            break;

            case EV_CHARGING_DB_PRICE_CHARGE_TYPES:
            {
                pun8PositionFuelTypes = (UN8 *)psColumn[eCurrentField].uData.sBLOB.pvData;

                if (pun8PositionFuelTypes == NULL)
                {
                    // This shouldn't happen
                    bOk = FALSE;
                }
            }
            break;

            case EV_CHARGING_DB_PRICE_CHARGE_NUMS:
            {
                pun8PositionsAvailable = (UN8 *)psColumn[eCurrentField].uData.sBLOB.pvData;

                if (pun8PositionsAvailable == NULL)
                {
                    // This shouldn't happen
                    bOk = FALSE;
                }
            }
            break;

            case EV_CHARGING_DB_PRICE_AMENITIES:
            {
                sStationRow.un32Amenities =
                    psColumn[eCurrentField].uData.sUN32.un32Data;
            }
            break;

            default: // Shouldn't happen
            break;
        }

        if (bOk == FALSE)
        {
            break;
        }
    } while (++eCurrentField < EV_CHARGING_DB_PRICE_MAX_FIELDS);

    if (bOk == TRUE)
    {
        UN8 un8Index;

        // Populate the refueling types structure
        for (un8Index = 0;
            un8Index < sStationRow.un8NumPermanentTypes;
            un8Index++)
        {
            // Process each entry now
            asTypes[un8Index].un8FuelType = pun8PositionFuelTypes[un8Index];
            asTypes[un8Index].eFuelType = eMatchFuelType(pun8PositionFuelTypes[un8Index]);
            asTypes[un8Index].un8NumPositions = pun8PositionsAvailable[un8Index];
            asTypes[un8Index].hLongFuelName = STRING_INVALID_OBJECT;
            asTypes[un8Index].hShortFuelName = STRING_INVALID_OBJECT;
        }

        // Provide our types array
        sStationRow.psTypes = &asTypes[0];

        if (hStation == FUEL_STATION_INVALID_OBJECT)
        {
            // Now, create a station object and take ownership
            // of the objects in the row
            hStation = GsFuelMgrIntf.hCreateStation(
                psResult->psObj->hFuelService, &sStationRow);
        }

        // Have the manager put this station into service
        bStationInUse |= GsFuelMgrIntf.bPutStationInService(
            psResult->psObj->hFuelService,
            hStation, FALSE);

        // Is the station in use now?
        if (bStationInUse == FALSE)
        {
            // Manager didn't want this station and
            // it isn't otherwise in use.
            // We don't need it anymore
            FUEL_STATION_vDestroy(hStation);
            hStation = FUEL_STATION_INVALID_OBJECT;
        }
    }

    if (bOk == FALSE)
    {
        SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
            EV_CHARGING_NAME": Unable to update station with price from DB.");

        if (hStation == FUEL_STATION_INVALID_OBJECT)
        {
            // Destroy the strings
            if (sStationRow.hAddr != STRING_INVALID_OBJECT)
            {
                STRING_vDestroy(sStationRow.hAddr);
                sStationRow.hAddr = STRING_INVALID_OBJECT;
            }

            if (sStationRow.hBrand != STRING_INVALID_OBJECT)
            {
                STRING_vDestroy(sStationRow.hBrand);
                sStationRow.hBrand = STRING_INVALID_OBJECT;
            }

            if (sStationRow.hCity != STRING_INVALID_OBJECT)
            {
                STRING_vDestroy(sStationRow.hCity);
                sStationRow.hCity = STRING_INVALID_OBJECT;
            }

            if (sStationRow.hDesc != STRING_INVALID_OBJECT)
            {
                STRING_vDestroy(sStationRow.hDesc);
                sStationRow.hDesc = STRING_INVALID_OBJECT;
            }

            if (sStationRow.hName != STRING_INVALID_OBJECT)
            {
                STRING_vDestroy(sStationRow.hName);
                sStationRow.hName = STRING_INVALID_OBJECT;
            }

            if (sStationRow.hZIP != STRING_INVALID_OBJECT)
            {
                STRING_vDestroy(sStationRow.hZIP);
                sStationRow.hZIP = STRING_INVALID_OBJECT;
            }
        }
        else if (bStationInUse == FALSE)
        {
            // Destroy the fuel station
            FUEL_STATION_vDestroy(hStation);
        }

        psResult->bSuccess = FALSE;
    }

    // Keep iterating
    return TRUE;
}
