/******************************************************************************/
/*                    Copyright (c) Sirius XM Radio, Inc.                     */
/*                            All Rights Reserved                             */
/*         Licensed Materials - Property of Sirius XM Radio, Inc.             */
/******************************************************************************/
/*******************************************************************************
 *
 * DESCRIPTION
 *
 *  This module contains the Object:TOPIC implementation for the
 *  Sirius Module Services (SMS)
 *
 ******************************************************************************/
#include <standard.h>
#include <string.h>

#include "osal.h"
#include "sms_api.h"
#include "sms_obj.h"
#include "string_obj.h"
#include "sms_api_debug.h"

#include "_epg_topic_obj.h"

/*****************************************************************************
 PUBLIC FUNCTIONS
 *****************************************************************************/
/*****************************************************************************
 *
 *   tTopicId
 *
 *****************************************************************************/
static TOPIC_ID tTopicId(TOPIC_OBJECT hEpgTopic)
{
    EPG_TOPIC_STRUCT *psEpgTopic = (EPG_TOPIC_STRUCT*) hEpgTopic;
    BOOLEAN bValid;

    // Verify SMS Object
    bValid = SMSO_bValid((SMS_OBJECT) hEpgTopic);
    if( bValid == FALSE )
    {
        return UN16_MAX;
    }

    return psEpgTopic->tTopicId;
}

/*****************************************************************************
 *
 *   hTopicName
 *
 *****************************************************************************/
static STRING_OBJECT hTopicName(TOPIC_OBJECT hEpgTopic)
{
    EPG_TOPIC_STRUCT *psEpgTopic = (EPG_TOPIC_STRUCT*) hEpgTopic;
    BOOLEAN bValid;

    // Verify SMS Object
    bValid = SMSO_bValid((SMS_OBJECT) hEpgTopic);
    if( bValid == FALSE )
    {
        return STRING_INVALID_OBJECT;
    }

    return psEpgTopic->hTopicName;
}

/*****************************************************************************
 *
 *   n32FPrintf
 *
 *****************************************************************************/
static N32 n32FPrintf(TOPIC_OBJECT hEpgTopic, FILE *psFile)
{
    BOOLEAN bValid;
    N32 n32Return = 0;
    EPG_TOPIC_STRUCT *psObj = (EPG_TOPIC_STRUCT *) hEpgTopic;

    bValid = SMSO_bValid((SMS_OBJECT) hEpgTopic);
    if (( bValid == FALSE ) || ( psFile == NULL ))
    {
        return EOF;
    }

    // Print EPG topic
    n32Return += fprintf(psFile,
                "\n\t\tEPG topic:\n\t\t\thEpgTopic = %p\n",
                psObj);

    n32Return += fprintf(psFile, "\t\t\tTopicId = %d", psObj->tTopicId);

    if( psObj->hTopicName != STRING_INVALID_OBJECT )
    {
        char acTmpStrBuf[EPG_TOPIC_NAME_LEN_MAX];
        size_t tStrLen = 0;

        tStrLen = STRING.tSize(psObj->hTopicName) + 1;
        STRING.tCopyToCStr(psObj->hTopicName, acTmpStrBuf, tStrLen);
        n32Return += fprintf(psFile, "\t\t\tTopic name: %s\n", acTmpStrBuf);
    }

    return n32Return;
}

/*****************************************************************************
 FRIEND FUNCTIONS
 *****************************************************************************/

/*****************************************************************************
 *
 *  EPG_TOPIC_psCreate
 *
 *****************************************************************************/
EPG_TOPIC_STRUCT *EPG_TOPIC_psCreate(SMS_OBJECT hParent,
                                     TOPIC_ID tTopicId,
                                     char *pcTopicName)
{
    EPG_TOPIC_STRUCT *psEpgTopic = NULL;

    do
    {
        psEpgTopic = (EPG_TOPIC_STRUCT *) SMSO_hCreate(EPG_TOPIC_OBJECT_NAME,
                                                       sizeof(EPG_TOPIC_STRUCT),
                                                       hParent,
                                                       FALSE);
        if ((SMS_OBJECT) psEpgTopic == SMS_INVALID_OBJECT)
        {
            SMSAPI_DEBUG_vPrint(EPG_TOPIC_DBG_PREFIX, 0,
                PC_RED"ERROR: Failed to create EPG Topic object.\n"PC_RESET);
            break;
        }

        if (pcTopicName != NULL)
        {
            psEpgTopic->hTopicName = STRING.hCreate(pcTopicName, 0);
            if (psEpgTopic->hTopicName == STRING_INVALID_OBJECT)
            {
                SMSAPI_DEBUG_vPrint(EPG_TOPIC_DBG_PREFIX, 0,
                    PC_RED"ERROR: Failed to create EPG Topic name.\n"PC_RESET);
                break;
            }
        }
        else
        {
            psEpgTopic->hTopicName = STRING_INVALID_OBJECT;
        }

        psEpgTopic->tTopicId = tTopicId;

        // Everything's OK
        return psEpgTopic;

    } while (FALSE);

    // Error case:
    EPG_TOPIC_vDestroy(psEpgTopic);
    return NULL;
}

/*****************************************************************************
 *
 *  EPG_TOPIC_vDestroy
 *
 *****************************************************************************/
void EPG_TOPIC_vDestroy(EPG_TOPIC_STRUCT *psEpgTopic)
{
    if (psEpgTopic == NULL)
    {
        return;
    }

    if (psEpgTopic->hTopicName != STRING_INVALID_OBJECT)
    {
        STRING.vDestroy(psEpgTopic->hTopicName);
    }

    SMSO_vDestroy((SMS_OBJECT) psEpgTopic);

    return;
}

/*****************************************************************************
 *
 *  EPG_TOPIC_n16CompareEpgTopicsByTid
 *
 *****************************************************************************/
N16 EPG_TOPIC_n16CompareEpgTopicsByTid(EPG_TOPIC_STRUCT *psTopic1,
                                       EPG_TOPIC_STRUCT *psTopic2)
{
    if (psTopic1->tTopicId < psTopic2->tTopicId)
    {
        return -1;
    }

    if (psTopic1->tTopicId > psTopic2->tTopicId)
    {
        return 1;
    }

    return 0;
}

/*****************************************************************************
 *
 *  EPG_TOPIC_n16CompareEpgTopicAndTid
 *
 *****************************************************************************/
N16 EPG_TOPIC_n16CompareEpgTopicAndTid(EPG_TOPIC_STRUCT *psTopic,
                                       TOPIC_ID *ptTopicId)
{
    if (psTopic->tTopicId < *ptTopicId)
    {
        return -1;
    }

    if (psTopic->tTopicId > *ptTopicId)
    {
        return 1;
    }

    return 0;
}

/*****************************************************************************
 PRIVATE FUNCTIONS
 *****************************************************************************/

/*****************************************************************************
 INLINE FUNCTIONS
 *****************************************************************************/
