/******************************************************************************/
/*                    Copyright (c) Sirius XM Radio, Inc.                     */
/*                            All Rights Reserved                             */
/*         Licensed Materials - Property of Sirius XM Radio, Inc.             */
/******************************************************************************/
/*******************************************************************************
 *
 * DESCRIPTION
 *
 *  This module contains the Object:EPG_TEXT_CACHE implementation.
 *
 ******************************************************************************/
#include <standard.h>
#include <string.h>

#include "osal.h"
#include "sms_version.h"
#include "sms_api.h"
#include "sms_obj.h"
#include "string_obj.h"
#include "sms_api_debug.h"
#include "zlib.h"

#include "epg_cache_obj.h"
#include "_epg_cache_obj.h"

/*****************************************************************************
 PUBLIC FUNCTIONS
 *****************************************************************************/

/*****************************************************************************
 FRIEND FUNCTIONS
 *****************************************************************************/

/*******************************************************************************
 *
 *  EPG_TEXT_CACHE_psCreate
 *
 *******************************************************************************/
EPG_TEXT_CACHE_STRUCT *EPG_TEXT_CACHE_psCreate(SMS_OBJECT hParent,
                                                UN8 un8NumSegments,
                                                UN8 un8NumCachedMax)
{
    EPG_TEXT_CACHE_STRUCT *psCacheObj = NULL;

    do
    {
        // Verify input
        if (un8NumSegments == 0 ||
            un8NumCachedMax == 0 ||
            un8NumCachedMax > un8NumSegments)
        {
            break;
        }

        // Create cache object
        psCacheObj =
            (EPG_TEXT_CACHE_STRUCT *) SMSO_hCreate(EPG_TEXT_CACHE_OBJECT_NAME,
                                                    sizeof(EPG_TEXT_CACHE_STRUCT),
                                                    hParent,
                                                    FALSE);
        if ((SMS_OBJECT) psCacheObj == SMS_INVALID_OBJECT)
        {
            SMSAPI_DEBUG_vPrint(EPG_CACHE_DBG_PREFIX, 0,
                PC_RED"ERROR: Failed to create Text Cache object.\n"PC_RESET);
            psCacheObj = NULL;
            break;
        }

        // Create text data array
        psCacheObj->asTextCacheData =
            (EPG_SEGMENT_TEXT_DATA_STRUCT *)
                SMSO_hCreate(EPG_TEXT_CACHE_OBJECT_NAME,
                             sizeof(EPG_SEGMENT_TEXT_DATA_STRUCT) * un8NumSegments,
                             (SMS_OBJECT) psCacheObj, FALSE);
        if ((SMS_OBJECT) psCacheObj->asTextCacheData == SMS_INVALID_OBJECT)
        {
            SMSAPI_DEBUG_vPrint(EPG_CACHE_DBG_PREFIX, 0,
                PC_RED"ERROR: Failed to create Text Cache object.\n"PC_RESET);
            psCacheObj->asTextCacheData = NULL;
            break;
        }

        psCacheObj->un8NumSegments = un8NumSegments;
        psCacheObj->un8NumCachedMax = un8NumCachedMax;

        // Everything's OK
        return psCacheObj;

    } while (FALSE);

    // Error case:
    EPG_TEXT_CACHE_vDestroy(psCacheObj);
    return NULL;
}

/*******************************************************************************
 *
 *  EPG_TEXT_CACHE_vDestroy
 *
 *******************************************************************************/
void EPG_TEXT_CACHE_vDestroy(EPG_TEXT_CACHE_STRUCT *psCacheObj)
{
    if (psCacheObj == NULL)
    {
        return;
    }

    if (psCacheObj->asTextCacheData != NULL)
    {
        SMSO_vDestroy((SMS_OBJECT) psCacheObj->asTextCacheData);
    }

    SMSO_vDestroy((SMS_OBJECT) psCacheObj);

    return;
}

/*******************************************************************************
 *
 *  EPG_TEXT_CACHE_bLoadCompressedTextFile
 *
 *******************************************************************************/
BOOLEAN EPG_TEXT_CACHE_bLoadCompressedTextFile(EPG_TEXT_CACHE_STRUCT *psCacheObj,
                                               UN8 un8SegNum,
                                               FILE *pFile,
                                               UN32 un32CompressedSize,
                                               UN32 un32DecompressedSize)
{
    BOOLEAN bResult = FALSE;

    do
    {
        // Check if segment is already cached
        bResult = EPG_TEXT_CACHE_bIsSegmentCached(psCacheObj, un8SegNum);
        if (bResult == TRUE)
        {
            EPG_TEXT_CACHE_vCleanSegment(psCacheObj, un8SegNum);
        }

        // Check if cache is full
        if (psCacheObj->un8NumCached == psCacheObj->un8NumCachedMax)
        {
            // Cleanup segment with the highest index
            UN8 un8Index = psCacheObj->un8NumSegments;
            SMSAPI_DEBUG_vPrint(EPG_CACHE_DBG_PREFIX, 4,
                PC_GREY"Cache is full. Total cached = %u.\n"PC_RESET,
                psCacheObj->un8NumCached);
            while (un8Index > 0)
            {
                un8Index--;
                bResult = EPG_TEXT_CACHE_bIsSegmentCached(psCacheObj, un8Index);
                if (bResult == TRUE)
                {
                    SMSAPI_DEBUG_vPrint(EPG_CACHE_DBG_PREFIX, 4,
                        PC_GREY"Cleanup cache segment %u.\n"PC_RESET,
                        un8Index);
                    EPG_TEXT_CACHE_vCleanSegment(psCacheObj, un8Index);
                    un8Index = 0;
                }
            }
        }

        // Load segment cache
        bResult = EPG_TEXT_CACHE_bLoadCompressedFile(psCacheObj,
                                                     un8SegNum,
                                                     pFile,
                                                     un32CompressedSize,
                                                     un32DecompressedSize);
        if (bResult == FALSE)
        {
            break;
        }

    } while (FALSE);

    return bResult;
}

/*******************************************************************************
 *
 *  EPG_TEXT_CACHE_bGetString
 *
 *******************************************************************************/
BOOLEAN EPG_TEXT_CACHE_bGetString(EPG_TEXT_CACHE_STRUCT *psCacheObj,
                                  UN8 un8SegNum,
                                  UN32 un32StringIndex,
                                  STRING_OBJECT hResultString)
{
    EPG_SEGMENT_TEXT_DATA_STRUCT *psTextData = &(psCacheObj->asTextCacheData[un8SegNum]);
    char *pcString = "";
    BOOLEAN bResult = FALSE;

    do
    {
        if (psTextData->pcDecompressedText == NULL ||
            psTextData->asTextIndex == NULL)
        {
            break;
        }

        if (un32StringIndex >= psTextData->un16NumStrings)
        {
            break;
        }

        pcString = psTextData->pcDecompressedText +
                psTextData->asTextIndex[un32StringIndex].un32Offset;

        STRING.bModifyCStr(hResultString, pcString);

        bResult = TRUE;

    } while (FALSE);

    return bResult;
}

/*******************************************************************************
 *
 *  EPG_TEXT_CACHE_vCleanSegment
 *
 *******************************************************************************/
void EPG_TEXT_CACHE_vCleanSegment(EPG_TEXT_CACHE_STRUCT *psCacheObj,
                                  UN8 un8SegNum)
{
    EPG_SEGMENT_TEXT_DATA_STRUCT *psTextData = &(psCacheObj->asTextCacheData[un8SegNum]);

    if (psTextData->pcDecompressedText != NULL)
    {
        SMSO_vDestroy((SMS_OBJECT) psTextData->pcDecompressedText);
        psTextData->pcDecompressedText = NULL;

        if (psTextData->asTextIndex != NULL)
        {
            SMSO_vDestroy((SMS_OBJECT) psTextData->asTextIndex);
            psTextData->asTextIndex = NULL;
        }
        else
        {
            SMSAPI_DEBUG_vPrint(EPG_CACHE_DBG_PREFIX, 0,
                PC_RED"ASSERT: Cache indexes don't exist!\n"PC_RESET);
        }

        if (psCacheObj->un8NumCached > 0)
        {
            psCacheObj->un8NumCached--;
        }
        else
        {
            SMSAPI_DEBUG_vPrint(EPG_CACHE_DBG_PREFIX, 0,
                PC_RED"ASSERT: Number of cached segments = 0!\n"PC_RESET);
        }
    }

    return;
}

/*******************************************************************************
 *
 *  EPG_TEXT_CACHE_vCleanWholeCache
 *
 *******************************************************************************/
void EPG_TEXT_CACHE_vCleanWholeCache(EPG_TEXT_CACHE_STRUCT *psCacheObj)
{
    UN8 un8SegNum = 0;

    while (un8SegNum < psCacheObj->un8NumSegments)
    {
        EPG_TEXT_CACHE_vCleanSegment(psCacheObj, un8SegNum);
        un8SegNum++;
    }

    return;
}

/*******************************************************************************
 *
 *  EPG_TEXT_CACHE_bIsSegmentCached
 *
 *******************************************************************************/
BOOLEAN EPG_TEXT_CACHE_bIsSegmentCached(EPG_TEXT_CACHE_STRUCT *psCacheObj,
                                        UN8 un8SegNum)
{
    BOOLEAN bResult = FALSE;

    if (psCacheObj->asTextCacheData[un8SegNum].pcDecompressedText != NULL )
    {
        bResult = TRUE;
    }

    return bResult;
}

/*****************************************************************************
 PRIVATE FUNCTIONS
 *****************************************************************************/

/*******************************************************************************
 *
 *  EPG_TEXT_CACHE_bLoadCompressedTextData
 *
 *******************************************************************************/
BOOLEAN EPG_TEXT_CACHE_bLoadCompressedTextData(EPG_TEXT_CACHE_STRUCT *psCacheObj,
                                               UN8 un8SegNum,
                                               void *pvCompressedData,
                                               UN32 un32CompressedSize,
                                               UN32 un32DecompressedSize)
{
    EPG_SEGMENT_TEXT_DATA_STRUCT *psTextData =
                    &(psCacheObj->asTextCacheData[un8SegNum]);
    UN32 un32ActualDecompressedSize = un32DecompressedSize;
    size_t tOffset = 0;
    size_t tHead = 0;
    UN16 un16StringsCount = 0;
    UN16 un16Index = 0;
    int iRet = 0;

    do
    {
        if (psTextData->pcDecompressedText != NULL ||
            psTextData->asTextIndex != NULL)
        {
            SMSAPI_DEBUG_vPrint(EPG_CACHE_DBG_PREFIX, 0,
                PC_RED"ASSERT: Try to overwrite segment %u cache!\n"PC_RESET,
                un8SegNum);
            break;
        }

        // Create buffer to save decompressed text
        psTextData->pcDecompressedText =
            (char*) SMSO_hCreate(EPG_DECOMPRESSED_TEXT_BUF_OBJECT_NAME,
                                 un32DecompressedSize + 1, // + '\0',
                                 (SMS_OBJECT) psCacheObj, FALSE);
        if ((SMS_OBJECT) psTextData->pcDecompressedText == SMS_INVALID_OBJECT)
        {
            SMSAPI_DEBUG_vPrint(EPG_CACHE_DBG_PREFIX, 0,
                PC_RED"ERROR: Failed to allocate decompressed text buffer.\n"
                PC_RESET);
            psTextData->pcDecompressedText = NULL;
            break;
        }

        // Decompress
        iRet = uncompress_m((Bytef*)psTextData->pcDecompressedText,
                            (uLongf*)&un32ActualDecompressedSize,
                            (const Bytef*)pvCompressedData,
                            (uLong)un32CompressedSize);

        // Add NULL terminator after the last string
        psTextData->pcDecompressedText[un32ActualDecompressedSize] = '\0';
        psTextData->un32DecompressedFileSize = un32ActualDecompressedSize + 1;

        if (iRet != Z_OK && iRet != Z_BUF_ERROR)
        {
            SMSAPI_DEBUG_vPrint(EPG_CACHE_DBG_PREFIX, 0,
                PC_RED"ERROR: Decompression failed!\n"PC_RESET);
            break;
        }

        // Count number of strings in decompressed data buffer
        while (tOffset < psTextData->un32DecompressedFileSize)
        {
            if (psTextData->pcDecompressedText[tOffset] == '\0')
            {
                un16StringsCount++;
            }

            tOffset++;
        }

        if (un16StringsCount == 0)
        {
            break;
        }

        // Create Cache indexes array
        psTextData->asTextIndex =
            (EPG_TEXT_INDEX_ITEM_STRUCT *)
                SMSO_hCreate(EPG_CACHE_INDEX_ARRAY_OBJECT_NAME,
                             sizeof(EPG_TEXT_INDEX_ITEM_STRUCT) * un16StringsCount,
                             (SMS_OBJECT) psCacheObj, FALSE);
        if ((SMS_OBJECT) psTextData->asTextIndex == SMS_INVALID_OBJECT)
        {
            SMSAPI_DEBUG_vPrint(EPG_CACHE_DBG_PREFIX, 0,
                PC_RED"ERROR: Failed to allocate text index array.\n"PC_RESET);
            psTextData->asTextIndex = NULL;
            break;
        }

        // Save strings count
        psTextData->un16NumStrings = un16StringsCount;

        // Fill indexes
        tOffset = 0;
        while (un16Index < un16StringsCount)
        {
            if (psTextData->pcDecompressedText[tOffset] == '\0')
            {
                psTextData->asTextIndex[un16Index].un32Offset = tHead;
                psTextData->asTextIndex[un16Index].un32Size = tOffset - tHead;
                un16Index++;
                tHead = tOffset + 1; // Skip '\0'
            }

            tOffset++;
        };

        // Everything's OK
        psCacheObj->un8NumCached++;
        return TRUE;

    } while (FALSE);

    // Error case:
    EPG_TEXT_CACHE_vCleanSegment(psCacheObj, un8SegNum);
    return FALSE;
}

/*******************************************************************************
 *
 *  EPG_TEXT_CACHE_bLoadCompressedFile
 *
 *******************************************************************************/
BOOLEAN EPG_TEXT_CACHE_bLoadCompressedFile(EPG_TEXT_CACHE_STRUCT *psCacheObj,
                                           UN8 un8SegNum,
                                           FILE *pFile,
                                           UN32 un32CompressedSize,
                                           UN32 un32DecompressedSize)
{
    size_t tElementsRead = 0;
    UN8 *pun8CompressedText = NULL;
    BOOLEAN bResult = FALSE;

    do
    {
        // Create temporary buffer to read compressed text
        pun8CompressedText =
            (UN8*) SMSO_hCreate(EPG_COMPRESSED_TEXT_BUF_OBJECT_NAME,
                                un32CompressedSize,
                                SMS_INVALID_OBJECT,
                                FALSE);
        if ((SMS_OBJECT) pun8CompressedText == SMS_INVALID_OBJECT)
        {
            SMSAPI_DEBUG_vPrint(EPG_CACHE_DBG_PREFIX, 0,
                PC_RED"ERROR: Failed to allocate CT buffer.\n"PC_RESET);
            pun8CompressedText = NULL;
            break;
        }

        // Read compressed text
        tElementsRead = fread(pun8CompressedText,
                              un32CompressedSize,
                              1,
                              pFile);
        if (tElementsRead != 1)
        {
            SMSAPI_DEBUG_vPrint(EPG_CACHE_DBG_PREFIX, 0,
                PC_RED"ERROR: Failed to read CT data from file.\n"PC_RESET);
            break;
        }

        // Load data
        bResult = EPG_TEXT_CACHE_bLoadCompressedTextData(psCacheObj,
                                                         un8SegNum,
                                                         pun8CompressedText,
                                                         un32CompressedSize,
                                                         un32DecompressedSize);
        if (bResult == FALSE)
        {
            SMSAPI_DEBUG_vPrint(EPG_CACHE_DBG_PREFIX, 0,
                PC_RED"ERROR: Failed to load CT data.\n"PC_RESET);
            break;
        }

    } while (FALSE);

    if (pun8CompressedText != NULL)
    {
        SMSO_vDestroy((SMS_OBJECT) pun8CompressedText);
    }

    return bResult;
}

/*****************************************************************************
 INLINE FUNCTIONS
 *****************************************************************************/
