/******************************************************************************/
/*                    Copyright (c) Sirius XM Radio, Inc.                     */
/*                            All Rights Reserved                             */
/*         Licensed Materials - Property of Sirius XM Radio, Inc.             */
/******************************************************************************/
/*******************************************************************************
 *
 * DESCRIPTION
 *
 *  This module contains the Object:Engineering Data implementation for the
 *  Sirius Module Services (SMS)
 *
 ******************************************************************************/
#include "sms_api.h"
#include "decoder_obj.h"
#include "module_obj.h"
#include "_engineering_data_obj.h"
#include "engineering_data_obj.h"
#include "detailed_signal_quality_obj.h"
#include "detailed_overlay_signal_quality_obj.h"
#include "link_status_information_obj.h"

#include "sms_api_debug.h"
static const char *gpacThisFile = __FILE__;

/*****************************************************************************
                             PUBLIC FUNCTIONS
*****************************************************************************/

/*****************************************************************************
*
*   tEventMask
*
*   Where:
*       hDecoder - a valid handle to an exisiting DECODER_OBJECT
*
*   Return:
*
*****************************************************************************/
static ENGINEERING_DATA_EVENT_MASK tEventMask (
    DECODER_OBJECT hDecoder
        )
{
    ENGINEERING_DATA_OBJECT_STRUCT *psObj;
    ENGINEERING_DATA_EVENT_MASK tEventMask =
            ENGINEERING_DATA_OBJECT_EVENT_NONE;
    SMSAPI_RETURN_CODE_ENUM eReturnCode;

    eReturnCode = eAccessObjectFromDecoder(hDecoder, &psObj);

    if ( eReturnCode == SMSAPI_RETURN_CODE_SUCCESS )
    {
        tEventMask = SMSU_tMask(&psObj->sEvent);
    }
    return tEventMask;
}


/*****************************************************************************
*
*   hSignalQualityDetailed
*
*   Where:
*       hDecoder - a valid handle to an exisiting DECODER_OBJECT
*
*   Return:
*
*****************************************************************************/
static DETAILED_SIGNAL_QUALITY_OBJECT hSignalQualityDetailed (
    DECODER_OBJECT hDecoder
        )
{
    ENGINEERING_DATA_OBJECT_STRUCT *psObj;
    DETAILED_SIGNAL_QUALITY_OBJECT hSignalQualityDetailed =
            DETAILED_SIGNAL_QUALITY_INVALID_OBJECT;
    SMSAPI_RETURN_CODE_ENUM eReturnCode;

    eReturnCode = eAccessObjectFromDecoder(hDecoder, &psObj);

    if ( eReturnCode == SMSAPI_RETURN_CODE_SUCCESS )
    {
        hSignalQualityDetailed = psObj->hSignalQualityDetailed;
    }

    return hSignalQualityDetailed;
}

/*****************************************************************************
*
*   hOverlaySignalQualityDetailed
*
*   Where:
*       hDecoder - a valid handle to an existing DECODER_OBJECT
*
*   Return:
*
*****************************************************************************/
static DETAILED_OVERLAY_SIGNAL_QUALITY_OBJECT hOverlaySignalQualityDetailed (
		DECODER_OBJECT hDecoder
			)
{
    ENGINEERING_DATA_OBJECT_STRUCT *psObj;
    DETAILED_OVERLAY_SIGNAL_QUALITY_OBJECT hOverlaySignalQualityDetailed =
            DETAILED_OVERLAY_SIGNAL_QUALITY_INVALID_OBJECT;
    SMSAPI_RETURN_CODE_ENUM eReturnCode;

    eReturnCode = eAccessObjectFromDecoder(hDecoder, &psObj);

    if ( eReturnCode == SMSAPI_RETURN_CODE_SUCCESS )
    {
        hOverlaySignalQualityDetailed = psObj->hOverlaySignalQualityDetailed;
    }

    return hOverlaySignalQualityDetailed;
}

/*****************************************************************************
*
*   eAudioDecoderBitrate
*
*   Where:
*       hDecoder - a valid handle to an existing DECODER_OBJECT
*
*   Return:
*
*****************************************************************************/
static SMSAPI_RETURN_CODE_ENUM eAudioDecoderBitrate (
    DECODER_OBJECT hDecoder,
    AUDIO_DECODER_BITRATE *ptBitrate
        )
{
    ENGINEERING_DATA_OBJECT_STRUCT *psObj;
    SMSAPI_RETURN_CODE_ENUM eReturnCode;

    eReturnCode = eAccessObjectFromDecoder(hDecoder, &psObj);

    if ( eReturnCode == SMSAPI_RETURN_CODE_SUCCESS )
    {
        *ptBitrate = psObj->tBitrate;
    }
    else
    {
        *ptBitrate = 0;
    }

    return eReturnCode;
}

/*****************************************************************************
*
*   hGetLinkStatusInformation
*
*   Where:
*       hDecoder - a valid handle to an existing DECODER_OBJECT
*
*   Return:
*
*****************************************************************************/
static LINK_STATUS_INFORMATION_OBJECT hLinkStatusInformation (
    DECODER_OBJECT hDecoder
        )
{
    ENGINEERING_DATA_OBJECT_STRUCT *psObj;
    LINK_STATUS_INFORMATION_OBJECT hLinkStatusInformation =
            LINK_STATUS_INFORMATION_INVALID_OBJECT;
    SMSAPI_RETURN_CODE_ENUM eReturnCode;

    eReturnCode = eAccessObjectFromDecoder(hDecoder, &psObj);

    if ( eReturnCode == SMSAPI_RETURN_CODE_SUCCESS )
    {
        hLinkStatusInformation = psObj->hLinkStatusInformation;
    }

    return hLinkStatusInformation;
}


/*****************************************************************************
*
*   n32FPrintf
*
*   Where:
*       hDecoder - a valid handle to an existing DECODER_OBJECT
*
*   Return:
*
*****************************************************************************/
static N32 n32FPrintf (
    DECODER_OBJECT hDecoder,
    FILE *psFile
        )
{
    N32 n32Return = 0;
    SMSAPI_RETURN_CODE_ENUM eReturnCode;
    AUDIO_DECODER_BITRATE tBitrate;

    eReturnCode = ENGINEERING_DATA.eAudioDecoderBitrate(hDecoder, &tBitrate);
    if (( eReturnCode != SMSAPI_RETURN_CODE_SUCCESS ) || ( psFile == NULL ))
    {
        return EOF;
    }

    n32Return += fprintf(psFile, "ENGINEERING_DATA_OBJECT\n");
    n32Return += fprintf(psFile,
                        "Audio Decoder Bitrate = %u\n",
                        tBitrate);

    return n32Return;
}


/*****************************************************************************
                             FRIEND FUNCTIONS
*****************************************************************************/

/*****************************************************************************
*
*   ENGINEERING_DATA_vInitializeObject
*
*****************************************************************************/
void ENGINEERING_DATA_vInitializeObject(
    ENGINEERING_DATA_OBJECT hEngineeringData
        )
{
    ENGINEERING_DATA_OBJECT_STRUCT *psObj;

    psObj = (ENGINEERING_DATA_OBJECT_STRUCT *)hEngineeringData;

    psObj->tBitrate = 0;
    LINK_STATUS_INFORMATION_vInitializeObject(
                                        psObj->hLinkStatusInformation);
    DETAILED_OVERLAY_SIGNAL_QUALITY_vInitializeObject(
                                        psObj->hOverlaySignalQualityDetailed);
    DETAILED_SIGNAL_QUALITY_vInitializeObject(
                                        psObj->hSignalQualityDetailed);
    return;
}
/*****************************************************************************
*
*   ENGINEERING_DATA_bUpdateLinkStatusInformation
*
*****************************************************************************/
BOOLEAN ENGINEERING_DATA_bUpdateLinkStatusInformation(
    DECODER_OBJECT hDecoder,
    LINK_INFORMATION_STRUCT const *psNewLinkStatusInformation
        )
{
    BOOLEAN bReturn = FALSE;
    ENGINEERING_DATA_OBJECT_STRUCT *psObj;
    SMSAPI_RETURN_CODE_ENUM eReturnCode;

    eReturnCode = eAccessObjectFromDecoder(hDecoder, &psObj);

	if ( eReturnCode == SMSAPI_RETURN_CODE_SUCCESS )
    {
        bReturn =
            LINK_STATUS_INFORMATION_bUpdateLinkStatusInformation(
                hDecoder,
                psNewLinkStatusInformation
                    );

        if ( bReturn == TRUE )
        {
            //update event mask
            SMSU_tUpdate(&psObj->sEvent,
                         ENGINEERING_DATA_OBJECT_EVENT_LINK_STATUS);
        }
        else
        {
            puts( ENGINEERING_DATA_OBJECT_NAME
                ": No change in link status information\n");
        }
    }
    else
    {
        SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    ENGINEERING_DATA_OBJECT_NAME
                    ": Unable to determine Engineering Data Object.");
    }
    return bReturn;
}

/*****************************************************************************
*
*   ENGINEERING_DATA_bUpdateAudioDecoderBitrate
*
*****************************************************************************/
BOOLEAN ENGINEERING_DATA_bUpdateAudioDecoderBitrate(
    DECODER_OBJECT hDecoder,
    AUDIO_DECODER_BITRATE tNewAudioDecoderBitrate
        )
{
    ENGINEERING_DATA_OBJECT_STRUCT *psObj;
	BOOLEAN bReturn = FALSE;
    SMSAPI_RETURN_CODE_ENUM eReturnCode;

    eReturnCode = eAccessObjectFromDecoder(hDecoder, &psObj);
	if ( eReturnCode == SMSAPI_RETURN_CODE_SUCCESS )
    {
        if ( psObj->tBitrate != tNewAudioDecoderBitrate )
        {
            bReturn = TRUE;
            psObj->tBitrate = tNewAudioDecoderBitrate;

            //update event mask
            SMSU_tUpdate(&psObj->sEvent,
                    ENGINEERING_DATA_OBJECT_EVENT_AUDIO_DECODER_BITRATE);
        }
    }

    return bReturn;
}

/*****************************************************************************
*
*   ENGINEERING_DATA_vUpdateNewDetailedSignalQuality
*
*****************************************************************************/
BOOLEAN ENGINEERING_DATA_bUpdateNewDetailedSignalQuality(
    DECODER_OBJECT hDecoder,
    DETAILED_SIGNAL_QUALITY_STRUCT const *psNewDetailedSignalQuality
        )
{
    BOOLEAN bReturn = FALSE;
    ENGINEERING_DATA_OBJECT_STRUCT *psObj;
    SMSAPI_RETURN_CODE_ENUM eReturnCode;

    eReturnCode = eAccessObjectFromDecoder(hDecoder, &psObj);

	if ( eReturnCode == SMSAPI_RETURN_CODE_SUCCESS )
    {
        bReturn =
            DETAILED_SIGNAL_QUALITY_bUpdateDetailedSignalQuality(
                hDecoder,
                psNewDetailedSignalQuality
                    );

        if (bReturn == TRUE)
        {
            //update event mask
            SMSU_tUpdate(&psObj->sEvent,
                    ENGINEERING_DATA_OBJECT_EVENT_SIGNAL_DETAILED);
        }
        else
        {
            puts(ENGINEERING_DATA_OBJECT_NAME
            ": No change in detailed signal quality\n");
        }
    }
    else
    {
        SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile,__LINE__,
                ENGINEERING_DATA_OBJECT_NAME
                ": Unable to determine Engineering Data Object.");
    }
    return bReturn;
}

/*****************************************************************************
*
*   ENGINEERING_DATA_bUpdateNewDetailedOverlaySignalQuality
*
*****************************************************************************/
BOOLEAN ENGINEERING_DATA_bUpdateNewDetailedOverlaySignalQuality(
    DECODER_OBJECT hDecoder,
    DETAILED_OVERLAY_SIGNAL_QUALITY_STRUCT const *psNewDetailedOverlaySignalQuality
        )
{
    BOOLEAN bReturn = FALSE;
    ENGINEERING_DATA_OBJECT_STRUCT *psObj;
    SMSAPI_RETURN_CODE_ENUM eReturnCode;

    eReturnCode = eAccessObjectFromDecoder(hDecoder, &psObj);

	if ( eReturnCode == SMSAPI_RETURN_CODE_SUCCESS )
    {
        bReturn =
            DETAILED_OVERLAY_SIGNAL_QUALITY_bUpdateDetailedOverlaySignalQuality(
                hDecoder,
                psNewDetailedOverlaySignalQuality
                    );

        if ( bReturn == TRUE )
        {
            //update event mask
            SMSU_tUpdate(&psObj->sEvent,
                ENGINEERING_DATA_OBJECT_EVENT_OVERLAY_SIGNAL_DETAILED);
        }
        else
        {
            puts(ENGINEERING_DATA_OBJECT_NAME
            ": No change in detailed overlay signal quality\n");
        }
    }
    else
    {
        SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                ENGINEERING_DATA_OBJECT_NAME
                ": Unable to determine Engineering Data Object.");
    }
    return bReturn;
}

/*****************************************************************************
*
*   ENGINEERING_DATA_hCreate
*
*****************************************************************************/
ENGINEERING_DATA_OBJECT ENGINEERING_DATA_hCreate(
    DECODER_OBJECT hOwner
        )
{
    ENGINEERING_DATA_OBJECT_STRUCT *psObj;
    ENGINEERING_DATA_OBJECT hObj;
    MODULE_OBJECT hModule;
    BOOLEAN bOverlaySupported;
    BOOLEAN bOwner;

    // Verify ownership of parent object
    bOwner = SMSO_bOwner((SMS_OBJECT)hOwner);

    if (bOwner == FALSE)
    {
        return ENGINEERING_DATA_INVALID_OBJECT;
    }

    // Create an instance of this object
    psObj = (ENGINEERING_DATA_OBJECT_STRUCT *)
                SMSO_hCreate( ENGINEERING_DATA_OBJECT_NAME,
                              sizeof(ENGINEERING_DATA_OBJECT_STRUCT),
                             (SMS_OBJECT)hOwner, FALSE);

    if( psObj == NULL )
    {
        // Error!
        return ENGINEERING_DATA_INVALID_OBJECT;
    }

    // Initialize state attributes
    *psObj = gsObjectDefaults;

    // Initialize asynchronous update configuration and
    // save inputs
    SMSU_vInitialize(
                    &psObj->sEvent,
                    psObj,
                    ENGINEERING_DATA_OBJECT_EVENT_NONE,
                    ENGINEERING_DATA_OBJECT_EVENT_NONE,
                    (SMSAPI_OBJECT_EVENT_CALLBACK)NULL,
                    NULL);

    // Create the sub Objects
    hObj = (ENGINEERING_DATA_OBJECT)psObj;
    psObj->hLinkStatusInformation =
                    LINK_STATUS_INFORMATION_hCreate(
                        hObj);

    //figure out if we should create he overlay object
    hModule = DECODER_hModule((DECODER_OBJECT) hOwner);
    bOverlaySupported = MODULE_bOverlaySupported(hModule);
    if (bOverlaySupported == TRUE)
    {
        psObj->hOverlaySignalQualityDetailed =
                        DETAILED_OVERLAY_SIGNAL_QUALITY_hCreate(
                            hObj);
    }
    psObj->hSignalQualityDetailed =
                    DETAILED_SIGNAL_QUALITY_hCreate(
                        hObj);

    if (  ( psObj->hLinkStatusInformation
            == LINK_STATUS_INFORMATION_INVALID_OBJECT
          )
       || (( psObj->hOverlaySignalQualityDetailed
            == DETAILED_OVERLAY_SIGNAL_QUALITY_INVALID_OBJECT
          ) && (bOverlaySupported == TRUE))
       || ( psObj->hSignalQualityDetailed
            == DETAILED_SIGNAL_QUALITY_INVALID_OBJECT
          )
       )
    {
        SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                         ENGINEERING_DATA_OBJECT_NAME
                         ": Unable to create Engineering Data child Objects.");
        ENGINEERING_DATA_vDestroy((ENGINEERING_DATA_OBJECT) psObj);
        return ENGINEERING_DATA_INVALID_OBJECT;
    }

    return (ENGINEERING_DATA_OBJECT)psObj;
}

/*****************************************************************************
*
*   ENGINEERING_DATA_vDestroy
*
*****************************************************************************/
void ENGINEERING_DATA_vDestroy (
    ENGINEERING_DATA_OBJECT hEngineeringData
        )
{
    ENGINEERING_DATA_OBJECT_STRUCT *psObj;
    BOOLEAN bOwner;

    psObj = (ENGINEERING_DATA_OBJECT_STRUCT *)hEngineeringData;
    // Verify caller is already the owner of the object.
    // If not this function may not be called.
    bOwner = SMSO_bOwner( (SMS_OBJECT)hEngineeringData );
    if( bOwner == TRUE )
    {
        //Destroy child objects
        DETAILED_OVERLAY_SIGNAL_QUALITY_vDestroy(
                        psObj->hOverlaySignalQualityDetailed);
        DETAILED_SIGNAL_QUALITY_vDestroy(psObj->hSignalQualityDetailed);
        LINK_STATUS_INFORMATION_vDestroy(psObj->hLinkStatusInformation);
        SMSU_vDestroy(&psObj->sEvent);

        psObj->hOverlaySignalQualityDetailed =
                        DETAILED_OVERLAY_SIGNAL_QUALITY_INVALID_OBJECT;
        psObj->hSignalQualityDetailed =
                        DETAILED_SIGNAL_QUALITY_INVALID_OBJECT;
        psObj->hLinkStatusInformation =
                        LINK_STATUS_INFORMATION_INVALID_OBJECT;

        // Free object instance
        SMSO_vDestroy((SMS_OBJECT)hEngineeringData);
    }
    return;
}

/*****************************************************************************
                             PRIVATE FUNCTIONS
*****************************************************************************/

/*****************************************************************************
*
*   eAccessObjectFromDecoder
*
*****************************************************************************/
static SMSAPI_RETURN_CODE_ENUM eAccessObjectFromDecoder(
    DECODER_OBJECT hDecoder,
    ENGINEERING_DATA_OBJECT_STRUCT **ppsObj
        )
{
    BOOLEAN bOwner;
    SMSAPI_RETURN_CODE_ENUM eReturnCode;

    // Verify caller is already the owner of the object.
    // If not this function may not be called.
    bOwner = SMSO_bOwner((SMS_OBJECT)hDecoder);
    if( bOwner == TRUE )
    {
        do
        {
            // Retrieve the DECODER's ENGINEERING_DATA_OBJECT instance
            // (if one exists)
            *ppsObj = (ENGINEERING_DATA_OBJECT_STRUCT *)
                        DECODER_hGetEngineeringData(hDecoder);
            if( (ENGINEERING_DATA_OBJECT)*ppsObj
                 == ENGINEERING_DATA_INVALID_OBJECT
              )
            {
                eReturnCode = SMSAPI_RETURN_CODE_NOT_FOUND;
                break;
            }

            // If we made it this far all is well
            eReturnCode = SMSAPI_RETURN_CODE_SUCCESS;

        } while(0);
    }
    else
    {
        eReturnCode = SMSAPI_RETURN_CODE_NOT_OWNER;
    }
    return eReturnCode;
}

