/******************************************************************************/
/*                    Copyright (c) Sirius XM Radio, Inc.                     */
/*                            All Rights Reserved                             */
/*         Licensed Materials - Property of Sirius XM Radio, Inc.             */
/******************************************************************************/
/*******************************************************************************
 *
 * DESCRIPTION
 *
 *  This module contains the Object:DSRL implementation for the
 *  Sirius Module Services (SMS)
 *
 ******************************************************************************/

#include "standard.h"
#include "osal.h"

#include "sms_api.h"
#include "sms_obj.h"

#include "dataservice_mgr_obj.h"
#include "dsrl_target_obj.h"
#include "dsrl_entry_obj.h"
#include "location_obj.h"

#include "dsrl_obj.h"
#include "_dsrl_obj.h"

#include "sms_api_debug.h"
static const char *gpacThisFile = __FILE__;

/*****************************************************************************
                             PUBLIC FUNCTIONS
*****************************************************************************/

/*****************************************************************************
*
*   hCreate
*
* This object interface method is a function that is used to create a
* DSRL object for a specific data service
*
* Inputs:
*
*   hDataServiceMgr - The Data Service object this object is being used with
*   phTargetList - A pointer to an array of DSRL_TARGET_OBJECTs that will
*       serve as the target critera for this DSRL
*   tNumTargets - The number of elements in the array pointed to by phTargetList
*   tCapacity - The max number of items that can be contained in this DSRL
*   n16Sort - A user-supplied sort function that will override the
*       service-specific sort function.  A value of NULL would indicate that
*       the caller wants to reply on the service-specific sort function
*   pvSortArgs - Arguments that will be present in the sort function.
*   bFilterItem - A user-supplied function that can be used to filter items
*       from the DSRL.  A value of NULL indicates that this DSRL does not
*       need a filter function.
*   pvFilterArgs - Arguments that will be present in the filter function.
*   vCallback - The callback function that is called when the DSRL wants to
*               notify of any changes to the contents.
*   pvCallbackArgs - Arguments that will be present in the callback function.
*
* Outputs:
*
*   A valid DSRL on success, a DSRL_INVALID_OBJECT on error.
*****************************************************************************/
static DSRL_OBJECT hCreate (
    DATASERVICE_MGR_OBJECT hManager,
    DSRL_TARGET_OBJECT *phTargetList,
    size_t tNumTargets,
    size_t tCapacity,
    DSRL_SORT_FUNCTION n16Sort,
    void *pvSortArgs,
    DSRL_FILTER_FUNCTION bFilterItem,
    void *pvFilterArgs,
    DSRL_CALLBACK vCallback,
    void *pvCallbackArgs
        )
{
    BOOLEAN bSuccess;
    DSRL_OBJECT_STRUCT *psObj;

    // If the capacity or the number of targets is zero, error out.
    if ((tCapacity == 0) || (tNumTargets == 0))
    {
        return DSRL_INVALID_OBJECT;
    }

    // If the target list is null, error out
    if (phTargetList == NULL)
    {
        return DSRL_INVALID_OBJECT;
    }

    // Create the object and initialize it
    psObj = psCreate(
        hManager,
        DSRL_TYPE_STANDARD,
        tCapacity,
        n16Sort, pvSortArgs,
        bFilterItem, pvFilterArgs,
        vCallback, pvCallbackArgs);
    if (psObj == NULL)
    {
        return DSRL_INVALID_OBJECT;
    }

    // Send the create target arg to the manager
    bSuccess = bPostTargetArgEvent(psObj,
        DSRL_ACTION_ADD,
        DSRL_MODIFY_OPERATION_INVALID,
        phTargetList, tNumTargets);
    if (bSuccess == FALSE)
    {
        // Couldn't do it -- destroy the DSRL
        DSRL_vDestroy((DSRL_OBJECT)psObj);
        psObj = NULL;
    }

    return (DSRL_OBJECT)psObj;
}

/*****************************************************************************
*
*   hCreateForDeviceLocation
*
* This object interface method is a function that is used to create a
* DSRL object for a specific data service that is attached to the
* device location
*
* Inputs:
*
*   hManager - The Data Service object this object is being used with
*   hRadiusOfInterest - A valid DISTANCE_OBJECT handle which indicates the
*       radius of interest around the device location for this list.
*   tCapacity - The max number of items that can be contained in this DSRL
*   n16Sort - A user-supplied sort function that will override the
*       service-specific sort function.  A value of NULL would indicate that
*       the caller wants to reply on the service-specific sort function
*   pvSortArg - Arguments that will be present in the sort function.
*   bFilterItem - A user-supplied function that can be used to filter items
*       from the DSRL.  A value of NULL indicates that this DSRL does not
*       need a filter function.
*   pvFilterArg - Arguments that will be present in the filter function.
*   vCallback - The callback function that is called when the DSRL wants to
*               notify of any changes to the contents.
*   pvCallbackArgs - Arguments that will be present in the callback function.
*
* Outputs:
*
*   A valid DSRL on success, a DSRL_INVALID_OBJECT on error.
*****************************************************************************/
static DSRL_OBJECT hCreateForDeviceLocation (
    DATASERVICE_MGR_OBJECT hManager,
    DISTANCE_OBJECT hRadiusOfInterest,
    size_t tCapacity,
    DSRL_SORT_FUNCTION n16Sort,
    void *pvSortArg,
    DSRL_FILTER_FUNCTION bFilterItem,
    void *pvFilterArg,
    DSRL_CALLBACK vCallback,
    void *pvCallbackArg
        )
{
    BOOLEAN bSuccess = FALSE;
    DSRL_OBJECT_STRUCT *psObj = (DSRL_OBJECT_STRUCT*)DSRL_INVALID_OBJECT;
    LOCATION_OBJECT hTargetLocation = LOCATION_INVALID_OBJECT;

    do
    {
        // Capacity must be something useful!
        if (tCapacity == 0)
        {
            break;
        }

        // Create a target based on the current device location
        hTargetLocation =
            LOCATION_hCreateForDeviceArea( hRadiusOfInterest );
        if (hTargetLocation == LOCATION_INVALID_OBJECT)
        {
            break;
        }

        // Create the object and initialize it
        psObj = psCreate(
            hManager,
            DSRL_TYPE_DEVICE,
            tCapacity,
            n16Sort, pvSortArg,
            bFilterItem, pvFilterArg,
            vCallback, pvCallbackArg);
        if (psObj == NULL)
        {
            break;
        }

        // Send the create target arg to the manager
        bSuccess = bPostTargetArgEvent(psObj,
            DSRL_ACTION_ADD,
            DSRL_MODIFY_OPERATION_INVALID,
            &hTargetLocation, 1);
    } while (FALSE);

    if (bSuccess == FALSE)
    {
        if (hTargetLocation != LOCATION_INVALID_OBJECT)
        {
            LOCATION.vDestroy(hTargetLocation);
            hTargetLocation = LOCATION_INVALID_OBJECT;
        }

        if (psObj != NULL)
        {
            // Couldn't do it -- destroy the DSRL
            DSRL_vDestroy((DSRL_OBJECT)psObj);
            psObj = (DSRL_OBJECT_STRUCT*)DSRL_INVALID_OBJECT;
        }
    }

    return (DSRL_OBJECT)psObj;
}

/*****************************************************************************
*
*   hCreateFavorites
*
* This object interface method is a function that is used to create a
* "favorites" DSRL object for a specific data service
*
* Inputs:
*
*   hDataServiceMgr - The Data Service object this object is being used with
*   n16Sort - A user-supplied sort function that will override the
*       service-specific sort function.  A value of NULL would indicate that
*       the caller wants to reply on the service-specific sort function
*   pvSortArgs - Arguments that will be present in the sort function.
*   vCallback - The callback function that is called when the DSRL wants to
*               notify of any changes to the contents.
*   pvCallbackArgs - Arguments that will be present in the callback function.
*
* Outputs:
*
*   A valid DSRL on success, a DSRL_INVALID_OBJECT on error.
*****************************************************************************/
static DSRL_OBJECT hCreateFavorites (
    DATASERVICE_MGR_OBJECT hManager,
    DSRL_SORT_FUNCTION n16Sort,
    void *pvSortArgs,
    DSRL_CALLBACK vCallback,
    void *pvCallbackArgs
        )
{
    BOOLEAN bSuccess;
    DSRL_OBJECT_STRUCT *psObj;

    // Create the object and initialize it
    psObj = psCreate(
        hManager,
        DSRL_TYPE_FAVORITES,
        SIZE_T_MAX,
        n16Sort, pvSortArgs,
        NULL, NULL,
        vCallback, pvCallbackArgs);
    if (psObj == NULL)
    {
        return DSRL_INVALID_OBJECT;
    }

    // Send the create target arg to the manager,
    // but we don't have any targets to go along with it
    bSuccess = bPostTargetArgEvent(psObj,
        DSRL_ACTION_ADD,
        DSRL_MODIFY_OPERATION_INVALID,
        NULL, 0);
    if (bSuccess == FALSE)
    {
        // Couldn't do it -- destroy the DSRL
        DSRL_vDestroy((DSRL_OBJECT)psObj);
        psObj = NULL;
    }

    return (DSRL_OBJECT)psObj;
}

/*****************************************************************************
*
*   eModifyTargets
*
* This object interface method is a function that is used to modify
* the targets for a DSRL object.
*
* Inputs:
*
*   hDSRL - A valid DSRL handle
*   eOperation - The alteration to make
*   phTargetList - A pointer to an array of DSRL_TARGET_OBJECTs that will
*       serve as the target criteria for this DSRL
*   tNumTargets - The number of elements in the array pointed to by phTargetList
*
* Outputs:
*
*   A SMSAPI_RETURN_CODE_SUCCESS on success, another error code for failure.
*****************************************************************************/
static SMSAPI_RETURN_CODE_ENUM eModifyTargets (
    DSRL_OBJECT hDSRL,
    DSRL_MODIFY_OPERATION_ENUM eOperation,
    DSRL_TARGET_OBJECT *phTargetList,
    size_t tNumTargets
        )
{
    BOOLEAN bSuccess, bLocked;
    SMSAPI_RETURN_CODE_ENUM eErrorCode;
    DSRL_OBJECT_STRUCT *psObj = (DSRL_OBJECT_STRUCT *)hDSRL;

    // Verify and lock SMS Object
    bLocked =
        SMSO_bLock((SMS_OBJECT)hDSRL, OSAL_OBJ_TIMEOUT_INFINITE);
    if(bLocked == FALSE)
    {
        // Unable to lock the provided object
        return SMSAPI_RETURN_CODE_ERROR;
    }

    do
    {
        // Ensure this isn't a DEVICE-based DSRL
        if (psObj->eDSRLType == DSRL_TYPE_DEVICE)
        {
            eErrorCode = SMSAPI_RETURN_CODE_API_NOT_ALLOWED;
            break;
        }

        // Ensure the DSRL isn't still initializing
        if ((psObj->eState == DSRL_STATE_UNKNOWN) ||
            (psObj->eState == DSRL_STATE_INITIAL))
        {
            eErrorCode = SMSAPI_RETURN_CODE_API_NOT_ALLOWED;
            break;
        }

        // Ensure the DSRL isn't stopping
        if (psObj->bStopping == TRUE)
        {
            eErrorCode = SMSAPI_RETURN_CODE_API_NOT_ALLOWED;
            break;
        }

        bSuccess = bPostTargetArgEvent(psObj,
            DSRL_ACTION_MODIFY, eOperation,
            phTargetList, tNumTargets);

        if (bSuccess == FALSE)
        {
            eErrorCode = SMSAPI_RETURN_CODE_ERROR;
            break;
        }

        // All is good
        eErrorCode = SMSAPI_RETURN_CODE_SUCCESS;

    } while (FALSE);

    SMSO_vUnlock((SMS_OBJECT)hDSRL);

    return eErrorCode;
}

/*****************************************************************************
*
*   eModifyCapacity
*
* This object interface method is a function that is used to modify the
* capacity of a given DSRL
*
* Inputs:
*
*   hDSRL - A valid DSRL handle
*   tCapacity - The new capacity for this DSRL
*
* Outputs:
*
*   A SMSAPI_RETURN_CODE_SUCCESS on success, another error code for failure.
*****************************************************************************/
static SMSAPI_RETURN_CODE_ENUM eModifyCapacity (
    DSRL_OBJECT hDSRL,
    size_t tCapacity
        )
{
    BOOLEAN bLocked;
    SMSAPI_RETURN_CODE_ENUM eErrorCode = SMSAPI_RETURN_CODE_ERROR;
    DSRL_OBJECT_STRUCT *psObj = (DSRL_OBJECT_STRUCT *)hDSRL;

    // Verify and lock SMS Object
    bLocked =
        SMSO_bLock((SMS_OBJECT)hDSRL, OSAL_OBJ_TIMEOUT_INFINITE);
    if(bLocked == FALSE)
    {
        // Unable to lock the provided object
        return SMSAPI_RETURN_CODE_ERROR;
    }

    do
    {
        BOOLEAN bDecrease = FALSE;

        // Verify DSRL state & stopping status
        if ((psObj->eState != DSRL_STATE_READY) ||
            (psObj->bStopping == TRUE))
        {
            // Don't try this when the DSRL isn't stable
            eErrorCode = SMSAPI_RETURN_CODE_API_NOT_ALLOWED;

            break;
        }

        // Verify capacity actually changed
        if (psObj->tReportedCapacity == tCapacity)
        {
            // Not a problem
            eErrorCode = SMSAPI_RETURN_CODE_SUCCESS;
            break;
        }

        // Did the DSRL capacity decrease?
        if (tCapacity < psObj->tReportedCapacity)
        {
            bDecrease = TRUE;
        }

        // Set the reported capacity and compute the
        // working capacity
        psObj->tReportedCapacity = tCapacity;
        psObj->tWorkingCapacity = DSRL_COMPUTE_WORKING_CAPACITY(tCapacity);

        if (bDecrease == TRUE)
        {
            // Send the DSRL to UPDATING
            DSRL_vSetState(hDSRL, DSRL_STATE_UPDATING);

            // Transition to the next state (we prune
            // the list when we go to ready, so we
            // don't have any more work to do)
            DSRL_vSetState(hDSRL, DSRL_STATE_READY);
        }
        else
        {
            BOOLEAN bPosted;

            // Post an event to grow the list since we
            // can't do it ourselves
            bPosted = bPostTargetArgEvent(psObj,
                DSRL_ACTION_REFRESH,
                DSRL_MODIFY_OPERATION_INVALID,
                NULL, 0);

            if (bPosted == FALSE)
            {
                eErrorCode = SMSAPI_RETURN_CODE_ERROR;
                break;
            }
        }

        // All is good
        eErrorCode = SMSAPI_RETURN_CODE_SUCCESS;

    } while (FALSE);

    SMSO_vUnlock((SMS_OBJECT)hDSRL);

    return eErrorCode;
}

/*****************************************************************************
*
*   eModifyFilter
*
* This object interface method is a function that is used to modify the
* filter function used
*
* Inputs:
*
*   hDSRL - A valid DSRL handle
*   bFilterItem - A user-supplied function that can be used to filter items
*       from the DSRL.  A value of NULL indicates that this DSRL does not
*       need a filter function.
*   pvFilterArgs - Arguments that will be present in the filter function.
*
* Outputs:
*
*   A SMSAPI_RETURN_CODE_SUCCESS on success, another error code for failure.
*****************************************************************************/
static SMSAPI_RETURN_CODE_ENUM eModifyFilter (
    DSRL_OBJECT hDSRL,
    DSRL_FILTER_FUNCTION bFilterItem,
    void *pvFilterArgs
        )
{
    BOOLEAN bLocked;
    SMSAPI_RETURN_CODE_ENUM eErrorCode;
    DSRL_OBJECT_STRUCT *psObj = (DSRL_OBJECT_STRUCT *)hDSRL;

    // Verify and lock SMS Object
    bLocked =
        SMSO_bLock((SMS_OBJECT)hDSRL, OSAL_OBJ_TIMEOUT_INFINITE);
    if(bLocked == FALSE)
    {
        // Unable to lock the provided object
        return SMSAPI_RETURN_CODE_ERROR;
    }

    do
    {
        BOOLEAN bSuccess;
        DSRL_STATE_ENUM eEndState = DSRL_STATE_READY;

        // Verify DSRL state & stopping status
        if ((psObj->eState != DSRL_STATE_READY) ||
            (psObj->bStopping == TRUE))
        {
            // Don't try this when the DSRL isn't stable
            eErrorCode = SMSAPI_RETURN_CODE_API_NOT_ALLOWED;

            break;
        }

        // update the filter function and argument
        psObj->bFilterItem = bFilterItem;
        psObj->pvFilterArgs = pvFilterArgs;

        // We need to apply this new filter configuration

        // Send the DSRL to UPDATING
        DSRL_vSetState(hDSRL, DSRL_STATE_UPDATING);

        // Apply the new filter
        bSuccess = bApplyNewFilter(psObj);
        if (bSuccess == FALSE)
        {
            eEndState = DSRL_STATE_ERROR;
        }

        // Transition to the next state
        DSRL_vSetState(hDSRL, eEndState);

        bSuccess = bPostTargetArgEvent(psObj,
            DSRL_ACTION_REFRESH,
            DSRL_MODIFY_OPERATION_INVALID,
            NULL, 0);

        if (bSuccess == FALSE)
        {
            eErrorCode = SMSAPI_RETURN_CODE_ERROR;
            break;
        }

        // All is good
        eErrorCode = SMSAPI_RETURN_CODE_SUCCESS;

    } while (FALSE);

    SMSO_vUnlock((SMS_OBJECT)hDSRL);

    return eErrorCode;
}

/*****************************************************************************
*
*   eModifySort
*
* This object interface method is a function that is used to modify the
* sort function used
*
* Inputs:
*
*   hDSRL - A valid DSRL handle
*   n16Sort - A user-supplied function that can be used to sort items
*       in the DSRL.  A value of NULL indicates that this DSRL does not
*       need a sort function.
*   pvSortArg - Argument that will be presented to the sort function.
*
* Outputs:
*
*   A SMSAPI_RETURN_CODE_SUCCESS on success, another error code for failure.
*****************************************************************************/
static SMSAPI_RETURN_CODE_ENUM eModifySort (
    DSRL_OBJECT hDSRL,
    DSRL_SORT_FUNCTION n16Sort,
    void *pvSortArgs
        )
{
    BOOLEAN bLocked;
    SMSAPI_RETURN_CODE_ENUM eErrorCode = SMSAPI_RETURN_CODE_ERROR;
    DSRL_OBJECT_STRUCT *psObj = (DSRL_OBJECT_STRUCT *)hDSRL;

    // Verify and lock SMS Object
    bLocked =
        SMSO_bLock((SMS_OBJECT)hDSRL, OSAL_OBJ_TIMEOUT_INFINITE);
    if(bLocked == FALSE)
    {
        // Unable to lock the provided object
        return SMSAPI_RETURN_CODE_ERROR;
    }

    do
    {
        BOOLEAN bSuccess;

        // Verify DSRL state & stopping status
        if ((psObj->eState != DSRL_STATE_READY) ||
            (psObj->bStopping == TRUE))
        {
            // Don't try this when the DSRL isn't stable
            eErrorCode = SMSAPI_RETURN_CODE_API_NOT_ALLOWED;

            break;
        }

        // Were we given a sort function?
        if (n16Sort == NULL)
        {
            // No -- so we'll use the default
            psObj->n16Sort = psObj->n16DefaultSort;
            psObj->pvSortArgs = psObj->pvDefaultSortArgs;
        }
        else
        {
            // Yeah, update the sort function and argument
            psObj->n16Sort = n16Sort;
            psObj->pvSortArgs = pvSortArgs;
        }

        // We need to apply this new sort configuration

        // Send the DSRL to UPDATING
        DSRL_vSetState(hDSRL, DSRL_STATE_UPDATING);

        // Re-sort all of this DSRL's contents
        vFullSort(psObj);

        // Transition to the next state
        DSRL_vSetState(hDSRL, DSRL_STATE_READY);

        // Post the event to refresh the DSRL now
        bSuccess = bPostTargetArgEvent(psObj,
            DSRL_ACTION_REFRESH,
            DSRL_MODIFY_OPERATION_INVALID,
            NULL, 0);

        if (bSuccess == FALSE)
        {
            break;
        }

        // All is good
        eErrorCode = SMSAPI_RETURN_CODE_SUCCESS;

    } while (FALSE);

    SMSO_vUnlock((SMS_OBJECT)hDSRL);

    return eErrorCode;
}

/*****************************************************************************
*
*   eModifyDeviceRadius
*
* This object interface method is used to modify the radius in use for a
* device-based DSRL.
*
* Inputs:
*
*   hDSRL - A valid DSRL handle to a DEVICE dsrl
*   hRadius - A user-supplied valid DISTANCE_OBJECT that indicates the
*       new radius to use for this DSRL.  This object is copied by
*       SMS to ensure object safety.
*
* Outputs:
*
*   A SMSAPI_RETURN_CODE_SUCCESS on success, an error code for failure.
*****************************************************************************/
static SMSAPI_RETURN_CODE_ENUM eModifyDeviceRadius (
    DSRL_OBJECT hDSRL,
    DISTANCE_OBJECT hRadius
        )
{
    BOOLEAN bLocked;
    SMSAPI_RETURN_CODE_ENUM eReturnCode = SMSAPI_RETURN_CODE_ERROR;

    // Verify and lock SMS Object
    bLocked =
        SMSO_bLock((SMS_OBJECT)hDSRL, OSAL_OBJ_TIMEOUT_INFINITE);
    if(bLocked == FALSE)
    {
        // Unable to lock the provided object
        return SMSAPI_RETURN_CODE_ERROR;
    }

    do
    {
        DSRL_OBJECT_STRUCT *psObj =
            (DSRL_OBJECT_STRUCT *)hDSRL;
        BOOLEAN bSuccess;

        // Ensure the DSRL isn't still initializing
        if ((psObj->eState == DSRL_STATE_UNKNOWN) ||
            (psObj->eState == DSRL_STATE_INITIAL))
        {
            eReturnCode = SMSAPI_RETURN_CODE_API_NOT_ALLOWED;

            break;
        }

        // Ensure the DSRL isn't stopping
        if (psObj->bStopping == TRUE)
        {
            eReturnCode = SMSAPI_RETURN_CODE_API_NOT_ALLOWED;

            break;
        }

        if (psObj->eDSRLType != DSRL_TYPE_DEVICE)
        {
            // Only good for device dsrls!
            eReturnCode = SMSAPI_RETURN_CODE_API_NOT_ALLOWED;

            break;
        }

        // Update the radius now
        bSuccess =
            DATASERVICE_IMPL_bDeviceRadiusUpdate(psObj->hService, hRadius);
        if (bSuccess == FALSE)
        {
            eReturnCode = SMSAPI_RETURN_CODE_ERROR;

            break;
        }

        // All is good
        eReturnCode = SMSAPI_RETURN_CODE_SUCCESS;

    } while (FALSE);

    // All done inspecting the DSRL
    SMSO_vUnlock((SMS_OBJECT)hDSRL);

    return eReturnCode;
}

/*****************************************************************************
*
*   vDestroy
*
* This object interface method is a function that is used to destroy
* the DSRL.  Please note that the objects in the DSRL are not destroyed, the
* data service manager is still responsible for destroying those.  However,
* the data service manager is notified that the objects are being removed
* via the finalize function if need be.
*
* Inputs:
*
*   hDSRL - A valid DSRL
*
* Outputs:
*
*   Nothing
*
*****************************************************************************/
static void vDestroy (
    DSRL_OBJECT hDSRL
        )
{
    BOOLEAN bLocked;

    // Verify and lock SMS Object
    bLocked =
        SMSO_bLock((SMS_OBJECT)hDSRL, OSAL_OBJ_TIMEOUT_INFINITE);
    if(bLocked == TRUE)
    {
        DSRL_OBJECT_STRUCT *psObj =
            (DSRL_OBJECT_STRUCT *)hDSRL;

        if (psObj->bStopping == FALSE)
        {
            BOOLEAN bPosted;

            // Create memory for our argument to be passed along with the event
            bPosted = bPostTargetArgEvent(psObj,
                DSRL_ACTION_REMOVE, DSRL_MODIFY_OPERATION_INVALID,
                NULL, 0);

            if (bPosted == FALSE)
            {
                SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    DSRL_OBJECT_NAME": Unable to post stop event");
            }

            // This DSRL is stopping if this event was posted
            psObj->bStopping = bPosted;
        }

        SMSO_vUnlock((SMS_OBJECT)hDSRL);
    }

    return;
}

/*****************************************************************************
*
*   hDataServiceMgr
*
* This object interface method is used by the caller to get the handle to the
* data service manager associated with this DSRL.
*
* Inputs:
*
*   hDSRL - A handle to a valid DSRL object
*
* Outputs:
*
*   A valid DATASERVICE_MGR_OBJECT handle for the DSRL on Success
*   A DATASERVICE_MGR_INVALID_OBJECT upon error
*
*****************************************************************************/
static DATASERVICE_MGR_OBJECT hDataServiceMgr (
    DSRL_OBJECT hDSRL
        )
{
    BOOLEAN bLocked;
    DATASERVICE_MGR_OBJECT hManager =
        DATASERVICE_MGR_INVALID_OBJECT;

    // Verify and lock SMS Object
    bLocked =
        SMSO_bLock((SMS_OBJECT)hDSRL, OSAL_OBJ_TIMEOUT_INFINITE);
    if(bLocked == TRUE)
    {
        DSRL_OBJECT_STRUCT *psObj =
            (DSRL_OBJECT_STRUCT *)hDSRL;

        // Extract the service handle now, call it a manager
        hManager = (DATASERVICE_MGR_OBJECT)psObj->hService;

        SMSO_vUnlock((SMS_OBJECT)hDSRL);
    }
    return hManager;
}

/*****************************************************************************
*
*   hDeviceLocation
*
* This object interface method is used by the caller to get the handle to the
* device location instance associated with this DSRL.
*
* Inputs:
*
*   hDSRL - A handle to a valid DSRL object
*
* Outputs:
*
*   A valid LOCATION_OBJECT handle for the DSRL on Success
*   A LOCATION_INVALID_OBJECT upon error
*
*****************************************************************************/
static LOCATION_OBJECT hDeviceLocation (
    DSRL_OBJECT hDSRL
        )
{
    BOOLEAN bOwner;
    LOCATION_OBJECT hDevLocation = LOCATION_INVALID_OBJECT;

    // Verify inputs. Object handle must be valid.
    // THIS MUST BE AN OWNERSHIP CHECK
    bOwner = SMSO_bOwner((SMS_OBJECT)hDSRL);
    if(bOwner == TRUE)
    {
        DSRL_OBJECT_STRUCT *psObj =
            (DSRL_OBJECT_STRUCT *)hDSRL;

        // Provide the caller with the data
        hDevLocation = psObj->hDeviceLocation;

        // Remember that they asked for this
        psObj->bSortUsesDeviceLocation = TRUE;
    }

    return hDevLocation;
}

/*****************************************************************************
*
*   eType
*
* This object interface method is used by the caller to get the type of
* data service this list is for.
*
* Inputs:
*
*   hDSRL - A handle to a valid DSRL object
*
* Outputs:
*
*   A DATASERVICE_TYPE_ENUM representing the type of this
*   DSRL on success. A DATASERVICE_TYPE_UNKNOWN upon error
*
*****************************************************************************/
static DATASERVICE_TYPE_ENUM eType (
    DSRL_OBJECT hDSRL
        )
{
    BOOLEAN bLocked;
    DATASERVICE_TYPE_ENUM eType = DATASERVICE_TYPE_UNKNOWN;

    // Verify and lock SMS Object
    bLocked =
        SMSO_bLock((SMS_OBJECT)hDSRL, OSAL_OBJ_TIMEOUT_INFINITE);
    if (bLocked == TRUE)
    {
        DSRL_OBJECT_STRUCT *psObj =
            (DSRL_OBJECT_STRUCT *)hDSRL;

        eType = psObj->eType;

        SMSO_vUnlock((SMS_OBJECT)hDSRL);
    }

    return eType;
}

/*****************************************************************************
*
*   eState
*
* This object interface method is used by the caller to get the state of
* data service this list is for.
*
* Inputs:
*
*   hDSRL - A handle to a valid DSRL object
*
* Outputs:
*
*   A DSRL_STATE_ENUM representing the type of this
*   DSRL on success. A DSRL_STATE_UNKNOWN upon error
*
*****************************************************************************/
static DSRL_STATE_ENUM eState (
    DSRL_OBJECT hDSRL
        )
{
    BOOLEAN bLocked;
    DSRL_STATE_ENUM eState = DSRL_STATE_UNKNOWN;

    // Verify and lock SMS Object
    bLocked =
        SMSO_bLock((SMS_OBJECT)hDSRL, OSAL_OBJ_TIMEOUT_INFINITE);
    if (bLocked == TRUE)
    {
        DSRL_OBJECT_STRUCT *psObj =
            (DSRL_OBJECT_STRUCT *)hDSRL;

        eState = psObj->eState;

        SMSO_vUnlock((SMS_OBJECT)hDSRL);
    }

    return eState;
}

/*****************************************************************************
*
*   tCapacity
*
* This object interface method is used by the caller to get the capacity of
* the DSRL.
*
* Inputs:
*
*   hDSRL - A handle to a valid DSRL object
*
* Outputs:
*
*   A size_t representing the capacity of this
*   DSRL on success. A 0 upon error
*
*****************************************************************************/
static size_t tCapacity (
    DSRL_OBJECT hDSRL
        )
{
    BOOLEAN bLocked;
    size_t tCapacity = 0;

    // Verify and lock SMS Object
    bLocked =
        SMSO_bLock((SMS_OBJECT)hDSRL, OSAL_OBJ_TIMEOUT_INFINITE);
    if (bLocked == TRUE)
    {
        DSRL_OBJECT_STRUCT *psObj =
            (DSRL_OBJECT_STRUCT *)hDSRL;

        tCapacity = psObj->tReportedCapacity;

        SMSO_vUnlock((SMS_OBJECT)hDSRL);
    }

    return tCapacity;
}

/*****************************************************************************
*
*   bTargetsUpdated
*
* This object interface method is used by the caller to determine if
* a target update is responsible for the current state transition
* of a DSRL
*
* Inputs:
*
*   hDSRL - A handle to a valid DSRL object
*
* Outputs:
*
*   A BOOLEAN indicating if the state transition currently being experienced
*   is due to a target update.
*
*****************************************************************************/
static BOOLEAN bTargetsUpdated (
    DSRL_OBJECT hDSRL
        )
{
    BOOLEAN bOwner, bUpdated = FALSE;

    bOwner = SMSO_bOwner((SMS_OBJECT)hDSRL);
    if (TRUE == bOwner)
    {
        DSRL_OBJECT_STRUCT *psObj =
            (DSRL_OBJECT_STRUCT *)hDSRL;

        // Extract the appropriate value from the
        // state dependent attributes
        bUpdated = psObj->asStateAttribs[
            DSRL_TARGET_UPDATE_ATTRIB_INDEX].bValue;
    }

    return bUpdated;
}

/*****************************************************************************
*
*   eIterate
*
* This public object interface method is used to iterate the contents of the
* DSRL using the specified arguments.  This function may be called in
* any context, but will not succeed if the DSRL is not in a ready state.
*
* Inputs:
*
*   hDSRL - A handle to a valid DSRL object
*   bEntryIterator - A DSRL_ITERATOR_CALLBACK function used to iterate over
*                    the contents of the DSRL.
*   pvIteratorArg - A callback argument.
*
* Outputs:
*
*   SMSAPI_RETURN_CODE_SUCCESS if iteration is successful.
*   SMSAPI_RETURN_CODE_ERROR upon error.
*
*****************************************************************************/
static SMSAPI_RETURN_CODE_ENUM eIterate (
    DSRL_OBJECT hDSRL,
    DSRL_ITERATOR_CALLBACK bEntryIterator,
    void *pvIteratorArg
        )
{
    BOOLEAN bLocked;
    DSRL_OBJECT_STRUCT *psObj =
        (DSRL_OBJECT_STRUCT *)hDSRL;
    SMSAPI_RETURN_CODE_ENUM eReturnCode;

    // Verify and lock SMS Object
    bLocked =
        SMSO_bLock((SMS_OBJECT)hDSRL, OSAL_OBJ_TIMEOUT_INFINITE);
    if(bLocked == FALSE)
    {
        // Unable to lock the provided object
        return SMSAPI_RETURN_CODE_ERROR;
    }

    // Call the local iterator -- ensure
    // that we are in the ready state
    eReturnCode = eIterateLocal(psObj, TRUE, bEntryIterator, pvIteratorArg);

    SMSO_vUnlock((SMS_OBJECT)hDSRL);

    return eReturnCode;
}

/*****************************************************************************
*
*   eExclusiveAccess
*
* This public object interface method is used to iterate the contents of the
* DSRL using the specified arguments.  This function may be called in
* any context, but will not succeed if the DSRL is not in a ready state.
*
* Inputs:
*
*   hDSRL - A handle to a valid DSRL object
*   bEntryIterator - A DSRL_ITERATOR_CALLBACK function used to iterate over
*                    the contents of the DSRL.
*   pvIteratorArg - A callback argument.
*
* Outputs:
*
*   SMSAPI_RETURN_CODE_SUCCESS if iteration is successful.
*   SMSAPI_RETURN_CODE_ERROR upon error.
*
*****************************************************************************/
static SMSAPI_RETURN_CODE_ENUM eExclusiveAccess (
   DSRL_OBJECT hDSRL,
   DSRL_EXCLUSIVE_ACCESS_CALLBACK vAccessCallback,
   void *pvAccessArg
      )
{
    BOOLEAN bLocked;

    // Verify callback
    if (vAccessCallback == NULL)
    {
        return SMSAPI_RETURN_CODE_INVALID_INPUT;
    }

    // Verify and lock SMS Object
    bLocked =
        SMSO_bLock((SMS_OBJECT)hDSRL, OSAL_OBJ_TIMEOUT_INFINITE);
    if(bLocked == FALSE)
    {
        return SMSAPI_RETURN_CODE_ERROR;
    }

    // Invoke the callback
    vAccessCallback(hDSRL, pvAccessArg);

    SMSO_vUnlock((SMS_OBJECT)hDSRL);

    return SMSAPI_RETURN_CODE_SUCCESS;
}

/*****************************************************************************
                             FRIEND FUNCTIONS
*****************************************************************************/
/*****************************************************************************
*
*   DSRL_vDestroy
*
*****************************************************************************/
void DSRL_vDestroy (
    DSRL_OBJECT hDSRL
        )
{
    BOOLEAN bLocked;

    // Lock the DSRL
    bLocked = SMSO_bLock((SMS_OBJECT)hDSRL, OSAL_OBJ_TIMEOUT_INFINITE);
    if(bLocked == TRUE)
    {
        OSAL_RETURN_CODE_ENUM eReturnCode;
        DSRL_OBJECT_STRUCT *psObj =
            (DSRL_OBJECT_STRUCT *)hDSRL;
        SMS_OBJECT hParent;

        // Get the parent object so we can
        // unlock it after DSRL destroying
        hParent = SMSO_hParent((SMS_OBJECT)hDSRL);
        if (hParent == SMS_INVALID_OBJECT)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                DSRL_OBJECT_NAME": Unable to get parent handle.");

            SMSO_vUnlock((SMS_OBJECT)hDSRL);
            return;
        }

        if (psObj->eState != DSRL_STATE_UNKNOWN)
        {
            // Tell the application that this DSRL has stopped
            // but only if we've made it out of
            // the hCreate API (since in that case
            // we never invoked the application callback
            // and it would not be helpful to call it now)
            DSRL_vSetState(hDSRL, DSRL_STATE_STOPPED);
        }

        eReturnCode = OSAL.eLinkedListRemoveAll(psObj->hObjectList, 
            (OSAL_LL_RELEASE_HANDLER)vDestroyResultEntry);
        if (eReturnCode == OSAL_SUCCESS)
        {
            eReturnCode = OSAL.eLinkedListDelete(psObj->hObjectList);
            if (eReturnCode == OSAL_SUCCESS)
            {
                psObj->hObjectList = OSAL_INVALID_OBJECT_HDL;
            }
        }

        eReturnCode = OSAL.eLinkedListRemoveAll(psObj->hRemovedObjList,
            OSAL.vLinkedListMemoryFree);
        if (eReturnCode == OSAL_SUCCESS)
        {
            eReturnCode = OSAL.eLinkedListDelete(psObj->hRemovedObjList);
            if (eReturnCode == OSAL_SUCCESS)
            {
                psObj->hRemovedObjList = OSAL_INVALID_OBJECT_HDL;
            }
        }

        // Report to the DSM that this dsrl is now gone
        DATASERVICE_IMPL_vReportDSRLDestroyed(
            psObj->hService, hDSRL, psObj->eDSRLType);

        // Free object instance
        SMSO_vDestroy((SMS_OBJECT)psObj);

        // Unlock parent object
        SMSO_vUnlock(hParent);
    }

    return;
}

/*****************************************************************************
*
*   DSRL_bSetDefaultSortFunction
*
* This object interface method is a friend function that is used to set the
* default service-specific sort function.  Can only be set once
*
* Inputs:
*
*   hDSRL - A valid DSRL
*   n16Sort - The sort function
*   pvSortArgs - Arguments that will be present in the sort function.
*
* Outputs:
*
*   TRUE if the sort function was set
*   FALSE if there was some error
*
*****************************************************************************/
BOOLEAN DSRL_bSetDefaultSortFunction (
    DSRL_OBJECT hDSRL,
    DSRL_SORT_FUNCTION n16Sort,
    void *pvSortArgs
        )
{
    DSRL_OBJECT_STRUCT *psObj =
        (DSRL_OBJECT_STRUCT *)hDSRL;
    BOOLEAN bOwner;

    // Verify inputs. Object handle must be valid.
    bOwner = SMSO_bOwner((SMS_OBJECT)hDSRL);
    if(FALSE == bOwner)
    {
        // Error!
        return FALSE;
    }

    // Only set the default sort once.
    // We want to prevent changing the default
    // sort function through the course of operation
    if (NULL != psObj->n16DefaultSort)
    {
        return FALSE;
    }

    do
    {
        OSAL_RETURN_CODE_ENUM eReturnCode;

        // Favorites don't get a default sort,
        // but just say it worked
        if (DSRL_TYPE_FAVORITES == psObj->eDSRLType)
        {
            break;
        }

        // Update the default
        psObj->n16DefaultSort = n16Sort;
        psObj->pvDefaultSortArgs = pvSortArgs;

        // Is the sort function set?
        if (NULL != psObj->n16Sort)
        {
            // Yes, leave it alone
            break;
        }

        // No sort is set - use the default
        psObj->n16Sort = n16Sort;
        psObj->pvSortArgs = pvSortArgs;

        // calling the sort to update the sort order
        eReturnCode = OSAL.eLinkedListSort(psObj->hObjectList, 
            (OSAL_LL_COMPARE_HANDLER)n16SortShim);
        if (eReturnCode != OSAL_SUCCESS)
        {
            // indicating the failure
            return FALSE;
        }

    } while (FALSE);

    return TRUE;
}

/*****************************************************************************
*
*   DSRL_bSort
*
*   Perform a full-resort of the DSRL without all the bells & whistles
*   that DSRL.eModifySort has.  This is called by service managers which
*   assume that the sort is set to whatever was configured as the
*   default.  So, if the sort used now isn't actually the default then
*   we can just ignore this request.
*
*****************************************************************************/
BOOLEAN DSRL_bSort (
    DSRL_OBJECT hDSRL
        )
{
    DSRL_OBJECT_STRUCT *psObj =
        (DSRL_OBJECT_STRUCT *)hDSRL;
    BOOLEAN bOwner;

    // Verify inputs. Object handle must be valid.
    bOwner = SMSO_bOwner((SMS_OBJECT)hDSRL);
    if(bOwner == FALSE)
    {
        // Error!
        return FALSE;
    }

    // Are we currently using the default sort?
    if (psObj->n16DefaultSort == psObj->n16Sort)
    {
        // Yes - perform the sort of the DSRL's contents
        vFullSort(psObj);
    }
    else
    {
        // No, which means we don't actually need to sort now
        // since the sort settings are under the control
        // of the application
    }

    return TRUE;
}

/*****************************************************************************
*
*   DSRL_bSetCompleteAddFunction
*
* This object interface method is a friend function that is used to set a
* service-specific "add completion" function.
*
* Function that is called just before the handle is stored in the DSRL
* Might be used to do something complex where you use a stack object
* to see if it *can* be added to the DSRL, then you provide a heap object
* when you know that it will be added.  Prevent a few mallocs.
*
* Inputs:
*
*   hDSRL - A valid DSRL
*   n16Sort - The sort function
*
* Outputs:
*
*   TRUE if the  function was set
*   FALSE if there was some error
*
*****************************************************************************/
BOOLEAN DSRL_bSetCompleteAddFunction (
    DSRL_OBJECT hDSRL,
    DSRL_COMPLETE_ADD_FUNCTION hCompleteAdd,
    void *pvCompleteAddArgs
        )
{
    DSRL_OBJECT_STRUCT *psObj =
        (DSRL_OBJECT_STRUCT *)hDSRL;
    BOOLEAN bOwner;

    // Verify inputs. Object handle must be valid.
    bOwner = SMSO_bOwner((SMS_OBJECT)hDSRL);
    if(bOwner == FALSE)
    {
        // Error!
        return FALSE;
    }

    psObj->hCompleteAdd = hCompleteAdd;
    psObj->pvCompleteAddArgs = pvCompleteAddArgs;

    return TRUE;
}

/*****************************************************************************
*
*   DSRL_bSetFinalizeFunction
*
* This object interface method is a friend function that is used to set a
* service-specific finlize function for the entries on the list.  This callback
* is called before the entry is finally removed from the DSRL due to various
* operations
*
* Inputs:
*
*   hDSRL - A valid DSRL
*   vFinalize - The finalize function
*   pvFinalizeArgs - The arguments passed to the finalize function
*
* Outputs:
*
*   TRUE if the finalize function was set
*   FALSE if there was some error
*
*****************************************************************************/
BOOLEAN DSRL_bSetFinalizeFunction(
    DSRL_OBJECT hDSRL,
    DSRL_FINALIZE_FUNCTION vFinalize,
    void *pvFinalizeArgs
        )
{
    DSRL_OBJECT_STRUCT *psObj =
        (DSRL_OBJECT_STRUCT *)hDSRL;
    BOOLEAN bOwner;

    // Verify inputs. Object handle must be valid.
    bOwner = SMSO_bOwner((SMS_OBJECT)hDSRL);
    if(bOwner == FALSE)
    {
        // Error!
        return FALSE;
    }

    psObj->vFinalize = vFinalize;
    psObj->pvFinalizeArgs = pvFinalizeArgs;

    return TRUE;
}

/*****************************************************************************
*
*   DSRL_bSetDeviceLocation
*
* This object interface method is a friend function that is used to set a
* device loation instance for this DSRL.
*
* Inputs:
*
*   hDSRL - A valid DSRL
*   hDeviceLocation - A valid location object which is this dsrl's
*       device location instance
*
* Outputs:
*
*   TRUE on succes, FALSE if there was some error
*
*****************************************************************************/
BOOLEAN DSRL_bSetDeviceLocation (
    DSRL_OBJECT hDSRL,
    LOCATION_OBJECT hDeviceLocation
        )
{
    BOOLEAN bLocked;

    // Verify and lock SMS Object
    bLocked =
        SMSO_bLock((SMS_OBJECT)hDSRL, OSAL_OBJ_TIMEOUT_INFINITE);
    if(bLocked == TRUE)
    {
        DSRL_OBJECT_STRUCT *psObj =
            (DSRL_OBJECT_STRUCT *)hDSRL;

        // Store the handle for later use
        psObj->hDeviceLocation = hDeviceLocation;

        SMSO_vUnlock((SMS_OBJECT)hDSRL);
    }

    return bLocked;
}

/*****************************************************************************
*
*   DSRL_eAddEntry
*
* This object interface method is a friend function that is used to add entries
* to the DSRL.  Duplicate entries will not be added to the list.
*
* Inputs:
*
*   hDSRL - A valid DSRL
*   hObject - The object to add to the list
*
* Outputs:
*
*   DSRL_ADD_REPLACE_OK if a new/duplicate item was added to the list.
*   DSRL_ADD_REPLACE_NOP if a new/duplicate item was not added to list
*   DSRL_ADD_REPLACE_ERROR if there was some error adding item to the list.
*
*****************************************************************************/
DSRL_ADD_REPLACE_RESULT_ENUM DSRL_eAddEntry (
    DSRL_OBJECT hDSRL,
    DSRL_ENTRY_OBJECT hObject
        )
{
    DSRL_ADD_REPLACE_RESULT_ENUM eReturn = DSRL_ADD_REPLACE_ERROR;
    DSRL_ENTRY_STRUCT *psResultEntry = NULL;
    OSAL_RETURN_CODE_ENUM eReturnCode;

    do
    {
        DSRL_OBJECT_STRUCT *psObj =
            (DSRL_OBJECT_STRUCT *)hDSRL;
        BOOLEAN bOwner, bSuccess;
        DSRL_ENTRY_STRUCT *psEntryToRemove = NULL;

        // Verify inputs. Object handle must be valid.
        bOwner = SMSO_bOwner((SMS_OBJECT)hDSRL);
        if(bOwner == FALSE)
        {
            // Error!
            break;
        }

        // Test this entry data now
        eReturn = eTestDSRLEntryData (
            psObj, hObject,
            TRUE, &psEntryToRemove);

        if (eReturn != DSRL_ADD_REPLACE_OK)
        {
            break;
        }

        if (psObj->hCompleteAdd != NULL)
        {
            // Give the service-specific manager a chance to do something
            // before this is added to our DSRL.
            hObject = psObj->hCompleteAdd(hDSRL, hObject,
                psObj->pvCompleteAddArgs);

            if (hObject == DSRL_ENTRY_INVALID_OBJECT)
            {
                // They no longer want to add a handle.
                eReturn = DSRL_ADD_REPLACE_NOP;
                break;
            }
        }

        // allocating the memory for new entry
        psResultEntry = (DSRL_ENTRY_STRUCT *)OSAL.pvLinkedListMemoryAllocate(
            DSRL_OBJECT_NAME":Entry", sizeof(*psResultEntry), TRUE);
        if (psResultEntry == NULL)
        {
            // error!
            break;
        }

        psResultEntry->tID = ++psObj->tCurrentID;
        psResultEntry->psDSRL = psObj;
        psResultEntry->hObject = hObject;

        // Add the new entry to the list
        eReturnCode = OSAL.eLinkedListAdd(psObj->hObjectList, 
            &psResultEntry->hEntry, psResultEntry);
        if (eReturnCode != OSAL_SUCCESS)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__, 
                DSRL_OBJECT_NAME": Failed to add entry to objects list "
                "(%s)", OSAL.pacGetReturnCodeName(eReturnCode));

            // Some big failure with our LL, fail out
            eReturn = DSRL_ADD_REPLACE_ERROR;
            break;
        }

        // store the LL entry handle
        bSuccess = DSRL_ENTRY_bAddToDSRL(hObject, hDSRL, 
            psResultEntry->hEntry);
        if (bSuccess == FALSE)
        {
            // error!
            break;
        }

        // Provide this entry with our service handle
        DSRL_ENTRY_vSetService(hObject, psObj->hService);

        if (psObj->eDSRLType == DSRL_TYPE_FAVORITES)
        {
            // This entry is now a favorite
            DSRL_ENTRY_vSetFavorite(hObject, TRUE);
        }

        // The eTestDSRLEntryData indicated an entry
        // should be removed due to capacity checks.
        // Since adding this entry worked, remove psEntryToRemove
        // NULL check happens inside the vRemoveResultEntry
        vRemoveResultEntry(psEntryToRemove);

        printf(DSRL_OBJECT_NAME": Object added is unique: 0x%p\n", hObject);
        return DSRL_ADD_REPLACE_OK;

    } while (FALSE);

    if ((psResultEntry != NULL) &&
        (psResultEntry->hEntry != OSAL_INVALID_LINKED_LIST_ENTRY))
    {
        // error case - need to remove DSRL entry from DSRL list
        eReturnCode = OSAL.eLinkedListRemove(psResultEntry->hEntry);
        if (eReturnCode != OSAL_SUCCESS)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__, 
                DSRL_OBJECT_NAME": Failed to remove entry from objects "
                "list (%s)", OSAL.pacGetReturnCodeName(eReturnCode));

            // skipping pointer to allocated memory to avoid memory
            // release since it was not really removed from the list
            psResultEntry = NULL;
        }
    }

    if (psResultEntry != NULL)
    {
        OSAL.vLinkedListMemoryFree(psResultEntry);
    }

    printf(DSRL_OBJECT_NAME": Object was not added to the list: %p\n", hObject);

    // Item not added,
    return eReturn;
}

/*****************************************************************************
*
*   DSRL_eReplaceEntry
*
* This object interface method is a friend function that will replace the
* object that currently exists in the DSRL with a new object. The object's
* unique ID stays the same and the status will be set to
* DSRL_ENTRY_STATUS_CHANGED
*
* Inputs:
*
*   hDSRL - A valid DSRL
*   hOldEntryData - The existing object to replace in the DSRL
*   hNewEntryData - The new object to use in the DSRL
*
* Outputs:
*
*   DSRL_ADD_REPLACE_OK if a new/duplicate item was added to the list.
*   DSRL_ADD_REPLACE_NOP if a new/duplicate item was not added to list
*   DSRL_ADD_REPLACE_ERROR if there was some error adding item to the list.
*
*****************************************************************************/
DSRL_ADD_REPLACE_RESULT_ENUM DSRL_eReplaceEntry (
    DSRL_OBJECT hDSRL,
    DSRL_ENTRY_OBJECT hOldEntryData,
    DSRL_ENTRY_OBJECT hNewEntryData
        )
{
    DSRL_OBJECT_STRUCT *psObj =
            (DSRL_OBJECT_STRUCT *)hDSRL;
    BOOLEAN bOwner;
    OSAL_RETURN_CODE_ENUM eReturnCode = OSAL_ERROR;
    DSRL_ADD_REPLACE_RESULT_ENUM eReturn = DSRL_ADD_REPLACE_ERROR;

    // Verify inputs. Object handle must be valid.
    bOwner = SMSO_bOwner((SMS_OBJECT)hDSRL);
    if(bOwner == FALSE)
    {
        return DSRL_ADD_REPLACE_ERROR;
    }

    do
    {
        DSRL_ENTRY_STRUCT *psResultEntry;
        OSAL_LINKED_LIST_ENTRY hDSRLListEntry, hNewDSRLListEntry;
        BOOLEAN bSuccess;

        if (hOldEntryData == hNewEntryData)
        {
            // looks like we just need to update the entry data
            printf(DSRL_OBJECT_NAME": performing an update of %p\n",
                hNewEntryData);

            return DSRL_eUpdateEntry(hDSRL, hNewEntryData);
        }

        // get the entry data
        hDSRLListEntry = DSRL_ENTRY_hDSRLListEntry(hOldEntryData, hDSRL);
        if (hDSRLListEntry == OSAL_INVALID_LINKED_LIST_ENTRY)
        {
            // Can't find the old entry, so we can't replace anything
            printf(DSRL_OBJECT_NAME": can't find %p, performing an add "
                "of %p\n", hOldEntryData, hNewEntryData);

            return DSRL_eAddEntry(hDSRL, hNewEntryData);
        }

        printf(DSRL_OBJECT_NAME": found %p\n", hOldEntryData);

        // checking if new entry is not in DSRL already
        hNewDSRLListEntry = DSRL_ENTRY_hDSRLListEntry(hNewEntryData, hDSRL);
        if (hNewDSRLListEntry != OSAL_INVALID_LINKED_LIST_ENTRY)
        {
            // Then its in DSRL already, we should just remove the old entry
            printf(DSRL_OBJECT_NAME": Entry %p already in DSRL, performing "
                "the removal of %p\n", hNewEntryData, hOldEntryData);

            DSRL_vRemoveEntry(hDSRL, hOldEntryData);

            return DSRL_ADD_REPLACE_NOP;
        }

        // Test the replaced entry data now
        eReturn = eTestDSRLEntryData (
            psObj, hNewEntryData,
            FALSE, (DSRL_ENTRY_STRUCT **)NULL);
        if (eReturn != DSRL_ADD_REPLACE_OK)
        {
            break;
        }

        if (psObj->hCompleteAdd != NULL)
        {
            // Give the service-specific manager a chance to do something
            // before this is added to our DSRL.
            hNewEntryData = psObj->hCompleteAdd(hDSRL, hNewEntryData,
                psObj->pvCompleteAddArgs);
            if (hNewEntryData == DSRL_ENTRY_INVALID_OBJECT)
            {
                // They no longer want to add a handle.
                eReturn = DSRL_ADD_REPLACE_NOP;
                break;
            }
        }

        // reusing old entry list element
        psResultEntry = (DSRL_ENTRY_STRUCT *)OSAL.pvLinkedListThis(
            hDSRLListEntry);
        psResultEntry->hObject = hNewEntryData;

        // Replace the entry in the list
        eReturnCode = OSAL.eLinkedListReplaceEntry(
            psObj->hObjectList, hDSRLListEntry,
            psResultEntry);
        if (eReturnCode != OSAL_SUCCESS)
        {
            eReturn = DSRL_ADD_REPLACE_ERROR;
            break;
        }

        printf(DSRL_OBJECT_NAME": replaced %p\n",
            hOldEntryData);

        bSuccess = DSRL_ENTRY_bAddToDSRL(hNewEntryData, hDSRL, 
            hDSRLListEntry);
        if (bSuccess == FALSE)
        {
            // error!
            break;
        }

        // Provide this entry with our manager handle
        DSRL_ENTRY_vSetService(hNewEntryData, psObj->hService);

        if (psObj->eDSRLType == DSRL_TYPE_FAVORITES)
        {
            // This entry is now a favorite
            DSRL_ENTRY_vSetFavorite(hNewEntryData, TRUE);

            // Reset favorite flag for an old entry
            DSRL_ENTRY_vSetFavorite(hOldEntryData, FALSE);
        }

        // Mark that an update happened, but only if we aren't
        // a new entry.  We always want the application to see
        // NEW before anything else.
        if (psResultEntry->eStatus != DSRL_ENTRY_STATUS_NEW)
        {
            psResultEntry->eStatus = DSRL_ENTRY_STATUS_CHANGED;
        }

        DSRL_ENTRY_bRemoveFromDSRL(hOldEntryData, hDSRL);

        printf(DSRL_OBJECT_NAME": replaced %p with %p\n",
            hOldEntryData, hNewEntryData);

        return DSRL_ADD_REPLACE_OK;

    } while (FALSE);

    printf(DSRL_OBJECT_NAME": Can't replace 0x%p -- %d\n", hOldEntryData, eReturnCode);

    return eReturn;
}

/*****************************************************************************
*
*   DSRL_eUpdateEntry
*
* This object interface method is a friend function that will update the
* object that currently exists in the DSRL. The object's
* unique ID stays the same and the status will be set to
* DSRL_ENTRY_STATUS_CHANGED
*
* Inputs:
*
*   hDSRL - A valid DSRL
*   hEntryData - The existing object to update in the DSRL
*
* Outputs:
*
*   DSRL_ADD_REPLACE_OK if a new/duplicate item was added to the list.
*   DSRL_ADD_REPLACE_NOP if a new/duplicate item was not added to list
*   DSRL_ADD_REPLACE_ERROR if there was some error adding item to the list.
*
*****************************************************************************/
DSRL_ADD_REPLACE_RESULT_ENUM DSRL_eUpdateEntry(
    DSRL_OBJECT hDSRL,
    DSRL_ENTRY_OBJECT hObject
        )
{
    DSRL_OBJECT_STRUCT *psObj =
            (DSRL_OBJECT_STRUCT *)hDSRL;
    BOOLEAN bOwner;
    OSAL_RETURN_CODE_ENUM eReturnCode = OSAL_ERROR;
    DSRL_ADD_REPLACE_RESULT_ENUM eReturn = DSRL_ADD_REPLACE_ERROR;

    // Verify inputs. Object handle must be valid.
    bOwner = SMSO_bOwner((SMS_OBJECT)hDSRL);
    if(bOwner == FALSE)
    {
        return DSRL_ADD_REPLACE_ERROR;
    }

    do
    {
        OSAL_LINKED_LIST_ENTRY hDSRLListEntry;
        DSRL_ENTRY_STRUCT *psResultEntry;

        // get the entry data
        hDSRLListEntry = DSRL_ENTRY_hDSRLListEntry(hObject, hDSRL);
        if (hDSRLListEntry == OSAL_INVALID_LINKED_LIST_ENTRY)
        {
            // Can't find the old entry, so we can't replace anything
            printf(DSRL_OBJECT_NAME": can't find, performing an add of %p\n",
                hObject);

            return DSRL_eAddEntry(hDSRL, hObject);
        }

        // getting the struct containing DSRL entry handle
        psResultEntry = (DSRL_ENTRY_STRUCT *)OSAL.pvLinkedListThis(
            hDSRLListEntry);

        // Replace the entry in the list
        eReturnCode = OSAL.eLinkedListReplaceEntry(psObj->hObjectList,
            hDSRLListEntry, psResultEntry);
        if (eReturnCode != OSAL_SUCCESS)
        {
            eReturn = DSRL_ADD_REPLACE_ERROR;
            break;
        }

        // Mark that an update happened, but only if we aren't
        // a new entry.  We always want the application to see
        // NEW before anything else.
        if (psResultEntry->eStatus != DSRL_ENTRY_STATUS_NEW)
        {
            psResultEntry->eStatus = DSRL_ENTRY_STATUS_CHANGED;
        }

        printf(DSRL_OBJECT_NAME": updated %p\n", hObject);

        return DSRL_ADD_REPLACE_OK;

    } while (FALSE);

    printf(DSRL_OBJECT_NAME": Can't update %p -- %d\n", hObject, eReturnCode);

    return eReturn;
}

/*****************************************************************************
*
*   DSRL_vRemoveEntry
*
* This object interface method is a friend function that is used to remove
* an entry from the DSRL.
*
* Inputs:
*
*   hDSRL - A valid DSRL
*   hObject - The object to remove to the list
*
* Outputs:
*
*   Nothing
*
*****************************************************************************/
void DSRL_vRemoveEntry (
    DSRL_OBJECT hDSRL,
    DSRL_ENTRY_OBJECT hObject
        )
{
    DSRL_OBJECT_STRUCT *psObj =
        (DSRL_OBJECT_STRUCT *)hDSRL;
    OSAL_LINKED_LIST_ENTRY hDSRLListEntry;
    BOOLEAN bOwner;

    // Verify inputs. Object handle must be valid.
    bOwner = SMSO_bOwner((SMS_OBJECT)hDSRL);
    if(bOwner == FALSE)
    {
        // Error!
        return;
    }

    // find the entry
    hDSRLListEntry = DSRL_ENTRY_hDSRLListEntry(hObject, 
        (DSRL_OBJECT)psObj);
    if (hDSRLListEntry != OSAL_INVALID_LINKED_LIST_ENTRY)
    {
        DSRL_ENTRY_STRUCT *psResultEntry;

        psResultEntry = (DSRL_ENTRY_STRUCT *)OSAL.pvLinkedListThis(
            hDSRLListEntry);
        printf(DSRL_OBJECT_NAME": found %p\n", psResultEntry->hObject);

        // Mark the entry as removed.
        vRemoveResultEntry(psResultEntry);
    }
    else
    {
        printf(DSRL_OBJECT_NAME": can't remove %p\n", hObject);
    }

    return;
}

/*****************************************************************************
*
*   DSRL_vRemoveAllEntries
*
* This object interface method is a friend function that is used to remove
* all the entries from the DSRL.
*
* Inputs:
*
*   hDSRL - A valid DSRL
*
* Outputs:
*
*   Nothing
*
*****************************************************************************/
void DSRL_vRemoveAllEntries (
    DSRL_OBJECT hDSRL
        )
{
    DSRL_OBJECT_STRUCT *psObj =
        (DSRL_OBJECT_STRUCT *)hDSRL;
    BOOLEAN bOwner;

    // Verify inputs. Object handle must be valid.
    bOwner = SMSO_bOwner((SMS_OBJECT)hDSRL);
    if(bOwner == FALSE)
    {
        // Error!
        return;
    }

    // Empty the list
    OSAL.eLinkedListRemoveAll(psObj->hObjectList, 
        (OSAL_LL_RELEASE_HANDLER)vRemoveAllResultEntries);

    return;
}

/*****************************************************************************
*
*   DSRL_vSetState
*
*   A friend function that updates the state variable, and if it is different,
*   notifies the caller via the callback
*
*****************************************************************************/
void DSRL_vSetState (
    DSRL_OBJECT hDSRL,
    DSRL_STATE_ENUM eState
        )
{
    DSRL_OBJECT_STRUCT *psObj =
        (DSRL_OBJECT_STRUCT *)hDSRL;
    BOOLEAN bOwner;

    // Verify inputs. Object handle must be valid.
    bOwner = SMSO_bOwner((SMS_OBJECT)hDSRL);
    if(bOwner == FALSE)
    {
        // Error!
        return;
    }

    if ( psObj->eState != eState)
    {
        DSRL_STATE_ENUM eLastState = psObj->eState;

        // update state value
        psObj->eState = eState;

        // Update our state attributes now
        vUpdateStateAttributes(psObj, eLastState);

        if (eState == DSRL_STATE_READY)
        {
            // Prepare the DSRL for the ready state
            vPrepareForReady(psObj);
        }

        // Only invoke the callback if:
        // 1) We aren't stopping, or
        // 2) We are stopping and we're reporting STOPPED
        // and the callback is valid
        if (  ( (psObj->bStopping == FALSE)
                ||
                ((psObj->bStopping == TRUE) && (eState == DSRL_STATE_STOPPED))
              )
              && (psObj->vCallback != NULL)
           )
        {
            // Notify the listener
            psObj->vCallback(
                hDSRL, psObj->eState, psObj->pvCallbackArgs);
        }
    }

    return;
}

/*****************************************************************************
*
*   DSRL_vSetType
*
*****************************************************************************/
void DSRL_vSetType (
    DSRL_OBJECT hDSRL,
    DATASERVICE_TYPE_ENUM eType
        )
{
    DSRL_OBJECT_STRUCT *psObj =
        (DSRL_OBJECT_STRUCT *)hDSRL;
    BOOLEAN bOwner;

    // Verify inputs. Object handle must be valid.
    bOwner = SMSO_bOwner((SMS_OBJECT)hDSRL);
    if(TRUE == bOwner)
    {
        psObj->eType = eType;
    }

    return;
}

/*****************************************************************************
*
*   DSRL_vTargetsUpdated
*
*****************************************************************************/
void DSRL_vTargetsUpdated (
    DSRL_OBJECT hDSRL
        )
{
    DSRL_OBJECT_STRUCT *psObj =
        (DSRL_OBJECT_STRUCT *)hDSRL;
    BOOLEAN bOwner;

    // Verify inputs. Object handle must be valid.
    bOwner = SMSO_bOwner((SMS_OBJECT)hDSRL);
    if(TRUE == bOwner)
    {
        // Update the staged value for this attribute
        psObj->asStateAttribs[
            DSRL_TARGET_UPDATE_ATTRIB_INDEX].bStagedValue = TRUE;

        // Is this DSRL's registered callback using
        // the device position and has any sort function
        // active?
        if ((TRUE == psObj->bSortUsesDeviceLocation) && 
            (psObj->n16Sort != NULL))
        {
            // Re-sort all of this DSRL's contents
            vFullSort(psObj);
        }
    }

    return;
}

/*****************************************************************************
*
*   DSRL_eIterate
*
* This  function is provided to allow data service managers the ability to
* iterate their DSRLs no matter what state they're in.  The caller must
* own the DSRL object.
*
*****************************************************************************/
SMSAPI_RETURN_CODE_ENUM DSRL_eIterate (
    DSRL_OBJECT hDSRL,
    DSRL_ITERATOR_CALLBACK bEntryIterator,
    void *pvIteratorArg
        )
{
    BOOLEAN bLocked;
    DSRL_OBJECT_STRUCT *psObj =
        (DSRL_OBJECT_STRUCT *)hDSRL;
    SMSAPI_RETURN_CODE_ENUM eReturnCode;

    // Verify and lock SMS Object
    bLocked =
        SMSO_bLock((SMS_OBJECT)hDSRL, OSAL_OBJ_TIMEOUT_INFINITE);
    if(bLocked == FALSE)
    {
        // Unable to lock the provided object
        return SMSAPI_RETURN_CODE_ERROR;
    }

    // Call the local iterator -- don't
    // worry about what state we're in
    eReturnCode = eIterateLocal(psObj, FALSE, bEntryIterator, pvIteratorArg);

    SMSO_vUnlock((SMS_OBJECT)hDSRL);

    return eReturnCode;
}

/*****************************************************************************
*
*   DSRL_psCreateArg
*
*****************************************************************************/
DSRL_ARG_STRUCT *DSRL_psCreateArg (
    size_t tNumTargets
        )
{
    DSRL_ARG_STRUCT *psArg;
    size_t tAdditionalSpace = sizeof(DSRL_TARGET_OBJECT) * tNumTargets;

    // Create the target argument for the caller
    psArg = (DSRL_ARG_STRUCT *)
        SMSO_hCreate (
            DSRL_OBJECT_NAME":EvtTgtArgs",
            sizeof(DSRL_ARG_STRUCT) + tAdditionalSpace,
            SMS_INVALID_OBJECT, FALSE );

    return psArg;
}

/*****************************************************************************
*
*   DSRL_vDestroyArg
*
*****************************************************************************/
void DSRL_vDestroyArg (
    DSRL_ARG_STRUCT *psArg
        )
{
    if (psArg != NULL)
    {
        size_t tCurTarget;

        // Iterate the size of the target list and destroy the targets
        for (tCurTarget = 0; tCurTarget < psArg->tNumTargets; tCurTarget++)
        {
            // If it is null, this function will handle that
            DSRL_TARGET_vDestroyByType(
                (DSRL_TARGET_OBJECT)psArg->ahTargetList[tCurTarget]);
        }

        SMSO_vDestroy((SMS_OBJECT)psArg);
    }

    return;
}

/*****************************************************************************
*
*   DSRL_pvServiceData
*
*   This function returns pointer to DSRL service data. NULL will be returned
*   in case of any error or if service data has not been allocated/set
*
*****************************************************************************/
void *DSRL_pvServiceData (
    DSRL_OBJECT hDSRL
        )
{
    BOOLEAN bOwner;
    void *pvServiceData = NULL;

    // Verify and lock SMS Object
    bOwner =
        SMSO_bOwner((SMS_OBJECT)hDSRL);
    if (bOwner == TRUE)
    {
        DSRL_OBJECT_STRUCT *psObj =
            (DSRL_OBJECT_STRUCT *)hDSRL;

        pvServiceData = psObj->pvServiceData;
    }

    return pvServiceData;
}

/*****************************************************************************
                             PRIVATE FUNCTIONS
*****************************************************************************/

/*****************************************************************************
*
*   psCreate
*
*****************************************************************************/
static DSRL_OBJECT_STRUCT *psCreate (
    DATASERVICE_MGR_OBJECT hManager,
    DSRL_TYPE_ENUM eType,
    size_t tCapacity,
    DSRL_SORT_FUNCTION n16Sort,
    void *pvSortArgs,
    DSRL_FILTER_FUNCTION bFilterItem,
    void *pvFilterArgs,
    DSRL_CALLBACK vCallback,
    void *pvCallbackArgs
    )
{
    DSRL_OBJECT_STRUCT *psObj =
        (DSRL_OBJECT_STRUCT *)NULL;
    OSAL_RETURN_CODE_ENUM eReturnCode;

    do
    {
        void *pvServiceData = NULL;

        // Ask the data service to create our dsrl object
        psObj = (DSRL_OBJECT_STRUCT *)
            DATASERVICE_IMPL_hCreateDSRL(
                (DATASERVICE_IMPL_HDL)hManager,
                sizeof(DSRL_OBJECT_STRUCT),
                DSRL_OBJECT_NAME,
                eType, &pvServiceData);

        if(psObj == NULL)
        {
            // Error!
            break;
        }

        // Initialize all state-dependent attributes
        vInitializeStateAttributes(psObj);

        eReturnCode = OSAL.eLinkedListCreate(
            &psObj->hObjectList,
            DSRL_OBJECT_NAME":EntryList",
            (OSAL_LL_COMPARE_HANDLER)NULL,
            OSAL_LL_OPTION_LINEAR | OSAL_LL_OPTION_USE_PRE_ALLOCATED_ELEMENTS
                );

        if(eReturnCode != OSAL_SUCCESS)
        {
            // Error!
            break;
        }

        eReturnCode = OSAL.eLinkedListCreate(
            &psObj->hRemovedObjList,
            DSRL_OBJECT_NAME":RemEntryList",
            (OSAL_LL_COMPARE_HANDLER)NULL,
            OSAL_LL_OPTION_USE_PRE_ALLOCATED_ELEMENTS
                );

        if(eReturnCode != OSAL_SUCCESS)
        {
            // Error!
            break;
        }

        // Setup member variables
        psObj->hService = (DATASERVICE_IMPL_HDL)hManager;
        psObj->eType = DATASERVICE_TYPE_UNKNOWN;
        psObj->eDSRLType = eType;
        psObj->eState = DSRL_STATE_UNKNOWN;
        psObj->n16Sort = n16Sort;
        psObj->pvSortArgs = pvSortArgs;
        psObj->bFilterItem = bFilterItem;
        psObj->pvFilterArgs = pvFilterArgs;
        psObj->vCallback = vCallback;
        psObj->pvCallbackArgs = pvCallbackArgs;
        psObj->tCurrentID = DSRL_ENTRY_INVALID_ID;
        psObj->vFinalize = NULL;
        psObj->pvFinalizeArgs = NULL;
        psObj->bStopping = FALSE;
        psObj->bSortUsesDeviceLocation = FALSE;

        // Set the reported capacity and compute the
        // working capacity
        psObj->tReportedCapacity = tCapacity;
        psObj->tWorkingCapacity = DSRL_COMPUTE_WORKING_CAPACITY(tCapacity);
        psObj->pvServiceData = pvServiceData;

        // enabling sort function for DSRL
        if (psObj->n16Sort != NULL)
        {
            eReturnCode = OSAL.eLinkedListSort(psObj->hObjectList, 
                (OSAL_LL_COMPARE_HANDLER)n16SortShim);
            if (eReturnCode != OSAL_SUCCESS)
            {
                // error!
                break;
            }
        }

        return psObj;
    } while (FALSE);

    // This didn't work out
    DSRL_vDestroy((DSRL_OBJECT)psObj);

    return (DSRL_OBJECT_STRUCT *)NULL;
}

/*****************************************************************************
*
*   bPostTargetArgEvent
*
*****************************************************************************/
static BOOLEAN bPostTargetArgEvent (
    DSRL_OBJECT_STRUCT *psObj,
    DSRL_ACTION_ENUM eAction,
    DSRL_MODIFY_OPERATION_ENUM eOperation,
    DSRL_TARGET_OBJECT *phTargetList,
    size_t tNumTargets
    )
{
    DSRL_ARG_STRUCT *psEventArg;
    DSRL_TARGET_TYPE_ENUM eTargetType;
    DSRL_TARGET_OBJECT hTarget;
    BOOLEAN bSuccess = TRUE;

    if ( (psObj->eState == DSRL_STATE_ERROR) &&
         (eAction != DSRL_ACTION_REMOVE) )
    {
        // We can't work on ERROR'd lists unless
        // we are destroying them.
        return FALSE;
    }

    // Create memory for our argument to be passed along with the event
    psEventArg = DSRL_psCreateArg( tNumTargets );
    if (psEventArg == NULL)
    {
        // We can't allocate memory for this argument!
        return FALSE;
    }

    // Set arguments
    psEventArg->eAction = eAction;
    psEventArg->eDSRLType = psObj->eDSRLType;
    psEventArg->hDSRL = (DSRL_OBJECT)psObj;
    psEventArg->tNumTargets = tNumTargets;

    // Make sure to specify the additional arguments if this is a modify event
    if (eAction == DSRL_ACTION_MODIFY)
    {
        // Perform this operation
        psEventArg->uAction.sModify.eModifyType = eOperation;

        // Force a state change since this comes from the app and
        // they expect to see the dsrl do something if they
        // modified it via a public API
        psEventArg->uAction.sModify.bForceDSRLStateChange = TRUE;
    }

    // We only need to deal with a target list if
    // this is an add or a modify event
    if ( ((eAction == DSRL_ACTION_MODIFY) ||
          (eAction == DSRL_ACTION_ADD))
         &&
         (tNumTargets != 0))
    {
        size_t tCurTarget;
        BOOLEAN bDeviceLoc;

        // Iterate through all the targets
        for (tCurTarget = 0; tCurTarget < psEventArg->tNumTargets; tCurTarget++)
        {
            // Grab the handle
            hTarget = phTargetList[tCurTarget];

            if (hTarget == DSRL_TARGET_INVALID_OBJECT)
            {
                bSuccess = FALSE;
                break;
            }

            // Get the current target type
            eTargetType = DSRL_TARGET.eType(hTarget);

            // Location-based targets
            if (eTargetType == DSRL_TARGET_TYPE_LOCATION)
            {
                // Verify correct use of device location
                bDeviceLoc = LOCATION_bIsDevicePosition(hTarget);
                if (bDeviceLoc == TRUE)
                {
                    // This must be a device dsrl and this
                    // must be a dsrl creation event
                    if ((psObj->eDSRLType != DSRL_TYPE_DEVICE) ||
                        (eAction != DSRL_ACTION_ADD))
                    {
                        bSuccess = FALSE;
                        break;
                    }

                    // Take the device location target since it is
                    // a copy that we made privately
                    psEventArg->ahTargetList[tCurTarget] = hTarget;
                }
                else
                {
                    // Duplicate the target and use that
                    psEventArg->ahTargetList[tCurTarget] =
                        DSRL_TARGET_hDuplicate(SMS_INVALID_OBJECT, hTarget);
                }
            }
            // Stock symbol targets
            else if (eTargetType == DSRL_TARGET_TYPE_STOCK_SYMBOL)
            {
                // Duplicate the target and use that
                psEventArg->ahTargetList[tCurTarget] =
                    DSRL_TARGET_hDuplicate(SMS_INVALID_OBJECT, hTarget);
            }
            else
            {
                bSuccess = FALSE;
                break;
            }

            // Validate whatever it is we just copied
            if (psEventArg->ahTargetList[tCurTarget] == DSRL_TARGET_INVALID_OBJECT)
            {
                bSuccess = FALSE;
                break;
            }
        }
    }

    if (bSuccess == TRUE)
    {
        // Inform the data service of the DSRL update
        bSuccess = DATASERVICE_IMPL_bDSRLUpdate(
            psObj->hService,
            psEventArg);
    }

    if (bSuccess == FALSE)
    {
        // can't post the event
        DSRL_vDestroyArg(psEventArg);
    }

    return bSuccess;
}

/*****************************************************************************
*
*   eIterateLocal
*
* This is the main iteration function which provides iteration services to
* both the public API and the friend function utilized within SMS.
*
*****************************************************************************/
static SMSAPI_RETURN_CODE_ENUM eIterateLocal (
    DSRL_OBJECT_STRUCT *psObj,
    BOOLEAN bEnforceReady,
    DSRL_ITERATOR_CALLBACK bEntryIterator,
    void *pvIteratorArg
        )
{
    DSRL_ITERATOR_STRUCT sIterator;
    OSAL_RETURN_CODE_ENUM eOsalReturnCode;
    SMSAPI_RETURN_CODE_ENUM eReturnCode =
        SMSAPI_RETURN_CODE_ERROR;

    // If we're enforcing the dsrl to be ready
    // then ensure our state indicates we are
    if (bEnforceReady == TRUE)
    {
        if ((psObj->eState != DSRL_STATE_READY) ||
            (psObj->bStopping == TRUE))
        {
            return SMSAPI_RETURN_CODE_API_NOT_ALLOWED;
        }
    }

    if (bEntryIterator == NULL)
    {
        return SMSAPI_RETURN_CODE_INVALID_INPUT;
    }

    // Fill out our iterator struct
    sIterator.hDSRL = (DSRL_OBJECT)psObj;
    sIterator.bIterator = bEntryIterator;
    sIterator.pvIteratorArg = pvIteratorArg;
    sIterator.bContinue = TRUE;

    // Only prune the list if we're utilizing
    // the public iterator
    sIterator.bPrune = bEnforceReady;

    // Begin iterating the entries we have
    eOsalReturnCode = OSAL.eLinkedListIterate(
        psObj->hObjectList,
        (OSAL_LL_ITERATOR_HANDLER)bResultEntryIterator,
        (void *)&sIterator);

    // Continue iteration of removed list
    // only if we didn't experience an error
    if ((eOsalReturnCode == OSAL_SUCCESS) ||
        (eOsalReturnCode == OSAL_NO_OBJECTS))
    {
        // Did the iterator tell us to continue
        // iterating?
        if (sIterator.bContinue == TRUE)
        {
            // Yes! Iterate the removed objects now
            eOsalReturnCode = OSAL.eLinkedListIterate(
                psObj->hRemovedObjList,
                (OSAL_LL_ITERATOR_HANDLER)bRemovedEntriesIterator,
                (void *)&sIterator);
        }

        if ((eOsalReturnCode == OSAL_SUCCESS) ||
            (eOsalReturnCode == OSAL_NO_OBJECTS))
        {
            eReturnCode = SMSAPI_RETURN_CODE_SUCCESS;
        }
    }

    return eReturnCode;
}

/*****************************************************************************
*
*   bRemovedEntriesIterator
*
*****************************************************************************/
static BOOLEAN bRemovedEntriesIterator(
    DSRL_ENTRY_STRUCT *psEntry,
    DSRL_ITERATOR_STRUCT *psIterator
        )
{
    // Return whatever their callback returns back to the iterator
    psIterator->bContinue = psIterator->bIterator(
        psIterator->hDSRL, psEntry->tID, DSRL_ENTRY_STATUS_REMOVED,
        DSRL_ENTRY_INVALID_OBJECT, psIterator->pvIteratorArg);

    if (psIterator->bPrune == TRUE)
    {
        OSAL_RETURN_CODE_ENUM eReturnCode;

        printf(DSRL_OBJECT_NAME": destroying entry struct ID %u\n", 
            psEntry->tID);

        // Since they are done, call the prune 
        // function to remove the entry data
        eReturnCode = OSAL.eLinkedListRemove(psEntry->hEntry);
        if (eReturnCode == OSAL_SUCCESS)
        {
           OSAL.vLinkedListMemoryFree(psEntry);
        }
        else
        {
            return FALSE;
        }
    }

    return psIterator->bContinue;
}

/*****************************************************************************
*
*   eTestDSRLEntryData
*
*****************************************************************************/
static DSRL_ADD_REPLACE_RESULT_ENUM eTestDSRLEntryData (
    DSRL_OBJECT_STRUCT *psObj,
    DSRL_ENTRY_OBJECT hObject,
    BOOLEAN bTestingNewEntry,
    DSRL_ENTRY_STRUCT **ppsEntryToRemove
        )
{
    DSRL_ADD_REPLACE_RESULT_ENUM eReturn = DSRL_ADD_REPLACE_ERROR;

    // Are we testing a new entry?
    if (TRUE == bTestingNewEntry)
    {
        OSAL_LINKED_LIST_ENTRY hDSRLListEntry;

        // See if this entry exists
        hDSRLListEntry = DSRL_ENTRY_hDSRLListEntry(hObject, 
            (DSRL_OBJECT)psObj);

        // Did we find something?
        if (OSAL_INVALID_LINKED_LIST_ENTRY != hDSRLListEntry)
        {
            // This entry already exists!
            return DSRL_ADD_REPLACE_NOP;
        }
    }
    else // Test an existing entry
    {
        // Don't need to search for it. We just need
        // to validate the entry data against the filter.
    }

    do
    {
        // See if this passes the provided filter function
        // but only call the filter function if we're not stopping
        if ((psObj->bFilterItem != NULL) && (psObj->bStopping == FALSE))
        {
            BOOLEAN bAddItem;

            // Pass this entry through the filter now
            bAddItem = psObj->bFilterItem(
                    (DSRL_OBJECT)psObj, hObject, psObj->pvFilterArgs);

            if (bAddItem == FALSE)
            {
                // We can't add this item, so return
                printf(DSRL_OBJECT_NAME": Object failed Application's filter 0x%p\n", hObject);
                eReturn = DSRL_ADD_REPLACE_NOP;
                break;
            }
        }

        // We need to do a sort/capacity check only if we're
        // attempting to add a new entry to the list
        if (TRUE == bTestingNewEntry)
        {
            OSAL_RETURN_CODE_ENUM eReturnCode;
            UN32 un32NumItems;

            // How many items in this list now?
            eReturnCode = OSAL.eLinkedListItems(
                psObj->hObjectList, &un32NumItems);

            if (eReturnCode != OSAL_SUCCESS)
            {
                printf(DSRL_OBJECT_NAME": eLinkedListItems %d\n", eReturnCode);
                eReturn = DSRL_ADD_REPLACE_ERROR;
                break;
            }

            // Are we at our limit?
            if (un32NumItems >= psObj->tWorkingCapacity)
            {
                DSRL_ENTRY_STRUCT *psLastEntry = NULL;
                N16 n16Compare;

                // We are already at capacity

                // If we are sorted, there is a chance that this item
                // may knock another below our capacity line, Biggest Loser style
                if (psObj->n16Sort == NULL)
                {
                    // we don't have a sort function, so move on
                    eReturn = DSRL_ADD_REPLACE_NOP;
                    break;
                }

                // Compare this item to our last item in the list to find
                // out if this entry is now sorted out or if the current
                // last entry will get sorted out
                OSAL.hLinkedListLast(psObj->hObjectList, (void**)&psLastEntry);

                if ((DSRL_ENTRY_STRUCT *)NULL == psLastEntry)
                {
                    // Odd?  We think so
                    eReturn = DSRL_ADD_REPLACE_ERROR;
                    break;
                }

                // the result entry to our object
                n16Compare = psObj->n16Sort(
                    (DSRL_OBJECT)psObj,
                    psLastEntry->hObject, hObject, psObj->pvSortArgs);
                if (n16Compare <= 0)
                {
                    // Our old entry doesn't meet the sort requirements
                    eReturn = DSRL_ADD_REPLACE_NOP;
                    break;
                }

                // Our last entry is greater than our new object, so indicate
                // that the entry should be removed.
                // This will free up a spot for our new item.
                *ppsEntryToRemove = psLastEntry;
            }
        }

        // We passed all our checks
        return DSRL_ADD_REPLACE_OK;

    } while (FALSE);

    // A check failed (or an error occurred)
    return eReturn;
}

/*****************************************************************************
*
*   vPrepareForReady
*
*****************************************************************************/
static void vPrepareForReady (
    DSRL_OBJECT_STRUCT *psObj
        )
{

    // Trim the contents to meet the capacity constraint

    do 
    {
        OSAL_RETURN_CODE_ENUM eReturnCode;
        UN32 un32Items;
        OSAL_LINKED_LIST_ENTRY hEntry = OSAL_INVALID_LINKED_LIST_ENTRY;
        DSRL_ENTRY_STRUCT *psEntry;

        if (psObj->tReportedCapacity == SIZE_T_MAX)
        {
            break;
        }

        // check if list is out of capacity
        eReturnCode = OSAL.eLinkedListItems(psObj->hObjectList, &un32Items);
        if (eReturnCode != OSAL_SUCCESS)
        {
            printf(DSRL_OBJECT_NAME": Failed to get number of entries\n");
            break;
        }

        if (un32Items <= psObj->tReportedCapacity)
        {
            printf(DSRL_OBJECT_NAME": Number of items is %u which is less or "
                "equal to capacity %u\n", un32Items, psObj->tReportedCapacity);
            break;
        }

        hEntry = OSAL.hLinkedListLast(psObj->hObjectList, (void**)&psEntry);
        while ((hEntry != OSAL_INVALID_LINKED_LIST_ENTRY) &&
            (un32Items-- > psObj->tReportedCapacity))
        {
            vRemoveResultEntry(psEntry);
            hEntry = OSAL.hLinkedListLast(psObj->hObjectList, 
                (void**)&psEntry);
        }

    } while (FALSE);

    return;
}

/*****************************************************************************
*
*   vFullSort
*
*****************************************************************************/
static void vFullSort (
    DSRL_OBJECT_STRUCT *psObj
        )
{
    OSAL_RETURN_CODE_ENUM eReturnCode;

    // The sort callback will tell us if it's
    // using the device location API
    psObj->bSortUsesDeviceLocation = FALSE;

    if (psObj->n16Sort != NULL)
    {
        // Sort the list now with these new parameters
        eReturnCode = OSAL.eLinkedListSort(psObj->hObjectList,
            (OSAL_LL_COMPARE_HANDLER)n16SortShim);
    }
    else
    {
        // Sort the list now with ID comparator
        eReturnCode = OSAL.eLinkedListSort(psObj->hObjectList,
            (OSAL_LL_COMPARE_HANDLER)n16CompareEntryId);
    }

    if (OSAL_SUCCESS != eReturnCode)
    {
        puts(DSRL_OBJECT_NAME": unable to sort list");
    }

    return;
}

/*****************************************************************************
*
*   vInitializeStateAttributes
*
*****************************************************************************/
static void vInitializeStateAttributes (
    DSRL_OBJECT_STRUCT *psObj
        )
{
    size_t tIndex;

    // Initialize all of these attributes
    for (tIndex = 0; tIndex < DSRL_NUM_STATE_DEPENDENT_ATTRIBS; tIndex++)
    {
        // Intialize the value of the attrib
        psObj->asStateAttribs[tIndex].bValue =
            psObj->asStateAttribs[tIndex].bInitializerValue;

        // Intialize the staged value as well
        psObj->asStateAttribs[tIndex].bStagedValue =
            psObj->asStateAttribs[tIndex].bInitializerValue;

        psObj->asStateAttribs[tIndex].eReportingState = DSRL_STATE_READY;
    }

    return;
}

/*****************************************************************************
*
*   vUpdateStateAttributes
*
*   Iterate through all state-dependent attributes. Initialize an attribute
*   if we've exited the reporting state for it
*
*****************************************************************************/
static void vUpdateStateAttributes (
    DSRL_OBJECT_STRUCT *psObj,
    DSRL_STATE_ENUM eLastState
        )
{
    size_t tIndex;

    // Iterate all state dependent attributes
    for (tIndex = 0; tIndex < DSRL_NUM_STATE_DEPENDENT_ATTRIBS; tIndex++)
    {
        // We must reset the value of all attributes which report
        // in the state we just left
        if (eLastState == psObj->asStateAttribs[tIndex].eReportingState)
        {
            // Reset the value
            psObj->asStateAttribs[tIndex].bValue =
                psObj->asStateAttribs[tIndex].bInitializerValue;
        }

        // Activate the staged value of all attributes which report
        // in the state we're entering
        if (psObj->eState == psObj->asStateAttribs[tIndex].eReportingState)
        {
            // Apply the staged value
            psObj->asStateAttribs[tIndex].bValue =
                psObj->asStateAttribs[tIndex].bStagedValue;

            // Reset the staged value now
            psObj->asStateAttribs[tIndex].bStagedValue =
                psObj->asStateAttribs[tIndex].bInitializerValue;

        }
    }

    return;
}

/*****************************************************************************
*
*   bApplyNewFilter
*
*****************************************************************************/
static BOOLEAN bApplyNewFilter (
    DSRL_OBJECT_STRUCT *psObj
        )
{
    OSAL_RETURN_CODE_ENUM eReturnCode;
    BOOLEAN bSuccess = TRUE;

    // Iterate the list to put all the entries through
    // the new filter
    eReturnCode = OSAL.eLinkedListIterate(psObj->hObjectList, 
        (OSAL_LL_ITERATOR_HANDLER)bApplyNewFilterToEntry, &bSuccess);

    if ((eReturnCode != OSAL_SUCCESS) && (eReturnCode != OSAL_NO_OBJECTS))
    {
        // Make sure this is false
        bSuccess = FALSE;
    }

    return bSuccess;
}

/*****************************************************************************
*
*   bResultEntryIterator
*
*****************************************************************************/
static BOOLEAN bResultEntryIterator(
    DSRL_ENTRY_STRUCT *psEntry,
    DSRL_ITERATOR_STRUCT *psIterator
        )
{
    // Return whatever their callback returns back to the iterator
    psIterator->bContinue = psIterator->bIterator (
        psIterator->hDSRL, psEntry->tID, psEntry->eStatus,
        psEntry->hObject, psIterator->pvIteratorArg);

    if (psIterator->bPrune == TRUE)
    {
        // Since they are done, call the prune 
        // function to reset the entry status
        vPruneResultEntry(psEntry);
    }

    return psIterator->bContinue;
}

/*****************************************************************************
*
*   n16SortShim
*
*****************************************************************************/
static N16 n16SortShim (
    DSRL_ENTRY_STRUCT *psDSRLEntry1,
    DSRL_ENTRY_STRUCT *psDSRLEntry2
        )
{
    N16 n16Return = N16_MIN;

    if ((psDSRLEntry1 != NULL) && (psDSRLEntry2 != NULL))
    {
        if ((NULL != psDSRLEntry1->psDSRL->n16Sort) && (
            (FALSE == psDSRLEntry1->psDSRL->bStopping)))
        {
            // Use the provided sort function
            n16Return = psDSRLEntry1->psDSRL->n16Sort(
                (DSRL_OBJECT)psDSRLEntry1->psDSRL, 
                psDSRLEntry1->hObject, 
                psDSRLEntry2->hObject, 
                psDSRLEntry1->psDSRL->pvSortArgs);
        }
    }

    return n16Return;
}

/*****************************************************************************
*
*   n16CompareEntryId
*
*****************************************************************************/
static N16 n16CompareEntryId (
    DSRL_ENTRY_STRUCT *psListEntry1,
    DSRL_ENTRY_STRUCT *psListEntry2
        )
{
    N16 n16Return = N16_MIN;

    if ((psListEntry1 != NULL) && (psListEntry2 != NULL))
    {
        n16Return = COMPARE(psListEntry1->tID, psListEntry2->tID);
    }

    return n16Return;
}

/*****************************************************************************
*
*   vPruneResultEntry
*
*****************************************************************************/
static void vPruneResultEntry (
    DSRL_ENTRY_STRUCT *psResultEntry
        )
{
    if (psResultEntry != NULL)
    {
        switch (psResultEntry->eStatus)
        {
            case DSRL_ENTRY_STATUS_CHANGED:
            case DSRL_ENTRY_STATUS_UNCHANGED:
            case DSRL_ENTRY_STATUS_NEW:
            {
                // Set the state to the UNCHANGED state
                psResultEntry->eStatus = DSRL_ENTRY_STATUS_UNCHANGED;
            }
            break;

            case DSRL_ENTRY_STATUS_REMOVED:
            {
                //nothing to do
            }
            break;

            case DSRL_ENTRY_STATUS_UNKNOWN:
            default:
            {
                puts(DSRL_OBJECT_NAME": Error, pruning entry of unknown state");
            }
            break;
        }
    }

    return;
}

/*****************************************************************************
*
*   vDestroyResultEntry
*
*****************************************************************************/
static void vDestroyResultEntry (
    DSRL_ENTRY_STRUCT *psResultEntry
        )
{
    // Validate pointer
    if (NULL == psResultEntry)
    {
        return;
    }

    // unlinking the entry from DSRL
    DSRL_ENTRY_bRemoveFromDSRL(psResultEntry->hObject, 
        (DSRL_OBJECT)psResultEntry->psDSRL);

    // We are removing an entry that the app never saw
    // so just get rid of it
    OSAL.vLinkedListMemoryFree(psResultEntry);

    return;
}

/*****************************************************************************
*
*   vRemoveResultEntry
*
*****************************************************************************/
static void vRemoveResultEntry (
    DSRL_ENTRY_STRUCT *psResultEntry
        )
{
    // Validate pointer
    if (NULL == psResultEntry)
    {
        return;
    }

    // unlinking the entry from DSRL
    DSRL_ENTRY_bRemoveFromDSRL(psResultEntry->hObject, 
        (DSRL_OBJECT)psResultEntry->psDSRL);

    // if entry is removed from favorites DSRL
    // lets reset favorite flag
    if (psResultEntry->psDSRL->eDSRLType == DSRL_TYPE_FAVORITES)
    {
        DSRL_ENTRY_vSetFavorite(psResultEntry->hObject, FALSE);
    }

    // Call the finalizer method, if provided
    if (psResultEntry->psDSRL->vFinalize != NULL)
    {
        psResultEntry->psDSRL->vFinalize((DSRL_OBJECT)psResultEntry->psDSRL,
            psResultEntry->hObject, psResultEntry->psDSRL->pvFinalizeArgs);
    }

    // Remove this entry from the main object list
    OSAL.eLinkedListRemove(psResultEntry->hEntry);

    if (psResultEntry->eStatus == DSRL_ENTRY_STATUS_NEW)
    {
        // We are removing an entry that the app never saw
        // so just get rid of it
        OSAL.vLinkedListMemoryFree(psResultEntry);
    }
    else
    {
        printf(DSRL_OBJECT_NAME": removed entry %p; adding to removed objects "
            "list\n", psResultEntry->hObject);

        // pre-cleaning the entry handle
        psResultEntry->hEntry = OSAL_INVALID_LINKED_LIST_ENTRY;

        // Add tID to the removed entries list
        OSAL.eLinkedListAdd(psResultEntry->psDSRL->hRemovedObjList, 
            &psResultEntry->hEntry, psResultEntry);
    }

    return;
}

/*****************************************************************************
*
*   vRemoveAllResultEntries
*
*   Should be used for OSAL.eLinkedListRemoveAll
*
*****************************************************************************/
static void vRemoveAllResultEntries (
    DSRL_ENTRY_STRUCT *psResultEntry
        )
{
    // Validate pointer
    if (NULL == psResultEntry)
    {
        return;
    }

    // unlinking the entry from DSRL
    DSRL_ENTRY_bRemoveFromDSRL(psResultEntry->hObject, 
        (DSRL_OBJECT)psResultEntry->psDSRL);

    // if entry is removed from favorites DSRL
    // lets reset favorite flag
    if (psResultEntry->psDSRL->eDSRLType == DSRL_TYPE_FAVORITES)
    {
        DSRL_ENTRY_vSetFavorite(psResultEntry->hObject, FALSE);
    }

    // Call the finalizer method, if provided
    if (psResultEntry->psDSRL->vFinalize != NULL)
    {
        psResultEntry->psDSRL->vFinalize((DSRL_OBJECT)psResultEntry->psDSRL,
            psResultEntry->hObject, psResultEntry->psDSRL->pvFinalizeArgs);
    }

    if (psResultEntry->eStatus == DSRL_ENTRY_STATUS_NEW)
    {
        // We are removing an entry that the app never saw
        // so just get rid of it
        OSAL.vLinkedListMemoryFree(psResultEntry);
    }
    else
    {
        printf(DSRL_OBJECT_NAME": removed entry %p; adding to removed objects "
            "list\n", psResultEntry->hObject);

        // pre-cleaning the entry handle
        psResultEntry->hEntry = OSAL_INVALID_LINKED_LIST_ENTRY;

        // Add tID to the removed entries list
        OSAL.eLinkedListAdd(psResultEntry->psDSRL->hRemovedObjList, 
            &psResultEntry->hEntry, psResultEntry);
    }

    return;
}

/*****************************************************************************
*
*   bApplyNewFilterToEntry
*
*****************************************************************************/
static BOOLEAN bApplyNewFilterToEntry (
    DSRL_ENTRY_STRUCT *psResultEntry,
    BOOLEAN *pbSuccess
        )
{
    DSRL_ADD_REPLACE_RESULT_ENUM eResult;

    // Test this entry data now
    eResult = eTestDSRLEntryData(psResultEntry->psDSRL, 
        psResultEntry->hObject, FALSE, NULL);
    if (eResult == DSRL_ADD_REPLACE_NOP)
    {
        // Entry is filtered out. Lets remove it from DSRL
        vRemoveResultEntry(psResultEntry);
    }
    else if (eResult == DSRL_ADD_REPLACE_ERROR)
    {
        *pbSuccess = FALSE;
    }

    return *pbSuccess;
}

