/******************************************************************************/
/*                    Copyright (c) Sirius XM Radio, Inc.                     */
/*                            All Rights Reserved                             */
/*         Licensed Materials - Property of Sirius XM Radio, Inc.             */
/******************************************************************************/
/*******************************************************************************
 *
 * DESCRIPTION
 *
 *  This module contains the Object:DISTANCE implementation for the
 *  Sirius Module Services (SMS)
 *
 ******************************************************************************/
#include <string.h>

#include "standard.h"
#include "osal.h"

#include "sms_api.h"
#include "sms_obj.h"

#include "distance_obj.h"
#include "_distance_obj.h"


/*****************************************************************************
                             PUBLIC FUNCTIONS
*****************************************************************************/

/*****************************************************************************
*
*   hCreate
*
* This object interface method is used by the caller to create a DISTANCE
* object that represents a whole value distance of a type of units.
*
* Inputs:
*
*   n32Value - The whole value for the distance object specfied in eUnits
*   eUnits - The type of units that un32value is.
*
* Outputs:
*
*   A handle to a DISTANCE object on success
*   A DISTANCE_INVALID_OBJECT on failrue
*
*****************************************************************************/
static DISTANCE_OBJECT hCreate (
    N32 n32Value,
    DISTANCE_UNIT_TYPE_ENUM eUnits
        )
{
    DISTANCE_OBJECT hDistance = DISTANCE_INVALID_OBJECT;
    OSAL_FIXED_OBJECT hFixedValue;
    OSAL_FIXED_OBJECT_DATA atData[OSAL_FIXED_OBJECT_SIZE];

    // Checking input parameters first
    if ((n32Value >= 0) && 
        ((eUnits == DISTANCE_UNIT_TYPE_MILES) ||
         (eUnits == DISTANCE_UNIT_TYPE_KILOMETERS)))
    {
        hFixedValue = OSAL_FIXED.hCreateInMemory(n32Value, 0, atData);

        hDistance = DISTANCE_hCreateFromFixed(hFixedValue, eUnits);
    }

    return hDistance;
}

/*****************************************************************************
*
*   hDuplicate
*
* This object interface method is used to access, then do a "deep-copy" of a
* DISTANCE.
*
* Inputs:
*
*   hDistance - The DISTANCE_OBJECT object to duplicate
*
* Outputs:
*
*   A DISTANCE that is the duplicate of hDistance
*
*
*****************************************************************************/
static DISTANCE_OBJECT hDuplicate (
    DISTANCE_OBJECT hDistance
        )
{
    DISTANCE_OBJECT_STRUCT *psObj =
        (DISTANCE_OBJECT_STRUCT *)hDistance;

    // Verify inputs
    if(SMSO_bValid((SMS_OBJECT)hDistance) == FALSE )
    {
        // Error!
        return DISTANCE_INVALID_OBJECT;
    }

    if (psObj->bIsVector == FALSE)
    {
        // Create a distance from the whole value stored in the
        // fixed object
        return DISTANCE_hCreateFromFixed(
            psObj->hValue, psObj->eUnits);
    }
    else
    {
        return DISTANCE_hCreateFromDistanceVector(psObj->hValue);
    }

}

/*****************************************************************************
*
*   hValue
*
* This function returns the value of the DISTANCE rounded to the next largest
* whole number in the specified units
*
* Inputs:
*
*   hDistance - A handle to a valid DISTANCE object
*   eUnits - The units to get the value of the DISTANCE as
*
* Outputs:
*
*   A OSAL_FIXED_OBJECT representing the value for this DISTANCE.
*   OSAL_FIXED_INVALID_OBJECT will be returned on error.
*
*****************************************************************************/
static OSAL_FIXED_OBJECT hValue (
      DISTANCE_OBJECT hDistance,
      DISTANCE_UNIT_TYPE_ENUM eUnits
          )
{
    BOOLEAN bValid;
    OSAL_FIXED_OBJECT hFixed = OSAL_FIXED_INVALID_OBJECT;

    // Verify ownership of this object
    bValid =
        SMSO_bValid((SMS_OBJECT)hDistance);
    if (bValid == TRUE)
    {
        DISTANCE_OBJECT_STRUCT *psObj =
            (DISTANCE_OBJECT_STRUCT *)hDistance;

        if (psObj->eUnits == eUnits &&
            psObj->bIsVector == FALSE)
        {
            // No conversion needed, just return the whole value
            // stored in the structure
            hFixed = psObj->hValue;
        }
        else
        {
            BOOLEAN bSuccess;
            psObj->hConvertedValue = OSAL_FIXED.hCreateInMemory(
                0,0, &psObj->atConvertedValue[0]);
            bSuccess = bConvertUnits(
                psObj, eUnits, psObj->hConvertedValue);

            if (bSuccess == TRUE)
            {
                hFixed = psObj->hConvertedValue;
            }
        }
    }

    return hFixed;
}

/*****************************************************************************
*
*   n16Compare
*
* This method is used to compare two DISTANCE objects.  This function compares
* the values of the function, regardless of the units.  If the units don't
* match, hDistance2 is converted to hDistance1 and compared.
*
* Inputs:
*
*   hDistance1 - A handle to a valid Distance object
*   hDistance2 - A handle to a valid Distance object
*
* Outputs:
*               0   - DISTANCEs have the same value (equal)
*               > 0 - hDistance1 is greater than (after) hDistance2
*               < 0 - hDistance1 is less than (before) hDistance2 (or error)
*
*****************************************************************************/
static N16 n16Compare (
    DISTANCE_OBJECT hDistance1,
    DISTANCE_OBJECT hDistance2
        )
{
    N16 n16Result = N16_MIN;
    DISTANCE_OBJECT_STRUCT *psObj1 = (DISTANCE_OBJECT_STRUCT *)hDistance1,
                           *psObj2 = (DISTANCE_OBJECT_STRUCT *)hDistance2;
    BOOLEAN bOk = FALSE;

    // Verify ownership of these object
    bOk = SMSO_bValid((SMS_OBJECT)hDistance1);
    if(bOk == TRUE)
    {
        bOk = SMSO_bValid((SMS_OBJECT)hDistance2);
    }
    if(bOk == FALSE)
    {
        // Error!
        return N16_MIN;
    }

    if (psObj1->eUnits == psObj2->eUnits &&
        psObj1->bIsVector == FALSE &&
        psObj2->bIsVector == FALSE)
    {
        // Do a straight-up compare
        n16Result = OSAL_FIXED.n16Compare(psObj1->hValue, psObj2->hValue);
    }
    else
    {

        DISTANCE_UNIT_TYPE_ENUM eUnitsToCompare = psObj1->eUnits;
        OSAL_FIXED_OBJECT_DATA atFixedData[OSAL_FIXED_OBJECT_SIZE*2];
        OSAL_FIXED_OBJECT hFixed1, hFixed2;

        // Convert both objects to the units of psObj1.
        // If psObj1 has an unknown units type (if it is a distance vector)
        // then use psObj2's units.  Otherwise, use the default units type
        if ( psObj1->eUnits == DISTANCE_UNIT_TYPE_UNKNOWN )
        {
            if ( psObj2->eUnits != DISTANCE_UNIT_TYPE_UNKNOWN )
            {
                eUnitsToCompare = psObj2->eUnits;
            }
            else
            {
                eUnitsToCompare = DEFAULT_DISTANCE_TYPE;
            }
        }

        hFixed1 = OSAL_FIXED.hCreateInMemory(
            0, 0, &atFixedData[OSAL_FIXED_OBJECT_SIZE*0]);
        hFixed2 = OSAL_FIXED.hCreateInMemory(
            0, 0, &atFixedData[OSAL_FIXED_OBJECT_SIZE*1]);

        if (hFixed1 != OSAL_FIXED_INVALID_OBJECT &&
            hFixed2 != OSAL_FIXED_INVALID_OBJECT)
        {
            // Convert to the same unit
            bOk = bConvertUnits(psObj1, eUnitsToCompare, hFixed1);
            bOk &= bConvertUnits(psObj2, eUnitsToCompare, hFixed2);

            if (bOk == TRUE)
            {
                // Compare the fixed values
                n16Result = OSAL_FIXED.n16Compare(hFixed1, hFixed2);
            }
        }
    }

    return n16Result;
}


/*****************************************************************************
*
*   n32FWrite
*
* This object interface method is used to write the contents of a DISTANCE
* (performing a deep write) to a device specified by psFile. The intent of
* this method is to effectively store the contents of a DISTANCE for later
* retrieval (perhaps after a power cycle). The entire DISTANCE contents are written
* to the device, with enough information to recreate the exact DISTANCE
* when the hFRead() method is called.
*
* Inputs:
*
*   hDistance - The DISTANCE handle the caller wishes to write.
*   psFile - The device to write the LOCATION contents to.
*
* Outputs:
*
*   The number of characters written or EOF on error.
*
*****************************************************************************/
static N32 n32FWrite (
    DISTANCE_OBJECT hDistance,
    FILE *psFile
        )
{
    DISTANCE_OBJECT_STRUCT *psObj =
        (DISTANCE_OBJECT_STRUCT *)hDistance;
    N32 n32Return = 0;
    UN8 un8ByteToWrite, un8FracBits;
    UN32 un32Type;
    UN32 un32Value;

    // Verify inputs
    if((SMSO_bValid((SMS_OBJECT)hDistance) == FALSE) || (psFile == NULL))
    {
        // Error!
        return EOF;
    }

    // Write DISTANCE to file...
    if (psObj->bIsVector == TRUE)
    {
        // Convert this for a typeless distance vector to a
        // distance in miles
        //
        // Not sure who would be writing out a typeless value anyways....
        OSAL_FIXED_OBJECT_DATA atFixedData[OSAL_FIXED_OBJECT_SIZE];
        OSAL_FIXED_OBJECT hFixed;
        BOOLEAN bOk;

        hFixed = OSAL_FIXED.hCreateInMemory(0, 0, &atFixedData[0]);
        bOk = bConvertUnits(psObj, DEFAULT_DISTANCE_TYPE, hFixed);

        if (bOk == FALSE)
        {
            // Error!
            return EOF;
        }

        un32Value = OSAL_FIXED.n32Value(hFixed);
        un8FracBits = OSAL_FIXED.un8NumFractionalBits(hFixed);
        psObj->eUnits = DEFAULT_DISTANCE_TYPE;

    }
    else
    {
        un32Value = (UN32)OSAL_FIXED.n32Value(psObj->hValue);
        un8FracBits = OSAL_FIXED.un8NumFractionalBits(psObj->hValue);
    }

    // Write un32Value
    un8ByteToWrite = BYTE0(un32Value);
    n32Return += (N32)fwrite(&un8ByteToWrite, sizeof(UN8), 1, psFile);
    un8ByteToWrite = BYTE1(un32Value);
    n32Return += (N32)fwrite(&un8ByteToWrite, sizeof(UN8), 1, psFile);
    un8ByteToWrite = BYTE2(un32Value);
    n32Return += (N32)fwrite(&un8ByteToWrite, sizeof(UN8), 1, psFile);
    un8ByteToWrite = BYTE3(un32Value);
    n32Return += (N32)fwrite(&un8ByteToWrite, sizeof(UN8), 1, psFile);
    n32Return += (N32)fwrite(&un8FracBits, sizeof(UN8), 1, psFile);

    // Write the type of units
    un32Type = (UN32)psObj->eUnits;
    un8ByteToWrite = BYTE0(un32Type);
    n32Return += (N32)fwrite(&un8ByteToWrite, sizeof(UN8), 1, psFile);
    un8ByteToWrite = BYTE1(un32Type);
    n32Return += (N32)fwrite(&un8ByteToWrite, sizeof(UN8), 1, psFile);
    un8ByteToWrite = BYTE2(un32Type);
    n32Return += (N32)fwrite(&un8ByteToWrite, sizeof(UN8), 1, psFile);
    un8ByteToWrite = BYTE3(un32Type);
    n32Return += (N32)fwrite(&un8ByteToWrite, sizeof(UN8), 1, psFile);

    return n32Return;
}

/*****************************************************************************
*
*   hFRead
*
* This object interface method is used to read from a specified device psFile
* and generate a DISTANCE from that information. The data read from the device
* must have been previously written by the n32FWrite method. Upon
* successful execution of this method a new DISTANCE is created (created by the
* caller) which may be used to register for events or presented to the
* UI for display, etc. This method allows the caller to effectively read
* a previously stored DISTANCE regenerating the original DISTANCE written(saved) at
* an earlier time.
*
* Inputs:
*
*   psFile - The device to read the DISTANCE contents from.
*
* Outputs:
*
*   A new DISTANCE on success, otherwise DISTANCE_INVALID_OBJECT on failure.
*
*****************************************************************************/
static DISTANCE_OBJECT hFRead (
    FILE *psFile
        )
{
    DISTANCE_OBJECT hDistance = DISTANCE_INVALID_OBJECT;
    DISTANCE_UNIT_TYPE_ENUM eUnits;
    UN32 un32Type = 0;
    N32 n32Value = 0;
    size_t tTotal, tLen;
    UN8 un8FracBits;
    OSAL_FIXED_OBJECT hDistanceFixed = OSAL_FIXED_INVALID_OBJECT;
    OSAL_FIXED_OBJECT_DATA atDistData[OSAL_FIXED_OBJECT_SIZE];

    // Verify input
    if(psFile == NULL)
    {
        return DISTANCE_INVALID_OBJECT;
    }

    // Read DISTANCE from file...

    // Read n32Value
    tTotal = 0;
    do
    {
        tLen =
            fread(((UN8*)&n32Value) + tTotal, 1, 1, psFile);
        tTotal += tLen;
    } while((tLen != 0) && (tTotal <  sizeof(N32)));

    if(tTotal != sizeof(N32))
    {
        // End of file or error occured
        return DISTANCE_INVALID_OBJECT;
    }

    tTotal = fread(&un8FracBits, sizeof(UN8), 1, psFile);

    if(tTotal != 1)
    {
        // End of file or error occurred
        return DISTANCE_INVALID_OBJECT;
    }

    // Read eUnits
    tTotal = 0;
    do
    {
        tLen =
            fread(((UN8*)&un32Type) + tTotal, 1, 1, psFile);
        tTotal += tLen;
    } while((tLen != 0) && (tTotal <  sizeof(UN32)));

    if(tTotal != sizeof(UN32))
    {
        // End of file or error occured
        return DISTANCE_INVALID_OBJECT;
    }
    eUnits = (DISTANCE_UNIT_TYPE_ENUM)un32Type;

     if (n32Value != 0)
    {
        hDistanceFixed =
            OSAL_FIXED.hCreateInMemory(n32Value, un8FracBits,
                &atDistData[0]);
    }

    // Create an instance of the DISTANCE object
    hDistance = DISTANCE_hCreateFromFixed(hDistanceFixed, eUnits);

    return hDistance;
}

/*****************************************************************************
*
*   n32FPrintf
*
* This object interface method is used by the caller to send formatted
* output of a DISTANCE's contents to a specified file or device.
* This is mainly helpful during debugging of DISTANCE's but could be used by
* an application for any reason. This API is different than the n32FWrite()
* method which instead writes the contents of a DISTANCE to a file for the
* purposes of later re-generating the DISTANCE (for storage of the object).
* This API instead sends the DISTANCE as a verbose formatted output version.
*
* Inputs:
*
*   hDistance - The DISTANCE handle the caller wishes to write.
*   psFile - The device to write the LOCATION contents to.
*
* Outputs:
*
*   The number of characters written or EOF on error.
*
*****************************************************************************/
static N32 n32FPrintf (
    DISTANCE_OBJECT hDistance,
    FILE *psFile
        )
{
    DISTANCE_OBJECT_STRUCT *psObj =
        (DISTANCE_OBJECT_STRUCT *)hDistance;
    N32 n32Return = 0;
    BOOLEAN bValid;
    DISTANCE_UNIT_TYPE_ENUM eUnits;
    UN32 un32Value;


    // Determine if the caller owns this resource
    bValid = SMSO_bValid((SMS_OBJECT)hDistance);

    // Verify inputs. Object handle must be valid as well as the file handle.
    if((bValid == FALSE) || (psFile == NULL))
    {
        // Error!
        return EOF;
    }

    // Print DISTANCE information header
    n32Return += fprintf(psFile, "DISTANCE: hDistance = 0x%p\n",
        psObj);

    if (psObj->bIsVector == TRUE)
    {
        // Convert this from a typeless distance vector to a
        // distance in miles.  Want to hide this typless data for now
        OSAL_FIXED_OBJECT_DATA atFixedData[OSAL_FIXED_OBJECT_SIZE];
        OSAL_FIXED_OBJECT hFixed;
        BOOLEAN bOk;

        hFixed = OSAL_FIXED.hCreateInMemory(0, 0, &atFixedData[0]);
        bOk = bConvertUnits(psObj, DEFAULT_DISTANCE_TYPE, hFixed);

        if (bOk == FALSE)
        {
            // Error!
            return EOF;
        }

        un32Value = OSAL_FIXED.n32Ceiling(hFixed);
        eUnits = DEFAULT_DISTANCE_TYPE;
    }
    else
    {
        un32Value = (UN32)OSAL_FIXED.n32Value(psObj->hValue);
        eUnits = psObj->eUnits;
    }


    // Print un32Value
    n32Return += fprintf(psFile, "\tValue = %d\n", un32Value);

    // Print eUnits
    n32Return += fprintf(psFile, "\tUnits = %s\n",
        pacUnitsText(eUnits));


    return n32Return;
}

/*****************************************************************************
*
*   vDestroy
*
* This object interface method is used by the caller to destroy the specified
* DISTANCE object and all members of the object
*

* Inputs:
*
*   hDistance - A handle to a valid DISTANCE object that the caller wants
*                 to get rid of
*
* Outputs:
*
*   Nada
*
*****************************************************************************/
static void vDestroy (
    DISTANCE_OBJECT hDistance
        )
{
    BOOLEAN bValid;

    // Verify inputs. Object handle must be valid.
    bValid = SMSO_bValid((SMS_OBJECT)hDistance);
    if(bValid == FALSE)
    {
        // Error!
        return;
    }

    // Free object instance
    SMSO_vDestroy((SMS_OBJECT)hDistance);

    return;
}

/*****************************************************************************
                             FRIEND FUNCTIONS
*****************************************************************************/

/*****************************************************************************
*
*   DISTANCE_hCreateFromDistanceVector
*
* This object interface method is a friend function that is used to create a
* DISTANCE object that represents the final value from the distance calculation
* before it is multiplied by the earth's radius.  The radius value can be in
* whatever units desired, so we keep this DISTANCE "typeless" to provide the
* best accuracy, if needed.  The value is converted in to the desired type
* when a call to DISTANCE.un32Value or DISTANCE.fValue is made.
*
* Inputs:
*
*   hValue - A FIXED that is the final part of the distance calculation
*            function before multiplying by the Eearth's radius in desired
*            miles.
*
* Outputs:
*
*   A handle to a valid DISTANCE_OBJECT on success.
*   A DISTANCE_INVALID_OBJECT handle on failure
*
*****************************************************************************/
DISTANCE_OBJECT DISTANCE_hCreateFromDistanceVector  (
    OSAL_FIXED_OBJECT  hValue
        )
{
    DISTANCE_OBJECT_STRUCT *psObj =
        (DISTANCE_OBJECT_STRUCT *)DISTANCE_INVALID_OBJECT;
    BOOLEAN bOk;

    // Create an instance of the DISTANCE object
    psObj = (DISTANCE_OBJECT_STRUCT *)
        SMSO_hCreate(
            DISTANCE_OBJECT_NAME,
            sizeof(DISTANCE_OBJECT_STRUCT),
            SMS_INVALID_OBJECT, // No parent
            FALSE); // No Lock (ignored)

    if(psObj == NULL)
    {
        // Error!
        return DISTANCE_INVALID_OBJECT;
    }

    // Copy fixed data to our structure
    bOk = OSAL_FIXED.bCopyToMemory(hValue, &psObj->atValueData[0]);

    if (bOk == FALSE)
    {
        // Error!
        DISTANCE.vDestroy((DISTANCE_OBJECT)psObj);
        return DISTANCE_INVALID_OBJECT;
    }

    // Set the unit type as unknown since this is a
    // "typeless" DISTANCE
    psObj->eUnits = DISTANCE_UNIT_TYPE_UNKNOWN;
    psObj->bIsVector = TRUE;

    // Point our vector handle to our memory structure
    psObj->hValue =
        (OSAL_FIXED_OBJECT)&psObj->atValueData[0];

    return (DISTANCE_OBJECT)psObj;
}

/*****************************************************************************
*
*   DISTANCE_n16CompareToDistanceVector
*
* This object interface method is a friend function that is used to compare a
* DISTANCE object with a given distance vector.  The units used to describe
* the DISTANCE object are used to perform the comparison.
*
* Inputs:
*
*   hDistance - A valid DISTANCE_OBJECT handle to compare against
*   hVector - A valid FIXED handle that contains a "unitless" distance
*       vector which must be converted (locally) to the same units used
*       to describe the value held in the distance object.
*
* Outputs:
*
*   An N16 representing the relationship between the two values, as gived
*   by OSAL_FIXED.n16Compare
*
*****************************************************************************/
BOOLEAN DISTANCE_bDistanceOverRadius (
    DISTANCE_OBJECT hDistance,
    OSAL_FIXED_OBJECT hResult
        )
{
    DISTANCE_OBJECT_STRUCT *psObj = (DISTANCE_OBJECT_STRUCT *)hDistance;
    OSAL_RETURN_CODE_ENUM eReturnCode;
    OSAL_FIXED_OBJECT_DATA atFixedData[OSAL_FIXED_OBJECT_SIZE];
    OSAL_FIXED_OBJECT hRadius = OSAL_FIXED_INVALID_OBJECT;

    BOOLEAN bOwner = SMSO_bOwner((SMS_OBJECT)hDistance);

    do
    {
        if (bOwner == FALSE || hResult == OSAL_FIXED_INVALID_OBJECT)
        {
            break;
        }

        // Create a fixed object for our radius in specified units
        if (psObj->eUnits == DISTANCE_UNIT_TYPE_KILOMETERS)
        {
            hRadius = OSAL_FIXED.hCreateInMemory(
                FIXED_EARTH_RADIUS_KM,
                DISTANCE_FRACTIONAL_BITS,
                &atFixedData[0]);
        }
        else if  (psObj->eUnits == DISTANCE_UNIT_TYPE_MILES)
        {
            hRadius = OSAL_FIXED.hCreateInMemory(
                FIXED_EARTH_RADIUS_MILES,
                DISTANCE_FRACTIONAL_BITS,
                &atFixedData[0]);
        }
        else
        {
            // Unknown units
            break;
        }

        // Divide our distance by the radius
        eReturnCode = OSAL_FIXED.eDivide(
            psObj->hValue, hRadius, hResult);

        if (eReturnCode != OSAL_SUCCESS)
        {
            break;
        }

        return TRUE;

    } while (FALSE);

    return FALSE;
}


/*****************************************************************************
*
*   DISTANCE_n16CompareToDistanceVector
*
* This object interface method is a friend function that is used to compare a
* DISTANCE object with a given distance vector.  The units used to describe
* the DISTANCE object are used to perform the comparison.
*
* Inputs:
*
*   hDistance - A valid DISTANCE_OBJECT handle to compare against
*   hVector - A valid FIXED handle that contains a "unitless" distance
*       vector which must be converted (locally) to the same units used
*       to describe the value held in the distance object.
*
* Outputs:
*
*   An N16 representing the relationship between the two values, as gived
*   by OSAL_FIXED.n16Compare
*
*****************************************************************************/
N16 DISTANCE_n16CompareToDistanceVector (
    DISTANCE_OBJECT hDistance,
    OSAL_FIXED_OBJECT hVector
        )
{
    DISTANCE_OBJECT_STRUCT *psObj =
        (DISTANCE_OBJECT_STRUCT *)hDistance;
    BOOLEAN bOk;
    N16 n16Result = N16_MIN;

    // Verify inputs
    bOk = SMSO_bValid((SMS_OBJECT)hDistance);
    if (bOk == TRUE)
    {
        OSAL_FIXED_OBJECT_DATA atFixedData[OSAL_FIXED_OBJECT_SIZE];
        OSAL_FIXED_OBJECT hResult;

        // Create the result fixed object
        hResult = OSAL_FIXED.hCreateInMemory(0, 0, &atFixedData[0]);

        // Convert the distance vector to the
        // units used by this object
        bOk = bMultiplyByEarthsRadius(
            hVector, psObj->eUnits, hResult);
        if (bOk == TRUE)
        {
            // Now, compare the two distances
            n16Result = OSAL_FIXED.n16Compare(psObj->hValue, hResult);
        }
    }

    return n16Result;
}

/*****************************************************************************
*
*   DISTANCE_hCreateFromFixed
*
* This object interface method is a friend function that is used to create a
* DISTANCE object from OSAL_FIXED_OBJECT
*
* Inputs:
*
*   hValue - for the distance object
*   eUnits - The type of units
*
* Outputs:
*
*   A handle to a valid DISTANCE_OBJECT on success.
*   A DISTANCE_INVALID_OBJECT handle on failure
*
*****************************************************************************/
DISTANCE_OBJECT DISTANCE_hCreateFromFixed  (
    OSAL_FIXED_OBJECT  hValue,
    DISTANCE_UNIT_TYPE_ENUM eUnits
        )
{
    DISTANCE_OBJECT_STRUCT *psObj =
        (DISTANCE_OBJECT_STRUCT *)DISTANCE_INVALID_OBJECT;
    N32 n32Value;

    n32Value = OSAL_FIXED.n32Value(hValue);
    if(n32Value < 0)
    {
        // Error!Cannot create object with negative distance value
        return DISTANCE_INVALID_OBJECT;
    }

    // Create an instance of the DISTANCE object
    psObj = (DISTANCE_OBJECT_STRUCT *)
        SMSO_hCreate(
            DISTANCE_OBJECT_NAME,
            sizeof(DISTANCE_OBJECT_STRUCT),
            SMS_INVALID_OBJECT, // No parent
            FALSE); // No Lock (ignored)

    if(psObj == NULL)
    {
        // Error!
        return DISTANCE_INVALID_OBJECT;
    }

    psObj->eUnits = eUnits;
    psObj->bIsVector = FALSE;
    psObj->hValue = OSAL_FIXED.hCreateInMemory(OSAL_FIXED.n32Value(hValue),
                                               OSAL_FIXED.un8NumFractionalBits(hValue),
                                               &psObj->atValueData[0]);

    return (DISTANCE_OBJECT)psObj;
}


/*****************************************************************************
                             PRIVATE FUNCTIONS
*****************************************************************************/

/*****************************************************************************
*
*   bConvertUnits
*
*****************************************************************************/
static BOOLEAN bConvertUnits (
    DISTANCE_OBJECT_STRUCT *psObj,
    DISTANCE_UNIT_TYPE_ENUM eUnitsToConvertTo,
    OSAL_FIXED_OBJECT hResult
        )
{
    BOOLEAN bSuccess = FALSE;
    OSAL_RETURN_CODE_ENUM eReturnCode;

    if (psObj->bIsVector == TRUE)
    {
        bSuccess = bMultiplyByEarthsRadius(psObj->hValue,
            eUnitsToConvertTo, hResult);
    }
    else
    {
        if (eUnitsToConvertTo == psObj->eUnits)
        {
            bSuccess = OSAL_FIXED.bCopyToMemory(psObj->hValue, (void *)hResult);
        }
        else
        {
            // Convert from one unit to the other
            OSAL_FIXED_OBJECT_DATA atFixedData[OSAL_FIXED_OBJECT_SIZE];
            OSAL_FIXED_OBJECT hConversionMultipler = OSAL_FIXED_INVALID_OBJECT;

            if (eUnitsToConvertTo == DISTANCE_UNIT_TYPE_MILES &&
                psObj->eUnits == DISTANCE_UNIT_TYPE_KILOMETERS)
            {
                // 1 kilometer = 0.621371192 miles
                hConversionMultipler = OSAL_FIXED.hCreateInMemory(
                        KILOMETERS_TO_MILES, DISTANCE_FRACTIONAL_BITS,
                        &atFixedData[0]);
            }
            else if  (eUnitsToConvertTo == DISTANCE_UNIT_TYPE_KILOMETERS &&
                      psObj->eUnits == DISTANCE_UNIT_TYPE_MILES)
            {
                // 1 Mile = 1.609344 Kilometers
                hConversionMultipler = OSAL_FIXED.hCreateInMemory(
                        MILES_TO_KILOMETERS, DISTANCE_FRACTIONAL_BITS,
                        &atFixedData[0]);
            }
            else
            {
                // Can't convert  Unknown combo
                return FALSE;
            }

            // Multiply our value by our conversion factor
            eReturnCode = OSAL_FIXED.eMultiply(psObj->hValue,
                                          hConversionMultipler,
                                          hResult);

            if (eReturnCode == OSAL_SUCCESS)
            {
                bSuccess = TRUE;
            }
        }
    }

    return bSuccess;
}

/*****************************************************************************
*
*   bMultiplyByEarthsRadius
*
* A private helper function that takes a FIXED value and multiples it by the
* earth's radius in tjhe specified units.
*
* Inputs:
*
*   hDistanceVector - The output, as a FIXED, of the distance calculation after
*                    all the trig is done (from the LOCATION object)
*   eUnits - The desired units the distance should be in.
*   hResult - The OSAL_FIXED_OBJECT which will store the result of multiplying
*       the hDistanceVector fixed value by the earth's radius in eUnits.
*
* Outputs:
*
*   TRUE on succes, FALSE is returned on error.
*
*****************************************************************************/
static BOOLEAN bMultiplyByEarthsRadius(
    OSAL_FIXED_OBJECT hDistanceVector,
    DISTANCE_UNIT_TYPE_ENUM eUnits,
    OSAL_FIXED_OBJECT hResult
        )
{
    BOOLEAN bSuccess = FALSE;
    OSAL_RETURN_CODE_ENUM eReturnCode;
    OSAL_FIXED_OBJECT_DATA atFixedData[OSAL_FIXED_OBJECT_SIZE];
    OSAL_FIXED_OBJECT hEarthsRadius;

    switch (eUnits)
    {
        case DISTANCE_UNIT_TYPE_MILES:
        {
            hEarthsRadius =
                OSAL_FIXED.hCreateInMemory(
                    FIXED_EARTH_RADIUS_MILES, DISTANCE_FRACTIONAL_BITS,
                    &atFixedData[0]);
        }
        break;
        case DISTANCE_UNIT_TYPE_KILOMETERS:
        {
            hEarthsRadius =
                OSAL_FIXED.hCreateInMemory(
                    FIXED_EARTH_RADIUS_KM, DISTANCE_FRACTIONAL_BITS,
                    &atFixedData[0]);
        }
        break;
        case DISTANCE_UNIT_TYPE_UNKNOWN:
        default:
        {
            hEarthsRadius = OSAL_FIXED_INVALID_OBJECT;
        }
        break;
    }

    // Multiply by the earth's radius
    eReturnCode = OSAL_FIXED.eMultiply(hDistanceVector,
                                  hEarthsRadius,
                                  hResult);
    if (eReturnCode == OSAL_SUCCESS)
    {
        bSuccess = TRUE;
    }

    return bSuccess;
}

/*****************************************************************************
*
*       pacUnitsText
*
* This is a local function which simply maps an enumerated type to
* a textual representation for formatting the enumerated type.
*
*****************************************************************************/
static const char *pacUnitsText(
    DISTANCE_UNIT_TYPE_ENUM eUnits
        )
{
    const char *pacReturnString;

    switch (eUnits)
    {
        case DISTANCE_UNIT_TYPE_MILES:
            pacReturnString = MACRO_TO_STRING(DISTANCE_UNIT_TYPE_MILES);
        break;
        case DISTANCE_UNIT_TYPE_KILOMETERS:
            pacReturnString = MACRO_TO_STRING(DISTANCE_UNIT_TYPE_KILOMETERS);
        break;
        case DISTANCE_UNIT_TYPE_UNKNOWN:
        default:
            pacReturnString = MACRO_TO_STRING(DISTANCE_UNIT_TYPE_UNKNOWN);
        break;
    }

    return pacReturnString;
}

#ifdef SUPPORT_CUNIT
#include <distance_obj.cunit>
#endif // SUPPORT_CUNIT
