/******************************************************************************/
/*                    Copyright (c) Sirius XM Radio, Inc.                     */
/*                            All Rights Reserved                             */
/*         Licensed Materials - Property of Sirius XM Radio, Inc.             */
/******************************************************************************/
/*******************************************************************************
 *
 * DESCRIPTION
 *
 *  This module contains the Object:DEVICE implementation for the
 *  Satellite Module Services (SMS)
 *
 ******************************************************************************/

#include "standard.h"
#include "osal.h"

#include "sms_api.h"
#include "sms_obj.h"
#include "sms.h"
#include "sms_version.h"

#include "dataservice_mgr_obj.h"
#include "location_obj.h"

#include "device_obj.h"
#include "_device_obj.h"

#include "sms_api_debug.h"
static const char *gpacThisFile = __FILE__;

/*****************************************************************************
 *                           PUBLIC FUNCTIONS
 *****************************************************************************/

/*****************************************************************************
 *
 *   bSetPosition
 *
 *****************************************************************************/
static BOOLEAN bSetPosition (
    OSAL_FIXED_OBJECT hLat,
    OSAL_FIXED_OBJECT hLon
        )
{
    BOOLEAN bPosted = FALSE;

    // Check input
    if ((hLat != OSAL_FIXED_INVALID_OBJECT) &&
        (hLon != OSAL_FIXED_INVALID_OBJECT))
    {
        // Post the event!
        bPosted = DATASERVICE_MGR_bPostDevicePositionEvent(hLat, hLon);
    }

    return bPosted;
}

/*****************************************************************************
 *
 *   bGetPosition
 *
 *****************************************************************************/
static BOOLEAN bGetPosition (
    OSAL_FIXED_OBJECT hLat,
    OSAL_FIXED_OBJECT hLon
        )
{
    DEVICE_OBJECT_STRUCT *psObj;
    BOOLEAN bSuccess = FALSE;

    // Check inputs
    if ((hLat == OSAL_FIXED_INVALID_OBJECT) ||
        (hLon == OSAL_FIXED_INVALID_OBJECT))
    {
        SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
            DEVICE_OBJECT_NAME": invalid input");

        return FALSE;
    }

    // Get the device object from the DSM
    psObj = psGetDevFromDSM();

    // Did that work?
    if (psObj == (DEVICE_OBJECT_STRUCT *)NULL)
    {
        return FALSE;
    }

    do
    {
        OSAL_FIXED_OBJECT hCoord;
        BOOLEAN bOk;

        if (psObj->bPositionAvailable != TRUE)
        {
            puts(DEVICE_OBJECT_NAME": Position has not been set");
            break;
        }

        /* Update our lattitude */

        // Grab the fixed object
        hCoord = LOCATION.hLat(psObj->hCurrentLocation);
        if (hCoord == OSAL_FIXED_INVALID_OBJECT)
        {
            break;
        }

        // Update it
        bOk = bUpdateFixed( hCoord, hLat);
        if (bOk == FALSE)
        {
            printf(DEVICE_OBJECT_NAME": failed to update LAT\n");
            break;
        }

        /* Update our longitude */

        // Grab the fixed object
        hCoord = LOCATION.hLon(psObj->hCurrentLocation);
        if (hCoord == OSAL_FIXED_INVALID_OBJECT)
        {
            break;
        }

        // Update it
        bOk = bUpdateFixed( hCoord, hLon);
        if (bOk == FALSE)
        {
            printf(DEVICE_OBJECT_NAME": failed to update LON\n");
            break;
        }

        // All done!
        bSuccess = TRUE;
    } while (0);

    SMSO_vUnlock((SMS_OBJECT)psObj);

    return bSuccess;
}

/*****************************************************************************
 *
 *   hRegisterForUpdates
 *
 *****************************************************************************/
static DEVICE_REGISTRATION_OBJECT hRegisterForUpdates (
    DISTANCE_OBJECT hDistance,
    void *pvRegistrationArgument,
    DEVICE_UPDATE_CALLBACK vCallback,
    void *pvCallbackArg
        )
{
    DEVICE_REGISTRATION_OBJECT hRegistration =
        DEVICE_REGISTRATION_INVALID_OBJECT;
    DEVICE_OBJECT_STRUCT *psObj;
    DISTANCE_OBJECT hDistanceCopy = DISTANCE_INVALID_OBJECT;

    // Check inputs
    if ((vCallback == NULL) ||
        (hDistance == DISTANCE_INVALID_OBJECT))
    {
        SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
             DEVICE_OBJECT_NAME": invalid input");

        return DEVICE_REGISTRATION_INVALID_OBJECT;
    }

    // Get the device object from the DSM
    psObj = psGetDevFromDSM();

    // Did that work?
    if (psObj !=  (DEVICE_OBJECT_STRUCT *)NULL)
    {
        // Duplicate distance
        hDistanceCopy = DISTANCE.hDuplicate(hDistance);
        if (hDistanceCopy != DISTANCE_INVALID_OBJECT)
        {
            // Register for notification with this distance
            hRegistration = hRegisterLocal(
                psObj, hDistanceCopy,
                pvRegistrationArgument,
                vCallback, pvCallbackArg);
        }

        SMSO_vUnlock((SMS_OBJECT)psObj);
    }

    // Clean up memory if there was an error
    if ((hRegistration == DEVICE_REGISTRATION_INVALID_OBJECT) &&
        (hDistanceCopy != DISTANCE_INVALID_OBJECT))
    {
        DISTANCE.vDestroy(hDistanceCopy);
    }

    return hRegistration;
}

/*****************************************************************************
 *
 *   vUnregisterForUpdates
 *
 *****************************************************************************/
static void vUnregisterForUpdates (
    DEVICE_REGISTRATION_OBJECT hRegistration
        )
{
    DEVICE_OBJECT hDevice = DEVICE_INVALID_OBJECT;
    BOOLEAN bLocked = FALSE;

    do
    {
        DEVICE_REGISTRATION_STRUCT *psReg =
            (DEVICE_REGISTRATION_STRUCT *)hRegistration;
        OSAL_RETURN_CODE_ENUM eReturnCode;

        // Lock the registration object
        // (and validate the registration handle)
        bLocked = SMSO_bLock((SMS_OBJECT)hRegistration,
            OSAL_OBJ_TIMEOUT_INFINITE);
        if (bLocked == FALSE)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                DEVICE_OBJECT_NAME": Unable to lock registration");
            break;
        }

        // Get the device object from this registration
        hDevice = (DEVICE_OBJECT)SMSO_hParent((SMS_OBJECT)hRegistration);
        if (hDevice == DEVICE_INVALID_OBJECT)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                DEVICE_OBJECT_NAME": Unable to get device from registration");

            // This is really bad. Not sure how it could happen. DEVICE
            // object was supposed to exist, since it was about to be 
            // locked while locking Registration object.
            SMSO_vUnlock((SMS_OBJECT)hRegistration);
            bLocked = FALSE;
            break;
        }

        // Remove this from the registration list
        eReturnCode = OSAL.eLinkedListRemove(psReg->hEntry);
        if (eReturnCode != OSAL_SUCCESS)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    DEVICE_OBJECT_NAME
                    ": failed to remove desc from list (%s)\n",
                    OSAL.pacGetReturnCodeName(eReturnCode));
            break;
        }

        // Destroy this registration now
        vDestroyReg(psReg);

    } while (0);

    if (bLocked == TRUE)
    {
        SMSO_vUnlock((SMS_OBJECT)hDevice);
    }

    return;
}

/*****************************************************************************
 *
 *   pvRegistrationArgument
 *
 *****************************************************************************/
static void *pvRegistrationArgument (
    DEVICE_REGISTRATION_OBJECT hRegistration
        )
{
    BOOLEAN bLocked;
    void *pvRegistrationArg = NULL;

    // Lock the device via this registration
    bLocked = SMSO_bLock(
        (SMS_OBJECT)hRegistration, OSAL_OBJ_TIMEOUT_INFINITE );
    if (bLocked == TRUE)
    {
        DEVICE_REGISTRATION_STRUCT *psReg =
            (DEVICE_REGISTRATION_STRUCT *)hRegistration;

        pvRegistrationArg = psReg->pvRegistrationArg;

        SMSO_vUnlock((SMS_OBJECT)hRegistration);
    }

    return pvRegistrationArg;
}

/*****************************************************************************
 *
 *   bReplaceRegistrationArgument
 *
 *****************************************************************************/
static BOOLEAN bReplaceRegistrationArgument (
    DEVICE_REGISTRATION_OBJECT hRegistration,
    BOOLEAN bNotify,
    void *pvRegistrationArg
        )
{
    BOOLEAN bLocked;

    // Lock the device via this registration
    bLocked = SMSO_bLock(
        (SMS_OBJECT)hRegistration, OSAL_OBJ_TIMEOUT_INFINITE );
    if (bLocked == TRUE)
    {
        DEVICE_REGISTRATION_STRUCT *psReg =
            (DEVICE_REGISTRATION_STRUCT *)hRegistration;

        // Update the registration argument
        psReg->pvRegistrationArg = pvRegistrationArg;

        // Did the caller want us to notify
        // for this change?
        if (bNotify == TRUE)
        {
            // Perform the notification now,
            // and indicate we don't have a distance
            // object -- just the registration arg
            psReg->vCallback(
                psReg->pvRegistrationArg,
                DEVICE_EVENT_REGISTRATION_UPDATE,
                DISTANCE_INVALID_OBJECT,
                psReg->pvCallbackArg);
        }

        SMSO_vUnlock((SMS_OBJECT)hRegistration);
    }

    return bLocked;
}

/*****************************************************************************
 *                             FRIEND FUNCTIONS
 *****************************************************************************/

/*****************************************************************************
 *
 *   DEVICE_hInstall
 *
 *****************************************************************************/
DEVICE_OBJECT DEVICE_hInstall(
    SMS_OBJECT hParent
        )
{
    DEVICE_OBJECT_STRUCT *psObj =
        (DEVICE_OBJECT_STRUCT *)NULL;
    BOOLEAN bOwner;
    OSAL_RETURN_CODE_ENUM eReturnCode;

    bOwner = SMSO_bOwner(hParent);
    if (bOwner == FALSE)
    {
        return DEVICE_INVALID_OBJECT;
    }

    do
    {
        // Create the global device object
        psObj = (DEVICE_OBJECT_STRUCT *)
            SMSO_hCreate(
                DEVICE_OBJECT_NAME,
                sizeof(DEVICE_OBJECT_STRUCT),
                hParent, FALSE);

        if (psObj == NULL)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    DEVICE_OBJECT_NAME": failed to create object");
            break;
        }

        // Initialize attributes
        psObj->hCurrentLocation = LOCATION_INVALID_OBJECT;
        psObj->hRegistrationList = OSAL_INVALID_OBJECT_HDL;
        psObj->bPositionAvailable = FALSE;

        // Create the linked list of registrants
        eReturnCode = OSAL.eLinkedListCreate(
            &psObj->hRegistrationList,
            DEVICE_OBJECT_NAME": Regs",
            (OSAL_LL_COMPARE_HANDLER)NULL,
            OSAL_LL_OPTION_NONE );
        if (eReturnCode != OSAL_SUCCESS)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    DEVICE_OBJECT_NAME
                    ": failed to create registration list (%s)\n",
                    OSAL.pacGetReturnCodeName(eReturnCode));
            break;
        }

        return (DEVICE_OBJECT)psObj;
    }
    while (0);

    // Something went wrong -- uninstall
    DEVICE_vUninstall((DEVICE_OBJECT)psObj);

    return DEVICE_INVALID_OBJECT;
}

/*****************************************************************************
 *
 *   DEVICE_vUninstall
 *
 *****************************************************************************/
void DEVICE_vUninstall(
    DEVICE_OBJECT hDevice
        )
{
    BOOLEAN bOwner;

    bOwner = SMSO_bOwner((SMS_OBJECT)hDevice);
    if (bOwner == TRUE)
    {
        DEVICE_OBJECT_STRUCT *psObj =
            (DEVICE_OBJECT_STRUCT *)hDevice;

        // Destroy the "current location"
        if (psObj->hCurrentLocation != LOCATION_INVALID_OBJECT)
        {
            LOCATION.vDestroy(psObj->hCurrentLocation);
            psObj->hCurrentLocation = LOCATION_INVALID_OBJECT;
        }

        // Empty / destroy the registration list
        if (psObj->hRegistrationList != OSAL_INVALID_OBJECT_HDL)
        {
            OSAL_RETURN_CODE_ENUM eReturnCode;
#if (SMS_DEBUG == 1)
            UN32 un32Items = 0;

            eReturnCode = OSAL.eLinkedListItems(psObj->hRegistrationList, &un32Items);
            if (eReturnCode != OSAL_SUCCESS)
            {
                SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    DEVICE_OBJECT_NAME": failed to get number of items in "
                        "registration list (%s)", OSAL.pacGetReturnCodeName(eReturnCode)
                            );
            }
            else if (un32Items > 0)
            {
                SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    DEVICE_OBJECT_NAME": not all registered managers/callbacks "
                        "are unregistered. Still has %d item(s)", un32Items);
            }
#endif

            // Remove all entries from the list (dirty clean up)
            eReturnCode = OSAL.eLinkedListRemoveAll(
                psObj->hRegistrationList,
                (OSAL_LL_RELEASE_HANDLER)vDestroyReg);
            if (eReturnCode != OSAL_SUCCESS)
            {
                SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                     DEVICE_OBJECT_NAME": OSAL.eLinkedListRemoveAll Failed: %u (%s)",
                     eReturnCode, OSAL.pacGetReturnCodeName(eReturnCode));
            }
            else
            {
                // Delete the list now that it's empty
                eReturnCode = OSAL.eLinkedListDelete(psObj->hRegistrationList);
                if (eReturnCode != OSAL_SUCCESS)
                {
                    SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                         DEVICE_OBJECT_NAME": OSAL.eLinkedListDelete Failed: %u (%s)",
                         eReturnCode, OSAL.pacGetReturnCodeName(eReturnCode));
                }

                psObj->hRegistrationList = OSAL_INVALID_OBJECT_HDL;
            }
        }

        SMSO_vDestroy((SMS_OBJECT)psObj);
    }

    return;
}

/*****************************************************************************
 *
 *   DEVICE_bSetPosition
 *
 *****************************************************************************/
BOOLEAN DEVICE_bSetPosition (
    DEVICE_OBJECT hDevice,
    OSAL_FIXED_OBJECT hLat,
    OSAL_FIXED_OBJECT hLon
        )
{
    BOOLEAN bOwner;

    // Check input
    if ((hLat == OSAL_FIXED_INVALID_OBJECT) ||
        (hLon == OSAL_FIXED_INVALID_OBJECT))
    {
        SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
            DEVICE_OBJECT_NAME": invalid input");

        return FALSE;
    }

    bOwner = SMSO_bOwner((SMS_OBJECT)hDevice);
    if (bOwner == TRUE)
    {
        DEVICE_OBJECT_STRUCT *psObj =
            (DEVICE_OBJECT_STRUCT *)hDevice;

        do
        {
            OSAL_RETURN_CODE_ENUM eReturnCode = OSAL_ERROR;
            DEVICE_POSITION_NOTIFY_STRUCT sNotify;
            BOOLEAN bUpdated;

            // The event here will always include the position update flag
            sNotify.tEventMask = DEVICE_EVENT_POSITION_UPDATE;

            // If the current position tracked by this object
            // is invalid it means that we're setting the
            // initial position
            if (psObj->hCurrentLocation == LOCATION_INVALID_OBJECT)
            {
                // Alter the mask to indicate initial position configured
                sNotify.tEventMask = DEVICE_EVENT_INITIAL_POSITION_CONFIGURED;
            }

            // Update current location object
            bUpdated  = bUpdateLocation(&psObj->hCurrentLocation, hLat, hLon);
            if (bUpdated == FALSE)
            {
                SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    DEVICE_OBJECT_NAME
                    ": failed to update current LOCATION object");
                break;
            }

            // We have a position now
            psObj->bPositionAvailable = TRUE;

            // Now, use the current location in our notifier structure
            sNotify.hLocation = psObj->hCurrentLocation;

            // Iterate all registrations to notify them about new coordinates
            // if it meets their distance criteria
            eReturnCode = OSAL.eLinkedListIterate(
                psObj->hRegistrationList,
                (OSAL_LL_ITERATOR_HANDLER)bNotifyByLocation,
                &sNotify
                    );
            if ((eReturnCode != OSAL_SUCCESS) &&
                (eReturnCode != OSAL_NO_OBJECTS))
            {
                SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    DEVICE_OBJECT_NAME
                    ": failed to iterate list to notify "
                    "registrations (%s)\n",
                    OSAL.pacGetReturnCodeName(eReturnCode)
                        );
                break;
            }

            return TRUE;
        } while (0);

        // There was a failure above which means something really
        // went wrong here, and we don't know what the state of
        // this object is. In that case we wont't let callers ask
        // us for position info.
        psObj->bPositionAvailable = FALSE;
    }

    return FALSE;
}

/*****************************************************************************
 *
 *   DEVICE_hRegisterForUpdates
 *
 *****************************************************************************/
DEVICE_REGISTRATION_OBJECT DEVICE_hRegisterForUpdates (
    DEVICE_OBJECT hDevice,
    N32 n32DistanceValue,
    DISTANCE_UNIT_TYPE_ENUM eDistanceUnits,
    void *pvRegistrationArgument,
    DEVICE_UPDATE_CALLBACK vCallback,
    void *pvCallbackArg
        )
{
    DEVICE_REGISTRATION_OBJECT hRegistration =
        DEVICE_REGISTRATION_INVALID_OBJECT;
    DISTANCE_OBJECT hDistance = DISTANCE_INVALID_OBJECT;
    BOOLEAN bOwner;

    // Check inputs
    if (vCallback == NULL)
    {
        SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
             DEVICE_OBJECT_NAME": invalid input");

        return DEVICE_REGISTRATION_INVALID_OBJECT;
    }

    bOwner = SMSO_bOwner((SMS_OBJECT)hDevice);
    if (bOwner == TRUE)
    {
        DEVICE_OBJECT_STRUCT *psObj =
            (DEVICE_OBJECT_STRUCT *)hDevice;

        // Create the distance object
        hDistance = DISTANCE.hCreate(n32DistanceValue, eDistanceUnits);

        // Check how that went
        if (hDistance != DISTANCE_INVALID_OBJECT)
        {
            // Register for notification with this distance
            hRegistration = hRegisterLocal(
                psObj, hDistance,
                pvRegistrationArgument,
                vCallback, pvCallbackArg);
        }
    }

    // Clean up memory if there was an error
    if ((hRegistration == DEVICE_REGISTRATION_INVALID_OBJECT) &&
        (hDistance != DISTANCE_INVALID_OBJECT))
    {
        DISTANCE.vDestroy(hDistance);
    }

    return hRegistration;
}

/*****************************************************************************
 *
 *   DEVICE_bUpdateLocWithCurrentDevLoc
 *
 *****************************************************************************/
BOOLEAN DEVICE_bUpdateLocWithCurrentDevLoc (
    DEVICE_OBJECT hDevice,
    LOCATION_OBJECT hLocation
        )
{
    BOOLEAN bOwner, bUpdated = FALSE;

    // Verify input
    if (hLocation == LOCATION_INVALID_OBJECT)
    {
        return FALSE;
    }

    // Verify object ownership
    bOwner = SMSO_bOwner((SMS_OBJECT)hDevice);
    if (bOwner == TRUE)
    {
        DEVICE_OBJECT_STRUCT *psObj =
            (DEVICE_OBJECT_STRUCT *)hDevice;

        do
        {
            BOOLEAN bOk;
            OSAL_FIXED_OBJECT_DATA atFixedData[OSAL_FIXED_OBJECT_SIZE * 2];
            OSAL_FIXED_OBJECT hLat, hLon;

            if (psObj->bPositionAvailable != TRUE)
            {
                break;
            }

            // Place the fixed objects into stack memory
            hLat = OSAL_FIXED.hCreateInMemory(0, 0,
                &atFixedData[OSAL_FIXED_OBJECT_SIZE*0]);

            if (hLat == OSAL_FIXED_INVALID_OBJECT)
            {
                break;
            }

            hLon = OSAL_FIXED.hCreateInMemory(0, 0,
                &atFixedData[OSAL_FIXED_OBJECT_SIZE*1]);

            if (hLon == OSAL_FIXED_INVALID_OBJECT)
            {
                break;
            }

            // Get the current position
            bOk = DEVICE.bGetPosition(hLat, hLon);
            if (bOk == FALSE)
            {
                break;
            }

            // Now it's time to update the location object
            bUpdated = bUpdateLocation(&hLocation, hLat, hLon);
        } while (FALSE);
    }

    return bUpdated;
}

/*****************************************************************************
 *
 *   DEVICE_n32FPrintf
 *
 *   Friend function to print out contents fo DEVICE object
 *
 *****************************************************************************/
N32 DEVICE_n32FPrintf (
    FILE *psFile,
    SMSAPI_OUTPUT_OPTION_ENUM eOutputOption
        )
{
    N32 n32Return = 0;
    DEVICE_OBJECT_STRUCT *psObj;

    if (psFile == NULL)
    {
        return EOF;
    }

    // Get the device object from the DSM
    psObj = psGetDevFromDSM();
    if (psObj == NULL)
    {
        return EOF;
    }

    // Printing title
    n32Return += fprintf(psFile,DEVICE_OBJECT_NAME":\n");
    if (psObj->bPositionAvailable == TRUE)
    {
        N32 n32LocationResult;
        n32Return += fprintf(psFile,"Current position:\n");

        n32LocationResult = LOCATION.n32FPrintf(psObj->hCurrentLocation, psFile);
        if (n32LocationResult > 0)
        {
            n32Return += n32LocationResult;
        }
    }
    else
    {
        n32Return += fprintf(psFile,"Position unknown\n");
    }

    if( psObj->hRegistrationList != NULL)
    {
        DEVICE_REG_PRINT_ITERATOR_STRUCT sIterator;

        // Printing registration list
        n32Return += fprintf(psFile,"Registration list:\n");

        sIterator.psFile = psFile;
        sIterator.pn32BytesWritten = &n32Return;
        sIterator.un8Counter = 0;

        OSAL.eLinkedListIterate(
            psObj->hRegistrationList,
            (OSAL_LL_ITERATOR_HANDLER)bPrintRegEntry,
            &sIterator);
    }

    SMSO_vUnlock((SMS_OBJECT)psObj);

    return n32Return;
}

/*****************************************************************************
 *                             PRIVATE FUNCTIONS
 *****************************************************************************/
/*****************************************************************************
 *
 *   psGetDevFromDSM
 *
 *****************************************************************************/
static DEVICE_OBJECT_STRUCT *psGetDevFromDSM ( void )
{
    BOOLEAN bLocked;
    DEVICE_OBJECT_STRUCT *psObj =
        (DEVICE_OBJECT_STRUCT *)NULL;
    DEVICE_OBJECT hDevice;

    // Get the device handle from the DSM
    hDevice = SMS_hDevice();
    if (hDevice != DEVICE_INVALID_OBJECT)
    {
        // Lock the device now
        bLocked = SMSO_bLock(
            (SMS_OBJECT)hDevice, OSAL_OBJ_TIMEOUT_INFINITE );

        if (bLocked == TRUE)
        {
            // Copy the pointer
            psObj = (DEVICE_OBJECT_STRUCT *)hDevice;
        }
    }

    // Provide result to caller
    return psObj;
}

/*****************************************************************************
 *
 *   hRegisterLocal
 *
 *****************************************************************************/
static DEVICE_REGISTRATION_OBJECT hRegisterLocal (
    DEVICE_OBJECT_STRUCT *psObj,
    DISTANCE_OBJECT hDistance,
    void *pvRegistrationArgument,
    DEVICE_UPDATE_CALLBACK vCallback,
    void *pvCallbackArg
        )
{
    DEVICE_REGISTRATION_STRUCT *psReg =
        (DEVICE_REGISTRATION_STRUCT *)NULL;

    do
    {
        BOOLEAN bOk;

        // First, check that our callback is non-null
        if (vCallback == NULL)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                DEVICE_OBJECT_NAME": cannot create device registration"
                "with NULL callback.");
            break;
        }

        // If okay, create the registration
        psReg = psCreateReg(
            psObj, hDistance,
            pvRegistrationArgument,
            vCallback, pvCallbackArg);
        if (psReg == (DEVICE_REGISTRATION_STRUCT *)NULL)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                DEVICE_OBJECT_NAME": failed to create desc");
            break;
        }

        // Check if device location was ever set
        if(psObj->bPositionAvailable == TRUE)
        {
            // If it was, set current location as "last notified" to
            // bypass first immediate notification for any coordinate update
            psReg->hLastNotifiedLocation = LOCATION.hDuplicate(psObj->hCurrentLocation);
        }

        // Add this to the list of registrations we handle
        bOk = bAddRegistrationToList(psObj, psReg);

        if (bOk == FALSE)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                DEVICE_OBJECT_NAME": failed to register for notification callback %p",
                vCallback);
            break;
        }

        return (DEVICE_REGISTRATION_OBJECT)psReg;
    }
    while (0);

    if (psReg != NULL)
    {
        vDestroyReg(psReg);
    }

    return DEVICE_REGISTRATION_INVALID_OBJECT;
}

/*****************************************************************************
 *
 *   psCreateReg
 *
 *****************************************************************************/
static DEVICE_REGISTRATION_STRUCT *psCreateReg (
    DEVICE_OBJECT_STRUCT *psObj,
    DISTANCE_OBJECT hDistance,
    void *pvRegistrationArg,
    DEVICE_UPDATE_CALLBACK vCallback,
    void *pvCallbackArg
        )
{
    DEVICE_REGISTRATION_STRUCT *psReg;

    // Create object
    psReg = (DEVICE_REGISTRATION_STRUCT*)
        SMSO_hCreate(
            DEVICE_OBJECT_NAME": RegStr",
            sizeof(DEVICE_REGISTRATION_STRUCT),
            (SMS_OBJECT)psObj, FALSE);

    if (psReg != NULL)
    {
        // Use the given distance object
        psReg->hDistance = hDistance;

        // Initialize location handle
        psReg->hLastNotifiedLocation = LOCATION_INVALID_OBJECT;

        // Register supplied arguments
        psReg->pvRegistrationArg = pvRegistrationArg;
        psReg->vCallback = vCallback;
        psReg->pvCallbackArg = pvCallbackArg;

        // Initialize entry handle
        psReg->hEntry = OSAL_INVALID_LINKED_LIST_ENTRY;
    }

    return psReg;
}

/*****************************************************************************
 *
 *   bAddRegistrationToList
 *
 *****************************************************************************/
static BOOLEAN bAddRegistrationToList(
    DEVICE_OBJECT_STRUCT *psObj,
    DEVICE_REGISTRATION_STRUCT *psReg
        )
{
    OSAL_RETURN_CODE_ENUM eReturnCode;

    // Check input
    if (psReg == NULL)
    {
        SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
            DEVICE_OBJECT_NAME": invalid input\n");
        return FALSE;
    }

    // Add this registration
    eReturnCode = OSAL.eLinkedListAdd(
        psObj->hRegistrationList,
        &psReg->hEntry, psReg);
    if (eReturnCode != OSAL_SUCCESS)
    {
        SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
            DEVICE_OBJECT_NAME
            ": failed to add desc %p to list %p (%s)",
            psReg, psObj->hRegistrationList,
            OSAL.pacGetReturnCodeName(eReturnCode));

        return FALSE;
    }

    return TRUE;
}

/*****************************************************************************
 *
 *   vDestroyReg
 *
 *****************************************************************************/
static void vDestroyReg (
    DEVICE_REGISTRATION_STRUCT *psReg
        )
{
    if (psReg == NULL)
    {
        return;
    }

    if (psReg->hDistance != DISTANCE_INVALID_OBJECT)
    {
        DISTANCE.vDestroy(psReg->hDistance);
        psReg->hDistance = DISTANCE_INVALID_OBJECT;
    }

    if (psReg->hLastNotifiedLocation != LOCATION_INVALID_OBJECT)
    {
        LOCATION.vDestroy(psReg->hLastNotifiedLocation);
        psReg->hLastNotifiedLocation = LOCATION_INVALID_OBJECT;
    }

    psReg->pvRegistrationArg = NULL;
    psReg->pvCallbackArg = NULL;
    psReg->vCallback = NULL;
    psReg->hEntry = OSAL_INVALID_LINKED_LIST_ENTRY;

    SMSO_vDestroy((SMS_OBJECT)psReg);

    return;
}

/*****************************************************************************
 *
 *   bPrintRegEntry
 *
 *****************************************************************************/
static BOOLEAN bPrintRegEntry(
    DEVICE_REGISTRATION_STRUCT *psReg,
    DEVICE_REG_PRINT_ITERATOR_STRUCT *psIterator
        )
{
    BOOLEAN bContinue = TRUE;

    if ((psReg == NULL) || (psIterator == NULL ))
    {
        // Invalid input parameters
        bContinue = FALSE;
    }
    else
    {
        UN32 n32BytesWritten = 0;

        // Incrementing the counter
        psIterator->un8Counter++;

        // Do it this way for a while...
        n32BytesWritten = fprintf(
            psIterator->psFile,
            "%3d\tCallback:%8.8p\tdistance:%8.8p\n",
            psIterator->un8Counter,
            psReg->vCallback,
            psReg->hDistance);

        if(psIterator->pn32BytesWritten != NULL)
        {
            // Add number of printed characters to global counter
            *psIterator->pn32BytesWritten += n32BytesWritten;
        }
    }

    return bContinue;
}

/*****************************************************************************
 *
 *   bNotifyByLocation
 *
 *****************************************************************************/
static BOOLEAN bNotifyByLocation(
    DEVICE_REGISTRATION_STRUCT *psReg,
    DEVICE_POSITION_NOTIFY_STRUCT *psNotify
        )
{
    BOOLEAN bDoNotification = FALSE;
    BOOLEAN bOk = FALSE;
    DISTANCE_OBJECT hDistance = DISTANCE_INVALID_OBJECT;

    if ((psNotify == NULL) ||
        (psNotify->hLocation == LOCATION_INVALID_OBJECT))
    {
        SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
            DEVICE_OBJECT_NAME
            ": bNotifyByLocation called with invalid psNotify argument.");

        return FALSE;
    }

    do
    {
        // Is this the first location provided to this notification?
        if (psReg->hLastNotifiedLocation == LOCATION_INVALID_OBJECT)
        {
            // Yes! Our last notified location is now this one
            psReg->hLastNotifiedLocation = LOCATION.hDuplicate(psNotify->hLocation);
            if (psReg->hLastNotifiedLocation == LOCATION_INVALID_OBJECT)
            {
                SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                        DEVICE_OBJECT_NAME
                        ": failed to duplicate current LOCATION object "
                        "to last notified for callback (%p)",
                        psReg->vCallback);
                break;
            }

            // We don't have an actual distance, so just use zero
            hDistance = DISTANCE.hCreate(0, DISTANCE_UNIT_TYPE_MILES);
            if (hDistance == DISTANCE_INVALID_OBJECT)
            {
                SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    DEVICE_OBJECT_NAME
                    ": failed to create distance object",
                    psReg->vCallback);

                break;
            }

            // Need to perform this notification
            bDoNotification = TRUE;
        }
        else // No, this registration has a previous location
        {
            BOOLEAN bSameLocation;

            // Make sure that we don't use the same location for distance
            // calculation.
            bSameLocation = LOCATION_bCompare(
                psNotify->hLocation, psReg->hLastNotifiedLocation);
            if (bSameLocation != TRUE)
            {
                N16 n16CompareResult;

                // Calculate distance between latest notified location and current one
                hDistance = LOCATION.hDistance(
                    psNotify->hLocation,
                    psReg->hLastNotifiedLocation);
                if (hDistance == DISTANCE_INVALID_OBJECT)
                {
                    SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                            DEVICE_OBJECT_NAME
                            ": failed to calculate distance between current "
                            "location and latest notified location for "
                            "callback %p",
                            psReg->vCallback);
                    break;
                }

                // Check traveled distance between last notified location
                // and the current location
                n16CompareResult =
                    DISTANCE.n16Compare(psReg->hDistance, hDistance);

                // Is the distance greater than the registered
                // distance?
                if (n16CompareResult <= 0)
                {
                    // Yes! Update latest notified location
                    bOk = bUpdateLocation(
                        &psReg->hLastNotifiedLocation,
                        LOCATION.hLat(psNotify->hLocation),
                        LOCATION.hLon(psNotify->hLocation));
                    if (bOk == FALSE)
                    {
                        SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                                DEVICE_OBJECT_NAME
                               ": failed to update latest notified location "
                               "for callback %p", psReg->vCallback);
                        break;
                    }

                    bDoNotification = TRUE;
                }
            }
        }

        // Do notification if requested
        if (bDoNotification == TRUE)
        {
            if (hDistance != DISTANCE_INVALID_OBJECT)
            {
                psReg->vCallback(
                    psReg->pvRegistrationArg,
                    psNotify->tEventMask,
                    hDistance,
                    psReg->pvCallbackArg);
            }
            else
            {
                SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    DEVICE_OBJECT_NAME
                   ": attempt to notify with invalid distance object");
            }
        }
    }
    while (FALSE);

    if (hDistance != DISTANCE_INVALID_OBJECT)
    {
        DISTANCE.vDestroy(hDistance);
    }

    return TRUE;
}

/*****************************************************************************
 *
 *   bUpdateLocation
 *
 *****************************************************************************/
static BOOLEAN bUpdateLocation(
    LOCATION_OBJECT *phLocation,
    OSAL_FIXED_OBJECT hLat,
    OSAL_FIXED_OBJECT hLon
        )
{
    BOOLEAN bOk = FALSE;

    if (phLocation == NULL)
    {
        return FALSE;
    }

    // Are we updating an existing location?
    if (*phLocation != LOCATION_INVALID_OBJECT)
    {
        // Yes, we need to update it
        N32 n32Lat, n32Lon;
        UN8 un8LatBits, un8LonBits;

        // Extract the raw values from the fixed objects
        n32Lat = OSAL_FIXED.n32Value(hLat);
        un8LatBits = OSAL_FIXED.un8NumFractionalBits(hLat);
        n32Lon = OSAL_FIXED.n32Value(hLon);
        un8LonBits = OSAL_FIXED.un8NumFractionalBits(hLon);

        // Update existing location with that data
        bOk = LOCATION_bUpdateCoordinates(
            *phLocation,
            n32Lat, un8LatBits,
            n32Lon, un8LonBits );
        if (bOk == FALSE)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                DEVICE_OBJECT_NAME
                ": failed to update LOCATION coordinates");
        }
    }
    else // No, we need to create a new location
    {
        // Create the new LOCATION object
        *phLocation = LOCATION.hCreateForRadius(
            hLat, hLon, DISTANCE_INVALID_OBJECT
                );

        if (*phLocation != LOCATION_INVALID_OBJECT)
        {
            bOk = TRUE;
        }
        else
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    DEVICE_OBJECT_NAME
                    ": failed to create LOCATION object");
        }
    }

    return bOk;
}

/*****************************************************************************
 *
 *   bUpdateFixed
 *
 *****************************************************************************/
static BOOLEAN bUpdateFixed(
    OSAL_FIXED_OBJECT hFrom,
    OSAL_FIXED_OBJECT hTo
        )
{
    BOOLEAN bOk = FALSE;

    if ((hFrom == OSAL_FIXED_INVALID_OBJECT) ||
        (hTo == OSAL_FIXED_INVALID_OBJECT))
    {
        return FALSE;
    }

    bOk = OSAL_FIXED.bCopyToMemory(hFrom, hTo);
    if (bOk == FALSE)
    {
        SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                DEVICE_OBJECT_NAME": failed to copy data from %p to %p",
               hFrom, hTo);
    }

    return bOk;
}
