/******************************************************************************/
/*                    Copyright (c) Sirius XM Radio, Inc.                     */
/*                            All Rights Reserved                             */
/*         Licensed Materials - Property of Sirius XM Radio, Inc.             */
/******************************************************************************/
/*******************************************************************************
 *
 * DESCRIPTION
 *
 *  This module contains the DB_UTIL implementation for the
 *  Sirius Module Services (SMS)
 *
 ******************************************************************************/
#include <stdio.h>
#include <string.h>

#include "standard.h"
#include "osal.h"

#include "sms_api.h"
#include "sms.h"
#include "sms_obj.h"

#include "db_util.h"
#include "_db_util.h"

#include "sms_api_debug.h"
static const char *gpacThisFile = __FILE__;

/*****************************************************************************
                             PUBLIC FUNCTIONS
*****************************************************************************/

/*****************************************************************************
                             FRIEND FUNCTIONS
*****************************************************************************/

/*****************************************************************************
*
*   DB_UTIL_hConnectToPersistent
*
* This friend function helps to simplify connections to persistent storage
* databases by different SMS objects.  A persistent storage database has the
* following attributes:
*   -- Doesn't matter if it gets corrupt
*   -- The initial state of the database is blank
*   -- The tables can be re-built at run-time
*   -- If the schema version doesn't match, the tables can be dropped and
*      rebuilt.
*   -- You always want to run a corruption check at the initial connection
*
* Inputs:
*
*   pacDatabaseFile - The path to the persistent storage database
*   bCreateDBHandler - A pointer to a function that would issue all the
*       SQL statements to recreate the database
*   pvCreateArg - An argument that will be supplied to bCreateDBHandler
*   bCheckVersionHandler - A pointer to a function that would issue all the
*       SQL statements to perform a version check of the database.  If the
*       version check fails, the database will be deleted and recreated.
*       The version check is not called if the database was just created.
*   pvCheckArg - An argument that will be supplied to bCheckVersionHandler
*   *peErrorCode - A valid pointer to the error code if the database connection
*       couldn't be made, if the database couldn't be created or if the database
*       couldn't be re-built.
*
* Outputs:
*
*   A valid SQL_INTERFACE_OBJECT on success,
*   otherwise SQL_INTERFACE_INVALID_OBJECT
*
*****************************************************************************/
SQL_INTERFACE_OBJECT DB_UTIL_hConnectToPersistent (
    const char *pacDatabaseFile,
    DB_UTIL_CREATE_DB_HANDLER bCreateDBHandler,
    void *pvCreateArg,
    DB_UTIL_CHECK_VERSION_HANDLER bCheckVersionHandler,
    void *pvCheckArg,
    DATASERVICE_ERROR_CODE_ENUM *peErrorCode
        )
{
    DATASERVICE_ERROR_CODE_ENUM eErrorCode = DATASERVICE_ERROR_CODE_NONE;
    SQL_INTERFACE_OBJECT hSQLConnection = SQL_INTERFACE_INVALID_OBJECT;
    BOOLEAN bOk;
    int iResult;

    do
    {
        if (peErrorCode == NULL)
        {
            peErrorCode = &eErrorCode;
        }

        if (pacDatabaseFile == NULL)
        {
            *peErrorCode = DATASERVICE_ERROR_UNKNOWN;
            break;
        }

        hSQLConnection = hOpenPersistentDB(
            pacDatabaseFile, bCreateDBHandler, pvCreateArg, peErrorCode);

        if (hSQLConnection == SQL_INTERFACE_INVALID_OBJECT)
        {
            if (*peErrorCode == DATASERVICE_ERROR_CODE_DATABASE_CORRUPT)
            {
                int iResult;

                iResult = remove(pacDatabaseFile);

                if (iResult != 0)
                {
                    SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                        DB_UTIL_OBJECT_NAME
                        ": couldn't remove corrupt persistent database %s -- %d",
                        pacDatabaseFile, iResult);

                    *peErrorCode = DATASERVICE_ERROR_CODE_DATABASE_ACCESS_FAILURE;
                    break;
                }

                // Open the connection and recreate the database
                hSQLConnection = hOpenPersistentDB(
                    pacDatabaseFile, bCreateDBHandler, pvCreateArg, peErrorCode);

                if (hSQLConnection == SQL_INTERFACE_INVALID_OBJECT)
                {
                    SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                        DB_UTIL_OBJECT_NAME
                        ": couldn't recreate persistent database after corruption check failed -- %s ",
                        pacDatabaseFile);
                    break;
                }
            }

            break;
        }

        if (*peErrorCode != DATASERVICE_ERROR_CODE_NONE)
        {
            // Not sure what is going on
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                DB_UTIL_OBJECT_NAME
                ": unexpected error code after opening %s",
                pacDatabaseFile);

            SQL_INTERFACE.vDisconnect(hSQLConnection);
            hSQLConnection = SQL_INTERFACE_INVALID_OBJECT;
            break;
        }

        if (bCheckVersionHandler == NULL)
        {
            // We aren't erroring out, just stopping
            // since we don't have anymore work to do with this
            // connection
            break;
        }

        bOk = bCheckVersionHandler(hSQLConnection, pvCheckArg);

        if (bOk == TRUE)
        {
            // We aren't erroring out, just stopping
            // since we don't have anymore work to do with this
            // connection
            break;
        }

        SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                DB_UTIL_OBJECT_NAME
                ": call to bCheckVersionHandler failed while verifying the version of %s",
                pacDatabaseFile);

        if (bCreateDBHandler == NULL)
        {
            // We failed a version check and there is no way
            // to recreate this database.  Indicate the error
            // and stop here.
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                DB_UTIL_OBJECT_NAME
                ": no way to recreate database %s",
                pacDatabaseFile);

            *peErrorCode = DATASERVICE_ERROR_CODE_DATABASE_VERSION_MISMATCH;

            SQL_INTERFACE.vDisconnect(hSQLConnection);
            hSQLConnection = SQL_INTERFACE_INVALID_OBJECT;

            break;
        }

        // Close the database, delete the database since the
        // version doesn't match, then rebuild it.
        SQL_INTERFACE.vDisconnect(hSQLConnection);
        hSQLConnection = SQL_INTERFACE_INVALID_OBJECT;

        iResult = remove(pacDatabaseFile);

        if (iResult != 0)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                DB_UTIL_OBJECT_NAME
                ": couldn't remove persistent database %s -- %d",
                pacDatabaseFile, iResult);

            *peErrorCode = DATASERVICE_ERROR_CODE_DATABASE_ACCESS_FAILURE;
            break;
        }

        // Open the connection and recreate the database
        hSQLConnection = hOpenPersistentDB(
            pacDatabaseFile, bCreateDBHandler, pvCreateArg, peErrorCode);

        if (hSQLConnection == SQL_INTERFACE_INVALID_OBJECT)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                DB_UTIL_OBJECT_NAME
                ": couldn't recreate persistent database after version check failed -- %s ",
                pacDatabaseFile);
            break;
        }

    } while (FALSE);

    return hSQLConnection;
}

/*****************************************************************************
*
*   DB_UTIL_eCheckReferenceBanks
*
*****************************************************************************/
DATASERVICE_ERROR_CODE_ENUM DB_UTIL_eCheckReferenceBanks (
    const char *pacDatabaseFileA,
    const char *pacDatabaseFileB,
    DB_UTIL_EXTRACT_CONTENT_VERSION_HANDLER n32ExtractVersionHandler,
    void *pvExtractVersionHandlerArg,
    size_t tVersionBitWidth,
    DATASERVICE_REF_DATA_VER *ptCurrentVer,
    DATASERVICE_REF_DATA_VER *ptNextVer
        )
{
    return eCheckReferenceBanks(
        pacDatabaseFileA, pacDatabaseFileB,
        NULL, NULL,
        n32ExtractVersionHandler,
        pvExtractVersionHandlerArg,
        ptCurrentVer, ptNextVer,
        NULL, tVersionBitWidth);
}

/*****************************************************************************
*
*   DB_UTIL_hConnectToReferenceBank
*
* This friend function helps to simplify connections to reference storage
* database banks.  This function selects which database a caller should
* connect to and cleans up any database which has been deprecated.
*
*****************************************************************************/
SQL_INTERFACE_OBJECT DB_UTIL_hConnectToReferenceBank (
    const char *pacDatabaseFileA,
    const char *pacDatabaseFileB,
    const char **ppacSelectedFile,
    DB_UTIL_EXTRACT_CONTENT_VERSION_HANDLER n32ExtractVersionHandler,
    void *pvn32ExtractVersionHandlerArg,
    BOOLEAN bDeleteExpiredDB,
    BOOLEAN *pbDatabaseSwapped,
    size_t tVersionBitWidth,
    DB_UTIL_CHECK_VERSION_HANDLER bCheckVersionHandler,
    void *pvCheckArg,
    DATASERVICE_ERROR_CODE_ENUM *peErrorCode,
    SQL_INTERFACE_OPTIONS_MASK tOptions
        )
{
    DATASERVICE_ERROR_CODE_ENUM eErrorCode = DATASERVICE_ERROR_CODE_GENERAL;
    const char *pacSelectedDatabaseFile = NULL, 
               *pacDeletedDatabaseFile = NULL;

    // Verify inputs
    // Check that both dbs exist
    if ((pacDatabaseFileA == NULL) ||
        (pacDatabaseFileB == NULL) ||
        (ppacSelectedFile == NULL))
    {
        return SQL_INTERFACE_INVALID_OBJECT;
    }

    if (peErrorCode == NULL)
    {
        peErrorCode = &eErrorCode;
    }

    *peErrorCode = eCheckReferenceBanks(
        pacDatabaseFileA, pacDatabaseFileB,
        &pacDeletedDatabaseFile, &pacSelectedDatabaseFile,
        n32ExtractVersionHandler, pvn32ExtractVersionHandlerArg, 
        NULL, NULL, pbDatabaseSwapped,
        tVersionBitWidth);

    // If we have a file which is old and ready
    // to be deleted ensure the caller wants
    // old files deleted
    if ((pacDeletedDatabaseFile != NULL) &&
        (bDeleteExpiredDB == TRUE))
    {
        remove(pacDeletedDatabaseFile);
    }

    if (pacSelectedDatabaseFile != NULL)
    {
        *ppacSelectedFile = pacSelectedDatabaseFile;

        // Finally make the connection
        return DB_UTIL_hConnectToReference(
            pacSelectedDatabaseFile,
            bCheckVersionHandler, pvCheckArg,
            peErrorCode, tOptions);
    }

    // Failed -- we already set the error code above
    return SQL_INTERFACE_INVALID_OBJECT;
}

/*****************************************************************************
*
*   DB_UTIL_hConnectToReference
*
* This friend function helps to simplfy connections to reference storage
* databases by different SMS objects.  A reference storage database has the
* following attributes:
*   -- You can't rebuild it at run-time
*   -- It matters if it is corrupt
*   -- It matters if the version check fails
*   -- You only want to run corruption checks when the database has been
*      written to.
*
* Inputs:
*
*   pacDatabaseFile - The path to the reference storage database
*   bCheckVersionHandler - A pointer to a function that would issue all the
*       SQL statements to perform a version check of the database.  If the
*       version check fails, an error code will be indicated
*   pvCheckArg - An argument that will be supplied to bCheckVersionHandler
*   *peErrorCode - A valid pointer to the error code if the database connection
*       couldn't be made, if the database couldn't be found, if the
*       version check failed or if the database was corrupt.
*   tOptions - A SQL_INTERFACE_OPTIONS_MASK specifing the option mask to use
*              when opening the database.
*
* Outputs:
*
*   A valid SQL_INTERFACE_OBJECT on success,
*   otherwise SQL_INTERFACE_INVALID_OBJECT
*
*****************************************************************************/
SQL_INTERFACE_OBJECT DB_UTIL_hConnectToReference (
    const char *pacDatabaseFile,
    DB_UTIL_CHECK_VERSION_HANDLER bCheckVersionHandler,
    void *pvCheckArg,
    DATASERVICE_ERROR_CODE_ENUM *peErrorCode,
    SQL_INTERFACE_OPTIONS_MASK tOptions
        )
{
    DATASERVICE_ERROR_CODE_ENUM eErrorCode = DATASERVICE_ERROR_CODE_NONE;
    SQL_INTERFACE_OBJECT hSQLConnection = SQL_INTERFACE_INVALID_OBJECT;
    BOOLEAN bOk;

    do
    {
        if (peErrorCode == NULL)
        {
            peErrorCode = &eErrorCode;
        }

        if (pacDatabaseFile == NULL)
        {
            *peErrorCode = DATASERVICE_ERROR_UNKNOWN;
            break;
        }

        hSQLConnection = SQL_INTERFACE.hConnect(
                    pacDatabaseFile,
                    tOptions,
                    peErrorCode);

        if (hSQLConnection == SQL_INTERFACE_INVALID_OBJECT)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                DB_UTIL_OBJECT_NAME
                ": failed to connect to reference database %s",
                pacDatabaseFile);
            break;
        }

        if (*peErrorCode != DATASERVICE_ERROR_CODE_NONE)
        {
            // Not sure what is going on
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                DB_UTIL_OBJECT_NAME
                ": unexpected error code after opening %s",
                pacDatabaseFile);

            SQL_INTERFACE.vDisconnect(hSQLConnection);
            hSQLConnection = SQL_INTERFACE_INVALID_OBJECT;

            break;
        }

        if (bCheckVersionHandler == NULL)
        {
            // Not an error, we are just all done
            break;
        }

        bOk = bCheckVersionHandler(hSQLConnection, pvCheckArg);

        if (bOk == FALSE)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                DB_UTIL_OBJECT_NAME
                ": call to bCheckVersionHandler failed while verifying the version of %s",
                pacDatabaseFile);

            *peErrorCode = DATASERVICE_ERROR_CODE_DATABASE_VERSION_MISMATCH;

            SQL_INTERFACE.vDisconnect(hSQLConnection);
            hSQLConnection = SQL_INTERFACE_INVALID_OBJECT;

            break;
        }

    } while (FALSE);

    return hSQLConnection;
}

/*****************************************************************************
*
*   DB_UTIL_bUpdateTimestamp
*
*   This friend function allows a data service to
*   the interface to indicate a database update has been completed and
*   a report the new database version number.
*
* Inputs:
*
*   pacDatabaseFile - The path to the reference storage database
*
* Outputs:
*
*   TRUE if the flag was set.  FALSE if there was an error
*
*****************************************************************************/
BOOLEAN DB_UTIL_bUpdateTimestamp(
    SQL_INTERFACE_OBJECT hConnection,
    char *pcSQLCommandBuffer,
    size_t tBufferSize
        )
{
    BOOLEAN  bResult = FALSE;

    do
    {
        N32                   n32Result    = 0;
        OSAL_RETURN_CODE_ENUM eResult      = OSAL_ERROR_UNKNOWN;
        UN32                  un32UnixTime = 0;


        // Grab the ( Unix Epoch ) time from OSAL
        eResult = OSAL.eTimeGet( &un32UnixTime );
        if ( eResult != OSAL_SUCCESS )
        {
            SMSAPI_DEBUG_vPrintErrorFull( gpacThisFile, __LINE__,
                    DB_UTIL_OBJECT_NAME": Unable to get OSAL time." );
            break;
        }

        // Generate the SQL command to update the timestamp row
        // in the database
        n32Result = snprintf( pcSQLCommandBuffer, tBufferSize,
                              DB_UTIL_UPDATE_TIMESTAMP_ROW,
                              un32UnixTime, DB_UTIL_TIMESTAMP_TYPE_DB );
        if ( n32Result <= 0 )
        {
            break;
        }

        // Update the database timestamp row
        bResult = SQL_INTERFACE.bExecuteCommand( hConnection, pcSQLCommandBuffer );
        if (bResult == FALSE)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    DB_UTIL_OBJECT_NAME": Unable to set database timestamp.");
        }

    } while ( FALSE );

    return bResult;
}

/*****************************************************************************
*
*   DB_UTIL_bCreateFullFilePath
*
* This friend function builds a database path using the parameters given.
*
* Inputs:
*
*   pcContainingDirectoryPath - The path to the containing directory,
*           if known
*   pcFileDirectory - The folder path that is relative to the top-level
*           SMS directory. Only used if pcContainingDirectoryPath==NULL
*   pcFilename - The name of the database
*   ppcFullPath - The full path will be stored here.  It is up to the caller
*                   to free this memory with SMSO_vDestroy
*
* Outputs:
*
*   TRUE on success, FALSE on error
*
* Examples:
*   pcContainingDirectoryPath is: "/pathA"
*   pcFileDirectory is: "serviceDir" <-- IGNORED IN THIS CASE
*   pcFilename is :service.db
*      Result: /pathA/service.db
*
*
*   pcContainingDirectoryPath is: "/pathA/"
*   pcFileDirectory is: "serviceDir" <-- IGNORED IN THIS CASE
*   pcFilename is :service.db
*      Result: /pathA/service.db
*
*
*   pcContainingDirectoryPath is: NULL
*   pcFileDirectory is: "serviceDir"
*   pcFilename is :service.db
*      Result: <sms_base_path/serviceDir/service.db
*
*
*****************************************************************************/
BOOLEAN DB_UTIL_bCreateFilePath (
    const char *pcContainingDirectoryPath,
    const char *pcFileDirectory,
    const char *pcFilename,
    char **ppcFullPath
        )
{
    size_t tDirNameLen = 0,
           tBasePathLen = 0,
           tFilenameLen = 0,
           tDBPathLen = 0;
    BOOLEAN bSuccess = FALSE;

    // These two must always be present
    if (ppcFullPath == NULL)
    {
        return FALSE;
    }

    if (pcContainingDirectoryPath == NULL)
    {
        // Validate this pointer only when
        // necessary
        if (pcFileDirectory == NULL)
        {
            return FALSE;
        }

        // Caller doesn't know where to find
        // this file & directory -- so point
        // them to the SMS working directory
        pcContainingDirectoryPath = SMS_pacGetPath();

        // Compute the length of this string now
        // + 1 for delimiter that we'll add
        tDirNameLen = strlen(pcFileDirectory) + 1;
    }

    // This has to be valid now
    if (pcContainingDirectoryPath == NULL)
    {
        return FALSE;
    }

    // Compute the length of each string

    // + 1 For the delimiter we'll add
    tBasePathLen = strlen(pcContainingDirectoryPath) + 1;

    if ((const char *)NULL != pcFilename)
    {
        tFilenameLen = strlen(pcFilename);
    }

    // + 1 for the NULL character at the end
    tDBPathLen = tBasePathLen +
                 tDirNameLen +
                 tFilenameLen + 1;

    // Now we know the size of the path, so we can now
    // allocate the proper amount of memory
    *ppcFullPath =
        (char *) SMSO_hCreate(
            DB_UTIL_OBJECT_NAME":DBFilePath",
            tDBPathLen,
            SMS_INVALID_OBJECT, FALSE );

    // Ensure allocation succeeded
    if (*ppcFullPath != NULL)
    {
        size_t tWriteIndex;
        char *pcFullPath = *ppcFullPath;

        // Write the containing directory path
        tWriteIndex = snprintf(
            &pcFullPath[0],
            tBasePathLen + 1,
            "%s/",
            pcContainingDirectoryPath);

        if ((pcContainingDirectoryPath[tBasePathLen - 2] == '\\') ||
            (pcContainingDirectoryPath[tBasePathLen - 2] == '/'))
        {
            // Normalize this so it never ends in a delimiter
            tWriteIndex--;
        }

        // Write the directory name, if necessary
        if (tDirNameLen != 0)
        {
            tWriteIndex += snprintf(
                &pcFullPath[tWriteIndex],
                tDirNameLen + 1,
                "%s/",
                pcFileDirectory);
        }

        if ((const char *)NULL != pcFilename)
        {
            // Write the file name
            snprintf(
                &pcFullPath[tWriteIndex],
                tFilenameLen + 1,
                "%s",
                pcFilename);
        }

        bSuccess = TRUE;
    }

    return bSuccess;
}

/*****************************************************************************
*
*   DB_UTIL_bCopyDB
*
* This friend function copies the pre-existing SQLite DB found at pcDstDB
* to the path pcSrcDB, ensuring page sizes are equal between the two.
*
* Inputs:
*
*   pcDstDB - The DB file to copy.
*   pcSrcDB - The path at which to generate a copy pcDstDB.
*
* Outputs:
*
*   TRUE on success, FALSE on error
*
*****************************************************************************/
BOOLEAN DB_UTIL_bCopyDB (
    const char *pcDstDB,
    const char *pcSrcDB
        )
{
    SQL_INTERFACE_OBJECT hSrcConnection = SQL_INTERFACE_INVALID_OBJECT,
                         hDstConnection = SQL_INTERFACE_INVALID_OBJECT;
    BOOLEAN bCopied = FALSE;

    if ((pcDstDB == NULL) || (pcSrcDB == NULL))
    {
        return FALSE;
    }

    do
    {
        // Remove any file already located at psDstDB
        remove(pcDstDB);

        // Open a connection to Src DB as read-only
        hSrcConnection = SQL_INTERFACE.hConnect(
            pcSrcDB,
            SQL_INTERFACE_OPTIONS_READONLY |
            SQL_INTERFACE_OPTIONS_SKIP_CORRUPTION_CHECK,
            NULL);

        if (hSrcConnection == SQL_INTERFACE_INVALID_OBJECT)
        {
            break;
        }

        // Open a connection to dst DB, create if it
        // isn't there
        hDstConnection = SQL_INTERFACE.hConnect(
            pcDstDB,
            SQL_INTERFACE_OPTIONS_CREATE_IF_NOT_FOUND |
            SQL_INTERFACE_OPTIONS_SKIP_CORRUPTION_CHECK,
            NULL);

        if (hDstConnection == SQL_INTERFACE_INVALID_OBJECT)
        {
            break;
        }

        // Perform the copy
        bCopied = SQL_INTERFACE.bCopyMainDatabase(hDstConnection, hSrcConnection);
    } while (FALSE);

    if (hSrcConnection != SQL_INTERFACE_INVALID_OBJECT)
    {
        SQL_INTERFACE.vDisconnect(hSrcConnection);
    }

    if (hDstConnection != SQL_INTERFACE_INVALID_OBJECT)
    {
        SQL_INTERFACE.vDisconnect(hDstConnection);
    }

    return bCopied;
}

/*****************************************************************************
                             PRIVATE FUNCTIONS
*****************************************************************************/


/*******************************************************************************
*
*   bProcessSelectTimestamp
*
*   This function is provided to the SQL_INTERFACE_OBJECT in order to process
*   a "select all" query made on the database timestamp table.  It populates
*   a pointer to a DB_UTIL_TIMESTAMP_ROW_STRUCT with the information found
*   in the database as well as performs timestamp verification.
*
*   Although we (currently) only have a timestamp entry for the last database
*   update, this infrastructure should allow for easy expansion wrt/any new
*   _TIMESTAMP_TYPES_ENUM values.
*
*******************************************************************************/
static BOOLEAN bProcessSelectTimestamp (
    SQL_QUERY_COLUMN_STRUCT *psColumn,
    N32 n32NumberOfColumns,
    DB_UTIL_TIMESTAMP_RESULT_STRUCT *psResult
        )
{
    DB_UTIL_TIMESTAMP_FIELDS_ENUM eCurrentField =
        (DB_UTIL_TIMESTAMP_FIELDS_ENUM)0;
    DB_UTIL_TIMESTAMP_ROW_STRUCT sCurrentRow;

    // Verify input
    if (psResult == NULL)
    {
        return FALSE;
    }

    // Initialize row structure
    sCurrentRow.eType = DB_UTIL_TIMESTAMP_MAX_TYPES;
    sCurrentRow.un32Timestamp = DB_UTIL_INVALID_TIMESTAMP;

    // If we have the correct number of columns, then we have good results
    if ( n32NumberOfColumns == DB_UTIL_TIMESTAMP_MAX_FIELDS )
    {
        // Start off by marking this a success
        psResult->bSuccess = TRUE;
    }
    else
    {
        // This is definitely not a match
        psResult->bSuccess = FALSE;
        return FALSE;
    }

    do
    {
        // Decode the current field based on it's type (we only have a
        // whole-database timestamp at present.)
        switch (eCurrentField)
        {
            case DB_UTIL_TIMESTAMP_TYPE:
            {
                sCurrentRow.eType = (DB_UTIL_TIMESTAMP_TYPES_ENUM)
                    psColumn[eCurrentField].uData.sUN32.un32Data;
            }
            break;

            case DB_UTIL_TIMESTAMP_TIME:
            {
                sCurrentRow.un32Timestamp =
                    psColumn[eCurrentField].uData.sUN32.un32Data;
            }
            break;

            default:
            {
                SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                        DB_UTIL_OBJECT_NAME": bad field in the timestamp table");

                psResult->bSuccess = FALSE;
            }
            break;
        }

    } while ( ++eCurrentField < DB_UTIL_TIMESTAMP_MAX_FIELDS);

    // Now process the version data
    switch(sCurrentRow.eType)
    {
        case DB_UTIL_TIMESTAMP_TYPE_DB:
        {
            // Just pull this value in
            psResult->tTimestamp =
                    (TIME_T)sCurrentRow.un32Timestamp;
        }
        break;

        default:
        {
            psResult->bSuccess = FALSE;
        }
    }

    // FALSE will stop iteration; TRUE will keep going
    return psResult->bSuccess;
}

/*****************************************************************************
*
*   un32ExtractDatabaseTimestamp
*
*****************************************************************************/
static TIME_T tExtractDatabaseTimestamp  (
    SQL_INTERFACE_OBJECT hConnection
        )
{
    BOOLEAN bOk;
    DB_UTIL_TIMESTAMP_RESULT_STRUCT sTimestamp;
    TIME_T tResult = DB_UTIL_INVALID_TIMESTAMP;

    if (hConnection == SQL_INTERFACE_INVALID_OBJECT)
    {
        return DB_UTIL_INVALID_TIMESTAMP;
    }

    // Init the result struct
    OSAL.bMemSet(&sTimestamp, 0, sizeof(sTimestamp));

    // Perform the SQL query and process the result
    // (it will provide us with a data row)
    bOk = SQL_INTERFACE.bQuery(
            hConnection, DB_UTIL_SELECT_ALL_TIMESTAMPS,
            (SQL_QUERY_RESULT_HANDLER)bProcessSelectTimestamp,
            &sTimestamp ) ;

    if ( ( bOk == TRUE ) &&
         ( sTimestamp.bSuccess == TRUE ) )
    {
        tResult = sTimestamp.tTimestamp;
    }

    return tResult;
}


/*****************************************************************************
*
*   hOpenPersistentDB
*
*****************************************************************************/
static SQL_INTERFACE_OBJECT hOpenPersistentDB (
    const char *pacDatabaseFile,
    DB_UTIL_CREATE_DB_HANDLER bCreateDBHandler,
    void *pvCreateArg,
    DATASERVICE_ERROR_CODE_ENUM *peErrorCode
        )
{
    SQL_INTERFACE_OBJECT hSQLConnection = SQL_INTERFACE_INVALID_OBJECT;
    BOOLEAN bOk;

    do
    {
        hSQLConnection = SQL_INTERFACE.hConnect(
                    pacDatabaseFile,
                    SQL_INTERFACE_OPTIONS_CREATE_IF_NOT_FOUND,
                    peErrorCode);

        if (hSQLConnection == SQL_INTERFACE_INVALID_OBJECT)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                DB_UTIL_OBJECT_NAME
                ": failed to connect to persistent database %s",
                pacDatabaseFile);
            break;
        }

        if (*peErrorCode == DATASERVICE_ERROR_CODE_DATABASE_NOT_FOUND)
        {
            // We created a connection, but the error code indicates
            // that a new database was created.  Tell the caller
            // to initialize the new database how they see fit

            if (bCreateDBHandler != NULL)
            {
                bOk = bCreateDBHandler(hSQLConnection, pvCreateArg);

                if (bOk == FALSE)
                {
                    int iResult;

                    SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                        DB_UTIL_OBJECT_NAME
                        ": call to bCreateDBHandler failed while creating %s",
                        pacDatabaseFile);

                    // Assuming since we cannot create this DB we should not use
                    // it from here
                    SQL_INTERFACE.vDisconnect(hSQLConnection);
                    hSQLConnection = SQL_INTERFACE_INVALID_OBJECT;

                    // Remove file since we've create it here from scratch
                    iResult = remove(pacDatabaseFile);
                    if (iResult != 0)
                    {
                        SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                            DB_UTIL_OBJECT_NAME
                            ": couldn't remove persistent database %s -- %d",
                            pacDatabaseFile, iResult);
                    }

                    *peErrorCode = DATASERVICE_ERROR_CODE_DATABASE_ACCESS_FAILURE;
                }
                else
                {
                    *peErrorCode = DATASERVICE_ERROR_CODE_NONE;
                }
            }
        }
    } while (FALSE);

    return hSQLConnection;
}

/*****************************************************************************
*
*   eCheckReferenceBanks
*
*****************************************************************************/
static DATASERVICE_ERROR_CODE_ENUM eCheckReferenceBanks (
    const char *pacDatabaseFileA,
    const char *pacDatabaseFileB,
    const char **ppacOldFile,
    const char **ppacNewFile,
    DB_UTIL_EXTRACT_CONTENT_VERSION_HANDLER n32ExtractVersionHandler,
    void *pvExtractVersionHandlerArg,
    DATASERVICE_REF_DATA_VER *ptCurrentVer,
    DATASERVICE_REF_DATA_VER *ptNextVer,
    BOOLEAN *pbDatabaseSwapped,
    size_t tVersionBitWidth
        )
{
    DATASERVICE_ERROR_CODE_ENUM eErrorCode = DATASERVICE_ERROR_CODE_NONE;
    SQL_INTERFACE_OBJECT hSQLConnectionA = SQL_INTERFACE_INVALID_OBJECT,
        hSQLConnectionB = SQL_INTERFACE_INVALID_OBJECT;
    const char *pacSelectedDatabaseFile, *pacDeletedDatabaseFile;
    N32 n32VerA = DB_UTIL_DB_UNDER_CONSTRUCTION_VER, 
        n32VerB = DB_UTIL_DB_UNDER_CONSTRUCTION_VER,
        n32CurrentVer, n32NextVer;
    BOOLEAN bDBSwapped;

    // Using own variables if caller does not want some optional info
    if (ptCurrentVer == NULL)
    {
        ptCurrentVer = (DATASERVICE_REF_DATA_VER *)&n32CurrentVer;
    }

    if (ptNextVer == NULL)
    {
        ptNextVer = (DATASERVICE_REF_DATA_VER *)&n32NextVer;
    }

    if (pbDatabaseSwapped == NULL)
    {
        pbDatabaseSwapped = &bDBSwapped;
    }

    if (ppacOldFile == NULL)
    {
        ppacOldFile = &pacDeletedDatabaseFile;
    }

    if (ppacNewFile == NULL)
    {
        ppacNewFile = &pacSelectedDatabaseFile;
    }

    do
    {
        SMS_VERSION_COMPARE_ENUM eCompareResult = SMS_VERSION_EQUAL;
        BOOLEAN bRolloverIndication = FALSE;

        // Initialize DB versions
        *ptCurrentVer = DB_UTIL_DB_UNDER_CONSTRUCTION_VER;
        *ptNextVer = DB_UTIL_DB_UNDER_CONSTRUCTION_VER;

        // Initialize return paths
        *ppacNewFile = NULL;
        *ppacOldFile = NULL;

        // Verify inputs
        if ((pacDatabaseFileA == NULL) ||
            (pacDatabaseFileB == NULL))
        {
            eErrorCode = DATASERVICE_ERROR_CODE_GENERAL;
            break;
        }

        // Connect to Database A
        hSQLConnectionA = SQL_INTERFACE.hConnect(
            pacDatabaseFileA,
            SQL_INTERFACE_OPTIONS_READONLY |
            SQL_INTERFACE_OPTIONS_SKIP_CORRUPTION_CHECK,
            NULL);

        // Connect to Database B
        hSQLConnectionB = SQL_INTERFACE.hConnect(
            pacDatabaseFileB,
            SQL_INTERFACE_OPTIONS_READONLY |
            SQL_INTERFACE_OPTIONS_SKIP_CORRUPTION_CHECK,
            NULL);

        // If we only have 1 usable database, then we're all done
        if (hSQLConnectionB == SQL_INTERFACE_INVALID_OBJECT)
        {
            // Use database file A
            *ppacNewFile = pacDatabaseFileA;

            if (hSQLConnectionA == SQL_INTERFACE_INVALID_OBJECT)
            {
                eErrorCode = DATASERVICE_ERROR_CODE_DATABASE_NOT_FOUND;
            }
            else
            {
                // Set both current and next versions to the same value
                *ptCurrentVer = *ptNextVer = (DATASERVICE_REF_DATA_VER)
                    n32ExtractVersionHandler(hSQLConnectionA,
                                             pvExtractVersionHandlerArg);
            }
            break;
        }

        if (hSQLConnectionA == SQL_INTERFACE_INVALID_OBJECT)
        {
            // Use database file B
            *ppacNewFile = pacDatabaseFileB;
            // Set both current and next versions to the same value
            *ptCurrentVer = *ptNextVer = (DATASERVICE_REF_DATA_VER)
                n32ExtractVersionHandler(hSQLConnectionB,
                                         pvExtractVersionHandlerArg);
            break;
        }

        // We have two databases -- compare them now

        // Get the version/status of each db
        n32VerA = n32ExtractVersionHandler(hSQLConnectionA,
                                           pvExtractVersionHandlerArg);
        n32VerB = n32ExtractVersionHandler(hSQLConnectionB,
                                           pvExtractVersionHandlerArg);

        if ((n32VerA == (N32)DB_UTIL_DB_UNDER_CONSTRUCTION_VER) &&
            (n32VerB == (N32)DB_UTIL_DB_UNDER_CONSTRUCTION_VER))
        {
            // Well, that's not good
            eErrorCode = DATASERVICE_ERROR_CODE_DATABASE_NOT_FOUND;
            break;
        }

        if (n32VerA == (N32)DB_UTIL_DB_UNDER_CONSTRUCTION_VER)
        {
            // Use database file B
            *ppacNewFile = pacDatabaseFileB;
            *ptCurrentVer = *ptNextVer = (DATASERVICE_REF_DATA_VER)n32VerB;
            break;
        }
        else if (n32VerB == (N32)DB_UTIL_DB_UNDER_CONSTRUCTION_VER)
        {
            // Use database file A
            *ppacNewFile = pacDatabaseFileA;
            *ptCurrentVer = *ptNextVer = (DATASERVICE_REF_DATA_VER)n32VerA;
            break;
        }

        // If neither database is under construction, then
        // do a numeric version compare, checking for rollover.
        // The version bit width will vary by service.

        eCompareResult = SMS_eCompareVersions(
            tVersionBitWidth,
            n32VerB,
            n32VerA,
            &bRolloverIndication);

        // Ensure that the comparison occurred
        if ( eCompareResult == SMS_VERSION_COMPARISON_INVALID )
        {
            // Well that's not good!
            eErrorCode = DATASERVICE_ERROR_CODE_DATABASE_ACCESS_FAILURE;
            break;
        }

        //
        // If a rollover may have occurred, then we fall back to comparing
        // database update timestamps.
        //

        if ( bRolloverIndication == TRUE )
        {
            TIME_T tFileModificationTimeA;
            TIME_T tFileModificationTimeB;

            tFileModificationTimeA = tExtractDatabaseTimestamp(hSQLConnectionA);
            tFileModificationTimeB = tExtractDatabaseTimestamp(hSQLConnectionB);

            // Compare modification dates for each database
            if ( tFileModificationTimeA >= tFileModificationTimeB )
            {
                // Use database file A, trash B
                *ppacNewFile = pacDatabaseFileA;
                *ppacOldFile = pacDatabaseFileB;
                *ptNextVer = (DATASERVICE_REF_DATA_VER)n32VerA;
                *ptCurrentVer = (DATASERVICE_REF_DATA_VER)n32VerB;
            }
            else
            {
                // Use database file B, trash A
                *ppacNewFile = pacDatabaseFileB;
                *ppacOldFile = pacDatabaseFileA;
                *ptNextVer = (DATASERVICE_REF_DATA_VER)n32VerB;
                *ptCurrentVer = (DATASERVICE_REF_DATA_VER)n32VerA;
            }
        }
        else
        {
            // If we're here, there's no sign of version rollover, so
            // we're just going to do a straight numeric version
            // comparison.

            // n32VerA < n32VerB
            if ( eCompareResult == SMS_VERSION_LESSER )
            {
                // Use database file B, trash A
                *ppacNewFile = pacDatabaseFileB;
                *ppacOldFile = pacDatabaseFileA;
                *ptNextVer = (DATASERVICE_REF_DATA_VER)n32VerB;
                *ptCurrentVer = (DATASERVICE_REF_DATA_VER)n32VerA;
            }
            // n32VerA >= n32VerB
            else
            {
                // Use database file A, trash B
                *ppacNewFile = pacDatabaseFileA;
                *ppacOldFile = pacDatabaseFileB;
                *ptNextVer = (DATASERVICE_REF_DATA_VER)n32VerA;
                *ptCurrentVer = (DATASERVICE_REF_DATA_VER)n32VerB;
            }
        }

        // We've swapped the databases!
        *pbDatabaseSwapped = TRUE;

    } while (FALSE);

    if (hSQLConnectionA != SQL_INTERFACE_INVALID_OBJECT)
    {
        SQL_INTERFACE.vDisconnect(hSQLConnectionA);
    }

    if (hSQLConnectionB != SQL_INTERFACE_INVALID_OBJECT)
    {
        SQL_INTERFACE.vDisconnect(hSQLConnectionB);
    }

    return eErrorCode;
}
